#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#   abaqus2fistr.py
#
#       abaqus から fistr へメッシュ変換
#
#       abaqusType  node    fistrType
#   -----------------------------------------------------------
#       C3D4        4       341             四面体1次（tetra）
#       C3D10       10      342             四面体2次
#       C3D8        8       361             六面体1次（hexa）
#       C3D8R       8       361              ↑
#       C3D20       20      362             六面体2次
#       C3D20R      20       ↑              ↑
#       C3D6        6       351             五面体1次（prism）
#       C3D15       15      352             五面体2次
#       STRI35      3       731,761(S341)   三角形1次表面要素
#       S3          3       731,761(S341)    ↑
#       CPS3        3       731,761(S341)    ↑
#       S6          6       732(S342)       三角形2次表面要素
#       S4R5        4       741,781(S361)   四角形1次表面要素
#       S4          4       741,781(S361)    ↑
#       S8          8       (S362)          四角形2次表面要素
#       B31         2       611,641         beam
#       T3D2        2       611,641         beam (トラス要素をbeamとして変換）
#
#       *NODE               !NODE
#       *ELEMENT            !ELEMENT
#       *ELSET              !EGROUP
#       *SURFACE            !SGROUP
#       *NSET               !NGROUP
#
#---------------------------
#   15/03/19    新規作成
#      03/27    mashDataを数値で読み込みに変更
#      03/29    SGROUPを設定
#      03/30    5面体要素を追加
#      04/09    selValNameを修正
#      04/13    separateElset:高速化（直接global変数を操作する様に修正）
#      04/29    elementNoをインデックスでそのまま保存して検索を高速化
#      05/07    elementListデータをsortしてface面を検索（高速化の為）
#      05/08    nodeNoをインデクスでそのnodeを含むelement、typeのlist作成（高速化）
#      05/21    その他のsurGroupList「otherS」グループを作成
#      09/01    windows対応のため、os.sepに修正
#      09/02    "\n"→os.linesepに修正
#      10/18    getDataList:len(word)>0 and word!="\n"に修正
#               file読み込み時、"\n"行は読み飛ばす。
#      10/30    deleteLsOfLine:追加
#      12/22    changeNodeOrder351,changeNodeOrder352:バグ修正。
#               prismのnode順番が間違っていた為、四角形部にCLOADを設定して計算させた時
#               変位の正負が反対に計算されていた。これを修正。
#      12/24    getOtherSurGroupList:inpファイル内でsurfaceが定義されていない場合は、
#               エラーが発生していたので、修正。
#      12/25    行末コードlsを修正
#   16/01/08    getNodeElementData:inpファイルに「*NODE PRINT」が存在した場合エラー
#               発生したので、修正
#               saveData,checkNodeGroup:NSETのGENERATEに対応。
#      01/10    windows用のinpファイルを読み込んだ時、変換できない。
#               inpファイル読み込み時に、行末を"\n"に統一。
#               getDataList:座標が省略された場合にも対応できる様に修正
#      01/15    「*SURFACE」の定義の変換を追加
#               六面体1次2次要素の節点No順を修正（abaqusと同じ順に修正）
#      01/16    remakeElementHeaderData:EGRP名の設定を修正
#      02/03    getNodeElementData:大文字小文字の区別なくkeywordを取得
#      02/04    getDataList:deleteLsOfLine→deleteSpに変更
#      02/05    小文字のgroup名を残すように修正
#      03/24    createSurfaceNodesList:surfaceの定義が無い場合、ｴﾗｰ発生を修正
#      03/27    シェル要素（731、741）を追加
#      03/31    checkElementGroup:elTypeが複数ある場合にfindElementTypeを追加
#      04/01    複数のelTypeがあり、egrpを設定している場合、otherグループを設定
#      04/08    シェル要素（761、781）追加
#      04/11    getElementFaceNo:バグ修正（surData=[]の場所変更）
#      06/21    remakeSolidShellModel:追加（「!EQUATION」を追加する）
#      09/26    checkElementListOrder:node順の修正する位置を元に戻す
#               checkNoElementGroup:追加（EGRPが空白の場合、EGRPとしてeltypeを設定）
#      11/23    check641BeamElementGroup:NGRP名のdummyをdummyBeamに変更
#      11/27    remakeSolidShellModel:beamを追加
#      12/11    getNumberOfDummyNodesを追加（NGRPのdummyNGRPを作成）
#      12/18    checkDummyNodeGroup:dummyのNGRP中の最初と最後のNodeNoから要素を取得
#               （節点を共有する場合がある為、2点で要素を決定する）
#      12/23    checkDummyNodeGroup, getDummyNodes:高速化の為修正。
#               dummyNodes作成時にnodeToDummyListでnodeとdummy間のひも付け実施
#      12/25    checkDummyNodeGroup:バグ修正（dummyが必要ない要素の場合、ｴﾗｰ発生）
#   17/07/06    check641BeamElementGroup:beamを全て641に設定。
#      09/28    convNum2str:追加。座標などの実数を文字に変換する時、小数点有無を
#               チェックし、修正する。
#      10/28    getValName:大文字同士で検索する様に修正
#   18/03/30    getFaceNoOfElement:高速化の為、修正。
#      03/31    isThereSolidElement:dictを使って判断するように修正
#      06/28    abaqus2fistr:inpファイル読み込み時、keyword行は、大文字に変換する
#      07/16    changeKeywordUpper:追加。（小文字keywordを大文字に変換を追加）
#      11/29    checkElementGroup:PrePoMaxが吐き出したinpファイル用に修正
#               getDataList:同上（surfaceの定義で要素group名で定義できるように修正）
#      12/14    checkElementGroup:1ヶの要素group中に複数の要素typeが存在する場合に対応
#               makeElement:section追加時、EGRP名毎に追記する様修正
#   19/01/04    python3用に書き換え
#      04/06    pythonスクリプト起動時にフルパスでスクリプトを指定する様に修正
#      04/20    vsCodeを使って修正。
#      05/04    writeData:データ作成方法を修正（windowsは、作成に時間が掛かる為）
#      05/05    check641BeamElementGroup:元に戻す。solidがある時のみ611を641に変更する。
#   20/06/23    checkElementGroup:elsetの処理方法を修正。
#               elsetで洩れている要素が未定義になる為、これを修正
#      06/24    headerの内容を修正。（PARTNAME, NUMが残ってしまうので。）
#      09/21    要素タイプCPS3, CPS4, T3D2を追加
#   21/05/17    remakeSolidShellModel:os.systemをsubprocessに変更
#   22/04/12    log表示修正（import logFileCreaterを追加）
#      12/07    abaqus2fistr:fileが存在しない場合、エラー発生の為、エラー出力追加。
#   23/08/15    addIncludeHeaders:「*INCLUDE」行の処理追加
#               getNodeElementData:keywordを大文字に変換して辞書内容と比較する様に修正。
#               辞書は、全て大文字の為。
#   24/07/07    openにencoding="tuf-8"を追加
#      10/06    deleteSp:deleteSp:空白削除後、CONTACT PAIR, INITIAL CONDITIONを再作成。
#   25/04/03    elementAbaToFistrDict:「R3D3」を追加
#      04/06    elementAbaToFistrDict:「B31H」を追加
#      04/14    checkBeamElementGroup:solid、shellと混在しているbeam要素が取得
#               できていなかった為、修正。
#               elementAbaToFistrDict,elementContsDict,changeOrderDict:
#               辞書内から612→112に変更。612要素は存在しない為
#      04/17    作成したmeshの保存file名変更
#      04/22    入力の引数（newName）を追加
#      04/26    convertFaceNo:faceNoの取得をabaqusと同じNoに変更。
#               shellを変換した時、faceNoが取得できておらず、エラー発生の為、見直した。
#

import os
import sys
import subprocess
import convertMesh as cm
import pyFistr

#ls = os.linesep           #行末コードを定義
ls = "\n"

abaName = ""
nodeList = []           #NODE
elementList = []        #ELEMENT
surfaceList = []        #ELEMENT（表面要素）
surGroupList = []       #SGROUP
surfaceGroupList = []   #SGROUP（*SURFACEで定義）
elsetList = []
ndsetList = []          #NGROUP
nElementType = 0        #elementTypeの種類
nElsetSur = []
nElsetEl = []
elNodeList = []         #element(表面、内部要素)ｘnodeNoのリスト（検索を早めるため）
nodeElsetList = []      #nodeNo?elementNoのリスト（検索を早めるため）
nodeToDummyList = []    #nodeNoに対するdummyNodeNoのリスト
dummyNodeList = []      #dummyNode取得の為のリスト


#  abaqusのkeyword取得用Dict
keywordItemContsDict = {
        #keyword     item    wordLength
        "*NODE":    ["NODE",    4],         #節点定義
        "*ELEMENT": ["ELEMENT", 0],         #要素定義
        "*ELSET":   ["ELSET",   1],         #要素group定義
        "*NSET":    ["NSET",    1],         #節点group定義
        "*SURFACE": ["SURFACE", 0]          #面group定義
        }

#  abaqus要素内容
elementAbaToFistrDict = {
        #abaType    item   fistrType length 次数 種別
        "C3D4":   ["ELEMENT", "341",  5,     1, "tet"],
        "C3D10":  ["ELEMENT", "342",  11,    2, "tet"],
        "C3D10MH":["ELEMENT", "342",  11,    2, "tet"],
        "C3D8":   ["ELEMENT", "361",  9,     1, "hex"],
        "C3D8R":  ["ELEMENT", "361",  9,     1, "hex"],
        "C3D8RH": ["ELEMENT", "361",  9,     1, "hex"],
        "C3D20":  ["ELEMENT", "362",  21,    2, "hex"],
        "C3D20R": ["ELEMENT", "362",  21,    2, "hex"],
        "C3D20RH":["ELEMENT", "362",  21,    2, "hex"],
        "C3D6":   ["ELEMENT", "351",  7,     1, "prism"],
        "C3D15":  ["ELEMENT", "352",  16,    2, "prism"],
        "STRI35": ["SURFACE", "S341", 4,     1, "tri"],
        "S3":     ["SURFACE", "S341", 4,     1, "tri"],
        "CPS3":   ["SURFACE", "S341", 4,     1, "tri"],
        "R3D3":   ["SURFACE", "S341", 4,     1, "tri"],
        "S6":     ["SURFACE", "S342", 7,     2, "tri"],
        "CPE6":   ["SURFACE", "S342", 7,     2, "tri"],
        "S4R5":   ["SURFACE", "S361", 5,     1, "quad"],
        "S4":     ["SURFACE", "S361", 5,     1, "quad"],
        "CPS4":   ["SURFACE", "S361", 5,     1, "quad"],
        "CAX4P":  ["SURFACE", "S361", 5,     1, "quad"],
        "S8":     ["SURFACE", "S362", 9,     2, "quad"],
        "S8R":    ["SURFACE", "S362", 9,     2, "quad"],
        "S8R5":   ["SURFACE", "S362", 9,     2, "quad"],
        "B31":    ["ELEMENT", "611",  3,     1, "beam"],
        "B32":    ["ELEMENT", "112",  4,     2, "beam"],
        "B33H":   ["ELEMENT", "112",  4,     2, "beam"],
        "B31H":   ["ELEMENT", "611",  3,     1, "beam"],
        "T3D2":   ["ELEMENT", "611",  3,     1, "beam"]
        }

#  要素内容を定義
elementContsDict = {
        #name  type    次数　種別　　節点数  
        "341":["solid",  1, "tet",   4],
        "342":["solid",  2, "tet",   10],
        "361":["solid",  1, "hex",   8],
        "362":["solid",  2, "hex",   20],
        "351":["solid",  1, "prism", 6],
        "352":["solid",  2, "prism", 15],
        "731":["shell",  1, "tri",   3],
        "732":["shell",  2, "tri",   6],
        "761":["shell3", 1, "tri",   6],        #3自由度shell
        "741":["shell",  1, "quad",  4],
        "781":["shell3", 1, "quad",  8],        #3自由度shell
        "611":["beam",   1, "beam",  2],
        "112":["beam",   2, "beam",  3],
        "641":["beam3",  1, "beam",  4]         #3自由度beam
        }


#  printMsg
#    print help
def printMsg():
    msg = """
<abaqus2fistr.py>
abaqusフォーマットのmeshをFrontISTR用meshに変換する。

<使い方>
abaqus2fistr.py <inpファイル> <fistrファイル>

例：
abaqus2fistr.py name.inp test
abaqus2fistr.py name test.msh
"""
    print (msg)

#
#  getNodeElementData
#  ------------------
#    必要なdataを取得
def getNodeElementData(lines):
    item = ""
    header = ""
    data = []
    n = 0
    while n < len(lines):
        line = lines[n]
        if line[0] == "*":
            if len(data) > 0:
                #dataをlistに保存
                saveData(item, header, data)

            #abaKeywordを取得
            words = deleteSp(line).split(",")
            keyword = words[0].upper()
            #keywordから処理開始
            try:
                #登録内容を取得
                keyCont = keywordItemContsDict[keyword]
                item, length = keyCont[0], keyCont[1]
            except:
                #list登録以外
                item = ""
            if item != "":
                #要素？
                if item == "ELEMENT":
                    #要素の取得
                    typeName = getValName(deleteSp(line), "TYPE")
                    #要素内容を取得
                    elCont = elementAbaToFistrDict[typeName]
                    item, elType, length = elCont[0], elCont[1], elCont[2]
                    header = setValName(line, "TYPE="+elType)
                    data = []
                #面？
                elif item == "SURFACE":
                    #surfaceGroupの取得
                    typeName = getValName(line, "TYPE")
                    if typeName.upper() == "NODE":
                        header = "*NSET\n"
                        item, length  = "NSET", 1
                        data = []
                    else:
                        header = line
                        item, length = "SURSET", 2
                        data = []
                #その他
                else:
                    #NODE、ELSET、NSETを取得
                    header = line
                    data = []
            n += 1
        else:
            if item != "":
                #itemのdataを取得
                [data, n] = getDataList(lines, n, item, length)
            else:
                n += 1
    if len(data) > 0:
        #取得したdataをlistに保存
        saveData(item, header, data)

#  getDataList
#    data部を取得
def getDataList(lines, n, item, length):
    data = []
    if item == "NODE":
        while n < len(lines) and lines[n][0] != "*":
            line = lines[n]
            words = line.split(",")
            a = [1, 0.0, 0.0, 0.0]
            a[0] = int(words[0])
            for i in range(1, len(words)):
                a[i] = float(words[i])
            data.append(a)
            n += 1
    elif item == "ELEMENT" or item == "SURFACE":
        while n < len(lines) and lines[n][0] != "*":
            line = lines[n]
            words = deleteSp(line).split(",")
            for word in words:
                if len(word) > 0 and word != "\n":
                    data.append(int(word))
            n += 1
        ans = []
        line = []
        p = 0
        for a in data:
            if p < length:
                line.append(a)
            else:
                ans.append(line)
                p = 0
                line = []
                line.append(a)
            p += 1
        ans.append(line)
        data = ans
    elif item == "ELSET" or item == "NSET":
        while n < len(lines) and lines[n][0] != "*":
            line = lines[n]
            words = deleteSp(line).split(",")
            for word in words:
                if len(word) > 0 and word != "\n":
                    try:
                        data.append(int(word))
                    except:
                        pass
            n += 1
    elif item == "SURSET":
        while n < len(lines) and lines[n][0] != "*":
            line = lines[n]
            words = deleteSp(line).split(",")
            if len(words) == 2:
                try:
                    data.append([int(words[0]), words[1]])
                except:
                    #要素グループ名で定義されている場合
                    elNos = []
                    elGrpName = words[0]
                    #要素grp内から該当するgrrpを検索
                    for elsetCont in elsetList:
                        header = elsetCont[0]
                        grpName = getValName(header, "ELSET")
                        if grpName == elGrpName:
                            #要素grp内の要素Noを取得
                            elNos = elsetCont[1]
                            break
                    #要素grpの要素Noを取得
                    for elNo in elNos:
                        #dataを取得
                        data.append([elNo, words[1]])
            n += 1
    return [data, n]

#  saveData
#    読み込んだdataを取得する
def saveData(item, header, data):
    global nodeList, elementList, surfaceList, elsetList, ndsetList
    global surfaceGroupList
    if item == "NODE":
        block = []
        num = str(len(data))
        #header = "!NODE, NUM=" + num + ls
        header = "!NODE" + ls
        block.append(header)
        block.append(data)
        nodeList.append(block)
        print ("found NODE... num " + num)
    elif item == "ELEMENT":
        block = []
        num = str(len(data))
        header = "!" + header[1:]
        header = deleteSp(header).split()[0]
        words = header.split(",")
        header = ", ".join(words) + ", NUM=" + num + ls
        block.append(header)
        block.append(data)
        elementList.append(block)
        name = getValName(header, "TYPE")
        print ("found ELEMENT... TYPE " + name + " num " + num)
    elif item == "SURFACE":
        block = []
        num = str(len(data))
        header = "!" + header[1:]
        header = deleteSp(header).split()[0]
        words = header.split(",")
        header = ", ".join(words) + ", NUM=" + num + ls
        block.append(header)
        block.append(data)
        surfaceList.append(block)
        name = getValName(header, "TYPE")
        print ("found surfaceELEMENT... TYPE " + name + " num " + num)
    elif item == "ELSET":
        block = []
        num = str(len(data))
        header = "!" + header[1:]
        header = deleteSp(header).split()[0]
        words = header.split(",")
        newWords = []
        for word in words:
            if word == "GENERATE":
                data = generateNodeGroup(data)
                num = str(len(data))
            else:
                newWords.append(word)
        header = ", ".join(newWords) + ", NUM=" + num + ls
        block.append(header)
        block.append(data)
        elsetList.append(block)
        name = getValName(header, "ELSET")
        print ("found ELSET... " + name + " num " + num)
    elif item == "NSET":
        block = []
        num = str(len(data))
        header = "!" + header[1:]
        header = deleteSp(header).split()[0]
        words = header.split(",")
        newWords = []
        for word in words:
            if word == "GENERATE":
                data = generateNodeGroup(data)
                num = str(len(data))
            else:
                newWords.append(word) 
        header = ", ".join(newWords) + ", NUM=" + num + ls
        block.append(header)
        block.append(data)
        ndsetList.append(block)
        name = getValName(header, "NSET")
        print ("found NSET... " + name + " num " + num)
    elif item == "SURSET":
        block = []
        num = str(len(data))
        name = getValName(header, "NAME")
        #header = "!SGROUP, SGRP=" + name + ", NUM=" + num + ls
        header = "!SGROUP, SGRP=" + name + ls
        block.append(header)
        data = convertFaceNo(data)
        block.append(data)
        surfaceGroupList.append(block)
        print ("found surfaceGroup... NAME " + name + " num " + num)

#
#  checkShellElementData
#  ---------------------
#    シェル要素のチェック。シェル要素がある場合、シェル要素を取得
def checkShellElementData():
    global elementList, surfaceList
    #solid要素が定義されているか
    if isThereSolidElement(elementList) == True:
        #solid要素が存在する場合、そのまま戻る
        return

    #シェル要素を取得（solid要素が存在しない場合）
    for n in range(len(surfaceList)):
        surface = surfaceList[n]
        surType = getValName(surface[0], "TYPE")
        if surType == "S341":
            #三角形1次
            surface[0] = setValName(surface[0], "TYPE=731")
            elementList.append(surface)
            surfaceList[n] = []
        elif surType == "S342":
            #三角形2次
            surface[0] = setValName(surface[0], "TYPE=732")
            elementList.append(surface)
            surfaceList[n] = []
        elif surType == "S361":
            #四角形1次
            surface[0] = setValName(surface[0], "TYPE=741")
            elementList.append(surface)
            surfaceList[n] = []

    #surfaceListを整形
    flag = 0
    newSurfaceList = []
    for surface in surfaceList:
        if len(surface) != 0:
            newSurfaceList.append(surface)
            flag = 1
    if flag == 0:
        surfaceList = []
    else:
        surfaceList = newSurfaceList

#  isThereSolidElement
#    solid要素が存在するかどうか確認する
def isThereSolidElement(elementList):
    ans = False
    for element in elementList:
        header = element[0]
        elType = getValName(header, "TYPE")
        elCont = elementContsDict[elType]
        if elCont[0] == "solid":
            ans = True
            break
    return ans

#
#  convertFaceNo
#    faceNoをfistr用に変換
def convertFaceNo(data):
    """ faceNoは、abaqusのfaceNo表記「S2」をそのまま「faceNo=2」として取得"""

    # def getFaceNo(faceNo, nodes):
    #     face = int(faceNo[1:])
    #     nNode = len(nodes)
    #     if nNode == 4 or nNode == 10:
    #         #四面体1次2次要素
    #         faces = [1,2,3,4]
    #         cFace = faces[face-1]
    #     elif nNode == 8 or nNode == 20:
    #         #六面体1次2次要素
    #         #faces = [6,4,1,5,2,3]
    #         faces = [1,2,3,4,5,6]
    #         cFace = faces[face-1]
    #     elif nNode == 6 or nNode == 15:
    #         #五面体1次2次要素
    #         faces = [1,2,3,4,5]
    #         cFace = faces[face-1]
    #     return cFace

    for i in range(len(data)):
        elNo = data[i][0]
        faceNo = data[i][1]
        # flag = 0
        # for [dummy_header, elements] in elementList:
        #     for element in elements:
        #         if element[0] == elNo:
        #             nodes = element[1:]
        #             sNo = getFaceNo(faceNo, nodes)
        #             data[i][1] = sNo
        #             flag = 1
        #             break
        #     if flag == 1:
        #         break
        # if flag == 0:
        #     #sNoが設定できない場合（shellの場合）
        #     #  faceNoをそのまま設定する
        #     data[i][1] = faceNo[1]
        data[i][1] = faceNo[1]
    return data

#
#  generateNodeGroup
#    GENERATEコマンドでnodeGroupを作成する
def generateNodeGroup(data):
    newData = []
    for i in range(0, len(data), 3):
        currData = data[i:i+3]
        if len(currData) == 3:
            ist = currData[0]
            ied = currData[1]
            inc = currData[2]
            ii = ist
            while ii <= ied:
                newData.append(ii)
                ii += inc
    return newData

#
#  checkElsetInElement
#    要素定義中のelsetをチェック
#    同じelsetが定義されていれば、これをまとめる
def checkElsetInElement():
    global elementList

    #  addToNewElementList
    #    同じelsetがある場合、newElementListに追加する
    def addToNewElementList(element, newElementList):
        flag = 0
        currElsetName = getValName(element[0], "ELSET")
        if currElsetName != "":
            for newElement in newElementList:
                elsetName = getValName(newElement[0], "ELSET")
                if currElsetName == elsetName:
                    #同じelsetに追加
                    newElement[1] += element[1]
                    flag = 1
                    break
        if flag == 0:
            #新規に追加
            newElementList += [element]
        return newElementList

    newElementList = []
    for element in elementList:
        newElementList = addToNewElementList(element, newElementList)
    elementList = newElementList

#   --------- node順変更関数 -------------

#  changeNodeOrder341
#    四面体1次メッシュの節点順を変更
def changeNodeOrder341(data):
    return data

#  changeNodeOrder342
#    四面体2次メッシュの節点順を変更
def changeNodeOrder342(data):
    i = 0
    while i < len(data):
        nums = data[i]
        newOder = [nums[0], nums[1], nums[2], nums[3], nums[4], nums[6], nums[7],
                nums[5], nums[8], nums[9], nums[10]]
        data[i] = newOder
        i += 1
    return data

#  changeNodeOrder361
#    六面体1次メッシュの節点順を変更
def changeNodeOrder361(data):
    i = 0
    while i < len(data):
        nums = data[i]
        #newOder = [nums[0], nums[1], nums[5], nums[6], nums[2], nums[4], nums[8],
        #        nums[7], nums[3]]
        newOder = [nums[0], nums[1], nums[2], nums[3], nums[4], nums[5], nums[6],
                nums[7], nums[8]]
        data[i] = newOder
        i += 1
    return data

#  changeNodeOrder362
#    六面体2次メッシュの節点順を変更
def changeNodeOrder362(data):
    i = 0
    while i < len(data):
        nums = data[i]
        #newOder = [nums[0], nums[1], nums[5], nums[6], nums[2], nums[4], nums[8],
        #        nums[7], nums[3], nums[17], nums[13], nums[18], nums[9], nums[20],
        #        nums[15], nums[19], nums[11], nums[12], nums[16], nums[14], nums[10]]
        newOder = [nums[0], nums[1], nums[2], nums[3], nums[4], nums[5], nums[6],
                nums[7], nums[8], nums[9], nums[10], nums[11], nums[12], nums[13],
                nums[14], nums[15], nums[16], nums[17], nums[18], nums[19], nums[20]]
        data[i] = newOder
        i += 1
    return data

#  changeNodeOrder351
#    五面体1次メッシュの節点順を変更
def changeNodeOrder351(data):
    i = 0
    while i < len(data):
        nums = data[i]
        #newOrder = [nums[0], nums[1], nums[3], nums[2], nums[4], nums[6], nums[5]]
        newOrder = [nums[0], nums[1], nums[2], nums[3], nums[4], nums[5], nums[6]]
        data[i] = newOrder
        i += 1
    return data

#  changeNodeOrder352
#    五面体2次メッシュの節点順を変更
def changeNodeOrder352(data):
    i = 0
    while i < len(data):
        nums = data[i]
        #newOrder = [nums[0], nums[1], nums[3], nums[2], nums[4], nums[6], nums[5],
        #        nums[8], nums[7], nums[9], nums[11], nums[10], nums[12], nums[13],
        #        nums[15], nums[14]]
        newOrder = [nums[0], nums[1], nums[2], nums[3], nums[4], nums[5], nums[6],
                nums[8], nums[9], nums[7], nums[11], nums[12], nums[10], nums[13],
                nums[14], nums[15]]
        data[i] = newOrder
        i += 1
    return data

#  changeNodeOrder731
#    シェル三角形1次メッシュの節点順を変更
def changeNodeOrder731(data):
    i = 0
    while i < len(data):
        nums = data[i]
        newOrder = [nums[0], nums[1], nums[2], nums[3]]
        data[i] = newOrder
        i += 1
    return data

#  changeNodeOrder732
#    シェル三角形2次要素の節点順を変更
def changeNodeOrder732(data):
    i = 0
    while i < len(data):
        nums = data[i]
        newOrder = [nums[0], nums[1], nums[2], nums[3], nums[5], nums[6], nums[4]]
        data[i] = newOrder
        i += 1
    return data

#  changeNodeOrder741
#    シェル四角形1次要素の節点順を変更
def changeNodeOrder741(data):
    i = 0
    while i < len(data):
        nums = data[i]
        newOrder = [nums[0], nums[1], nums[2], nums[3], nums[4]]
        data[i] = newOrder
        i += 1
    return data

#  changeNodeOrder761
#    761要素の節点順を変更
def changeNodeOrder761(data):
    return data

#  changeNodeOrder781
#    781要素の節点順を変更
def changeNodeOrder781(data):
    return data

#  noChangeNodeOrder
#    要素順を変更しない
def noChangeNodeOrder(data):
    return data

#
#  checkElementListOrder
#  ---------------------
#    要素の並び順をチェック
def checkElementListOrder():

    changeOrderDict = {
        #type   変換関数名
        "341":changeNodeOrder341,
        "342":changeNodeOrder342,
        "361":changeNodeOrder361,
        "362":changeNodeOrder362,
        "351":changeNodeOrder351,
        "352":changeNodeOrder352,
        "731":changeNodeOrder731,
        "732":changeNodeOrder732,
        "741":changeNodeOrder741,
        "761":changeNodeOrder761,
        "781":changeNodeOrder781,
        "611":noChangeNodeOrder,
        #"612":noChangeNodeOrder,
        "112":noChangeNodeOrder,
        "641":noChangeNodeOrder
        }

    global elementList, nElementType
    #elementの種類の個数を取得
    nElementType = len(elementList)
    #メッシュの並び順をチェック
    i = 0
    while i < len(elementList):
        element = elementList[i]
        header = element[0]
        typeName = getValName(header, "TYPE")
        func = changeOrderDict[typeName]        #node順変更関数取得
        data = func(element[1])                 #node順変更
        elementList[i][1] = data
        i += 1
    #headerの修正
    remakeElementHeaderData()

#
#  remakeElementHeaderData
#    data整形（headerをfistr様に変更）
def remakeElementHeaderData():
    global elementList
    i = 0
    while i < len(elementList):
        header = elementList[i][0]
        typeName = getValName(header, "TYPE")
        num = getValName(header, "NUM")
        name = abaName.split(".")[0]
        egrpName = getValName(header, "ELSET")
        header = "!ELEMENT, TYPE=" + typeName + ", PARTNAME=" + name + ", EGRP=" + egrpName + ", NUM=" + num + ls
        elementList[i][0] = header
        i += 1

#
#  checkSurfaceElementGroup
#  ------------------------
#    surfaceのgroup分けをチェック
def checkSurfaceElementGroup():
    print ("checking data...")
    global surGroupList, elsetList, surfaceList, elementList

    #surfaceのlist（elNodeList）を作成
    createSurfaceNodesList()

    #elsetをEGROUPとSGROUPに分ける
    surGroupList = []       #表面要素（surface）
    newElset = []           #内部要素（element）
    for elset in elsetList:
        surfaceNo = elset[1][0]
        if elNodeList[surfaceNo][0][0] == "S":
            #表面要素
            surGroupList += [elset]
        else:
            #内部要素
            newElset += [elset]
    elsetList = newElset
    #その他の表面要素を取得
    getOtherSurGroupList()
    #nodeを含むelementのlistを作成
    createNodeElementSet()

    print ("creating SGROUP(surface)...", end=" ")
    #各要素にnodeNoを追加
    newSurElset = addNodesSurElset(surGroupList)
    #elementを取得
    [surfaceGroup, surElementSet] = getElementFaceNo(newSurElset)
    #header部を修正
    surGroupList = remakeHeaderSurGroup(surfaceGroup)
    if len(surfaceGroupList) > 0:
        #「*SURFACE」内を追加
        for surfaceGroup in surfaceGroupList:
            if len(surfaceGroup[1]) > 0:
                surGroupList += [surfaceGroup]
                name = getValName(surfaceGroup[0], "SGRP")
                print (name, end=" ")
    if len(surElementSet) > 0:
        #header部を修正（shellElement）
        shellElementList = remakeHeaderShellElement(surElementSet, newSurElset)
        elementList += shellElementList
        if len(elsetList) > 0:
            shellGroups = getShellElementGroupList(shellElementList, newSurElset)
            elsetList += shellGroups
    print (ls)

#
#  getSolidElementGroupList
#    soild要素のelsetを取得
def getSolidElementGroupList(elementList):
    elset = []
    for element in elementList:
        line = element[0]
        egrp = getValName(line, "ELSET")
        header = "!ELSET, ELSET=" + egrp + ls
        data = []
        for elLine in element[1]:
            elNo = elLine[0]
            data.append(elNo)
        elset.append([header, data])
    return elset

#
#  getShellElementGroupList
#    shellのgroupを取得
def getShellElementGroupList(shellElementList, surElset):
    newElsetList = []
    for elset in surElset:
        setElNo = elset[1][0][0]
        for element in shellElementList:
            for elNode in element[1]:
                if setElNo == elNode[0]:
                    newElsetList.append(elset)
                    break
    ans = []
    if len(newElsetList) > 0:
        for elset in newElsetList:
            header = elset[0]
            data = []
            for element in elset[1]:
                data.append(element[0])
            ans.append([header, data])
    return ans

#
#  getOtherSurGroupList
#    その他の表面要素を取得
def getOtherSurGroupList():
    global surGroupList
    #配列を準備
    dummy = []
    for surface in surfaceList:
        for surNodes in surface[1]:
            dummy.append(surNodes[0])
    #surfaceを定義していない場合は、戻る
    if len(dummy) == 0:
        return

    maxSurNo = max(dummy)
    surElement = [ -1 for i in range(maxSurNo+1) ]
    # surElement = []
    # i = 0
    # while i < maxSurNo+1:
    #     surElement.append(-1)
    #     i += 1
    #elNoを取得
    for surface in surfaceList:
        for surNodes in surface[1]:
            elNo = surNodes[0]
            surElement[elNo] = 0
    #設定済の表面要素Noを取得
    for surGroup in surGroupList:
        for surElements in surGroup[1]:
            elNo = surElements
            surElement[elNo] = 1
    #その他の表面要素を取得
    otherData = []
    i = 0
    for surElNo in surElement:
        if surElNo == 0:
            otherData.append(i)
        i += 1
    #otherSのelsetを作成
    addList = []
    addList += ["!ELSET, ELSET=otherS, NUM=" + str(len(otherData)) + ls]
    addList += [otherData]
    surGroupList = [addList] + surGroupList

#
#  createNodeElementSet
#    nodeNoをindexにしてそのnodeを含むelementのlist作成
def createNodeElementSet():
    global nodeElsetList
    tmp = []
    for nodes in nodeList:
        for nodeLine in nodes[1]:
            tmp.append(nodeLine[0])
    if len(tmp) == 0:
        return

    maxNo = max(tmp)
    #配列の作成
    nodeElsetList = [[[],[]] for i in range(maxNo+1)]
    # nodeElsetList = []
    # i = 0
    # while i < maxNo+1:
    #     nodeElsetList += [[[],[]]]
    #     i += 1
    #nodeLocを取得
    for nodes in nodeList:
        for nodeLine in nodes[1]:
            node = nodeLine[0]
            nodeElsetList[node][0] = nodeLine[1:]
    #elementNoを取得
    for elements in elementList:
        for nodesLine in elements[1]:
            elNo = nodesLine[0]
            nodes = nodesLine[1:]
            for node in nodes:
                nodeElsetList[node][1] += [elNo]

#
#  createSurfaceNodesList
#    表面要素の要素NoとnodeNoのlist作成
#    [[<"S" or "E">, elType], [node1, node2,...]]
def createSurfaceNodesList():
    global elNodeList
    #最大要素Noを取得
    tmp = []
    for surface in surfaceList:
        for surfaceNode in surface[1]:
            tmp.append(surfaceNode[0])
    for element in elementList:
        for elementNode in element[1]:
            tmp.append(elementNode[0])

    if len(tmp) == 0:
        return

    nMaxSurNo = max(tmp)
    #list作成
    elNodeList = [ [] for i in range(nMaxSurNo+1) ]
    # elNodeList = []
    # i = 0
    # while i < nMaxSurNo+1:
    #     elNodeList.append([])
    #     i += 1
    #表面要素、要素typeを保存
    for surface in surfaceList:
        header = surface[0]
        elType = getValName(header, "TYPE")
        for surfaceNode in surface[1]:
            elNo = surfaceNode[0]
            elNodeList[elNo] = [["S", elType]] + surfaceNode[1:]
    #内部要素、要素typeを保存
    for element in elementList:
        header = element[0]
        elType = getValName(header, "TYPE")
        for elementNode in element[1]:
            elNo = elementNode[0]
            elNodeList[elNo] = [["E", elType]] + elementNode[1:]

#  separateElset
#    elsetをEGROUPとSGROUPに分ける
def separateElset():
    global surfaceList, surGroupList, elsetList
    surGroupList = []
    newElset = []
    i = 0
    while i < len(elsetList):
        elset = elsetList[i]
        surfaceNo = elset[1][0]
        flag = 0
        for surElement in surfaceList:
            for surElNo in surElement[1]:
                if surfaceNo == surElNo[0]:
                    flag = 1
                    break
            if flag == 1:
                break
        if flag == 1:
            surGroupList += [elset]
        else:
            newElset += [elset]
        i += 1
    elsetList = newElset
    return

#  addNodesSurElset
#    各表面要素にnode番号を追加
def addNodesSurElset(elsets):
    newSurElset = []
    for elset in elsets:
        data = []
        for elNo in elset[1]:
            nodes = elNodeList[elNo][1:]
            newNodes = [elNo] + nodes
            data.append(newNodes)
        data = [elset[0]] + [data]
        newSurElset.append(data)
    return newSurElset

#  findSurfaceNodes
def findSurfaceNodes(elNo):
    ans = []
    flag = 0
    for surface in surfaceList:
        for nodeNo in surface[1]:
            if nodeNo[0] == elNo:
                flag = 1
                ans = nodeNo[1:]
                break
        if flag == 1:
            break
    return ans

#  getElementFaceNo
#    elementとfaceNo取得、shell要素を取得
def getElementFaceNo(elsets):
    #elementNo取得
    elementNoSet = []       #solidElements
    surElementSet = []      #shellElements
    surData = []
    for elset in elsets:
        header = elset[0]
        name = getValName(header, "ELSET")
        data = []
        #surData = []
        #ellNo = 0
        for line in elset[1]:
            setNodes = line[1:]
            [elementNo, faceNo] = findElementNo(setNodes)
            if elementNo != "":
                data += [[elementNo] + [faceNo]]
            else:
                surData += [line]
        if len(data) > 0:
            print (name, end=" ")
            data.sort()
            elementNoSet += [[header] + [data]]
    if len(surData) > 0:
        surData.sort()
        surElementSet = makeShellElementSet(surData)
    else:
        surElementSet = []
    return [elementNoSet, surElementSet]

#
#  makeShellElementSet
#    shell要素のsetを取得
def makeShellElementSet(surData):
    elTypes = [["", []]]
    for surface in surData:
        elNo = surface[0]
        elType = elNodeList[elNo][0][1]
        flag = 0
        for i in range(len(elTypes)):
            if elTypes[i][0] == elType:
                elTypes[i][1].append(surface)
                flag = 1
                break
        if flag == 0:
            elTypes.append([elType, [surface]])
    return elTypes[1:]

#  findElementNo
#    elementNoを取得し、そのfaceNoを取得して返す
def findElementNo(nodes):
    flag = 0
    i = 0
    #最初のnodeNoを含むelementListを取得
    elements = nodeElsetList[nodes[0]][1]
    while i < len(elements):
        if i < len(elements):
            elementNo = elements[i]
            line = elNodeList[elementNo]
            if isInNodes(line[1:], nodes) == True:
                elType = elNodeList[elementNo][0][1]
                faceNo = getFaceNoOfElement(line[1:], nodes, elType)
                flag = 1
                break
        i += 1
    if flag == 1:
        ans = [elementNo, faceNo]
    else:
        ans = ["", ""]
    return ans

#  isInNodes
def isInNodes(line, nodes):
    flag = 0
    ans = False
    for node in nodes:
        if node in line:
            flag += 1
        else:
            break
    if flag == len(nodes):
        ans = True
    return ans

#  remakeHeaderSurGroup
#    headerを書き換え
def remakeHeaderSurGroup(elsets):
    i = 0
    while i < len(elsets):
        elset = elsets[i]
        header = elset[0]
        groupName = getValName(header, "ELSET")
        header = "!SGROUP, SGRP=" + groupName + ls
        # num = getValName(header, "NUM")
        # partName = abaName.split(".")[0]
        # header = "!SGROUP, PARTNAME=" + partName + ", SGRP=" + groupName + ", NUM=" + num + ls
        elsets[i][0] = header
        i += 1
    return elsets

#  remakeHeaderShellElement
#    headerを書き換え
def remakeHeaderShellElement(surElementSet, newElset):
    global elNodeList, nodeList, dummyNodeList, nodeToDummyList
    #配列を準備
    maxNodeNo = len(nodeElsetList)
    dummyNodeList = [ "" for i in range(maxNodeNo+1) ]
    nodeToDummyList += [ "" for i in range(maxNodeNo+1) ]
    # dummyNodeList = []
    # for node in range(maxNodeNo+1):
    #     dummyNodeList.append("")
    #     nodeToDummyList.append("")
    currNo = maxNodeNo
    newNodes = []
    for i in range(len(surElementSet)):
        surType = surElementSet[i][0]
        if surType == "S341":
            elType = "761"
            elset = "S3"
        elif surType == "S361":
            elType = "781"
            elset = "S4"
        #surElementListのデータを修正
        header = "!ELEMENT, TYPE=" + elType + ", ELSET=" + elset + ls
        surElementSet[i][0] = header
        #元々のデータ（elNodeList）も修正
        for j in range(len(surElementSet[i][1])):
            elNo = surElementSet[i][1][j][0]
            [addNodes, currNo] = createNewNodes(surElementSet[i][1][j], currNo)
            elNodeList[elNo][0] = ["E", elType]
            elNodeList[elNo] += addNodes
            surElementSet[i][1][j] += addNodes
    newNodes = getDummyNodes()
    header = "!NODE, NGRP=dummy" + ls
    nodeList.append([header, newNodes])
    return surElementSet

#  getDummyNodes
#
def getDummyNodes():
    global nodeToDummyList
    nodeData = []
    for i in range(len(dummyNodeList)):
        nodeNo = dummyNodeList[i]
        if nodeNo != "":
            nodeLoc = nodeElsetList[i][0]
            nodeData.append([nodeNo] + nodeLoc)
            nodeToDummyList[i] = nodeNo
    nodeData.sort()
    return nodeData

#  createNewNodes
#    同じ座標のnodeを新しく作成する
def createNewNodes(nodes, currNo):
    global nodeElsetList, dummyNodeList
    #newNo = len(nodeElsetList)
    nums = []
    for node in nodes[1:]:
        if dummyNodeList[node] == "":
            nodeLoc = nodeElsetList[node][0]
            nodeElsetList.append([nodeLoc,[nodes[0]]])
            nums.append(currNo)
            dummyNodeList[node] = currNo
            currNo += 1
        else:
            dummy = dummyNodeList[node]
            nums.append(dummy)
    return [nums, currNo]

#  getFaceNoOfElement
#    elementのfaceNo取得
def getFaceNoOfElement(elNodes, faceNodes, elType):
    faceNo = 0
    faceNo = cm.fistr_elFace(faceNodes, elNodes)
    return faceNo

#
#  checkBeamElementGroup
#  ---------------------
#    beam要素（611）の取得
def checkBeamElementGroup():
    global elementList
    #全てのbeam要素を取得
    beamList = []
    for element in elementList:
        header = element[0]
        elType = getValName(header, "TYPE")
        elCont = elementContsDict[elType]
        if elCont[0] == "beam" or elCont[0] == "beam3":
            beamList.append(element)
    #表面要素に含まれるbeam要素かどうかチェック
    for beam in beamList:
        for i in range(len(beam[1])):
            beam[1][i] += ["N"]
    newBeamList = []
    #表面要素を取得
    surfaceElement = []
    for surface in surfaceList:
        if len(surface) >= 0:
            surfaceElement += surfaceList
    for element in elementList:
        header = element[0]
        elType = getValName(header, "TYPE")
        elCont = elementContsDict[elType]
        if elCont[0] == "shell" or elCont[0] == "shell3":
            surfaceElement.append(element)
    #表面要素に含まれないbeam要素を取得
    if len(surfaceElement) == 0:
        #表面要素が存在しない場合
        pass
    else:
        #表面要素に含まれないbeam要素を取得
        for surfaces in surfaceElement:
            for surface in surfaces[1]:
                surElNo = surface[0]
                surNodes = surface[1:]
                for i in range(len(beamList)):
                    beamData = beamList[i][1]
                    for ii in range(len(beamData)):
                        beam = beamData[ii]
                        beamNodes = beam[1:]
                        if beamNodes[-1] == "N":
                            remNodes = list(set(beamNodes[:-1]) - set(surNodes))
                            if len(remNodes) == 0:
                                beamData[ii][-1] = "Y"
    #beam要素を取得
    beamData = []
    for header, data in beamList:
        for beam in data:
            if beam[-1] == "N":
                beamData.append(beam[:-1])
    if len(beamData) > 0:
        header = "!ELEMENT, TYPE=611, EGRP=E611\n"
        newBeamList.append([header, beamData])
    #elementListからbeam要素を削除
    newElementList = []
    for element in elementList:
        header = element[0]
        elType = getValName(header, "TYPE")
        elCont = elementContsDict[elType]
        if not (elCont[0] == "beam" or elCont[0] == "beam3"):
            newElementList.append(element)
    elementList = newElementList
    if len(newBeamList) != 0:
        #beam要素を追加
        elementList = newBeamList + elementList
    return

#
#  check641BeamElementGroup
#  ------------------------
#    641要素（solidとbeam混在型）を取得
def check641BeamElementGroup():
    global elementList, nodeList, elNodeList, dummyNodeList, nodeToDummyList
    #solid有無をチェック
    flag = 0
    for element in elementList:
        elType = getValName(element[0], "TYPE")
        if elementContsDict[elType][0] == "solid":
            flag = 1
            break
    if flag == 0:
        #solidが無い場合は直ぐに戻る
        return

    #beam要素611を641に修正（solidが存在する場合）
    flag = 0
    i = 0
    for element in elementList:
        elType = getValName(element[0], "TYPE")
        if elType == "611":
            flag = 1
            header = element[0]
            header = setValName(header, "TYPE=641")
            elementList[i][0] = header
        i += 1
    #beamが無い場合は、すぐに戻る
    if flag == 0:
        return

    #dummyNodeを追加
    #  配列を準備
    maxNodeNo = len(nodeElsetList)
    dummyNodeList = [ "" for i in range(maxNodeNo+1) ]
    nodeToDummyList += [ "" for i in range(maxNodeNo+1)]
    currNo = maxNodeNo
    newNodes = []
    for i in range(len(elementList)):
        header = elementList[i][0]
        elType = getValName(header, "TYPE")
        if elType == "641":
            for j in range(len(elementList[i][1])):
                elNo = elementList[i][1][j][0]
                [addNodes, currNo] = createNewNodes(elementList[i][1][j], currNo)
                elNodeList[elNo] += addNodes
                elementList[i][1][j] += addNodes
    newNodes = getDummyNodes()
    header = "!NODE, NGRP=dummyBeam" + ls
    nodeList.append([header, newNodes])
    return

#
#  checkElementGroup
#  -----------------
#    elementのgroup分けをチェック
def checkElementGroup():
    global elsetList, elementList, nElementType
    if len(elsetList) == 0:
        return

    print ("creating EGROUP(element)...", end=" ")
    #elementListを整形
    for i in range(len(elementList)):
        header = elementList[i][0]
        elType = getValName(header, "TYPE")
        egrpName = getValName(header, "EGRP")
        if egrpName == "":
            egrpName = elType
            header = setValName(header, "EGRP="+egrpName)
            elementList[i][0]
    #elSetを作成
    for i in range(len(elsetList)):
        elset = elsetList[i]
        header = elset[0]
        egrpName = getValName(header, "ELSET")
        elNos = elset[1]
        header = "!EGROUP, EGRP=" + egrpName + ls
        elsetList[i][0] = header
    print

#  findElementType
#    要素Noからeltypeを検索する
def findElementType(elData):
    elNo = elData[0][0]
    element = elNodeList[elNo]
    elType = element[0][1]
    return elType

#  createOtherElementForElType
#    elType毎のotherElemntを取得
def createOtherElementForElType(newElementList):
    #配列を準備
    elset = [ 0 for i in range(len(elNodeList)) ]
    #定義済みの場合、配列に「1」をセット
    for element in newElementList:
        for elNode in element[1]:
            elNo = elNode[0]
            elset[elNo] = 1
    #未定義のelementNoを取得
    otherData = []
    for i in range(len(elset)):
        if elset[i] == 0:
            otherData.append(i)
    #otherElementを取得
    otherElement = []
    for i in range(len(elementList)):
        header = elementList[i][0]
        elType = getValName(header, "TYPE")
        otherElement.append([elType, []])
    nType = len(otherElement)
    for elNo in otherData:
        element = elNodeList[elNo]
        if len(element) > 0:
            for i in range(nType):
                if otherElement[i][0] == element[0][1]:
                    otherElement[i][1].append([elNo] + element[1:])
                    break
    #headerを整形
    otherElementList = []
    for element in otherElement:
        if len(element[1]) > 0:
            otherElementList.append(element)
    for i in range(len(otherElementList)):
        elType = otherElementList[i][0]
        egrp = "other" + elType
        header = "!ELEMENT, TYPE=" + elType + ", EGRP=" + egrp + ls        
        otherElementList[i][0] = header
        print (egrp, end=" ")
    return otherElementList

#  createNewElGroup
#    elsetを作成する
def createNewElGroup(elset):
    line = elset[0]
    name = getValName(line, "ELSET")
    partName = abaName.split(".")[0]
    num = getValName(line, "NUM")
    header = "!EGROUP, PARTNAME=" + partName + ", EGRP=" + name + ", NUM=" + num + ls
    elset[0] = header
    return elset

#  createNewElementData
#    elsetのelementDataを作成する
def createNewElementData(elset):
    line = elset[0]
    name = getValName(line, "ELSET")
    if name == "":
        name = getValName(line, "EGRP")
    line = elementList[0][0]
    header = setValName(line, "EGRP="+name)
    print (name, end=" ")
    ans = []
    newElset = []
    for elNo in elset[1]:
        nodes = elNodeList[elNo][1:]
        newElset += [[elNo] + nodes]
    num = str(len(newElset))
    header = setValName(header, "NUM="+num)
    ans.append(header)
    ans.append(newElset)
    return ans

#  createOtherElementData
#    otherElementを追加する
def createOtherElementData(newElementList):
    nEle = 0
    for element in newElementList:
        nEle += len(element[1])
    if nEle == len(elementList[0][1]):
        ans = ["", []]
        return ans

    #既に設定してあるELSETを取得
    #  配列を準備
    elset = [ 0 for i in range(len(elNodeList)) ]
    #  定義済の場合、「1」をセット
    for element in newElementList:
        for elNode in element[1]:
            elNo = elNode[0]
            elset[elNo] = 1

    #otherを取得
    print ("other", end=" ")
    otherData = []
    for elLine in elementList[0][1]:
        elNo = elLine[0]
        if elset[elNo] == 0:
            otherData.append(elLine)
    header = elementList[0][0]
    header= setValName(header, "EGRP=other")
    num = str(len(otherData))
    header = setValName(header, "NUM="+num)
    ans = [header]
    ans.append(otherData)
    return ans

#
#  checkNodeGroup
#  --------------
#    nodeGroupをチェック
def checkNodeGroup():
    global ndsetList
    print ("creating NGROUP(node)...", end=" ")
    newNdset = []
    for ndSet in ndsetList:
        data = []
        header = ndSet[0]
        name = getValName(header, "NSET")
        print (name, end=" ")
        newHeader = "!NGROUP, NGRP=" + name
        if header.find("GENERATE") >= 0:
            newHeader += ", GENERATE"
        newHeader += ls
        data.append(newHeader)
        data.append(ndSet[1])
        data[1].sort()
        newNdset.append(data)
    ndsetList = newNdset
    print()

#
#  checkDummyNodeGroup
#  -------------------
#    NGRPに対応したdummyのnodeGroupを作成する。
def checkDummyNodeGroup():
    global ndsetList
    print ("creating dummyNGROUP(node)...", end=" ")
    #dummyNodeを取得
    newNdset = []
    for ndset in ndsetList:
        header = ndset[0]
        data = ndset[1]
        newData = []
        for nodeNo in data:
            if nodeNo < len(nodeToDummyList):
                dummyNode = nodeToDummyList[nodeNo]
                if dummyNode != "":
                    newData.append(dummyNode)
                #node数が同じ場合、取得
                if len(newData) == len(data):
                    ngrpName = "dummy_" + getValName(header, "NGRP")
                    print (ngrpName, end=" ")
                    header = setValName(header, "NGRP="+ngrpName)
                    newNdset.append([header, newData])
    print(" done")
    ndsetList += newNdset

#
#  checkNoElementGroup
#  -----------------
#    メッシュに要素groupを設定しなかった場合の対処
def checkNoElementGroup():
    global elementList
    for element in elementList:
        header = element[0]
        name = getValName(header, "EGRP")
        if name == "":
            #EGRPが空白の場合は、EGRPとしてelTypeを設定する
            elType = getValName(header, "TYPE")
            newHeader = "!ELEMENT, TYPE=" + elType + "\n"            
            newHeader = setValName(newHeader, "EGRP=E"+elType)
            element[0] = newHeader

#
#  witeData
#  --------
#    dataの書き込み
def writeData(fileName):
    print ("creating FrontISTR mesh file data...")
    cont = makeHeader()
    cont += makeNode()
    cont += makeElement()
    cont += makeElementGroup()
    cont += makeSurfaceGroup()
    cont += makeNodeGroup()
    cont += "!END"
    print ("writing...", end=" ")
    f=open(fileName, "w"); f.write(cont); f.close()
    print ("done")

#  makeHeader
#    header
def makeHeader():
    cont = "!HEADER" + ls
    cont += "  generated by abaqus2fistr.py" + ls
    return cont

#  convLines2Cont
#    linesを「\n」で接続し、最後に「\n」を追加する
def convLines2Cont(words):
    if len(words) > 0:
        cont = ls.join(words) + ls
    else:
        cont = ""
    return cont

#  makeNode
#    node
def makeNode():
    cont = ""
    for node in nodeList:
        cont += node[0]
        lines = []
        for nums in node[1]:
            strs = []
            for num in nums:
                word = convNum2str(num)     #指数表記の「.」をチェックして変換
                strs.append(word)
            lines += [", ".join(strs)]
        cont += convLines2Cont(lines)
    return cont        

#  makeElement
#    element
def makeElement():
    cont = ""
    names = []
    for element in elementList:
        header = element[0]
        name = getValName(header, "EGRP")
        elType = getValName(header, "TYPE")
        cont += header
        lines = []
        for nums in element[1]:
            strs = list(map(str, nums))     #numsをstrListに変換
            lines += [", ".join(strs)]
        cont += convLines2Cont(lines)
    return cont

#  makeElementGroup
#    elementGroup
def makeElementGroup():
    cont = ""
    if nElementType == 1 and len(elsetList) == 0:
        return cont

    for elGroup in elsetList:
        cont += elGroup[0]
        cont += makeStringFromData(elGroup[1], 8)
    return cont

#  makeSurfaceGroup
#    表面要素group
def makeSurfaceGroup():
    cont = ""
    for surGroup in surGroupList:
        cont += surGroup[0]
        lines = []
        for surSet in surGroup[1]:
            strs = list(map(str, surSet))       #surSetをstrListに変換
            lines += [", ".join(strs)]
        cont += convLines2Cont(lines)
    return cont

#  makeNodeGroup
#    nodeGroup
def makeNodeGroup():
    cont = ""
    if len(ndsetList) == 0:
        return cont

    for ndset in ndsetList:
        cont += ndset[0]
        cont += makeStringFromData(ndset[1], 8)
    return cont

#  makeStringFromData
#    dataSetをstrに変換
def makeStringFromData(data, w):
    cont = ""
    p = 0
    lines = []
    while p+w < len(data):
        nums = data[p:p+w]
        strs = list(map(str, nums))     #numsをstrListに変換
        lines += [", ".join(strs)]
        p += w
    nums = data[p:]
    strs = list(map(str, nums))         #numsをstrListに変換
    lines += [", ".join(strs)]
    cont += convLines2Cont(lines)
    return cont

#  getValName
#    valName(TYPEなど）を取得
def getValName(line, valName):
    valName = valName.upper()
    words = deleteSp(line).split(",")
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        word = deleteSp(word)
        if word.upper().find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#  deleteSp
#    空白、行末コードを削除
def deleteSp(line):
    """ 空白を削除。ただし、「!CONTACT PAIR」「!INITIAL CONDITION」は、
    空白を削除せずそのまま。"""
    newLine = ""
    for chara in line:
        if chara != " " and chara != "\n" and chara != "\r":
            newLine += chara
    #「!CONTACT PAIR」「!INITIAL CONDITION」をチェック修正
    if newLine[:len("!CONTACTPAIR,")] == "!CONTACTPAIR,":
        newLine = newLine.replace("!CONTACTPAIR,", "!CONTACT PAIR,")
    elif newLine[:len("!INITIALCONDITION,")] == "!INITIALCONDITION,":
        newLine = newLine.replace("!INITIALCONDITION,", "!INITIAL CONDITION,")
    return newLine

#  setValName
#    変数の値をセット
#    行末コードにより処理を変更（importして使う事がある為）
#    　abaqus2fistrの中のみなら問題なし
def setValName(line, val):
    valNames = val.split("=")
    words = deleteSp(line).split(",")
    i = 0
    flag = 0
    for word in words:
        word = deleteSp(word)
        #name = (word.split()[0]).split("=")[0].upper()
        name = word.split("=")[0].upper()
        if name == valNames[0]:
            #words[i] = " " + val
            words[i] = val
            flag = 1
            break
        i += 1
    if flag == 0:
        #words.append(" "+val)
        words.append(val)
    newLine = ", ".join(words) + ls
    return newLine

#  deleteLsOfLine
#    行末コードを削除する
def deleteLsOfLine(line):
    if line[-2:] == "\r\n":
        newLine = line[:-2]
    else:
        newLine = line[:-1]
    return newLine

#
#  convNum2str
#    数値を文字列に変換する
#    指数表記で小数点が無い場合は、少数点を追加する
def convNum2str(num):
    word = str(num)
    if word.find(".") >= 0:
        return word
    n = word.find("e")
    if n >= 0:
        word = word[:n] + ".0" + word[n:]
    return word

#
#  remakeSolidShellModel
#    solidSellの混在モデルの場合、EQUATIONを追加
def remakeSolidShellModel(fileName):
    comm = "python3 " + pyFistr.binAppPath + "remakeSolidShellBeamModel.py " + fileName
    #os.system(comm)
    proc = subprocess.run(comm, shell=True)

#
#  changeKeywordUpper
#    keywordを大文字に変換
def changeKeywordUpper(line):
    words = line.split(",")
    words[0] = words[0].upper()
    line = ",".join(words)
    return line

#
#  remakeHeaderNEG
#------------------
def remakeHeaderNEG():
    """ headerの内容を修正する。
    !NODE, !ELEMENT, !SGROUP, !NGROUPについて修正する"""
    global nodeList, elementList, surGroupList, ndsetList
    #nodeを修正(!NODE, NGRP=)
    for i in range(len(nodeList)):
        header = nodeList[i][0]
        ngrp = getValName(header, "NGRP")
        if ngrp != "":
            header = "!NODE, NGRP=" + ngrp + ls
        else:
            header = "!NODE" + ls
        nodeList[i][0]
    #elementを修正(!ELEMENT, TYPE=, EGRP=)
    for i in range(len(elementList)):
        header = elementList[i][0]
        elType = getValName(header, "TYPE")
        egrp = getValName(header, "EGRP")
        header = "!ELEMENT, TYPE=" + elType + ", EGRP=" + egrp + ls
        elementList[i][0] = header
    #sgroupを修正（!SGROUP, SGRP=)
    for i in range(len(surGroupList)):
        header = surGroupList[i][0]
        sgrp = getValName(header, "SGRP")
        header = "!SGROUP, SGRP=" + sgrp + ls
        surGroupList[i][0] = header
    #ngroupを修正(!NGROUP, NGRP=)
    for i in range(len(ndsetList)):
        header = ndsetList[i][0]
        ngrp = getValName(header, "NGRP")
        header = "!NGROUP, NGRP=" + ngrp + ls
        ndsetList[i][0] = header

#
#  addIncludeHeaders
#--------------------
def addIncludeHeaders(fileDir, lines):
    """ *INCLUDE行を追加する。"""
    newLines = []
    for line in lines:
        words = deleteSp(line).split(",")
        if len(words) >= 2:
            if words[0].upper() == "*INCLUDE":
                name = getValName(line, "INPUT")
                if name[0] == "/":
                    fileName = name
                else:
                    fileName = fileDir + "/../" + name
                f = open(fileName, encoding="utf-8"); addLines = f.readlines(); f.close()
                newLines += addLines
            else:
                newLines += [line]
        else:
            newLines += [line]
    return newLines

#
#  abaqus2fistr
#  ------------
#    convert abaqus to fistr
#def abaqus2fistr(fileDir, abaName):
def abaqus2fistr(fileName, newName):
    print
    print ("abaqus mesh will be convert to FrontISTR.")
    #fileName = fileDir + os.sep + abaName + ".inp"
    if os.path.exists(fileName) == False:
        print("error: could not open '" + fileName + "'.")
        return
    f = open(fileName, encoding="utf-8")
    lines = []
    for line in f.readlines():
        if line != "\n" and line !="\r\n" :
            if line[-2:] == "\r\n":
                line = line[:-2] + "\n"
            if line[0] == "*":
                #keywordを大文字に変換する
                line = changeKeywordUpper(line)
            lines.append(line)
    f.close()
    #データ取得
    #includeを処理
    #lines = addIncludeHeaders(fileDir, lines)
    lines = addIncludeHeaders(fileName, lines)
    getNodeElementData(lines)
    #シェル要素をチェックし、シェル要素（731、741）があれば取得
    checkShellElementData()
    #要素定義中のelsetをチェック（同じ要素名の場合、まとめる）
    checkElsetInElement()
    #nodeの並びをチェック
    checkElementListOrder()
    #表面groupの取得,shell要素（761、781）取得
    checkSurfaceElementGroup()
    #beam要素（611）取得
    checkBeamElementGroup()
    #EGRPのチェック修正（要素groupを設定しなかった場合）
    checkNoElementGroup()
    #要素groupの取得（単一要素の場合は、要素定義をgroup分けする）
    checkElementGroup()

    #beam要素（641）を取得
    check641BeamElementGroup()

    #nodeGroupの取得
    checkNodeGroup()
    checkDummyNodeGroup()

    #headerを修正(NODE, ELEMENT, SGROUP, NGROUP)
    remakeHeaderNEG()
    #書き出し(header等をremake）
    #newName = fileDir + os.sep + abaName + "_fistr.msh"
    writeData(newName)
    #EQUATIONを追加（solidShellの場合）
    remakeSolidShellModel(newName)


if __name__=='__main__':
    #log表示
    import logFileCreater
    logFileCreater.log()

    #abaNameDir = sys.argv[1]
    abaName = sys.argv[1]
    newName = sys.argv[2]
    #if abaNameDir[:len("-h")] == "-h" or abaNameDir[:len("--help")] == "--help":
    if abaName[:len("-h")] == "-h" or abaName[:len("--help")] == "--help":
        printMsg()
    else:
        if newName.split(".")[-1] != "msh":
            newName = newName + ".msh"
        abaqus2fistr(abaName, newName)
        # fileDir = os.sep.join(abaNameDir.split(os.sep)[:-1])
        # if fileDir == "":
        #     fileDir = os.getcwd()
        # abaName = abaNameDir.split(os.sep)[-1]
        # if abaName.split(".")[-1] == "inp":
        #     name = abaName[:-4]
        # else:
        #     name = abaName
        # abaqus2fistr(fileDir, name)

