#!/usr/bin/python3
# coding: utf-8
#
#   topoSetEditor
#
#       topoSetDictを編集する為のeditor
#
#   12/09/06    新規作成
#      10/10    actionの位置変更
#      10/13    sourceTypeのsetsを追加。
#               設定したcodeが出力されているかﾁｪｯｸを追加
#      11/24    polyMeshフォルダが無かった場合、ｴﾗｰ発生していたのを修正
#      12/01    openSlFolderボタン追加
#      12/16    combinedActionを追加
#      12/17    表示・非表示の設定修正
#   13/01/27    regionToCellがセットできる様に修正
#      02/03    newCellToFace(cellSet(Zone)の表面のfaceを抽出）を追加
#      04/15    regionsの取得方法を変更（未だregionの取得結果は使っていない）
#      04/17    getCellZoneNames、getFaceZonNmesを高速化
#      04/20    newZonesToSetsコマンドを追加
#      05/13    meshデータの場所をcontrolDictのstartFromの値によって変える。
#      06/25    操作対象のfieldやmeshを選択できるように修正。
#      07/09    topoSetの「-constant」オプションを削除
#      07/10    toposetの「-time」オプションを削除（controlDictに従う）
#      07/13    sourceMesh, resultMeshの場所を表示
#      07/15    ﾊﾞｸﾞ修正
#      10/15    国際化の為、修正
#      11/23    ﾊﾞｸﾞ修正（相対pathで起動できなかった）
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/12/02    getSetsNames:setsDirの取得方法を修正（zoneよりも遡らない）
#   15/07/01    topoSetDictのをOF-2.4.0に合わせる 
#      07/02    resultTypeに「sets」と「zones」を追加。
#      07/03    「sets」「zones」に、add, delete, clear, invert, remove actionを追加
#      10/11    sourceName regionのradio_btnの処理を修正。regionBtnがONの時、
#               resultName cellZone,faceZoneの処理を追加。
#   16/12/14    checkTopoSetCommandDBを追加（topoSetが使えるかどうかをチェック）
#   19/12/26    GTK+3, python3用に大幅修正
#   20/04/22    多言語化対応
#   21/08/21    meshViewerDialog起動ボタンを追加。
#               runMeshViewer:追加。
#      11/23    runMeshViewer:runMeshViewer:mehsViewer起動時にmeshViewerが既に起動
#               しているか確認する事を追加。（複数起動を避けるため）
#      11/25    runMeshViewer:viewerの起動方法を修正
#      12/03    showSelectItemsInViewer:item選択が適正に選択できない事がある為修正
#      12/08    showSelectItemsInViewer_sub:regionの表示を追加
#   22/01/31    runMeshViewer:stlDirの指定を追加（"stl stlDir\n"）を追加
#               getStlFileNames:stlDirが相対pathの場合、stlFileが取得できないを修正
#               getStlDir:stlDirが相対pathの場合、selDirが取得できないを修正
#      02/01    getStlDir:selDirが取得できないをgetOpenFolderDDialog側で修正した
#               ので、元に戻した。
#      07/18    getTopoSetClass:OF-10用のclassの取得を追加
#      08/13    getTopoSetClass:OF-v2206用でエラー発生する為修正
#               OF-10用での修正でバグ発生した。
#   24/08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib
import os, sys
import glob

import pyTreeFoam
import topoSetEditorCommand
import getOpenFolderDDialog
import universalDialogs as unvDlg
import meshViewerControl

#import locale
#localeDir = os.getenv("TreeFoamPath") + "/data/locale"
#locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()


#------------------
#  windowApp class
#------------------
class windowApp:

    def __init__(self, caseDir, stlDir, currTime, selRegion):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"topoSetEditorDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.caseDir = caseDir
        self.stlDir = stlDir
        self.currTime = currTime
        self.selRegion = selRegion
        case = pyTreeFoam.case(caseDir)
        self.contDict = case.getControlDict()
        self.timeFolders = case.getTimeFolders()
        self.currTime = ""          #currentTimeFolder
        self.currMeshDir = ""       #currentMeshDir
        #self.selRegion = ""         #選択しているregion
        self.regions = []           #regionのリスト
        self.maskEvent = True       #eventのmaskFlag
        self.meshViewer = ""        #meshViewerの定義
        #GUIのobject名を取得
        self.setGtkObject()
        #radioボタンのgroup化
        self.sourceTypes = [
            self.radio_cellSet, self.radio_faceSet, self.radio_pointSet,
            self.radio_sets, self.radio_cellZone, self.radio_faceZone,
            self.radio_pointZone, self.radio_box, self.radio_cylinder,
            self.radio_sphere, self.radio_surface, self.radio_rotatedBox,
            self.radio_targetVolume, self.radio_searchableSurface,
            self.radio_region,
            self.radio_boundary, self.radio_field, self.radio_patch,
            self.radio_label, self.radio_shape, self.radio_normal,
            self.radio_nearest
            ]
        self.resultTypes = [
            self.radio_resCellSet, self.radio_resFaceSet, self.radio_resPointSet,
            self.radio_resSets, self.radio_resCellZoneSet, self.radio_resFaceZoneSet,
            self.radio_resPointZoneSet, self.radio_resZones
            ]
        self.actionTypes = [
            self.radio_new, self.radio_add, self.radio_delete,
            self.radio_subset, self.radio_clear, self.radio_invert,
            self.radio_remove
            ]
        self.combiTypes = [
            self.radio_renameSetZone, self.radio_newAddsSet, self.radio_newCellToFace,
            self.radio_newZonesToSets
            ]
        #GUIの初期化
        self.initializeGui()
        #configDictを取得
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        #timeFolder, meshDirを取得
        self.getTimeFolderMeshDir()
        #titleを設定
        self.setTitle()
        #treeListの初期化
        self.iniTreeList()
        #全radioボタンを表示
        self.showAllRadioButtons()
        #topoSetDBを確認
        self.makeRunTopoSet =  self.getTopoSetClass()


    def setGtkObject(self):
        """ glade内のobject名を取得する"""
        self.box_action = self.builder.get_object("box_action")
        self.box_source = self.builder.get_object("box_source")
        self.box_result = self.builder.get_object("box_result")
        self.label_action = self.builder.get_object("label_action")
        self.label_source = self.builder.get_object("label_source")
        self.label_result = self.builder.get_object("label_result")
        self.combo_time = self.builder.get_object("combo_time")
        self.combo_region = self.builder.get_object("combo_region")
        self.radio_new = self.builder.get_object("radio_new")
        self.radio_add = self.builder.get_object("radio_add")
        self.radio_delete = self.builder.get_object("radio_delete")
        self.radio_subset = self.builder.get_object("radio_subset")
        self.radio_clear = self.builder.get_object("radio_clear")
        self.radio_invert = self.builder.get_object("radio_invert")
        self.radio_remove = self.builder.get_object("radio_remove")
        self.radio_renameSetZone = self.builder.get_object("radio_renameSetZone")
        self.radio_newAddsSet = self.builder.get_object("radio_newAddsSet")
        self.radio_newCellToFace = self.builder.get_object("radio_newCellToFace")
        self.radio_newZonesToSets = self.builder.get_object("radio_newZonesToSets")
        self.radio_cellSet = self.builder.get_object("radio_cellSet")
        self.radio_faceSet = self.builder.get_object("radio_faceSet")
        self.radio_pointSet = self.builder.get_object("radio_pointSet")
        self.radio_sets = self.builder.get_object("radio_sets")
        self.radio_cellZone = self.builder.get_object("radio_cellZone")
        self.radio_faceZone = self.builder.get_object("radio_faceZone")
        self.radio_pointZone = self.builder.get_object("radio_pointZone")
        self.radio_box = self.builder.get_object("radio_box")
        self.radio_cylinder = self.builder.get_object("radio_cylinder")
        self.radio_sphere = self.builder.get_object("radio_sphere")
        self.radio_surface = self.builder.get_object("radio_surface")
        self.radio_rotatedBox = self.builder.get_object("radio_rotatedBox")
        self.radio_targetVolume = self.builder.get_object("radio_targetVolume")
        self.radio_searchableSurface = self.builder.get_object("radio_searchableSurface")
        self.radio_region = self.builder.get_object("radio_region")
        self.radio_boundary = self.builder.get_object("radio_boundary")
        self.radio_field = self.builder.get_object("radio_field")
        self.radio_patch = self.builder.get_object("radio_patch")
        self.radio_label = self.builder.get_object("radio_label")
        self.radio_shape = self.builder.get_object("radio_shape")
        self.radio_normal = self.builder.get_object("radio_normal")
        self.radio_nearest = self.builder.get_object("radio_nearest")
        self.radio_resCellSet = self.builder.get_object("radio_resCellSet")
        self.radio_resFaceSet = self.builder.get_object("radio_resFaceSet")
        self.radio_resPointSet = self.builder.get_object("radio_resPointSet")
        self.radio_resSets = self.builder.get_object("radio_resSets")
        self.radio_resCellZoneSet = self.builder.get_object("radio_resCellZoneSet")
        self.radio_resFaceZoneSet = self.builder.get_object("radio_resFaceZoneSet")
        self.radio_resPointZoneSet = self.builder.get_object("radio_resPointZoneSet")
        self.radio_resZones = self.builder.get_object("radio_resZones")
        self.tree_source = self.builder.get_object("tree_source")
        self.tree_result = self.builder.get_object("tree_result")
        self.btn_surface = self.builder.get_object("btn_surface")
        self.btn_getFromList = self.builder.get_object("btn_getFromList")
        self.entry_resultName = self.builder.get_object("entry_resultName")
        self.text_code = self.builder.get_object("text_code")

    #
    #  initializeGui
    #---------------
    def initializeGui(self):
        """ GUIの初期化"""
        #Gtk.Boxの背景色をlightgrayに変更
        gray_css = Gtk.CssProvider.new()
        cont  = b"box\n"
        cont += b"{\n"
        cont += b"    background-color: gainsboro;\n"
        cont += b"}\n"
        gray_css.load_from_data(cont)
        style_action = self.box_action.get_style_context()
        style_action.add_provider(gray_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_source = self.box_source.get_style_context()
        style_source.add_provider(gray_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_result = self.box_result.get_style_context()
        style_result.add_provider(gray_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        #labelのfontをboldに変更
        bold_css = Gtk.CssProvider.new()
        cont  = b"label\n"
        cont += b"{\n"
        cont += b"    font-weight: bold;\n"
        cont += b"}\n"
        bold_css.load_from_data(cont)
        style_actionFont = self.label_action.get_style_context()
        style_actionFont.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_sourceFont = self.label_source.get_style_context()
        style_sourceFont.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_resultFont = self.label_result.get_style_context()
        style_resultFont.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        #radioBtnを選択
        self.radio_cellSet.set_active(True)
        self.radio_resCellSet.set_active(True)
        self.radio_new.set_active(True)

    #
    #  getTimeFolderMeshDir
    #-----------------------
    def getTimeFolderMeshDir(self):
        """ timeFolder, region, meshDirを取得する"""
        case = pyTreeFoam.case(self.caseDir)
        if self.currTime == "":
            self.currTime = case.getCurrTimeFolder()
        else:
            if len(glob.glob(self.currTime)) == 0:
                self.currTime = case.getCurrTimeFolder()
        (_flag, _loc, self.regions) = case.isMultiRegion()
        if self.selRegion == "":
            self.selRegion = "."
        elif len(glob.glob(self.caseDir + "/constant/" + self.selRegion)) == 0:
            self.selRegion = "."
        self.regions = ["."] + self.regions
        relMeshDir = case.getCurrMeshDir(self.currTime, self.selRegion, "boundary")
        self.currMeshDir = self.caseDir + "/" + relMeshDir

    #
    #  setTitle
    #-----------
    def setTitle(self):
        """ windowのtitleを設定する"""
        subTitle = "/".join(self.caseDir.split("/")[-2:])
        title = _("mesh抽出    :") + subTitle
        self.mainWindow.set_title(title)

    #
    #  iniTreeList
    #--------------
    def iniTreeList(self):
        """ treeListの初期化"""
        #tree_sourceの初期化
        self.tree_source.get_selection().set_mode(Gtk.SelectionMode.MULTIPLE)
        self.tree_source.set_headers_visible(False)
        cell = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn()
        col.pack_start(cell, True)
        col.add_attribute(cell, "text", 0)
        self.tree_source.append_column(col)
        #tree_resultの初期化
        self.tree_result.set_headers_visible(False)
        cell = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn()
        col.pack_start(cell, True)
        col.add_attribute(cell, "text", 0)
        self.tree_result.append_column(col)
        selection = self.tree_result.get_selection()
        selection.set_mode(Gtk.SelectionMode.MULTIPLE)

    #
    #  main
    #--------
    def main(self):
        """ GUIの表示"""
        #comboBoxにデータセット
        self.setComboBoxes()
        #maskEventを解除
        self.maskEvent = False
        #listBoxにデータセット
        self.setNameList()
        #resultのBtnを隠す
        self.hideResultRadioBtnsFromSourceBtns()
        #GUIを表示
        self.mainWindow.show()
        Gtk.main()

    #
    #  close
    #--------
    def close(self, *args):
        """ 閉じる"""
        #meshViewerの処理
        if self.meshViewer != "":
            self.meshViewer.closeViewer()
        Gtk.main_quit()

    #------- event handler ------------------------
    #閉じる
    def onClose(self, event):
        self.close()
    #actionのradioを選択した時
    def onChangeRadioBtn(self, radioBtn):
        if self.maskEvent == True:
            return
        if radioBtn.get_active() == True:
            self.changeRadioButton()
    #sourceのradioを選択した時
    def onSelectSourceRadioBtn(self, radioBtn):
        if self.maskEvent == True:
            return
        if radioBtn.get_active() == True:
            self.selectSourceRadioButton()
    #resultのradioを選択した時
    def onSelectResultRadioBtn(self, radioBtn):
        if self.maskEvent == True:
            return
        if radioBtn.get_active() == True:
            self.selectResultRadioButton()
    #time変更
    def onChangeTime(self, event):
        if self.maskEvent == True:
            return
        self.changeTime()
    #region変更
    def onChangeRegion(self, event):
        if self.maskEvent == True:
            return
        self.changeRegion()
    #sourceName選択
    def onChangeItem(self, event):
        if self.maskEvent == True:
            return
        self.showSelectItemsInViewer()
    #----- entry ------------
    #resultName入力
    def onInputResultName(self, event, *args):
        self.setTopoSetCode()
    #---- button -----------
    #↑listから取得
    def onGetResultName(self, event):
        self.resultName()
    #code出力
    def onSetTopoSetCode(self, event):
        self.setTopoSetCode()
    #clear
    def onClearTopoSetDict(self, event):
        self.clearTopoSetDict()
    #追加
    def onAddTopoSetDict(self, event):
        self.addTopoSetDict()
    #実行
    def onRunTopoSetDict(self, event):
        self.runTopoSetDict()
    #clearAddRun
    def onClearAddRunTopoSetDict(self, event):
        self.clearTopoSetDict()
        self.addTopoSetDict()
        self.runTopoSetDict()
    #topoSetDict編集
    def onEditTopoSetDict(self, event):
        self.editTopoSetDict()
    #paraFoamの起動
    def onRunParaFoam(self, event):
        self.runParaFoam()
    #stlDirを取得
    def onGetStlDir(self, event):
        self.getStlDir()
    #meshViewer起動
    def onRunMeshViewer(self, event):
        self.runMeshViewer()

    #
    #  getTopoSetClass
    #------------------
    def getTopoSetClass(self):
        """ OFversionを確認し、versionに応じたtopoSetDBを取得"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        #if OFversion <= "v1806":
        if ( (0 < numVer and numVer < 10.0) or 
            (OFversion[0] == "v" and OFversion <= "v1806") ):
            #標準のOF（v1806も含む）
            #topoSetDictを取得する
            topoSetDict = pyTreeFoam.getFoamContents().defaultDictPathInDataFolder(OFversion, "system/topoSetDict")
            #originalのdictは、
            #   <OpenFOAM>/applications/utilities/mesh/manipulation/topoSet/topoSetDict
            #   <OpenFOAM>/etc/caseDicts/annotated/topoSetDict
            #のいずれかに保存されている
            if topoSetDict == "":
                print("error: does not exist topoSet.")
                return ""
            #topoSetDictからcommandDB作成
            makeRunTopoSet = topoSetEditorCommand.makeTopoSet(topoSetDict)
        elif numVer >= 10.0 :
            #OF-10以降
            #topoSetDictを取得する
            topoSetDict = pyTreeFoam.getFoamContents().defaultDictPathInDataFolder(OFversion, "system/topoSetDict")
            #originalのdictは、
            #   <OpenFOAM>/applications/utilities/mesh/manipulation/topoSet/topoSetDict
            #   <OpenFOAM>/etc/caseDicts/annotated/topoSetDict
            #のいずれかに保存されている
            if topoSetDict == "":
                print("error: does not exist topoSet.")
                return ""
            #topoSetDictからcommandDB作成
            makeRunTopoSet = topoSetEditorCommand.makeTopoSet_10(topoSetDict)
        else:
            #v1812以上のOF
            #topoSetDictを取得する
            topoSetDict = pyTreeFoam.getFoamContents().defaultDictPathInDataFolder(OFversion, "system/topoSetSourcesDict")
            #originalのdictは、
            #   <OpenFOAM>/etc/caseDicts/annotated/topoSetSourceDict
            #に保存されている
            if topoSetDict == "":
                print("error: does not exist topoSet.")
                return ""
            #topoSetDictからcommandDB作成
            makeRunTopoSet = topoSetEditorCommand.makeTopoSet1812(topoSetDict)
        return makeRunTopoSet

    #
    #  setComboBoxes
    #----------------
    def setComboBoxes(self):
        """ comboBoxにデータをセットする"""
        self.setTimeToComboBox()
        self.setRegionToComboBox()

    def setTimeToComboBox(self):
        items = []
        firstTime = self.timeFolders[0]
        startTime = self.contDict["startTime"]
        latestTime = self.timeFolders[-1]
        items.append("firstTime:" + firstTime)
        items.append("startTime:" + startTime)
        items.append("latestTime:" + latestTime)
        self.setComboDataInComboBox(self.combo_time, items)
        if self.currTime == startTime:
            selNo = 1
        elif self.currTime == firstTime:
            selNo = 0
        else:
            selNo = 2
        self.combo_time.set_active(selNo)

    def setRegionToComboBox(self):
        items = self.regions[:]
        selNo = items.index(self.selRegion)
        items[0] = "(region0)"
        self.setComboDataInComboBox(self.combo_region, items)
        self.combo_region.set_active(selNo)

    def setComboDataInComboBox(self, comboBox, items):
        """ comboBoxにデータをセットする"""
        comboBox.clear()
        listModel = Gtk.ListStore(str)
        for name in items:
            listModel.append([name])
        cell = Gtk.CellRendererText()
        comboBox.pack_start(cell, True)
        comboBox.add_attribute(cell, "text", 0)
        comboBox.set_model(listModel)
        comboBox.set_entry_text_column(0)

    #
    #  showAllRadioButtons
    #----------------------
    def showAllRadioButtons(self):
        """ 全radioButtonsを表示させる"""
        #actionの表示
        for radio in self.actionTypes:
            radio.set_sensitive(True)
        #combineの表示
        for radio in self.combiTypes:
            radio.set_sensitive(True)
        #sourceの表示
        for radio in self.sourceTypes:
            radio.set_sensitive(True)
        self.btn_surface.set_sensitive(True)
        self.tree_source.set_sensitive(True)
        #resultの表示
        for radio in self.resultTypes:
            radio.set_sensitive(True)
        self.tree_result.set_sensitive(True)
        self.btn_getFromList.set_sensitive(True)

    #
    #  hideSourceRadioButtons
    def hideSourceRadioButtons(self):
        """ sourceTypeのRadioボタンを隠す"""
        hideBtns = []
        #no source？
        if (self.radio_clear.get_active() == True or
            self.radio_invert.get_active() == True or
            self.radio_remove.get_active() == True):
            hideBtns = self.sourceTypes[:] + [self.btn_surface, self.tree_source]
        #combi renameSetZone?
        elif self.radio_renameSetZone.get_active() == True:
            hideBtns = [
                self.radio_pointSet, self.radio_sets, self.radio_pointZone,
                self.radio_box, self.radio_cylinder, self.radio_sphere,
                self.radio_surface, self.btn_surface, self.radio_rotatedBox,
                self.radio_targetVolume, self.radio_searchableSurface, self.radio_region,
                self.radio_boundary, self.radio_field, self.radio_patch,
                self.radio_label, self.radio_shape, self.radio_normal,
                self.radio_nearest
                ]
        #combi newAddsSet?
        elif self.radio_newAddsSet.get_active() == True:
            hideBtns = [
                self.radio_pointSet,
                self.radio_sets, self.radio_cellZone, self.radio_faceZone,
                self.radio_pointZone, self.radio_box, self.radio_cylinder,
                self.radio_sphere, self.radio_surface, self.btn_surface, self.radio_rotatedBox,
                self.radio_targetVolume, self.radio_searchableSurface, self.radio_region,
                self.radio_boundary, self.radio_field, self.radio_patch,
                self.radio_label, self.radio_shape, self.radio_normal,
                self.radio_nearest
                ]
        #combi newCellToFace?
        elif self.radio_newCellToFace.get_active() == True:
            hideBtns = [
                self.radio_pointSet,
                self.radio_sets,
                self.radio_pointZone, self.radio_box, self.radio_cylinder,
                self.radio_sphere, self.radio_surface, self.btn_surface, self.radio_rotatedBox,
                self.radio_targetVolume, self.radio_searchableSurface, self.radio_region,
                self.radio_boundary, self.radio_field, self.radio_patch,
                self.radio_label, self.radio_shape, self.radio_normal,
                self.radio_nearest
                ]
        #combi newZonesToSets?
        elif self.radio_newZonesToSets.get_active() == True:
            hideBtns = [
                self.radio_cellSet, self.radio_faceSet, self.radio_pointSet,
                self.radio_sets,
                self.radio_pointZone, self.radio_box, self.radio_cylinder,
                self.radio_sphere, self.radio_surface, self.btn_surface, self.radio_rotatedBox,
                self.radio_targetVolume, self.radio_searchableSurface, self.radio_region,
                self.radio_boundary, self.radio_field, self.radio_patch,
                self.radio_label, self.radio_shape, self.radio_normal,
                self.radio_nearest
                ]
        #ボタンを隠す
        if len(hideBtns) != 0:
             for btn in hideBtns:
                 btn.set_sensitive(False)

    #
    #  changeRadioButton
    #--------------------
    def changeRadioButton(self):
        """ actionのradioボタンを変更した時の処理。
        onしているradioボタンを確認し、ボタンを非表示に設定する。"""
        self.showAllRadioButtons()
        self.hideSourceRadioButtons()
        self.hideResultRadioBtnsFromSourceBtns()

    #
    #  selectSourceRadioButton
    #----------------
    def selectSourceRadioButton(self):
        """ sourceのradioBtnを選択した時の処理。"""
        #sourceのlistBoxにデータセット
        self.setSourceNameList()
        #resultの全radioボタンを表示
        for radio in self.resultTypes:
            radio.set_sensitive(True)
        #resultのradioボタンを隠す
        self.hideResultRadioBtnsFromSourceBtns()

    #
    #  hideResultRadioBtnsFromSourceBtns
    def hideResultRadioBtnsFromSourceBtns(self):
        """ resultBtnを隠す。
        sourceの選択BtnからtopoSetDBを検索して、必要なresultBtnを取得し、不要なBtnを隠す。"""
        #sourceの選択btnから検索wordを取得する
        radios = list(filter(lambda x: x.get_active(), self.sourceTypes))
        selName = radios[0].get_label()             #選択Btn
        if selName == "cellSet":
            searchNames = ["cellTo", "setTo"]       #検索word
        elif selName == "faceSet":
            searchNames = ["faceTo", "setTo"]
        elif selName == "pointSet":
            searchNames = ["pointTo", "setTo"]
        elif selName == "sets":
            searchNames = ["setsTo"]       
        elif selName == "cellZone":
            searchNames = ["zoneTo", "cellZoneTo"]
        elif selName == "faceZone":
            searchNames = ["zoneTo", "faceZoneTo"]
        elif selName == "pointZone":
            searchNames = ["zoneTo", "pointZoneTo"]
        else:
            searchNames = [selName + "To"]       
        #topoSetDBを検索する
        lines = self.makeRunTopoSet.topoSetDB.split("\n")
        resNames = []
        for line in lines:
            #「source cellToCell;」の様なsource行を確認する
            words = line.split()
            if len(words) >= 2:
                if words[0] == "source":
                    for searchName in searchNames:
                        if words[1][:len(searchName)] == searchName:
                            resName = words[1][len(searchName):]
                            resName = resName.split(";")[0].lower()
                            resNames.append(resName)
        resNames = list(set(resNames))
        #辞書作成
        radioDict = {
            "cell": self.radio_resCellSet,
            "face": self.radio_resFaceSet,
            "point": self.radio_resPointSet,
            "cellzone": self.radio_resCellZoneSet,
            "facezone": self.radio_resFaceZoneSet,
            "pointzone": self.radio_resPointZoneSet
            }
        #表示させる（必要な）btnを取得する
        showRadioBtn = list(map(lambda x: radioDict[x], resNames))
        if self.radio_new.get_active() == True:
            if selName[-3:] == "Set":
                showRadioBtn.append(self.radio_resZones)
            elif selName[-4:] == "Zone":
                showRadioBtn.append(self.radio_resSets)
        #combiBtnをチェック（全て隠す）
        radios = list(filter(lambda x: x.get_active(), self.combiTypes))
        if len(radios) != 0:
            showRadioBtn = []
            self.btn_getFromList.set_sensitive(False)
            self.tree_result.set_sensitive(False)
        #no sourceをチェック（全て表示）
        if (self.radio_clear.get_active() or
            self.radio_invert.get_active() or
            self.radio_remove.get_active()):
            showRadioBtn = self.resultTypes[:]
        #隠すbtnを取得し、btnを隠す
        for radio in self.resultTypes:
            if not radio in showRadioBtn:
                radio.set_sensitive(False)

    #
    #  selectResultRadioButton
    #--------------------------
    def selectResultRadioButton(self):
        """ sourceのradioBtnを選択した時の処理。"""
        #resultのlistBoxにデータをセット
        self.setResultNameList()

    #
    #  changeTime
    #-------------
    def changeTime(self):
        """ timeを変更した時の処理"""
        entry = self.combo_time.get_child()
        text = entry.get_text()
        selTime = "".join(text.split()).split(":")[-1]
        if len(glob.glob(self.caseDir + "/" + selTime)) == 0:
            self.combo_time.set_active(0)
            title = _("エラー")
            mess = _("選択したtimeFolderは、存在しません。") + "\n"
            mess += _("  firstTime に変更しました。")
            self.errDialog(title, mess)
            return
        #timeを変更
        self.currTime = selTime
        case = pyTreeFoam.case(self.caseDir)
        relMeshDir = case.getCurrMeshDir(self.currTime, self.selRegion, "boundary")
        self.currMeshDir = self.caseDir + "/" + relMeshDir
        #データを再読み込み
        self.setNameList()
        #meshViewerの処理
        if self.meshViewer != "":
            commLines = ["start\n"]
            commLines += ["time " + self.currTime + "\n"]
            commLines += ["load patchesZones sets\n"]
            commLines += ["resetCamera\n"]
            self.meshViewer.runMultiCommands(commLines)

    #
    #  changeRegion
    #---------------
    def changeRegion(self):
        """ regionを変更した時の処理"""
        entry = self.combo_region.get_child()
        region = entry.get_text()
        #regionを変更
        if region == "(region0)":
            self.selRegion = "."
        else:
            self.selRegion = region
        case = pyTreeFoam.case(self.caseDir)
        relMeshDir = case.getCurrMeshDir(self.currTime, self.selRegion, "boundary")
        self.currMeshDir = self.caseDir + "/" + relMeshDir
        #データ再読み込み
        self.setNameList()
        #meshViewerの処理
        if self.meshViewer != "":
            commLines = ["start\n"]
            commLines += ["region " + self.selRegion + "\n"]
            commLines += ["load patchesZones sets\n"]
            commLines += ["resetCamera\n"]
            self.meshViewer.runMultiCommands(commLines)

    #
    #  resultName
    #-------------
    def resultName(self):
        """ resultNameを取得し、textBoxにセットする"""
        selection = self.tree_result.get_selection()
        treeStore, pathes = selection.get_selected_rows()
        if len(pathes) == 0:
            title = _("エラー")
            mess = _("resultNameを選択してください。")
            self.errDialog(title, mess)
        path = pathes[0]
        treeIter = treeStore.get_iter(path)
        resName = treeStore.get_value(treeIter, 0)
        self.entry_resultName.set_text(resName)

    #
    #  getselectNamesInTreeList
    def getSelectNamesInTreeList(self, treeList):
        """ treeList内の選択したnameを取得する。"""
        treeSelection = treeList.get_selection()
        treeStore, pathes = treeSelection.get_selected_rows()
        selNames = []
        for path in pathes:
            treeIter = treeStore.get_iter(path)
            value = treeStore.get_value(treeIter, 0)
            selNames.append(value)
        return selNames

    #
    #  setTopoSetCode
    #-----------------
    def setTopoSetCode(self):
        """ topoSetCodeをtextBoxに出力する。"""
        (sourceType, sourceName, action, resultType, resultName) = self.readSettingItem()
        #combi?
        if self.isCombinedAction() == True:
            code = self.makeRunTopoSet.makeCombiContents(sourceType, sourceName, action,
                        resultType, resultName)
        elif self.isSetsOrZonesResult() == True:
            if (self.radio_clear.get_active() or
                self.radio_invert.get_active() or
                self.radio_remove.get_active()):
                resultName = self.getSelectNamesInTreeList(self.tree_result)
            code = self.makeRunTopoSet.makeSetsZonesContents(sourceType, sourceName, action,
                        resultType, resultName)
        else:
            code = self.makeRunTopoSet.makeContents(sourceType, sourceName, action,
                        resultType, resultName)
        #textViewにcodeをセット
        buffer = self.text_code.get_buffer()
        buffer.set_text(code)

    #
    #  isSetsOrZonesResult
    def isSetsOrZonesResult(self):
        """ resultが sets or zones の場合か"""
        flag = False
        if self.radio_resSets.get_active() or self.radio_resZones.get_active():
            flag = True
        return flag

    #
    #  isCombinedAction
    def isCombinedAction(self):
        """ combinedActionを選択しているか"""
        radios = list(filter(lambda x: x.get_active(), self.combiTypes))
        if len(radios) != 0:
            return True
        else:
            return False

    #
    #  readSettingItem
    def readSettingItem(self):
        """ setした項目を読み込む"""
        #sourceTypeを取得
        radios = list(filter(lambda x: x.get_active(), self.sourceTypes))
        if len(radios) != 0:
            sourceType = radios[0].get_label()
        else:
            sourceType = ""
        #sourceNamesを取得
        sourceNames = self.getSelectNamesInTreeList(self.tree_source)
        if len(sourceNames) == 1:
            words = sourceNames[0].split(".")
            if words[-1] == "stl":
                sourceName = self.stlDir + "/" + sourceNames[0]
            else:
                sourceName = sourceNames[0]
        elif len(sourceNames) == 0:
            if sourceType == "sets":
                sourceName = []
            else:
                sourceName = ""
        else:
            sourceName = sourceNames
        #actionを取得
        radios = list(filter(lambda x: x.get_active(), self.actionTypes+self.combiTypes))
        action = radios[0].get_label()
        #resultTypeを取得
        radios = list(filter(lambda x: x.get_active(), self.resultTypes))
        resultType = radios[0].get_label()
        #resultNameを取得
        resultName = self.entry_resultName.get_text()
        return (sourceType, sourceName, action, resultType, resultName)

    #
    #  clearTopoSetDict
    #-------------------
    def clearTopoSetDict(self):
        """ topoSetDictの内容（actionsの内容）をクリアする。"""
        getFoamContents = pyTreeFoam.getFoamContents(self.caseDir)
        OFversion = self.configDict["OFversion"]
        header = getFoamContents.makeFoamHeaderVersion(OFversion)
        version = "2.0"
        format = "ascii"
        className = "dictionary"
        location = "system/" + self.selRegion
        objectName = "topoSetDict"
        header += getFoamContents.makeFoamFile(version, format, className, location, objectName)
        actions = "\nactions\n(\n);\n\n"
        footer = getFoamContents.makeFoamFooter()
        dictCont = header + actions + footer
        self.writeTopoSetDict(dictCont)

    #
    #  writeTopoSetDict
    def writeTopoSetDict(self, cont):
        """ topoSetDictに書き込み"""
        fileName = self.caseDir + "/system/" + self.selRegion + "/topoSetDict"
        f = open(fileName, "w"); f.write(cont); f.close()

    #
    #  addTopoSetDict
    #-----------------
    def addTopoSetDict(self):
        """ 確定した項目をtopoSetDictに追加する。"""

        def errorCheck():
            (sourceType, sourceName, action, resultType, resultName) = self.readSettingItem()
            msg = ""
            #actionに応じたチェック
            if (action != "clear" and action != "invert" and action != "remove" and
                action != "renameSetZone" and action != "newAddsSet" and
                action != "newCellToFace" and action != "newZonesToSets" and
                resultType != "sets" and resultType != "zones"):
                #sourceInfoのチェック
                sourceInfo = self.makeRunTopoSet.getSourceInfo(sourceType, sourceName,
                                action, resultType)
                if sourceInfo == "":
                    msg = u"(" + action + u" " + sourceType + u" " + resultType + u") " + _(u"の組み合わせはできません。")
            elif (action == "renameSetZone" or action == "newAddsSet" or
                action == "newInvertSet"  or action == "newZonesToSets"):
                sourceInfo = self.makeRunTopoSet.makeCombiContents(sourceType, sourceName,
                                action, resultType, resultName)
                if sourceInfo == "":
                    msg = u"(" + action + u" " + sourceType + u" " + resultType + u") " + _(u"の組み合わせはできません。")

            #resultNameのチェック
            if (len(resultName.split()) == 0):
                if (action != "newZonesToSets" and resultType != "sets" and resultType != "zones"):
                    msg = _(u"「result name」がセットされていません。")

            #最新のcodeかどうかのチェック（sourceTypeのチェックはせず）
            if (action != "renameSetZone" and action != "newAddsSet" and
                action != "newCellToFace" and action != "newZonesToSets" and 
                resultType != "sets" and resultType != "zones"):
                #currCont = self.text_ctrl_2.GetValue()
                buffer = self.text_code.get_buffer()
                start = buffer.get_start_iter()
                end = buffer.get_end_iter()
                currCont = buffer.get_text(start, end, True)
                errMsg = _(u"設定した組み合わせのcodeが表示されていません。\n「code出力」ボタンをクリックしてcodeを出力してください。")
                if (currCont.find(resultType) < 0 or currCont.find(resultName) < 0
                        or currCont.find(action) < 0):
                    msg = errMsg
            #error表示
            if msg != "":
                title = _(u"エラー")
                print("error:" + msg)
                self.errDialog(title, msg)
            return msg

        error = errorCheck()
        if error != "":
            return
        buffer = self.text_code.get_buffer()
        start = buffer.get_start_iter()
        end = buffer.get_end_iter()
        code = buffer.get_text(start, end, True)
        cont = self.readTopoSetDict()
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        p = contOp.getKeywordPointer("actions", p)
        sp = p
        (_actionCont, p) = contOp.getSmallPair(p)
        ep = p
        actions = cont[sp:ep-1] + code + ")"
        dictCont = cont[:sp] + actions + cont[ep:]
        self.writeTopoSetDict(dictCont)

    #
    #  readTopoSetDict
    def readTopoSetDict(self):
        """ topoSetDictの読み込み"""
        fileName = self.caseDir + "/system/" + self.selRegion + "/topoSetDict"
        f = open(fileName); cont = f.read(); f.close()
        return cont

    #
    #  runTopoSetDict
    #------------------
    def runTopoSetDict(self):
        """ topoSetDictの実行"""
        #comm作成
        bashrcFOAM = self.configDict["bashrcFOAM"]
        comm = ". " + bashrcFOAM + "; "
        comm += "topoSet -region " + self.selRegion
        print(comm)
        (stat, _res, _err) = pyTreeFoam.run(self.caseDir).commandReturnCont(comm)
        title = _("topoSet実行")
        if stat == "OK":
            mess = _("topoSetを実行しました")
            self.okDialog(title, mess)
        else:
            mess = _("error または warning が発生しています。\n  logを確認してください。")
            self.errDialog(title, mess)
        self.setNameList()

    #
    #  editTopoSetDict
    #------------------
    def editTopoSetDict(self):
        """ topoSetDictを編集"""
        fileName = self.caseDir + "/system/" + self.selRegion + "/topoSetDict"
        pyTreeFoam.run(self.caseDir).editor([fileName])

    #
    #  runParaFoam
    #--------------
    def runParaFoam(self):
        """ paraFoamの起動"""
        paraFoam = self.configDict["paraFoam"]
        comm = paraFoam + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  getStlDir
    #------------
    def getStlDir(self):
        """ stlDirを取得する。"""
        title = _(u"stlファイルの場所")
        mess = _(u"stlファイルの場所を設定")
        rootDir = self.configDict["rootDir"]
        currDir = self.caseDir
        selDir = self.stlDir
        okArgs = [self.getStlDir_setStl]
        dialog = getOpenFolderDDialog.getOpenFolder(
            title, mess, rootDir, currDir, selDir, 
            funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #  getStlDir_setStl
    def getStlDir_setStl(self, selDir):
        self.stlDir = selDir
        self.setSourceNameList()
        #meshViewerへの処理
        if self.meshViewer != "":
            commLines  = ["start\n"]
            commLines += ["stl " + self.stlDir + "\n"]
            commLines += ["load stls\n"]
            commLines += ["end\n"]
            self.meshViewer.runMultiCommands(commLines)

    #
    #  runMeshViewer
    #----------------
    def runMeshViewer(self):
        """ meshViewerを起動する。"""
        if self.meshViewer != "":
            if self.meshViewer.isAlive() == True:
                title = _("meshViewer")
                mess = _("meshViewerは、既に起動しています。")
                self.errDialog(title, mess)
                return
        caseDir = self.caseDir
        region = self.selRegion
        timeFolder = self.currTime
        title = "meshViewer_topoSetEditor_" + str(os.getpid())
        #viewer定義
        self.meshViewer = meshViewerControl.control(caseDir, timeFolder, region, title)
        #viewer起動
        self.meshViewer.openViewer("normal")
        #コマンド入力
        commLines  = [_("start\n")]
        commLines += [_("mess topoSetEditorにlinkしています。\n")]
        commLines += [_("addMess '  topoSetEditor上で選択した項目が表示されます。'\n")]
        commLines += [_("addMess '  patch, zone, set, surface(stl)が表示できます。'\n")]
        commLines += [_("print 'loading ...'\n")]
        commLines += [_("wait 0.2s\n")]     #GUI表示させる為
        commLines += [_("stl " + self.stlDir + "\n")]
        commLines += [_("load patchesZones sets stls\n")]
        commLines += [_("show\n")]
        commLines += [_("resetCamera\n")]
        commLines += [_("end\n")]
        stat = self.meshViewer.runMultiCommands(commLines)
        #選択項目を表示
        self.showSelectItemsInViewer()

    #
    #  showSelectItemsInViewer
    #--------------------------
    def showSelectItemsInViewer(self):
        """ 選択項目を表示させる。"""
        #viewerが定義されているか？
        if self.meshViewer == "":
            return
        #直ぐに処理するとtreeList内の選択nameが読み取れない
        GLib.timeout_add(50, self.showSelectItemsInViewer_sub)
        #self.showSelectItemsInViewer_sub()
        return

    #
    #  showSelectItemsInViewer_sub
    def showSelectItemsInViewer_sub(self):
        #選択項目を取得
        (sourceType, sourceName,
         action, resultType, resultName) = self.readSettingItem()
        #sourceNameが非選択？
        if sourceName == "":
            #非選択の場合は、全actorを消す
            self.meshViewer.runCommand("show\n")
            return
        #sourceNamesを取得
        sourceNames = []
        if type(sourceName) == str:
            sourceNames = [sourceName]
        else:
            sourceNames = sourceName
        #command（選択項目を表示させる）を取得
        commands = []
        #patch, zone, set?
        if (sourceType == "patch" or sourceType == "cellZone" or
            sourceType == "faceZone" or sourceType == "pointZone" or
            sourceType == "cellSet" or sourceType == "faceSet" or
            sourceType == "pointSet" or sourceType == "region"):
            showItems = []
            #patch, zone, setの場合
            for name in sourceNames:
                item = sourceType + ":" + name
                showItems.append(item)
            command = "show " + " ".join(showItems) + "\n"
            commands.append(command)
        #sets?
        elif sourceType == "sets":
            #setsの場合
            showItems = []
            for setName in sourceNames:
                cfp, name = setName.split(":")
                if cfp == "c":
                    item = "cellSet:" + name
                elif cfp == "f":
                    item = "faceSet:" + name
                elif cfp == "p":
                    item = "pointSet:" + name
                showItems.append(item)
            command = "show " + " ".join(showItems) + "\n"
            commands.append(command)
        #surface?
        elif sourceType == "surface":
            #stlの場合
            showItems = []
            for namePath in sourceNames:
                stlName = os.path.basename(namePath)
                name = stlName[:-4]
                item = "stl:" + name
                showItems.append(item)
            command = "show " + " ".join(showItems) + "\n"
            commands.append(command)
        #command送出
        self.meshViewer.runMultiCommands(commands)

    #
    #  setNameList
    #--------------
    def setNameList(self):
        """ listBoxにデータをセット"""
        self.setSourceNameList()
        self.setResultNameList()

    #
    #  setSourceNameList
    def setSourceNameList(self):
        """ sourceNameをlistBoxにセット"""
        if self.radio_cellSet.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setSourceNameInListBox(cellSets)
        elif self.radio_faceSet.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setSourceNameInListBox(faceSets)
        elif self.radio_pointSet.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setSourceNameInListBox(pointSets)
        elif self.radio_cellZone.get_active() == True:
            cellZones = self.getCellZoneNames()
            self.setSourceNameInListBox(cellZones)
        elif self.radio_faceZone.get_active() == True:
            faceZones = self.getFaceZoneNames()
            self.setSourceNameInListBox(faceZones)
        elif self.radio_pointZone.get_active() == True:
            pointZones = self.getPointZoneNames()
            self.setSourceNameInListBox(pointZones)
        elif self.radio_surface.get_active() == True:
            stlFiles = self.getStlFileNames()
            self.setSourceNameInListBox(stlFiles)
        elif self.radio_region.get_active() == True:
            regions = self.getRegionNames()
            self.setSourceNameInListBox(regions)
        elif self.radio_field.get_active() == True:
            fields = self.getFieldNames()
            self.setSourceNameInListBox(fields)
        elif self.radio_patch.get_active() == True:
            patches = self.getPatchNames()
            self.setSourceNameInListBox(patches)
        elif self.radio_sets.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setSourceSetsNamesInListBox(cellSets, faceSets, pointSets)
        else:
            self.setSourceNameInListBox([])

    def setSourceNameInListBox(self, items):
        """ sourceListBoxにデータをセットする"""
        listStore = Gtk.ListStore(str)
        for item in items:
            listStore.append([item])
        self.tree_source.set_model(listStore)

    def setSourceSetsNamesInListBox(self, cellSets, faceSets, pointSets):
        """ setsの場合のnameをセット"""
        names  = list(map(lambda x: "c:" + x, cellSets))
        names += list(map(lambda x: "f:" + x, faceSets))
        names += list(map(lambda x: "p:" + x, pointSets))
        self.setSourceNameInListBox(names)

    #
    #  setResultNameList
    def setResultNameList(self):
        """ resultNamesをlistBoxにセット"""
        if self.radio_resCellSet.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setResultNameInListBox(cellSets)
        elif self.radio_resFaceSet.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setResultNameInListBox(faceSets)
        elif self.radio_resPointSet.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setResultNameInListBox(pointSets)
        elif self.radio_resSets.get_active() == True:
            (cellSets, faceSets, pointSets) = self.getSetsNames()
            self.setResultAllNamesInListBox(cellSets, faceSets, pointSets)
        elif self.radio_resCellZoneSet.get_active() == True:
            cellZones = self.getCellZoneNames()
            self.setResultNameInListBox(cellZones)
        elif self.radio_resFaceZoneSet.get_active() == True:
            faceZones = self.getFaceZoneNames()
            self.setResultNameInListBox(faceZones)
        elif self.radio_resZones.get_active() == True:
            cellZones = self.getCellZoneNames()
            faceZones = self.getFaceZoneNames()
            pointZones = self.getPointZoneNames()
            self.setResultAllNamesInListBox(cellZones, faceZones, pointZones)
        else:
            self.setResultNameInListBox([])

    def setResultNameInListBox(self, items):
        """ resultListBoxにデータをセットする"""
        listStore = Gtk.ListStore(str)
        for item in items:
            listStore.append([item])
        self.tree_result.set_model(listStore)

    def setResultAllNamesInListBox(self, cells, faces, points):
        """ setsの場合のnameをセット"""
        names  = list(map(lambda x: "c:" + x, cells))
        names += list(map(lambda x: "f:" + x, faces))
        names += list(map(lambda x: "p:" + x, points))
        self.setResultNameInListBox(names)

    def getSetsNames(self):
        """ setsの内容を返す。"""
        relMeshDir = self.currMeshDir[len(self.caseDir):]
        print(_("setsを ") + relMeshDir + _(" から取得中..."))
        cellSets = []
        faceSets = []
        pointSets = []
        wkDir = self.currMeshDir + "/sets"
        files = pyTreeFoam.getFileNames(wkDir)
        if len(files) > 0:
            for fileName in files:
                foamFile = pyTreeFoam.foamFile()
                cont = foamFile.read(wkDir + "/" + fileName)
                className = foamFile.getClassInFoamFile(cont)
                className = className.decode()
                if className == "cellSet":
                    cellSets.append(fileName)
                elif className == "faceSet":
                    faceSets.append(fileName)
                elif className == "pointSet":
                    pointSets.append(fileName)
            cellSets.sort()
            faceSets.sort()
            pointSets.sort()
        else:
            print(_(u"「") + wkDir + _(u"」にファイルがありません。"))
        return (cellSets, faceSets, pointSets)

    def getCellZoneNames(self):
        """ polyMesh/cellZoneからcellZoneNameを取得する。"""
        print("cellZoneを " + self.currMeshDir[len(self.caseDir):] + _(" から取得中..."))
        cellZones = pyTreeFoam.getFoamContents(self.caseDir).cellZoneNames(self.currMeshDir)
        cellZones.sort()
        return cellZones

    def getFaceZoneNames(self):
        """ polyMesh/faceZoneからfaceZoneNameを取得する。"""
        print("faceZoneを " + self.currMeshDir[len(self.caseDir):] + _(" から取得中..."))
        faceZones = pyTreeFoam.getFoamContents(self.caseDir).faceZoneNames(self.currMeshDir)
        faceZones.sort()
        return faceZones

    def getPointZoneNames(self):
        """ polyMesh/pointZoneからpointZoneNameを取得する。"""
        print("pointZoneを " + self.currMeshDir[len(self.caseDir):] + _(" から取得中..."))
        pointZones = pyTreeFoam.getFoamContents(self.caseDir).pointZoneNames(self.currMeshDir)
        pointZones.sort()
        return pointZones

    def getStlFileNames(self):
        """ stlFile取得する。"""
        print(_("stlファイルを取得中..."))
        stlDir = self.stlDir
        if stlDir[0] != "/":
            stlDir = self.caseDir + "/" + stlDir
        wkDir = stlDir + "/*.stl"
        files = glob.glob(wkDir)
        if len(files) == 0:
            return []
        fileNames = list(map(lambda x: x.split("/")[-1], files))
        fileNames.sort()
        return fileNames

    def getRegionNames(self):
        """ regionを取得"""
        print("regionを取得中...")
        (flag, location, regions) = pyTreeFoam.case(self.caseDir).isMultiRegion()
        return regions

    def getFieldNames(self):
        """ field名を取得する"""
        wkDir = self.currTime + "/" + self.selRegion
        print(_("field名を ") + wkDir + _(" から取得中..."))
        fields = pyTreeFoam.case(self.caseDir).getFieldNames(self.currTime, self.selRegion)
        fields.sort()
        return fields

    def getPatchNames(self):
        """ patch名を取得する"""
        wkDir = self.currMeshDir[len(self.caseDir):]
        print(_("patchを " + wkDir + "/boundary から取得中..."))
        patchConts = pyTreeFoam.case(self.caseDir).getPatchNameFromBoundary(self.currMeshDir)
        patches = list(map(lambda x: x[0], patchConts))
        patches.sort()
        return patches


    #---- universalDialogs ---------------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()


#
#  getOption
#------------
def getOption():
    """ 起動時のoptionを取得する"""
    stlDir = ""
    timeFolder = ""
    region = ""
    i = 2
    while i < len(sys.argv):
        if sys.argv[i]== "--stl":
            i += 1
            stlDir = sys.argv[i]
        elif sys.argv[i] == "--time":
            i += 1
            timeFolder = sys.argv[i]
        elif sys.argv[i] == "--region":
            i += 1
            region = sys.argv[i]
        i += 1
    return (stlDir, timeFolder, region)            


if __name__ == "__main__":
    import gettext
    import locale
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    locale.bindtextdomain("treefoam", localeDir)
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    (stlDir, timeFolder, region) = getOption()

    #stlDir = sys.argv[2]
    winApp = windowApp(caseDir, stlDir, timeFolder, region)
    winApp.main()

