#!/usr/bin/python3
# coding: utf-8
""" TreeFoam用のモジュール。"""
#
#   pyTreeFoam.py
#
#   11/04/18    新規作成
#   12/01/01    replaceBoundaryFieldを修正（ｺﾒﾝﾄ文に対応）
#      01/04    deletePatch（パッチ削除）を追加
#      01/08    changePatchName（パッチ名の変更）を追加
#      01/15    delpatchを修正（最初のpatch削除ができなかった）
#      04/04    replacePatchDataを修正（patchデータの最初の行が"\n"の場合削除）
#      07/20    changePatchNameを修正（最初のpatchNameの場合、変更できなかった）
#      09/12    readConfigTreeFoamを追加
#      09/17    compressSPCRを修正（listが1行の場合ｴﾗｰが発生）
#      09/22    gridEditorの非表示設定関係のmodule追加
#      10/29    FFlistの取得を追加
#      11/07    空patch追加のルーチン（addPtchAtBoundaryField）を追加
#   13/01/11    changePatchNameを修正（patch名がpatchTypeと同じ名前の場合ｴﾗｰ）
#      03/04    makeDefineDataを修正（patch内容にpatch名が含まれるとｴﾗｰ）
#      03/16    regionPropertiesの内容取得を追加
#      04/14    checkMultiRegionのバグ修正
#      04/18    makeNameListFile:cellZoneNamesFの高速検索に変更
#      04/19    addPatchAtBoundaryField:バグ取り（fieldが無い時、ｴﾗｰ発生）
#      05/20    getZonesInRegionProperties:OF-2.2.0に対応
#      06/10    makeDefineData:nonuniform形式を非表示にしたので、元々あれば追加。
#               isMatchName:patternチェックを追加
#      06/26    changePatchNameInBoundaryFieldのﾊﾞｸﾞ修正（";"文字たまに欠落）
#      07/11    checkMultiRegion:regionをcurrTimeFolderから探す様に修正
#      09/07    fieldの圧縮ファイルに対応する為、fieldの読込を修正
#      09/18    checkMultiRegionの検索方法を変更
#      09/19    delPatchBoundaryFieldのバグ（削除後、\nが1ヶ無くなる）
#      10/04    getSaveDisplayFieldsFileNameのバグ（設定したdirがない場合あり）
#      10/15    国際化のため修正
#      11/15    modelフォルダ場所が変更できるようにtextBox追加
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#      12/28    openFolderNautilus:windowをwmctrlで制御
#   14/06/04    deleteNullPatchesを追加
#      07/10    checkMultiRegion:regNamesをsortして戻す様に修正
#      08/23    IsOldPatchNameInBoundary:ｴﾗｰﾒｯｾｰｼﾞを修正
#      09/22    addPatchAtBoundaryField:バグ修正（「\n」を追加）
#      10/01    isMatchName:reﾓｼﾞｭｰﾙでmatchするかどうかを確認
#      10/23    replacePatchData:patchContがnonuniform形式の場合、ｴﾗｰ発生を修正
#      11/07    readCsvData:追加
#   15/02/04    selectNeedFiles:追加
#      02/14    selectNeedFolders:追加
#      07/13    convAbsDir:環境変数が使えるように修正。tempFileの場所を変更
#   16/03/03    readSshfsDataを追加
#      05/26    readSshfsDataを修正。（コメント文の削除を追加）
#      08/28    getTreeFoamVersionを追加
#      10/31    getTreeFoamServerResult:追加
#      12/31    getContentsInRegionProperties:OF-ext用に修正
#   17/03/06    selectNeedFolders:不要なfolder名に「postPro*」を追加
#      03/20    selectNeedFiles:unneedfilesを追加
#      04/25    sendSshCommand:追加
#      05/20    readSshfsData:sshHeaderの読み込み追加
#      05/28    readSshfsData:sshHeaderの最後に「" "」を追加
#               getTreeFoamServerResult, runLoginTerm, sshfsCommand, sendSshCommand,
#               scpCommand, scpCommandNoTerm:実行時に「" "」を削除
#      06/13    getTreeFoamServerResult, scpCommand:ssh, scpコマンド実行時に
#               errorCheckを廃止
#      06/17    readSshfsData:sshfs_dataファイルからuserName、login項目を削除。
#               これら項目が未入力の場合、内容を作り出して返す。
#      06/30    複数のserverがマウントできる様に修正。
#               sshfs_dataファイルの書式変更（UserName, loginを削除）
#      07/01    getTreeFoamServerResult:treefoamServerの結果をsubprocessで直接取得
#               runShellCommandReturnCont:追加。（subprocessで実行結果を直接取得）
#      07/14    openFolderNautilus:folder名を「'」で囲みnautilusを実行する様に修正。
#      07/25    runShellCommandReturnSeparate:追加（標準出力、error出力を分けて取得）
#      08/02    readSshConfig:.ssh/congigが存在しない場合、空白を返す。
#      08/04    getTreeDirUntilCurrCase:unicodeエラー発生のため、修正。
#      09/08    selectNeedFiles:run,Allrun等はneedに変更
#      09/09    runCommandErrorCheck:標準出力をそのまま出力に変更（1>logfileを廃止）
#               runOFUtility,runOFUtilityReturnStat:commをfileに保存し、
#               そのfileを実行する様に修正
#      09/10    getContentsInRegionProperties:OFversion毎の処理を修正
#   18/06/12    gnome-terminalのオプション「-e」を「--」に変更
#      06/13    getFoldersOnlyCurrDir:folder取得方法を修正
#   19/09/11    python3, GTK+3へ置き換え、大幅修正
#   20/02/10    getBoundaryContFromPatchType:patchTypeが適切に設定されていなかった
#               ので、修正。（emptyがzeroGradientに設定されていた）
#      04/22    多言語化対応
#      05/29    separateBinaryContents:OF-7.0からboundaryの内容に「List<word>」が追加
#               されたので、binaryモードでは、読めなくなったので修正。
#      07/17    getControlDict:data取得方法を修正（バグ修正）
#               setControlDict:data保存方法を修正（バグ修正）
#      07/25    getNumberResult:並列処理のmaxResNumberが取得できていない為、修正（バグ）
#               文字の最大値を取得→数値の最大値に修正
#      11/15    separateAsciiContents:バグ修正
#               OF-v2006では、「List<scalar> 0()」が出力されるので、対応した。
#      11/20    reconstFuseDataAtMaxLinesField, separateAsciiContents, separateBinaryContents:
#               OF-v2006では、「List<scalar> 0()」と「List<scalar> 0」が出力されるので
#               関係する3ヶの関数を修正（データ数を確認し、データ数「0」は処理しない）
#   21/05/11    getHddVals:HDD容量取得を追加
#      05/30    defaultSurfaceFeaturesDict:OFversion 8 へ対応して追加
#      07/28    stl読込関係、有効数字3桁取得関数追加。
#      08/04    delFolderPatternsIniCaseに「"VTK"」を追加。
#      08/27    reconstFuseDataAtMaxLinesField:patchDataが1行で終わるものが存在する時
#               「...」のpatchDataが元に戻らない為、修正。
#      09/23    getValue3digit_shrink:有効数字3桁取得（末尾の「0」の削除）を追加
#      10/31    commandWithLog:実行方法を修正。
#      12/14    readWindowSize, writeWindowSize:meshViewerのwinSize保存を追加。
#   22/01/07    writeConfigTreeFoam:configTreeFoamファイルに必要な項目が存在しなかった
#               場合、書き込まれなかったので修正。
#      01/13    getFoldersBetweenDir:取得漏れが生じた為、currDirの取得方法を修正。    
#      01/23    getCurrMeshDir:maxTimeが数字folderでない場合は、constant/polyMeshを返す
#      02/26    isCaseDir:追加
#      03/27    getMountedServers:読み込みエラー発生の為、encoding="utf-8"を追加
#      07/17    getOFversionFolder:数値に変換してOFversionチェック(OF-10対応)
#               defaultDictPathInDataFolder:getOFversionFolderで取得
#      07/18    getNumVersion:OFversionをfloatの数値に変換するを追加
#      07/19    getZonesInRegionProperties,makeRegionProperties:OFversionのcheck方法修正（OF-10対応）
#      08/14    command,commandReturnCont:subprocess.Popen内に実行するシェルを指定する為、
#               「executable="bash"」を追加。
#      08/28    getNumVersion:OF-5.xの様なversionを判読させる。
#      11/26    commandWithLog:error出力を追加(「2>&1」を追加)
#   23/01/17    separateBinaryContents:nonuniformListの場合、data有無に関わらずbinaryDataをセット
#               dataが存在しないnonuniformListでemptyが途中にあった場合、エラー発生の為。
#               separateAsciiContents:同上
#               asciiの場合はエラー発生していないが、binaryと同様に修正。
#               readSeparate:surfaceVectorFieldなどを追加
#               reconstFuseDataAtMaxLinesAB:funcDictに変更、surfaceVectorなど追加
#      03/21    readConfigTreeFoam:fistr1, easyistrの項目を追加
#      04/07    findDataListPosition:追加。検索文字列を'List<'→'nonuniform List<'に変更。
#               codeを記述する時'constant List<scalar>&'の文字列がある為。
#      05/05    paraCase:paraCase classを追加
#      06/05    skipNextCR:次の行頭までskipを追加
#   24/01/16    getSolverBC:folderContのsolver名取得方法を変更（OF-11対応）
#      01/19    findDataListPosition:regIOobejectの読み込みエラー発生のため修正
#               「List<bool>75{true};」が読み込めない。
#      01/22    findDataListPosition:asciiのvolScalarField（alpha.water）が読めなくなった
#               為、修正（バグ）
#      01/25    findDataListPosition:「List<word>」の場合、該当せず、次を検索する様に修正
#               バグ修正
#      01/26    skipBackCR:新規追加。（前の行頭まで戻る）
#      02/05    getControlDict:controlDict内の「include」「includeIfPresent」の読込追加。
#               getZonesInRegionProperties:OF-11用を追加。
#               makeRegionProperties:OF-11用を追加
#      02/07    getIncludeCont:includeのnestingを修正
#               getAbsPath:追加（pyTreeFoam.py側に移動）
#      02/08    setControlDict:setDictの内容を削除、修正、追加を可能にした。
#      02/13    getZonesInRegionProperties:OF-11の場合、solver名で流体or固体を判断
#      07/11    makeFoamHeaderVersion:"\*..." を "\\*..." に変更。
#      07/17    readSeparate:foamFileのclass名が変わった為、funcDictのclass名を追加。
#               Zonesのclass名:regIOobject→cellZoneList,faceZoneList,pointZoneListに変更
#      07/19    getNextLinePosition:regionSolversの内容が{}の時、エラー発生。バグ修正
#      08/04    makeRegionProperties:バグ修正。
#      08/31    isMultiRegion:timeFolder内のregion取得アルゴリズム修正。（バグ修正）
#      10/04    float2strAuto:新規追加。EasyISTR側で作成したものを流用。
#      10/15    getAbsDir:新規追加。相対pathを絶対pathに変換して返す。
#      11/10    chengeEmptyToZeroGradient:emptyをzeroGradientに修正するを追加
#      11/26    getCreateMeshData:createMesh_featureAngle,addLayer_featureAngleを追加
#      12/08    replace1lineKeyword:comment文中も検索していた為、commentが置き換わって
#               いた事があった。バグ修正。
#   25/02/18    readConfigTreeFoam,writeConfigTreeFoam_runApp:plotWatcherの起動を追加。
#      07/02    getCreateMeshData:featureAngleのdefault値を90→80に修正。
#      07/08    getAllRegions:新規追加。「.」を含む全region名の取得
#      07/11    setControlDict:項目追加方法を修正（バグ修正）
#      07/14    setControlDict:項目追加方法を修正（バグ再修正）
#               expandIncludeFile:includeIfPresentのFileが無い場合は、includeIfPresentの
#               fileを元に戻す事を追加。
#      07/21    getControlDict:applicationをチェック修正するを追加。
#               getZonesInRegionProperties:OF-11以上の場合の区分け方法を修正。
#               setControlDict:全region削除時、削除範囲を1文字追加（バグ修正。）
#      07/26    getControlDict:OF-v系でtutorialsが正しく取得できず修正。
#               「itemDict["regionSolvers"]={}」が認識できなかった。
#


import csv
import glob
import gzip
import os
import re
import struct
import subprocess
import shutil
import time
import threading
import decimal
import math
import select
import sys

#import gi
#gi.require_version("Gtk", "3.0")
from gi.repository import Gio  # ゴミ箱で使用

try:
    import stringOp
except:
    import python.stringOp as stringOp

#
#appのpathの定義
#  pyTreeFoamのimport時にconfigTreeFoamの設定内容に置き換える
#   （$TreeFoamUserPathが設定されていない場合はそのまま）
#  readConfigTreeFoam時に書き換える
fileManager = "nautilus"
Terminal = "gnome-terminal"
TerminalRun = "gnome-terminal -- "
foamTerminal = "gnome-terminal --geometry=110x24 -- bash --rcfile"
foamTerminalRun = "gnome-terminal --geometry=110x24 -- "
Editor = "gedit --standalone"
Office = "loffice"
fistr1 = "fistr1"
easyistr = "easyistr"
PlotWatcher = "$TreeFoamUserPath/app/runPlotWatcher-venv"

#iniCase時に削除するfolder、file名のpattern
delFolderPatternsIniCase = ["processor.*", ".*\\.log.*", "log\\..*",
            "postPro*", "VTK"]
delFilePatternsIniCase = ["log\\..*|.*\\.log", ".*~",
            ".*\\.bak|.*\\.png|.*\\.obj|.*\\.jpg|.*\\.OpenFOAM|.*\\.foam|.*\\.blockMesh",
            "test.*|.*test",
            #"plotWatcher", "README.*",
            "plotWatcher",
            #".*\\.ods|.*\\.odt|.\\*.odp|.*\\.csv",
            ".*\\.ods|.*\\.odt|.\\*.odp",
            "job*"]

#  ----------- csvデータの読み込み ---------
#
#  readCsvData
#
def readCsvData(fileName):
    """ csvファイルを読み込み、list形式で返す"""
    listData = []
    with open(fileName) as file:
        conts = csv.reader(file)
        for row in conts:
            listData.append(row)
    return listData

#  ---------- stlファイルの読込 ------------
def isBinaryStl(fileName):
    """ binaryファイルかどうかチェック。header(80bytes)を除いた9文字をチェック
    
    Args:
        fileName (str) :stlFile
    Return:
        bool    :True or False"""
    f = open(fileName, "rb"); cont = f.read(100); f.close()
    if len(cont) < 100:
        return False
    ans = False
    for i in range(81, 90):
        if ((ord(" ") <= cont[i] and (cont[i] <= ord("z")) or
            cont[i] == ord("\n") or cont[i] == ord("\r") or cont[i] == ord("\t"))):
            pass
        else:
            ans = True
            break
    return ans

def getStlContFromAscii(fileName):
    """ asciiのstlファイルからsolidNameとxyz座標を取得
    
    Args:
        stlFile (str)   :stlファイル
    Return:
        solidName (str) :solid名
        xyzVals (list)  :[xVlas, yVals, zVals]"""
    f=open(fileName); lines=f.readlines(); f.close()
    #solidName取得
    names = []
    for line in lines:
        if line[:5] == "solid":
            name = " ".join(line.split()[1:])
            names.append(name)
    solidName = ",".join(names)
    #stlSize取得
    xVals=[]; yVals=[]; zVals=[]
    for line in lines:
        if line.find("vertex") >= 0:
            values = line.split()
            xVals.append(float(values[1]))
            yVals.append(float(values[2]))
            zVals.append(float(values[3]))
    xyzVals = [xVals, yVals, zVals]
    return (solidName, xyzVals)

def getStlContFromBin(fileName):
    """ binaryのstlファイルからsolidNameとxyz座標を取得
    
    Args:
        stlFile (str)   :stlファイル
    Return:
        solidName (str) :solid名
        xyzVals (list)  :[xVlas, yVals, zVals]"""
    f = open(fileName, "rb"); cont = f.read(); f.close()
    #header部を取得
    i = 80
    header = cont[:i]
    #solidNameを取得
    solidName = ""
    for chara in header:
        #binaryの場合、数値でで大小を比較する
        if ord(" ") <= chara and chara <= ord("z"):
            solidName += chr(chara)
        else:
            break
    #三角形の数
    num = struct.unpack("1i", cont[i:i+4])
    _nTet = num[0]
    i += 4
    #座標を取得
    xVals = []; yVals = []; zVals = []
    while i < len(cont):
        num = struct.unpack("12f", cont[i:i+48])
        #単位ベクトルはskipする
        xVals += [num[3]] + [num[6]] + [num[9]]
        yVals += [num[4]] + [num[7]] + [num[10]]
        zVals += [num[5]] + [num[8]] + [num[11]]
        #最後の2byte分も加えてskip
        i += 48 + 2
    xyzVals = [xVals, yVals, zVals]
    return (solidName, xyzVals)

# ----------- folder取得関連----------------
#
#　getFoldersOnlyCurrDir
#
def getFoldersOnlyCurrDir(currDir, maskDirs=[]):
    """ currDir内のfolderDir（fullPath）を取得する。
    ただし、maskDirsに含まれるDirは取得しない。"""
    if currDir in maskDirs:
        return []
    items = glob.glob(currDir+"/*")
    folders = [f for f in items if os.path.isdir(f)]
    #maskDisに含まれるDirを削除する
    #folders = list(filter(lambda x: not x in maskDirs, folders))
    return folders
    
def getFolders(currDir, maskDirs=[]):
    """ currDir内のfolderDir（fullPath）を取得する。
    ただし、maskDisに含まれるDirは取得しない。"""
    return getFoldersOnlyCurrDir(currDir, maskDirs)

def getFiles(currDir):
    """ currDir内のfile（fullPath）を取得する"""
    items = glob.glob(currDir+"/*")
    files = [f for f in items if os.path.isfile(f)]
    return files

def getFolderNames(currDir):
    """ currDir内のfolder名（baseName）を取得する"""
    folNames = [f for f in os.listdir(currDir) if os.path.isdir(currDir+"/"+f)]
    return folNames

def getFileNames(currDir):
    """ currDir内のfile名（baseName）を取得する"""
    items = glob.glob(currDir+"/*")
    fileDirs = [f for f in items if os.path.isfile(f)]
    fileNames = list(map(lambda x: x.split(os.sep)[-1], fileDirs))
    return fileNames

def isNeedFolder(path):
    """ pathが必要なfolderかどうかチェックする。
    TreeViewに表示させるfolderかどうかチェック。"""
    name = os.path.basename(path)
    flag = False
    if not (name == "system" or name == "constant"
            or name == "solve.log.analyzed" or name[0] == "."
            or name[:len("processor")] == "processor"):
        try:
            float(name)
        except:
            flag = True
    return flag

def isCaseDir(path):
    """ OpenFOAMのcaseDirかどうかを確認する"""
    if os.path.exists(path + "/constant") == False:
        if os.path.exists(path + "/system") == False:
            return False
    return True

def getNeedFolders(currDir, maskDirs=[]):
    """ currDir直下の必要なfolderのみ取得する。
    ただし、maskDirsに含まれるdirは取得しない。"""
    folders = getFoldersOnlyCurrDir(currDir, maskDirs)
    folders = list(filter(isNeedFolder, folders))
    return folders

def getFoldersBetweenDir(startDir, endDir, maskDirs=[]):
    """ startDirからendDirまでの必要なfolderを取得する。
    ただし、maskDirsに含まれるdirは取得しない。"""
    folderDir = []
    folders = getNeedFolders(startDir, maskDirs)
    folderDir = folders[:]
    if startDir == endDir:
        return folderDir
    currDir = ""
    for folder in folderDir:
        n = len(folder)
        if folder == endDir[:n]:
            if endDir[:n+1][-1] == "/" or folder == endDir:
                currDir = folder
                break
    folderDir += getNeedFolders(currDir, maskDirs)
    names = endDir[len(currDir)+1:].split("/")
    for name in names:
        currDir += "/" + name
        folders = getNeedFolders(currDir, maskDirs)
        folderDir += folders[:]
    folderDir = list(set(folderDir))
    return folderDir

def getChildFolders(folders, maskDirs=[]):
    """ folders(list)のchildFoldersを取得する。
    ただし、maskDirsに含まれるDirは取得しない。"""
    subFolders = []
    for folder in folders:
        subFolders += getNeedFolders(folder, maskDirs)
    return subFolders

def getFolderType(currDir):
    """ 対象のfolderを 'case(Link)', 'salome(Link)', 'folder(Link)'の区分に分ける。
    
    Args:
        None
    Returns:
        (str)   :"case", "caseLink", "salome", "salomeLink, "folder", "folderLink"
    """
    ans = "folder"
    if os.path.exists(currDir+"/system/controlDict"):
        ans = "case"
    elif len(glob.glob(currDir+"/*hdf")) != 0:
        ans = "salome"
    if os.path.islink(currDir):
        ans += "Link"
    return ans

def getHddVals(currDir):
    """ HDDの容量を取得して返す"""
    if os.path.exists(currDir) == False:
        currDir = os.getenv("HOME")
    else:
        if os.path.isfile(currDir) == True:
            currDir = os.path.dirname(currDir)
    disk = shutil.disk_usage(currDir)
    total = disk.total
    free = disk.free
    return (total, free)

#  --------ゴミ箱へ移動 ------------------
def moveToTrash(pathes):
    """ pathes（folder, folder）をゴミ箱へ移動する
    
    Args:
        pathes (list(str))  :ゴミ箱へ移動するpath名のlist
    Returns:
        unPathes (list(str))  :ゴミ箱へ移動できなかったpath名のlist"""
    unPathes = []
    for path in pathes:
        try:
            obj = Gio.File.new_for_path(path)
            res = obj.trash()
            if res == False:
                unPathes.append(path)
        except:
            unPathes.append(path)
    return unPathes

#  ---------- 数字の丸め -----------------
def getValue3digit(x):
    """ 有効数字3桁の文字列を取得する。
    有効数字4桁目を四捨五入して3桁に丸める。"""
    xx = float2strAuto(x, prec=3)
    return xx

def getValue3digit_shrink(x):
    """ 有効数字3桁の文字列を取得する。末尾の「0」を削除"""
    xx = float2strAuto(x, prec=3, showPoint=False)
    return xx

#
#  float2strAuto
def float2strAuto(num, prec=8, minNum=0.00001, maxNum=100000.0,
                  zeroNum=1e-20, showPoint=True, expStr="e"):
    """ floatを指定の有効桁数(default=8)で文字変換して返す。
    変換できない場合は、「""」を返す。
    minNum < abs(num) < maxNum: 通常の小数点表記の文字列を返す。
    zeroNum < abs(num): 「"0.0"」を返す。
    以外: 指数表記の文字列を返す。
    
    Args:
        num(float, str) :変換する数値
        prec(int)       :有効桁数(default=8)
        minNum(float)   :小数点表記の下限値(default=0.00001)
        maxNum(float)   :小数点表記の上限値(default=100000)
        zeroNum(float)  :zeroNum以下をzeroに設定(default=1e-20)
        showPoint(bool) :小数点の表示、「0.0」の場合「0」を表示(default=True)
        expStr(str)     :指数を「E」or「e」(default="e")"""
    
    def deletePointZero(numStr):
        """ 不要な ".0" を削除する"""
        n = numStr.find(".")
        if n < 0:
            return numStr
        else:
            newNum = "0"
            for i in range(len(numStr)):
                ii = len(numStr) - i - 1
                if numStr[ii] != "0":
                    if numStr[ii] == ".":
                        newNum = numStr[:ii]
                    else:
                        newNum = numStr[:ii+1]
                    break
            return newNum
            
    def deleteZero(numStr):
        """ 不要な "0" を削除する"""
        newHead = "0.0"
        n = len(numStr)
        for i in range(n):
            ii = n - i - 1
            if numStr[ii] != "0":
                newHead = numStr[:ii+1]
                if numStr[ii] == ".":
                    newHead += "0"
                break
        return newHead

    def remakeFloat(numStr):
        """ 小数点有無確認し、無ければ追加"""
        if numStr.find(".") < 0:
            numStr += ".0"
        return numStr

    def changeExpToFloat(numStr):
        """ 指数表示を小数点表示に変換する"""
        #floatに戻してstr変換
        num = float(numStr)
        numStr = str(num).upper()
        if numStr.find("E") < 0:
            #小数点表記なら直ぐに戻る
            return numStr
        #指数表記の場合、桁数から小数点表記に変換する
        e10 = math.log(num, 10)
        n = prec - int(e10)
        if n < 0:
            #小数の桁数が大きい場合
            nn = -n
            fmt = "{:." + str(nn) + "f}"
        elif n == 0 or n == 1:
            #小数の桁数と有効桁数が等しい場合
            fmt = fmt = "{:.1f}"
        else:
            #有効桁数が大きい場合
            nn = n - 1
            fmt = "{:." + str(nn) + "f}"
        numStr = fmt.format(num)
        #不要なzeroを削除
        numStr = deleteZero(numStr)        
        return numStr
    
    def changeFloatToExp(numStr):
        """ 小数点表記を指数表記に変換する"""
        num = float(numStr)
        fmt = "{:." + str(prec-1) + "E}"
        numStr = fmt.format(num)
        #余分なzeroを削除
        numStr = deleteZeroExp(numStr)
        return numStr
        
    def remakeExp(numStr):
        """ 小数点有無確認し、無ければ追加"""
        n = numStr.find("E")
        head = numStr[:n]
        if head.find(".") < 0:
            head += ".0"
            numStr = head + numStr[n:]
        return numStr
        
    def deleteZeroExp(numStr):
        """ 指数表記で、小数表記部と指数部のzeroを削除"""
        n = numStr.find("E")
        #小数点表記部のzero削除
        head = numStr[:n]
        newHead = deleteZero(head)
        #指数部のzero削除
        newExp = "0"
        for i in range(n+2, len(numStr)):
            if numStr[i] != "0":
                newExp = numStr[i:]
                break
        numStr = newHead + numStr[n:n+2] + newExp
        return numStr
       
    def deletePointZeroFromNum(numStr):
        """ 小数点を非表示に設定する。"""
        ne = numStr.find("E")
        if ne < 0:
            #通常の小数点表記の場合
            numStr = deletePointZero(numStr)
        else:
            #指数表記の場合
            newHead = numStr[:ne]
            newHead = deletePointZero(newHead)
            numStr = newHead + numStr[ne:]
        return numStr

    #float以外は、floatに変換
    if type(num) != float:
        try:
            num = float(num)
        except:
            return ""
    #符号を確認
    minus = False
    if num < 0:
        minus = True
    num = abs(num)
    #有効数字で丸める
    decimal.getcontext().prec = prec
    decNum = decimal.Decimal(num)
    decNum = decimal.Decimal.normalize(decNum)
    numStr = str(decNum).upper()
    #numの値により処理
    if minNum < num and num < maxNum:
        #通常の少数点表記
        if numStr.find("E") < 0:
            #少数点有無チェック
            numStr = remakeFloat(numStr) 
        else:
            #指数表記を小数点表記に変換
            numStr = changeExpToFloat(numStr)
    elif num < zeroNum:
        #zero
        numStr = "0.0"
    else:
        #指数表記
        if numStr.find("E") < 0:
            #小数点表記を指数表記に変換
            numStr = changeFloatToExp(numStr)
        else:
            #小数点有無チェック
            numStr = remakeExp(numStr)
    #小数点表示チェック
    if showPoint == False:
        #小数点を非表示にする
        numStr = deletePointZeroFromNum(numStr)
    #指数文字を設定（大文字小文字）
    if expStr == "e":
        numStr = numStr.lower()
    #符号を設定
    if minus == True:
        numStr = "-" + numStr
    return numStr

# ----------- treeFoam関連----------------

#  raedConfigTreeFoam
#
def readConfigTreeFoam():
    """ configTreeFoamの内容を読み取り、結果を辞書形式で返す。
    appの内容をconfigTreeFoamに合わせる
    辞書keys: language, logFile, OFversion, rootDir, workDir, bashrcFOAM,
    paraFoam, plotWatcher, salomeMeca, CAD, editor, fileManager, Terminal, foamTerminal"""
    global fileManager, Terminal, Editor, foamTerminal, Office, PlotWatcher
    global fistr1, easyistr
    configDict = {
        "language": "",
        "logFile": "",
        "OFversion": "",
        "rootDir": "",
        "workDir": "",
        "bashrcFOAM": "",
        "paraFoam": "",
        "plotWatcher": "",
        "salomeMeca": "",
        "CAD": "",
        "editor": "",
        "fileManager": "",
        "Terminal": "",
        "TerminalRun": "",
        "foamTerminal": "",
        "foamTerminalRun": "",
        "office": "",
        "fistr1": "",
        "easyistr": ""
        }
    fileName = os.getenv("TreeFoamUserPath") + os.sep + "configTreeFoam"
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    for line in lines:
        words = line.split()
        if len(words) > 1 and words[0][0] != "#":
            if words[0] in configDict.keys():
                configDict[words[0]] = " ".join(words[1:])
    #appの設定
    flags = [0,0,0,0,0,0,0,0,0,0]
    #  fileManagerの設定
    if configDict["fileManager"] == "":
        configDict["fileManager"] = fileManager
        flags[0] = 1
    else:
        fileManager = configDict["fileManager"]
    #  Terminalの設定
    if configDict["Terminal"] == "":
        configDict["Terminal"] = Terminal
        flags[1] = 1
    else:
        Terminal = configDict["Terminal"]
    #  editorの設定
    if configDict["editor"] == "":
        configDict["editor"] = Editor
        flags[2] = 1
    else:
        Editor = configDict["editor"]
    #  foamTerminalの設定
    if configDict["foamTerminal"] == "":
        configDict["foamTerminal"] = foamTerminal
        flags[3] = 1
    else:
        foamTerminal = configDict["foamTerminal"]
    #  officeの設定
    if configDict["office"] == "":
        configDict["office"] = Office
        flags[4] = 1
    else:
        Office = configDict["office"]
    #  TerminalRunの設定
    if configDict["TerminalRun"] == "":
        configDict["TerminalRun"] = TerminalRun
        flags[5] = 1
    #  foamTerminalRunの設定
    if configDict["foamTerminalRun"] == "":
        configDict["foamTerminalRun"] = foamTerminalRun
        flags[6] = 1
    #  fistr1の設定
    if configDict["fistr1"] == "":
        configDict["fistr1"] = fistr1
        flags[7] = 1
    #  easyistr
    if configDict["easyistr"] == "":
        configDict["easyistr"] = easyistr
        flags[8] = 1
    #  plotWatcher
    if configDict["plotWatcher"] == "":
        configDict["plotWatcher"] = PlotWatcher
        flags[9] = 1
    #  未定義appをconfigに書き込む
    if sum(flags) != 0:
        writeConfigTreeFoam_runApp(flags)
    return configDict

#  writeConfigTreeFoam
#
def writeConfigTreeFoam(configDict):
    """ 辞書形式で受け取った内容をconnfigTreeFoamに書き込む"""
    fileName = os.getenv("TreeFoamUserPath") + os.sep + "configTreeFoam"
    f = open(fileName); lines = f.readlines(); f.close()
    items = list(configDict.keys())
    for i in range(len(lines)):
        line = lines[i]
        words = line.split()
        if len(words) > 0 and words[0][0] != "#":
            if words[0] in items:
                newCont = configDict[words[0]]
                lines[i] = words[0] + " " + newCont + "\n"
                #合致分を削除
                idx = items.index(words[0])
                a = items.pop(idx)
    #合致しなかった分を追記
    for item in items:
        line = item + " " + configDict[item] + "\n"
        lines += [line]
    f = open(fileName, "w"); f.writelines(lines); f.close()

#  writeConfigTreeFoam_runApp
#
def writeConfigTreeFoam_runApp(flags):
    """ 実行コマンドのdefaultをconfigTreeFoamに書き込む"""
    cont = ""
    if flags[0] == 1:
        cont += "#fileManagerの実行コマンド\n"
        cont += "fileManager " + fileManager + "\n\n"
    if flags[1] == 1:
        cont += "#terminalの実行コマンド\n"
        cont += "Terminal " + Terminal + "\n\n"
    if flags[2] == 1:
        cont += "#editorの実行コマンド\n"
        cont += "#  editorがcloseするまで、待つ設定にする。裏で起動しない。\n"
        cont += "editor " + Editor + "\n\n"
    if flags[3] == 1:
        cont += "#FOAM端末の実行コマンド\n"
        cont += "#  行末に環境設定fileを付加して実行する\n"
        cont += "foamTerminal " + foamTerminal + "\n\n"
    if flags[4] == 1:
        cont += "#officeの実行コマンド\n"
        cont += "#  csvファイル編集用\n"
        cont += "#  「<officeコマンド> test.csv」で編集する\n"
        cont += "office " + Office + "\n\n"
    if flags[5] == 1:
        cont += "#terminalの実行コマンド\n"
        cont += "#  行末に実行fileを付加して、terminal上で実行し、実行後はterminalを閉じる。\n"
        cont += "#  「<terminalコマンド> <実行file>」でterminal上で実行する\n"
        cont += "TerminalRun " + TerminalRun + "\n\n"
    if flags[6] == 1:
        cont += "#FOAM端末の実行コマンド\n"
        cont += "#  行末に実行fileを付加して、terminal上で実行し、実行後はFOAM端末を閉じる。\n"
        cont += "#  「<foamTerminalコマンド> <実行file>」でterminal上で実行する\n"
        cont += "#  FOAM端末の環境設定は、実行file内で設定しておく。\n"
        cont += "foamTerminalRun " + foamTerminalRun + "\n\n"
    if flags[7] == 1:
        cont += "#FrontISTRの実行コマンド\n"
        cont += "fistr1 " + fistr1 + "\n\n"
    if flags[8] == 1:
        cont += "#EasyISTRの実行コマンド\n"
        cont += "easyistr " + easyistr + "\n\n"
    if flags[9] == 1:
        cont += "#plotWatcherの実行コマンド\n"
        cont += "#  pythonの仮想環境で起動する場合は、「runPlotWatcher-venv」を選択\n"
        cont += "plotWatcher " + PlotWatcher + "\n\n"
    fileName = fileName = os.getenv("TreeFoamUserPath") + os.sep + "configTreeFoam"
    f = open(fileName, "a"); f.write(cont); f.close()

#  readWindowSize
#
def readWindowSize():
    """ windowサイズをfileから辞書形式で読み込む"""
    winDict = {
        "treeFoam": "",
        "treeFoam_colWidth": "",
        "treeFoam_splitter": "",
        "gridEditor": "",
        "createSetFields": "",
        "editStlFiles": "",
        "editStlFiles_colWidth": "",
        "editStlFiles_splitter": "",
        "meshViewer": "",
        "meshViewer_splitter": ""
        }
    fileName = os.getenv("TreeFoamUserPath") + os.sep + "data" + os.sep + "winSize"
    f = open(fileName); lines = f.readlines(); f.close()
    for line in lines:
        words = line.split()
        if len(words) >= 2:
            if words[0][0] != "#":
                winDict[words[0]] = " ".join(words[1:])
    return winDict

#  writeWindowSize
#
def writeWindowSize(winDict):
    """ windowサイスを書き込む"""
    fileName = os.getenv("TreeFoamUserPath") + os.sep + "data" + os.sep + "winSize"    
    f = open(fileName, "r"); lines = f.readlines(); f.close()
    items = list(winDict.keys())
    for i in range(len(lines)):
        words = lines[i].split()
        if len(words) > 1:
            if words[0] in items:
                newCont = winDict[words[0]]
                lines[i] = words[0] + " " + newCont + "\n"
                idx = items.index(words[0])
                items.pop(idx)
    if len(items) > 0:
        for item in items:
            line = item + " " + winDict[item] + "\n"
            lines.append(line)
    f = open(fileName, "w"); f.writelines(lines); f.close()

#  getNumVersion
#
def getNumVersion(OFversion):
    """ versionを数値(float)に変換する。
    数値変換できない場合は、「-1.0」を返す"""
    numStr = ".".join(OFversion.split(".")[:2])
    if numStr[-1].lower() == "x":
        numStr = numStr[:-1]
    try:
        numVer = float(numStr)
    except:
        numVer = -1.0
    return numVer

#  getIntVersion
#
def getIntVersion(OFversion):
    """ versionを整数(int)に変換する。
    数値変換できない場合は、「-1」を返す"""
    intStr = OFversion.split(".")[0]
    try:
        intVer = int(intStr)
    except:
        intVer = -1
    return intVer

#  ------ pattern match ------------------

def isPatternMatch(patterns, name):
    """ nameがpattern(list)にmatchするかどうか
    
    Args:
        patterns (list(pattern))    :patternのlist
        name (str)                  :対象の文字列
    Returns:
        flag (bool)                 :True(match) or False(no match)"""
    flag = False
    for pattern in patterns:
        if re.match(pattern, name) != None:
            flag = True
            break
    return flag

#
#  getAbsDir
#------------
def getAbsDir(currDir, path):
    """ 相対pathを絶対pathに変換する。"""
    if path[0] == os.sep:
        return path
    #絶対pathに変換
    path = currDir + os.sep + path
    path = os.path.abspath(path)
    return path

#
#  ---------
#  run class
#  ---------
#
class run:
    """ app, commandの実行関連"""

    def __init__(self, caseDir=os.getcwd()):
        self.caseDir = caseDir

    def fileManager(self):
        """ fileManagerでcaseDirを開く"""
        comm = fileManager + " " + self.caseDir + " &"
        self.command(comm)

    def editControlDict(self):
        """ editorでsystem内のcontrolDict、fvSchemes、fvSolutionを開く"""
        fileNames = []
        fileNames.append(os.sep.join([self.caseDir, "system", "controlDict"]))
        fileNames.append(os.sep.join([self.caseDir, "system", "fvSchemes"]))
        fileNames.append(os.sep.join([self.caseDir, "system", "fvSolution"]))
        self.editor(fileNames)

    def editor(self, files):
        """ editorでfilesをopenする。裏で動かす。"""
        comm = Editor + " " + " ".join(files) + " &"
        self.command(comm)

    def editorWait(self, files):
        """ editorでfilesをopenする。editorを閉じるまで戻らない"""
        comm = Editor + " " + " ".join(files)
        self.command(comm)

    def foamEditor(self, files):
        """ editorでfoamFileを開き編集する。
        tempFileを作成し編集する。timeStampを確認し、更新した場合は、元のfileを書き換える。
        対象のfoamFileは、ascii, binary, compressでも編集できる。"""
        th = threading.Thread(target=self.foamEditorWait, args=(files,))
        th.start()

    def foamEditorWait(self, files):
        #tempFolderを作成する
        tempFolder = "temp" + str(time.time())
        fileDir = os.getenv("TreeFoamUserPath") + "/temp/" + tempFolder + "/"
        os.mkdir(fileDir)
        #編集するfileとtimeStampを取得
        tempFiles = list(map(lambda x: fileDir + os.path.basename(x), files))
        foam = foamFile()
        timeStamps = []
        for i in range(len(tempFiles)):
            cont = foam.read(files[i])
            f = open(tempFiles[i], "w"); f.write(cont); f.close()
            timeStamp = os.stat(tempFiles[i]).st_mtime
            timeStamps.append(timeStamp)
        #editorでtempFileを開く
        self.editorWait(tempFiles)
        for i in range(len(tempFiles)):
            if os.stat(tempFiles[i]).st_mtime != timeStamps[i]:
                #更新した場合は、元のfileを更新する
                cont = foam.read(tempFiles[i])
                foam.write(files[i], cont)
        #tempFolderを削除
        shutil.rmtree(fileDir)

    def terminal(self):
        """ 端末を起動する"""
        #os.chdir(self.caseDir)
        comm = Terminal
        self.command(comm)

    def foamTerminal(self, envFile):
        """ FOAM端末を起動する"""
        #os.chdir(self.caseDir)
        comm = foamTerminal + " " + envFile + " &"
        self.command(comm)

    def command(self, comm):
        """ commandを実行する。終了するまで待ち、終了後に戻る。
        終了を待たずに戻る場合は、commandの最後に「&」を追加する。"""
        os.chdir(self.caseDir)
        dummy_proc = subprocess.run(comm, shell=True, executable="bash")
        
    def commandWithLog(self, comm):
        """ commandを実行し、logFoileを更新する。
        「2>&1 | tee -a <logFile>」を追加する"""
        #os.chdir(self.caseDir)
        #最終文字が「&」かどうか確認
        lastWord = ""
        for i in range(len(comm), 0, -1):
            if comm[i-1] > " ":
                lastWord = comm[i-1]
                break
        if lastWord != "":
            if lastWord == "&":
                #最終文字が「&」の場合
                comm = comm[:i-1]
                comm = comm + " 2>&1 | tee -a " + os.getenv("LogFile") + " &"
                #_proc = subprocess.run(comm, shell=True)
                self.command(comm)
                return
        #最終文字が「&」ではない場合
        comm = comm + " 2>&1 | tee -a " + os.getenv("LogFile")
        #_proc = subprocess.run(comm, shell=True)
        self.command(comm)
        return

    def commandReturnCont(self, comm, isOut=True):
        """ commandを実行しその結果を受け取る
        
        Args:
            comm (str)  :コマンド
            isOut (bool):標準出力にも出力する（default=True)
        Returns:
            stat (str)      :"OK" or "ERROR"
            resCont (str)   :実行結果
            errCont (str) ) :エラーの内容"""
        #コマンド実行
        results = []
        os.chdir(self.caseDir)
        proc = subprocess.Popen(
            comm,
            shell=True,
            executable="bash",
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE )
        #結果をrealTimeで取得し、出力
        for line in iter(proc.stdout.readline, b""):
            resLine = line.rstrip().decode("utf-8")
            results.append(resLine)
            if isOut == True:
                print(resLine)
        #実行結果を確認
        errCont = proc.stderr.read().decode("utf-8")
        resCont = "\n".join(results) + "\n"
        if errCont == "":
            stat = "OK"
        else:
            if isOut == True:
                print(errCont)
            stat = "ERROR"
        return (stat, resCont, errCont)


#  ----------
#  case class
#  ----------
#
class case:
    """ 解析case内のfileやfolderを取得する。

    Attribute:
        caseDir (str)   :対象の解析caseのdirectory
    """
    def __init__(self, caseDir=os.getcwd()):
        self.caseDir = caseDir
        #controlDict内の読込項目
        self.itemsDict = {
            "application": "",
            "solver": "",
            #"subSolver": "",
            "regionSolvers": {},        #{<region>:<solver>}
            "startFrom": "",
            "startTime": "",
            "stopAt": "",
            "endTime": "",
            "deltaT": "",
            "timePrecision": "",
            "writeInterval": "",
            "writeFormat": "",
            "writePrecision": "",
            "writeCompression": "",
            "runTimeModifiable": "",
            "adjustTimeStep": ""
            }

    def getTimeFolders(self, processor="."):
        """ 対象の解析case内のtimeFolderを取得する。
        timeFolderは、数値順にsortした状態で返す。
        
        Args:
            processor (str) :processor名（default="."）
        Returns:
            (list)  :timeFolder名のlistを返す
        """
        wkDir = os.path.abspath(self.caseDir + os.sep + processor)
        folders = getFolderNames(wkDir)
        numFolders = []
        for folder in folders:
            try:
                a = float(folder)
                numFolders.append([a, folder])
            except:
                pass
        numFolders.sort()
        timeFolders = list(map(lambda x: x[1], numFolders))
        return timeFolders

    def getCurrTimeFolder(self):
        """ 解析case内のcurrTimeFolder(計算開始時のtimeFolder)を取得する。
            processor内のtimeFolderは、無視する"""
        timeFolder = ""
        timeFolders = self.getTimeFolders()
        contDict = self.getControlDict()
        if len(timeFolders) != 0 and contDict["startFrom"] != "" and contDict["startTime"]: 
            if contDict["startFrom"] == "firstTime":
                timeFolder = timeFolders[0]
            elif contDict["startFrom"] == "latestTime":
                timeFolder = timeFolders[-1]
            else:
                timeFolder = contDict["startTime"]
                if len(glob.glob(self.caseDir+os.sep+timeFolder)) == 0:
                    timeFolder = timeFolders[0]
        return timeFolder

    def getCurrMeshDir(self, maxTime, region, meshName):
        """case内のregionからpolyMesh内のmeshName(boundary等)を探す。
        検索方法は、maxTimeから遡って、検索する。
        region0は、"."で探す。
        戻り値は、caseDirからの相対Dirを返す。
            [例:constant/polyMesh]"""
        savePath = os.getcwd()
        os.chdir(self.caseDir)
        try:
            maxTimeNum = float(maxTime)
        except:
            meshDir = "constant/" + region + "/polyMesh"
            return meshDir
        timeFlag = 0
        meshDirs = glob.glob("*/" + region + "/polyMesh/" + meshName)
        if len(meshDirs) > 0:
            ans = []
            for meshDir in meshDirs:
                dirs = meshDir.split("/")
                try:
                    a = float(dirs[0])
                    if a <= maxTimeNum:
                        ans.append(dirs[:-1])
                except:
                    pass
            if len(ans) > 0:
                ans.reverse()
                meshDir = "/".join(ans[0])
                timeFlag = 1
        if timeFlag == 0:
            meshDir = "constant/" + region + "/polyMesh"
        os.chdir(savePath)
        return meshDir

    def getFieldNames(self, timeFolder, region="."):
        """ 指定したtimeFolder内のfield名を取得する。
        圧縮・非圧縮ファイルを識別してfield名を取得する。"""
        fieldDir = self.caseDir + os.sep + timeFolder + os.sep + region
        names = getFileNames(fieldDir)
        fields = []
        for name in names:
            name = self.getOrgFileName(name)
            if name[-1] != "~" and name[0] != ".":
                #バックアップ、隠しファイルを除く
                fields.append(name)
        #同名を削除
        fields = list(set(fields))
        return fields

    def getOrgFileName(self, fileName):
        """ 非圧縮のfileNameを返す"""
        name = fileName
        if name[-3:] == ".gz":
            name = fileName[:-3]
        return name

    def checkFileName(self, fileName):
        """ 実在するfile名に合わせて、圧縮or非圧縮のfile名に設定する
        非圧縮fileが存在する場合は、非圧縮のfile名を返す。
        両方存在する場合は、非圧縮のfile名を返す。
        両方存在しない場合は、そのままfile名を返す。
        fileNameはfullPathで指定"""
        name = fileName
        if name[-3:] == ".gz":
            name = name[:-3]
        if len(glob.glob(name)) != 0:
            return name
        name += ".gz"
        if len(glob.glob(name)) != 0:
            return name
        return fileName

    def clearNullBoundaryAllFields(self, timeFolder, meshDir, region="."):
        """ 空白patchのみのboundaryFieldのクリア
        fieldを取得してクリアする。

        Args:
            timeFolder (str)    :時間folder
            meshDir (str)       :polyMeshの親dir(fullPath)
            region (str)        :region（default:「.」
        Returns:
            None"""
        fields = self.getFieldNames(timeFolder, region)
        if len(fields) == 0:
            return
        #fieldのクリア
        for field in fields:
            print(field + ": ", end="")
            self.clearNullBoundaryFields(timeFolder, meshDir, [field], region)

    def clearNullBoundaryFields(self, timeFolder, meshDir, fields, region="."):
        """ 空白patchのみboundaryFieldをクリアする。
        (各fieldのboundaryFieldのpatch名が同じ場合に、使う。)
        boundaryとboundaryField間の整合性をとる。
        boundary中の空patchは、そのまま残る。
        クリアするfieldを指定してクリアする。
        
        Args:
            timeFolder (str)    :時間フォルダ
            meshDir (str)       :polyMeshの親dir(fullPath)
            fileds (list(str))  :field名のlist
            region (str)        :region（default:「.」
        Returns:
            None"""
        polyMeshDir = meshDir
        #patch名の取得
        patches = self.getPatchNameFromBoundary(polyMeshDir)
        if len(patches) != 0:
            Pname = []
            Ptype = []
            for name in patches:
                Pname.append(name[0])
                Ptype.append(name[1])
            #boundaryFieldの作成
            (dummy, patches, Bcont) = self.getFieldsPatchNames(timeFolder, fields, region)
            #追加patchのチェック
            setFlag = 0
            print(_("以下の追加patchを「zeroGradient, empty, symmetry」で定義します。"))
            for newPatch in Pname:
                same = "no"
                for oldPatch in patches:
                    if newPatch == oldPatch:
                        same = "yes"
                        break
                if same == "no":
                    setFlag = 1
                    print("  " + newPatch)
            if setFlag == 0:
                print(_("  追加patchはありません。"))
            else:
                [dummy, newBcont] = self.setNullBoundaryField(fields, Pname, patches,
                                        Bcont, Ptype)
                error = self.makeNullBoundaryFields(timeFolder, region, fields, Pname, newBcont)
                if error[0] != "OK":
                    print(error[1])

    def setNullBoundaryField(self, fields, newPatches, patches, Bcont, Ptype):
        newBcontents = []
        f=0
        for _field in fields:
            newBcont = []
            p = 0
            for newPatch in newPatches:
                same = 0
                i=0
                for patch in patches:
                    if newPatch == patch:
                        value = self.reformTabContents(Bcont[f][i] + "\n")
                        newBcont.append(value)
                        same = 1
                        break
                    i += 1
                if same == 0:
                    value = self.getBoundaryContFromPatchType(Ptype[p])
                    #valueを整形する。
                    value = self.reformTabContents(value + "\n")  #追加
                    newBcont.append(value)
                p += 1
            newBcontents.append(newBcont)
            f += 1
        return (fields, newBcontents)

    def makeNullBoundaryFields(self, timeFolder, region, fields, patches, Bconts):
        fieldDir = os.sep.join([self.caseDir, timeFolder, region])
        #patchNameを整形（空白追加）
        patchesSp = []
        for patch in patches:
            patchesSp.append("    " + patch)
        #各fieldのboundaryFieldを作成
        i=0
        mess = ""
        while i<len(fields):
            #fieldをオープン
            fileName = fieldDir + "/" + fields[i]
            contents = foamFile().read(fileName)
            #patchDataから「boundaryField」を作成
            newBoundary = self.makeBoundaryField(patchesSp, Bconts[i])
            #「boundaryField」を置き換える
            newContents = self.replaceBoundaryField(newBoundary, contents)
            if newContents != "":
                foamFile().write(fileName, newContents)
            else:
                print("error:", fields[i], _("が置き換えできませんでした。"))
                mess += fields[i] + ","
            i += 1
        if mess == "":
            error = ["OK", ""]
        else:
            error = ["NG", mess[:-1] + _("が書き込めません。")]
        return error

    #  reformTabContents
    #
    def reformTabContents(self, value):
        """ 行頭に空白、行中にtab追加。tab:8,20で設定"""
        tab1=8; tab2=20
        #各行に分割
        lines = value.split("\n")
        #各行のnul行を削除
        words = []
        for line in lines:
            ll = line.split()
            if len(ll) > 0:
                if ll[0] != "":
                    words.append(line)
        #各行にtab追加
        i = 0
        line = ""
        while i < len(words):
            #行を単語に分割
            wds = words[i].split()
            if len(wds) > 1:
                #2単語以上の場合、行頭に空白(tab1個)追加
                wds[0] = tab1*" " + wds[0]
                if len(wds[0]) < tab2-1:
                    #第1単語がtab2より短い場合、tab2位置まで空白追加
                    wds[0] += (tab2 - len(wds[0]))*" "
                else:
                    #長い場合は、空白1ヶを追加
                    wds[0] += " "
                #1行を作成
                line += wds[0] + " ".join(wds[1:]) + "\n"
            elif len(wds) == 1:
                #行内に1単語のみの場合
                line += tab1*" " + wds[0] + "\n"
            i += 1
        return line

    def getBoundaryContFromPatchType(self, patchCont):
        #patchCont="type empty"の為、empyのみ取得する
        words = patchCont.split()
        if len(words) >= 2:
            pType = words[1]
        else:
            pType = words[0]
        #pType別に内容を取得して返す
        if (pType == "empty;" or pType == "symmetry;" or pType == "symmetryPlane;" or
            pType == "cyclic;" or pType == "cyclicAMI;" or pType == "cyclicACMI;" or
            pType == "cyclicSlip;" or pType == "wedge;"):
            pCont = "type    " + pType
        else:
            pCont = "type    zeroGradient;"
        return pCont

    def makeBoundaryField(self, patches, patchData):
        boundary = ""
        i = 0
        lenPD = len(patchData)
        while i < lenPD:
            #patchDataが空文の場合、作成しない。
            if (patchData[i] != ""):
                patch = patches[i]
                boundary += patch + "\n    {\n"
                boundary += patchData[i]
                boundary += "    }\n"
            i += 1
        return boundary

    def replaceBoundaryField(self, newBoundary, conts):
        """ conts(field内容)から該当するpatch内容に置き換える"""
        #boundaryのポインタを取得
        p = 0
        lineOp = strOp(conts)
        pp = lineOp.getKeywordPointer("boundaryField", p)
        #boundaryの内容を取得
        (boundary, p) = lineOp.getKeywordContents("boundaryField", pp)
        #ポインタをboundaryの頭に戻す
        if boundary != "":
            p = pp
            conts = (conts[:p]
                        + "boundaryField\n{\n"
                        + newBoundary
                        + "}\n\n// ************************************************************************* //")
        else:
            conts = ""
        return conts

    def getPatchNameFromBoundary(self, meshDir):
        """ boundaryからpatch名、patch内容を取得して戻る。
        
        Args:
            meshDir (str)   :polyMeshの親dir、又はboundaryの親dir(fullPath)
        Returns:
            ans (list(list)):[[patchName, type etc, nFaces, startFace], ...]
        """
        if meshDir.split(os.sep)[-1] == "polyMesh":
            fileName = meshDir + "/boundary"
        else:
            fileName = meshDir + os.sep + "polyMesh/boundary"
        if len(glob.glob(fileName)) == 0:
            if len(glob.glob(fileName + ".gz")) == 0:
                print("error: " + fileName + " " + _("は、polyMeshのDirでは、ありません。"))
                return []
            else:
                #圧縮ファイルを展開する
                cont = foamFile().read(fileName)
                if cont.find("...") < 0:
                    #binaryが存在しないので、そのまま保存
                    f = open(fileName, "w"); f.write(cont); f.close()
                else:
                    #binaryが存在するので、戻る
                    print("error: " + fileName + " " + _("は、polyMeshのDirでは、ありません。"))
                    return []
        ans = []
        #boundar読み込み
        f = open(fileName); contents = f.read(); f.close()
        contOp = strOp(contents)
        p = contOp.skipFoamFile()
        (bndConts, p) = contOp.getSmallPair(p)
        bndOp = strOp(bndConts)
        ((keyword, bndCont), p) = bndOp.getNextKeywordAndContents(0)
        #patch名毎に処理
        while keyword != "":
            bndContOp = strOp(bndCont)
            #      patchName, type etc., nFaces, strtFace
            name = ["",       "",        "",     ""]
            name[0] = keyword
            #patch内をline毎に処理
            (lineCont, pp, _kind) = bndContOp.get1line(0)
            while lineCont != "":
                words = lineCont.split()
                if len(words) > 1:
                    if words[0] == "type":
                        name[1] += " ".join(words) + "\n"
                    elif words[0] == "nFaces":
                        name[2] = words[1].split(";")[0]
                    elif words[0] == "startFace":
                        name[3] = words[1].split(";")[0]
                    else:
                        line = " ".join(words) + "\n"
                        line = line.replace(" ( ", "(")
                        line = line.replace(" ) ", ")")
                        name[1] += line
                (lineCont, pp, _kind) = bndContOp.get1line(pp)
            ans.append(name)
            ((keyword, bndCont), p) = bndOp.getNextKeywordAndContents(p)
        return ans

    def getFieldsPatchNames(self, timeFolder, fields, region="."):
        """ 指定したfieldのpatch名とboundaryConditionを取得
        
        Args:
            timeFolder (str)    :時間folder
            fields (list(str))  :field名のlist
            region (str)        :region名(default:".")
        Returns:
            fields (list(str))    :field名のlist
            patches (list(str))   :patch名のlist
            patchData (list(str)) :patch内容のlist"""
        fieldsPatch = []
        patchData = []
        fieldDir = os.sep.join([self.caseDir, timeFolder, region])
        for field in fields:
            patchesCont = []
            fileName = fieldDir + os.sep + field
            contents = foamFile().read(fileName)
            contOp = strOp(contents)
            p = 0
            (boundary, p) = contOp.getKeywordContents("boundaryField", p)
            bndOp = strOp(boundary)
            ((patch, cont), p) = bndOp.getNextKeywordAndContents(0)
            while patch != "":
                patchesCont.append([patch, cont])
                ((patch, cont), p) = bndOp.getNextKeywordAndContents(p)
            fieldsPatch.append(patchesCont)
        patches = []
        for patchC in fieldsPatch[0]:
            patches.append(patchC[0])
        for fieldPatch in fieldsPatch:
            col = []
            for patch in patches:
                col.append("")
            for fpatch in fieldPatch:
                flag = 0
                i = 0
                for patch in patches:
                    if fpatch[0] == patch:
                        col[i] = self.compressSPCR(fpatch[1])
                        flag = 1
                        break
                    i += 1
                if flag == 0:
                    patches.append(fpatch[0])
                    col.append(self.compressSPCR(fpatch[1]))
            patchData.append(col)
        return (fields, patches, patchData)

    #
    #  changeEmptyToZeroGradient
    #
    def chengeEmptyToZeroGradient(self, timeFolder, meshDir, region="."):
        """ boundary内のpatchTypeをempty→patchに変更。
        各field内のempty→zeroGradientに変更する。
    
        Args:
            timeFolder (str)    :時間folder
            meshDir (str)       :polyMeshの親dir(fullPath)
            region (str)        :region（default:「.」
        Returns:
            None"""
        fields = self.getFieldNames(timeFolder, region)
        if len(fields) == 0:
            return
        #boundaryのcheck
        self.changeBoundaryEmptyToPatch(timeFolder, meshDir, region)
        #fieldのcheck
        self.changeFieldsEmptyToZeroGradient(timeFolder, meshDir, region)

    def changeBoundaryEmptyToPatch(self, timeFolder, meshDir, region):
        """ boundary内のemptyをpatchに変更する"""
        if meshDir.split(os.sep)[-1] == "polyMesh":
            fileName = meshDir + "/boundary"
        else:
            fileName = meshDir + os.sep + "polyMesh/boundary"
        f = open(fileName); contents = f.read(); f.close()
        contOp = strOp(contents)
        p = contOp.skipFoamFile()
        (bndConts, p) = contOp.getSmallPair(p)
        sp = p - len(bndConts) -1
        ep = p - 1
        ap = self.searchTypeEmpty(bndConts)
        if ap >= 0:
            bndOp = strOp(bndConts)
            newLine = "type        patch;"
            bndConts, p, keyword = bndOp.replace1lineKeyword(ap, newLine)
            newConts = contents[:sp] + bndConts + contents[ep:]
            f = open(fileName, "w"); f.write(newConts); f.close()

    def changeFieldsEmptyToZeroGradient(self, timeFolder, meshDir, region):
        """ 各field内のemptyをzeroGradientに修正する"""
        fields = self.getFieldNames(timeFolder, region)
        if len(fields) == 0:
            return
        #fieldを修正
        fieldDir = os.sep.join([self.caseDir, timeFolder, region])
        for field in fields:
            fileName = fieldDir + os.sep + field
            foam = foamFile()
            contents = foam.read(fileName)
            contOp = strOp(contents)
            p = 0
            (boundary, p) = contOp.getKeywordContents("boundaryField", p)
            ap = self.searchTypeEmpty(boundary)
            if ap >= 0:
                boundaryOp = strOp(boundary)
                newLine = newLine = "type        zeroGradient;"
                newBnd, p, keyword = boundaryOp.replace1lineKeyword(ap, newLine)
                newBoundary  = "boundaryField\n"
                newBoundary += "{"
                newBoundary += newBnd
                newBoundary += "}"
                newContents, p, _keyword = contOp.replace1lineKeyword(0, newBoundary)
                foam.write(fileName, newContents)

    def searchTypeEmpty(self, conts):
        """ conts内から「type empty」を検索し、
        有れば、そのpointerを返す。
        無ければ、-1を返す。"""
        contOp = strOp(conts)
        ans = -1
        p = 0
        while True:
            p = contOp.getKeywordPointer("type", p)
            if p > 0:
                lineCont, pp, kind = contOp.get1line(p)
                if lineCont.find("empty") >= 0:
                    ans = p
                    break
                else:
                    p = pp
            else:
                ans = -1
                break
        return ans

    #
    #  changePatchName
    #
    def changePatchName(self, boundaryDir, fieldDir, fields, oldPatchName, newPatchName):
        """ patch名を変更する。
        
        Args:
            boundaryDir (str)   :boundaryの親dir
            fieldDir (str)      :fieldファイルの親dir
            fields (list(str))  :対象のfield [field1, field2, ...]
            oldPatchName (str)  :修正前のpatchName
            newPatchName (str)  :修正後のpatchName
        Returns:
            error (list(str))   :(<"OK" or "NG">, error内容)
            """
        #文字種をチェック（使える文字かどうか）
        error = self.checkPatchName(newPatchName)
        if error[0] != "OK":
            return error
        #oldPatchNameがboundaryFieldにあるかどうかチェック
        error = self.IsOldPatchNameInBoundary(boundaryDir, fieldDir, fields, oldPatchName)
        if error[0] != "OK":
            return error
        #boundary中のpatchNameを修正
        error = self.changePatchNameInBoundary(boundaryDir, oldPatchName, newPatchName)
        if error[0] != "OK":
            return error
        #boundaryField中のpatchNameを修正
        error = self.changePatchNameInBoundaryField(fieldDir, fields, oldPatchName, newPatchName)
        return error

    def checkPatchName(self, patchName):
        """ patchNameの文字種をチェックする。"""
        error = ("OK", "")
        flag = 0
        nameOp = strOp(patchName)
        for chara in patchName:
            if nameOp.checkChara(chara) == False:
                flag = 1
                break
        if flag == 0:
            if chara == "/" or chara == "\\" or chara == "'" or chara == '"':
                flag = 1
        if flag == 1:
            error = ("NG", _(u"使えない文字「") + chara + _(u"」が使われています"))
        return error

    def IsOldPatchNameInBoundary(self, boundaryDir, fieldDir, fields, oldPatchName):
        """ oldPatchNameがboundaryにあるかどうかチェック"""
        error = ("OK", "")
        #boundaryのチェック
        fileName = boundaryDir + "/boundary"
        if len(glob.glob(fileName)) != 0:
            f = open(fileName); cont = f.read(); f.close()
            contOp = strOp(cont)
            #FoamFileをskip
            p = contOp.skipFoamFile()
            #boundary中にoldPatchNameがあるかどうかチェック
            (boundCont, p) = contOp.getSmallPair(p)
            boundContOp = strOp(boundCont)
            (lineCont, p, _kind) = boundContOp.get1line(0)
            (keyword, _pp) = strOp(lineCont).getKeyword(0)
            while lineCont != "" and keyword != oldPatchName:
                (lineCont, p, _kind) = boundContOp.get1line(p)
                (keyword, _pp) = strOp(lineCont).getKeyword(0)
            if keyword != oldPatchName:
                error = ("NG", _(u"boundary中に「") + oldPatchName + _(u"」が存在しません"))
                return error
        else:
            error = ("NG", fileName+_(u"が存在しません。"))
            return error
        #boundaryFieldのチェック
        for field in fields:
            fileName = fieldDir + "/" + field
            cont = foamFile().read(fileName)
            contOp = strOp(cont)
            #FoamFileをskip
            p = contOp.skipFoamFile()
            #boundaryFieldを取得
            (boundCont, p) = contOp.getKeywordContents("boundaryField", p)
            #boundaryField中にoldPatchNameがあるかどうかチェック
            boundContOp = strOp(boundCont)
            (lineCont, p, _kind) = boundContOp.get1line(0)
            (keyword, _pp) = strOp(lineCont).getKeyword(0)
            while lineCont != "" and keyword != oldPatchName:
                (lineCont, p, _kind) = boundContOp.get1line(p)
                (keyword, _pp) = strOp(lineCont).getKeyword(0)
            if keyword != oldPatchName:
                error = ("NG", _(u"field ") + field + _(u" のboundaryField中に「") + oldPatchName + _(u"」が存在しません"))
                return error
        return error

    def changePatchNameInBoundary(self, boundaryDir, oldPatchName, newPatchName):
        """ boundary中のpatchNameを修正する"""
        error = ("OK", "")
        fileName = boundaryDir + "/boundary"
        f= open(fileName); cont = f.read(); f.close()
        contOp = strOp(cont)
        #FoamFileをskip
        p = contOp.skipFoamFile()
        #oldPatchNameを検索
        newCont = cont
        (bndCont, p) = contOp.getSmallPair(p)
        ps = p - len(bndCont) - 1
        bndContOp = strOp(bndCont)
        (lineCont, p, _kind) = bndContOp.get1line(0)
        (keyword, pp) = strOp(lineCont).getKeyword(0)
        while lineCont != "":
            if keyword == oldPatchName:
                newLineCont = newPatchName + lineCont[pp:]
                (newCont, _p, _keyFlag) = contOp.replace1line(ps, lineCont, newLineCont)
                break
            (lineCont, p, _kind) = bndContOp.get1line(p)
            (keyword, pp) = strOp(lineCont).getKeyword(0)
        if lineCont == "":
            error = ("NG", _(u"「") + oldPatchName + _(u"」がboundary中に存在しません"))
            return error
        f = open(fileName, "w"); f.write(newCont); f.close()
        return error

    def changePatchNameInBoundaryField(self, fieldDir, fields, oldPatchName, newPatchName):
        """ 各field内のboundaryFieldのpatchNameを修正する。"""
        error = ("OK", "")
        for field in fields:
            fileName = fieldDir + "/" + field
            cont = foamFile().read(fileName)
            contOp = strOp(cont)
            #FoamFileをskip
            p = contOp.skipFoamFile()
            #oldPatchNameを検索
            newCont = cont
            (lineCont, p, _kind) = contOp.get1line(p)
            (keyword, pp) = strOp(lineCont).getKeyword(0)
            while lineCont != "" and keyword != "boundaryField":
                (lineCont, p, _kind) = contOp.get1line(p)
                (keyword, pp) = strOp(lineCont).getKeyword(0)
            if keyword == "boundaryField":
                ps = p - len(lineCont) - 1
                (bndCont, p) = strOp(lineCont).getMiddlePair(0)
                bndContOp = strOp(bndCont)
                (lineCont, p, _kind) = bndContOp.get1line(0)
                (keyword, pp) = strOp(lineCont).getKeyword(0)
                while lineCont != "":
                    if keyword == oldPatchName:
                        newLineCont = newPatchName + lineCont[pp:]
                        (newCont, _p, _keyFlag) = contOp.replace1line(ps, lineCont, newLineCont)
                        break
                    (lineCont, p, _kind) = bndContOp.get1line(p)
                    (keyword, pp) = strOp(lineCont).getKeyword(0)
            foamFile().write(fileName, newCont)
        return error

    #
    #  deletePatch
    #
    def deletePatch(self, boundaryDir, fieldDir, fields, delPatch):
        """ boundary, fieldのファイルからpatchを削除する。
        
        Args:
            boundaryDir (str)   :boundaryの親dir
            fieldDir (str)      :fieldの親dir
            fields (list(str))  :fieldのlist
            delPatch (str)      :削除するpatch名
        Returns:
            error (str)     :errorMessage(「""」の場合は、error無し)
        """
        error = self.delPatchBoundary(boundaryDir, delPatch)
        if error == "":
            error = self.delPatchBoundaryFields(fieldDir, fields, delPatch)
        return error

    def delPatchBoundary(self, boundaryDir, delPatch):
        """ bondaryからpatchを削除する"""
        error = ""
        fileName = boundaryDir + "/boundary"
        if len(glob.glob(fileName)) == 0:
            error = fileName + _(u"が存在しません。")
            return error

        f = open(fileName); cont = f.read(); f.close()
        contOp = strOp(cont)
        p = contOp.skipFoamFile()
        #patch数を減らす
        (nPatch, p) = contOp.getKeyword(p)
        pst = p - (len(nPatch))             #pointerをkeywordの前に戻す
        newCont = cont[:pst]                #pointerまでを書き込む
        n = int(nPatch) - 1
        newCont += str(n) + "\n("           #patch数を書き込む
        p = pst + len(str(n))               #pointerをpatch数に設定
        #patchを削除
        (bndCont, p) = contOp.getSmallPair(p)
        bndContOp = strOp(bndCont)
        p = 0; flag = 0
        patchName = ("", "")
        while (patchName[0] != delPatch) and (p < len(bndCont)):
            pbak = p
            (patchName, p) = bndContOp.getNextKeywordAndContents(p)
            if patchName[0] == delPatch:
                newCont += bndCont[:pbak] + bndCont[p:]
                flag = 1
        if flag == 1:
            newCont += ")\n\n// ************************************************************************* //"
            f = open(fileName, "w"); f.write(newCont); f.close()
        else:
            error = _(u"「") + delPatch + _(u"」のpatchがboundary中に存在しません")
            print(error)
        return error

    def delPatchBoundaryFields(self, fieldDir, fields, delPatch):
        """ fieldのpatchを削除する"""
        error = ""
        for field in fields:
            fileName = fieldDir + "/" + field
            cont = foamFile().read(fileName)
            contOp = strOp(cont)
            p = contOp.skipFoamFile()
            p = contOp.getKeywordPointer("boundaryField", p)
            newCont = cont[:p]
            (bndCont, p) = contOp.getKeywordContents("boundaryField", p)
            newCont += "boundaryField\n{"
            bndContOp = strOp(bndCont)
            p = 0; flag = 0
            patchName = ("", "")
            while (patchName[0] != delPatch) and (p < len(bndCont)):
                pbak = p
                (patchName, p) = bndContOp.getNextKeywordAndContents(p)
                if patchName[0] == delPatch:
                    newCont += bndCont[:pbak] + bndCont[p:]
                    flag = 1
            if flag == 1:
                newCont += "}\n\n// ************************************************************************* //"
                foamFile().write(fileName, newCont)
            else:
                error = _(u"「") + delPatch + _(u"」のpatchが「") + field + _(u"」のboundaryField中に存在しません")
                print(error)
                break
        return error

    #
    #  clearInternalFields
    #
    def clearInternalFields(self, fieldDir, fields):
        """ internalFieldをクリアする
        
        Args:
            fieldDir (str)      :対象fieldの親dir(fullPath)
            fields (list(str))  :対象field名
        Returns:
            None"""
        if len(fields) != 0:
            print(_("internal field をクリアします..."))
            for name in fields:
                print("    " + name + "...")
                fileName = fieldDir + "/" + name
                cont = foamFile().read(fileName)
                lineOp = strOp(cont)
                p = lineOp.skipFoamFile(0)
                n = cont[p:].find("internalField")
                if n >= 0:
                    ps = p + n
                    n = cont[ps:].find(";")
                    if n >= 0:
                        pe = ps + n + 1
                        line = cont[ps:pe]
                        newLine = self.clearInternalField(line)
                        newCont = cont[:ps] + newLine + cont[pe:]
                        foamFile().write(fileName, newCont)
            print(_("internal field をクリアしました。"))

    def clearInternalField(self, line):
        lineOp = strOp(line)
        (dummy, ps) = lineOp.getKeyword(0)
        (valForm, p) = lineOp.getKeyword(ps)
        if valForm == "nonuniform":
            valType = line[p:p+100]
            if valType.find("scalar") >= 0:
                val = "0"
            elif valType.find("vector") >= 0:
                val = "(0 0 0)"
            elif valType.find("symmTensor") >= 0:
                val = "(0 0 0 0 0 0)"
            elif valType.find("tensor") >= 0:
                val = "(0 0 0 0 0 0 0 0 0)"
            else:
                val = "0"
            newLine = line[:ps] + " uniform " + val + ";\n"
            line = newLine
        return line

    def compressSPCR(self, contents):
        """ 余分な空白と改行を削除する。"""
        lines = contents.split("\n")
        ans = []
        for line in lines:
            if line != "":
                words = line.split()
                ans.append(" ".join(words))
        ansR = "\n".join(ans)
        if len(ansR) > 0:
            if ansR[-1] == "\n":
                ansR = "\n".join(ans)[:-1]
        else:
            ansR = "\n"
        return ansR

    def isMultiRegion(self):
        """ 解析caseがmultiRegionかどうか確認する
        
        Aegs:
            None
        Returns:
            ans (bool)              :True(multiRegion) or False(normalCase)
            location (str)          :polyMeshの親folder名(time or constant)
            reginNames (list(str))  :region名のリスト"""
        regions = []
        os.chdir(self.caseDir)
        regMeshDirs = glob.glob("*/*/polyMesh/boundary")
        #timeFolder内のmeshを確認
        currtime = self.getCurrTimeFolder()
        timeMeshes = []
        location = ""
        for meshDir in regMeshDirs:
            time = meshDir.split(os.sep)[0]
            try:
                a = float(time)
                timeMeshes.append([a, meshDir])
            except:
                pass
        #timeFolder内のmeshを確認
        if len(timeMeshes) > 0:
            #locationを取得
            timeMeshes.reverse()
            for timeMesh in timeMeshes:
                locTime = timeMesh[1].split(os.sep)[0]
                if locTime == currtime:
                    #regionを取得
                    location = locTime
                    regions.append(timeMesh[1].split(os.sep)[1])
                    break
        #constantを確認
        if len(regions) == 0:
            location = "constant"
            for meshDir in regMeshDirs:
                if meshDir.split(os.sep)[0] == location:
                    regions.append(meshDir.split(os.sep)[1])
        #戻り値を設定
        ans = False
        regNames = []
        if len(regions) > 0:
            ans = True
            regNames = regions
            regNames.sort()
            print(_("folder「") + location + _("」内に "), regNames, _("のregionがあります。"))
        return [ans, location, regNames]

    #  getAllRegions
    def getAllRegions(self):
        """ constantフォルダ内から「.」を含め全region名を取得する"""
        regNames = []
        os.chdir(self.caseDir)
        if os.path.exists("constant/polyMesh/boundary") == True:
            regNames.append(".")
        meshDirs = glob.glob("constant/*/polyMesh/boundary")
        for meshDir in meshDirs:
            words = meshDir.split("/")
            regName = words[1]
            regNames.append(regName)
        regNames.sort()
        return regNames

    #
    #  getZonesInRegionProperties
    #
    def getZonesInRegionProperties(self, OFversion):
        """ regionPropertiesからfluidRegion, solidRegionを取得する"""
        
        def readRegionProperties():
            fileName = self.caseDir + "/constant/regionProperties"
            if os.path.exists(fileName) == False:
                cont = ""
            else:
                f = open(fileName); cont = f.read(); f.close()
            return cont

        fluidRegions = []
        solidRegions = []
        #OFversionチェック
        numVer = getNumVersion(OFversion)
        #OF-2.1以下、OF-ext？
        if (0.0 < numVer and numVer <= 2.1) or OFversion[:3] == "ext":
            cont = readRegionProperties()
            if cont == "":
                return ("", "")
            #fluidRegionsを検索
            contOp = strOp(cont)
            ps = contOp.skipFoamFile()
            keyword = " "
            while (keyword != "fluidRegionNames") and (keyword != "") and (p < len(cont)):
                (keyword, p) = contOp.getKeyword(p)
            if keyword == "fluidRegionNames":
                (pairCont, p) = contOp.getSmallPair(p)
                fluidRegions = pairCont.split()
            else:
                fluidRegions = []
            #solidRegionsを検索
            p = ps
            keyword = " "
            while (keyword != "solidRegionNames") and (keyword != "") and (p<len(cont)):
                (keyword, p) = contOp.getKeyword(p)
            if keyword == "solidRegionNames":
                (pairCont, p) = contOp.getSmallPair(p)
                solidRegions = pairCont.split()
            else:
                solidRegions = []
        #OF-11以上
        if numVer >= 11.0:
            regionSolversDict = self.getControlDict()["regionSolvers"]
            fluidRegions = []
            solidRegions = []
            for region in regionSolversDict.keys():
                solver = regionSolversDict[region].lower()
                #solver名からsolid or fluidを判断
                if solver == "solid":
                    solidRegions.append(region)     #solid
                else:
                    fluidRegions.append(region)     #fluid
                # elif solver.find("fluid") >= 0:
                #     fluidRegions.append(region)     #fluid
                # elif solver.find("incompressible") >= 0:
                #     fluidRegions.append(region)     #fluid
                # elif solver.find("compressible") >= 0:
                #     fluidRegions.append(region)     #fluid
                # elif solver.find("multicomponent") >= 0:
                #     fluidRegions.append(region)     #fluid
                # elif solver.find("multiphase") >= 0:
                #     fluidRegions.append(region)     #fluid
        #OF-2.2.0〜OF-10、OF-v関連
        else:
            #ver 2.2.0以降
            cont = readRegionProperties()
            if cont == "":
                return ("", "")
            contOp = strOp(cont)
            p = contOp.skipFoamFile()
            # #regionを取得
            (keyword, p) = contOp.getKeyword(p)
            while (keyword != "" and keyword != "regions"):
                (keyword, p) = contOp.getKeyword(p)
            if keyword == "regions":
                (contents, p) = contOp.getSmallPair(p)
                contentsOp = strOp(contents)
                #fluidを取得
                (keyword, p) = contentsOp.getKeyword(0)
                while (keyword != "" and keyword != "fluid"):
                    (keyword, p) = contentsOp.getKeyword(p)
                if keyword == "fluid":
                    (regionCont, p) = contentsOp.getSmallPair(p)
                    fluidRegions = regionCont.split()
                #solidを取得
                (keyword, p) = contentsOp.getKeyword(0)
                while (keyword != "" and keyword != "solid"):
                    (keyword, p) = contentsOp.getKeyword(p)
                if keyword == "solid":
                    (regionCont, p) = contentsOp.getSmallPair(p)
                    solidRegions = regionCont.split()
        return (fluidRegions, solidRegions)

    #
    #  getSolverBC
    #    solverとBC（ascii/binary, normal/compression）を取得
    def getSolverBC(self):
        """ caseのsolverとBCをcontrolDictから取得"""
        contsDict = self.getControlDict()
        solver = contsDict["application"]
        if contsDict["solver"] != "":
            solver += "." + contsDict["solver"]
        wFormat = contsDict["writeFormat"]
        if wFormat == "binary":
            BCb = "B"
        else:
            BCb = "a"
        comp = contsDict["writeCompression"]
        if comp == "compressed" or comp == "yes" or comp == "on":
            BCc = "C"
        else:
            BCc = "n"
        BC = BCb + BCc + "P"
        return [solver, BC]

    #
    #  getNumberResult
    #
    """ 計算結果の nR, st, ed を取得"""
    def getNumberResult(self):
        folNames = getFolderNames(self.caseDir)
        #並列数を確認
        nProc = len(list(filter(lambda x: x[:len("processor")] == "processor", folNames)))
        # nProc = 0
        # for folder in folNames:
        #     if folder[:len("processor")] == "processor":
        #         nProc += 1
        #結果folder数を取得
        ans = []
        for folder in folNames:
            try:
                a = float(folder)
                ans.append(a)
            except:
                pass
        nRst = len(ans)
        if nProc > 1:
            #並列処理の計算結果を確認
            ansP = self.getTimeFolders("processor0")
            if len(ansP) > nRst:
                nRst = len(ansP)
                ans = list(map(float, ansP))
        #戻り値を設定
        ansR = ["", "", "", ""]
        if nRst != 0:
            ansR[0] = " " + str(nRst)           # nR
            ansR[1] = " " + str(min(ans))       # st
            if min(ans) != max(ans):
                ansR[2] = " " + str(max(ans))   # ed
            if nProc != 0:
                ansR[3] = str(nProc)            # nProc
        return ansR

    #
    #  getCaseCont
    #
    def getCaseCont(self):
        """ case内の solver, BCPn, nR, st, ed を取得する
        
        Args:
            None
        Returns
            caseCont (list) :[solver, BCPn, nR, st, ed]を返す"""
        [solver, BC] = self.getSolverBC()
        nRst = self.getNumberResult()
        #並列処理?
        if nRst[-1] != "":
            #並列数を追加
            BCPn = BC + nRst[-1]
        else:
            BCPn = BC
        nR=nRst[0]; st=nRst[1]; ed=nRst[2]
        caseCont = [solver, BCPn, nR, st, ed]
        return caseCont

    def getControlDict(self, name="controlDict", remake=True):
        """ controlDict内容の読み込み
        
        Args:
            None
        Returns:
            dict    :辞書形式で戻す
        """
        fileName = os.sep.join([self.caseDir, "system", name])
        # self.itemsDict = {
        #     "application": "",
        #     "solver": "",
        #     #"subSolver": "",
        #     "regionSolvers": {},        #{<region>:<solver>}
        #     "startFrom": "",
        #     "startTime": "",
        #     "stopAt": "",
        #     "endTime": "",
        #     "deltaT": "",
        #     "timePrecision": "",
        #     "writeInterval": "",
        #     "writeFormat": "",
        #     "writePrecision": "",
        #     "writeCompression": "",
        #     "runTimeModifiable": "",
        #     "adjustTimeStep": ""
        #     }
        f = open(fileName); cont = f.read(); f.close()
        #cont内の#include行を展開して取得する
        cont = self.expandIncludeFile(cont)
        contOp = strOp(cont)
        p = contOp.skipFoamFile()
        newCont = cont[p:]
        itemsDict = self.getItemsDict(newCont)
        #辞書内容を整理(self.itemsDictの項目のみ）
        for key in self.itemsDict.keys():
            if key in itemsDict.keys():
                self.itemsDict[key] = itemsDict[key]
        #applicationをチェック
        if remake == True:
            #修正
            if self.itemsDict["application"] == "":
                app = ""
                if self.itemsDict["solver"] != "":
                    app = "foamRun"
                elif self.itemsDict["regionSolvers"] != "":
                    if self.itemsDict["regionSolvers"] != {}:
                        app = "foamMultiRun"
                #applicationをセット
                if app != "":
                    self.itemsDict["application"] = app
                    setDict = {"application": app}
                    self.setControlDict(setDict)
        return self.itemsDict

    def getItemsDict(self, cont):
        """ cont内のitemsDictを取得して返す。
        include, includeIfPresentがあれば再帰的呼び出しで読み込む。"""
        contOp = strOp(cont)
        itemsDict = {}
        p = 0
        line = " "
        while line != "":
            (line, p, kind) = contOp.get1line(p)
            #通常のline
            if kind == "line":
                line = line.replace(";", "")
                words = line.split()
                if len(words) >= 2:
                    itemsDict[words[0]] = words[1]
            #keyword { }
            elif kind == "keyword":
                lineOp = strOp(line)
                keyName, _p = lineOp.getKeyword(0)
                if keyName == "regionSolvers":
                    newCont, _p = lineOp.getMiddlePair(0)
                    regionSolversDict = self.getItemsDict(newCont)
                    itemsDict[keyName] = regionSolversDict
        return itemsDict

    def expandIncludeFile(self, cont):
        """ cont内のinclude行を展開して返す"""

        def getIncludeCont(cont):
            currDir = self.caseDir + "/system"
            allCont = ""
            contOp = strOp(cont)
            p = 0
            line = " "
            while line != "":
                sp = p
                (line, p, kind) = contOp.get1line(p)
                pp = p - len(line)
                #include?
                if kind == "include":
                    #行頭の「#」まで遡る
                    pp = cont[:pp].rfind("#")
                    #前回の行末から今回の行頭までを取得
                    allCont += cont[sp:pp]
                    #includeFile名を取得
                    incName = line.split('"')[1]
                    incFile = self.getAbsPath(currDir, incName)
                    f = open(incFile); newCont = f.read(); f.close()
                    #include内容を取得
                    subCont = getIncludeCont(newCont)
                    #include行をcommentOut
                    repLine = "//" + line + "\n"
                    allCont += repLine
                    #include内容をセット
                    allCont += subCont + "\n"
                #includeIfPresent?
                elif kind == "includeIfPresent":
                    #行頭の「#」まで遡る
                    pp = cont[:pp].rfind("#")
                    #前回の行末から今回の行頭までを取得
                    allCont += cont[sp:pp]
                    #includeFile名を取得
                    incName = line.split('"')[1]
                    incFile = self.getAbsPath(currDir, incName)
                    #includefile有無確認
                    if os.path.exists(incFile) == True:
                        f = open(incFile); newCont = f.read(); f.close()
                        #include内容を取得
                        subCont = getIncludeCont(newCont)
                        #include行をcommentOut
                        repLine = "//" + line + "\n"
                        allCont += repLine
                        #include内容をセット
                        allCont += subCont + "\n"
                    else:
                        allCont += line + "\n"
                #その他？
                else:
                    allCont += cont[sp:pp]
                    item, _p = strOp(line).getKeyword(0)
                    if item in self.itemsDict.keys():
                        if kind == "keyword":
                            newLine, _p = strOp(line).getMiddlePair(0)
                            subCont = getIncludeCont(newLine)
                            line = item + "\n"
                            line += "{"
                            line += subCont + "\n"
                            line += "}"
                    allCont += line
            return allCont

        fullCont = ""
        contOp = strOp(cont)
        p = contOp.skipFoamFile()
        fullCont += cont[:p]
        subCont = getIncludeCont(cont[p:])
        fullCont += subCont
        return fullCont

    def setControlDict(self, setDict):
        """ controlDictの書き込み
        
        Args:
            setDict (dict)  :設定したい項目の辞書
        Returns:
            None
        """

        def getNextLinePosition(cont, key):
            """ 最終行の最後の位置を返す"""
            contOp = strOp(cont)
            ip = -1
            p = 0
            lineCont = " "
            while lineCont != "":
                ps = p
                lastLine = lineCont
                (lineCont, p, kind) = contOp.get1line(p)
                if lineCont != "":
                    (keyword, _p) = strOp(lineCont).getKeyword(0)
                    if keyword == key:
                        ip = p - len(lineCont)
                        #ip = p + 1
            if ip < 0:
                ip = p + len(lastLine)
            return ip  

        def getFromTopLinePosition(cont, n):
            contOp = strOp(cont)
            p = contOp.skipFoamFile()
            i = -1
            lineCont = " "
            while lineCont != "":
                lastLine = lineCont
                ps = p
                (lineCont, p, kind) = contOp.get1line(p)
                if lineCont != "":
                    i += 1
                    if i == n:
                        #指定した「n」番目に挿入
                        #ip = p - len(lineCont) - 1
                        ip = p - len(lineCont)
                        ip = contOp.skipBackCR(ip)
                        break
                else:
                    #最終行に挿入
                    ip = ps + 1
                    break
            return ip

        fileName = os.sep.join([self.caseDir, "system", "controlDict"])
        f = open(fileName); cont = f.read(); f.close()
        cont = self.expandIncludeFile(cont)
        contOp = strOp(cont)
        p = contOp.skipFoamFile()
        lineCont = " "
        while lineCont != "":
            (lineCont, p, kind) = contOp.get1line(p)
            if kind == "line":
                words = lineCont.split()
                if len(words) >= 2:
                    if words[0] in setDict.keys():
                        pp = p - len(lineCont)
                        if setDict[words[0]] == "":
                            #項目削除
                            cont = contOp.line
                            cont = cont[:pp] + cont[p:]
                            contOp.line = cont
                        else:
                            #置き換え
                            newWord = setDict[words[0]] + ";"
                            newLineCont = lineCont.replace(words[1], newWord)
                            (cont, p, keyword) = contOp.replace1lineKeyword(pp, newLineCont)
                        _cont = setDict.pop(words[0])
            elif kind == "keyword":
                keyName, _p = strOp(lineCont).getKeyword(0)
                if keyName in setDict.keys():
                    pp = p - len(lineCont)
                    if setDict[keyName] == "":
                        #項目削除
                        cont = contOp.line
                        cont = cont[:pp] + cont[p+1:]
                        contOp.line = cont
                    else:
                        #置き換え
                        newWord = setDict[keyName]
                        lines = newWord.split("\n")
                        lines = list(map(lambda x: "    " + x, lines))
                        newWord = "\n".join(lines) + "\n"
                        newWordOp = strOp(newWord)
                        newWordOp.deleteNullLines()
                        newWord = newWordOp.line
                        newLine  = keyName + "\n"
                        newLine += "{\n"
                        newLine += newWord
                        newLine += "}"
                        (cont, p, keyword) = contOp.replace1lineKeyword(pp, newLine)
                    _cont = setDict.pop(keyName)
        #setDict内は空？
        if len(setDict.keys()) > 0:
            #空でない場合
            #追加する
            addKeys = list(set(setDict.keys()) & set(self.itemsDict.keys()))
            cont = contOp.line
            for key in addKeys:
                if key == "regionSolvers":
                    newLine = ""
                    #regionSolversの内容が空？
                    if len(setDict[key]) > 0:
                        #空でない場合取得して設定
                        #ip = getNextLinePosition(cont, "application")
                        ip = getFromTopLinePosition(cont, 1)
                        newWord = setDict[key]
                        lines = newWord.split("\n")
                        for i in range(len(lines)):
                            words = lines[i].split()
                            if len(words) > 1:
                                lines[i] = "    " + ("%-15s" % words[0]) + " " + "".join(words[1:])
                        newWord = "\n".join(lines)
                        #newLine  = "\n" + key + "\n"
                        newLine  = key + "\n"
                        newLine += "{\n"
                        newLine += newWord
                        newLine += "}"
                else:
                    #通常のkeyを挿入
                    if key == "application":
                        ip = getFromTopLinePosition(cont, 0)
                    else:
                        ip = getFromTopLinePosition(cont, -1)
                    newLine = ("%-15s" % key) + " " + setDict[key] + ";"
                if newLine != "":
                    cont = cont[:ip] + newLine + "\n\n" + cont[ip:]
                contOp.line = cont

        #controlDictに書き込み
        cont = contOp.line
        fw = open(fileName, "w"); fw.write(cont); fw.close()
        #追加を加えること
        return

    def getAbsPath(self, currDir, cont):
        """ contの内容を絶対pathに置き換える。
        currDirは、caseDirを想定しているが、caseDirでなくても可。
        環境変数を判断して、置き換える。"""
        #環境変数S{xxx}を取得
        sp = cont.find("${")
        if sp >= 0:
            # ${xxx}記述の解釈
            (envPath, p) = stringOp.strings(cont).getMiddlePair(sp)
            envPath = envPath.decode()
            if os.getenv(envPath) != None:
                cont = cont[:sp] + os.getenv(envPath) + cont[p:]
            else:
                if envPath == "FOAM_CASE":
                    # $FOAM_CASEは、bashの環境変数ではないのでここでセット
                    cont = cont[:sp] + currDir + cont[p:]
        #環境変数$xxxを取得
        elif cont.find("$") >= 0:
            # $xxx記述の解釈
            sp = cont.find("$")
            newCont = cont[sp+1:]
            envPath = newCont.split("/")[0]
            p = sp + len(envPath) + 1
            if os.getenv(envPath) != None:
                cont = cont[:sp] + os.getenv(envPath) + cont[p:]
            else:
                if envPath == "FOAM_CASE":
                    cont = cont[:sp] + currDir + cont[p:]
        #<constant>の解釈
        if cont.find("<constant>") >= 0:
            constantDir = currDir + "/constant"
            cont = cont.replace("<constant>", constantDir)
        #<system>の解釈
        if cont.find("<system>") >= 0:
            systemDir = currDir + "/system"
            cont = cont.replace("<system>", systemDir)
        #絶対pathの取得
        if cont[0] != "/":
            cont = currDir + "/" + cont
        return cont


    def getCreateMeshData(self):
        """ メッシュ作成用データをhelyxOS_dataファイルから取得する
        
        Args:
            None
        Returns:
            dict    :辞書形式で戻す"""
        #初期値
        dataDict = {
            "stlDir": "./model",
            "surfaceAngle": "150",
            "clearMesh": "yes",
            "parallel": "no",
            "nCpu": "4",
            "nSeparate": "2 2 1",
            #"createMesh_featureAngle": "90",
            "createMesh_featureAngle": "80",
            "addLayer_featureAngle": "100"
            }
        helyxDataFile = "helyxOS_data"
        fileNameCase = self.caseDir + os.sep + helyxDataFile
        fileNameTree = os.sep.join([os.getenv("TreeFoamUserPath"), "data", helyxDataFile])
        if len(glob.glob(fileNameCase)) == 0:
            if len(glob.glob(fileNameTree)) == 0:
                #初期値を設定
                cont = ""
                for key in dataDict.keys():
                    cont += key + " " + dataDict[key] + "\n"
                # cont = "stlDir  ./model\n"
                # cont += "surfaceAngle  150\n"
                # cont += "clearMesh  yes\n"
                # cont += "parallel no\n"
                # cont += "nCpu 4\n"
                # cont += "nSeparate 2 2 1\n"
                # cont += "createMesh_featureAngle 90\n"
                # cont += "addLayer_featureAngle 100\n"
                f=open(fileNameCase, "w"); f.write(cont); f.close()
            else:
                shutil.copy(fileNameTree, self.caseDir)
        f = open(fileNameCase); lines = f.readlines(); f.close()
        for line in lines:
            words = line.split()
            if len(words) > 1:
                if words[0] in dataDict.keys():
                    dataDict[words[0]] = " ".join(words[1:])
        return dataDict

    def setCreateMeshData(self, dataDict):
        """ メッシュ作成用データをhelyxOS_dataファイルに書き込む
        
        Args:
            dataDict (dict) :辞書形式のデータ
        Returns:
            None"""
        lines = []
        helyxDataFile = "helyxOS_data"
        fileName = self.caseDir + os.sep + helyxDataFile
        if len(glob.glob(fileName)) == 1:
            f = open(fileName); lines = f.readlines(); f.close()
        for key in dataDict.keys():
            if dataDict[key] != "":
                newLine = key + " " + dataDict[key] + "\n"
                flag = 0
                for i in range(len(lines)):
                    words = lines[i].split()
                    if len(words) > 1:
                        if words[0] == key:
                            lines[i] = newLine
                            flag = 1
                            break
                if flag == 0:
                    lines.append(newLine)
        f = open(fileName, "w"); f.writelines(lines); f.close()

    def getIniCaseFolders(self):
        """ case初期化時に残すfolder名(fullPath)を返す"""
        #削除する不要なfolder名を定義
        delPatterns = delFolderPatternsIniCase
        #folderを取得
        folders = getFolders(self.caseDir)
        #baseNameを抜き出し
        names = list(map(os.path.basename, folders))
        needNames = []      #必要なfolder名
        numFolders = []     #numFolder
        for name in names:
            #削除対象のname？
            if isPatternMatch(delPatterns, name) == False:
                #必要なnameの場合
                try:
                    #numFolderを取得
                    num = float(name)
                    numFolders.append([num, name])
                except:
                    #数値Folder以外の場合
                    needNames.append(name)
        #numFolderが存在するか？
        if len(numFolders) > 0:
            #存在する場合、minのnumFolderを取得
            numFolders.sort()
            numFolder = numFolders[0][1]
            needNames.append(numFolder)
        #取得した必要なfolderをfullPathに変換
        needFolders = list(map(lambda x: self.caseDir+os.sep+x, needNames))
        return needFolders

    def getIniCaseFiles(self):
        """ case初期化時に残すfolder名(fullPath)を返す"""
        files = getFiles(self.caseDir)
        names = list(map(os.path.basename, files))
        #削除対象のpatternを定義
        delPatterns = delFilePatternsIniCase
        needNames = []
        for name in names:
            if isPatternMatch(delPatterns, name) == False:
                needNames.append(name)
        needFiles = list(map(lambda x: self.caseDir+os.sep+x, needNames))
        return needFiles

#------------------
#  paraCase class
#------------------
class paraCase(case):

    def __init__(self, caseDir=os.getcwd()):
        self.caseDir = caseDir

    def getCurrTimeFolder(self):
        timeFolder = ""
        timeFolders = self.getTimeFolders(processor="processor0")
        contDict = self.getControlDict()
        if len(timeFolders) != 0 and contDict["startFrom"] != "" and contDict["startTime"]: 
            if contDict["startFrom"] == "firstTime":
                timeFolder = timeFolders[0]
            elif contDict["startFrom"] == "latestTime":
                timeFolder = timeFolders[-1]
            else:
                timeFolder = contDict["startTime"]
                if len(glob.glob(self.caseDir+os.sep+timeFolder)) == 0:
                    timeFolder = timeFolders[0]
        return timeFolder

    def getCurrMeshDir(self, maxTime, region, meshName):
        savePath = os.getcwd()
        try:
            maxTimeNum = float(maxTime)
        except:
            meshDir = "constant" + os.sep + region + os.sep + "polyMesh"
            return meshDir
        os.chdir(self.caseDir + "/processor0")
        timeFlag = 0
        meshDirs = glob.glob("*/" + region + "/polyMesh/" + meshName)
        if len(meshDirs) > 0:
            ans = []
            for meshDir in meshDirs:
                dirs = meshDir.split(os.sep)
                try:
                    a = float(dirs[0])
                    if a <= maxTimeNum:
                        ans.append(dirs[:-1])
                except:
                    pass
            if len(ans) > 0:
                ans.reverse()
                meshDir = os.sep.join(ans[0])
                timeFlag = 1
        if timeFlag == 0:
            meshDir = "constant" + os.sep + region + os.sep + "polyMesh"
        os.chdir(savePath)
        return meshDir

    def getFieldsPatchNames(self, timeFolder, fields, region=".", proc="."):
        """ 指定したprocessor内の指定したfieldのpatch名とboundaryConditionを取得
        
        Args:
            timeFolder (str)    :時間folder
            fields (list(str))  :field名のlist
            region (str)        :region名(default:".")
            proc (str)          :processor名(default:".")
        Returns:
            fields (list(str))    :field名のlist
            patches (list(str))   :patch名のlist
            patchData (list(str)) :patch内容のlist"""
        fieldsPatch = []
        patchData = []
        fieldDir = os.sep.join([self.caseDir, proc, timeFolder, region])
        for field in fields:
            patchesCont = []
            fileName = fieldDir + os.sep + field
            contents = foamFile().read(fileName)
            contOp = strOp(contents)
            p = 0
            (boundary, p) = contOp.getKeywordContents("boundaryField", p)
            bndOp = strOp(boundary)
            ((patch, cont), p) = bndOp.getNextKeywordAndContents(0)
            while patch != "":
                patchesCont.append([patch, cont])
                ((patch, cont), p) = bndOp.getNextKeywordAndContents(p)
            fieldsPatch.append(patchesCont)
        patches = []
        for patchC in fieldsPatch[0]:
            patches.append(patchC[0])
        for fieldPatch in fieldsPatch:
            col = []
            for patch in patches:
                col.append("")
            for fpatch in fieldPatch:
                flag = 0
                i = 0
                for patch in patches:
                    if fpatch[0] == patch:
                        col[i] = self.compressSPCR(fpatch[1])
                        flag = 1
                        break
                    i += 1
                if flag == 0:
                    patches.append(fpatch[0])
                    col.append(self.compressSPCR(fpatch[1]))
            patchData.append(col)
        return (fields, patches, patchData)

#  --------------
#  foamFile class
#  --------------
class foamFile:
    """ OpenFOAMのfoamFileの読み込み、書き込み関連。
    読み書きは、圧縮、binaryにも適用。
    
    Attribute:
        nMaxLines (int) :読み込む行数(default:20)"""

    def __init__(self, nMaxLines=20):
        self.nMaxLines = nMaxLines      #読み込む行数

    def read(self, fieldName):
        """ fieldデータの読み込み。戻り値は、ascii(str)データ。
        nMaxLinesの行数まで読み込む
        
        Args:
            fieldName (str) :fullPathで指定
        Returns:
            cont (str)      :読み込み結果"""
        (ascii, data) = self.readSeparate(fieldName)
        cont = self.getFuseDataAtMaxLinesAB(fieldName, ascii, data)
        cont = cont.decode()    #結果をstrに変換
        return cont

    def readSeparate(self, fieldName):
        """ fileをascii部、データ部に分けて読み込む
        
        Args:
            fieldName (str) :fullPathで指定
        Returns:
            ascii (str)     :ascii部の内容
            data (bytes)    :データ部（ascii or binary)"""
        funcDict = {
            #  class                関数名
            "regIOobject"       : self.separateContents,
            "cellZoneList"      : self.separateContents,
            "faceZoneList"      : self.separateContents,
            "pointZoneList"     : self.separateContents,
            "labelList"         : self.separateLabelListContents,
            "faceList"          : self.separateFaceListContents,
            "faceCompactList"   : self.separateFaceListContents,
            "refinementHistory" : self.separateFaceListContents,
            "vectorField"       : self.separateVectorFieldContents,
            "cellSet"           : self.separateLabelListContents,
            "faceSet"           : self.separateLabelListContents,
            "pointSet"          : self.separateLabelListContents,
            "polyBoundaryMesh"  : self.separateContents,
            "volScalarField"    : self.separateContents,
            "volVectorField"    : self.separateContents,
            "volTensorField"    : self.separateContents,
            "volSymmTensorField": self.separateContents,
            "surfaceScalarField": self.separateContents,
            "surfaceVectorField": self.separateContents,
            "pointScalarField"  : self.separateContents,
            "pointVectorField"  : self.separateContents
        }
        cont = self.readFull(fieldName)
        classType = self.getClassInFoamFile(cont)
        classType = classType.decode()
        ascii = ""
        data = []
        if classType in funcDict.keys():
            (ascii, data) = funcDict[classType](cont)
        else:
            if classType[-4:] == "List" or classType[-4:] == "list":
                (ascii, data) = self.separateFaceListContents(cont)
            else:
                (ascii, data) = self.separateContents(cont)
        ascii = ascii.decode()
        return (ascii, data)
        
    #
    #  readFull
    #
    def readFull(self, fileName):
        """ fieldデータを全て読み込む（ascii、binaryの区別なく）
        
        Args:
            fileName (str)  :fullpathで指定
        Returns:
            cont (bytes)    :bytes型で取得する"""
        fileName = case().checkFileName(fileName)
        if fileName[-3:] == ".gz":
            #gzip形式でopen、read
            f = gzip.open(fileName, "rb")
            cont = f.read()
            f.close()
        else:
            #非圧縮でopen、read
            f=open(fileName, "rb")
            cont = f.read()
            f.close()
        return cont

    #
    #  write
    #
    def write(self, fileName, cont):
        """ readメソッドで読み込んだcont(ascii, dataをfuseした結果)を
        fileに書き込む。contをascii, dataに分離し、dataは、fileを再読み込み
        して全内容を再構築し、fileに書き込む。
        
        Args:
            fileName (str)  :fullPathで指定
            cont (str)      :ascii, dataをfuseした内容
        Returns:
            None"""
        allData = self.reconstFuseDataAtMaxLinesAB(fileName, cont)
        #全データの書き込み
        self.writeFull(fileName, allData)
        return

    #
    #  writeFull
    #
    def writeFull(self, fileName, cont):
        """ fieldデータを全て書き込み（ascii、binaryの区別なく）
        
        Args:
            fileName (str)  :fullPathで指定
            cont (bytes)    :書き込む内容
        Returns:
            None"""
        if type(cont) == str:
            cont = cont.encode()
        fileName = case().checkFileName(fileName)
        if fileName[-3:] == ".gz":
            #gzip形式でopen、write
            f = gzip.open(fileName, "wb")
            f.write(cont)
            f.close()
        else:
            #非圧縮でopen、write
            f = open(fileName, "wb")
            f.write(cont)
            f.close()
        return

    #  separateContents
    #
    def separateContents(self, cont):
        """ field内容をasciiとdata部に分ける"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiContents(cont)
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryContents(cont)
        else:
            (ascii, data, error) = self.separateBinaryContents(cont)
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiContents(cont)
        return (ascii, data)


    #  separateLabelListContents
    #
    def separateLabelListContents(self, cont):
        """ labelList形式の内容を分離する"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiMultiListContents(cont, "label")
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
        else:
            (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiMultiListContents(cont, "label")
        return (ascii, data)

    #  separateFaceListContents
    #
    def separateFaceListContents(self, cont):
        """ faceList形式の内容を分離する"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiMultiListContents(cont, "faceList")
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
        else:
            (ascii, data, error) = self.separateBinaryMultiListContents(cont, "label")
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiMultiListContents(cont, "faceList")
        return (ascii, data)

    #  separateVectorFieldContents
    #
    def separateVectorFieldContents(self, cont):
        """ vectorField形式の内容を分離する"""
        if self.checkBinary(cont) == "ascii":
            (ascii, data, error) = self.separateAsciiMultiListContents(cont, "vector")
            if error == 1:
                cont = self.remakeDataFormat(cont, "ascii", "binary")
                (ascii, data, error) = self.separateBinaryMultiListContents(cont, "vector")
        else:
            (ascii, data, error) = self.separateBinaryMultiListContents(cont, "vector")
            if error == 1:
                cont = self.remakeDataFormat(cont, "binary", "ascii")
                (ascii, data, error) = self.separateAsciiMultiListContents(cont, "vector")
        return (ascii, data)

    #
    #  findDataListPosition
    def findDataListPosition(self, cont):
        """ contの中から'nonuniform List<'の文字を探す"""
        flag = 0
        i = 0
        while flag == 0:
            n = cont.find(b" List<", i)
            if n >= 0:
                #List<word>のチェック
                newCont = cont[n:]
                #  List<word>は、該当しない。次を検索させる。
                if newCont[:len(b" List<word>")] != b" List<word>":
                    #該当部か50文字をチェック
                    for i in range(1, 50):
                        #1文字を取得するとintになる為、1バイトのbytsに変換
                        chara = cont[n + i].to_bytes(1, "big")
                        if chara == b"{":
                            #次を検索
                            i += n + 1
                            break
                        elif chara == b"(":
                            #終了
                            flag = 1
                            break
                #次を検索
                i += n + 1
            else:
                #終了
                flag = 1
        return n

        # n = cont.find(b" List<")
        # if n >= 0:
        #     #最大50文字をチェック
        #     for i in range(1, 50):
        #         #1文字を取得するとintになる為、1バイトのbytsに変換
        #         chara = cont[n + i].to_bytes(1, "big")
        #         if chara == b"{":
        #             return -1           #文字列なし
        #         elif chara == b"(":
        #             return n            #該当（文字列発見）
        #     return -1           #文字列なし
        # else:
        #     return -1           #文字列なし

        # if n >= 0:
        #     nn = cont[n-15:n].find(b"nonuniform")
        #     if nn >= 0:
        #         #発見時
        #         return n
        #     else:
        #         #文字列無し
        #         return -1
        # else:
        #     #文字列無し
        #     return -1

    #  separateAsciiContents
    #
    def separateAsciiContents(self, cont):
        """ asciiFile内容をasciiとdata部に分ける"""
        error = 0
        obj = self.getObjectInFoamFile(cont)
        ascii = ""
        data = []
        strOp = stringOp.strings(cont)
        p = strOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n >= 0 and error == 0:
            ps = p
            #n = cont[p:].find(b" List<")
            n = self.findDataListPosition(cont[p:])
            if n >= 0:
                p += n
                (dataType, nData) = self.getDataTypeNumData(cont, p)
                #データ数を確認
                if nData == b"0":
                    #データが存在しない場合
                    #  List<scalar> 0の場合（データ数「0」の場合
                    p += 1
                    ascii += cont[ps:p]
                    dataHeader = [dataType, nData]
                    data.append([dataHeader, []])   #追加(23/01/17)
                else:
                    #データが存在する場合
                    dataHeader = [dataType, nData]
                    n = cont[p:p+100].find(b"(")
                    if n >= 0:
                        p += n + 1
                        pe = p
                        if int(nData) < 100:
                            #データ数が少ない場合
                            (contData, p) = strOp.getSmallPair(p-1)
                            spId = obj + b"." + str(i).encode()
                            ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                            data.append([dataHeader, contData])
                            error = self.confirmAscii(contData)
                            p -= 1
                            i += 1
                        else:
                            #データ数が多い場合
                            n = cont[p:].find(b"\n)")
                            if n>=0:
                                spId = obj + b"." + str(i).encode()
                                ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                                p += n + 1
                                contData = cont[pe:p]
                                data.append([dataHeader, contData])
                                error = self.confirmAscii(contData)
                                i += 1
            #i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  separateBinaryContents
    #
    def separateBinaryContents(self, cont):
        """ binaryFile内容をasciiとdata部に分ける"""

        def findListPosition(cont):
            """ 'List<word>'を除く'List<'を検索する。"""
            i = 0
            while True:
                #n = cont.find(b"List<", i)
                n = self.findDataListPosition(cont)
                if n >= 0:
                    block = cont[n:]
                    ps = block.find(b"<") + 1
                    pe = block.find(b">")
                    dataType = block[ps:pe]
                    if dataType != b"word":
                        break
                    else:
                        i = n + 5
                else:
                    break
            return n

        error = 0
        obj = self.getObjectInFoamFile(cont)
        data = []
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n>=0 and error == 0:
            ps = p
            #n = cont[p:].find(b" List<")
            n = findListPosition(cont[p:])
            if n >= 0:
                p += n
                (binaryType, nData) = self.getDataTypeNumData(cont, p)
                #データ数を確認
                if nData == b"0":
                    #データが存在しない場合
                    #  List<scalar> 0の場合（データ数「0」の場合）
                    p += 1
                    ascii += cont[ps:p]
                    dataHeader = [binaryType, nData]
                    data.append([dataHeader, []])   #追加(23/01/17)
                else:
                    #データが存在する場合
                    dataHeader = [binaryType, nData]
                    n = cont[p:p+100].find(b"(")
                    if n >= 0:
                        p += n + 1
                        pe = p
                        (binary, p) = self.getBinaryData(binaryType, nData, cont, p)
                        if len(binary) != 0:
                            spId = obj + b"." + str(i).encode()
                            ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                            data.append([dataHeader, binary])
                            #errorチェック
                            if not (cont[p] == b")" or cont[p] == ord(")")):
                                error = 1
            i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  separateAsciiMultiListContents
    #
    def separateAsciiMultiListContents(self, cont, dataType):
        """ 複数のlist形式のascii内容を分離する"""
        error = 0
        obj = self.getObjectInFoamFile(cont)
        data = []
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n >= 0 and error == 0:
            ps = p
            n = cont[p:].find(b"(")
            if n >= 0:
                pp = p + n
                [nData, p] = lineOp.getBackKeyword(p+n)
                dataHeader = [dataType, nData]
                if int(nData) < 100:
                    #データ数少ない場合
                    p = pp
                    pe = pp + 1
                    [dataCont, p] = lineOp.getSmallPair(p)
                    ascii += cont[ps:pe] + (b"\0" + obj + b"." + str(i).encode() + b"\0")
                    p -= 1
                    data.append([dataHeader, dataCont])
                    error = self.confirmAscii(dataCont)
                else:
                    #データ数多い場合
                    p = pp
                    pe = pp + 1
                    ascii += cont[ps:pe] + (b"\0" + obj + b"." + str(i).encode() + b"\0")
                    ps = p + 1
                    n = cont[p:].find(b"\n)")
                    p += n + 1
                    dataCont = cont[ps:p]
                    data.append([dataHeader, dataCont])
                    error = self.confirmAscii(dataCont)
            i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  separateBinaryMultiListContents
    #
    def separateBinaryMultiListContents(self, cont, dataType):
        """ 複数のlist形式のbinary内容を分離する"""
        (ascii, data, error) = self.separateBinaryFaceListContents(cont, dataType)
        return (ascii, data, error)

    #  separateBinaryFaceListContents
    #
    def separateBinaryFaceListContents(self, cont, dataType):
        """ faceCompactList形式のbinary内容を分離する"""
        error = 0
        obj = self.getObjectInFoamFile(cont)
        data = []
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        ascii = cont[:p]
        n = 0
        i = 0
        while n >= 0 and error == 0:
            ps = p
            n = cont[p:].find(b"(")
            if n >= 0:
                pss = p + n
                (nData, _p) = lineOp.getBackKeyword(p+n)
                dataHeader = [dataType, nData]
                p = pss + 1
                pe = p
                (binary, p) = self.getBinaryData(dataType, nData, cont, p)
                if len(binary) != 0:
                    spId = obj + b"." + str(i).encode()
                    ascii += cont[ps:pe] + (b"\0" + spId + b"\0")
                    data.append([dataHeader, binary])
                    #errorチェック(bytes型の1文字チェック)
                    if not (cont[p] == b")" or cont[p] == ord(")")):
                        error = 1
                else:
                    ascii += cont[ps:pe]
            i += 1
        ascii += cont[ps:]
        return (ascii, data, error)

    #  reconstFuseDataAtMaxLinesAB
    #
    def reconstFuseDataAtMaxLinesAB(self, fileName, cont):
        """ nMaxLinesまで結合されたfuseDataを再構築して全データを取得"""
        funcDict = {
            #  class                関数名
            "regIOobject":          self.reconstFuseDataAtMaxLinesField,
            "labelList":            self.reconstFuseDataAtMaxLinesMultiList,
            "faceList":             self.reconstFuseDataAtMaxLinesMultiList,
            "faceCompactList":      self.reconstFuseDataAtMaxLinesMultiList,
            "refinementHistory":    self.reconstFuseDataAtMaxLinesMultiList,
            "vectorField":          self.reconstFuseDataAtMaxLinesMultiList,
            "cellSet":              self.reconstFuseDataAtMaxLinesMultiList,
            "faceSet":              self.reconstFuseDataAtMaxLinesMultiList,
            "pointSet":             self.reconstFuseDataAtMaxLinesMultiList,
            "polyBoundaryMesh":     self.reconstFuseDataAtMaxLinesField,
            "volScalarField":       self.reconstFuseDataAtMaxLinesField,
            "volVectorField":       self.reconstFuseDataAtMaxLinesField,
            "volTensorField":       self.reconstFuseDataAtMaxLinesField,
            "volSymmTensorField":   self.reconstFuseDataAtMaxLinesField,
            "surfaceScalarField":   self.reconstFuseDataAtMaxLinesField,
            "surfaceVectorField":   self.reconstFuseDataAtMaxLinesField,
            "pointScalarField":     self.reconstFuseDataAtMaxLinesField,
            "pointVectorField":     self.reconstFuseDataAtMaxLinesField
        }
        classType = self.getClassInFoamFile(cont)
        if classType in funcDict.keys():
            allCont = funcDict[classType](fileName, cont)
        else:
            #Listタイプのreconst（複数のListが存在する場合に設定）
            if (classType[-4:] == "List" or classType[-4:] == "list"):
                allCont = self.reconstFuseDataAtMaxLinesMultiList(fileName, cont)
            #その他のreconst（fieldタイプに設定　List<**> **(** **..); ）
            else:
                allCont = self.reconstFuseDataAtMaxLinesField(fileName, cont)
        return allCont

    #  reconstFuseDataAtMaxLinesMultiList
    #
    def reconstFuseDataAtMaxLinesMultiList(self, fileName, cont):
        """ fuseDataをreconstructして元のデータに戻す
        複数のlist形式に対応"""
        cont = cont.encode()
        #separateデータの読込
        (_ascii, data) = self.readSeparate(fileName)
        #data部までskipしてdataを入れ替え
        obj = self.getObjectInFoamFile(cont)
        lineOp = stringOp.strings(cont)
        p = lineOp.skipFoamFile(0)
        #p = skipFoamFileMem(cont, 0)
        allCont = cont[:p]
        n = 0
        while n >= 0 and p < len(cont):
            #data部までskip
            ps = p
            n = cont[p:].find("\n(")
            if n >= 0:
                p += n + 2
                pks = p
                allCont += cont[ps:p]
                n = cont[pks:].find("\n)")
                p += n + 1
                ple = p
                #keyを取得
                n = cont[pks:p].rfind("...")
                if n >= 0:
                    p = pks + n
                    n = cont[pks:p].rfind("...")
                    p = pks + n
                    key = cont[p:ple].split("...")[1]
                    #recObj = key.split(".")[0]
                    #dataNo = int(key.split(".")[1])
                    recObj = ".".join(key.split(".")[:-1])
                    dataNo = int(key.split(".")[-1])
                    #dataを入れ替え
                    if obj == recObj:
                        #自身のdataに入れ替え
                        allCont += data[dataNo][1]
                        p = ple
                    else:
                        #他のdataに入れ替え
                        newFile = "/".join(fileName.split("/")[:-1]) + "/" + recObj
                        (_newAscii, newData) = self.readSeparate(newFile)
                        allCont += newData[dataNo][1]
                        p = ple
                else:
                    #keyが無い場合は、そのまま取得
                    allCont += cont[pks:ple]
                    p = ple
        allCont += cont[p:]
        return allCont  

    #  reconstFuseDataAtMaxLinesField
    #
    def reconstFuseDataAtMaxLinesField(self, fileName, cont):
        """ fuseDataのcontをreconstructして元のデータに戻す"""

        def skipLineEnd(cont):
            """ 行末「);」までskip"""
            ne = cont.find(b";")
            n = cont[:ne].rfind(b")")
            return n

        #編集内容の文字コードを「bytes」に設定
        if type(cont) == str:
            cont = cont.encode()
        #separateデータを読込
        (_ascii, data) = self.readSeparate(fileName)
        #data部までskipしてdataを入れ替え
        obj = self.getObjectInFoamFile(cont)
        allCont = b""
        n = 0
        p = 0
        while n >= 0 and p < len(cont):
            #data部までskip
            ps = p
            #n = cont[p:].find(b" List<")
            n = self.findDataListPosition(cont[p:])
            if n >= 0:
                p += n
                (_dataType, nData) = self.getDataTypeNumData(cont, p)
                #データ数を確認
                if nData == b"0":
                    #データが存在しない場合
                    #  List<scalar> 0の場合
                    p += 1
                    allCont += cont[ps:p]
                else:
                    #データが存在する場合
                    n = cont[p:].find(b"(")
                    if n >= 0:
                        p += n +1
                        pks = p
                        allCont += cont[ps:p]
                        n = skipLineEnd(cont[pks:])
                        p += n
                        ple = p
                        #keyを取得
                        n = cont[pks:p].rfind(b"...")
                        if n >= 0:
                            p = pks + n
                            n = cont[pks:p].rfind(b"...")
                            p = pks + n
                            key = cont[p:ple].split(b"...")[1]
                            recObj = b".".join(key.split(b".")[:-1])
                            dataNo = int(key.split(b".")[-1])
                            #dataを入れ替え
                            if obj == recObj:
                                #自身のdataに入れ替え
                                allCont += data[dataNo][1]
                                p = ple
                                n = 0
                            else:
                                #他のdataに入れ替え
                                newFile = "/".join(fileName.split("/")[:-1]) + "/" + recObj
                                (_newAscii, newData) = self.readSeparate(newFile)
                                allCont += newData[dataNo][1]
                                p = ple
                                n = 0  
                        else:
                            #keyが無い場合は、そのまま取得
                            allCont += cont[pks:ple]
                            p = ple
                            n = 0
        allCont += cont[p:]
        return allCont

    #  getFuseDataAtMaxLinesAB
    #
    def getFuseDataAtMaxLinesAB(self, fileName, ascii, data):
        """ ascii binary dataをnMaxLinesまで結合して返す"""
        #obj = checkObject(ascii)
        if type(ascii) == str:
            ascii = ascii.encode()
        obj = self.getObjectInFoamFile(ascii)
        cont = b""
        #i = 0
        p = 0
        n = 0
        while n >= 0:
            #separatorまでskip
            ps = p
            n = ascii[p:].find(b"\0")
            if n >= 0:
                #separatorまでのasciiを取得
                p += n
                cont += ascii[ps:p]
                #次のseparatorまでskip
                p += 1
                pks = p
                n = ascii[p:].find(b"\0")
                if n >= 0:
                    #dataObjectを取得
                    p += n
                    key = ascii[pks:p]
                    recObj = b".".join(key.split(b".")[:-1])
                    dataNo = int(key.split(b".")[-1])
                    #dataObjectのdataを取得
                    if obj == recObj:
                        #自身のdataをnMaxLinesまで取得
                        sourceData = data[dataNo]
                    else:
                        #他のdataを取得
                        fileDir = "/".join(fileName.split("/")[:-1])
                        (_newAscii, newData) = self.readSeparate(fileDir + "/" + recObj)
                        sourceData = newData[dataNo]
                    #dataをsourceから取得し、set
                    if self.checkBinary(ascii) == "binary":
                        #binary
                        #smallData = self.getDataAtMaxLinesBinary(data[dataNo])
                        smallData = self.getDataAtMaxLinesBinary(sourceData)
                        cont += smallData + b"..." + key + b"...\n"
                    else:
                        #ascii
                        #smallData = self.getDataAtMaxLinesAscii(data[dataNo])
                        smallData = self.getDataAtMaxLinesAscii(sourceData)
                        if smallData[-3:] == b"...":
                            cont += smallData + key + b"...\n"
                        else:
                            cont += smallData
                    p += 1
        cont += ascii[p:]
        return cont

    #  getDataAtMaxLinesAscii
    #
    def getDataAtMaxLinesAscii(self, data):
        """ ascii data をnMxLinesまで取得"""

        def getCRLine(cont, p):
            """ CRまでの1行を取得"""
            ps = p
            n = cont[ps:].find(b"\n")
            if n >= 0:
                p += n + 1
                line = cont[ps:p]
            else:
                line = cont[ps:]
                p = len(cont)
            return (line, p)

        asciiData = data[1]
        if self.nMaxLines == -1:
            self.nMaxLines = len(asciiData)
        i = 0
        cont = b""
        p = 0
        while i <= self.nMaxLines and p < len(asciiData):
            [line , p] = getCRLine(asciiData, p)
            cont += line
            i += 1
        if i > self.nMaxLines:
            cont += b"..."
        return cont

    #  getDataAtMaxLinesBinary
    #
    def getDataAtMaxLinesBinary(self, data):
        """ binary dataをnMxLinesまで取得"""
        (nNum, nByte, valType) = self.getDataFormat(data[0][0])
        #binaryの読み込みの為、valTypeをbinaryに変換する
        valType = valType.encode()
        if self.nMaxLines == -1:
            convByte = len(data[1])
        else:
            convByte = nNum * nByte * self.nMaxLines

        if len(data[1]) > convByte:
            #nMaxLinesまで変換
            convData = data[1][:convByte]
        else:
            #全データを変換
            convData = data[1]
        n = int(len(convData)/nByte)        #valTypeのデータ数
        numData = struct.unpack(str(n).encode()+valType, convData)
        if nNum == 1:
            stLine = b""; edLine = b""
        else:
            stLine = b"("; edLine = b")"
        cont = b"\n"
        i = 0
        while i<len(numData):
            cont += stLine
            addCont = b""
            for num in numData[i:i+nNum]:
                addCont += str(num).encode() + b" "
            cont += addCont[:-1] + edLine + b"\n"
            i += nNum
        return cont

    #
    #  structBinData
    #
    def structBinData(self, data):
        """ binaryDataから数値を取得"""
        dataType = data[0][0]
        _nData = data[0][1]
        binData = data[1]
        #valTypeを取得
        (_nNum, nByte, valType) = self.getDataFormat(dataType)
        #数値変換する数
        num = int(len(binData) / nByte)
        #数値変換
        numData = struct.unpack(str(num)+valType, binData)
        return numData

    #  remakeDataFormat
    #
    def remakeDataFormat(self, ascii, form, newForm):
        """ dataのformatを書き換え"""
        lineOp = stringOp.strings(ascii)
        p = lineOp.getKeywordPointer(form, 0)
        if type(newForm) == str:
            newForm = newForm.encode()
        if p >= 0:
            newCont = ascii[:p] + newForm + ascii[p+len(form):]
            ascii = newCont
        return ascii

    def getClassInFoamFile(self, cont):
        """ FoamFileからclassを取得する"""
        lineOp = stringOp.strings(cont)
        (foamFile, _p) = lineOp.getKeywordContents("FoamFile", 0)
        #foamFile = foamFile.decode()
        lines = foamFile.split(b";")
        ans = b""
        for line in lines:
            words = line.split()
            if len(words) >= 2:
                if words[0] == b"class":
                    ans = words[1]
                    break
        return ans

    def getObjectInFoamFile(self, cont):
        """ FoamFileからobjectを取得する"""
        strOp = stringOp.strings(cont)
        (foamFile, _p) = strOp.getKeywordContents("FoamFile", 0)
        lines = foamFile.split(b";")
        ans = ""
        for line in lines:
            items = line.split()
            if len(items) >= 2:
                if items[0] == b"object":
                    ans = items[1]
                    break
        return ans

    def getDataTypeNumData(self, cont, p):
        """ dataTypeとnDataを取得"""
        block = cont[p:p+100]
        ps = block.find(b"<") + 1
        pe = block.find(b">")
        dataType = block[ps:pe]
        lineOp = stringOp.strings(block)
        (nData, p) = lineOp.getKeyword(pe+1)
        #[nData, p] = getKeywordMem(block, pe+1)
        return (dataType, nData)

    #  getBinaryData
    #
    def getBinaryData(self, binaryType, nData, cont, p):
        """ typeに応じてbinaryを取得"""
        (nNum, nByte, _valType) = self.getDataFormat(binaryType)
        pe = p + (nNum * int(nData) * nByte)
        dataCont = cont[p:pe]
        return (dataCont, pe)    

    #  getDataFormat
    #
    def getDataFormat(self, dataFormat):
        """ dataのformatを取得
        戻り値：[nNum, nByte, valType]"""
        dataDict = {
        #  type     nNum nByte valtype
        "scalar":     [1,  8,   "d"],
        "vector":     [3,  8,   "d"],
        "symmTensor": [6,  8,   "d"],
        "tensor":     [9,  8,   "d"],
        "label":      [1,  4,   "i"],
        "faceList":   [1,  4,   "i"],
        "bool":       [1,  1,   "b"]
        }
        if type(dataFormat) == bytes:
            dataFormat = dataFormat.decode()
        if dataFormat in dataDict.keys():
            ans = dataDict[dataFormat]
        else:
            ans = [0, 0, b""]
        return ans

    def confirmAscii(self, cont):
        """ dataがasciiかどうかチェック"""
        error = 0
        data = cont[:10]
        for chara in data:
            if chara < ord(chr(0x08)) or chara > ord(chr(0x80)):
                error = 1
                break
        return error

    def checkBinary(self, cont):
        """ データがbinaryかどうかをチェック"""
        lineOp = stringOp.strings(cont)
        (foamFile, _p) = lineOp.getKeywordContents("FoamFile", 0)
        foamFile = foamFile.decode()
        lines = foamFile.split(";")
        ans = "ascii"
        for line in lines:
            items = line.split()
            if len(items) >= 2:
                if items[0] == "format":
                    ans = items[1]
                    break
        return ans


#  -----------
#  strOp class
#  -----------
class strOp:
    """ 文字列操作。文字列は、str型で検索操作する。結果もstr型文字列が戻る。
    対象の文字列が変わる操作（replace、delete関連）は、対象の文字列も変える。

    Attribute:
        line (str)  :対象の文字列"""

    def __init__(self, line):
        if type(line) != str:
            line = line.decode()
        self.line = line

    #
    #  skipFoamFile
    #
    def skipFoamFile(self, p=0):
        """ FoamFileブロックをskipしたpointerを返す。
        FoamFileが存在しない場合は、skipしない。"""
        (_line, pp) = self.getKeywordContents("FoamFile", p)
        #FoamFileの有無チェック
        if pp >= len(self.line):
            #FoamFileが存在しない場合
            pp = p
        return pp

    #
    #  replace1line
    #
    def replace1line(self, ps, oldLine, newLine):
        """ line[ps:]以降を検索し、のoldLineをnewLineに入れ替える。
        置き換え後は、元のself.lineの内容も置き換える。
        戻り値:[newCont, p, keyword]。
        失敗した場合、keywordが「""」になる。
        pointerは、更新される。
        str.replace()で置き換える為、commentも含めて検索する。"""
        #置き換える
        newCont = self.line[:ps] + self.line[ps:].replace(oldLine, newLine, 1)
        #置き換え結果を確認
        p = newCont[ps:].find(newLine)
        if p >= 0:
            #置き換え成功
            p = ps + p + len(newLine)
            keyword = newLine
            self.line = newCont
        else:
            #置き換え失敗
            p = len(newCont)
            keyword = ""
        return (newCont, p, keyword)

    #
    #  replace1lineKeyword
    #
    def replace1lineKeyword(self, ps, newLine):
        """ line[ps:]を検索してnewLineに置き換える。検索は、newLineで使われているkeywordを検索する。
        置き換え後は、元のself.lineも置き換える。
        戻り値:(newCont, p, keyword)。
        失敗した場合、keywordが「""」になる。"""
        (keyword, _pp) = strOp(newLine).getKeyword(0)
        (lineCont, p, _kind) = self.get1line(ps)
        while lineCont != "":
            (tempKey, _pp) = strOp(lineCont).getKeyword(0)
            if tempKey == keyword:
                #pointerをlineCont(oldLine)の頭に戻す
                oldLine = lineCont
                pss = p - len(lineCont)
                break
            ps = p
            (lineCont, p, _kind) = self.get1line(ps)
        if lineCont == "":
            return ("", -1, "")
        else:
            #oldLineの頭から検索して置き換える
            (newCont, p, keyword) = self.replace1line(pss, oldLine, newLine)
            self.line = newCont
            return (newCont, p, keyword)

    #
    #  get1line
    #
    def get1line(self, p):
        """ 指定したpointerから1行を取得する
        pointerがeofの場合は、lineCont=""が戻る。
        コメント文もそのまま取得する。
        戻り値:(lineCont, p, kind)
        
        kindの種類:
            line        :「...;」
            keyword     :「keyword{}」
            include     :「# include "xxx.H"」
            includeEtc  :「# includeEtc "xxxx.H"」
            includeIfPresent :「# includeIfPresent "xxxx"」
            other       :「# xxxx yyyy」
            """
        lineCont = ""
        kind = ""
        if p >= len(self.line):
            return (lineCont, p, kind)
        
        chara, p = self.skipNoChara(p)      #pointerを空白以外に設定
        if p >= len(self.line) or chara == "":
            return (lineCont, p, kind)
        
        if chara == ";":
            lineCont = ";"
            p += 1
        elif chara == "#":
            #「#include ...」の取得
            sp = p
            p += 1                          #pointerを「#」の次に進める
            (chara, p) = self.skipNoChara(p)
            (name, p) = self.getKeyword(p)      #「include」取得
            (_keyword, p) = self.getKeyword(p)   #「"xxx.H"」取得
            lineCont = self.line[sp:p-1]
            if name == "include" or name == "includeEtc" or name == "includeIfPresent":
                kind = name
            else:
                kind = "other"
        else:
            sp = p
            (_keyword, p) = self.getKeyword(p)
            (chara, p) = self.skipNoChara(p)
            if chara == "{":
                #「keyword{}」の取得
                p -= 1                      #pointerを戻す
                (_cont, p) = self.getMiddlePair(p)
                lineCont = self.line[sp:p]
                kind = "keyword"
            else:
                #「...;」を取得
                while chara != ";" and chara != "":
                    if chara == "(":
                        p -= 1              #pointerを戻す
                        (dummy, p) = self.getSmallPair(p)
                    elif chara == "{":
                        p -= 1              #pointerを戻す
                        (dummy, p) = self.getMiddlePair(p)
                    (chara, p) = self.get1chara(p)
                if chara != "" and p < len(self.line):
                    if self.line[p] == ";":
                        p += 1
                lineCont = self.line[sp:p]
                kind = "line"
        return (lineCont, p, kind)

    #
    #  getKeywordPointer
    #
    def getKeywordPointer(self, keyword, p):
        """ 指定したkeywordのpointerを取得する。
        （高速検索している為、keywordがコメント中でも、検索するの注意！）
        pointerは、keywordの頭の文字位置。
        合致しない場合は、戻り値「-1」になる。"""
        if p >= 0 and p < len(self.line):
            #高速検索のため、string.find()を使用してポインタを設定
            flag = 0
            while (p >= 0) and (flag == 0):
                n = self.line[p:].find(keyword)
                if n >= 0:
                    #keyword前後の文字をチェック
                    ps = p + n - 1
                    if ps < 0:
                        stChara = self.line[ps+1]
                    else:
                        stChara = self.line[ps]
                    pe = p + n + len(keyword)
                    if pe >= len(self.line):
                        edChara = "\n"
                    else:
                        edChara = self.line[pe]
                    if ((self.checkChara(stChara) == False) and
                        (self.checkChara(edChara) == False)):
                        flag = 1
                        p = pe
                        break
                    else:
                        p = p + n + 1
                else:
                    break
            if flag == 0:
                return -1
            else:
                p = ps + 1
                return p
        return -1

    #
    #  getKeywordContents
    #
    def getKeywordContents(self, keyword, p):
        """ 指定したkeywordの内容を取得する。
        str.find()で検索するため、コメント内の文字列も検索する。
        {}内のデータを取得する。()内は取得しない。"""
        if p >= 0 and p < len(self.line):
            #高速検索のため、str.find()で検索する
            flag = 0
            while p >= 0 and flag == 0:
                n = self.line[p:].find(keyword)
                if n >= 0:
                    #keyword前後の文字をチェック
                    ps = p + n - 1
                    if ps < 0:
                        #stChara = self.line[ps+1]
                        stChara = "\n"
                    else:
                        stChara = self.line[ps]
                    pe = p + n + len(keyword)
                    if pe >= len(self.line):
                        edChara = "\n"
                    else:
                        edChara = self.line[pe]
                    if (self.checkChara(stChara) == False and
                        self.checkChara(edChara) == False):
                        #ループ脱出
                        flag = 1
                        p = pe
                        break
                    else:
                        p = p + n + 1
                else:
                    break
            if flag == 0:
                contents = ""
                p = len(self.line) + 1
            else:
                #内容を取得
                (contents, p) = self.getMiddlePair(p)
        else:
            contents = ""
            p = len(self.line) + 1
        return (contents, p)

    #
    #  getBackKeyword
    #
    def getBackKeyword(self, p):
        """ 遡ってkeywordを取得"""
        #文字をbackする
        chara = self.line[p]
        while self.checkChara(chara) == True:
            p -= 1
            chara = self.line[p]
        #空白をbackSkipする
        while self.checkChara(chara) == False:
            p -= 1
            chara = self.line[p]
        #文字をbackする
        while self.checkChara(chara) == True:
            p -= 1
            chara = self.line[p]
        #keywordを取得
        (keyword, p) = self.getKeyword(p)
        return (keyword, p)

    #
    #  getNextKeywordAndContents
    #
    def getNextKeywordAndContents(self, p):
        """ 次のkeywordと{}の内容を取得する
        戻り値：((keyword, contents), p)"""
        flag = 0
        chara = " "
        while flag == 0 and chara != "":
            (keyword, p) = self.getKeyword(p)
            (chara, p) = self.skipNoChara(p)
            if chara == "{":
                flag = 1
        p -= 1
        (contents, p) = self.getMiddlePair(p)
        return ((keyword, contents), p)

    #
    #  getMiddlePair
    #
    def getMiddlePair(self, p):
        """ {}内の文字列を取得し、その内容とpointerを返す。
        検索はstr.find()で検索する。"""
        pair = ""
        while pair != "{" and p < len(self.line):
            #"{" or "}" を探す
            (pair, p) = self.getNextMiddlePair(p)
        #p += 1
        ap = p
        count  = 1
        while count > 0 and p < len(self.line):
            (pair, p) = self.getNextMiddlePair(p)
            if pair == "{":
                count += 1
            elif pair == "}":
                count -= 1
        contents = self.line[ap:p-1]
        return (contents, p)
    
    #
    #  getNextMiddlePair
    #
    def getNextMiddlePair(self, p):
        """ pointerに近い"{" or "}" 文字を検索し、文字種とpointerを返す。
        検索はstr.finf()で検索する。"""
        lp = self.line[p:].find("{")
        if lp < 0:
            lp = len(self.line)
        rp = self.line[p:].find("}")
        if rp < 0:
            rp = len(self.line)
        if lp < rp:
            ans = "{"
            ap = p + lp + 1
        elif rp < lp:
            ans = "}"
            ap = p + rp + 1
        else:
            ans = ""
            ap = len(self.line)
        return (ans, ap)
    
    #
    #  getSmallPair
    #
    def getSmallPair(self, p):
        """ ()内データを取得し返す。"""
        pair = ""
        while pair != "(" and p < len(self.line):
            #"(" or ")" を探す
            (pair, p) = self.getNextSmallPair(p)
        ap = p
        count  = 1
        while count > 0 and p < len(self.line):
            (pair, p) = self.getNextSmallPair(p)
            if pair == "(":
                count += 1
            elif pair == ")":
                count -= 1
        contents = self.line[ap:p-1]
        return (contents, p)

    #
    #  getNextSmallPair
    #
    def getNextSmallPair(self, p):
        """ pointerに近い"(" or ")" 文字を検索し、文字種とpointerを返す。
        検索はstr.finf()で検索する。"""
        lp = self.line[p:].find("(")
        if lp < 0:
            lp = len(self.line)
        rp = self.line[p:].find(")")
        if rp < 0:
            rp = len(self.line)
        if lp < rp:
            ans = "("
            ap = p + lp + 1
        elif rp < lp:
            ans = ")"
            ap = p + rp + 1
        else:
            ans = ""
            ap = len(self.line)
        return (ans, ap)

    #
    #  getKeyword
    #
    def getKeyword(self, p):
        """ keywordを取得して返す"""
        (chara, p) = self.get1chara(p)
        #文字まで進む
        while (self.checkChara(chara) != True and chara != ""):
            (chara, p) = self.get1chara(p)
        #keyword取得
        keyword = ""
        if chara == '"':
            #""で囲まれる場合
            keyword += chara
            (chara, p) = self.get1chara(p)
            while (chara != '"') and (chara != ""):
                keyword += chara
                (chara, p) = self.get1chara(p)
            #最後の"をkeywordに追加しポインタを1ヶ進める
            keyword += chara
            p += 1
        else:
            #通常の文字列
            while ((self.checkChara(chara) == True) and (chara != "")
                        and p < len(self.line)):
                keyword += chara
                (chara, p) = self.get1chara(p)
            if chara != "":
                p -= 1      #1文字読みすぎているので、ポインタを1ヶ戻す
        return (keyword, p)

    #
    #  get1chara
    #
    def get1chara(self, p):
        """ 1文字を取得"""
        if p >= len(self.line):
            return ("", p)

        (chara, p) = self.get1c(p)
        if chara == "/":
            ps = p
            (nextChara, p) = self.get1c(p)
            if nextChara == "/":
                p = self.skipUntilCR(p)      #行末までスキップ
                (chara, p) = self.get1c(p)
            elif nextChara == "*":
                p= self.skipUntilSS(p)       #"*/"までスキップ
                (chara, p) = self.get1c(p)
            else:
                p = ps                       #pointerを戻す
        return (chara, p)

    #次の行頭までスキップ
    def skipNextCR(self, p):
        """ 次の行頭までskipする"""
        if p >= 0 and p < len(self.line):
            chara, p = self.get1c(p)
            while (chara != "\n") and (chara != ""):
                [chara, p] = self.get1c(p)
            if chara == "":
                p = len(self.line)    
        else:
            p = len(self.line)
        return p

    #前の行頭までスキップ
    def skipBackCR(self, p):
        """ 前の行頭までskipする。"""
        if p >= 0:
            if p >= len(self.line):
                p = len(self.line) - 1
            if self.line[p] == "\n":
                p -= 1
            chara = " "
            while (chara != "\n") and (p >= 0):
                chara = self.line[p]
                p -= 1
            if p < 0:
                p = 0
            else:
                p += 2      #2文字戻す
        else:
            p = 0
        return p

    #行末までスキップ
    def skipUntilCR(self, p):
        """ 行末まで(CRの直前)skipする"""
        if p>=0 and p<len(self.line):
            (chara, p) = self.get1c(p)
            while (chara != "\n") and (chara != ""):
                [chara, p] = self.get1c(p)
            if chara == "\n":
                p-=1                            #1文字戻す
        else:
            p=len(self.line)
        return p

    #*/までスキップ
    def skipUntilSS(self, p):
        """ 「*/」までskipする"""
        if p >= 0 and p < len(self.line)-1:
            charas = self.line[p:p+2]; p+=1
            while (charas != "*/") and (p < len(self.line)-1):
                charas = self.line[p:p+2]; p+=1
            if charas == "*/":
                p+=1
            else:
                p=len(self.line)
        else:
            p=len(self.line)
        return p

    #spスキップ
    def skipSP(self, p):
        """ 空白をskipする"""
        if p >= 0 and p < len(self.line)-1:
            while self.line[p] == " ":
                p += 1
        else:
            p=len(self.line)
        return p

    #1文字取得
    def get1c(self, p):
        """ 1文字取得"""
        if p >= 0 and p < len(self.line):
            chara = self.line[p]; p += 1
        else:
            chara = ""; p = len(self.line)
        return (chara, p)

    #
    #  skipNoChara
    #
    def skipNoChara(self, p):
        """ 空白以外の文字までskipする
        ﾎﾟｲﾝﾀは空白以外の文字位置を返す。charaはﾎﾟｲﾝﾀが示す文字を返す"""
        if p >= len(self.line):
            return ("", p)
        (chara, p) = self.get1chara(p)
        while (chara == " " or chara == "\n" or chara == "\t"):
            (chara, p) = self.get1chara(p)
        p -= 1
        return (chara, p)

    #
    #  checkChara
    #
    def checkChara(self, chara):
        """ keywordとして使える文字のチェック"""
        # if type(chara) == int:
        #     chara = chr(chara).encode()
        if chara > " " and chara <= "z":
            if chara == "{" or chara == "}":
                flag = False
            elif chara == "(" or chara == ")":
                flag = False
            elif chara == ";":
                flag = False
            else:
                flag = True
        else:
            flag = False
        return flag

    #
    #  skipIntoPair
    #
    def skipIntoPair(self, p, pair="{"):
        """ 現在のpointerからpair文字（default:"{"）までskipしてpointerを返す """
        if p >= 0 and p < len(self.line)-1:
            #asciiコードで比較
            while (self.line[p] != pair) and (p < len(self.line)-1):
                p += 1
            p += 1
        else:
            p=len(self.line)
        return p

    #
    #  deleteCommentsNullLines
    #
    def deleteCommentsNullLines(self):
        """ コメント、空行の削除。削除後は、元のself.lineも置き換わる"""
        contents = self.deleteComments()
        contents = self.deleteNullLines()
        return contents

    #
    #  deleteComments
    #
    def deleteComments(self):
        """ commentを削除して返す。削除後は、元のself.lineも置き換わる"""
        #/*を削除
        nocomm = ""
        st = 0
        loop = True
        while loop == True:
            # "/" を検索
            pp = self.line[st:].find("/")
            if pp >= 0:
                # "/*" を検索
                if self.line[st:st+2] == "/*":
                    commS = st +  pp
                    nocomm += self.line[st:pp]
                    ped = self.line[commS:].find("*/")
                    if ped >= 0:
                        st = commS + ped + len("*/")
                    else:
                        break
                # "//" を検索
                elif self.line[st:st+2] == "//":
                    commS = st + pp
                    nocomm += self.line[st:commS]
                    ped = self.line[commS:].find("\n")
                    if ped >= 0:
                        st = commS + ped
                    else:
                        break
                else:
                    #pointerを更新
                    nocomm += self.line[st]
                    st += 1
            else:
                nocomm += self.line[st:]
                break
        self.line = nocomm
        return nocomm

    #
    #  deleteNullLines
    #
    def deleteNullLines(self):
        """ 空行を削除する。削除後は、元のself.lineも置き換わる"""
        lines = self.line.split("\n")
        newLines = []
        for line in lines:
            if len(line.split()) != 0:
                newLines.append(line)
        newCont = "\n".join(newLines) + "\n"
        self.line = newCont
        return newCont

    #
    #  compressSPCR
    #
    def compressSPCR(self):
        """ 余分なSPとCRを削除する。削除後は、元のself.lineも置き換える"""
        lines = self.line.split("\n")
        ans = []
        for line in lines:
            if line != "":
                words = line.split()
                ans.append(" ".join(words))
        ansR = "\n".join(ans)
        if len(ansR) > 0:
            if ansR[-1] == "\n":
                ansR = ansR[:-1]
        else:
            ansR = "\n"
        self.line = ansR        
        return ansR
        

#  ------------
#  server class
#  ------------
class server:
    """ 特定のserverの操作を行う。
    
    Attribute:
        serverName (str)    :server名
        sshfsDict (dict)    :serverの詳細データ
        sshDict (dict)      :sshConfigの内容
        sshfsFile (str)     :sshfs_dataのfilePath
        sshConfigFile (str) :sshConfigのfilePath
    
    sshfsdictの内容:
        hostName    :sshConfigで定義しているhost名
        hostDir     :mountする時のhost側Dir
        mountPoint  :mountする時のlocal側Dir
        login       :loginサーバ名(hostNameと同じ)
        setEnviron  :login時の環境設定内容
        sshHeader   :sshのheader内容
        sshPassFile :秘密鍵を使わずに接続する時のpasswordファイル名
    
    sshDictの内容
        url     :hostのurl
        user    :user名
    """
    def __init__(self, serverName):
        self.serverName = serverName
        self.sshfsFile = os.getenv("TreeFoamUserPath") + "/data/sshfs_data"
        self.sshConfigFile = os.getenv("HOME") + "/.ssh/config"
        #hostの設定を読み込む
        self.sshfsDict = self.readSshfsData()
        hostName = self.sshfsDict["hostName"]
        self.sshDict = self.readSshConfig(hostName)

    #
    #  readSshConfig
    def readSshConfig(self, host):
        """ .ssh/configの読み込み。
        urlとuserを取得"""
        sshDict = {
            "url": "",
            "user": ""
            }
        fileName = self.sshConfigFile
        f = open(fileName); lines = f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        flag = 0; edFlag = 0
        for line in lines:
            if line[:len("Host ")].lower() == "Host ".lower():
                words = line.split()
                if words[1] == host:
                    flag = 1
                else:
                    if flag == 1:
                        edFlag = 1
                    flag = 0
            else:
                if flag == 1:
                    words = line.split()
                    if len(words) >= 2:
                        if words[0].lower() == "HostName".lower():
                            sshDict["url"] = words[1]
                        elif words[0].lower() == "User".lower():
                            sshDict["user"] = words[1]
            if edFlag == 1:
                break
        return sshDict

    #
    #  readSshfsData
    def readSshfsData(self):
        """ 該当serverのsshfsData（接続内容）を取得する"""
        sshfsDict = {
            "hostName": "",
            "hostDir": "",
            "mountPoint": "",
            "login": "",
            "setEnviron": "",
            "sshHeader": "",
            "sshPassFile": ""
            }
        fileName = self.sshfsFile
        f=open(fileName); lines=f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        #必要なserverName部を取得
        newLines = []
        flag = 0
        for line in lines:
            if line[:len("Host ")] == "Host ":
                words = line.split()
                if words[1] == self.serverName:
                    flag = 1
                else:
                    flag = 0
            elif flag == 1:
                newLines.append(line)
        #serverNameの設定値を取得
        setEnv = 0
        for i in range(len(newLines)):
            line = newLines[i]
            words = line.split()
            if line[0] != " ":
                if len(words) > 1:
                    if line[:len("HostName")].lower() == "HostName".lower():
                        sshfsDict["hostName"] = words[1]
                    elif line[:len("HostDir")].lower() == "HostDir".lower():
                        sshfsDict["hostDir"] = words[1]
                    elif line[:len("MountPoint")].lower() == "MountPoint".lower():
                        sshfsDict["mountPoint"] = words[1]
                    elif line[:len("sshHeader")].lower() == "sshHeader".lower():
                        sshfsDict["sshHeader"] = " ".join(words[1:]) + " "    #空白1文字を追加
                    elif line[:len("sshPassFile")].lower() == "sshPassFile".lower():
                        sshfsDict["sshPassFile"] = words[1]
                if line[:len("login")].lower() == "login".lower():
                    words = lines[i+1].split()
                    sshfsDict["login"] = " ".join(words) 
                elif line[:len("setEnviron")].lower() == "setEnviron".lower():
                    setEnv = 1
                else:
                    setEnv = 0
            else:
                if setEnv == 1:
                    if len(words) > 0:
                        sshfsDict["setEnviron"] += " ".join(words) + "\n"
        #loginを設定
        if sshfsDict["login"] == "":
            sshfsDict["login"] = "ssh " + sshfsDict["hostName"]
        #環境変数をセット、fullpathで設定（sshPassFile）
        sshPassFile = sshfsDict["sshPassFile"]
        sshPassFile = os.path.expanduser(sshPassFile)
        sshPassFile = os.path.abspath(sshPassFile)
        os.environ["sshPassFile"] = sshPassFile
        return sshfsDict

    #
    #  writeSshfsData
    def writeSshfsData(self):
        """ 該当serverのsshfsData（接続内容）を書き込む"""
        fileName = self.sshfsFile
        f=open(fileName); lines=f.readlines(); f.close()
        #必要なserverName部を検索
        #  serverNameの前側linesを取得
        uLines = []
        ip = 0
        for i in range(len(lines)):
            line = lines[i]
            if line[:len("Host ")] == "Host ":
                words = line.split()
                if words[1] == self.serverName:
                    uLines = lines[:i]
                    ip = i
                    break
        #  serverNameの後側linesを取得
        dLines = []
        ie = len(lines)
        i = ip + 1
        while i < len(lines):
            line = lines[i]
            if line[:len("Host ")] == "Host ":
                dLines = lines[i:]
                ie = i
                break
            i += 1
        #  serverName部を取得
        mLines = lines[ip:ie]
        #severName部に上書き
        flags = [0,0,0,0]
        for i in range(len(mLines)):
            line = mLines[i]
            words = line.split()
            if len(words) > 0:
                if words[0] == "HostName":
                    flags[0] = 1
                    mLines[i] = "HostName " + self.sshfsDict["hostName"] + "\n"
                elif words[0] == "HostDir":
                    flags[1] = 1
                    mLines[i] = "HostDir " + self.sshfsDict["hostDir"] + "\n"
                elif words[0] == "MountPoint":
                    flags[2] = 1
                    mLines[i] = "MountPoint " + self.sshfsDict["mountPoint"] + "\n"
                elif words[0] == "sshHeader":
                    flags[3] = 1
                    mLines[i] = "sshHeader " + self.sshfsDict["sshHeader"] + "\n"
        if flags[0] == 0 and self.sshfsDict["hostName"] != "":
            mLines.append("HostName " + self.sshfsDict["hostName"] + "\n")
        if flags[1] == 0 and self.sshfsDict["hostDir"] != "":
            mLines.append("HostDir " + self.sshfsDict["hostname"] + "\n")
        if flags[2] == 0 and self.sshfsDict["mountPoint"] != "":
            mLines.append("MountPoint " + self.sshfsDict["mountpoint"] + "\n")
        if flags[3] == 0 and self.sshfsDict["sshHeader"] != "":
            mLines.append("sshHeader " + self.sshfsDict["sshHeader"] + "\n")
        #setEnvironを設定
        #  setEnvironの内容を削除
        newLines = []
        si = 0; flag = 0
        for i in range(len(mLines)):
            line = mLines[i]
            words = line.split()
            if len(words) > 0:
                if words[0] == "setEnviron":
                    newLines.append(line)
                    flag = 1
                    si = i + 1
                else:
                    if line[0] == " " and flag == 1:
                        pass
                    else:
                        newLines.append(line)
            else:
                newLines.append(line)
        #  setEnvironを設定
        if flag == 1:
            envLines = self.sshfsDict["setEnviron"].split("\n")
            if envLines[-1] == "":
                envLines = envLines[:-1]
            for i in range(len(envLines)):
                envLines[i] = "    " + envLines[i] + "\n"
            mLines = newLines[:si] + envLines + newLines[si:]
        else:
            if self.sshfsDict["setEnviron"] != "":
                envLines = self.sshfsDict["setenviron"].split("\n")
                if envLines[-1] == "":
                    envLines = envLines[:-1]
                for i in range(len(envLines)):
                    envLines[i] = "    " + envLines[i] + "\n"
                mLines += envLines
        #  linesを再作成して保存
        lines = uLines + mLines + dLines
        f = open(fileName, "w"); f.writelines(lines); f.close()

    #
    #  mount
    def mount(self, mountDir):
        self.sshfsDict["mountPoint"] = mountDir
        self.writeSshfsData()
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        hostDir = self.sshfsDict["hostDir"]
        sshHeader = self.sshfsDict["sshHeader"]
        comm = "sshfs " + userName + "@" + hostName + ":" + hostDir
        comm += " " + mountDir
        commLine = sshHeader + comm
        print(commLine)
        (stat, _res, err) = run().commandReturnCont(commLine)
        return (stat, err)

    #
    #  sendSshCommand
    def sendSshCommand(self, comm):
        """ serverに対し、sshでコマンドを送る"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        #hostDir = self.sshfsDict["hostDir"]
        #mountDir = self.sshfsDict["mountPoint"]
        sshHeader = self.sshfsDict["sshHeader"]
        commLine = "ssh " + userName + "@" + hostName + " '" + comm + "'"
        commLine = sshHeader + commLine
        print(commLine)
        run().command(commLine)

    #
    #  scpCommandNoTerm
    def scpCommandNoTerm(self, source, paste):
        """ 端末を開かずにscpを実行する。
        
        Args:
            source (str)    :転送するfile名
            paste (str)     :転送先"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        #hostDir = self.sshfsDict["hostDir"]
        #mountDir = self.sshfsDict["mountPoint"]
        sshHeader = self.sshfsDict["sshHeader"]
        header = userName + "@" + hostName + ":"
        source = self.checkServerDir(source)
        paste = self.checkServerDir(paste)
        #errorCheck
        if source[:len(header)] == header and paste[:len(header)] == header:
            return ["ERROR", "unmatch server directory"]
        elif source[:len(header)] != header and paste[:len(header)] != header:
            return ["ERROR", "unmatch server directory"]
        #実行
        commLine = "scp -Cr " + source + " " + paste
        commLine = sshHeader + commLine
        print(commLine)
        run().command(commLine)

    #
    #  moveRunPython
    def moveRunPython(self, scriptPath, args):
        """ python3のスクリプトをserverに転送し、そのスクリプトを実行する。
        実行は、python2でスクリプトを実行する。
        引数（args）は、文字列として入力する。"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        host = userName + "@" + hostName
        comm = "scp " + scriptPath + " " + host + ":~/runPython.py"
        comm += "; ssh " + host + " python runPython.py " + args
        run().command(comm)

    #
    #  moveRunPythonReturnCont
    def moveRunPythonReturnCont(self, scriptPath, args):
        """ python3のスクリプトをserverに転送し、そのスクリプトを実行し、結果を返す。
        実行は、python2でスクリプトを実行する。
        引数（args）は、文字列として入力する。"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        host = userName + "@" + hostName
        comm = "scp " + scriptPath + " " + host + ":~/runPython.py"
        comm += "; ssh " + host + " python runPython.py " + args
        result = run().commandReturnCont(comm)
        return result

    #
    #  runLoginTerm
    def runLoginTerm(self):
        """ login端末を起動"""
        sshHeader = self.sshfsDict["sshHeader"]
        login = self.sshfsDict["login"]
        #端末を起動
        login = sshHeader + login
        fileName = os.getenv("TreeFoamUserPath") + "/temp/loginTerm"
        line = "#!/bin/bash\n"
        line += login + "\n"                                            #login
        f=open(fileName, "w"); f.write(line); f.close()
        run().command("chmod a+x " + fileName)
        comm = foamTerminalRun + " " + fileName
        run().command(comm)

    #
    #  checkServerDir
    def checkServerDir(self, dirName):
        """ serverDirかどうか調べ、serverの場合は、userName等を付加する。"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        hostDir = self.sshfsDict["hostDir"]
        mountDir = self.sshfsDict["mountPoint"]
        header = userName + "@" + hostName + ":"
        if dirName[:len(mountDir)] == mountDir:
            dirName = header + hostDir + dirName[len(mountDir):]
        elif dirName[:len(hostDir)] == hostDir:
            dirName = header + dirName
        elif dirName[:len("~/TreeFoamServer/")] == "~/TreeFoamServer/":
            dirName = header + dirName
        elif dirName[0] == "~":
            dirName = header + dirName
        return dirName

    #
    #  deleteBashComment
    def deleteBashComment(self, lines):
        """ lines内からコメント部「#」を削除する"""
        newLines = []
        for line in lines:
            if line[0] != "#":
                n = line.find("#")
                if n < 0:
                    newLines.append(line)
                else:
                    newLine = line[:n] + "\n"
                    newLines.append(newLine)
        return newLines


#  ------------
#  servers class
#  ------------
class servers:
    """ 登録してある全serverの内容を取得する。

    Attribute:
        sshfsFile (str)     :sshfs_dataのfilePath
        sshConfigFile (str) :sshConfigのfilePath
    """
    def __init__(self):
        self.sshfsFile = os.getenv("TreeFoamUserPath") + "/data/sshfs_data"
        self.sshConfigFile = os.getenv("HOME") + "/.ssh/config"

    #
    #  getAllServerNames
    def getAllServerNames(self):
        """ TreeFoamに登録されているserverを取得する。
        $TreeFoamUserPath/data/sshfs_data から取得
        
        Returns:
            names (list(str))       :全serverName
            mountNames (list(str))  :mountされているserver名
        """
        fileName = self.sshfsFile
        f=open(fileName); lines=f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        names = []; hostName = ""; mountNames = []
        for line in lines:
            words = line.split()
            if len(words) >= 1:
                if words[0] == "Host":
                    hostName = words[1]
                    names.append(hostName)
                elif words[0] == "MountPoint":
                    if len(words) > 1:
                        mountNames.append(hostName)
        names.sort()
        mountNames.sort()
        return (names, mountNames)
        
    #
    #  getMountedServers
    def getMountedServers(self):
        """ マウントされているserver(server, host, mountPoint)を検索し返す。
        mountPointが設定されているserver, hostNameを取得。
        
        Returns:
            mountServers (list(str, str, str)) :[[serverName, hostName, mountPoint]]
        """
        fileName = self.sshfsFile
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        mountServers = []
        serverName = ""; hostName = ""; mountPoint = ""
        for line in lines:
            words = line.split()
            if len(words) >= 1:
                if words[0] == "Host":
                    if hostName != "" and mountPoint != "":
                        mountServers.append([serverName, hostName, mountPoint]) 
                    serverName = words[1]
                    hostName = ""; mountPoint = ""
                elif words[0] == "HostName":
                    hostName = words[1]
                elif words[0] == "MountPoint":
                    if len(words) > 1:
                        mountPoint = words[1]
        if hostName != "" and mountPoint != "":
            mountServers.append([serverName, hostName, mountPoint])
        return mountServers

    #
    #  getServerName
    def getServerName(self, getDir):
        """ directoryからserverNameを取得する。
        
        Attribute:
            getServer (str)  :Dir系列にmountされているserver名を取得する"""
        mountServers = self.getMountedServers()
        getServer = ""
        getDirList = getDir.split("/")
        for (serverName, _hostName, mountP) in mountServers:
            mountList = mountP.split("/")
            if mountList == getDirList[:len(mountList)]:
                getServer = serverName
                break
        return getServer

    #
    #  deleteBashComment
    def deleteBashComment(self, lines):
        """ lines内からコメント部「#」を削除する"""
        newLines = []
        for line in lines:
            if line[0] != "#":
                n = line.find("#")
                if n < 0:
                    newLines.append(line)
                else:
                    newLine = line[:n] + "\n"
                    newLines.append(newLine)
        return newLines


#  ---------------------
#  getFoamContents class
#  ---------------------
class getFoamContents:

    def __init__(self, caseDir=os.getcwd()):
        self.caseDir = caseDir
    
    def defaultSurfaceFeatureExtractDict(self):
        """ デフォルトのsurfaceFeatureExtructDictを返す"""
        cont = "xxxx.stl         //stl file. if many files, copy this area at each stl file.\n"
        cont += "{\n"
        cont += "    // How to obtain raw features (extractFromFile || extractFromSurface)\n"
        cont += "    extractionMethod    extractFromSurface;\n"
        cont += "\n"
        cont += "    extractFromSurfaceCoeffs\n"
        cont += "    {\n"
        cont += "        // Mark edges whose adjacent surface normals are at an angle less\n"
        cont += "        // than includedAngle as features\n"
        cont += "        // - 0  : selects no edges\n"
        cont += "        // - 180: selects all edges\n"
        cont += "        includedAngle   150;\n"
        cont += "    }\n"
        cont += "\n"
        cont += "    subsetFeatures\n"
        cont += "    {\n"
        cont += "        // Keep nonManifold edges (edges with >2 connected faces)\n"
        cont += "        nonManifoldEdges       no;\n"
        cont += "\n"
        cont += "        // Keep open edges (edges with 1 connected face)\n"
        cont += "        openEdges              yes;\n"
        cont += "    }\n"
        cont += "\n"
        cont += "    // Write options\n"
        cont += "\n"
        cont += "    //   Write features to obj format for postprocessing\n"
        cont += "    writeObj               yes;\n"
        cont += "}\n"
        return cont

    def defaultSurfaceFeaturesDict(self):
        cont  = "surface1\n"
        cont += "{\n"
        cont += "    surfaces\n"
        cont += "    (\n"
        cont += '        "xxxx.stl"\n'
        cont += "    );\n"
        cont += "\n"
        cont += "    // Identify a feature when angle between faces < includedAngle\n"
        cont += "    includedAngle   150;\n"
        cont += "\n"
        cont += "    // Do not mark region edges\n"
        cont += "    geometricTestOnly       yes;\n"
        cont += "\n"
        cont += "    // Write options\n"
        cont += "\n"
        cont += "    // Write features to obj format for postprocessing\n"
        cont += "    writeObj                yes;\n"
        cont += "    verboseObj              no;\n"
        cont += "}\n"
        return cont

    def makeDefaultmachines(self):
        """ defaultのmachinesファイルを作成する"""
        cont  = "#  discript subDomain name and number of CPUs.\n"
        cont += "#  subDomain name is discripted host name in host of '/etc/hosts'.\n"
        cont += "#\n"
        cont += "# Example:\n"
        cont += "#localhost cpu=4\n"
        cont += "#host1 cpu=2\n"
        cont += "#host2\n"
        cont += "\n"
        cont += "localhost cpu=4\n"
        return cont

    def defaultSetFieldsDict(self, OFversion):
        """ defaultのsetFieldsDictを返す"""
        numVer = getNumVersion(OFversion)
        if numVer >= 13.0:
            #""" OF-13用のsetFirldsDict"""
            a = "defaultValues\n"
            a += "{\n"
            a += "    xxxxx     0;                 // scalar\n"
            a += "    yyyyy     (0 0 0);           // vector\n"
            a += "}\n"
            a += "\n"
            a += "zones\n"
            a += "{\n"
            a += "    water                         //cellZoneName\n"
            a += "    {\n"
            a += "        type          lookup;\n"
            a += "        zoneType      cell;\n"
            a += "        values\n"
            a += "        {\n"
            a += "            xxxxx     10;\n"
            a += "            yyyyy     (1 1 1);\n"
            a += "        }\n"
            a += "    }\n"
            a += "\n"
            a += "    waterBox                      //任意名\n"
            a += "    {\n"
            a += "        type          box;\n"
            a += "        zoneType      cell;\n"
            a += "        box           (0 0 0) (1 1 1);\n"
            a += "        values\n"
            a += "        {\n"
            a += "            xxxxx     1;\n"
            a += "            yyyyy     (2 2 2);\n"
            a += "        }\n"
            a += "    }\n"
            a += "}\n"
            return a
        else:
            #OF-12以下、v、ext形式のsetFields
            a = "defaultFieldValues\n"
            a += "(\n"
            a += "    volScalarFieldValue xxxxx 0                 // scalar\n"
            a += "    volVectorFieldValue yyyyy (0 0 0)           // vector\n"
            a += ");\n"
            a += "\n"
            a += "regions\n"
            a += "(\n"
            a += "    cellToCell\n"
            a += "    {\n"
            a += "        set cellset_name1;                     //cellSetName\n"
            a += "        fieldValues\n"
            a += "        (\n"
            a += "            volScalarFieldValue xxxxx 10\n"
            a += "            volVcalarFieldValue yyyyy (1 1 1)\n"
            a += "        );\n"
            a += "    }\n"
            a += "\n"
            a += "    boxToCell\n"
            a += "    {\n"
            a += "        box (0 0 0) (1 1 1);                     //cellSetName\n"
            a += "        fieldValues\n"
            a += "        (\n"
            a += "            volScalarFieldValue xxxxx 1\n"
            a += "            volVcalarFieldValue yyyyy (2 2 2)\n"
            a += "        );\n"
            a += "    }\n"
            a += ");\n"
            return a

    def makeFoamHeaderVersion(self, version):
        """ OFversionを書き込んだHeaderを返す"""

        a  = "/*--------------------------------*- C++ -*----------------------------------*\\\n"
        a += "| =========                 |                                                 |\n"
        a += "| \\\\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox           |\n"
        a += "|  \\\\    /   O peration     | Version:  " + version + "                                 |\n"
        a += "|   \\\\  /    A nd           | Web:      www.OpenFOAM.org                      |\n"
        a += "|    \\\\/     M anipulation  |                                                 |\n"
        a += "\\*---------------------------------------------------------------------------*/\n"
        return a

    def makeFoamFile(self, version, format, className, location, objectName):
        """ foamFileを作成し、返す"""
        if version == "":
            version = "2.0"
        a=  "FoamFile\n"
        a=a+"{\n"
        a=a+"    version     " + version + ";\n"
        a=a+"    format      " + format + ";\n"
        a=a+"    class       " + className + ";\n"
        a=a+'    location    "' + location + '";\n'
        a=a+"    object      " + objectName + ";\n"
        a=a+"}\n"
        a=a+"// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //\n"
        a=a+"\n"
        return a

    def makeFoamFooter(self):
        """ footerを作成し返す"""
        a="\n// ************************************************************************* //\n"
        return a

    def defaultDictPathInDataFolder(self, OFversion, dictFile):
        """ OFversionに応じたdictFileをTreeFoam/dataフォルダ内から取得する。
        取得できなかった場合、「""」を返す。

        Args:
            OFversion (str) :OpenFOAMのバージョン
            dictFile (str)  :dictFile名(例:system/decomposeParDict)
        Returns:
            path (str)  :dictFileのpathを返す"""
        dictFolder = self.getOFversionFolder(OFversion)
        dictDir = dictFolder + "/" + dictFile
        if os.path.exists(dictDir) == True:
            return dictDir
        else:
            return ""

    def getOFversionFolder(self, OFversion):
        """ OFversionのfolder名を取得する。
        このfolder内にOFversionに応じた各種Dictが保存されている。
        
        Args:
            OFversion (str) :OpenFOAMのバージョン
        Returns:
            folderDir (str) :実在する最新のfolderDir(fullPath)"""
        verDir = os.getenv("TreeFoamPath") + "/data/OFDict"
        folders = glob.glob(verDir + "/*")
        vers = []
        for folder in folders:
            name = folder.split(os.sep)[-1]
            ver = "-".join(name.split("-")[1:])
            intVer = getIntVersion(ver)
            if intVer < 0:
                intVer = 9999
            vers.append([intVer, ver])
        vers.sort()
        #近いversionを取得
        intOF = getIntVersion(OFversion)
        if intOF < 0:
            intOF = 9999
        currVer = vers[0]
        for [intVer, ver] in vers:
            #OF-v, OF-extの場合は、文字をそのままチェック
            if intOF == 9999:
                if ver <= OFversion:    #文字同士をチェック
                    currVer = ver
            #OF-10の場合、数値でチェック
            elif intVer <= intOF:       #数値をチェック
                currVer = ver
            else:
                break
        folderDir = verDir + "/OF-" + currVer
        return folderDir

    def cellZoneNames(self, polyMeshDir):
        """ cellZonesファイルからcellZone名を取得する
        
        Args:
            polyMeshDir (str)   :polyMeshDir(fullPath)
        returns:
            cellZoneNames (list(str))   :cellZone名のlist"""
        #fileが無ければ直ぐに戻る
        fileName = polyMeshDir + "/cellZones"
        if len(glob.glob(fileName)) == 0:
            fileName += ".gz"
            if len(glob.glob(fileName)) == 0:
                return []
        (contents, _data) = foamFile().readSeparate(fileName)
        contentsOp = stringOp.strings(contents)
        contents = contentsOp.deleteCommentsNullLines()
        #formFileをskip
        p = contentsOp.skipFoamFile()
        #zone名の取得
        zones = []
        (keyword, p) = contentsOp.getKeyword(p)
        try:
            int(keyword)
        except:
            zones.append(keyword.decode())
            (dummy, p) = contentsOp.getMiddlePair(p)
        while keyword != b"" and p < len(contents):
            (keyword, p) = contentsOp.getKeyword(p)
            if keyword != b"":
                zones.append(keyword.decode())
                (dummy, p) = contentsOp.getMiddlePair(p)
        zones.sort()
        return zones

    def faceZoneNames(self, polyMeshDir):
        """ faceZonesファイルからfaceZone名を取得する
        
        Args:
            polyMeshDir (str)   :polyMeshDir(fullPath)
        returns:
            faceZoneNames (list(str))   :faceZone名のlist"""
        #fileが無ければ直ぐに戻る
        fileName = polyMeshDir + "/faceZones"
        zones = self.zoneNames(fileName)
        return zones

    def pointZoneNames(self, polyMeshDir):
        """ pointZonesファイルからpointZone名を取得する
        
        Args:
            polyMeshDir (str)   :polyMeshDir(fullPath)
        returns:
            pointZoneNames (list(str))   :pointZone名のlist"""
        #fileが無ければ直ぐに戻る
        fileName = polyMeshDir + "/pointZones"
        zones = self.zoneNames(fileName)
        return zones

    def zoneNames(self, zoneNameFile):
        fileName = zoneNameFile
        if len(glob.glob(fileName)) == 0:
            fileName += ".gz"
            if len(glob.glob(fileName)) == 0:
                return []
        
        (contents, _data) = foamFile().readSeparate(fileName)
        contentsOp = strOp(contents)
        contents = contentsOp.deleteCommentsNullLines()
        p = contentsOp.skipFoamFile()
        #zone名の取得
        zones = []
        (keyword, p) = contentsOp.getKeyword(p)
        try:
            int(keyword)
        except:
            zones.append(keyword)
            (dummy, p) = contentsOp.getMiddlePair(p)
        while keyword != "" and p < len(contents):
            (keyword, p) = contentsOp.getKeyword(p)
            if keyword != "":
                zones.append(keyword)
                (dummy, p) = contentsOp.getMiddlePair(p)
        zones.sort()
        return zones

    def makeRegionProperties(self, fluidRegions, solidRegions):
        """ regionPropertiesを作成して返す"""
        configDict = readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = getNumVersion(OFversion)
        if (0.0 < numVer and numVer < 2.2) or OFversion[:3] == "ext":
            #version 2.1.1以前、ext版の場合
            #foamFile作成
            foamFile = self.makeFoamHeaderVersion(OFversion)
            foamFile += self.makeFoamFile("2.0", "ascii", "dictionary",
                        "constant", "regionProperties")
            footer = self.makeFoamFooter()
            #regionPropertiesの内容を作成
            contents = "fluidRegionNames ( "
            for name in fluidRegions:
                contents += name + " "
            contents += ");\n\n"
            contents += "solidRegionNames ( "
            for name in solidRegions:
                contents += name + " "
            contents += ");\n"
            cont = foamFile + contents + footer
        elif numVer >= 11.0:
            #OF-11以降の場合
            contents = ""
            for name in fluidRegions:
                contents += "    " + ("%-15s" % name) + " fluid;\n"
            for name in solidRegions:
                contents += "    " + ("%-15s" % name) + " solid;\n"
            cont  = "regionSolvers\n"
            cont += "{\n"
            cont += contents
            cont += "}\n"
        else:
            #version 2.2.0以降の場合
            #foamFile作成
            foamFile = self.makeFoamHeaderVersion(OFversion)
            foamFile += self.makeFoamFile("2.2", "ascii", "dictionary",
                        "constant", "regionProperties")
            footer = self.makeFoamFooter()
            #regionPropertiesの内容作成
            contents = "regions\n(\n"
            contents += "    fluid       ( "
            for name in fluidRegions:
                contents += name + " "
            contents += ")\n"
            contents += "    solid       ( "
            for name in solidRegions:
                contents += name + " "
            contents += ")\n);\n"
            cont = foamFile + contents + footer
        return cont

#
#  pyTreeFoamのimport時に、appの設定を行う
#
try:
    configDict = readConfigTreeFoam()
    fileManager  = configDict["fileManager"]
    Terminal     = configDict["Terminal"]
    foamTerminal = configDict["foamTerminal"]
    Editor       = configDict["editor"]
    Office       = configDict["Office"]

    print("set pyTreeFoam.application(fileMnager, Terminal, foamTerminal, editor, Office).")
except:
    pass


if __name__ == "__main__":
    import gettext
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    currDir = '/home/caeuser/CAE/CAE-FOAM/OF-v1906/pitzDaily_binary'
    fileName = currDir + "/100/U"
    cont = foamFile().read(fileName)
    print(cont)
    foamFile().write(fileName, cont)
