#!/usr/bin/python3
#  coding: utf-8
#
#   pasteFieldsMappingDialog.py
#
#       copyDirのfieldをpasteDirにmappingする
#
#   22/01/29    新規作成
#      02/02    renameUnmappedFiles:「.unmapped」付きファイルの修正方法
#               を変更。
#      02/03    setTargetRegionTime,setSourceRegionTime:timeFolders=[]の場合
#               timeFolders=[""]を設定。
#      06/09    onStartPaste:paste後、close()を追加
#      07/19    startPaste:OFversionチェック方法修正（OF-10対応）
#   24/08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import sys, os
import glob
import shutil

import GtkParts
import pyTreeFoam
import universalDialogs as unvDlg

import logFileCreater
logFileCreater.log()


#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, copyDir, pasteDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"pasteFieldsMappingDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を定義
        self.setGtkObject()
        #変数の定義
        self.copyDir = copyDir
        self.pasteDir = pasteDir

    #
    #  main
    #-------
    def main(self):
        """ GUIの表示"""
        self.initialize()
        self.mainWindow.show()
        Gtk.main()

    #
    #  close
    #--------
    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ glade内のobject名を定義"""
        self.label_copyDir = self.builder.get_object("label_copyDir")
        self.label_pasteDir = self.builder.get_object("label_pasteDir")
        self.label_copyCase = self.builder.get_object("label_copyCase")
        self.label_pasteCase = self.builder.get_object("label_pasteCase")
        tree_copy = self.builder.get_object("tree_copy")
        self.listBoxCopy = GtkParts.treeList(tree_copy)
        self.listBoxCopy.create(multi=False)
        tree_paste = self.builder.get_object("tree_paste")
        self.listBoxPaste = GtkParts.treeList(tree_paste)
        self.listBoxPaste.create(multi=False)
        self.check_paraSource = self.builder.get_object("check_paraSource")
        self.check_paraTarget = self.builder.get_object("check_paraTarget")
        combo_regionSource = self.builder.get_object("combo_regionSource")
        self.combo_regionSource = GtkParts.comboBoxText(combo_regionSource)
        combo_regionTarget = self.builder.get_object("combo_regionTarget")
        self.combo_regionTarget = GtkParts.comboBoxText(combo_regionTarget)
        combo_mapMethod = self.builder.get_object("combo_mapMethod")
        self.combo_mapMethod = GtkParts.comboBoxText(combo_mapMethod)



    #---------- event handler -----------
    #changeSourcePara
    def onChangeSourcePara(self, event):
        self.changeSourcePara()
    #changeTargetPara
    def onChangeTargetPara(self, event):
        self.changeTargetPara()
    #貼り付け開始
    def onStartPaste(self, event):
        self.startPaste()
        #self.close()
    #キャンセル
    def onCancel(self, event):
        self.close()


    #
    #  changeSourcePara
    #-------------------
    def changeSourcePara(self):
        """ source側の並列checkBoxを修正した時"""
        if self.check_paraSource.get_active() == True:
            if self.isParallelCase(self.copyDir) == False:
                self.check_paraSource.set_active(False)
                title = _("エラー")
                mess = _("source側が並列処理用にmesh分割されていません。")
                self.errDialog(title, mess)
                return
        #regioとtimeを再設定
        self.setSourceRegionTime()

    #
    #  changeTargetPara
    #-------------------
    def changeTargetPara(self):
        """ target側の並列checkBoxを修正した時"""
        if self.check_paraTarget.get_active() == True:
            if self.isParallelCase(self.pasteDir) == False:
                self.check_paraTarget.set_active(False)
                title = _("エラー")
                mess = _("target側が並列処理用にmesh分割されていません。")
                self.errDialog(title, mess)
                return
        #regionとtimeを再設定
        self.setTargetRegionTime()

    #
    #  startPaste
    #-------------
    def startPaste(self):
        """ mapFieldsを実行"""
        #バージョンチェック---------------
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        #if OFversion < "4.0" or OFversion[:len("ext")] == "ext":
        if ( (0.0 < numVer and numVer < 4.0) or 
              OFversion[:3] == "ext" ):
            title = _("エラー")
            mess = _("現在のOpenFOAMのバージョンでは、使用できません。\n")
            mess += _("  現在のバージョン: ") + OFversion
            self.errDialog(title, mess)
            return
        #command行を作成-----------------
        envFile = configDict["bashrcFOAM"]
        comm = ". " + envFile + "; "
        comm += "mapFields "
        #  sourcePara
        if self.check_paraSource.get_active() == True:
            comm += "-parallelSource "
            if self.isParallelCase(self.copyDir) == False:
                #error
                title = _("エラー")
                mess = _("source側が並列処理用にmesh分割されていません。")
                self.errDialog(title, mess)
                return
        #  sourceRegion
        selRegion = self.combo_regionSource.getSelectedItem()
        if selRegion == "(region0)":
            selRegion = "."
        comm += "-sourceRegion " + selRegion + " "
        #  sourceTime
        selTime = self.listBoxCopy.getSelectedItems()[0]
        words = selTime.split(":")
        if len(words) == 2:
            selTime = words[1]
        comm += "-sourceTime " + selTime + " "
        error = 0
        if self.check_paraSource.get_active() == False:
            if os.path.exists(self.copyDir+"/"+selTime) == False:
                error = 1
        else:
            if os.path.exists(self.copyDir + "/processor0/" + selTime) == False:
                error = 1
        if error == 1:
            title = _("エラー")
            mess = _("source側の指定したtimeFolderが存在しません")
            self.errDialog(title, mess)
            return
        #  targetPara
        if self.check_paraTarget.get_active() == True:
            comm += "-parallelTarget "
            if self.isParallelCase(self.pasteDir) == False:
                title = _("エラー")
                mess = _("target側が並列処理用にmesh分割されていません。")
                self.errDialog(title, mess)
                return
        #  targetRegion
        selRegion = self.combo_regionTarget.getSelectedItem()
        if selRegion == "(region0)":
            selRegion = "."
        comm += "-targetRegion " + selRegion + " "
        #  targetTime
        selTime = self.listBoxPaste.getSelectedItems()[0]
        start = selTime.split(":")[0]
        case = pyTreeFoam.case(self.pasteDir)
        contDict = case.getControlDict()
        if contDict["startFrom"] != start:
            if (start == "firstTime" or start == "startTime"
                or start == "latestTime"):
                contDict["startFrom"] = start
            else:
                contDict["startFrom"] = "startTime"
                contDict["startTime"] = start
            case.setControlDict(contDict)
        #  mapMethod
        method = self.combo_mapMethod.getSelectedItem()
        if method != "(default)":
            comm += "-mapMethod " + method + " "
        #  sourceDir
        comm += self.copyDir
        #実行確認----------------
        title = _("mapFields実行")
        mess = _("mapFieldsを実行します。")
        okArgs = [self.startPaste_runMapping, comm]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return
        # if stat != "OK":
        #     return
        
    #  startPaste_runMapping
    def startPaste_runMapping(self, comm):
        """ mapFieldsを実行"""

        #  deleteUnmapped
        def deleteUnmapped(oldName):
            """ oldNameから「unmapped」の文字を削除"""
            while oldName[-len(".unmapped"):] == ".unmapped":
                oldName = oldName[:-len(".unmapped")]
            return oldName

        #  renameUnmappedFiles
        def renameUnmappedFiles(fieldDir):
            """ unmappedFileをrenameして元に戻し、そのfile名を返す。"""
            unmapFields = []
            fileNames = glob.glob(fieldDir + "/*.unmapped")
            fileNames += glob.glob(fieldDir + "/*/*.unmapped")
            for oldName in fileNames:
                newName = deleteUnmapped(oldName)
                os.rename(oldName, newName)
                unName = newName.split("/")[-1]
                unmapFields.append(unName)
            return unmapFields

        #  getCurrTimeInProc
        def getCurrTimeInProc(proc):
            """ currTimeを取得して返す"""
            case = pyTreeFoam.case(self.pasteDir)
            contDict = case.getControlDict()
            startFrom = contDict["startFrom"]
            if startFrom == "startTime":
                currTime = contDict["startTime"]
            else:
                timeFolders = case.getTimeFolders(proc)
                if startFrom == "latestTime":
                    currTime = timeFolders[-1]
                else:
                    currTime = timeFolders[0]
            return currTime

        #  command実行
        print(comm)
        (stat, _res, _err) = pyTreeFoam.run(self.pasteDir).commandReturnCont(comm)
        if stat != "OK":
            title = _("エラー")
            mess = _("エラーが発生しました。\n")
            mess += _("logを確認してください。")
            self.errDialog(title, mess)
            return
        #処理後の確認----------------
        if self.check_paraTarget.get_active() == False:
            #singleCoreの場合
            currTime = getCurrTimeInProc(".")
            fieldDir = self.pasteDir + "/" + currTime
            unmapFields = renameUnmappedFiles(fieldDir)
        else:
            #並列処理の場合
            currTime = getCurrTimeInProc("processor0")
            procDirs = glob.glob(self.pasteDir + "/processor*")
            for procDir in procDirs:
                fieldDir = procDir + "/" + currTime
                unmapFields = renameUnmappedFiles(fieldDir)
        title = _("mapFields実行")
        mess = _("mapFieldsを実行して、fieldにデータをmappingしました。")
        if len(unmapFields) > 0:
            mess += _("\n以下のfieldは、mappingしていません。\n")
            unmapFields = list(map(lambda x: "  "+x, unmapFields))
            mess += "\n".join(unmapFields)
        okArgs = [self.close]       #dialogを閉じる
        self.okDialog(title, mess, funcOk=okArgs)

    #
    #  shrinkLenStr
    def shrinkLenStr(self, name, nMax):
        """ 文字列を指定の長さに縮める。"""
        if len(name) > nMax:
            name = "..." + name[-(nMax-3):]
        return name

    #
    #  setCaseNameToLabel
    def setCaseNameToLabel(self):
        """ copy, pasteのcase名をlabelに設定する"""
        copyDir = self.shrinkLenStr(self.copyDir, 80)
        self.label_copyDir.set_text(copyDir)
        pasteDir = self.shrinkLenStr(self.pasteDir, 80)
        self.label_pasteDir.set_text(pasteDir)
        copyName = self.copyDir.split("/")[-1]
        copyName = self.shrinkLenStr(copyName, 40)
        pasteName = self.pasteDir.split("/")[-1]
        pasteName = self.shrinkLenStr(pasteName, 40)
        self.label_copyCase.set_text(_("case名: ") + copyName)
        self.label_pasteCase.set_text(_("case名: ") + pasteName)

    #
    #  isParallelCase
    def isParallelCase(self, caseDir):
        """ caseが並列用にmesh分割されているかcheck"""
        if len(glob.glob(caseDir + "/processor*")) != 0:
            return True
        return False

    #
    #  setSourceRegionTime
    def setSourceRegionTime(self):
        """ source側のregionとtimeを設定する"""
        #regionをチェック
        case = pyTreeFoam.case(self.copyDir)
        flag, _loc, regions = case.isMultiRegion()
        if flag == False:
            regions = ["(region0)"]
        else:
            regions = ["(region0)"] + regions
        self.combo_regionSource.setItems(regions)
        if len(regions) == 1:
            self.combo_regionSource.selectItem(0)
        else:
            self.combo_regionSource.selectItem(1)
        #timeをチェック
        if self.check_paraSource.get_active() == False:
            timeFolders = case.getTimeFolders()
        else:
            timeFolders = case.getTimeFolders("processor0")
        if len(timeFolders) == 0:
            timeFolders = [""]
        contDict = case.getControlDict()
        firstTime = timeFolders[0]
        startTime = contDict["startTime"]
        latestTime = timeFolders[-1]
        addItems = ["firstTime:" + firstTime]
        addItems += ["startTime:" + startTime]
        addItems += ["latestTime:" + latestTime]
        items = addItems + timeFolders
        self.listBoxCopy.setItems(items)
        self.listBoxCopy.selectItemNos([2])

    #
    #  setSourceContents
    def setSourceContents(self):
        """ source側の内容をセットする"""
        #checkBoxを設定
        paraFlag = self.isParallelCase(self.copyDir)
        if paraFlag == True:
            self.check_paraSource.set_active(True)
        else:
            self.check_paraSource.set_active(False)
        #region, timeをセット
        self.setSourceRegionTime()

    #
    #  setTargetRegionTime
    def setTargetRegionTime(self):
        """ target側のregionとtimeを設定"""
        #regionをチェック
        case = pyTreeFoam.case(self.pasteDir)
        flag, _loc, regions = case.isMultiRegion()
        if flag == False:
            regions = ["(region0)"]
        else:
            regions = ["(region0)"] + regions
        self.combo_regionTarget.setItems(regions)
        if len(regions) == 1:
            self.combo_regionTarget.selectItem(0)
        else:
            self.combo_regionTarget.selectItem(1)
        #timeをチェック
        if self.check_paraTarget.get_active() == False:
            timeFolders = case.getTimeFolders()
        else:
            timeFolders = case.getTimeFolders("processor0")
        if len(timeFolders) == 0:
            timeFolders = [""]
        contDict = case.getControlDict()
        firstTime = timeFolders[0]
        startTime = contDict["startTime"]
        latestTime = timeFolders[-1]
        addItems = ["firstTime:" + firstTime]
        addItems += ["startTime:" + startTime]
        addItems += ["latestTime:" + latestTime]
        items = addItems + timeFolders
        self.listBoxPaste.setItems(items)
        selNo = 1
        if contDict["startFrom"] == "firstTime":
            selNo = 0
        elif contDict["startFrom"] == "startTime":
            selNo = 1
        elif contDict["startFrom"] == "latestTime":
            selNo = 2
        self.listBoxPaste.selectItemNos([selNo])

    #
    #  setTargetContents
    def setTargetContents(self):
        """ target側の内容をセット"""
        #checkBoxを設定
        paraFlag = self.isParallelCase(self.pasteDir)
        if paraFlag == True:
            self.check_paraTarget.set_active(True)
        else:
            self.check_paraTarget.set_active(False)
        #region, timeをセット
        self.setTargetRegionTime()

    #
    #  setMapMethod
    def setMapMethod(self):
        """ mapMethodをセット"""
        items = ["(default)", "mapNearest", "interpolate",
                 "cellPointInterpolate"]
        self.combo_mapMethod.setItems(items)
        self.combo_mapMethod.selectItem(0)

    #
    #  checkMapFieldsDict
    def checkMapFieldsDict(self):
        """ mapFieldsDict有無を確認し、無ければコピーする。"""
        mapFieldsDict = self.pasteDir + "/system/mapFieldsDict"
        if os.path.exists(mapFieldsDict) == False:
            configDict = pyTreeFoam.readConfigTreeFoam()
            OFversion = configDict["OFversion"]
            sourceDir = pyTreeFoam.getFoamContents().getOFversionFolder(OFversion)
            sourceFile = sourceDir + "/system/mapFieldsDict"
            pasteDir = self.pasteDir + "/system"
            shutil.copy(sourceFile, pasteDir)
            title = _("mapFieldsの実行")
            mess = _("mapFieldsDictが存在しなかったので、default設定のmapFieldsDictをコピーしました。")
            self.okDialog(title, mess)

    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        #mapFieldsDictを確認
        self.checkMapFieldsDict()
        #case名をセット
        self.setCaseNameToLabel()
        #source側をセット
        self.setSourceContents()
        #target側をセット
        self.setTargetContents()
        #mapMethodをセット
        self.setMapMethod()


    #  ------- universal dialog ---------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog =  unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()


if __name__ == "__main__":
    import gettext
    import locale
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    locale.bindtextdomain("treefoam", localeDir)
    gettext.install("treefoam", localeDir)

    copyDir = sys.argv[1]
    pasteDir = sys.argv[2]

    winApp = windowApp(copyDir, pasteDir)
    winApp.main()
