#!/usr/bin/python3
# coding: utf-8
#
#       meshViewerCreateSetsVtk.py
#
#           sets(cellSet, faceSet, pointSet)のvtkを作成する
#
#   21/09/15    新規作成
#      09/17    高速化の為、dictをlistに変更
#      10/30    getOutSideFacesOfCellSet:ownerCellのfaceを取得する時
#               全てのcellNoがownerCellに入っていないので、エラー発生
#               する場合ある為、修正。
#   23/08/03    createAllSetsVtkFiles:vtkファイルが余る場合があるので、
#               setsフォルダ内と合致するかチェックし、余る場合は削除を追加。
#   24/02/12    createAllSetsVtkFiles:message出力にregion名を追加
#

import os, sys
import glob
import pyTreeFoam


#
#  getMeshValues
#-----------------
def getMeshValues(nameDir):
    """ owner, neighbourの様な形式のファイルのデータ部を読み込んで返す。
    戻り値は、int形式に変換して戻す。
    file形式がascii、binaryでも読み込める。
    
    Args:
        nameDir(str):   fullPathで指定（fileName）
    Return:
        cellNos(list(int)): データの値"""
    #ellNoを取得
    fileName = nameDir
    foamFile = pyTreeFoam.foamFile()
    ascii, data = foamFile.readSeparate(fileName)
    dataHeader = data[0][0]
    #ascii or binaryをチェック
    binData = data[0][1]
    error = foamFile.confirmAscii(binData)
    if error == 0:
        #ascii形式
        cont = binData.decode()
        words = cont.split()
        cellNos = list(map(int, words))
        return cellNos
    else:
        #binary形式
        valData = data[0]
        #dataType = valData[0][0]
        #nData = valData[0][1]
        cellNos = foamFile.structBinData(valData)
        return cellNos

#
#  getMeshFaces
#---------------
def getMeshFaces(nameDir):
    """ polyMesh内のfaceファイルを読み込む。
    faceを構成するpointNoのlistを返す。
    fileはascii、binary形式でも読み込める。
    戻り値は、int形式のリストで返す。
    
    Args:
        nameDir(str):   fullPathで指定（fileName）
    Return:
        faces(list(face)): face(list(int))"""
    fileName = nameDir
    foamFile = pyTreeFoam.foamFile()
    ascii, data = foamFile.readSeparate(fileName)
    dataHeader = data[0][0]
    #ascii or binaryをチェック
    binData = data[0][1]
    error = foamFile.confirmAscii(binData)
    if error == 0:
        #ascii形式
        cont = data[0][1].decode()
        words = cont.split("\n")
        if len(words) > 20:
            #数が多い場合
            faces = []
            for word in words:
                p = word.find("(") + 1
                if p > 0:
                    pe = word.rfind(")")
                    nums = list(map(int, word[p:pe].split()))
                    faces.append(nums)
                    #faces.append(word[p:pe])
        else:
            #数が少ない場合
            contOp = pyTreeFoam.strOp(cont)
            p = 0
            faces = []
            while p < len(cont):
                face, p = contOp.getSmallPair(p)
                if face != "":
                    nums = list(map(int, face.split()))
                    faces.append(nums)
                    #faces.append(face)
        return faces
    else:
        #binary形式
        #  n(a,b,c)の「n」の累計を取得
        valData = data[0]
        nNums = foamFile.structBinData(valData)
        #  n(a,b,c)の(a,b,c)を取得
        valData = data[1]
        nums = foamFile.structBinData(valData)
        #  faceを取得
        faces = []
        for i in range(len(nNums)-1):
            ps = nNums[i]
            pe = nNums[i+1]
            face = nums[ps:pe]
            faces.append(face)
        return faces

#
#  getMeshPoints
#----------------
def getMeshPoints(nameDir):
    """ polyMesh内のpontsファイルを読み込む。
    pointを構成する座標のlistを返す。
    fileはascii、binary形式でも読み込める。
    戻り値は、float形式のlistで返す。
    
    Args:
        nameDir(str):   fullPathで指定（fileName）
    Return:
        faces(list(point)): point(list(float))"""
    fileName = nameDir
    foamFile = pyTreeFoam.foamFile()
    ascii, data = foamFile.readSeparate(fileName)
    dataHeader = data[0][0]
    #ascii or binaryをチェック
    binData = data[0][1]
    error = foamFile.confirmAscii(binData)
    if error == 0:
        #ascii形式
        cont = binData.decode()
        words = cont.split("\n")
        if len(words) > 20:
            #数が多い場合
            points = []
            for word in words:
                p = word.find("(") + 1
                if p > 0:
                    pe = word.rfind(")")
                    nums = list(map(float, word[p:pe].split()))
                    points.append(nums)
                    #points.append(word[p:pe])
        else:
            #数が少ない場合
            contOp = pyTreeFoam.strOp(cont)
            p = 0
            points = []
            while p < len(cont):
                point, p = contOp.getSmallPair(p)
                if point != "":
                    #points.append(point)
                    nums = list(map(float, point.split()))
                    points.append(nums)
        return points
    else:
        #binary形式
        valData = data[0]
        nums = foamFile.structBinData(valData)
        points = []
        for i in range(0, len(nums), 3):
            point = nums[i:i+3]
            points.append(point)
        return points

#
#  getMeshIdDict
def getMeshIdDict(vals):
    """ cellNo→faceNoのlistを作成する"""
    maxNo = max(vals) + 1
    valDict = [[] for i in range(maxNo)]
    for i in range(len(vals)):
        cellNo = vals[i]
        valDict[cellNo].append(i)
    return valDict

#
#  getVtkPointsLines
def getVtkPointsLines(newPointLocs):
    """ POINTSのvtkを作成する"""
    nPoints = len(newPointLocs)
    lines = ["POINTS " + str(nPoints) + " float\n"]
    for loc in newPointLocs:
        line = " ".join(map(str, loc)) + "\n"
        lines += [line]
    return lines

#
#  getVtkCells
def getVtkCells(newFaces):
    """ CELLSのvtkを作成する"""
    nFaces = len(newFaces)
    nData = 0
    lines = []
    for points in newFaces:
        nData += len(points) + 1
        words = list(map(str, points))
        lines += [str(len(points)) + " " + " ".join(words) + "\n"]
    line = "CELLS " + str(nFaces) + " " + str(nData) + "\n"
    lines = [line] + lines
    return lines

#
#  getVtkCellTypes
def getVtkCellTypes(newFaces):
    """ CELL_TYPEのvtkを作成する"""
    lines = ["CELL_TYPES " + str(len(newFaces)) + "\n"]
    for i in range(len(newFaces)):
        lines += ["7\n"]
    return lines

#
#  getOutSideFacesOfCellSet
#--------------------------
def getOutSideFacesOfCellSet(folderDir, name, ownerFaces, neibFaces, allOwners):
    """ cellSetの外表面faceを取得する。"""
    #cellSet内のcellNoを取得
    setsDir = folderDir + "/sets"
    cellNos = getMeshValues(setsDir + "/" + name)
    #cellNosからfaceDict辞書作成（faceNo→cellNoを取得）
    faceDict = {}
    for cellNo in cellNos:
        #owner側をチェック
        try:
            faceNos = ownerFaces[cellNo]    #ここでerror発生する事あり
            for faceNo in faceNos:
                if faceNo in faceDict.keys():
                    faceDict[faceNo].append(cellNo)
                else:
                    faceDict[faceNo] = [cellNo]
        except:
            pass
        #neighbour側をチェック
        try:
            faceNos = neibFaces[cellNo]     #ここでerror発生する事あり
            for faceNo in faceNos:
                if faceNo in faceDict.keys():
                    faceDict[faceNo].append(cellNo)
                else:
                    faceDict[faceNo] = [cellNo]
        except:
            pass
    #外表面のfaceを取得する
    singleFaces = []
    for faceNo in faceDict.keys():
        if len(faceDict[faceNo]) == 1:
            singleFaces.append(faceNo)
    #外表面の向き反転させるfaceNoを取得
    reverseFaces = []
    cells = set(cellNos)
    for i in range(len(singleFaces)):
        faceNo = singleFaces[i]
        if allOwners[faceNo] in cells:
            pass
        else:
            #向き反転
            reverseFaces.append(faceNo)
    return singleFaces, reverseFaces

#
#  reverseFaceDirection
def reverseFaceDirection(points):
    """ faceの向きを反転して返す"""
    newPoints = []
    for i in range(len(points)):
        idx = len(points) - 1 -i
        newPoints.append(points[idx])
    return newPoints

#
#  remakePointNoOfCellSet
#-------------------------
def remakePointNoOfCellSet(singleFaces, reverseFaces, allFaces, allPoints):
    """ pointNoを更新する。"""
    #使用するpointsを取得
    newPoints = []
    for faceNo in singleFaces:
        points = allFaces[faceNo]
        newPoints += points
    newPoints = list(set(newPoints))
    #pointNoを更新する為の辞書を作成
    newPointDict = {}
    for i in range(len(newPoints)):
        pointNo = newPoints[i]
        newPointDict[pointNo] = i
    #pointNoを更新したnewFaceを作成
    revFaces = set(reverseFaces)
    newFaces = []
    for i in range(len(singleFaces)):
        faceNo = singleFaces[i]
        points = allFaces[faceNo]
        points = list(map(lambda x: newPointDict[x], points))
        #face反転チェック
        if faceNo in revFaces:
            #faceを反転させる
            points = reverseFaceDirection(points)
        newFaces.append(points)
    #newPointLocsを作成
    newPointLocs = []
    for pointNo in newPointDict.keys():
        loc = allPoints[pointNo]
        newPointLocs.append(loc)
    return newPointLocs, newFaces

#
#  remakePointNoOfPointSet
#--------------------------
def remakePointNoOfPointSet(points, allPoints):
    """ pointNoを更新する"""
    newPointDict = {}
    for i in range(len(points)):
        pointNo = points[i]
        newPointDict[pointNo] = i
    #pointNoを更新したnewPointLocsを作成
    newPointLocs = []
    for pointNo in newPointDict.keys():
        loc = allPoints[pointNo]
        newPointLocs.append(loc)
    return newPointLocs

#
#  createCellSetVtkFile
#-----------------------
def createCellSetVtkFile(folderDir, name, newPointLocs, newFaces):
    """ cellSetのvtkFileを作成する"""
    #vtkFileを作成
    header = "# vtk DataFile Version 2.0\n"
    title = "cellSet " + name + "\n"
    vtkLines = [header]
    vtkLines += [title]
    vtkLines += ["ASCII\n"]
    vtkLines += ["DATASET UNSTRUCTURED_GRID\n"]
    vtkLines += getVtkPointsLines(newPointLocs)
    vtkLines += getVtkCells(newFaces)
    vtkLines += getVtkCellTypes(newFaces)
    vtkDir = folderDir + "/VTK"
    if os.path.exists(vtkDir) == False:
        os.mkdir(vtkDir)
    fileName = vtkDir + "/cellSet_" + name + ".vtk"
    f = open(fileName, "w"); f.writelines(vtkLines); f.close()

#
#  createFaceSetVtkFile
#-----------------------
def createFaceSetVtkFile(folderDir, name, newPointLocs, newFaces):
    """ faceSetのvtkFileを作成する"""
    header = "# vtk DataFile Version 2.0\n"
    title = "faceSet " + name + "\n"
    vtkLines = [header]
    vtkLines += [title]
    vtkLines += ["ASCII\n"]
    vtkLines += ["DATASET UNSTRUCTURED_GRID\n"]
    vtkLines += getVtkPointsLines(newPointLocs)
    vtkLines += getVtkCells(newFaces)
    vtkLines += getVtkCellTypes(newFaces)
    vtkDir = folderDir + "/VTK"
    if os.path.exists(vtkDir) == False:
        os.mkdir(vtkDir)
    fileName = vtkDir + "/faceSet_" + name + ".vtk"
    f = open(fileName, "w"); f.writelines(vtkLines); f.close()

#
#  createPointSetVtkFile
#------------------------
def createPointSetVtkFile(folderDir, name, newPointLocs):
    """ pointSetのvtkFileを作成する"""
    header = "# vtk DataFile Version 2.0\n"
    title = "pointSet " + name + "\n"
    vtkLines = [header]
    vtkLines += [title]
    vtkLines += ["ASCII\n"]
    vtkLines += ["DATASET POLYDATA\n"]
    vtkLines += getVtkPointsLines(newPointLocs)
    vtkDir = folderDir + "/VTK"
    if os.path.exists(vtkDir) == False:
        os.mkdir(vtkDir)
    fileName = vtkDir + "/pointSet_" + name + ".vtk"
    f = open(fileName, "w"); f.writelines(vtkLines); f.close()

#---------------------------
#  getCellFacePontSetsNames
#---------------------------
def getCellFacePontSetsNames(folderDir):
    """ cellSet, faceSet, pointSetのfileNameを取得する"""
    setsDir = folderDir + "/sets"
    fileNames = glob.glob(setsDir+"/*")
    cellSets = []
    faceSets = []
    pointSets = []
    foamFile = pyTreeFoam.foamFile()
    for fileName in fileNames:
        if os.path.isfile(fileName) == True:
            name = os.path.basename(fileName)
            cont = foamFile.read(fileName)
            className = foamFile.getClassInFoamFile(cont).decode()
            objName = foamFile.getObjectInFoamFile(cont).decode()
            if className == "cellSet":
                cellSets.append(name)
            elif className == "faceSet":
                faceSets.append(name)
            elif className == "pointSet":
                pointSets.append(name)
    return cellSets, faceSets, pointSets

#
#  getNewSetsFile
#-----------------
def getNewSetsFile(folderDir, cellSets, faceSets, pointSets):
    """ setsFileとvtkファイルのtimeStampを確認し、新しいsetsFile
    のみ取得して返す。"""
    #cellSetを確認
    flag = 0
    cells = []
    for cellSet in cellSets:
        setFile = folderDir + "/sets/" + cellSet
        vtkFile = folderDir + "/VTK/" + "cellSet_" + cellSet + ".vtk"
        if os.path.exists(vtkFile) == False:
            flag = 1
        else:
            timeStampSet = os.path.getmtime(setFile)
            timeStampVtk = os.path.getmtime(vtkFile)
            if timeStampSet > timeStampVtk:
                flag = 1
        if flag == 1:
            cells.append(cellSet)
    #faceSetを確認
    flag = 0
    faces = []
    for faceSet in faceSets:
        setFile = folderDir + "/sets/" + faceSet
        vtkFile = folderDir + "/VTK/" + "faceSet_" + faceSet + ".vtk"
        if os.path.exists(vtkFile) == False:
            flag = 1
        else:
            timeStampSet = os.path.getmtime(setFile)
            timeStampVtk = os.path.getmtime(vtkFile)
            if timeStampSet > timeStampVtk:
                flag = 1
        if flag == 1:
            faces.append(faceSet)
    #pointSetを確認
    flag = 0
    points = []
    for pointSet in pointSets:
        setFile = folderDir + "/sets/" + pointSet
        vtkFile = folderDir + "/VTK/" + "pointSet_" + pointSet + ".vtk"
        if os.path.exists(vtkFile) == False:
            flag = 1
        else:
            timeStampSet = os.path.getmtime(setFile)
            timeStampVtk = os.path.getmtime(vtkFile)
            if timeStampSet > timeStampVtk:
                flag = 1
        if flag == 1:
            points.append(pointSet)
    return cells, faces, points

#
#  getMeshDataFileNames
#-----------------------
def getMeshDataFileNames(caseDir, timeFolder, region):
    """ owner, neighbour, faces, pointsを探す"""
    case = pyTreeFoam.case(caseDir)
    relDir = case.getCurrMeshDir(timeFolder, region, "owner")
    owner = os.path.normpath(caseDir + "/" + relDir + "/owner")
    relDir = case.getCurrMeshDir(timeFolder, region, "neighbour")
    neighbour = os.path.normpath(caseDir + "/" + relDir + "/neighbour")
    relDir = case.getCurrMeshDir(timeFolder, region, "faces")
    faces = os.path.normpath(caseDir + "/" + relDir + "/faces")
    relDir = case.getCurrMeshDir(timeFolder, region, "points")
    points = os.path.normpath(caseDir + "/" + relDir + "/points")
    return [owner, neighbour, faces, points]

#---------------------
#  getOwnerNeighbour
#---------------------
def getOwnerNeighbour(owner, neighbour):
    """ owner, neighbourの読込、取得
    
    Args:
        owner(str)      :fullPathのownerFile
        neighbour(str)  :fullPathのneighbourFile
    Returns:
        ownerFaces(list(list(int))) :cellNo→facesのlist(owner)
        neibFaces(list(list(int)))  :cellNo→facesのlist(neighbour)
        allOwners(list(int))        :faceNo→cellNoのlist(owner)
        """
    #owner
    print("getting owner...")
    vals = getMeshValues(owner)
    ownerFaces = getMeshIdDict(vals)
    allOwners = vals
    #neighbour
    print("getting neighbour...")
    vals = getMeshValues(neighbour)
    neibFaces = getMeshIdDict(vals)
    return ownerFaces, neibFaces, allOwners

#------------------
#  getFacesPoints
#------------------
def getFacesPoints(faces, points):
    """ faces, pointsの読込、取得
    
    Args:
        faces(str)  :fullPathのfacesFile名
        points(str) :fullPathのpointsFile名
    Returns:
        allFaces(list(list(int)))       :faceNo→pointsNos
        allPoints(list(list(float)))    :pointNo→xyz location"""
    #faces
    print("getting faces...")
    allFaces = getMeshFaces(faces)
    #points
    print("getting points...")
    allPoints = getMeshPoints(points)
    return allFaces, allPoints

#---------------------------
#  createAllCellSetVtkFiles
#---------------------------
def createAllCellSetVtkFiles(folderDir, names, ownerFaces, neibFaces, allOwners, allFaces, allPoints):
    """ 全cellSetのvtkFileを作成する"""
    for name in names:
        print("creating vtk of cellSet " + name + "...")
        #cellSetの外表面のfaceを取得
        singleFaces, reverseFaces = getOutSideFacesOfCellSet(folderDir, name, ownerFaces, neibFaces, allOwners)
        #pointNoを更新
        newPointLocs, newFaces = remakePointNoOfCellSet(singleFaces, reverseFaces, allFaces, allPoints)
        #vtkFileを作成する
        createCellSetVtkFile(folderDir, name, newPointLocs, newFaces)

#---------------------------
#  createAllFaceSetVtkFiles
#---------------------------
def createAllFaceSetVtkFiles(folderDir, names, allFaces, allPoints):
    """ 全faceSetのvtkFileを作成する"""
    setsDir = folderDir + "/sets"
    for name in names:
        print("creating vtk of faceSet " + name + "...")
        singleFaces = getMeshValues(setsDir + "/" + name)
        reverseFaces = []
        #pointNoを更新
        newPointLocs, newFaces = remakePointNoOfCellSet(singleFaces, reverseFaces, allFaces, allPoints)
        #vtkFileを作成する
        createFaceSetVtkFile(folderDir, name, newPointLocs, newFaces)

#----------------------------
#  createAllPointSetVtkFiles
#----------------------------
def createAllPointSetVtkFiles(folderDir, names, allPoints):
    """ 全pointSetのvtkFileを作成する"""
    setsDir = folderDir + "/sets"
    for name in names:
        print("creating vtk of pointSet " + name + "...")
        setPoints = getMeshValues(setsDir + "/" + name)
        #pointNoを更新
        newPointLocs = remakePointNoOfPointSet(setPoints, allPoints)
        #vtkFileを作成する
        createPointSetVtkFile(folderDir, name, newPointLocs)

#-------------------
#  fitAllSetsFiles
#-------------------
def fitAllSetsFiles(folderDir):
    """ setsfolderの内容とvtkfolderの内容を合わせる。
    （余分なvtkを削除する。）
    setsを削除した場合、vtk側が削除されない為"""
    delFiles = []
    vtkFiles = glob.glob(folderDir + "/VTK/*")
    setsFiles = glob.glob(folderDir + "/sets/*")
    if len(setsFiles) > 0:
        setsNames = list(map(lambda x: os.path.basename(x), setsFiles))
    for vtkFile in vtkFiles:
        setsVtk = os.path.basename(vtkFile)
        words = setsVtk[:-4].split("_")[1:]
        name = "_".join(words)
        if not (name in setsNames):
            delFiles.append(vtkFile)
    if len(delFiles) > 0:
        for delFile in delFiles:
            os.remove(delFile)

#-------------
#  getMeshDir
#-------------
def getMeshDir(caseDir, timeFolder, region):
    """ polyMeshフォルダを検索して返す"""
    #polyMeshフォルダ（folderDir）を探す
    relDir = pyTreeFoam.case(caseDir).getCurrMeshDir(timeFolder, region, "*")
    folderDir = caseDir + "/" + relDir
    folderDir = os.path.normpath(folderDir)
    return folderDir

#========================
#  createAllSetsVtkFiles
#========================
def createAllSetsVtkFiles(caseDir, timeFolder, region, checkFlag=True):
    """ 該当するtimeFolder、regionに存在するsetsからvtkFileを作成する
    checkFlagがTrueの場合、setsFileのtimeStampを確認してvtk作成有無を決定。
    
    Args:
        caseDir(str)        :caseDir名
        timeFolder(str)     :timeFolder名（ここから遡ってmeshFileを検索）
        region(str)         :region名（regin0は、「.」を指定
        checkFlag(bool)     :True=timeStampを確認してvtkを作成。
                            :False=無条件で作成
                            :(default=true)
    """
    #polyMeshフォルダ（folderDir）を探す
    folderDir = getMeshDir(caseDir, timeFolder, region)
    #sets名を取得
    cellSets, faceSets, pointSets = getCellFacePontSetsNames(folderDir)
    if len(cellSets) + len(faceSets) + len(pointSets) == 0:
        print("region '" + region +"' : no sets files. could not create vtkFile of sets.")
        return
    #timeStampをチェックする？
    if checkFlag == True:
        cellSets, faceSets, pointSets = getNewSetsFile(folderDir, cellSets, faceSets, pointSets)
    #setsが存在するか？
    if len(cellSets) + len(faceSets) + len(pointSets) > 0:
        #setsが存在する場合
        print("create sets vtk file.")
        #fullpathのmeshFileを取得
        (owner, neighbour,
            faces, points) = getMeshDataFileNames(caseDir, timeFolder, region)
        #owner, neighbourの読込
        ownerFaces, neibFaces, allOwners = getOwnerNeighbour(owner, neighbour)
        #points, facesの読込
        allFaces, allPoints = getFacesPoints(faces, points)
        #Setsのvtkを作成する(作成場所はpolyMesh内)
        createAllCellSetVtkFiles(folderDir, cellSets, ownerFaces, neibFaces, allOwners, allFaces, allPoints)
        createAllFaceSetVtkFiles(folderDir, faceSets, allFaces, allPoints)
        createAllPointSetVtkFiles(folderDir, pointSets, allPoints)
        #余分なvtkがあれば削除
        fitAllSetsFiles(folderDir)
        print("created vtk in " + folderDir + "/VTK.")
    else:
        #setsが存在しない場合
        #　余分なvtkがあれば削除
        fitAllSetsFiles(folderDir)
        print("vtk files of sets have been created.")

#
# getArguments
#
def getArguments(caseDir, timeFolder, region, check):
    """ 引数を取得する。"""
    flags = [0, 0, 0, 0]
    for i in range(1, len(sys.argv)):
        arg = sys.argv[i]
        if arg == "-case":
            flags = [1, 0, 0, 0]
        elif arg == "-time":
            flags = [0, 1, 0, 0]
        elif arg == "-region":
            flags = [0, 0, 1, 0]
        elif arg == "-check":
            flags = [0, 0, 0, 1]
        else:
            if flags[0] == 1:
                caseDir = arg
            elif flags[1] == 1:
                timeFolder = arg
            elif flags[2] == 1:
                region = arg
            elif flags[3] == 1:
                if arg == "no" or arg == "off":
                    check = False
                else:
                    check = True
    return caseDir, timeFolder, region, check


if __name__ == "__main__":
    #default値を設定
    caseDir = os.getcwd()
    timeFolder = "0"
    region = "."
    check = True
    #引数を取得
    (caseDir, timeFolder,
        region, check) = getArguments(caseDir, timeFolder,
                                      region, check)
    #vtkFileを作成する。
    createAllSetsVtkFiles(caseDir, timeFolder, region, check)


