#!/usr/bin/python3
# coding: utf-8
#
#       meshOperationDialog.py
#
#       メッシュ操作に関する処理
#
#   11/04/10    新規作成
#   12/12/09    領域分割方法をOFver2.0.0以降のchtMultiRegionに対応
#   13/01/08    changeDict編集ボタンを追加
#      01/25    ﾊﾞｸﾞ修正（zone名変更時に違うzoneが変更される事がある）
#      02/09    changeDictの編集、実行方法を変更
#      04/07    メッシュ抽出を削除
#      04/21    pyOFLauncherをpyMeshOperationに変更
#      05/20    makeRegionPropertiesの内容をversionで変更する。
#      05/21    changeDictにUのfixedValueを追加
#      06/29    makeInternalPatchをOF-2.2.0用を追加
#      06/30    setsファイルを各regionに配布する様に修正
#      09/25    makeSetsFileInRegionをgzip、binaryに対応させる
#      10/15    国際化のため修正
#      11/15    modelの場所が変更できるようtextBox追加
#      12/08    csvによるmesh作成を追加
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/02/08    snappyDialogを表示させる時、closeする様に修正
#      05/03    領域分割方法をOFver2.3.0に対応
#      07/21    メッシュ分割時、T fieldが存在しない場合ｴﾗｰ発生を修正
#      07/26    meshSplitRegion:constant/polyMeshの削除を追加
#      10/05    meshSplitRegion:fluidRegions, solidRegions内のfileｺﾋﾟｰと
#               includefileのｺﾋﾟｰ。界面の境界条件設定を追加
#      10/13    copyIncludeFile:バグ修正
#      10/24    checkNullPatchInAddFieldsを追加
#      11/11    copyIncludeFile:errorメッセージ修正
#      12/09    changeCellZoneName, changeFaceZoneName, openPolyMesh,
#               meshSplitCellZones:meshDirをcurrMeshに変更
#               onOpenPolyMesh:meshが無かった場合、caseDirを開く
#      12/25    copyFilesInRegions:fluidSolidRegions内の「***.**」ﾌｧｲﾙｺﾋﾟｰの
#               バグ修正。
#   15/02/07    meshSplitRegion:timeFolder内のfield作成方法を修正。
#               constant、system、timeFolderとも同じ方法に設定した。
#      02/11    runBlockMesh:FOAM端末上で実行する様に修正
#      07/13    helyxOS_dataの場所変更
#      07/17    dialog起動方法を修正
#      10/02    changeCellZoneName, changeFaceZoneName, openPolyMeshを削除
#               cfMesh、checkMeshボタンを追加
#      11/28    snappyHexMesh:OF-3.0.0用を追加
#   16/09/08    meshSplitRegion:mesh分割をOF-4.0用を追加
#      12/30    replacementDict400,replacementSolidDict400:OF-4.0用に
#               changeDictionaryの書式を修正（バグ）
#      12/31    snappyHexMesh,makeInternalPatch,makeRegionProperties,
#               copyIncludeFile,replacementDict,replacementSolidDict:
#               OF-ext用に修正
#   17/02/27    runBlockMesh,checkMesh:gnome-terminalの起動オプション「-t」を
#               「--name=」に変更。（警告発生するため）
#      08/04    caseDirを引数で取得するように修正
#               getModelDir:相対参照でdirを取得する方法を変更。
#   18/06/05    gnome-terminalのオプション「-x」を「--」に変更
#   19/01/23    meshSplitRegion:multiRegion用のmesh分割時の処理を修正。
#               setFields後に分割した時、setFieldsの結果が反映されない事を修正
#      02/27    remainFieldsInTimeFolder,copyFilesInRegions:
#               対象のfile名に「.」が有った場合、コピーされないので修正。
#      05/22    makeSetsFileInRegion:mkdirをos.mkdirに修正
#   19/10/09    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#      05/28    setFolderDir:絶対バスに変更後処理する様に修正。
#   21/06/03    runBlockMesh,runCheckMesh:バグ修正（chmode→chmodに修正） 
#   22/01/09    scaleChange:OF-9.0対応。
#               OF-9.0からtransformPointsコマンドが変更された為修正。
#      01/25    makeRegionProperties:バグ修正
#               （regionPropertiesが別な場所に保存されていた）
#      02/01    setFolderDir:dirをそのまま設定する様に修正
#      07/18    createTransformCommLine:OFversionのcheck方法を変更（OF-10対応） 
#      07/19    replacementDict,makeRegionProperties,replacementSolidDict,
#               copyIncludeFile:OFversionのcheck方法を修正（OF-10対応）
#   23/05/27    ideasUnvToFoam:mesh変換後、renumberMeshを追加。
#      08/09    ideasUnvToFoam:renumberを削除。2次元用のmeshでエラー発生する時ある為。
#      11/30    scaleChange:モデルサイズ取得方法をvtkのOpenFOAMReaderに変更。
#   24/02/08    makeRegionProperties:OF-11用を追加。
#      07/29    openFileDialog:universalDialogs内の関数名変更による修正
#      07/31    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/05    inputTextDDialog:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#   25/07/03    createNewRegionCaseDDialog:新規作成。
#               regNameがrenameできる、終了後treefoam.pyにsignalを発信。
#      07/06    ideasUnvToFoam:okCancelDialog表示の順番を修正。
#


import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os, sys
import glob
import shutil
import signal

import pyTreeFoam
import getOpenFolderDDialog
import universalDialogs as unvDlg
import stringOp
import inputTextDDialog
import splitMeshCellZones
import separateCellZoneGroupDDialog
from createBCsForMultiRegionDialog import replaceRegionFieldBoundaryConditions
import createBafflesDialog
import createNewRegionCaseDDialog

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()

treefoamPid = ""        #treefoamのPID（regCase作成後TreeFoamにsignal送信の為）

#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"meshOperationDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.caseDir = caseDir
        #GUIのobject名を取得
        self.setGtkObject()
        #初期化
        self.initialize()

    def setGtkObject(self):
        """ glade内のobject名を取得"""
        self.entryMeshFolder = self.builder.get_object("entryMeshFolder")

    def main(self):
        """ GUIの表示"""
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    def initialize(self):
        """ 初期化
        blockMeshDictの場所を確認する。
        textBoxへの初期値を設定する。"""
        #blockMeshDictの場所確認
        fileName = os.sep.join([self.caseDir, "system", "blockMeshDict"])
        if len(glob.glob(fileName)) == 0:
            fileName = os.sep.join([self.caseDir, "constant", "polyMesh", "blockMeshDict"])
            if len(glob.glob(fileName)) == 0:
                self.blockMeshDir = ""
            else:
                self.blockMeshDir = "constant" + os.sep + "polyMesh"
        else:
            self.blockMeshDir = "system"
        #textBoxへの初期値を設定
        dataDict = pyTreeFoam.case(self.caseDir).getCreateMeshData()
        modelDir = dataDict["stlDir"]
        if modelDir == "":
            modelDir = "./model"
        self.entryMeshFolder.set_text(modelDir)
        #rootDirを取得
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        self.rootDir = self.configDict["rootDir"]
        return


    #------ event handler -------------------
    #blockMesh編集
    def onEditBlockMeshDict(self, event):
        self.editBlockMeshDict()
    #blockMesh実行
    def onRunBlockMesh(self, event):
        self.runBlockMesh()
    #checkMesh
    def onCheckMesh(self, event):
        self.runCheckMesh()
    #snappyHexMesh
    def onSnappyHexMesh(self, event):
        self.snappyHexMesh()
        self.close()
    #cfMesh
    def onCfMesh(self, event):
        self.cfMesh()
        self.close()
    #開く
    def onOpenFolder(self, event):
        self.openFolder()
    #参照
    def onSetFolderDir(self, event):
        self.setFolderDir()
    #ideasUnv2Foam
    def onIdeasUnvToFoam(self, event):
        self.ideasUnvToFoam()
    #スケール変更
    def onScaleChange(self, event):
        self.scaleChange()
    #内部patch作成
    def onMakeInternalPatch(self, event):
        self.makeInternalPatch()
    #領域分割（Region）
    def onMeshSplitCellZones(self, event):
        self.meshSplitCellZone()
    #閉じる
    def onClose(self, event):
        modelDir = self.entryMeshFolder.get_text()
        if modelDir != "":
            dataDict = {}
            dataDict["stlDir"] = modelDir
            pyTreeFoam.case(self.caseDir).setCreateMeshData(dataDict)
        self.close() 

    #
    #  editBlockMeshDict
    #  -----------------
    def editBlockMeshDict(self):
        """ blockMeshの編集"""
        print(_("blockMeshを編集"))
        fileName = os.sep.join([self.caseDir, self.blockMeshDir, "blockMeshDict"])
        if len(glob.glob(fileName)) == 0:
            title = _("エラー")
            mess = _("blockMeshDictが存在しません。\n")
            self.errDialog(title, mess)
            return
        pyTreeFoam.run().editor([fileName])

    #
    #  runBlockMesh
    #  ------------
    def runBlockMesh(self):
        """ blockMeshをterminal上で実行する。"""
        print(_("blockMeshを実行"))
        #configDict = pyTreeFoam.readConfigTreeFoam()
        envOpenFoam = self.configDict["bashrcFOAM"]
        blockDict = self.blockMeshDir + os.sep + "blockMeshDict"
        cont = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        cont += "blockMesh -dict " + blockDict + " 2>&1 | tee solve.log\n"
        #スクリプト作成
        comm = "echo " + "'" + cont + "'" + " > run"
        runCase = pyTreeFoam.run(self.caseDir)
        runCase.command(comm)
        runCase.command("chmod a+x run")
        runCase.foamTerminal("run")

    #
    #  runCheckMesh
    #  ------------
    def runCheckMesh(self):
        print(_("checkMeshを実行"))
        #configDict = pyTreeFoam.readConfigTreeFoam()
        envOpenFoam = self.configDict["bashrcFOAM"]
        cont = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        cont += "checkMesh 2>&1 | tee solve.log\n"
        #スクリプト作成
        comm = "echo " + "'" + cont + "'" + " > run"
        runCase = pyTreeFoam.run(self.caseDir)
        runCase.command(comm)
        runCase.command("chmod a+x run")
        runCase.foamTerminal("run")

    #
    #  snappyHexMesh
    #  -------------
    def snappyHexMesh(self):
        """ snappyHexMeshによるメッシュ作成"""
        print(_("snappyHexMeshによるメッシュ作成用dialogを表示"))
        comm = "editSnappyHexMeshDialog.py " + self.caseDir + " &"
        pyTreeFoam.run(self.caseDir).command(comm)
        # path = os.getenv("TreeFoamPath") + os.sep + "python"
        # modelDir = self.entryMeshFolder.get_text()
        # comm = "./snappyHexMeshDialog.py " + self.caseDir + " " + modelDir + " &"
        # pyTreeFoam.run(path).command(comm)

    #
    #  cfMesh
    #  ------
    def cfMesh(self):
        """ cfMeshによるメッシュっ作成"""
        print(_("cfMeshによるメッシュ作成用dialogを表示"))
        comm = "editCfMeshDialog.py " + self.caseDir + " &"
        pyTreeFoam.run(self.caseDir).command(comm)
        # path = os.getenv("TreeFoamPath") + os.sep + "python"
        # modelDir = self.entryMeshFolder.get_text()
        # comm = "./cfMeshDialog.py " + self.caseDir + " " + modelDir + " &"
        # pyTreeFoam.run(path).command(comm)


    #
    #  openFolder
    #  ----------
    def openFolder(self):
        """ folderを開く"""
        print(_("folderを開く"))
        modelDir = self.entryMeshFolder.get_text()
        if modelDir[0] != os.sep:
            openDir = self.caseDir + os.sep + modelDir
            openDir = os.path.abspath(openDir)
        else:
            openDir = modelDir
        if len(glob.glob(openDir)) == 0:
            mess = _("設定したfolderは存在しません。\n場所を再設定してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        fileManager = pyTreeFoam.fileManager
        comm = fileManager + " " + openDir + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  setFolderDir
    #  ------------
    def setFolderDir(self):
        """ modelFolderをtextBoxにセットする"""
        print(_("modelFolderを取得する"))
        selDir = self.entryMeshFolder.get_text()
        title = _("開く")
        cont = _("folderを選択する")
        rootDir = self.rootDir
        currDir = self.caseDir
        okArgs = [self.setFolderDir_setTextBox]
        dialog = getOpenFolderDDialog.getOpenFolder(title, cont, rootDir, currDir, selDir, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #  setFolderDir_setTextBox
    def setFolderDir_setTextBox(self, openDir):
        self.entryMeshFolder.set_text(openDir)

    #
    #  ideasUnvToFoam
    #  --------------
    def ideasUnvToFoam(self):
        """ unv形式のメッシュをfoam形式に変換する"""
        self.ideasUnvToFoam_getUnv()
        # print(_("mesh変換します。"))
        # title = _("メッシュ変換")
        # message = _("「ideasUnvToFoam」でメッシュ変換します。")
        # okArgs = [self.ideasUnvToFoam_getUnv]
        # self.okCancelDialog(title, message, funcOk=okArgs)
        # return

    #  ideasUnvToFoam_getUnv
    def ideasUnvToFoam_getUnv(self):
        """ unv形式のメッシュをfoam形式に変換する"""
        modelDir = self.entryMeshFolder.get_text()
        if modelDir[0] != os.sep:
            modelDir = self.caseDir + os.sep + modelDir
            modelDir = os.path.abspath(modelDir)
        #unvメッシュファイルを選択
        title = _("unvファイル選択")
        path = modelDir
        patterns = [["unv", "*.unv"], ["all", "*"]]
        #okArgs = [self.ideasUnvToFoam_run]
        okArgs = [self.ideasUnvToFoam_okCancel]
        self.openFileDialog(title, path, funcOk=okArgs, patterns=patterns)
        return
    
    #  ideasUnvToFoam_okCancel
    def ideasUnvToFoam_okCancel(self, fileName):
        print(_("mesh変換します。"))
        title = _("メッシュ変換")
        message = _("「ideasUnvToFoam」でメッシュ変換します。")
        okArgs = [self.ideasUnvToFoam_run, fileName]
        self.okCancelDialog(title, message, funcOk=okArgs)
        return

    
    #  ideasUnvToFoam_run
    def ideasUnvToFoam_run(self, fileName):
        #メッシュ変換
        envOpenFoam = self.configDict["bashrcFOAM"]
        caseRun = pyTreeFoam.run(self.caseDir)
        comm  = ". " + envOpenFoam + "; "
        comm += "ideasUnvToFoam " + fileName
        (stat, _res, _err) = caseRun.commandReturnCont(comm)
        if stat != "OK":
            title = _("ideasUnvToFoam 変換エラー")
            mess =_("変換エラーが発生しました。")
            self.errDialog(title, mess)
            return
        #internalFieldをクリアする
        case = pyTreeFoam.case(self.caseDir)
        timeFolder = case.getCurrTimeFolder()
        fields = case.getFieldNames(timeFolder)
        fieldDir = self.caseDir + os.sep + timeFolder
        case.clearInternalFields(fieldDir, fields)
        #boundaryFieldをクリアする
        polyMeshDir = case.getCurrMeshDir(timeFolder, ".", "boundary")
        meshDir = self.caseDir + os.sep + os.sep.join(polyMeshDir.split(os.sep)[:-1])
        case.clearNullBoundaryAllFields(timeFolder, meshDir)
        self.scaleChange()

    #
    #  scaleChange
    #  -----------
    def scaleChange(self):
        """ モデルのスケールを変更する"""
        import meshViewerDialog
        #モデルサイズを取得
        case = pyTreeFoam.case(self.caseDir)
        timeFolder = case.getCurrTimeFolder()
        caseDir = self.caseDir
        region = "."
        inputFile = ""
        title = ""
        viewer = meshViewerDialog.windowApp(caseDir, timeFolder, region, inputFile, title)
        xyz = self.getOutlineActorBounds(viewer, region)
        #xyz = self.getModelSize(polyMeshDir)
        #scaleを取得
        title = _("モデルのscale変更")
        #mess = _("現在のmodelSize XYZ: ") + " x ".join(list(map(str, xyz)))
        mess = _("現在のmodelSize XYZ: ") + " x ".join(xyz)
        mess += "\n\n" + _("変更するscaleの値を入力してください")
        iniText = "1.0"
        okArgs = [self.scaleChange_run, viewer, region]
        dialog = inputTextDDialog.getInputText(title, mess, iniText, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return
        
    #  scaleChange_run
    def scaleChange_run(self, scale, viewer, region):
        try:
            scale = float(scale)
        except:
            return
        if scale == 1.0:
            return
        #スクリプトを作成、実行
        scale = str(scale)
        envOpenFoam = self.configDict["bashrcFOAM"]
        caseRun = pyTreeFoam.run(self.caseDir)
        comm  = ". " + envOpenFoam + "; "
        comm += self.createTransformCommLine(scale)
        (stat, _cont, _err) = caseRun.commandReturnCont(comm)
        if stat != "OK":
            title = _("スケールの変更")
            mess = _("エラーが発生しました。")
            self.errDialog(title, mess)
            return
        xyz = self.getOutlineActorBounds(viewer, region)
        newSize = " x ".join(xyz)
        title = _("スケールの変更")
        mess = _("モデルのスケールは、「") + scale + _("」倍に変更されました。")
        mess += "\n" + _("現在のmodelSize: ") + newSize
        self.okDialog(title, mess)
        
    def getOutlineActorBounds(self, viewer, region):
        """ meshViewerDialog内の関数を使って、サイズを取得"""
        viewer.iniGetPatchesZonesActors()
        viewer.getPatchesZonesActorsAllRegions()
        outlineDict = viewer.outlineActorDict[region]
        outlineActors = list(outlineDict.values())
        actors = outlineActors
        bounds = []
        for actor in actors:
            bound = actor.GetBounds()
            bounds.append(bound)
        Xmin = min(map(lambda x: x[0], bounds))
        Xmax = max(map(lambda x: x[1], bounds))
        Ymin = min(map(lambda x: x[2], bounds))
        Ymax = max(map(lambda x: x[3], bounds))
        Zmin = min(map(lambda x: x[4], bounds))
        Zmax = max(map(lambda x: x[5], bounds))
        x, y, z = Xmax-Xmin, Ymax-Ymin, Zmax-Zmin
        #3桁の文字列に変換
        xyz = [
            pyTreeFoam.getValue3digit_shrink(x),
            pyTreeFoam.getValue3digit_shrink(y),
            pyTreeFoam.getValue3digit_shrink(z)
            ]
        return xyz

    #
    #  createTransformCommLine
    def createTransformCommLine(self, scale):
        """ transformPoints コマンド行を作成する。
        OF-9からコマンド変更"""

        def createCommLine_org(scale):
            """
            transformPoints -scale '(2 2 2)'
            """
            argLine = " '(" + scale + " " + scale + " " + scale + ")'"
            line = "transformPoints -scale" + argLine
            return line

        def createCommLine_9(scale):
            """
            transformPoints 'scale=(2 2 2)'
            """
            argLine = " 'scale=(" + scale + " " + scale + " " + scale + ")'"
            line = "transformPoints" + argLine
            return line

        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion >= "v1804":
            line = createCommLine_org(scale)
        elif OFversion[:3] == "ext":
            line = createCommLine_org(scale)
        #elif OFversion >= "9":
        elif numVer >= 9.0:
            line = createCommLine_9(scale)
        else:
            line = createCommLine_org(scale)
        return line

    #
    #  makeRegionProperties
    def makeRegionProperties(self, fluidRegions, solidRegions):
        """ regionPropertiesを作成する"""

        def writeRegionProperties(contents):
            """ regionPropertiesに書き込む"""
            fileName = self.caseDir + "/regCase/constant/regionProperties"
            f = open(fileName, "w")
            f.write(contents)
            f.close()

        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion[:3] == "ext":
            #version extの場合
            #foamFile作成
            getFoamContents = pyTreeFoam.getFoamContents(self.caseDir)
            header = getFoamContents.makeFoamHeaderVersion(OFversion)
            header += getFoamContents.makeFoamFile("2.0", "ascii", "dictionary",
                    "constant", "regionProperties")
            footer = getFoamContents.makeFoamFooter()
            #regionPropertiesの内容作成
            contents = "fluidRegionNames ( "
            for name in fluidRegions:
                contents += name + " "
            contents += ");\n\n"
            contents += "solidRegionNames ( "
            for name in solidRegions:
                contents += name + " "
            contents += ");\n"
            contentsDict = header + contents + footer
            writeRegionProperties(contentsDict)

        elif numVer >= 11.0:
            #version 11.0以降の場合
            #regionSolversの内容を作成
            cont = ""
            for name in fluidRegions:
                cont += name + " fluid;\n"
            for name in solidRegions:
                cont += name + " solid;\n"
            #controlDictへの書き込み
            setDict = {"regionSolvers": cont}
            regCase = self.caseDir + "/regCase"
            case = pyTreeFoam.case(regCase)
            case.setControlDict(setDict)

        elif numVer >= 2.2 or OFversion[0] == "v":
            #version 2.2.0以降、OF-vの場合
            #foamFile作成
            getFoamContents = pyTreeFoam.getFoamContents(self.caseDir)
            header = getFoamContents.makeFoamHeaderVersion(OFversion)
            header += getFoamContents.makeFoamFile("2.2", "ascii", "dictionary",
                    "constant", "regionProperties")
            footer = getFoamContents.makeFoamFooter()
            #regionPropertiesの内容作成
            contents = "regions\n(\n"
            contents += "    fluid       ( "
            for name in fluidRegions:
                contents += name + " "
            contents += ")\n"
            contents += "    solid       ( "
            for name in solidRegions:
                contents += name + " "
            contents += ")\n);\n"
            contentsDict = header + contents + footer
            writeRegionProperties(contentsDict)

        else:
            #version 2.1.1以前の場合
            #foamFile作成
            getFoamContents = pyTreeFoam.getFoamContents(self.caseDir)
            header = getFoamContents.makeFoamHeaderVersion(OFversion)
            header += getFoamContents.makeFoamFile("2.0", "ascii", "dictionary",
                    "constant", "regionProperties")
            footer = getFoamContents.makeFoamFooter()
            #regionPropertiesの内容作成
            contents = "fluidRegionNames ( "
            for name in fluidRegions:
                contents += name + " "
            contents += ");\n\n"
            contents += "solidRegionNames ( "
            for name in solidRegions:
                contents += name + " "
            contents += ");\n"
            contentsDict = header + contents + footer
            writeRegionProperties(contentsDict)

    #
    #  makeChangeDictionaryAllRegionsDict
    def makeChangeDictionaryAllRegionsDict(self, regCaseDir, fluidRegions, solidRegions):
        """ changeDictionaryAllRegionsDictを作成する。
        OpenFOAMのバージョンに応じて作成"""
        OFversion = self.configDict["OFversion"]
        times = pyTreeFoam.case(regCaseDir).getTimeFolders()
        timeFolder = times[0]
        #各region毎にchangeDictionaryRegionsDict作成
        c = []
        c.append(self.headerDict(OFversion))
        #fluidRegionの設定
        for regName in fluidRegions:
            field = "T"
            fieldDir = regCaseDir + "/" + timeFolder + "/" + regName + "/" + field
            if len(glob.glob(fieldDir)) != 0:
                patches = self.getPatches(fieldDir)
                repContents = self.replacementDict(regName, field, patches, OFversion)
                c.append(regName + "\n")
                c.append("{\n")
                c.append(repContents)
                c.append("}\n")
                c.append("\n")
        #solidRegionの設定
        for regName in solidRegions:
            field = "T"
            fieldDir = regCaseDir + "/" + timeFolder + "/" + regName + "/" + field
            if len(glob.glob(fieldDir)) != 0:
                patches = self.getPatches(fieldDir)
                repContents = self.replacementSolidDict(regName, field, patches, OFversion)
                c.append(regName + "\n")
                c.append("{\n")
                c.append(repContents)
                c.append("}\n")
                c.append("\n")
        c.append(self.footerDict())
        f=open(regCaseDir + "/system/changeDictionaryAllRegionsDict", "w")
        for line in c:
            f.write(line)
        f.close()
        print(_(u"\nsystemフォルダ内に「changeDictionaryAllRegionsDict」を作成しました。"))

    def headerDict(self, OFversion):
        """ header部を作成する"""
        getFoamContents = pyTreeFoam.getFoamContents()
        header = getFoamContents.makeFoamHeaderVersion(OFversion)
        #foamFile = getFoamContents.makeFoamFile(OFversion, "ascii", "dictionary",
        #    "system", "changeDictionaryAllRegionsDict")
        foamFile = getFoamContents.makeFoamFile("2.0", "ascii", "dictionary",
            "system", "changeDictionaryAllRegionsDict")
        header += foamFile
        return header
        
    def footerDict(self):
        getFoamContents = pyTreeFoam.getFoamContents()
        footer = getFoamContents.makeFoamFooter()
        return footer

    def getPatches(self, fieldDir):
        """ fieldのpatch（region同士の界面のpatch）を取得"""
        regName = fieldDir.split("/")[-2]
        fileName = fieldDir
        foamFile = pyTreeFoam.foamFile()
        cont = foamFile.read(fileName)
        contOp = stringOp.strings(cont)
        (boundaryCont, _p) = contOp.getKeywordContents("boundaryField", 0)
        lines = boundaryCont.decode().split("\n")
        patches = []
        for line in lines:
            n = line.find(regName + "_to_")
            if n >= 0:
                patch = line.split()[0]
                patches.append(patch)
        return patches

    #  replacementDict
    def replacementDict(self, regName, field, patches, OFversion):
        """ OpenFOAMのバージョンに応じたfluid側のdictを作成する"""
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion[:3] == "ext":
            cont = self.replacementDict171(field, patches)
        #elif OFversion >= "4.0":
        elif numVer >= 4.0 or OFversion[0] == "v":
            cont = self.replacementDict400(field, patches)
        #elif OFversion >= "2.3.0":
        elif numVer >= 2.3:
            cont = self.replacementDict230(field, patches)
        #elif OFversion >= "2.2.0":
        elif numVer >= 2.2:
            cont = self.replacementDict220(field, patches)
        #elif OFversion >= "2.0.0":
        elif numVer >= 2.0:
            cont = self.replacementDict200(field, patches)
        else:
            cont = self.replacementDict171(field, patches)
        return cont

    def replacementDict400(self, field, patches):
        c=""
        c=c+    "    " + field + "\n"
        c=c+    "    {\n" 
        c=c+    "        boundaryField\n"
        c=c+    "        {\n"
        for patch in patches:
            c=c+"            " + patch + "\n"
            c=c+"            {\n"
            c=c+"                type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                value           uniform 300;\n"
            c=c+"                Tnbr            " + field + ";\n"
            c=c+"                kappaMethod     fluidThermo;\n"
            c=c+"            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        c=c+    "    U\n"
        c=c+    "    {\n" 
        c=c+    "        boundaryField\n"
        c=c+    "        {\n"
        for patch in patches:
            c=c+"            " + patch + "\n"
            c=c+"            {\n"
            c=c+"                type        fixedValue;\n"
            c=c+"                value       uniform (0 0 0);\n"
            c=c+"            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementDict230(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                    value           uniform 300;\n"
            c=c+"                    Tnbr            " + field + ";\n"
            c=c+"                    kappa           fluidThermo;\n"
            c=c+"                    kappaName       none;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "        U\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type        fixedValue;\n"
            c=c+"                    value       uniform (0 0 0);\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementDict220(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                    value           uniform 300;\n"
            c=c+"                    neighbourFieldName  " + field + ";\n"
            c=c+"                    kappa           fluidThermo;\n"
            c=c+"                    kappaName       none;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "        U\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type        fixedValue;\n"
            c=c+"                    value       uniform (0 0 0);\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementDict200(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                    value           uniform 300;\n"
            c=c+"                    neighbourFieldName  " + field + ";\n"
            c=c+"                    K               basicThermo;\n"
            c=c+"                    KName           none;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "        U\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type        fixedValue;\n"
            c=c+"                    value       uniform (0 0 0);\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementDict171(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            solidWallMixedTemperatureCoupled;\n"
            c=c+"                    neighbourFieldName  " + field + ";\n"
            c=c+"                    K               K;\n"
            c=c+"                    value           uniform 0;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "        U\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type        fixedValue;\n"
            c=c+"                    value       uniform (0 0 0);\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    #  replacementSolidDict
    def replacementSolidDict(self, regName, field, patches, OFversion):
        """ OpenFOAMのバージョンに応じたsolid側のdictを作成する"""
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion[:3] == "ext":
            cont = self.replacementSolidDict171(field, patches)
        #elif OFversion >= "4.0":
        elif numVer >= 4.0 or OFversion[0] == "v":
            cont = self.replacementSolidDict400(field, patches)
        #elif OFversion >= "2.3.0":
        elif numVer >= 2.3:
            cont = self.replacementSolidDict230(field, patches)
        #elif OFversion >= "2.2.0":
        elif numVer >= 2.2:
            cont = self.replacementSolidDict220(field, patches)
        #elif OFversion >= "2.0.0":
        elif numVer >= 2.0:
            cont = self.replacementSolidDict200(field, patches)
        else:
            cont = self.replacementSolidDict171(field, patches)
        return cont

    def replacementSolidDict171(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            solidWallMixedTemperatureCoupled;\n"
            c=c+"                    neighbourFieldName  " + field + ";\n"
            c=c+"                    K               K;\n"
            c=c+"                    value           uniform 0;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementSolidDict400(self, field, patches):
        c=""
        c=c+    "    " + field + "\n"
        c=c+    "    {\n" 
        c=c+    "        boundaryField\n"
        c=c+    "        {\n"
        for patch in patches:
            c=c+"            " + patch + "\n"
            c=c+"            {\n"
            c=c+"                type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                value           uniform 300;\n"
            c=c+"                Tnbr            " + field + ";\n"
            c=c+"                kappaMethod     solidThermo;\n"
            c=c+"            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementSolidDict230(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                    value           uniform 300;\n"
            c=c+"                    Tnbr            " + field + ";\n"
            c=c+"                    kappa           solidThermo;\n"
            c=c+"                    kappaName       none;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementSolidDict220(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                    value           uniform 300;\n"
            c=c+"                    neighbourFieldName  " + field + ";\n"
            c=c+"                    kappa           solidThermo;\n"
            c=c+"                    kappaName       none;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    def replacementSolidDict200(self, field, patches):
        c=""
        c=c+    "    dictionaryReplacement\n"
        c=c+    "    {\n"
        c=c+    "        " + field + "\n"
        c=c+    "        {\n" 
        c=c+    "            boundaryField\n"
        c=c+    "            {\n"
        for patch in patches:
            c=c+"                " + patch + "\n"
            c=c+"                {\n"
            c=c+"                    type            compressible::turbulentTemperatureCoupledBaffleMixed;\n"
            c=c+"                    value           uniform 300;\n"
            c=c+"                    neighbourFieldName  " + field + ";\n"
            c=c+"                    K               solidThermo;\n"
            c=c+"                    KName           none;\n"
            c=c+"                }\n"
        c=c+    "            }\n"
        c=c+    "        }\n"
        c=c+    "    }\n"
        return c

    #  runChangeDictionaryAllRegions
    def runChangeDictionaryAllRegions(self, regDir):
        """ changeDictionaryの実行"""
        envOpenFoam = self.configDict["bashrcFOAM"]
        path = os.getenv("TreeFoamPath") + os.sep + "python"
        comm = ". " + envOpenFoam + "; "
        comm += path + "/changeDictionaryAllRegions.py"
        pyTreeFoam.run(regDir).command(comm)

    #  makeSetsFileInRegion
    def makeSetsFileInRegion(self, caseDir, regions):
        """ 各region毎にsetsfile作成"""
        iniTime = pyTreeFoam.case(caseDir).getTimeFolders()[0]
        #分割前のmesh
        meshDir = caseDir + "/" + iniTime
        if len(glob.glob(meshDir+"/polyMesh/boundary")) == 0:
            meshDir = caseDir + "/constant"
        #regionのmesh
        regMeshDir = caseDir + "/" + iniTime
        if len(glob.glob(regMeshDir+"/"+regions[0]+"/polyMesh/boundary")) == 0:
            regMeshDir = caseDir + "/constant"
        #分割前のsetsFile取得
        setsFiles = pyTreeFoam.getFileNames(meshDir+"/polyMesh/sets")
        cellSets = []
        faceSets = []
        pointSets = []
        for setsFile in setsFiles:
            fileName = meshDir + "/polyMesh/sets/" + setsFile
            cont = pyTreeFoam.foamFile().readFull(fileName)
            (foamCont, _p) = stringOp.strings(cont).getKeywordContents("FoamFile", 0)
            foamLines = foamCont.decode().split("\n")
            for line in foamLines:
                words = line.split()
                if len(words) == 2:
                    if words[0] == "class":
                        (name, _p) = stringOp.strings(words[1]).getKeyword(0)
                        name = name.decode()
                        if name == "cellSet":
                            cellSets.append(setsFile)
                        elif name == "faceSet":
                            faceSets.append(setsFile)
                        elif name == "pointSet":
                            pointSets.append(setsFile)
        #各regionにsets作成
        for region in regions:
            #setsフォルダ作成
            folderDir = regMeshDir + "/" + region + "/polyMesh/sets"
            os.mkdir(folderDir)
        #cellSet作成
        OFversion = pyTreeFoam.readConfigTreeFoam()["OFversion"]
        for cellSet in cellSets:
            print("cellSet:", cellSet, _("を確認中..."))
            #allCell取得
            sourceFile = meshDir + "/polyMesh/sets/" + cellSet
            cells = self.getFileData(sourceFile)
            #各region毎に作成
            for region in regions:
                #meetするcellを取得
                IdFile = regMeshDir + "/" + region + "/polyMesh/cellRegionAddressing"
                cellAdds = self.getFileData(IdFile)
                newCells = self.getMeetSetsData(cells, cellAdds)
                if len(newCells) > 0:
                    #meetするcellを保存
                    data = ["cellSet", cellSet, newCells]
                    fileName = regMeshDir + "/" + region + "/polyMesh/sets/" + cellSet
                    self.createNewSetsFile(OFversion, fileName, data)
        #faceSet作成
        for faceSet in faceSets:
            print("faceSet:", faceSet, _(u"を確認中..."))
            sourceFile = meshDir + "/polyMesh/sets/" + faceSet
            faces = self.getFileData(sourceFile)
            #各region毎に作成
            for region in regions:
                #meetするcellを取得
                IdFile = regMeshDir + "/" + region + "/polyMesh/faceRegionAddressing"
                faceAdds = self.getFileData(IdFile)
                newFaces = self.getMeetSetsData(faces, faceAdds)
                if len(newFaces) > 0:
                    #meetするcellを保存
                    data = ["faceSet", faceSet, newFaces]
                    fileName = regMeshDir + "/" + region + "/polyMesh/sets/" + faceSet
                    self.createNewSetsFile(OFversion, fileName, data)
        #pointSet作成
        for pointSet in pointSets:
            print("pointSet:", faceSet, _(u"を確認中..."))
            sourceFile = meshDir + "/polyMesh/sets/" + pointSet
            points = self.getFileData(sourceFile)
            #各region毎に作成
            for region in regions:
                #meetするcellを取得
                IdFile = regMeshDir + "/" + region + "/polyMesh/pointRegionAddressing"
                pointAdds = self.getFileData(IdFile)
                newPoints = self.getMeetSetsData(points, pointAdds)
                if len(newPoints) > 0:
                    #meetするcellを保存
                    data = ["pointSet", pointSet, newPoints]
                    fileName = regMeshDir + "/" + region + "/polyMesh/sets/" + pointSet
                    self.createNewSetsFile(OFversion, fileName, data)

    def getFileData(self, fileName):
        """ file内のデータをlist形式の整数で取得"""
        (ascii, data) = pyTreeFoam.foamFile().readSeparate(fileName)
        ans = []
        foamFile = pyTreeFoam.foamFile()
        if foamFile.checkBinary(ascii) == "binary":
            numData = foamFile.structBinData(data[0])
            for n in numData:
                ans.append(n)
        else:
            numList = data[0][1].split()
            for n in numList:
                if len(n) > 0:
                    ans.append(int(n))
        ans.sort()
        return ans

    def getMeetSetsData(self, cells, cellAdds):
        """ 合致するSetsDataを取得"""
        newCells = []
        n = 0
        i = 0
        while n < len(cells) and i < len(cellAdds):
            cell = cells[n]
            cellAdd = cellAdds[i]
            if cell == cellAdd:
                newCells.append(i)
                i += 1
                n += 1
            elif cell > cellAdd:
                i += 1
            else:
                n += 1
        return newCells        

    def createNewSetsFile(self, OFversion, fileName, data):
        """ setsFileを保存"""
        className = data[0]
        name = data[1]
        numCells = data[2]
        #数字を文字に変換
        newCells = []
        for numCell in numCells:
            newCells.append(str(numCell))
        #保存内容を取得
        foamContents = pyTreeFoam.getFoamContents()
        header = foamContents.makeFoamHeaderVersion(OFversion)
        name = pyTreeFoam.case(caseDir).getOrgFileName(name)    #非圧縮のfile名に戻す
        foamFile = foamContents.makeFoamFile("2.0", "ascii", className, "", name)
        cont = self.makeFileData(newCells)
        footer = foamContents.makeFoamFooter()
        #保存
        writeCont = header + foamFile + cont + footer
        pyTreeFoam.foamFile().writeFull(fileName, writeCont)

    def makeFileData(self, cells):
        """ fileData作成"""
        cont = str(len(cells)) + "\n"
        cont += "(\n"
        cont += "\n".join(cells) + "\n"
        cont += ")\n"
        return cont

    #  copyFilesInCase
    def copyFilesInCase(self, regDir):
        """ case直下のfileをコピーする"""
        caseDir = "/".join(regDir.split("/")[:-1])
        files = glob.glob(caseDir + "/*")
        for fileName in files:
            if os.path.isfile(fileName) == True:
                shutil.copy(fileName, regDir)

    #  copyFilesInFluidRegions
    def copyFilesInFluidRegions(self, regCaseDir, timeFolder, fluidRegions):
        """ fluidRegions内のfileを各regionにコピー"""
        if len(fluidRegions) == 0:
            return
        baseDir = regCaseDir + "/" + timeFolder
        self.remainFieldsInTimeFolder(baseDir, "fluidRegions", fluidRegions)
        self.clearBoundaryFieldsAllRegions(regCaseDir, timeFolder, fluidRegions)
        baseDir = regCaseDir + "/constant"
        self.copyFilesInRegions(baseDir, "fluidRegions", fluidRegions)
        baseDir = regCaseDir + "/system"
        self.copyFilesInRegions(baseDir, "fluidRegions", fluidRegions)

    #  copyFilesInSolidRegions
    def copyFilesInSolidRegions(self, regCaseDir, timeFolder, solidRegions):
        """ solidRegions内のfileを各regionにコピー"""
        if len(solidRegions) == 0:
            return
        baseDir = regCaseDir + "/" + timeFolder
        self.remainFieldsInTimeFolder(baseDir, "solidRegions", solidRegions)
        self.clearBoundaryFieldsAllRegions(regCaseDir, timeFolder, solidRegions)
        baseDir = regCaseDir + "/constant"
        self.copyFilesInRegions(baseDir, "solidRegions", solidRegions)
        baseDir = regCaseDir + "/system"
        self.copyFilesInRegions(baseDir, "solidRegions", solidRegions)

    #  remainFieldsInTimeFolder
    def remainFieldsInTimeFolder(self, baseDir, fluidSolidRegions, regNames):
        """ timeFolder内の必要なfieldを残す"""
        sourceDir = baseDir + "/" + fluidSolidRegions
        remFields = glob.glob(sourceDir + "/*")     #残すfield名
        #残すfieldが無い場合は、戻る（全て残す）
        if len(remFields) == 0:
            return
        #残すfield以外を削除する
        for regName in regNames:
            nameDir = baseDir + "/" + regName
            allFields = glob.glob(nameDir + "/*")
            for field in allFields:
                flag = 0
                fieldName = field.split("/")[-1]
                for remField in remFields:
                    remName = remField.split("/")[-1]
                    nameList = remName.split(".")
                    if len(nameList) == 1 or not (nameList[-1] in regNames):
                        if fieldName == remName:
                            flag = 1
                            break
                    elif nameList[-1] in regNames:
                        if regName == ".".join(nameList[:-1]):
                            if fieldName == remName:
                                flag = 1
                                break
                if flag == 0:
                    #削除
                    os.remove(field)
            #足りないfieldを追加する
            resFieldsDir = glob.glob(nameDir + "/*")
            resFields = list(map(lambda x: os.path.basename(x), resFieldsDir))
            needFields = list(map(lambda x: os.path.basename(x), remFields))
            addFields = set(needFields).difference(resFields)
            for addField in addFields:
                addFieldPath = sourceDir + "/" + addField
                pasteDir = nameDir
                shutil.copy(addFieldPath, pasteDir)

    #clearBoundaryFieldsAllRegions
    def clearBoundaryFieldsAllRegions(self, regCaseDir, timeFolder, regions):
        """ boundaryFieldの整合性を確認し、整合をとる"""
        for region in regions:
            case = pyTreeFoam.case(regCaseDir)
            relMeshDir = case.getCurrMeshDir(timeFolder, region, "boundary")
            meshDir = regCaseDir + "/" + relMeshDir
            case.clearNullBoundaryAllFields(timeFolder, meshDir, region)

    #  copyFilesInRegions
    def copyFilesInRegions(self, baseDir, fluidSolidRegions, regNames):
        """ regionにfileをコピーする"""
        sourceDir = baseDir + "/" + fluidSolidRegions
        files = glob.glob(sourceDir+"/*")
        for fileName in files:
            nameList = (fileName.split("/")[-1]).split(".")
            if len(nameList) == 1 or not(nameList[-1] in regNames):
                for regName in regNames:
                    sourceName = fileName
                    pasteFolder = baseDir + "/" + regName
                    shutil.copy(sourceName, pasteFolder)
                    #comm = "cp -f " + sourceName + " " + pasteFolder
                    #os.system(comm)
        for fileName in files:
            nameList = (fileName.split("/")[-1]).split(".")
            if len(nameList) > 1 and (nameList[-1] in regNames):
                nameDir = "/".join(fileName.split("/")[:-1])
                sourceName = nameDir + "/" + ".".join(nameList)
                pasteFolder = baseDir + "/" + nameList[-1]
                pasteName = pasteFolder + "/" + ".".join(nameList[:-1])
                shutil.copy(sourceName, pasteName)
                #comm = "cp -f " + sourceName + " " + pasteName
                #os.system(comm)

    #  copyIncludeFile
    def copyIncludeFile(self, regCaseDir, fluidRegions, solidRegions):
        """ includeファイルや境界条件を再設定"""
        #config = pyTreeFoam.readConfigTreeFoam()
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        #if OFversion >= "2.2.0" and OFversion[:3] != "ext":
        if numVer >= 2.2 or OFversion[0] == "v":
            #境界条件を再設定
            (error, _mess) = replaceRegionFieldBoundaryConditions(regCaseDir,
                fluidRegions, solidRegions)
            if error != "OK":
                print(_("\nregionWallBoundaryConditionsDictのの内容は、反映されていません。"))
            else:
                print(_("\nregion間の境界条件を「regionWallBoundaryConditionsDict」に従って再設定しました。"))

    #
    #  makeInternalPatch
    #--------------------
    def makeInternalPatch(self):
        """ 内部patchの作成"""
        createBafflesDialog.Show(self.caseDir, treefoamPid)

    #
    #  meshSplitCellZone
    #  -----------------
    def meshSplitCellZone(self):
        """ cellZone毎にメッシュを領域分割する"""
        print(_("cellZone毎にメッシュを領域分割します。"))
        title = _("メッシュの領域分割")
        mess = _(u"メッシュをcellZone毎に領域分割します。")
        okArgs = [self.meshSplitCellZone_getFluidSolid]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return

    #  meshSplitCellZone_getFluidSolid        
    def meshSplitCellZone_getFluidSolid(self):
        """ cellZone毎にメッシュを領域分割する"""
        error = splitMeshCellZones.Run(self.caseDir)
        if error[0] != "OK":
            self.errDialog(_("エラー"), error[1])
            return
        print(_("領域（region）に分割しました。"))
        regDir = self.caseDir + "/regCase"
        case = pyTreeFoam.case(regDir)
        currTime = case.getCurrTimeFolder()
        reDir = case.getCurrMeshDir(currTime, ".", "boundary")
        polyMeshDir = regDir + "/" + reDir
        zoneNames = pyTreeFoam.getFoamContents().cellZoneNames(polyMeshDir)
        #流体、固体領域を設定
        okArgs = [self.meshSplitCellZone_run, regDir, zoneNames]
        dialog = separateCellZoneGroupDDialog.separateCellZoneGroup(zoneNames, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        
    #  meshSplitCellZone_run
    def meshSplitCellZone_run(self, result, regDir, zoneNames):
        #regionProperties作成
        (fluidZones, solidZones) = result
        self.makeRegionProperties(fluidZones, solidZones)

        #fluidRegions, solidRegionsフォルダ内のfileを各regionにコピー
        timeFolders = pyTreeFoam.case(regDir).getTimeFolders()
        self.copyFilesInCase(regDir)
        self.copyFilesInFluidRegions(regDir, timeFolders[0], fluidZones)
        self.copyFilesInSolidRegions(regDir, timeFolders[0], solidZones)
        #  regionWallBoundaryConditionsDictの内容を反映させる
        #  ここで、OFversionに応じたboundaryの内容がセットされる
        self.copyIncludeFile(regDir, fluidZones, solidZones)

        #changeDictionaryAllRegionsを作成
        self.makeChangeDictionaryAllRegionsDict(regDir, fluidZones, solidZones)
        #changeDictionaryAllRegionsを実行
        self.runChangeDictionaryAllRegions(regDir)
        #setsを各region毎に作成
        self.makeSetsFileInRegion(regDir, zoneNames)

        #終了処理
        newCaseName = "regCase"
        funcOk = [self.createNewRegionCase_run]
        parent = self.mainWindow
        dialog = createNewRegionCaseDDialog.dialogApp(newCaseName, funcOk=funcOk, parent=parent)
        dialog.show()

    #  createNewRegionCase_run
    def createNewRegionCase_run(self, newName):
        """ splitCellZone終了後の処理"""
        newRegName = self.caseDir + os.sep + newName
        if newName != "regCase":
            #case名変更
            currRegName = self.caseDir + os.sep + "regCase"
            if os.path.exists(newRegName) == True:
                if os.path.isfile(newRegName) == True:
                    os.remove(newRegName)
                else:
                    shutil.rmtree(newRegName)
            os.rename(currRegName, newRegName)
        #余分なfolderを削除
        #  constant/polyMeshを削除
        delFolder = newRegName + "/constant/polyMesh"
        shutil.rmtree(delFolder)
        #  数字folder内のfile
        timeFolder = pyTreeFoam.case(self.caseDir).getTimeFolders()[0]
        files = glob.glob(timeFolder+"/*")
        for fileName in files:
            if os.path.isfile(fileName) == True:
                os.remove(fileName)
        #変数を保存
        fileName = os.getenv("TreeFoamUserPath") + os.sep + "temp" + os.sep + "signalData"
        lines  = ["changeCursor\n"]
        lines += ["newDir " + newRegName + "\n"]
        f = open(fileName, "w"); f.writelines(lines); f.close()
        #TreeFoam側にsignal送信
        try:
            pid = int(treefoamPid)
            os.kill(pid, signal.SIGUSR1)
            print("send signal to TreeFoam.")
        except:
            pass

        # #終了message
        # title = _("メッシュの領域分割")
        # mess = _("メッシュをcellZone毎に領域分割し、新しい分割Case「regCase」を作成しました。") + "\n"
        # mess += _("解析caseを「regCase」に設定してください。")
        # self.okDialog(title, mess)

    #----- universalDialog -----------
    #  okDialog
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    #errDialog
    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    #okCancelDialog
    def okCancelDialog(self, title, mess, 
                       funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, 
            funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()

    #openFilaDialog
    def openFileDialog(self, title, path, 
                       funcOk=[], funcCancel=[], 
                       selName="", patterns=[]):
        dialog = unvDlg.openFileDialog(
            title, path, 
            funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow, selName=selName, 
            filterSets=patterns)
        dialog.show()



if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    treefoamPid = sys.argv[2]
    winApp = windowApp(caseDir)
    winApp.main()
