#!/usr/bin/python3
# coding: utf-8
#
#   editSnappyHexMeshDialog.py
#
#       snappyHexMeshDictを作成するためのcsvDaraを作成する
#
#   24/10/30    新規作成
#      12/04    release
#      12/08    csv内のfaceサイズを削除
#   25/01/29    setDefaultValueForCellSize:cellSize変更時の初期値を新規追加
#      01/30    reloadAtDefault,setDefaultValue:okCancelDialogを表示して継続有無確認を追加
#               setDefaultValueForCellSize:cellSize変更時、各行の設定済cellSizeに近い値に変更する。
#      02/07    menuCheckMesh,runCheckMesh:checkMeshのメニューを追加。
#      02/16    createMeshAddLayers,createDictPatch:decomposeParDict作成をcreateDictに移動
#               createDict,createMesh:decomposeParDict作成をcreateDictに移動。
#      03/12    setAdjustBndCheckBox:追加
#               createMesh:mesh作成後、boundaryFieldの調整有無を追加。
#               createDecomposeParNoFieldsCommLine:decomposeParコマンドにoptionを追加
#               追加したoptionがOFversionで異なるため、この関数を追加
#      03/30    setVtkModel,setVtkModelPatch:setVtkModel:vtk-9.4対応でGradientBackgroundOnを追加
#      03/31    setWindowSize:新規追加（ending時にwinSizeを保存し、initialize時にwinSizeを設定）
#      04/02    getWorldLocationFromScreenDot,changeLocInMesh,mouseClickedInViewer,mouseDragedInViewer:
#               locationInMesh位置を座標入力からマウスでdragする方法に変更
#      04/03    mouseClickedInViewer,mouseDragedInViewer:drag時のmouseCursor形状を変更
#      07/02    createCsvRowCol_sub:setSlave項目を最後に追加
#               setDataToTreeList,createDefaultSnappyCsvFile,readAddLayersCsvData,changeComboBox,
#               getStlConts,createDefaultSnappyCsvFile,getStlContsFromTreeList:
#               treeListに「setSlave」を追加。
#      07/03    addLayerSettingToSnappyDict,setDefaultValue_run:「setSlave」に対応させる。
#               editDict:編集するdictに「meshQualityDict」を追加。
#      07/08    setCaseDirToLabel:caseDirの表記文字数を100→80に変更。
#               showNoteBookPage:addLayerのpage移動時、patch読み込んでいない時、resetCamera追加
#               createPatchActorDict:「constant/polyMesh」無しのmultiRegionCaseに対応させる為修正。
#      07/11    createSnappyParallelCommLine:OF-13用を追加。
#               OF-13は、defaultがoverwriteに変わった。
#      07/12    createSnappyAddLayerCommLine:OF-13用を追加。
#               addLine用を新規追加。
#

import logFileCreater
logFileCreater.log()

import os, sys
sys.stdout.flush()
import checkVtkModule
isVtk = checkVtkModule.isVtk("editSnappyHexMesh")
if isVtk == True:
    import vtk
    from vtk.util import colors
    vtkVersion = vtk.vtkVersion.GetVTKSourceVersion().split()[-1]
else:
    print("error: could not import vtk !")
    exit()

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GdkPixbuf, GLib
import glob
import shutil
import math

import pyTreeFoam
import universalDialogs as unvDlg
import inputTextDDialog
import GtkParts
import GtkVtkRenderWindowInteractor as gtkVtk
import createBlockAndSnappyDict
import saveGoDDialog
import meshViewerControl
import getOpenFolderDDialog
import geometryFunc as geom

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

#csvFileのcolumn
csvColDict = {
    "stl":0,   "sect":1, "edge":2, 
    "cell":3,  "face":4, "note":5, 
    "layer":6, "expan":7, "slave":8
    }

#stlContsのcolumn(csvを読んだ後の保存形式）
stlColDict = {
    "stl":0,  "sect":1, "edge":2, 
    "cell":3, "layer":4, "expan":5, "slave":6
    }

#treeListのcolumn（treeList）
treeListColDict = {
    "stl":0,    "type":1,   "feature":2, 
    "edge":3,   "cell":4,   "addLayer":5, 
    "nLayer":6, "expRatio":7, "setSlave":8
    }


#treeListPatchのcolumn
treeListPatchColDict = {
    "patch":0, "addLayer":1, "nLayer":2, "expRatio":3, "setSlave":4
    } 

#noteBookのpage名とpageNo
pageDict = {"createMesh":0, "addLayer":1}


#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"editSnappyHexMeshDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を取得
        self.setGtkObject()
        #変数の設定
        self.windowTitle = ""
        self.editFlag = "no"
        self.newCsvFlag = 0
        self.caseDir = caseDir
        self.region = "."
        #self.region = ""
        self.stlDir = self.getStlDir()
        self.snappyCsvName = "snappyMeshDict.csv"
        self.addLayersCsvName = "addLayers.csv"
        self.csvRowCol = []
        self.block_cellSize = "0"
        self.block_overBlockSize = "5"
        self.locationInMesh = []
        self.stlConts = []
        self.blockMinMax = []
        self.angle = "150"           #featureEdge抽出時の角度
        self.patchConts = []
        #vtk関連
        self.maskEvent = True
        self.vtkSize = (300, 250)
        self.vtkInitialRun = True    #最初のvtk表示flag
        self.vtkObj = gtkVtk.GtkVtkRenderWindowInteractor(self)
        self.actorRegDict = {}
        self.outlineActorRegDict = {}
        self.readerDict = {}
        self.actorDict = {}
        self.outlineActorDict = {}
        self.renderer = ""
        self.originAxes = None
        self.allElms = []
        self.hideElms = []
        self.showElms = []
        self.currElms = []
        self.selectNos = 0
        self.scaleScreenDot = 1.0       #画面上のdot位置から座標に換算する係数
        self.radiusLocInMesh = 0.0      #赤点の半径
        self.vtkInitialize()
        #初期化
        self.initialize()

    def main(self):
        """ GUIの表示"""
        self.setVtkArea()           #vtkAreaをセット
        self.mainWindow.show_all()
        self.setVtkModel()          #vtkを表示
        self.treeList.treeList.grab_focus()  #treeViewにfocus（keyEventを効かす為）
        self.maskEvent = False
        self.clearEditFlag()        #修正flagをクリア
        if self.newCsvFlag != 0:
            GLib.timeout_add(100, self.openInitialMessageDialog)
        Gtk.main()

    def openInitialMessageDialog(self):
        if self.newCsvFlag == 1:
            title  = _("snappyCsvファイルの読み込み") + "\n"
            mess = _("snappyCsvファイルが存在しなかった為、新たに作成しました。")
            self.okDialog(title, mess)
        elif self.newCsvFlag == 2:
            title  = _("stlDir内の読み込み") + "\n"
            mess  = _("stlDirが存在しません。") + "\n"
            mess += _("  stlDirを変更してください。")
            self.okDialog(title, mess)
        elif self.newCsvFlag == 3:
            title  = _("snappyCsvファイルの読み込み") + "\n"
            mess  = _("csvDataの内容とstlFile名が合致しません。") + "\n"
            mess += _("  csvファイルをdefault値で再作成しました。")
            self.okDialog(title, mess)

    def close(self, *args):
        """ 閉じる"""
        self.ending()
        Gtk.main_quit()

    def setGtkObject(self):
        """ glade内のobject名を取得"""
        #右画面
        self.paned_win = self.builder.get_object("paned_win")
        self.label_caseDir = self.builder.get_object("label_caseDir")
        self.button_openStlDir = self.builder.get_object("button_openStlDir")
        self.entry_stlDir = self.builder.get_object("entry_stlDir")
        self.button_stlDirRef = self.builder.get_object("button_stlDirRef")
        self.button_caseDir = self.builder.get_object("button_openCaseDir")
        self.toggle_createMesh = self.builder.get_object("toggle_createMesh")
        self.toggle_addLayer = self.builder.get_object("toggle_addLayer")
        #noteBook
        self.noteBook = self.builder.get_object("noteBook")
        self.page_editStl = self.builder.get_object("page_editStl")
        self.page_createMesh = self.builder.get_object("page_createMesh")
        self.page_addLayer = self.builder.get_object("page_addLayer")

        #blockMesh
        self.box_blockMesh = self.builder.get_object("box_blockMesh")
        self.box_snappyHexMesh = self.builder.get_object("box_snappyHexMesh")
        self.label_blockMesh = self.builder.get_object("label_blockMesh")
        self.label_snappyHexMesh = self.builder.get_object("label_snappyHexMesh")
        self.label_createMesh = self.builder.get_object("label_createMesh")
        self.entry_cellSize = self.builder.get_object("entry_cellSize")
        self.entry_nCells = self.builder.get_object("entry_nCells")
        #snappyHexMeshの値をセット
        self.check_changeLocInMesh = self.builder.get_object("check_changeLocInMesh")
        self.entry_featureAngle = self.builder.get_object("entry_featureAngle")
        self.button_excludeStls = self.builder.get_object("button_excludeStls")
        self.button_setBoundaryDefault = self.builder.get_object("button_setBoundaryDefault")
        self.button_unselectStls = self.builder.get_object("button_unselectStls")
        treeView = self.builder.get_object("treeView")
        self.treeList = GtkParts.treeListCols(treeView)
        #mesh作成
        self.check_parallel = self.builder.get_object("check_parallel")
        self.entry_nProcs = self.builder.get_object("entry_nProcs")
        self.entry_separate = self.builder.get_object("entry_separate")
        self.button_createDict = self.builder.get_object("button_createDict")
        self.button_createMesh = self.builder.get_object("button_createMesh")
        self.button_editDict = self.builder.get_object("button_editDict")
        self.label_locationInMesh = self.builder.get_object("label_locationInMesh")
        self.label_locationXYZ = self.builder.get_object("label_locationXYZ")
        self.label_nProcs = self.builder.get_object("label_nProcs")
        self.label_separate = self.builder.get_object("label_separate")
        self.label_method = self.builder.get_object("label_method")
        self.check_adjustBnd = self.builder.get_object("check_adjustBnd")

        #vtk関連
        self.box_vtk = self.builder.get_object("box_vtk")
        self.check_showEdge = self.builder.get_object("check_showEdge")
        self.check_showOutline = self.builder.get_object("check_showOutline")
        self.check_showLocMesh = self.builder.get_object("check_showLocMesh")
        self.check_showBlockMesh = self.builder.get_object("check_showBlockMesh")
        self.check_showOrigin = self.builder.get_object("check_showOrigin")
        self.radio_both = self.builder.get_object("radio_both")
        self.radio_frontFace = self.builder.get_object("radio_frontFace")
        self.radio_backFace = self.builder.get_object("radio_backFace")
        self.label_vtkMess = self.builder.get_object("label_vtkMess")
        self.button_reset = self.builder.get_object("button_reset")
        self.button_Xaxis = self.builder.get_object("button_Xaxis")
        self.button_Yaxis = self.builder.get_object("button_Yaxis")
        self.button_Zaxis = self.builder.get_object("button_Zaxis")
        self.button_rightToLeft = self.builder.get_object("button_rightToLeft")
        self.button_rolling90 = self.builder.get_object("button_rolling90")
        #self.button_reloadVtk = self.builder.get_object("button_reloadVtk")
        
        #patchにlayer追加-----
        #region
        combo_regionPatch = self.builder.get_object("combo_regionPatch")
        self.combo_regionPatch = GtkParts.comboBoxText(combo_regionPatch)
        #snappyHexMesh
        self.label_regionPatch = self.builder.get_object("label_regionPatch")
        self.label_snappyHexMeshPatch = self.builder.get_object("label_snappyHexMeshPatch")
        self.label_addLayerPatch = self.builder.get_object("label_addLayerPatch")
        self.button_setDefaultPatches = self.builder.get_object("button_setDefaultPatches")
        self.button_unselectPatches = self.builder.get_object("button_unselectPatches")
        self.entry_featureAnglePatch = self.builder.get_object("entry_featureAnglePatch")
        treeViewPatch = self.builder.get_object("treeViewPatch")
        self.treeListPatch = GtkParts.treeListCols(treeViewPatch)
        #layer追加
        self.check_parallelPatch = self.builder.get_object("check_parallelPatch")
        self.entry_nProcsPatch = self.builder.get_object("entry_nProcsPatch")
        self.entry_separatePatch = self.builder.get_object("entry_separatePatch")
        self.button_createDictPatch = self.builder.get_object("button_createDictPatch")
        self.button_createMeshPatch = self.builder.get_object("button_createMeshPatch")
        self.button_editDictPatch = self.builder.get_object("button_editDictPatch")
        self.label_nProcsPatch = self.builder.get_object("label_nProcsPatch")
        self.label_separatePatch = self.builder.get_object("label_separatePatch")
        self.label_methodPatch = self.builder.get_object("label_methodPatch")

    #
    #  getStlDir
    def getStlDir(self):
        """ stlDir, csvNameを取得する"""
        case = pyTreeFoam.case(self.caseDir)
        dataDict = case.getCreateMeshData()
        stlDir = dataDict["stlDir"]
        if stlDir[0] != "/":
            stlDir = self.caseDir + "/" + stlDir
        return stlDir

    #--------- vtk関連 --------------------------
    #
    #  vtkInitialize
    #----------------
    def vtkInitialize(self):
        """ vtk関連の初期化"""
        self.setVtkMessageShowStl()
        self.setCheckBox()
        
        #toolTip
        self.check_showEdge.props.tooltip_text = _("stlのedgeを表示する")
        self.check_showOrigin.props.tooltip_text = _("原点を表示する")
        self.check_showOutline.props.tooltip_text = _("全stlファイルのoutline（feature edge）を表示する。")
        self.check_showLocMesh.props.tooltip_text = _("設定したlocationInMeshの座標を赤点表示する。")
        self.check_showBlockMesh.props.tooltip_text = _("blockMeshのサイズを表示する。")
        self.button_reset.props.tooltip_text = _("画面にフィットさせる（reset camera）")
        self.button_Xaxis.props.tooltip_text = _("x軸視（y-z 面）を表示")
        self.button_Yaxis.props.tooltip_text = _("y軸視（z-x 面）を表示")
        self.button_Zaxis.props.tooltip_text = _("z軸視（x-y 面）を表示")
        self.button_rightToLeft.props.tooltip_text = _("視点を反転(左右前後反転)")
        self.button_rolling90.props.tooltip_text = _("時計回りに90度回転")
        #createMeshPage
        self.button_openStlDir.props.tooltip_text = _("stlDirを開く")
        self.button_caseDir.props.tooltip_text = _("caseDirを開く")
        self.entry_cellSize.props.tooltip_text = _("blockMeshのcellSizeを設定する")
        mess = _("モデルのminMax値の両側にnCells分を加えたブロックサイズでblockMeshを作成する。")
        self.entry_nCells.props.tooltip_text = mess
        mess  = _("選択しているstlファイルを表から除外する。") + "\n"
        mess += _("ファイル名を'_'付きに変更して除外する。ファイルは削除されない。")
        self.button_excludeStls.props.tooltip_text = mess
        self.button_setBoundaryDefault.props.tooltip_text = _("blockMeshのcellSizeを使って、下表のdefault値を設定する。")
        mess = _("locationInMeshのdefault値は、モデルの中心座標が設定される。") + "\n"
        mess += _("中心に空洞があるモデルの場合は、座標を変更する。")
        self.label_locationInMesh.props.tooltip_text = mess
        self.label_locationXYZ.props.tooltip_text = mess
        mess = _("checkにより、赤点が黄点に変わる。") + "\n"
        mess += _("座標を修正する場合は、黄点をドラッグする。")
        self.check_changeLocInMesh.props.tooltip_text = mess
        mess  = _("90以下の設定は、角部にlayerが付かない。") + "\n"
        mess += _("90以上の設定は、外表面のedge部のcellが欠ける事がある。")
        self.entry_featureAngle.props.tooltip_text = mess
        self.button_unselectStls.props.tooltip_text = _("下表を非選択状態に設定する。")
        self.treeList.treeList.props.tooltip_text = _("変更したい項目をクリックすると、dropDownMenuを表示 or cell編集モードに移行するので、その項目が変更できる。")
        self.button_createDict.props.tooltip_text = _("blockMesh,snappyHexMeshの入力dataから、これらのDictを作成する。")
        self.button_createMesh.props.tooltip_text = _("特徴線を抽出、blockMesh実行、snappyhexMesh実行して、meshを作成する。")
        self.button_editDict.props.tooltip_text = _("snappyHexmeshDict, blockMeshDictを編集")
        self.radio_both.props.tooltip_text = _("faceの表裏両面を表示する")
        self.radio_frontFace.props.tooltip_text = _("faceの表面のみ表示する\nfaceの方向が確認きる")
        self.radio_backFace.props.tooltip_text = _("faceの裏面のみ表示する\n隠れた内部のstlが確認できる")
        
        #addLayersPage
        self.toggle_createMesh.props.tooltip_text = _("「mesh作成」「meshにlayer追加」のモード切り替えボタン")
        self.toggle_addLayer.props.tooltip_text = _("「mesh作成」「meshにlayer追加」のモード切り替えボタン")
        self.combo_regionPatch.comboBox.props.tooltip_text = _("layerを追加するregionを選択する。")
        self.button_setDefaultPatches.props.tooltip_text = _("全patchに対し、layer追加を外す。")
        self.button_unselectPatches.props.tooltip_text = _("全patchを非選択状態に設定する。")
        self.entry_featureAnglePatch.props.tooltip_text = _("90以下の設定では、角部にlayerが付かない。")
        self.treeListPatch.treeList.props.tooltip_text = _("layerを追加したいpatchを選択し、layer数と拡大率を設定する。")
        self.button_createDictPatch.props.tooltip_text = _("snappyHexMeshDictを作成する。")
        self.button_createMeshPatch.props.tooltip_text = _("snappyHexMeshを実行してlayerを追加する。")
        self.button_editDictPatch.props.tooltip_text = _("snappyHexMeshDictを編集する")

    #  setVtkMessageShowStl
    def setVtkMessageShowStl(self):
        """ vtk画面下にmessage(stl表示)を追加"""
        self.label_vtkMess.set_text(_("選択しているstlファイルを表示する。"))

    #  setVtkMessageShowPoint
    def setVtkMessageShowPoint(self):
        """ vtk画面下にmessage(stl表示)を追加"""
        self.label_vtkMess.set_text(_("locationInMesh位置を修正中..."))

    #
    #  setCheckBox
    def setCheckBox(self):
        """ checkBoxを設定する"""
        self.check_showEdge.set_active(True)
        self.check_showOutline.set_active(True)
        self.check_showLocMesh.set_active(True)
        self.check_showBlockMesh.set_active(False)
        self.check_showOrigin.set_active(False)

    #
    #  setVtkArea
    #-------------
    def setVtkArea(self):
        """ vtkAreaを設定する"""
        vtkObj = self.vtkObj
        vtkObj.set_size_request(*self.vtkSize)
        self.box_vtk.pack_start(vtkObj, True, True, 0)

    #
    #  addCornerAxis
    #----------------
    def addCornerAxis(self):
        """ 最初の1回のみ、XYZの軸（compas）をrendererに追加する。"""
        if self.vtkInitialRun == True:
            #add axis(最初の起動時のみ)
            interactor = self.renWin.GetInteractor()
            axesActor = vtk.vtkAxesActor()
            self.axes = vtk.vtkOrientationMarkerWidget()
            self.axes.SetOrientationMarker(axesActor)
            self.axes.SetInteractor(interactor)
            self.axes.EnabledOn()
            self.axes.InteractiveOn()
            #flagをセット
            self.vtkInitialRun = False
        return

    #
    #  addOriginAxis
    #----------------
    def addOriginAxis(self):
        """ 原点にaxisを表示する"""
        if self.check_showOrigin.get_active() == True:
            #原点を表示
            if self.originAxes == None:
                #原点を作成
                #bounds = self.getBoundsActors()
                bounds = self.getOriginBounds()
                if len(bounds) == 0:
                    return
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes = vtk.vtkAxesActor()
                axes.SetTotalLength(length, length, length)
                axes.GetXAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetXAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetYAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetYAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetZAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetZAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                self.renderer.AddActor(axes)
                self.originAxes = axes
            else:
                #原点を修正
                axes = self.originAxes
                #bounds = self.getBoundsActors()
                bounds = self.getOriginBounds()
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes.SetTotalLength(length, length, length)
                #削除して追加する（axesを最後に描画する為）
                self.renderer.RemoveActor(axes)
                self.renderer.AddActor(axes)
                self.originAxes = axes
        else:
            #原点を削除
            self.renderer.RemoveActor(self.originAxes)
            self.originAxes = None
        return

    #
    #  setVtkModel
    #---------------
    def setVtkModel(self):
        """ vtkを表示する"""
        #actorを作成する
        self.createActorDict()
        #rendererをチェック
        if self.renderer == "":
            self.renderer = vtk.vtkRenderer()
        else:
            self.renderer.NewInstance()
        #assyを定義
        self.assy = vtk.vtkAssembly()
        #stlを表示（選択しているstlを取得して）
        selNames = self.getSelectedStls()
        self.showSelectedItems(selNames)

        #backgroundColorを設定
        self.renderer.SetBackground2(0.5, 0.5, 1)
        self.renderer.SetBackground(0.8, 0.8, 1)
        self.renderer.GradientBackgroundOn()
        self.renderer.SetTexturedBackground(True)
        #self.renderer.ResetCamera()
        self.renWin = self.vtkObj.GetRenderWindow()
        self.renWin.NewInstance()
        #新しいrendererを追加
        self.renWin.AddRenderer(self.renderer)
        self.addCornerAxis()
        #vtk表示
        bounds = self.getCameraBounds()
        #self.renderer.ResetCamera(*self.modelBounds)
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.GlobalWarningDisplayOff()   #windowsのエラー対策
        self.vtkObj.Initialize()

        self.vtkObj.Start()
        return

    #
    #  createActorDict
    def createActorDict(self):
        """ actorを作成する"""
        #stlFileを取得
        blockMeshStl = self.createBlockMeshStlFile()
        blockName = os.path.basename(blockMeshStl)
        self.blockMeshName = ".".join(blockName.split(".")[:-1])
        allItems = self.treeList.getAllItemNames()
        stlNames = list(map(lambda x: x[0], allItems))
        stlFiles = list(map(lambda x: self.stlDir+"/"+x+".stl", stlNames))
        stlFiles.append(blockMeshStl)
        #全stlのactor辞書を作成
        self.actorDict = {}
        self.outlineActorDict = {}
        print("reading stl files...")
        #for stlName in stlNames:
        for fileName in stlFiles:
            stlFile = os.path.basename(fileName)
            stlName = ".".join(stlFile.split(".")[:-1])
            #reader
            reader = vtk.vtkSTLReader()
            reader.SetFileName(fileName)
            reader.Update()
            self.readerDict[stlName] = reader
            #filter
            geomFilter = vtk.vtkGeometryFilter()
            geomFilter.SetInputConnection(reader.GetOutputPort())
            #featureEdge
            featureEdge = vtk.vtkFeatureEdges()
            featureEdge.SetInputConnection(geomFilter.GetOutputPort())
            #mapper
            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputConnection(reader.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(featureEdge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            self.actorDict[stlName] = actor
            edgeActor = vtk.vtkActor()
            edgeActor.SetMapper(edgeMapper)
            self.outlineActorDict[stlName] = edgeActor
        #locMeshのactorを取得
        locMeshActor = self.getLocationInMeshActor()
        self.actorDict["locationInMesh"] = locMeshActor

    #
    #  createBlockMeshStlFile
    def createBlockMeshStlFile(self):
        """ blockMeshのstlファイルを作成する
        tempフォルダ内にtemp.stlで作成する"""

        def convNum2str(num):
            ans = pyTreeFoam.float2strAuto(num, 6)
            return ans

        def createTet(tet, normal):
            """ 三角形1個分を作成"""
            lines  = [" facet normal " + normal + "\n"]
            lines += ["  outer loop\n"]
            p0 = list(map(convNum2str, tet[0]))
            p1 = list(map(convNum2str, tet[1]))
            p2 = list(map(convNum2str, tet[2]))
            lines += ["   vertex " + " ".join(p0) + "\n"]
            lines += ["   vertex " + " ".join(p1) + "\n"]
            lines += ["   vertex " + " ".join(p2) + "\n"]
            lines += ["  endloop\n"]
            lines += [" endfacet\n"]
            return lines

        def createTets(tet1, tet2, normal):
            lines = createTet(tet1, normal)
            lines += createTet(tet2, normal)
            return lines

        [x0, y0, z0], [x1, y1, z1] = self.blockMinMax
        p0 = [x0, y0, z0]
        p1 = [x1, y0, z0]
        p2 = [x1, y0, z1]
        p3 = [x0, y0, z1]
        p4 = [x0, y1, z0]
        p5 = [x1, y1, z0]
        p6 = [x1, y1, z1]
        p7 = [x0, y1, z1]
        #x-z面
        tet1 = [p0, p1, p2]
        tet2 = [p0, p2, p3]
        tet3 = [p4, p6, p5]
        tet4 = [p4, p7, p6]
        #y-z面
        tet5 = [p1, p5, p6]
        tet6 = [p1, p6, p2]
        tet7 = [p0, p7, p4]
        tet8 = [p0, p3, p7]
        #x-y面
        tet9 = [p0, p5, p1]
        tet10 = [p0, p4, p5]
        tet11 = [p3, p2, p6]
        tet12 = [p3, p6, p7]
        #stlデータ作成
        lines = []
        lines += ["solid blockMesh\n"]
        normal = "0 -1 0"
        lines += createTets(tet1, tet2, normal)
        normal = "0 1 0"
        lines += createTets(tet3, tet4, normal)
        normal = "1 0 0"
        lines += createTets(tet5, tet6, normal)
        normal = "-1 0 0"
        lines += createTets(tet7, tet8, normal)
        normal = "0 0 -1"
        lines += createTets(tet9, tet10, normal)
        normal = "0 0 1"
        lines += createTets(tet11, tet12, normal)
        lines += ["endsolid blockMesh\n"]
        #stlファイル保存
        userDir = os.getenv("TreeFoamUserPath")
        fileName = userDir + "/temp/blockMesh.stl"
        f = open(fileName, "w"); f.writelines(lines); f.close
        return fileName

    #
    #  getLocationInMeshActor
    def getLocationInMeshActor(self):
        """ locationInMeshのactorを取得"""
        #source
        minVals, maxVals = self.blockMinMax
        length = [maxVals[0] - minVals[0], 
                  maxVals[1] - minVals[1],
                  maxVals[2] - minVals[2]]
        radius = sum(length) / 3.0 / 50.0
        self.radiusLocInMesh = radius
        location = list(map(float, self.locationInMesh))
        sphere = vtk.vtkSphereSource()
        sphere.SetRadius(radius)
        sphere.SetCenter(location)
        #mapper
        mapper = vtk.vtkPolyDataMapper()
        mapper.SetInputConnection(sphere.GetOutputPort())
        #actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper)
        prop = actor.GetProperty()
        prop.SetColor(colors.red)
        if self.vtkObj.canMove == False:
            prop.SetColor(colors.yellow)
        return actor

    #
    #  fitWindow
    #------------
    def fitWindow(self):
        """ 画面に合わせてvtk表示させる"""
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromX
    #------------
    def viewFromX(self):
        """ X軸方向から見る"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter+maxLeng*2.0, Ycenter, Zcenter)
        #X軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromY
    #------------
    def viewFromY(self):
        """ Y軸視（Z-X面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter+maxLeng*2.0, Zcenter)
        #Y軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(1, 0, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromZ
    #------------
    def viewFromZ(self):
        """ Z軸視（X-Y面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter, Zcenter+maxLeng*2.0)
        #Z軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 1, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  setReverseCamera
    #-------------------
    def setReverseCamera(self):
        """ camera位置を反転（左右反転）させる"""
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        center = (Xcenter, Ycenter, Zcenter)
        camera = self.renderer.GetActiveCamera()
        pos = camera.GetPosition()
        relPos = (pos[0]-center[0], pos[1]-center[1], pos[2]-center[2])
        newRelPos = list(map(lambda x: -x, relPos))
        newPos = (newRelPos[0]+center[0], newRelPos[1]+center[1], newRelPos[2]+center[2])
        camera.SetPosition(*newPos)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  rolling90
    #------------
    def rolling90(self):
        """ 視点を右回転（90°）させる。"""
        camera = self.renderer.GetActiveCamera()
        camera.Roll(-90.0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  changeShowEdge
    #-----------------
    def changeShowEdge(self):
        """ check_showEdgeを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowOutline
    #--------------------
    def changeShowOutline(self):
        """ check_showOutlineを変更したときのevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowOrigin
    #-------------------
    def changeShowOrigin(self):
        """ 原点表示チェックを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowBlockMesh
    #----------------------
    def changeShowBlockMesh(self):
        """ blockMeshsizeを表示する"""
        self.applyCheckContsToVtk()

    #
    #  changeShowLocMesh
    #--------------------
    def changeShowLocMesh(self):
        """ mesh位置locationInMeshを表示する"""
        self.applyCheckContsToVtk()

    #
    #  changeShowBoth
    #-----------------
    def changeShowBoth(self):
        """ radio_showBothを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowFront
    #------------------
    def changeShowFront(self):
        """ radio_showFrontFaceを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowBack
    #-----------------
    def changeShowBack(self):
        """ radio_showbackFaceを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  createShowSelectedItems
    #--------------------------
    def createShowSelectedItems(self, selNames):
        """ entryの値を取得し、vtkを作成し直す"""
        self.getDataFromEntryBox()  #entryの内容を取得
        self.createTreeList()       #TreeListを作成
        self.setDataToTreeList()    #TreeListにデータをセット
        self.selectRowsInTreeList(selNames)
        self.blockMinMax = self.getBlockMeshMinMax()    #blockサイズを取得
        self.createActorDict()      #actor辞書作成
        self.showSelectedItems(selNames)    #選択itemをvtk画面に表示

    #
    #  showSelectedItems
    #--------------------
    def showSelectedItems(self, selNames):
        """ TreeItemの選択したitemをvtk表示させる"""
        if self.renderer == "":
            return
        #選択item名を取得
        self.allElms = list(map(lambda x: x[0], self.stlConts))
        self.showElms = selNames
        self.hideElms = list(set(self.allElms) - set(self.showElms))
        #assyをclear
        parts = list(self.assy.GetParts())
        if len(parts) > 0:
            self.renderer.RemoveActor(self.assy)
            self.renderer.SetErase(True)
            self.assy = vtk.vtkAssembly()
        #actorを再設定
        for showElm in self.showElms:
            if showElm in self.actorDict.keys():
                actor = self.actorDict[showElm]
                actor = self.setActorProperty(actor)
                self.assy.AddPart(actor)
        #  outlineを追加
        if self.check_showOutline.get_active() == True:
            allItems = self.treeList.getAllItemNames()
            stlNames = map(lambda x: x[0], allItems)
            for stlName in stlNames:
                actor = self.outlineActorDict[stlName]
                self.assy.AddPart(actor)
        #blockMeshを追加
        if self.check_showBlockMesh.get_active() == True:
            actor = self.outlineActorDict[self.blockMeshName]
            self.assy.AddPart(actor)
        #locMeshを追加
        if self.check_showLocMesh.get_active() == True:
            actor = self.actorDict["locationInMesh"]
            self.assy.AddPart(actor)
        self.renderer.AddActor(self.assy)
        parts = list(self.assy.GetParts())
        #原点を表示
        self.addOriginAxis()
        #初めて（削除するactorが無かった）の場合、表示を合わせる
        #self.fitWindow()
        #vtk表示
        self.vtkObj.Initialize()
        return

    #
    #  setActorProperty
    def setActorProperty(self, actor):
        """ actorのpropertyを設定"""
        prop = actor.GetProperty()
        #edgeの表示
        if self.check_showEdge.get_active() == True:
            #edge表示
            prop.EdgeVisibilityOn()
        else:
            #edge非表示
            prop.EdgeVisibilityOff()
        #表裏面の表示
        if self.radio_both.get_active() == True:
            #両面表示
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOff()
        elif self.radio_frontFace.get_active() == True:
            #表面を表示
            prop.BackfaceCullingOn()
            prop.FrontfaceCullingOff()
        elif self.radio_backFace.get_active() == True:
            #裏面を表示
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOn()
        #色の設定
        prop.SetAmbient(0.3)
        prop.SetColor(colors.grey)
        return actor

    #
    #  getCameraBounds
    def getCameraBounds(self):
        """ 表示させるactorのboundsを取得する。
        原点、マージンも含めて取得する。"""
        bounds = self.getOriginBounds()
        # if self.originAxes == None:
        #     #原点無しの場合
        #     return bounds
        if len(bounds) == 0:
            return bounds
        #マージンを確保
        xmin = bounds[0]
        xmax = bounds[1]
        ymin = bounds[2]
        ymax = bounds[3]
        zmin = bounds[4]
        zmax = bounds[5]
        a = 0.1
        xmin = xmin - (xmax-xmin) * a
        xmax = xmax + (xmax-xmin) * a
        ymin = ymin - (ymax-ymin) * a
        ymax = ymax + (ymax-ymin) * a
        zmin = zmin - (zmax-xmin) * a
        zmax = zmax + (zmax-xmin) * a
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getOriginBounds
    def getOriginBounds(self):
        """ 表示させるactorと原点のboundsを取得する。"""
        bounds = self.getBoundsActors()
        if self.originAxes == None:
            #原点無しの場合
            return bounds
        #原点を追加する
        xmin = min([bounds[0], 0.0])
        xmax = max([bounds[1], 0.0])
        ymin = min([bounds[2], 0.0])
        ymax = max([bounds[3], 0.0])
        zmin = min([bounds[4], 0.0])
        zmax = max([bounds[5], 0.0])
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getBoundsActors
    def getBoundsActors(self):
        """ 全actorのboundsを取得して返す。"""
        actors = self.renderer.GetActors()
        bounds = []
        for actor in actors:
            bound = actor.GetBounds()
            bounds.append(bound)
        if len(bounds) == 0:
            return ()
        Xmin = min(map(lambda x: x[0], bounds))
        Xmax = max(map(lambda x: x[1], bounds))
        Ymin = min(map(lambda x: x[2], bounds))
        Ymax = max(map(lambda x: x[3], bounds))
        Zmin = min(map(lambda x: x[4], bounds))
        Zmax = max(map(lambda x: x[5], bounds))
        return (Xmin, Xmax, Ymin, Ymax, Zmin, Zmax)

    #---- locationInMesh（赤点）の位置操作 ---------------
    #
    #  getWorldLocationFromScreenDot
    def getWorldLocationFromScreenDot(self, dotXY, focalPt):
        """ 画面上のdot位置をworld座標に変換して返す
        原点は、画面中央(focalPt)。"""
        w, h = self.vtkObj._Iren.GetSize()
        locX = (dotXY[0] - w/2.0) * self.scaleScreenDot
        #Y方向は、方向が反対のため、「マイナス」を追加
        locY = -(dotXY[1] - h/2.0) * self.scaleScreenDot
        loc = [locX, locY, 0.0]
        #座標変換
        matrix = self.matrix_4x4
        vecWx = [matrix[0][0], matrix[1][0], matrix[2][0]]  #X軸
        vecWy = [matrix[0][1], matrix[1][1], matrix[2][1]]  #Y軸
        vecWz = [matrix[0][2], matrix[1][2], matrix[2][2]]  #Z軸
        #各軸方向の成分を取得
        locX = geom.dotProduct(loc, vecWx)
        locY = geom.dotProduct(loc, vecWy)
        locZ = geom.dotProduct(loc, vecWz)
        #原点（focalPt）に変化量を加える
        locX, locY, locZ = geom.addVec(focalPt, [locX, locY, locZ])
        return locX, locY, locZ

    #
    #  changeLocInMesh
    #------------------
    def changeLocInMesh(self):
        """ locationInMeshを変更する"""
        if self.check_changeLocInMesh.get_active() == True:
            #画面を停止
            self.vtkObj.canMove = False
            #赤点をfocus
            redPt = list(map(float, self.locationInMesh))
            camera = self.renderer.GetActiveCamera()
            camera.SetFocalPoint(redPt)
            #赤点を黄色に変更
            locActor = self.actorDict["locationInMesh"]
            locActor.GetProperty().SetColor(colors.yellow)
            #actor表示
            self.applyCheckContsToVtk()
            #vtk画面下部のmessageを変更
            self.setVtkMessageShowPoint()
        else:
            #画面停止を解除
            self.vtkObj.canMove = True
            #赤点をの色を戻す
            locActor = self.actorDict["locationInMesh"]
            locActor.GetProperty().SetColor(colors.red)
            #actor表示
            self.applyCheckContsToVtk()
            #vtk画面下部のmessageを戻す
            self.setVtkMessageShowStl()

    #
    #  mouseClickedInViewer
    #----------------------
    def mouseClickedInViewer(self):
        """ mouseをclickした時のevent
        画面のdot位置からactorの座標を取得"""
        if self.check_changeLocInMesh.get_active() == True:
            #画面上のdot位置からworld座標を求める倍率を算出
            camera = self.renderer.GetActiveCamera()
            camPos = camera.GetPosition()                
            viewAng = camera.GetViewAngle() * math.pi /180.0
            orgPt = camera.GetFocalPoint()      #原点
            l = geom.length(camPos, orgPt)      #視点-原点間長さ
            #world座標の幅（camera位置とviewAngleから算出）
            width = l * math.tan(viewAng/2.0) * 2.0
            #vtk画面上の幅,高さ
            w, h = self.vtkObj._Iren.GetSize()
            aw = h
            #倍率算出
            self.scaleScreenDot = width / aw
            #位置計算の為のmatrix取得
            matrixObj = camera.GetViewTransformMatrix()
            matrix = []
            for i in range(4):
                iVals = []
                for j in range(4):
                    val = matrixObj.GetElement(i,j)
                    iVals.append(val)
                matrix.append(iVals)
            self.matrix_4x4 = matrix
            #mouseCursor形状設定
            win = self.vtkObj.get_window()
            cursorHand = Gdk.Cursor(Gdk.CursorType.HAND2)
            win.set_cursor(cursorHand)
            #クリック位置を取得
            iniPos = self.vtkObj.get_pointer()
            xyz = self.getWorldLocationFromScreenDot(iniPos, orgPt)
            #クリック位置と赤点位置の差dlを取得
            locInMesh = list(map(float, self.locationInMesh))
            dl = geom.length(locInMesh, xyz)
            #dlが赤点半径*4以上か？
            if dl > self.radiusLocInMesh * 4.0:
                #赤点から離れた位置をクリックした場合
                #mouseCursor形状戻す
                win = self.vtkObj.get_window()
                win.set_cursor()
                #画面停止を解除
                self.check_changeLocInMesh.set_active(False)
                self.vtkObj.canMove = True
                locActor = self.actorDict["locationInMesh"]
                locActor.GetProperty().SetColor(colors.red)
                #actor表示
                self.applyCheckContsToVtk()
                #vtk画面下部message戻す
                self.setVtkMessageShowStl()
        return

    #
    #  mouseDragedInViewer
    #----------------------
    def mouseDragedInViewer(self):
        """ mouseをdragした後のevent
        座標を取得する"""
        if self.check_changeLocInMesh.get_active() == True:
            #mouseCursor形状戻す
            win = self.vtkObj.get_window()
            win.set_cursor()
            #drag後の位置を取得
            camera = self.renderer.GetActiveCamera()
            focalPt = camera.GetFocalPoint()
            dotPos = self.vtkObj.get_pointer()
            x, y, z = self.getWorldLocationFromScreenDot(dotPos, focalPt)
            #locationInMesh位置を修正
            locx = pyTreeFoam.float2strAuto(x, prec=3, minNum=0.001, maxNum=1000)
            locy = pyTreeFoam.float2strAuto(y, prec=3, minNum=0.001, maxNum=1000)
            locz = pyTreeFoam.float2strAuto(z, prec=3, minNum=0.001, maxNum=1000)
            self.locationInMesh = [locx, locy, locz]
            self.setDataToEntryBox()
            #赤点表示
            self.changelocxyz()
    #--------------------------------------------


    #----------- menuBar event handler ------
    def onMenuSave(self, event):            #保存
        self.saveSnappyCsv()
    def onMenuReload(self, event):          #再読込
        self.reloadSnappyCsv()
    def onMenuReloadNew(self, event):       #再読込（default）
        self.reloadAtDefault()
    def onMenuClose(self, event):           #閉じる
        self.close()
    def onMenuEditDict(self, event):        #dict編集
        self.menuEditDict()
    def onMenuCheckMesh(self, event):       #checkMesh実行
        self.menuCheckMesh()
    def onMenuStlViewer(self, event):       #stlViewer起動
        self.runStlViewer()
    def onMenuMeshViewer(self, event):      #meshViewer起動
        self.runMeshViewer()
    def onMenuParaview(self, event):        #paraview起動
        self.runParaview()
    #-----------toolBar event handler -------
    def onSaveSnappyCsv(self, event):       #保存
        self.saveSnappyCsv()
    def onReloadSnappyCsv(self, event):     #再読込
        self.reloadSnappyCsv()
    def onReloadAtDefault(self, event):     #再読込（default）
        self.reloadAtDefault()
    def onClose(self, event):               #閉じる
        if self.editFlag == "no":
            self.close()
            return
        self.selectEnding()
    def onRunCheckMesh(self, event):        #checkMesh起動
        self.runCheckMesh()
    def onRunStlViewer(self, event):        #stlViewer起動
        self.runStlViewer()
    def onRunMeshViewer(self, event):       #meshViewer起動
        self.runMeshViewer()
    def onRunParaview(self, event):         #paraview起動
        self.runParaview()
    def onChangeStlDir(self, event):        #stlDir変更時
        self.changeStlDir()
    def onStlDirRef(self, event):           #stlDir参照...
        self.stlDirRef()
    def onOpenStlDir(self, event):          #stlDir開く
        self.openStlDir()
    def onOpenCaseDir(self, event):         #caseDir開く
        self.openCaseDir()
    #------ noteBook event ---------------------
    def onShowCreateMeshPage(self, event):  #mesh作成page
        if self.maskEvent == False:
            self.showCreateMeshPage()
    def onShowAddLayerPage(self, event):    #layer追加page
        if self.maskEvent == False:
            self.showAddLayerPage()
    #------- createMesh内のevent--------------
    #------- entryのevent
    def onChangeCellSize(self, event):      #cellSize変更時
        self.changeCellSize()
    def onChangeNCells(self, event):        #nCells変更時
        self.changeNCells()
    def onChangeLocx(self, event):          #locX変更時
        self.changelocxyz()
    def onChangeLocy(self, event):          #locY変更時
        self.changelocxyz()
    def onChangeLocz(self, event):          #locZ変更時
        self.changelocxyz()
    def onChangeFAngle(self, event):        #featureAngle変更時
        self.changeFeatureAngle()
    #------ button, checkBoxのevent
    def onChangeLocInMesh(self, event):     #locationInMeshを変更
        self.changeLocInMesh()
    def onExcludeStls(self, event):         #stl除外
        self.excludeStls()
    def onSetDefaultValue(self, event):     #default値に設定
        self.setDefaultValue()
    def onUnselectStls(self, event):        #全て非選択
        self.unselectStls()
    def onCreateDict(self, event):          #dict作成
        self.createDict()
    def onCreateMesh(self, event):          #mesh作成
        self.createMesh()
    def onEditDict(self, event):            #dict編集
        self.editDict()
    def onChangeParallel(self, event):      #並列処理
        self.changeParallel()
    #------ treeList内のevent
    def onChangeCheckBox(self, path, col):  #checkBox変更時
        self.changeCheckBox(path, col)
    def onChangeComboBox(self, path, col):  #comboBox変更時
        self.changeComboBox(path, col)
    def onEditedLayers(self, path, col):    #cell値変更時
        self.editedLayers(path, col)
    #------ addLayer内のevent ------------------
    def onChangeRegionPatch(self, event):   #region変更時
        self.changeRegionPatch()
    def onSetDefaultValuePatch(self, event): #default値に設定
        self.setDefaultValuePatch()
    def onUnselectPatches(self, event):     #全て非選択
        self.unselectPatches()
    def onChangeFAnglePatch(self, event):   #featureAngle変更
        self.changeFeatureAnglePatch()
    def onCreateDictPatch(self, event):     #dict作成
        self.createDictPatch()
    def onCreateMeshPatch(self, event):     #layer追加
        self.createMeshAddLayers()
    def onEditDictPatch(self, event):       #dict編集
        self.editDictPatch()
    #------- treeListPatch内のevent
    def onChangeCheckBoxPatch(self, path, col): #checkBox変更時
        self.changeCheckBoxPatch(path, col)
    def onEditedLayersPatch(self, path, col):   #cell値変更時
        self.editedLayersPatch(path, col)
    #-------- vtk画面 ----------------------
    def onFitWindow(self, event):           #画面にfit
        self.fitWindow()
    def onViewFromX(self, event):           #X軸視
        self.viewFromX()
    def onViewFromY(self, event):           #Y軸視
        self.viewFromY()
    def onViewFromZ(self, event):           #Z軸視
        self.viewFromZ()
    def onSetReverseCamera(self, event):    #反転
        self.setReverseCamera()
    def onRolling90(self, event):           #90度回転
        self.rolling90()
    def onChangeShowEdge(self, event):      #edge表示
        if self.maskEvent == False:
            self.changeShowEdge()
    def onChangeShowOutline(self, event):   #outline表示
        if self.maskEvent == False:
            self.changeShowOutline()
    def onChangeShowOrigin(self, event):    #原点表示
        if self.maskEvent == False:
            self.changeShowOrigin()
    def onChangeShowBlockMesh(self, event): #blockMesh枠表示
        if self.maskEvent == False:
            self.changeShowBlockMesh()
    def onChangeShowLocMesh(self, event):   #mesh位置表示
        if self.maskEvent == False:
            self.changeShowLocMesh()
    def onChangeShowBoth(self, event):      #両面表示
        if self.maskEvent == False:
            self.changeShowBoth()
    def onChangeShowFront(self, event):     #表面表示
        if self.maskEvent == False:
            self.changeShowFront()
    def onChangeShowBack(self, event):      #裏面表示
        if self.maskEvent == False:
            self.changeShowBack()
    def onMouseClicked(self):
        if self.maskEvent == False:
            self.mouseClickedInViewer()
    def onMouseDraged(self):                #mouseDrag後
        if self.maskEvent == False:
            self.mouseDragedInViewer()
    #-------- treeList内のmouse,key操作------
    def onChangeCursor(self, widget):       #行選択時
        self.changeCursor(widget)
    #-------- treeListPatch内のmouse,key操作-----
    #selectRow
    def onChangeCursorPatch(self, widget):
        self.changeCursorPatch()            #行選択時



    #
    #  ending
    #---------
    def ending(self):
        """ endingの処理"""
        #設定を保存
        if self.check_parallel.get_active() == True:
            parallel = "yes"
        else:
            parallel = "no"
        nProcs = self.entry_nProcs.get_text()
        nSeparate = self.entry_separate.get_text()
        angle = self.angle
        stlDir = self.stlDir
        dataDict = {}
        dataDict["stlDir"] = stlDir
        dataDict["surfaceAngle"] = angle
        dataDict["nCpu"] = nProcs
        dataDict["nSeparate"] = nSeparate
        dataDict["parallel"] = parallel
        pyTreeFoam.case(self.caseDir).setCreateMeshData(dataDict)
        #windowSizeを保存
        (width, height) = self.mainWindow.get_size()
        splitWidth = self.paned_win.get_position()
        winDict = {}
        winDict["editSnappyHexMesh"] = str(width) + " " + str(height)
        winDict["editSnappyHexMesh_splitter"] = str(splitWidth)
        pyTreeFoam.writeWindowSize(winDict)

    #
    #  selectEnding
    def selectEnding(self):
        """ 保存して終了、終了、canceを選択して終了"""
        title = _("終了確認")
        mess = _("内容が変更されています。") + "\n"
        mess += _("  このまま処理を継続すると、変更内容が失われます。")
        funcSaveGo = [self.saveAndEnd]
        funcGo = [self.justEnd]
        funcCancel = []
        dialog = saveGoDDialog.saveGo(title, mess, 
                                     funcSaveGo=funcSaveGo,
                                     funcGo=funcGo,
                                     funcCancel=funcCancel)
        dialog.show()

    def saveAndEnd(self):
        """ 保存して終了"""
        self.saveSnappyCsv()
        self.close()

    def justEnd(self):
        """ 終了する"""
        self.close()

    #
    #  setEditFlag
    def setEditFlag(self):
        """ editFlagをセットする。titleに「*」を追加する"""
        self.editFlag = "edited"
        title = "*" + self.windowTitle
        self.mainWindow.set_title(title)

    #
    #  clearEditFlag
    def clearEditFlag(self):
        """ editFlagをクリアする。titleから「*」を削除"""
        self.editFlag = "no"
        self.mainWindow.set_title(self.windowTitle)

    #
    #  initialize
    #--------------
    def initialize(self):
        """ 初期化"""
        self.windowTitle = _("snappyHexMeshによるmesh作成")
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        self.setWindowSize()        #windowSizeを設定
        self.initializeGui()        #GUIの文字をゴシックに設定
        self.setCaseDirToLabel()    #labelにcaseDirを設定
        self.loadInitialSetting()   #helyxOS_data内の値をセットする
        #page_createMeshの初期化
        self.checkSnappyCsvFile()   #csvfile有無を確認し、なければ作成する
        self.getSnappyCsvConts()    #csvFileのデータを取得する
        self.setDataToEntryBox()    #entryに値をセット
        self.createTreeList()       #TreeListを作成
        self.setDataToTreeList()    #TreeListにデータをセット
        self.setAdjustBndCheckBox() #boundaryFieldの調整有無をセット
        #toggleSWを設定
        self.toggle_createMesh.set_active(True)
        #page_addLayerの初期化はaddLayerPageを開いた時
        #page表示
        page = pageDict["createMesh"]
        self.noteBook.set_current_page(page)

    #
    #  setWindowSize
    def setWindowSize(self):
        """ windowSizeを読み込み設定する"""
        winDict = pyTreeFoam.readWindowSize()
        #全体windowを設定
        if "editSnappyHexMesh" in winDict.keys():
            xy = winDict["editSnappyHexMesh"]
            if xy != "":
                x, y = map(int, xy.split()[:2])
                self.mainWindow.set_default_size(x, y)
        #panedSplitterを設定
        if "editSnappyHexMesh_splitter" in winDict.keys():
            splitWidth = winDict["editSnappyHexMesh_splitter"]
            if splitWidth != "":
                width = int(splitWidth.split()[0])
                self.paned_win.set_position(width)

    #
    #  initializeGui
    def initializeGui(self):
        """ frameのlabelをboldに設定する"""
        bold_css = Gtk.CssProvider.new()
        cont  = b"label\n"
        cont += b"{\n"
        cont += b"    font-weight: bold;\n"
        cont += b"}\n"
        bold_css.load_from_data(cont)
        #page_createMesh
        style_action = self.label_blockMesh.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_action = self.label_snappyHexMesh.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_action = self.label_createMesh.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        #page_addLayer
        style_action = self.label_regionPatch.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_action = self.label_snappyHexMeshPatch.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_action = self.label_addLayerPatch.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)

    #
    #  loadInitialSetting
    def loadInitialSetting(self):
        """ helyxOS_data内に保存されているdataを設定する"""
        case = pyTreeFoam.case(self.caseDir)
        dataDict = case.getCreateMeshData()
        stlDir = dataDict["stlDir"]
        if stlDir[0] == "/":
            self.stlDir = stlDir
            relDir = os.path.relpath(stlDir, self.caseDir)
        else:
            self.stlDir = self.caseDir + "/" + stlDir
            relDir = stlDir
        self.entry_stlDir.set_text(relDir)
        if os.path.exists(self.stlDir) == False:
            #起動後、messageDialogを表示する
            self.newCsvFlag = 2
        self.angle = dataDict["surfaceAngle"]
        nProcs = dataDict["nCpu"]
        self.entry_nProcs.set_text(nProcs)
        self.entry_nProcsPatch.set_text(nProcs)
        nSeparate = dataDict["nSeparate"]
        self.entry_separate.set_text(nSeparate)
        self.entry_separatePatch.set_text(nSeparate)
        parallel = dataDict["parallel"]
        if parallel == "yes":
            self.check_parallel.set_active(True)
            self.check_parallelPatch.set_active(True)
        else:
            self.check_parallel.set_active(False)
            self.check_parallelPatch.set_active(False)
        self.changeParallel()

    #
    #  setCaseDirToLabel
    def setCaseDirToLabel(self):
        """ caseDirをlabelに追記"""
        caseDir = self.caseDir
        nMax = 80
        if len(caseDir) > nMax:
            caseDir = "..." + caseDir[-(nMax-3):]
        self.label_caseDir.set_text(caseDir)

    #
    #  checkSnappyCsvFile
    def checkSnappyCsvFile(self):
        """ csvファイル有無を確認し、無ければ作成する"""
        if os.path.exists(self.stlDir) == False:
            return
        #stlNamesを取得
        stlNames = getStlFiles(self.stlDir)
        #csvをチェック        
        fileName = self.stlDir + "/" + self.snappyCsvName
        if os.path.exists(fileName) == True:
            #csv内容をチェック
            csvData = readCreateMeshCsvData(fileName)
            stlConts = csvData[3]
            names = list(map(lambda x: x[0], stlConts))
            names.sort()
            #stlFilesとcsvを比較
            if stlNames == names:
                #同じ場合は、直ぐに戻る
                return
            else:
                #異なる場合は、csvを削除
                os.remove(fileName)
                print(_("csvDataを再作成します。"))
                #csvをdefault値で作成する
                self.createDefaultSnappyCsvFile(fileName)
                #flagを設定(dialogを表示させるため)
                self.newCsvFlag = 3
        else:
            #csvをdefault値で作成する
            print(_("csvDataが存在しない為、default値で作成します。"))
            self.createDefaultSnappyCsvFile(fileName)
            #flagを設定(dialogを表示させるため)
            self.newCsvFlag = 1

    #
    #  createDefaultSnappyCsvFile
    def createDefaultSnappyCsvFile(self, fileName):
        """ defaultの内容でcsvを作成する"""
        stlNames = getStlFiles(self.stlDir)
        minLoc, maxLoc = self.getStlModelMinMax_sub(stlNames)
        x = maxLoc[0] - minLoc[0]
        y = maxLoc[1] - minLoc[1]
        z = maxLoc[2] - minLoc[2]
        size = max([x,y,z]) / 30.0
        self.block_cellSize = pyTreeFoam.float2strAuto(size, prec=3)
        #overBlockSizeを設定
        self.block_overBlockSize = "5"
        #locationInMeshを取得
        locx = (maxLoc[0] + minLoc[0]) / 2.0
        locy = (maxLoc[1] + minLoc[1]) / 2.0
        locz = (maxLoc[2] + minLoc[2]) / 2.0
        x = pyTreeFoam.float2strAuto(locx, prec=3)
        y = pyTreeFoam.float2strAuto(locy, prec=3)
        z = pyTreeFoam.float2strAuto(locz, prec=3)
        self.locationInMesh = [x, y, z]
        #stlConts作成
        self.stlConts = []
        for stlName in stlNames:
            #row = ["" for i in range(8)]
            row = ["" for i in range(9)]
            row[stlColDict["stl"]] = stlName
            row[stlColDict["sect"]] = "patch"
            row[stlColDict["edge"]] = self.block_cellSize
            row[stlColDict["cell"]] = self.block_cellSize
            row[stlColDict["slave"]] = ""
            self.stlConts.append(row)
        #file作成
        self.csvRowCol = self.createCsvRowCol_sub()
        lines = []
        for i in range(len(self.csvRowCol)):
            line = ",".join(self.csvRowCol[i]) + "\n"
            lines.append(line)
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

    #
    #  getDataFromEntryBox
    def getDataFromEntryBox(self):
        """ entryBox内のdataを取得する"""
        #blockMeshの値を設定
        self.block_cellSize = self.entry_cellSize.get_text()
        self.block_overBlockSize = self.entry_nCells.get_text()
        # #snappyHexMeshの値をセット
        # cont = self.label_locationXYZ.get_text()
        # cont = cont.replace(" ", "")
        # words = cont[1:-1].split(",")
        # self.locationInMesh = words[:3]

    #
    #  setDataToEntryBox
    def setDataToEntryBox(self):
        """ entryBox内にdataを設定する。"""
        #blockMeshの値を設定
        self.entry_cellSize.set_text(self.block_cellSize)
        self.entry_nCells.set_text(self.block_overBlockSize)
        #snappyHexMeshの値をセット
        loc = list(map(float, self.locationInMesh))
        locx = pyTreeFoam.float2strAuto(loc[0], prec=3, minNum=0.001, maxNum=1000)
        locy = pyTreeFoam.float2strAuto(loc[1], prec=3, minNum=0.001, maxNum=1000)
        locz = pyTreeFoam.float2strAuto(loc[2], prec=3, minNum=0.001, maxNum=1000)
        self.locationInMesh = [locx, locy, locz]
        cont = "( " + ", ".join(self.locationInMesh) + " )"
        cont = "%-32s" % cont
        self.label_locationXYZ.set_text(cont)
        #featureAngleをセット
        case = pyTreeFoam.case(self.caseDir)
        featureAngle = case.getCreateMeshData()["createMesh_featureAngle"]
        self.entry_featureAngle.set_text(featureAngle)

    #
    #  getCellSizesForCombo
    def getCellSizesForCombo(self, cellSize):
        """ comboBoxの候補となるcellSizeを取得する"""
        if type(cellSize) == str:
            cellSize = float(cellSize)
        comboSizes = []
        for i in range(10):
            size = pyTreeFoam.float2strAuto(cellSize, prec=3)
            comboSizes.append(size)
            cellSize /= 2.0
        return comboSizes        

    #
    #  createTreeList
    def createTreeList(self):
        """ treeListを作成（定義）する。"""
        widthes = []
        #cellSizeの候補を作成する
        cellSize = float(self.block_cellSize)
        comboSizes = self.getCellSizesForCombo(cellSize)
        #header文字を作成
        hd0 = "stl"
        hd1 = "patch\ntype"
        hd2 = _("特徴線") + "\n" + _(" 抽出")
        hd3 = _("特徴線") + "\n" + _("cellSize")
        hd4 = _("cellSize")
        hd5 = _("layer") + "\n" + _("追加")
        hd6 = _("layer") + "\n" + _("数")
        hd7 = _("layer") + "\n" + _("拡大率")
        hd8 = _("layer") + "\n" + _("setSlave")
        #treeListの定義
        #        text combo tggle text text  toggle   text    text   combo
        #        stl  type  FEdge edge cell  addLayer nLayer, expan, setSlave
        model = (str, str,  bool, str, str,  bool,    str,    str,   str)
        header= [hd0, hd1,  hd2,  hd3, hd4,  hd5,     hd6,    hd7,   hd8]
        comboTypes = ["patch", "wall", "empty", "symmetry", 
                 "symmetryPlane", "cellZone", "cellZoneOther", "faceZone", 
                 "region", "face"]
        #comboSizesFace = ["(clear)"] + comboSizes
        comboSlaves = ["(clear)", "setSlave"]
        tree = treeListColDict
        combos = [(tree["type"], tree["edge"], tree["cell"], tree["setSlave"]),      #comboのColNo 
                  (comboTypes,   comboSizes,   comboSizes, comboSlaves)]         #combo内容
        editables = [tree["nLayer"], tree["expRatio"]]      #colNo
        eventFuncs = [self.onChangeCheckBox, self.onChangeComboBox, self.onEditedLayers]
        #treeListを作成        
        self.treeList.create(
                model, 
                multi=True, 
                header=True, 
                headerConts=header,
                colWidthes=widthes,
                combos=combos,
                editables=editables,
                eventFuncs=eventFuncs
                )

    #
    #  setDataToTreeList
    def setDataToTreeList(self):
        """ TreeViewにdataをセットし直す"""
        items = []
        for stlCont in self.stlConts:
            (stlName, patchType, edgeSize, 
             cellSize, nLayers, expan, setSlave) = stlCont
            featureEdge = False
            if edgeSize != "":
                featureEdge = True
            addLayers = False
            if nLayers != "":
                addLayers = True
            item = [stlName, patchType, featureEdge, edgeSize, 
                    cellSize, addLayers, nLayers, expan, setSlave]
            items.append(item)            
        #treeListにセット
        self.treeList.setItems(items)

    #
    #  setAdjustBndCheckBox
    def setAdjustBndCheckBox(self):
        """ checkBoxを初期化する"""
        self.check_adjustBnd.set_active(True)

    #
    #  selectRowsInTreeList
    def selectRowsInTreeList(self, selNames):
        """ treeList内の項目を選択する"""
        if len(selNames) == 0:
            return
        allNames = list(map(lambda x: x[0], self.stlConts))
        selNos = []
        for selName in selNames:
            selNo = allNames.index(selName)
            selNos.append(selNo)
        self.treeList.selectItemNos(selNos)

    #
    #  getSnappyCsvConts
    def getSnappyCsvConts(self):
        """ snappyCsvの内容を取得する"""
        fileName = self.stlDir + "/" + self.snappyCsvName
        csvData = readCreateMeshCsvData(fileName)
        (self.block_cellSize, self.block_overBlockSize, 
         self.locationInMesh, self.stlConts) = csvData
        self.blockMinMax = self.getBlockMeshMinMax()

    #  getBlockMeshMinMax
    def getBlockMeshMinMax(self):
        """ bllockMeshのminMax座標を全stlから取得する"""
        #stlのminMax座標を取得
        minLoc, maxLoc = self.getStlModelMinMax()
        #blockのminMaxを取得
        overValue = float(self.block_cellSize) * float(self.block_overBlockSize)
        minLoc = list(map(lambda x: x - overValue, minLoc))
        maxLoc = list(map(lambda x: x + overValue, maxLoc))
        blockMinMax = [minLoc, maxLoc]
        return blockMinMax

    #  getStlModelMinMax
    def getStlModelMinMax(self):
        """ 全stlのminMaxを取得する"""
        #stlのminMax座標を取得
        stlNames = list(map(lambda x: x[0], self.stlConts))
        minLoc, maxLoc = self.getStlModelMinMax_sub(stlNames)
        return (minLoc, maxLoc)

    def getStlModelMinMax_sub(self, stlNames):
        if len(stlNames) == 0:
            return [[0,0,0], [0,0,0]]
        minValues = []; maxValues = []
        for stlName in stlNames:
            fileName = self.stlDir + "/" + stlName + ".stl"
            minMax = createBlockAndSnappyDict.getMinMaxValueStl(fileName)
            minValues.append(minMax[0])
            maxValues.append(minMax[1])
        minXs = list(map(lambda x: x[0], minValues))
        minYs = list(map(lambda x: x[1], minValues))
        minZs = list(map(lambda x: x[2], minValues))
        minX = min(minXs); minY = min(minYs); minZ = min(minZs)
        minLoc = [minX, minY, minZ]
        maxXs = list(map(lambda x: x[0], maxValues))
        maxYs = list(map(lambda x: x[1], maxValues))
        maxZs = list(map(lambda x: x[2], maxValues))
        maxX = max(maxXs); maxY = max(maxYs); maxZ = max(maxZs)
        maxLoc = [maxX, maxY, maxZ]
        return (minLoc, maxLoc) 

    #
    #  getStlContsFromTreeList
    def getStlContsFromTreeList(self):
        """ treeListからdata（self.stlConts）を取得し返す"""
        listStore = self.treeList.treeList.get_model()
        tree = treeListColDict
        getCols = [tree["stl"],  tree["type"], tree["edge"],
                   tree["cell"], tree["nLayer"], tree["expRatio"],
                   tree["setSlave"]]
        stlConts = []
        for i in range(len(listStore)):
            row = []
            for col in getCols:
                path = str(i)
                val = listStore[path][col]
                row.append(val)
            stlConts.append(row)
        return stlConts

    #
    #  changeCursor
    def changeCursor(self, widget):
        """ delayを設定して、treeListのselectItemsを取得し、表示する。"""
        GLib.timeout_add(50, self.getShowSelectedItems)

    #
    #  getShowSelecteditems
    def getShowSelectedItems(self):
        """ 選択stlを取得してvtk表示する"""
        selNames = self.getSelectedStls()
        self.showSelectedItems(selNames)

    #
    #  changeCheckBox
    #-----------------
    def changeCheckBox(self, path, col):
        """ treeListのcheckBoxを変更した時のevent"""
        self.setEditFlag()              #修正flagをセット
        if col == treeListColDict["feature"]:
            self.changeFeatureEdge(path, col)
        elif col == treeListColDict["addLayer"]:
            self.changeAddLayers(path, col)    

    #  changeFeatureEdge
    def changeFeatureEdge(self, path, col):
        """ featureEdgeのcheckを変更した時のevent"""
        if col == treeListColDict["feature"]:
            listStore = self.treeList.treeList.get_model()
            if listStore[path][col] == True:
                cellSize = listStore[path][col+2]
            else:
                cellSize = ""
            listStore[path][col+1] = cellSize
            self.stlConts[int(path)][stlColDict["edge"]] = cellSize

    #  changeAddLayers
    def changeAddLayers(self, path, col):
        """ addLayersのcheckを変更した時のevent"""
        if col == treeListColDict["addLayer"]:
            listStore = self.treeList.treeList.get_model()
            if listStore[path][col] == True:
                nLayers = "3"
                expan = "1.2"
            else:
                nLayers = ""
                expan = ""
            listStore[path][col+1] = nLayers
            listStore[path][col+2] = expan
            self.stlConts[int(path)][stlColDict["layer"]] = nLayers
            self.stlConts[int(path)][stlColDict["expan"]] = expan

    #
    #  changeComboBox
    #------------------
    def changeComboBox(self, path, col):
        """ treeViewのinternalCellSizeを変更したときのevent"""
        #修正Flag設定
        self.setEditFlag()              #修正flagをセット
        #type変更？
        if col == treeListColDict["type"]:
            listStore = self.treeList.treeList.get_model()
            patchType = listStore[path][col]
            self.stlConts[int(path)][stlColDict["sect"]] = patchType
        #cellSize変更？
        elif col == treeListColDict["cell"]:
            listStore = self.treeList.treeList.get_model()
            cellSize = listStore[path][col].split()[0]
            self.stlConts[int(path)][stlColDict["cell"]] = cellSize
            if listStore[path][col-2] == True:
                edgeCellSize = listStore[path][col-1]
                if float(edgeCellSize) > float(cellSize):
                    listStore[path][col-1] = cellSize
                    self.stlConts[int(path)][stlColDict["edge"]] = cellSize
        #edgeSize変更？
        elif col == treeListColDict["edge"]:
            listStore = self.treeList.treeList.get_model()
            edgeCellSize = listStore[path][col].split()[0]
            self.stlConts[int(path)][stlColDict["edge"]] = edgeCellSize            
            if listStore[path][col-1] == False:
                listStore[path][col-1] =True
        #setSlave変更？
        elif col == treeListColDict["setSlave"]:
            listStore = self.treeList.treeList.get_model()
            setSlave = listStore[path][col].split()[0]
            if setSlave == "(clear)":
                setSlave = ""
            listStore[path][col] = setSlave
            self.stlConts[int(path)][stlColDict["slave"]] = setSlave

    #
    #  editedLayers
    #----------------
    def editedLayers(self, path, col):
        """ treeViewのtextEdit終了時のevent"""
        #修正Flag設定
        self.setEditFlag()              #修正flagをセット
        #nLayer編集？
        if col == treeListColDict["nLayer"]:
            listStore = self.treeList.treeList.get_model()
            nLayers = listStore[path][col]
            self.stlConts[int(path)][stlColDict["layer"]] = nLayers
        #expansionRatio編集？
        elif col == treeListColDict["expRatio"]:
            listStore = self.treeList.treeList.get_model()
            expan = listStore[path][col]
            self.stlConts[int(path)][stlColDict["expan"]] = expan

    #
    #  saveSnappyCsv
    #-----------------
    def saveSnappyCsv(self):
        """ csvDataの保存"""
        #修正Flagクリア
        self.clearEditFlag()        #修正flagをクリア
        pageName = self.getCurrPageName()
        if pageName == "createMesh":
            fileName = self.stlDir + "/" + self.snappyCsvName
            #treeListのdataを取得
            self.csvRowCol = self.createCsvRowCol()
            lines = []
            for i in range(len(self.csvRowCol)):
                line = ",".join(self.csvRowCol[i]) + "\n"
                lines.append(line)
            f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
            nProcs = self.entry_nProcs.get_text()
            separate = self.entry_separate.get_text()
        elif pageName == "addLayer":
            fileName = self.caseDir + "/" + self.addLayersCsvName
            conts = self.createPatchCsv()
            #treeListPatchのdataを取得
            lines = []
            for i in range(len(conts)):
                line = ",".join(conts[i]) + "\n"
                lines.append(line)
            f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
            nProcs = self.entry_nProcsPatch.get_text()
            separate = self.entry_separatePatch.get_text()
        #featureAngleを保存
        featureAngle = self.entry_featureAngle.get_text()
        featureAnglePatch = self.entry_featureAnglePatch.get_text()
        dataDict = {}
        dataDict["createMesh_featureAngle"] = featureAngle
        dataDict["addLayer_featureAngle"] = featureAnglePatch
        dataDict["nCpu"] = nProcs
        dataDict["nSeparate"] = separate
        case = pyTreeFoam.case(self.caseDir)
        case.setCreateMeshData(dataDict)
        print(_("保存しました"))

    #
    #  createCsvRowCol
    def createCsvRowCol(self):
        """ GUIのdetaからcsvRowColを作成する"""
        #GUIdataを取得
        self.getDataFromEntryBox()
        self.stlConts = self.getStlContsFromTreeList()
        rows = self.createCsvRowCol_sub()
        return rows

    def createCsvRowCol_sub(self):
        """ cellSize, nCells, stlContsからrowListを作成する"""
        cellSize = self.block_cellSize
        nCells = self.block_overBlockSize
        locx, locy, locz = self.locationInMesh
        #取得結果をlist形式（csvRowCol）を作成
        rowCols = self.stlConts
        rows = []
        rows += [["","","","","","","","",""]]
        rows += [["<blockMesh>", "", "x", "y", "z", "備考","","",""]]
        rows += [["", "cellSize", cellSize, cellSize, cellSize, "blockMeshのcellSize","","",""]]
        rows += [["", "overBlockSize", nCells, nCells, nCells, "cells: stlのMinMax値を越えるcell数","","",""]]
        rows += [["","","","","","","","",""]]
        rows += [["<snappyHexMesh>", "", "", "", "", "","","",""]]
        rows += [["", "mesh", locx, locy, locz, "meshの位置(materialPoint)","","",""]]
        rows += [["stlFile", '"sect\n(patch/wall/\nempty/symmetry(Plane)/\nfaceZone/face/\ncellZone/reg)"', '"featureEdge\ncellSize"', '"base\ncellSize"', '"fine\ncellSize"', '"featureEdge: cellSizeを入力したstlのみ抽出。\nbase: surface, regionとも設定する。"',"nLayers","expansionRatio","setSlave"]]
        for rowData in rowCols:
            #途中に「"",""」を追加してcsvに合わせる(faceとnote)
            rows += [rowData[:4] + ["",""] + rowData[4:]]
        return rows

    #
    #  reloadSnappyCsv
    #------------------
    def reloadSnappyCsv(self):
        """ csvファイルを再読込する。"""
        self.clearEditFlag()        #修正flagをクリア
        pageName = self.getCurrPageName()
        if pageName == "createMesh":
            self.getSnappyCsvConts()    #csvFileのデータを取得する
            self.setDataToEntryBox()    #entryに値をセット
            self.createTreeList()       #TreeListを作成
            self.setDataToTreeList()    #TreeListにデータをセット
        elif pageName == "addLayer":
            self.getAddLayersConts()
            self.createTreeListPatch()
            self.setDataToTreeListPatch()
        print(_("再読込みしました。"))

    #
    #  reloadAtDefault
    #------------------
    def reloadAtDefault(self):
        """ csvを再読込し、各定数をdefault値に設定する"""
        title = _("default値の設定")
        mess  = _("stlファイルからdefault値を設定します。") + "\n\n" 
        mess += _("default値を設定すると、") + "\n"
        mess += "  " + _("現在の設定内容が全て初期値に戻ります。") + "\n\n"
        mess += _("このまま、default値を設定しますか？")
        funcOk = [self.reloadAtDefault_run]
        self.okCancelDialog(title, mess, funcOk)

    #
    #  reloadAtDefault_run
    def reloadAtDefault_run(self):
        """ csvを再読込し、各定数をdefault値に設定する"""
        pageName = self.getCurrPageName()
        if pageName == "createMesh":
            #csvFileのデータを取得する
            self.checkSnappyCsvFile()
            self.getSnappyCsvConts()    
            #cellSizeを取得
            minLoc, maxLoc = self.getStlModelMinMax()
            x = maxLoc[0] - minLoc[0]
            y = maxLoc[1] - minLoc[1]
            z = maxLoc[2] - minLoc[2]
            size = max([x,y,z]) / 30.0
            self.block_cellSize = pyTreeFoam.float2strAuto(size, prec=3)
            #overBlockSizeを設定
            self.block_overBlockSize = "5"
            #locationInMeshを取得
            locx = (maxLoc[0] + minLoc[0]) / 2.0
            locy = (maxLoc[1] + minLoc[1]) / 2.0
            locz = (maxLoc[2] + minLoc[2]) / 2.0
            x = pyTreeFoam.float2strAuto(locx, prec=3)
            y = pyTreeFoam.float2strAuto(locy, prec=3)
            z = pyTreeFoam.float2strAuto(locz, prec=3)
            self.locationInMesh = [x, y, z]
            #entryに値をセット
            self.setDataToEntryBox()
            self.treeList.unselectAllItems()
            self.setDefaultValue_run()
        elif pageName == "addLayer":
            self.getPatchConts()
            self.createTreeListPatch()
            self.setDataToTreeListPatch()

    #
    #  runCheckMesh
    #---------------
    def runCheckMesh(self):
        """ checkMeshの実行"""
        self.menuCheckMesh()

    #
    #  runStlViewer
    #---------------
    def runStlViewer(self):
        """ stlViewerの起動"""
        stlDir = self.stlDir
        stlDir = pyTreeFoam.getAbsDir(self.caseDir, stlDir)
        if os.path.exists(stlDir) == False or os.path.isdir(stlDir) == False:
            title = _("エラー")
            mess = _("stlDir「" + stlDir + "」は、存在しません。") + "\n"
            mess += _("stlDirを再設定してください。")
            self.errDialog(title, mess)
            return
        path = os.getenv("TreeFoamPath") + os.sep + "python"
        comm = "./editStlFilesDialog.py " + self.caseDir + " " + stlDir + " &"
        pyTreeFoam.run(path).command(comm)

    #
    #  runMeshViewer
    #---------------
    def runMeshViewer(self):
        """ meshViewerを起動"""
        caseDir = self.caseDir
        stlDir = self.stlDir
        try:
            timeFolder = pyTreeFoam.case(caseDir).getCurrTimeFolder()
        except:
            timeFolder = "0"
        region = "."
        title = "meshViewer"
        self.meshViewer = meshViewerControl.control(caseDir, timeFolder, region, title)
        self.meshViewer.openViewer("full")
        commLines  = [_("start\n")]
        commLines += [_("mess mesh作成caseから起動しています。\n")]
        commLines += [_("addMess '  patch, zone, set, stlが表示できます。'\n")]
        commLines += [_("addMess '  tree内の選択した項目が表示されます。'\n")]
        commLines += [_("time startFrom\n")]
        commLines += [_("stlDir ") + stlDir + "\n"]
        commLines += [_("print 'loading ...'\n")]
        commLines += [_("wait 0.2s\n")]     #GUIを完成させる為のwait
        commLines += [_("load all\n")]
        commLines += [_("show\n")]
        commLines += [_("resetCamera\n")]
        commLines += [_("setTreeList\n")]
        commLines += [_("end\n")]
        _stat = self.meshViewer.runMultiCommands(commLines)

    #
    #  runParaview
    #--------------
    def runParaview(self):
        """ paraviewを起動"""
        contDict = pyTreeFoam.readConfigTreeFoam()
        paraFoam = contDict["paraFoam"]
        comm = paraFoam + " &"
        print(comm)
        pyTreeFoam.run(self.caseDir).commandWithLog(comm)

    #
    #  changeStlDir
    #---------------
    def changeStlDir(self):
        """ stlDirを変更した時のevent"""
        stlDir = self.entry_stlDir.get_text()
        if stlDir[0] == "/":
            absDir = stlDir
            relDir = os.path.relpath(stlDir, self.caseDir)
            self.entry_stlDir.set_text(relDir)
        else:
            relDir = stlDir
            absDir = self.caseDir + "/" + stlDir
        self.stlDir = absDir
        self.checkSnappyCsvFile()   #csvfile有無を確認し、なければ作成する
        self.getSnappyCsvConts()    #csvFileのデータを取得する
        self.setDataToEntryBox()    #entryに値をセット
        self.createTreeList()       #TreeListを作成
        self.setDataToTreeList()    #TreeListにデータをセット
        #self.createActorDict()
        self.renderer = ""
        self.setVtkModel()

    #
    #  stlDirRef
    #------------
    def stlDirRef(self):
        """ 「参照...」strDirを選択する。"""
        relDir = self.entry_stlDir.get_text()
        stlDir = self.caseDir + "/" + relDir
        if os.path.exists(stlDir) == False:
            selDir = "."
        else:
            selDir = relDir
        title = _("stlDirの選択")
        cont = _("相対参照でstlDirを取得する。")
        rootDir = os.getenv("HOME")
        currDir = self.caseDir
        okArgs = [self.stlDirRef_setTextBox]
        dialog = getOpenFolderDDialog.getOpenFolder(
            title, cont, rootDir, currDir, selDir, 
            funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #
    #  stlDirRef_setTextBox
    def stlDirRef_setTextBox(self, openDir):
        self.entry_stlDir.set_text(openDir)
        self.changeStlDir()

    #
    #  openStlDir
    #-------------
    def openStlDir(self):
        """ stlDirを開く"""
        if os.path.exists(self.stlDir) == False:
            title  = _("stlDir内の読み込み") + "\n"
            mess  = _("stlDirが存在しません。") + "\n"
            mess += _("  stlDirを変更してください。")
            self.okDialog(title, mess)
            return
        run = pyTreeFoam.run()
        comm = pyTreeFoam.fileManager + " " + self.stlDir + " &"
        run.command(comm)

    #
    #  openCaseDir
    #--------------
    def openCaseDir(self):
        """ caseDirを開く"""
        run = pyTreeFoam.run()
        comm = pyTreeFoam.fileManager + " " + self.caseDir + " &"
        run.command(comm)

    #
    #  changeCellSize
    #----------------
    def changeCellSize(self):
        """ cellSizeを変更したときのevent"""
        self.setDefaultValueForCellSize()

    #
    #  changeNCells
    #---------------
    def changeNCells(self):
        """ blockMeshのnCellsを変更した時のevent"""
        self.setEditFlag()              #修正flagをセット
        selNames = self.getSelectedStls()
        self.createShowSelectedItems(selNames)

    #
    #  changelocxyz
    #---------------
    def changelocxyz(self):
        """ locationInMeshの座標を変更した時のevent"""
        #self.editFlag = "edited"    #修正Flagセット
        self.setEditFlag()              #修正flagをセット
        selNames = self.getSelectedStls()
        self.createShowSelectedItems(selNames)

    #
    #  changeFeatureAngle
    #---------------------
    def changeFeatureAngle(self):
        self.setEditFlag()

    #
    #  excludeStls
    #--------------
    def excludeStls(self):
        """ 選択stlをlistから除外する。
        file名を'_'付きに変更してlistから除外する。"""
        rows = self.treeList.getSelectedItemNames()
        selNames = []
        for row in rows:
            selName = row[0]
            selNames.append(selName)
            stlFile = self.stlDir + "/" + selName + ".stl"
            newName = self.stlDir + "/_" + selName + ".stl"
            os.rename (stlFile, newName)
        newConts = []
        for row in self.stlConts:
            if not row[0] in selNames:
                newConts.append(row)
        self.stlConts = newConts
        self.setDataToTreeList()
        self.saveSnappyCsv()

    #
    #  setDefaultValue
    #-------------------
    def setDefaultValue(self):
        """ treeView内をdefault値に設定する"""
        title = _("default値の設定")
        mess  = _("blockMeshのcellSizeからdefault値を設定します。") + "\n\n"
        mess += _("default値を設定すると、") + "\n"
        mess += "  " + _("現在の設定内容が全て初期値に戻ります。") + "\n\n"
        mess += _("このまま、default値を設定しますか？")
        funcOk = [self.setDefaultValue_run]
        self.okCancelDialog(title, mess, funcOk)

    #
    #  setDefaultValue_run
    def setDefaultValue_run(self):
        """ treeView内をdefault値に設定する"""
        self.setEditFlag()              #修正flagをセット
        stl = stlColDict
        #clearする項目
        blankCol = (stl["edge"], stl["layer"], stl["expan"], stl["slave"])
        #cellSizeを設定する項目
        setCol = (stl["edge"], stl["cell"])
        self.getDataFromEntryBox()
        cellSize = self.block_cellSize
        #patchTypeを設定
        patchType = "patch"
        for row in range(len(self.stlConts)):
            for col in range(len(self.stlConts[row])):
                if col in blankCol:
                    self.stlConts[row][col] = ""
                if col in setCol:
                    self.stlConts[row][col] = cellSize
                if col == stl["sect"]:
                    self.stlConts[row][col] = patchType
        selNames = self.getSelectedStls()
        self.createShowSelectedItems(selNames)

    #
    #  getNearestValeInCells
    def getNearestValeInCells(self, cellVals, currVal):
        """ cellVals内から、currValに近い値を取得して返す。
        比較は、対数で比較する。"""
        
        def calcDiff(a, b):
            """ 対数の差分を計算"""
            diff = abs(math.log10(a) - math.log10(b))
            return diff

        diff = calcDiff(float(cellVals[0]), float(currVal))
        si = 0
        for i in range(1, len(cellVals), 1):
            cellVal = cellVals[i]
            newDiff = calcDiff(float(cellVal), float(currVal))
            if diff >= newDiff:
                diff = newDiff
            else:
                si = i - 1
                break
        return cellVals[si]

    #
    #  setDefaultValueForCellSize
    #-----------------------------
    def setDefaultValueForCellSize(self):
        """ cellSizeを変更した時の初期値を設定する。"""
        self.setEditFlag()              #修正flagをセット
        stl = stlColDict
        #設定していたstlのみcellSizeを再設定
        blankCol = (stl["edge"],)
        #cellSizeを再設定する項目
        setCol = (stl["cell"],)
        self.getDataFromEntryBox()
        cellSize = self.block_cellSize
        cellVals = self.getCellSizesForCombo(cellSize)
        #cellSizeを設定
        for row in range(len(self.stlConts)):
            for col in range(len(self.stlConts[row])):
                if col in blankCol:
                    #設定済の場合、再設定する
                    if self.stlConts[row][col] != "":
                        currVal = self.stlConts[row][col]    
                        newValue = self.getNearestValeInCells(cellVals, currVal)
                        #self.stlConts[row][col] = cellSize
                        self.stlConts[row][col] = newValue
                if col in setCol:
                    #cellSizeを再設定する
                    currVal = self.stlConts[row][col]
                    newValue = self.getNearestValeInCells(cellVals, currVal)
                    #self.stlConts[row][col] = cellSize
                    self.stlConts[row][col] = newValue
        selNames = self.getSelectedStls()
        self.createShowSelectedItems(selNames)

    #
    #  getSelectedStls
    def getSelectedStls(self):
        """ 選択しているstlNameを取得する"""
        selRowConts = self.treeList.getSelectedItemNames()
        selNames = list(map(lambda x:x[0], selRowConts))
        return selNames

    #
    #  unselectStls
    #---------------
    def unselectStls(self):
        """ 全stlを非選択する"""
        self.treeList.unselectAllItems()
        selNames = []
        self.showSelectedItems(selNames)

    #
    #  createDict
    #------------
    def createDict(self):
        """ blockMesh, snappyHexMeshDictを作成する。"""
        #csvを保存
        self.saveSnappyCsv()
        #blockMesh, snappyHexMeshDict, decomposeParDict有無チェック
        fileName = self.stlDir + "/" + self.snappyCsvName
        copyFiles = self.checkBlockMeshSnappyHexMesh()
        #csvDataを取得
        (otherData, data) = createBlockAndSnappyDict.readCsvData(fileName)
        (blockCellSize, nAddCells, meshLoc) = otherData
        (error, baseCellSize) = createBlockAndSnappyDict.getBaseCellSize(blockCellSize)
        (error, stlMinMax) = createBlockAndSnappyDict.getValueCsvData(baseCellSize, data)
        [stl, sect, featureEdge, minLevel, maxLevel] = stlMinMax
        stlDir = self.stlDir
        caseDir = self.caseDir
        #blockMeshを作成
        modelSize = createBlockAndSnappyDict.getModelSize(stlDir, stl, sect) 
        blockSize = createBlockAndSnappyDict.getBlockSize(modelSize, baseCellSize, nAddCells)
        blockCont = createBlockAndSnappyDict.readBlockMeshDict(caseDir)
        blockCont = createBlockAndSnappyDict.setBlockPointData(blockCont, blockSize, baseCellSize)
        #blockMeshのscale（倍率）をチェック
        blockCont = createBlockAndSnappyDict.checkBlockMeshMag(blockCont)
        #blockMeshの「type empty」をチェック
        blockCont = createBlockAndSnappyDict.checkBlockMeshEmpty(blockCont)
        #blockMeshDictを書き込む
        createBlockAndSnappyDict.writeBlockMeshDict(caseDir, blockCont)
        #snappyHexMeshDict作成
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        snappyCont = createBlockAndSnappyDict.readSnappyHexMeshDict(caseDir)
        snappyCont = createBlockAndSnappyDict.setCastellatedSnapFlag(snappyCont)
        snappyCont = createBlockAndSnappyDict.setGeometryData(snappyCont, stl, sect, OFversion)
        snappyCont = createBlockAndSnappyDict.setFeatureEdgeData(snappyCont, stl, sect, featureEdge)
        snappyCont = createBlockAndSnappyDict.setRefinementSurfaces(snappyCont, stl, sect, minLevel,
                        maxLevel)
        snappyCont = createBlockAndSnappyDict.setRefinementRegions(snappyCont, stl, sect, minLevel, baseCellSize)
        snappyCont = createBlockAndSnappyDict.checkFaceZone(snappyCont, sect)
        snappyCont = createBlockAndSnappyDict.setLocationInMesh(snappyCont, meshLoc)
        #layer追加
        snappyCont = self.addLayerSettingToSnappyDict(snappyCont)
        #snappy保存
        createBlockAndSnappyDict.writeSnappyHexMeshDict(caseDir, snappyCont)
        #featureEdgeDictを作成
        featureDict = self.createFeatureEdgeDict()
        self.copyStlFilesToConstant()
        #parallel？
        if self.check_parallel.get_active() == True:
            #並列処理
            caseDir = self.caseDir
            region = "."
            nProcs = self.entry_nProcs.get_text()
            nSeparate = self.entry_separate.get_text()
            error = self.nProcsCheck(nProcs, nSeparate)
            if error != "":
                return
            #decomposeParDictを修正
            remakeDecomposeParDict(caseDir, region, nProcs, nSeparate)

        title = _("meshDict作成")
        mess = ""
        if len(copyFiles) > 0:
            files = ", ".join(copyFiles)
            mess += _(files + "\n")
            mess += _("  ファイルが存在しなかった為、コピーしました。\n\n")
        mess += _("blocMeshDict, snappyHexMeshDict, ") + featureDict + "\n"
        mess += _("  ファイルを作成しました。\n")
        print(mess)
        self.okDialog(title, mess)
        return

    #
    #  checkBlockMeshSnappyHexMesh
    def checkBlockMeshSnappyHexMesh(self):
        """ meshのDictファイルをチェック"""
        copyFiles = []
        #blockMeshのチェック
        fileName = self.caseDir + "/system/blockMeshDict"
        if len(glob.glob(fileName)) == 0:
            copyFiles.append("blockMeshDict")
        #snappyHexMeshDictのチェック
        fileName = self.caseDir + "/system/snappyHexMeshDict"
        if len(glob.glob(fileName)) == 0:
            copyFiles.append("snappyHexMeshDict")
        #decomposeParDictのチェック
        fileName = self.caseDir + "/system/decomposeParDict"
        if len(glob.glob(fileName)) == 0:
            copyFiles.append("decomposeParDict")
        #meshQualityDictのチェック
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 6.0 or OFversion >= "v1906":
            fileName = self.caseDir + "/system/meshQualityDict"
            if len(glob.glob(fileName)) == 0:
                copyFiles.append("meshQualityDict")
        #チェックしたファイルが存在しなかった場合、コピーする
        if len(copyFiles) != 0:
            OFversion = self.configDict["OFversion"]
            sourceDir = pyTreeFoam.getFoamContents().getOFversionFolder(OFversion)
            for name in copyFiles:
                sourceFile = sourceDir + "/system/" + name
                pasteDir = self.caseDir + "/system"
                shutil.copy(sourceFile, pasteDir)
        return copyFiles

    #
    #  addLayerSettingToSnappyDict
    def addLayerSettingToSnappyDict(self, snappyCont):
        """ layerSettingをsnappyDictに追加する。"""
        #layerSettingを取得
        layerPatchSetting = []
        col = stlColDict["layer"]
        for rowData in self.stlConts:
            if rowData[col] != "":
                #          patchName    nLayers       finalTh minTh  expantionRatio  setSlave
                setting = [rowData[0], [rowData[col], "0.3", "0.25", rowData[col+1], rowData[col+2]]]
                layerPatchSetting.append(setting)
        #featureAngleを取得
        featureAngle = self.entry_featureAngle.get_text()
        contents = snappyCont
        #addLayersをTrueに設定
        contentsOp = pyTreeFoam.strOp(contents)
        ps = contentsOp.skipFoamFile()
        newLine = "addLayers true;"
        (contents, p, _stat) = contentsOp.replace1lineKeyword(ps, newLine)
        #layerpatchをセット
        (lineCont, p, kind) = contentsOp.get1line(ps)
        while lineCont != "":
            if kind == "keyword":
                lineContOp = pyTreeFoam.strOp(lineCont)
                (keyword, _pp) = lineContOp.getKeyword(0)
                if keyword == "addLayersControls":
                    ps = p - len(lineCont)      #ポインタを頭に戻す
                    header = contents[:ps]
                    footer = contents[p:]
                    #layerPatchをセット
                    ps = lineContOp.skipIntoPair(0)
                    newLine  = "layers\n    {\n"
                    for (patch, setting) in layerPatchSetting:
                        if setting[4] == "setSlave":
                            patch += "_slave"
                        newLine += "        " + patch + "\n"
                        newLine += "        {\n"
                        newLine += "            nSurfaceLayers " + setting[0] + ";\n"
                        newLine += "            finalLayerThickness " + setting[1] + ";\n"
                        newLine += "            minThickness " + setting[2] + ";\n"
                        newLine += "            expansionRatio " + setting[3] + ";\n"
                        newLine += "        }\n"
                    newLine += "    }"
                    (lineCont, _pp, _keyword) = lineContOp.replace1lineKeyword(ps, newLine)
                    #featureAngleをセット
                    newLine = "featureAngle " + featureAngle + ";"
                    (lineCont, _pp, _keyword) = lineContOp.replace1lineKeyword(ps, newLine)                    
                    #relativeSizesをセット
                    newLine = "relativeSizes true;"
                    (lineCont, _pp, _keyword) = lineContOp.replace1lineKeyword(ps, newLine)
                    break
            (lineCont, p, kind) = contentsOp.get1line(p)
        contents = header + lineCont + footer
        return contents

    #
    #  createFeatureEdgeDict
    def createFeatureEdgeDict(self):
        """ 特徴線を週出した後、blockMesh実行、snappyHexMeshを実行してmeshを作成する"""
        #特徴線を抽出
        eMeshFiles = []
        col = stlColDict["edge"]
        for rowData in self.stlConts:
            if rowData[col] != "":
                stlName = rowData[0] + ".stl"
                eMeshFiles.append(stlName)
        OFversion = self.configDict["OFversion"]
        ver = pyTreeFoam.getNumVersion(OFversion)
        if ver >= 8.0:
            #OF-8以上
            featureDict = self.featureEdge_8(eMeshFiles)
        else:
            #OF-8未満、OF-V、OF-ext
            featureDict = self.featureEdge_7(eMeshFiles)
        return featureDict

    #  featureEdge_7
    def featureEdge_7(self, stlFiles):
        """ OF-7以下、OF-v、OF-ext用のfeatureEdge抽出用のDict作成"""
        fileName = self.caseDir + "/system/surfaceFeatureExtractDict"
        angle = self.angle
        #dictをコピーして新しく作成する
        makeNewFeatureDict_7(self.caseDir, fileName, stlFiles, angle)
        featureDict = os.path.basename(fileName)
        return featureDict

    #  featureEdge_8
    def featureEdge_8(self, stlFiles):
        """ ver 8.0 以降。特徴線抽出のためのDictを作成、編集する"""
        fileName = self.caseDir + "/system/surfaceFeaturesDict"
        angle = self.angle
        makeNewFeatureDict_8(self.caseDir, fileName, stlFiles, angle)
        featureDict = os.path.basename(fileName)
        return featureDict

    #  copyStlFilesToConstant
    def copyStlFilesToConstant(self):
        """ stlFilseをconstant/triSurfeceにコピーする"""
        sourceDir = self.stlDir
        pasteDir = self.caseDir + "/constant/triSurface"
        if os.path.exists(pasteDir) == False:
            os.mkdir(pasteDir)
        for i in range(len(self.stlConts)):
            stlName = self.stlConts[i][0] + ".stl"
            copyFile = sourceDir + "/" + stlName
            shutil.copy(copyFile, pasteDir)

    #
    #  createMesh
    #-------------
    def createMesh(self):
        """ 特徴線抽出、blockMesh実行、snappyHexMesh実行してmeshを作成する"""
        OFversion = self.configDict["OFversion"]
        ver = pyTreeFoam.getNumVersion(OFversion)
        if ver >= 8.0:
            #OF-8以上
            featureExe = "surfaceFeatures"
        else:
            #OF-7以下, OF-v, OF-ext
            featureExe = "surfaceFeatureExtract"
        blockDict = "system/blockMeshDict"
        envOpenFoam = self.configDict["bashrcFOAM"]
        cont  = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        if self.check_parallel.get_active() == True:
            #並列処理
            caseDir = self.caseDir
            region = "."
            nProcs = self.entry_nProcs.get_text()
            nSeparate = self.entry_separate.get_text()
            error = self.nProcsCheck(nProcs, nSeparate)
            if error != "":
                return
            #remakeDecomposeParDict(caseDir, region, nProcs, nSeparate)
            cont += featureExe + " 2>&1 | tee solve.log\n"
            cont += "blockMesh -dict " + blockDict + " 2>&1 | tee -a solve.log\n"
            cont += "rm -rf processor\n"
            #cont += os.getenv("TreeFoamPath") + "/python/meetPatchNameDeleteNullPatch.py " + self.caseDir + "\n"
            #cont += "decomposePar -no-fields 2>&1 | tee -a solve.log\n"
            cont += createDecomposeParNoFieldsCommLine(caseDir)
            #cont += "mpirun -np " + nProcs + " snappyHexMesh -parallel 2>&1 | tee -a solve.log\n"
            cont += createSnappyParallelCommLine(nProcs)
            cont += createReconstructMeshCommLine(caseDir, region)
            cont += os.getenv("TreeFoamPath") + "/python/paraSnappyPost.py " + self.caseDir + "\n"
            if self.check_adjustBnd.get_active() == True:
                cont += os.getenv("TreeFoamPath") + "/python/meetPatchNameDeleteNullPatch.py " + self.caseDir + "\n"
        else:
            #シングルコア
            cont += featureExe + " 2>&1 | tee solve.log\n"
            cont += "blockMesh -dict " + blockDict + " 2>&1 | tee -a solve.log\n"
            cont += "snappyHexMesh -overwrite 2>&1 | tee -a solve.log\n"
            if self.check_adjustBnd.get_active() == True:
                cont += os.getenv("TreeFoamPath") + "/python/meetPatchNameDeleteNullPatch.py " + self.caseDir + "\n"
        fileName = self.caseDir + "/run"
        f = open(fileName, "w"); f.write(cont); f.close()
        run = pyTreeFoam.run(self.caseDir)
        run.command("chmod a+x " + fileName)
        run.foamTerminal("./run")

    #
    #  nProcsCheck
    def nProcsCheck(self, nProcs, nSeparate):
        """ 並列数のチェック"""
        words = nSeparate.split()
        n = 1
        for word in words:
            try:
                n = n * int(word)
            except:
                n = -1
                break
        if n > 0 and n == int(nProcs):
            error = ""
        else:
            error = "error"
            title = _("エラー")
            mess = _("並列数と分割数が合致していません。") + "\n"
            mess += _("並列数を見直してください。")
            self.errDialog(title, mess)
        return error

    #
    #  menuEditDict
    #----------------
    def menuEditDict(self):
        """ dictを編集する"""
        pageName = self.getCurrPageName()
        if pageName == "createMesh":
            self.editDict()
        elif pageName == "addLayer":
            self.editDictPatch()

    #
    #  menuCheckMesh
    #----------------
    def menuCheckMesh(self):
        """ checkMeshを実行する"""
        print(_("checkMeshを実行"))
        envOpenFoam = self.configDict["bashrcFOAM"]
        cont = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        cont += "checkMesh 2>&1 | tee solve.log\n"
        #スクリプト作成
        comm = "echo " + "'" + cont + "'" + " > run"
        runCase = pyTreeFoam.run(self.caseDir)
        runCase.command(comm)
        runCase.command("chmod a+x run")
        runCase.foamTerminal("run")

    #
    #  editDict
    #-----------
    def editDict(self):
        """ dictを編集する"""
        path = self.caseDir + "/system"
        snappyDict = path + "/snappyHexMeshDict"
        blockDict = path + "/blockMeshDict"
        qualityDict = path + "/meshQualityDict"
        OFversion = self.configDict["OFversion"]
        featureDict = path + "/" + getFeatureEdgeDictName(OFversion)
        if os.path.exists(qualityDict) == True:
            files = [snappyDict, blockDict, qualityDict, featureDict]
        else:
            files = [snappyDict, blockDict, featureDict]
        run = pyTreeFoam.run(self.caseDir)
        run.editor(files)

    #
    #  changeParallel
    #-----------------
    def changeParallel(self):
        """ parallelのcheckBoxを変更した時のevent"""
        #paget_createMesh
        if self.check_parallel.get_active() == True:
            self.entry_nProcs.set_sensitive(True)
            self.entry_separate.set_sensitive(True)
            self.label_nProcs.set_sensitive(True)
            self.label_separate.set_sensitive(True)
            self.label_method.set_sensitive(True)
        else:
            self.entry_nProcs.set_sensitive(False)
            self.entry_separate.set_sensitive(False)
            self.label_nProcs.set_sensitive(False)
            self.label_separate.set_sensitive(False)
            self.label_method.set_sensitive(False)
        #page_addlayer
        if self.check_parallelPatch.get_active() == True:
            self.entry_nProcsPatch.set_sensitive(True)
            self.entry_separatePatch.set_sensitive(True)
            self.label_nProcsPatch.set_sensitive(True)
            self.label_separatePatch.set_sensitive(True)
            self.label_methodPatch.set_sensitive(True)
        else:
            self.entry_nProcsPatch.set_sensitive(False)
            self.entry_separatePatch.set_sensitive(False)
            self.label_nProcsPatch.set_sensitive(False)
            self.label_separatePatch.set_sensitive(False)
            self.label_methodPatch.set_sensitive(False)

    #
    #  showPopupMenu
    def showPopupMenu(self, pos):
        """ popupMenuを表示する"""
        #popupMenu作成
        menu = Gtk.Menu()
        menuItems = self.popupMenuItems
        for i in range(len(menuItems)):
            menuName = menuItems[i]
            if menuName[0] == "-":
                #セパレータをセット
                item = Gtk.SeparatorMenuItem()
            else:
                item = Gtk.MenuItem()
                item.set_label(menuName[0])
                item.connect("activate", menuName[1])
            menu.append(item)
        #menuを表示
        menu.show_all()
        menu.popup(None, None, None, None, 0, Gtk.get_current_event_time())

    #
    #  afterSelectPathes
    def afterSelectPathes(self, pathes, selection):
        """ 0.05s後に選択する"""
        GLib.timeout_add(50, self.afterSelectPathes_sub, pathes, selection)

    def afterSelectPathes_sub(self, pathes, selection):
        for path in pathes:
            selection.select_path(path)

    #----------------------
    #  page2 addLayer layer追加
    #----------------------
    #
    #  checkRegionsPatch
    def checkRegionsPatch(self):
        """ region有無をcheckしてcomboBoxにセットする"""
        case = pyTreeFoam.case(self.caseDir)
        #regions = ["(region0)"]
        regions = case.getAllRegions()
        if regions[0] == ".":
            regions[0] = "(region0)"
        #(flag, loc, regNames) = case.isMultiRegion()
        #if flag == True:
        #    regions += regNames
        self.combo_regionPatch.setItems(regions)

    #
    #  setDataToEntryPatch
    def setDataToEntryPatch(self):
        """ entryにdataをセットする"""
        case = pyTreeFoam.case(self.caseDir)
        featureAngle = case.getCreateMeshData()["addLayer_featureAngle"]
        self.entry_featureAnglePatch.set_text(featureAngle)

    #
    #  checkAddLayersCsvFile
    def checkAddLayersCsvFile(self):
        """ addLayers,csv file有無を確認し、無ければ作成する。"""
        fileName = self.caseDir + "/" + self.addLayersCsvName
        if os.path.exists(fileName) == True:
            #csvからpatchContsを取得
            self.region, self.patchConts = readAddLayersCsvData(fileName)
            patchNames = list(map(lambda x: x[0], self.patchConts))
            patchNames.sort()            
            #boundaryからpatchContsを取得
            names = readRegionPatchNames(self.caseDir, self.region)
            #csvとboundaryが同じか？
            if patchNames == names and len(names) > 0:
                #同じ場合戻る
                return
            else:
                #異なる場合、csvを削除
                os.remove(fileName)
        #file作成
        print(_("addLayersの保存データを再作成します。"))
        #region = self.region
        case = pyTreeFoam.case(self.caseDir)
        regions = case.getAllRegions()
        self.region = regions[0]
        # (flag, loc, regNames) = case.isMultiRegion()
        # if flag == False:
        #     self.region = "."
        # else:
        #     if not self.region in regNames:
        #         self.region = "."
        region = self.region
        self.getPatchConts()      
        conts = []
        conts.append(["", "", "", ""])
        conts.append(["region", region, "", ""])
        conts.append(["", "", "", ""])
        conts.append(["patch", "nLayers", "expansionRatio"])
        for i in range(len(self.patchConts)):
            patchName = self.patchConts[i][0]
            nLayers = self.patchConts[i][2]
            expRatio = self.patchConts[i][3]
            conts.append([patchName, nLayers, expRatio])
        lines = []
        for i in range(len(conts)):
            line = ",".join(conts[i]) + "\n"
            lines.append(line)
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
    
    #
    #  getAddLayersConts
    def getAddLayersConts(self):
        """ addLayersCsvの内容を取得する"""
        fileName = self.caseDir + "/" + self.addLayersCsvName
        self.region, self.patchConts = readAddLayersCsvData(fileName)
        if self.region == ".":
            self.combo_regionPatch.selectItem(0)
        else:
            self.combo_regionPatch.selectItem(self.region)

    #
    #  createPatchCsv
    def createPatchCsv(self):
        """ addLayerCsvを作成する"""
        #設定data読み込み
        region = self.combo_regionPatch.getSelectedItem()
        if region == "region0" or region == "(region0)":
            region = "."
        self.region = region
        textConts = self.treeListPatch.getAllItemNames()
        self.patchConts = []
        for conts in textConts:
            patchName, nLayers, expRatio = conts
            if nLayers != "" and expRatio != "":
                flag = True
            else:
                flag = False
            cont = [patchName, flag, nLayers, expRatio]
            self.patchConts.append(cont)
        #csvData作成
        conts = []
        conts.append(["", "", ""])
        conts.append(["region", self.region, ""])
        conts.append(["", "", ""])
        conts.append(["patch", "nLayers", "expansionRatio"])
        for i in range(len(self.patchConts)):
            patchName = self.patchConts[i][0]
            nLayers = self.patchConts[i][2]
            expRatio = self.patchConts[i][3]
            conts.append([patchName, nLayers, expRatio])
        return conts

    #
    #  createTreeListPatch
    def createTreeListPatch(self):
        """ treeListPatchを定義する"""
        widthes = []
        #header文字を作成
        hd0 = "patch"
        hd1 = "layer" + "\n" + _("追加")
        hd2 = "layer" + "\n" + _("数")
        hd3 = "layer" + "\n" + _("拡大率")
        #treeListの定義
        #       text  toggle   text   text
        #       patch addLayer nLayer expan
        model = (str, bool,    str,   str)
        header= [hd0, hd1,     hd2,   hd3]
        tree = treeListPatchColDict
        editables = [tree["nLayer"], tree["expRatio"]]      #colNo
        #             checkBox                   combo   edited
        eventFuncs = [self.onChangeCheckBoxPatch, None, self.onEditedLayersPatch]
        #treeListを作成
        self.treeListPatch.create(
                model, 
                multi=True, 
                header=True, 
                headerConts=header,
                colWidthes=widthes,
                editables=editables,
                eventFuncs=eventFuncs
                )

    #
    #  getPatchConts
    def getPatchConts(self):
        """ patchConts(patchNames)を取得する"""
        patchNames = readRegionPatchNames(self.caseDir, self.region)
        self.patchConts = []
        for patchName in patchNames:
            #       patch      addLayer nLayer, expLayer
            cont = [patchName, False,   "",     ""]
            self.patchConts.append(cont)

    #
    #  setDataToTreeListPatch
    def setDataToTreeListPatch(self):
        """ treeViewpatchにデータをセットする"""
        items = []
        for patchCont in self.patchConts:
            item = patchCont
            items.append(item)
        #treeListにセット
        self.treeListPatch.setItems(items)

    #
    #  changeCheckBoxPatch
    #----------------------
    def changeCheckBoxPatch(self, path, col):
        """ treeListPatchのcheckBoxを変更した時のevent"""
        self.setEditFlag()
        if col == treeListPatchColDict["addLayer"]:
            listStore = self.treeListPatch.treeList.get_model()
            if listStore[path][col] == True:
                nLayers = "3"
                expan = "1.2"
            else:
                nLayers = ""
                expan = ""
            listStore[path][col+1] = nLayers
            listStore[path][col+2] = expan
            self.patchConts[int(path)][col+1] = nLayers
            self.patchConts[int(path)][col+2] = expan

    #
    #  editedLayersPatch
    #--------------------
    def editedLayersPatch(self, path, col):
        """ treeViewのtextEdit終了時のevent"""
        self.setEditFlag()
        listStore = self.treeListPatch.treeList.get_model()
        if col == treeListPatchColDict["nLayer"]:
            nLayers = listStore[path][col]
            self.patchConts[int(path)][col] = nLayers
        elif col == treeListPatchColDict["expRatio"]:
            expan = listStore[path][col]
            self.patchConts[int(path)][col] = expan

    #
    #  setVtkModelPatch
    #-------------------
    def setVtkModelPatch(self):
        """ meshのvtkを表示する"""
        #actorを作成する
        self.createPatchActorDict()
        #rendererをチェック
        if self.renderer == "":
            self.renderer = vtk.vtkRenderer()
        else:
            self.renderer.NewInstance()
        #assyを定義
        self.assy = vtk.vtkAssembly()
        #stlを表示（選択しているstlを取得して）
        selNames = self.getSelectedPatches()
        self.showSelectedPatchItems(selNames)

        #backgroundColorを設定
        self.renderer.SetBackground(0.85, 0.85, 0.85)
        self.renderer.SetBackground2(0.4, 0.55, 0.75)
        self.renderer.GradientBackgroundOn()

    #
    #  createPatchActorDict
    def createPatchActorDict(self):
        """ meshのactorを作成する"""
        #全patchのactor辞書を作成
        self.actorRegDict = {}
        self.outlineActorRegDict = {}
        #reader
        contFile = self.caseDir + "/system/controlDict"
        readerOF = vtk.vtkOpenFOAMReader()
        readerOF.SetFileName(contFile)
        case = pyTreeFoam.case(self.caseDir)
        timeFolder = case.getCurrTimeFolder()
        try:
            #timeFoladerが存在する場合
            floatTime = float(timeFolder)
            readerOF.UpdateTimeStep(floatTime)
        except:
            #timeFolderが存在しない場合
            readerOF.Update()
        readerOF.EnableAllPatchArrays()
        readerOF.Update()

        blockDataDict = {}
        output = readerOF.GetOutput()
        blockData = None
        #region名とdataを取得
        metaData = output.GetMetaData(0)
        blockName = self.getBlockName(metaData)
        if blockName == "internalMesh":
            #普通のcase
            blockData = output
            blockDataDict["."] = blockData
        elif blockName == "film":
            #filmCase
            blockData = output.GetBlock(0)
            blockDataDict["film"] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData
        elif blockName == "defaultRegion":
            #mutiRegionCase
            blockData = output.GetBlock(0)
            blockDataDict["."] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData
        else:
            #multiRegionCase(constant/polyMesh無しの場合)
            blockData = output.GetBlock(0)
            blockDataDict[blockName] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData

        #region名を取得
        allRegions = list(blockDataDict.keys())
        #region毎にactorを取得
        patchActorDict = {}
        outlineActorDict = {}
        for region in allRegions:
            blockData = blockDataDict[region]
            if blockData != None:
                #actorを取得
                patchActorDicts = self.getRegionData(blockData)
                patchActorDict[region] = patchActorDicts[0]
                outlineActorDict[region] = patchActorDicts[1]
            else:
                patchActorDict[region] = {}
                outlineActorDict[region] = {}
        self.actorRegDict = patchActorDict
        self.outlineActorRegDict = outlineActorDict
        self.actorDict = patchActorDict[self.region]
        self.outlineActorDict = outlineActorDict[self.region]

    #
    #  getRegionData
    def getRegionData(self, regionBlock):
        """ regionのblockDataからactorDictを取得する"""
        nBlocks = regionBlock.GetNumberOfBlocks()
        #blockDataからpatchDataを取得
        patchActorDict = {}; edgeActorDict = {}
        for i in range(nBlocks):
            metaData = regionBlock.GetMetaData(i)
            blockName = self.getBlockName(metaData)
            #vtk 9.1.0から名称変更
            #           9.0.3以前               9.1.0以降
            if blockName == "Patches" or blockName == "boundary":
                patchBlock = regionBlock.GetBlock(i)
                patchNamesIds = self.getPatchNamesIds(patchBlock)
                patchActorDict, edgeActorDict = self.createPatchActors(regionBlock, patchNamesIds)
                break
        patchActorDicts = [patchActorDict, edgeActorDict]
        return patchActorDicts

    #
    #  getBlockName
    def getBlockName(self, metaData):
        """ metaDataからそのblockのblockNameを取得する"""
        lines = str(metaData).split("\n")
        blockName = ""
        for line in lines:
            words = line.split()
            if words[0] == "NAME:":
                blockName = words[1]
                break
        return blockName

    #
    #  getPatchNamesIds
    def getPatchNamesIds(self, patchBlock):
        """ blockDataからpatchNameとpatchIdを取得する"""
        patchIds = []
        patchNames = []
        index = 3
        nBlocks = patchBlock.GetNumberOfBlocks()
        for i in range(nBlocks):
            metaData = patchBlock.GetMetaData(i)
            patchName = self.getBlockName(metaData)
            patchIds.append(index)
            patchNames.append(patchName)
            index += 1
        return patchNames, patchIds

    #
    #  createPatchActors
    def createPatchActors(self, regionBlock, patchNamesIds):
        """ patchActor, outlineActor(featureEdge)を作成する"""
        (patchNames, patchIds) = patchNamesIds
        patchActorDict = {}
        outlineActorDict = {}
        for i in range(len(patchIds)):
            index = patchIds[i]
            extract_block = vtk.vtkExtractBlock()
            extract_block.SetInputData(regionBlock)
            extract_block.AddIndex(index)
            extract_block.Update()
            #filter
            geom_filter = vtk.vtkGeometryFilter()
            geom_filter.SetInputConnection(extract_block.GetOutputPort())
            #featureEdge
            feature_edge = vtk.vtkFeatureEdges()
            feature_edge.SetInputConnection(geom_filter.GetOutputPort())
            #mapper
            mapper = vtk.vtkCompositePolyDataMapper()
            mapper.SetInputConnection(geom_filter.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(feature_edge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            prop = actor.GetProperty()
            prop.SetColor(colors.green)
            name = patchNames[i]
            patchActorDict[name] = actor
            edgeActor = vtk.vtkActor()
            edgeActor.SetMapper(edgeMapper)
            outlineActorDict[name] = edgeActor
        return patchActorDict, outlineActorDict

    #
    #  getSelectedPatches
    def getSelectedPatches(self):
        """ 選択しているstlNameを取得する"""
        selRowConts = self.treeListPatch.getSelectedItemNames()
        selNames = list(map(lambda x:x[0], selRowConts))
        return selNames

    #
    #  showSelectedPatchItems
    def showSelectedPatchItems(self, selNames):
        """ TreeItemの選択したitemをvtk表示させる"""
        if self.renderer == "":
            return
        #選択item名を取得
        self.allElms = list(map(lambda x: x[0], self.patchConts))
        self.showElms = selNames
        self.hideElms = list(set(self.allElms) - set(self.showElms))
        #assyをclear
        parts = list(self.assy.GetParts())
        if len(parts) > 0:
            self.renderer.RemoveActor(self.assy)
            self.renderer.SetErase(True)
            self.assy = vtk.vtkAssembly()
        #actorを再設定
        for showElm in self.showElms:
            if showElm in self.actorDict.keys():
                actor = self.actorDict[showElm]
                actor = self.setActorProperty(actor)
                prop = actor.GetProperty()
                prop.SetColor(colors.green)
                self.assy.AddPart(actor)
        #  outlineを追加
        if self.check_showOutline.get_active() == True:
            allItems = self.treeListPatch.getAllItemNames()
            stlNames = list(map(lambda x: x[0], allItems))
            for stlName in stlNames:
                actor = self.outlineActorDict[stlName]
                self.assy.AddPart(actor)
        self.renderer.AddActor(self.assy)
        #原点を表示
        self.addOriginAxis()
        parts = list(self.assy.GetParts())
        #初めて（削除するactorが無かった）の場合、表示を合わせる
        #self.fitWindow()
        #vtk表示
        self.vtkObj.Initialize()
        return

    #
    #  changeCursorPatch
    #--------------------
    def changeCursorPatch(self):
        """ treeListPatch内の行を選択した時のevent"""
        GLib.timeout_add(50, self.applyCheckContsToVtk)

    #
    #  changeRegionPatch
    #--------------------
    def changeRegionPatch(self):
        """ comboのregionを変更したときのevent"""
        region = self.combo_regionPatch.getSelectedItem()
        if region == "region0" or region == "(region0)":
            region = "."
        if region == self.region:
            return
        self.region = region
        self.createPatchActorDict()
        self.getPatchConts()
        self.createTreeListPatch()
        self.setDataToTreeListPatch()

    #
    #  setDefaultValuePatch
    #-----------------------
    def setDefaultValuePatch(self):
        """ patchContsをdefault値に設定する"""
        for i in range(len(self.patchConts)):
            patchName = self.patchConts[i][0]
            cont = [patchName, False, "", ""]
            self.patchConts[i] = cont
        selNames = self.getSelectedPatches()
        self.setDataToTreeListPatch()
        self.treeListPatch.selectItemNames(selNames)
        self.showSelectedPatchItems(selNames)

    #
    #  unselectPatches
    #------------------
    def unselectPatches(self):
        """ 全patchを非選択に設定する"""
        self.treeListPatch.unselectAllItems()
        selNames = []
        self.showSelectedPatchItems(selNames)

    #
    #  changeFeatureAnglePatch
    #--------------------------
    def changeFeatureAnglePatch(self):
        self.setEditFlag()

    #
    #  createDictPatch
    #------------------
    def createDictPatch(self):
        """ addLayersのdictを作成する"""
        #設定dataを保存
        self.saveSnappyCsv()
        #snappyDictをチェック
        copyFiles = self.checkSnappyDictForAddLayers()
        #layersCont作成
        layersCont = self.getLayerSettingPatch()
        #layersCont置き換え
        self.replaceLayersContPatch(layersCont)
        #parallel？
        if self.check_parallelPatch.get_active() == True:
            #並列処理
            caseDir = self.caseDir
            region = self.region
            nProcs = self.entry_nProcsPatch.get_text()
            nSeparate = self.entry_separatePatch.get_text()
            error = self.nProcsCheck(nProcs, nSeparate)
            if error != "":
                return
            #decomposeParDictを修正
            remakeDecomposeParDict(caseDir, region, nProcs, nSeparate)

        title = _("addLayerDict作成")
        mess = ""
        if len(copyFiles) > 0:
            files = ", ".join(copyFiles)
            mess += files + "\n"
            mess += _("  ファイルが存在しなかった為、コピーしました。\n\n")
        mess += _("snappyHexMeshDict\n")
        mess += _("  ファイルを作成しました。")
        print(mess)
        self.okDialog(title, mess)

    #
    #  checkSnappyDictForAddLayers
    def checkSnappyDictForAddLayers(self):
        """ snappyDict有無をチェックする"""
        
        def checkFileName(sourceDir, fileName):
            """ file有無をチェックし無ければコピーする"""
            path = self.caseDir + "/system/" + self.region
            if os.path.exists(path + "/" + fileName) == False:
                copyFile = sourceDir + "/system/" + fileName
                pasteDir = path
                shutil.copy(copyFile, pasteDir)
                copyFiles.append(fileName)

        copyFiles = []
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        getFoamContents = pyTreeFoam.getFoamContents(self.caseDir)
        sourceDir = getFoamContents.getOFversionFolder(OFversion)
        checkFileName(sourceDir, "snappyHexMeshDict")
        checkFileName(sourceDir, "decomposeParDict")
        checkFileName(sourceDir, "meshQualityDict")
        return copyFiles

    #
    #  getLayerSettingPatch
    def getLayerSettingPatch(self):
        """ layerSettingを取得する"""
        cont = ""
        for patchCont in self.patchConts:
            layerFlag = patchCont[1]
            if layerFlag == True:
                patchName = patchCont[0]
                nLayers = patchCont[2]
                expRatio = patchCont[3]
                minThick = "0.1"
                finalThick = "0.3"
                cont += " "*8 + patchName + "\n"
                cont += " "*8 + "{\n"
                cont += " "*8 + "    nSurfaceLayers " + nLayers + ";\n"
                cont += " "*8 + "    finalLayerThickness " + finalThick + ";\n"
                cont += " "*8 + "    minThickness " + minThick + ";\n"
                cont += " "*8 + "    expansionRatio " + expRatio + ";\n"
                cont += " "*8 + "}\n\n"
        layersCont =  "layers\n"
        layersCont += "    {\n"
        layersCont += cont
        layersCont += "    }"
        return layersCont

    #
    #  replaceLayersContPatch
    def replaceLayersContPatch(self, layersCont):
        """ snappyDict内のlayersを置き換える"""
        snappyDict = self.caseDir + "/system/" + self.region + "/snappyHexMeshDict"
        f = open(snappyDict); cont = f.read(); f.close()
        contOp = pyTreeFoam.strOp(cont)
        #snappy内のstepRunを書き換え
        newLine = "castellatedMesh false;"
        newCont, p, keyword = contOp.replace1lineKeyword(0, newLine)
        newLine = "snap false;"
        newCont, p, keyword = contOp.replace1lineKeyword(0, newLine)
        newLine = "addLayers true;"
        newCont, p, keyword = contOp.replace1lineKeyword(0, newLine)
        #layersContに置き換え        
        p = 0
        while p < len(cont):
            ps = p
            lineCont, p, kind = contOp.get1line(p)
            lineContOp = pyTreeFoam.strOp(lineCont)
            keyword, pp = lineContOp.getKeyword(0)
            if keyword == "addLayersControls":
                break
        newCont, p = lineContOp.getMiddlePair(0)
        newContOp = pyTreeFoam.strOp(newCont)
        newCont, p, keyword = newContOp.replace1lineKeyword(0, layersCont)
        #featureAngleを置き換え
        featureAngle = self.entry_featureAnglePatch.get_text()
        newLine = "featureAngle " + featureAngle + ";"
        newCont, p, keyword = newContOp.replace1lineKeyword(0, newLine)
        #addLayersControls全体を置き換え
        addLayersCont  = "addLayersControls\n"
        addLayersCont += "{"
        addLayersCont += newCont
        addLayersCont += "}"
        newDict, p, keyword = contOp.replace1lineKeyword(0, addLayersCont)
        if keyword == "":
            print(_("ERROR: snappyHexMeshが修正できません。"))
            return
        #geometryの内容をクリア
        newLine  = "geometry\n"
        newLine += "{\n"
        newLine += "}"
        newDict, p, keyword = contOp.replace1lineKeyword(0, newLine)
        #newDictを保存
        f = open(snappyDict, "w"); f.write(newDict); f.close()
        print(_("snappyHexMeshDictを修正しました。"))
        return

    #
    #  createMeshAddLayers
    #----------------------
    def createMeshAddLayers(self):
        """ snappyHexMeshを実行して、layerを追加する"""
        envOpenFoam = self.configDict["bashrcFOAM"]
        cont  = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        if self.check_parallelPatch.get_active() == True:
            #並列処理
            caseDir = self.caseDir
            region = self.region
            nProcs = self.entry_nProcsPatch.get_text()
            nSeparate = self.entry_separatePatch.get_text()
            error = self.nProcsCheck(nProcs, nSeparate)
            if error != "":
                return
            #remakeDecomposeParDict(caseDir, region, nProcs, nSeparate)
            #cont += "rm -rf " + self.region + "/processor*\n"
            cont += "rm -rf processor*\n"
            #cont += "decomposePar -region " + self.region + " 2>&1 | tee -a solve.log\n"
            cont += createDecomposeParNoFieldsCommLine(caseDir)
            cont += "mpirun -np " + nProcs
            #cont += " snappyHexMesh -parallel -region " + self.region
            cont += createSnappyAddLayerCommLine(self.region) + " -parallel "
            cont += " 2>&1 | tee -a solve.log\n"
            cont += createReconstructMeshCommLine(caseDir, region)
            #cont += "rm -rf " + self.region + "/processor*\n"
            cont += "rm -rf processor*\n"
            #cont += os.getenv("TreeFoamPath") + "/python/paraSnappyPost.py " + self.caseDir + "\n"
        else:
            #シングルコア
            #cont += "snappyHexMesh -region " + self.region + " 2>&1 | tee -a solve.log\n"
            cont += createSnappyAddLayerCommLine(self.region) + " 2>&1 | tee -a solve.log\n"
        #実行ファイル作成
        fileName = self.caseDir + "/run"
        f = open(fileName, "w"); f.write(cont); f.close()
        run = pyTreeFoam.run(self.caseDir)
        run.command("chmod a+x " + fileName)
        run.foamTerminal("./run")

    #
    #  editDictPatch
    #----------------
    def editDictPatch(self):
        """ addLayerのdictを編集する。"""
        fileName = self.caseDir + "/system/" + self.region + "/snappyHexMeshDict"
        files = [fileName]
        run = pyTreeFoam.run(self.caseDir)
        run.editor(files)

    #--------------------
    #  noteBookの処理
    #--------------------
    #
    #  showNoteBookPage
    #------------------
    def showNoteBookPage(self, pageName):
        """ pageNameを表示する"""
        global pageDict

        def isReadActor():
            """ actorを既に読み込んでいるかどうか確認する"""
            patchNames = []
            if self.region in self.actorRegDict.keys():
                patchNames = list(self.actorRegDict[self.region].keys())
                patchNames.sort()
            else:
                #読み込んでいない場合
                return False
            names = readRegionPatchNames(self.caseDir, self.region)
            names.sort()
            #actorのpatch名とregionのpatch名が合致しているか？
            return patchNames == names

        if pageName in pageDict.keys():
            page = pageDict[pageName]
            #pageを表示
            self.noteBook.set_current_page(page)
            #page1のvtkを表示
            if page == 0:
                self.setVtkModel()
                self.setCheckBox()
            elif page == 1:
                #既にactorを読み込んでいるか？
                readFlag = isReadActor()
                if readFlag == False:
                    #addLayersの初期化
                    self.checkRegionsPatch()        #region有無をcheck
                    self.setDataToEntryPatch()
                    self.checkAddLayersCsvFile()    #csvFile有無確認
                self.getAddLayersConts()        #csvFileのデータを取得
                self.createTreeListPatch()      #treeListPatchを作成
                self.setDataToTreeListPatch()   #treeListPatchにデータセット
                #vtk画面を設定
                self.setVtkModelPatch()
                self.setCheckBoxForAddlayer()
                if readFlag == False:
                    self.fitWindow()

    #
    #  clearVtkModel
    def clearVtkModel(self):
        """ vtk画面をクリアする"""
        self.maskEvent = True
        self.check_showEdge.set_active(False)
        self.check_showOutline.set_active(False)
        self.check_showLocMesh.set_active(False)
        self.check_showBlockMesh.set_active(False)
        self.check_showOrigin.set_active(False)
        self.unselectStls()
        self.unselectPatches()
        self.maskEvent = False
        self.applyCheckContsToVtk()

    #
    #  applyCheckBoxContsToVtk
    def applyCheckContsToVtk(self):
        """ checkBoxの内容をvtkに適用する"""
        pageName = self.getCurrPageName()
        if pageName == "createMesh":
            selNames = self.getSelectedStls()
            self.showSelectedItems(selNames)
        elif pageName == "addLayer":
            selNames = self.getSelectedPatches()
            self.showSelectedPatchItems(selNames)

    #
    #  setCheckBoxForAddlayer
    def setCheckBoxForAddlayer(self):
        """ addLayer用の設定"""
        self.maskEvent = True
        self.check_showEdge.set_active(True)
        self.check_showOutline.set_active(True)
        self.check_showLocMesh.set_active(False)
        self.check_showBlockMesh.set_active(False)
        self.check_showOrigin.set_active(False)
        self.maskEvent = False
        self.applyCheckContsToVtk()

    #
    #  showCreateMeshPage
    #----------------------
    def showCreateMeshPage(self):
        """ メッシュ作成用のvtk、pageを表示させる"""
        if self.toggle_createMesh.props.active == True:
            #vtkをクリア
            self.clearVtkModel()
            #他のswをOFFする
            self.toggle_addLayer.set_active(False)
            self.showNoteBookPage("createMesh")
            #checkBox設定
            self.check_showBlockMesh.set_sensitive(True)
            self.check_showLocMesh.set_sensitive(True)
            self.button_openStlDir.set_sensitive(True)
            self.button_stlDirRef.set_sensitive(True)
            self.entry_stlDir.set_sensitive(True)
        else:
            #反転させる
            self.toggle_addLayer.set_active(True)

    #
    #  showAddLayerPage
    #-------------------
    def showAddLayerPage(self):
        """ layer追加用のvtk、pageを表示させる"""
            #他のswをOFFする
        if self.toggle_addLayer.props.active == True:
            #vtkをクリア
            self.clearVtkModel()
            #他のSWをOFFする
            self.toggle_createMesh.set_active(False)
            self.showNoteBookPage("addLayer")
            #checkBox設定
            self.check_showBlockMesh.set_sensitive(False)
            self.check_showLocMesh.set_sensitive(False)
            self.button_openStlDir.set_sensitive(False)
            self.button_stlDirRef.set_sensitive(False)
            self.entry_stlDir.set_sensitive(False)
        else:
            #反転させる
            self.toggle_createMesh.set_active(True)

    #
    #  getCurrPageName
    def getCurrPageName(self):
        """ noteBookの表示page名を返す"""
        global pageDict
        page = self.noteBook.get_current_page()
        pageName = ""
        for key in pageDict.keys():
            if pageDict[key] == page:
                pageName = key
                break
        return pageName

    #----- universalDialog -----------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def inputTextDialog(self, title, message, entryText, funcOk=[]):
        dialog = inputTextDDialog.getInputText(
            title, message, entryText, funcOk=funcOk, 
            parent=self.mainWindow)
        inputText = dialog.show()
        return inputText
    
    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()

#
#  getStlFiles
#--------------
def getStlFiles(stlDir):
    """ stlFileNameを取得して返す"""
    stlFiles = glob.glob(stlDir + "/*.stl")
    stlNames = []
    for stlFile in stlFiles:
        if os.path.isfile(stlFile) == True:
            name = os.path.basename(stlFile)
            stlName = ".".join(name.split(".")[:-1])
            if stlName[0] != "_":
                stlNames.append(stlName)
    stlNames.sort()
    return stlNames

#
#  readRegionPatchNames
#-----------------------
def readRegionPatchNames(caseDir, region):
    """ region内のpatchNamesを取得して返す"""
    case = pyTreeFoam.case(caseDir)
    currTime = case.getCurrTimeFolder()
    relMeshDir = case.getCurrMeshDir(currTime, region, "boundary")
    meshDir = caseDir + "/" + relMeshDir
    conts = case.getPatchNameFromBoundary(meshDir)
    patchNames = list(map(lambda x: x[0], conts))
    patchNames.sort()
    return patchNames

#
#  readCreateMeshCsvData
#------------------------
def readCreateMeshCsvData(fileName):
    """ createMeshのcsvDataを読み込み返す"""
    if os.path.exists(fileName) == False:
        block_cellSize = "0"
        block_overBlockSize = "0"
        locationInMesh = ["0", "0", "0"]
        stlConts = []
    else:
        csvRowCol = pyTreeFoam.readCsvData(fileName)
        block_cellSize = csvRowCol[2][2]
        block_overBlockSize = csvRowCol[3][2]
        locationInMesh = csvRowCol[6][2:5]
        stlConts = getStlConts(csvRowCol)
    ans = (block_cellSize, block_overBlockSize, 
           locationInMesh, stlConts)
    return ans

#
#  getStlConts
def getStlConts(csvRowCol):
    """ csvRowCol内容からstlに関するdataを取得する
    csvFileのcol(8個)
        stl, sect, edge, cell, face, note, layer, expan, slave
    stlContsのcol(6個)
        stl, sect, edge, cell, layer, expan, slave"""
    #取得するcolNoを取得
    csv = csvColDict
    getNos = (
        csv["stl"],  csv["sect"],  csv["edge"], csv["cell"],
        csv["layer"], csv["expan"], csv["slave"]
        )
    stlConts = []
    rowColData = csvRowCol[8:]
    for rowData in rowColData:
        data = []
        for getNo in getNos:
            try:
                val = rowData[getNo]
            except:
                val = ""
            data.append(val)
        stlConts.append(data)
    return stlConts

#
#  readAddLayersCsvData
#----------------------
def readAddLayersCsvData(fileName):
    """ addLayerのcsvDataを読み込み返す
    region, patchContsを返す"""
    conts = pyTreeFoam.readCsvData(fileName)
    region = conts[1][1]
    if region == "region0" or region == "(region0)":
        region = "."
    patchConts = []
    for i in range(4, len(conts), 1):
        patchName = conts[i][0] 
        nLayers = conts[i][1]
        expRatio = conts[i][2]
        if nLayers != "" and expRatio != "":
            addLayers = True
        else:
            addLayers = False
            nLayers = ""
            expRatio = ""
        row = [patchName, addLayers, nLayers, expRatio]
        patchConts.append(row)
    return region, patchConts

#
#  getStlFilesCont
#------------------
def getStlFilesCont(stlFiles):
    """ stlFileの内容（ascii or binary, size etc.)を取得する
    
    Args:
        stlFiles (list) :確認するstlファイル(fullPath)
    Returns:
        stlファイル毎に以下のlistを返す
        stlName (str)           :stlFileのbaseName
        fileFmts (str)          :"ascii" or "bin"
        solidNames (str)        :solid名
        stlSizes (list(float))  :stlファイルのサイズ[x,y,z]
        minLoc (list(float)     :stlファイルのmin座標"""
    stlConts = []
    for stlFile in stlFiles:
        stlCont = []
        #stlNameを取得
        stlName = stlFile.split(os.sep)[-1]
        stlCont.append(stlName)
        #solidName取得
        if pyTreeFoam.isBinaryStl(stlFile) == False:
            stlCont.append("ascii")
            (solidName, xyzVals) = pyTreeFoam.getStlContFromAscii(stlFile)
        else:
            stlCont.append("bin")
            (solidName, xyzVals) = pyTreeFoam.getStlContFromBin(stlFile)
        stlCont.append(solidName)
        #stlSizeを取得
        (xVals, yVals, zVals) = xyzVals
        if len(xVals) != 0:
            maxx = max(xVals); maxy = max(yVals); maxz = max(zVals)
            minx = min(xVals); miny = min(yVals); minz = min(zVals)
            dx = maxx - minx
            dy = maxy - miny
            dz = maxz - minz
        else:
            dx, dy, dz = 0.0, 0.0, 0.0
        #値を丸める
        dx = pyTreeFoam.getValue3digit_shrink(dx)
        dy = pyTreeFoam.getValue3digit_shrink(dy)
        dz = pyTreeFoam.getValue3digit_shrink(dz)
        stlCont.append([dx, dy, dz])
        #minLocを取得
        minxs = pyTreeFoam.getValue3digit_shrink(minx)
        minys = pyTreeFoam.getValue3digit_shrink(miny)
        minzs = pyTreeFoam.getValue3digit_shrink(minz)
        minLoc = [minxs, minys, minzs]
        stlCont.append(minLoc)
        stlConts.append(stlCont)
    stlConts.sort()
    return stlConts

#
#  getFeatureEdgeDictName
#-------------------------
def getFeatureEdgeDictName(OFversion):
    """ OFversionに応じたfeatureEdgeDict名を取得する"""
    ver = pyTreeFoam.getNumVersion(OFversion)
    if ver >= 8.0:
        #OF-8以上
        dictName = "surfaceFeaturesDict"
    else:
        #OF-8未満、OF-V、OF-ext
        dictName = "surfaceFeatureExtractDict"
    return dictName

#
#  makeNewFeatureDict_8
#-----------------------
def makeNewFeatureDict_8(caseDir, fileName, selFiles, angle):
    """ ver 8以降。
    デフォルトのsurfaceFeatureExtractDictを新しく作成する"""
    getFoamContents = pyTreeFoam.getFoamContents(caseDir)
    contents = getFoamContents.defaultSurfaceFeaturesDict()
    contentsOp = pyTreeFoam.strOp(contents)
    pairCont, pp = contentsOp.getMiddlePair(0)
    pe = pp - 1
    ps = pe - len(pairCont)
    header = contents[:ps]
    footer = contents[pe:]
    pairContOp = pyTreeFoam.strOp(pairCont)
    #stlFileをセット
    newLine = "surfaces\n"
    newLine += "    (\n"
    stlLine = ""
    for stlFile in selFiles:
        stlLine += '        "' + stlFile + '"\n'
    newLine += stlLine
    newLine += "    );"
    pairCont, _pp, keyword = pairContOp.replace1lineKeyword(0, newLine)
    #angleをセット
    newLine = "includedAngle  " + angle + ";"
    pairCont, _pp, keyword = pairContOp.replace1lineKeyword(0, newLine)
    contents = header + pairCont + footer
    #全内容を作成
    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    header = getFoamContents.makeFoamHeaderVersion(OFversion)
    foamFile = getFoamContents.makeFoamFile("2.0", "ascii", "dictionary",
                            "", "surfaceFeaturesDict")
    footer = getFoamContents.makeFoamFooter()
    contents = header + foamFile + contents + footer
    f = open(fileName, "w"); f.write(contents); f.close()

#
#  makeNewFeatureDict_7
#----------------------
def makeNewFeatureDict_7(caseDir, fileName, selFiles, angle):
    """ ver 7 以前、OF-v、OF-ext用。
    デフォルトのsurfaceFeatureExtractDictを新しく作成する"""
    contents = ""
    i = 0
    getFoamContents = pyTreeFoam.getFoamContents(caseDir)
    for selFile in selFiles:
        newLineCont = getDefaultFeatureDict(caseDir, selFile, angle)
        if i > 0:
            #コメント文を削除
            newLineCont = pyTreeFoam.strOp(newLineCont).deleteCommentsNullLines()
        contents += newLineCont
        i += 1
    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    header = getFoamContents.makeFoamHeaderVersion(OFversion)
    foamFile = getFoamContents.makeFoamFile("2.0", "ascii", "dictionary",
                            "", "surfaceFeatureExtractDict")
    footer = getFoamContents.makeFoamFooter()
    contents = header + foamFile + contents + footer
    f = open(fileName, "w"); f.write(contents); f.close()

#
#  getDefaultFeatureDict
def getDefaultFeatureDict(caseDir, stlFile, angle):
    """ OFver7以前、defaultのFeatureDictにstlファイル名と角度を設定して返す"""
    #defaultの内容を取得
    getFoamContents = pyTreeFoam.getFoamContents(caseDir)
    cont = getFoamContents.defaultSurfaceFeatureExtractDict()
    #stlファイル名をセット
    contOp = pyTreeFoam.strOp(cont)
    (_keyword, p) = contOp.getKeyword(0)
    cont = stlFile + cont[p:]
    #角度をセット
    contOp = pyTreeFoam.strOp(cont)
    p = contOp.skipIntoPair(0)              #ポインタを{}内に進める
    (lineCont, p, kind) = contOp.get1line(p)
    while lineCont != "":
        if kind == "keyword":
            lineContOp = pyTreeFoam.strOp(lineCont)
            (keyword, _pp) = lineContOp.getKeyword(0)
            if keyword == "extractFromSurfaceCoeffs":
                ps = p - len(lineCont)      #ポインタをkeywordに戻す
                header = cont[:ps]
                footer = cont[p:]
                #角度をセット
                ps = lineContOp.skipIntoPair(0)
                newLine = "includedAngle  " + angle + ";"
                (lineCont, _pp, keyword) = lineContOp.replace1lineKeyword(ps, newLine)
                break
        (lineCont, p, kind) = contOp.get1line(p)
    newCont = header + lineCont + footer
    return newCont

#
#  remakeDecomposeParDict
#-------------------------
def remakeDecomposeParDict(caseDir, region, nProcs, nSeparate):
    """ decomposeParDictを修正する.
    methodは、simple固定"""

    def getInsertPointFromEnd(cont):
        """ 挿入場所を検索する。挿入場所は、footerの上"""
        #footerを検索
        n = cont.find("*********")
        if n < 0:
            #footerが存在しない場合
            ip = len(cont)
            return ip
        #位置「n」から遡って、「\n」を検索
        ip = len(cont)
        i = n
        while i >= 0:
            i -= 1
            if cont[i] == "\n":
                ip = i
                break
        return ip

    fileName = caseDir + "/system/" + region + "/decomposeParDict"
    f = open(fileName); cont = f.read(); f.close()
    contOp = pyTreeFoam.strOp(cont)
    ps = contOp.skipFoamFile()
    #numberOfSubdomainsを書き換え
    newLine = "numberOfSubdomains  " + nProcs + ";"
    (cont, _p, _keyword) = contOp.replace1lineKeyword(ps, newLine)
    #methodを書き換え
    newLine = "method   simple;"
    (cont, _p, _keyword) = contOp.replace1lineKeyword(ps, newLine)
    #simpleCoffesを書き換え
    newLine  = "simpleCoeffs\n"
    newLine += "{\n"
    newLine += "    n    (" + nSeparate + ");\n"
    newLine += "    delta 0.001;\n"
    newLine += "}"
    (retCont, _p, keyword) = contOp.replace1lineKeyword(ps, newLine)
    if keyword == "":
        #最後に追加する
        p = getInsertPointFromEnd(cont)
        cont = cont[:p] + newLine + "\n" + cont[p:]
    else:
        cont = retCont
    #書き込み
    f = open(fileName, "w"); f.write(cont); f.close()

#
#  createReconstructMeshCommLine
#---------------------------------
def createReconstructMeshCommLine(caseDir, region):
    """ reconstructParMeshコマンド行を作成する。
    OF-9からコマンド変更"""

    def createCommLine_Org():
        contDict = pyTreeFoam.case(caseDir).getControlDict()
        pre = contDict["writePrecision"]
        line = "reconstructParMesh -latestTime -cellDist -mergeTol 1e-" + pre + " -region " + region + "\n"
        return line

    def createCommLine_9():
        line = "reconstructParMesh -latestTime -cellDist -region " + region + "\n"
        return line

    def createCommLine_11():
        line = "reconstructPar -latestTime -region " + region + "\n"
        return line

    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if OFversion >= "v1804":
        line = createCommLine_Org()
    elif OFversion[:3] == "ext":
        line = createCommLine_Org()
    else:
        if numVer >= 11.0:
            #OF-11以上
            line = createCommLine_11()
        elif numVer >= 9.0:
            #OF-9以上
            line = createCommLine_9()
        else:
            #OF-9未満
            line = createCommLine_Org()
    return line

#
#  createDecomposeParNoFieldsCommLine
#-------------------------------------
def createDecomposeParNoFieldsCommLine(caseDir):
    """ decomposePar -noFields コマンド行を作成する
    メッシュの分割のみでfieldは、分割しない"""

    def createCommLine_Org():
        line  = os.getenv("TreeFoamPath") + "/python/meetPatchNameDeleteNullPatch.py " + caseDir + "\n"
        line += "decomposePar 2>&1 | tee -a solve.log\n"
        return line

    def createCommLine_v():
        line = "decomposePar -no-fields 2>&1 | tee -a solve.log\n"
        return line
    
    def createCommLine_7():
        line = "decomposePar -noFields 2>&1 | tee -a solve.log\n"
        return line

    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if OFversion >= "v2206":
        line = createCommLine_v()
    elif OFversion[:3] == "ext":
        line = createCommLine_Org()
    else:
        if numVer >= 7:
            #OF-7以上
            line = createCommLine_7()
        else:
            #OF-7以下、v2206未満
            line = createCommLine_Org()
    return line

#
#  createSnappyParallelCommLine
#--------------------------------
def createSnappyParallelCommLine(nProcs):
    """ 並列計算用のsnappyHexMeshのcommLineを作成する"""
    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer >= 13.0:
        #noOverwriteを追加
        line = "mpirun -np " + nProcs + " snappyHexMesh -noOverwrite -parallel 2>&1 | tee -a solve.log\n"
    else:
        line = "mpirun -np " + nProcs + " snappyHexMesh -parallel 2>&1 | tee -a solve.log\n"
    return line

#
#  createSnappyAddLayerCommLine
#--------------------------------
def createSnappyAddLayerCommLine(region):
    """ addLayer用のsnappyHexMeshのcommLineを作成する"""
    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer >= 13.0:
        #noOverwriteを追加
        line = " snappyHexMesh -noOverwrite -region " + region
    else:
        line = " snappyHexMesh -region " + region
    return line

if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    #stlDir = sys.argv[2]
    #csvFileName = sys.argv[3]
    #caseDir = os.getenv("HOME") + "/vtkTest/normalMesh_copy0"
    #caseDir = "/mnt/hgfs/share/CAE/OpenFOAM/OF-7.0/taiyo/4_tanbayama/05_tabayamaMesh4_windTunnel_ESE/1_tanbayamaMesh_2_sqHouse"
    #stlDir = caseDir + "/model"
    winApp = windowApp(caseDir)
    winApp.main()
