#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#   editFoamFile.py
#
#       foamFile(field)をeditorで編集する
#       fileは、gzip、binaryでも編集できる
#
#   <使い方>
#       editFoamFile.py [option] <file name> ...
#       [option]
#           -t      temporary folderを指定
#           -n      データ部の表示行数の設定
#           -h      help
#
#   13/09/17    新規作成
#      09/28    optionを追加
#      09/30    バグ修正
#      10/07    file名のチェックを追加
#      10/13    file名チェックのバグ修正
#      10/15    国際化のため修正
#      11/13    dir違いの同名fileが同時にopenできなかったのを修正
#               folderを作成し、tempFile作成
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/06/23    helpが「--help」でも表示する様に修正
#   15/07/13    gridEditor_dataの保存場所変更
#   17/08/09    getNMaxLines:gridEditor_dataが存在しない場合エラー停止する為、修正
#   19/05/06    setEditTempFile:mkdirをos.mkdirに変更
#      11/24    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#

import sys
import os
import glob
import shutil

import pyTreeFoam

import logFileCreater; logFileCreater.log()


#
#  表示行数の読込
def getNMaxLines():
    """ gridEditorDataから表示行数を読み込む"""
    setFile = os.getenv("TreeFoamUserPath") + "/data/gridEDitor_data"
    if len(glob.glob(setFile)) == 0:
        nMaxLines = 20
        return nMaxLines

    f=open(setFile); cont=f.readlines(); f.close()
    for line in cont:
        words = line.split()
        if len(words) >= 2:
            if words[0] == "nMaxLinesBinToAscii":
                nMaxLines = int(words[1])
    return nMaxLines

#
#  setEditTempFile
def setEditTempFile(tempFolder, files, nMaxLines):
    """ 編集するfileのtempFile名とtimeStampを取得"""
    #編集するfileのtempFileの準備
    openFiles = []
    times = []
    error = 0
    for fileNameC in files:
        folNo = 0
        fileName = pyTreeFoam.case().checkFileName(fileNameC)
        if len(glob.glob(fileName)) != 0:
            cont = pyTreeFoam.foamFile(nMaxLines).read(fileName)
            name = pyTreeFoam.case().getOrgFileName(fileName).split("/")[-1]
            tempFile = tempFolder + "/" + name + "." + str(folNo) + "/" + name
            while len(glob.glob(tempFile)) != 0:
                folNo += 1
                tempFile = tempFolder + "/" + name + "." + str(folNo) + "/" + name
            os.mkdir(tempFolder + "/" + name + "." + str(folNo))
            f = open(tempFile, "w"); f.write(cont); f.close()
            openFiles.append(tempFile)
            #timeスタンプを保存
            times.append(os.path.getmtime(tempFile))
        else:
            error = 1
            print("\nerror: ", fileName, _(u"が開けません"))
            break
    return [error, openFiles, times]

#
#  saveEditTempFiles
def saveEditTempFiles(files, tempFiles, times):
    """ 編集内容を元のfileに戻す。
    editorで保存しなかった場合（当初のtimeStampのまま）、元のfileに戻さない"""
    i = 0
    for fileName in files:
        #file名を非圧縮のfile名に戻す
        sourceName = pyTreeFoam.case().getOrgFileName(fileName)
        if len(glob.glob(tempFiles[i])) != 0:
            #timeStampを取得
            newTime = os.path.getmtime(tempFiles[i])
            #書き換えていない場合は、保存しない
            if times[i] != newTime:
                f = open(tempFiles[i]); cont = f.read(); f.close()
                pyTreeFoam.foamFile().write(sourceName, cont)
        i += 1

#
#  deleteTempFiles
def deleteTempFiles(tempFiles):
    """ tempFileを削除する。"""
    for fileName in tempFiles:
        if len(glob.glob(fileName)) != 0:
            delFolder = "/".join(fileName.split("/")[:-1])
            shutil.rmtree(delFolder)

#
#  checkFiles
def checkFiles(files):
    """ file名のチェック"""
    newFiles = []
    for name in files:
        nameT = pyTreeFoam.case().checkFileName(name)
        if os.path.isfile(nameT) == True:
            newFiles.append(name)
    return newFiles

#  printHelp
def printHelp():
    """ helpを出力"""
    mess  = _(u"\n----- editFoamFile.py の使い方------------------------------\n")
    mess += _(u"OpenFOAMのファイルを編集する。\n")
    mess += _(u"ファイルが、gzip、binaryでもeditorで編集できる。\n")
    mess += _(u"データ部の行数を省略して表示するのでeditorの動作が軽くなるが\n")
    mess += _(u"データ部の編集はできない。\n\n")
    mess += _(u"使い方\n")
    mess += _(u"editFoamFile.py [option] <file0> <file1>...\n")
    mess += _(u"[option]\n")
    mess += _(u"  -t       temporary folderの指定\n")
    mess += _(u"           デフォルトは、「$TreeFoamPath/temp」\n")
    mess += _(u"  -n       データ部の表示行数を設定\n")
    mess += _(u"           デフォルトは、「$TreeFoamUserPath/data/griEditor_data」内の\n")
    mess += _(u"           nMaxLinesBinToAsciiの設定による。\n")
    mess += _(u"           全て表示させる場合は、表示行数を「-1」に設定。\n")
    mess += _(u"  -h       helpの表示\n\n")
    print (mess)


#------------------------
#  main
#------------------------

#言語の設定
import gettext
treeFoamPath = os.getenv("TreeFoamPath")
gettext.install("treefoam", treeFoamPath + "/data/locale")
#_ = gettext.gettext
if len(sys.argv) <= 1:
    print(_(u"\nerror:編集するfile名が入力されていません。\n"))
    printHelp()

else:
    endFlag = "no"
    files = []
    tempFolder = ""
    nMaxLines = 0
    nMaxFlag = 0
    i = 1
    while i<len(sys.argv):
        if sys.argv[i] == "-t":
            i += 1
            if i < len(sys.argv): 
                tempFolder = sys.argv[i]
            i += 1
        elif sys.argv[i] == "-n":
            i += 1
            if i<len(sys.argv):
                nMaxLines = int(sys.argv[i])
                nMaxFlag = 1
            i += 1
        elif sys.argv[i] == "-h" or sys.argv[i] == "--help":
            printHelp()
            i = len(sys.argv)
            endFlag = "end"
        else:
            files.append(sys.argv[i])
            i += 1

    if endFlag == "end":
        exit()
    if tempFolder == "":
        tempFolder = os.getenv("TreeFoamUserPath") + "/temp"
    if len(files) == 0:
        print(_(u"\nerror:編集するfile名が入力されていません。\n"))
        printHelp()
        exit()
    else:
        files = checkFiles(files)
        if len(files) == 0:
            print( _(u"\nerror:編集するfile名が入力されていません。\n"))
            printHelp()
            exit()

    #表示行数の設定
    if nMaxFlag == 0:
        nMaxLines = getNMaxLines()

    #編集するfileのtempFile名とtimeStampの取得
    (error, tempFiles, timeStamps) = setEditTempFile(tempFolder, files, nMaxLines)
    if error == 1:
        printHelp()
        exit()

    #編集ファイルをeditorでopenする
    pyTreeFoam.run().editorWait(tempFiles)

    #editorClose後、tempFileの内容をfileに書込
    saveEditTempFiles(files, tempFiles, timeStamps)

    #tempFileを削除
    deleteTempFiles(tempFiles)
