#!/usr/bin/python3
#  coding: utf-8
#
#           editCfMeshDialog.py
#
#       cfMeshDictを作成し、meshを作成する。
#
#   24/11/29    新規作成
#      12/04    release
#   25/01/29    setDefaultValueForCellSize:cellSize変更時の初期値設定を追加
#      01/30    reloadAtDefault,setDefaultValue:okCancelDialogを表示して継続有無確認を追加
#               setDefaultValueForCellSize:cellSize変更時、各行の設定済cellSizeに近い値に変更する。
#      02/07    menuCheckMesh,runCheckMesh:checkMeshのメニューを追加。
#      03/30    setVtkModel:vtk-9.4対応でGradientBackgroundOnを追加
#      03/31    createMesh:メッシュ作成後、boundaryFieldの整合性を取ることの有無を追加。
#               setWindowSize:新規追加。（endingでwindowSizeを保存し、initializeでwindowSizeを設定）
#

import logFileCreater
logFileCreater.log()

import os, sys
sys.stdout.flush()
import checkVtkModule
isVtk = checkVtkModule.isVtk("editCfMesh")
if isVtk == True:
    import vtk
    from vtk.util import colors
    vtkVersion = vtk.vtkVersion.GetVTKSourceVersion().split()[-1]
else:
    print("error: could not import vtk !")
    exit()

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GdkPixbuf, GLib
import glob
import shutil
import math

import pyTreeFoam
import universalDialogs as unvDlg
import inputTextDDialog
import GtkParts
import GtkVtkRenderWindowInteractor as gtkVtk
import createBlockAndSnappyDict
import saveGoDDialog
import meshViewerControl
import getOpenFolderDDialog
import createCfMeshDict

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

#csvFileのcolumn
csvColDict = {
    "stl":0,   "sect":1, "cell":2, 
    "nLayer":3,  "expan":4, "maxTh":5, 
    "note":6
    }

#treeListのcolumn
treeListColDict = {
    "stl":0,      "type":1,   "cell":2, 
    "addLayer":3, "nLayer":4, "expan":5, 
    "maxTh":6
    }


#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"editCfMeshDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を取得
        self.setGtkObject()
        #変数の定義
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        self.windowTitle = ""
        self.editFlag = "no"
        self.newCsvFlag = 0
        self.caseDir = caseDir
        self.stlDir = self.getStlDir()
        self.cfMeshCsvName = "cfMeshDict.csv"
        self.csvRowCol = []
        self.maxCellSize = "0"
        self.featureAngle = "30.0"
        self.stlConts = []
        self.parallelFlag = "no"
        #vtk関連
        self.maskEvent = True
        self.vtkSize = (300, 250)
        self.vtkInitialRun = True           #最初のvtk表示flag
        self.vtkObj = gtkVtk.GtkVtkRenderWindowInteractor(self)
        self.actorRegDict = {}
        self.outlineActorRegDict = {}
        self.readerDict = {}
        self.actorDict = {}
        self.outlineActorDict = {}
        self.renderer = ""
        self.originAxes = None
        self.allElms = []
        self.hideElms = []
        self.showElms = []
        self.currElms = []
        self.selectNos = 0
        #初期化
        self.vtkInitialize()
        self.initialize()


    def main(self):
        """ GUIの表示"""
        self.setVtkArea()           #vtkAreaをセット
        self.mainWindow.show_all()
        self.setVtkModel()          #vtkを表示
        self.treeList.treeList.grab_focus()  #treeViewにfocus（keyEventを効かす為）
        self.maskEvent = False
        self.clearEditFlag()        #修正flagをクリア
        if self.newCsvFlag != 0:
            GLib.timeout_add(100, self.openInitialMessageDialog)
        Gtk.main()

    def openInitialMessageDialog(self):
        if self.newCsvFlag == 1:
            title  = _("cfMeshCsvファイルの読み込み") + "\n"
            mess = _("cfMeshCsvファイルが存在しなかった為、新たに作成しました。")
            self.okDialog(title, mess)
        elif self.newCsvFlag == 2:
            title  = _("stlDir内の読み込み") + "\n"
            mess  = _("stlDirが存在しません。") + "\n"
            mess += _("  stlDirを変更してください。")
            self.okDialog(title, mess)
        elif self.newCsvFlag == 3:
            title  = _("cfMeshCsvファイルの読み込み") + "\n"
            mess  = _("csvDataの内容とstlFile名が合致しません。") + "\n"
            mess += _("  csvファイルをdefault値で再作成しました。")
            self.okDialog(title, mess)

    def close(self, *args):
        self.ending()
        Gtk.main_quit()

    def ending(self):
        """ 終了処理。windowsのサイズを保存する"""
        #windowSizeを取得
        width, height = self.mainWindow.get_size()
        splitWidth = self.paned_win.get_position()
        winDict = {}
        winDict["editCfMesh"] = str(width) + " " + str(height)
        winDict["editCfMesh_splitter"] = str(splitWidth)
        pyTreeFoam.writeWindowSize(winDict)

    def setGtkObject(self):
        """ glade内のobject名を取得"""
        #vtk関連
        self.box_vtk = self.builder.get_object("box_vtk")
        self.check_showEdge = self.builder.get_object("check_showEdge")
        self.check_showOutline = self.builder.get_object("check_showOutline")
        self.check_showOrigin = self.builder.get_object("check_showOrigin")
        self.radio_both = self.builder.get_object("radio_both")
        self.radio_frontFace = self.builder.get_object("radio_frontFace")
        self.radio_backFace = self.builder.get_object("radio_backFace")
        self.label_vtkMess = self.builder.get_object("label_vtkMess")
        self.button_reset = self.builder.get_object("button_reset")
        self.button_Xaxis = self.builder.get_object("button_Xaxis")
        self.button_Yaxis = self.builder.get_object("button_Yaxis")
        self.button_Zaxis = self.builder.get_object("button_Zaxis")
        self.button_rightToLeft = self.builder.get_object("button_rightToLeft")
        self.button_rolling90 = self.builder.get_object("button_rolling90")
        #右画面
        self.paned_win = self.builder.get_object("paned_win")
        self.button_openStlDir = self.builder.get_object("button_openStlDir")
        self.button_openCaseDir = self.builder.get_object("button_openCaseDir") 
        self.entry_stlDir = self.builder.get_object("entry_stlDir")
        self.entry_caseDir = self.builder.get_object("entry_caseDir")
        #cfMesh
        self.entry_maxCellSize = self.builder.get_object("entry_maxCellSize")
        self.entry_featureAngle = self.builder.get_object("entry_featureAngle")
        self.label_caseDir = self.builder.get_object("label_caseDir")
        self.label_cfMesh = self.builder.get_object("label_cfMesh")
        self.label_createMesh = self.builder.get_object("label_createMesh")
        self.entry_maxCellSize = self.builder.get_object("entry_maxCellSize")
        self.entry_featureAngle = self.builder.get_object("entry_featureAngle")
        self.button_excludeStls = self.builder.get_object("button_excludeStls")
        self.button_setBoundaryDefaule = self.builder.get_object("button_setBoundaryDefault")
        self.button_unselectStls = self.builder.get_object("button_unselectStls")
        #mesh作成
        self.entry_nThreads = self.builder.get_object("entry_nThreads")
        self.check_parallel = self.builder.get_object("check_parallel")
        treeView = self.builder.get_object("treeView")
        self.treeList = GtkParts.treeListCols(treeView)
        self.button_createDict = self.builder.get_object("button_createDict")
        self.button_createMesh = self.builder.get_object("button_createMesh")
        self.button_editDict = self.builder.get_object("button_editDict")
        self.check_adjustBnd = self.builder.get_object("check_adjustBnd")

    #
    #  getStlDir
    def getStlDir(self):
        """ stlDir, csvNameを取得する"""
        case = pyTreeFoam.case(self.caseDir)
        dataDict = case.getCreateMeshData()
        stlDir = dataDict["stlDir"]
        if stlDir[0] != "/":
            stlDir = self.caseDir + "/" + stlDir
        return stlDir

    #------------ vtk関連 --------------------
    #
    #  vtkInbitialize
    def vtkInitialize(self):
        """ vtk関連の初期化"""
        self.label_vtkMess.set_text(_("選択しているstlファイルを表示する。"))
        self.setCheckBox()
        #toolTip
        #vtk画面のtooltip
        self.check_showEdge.props.tooltip_text = _("stlのedgeを表示する")
        self.check_showOrigin.props.tooltip_text = _("原点を表示する")
        self.check_showOutline.props.tooltip_text = _("全stlファイルのoutline（feature edge）を表示する。")
        self.button_reset.props.tooltip_text = _("画面にフィットさせる（reset camera）")
        self.button_Xaxis.props.tooltip_text = _("x軸視（y-z 面）を表示")
        self.button_Yaxis.props.tooltip_text = _("y軸視（z-x 面）を表示")
        self.button_Zaxis.props.tooltip_text = _("z軸視（x-y 面）を表示")
        self.button_rightToLeft.props.tooltip_text = _("視点を反転(左右前後反転)")
        self.button_rolling90.props.tooltip_text = _("時計回りに90度回転")
        self.radio_both.props.tooltip_text = _("faceの表裏両面を表示する")
        self.radio_frontFace.props.tooltip_text = _("faceの表面のみ表示する\nfaceの方向が確認きる")
        self.radio_backFace.props.tooltip_text = _("faceの裏面のみ表示する\n隠れた内部のstlが確認できる")
        #設定画面
        self.button_openStlDir.props.tooltip_text = _("stlDirを開く")
        self.button_openCaseDir.props.tooltip_text = _("caseDirを開く")
        self.entry_stlDir.props.tooltip_text = _("stlDirをcaseDirからの相対pathで設定する。")
        #cfMesh
        self.entry_maxCellSize.props.tooltip_text = _("mesh作成時の最大cellSizeを設定")
        self.entry_featureAngle.props.tooltip_text = _("特徴線を抽出する時の角度を設定")
        mess  = _("選択しているstlファイルを表から除外する。") + "\n"
        mess += _("ファイル名を'_'付きに変更して除外する。ファイルは削除されない。")
        self.button_excludeStls.props.tooltip_text = mess
        self.button_setBoundaryDefaule.props.tooltip_text = _("表の値をdefault値に設定する")
        self.button_unselectStls.props.tooltip_text = _("表中の全stlを非選択状態に設定する")
        self.treeList.treeList.props.tooltip_text = _("変更したい項目をクリックすると、dropDownMenuを表示 or cell編集モードに移行するので、その項目が変更できる。")
        #mesh作成
        mess  = _("checkすると、thread並列数が設定できる。") + "\n"
        mess += _("checkしない場合、fullのthread数で実行する。")
        self.check_parallel.props.tooltip_text = mess
        self.entry_nThreads.props.tooltip_text = _("thread並列数を入力する。")
        self.button_createDict.props.tooltip_text = _("meshを作成する為のmeshDictを作成する。")
        self.button_createMesh.props.tooltip_text = _("meshDictに基づきmeshを作成する。")
        self.button_editDict.props.tooltip_text = _("meshDictを編集する。")

    #
    #  setCheckBox
    def setCheckBox(self):
        """ checkBoxを設定する"""
        self.check_showEdge.set_active(True)
        self.check_showOutline.set_active(True)
        self.check_showOrigin.set_active(False)

    #
    #  setVtkArea
    def setVtkArea(self):
        """ vtkAreaを設定する"""
        vtkObj = self.vtkObj
        vtkObj.set_size_request(*self.vtkSize)
        self.box_vtk.pack_start(vtkObj, True, True, 0)

    #
    #  setVtkModel
    #---------------
    def setVtkModel(self):
        """ vtkを表示する"""
        #actorを作成する
        self.createActorDict()
        #rendererをチェック
        if self.renderer == "":
            self.renderer = vtk.vtkRenderer()
        else:
            self.renderer.NewInstance()
        #assyを定義
        self.assy = vtk.vtkAssembly()
        #stlを表示（選択しているstlを取得して）
        selNames = self.getSelectedStls()
        self.showSelectedItems(selNames)

        #backgroundColorを設定
        self.renderer.SetBackground2(0.5, 0.5, 1)
        self.renderer.SetBackground(0.8, 0.8, 1)
        self.renderer.GradientBackgroundOn()
        self.renderer.SetTexturedBackground(True)
        #self.renderer.ResetCamera()
        self.renWin = self.vtkObj.GetRenderWindow()
        self.renWin.NewInstance()
        #新しいrendererを追加
        self.renWin.AddRenderer(self.renderer)
        self.addCornerAxis()
        #vtk表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.GlobalWarningDisplayOff()   #windowsのエラー対策
        self.vtkObj.Initialize()

        self.vtkObj.Start()
        return

    #
    #  createActorDict
    def createActorDict(self):
        """ actorを作成する"""
        #stlFileを取得
        allItems = self.treeList.getAllItemNames()
        stlNames = list(map(lambda x: x[0], allItems))
        stlFiles = list(map(lambda x: self.stlDir+"/"+x+".stl", stlNames))
        #全stlのactor辞書を作成
        self.actorDict = {}
        self.outlineActorDict = {}
        print("reading stl files...")
        for fileName in stlFiles:
            stlFile = os.path.basename(fileName)
            stlName = ".".join(stlFile.split(".")[:-1])
            #reader
            reader = vtk.vtkSTLReader()
            reader.SetFileName(fileName)
            reader.Update()
            self.readerDict[stlName] = reader
            #filter
            geomFilter = vtk.vtkGeometryFilter()
            geomFilter.SetInputConnection(reader.GetOutputPort())
            #featureEdge
            featureEdge = vtk.vtkFeatureEdges()
            featureEdge.SetInputConnection(geomFilter.GetOutputPort())
            #mapper
            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputConnection(reader.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(featureEdge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            self.actorDict[stlName] = actor
            edgeActor = vtk.vtkActor()
            edgeActor.SetMapper(edgeMapper)
            self.outlineActorDict[stlName] = edgeActor

    #
    #  showSelectedItems
    def showSelectedItems(self, selNames):
        """ TreeItemの選択したitemをvtk表示させる"""
        if self.renderer == "":
            return
        #選択item名を取得
        self.allElms = list(map(lambda x: x[0], self.stlConts))
        self.showElms = selNames
        self.hideElms = list(set(self.allElms) - set(self.showElms))
        #assyをclear
        parts = list(self.assy.GetParts())
        if len(parts) > 0:
            self.renderer.RemoveActor(self.assy)
            self.renderer.SetErase(True)
            self.assy = vtk.vtkAssembly()
        #actorを再設定
        for showElm in self.showElms:
            if showElm in self.actorDict.keys():
                actor = self.actorDict[showElm]
                actor = self.setActorProperty(actor)
                self.assy.AddPart(actor)
        #  outlineを追加
        if self.check_showOutline.get_active() == True:
            allItems = self.treeList.getAllItemNames()
            stlNames = map(lambda x: x[0], allItems)
            for stlName in stlNames:
                actor = self.outlineActorDict[stlName]
                self.assy.AddPart(actor)
        self.renderer.AddActor(self.assy)
        parts = list(self.assy.GetParts())
        #原点を表示
        self.addOriginAxis()
        #初めて（削除するactorが無かった）の場合、表示を合わせる
        #self.fitWindow()
        #vtk表示
        self.vtkObj.Initialize()
        return
    
    #
    #  setActorProperty
    def setActorProperty(self, actor):
        """ actorのpropertyを設定"""
        prop = actor.GetProperty()
        #edgeの表示
        if self.check_showEdge.get_active() == True:
            #edge表示
            prop.EdgeVisibilityOn()
        else:
            #edge非表示
            prop.EdgeVisibilityOff()
        #表裏面の表示
        if self.radio_both.get_active() == True:
            #両面表示
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOff()
        elif self.radio_frontFace.get_active() == True:
            #表面を表示
            prop.BackfaceCullingOn()
            prop.FrontfaceCullingOff()
        elif self.radio_backFace.get_active() == True:
            #裏面を表示
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOn()
        #色の設定
        prop.SetAmbient(0.3)
        prop.SetColor(colors.grey)
        return actor

    #
    #  addOriginAxis
    def addOriginAxis(self):
        if self.check_showOrigin.get_active() == True:
            #原点を表示
            if self.originAxes == None:
                #原点を作成
                #bounds = self.getBoundsActors()
                bounds = self.getOriginBounds()
                if len(bounds) == 0:
                    return
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes = vtk.vtkAxesActor()
                axes.SetTotalLength(length, length, length)
                axes.GetXAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetXAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetYAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetYAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetZAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetZAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                self.renderer.AddActor(axes)
                self.originAxes = axes
            else:
                #原点を修正
                axes = self.originAxes
                #bounds = self.getBoundsActors()
                bounds = self.getOriginBounds()
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes.SetTotalLength(length, length, length)
                #削除して追加する（axesを最後に描画する為）
                self.renderer.RemoveActor(axes)
                self.renderer.AddActor(axes)
                self.originAxes = axes
        else:
            #原点を削除
            self.renderer.RemoveActor(self.originAxes)
            self.originAxes = None
        return

    #
    #  addCornerAxis
    def addCornerAxis(self):
        """ 最初の1回のみ、XYZの軸（compas）をrendererに追加する。"""
        if self.vtkInitialRun == True:
            #add axis(最初の起動時のみ)
            interactor = self.renWin.GetInteractor()
            axesActor = vtk.vtkAxesActor()
            self.axes = vtk.vtkOrientationMarkerWidget()
            self.axes.SetOrientationMarker(axesActor)
            self.axes.SetInteractor(interactor)
            self.axes.EnabledOn()
            self.axes.InteractiveOn()
            #flagをセット
            self.vtkInitialRun = False
        return

    #
    #  getCameraBounds
    def getCameraBounds(self):
        """ 表示させるactorのboundsを取得する。
        原点、マージンも含めて取得する。"""
        bounds = self.getOriginBounds()
        # if self.originAxes == None:
        #     #原点無しの場合
        #     return bounds
        if len(bounds) == 0:
            return bounds
        #マージンを確保
        xmin = bounds[0]
        xmax = bounds[1]
        ymin = bounds[2]
        ymax = bounds[3]
        zmin = bounds[4]
        zmax = bounds[5]
        a = 0.1
        xmin = xmin - (xmax-xmin) * a
        xmax = xmax + (xmax-xmin) * a
        ymin = ymin - (ymax-ymin) * a
        ymax = ymax + (ymax-ymin) * a
        zmin = zmin - (zmax-xmin) * a
        zmax = zmax + (zmax-xmin) * a
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getOriginBounds
    def getOriginBounds(self):
        """ 表示させるactorと原点のboundsを取得する。"""
        bounds = self.getBoundsActors()
        if self.originAxes == None:
            #原点無しの場合
            return bounds
        #原点を追加する
        xmin = min([bounds[0], 0.0])
        xmax = max([bounds[1], 0.0])
        ymin = min([bounds[2], 0.0])
        ymax = max([bounds[3], 0.0])
        zmin = min([bounds[4], 0.0])
        zmax = max([bounds[5], 0.0])
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getBoundsActors
    def getBoundsActors(self):
        """ 全actorのboundsを取得して返す。"""
        actors = self.renderer.GetActors()
        bounds = []
        for actor in actors:
            bound = actor.GetBounds()
            bounds.append(bound)
        if len(bounds) == 0:
            return ()
        Xmin = min(map(lambda x: x[0], bounds))
        Xmax = max(map(lambda x: x[1], bounds))
        Ymin = min(map(lambda x: x[2], bounds))
        Ymax = max(map(lambda x: x[3], bounds))
        Zmin = min(map(lambda x: x[4], bounds))
        Zmax = max(map(lambda x: x[5], bounds))
        return (Xmin, Xmax, Ymin, Ymax, Zmin, Zmax)

    #
    #  fitWindow
    #-------------
    def fitWindow(self):
        """ 画面に合わせてvtk表示させる"""
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromX
    #------------
    def viewFromX(self):
        """ X軸方向から見る"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter+maxLeng*2.0, Ycenter, Zcenter)
        #X軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromY
    #------------
    def viewFromY(self):
        """ Y軸視（Z-X面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter+maxLeng*2.0, Zcenter)
        #Y軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(1, 0, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromZ
    #------------
    def viewFromZ(self):
        """ Z軸視（X-Y面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter, Zcenter+maxLeng*2.0)
        #Z軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 1, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  setReverseCamera
    #-------------------
    def setReverseCamera(self):
        """ camera位置を反転（左右反転）させる"""
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        center = (Xcenter, Ycenter, Zcenter)
        camera = self.renderer.GetActiveCamera()
        pos = camera.GetPosition()
        relPos = (pos[0]-center[0], pos[1]-center[1], pos[2]-center[2])
        newRelPos = list(map(lambda x: -x, relPos))
        newPos = (newRelPos[0]+center[0], newRelPos[1]+center[1], newRelPos[2]+center[2])
        camera.SetPosition(*newPos)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  rolling90
    #------------
    def rolling90(self):
        """ 視点を右回転（90°）させる。"""
        camera = self.renderer.GetActiveCamera()
        camera.Roll(-90.0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    # changeShowEdge
    #----------------
    def changeShowEdge(self):
        """ check_showEdgeを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowOutline
    #--------------------
    def changeShowOutline(self):
        """ check_showOutlineを変更したときのevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowOrigin
    #-------------------
    def changeShowOrigin(self):
        """ 原点表示チェックを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowBoth
    #-----------------
    def changeShowBoth(self):
        """ radio_showBothを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowFront
    #------------------
    def changeShowFront(self):
        """ radio_showFrontFaceを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  changeShowBack
    #-----------------
    def changeShowBack(self):
        """ radio_showbackFaceを変更した時のevent"""
        self.applyCheckContsToVtk()

    #
    #  applyCheckContsToVtk
    def applyCheckContsToVtk(self):
        """ checkBoxの内容をvtkに適用する"""
        selNames = self.getSelectedStls()
        self.showSelectedItems(selNames)


    #-------- event handler -----------
    #-------- vtk画面 -----------
    def onFitWindow(self, event):           #画面にfit
        self.fitWindow()
    def onViewFromX(self, event):           #X軸視
        self.viewFromX()
    def onViewFromY(self, event):           #Y軸視
        self.viewFromY()
    def onViewFromZ(self, event):           #Z軸視
        self.viewFromZ()
    def onSetReverseCamera(self, event):    #反転
        self.setReverseCamera()
    def onRolling90(self, event):           #90度回転
        self.rolling90()
    def onChangeShowEdge(self, event):      #edge表示
        if self.maskEvent == False:
            self.changeShowEdge()
    def onChangeShowOutline(self, event):   #outline表示
        if self.maskEvent == False:
            self.changeShowOutline()
    def onChangeShowOrigin(self, event):    #原点表示
        if self.maskEvent == False:
            self.changeShowOrigin()
    def onChangeShowBoth(self, event):      #両面表示
        if self.maskEvent == False:
            self.changeShowBoth()
    def onChangeShowFront(self, event):     #表面表示
        if self.maskEvent == False:
            self.changeShowFront()
    def onChangeShowBack(self, event):      #裏面表示
        if self.maskEvent == False:
            self.changeShowBack()
    #-------- cfMesh のevent ------------
    #---- menuBar 
    def onMenuSave(self, event):            #保存
        self.saveCsv()
    def onMenuReload(self, event):          #再読込
        self.reloadCsv()
    def onMenuReloadNew(self, event):       #再読込（default）
        self.reloadAtDefault()
    def onMenuClose(self, event):           #閉じる
        self.onClose(event)
    def onMenuEditDict(self, event):        #dict編集
        self.editDict()
    def onMenuCheckMesh(self, event):       #checkMesh実行
        self.menuCheckMesh()
    def onMenuStlViewer(self, event):       #stlViewer起動
        self.runStlViewer()
    def onMenuMeshViewer(self, event):      #meshViewer起動
        self.runMeshViewer()
    def onMenuParaview(self, event):        #paraview起動
        self.runParaview()
    #---- toolBar
    def onSaveCsv(self, event):             #保存
        self.saveCsv()
    def onReloadCsv(self, event):           #再読込
        self.reloadCsv()
    def onReloadAtDefault(self, event):     #歳読込（default）
        self.reloadAtDefault()
    def onClose(self, event):               #閉じる
        if self.editFlag == "no":
            self.close()
            return
        self.selectEndMethod()
    def onRunCheckMesh(self, event):        #checkMesh実行
        self.runCheckMesh()
    def onRunStlViewer(self, event):        #stlViewer起動
        self.runStlViewer()
    def onRunMeshViewer(self, event):       #meshViewer起動
        self.runMeshViewer()
    def onRunParaview(self, event):         #paraview起動
        self.runParaview()
    #---- button 
    def onOpenCaseDir(self, event):         #caseDir開く
        self.openCaseDir()
    def onOpenStlDir(self, event):          #stlDirを開く
        self.openStlDir()
    def onStlDirRef(self, event):           #参照...(stlDir)
        self.stlDirRef()
    def onExcludeStls(self, event):         #stl除外
        self.excludeStls()
    def onSetDefaultValue(self, event):     #defaultに戻す
        self.setDefaultValue()
    def onUnselectStls(self, event):        #全stl非選択
        self.unselectStls()
    def onCreateDict(self, event):          #meshDict作成
        self.createMeshDict()
    def onCreateMesh(self, event):          #mesh作成
        self.createMesh()
    def onEditDict(self, event):            #dict編集
        self.editDict()
    #---- entry
    def onChangeStlDir(self, event):        #stlDir変更時
        self.changeStlDir()
    def onChangeMaxCellSize(self, event):   #maxCellSize変更時
        self.changeMaxCellSize()
    def onChangeFeatureAngle(self, event):  #featureAngle変更時
        self.changeFeatureAngle()
    def onChangeNThreads(self, event):
        pass
    #---- treeView
    def onChangeCursor(self, event):        #行選択時
        self.changeCursor()
    def onSelectAll(self, event):           #全行選択時
        self.changeCursor()
    def onUnselectAll(self, event):         #全行非選択時
        self.changeCursor()
    def onChangeCheckBox(self, path, col):  #checkBox変更時
        self.changeCheckBox(path, col)
    def onChangeComboBox(self, path, col):  #combo変更時
        self.changeComboBox(path, col)
    def onChangeEditedBox(self, path, col): #edit終了時
        self.changeEditedBox(path, col)
    #---- checkBox
    def onSetThreadsParallel(self, event):  #checkParallel
        self.setThreadParallel()


    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        self.windowTitle = _("cfMeshによるmesh作成")
        self.setWindowSize()        #windowSizeを設定
        self.initializeGui()        #GUIの文字をゴシックに変更
        self.setCaseDirToLabel()    #labelにcaseDirを設定
        self.loadInitialSetting()   #helyxOS_data内の値をセットする
        self.createTreeList()       #TreeListを定義
        self.checkCfMeshCsvFile()   #csvfile有無を確認し、なければ作成する
        self.getCfMeshCsvConts()    #csvFileのデータを取得する
        self.setDataToEntryBox()    #entryに値をセット
        self.createTreeList()       #TreeListを再設定
        self.setDataToTreeList()    #TreeListにデータをセット
        self.setAdjustBndCheckBox() #boundaryFieldの調整有無をセット

    #
    #  setWindowSize
    def setWindowSize(self):
        """ windowSizeを設定する"""
        winDict = pyTreeFoam.readWindowSize()
        #全体windowSizeを設定
        if "editCfMesh" in winDict.keys():
            xy = winDict["editCfMesh"]
            if xy != "":
                x, y = map(int, xy.split()[:2])
                self.mainWindow.set_default_size(x, y)
        #panedSplitterを設定
        if "editCfMesh_splitter" in winDict.keys():
            splitWidth = winDict["editCfMesh_splitter"]
            if splitWidth != "":
                width = int(splitWidth.split()[0])
                self.paned_win.set_position(width)

    #
    #  initializeGui
    def initializeGui(self):
        """ frameのlabelをboldに設定する"""
        bold_css = Gtk.CssProvider.new()
        cont  = b"label\n"
        cont += b"{\n"
        cont += b"    font-weight: bold;\n"
        cont += b"}\n"
        bold_css.load_from_data(cont)
        #ゴシックの設定
        style_action = self.label_cfMesh.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
        style_action = self.label_createMesh.get_style_context()
        style_action.add_provider(bold_css, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)

    #
    #  setEditFlag
    def setEditFlag(self):
        """ editFlagをセットする。titleに「*」を追加する"""
        self.editFlag = "edited"
        title = "*" + self.windowTitle
        self.mainWindow.set_title(title)

    #
    #  clearEditFlag
    def clearEditFlag(self):
        """ editFlagをクリアする。titleから「*」を削除"""
        self.editFlag = "no"
        self.mainWindow.set_title(self.windowTitle)

    #
    #  setCaseDirToLabel
    def setCaseDirToLabel(self):
        """ caseDirをlabelに追記"""
        caseDir = self.caseDir
        nMax = 100
        if len(caseDir) > nMax:
            caseDir = "..." + caseDir[-(nMax-3):]
        self.label_caseDir.set_text(caseDir)

    #
    #  loadInitialSetting
    def loadInitialSetting(self):
        """ helyxOS_data内に保存されているdataを設定する"""
        case = pyTreeFoam.case(self.caseDir)
        dataDict = case.getCreateMeshData()
        stlDir = dataDict["stlDir"]
        if stlDir[0] == "/":
            self.stlDir = stlDir
            relDir = os.path.relpath(stlDir, self.caseDir)
        else:
            self.stlDir = self.caseDir + "/" + stlDir
            relDir = stlDir
        self.entry_stlDir.set_text(relDir)
        if os.path.exists(self.stlDir) == False:
            #起動後、messageDialogを表示する
            self.newCsvFlag = 2
        nProcs = dataDict["nCpu"]
        self.entry_nThreads.set_text(nProcs)
        parallel = dataDict["parallel"]
        if parallel == "yes":
            self.check_parallel.set_active(True)
        else:
            self.check_parallel.set_active(False)
        self.setThreadParallel()

    #
    #  checkCfMeshCsvFile
    def checkCfMeshCsvFile(self):
        if os.path.exists(self.stlDir) == False:
            return
        stlNames = getStlFiles(self.stlDir)
        #csvをチェック        
        fileName = self.stlDir + "/" + self.cfMeshCsvName
        if os.path.exists(fileName) == True:
            #csv内容をチェック
            csvData = readCreateCfMeshCsvData(fileName)
            stlConts = csvData[2]
            names = list(map(lambda x: x[0], stlConts))
            names.sort()
            #stlFilesとcsvを比較
            if stlNames == names:
                #同じ場合は、直ぐに戻る
                return
            else:
                #異なる場合は、csvを削除
                os.remove(fileName)
                print(_("csvDataを再作成します。"))
                #csvをdefault値で作成する
                self.createDefaultCfMeshCsvFile(fileName)
                #flagを設定(dialogを表示させるため)
                self.newCsvFlag = 3
        else:
            #csvをdefault値で作成する
            print(_("csvDataが存在しない為、default値で作成します。"))
            self.createDefaultCfMeshCsvFile(fileName)
            #flagを設定(dialogを表示させるため)
            self.newCsvFlag = 1

    #
    #  createDefaultCfMeshCsvFile
    def createDefaultCfMeshCsvFile(self, fileName):
        """ defaultの内容でcsvを作成する"""
        stlNames = getStlFiles(self.stlDir)
        self.setDefaultValueToEntryBox(stlNames)
        self.setDefaultValueToTreeList(stlNames)
        #csvFile作成
        csvRowCol = createCsvDataFromRowCol(
            self.maxCellSize, self.featureAngle, self.stlConts)
        lines = []
        for rowCol in csvRowCol:
            line = ",".join(rowCol) + "\n"
            lines.append(line)
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

    def setDefaultValueToEntryBox(self, stlNames):
        """ maxCellSize, featureAngleをdefault値に設定する"""
        minLoc, maxLoc = getStlModelMinMax(self.stlDir, stlNames)
        x = maxLoc[0] - minLoc[0]
        y = maxLoc[1] - minLoc[1]
        z = maxLoc[2] - minLoc[2]
        size = max([x,y,z]) / 30.0
        self.maxCellSize = pyTreeFoam.float2strAuto(size, prec=3)
        self.featureAngle = "30.0"
        #entryに値をセット
        self.setDataToEntryBox()

    def setDefaultValueToTreeList(self, stlNames):
        """ treeListのデータをdefault値に設定する"""
        #stlContsを作成
        self.stlConts = []
        for stlName in stlNames:
            row = ["" for i in range(6)]
            row[csvColDict["stl"]] = stlName
            row[csvColDict["sect"]] = "patch"
            row[csvColDict["cell"]] = self.maxCellSize
            self.stlConts.append(row)
        #treeListにセット
        self.setDataToTreeList()

    #
    #  getCfMeshCsvConts
    def getCfMeshCsvConts(self):
        """ csvの内容を取得する"""
        fileName = self.stlDir + "/" + self.cfMeshCsvName
        csvData = readCreateCfMeshCsvData(fileName)
        self.maxCellSize, self.featureAngle, self.stlConts = csvData

    #
    #  setDataToEntryBox
    def setDataToEntryBox(self):
        """ 値をentryBoxにセットする"""
        self.entry_maxCellSize.set_text(self.maxCellSize)
        self.entry_featureAngle.set_text(self.featureAngle)

    #
    #  getDataFromEntryBox
    def getDataFromEntryBox(self):
        """ entryBox内のdataを取得する"""
        self.maxCellSize = self.entry_maxCellSize.get_text()
        self.featureAngle = self.entry_featureAngle.get_text()

    #
    #  getCellSizesForCombo
    def getCellSizesForCombo(self, cellSize):
        """ comboBoxの候補となるcellSizeを取得する"""
        if type(cellSize) == str:
            cellSize = float(cellSize)
        comboSizes = []
        for i in range(10):
            size = pyTreeFoam.float2strAuto(cellSize, prec=3)
            comboSizes.append(size)
            cellSize /= 2.0
        return comboSizes        

    #
    #  createTreeList
    def createTreeList(self):
        """ treeListを定義する"""
        widthes = []
        #cellSizeを候補を作成する
        cellSize = float(self.maxCellSize)
        comboSizes = self.getCellSizesForCombo(cellSize)
        #header文字を作成
        hd0 = "stl"
        hd1 = "patch\ntype"
        hd2 = "cellSize"
        hd3 = "layer" + "\n" + _("追加")
        hd4 = "layer" + "\n" + _("数")
        hd5 = "layer" + "\n" + _("拡大率")
        hd6 = "layer" + "\n" + _("max厚")
        #treeListの定義
        #        text combo text toggle   text   text text
        #        stl  type  cell addLayer nLayer exp  maxTh
        model = (str, str,  str, bool,    str,   str, str)
        header= [hd0, hd1,  hd2, hd3,     hd4,   hd5, hd6]
        #header = ["stl", "patch\ntype", "cellSize", "layer\n追加",
        #          "layer\n数", "layer\n拡大率", "layer\nmax厚"]
        comboTypes = ["patch", "wall", "empty", "symmetry", 
                  "symmetryPlane", "regBox", "regSph", "face"]
        tree = treeListColDict
        combos = [(tree["type"], tree["cell"]),
                  (comboTypes, comboSizes)]
        editables = [tree["nLayer"], tree["expan"], tree["maxTh"]]
        eventFuncs = [self.onChangeCheckBox, self.onChangeComboBox, self.onChangeEditedBox]
        #treeListを作成
        self.treeList.create(
            model,
                multi=True, 
                header=True, 
                headerConts=header,
                colWidthes=widthes,
                combos=combos,
                editables=editables,
                eventFuncs=eventFuncs
                )

    #
    #  getSelectedStls
    def getSelectedStls(self):
        """ 選択しているstlNameを取得する"""
        selRowConts = self.treeList.getSelectedItemNames()
        selNames = list(map(lambda x:x[0], selRowConts))
        return selNames

    #
    #  setDataToTreeList
    def setDataToTreeList(self):
        """ TreeViewにdataをセットし直す"""
        items = []
        for stlCont in self.stlConts:
            (stlName, patchType, cell, 
             nLayers, expan, maxTh) = stlCont
            addLayers = False
            if nLayers != "":
                addLayers = True
            item = [stlName, patchType, cell, addLayers, 
                    nLayers, expan, maxTh]
            items.append(item)            
        #treeListにセット
        self.treeList.setItems(items)

    #
    #  setAdjustBndCheckBox
    def setAdjustBndCheckBox(self):
        """ checkBoxを初期化する"""
        self.check_adjustBnd.set_active(True)

    #
    #  getDataFromTreeList
    def getDataFromTreeList(self):
        """ treeListのdataをself.stlContsで取得する"""
        #treeListからstr形式のdataを取得
        items = self.treeList.getAllItemNames()
        #stlContsに反映
        self.stlConts = []
        for item in items:
            csvCont = item + [""]
            self.stlConts.append(csvCont)

    #
    #  createShowSelectedItems
    def createShowSelectedItems(self, selNames):
        """ entryの値を取得し、vtkを作成し直す"""
        self.getDataFromEntryBox()  #entryの内容を取得
        self.createTreeList()       #TreeListを作成
        self.setDataToTreeList()    #TreeListにデータをセット
        self.selectRowsInTreeList(selNames)
        self.createActorDict()      #actor辞書作成
        self.showSelectedItems(selNames)    #選択itemをvtk画面に表示

    #
    #  selectRowsInTreeList
    def selectRowsInTreeList(self, selNames):
        """ treeList内の項目を選択する"""
        if len(selNames) == 0:
            return
        allNames = list(map(lambda x: x[0], self.stlConts))
        selNos = []
        for selName in selNames:
            selNo = allNames.index(selName)
            selNos.append(selNo)
        self.treeList.selectItemNos(selNos)


    #
    #  saveCsv
    #----------
    def saveCsv(self):
        """ csvDataの保存"""
        #修正Flagクリア
        self.clearEditFlag()        #修正flagをクリア
        #csvで保存
        fileName = self.stlDir + "/" + self.cfMeshCsvName
        #entryのdataを取得
        self.getDataFromEntryBox()
        #treeListのdataを取得
        self.getDataFromTreeList()
        #csvを取得
        self.csvRowCol = createCsvDataFromRowCol(
            self.maxCellSize, self.featureAngle, self.stlConts)
        lines = []
        for i in range(len(self.csvRowCol)):
            line = ",".join(self.csvRowCol[i]) + "\n"
            lines.append(line)
        f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()
        #paralleの設定を取得
        dataDict = {}
        if self.check_parallel.get_active() == True:
            dataDict["parallel"] = "yes"
            dataDict["nCpu"] = self.entry_nThreads.get_text()
        else:
            dataDict["parallel"] = "no"
            dataDict["nCpu"] = self.entry_nThreads.get_text()
        case = pyTreeFoam.case(self.caseDir)
        case.setCreateMeshData(dataDict)
        print(_("保存しました。"))

    #
    #  reloadCsv
    #------------
    def reloadCsv(self):
        """ csvファイルを再読込する。"""
        self.clearEditFlag()        #修正flagをクリア
        self.getCfMeshCsvConts()
        self.setDataToEntryBox()
        self.createTreeList()
        self.setDataToTreeList()
        print(_("再読込みしました。"))

    #
    #
    #  reloadAtDefault
    #------------------
    def reloadAtDefault(self):
        """ csvを再読込し、各定数をdefault値に設定する"""
        title = _("default値の設定")
        mess  = _("default値を設定すると、") + "\n"
        mess += "  " + _("現在の設定内容が全て初期値に戻ります。") + "\n\n"
        mess += _("このまま、default値を設定しますか？")
        funcOk = [self.reloadAtDefault_run]
        self.okCancelDialog(title, mess, funcOk)

    #
    #  reloadAtDefault_run
    def reloadAtDefault_run(self):
        """ csvを再読込し、各定数をdefault値に設定する"""
        #csvFileのデータを取得する。
        self.checkCfMeshCsvFile()
        self.getCfMeshCsvConts()
        #maxCellSizeを設定
        stlNames = list(map(lambda x: x[0], self.stlConts))
        minLoc, maxLoc = getStlModelMinMax(self.stlDir, stlNames)
        x = maxLoc[0] - minLoc[0]
        y = maxLoc[1] - minLoc[1]
        z = maxLoc[2] - minLoc[2]
        size = max([x,y,z]) / 30.0
        self.maxCellSize = pyTreeFoam.float2strAuto(size, prec=3)
        self.featureAngle = "30.0"
        #entryに値をセット
        self.setDataToEntryBox()
        #treeListをdefault値に設定
        self.setDefaultValueToTreeList(stlNames)

    #
    #  selectEndMethod
    #------------------
    def selectEndMethod(self):
        """ 保存して終了、終了、canceを選択して終了"""
        title = _("終了確認")
        mess = _("内容が変更されています。") + "\n"
        mess += _("  このまま処理を継続すると、変更内容が失われます。")
        funcSaveGo = [self.saveAndEnd]
        funcGo = [self.justEnd]
        funcCancel = []
        dialog = saveGoDDialog.saveGo(title, mess, 
                                     funcSaveGo=funcSaveGo,
                                     funcGo=funcGo,
                                     funcCancel=funcCancel)
        dialog.show()

    def saveAndEnd(self):
        """ 保存して終了"""
        self.saveCsv()
        self.close()

    def justEnd(self):
        """ 終了する"""
        self.close()

    #
    #  runCheckMesh
    #---------------
    def runCheckMesh(self):
        """ checkMeshの実行"""
        self.menuCheckMesh()

    #
    #  runStlViewer
    #---------------
    def runStlViewer(self):
        """ stlViewerの起動"""
        stlDir = self.stlDir
        stlDir = pyTreeFoam.getAbsDir(self.caseDir, stlDir)
        if os.path.exists(stlDir) == False or os.path.isdir(stlDir) == False:
            title = _("エラー")
            mess = _("stlDir「" + stlDir + "」は、存在しません。") + "\n"
            mess += _("stlDirを再設定してください。")
            self.errDialog(title, mess)
            return
        path = os.getenv("TreeFoamPath") + os.sep + "python"
        comm = "./editStlFilesDialog.py " + self.caseDir + " " + stlDir + " &"
        pyTreeFoam.run(path).command(comm)

    #
    #  runMeshViewer
    #----------------
    def runMeshViewer(self):
        """ meshViewerを起動"""
        caseDir = self.caseDir
        stlDir = self.stlDir
        try:
            timeFolder = pyTreeFoam.case(caseDir).getCurrTimeFolder()
        except:
            timeFolder = "0"
        region = "."
        title = "meshViewer"
        self.meshViewer = meshViewerControl.control(caseDir, timeFolder, region, title)
        self.meshViewer.openViewer("full")
        commLines  = [_("start\n")]
        commLines += [_("mess mesh作成caseから起動しています。\n")]
        commLines += [_("addMess '  patch, zone, set, stlが表示できます。'\n")]
        commLines += [_("addMess '  tree内の選択した項目が表示されます。'\n")]
        commLines += [_("time startFrom\n")]
        commLines += [_("stlDir ") + stlDir + "\n"]
        commLines += [_("print 'loading ...'\n")]
        commLines += [_("wait 0.2s\n")]     #GUIを完成させる為のwait
        commLines += [_("load all\n")]
        commLines += [_("show\n")]
        commLines += [_("resetCamera\n")]
        commLines += [_("setTreeList\n")]
        commLines += [_("end\n")]
        _stat = self.meshViewer.runMultiCommands(commLines)

    #
    #  runParaview
    #--------------
    def runParaview(self):
        """ paraviewを起動"""
        contDict = pyTreeFoam.readConfigTreeFoam()
        paraFoam = contDict["paraFoam"]
        comm = paraFoam + " &"
        print(comm)
        pyTreeFoam.run(self.caseDir).commandWithLog(comm)

    #
    #  openCaseDir
    #--------------
    def openCaseDir(self):
        """ caseDirを開く"""
        run = pyTreeFoam.run()
        comm = pyTreeFoam.fileManager + " " + self.caseDir + " &"
        run.command(comm)

    #
    #  openStlDir
    #-------------
    def openStlDir(self):
        """ stlDirを開く"""
        run = pyTreeFoam.run()
        comm = pyTreeFoam.fileManager + " " + self.stlDir + " &"
        run.command(comm)

    #
    #  stlDirRef
    #------------
    def stlDirRef(self):
        """ 「参照...」strDirを選択する。"""
        relDir = self.entry_stlDir.get_text()
        stlDir = self.caseDir + "/" + relDir
        if os.path.exists(stlDir) == False:
            selDir = "."
        else:
            selDir = relDir
        title = _("stlDirの選択")
        cont = _("相対参照でstlDirを取得する。")
        rootDir = os.getenv("HOME")
        currDir = self.caseDir
        okArgs = [self.stlDirRef_setTextBox]
        dialog = getOpenFolderDDialog.getOpenFolder(
            title, cont, rootDir, currDir, selDir, 
            funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #  stlDirRef_setTextBox
    def stlDirRef_setTextBox(self, openDir):
        self.entry_stlDir.set_text(openDir)
        self.changeStlDir()

    #
    #  excludeStls
    #---------------
    def excludeStls(self):
        """ 選択stlをlistから除外する。
        file名を'_'付きに変更してlistから除外する。"""
        rows = self.treeList.getSelectedItemNames()
        selNames = []
        for row in rows:
            selName = row[0]
            selNames.append(selName)
            stlFile = self.stlDir + "/" + selName + ".stl"
            newName = self.stlDir + "/_" + selName + ".stl"
            os.rename (stlFile, newName)
        newConts = []
        for row in self.stlConts:
            if not row[0] in selNames:
                newConts.append(row)
        self.stlConts = newConts
        self.setDataToTreeList()
        self.saveCsv()

    #
    #  setDefaultValue
    #------------------
    def setDefaultValue(self):
        """ treeView内をdefault値に設定する"""
        title = _("default値の設定")
        mess  = _("default値を設定すると、") + "\n"
        mess += "  " + _("現在の設定内容が全て初期値に戻ります。") + "\n\n"
        mess += _("このまま、default値を設定しますか？")
        funcOk = [self.setDefaultValue_run]
        self.okCancelDialog(title, mess, funcOk)

    #
    #  setDefaultValue_run
    def setDefaultValue_run(self):
        """ treeView内をdefault値に設定する"""
        self.setEditFlag()              #修正flagをセット
        csv = csvColDict
        #clearする項目
        blankCol = (csv["nLayer"], csv["expan"], csv["maxTh"]) 
        #cellSizeを設定する項目
        setCol = (csv["cell"],)
        self.getDataFromEntryBox()
        #patchTypeを設定
        patchType = "patch"
        for row in range(len(self.stlConts)):
            for col in range(len(self.stlConts[row])):
                if col in blankCol:
                    self.stlConts[row][col] = ""
                if col in setCol:
                    self.stlConts[row][col] = self.maxCellSize
                if col == csv["sect"]:
                    self.stlConts[row][col] = patchType
        selNames = self.getSelectedStls()
        self.createShowSelectedItems(selNames)

    #
    #  getNearestValeInCells
    def getNearestValeInCells(self, cellVals, currVal):
        """ cellVals内から、currValに近い値を取得して返す。
        比較は、対数で比較する。"""
        
        def calcDiff(a, b):
            """ 対数の差分を計算"""
            diff = abs(math.log10(a) - math.log10(b))
            return diff

        diff = calcDiff(float(cellVals[0]), float(currVal))
        si = 0
        for i in range(1, len(cellVals), 1):
            cellVal = cellVals[i]
            newDiff = calcDiff(float(cellVal), float(currVal))
            if diff >= newDiff:
                diff = newDiff
            else:
                si = i - 1
                break
        return cellVals[si]

    #
    #  setDefaultValueForCellSize
    #-----------------------------
    def setDefaultValueForCellSize(self):
        """ cellSize変更時の初期値設定"""
        self.setEditFlag()              #修正flagをセット
        csv = csvColDict
        #cellSizeを設定する項目
        setCol = (csv["cell"],)
        self.getDataFromEntryBox()
        cellVals = self.getCellSizesForCombo(self.maxCellSize)
        for row in range(len(self.stlConts)):
            for col in range(len(self.stlConts[row])):
                if col in setCol:
                    currVal = self.stlConts[row][col]
                    newVal = self.getNearestValeInCells(cellVals, currVal)
                    #self.stlConts[row][col] = self.maxCellSize
                    self.stlConts[row][col] = newVal
        selNames = self.getSelectedStls()
        self.createShowSelectedItems(selNames)


    #
    #  unselectStls
    #---------------
    def unselectStls(self):
        """ 全stlを非選択する"""
        self.treeList.unselectAllItems()
        selNames = []
        self.showSelectedItems(selNames)

    #
    #  createMeshDict
    #-----------------
    def createMeshDict(self):
        """ meshDictを作成する"""
        #設定を保存
        self.saveCsv()
        #fmsファイル作成の為に必要なstlをmergeする
        stlDir = self.stlDir
        csvName = self.cfMeshCsvName
        mergeName = "_assy.stl"
        error = mergeStlFilesForFms(stlDir, csvName, mergeName)
        if error != "":
            title = _("エラー")
            mess = error
            self.errDialog(title, mess)
            return
        
        #fmsFileを作成
        envOpenFoam = self.configDict["bashrcFOAM"]
        angle = self.featureAngle
        relDir = self.entry_stlDir.get_text()
        mergeFile = relDir + "/" + mergeName
        surfaceName = "model.fms"
        surfaceFile = relDir + "/" + surfaceName
        comm =  ". " + envOpenFoam + "; " 
        comm += "surfaceFeatureEdges -angle " + angle + " " + mergeFile + " " + surfaceFile
        print(comm)
        (stat, _res, _err) = pyTreeFoam.run(self.caseDir).commandReturnCont(comm)
        if stat != "OK":
            title = _("エラー")
            error = _("特徴線が抽出出来ませんでした。")
            error += _("\nlogを確認してください。")
            self.errDialog(title, error)
            return

        #meshDict作成
        self.checkCfMeshDict()
        caseDir = self.caseDir
        otherData = [self.maxCellSize, "", self.featureAngle]
        data = self.stlConts
        createCfMeshDict.createMeshDict(caseDir, stlDir, surfaceName, otherData, data)
        title = _("meshDictの作成")
        mess = _("fmsファイル、meshDictを作成しました。")
        self.okDialog(title, mess)
        return 
    
    #
    #  checkCfMeshDict
    def checkCfMeshDict(self):
        """ cfMeshのmeshDict有無を確認し、無ければdefaultのmeshDictをコピーする。"""
        meshDict = caseDir + "/system/meshDict"
        if os.path.exists(meshDict) == True:
            return
        #defaultのmeshDictをコピーする
        TreeFoamPath = os.getenv("TreeFoamPath")
        cfMeshFolders = glob.glob(TreeFoamPath + "/data/cfMesh/*")
        cfMeshFolders.sort()
        latestFolder = cfMeshFolders[-1]
        copyFile = latestFolder + "/meshDict"
        pasteDir = self.caseDir + "/system"
        shutil.copy(copyFile, pasteDir)

    #
    #  createMesh
    #-------------
    def createMesh(self):
        """ cfMesh実行し、メッシュを作成する"""
        #並列処理の確認
        if self.parallelFlag == "yes":
            nThreads = self.entry_nThreads.get_text()
            parallel = "export OMP_NUM_THREADS=" + nThreads + "\n"
        else:
            parallel = ""
        #runFileを作成
        envOpenFoam = self.configDict["bashrcFOAM"]
        cont  = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        cont += parallel
        cont += "cartesianMesh | tee solve.log\n"
        if self.check_adjustBnd.get_active() == True:
            cont += os.getenv("TreeFoamPath") + "/python/meetPatchNameDeleteNullPatch.py " + self.caseDir + "\n"
        comm = "echo " + "'" + cont + "'" + " > run"
        #実行
        run = pyTreeFoam.run(self.caseDir)
        run.command(comm)
        run.command("chmod a+x run")
        run.foamTerminal("./run")
            

    #
    #  editDict
    #-----------
    def editDict(self):
        """ dictを編集する"""
        path = self.caseDir + "/system"
        cfMeshDict = path + "/meshDict"
        files = [cfMeshDict]
        run = pyTreeFoam.run(self.caseDir)
        run.editor(files)

    #
    #  menuCheckMesh
    #----------------
    def menuCheckMesh(self):
        """ checkMeshを実行する"""
        print(_("checkMeshを実行"))
        envOpenFoam = self.configDict["bashrcFOAM"]
        cont = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        cont += "checkMesh 2>&1 | tee solve.log\n"
        #スクリプト作成
        comm = "echo " + "'" + cont + "'" + " > run"
        runCase = pyTreeFoam.run(self.caseDir)
        runCase.command(comm)
        runCase.command("chmod a+x run")
        runCase.foamTerminal("run")

    #
    #  changeStlDir
    #---------------
    def changeStlDir(self):
        """ stlDirを変更した時のevent"""
        stlDir = self.entry_stlDir.get_text()
        if stlDir[0] == "/":
            absDir = stlDir
            relDir = os.path.relpath(stlDir, self.caseDir)
            self.entry_stlDir.set_text(relDir)
        else:
            relDir = stlDir
            absDir = self.caseDir + "/" + stlDir
        self.stlDir = absDir
        self.checkCfMeshCsvFile()   #csvfile有無を確認し、なければ作成する
        self.getCfMeshCsvConts()    #csvFileのデータを取得する
        self.setDataToEntryBox()    #entryに値をセット
        self.createTreeList()       #TreeListを作成
        self.setDataToTreeList()    #TreeListにデータをセット
        self.renderer = ""
        self.setVtkModel()
    
    #
    #  changeMaxCellSize
    #--------------------
    def changeMaxCellSize(self):
        """ cellSizeを変更したときのevent"""
        self.setDefaultValueForCellSize()

    #
    #  changeFeatureAngle
    #---------------------
    def changeFeatureAngle(self):
        """ featureAngleを変更したときのevent"""
        self.setEditFlag()
        self.featureAngle = self.entry_featureAngle.get_text()

    #
    #  changeCursor
    #---------------
    def changeCursor(self):
        """ delayを設定して、treeListのselectItemsを取得し、表示する。"""
        GLib.timeout_add(50, self.getShowSelectedItems)

    #  getShowSelectedItems
    def getShowSelectedItems(self):
        """ 選択stlを取得してvtk表示する"""
        selNames = self.getSelectedStls()
        self.showSelectedItems(selNames)

    #
    #  changeCheckBox
    #-----------------
    def changeCheckBox(self, path, col):
        """ treeViewのinternalCellSizeを変更したときのevent"""
        #修正Flag設定
        self.setEditFlag()              #修正flagをセット
        if col == treeListColDict["addLayer"]:
            self.changeAddLayers(path, col)    

    #
    #  changeAddLayers
    def changeAddLayers(self, path, col):
        """ addLayersのcheckを変更した時のevent"""
        if col == treeListColDict["addLayer"]:
            listStore = self.treeList.treeList.get_model()
            if listStore[path][col] == True:
                nLayers = "3"
                expan = "1.2"
            else:
                nLayers = ""
                expan = ""
            listStore[path][col+1] = nLayers
            listStore[path][col+2] = expan
            self.stlConts[int(path)][csvColDict["nLayer"]] = nLayers
            self.stlConts[int(path)][csvColDict["expan"]] = expan

    #
    #  changeComboBox
    #-----------------
    def changeComboBox(self, path, col):
        """ treeViewのinternalCellSizeを変更したときのevent"""
        #修正Flag設定
        self.setEditFlag()              #修正flagをセット
        #cellSize変更？
        if col == treeListColDict["type"]:
            listStore = self.treeList.treeList.get_model()
            patchType = listStore[path][col]
            self.stlConts[int(path)][csvColDict["sect"]] = patchType
        elif col == treeListColDict["cell"]:
            listStore = self.treeList.treeList.get_model()
            cellSize = listStore[path][col]
            self.stlConts[int(path)][csvColDict["cell"]] = cellSize

    #
    #  changeEditedBox
    #---------------------
    def changeEditedBox(self, path, col):
        """ treeViewのtextEdit終了時のevent"""
        #修正Flag設定
        self.setEditFlag()              #修正flagをセット
        #nLayer編集？
        if col == treeListColDict["nLayer"]:
            listStore = self.treeList.treeList.get_model()
            nLayers = listStore[path][col]
            self.stlConts[int(path)][csvColDict["nLayer"]] = nLayers
        elif col == treeListColDict["expan"]:
            listStore = self.treeList.treeList.get_model()
            expan = listStore[path][col]
            self.stlConts[int(path)][csvColDict["expan"]] = expan
        elif col == treeListColDict["maxTh"]:
            listStore = self.treeList.treeList.get_model()
            maxTh = listStore[path][col]
            self.stlConts[int(path)][csvColDict["maxTh"]] = maxTh

    #
    #  setThreadParallel
    #--------------------
    def setThreadParallel(self):
        """ thread並列するcheckBoxを変更したときのevent"""
        if self.check_parallel.get_active() == True:
            self.parallelFlag = "yes"
        else:
            self.parallelFlag = "no"


    #----- universalDialog -----------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def inputTextDialog(self, title, message, entryText, funcOk=[]):
        dialog = inputTextDDialog.getInputText(
            title, message, entryText, funcOk=funcOk, 
            parent=self.mainWindow)
        inputText = dialog.show()
        return inputText
    
    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()

#
#  getStlFiles
#---------------
def getStlFiles(stlDir):
    """ stlFileNameを取得して返す"""
    stlFiles = glob.glob(stlDir + "/*.stl")
    stlNames = []
    for stlFile in stlFiles:
        if os.path.isfile(stlFile) == True:
            name = os.path.basename(stlFile)
            stlName = ".".join(name.split(".")[:-1])
            if stlName[0] != "_":
                stlNames.append(stlName)
    stlNames.sort()
    return stlNames

#
#  createFmsFileForCfMesh
#-------------------------
def mergeStlFilesForFms(stlDir, csvName, mergeName):
    """ fmsファイルを作成する為に必要なstlFileを結合する。"""
    fileName = stlDir + "/" + csvName
    csvData = readCreateCfMeshCsvData(fileName)
    maxCellSize, featureAngle, stlConts = csvData
    #fmsを作成するstlFileを取得
    needPatches = ["patch", "wall", "empty", "symmetry", "symmetryPlane"]
    stlFiles = []
    error = ""
    for cont in stlConts:
        patchType = cont[1]
        if patchType in needPatches:
            stlFile = stlDir + "/" + cont[0] + ".stl"
            solidName = cont[0]
            error = setSolidNameInStlFile(stlFile, solidName)
            if error != "":
                break
            stlFiles.append(stlFile)
    if error != "":
        return error
    
    #stlFilesを結合
    mergeFile = stlDir + "/" + mergeName
    if os.path.exists(mergeFile) == True:
        os.remove(mergeFile)
    mergeStlFiles(stlFiles, mergeFile)
    return ""

#
#  mergeStlFiles
def mergeStlFiles(stlFiles, mergeFile):
    """ stlFilesを結合する"""
    fa = open(mergeFile, "a")
    for stlFile in stlFiles:
        f = open(stlFile); cont = f.read(); f.close()
        if cont[-1] != "\n":
            cont += "\n"
        fa.write(cont)
    fa.close()

#
#  setSolidNameInStlFiles
def setSolidNameInStlFile(stlFile, solidName):
    """ stlFileにsolidNameを設定する"""
    f = open(stlFile); lines = f.readlines(); f.close()
    flag = 0
    for i in range(len(lines)):
        line = lines[i]
        if line[:5] == "solid":
            lines[i] = "solid " + solidName + "\n"
            flag += 1
        elif line[:8] == "endsolid":
            lines[i] = "endsolid " + solidName + "\n"
            flag += 1
    if flag == 2:
        f = open(stlFile, "w"); f.writelines(lines); f.close()
        error = ""
    else:
        stlName = os.path.basename(stlFile)
        error =  _("stlFileのsolid名が変更できませんでした。") + "\n"
        error += "  " + stlName
    return error

#
#  readCreateCfMeshCsvData
#--------------------------
def readCreateCfMeshCsvData(fileName):
    """ cfMeshのcsvDataを読み込む"""
    if os.path.exists(fileName) == False:
        maxCellSize = "0"
        featureAngle = "0"
        stlConts = []
    else:
        csvRowCol = pyTreeFoam.readCsvData(fileName)
        maxCellSize = csvRowCol[2][2]
        featureAngle = csvRowCol[4][2]
        stlConts = getStlConts(csvRowCol)
    ans = (maxCellSize, featureAngle, stlConts)
    return ans

#
#  getStlConts
def getStlConts(csvRowCol):
    """ csvRowCol内容からstlに関するdataを取得する
    csvFileのcol(6+1個)
        stl, sect, cell, nLayer, expan, maxTh, note
    stlContsのcol(7個)
        stl, type, cell, addLayer, nLayer, expan, maxTh"""
    #取得するcolNoを取得
    csv = csvColDict
    getNos = (
        csv["stl"], csv["sect"], csv["cell"], csv["nLayer"],
        csv["expan"], csv["maxTh"]
        )
    stlConts = []
    rowColData = csvRowCol[6:]
    for rowData in rowColData:
        data = []
        for getNo in getNos:
            try:
                val = rowData[getNo]
            except:
                val = ""
            data.append(val)
        stlConts.append(data)
    return stlConts

#
#  createCsvDataFromRowCol
#-----------------------
def createCsvDataFromRowCol(maxCellSize, featureAngle, stlConts):
    """ csvDataを作成する"""
    rows = []
    rows += [["", "", "", "", "", "", ""]]
    rows += [["<cfMesh>", "", "", "", "", "", ""]]
    rows += [["", "maxCellSize", maxCellSize, "", "", "", ":最大のcellサイズ"]]
    rows += [["", "minCellSize", "", "", "", "", ":最小のcellサイズ"]]
    rows += [["", "featureAngle", featureAngle, "", "", "", ":特徴線抽出用の角度"]]
    rows += [["stlFile", '"sect\n(patch/wall/\nempty/symmetry(Plane)/\nregBox/regSph/face)"', "cellSize", "nLayer", "ratio", "maxThickness", ""]]
    for rowData in stlConts:
        rows += [rowData[:6] + [""]]
    return rows

#
#  getStlModelMinMax
#--------------------
def getStlModelMinMax(stlDir, stlNames):
    """ 全stlFileのminMaxを求める"""
    if len(stlNames) == 0:
        return [[0,0,0], [0,0,0]]
    minValues = []; maxValues = []
    for stlName in stlNames:
        fileName = stlDir + "/" + stlName + ".stl"
        minMax = createBlockAndSnappyDict.getMinMaxValueStl(fileName)
        minValues.append(minMax[0])
        maxValues.append(minMax[1])
    minXs = list(map(lambda x: x[0], minValues))
    minYs = list(map(lambda x: x[1], minValues))
    minZs = list(map(lambda x: x[2], minValues))
    minX = min(minXs); minY = min(minYs); minZ = min(minZs)
    minLoc = [minX, minY, minZ]
    maxXs = list(map(lambda x: x[0], maxValues))
    maxYs = list(map(lambda x: x[1], maxValues))
    maxZs = list(map(lambda x: x[2], maxValues))
    maxX = max(maxXs); maxY = max(maxYs); maxZ = max(maxZs)
    maxLoc = [maxX, maxY, maxZ]
    return (minLoc, maxLoc) 




if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()