#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
#   createSetFieldsDictDialog.py
#
#       setFieldsDictを作成する。
#
#   13/11/16    新規作成
#      11/24    field追加時、fieldTypesを取得するよう修正
#      12/02    grid上にpopupMenuを追加
#      12/10    paste時のバグを修正
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/03/06    readWinSize, saveWinSizeを修正（null行があった場合ｴﾗｰ発生）
#   15/01/19    saveWinSize:バグ修正（行末に「\n」を追加）
#      02/02    getDefaultFieldValues:バグ修正（値がvectorの場合ｴﾗｰ発生）
#      06/25    createMatrixでcellEditorのセットをwxのversionで変更
#               windowのサイズ設定も同様
#      12/02    「Dict保存・実行」ボタンを追加（saveExecを追加）
#   16/07/28    起動時のwindowsサイズをwxとTreeFoamのバージョンのセットで変更
#   17/02/02    copyToClipBoard,getPasteValsFromClipBoard:バグ修正
#               copy&pasteが作動しなかった事を修正
#      03/08    saveCsvData:誤記訂正。（scv→csvに修正）
#      03/09    addFields:fieldを同時に複数追加した時、作動がおかしかった。
#   19/02/01    windowsサイズ設定を修正(wx.vrsion()のgtkVerを追加）
#      03/03    saveExec:regionに対してsetFieldsできなかったので修正。
#               setFields ⇢ setFields -region <region名> に修正
#               saveDictData,getColRowData:region内のsetFieldsDictのpath名が
#               間違っていた為修正。
#   20/01/29    GTK+3, python3用に大幅修正
#      04/21    多言語化対応
#      05/02    PyQt4, PyQt5, PySide, PySide2に対応
#      05/24    PyQtのモジュール確認方法を変更（一元化）
#   22/07/19    getTopoSetDB:OFversionのcheck方法修正（OF-10対応）
#   24/08/23    __init__:windowの位置をmousePointerに合わせる様修正。
#               addCellSet,addField,closeCheck,checkSaveFlagGo,saveCsv:
#               self.MainWindowの中心位置を算出し、dialogに引数として与える事を追加。
#      08/26    runGridEditor:起動時に、bashrcFOAMの環境設定後、gridEditor起動。
#   25/07/12    getColRowData:OF-13用に修正
#      07/13    saveDictData,addBox,addCylinder,addSphere:OF-13用に修正
#      07/14    addCellSet,showPopupRowMenu:OF-13用に修正
#      07/20    setRegionsToDict13:setFieldsへの保存にaddGeomFileを追加
#               getColRowData:setFieldsの読み込みにgeomFileを追加
#


import sys
import os
import glob


import getPyQtModule
PyQtModule = getPyQtModule.PyQtModule
#  import されているPyQtに応じてimportする
if PyQtModule == "PyQt6":
    from PyQt6.QtCore import *
    from PyQt6.QtGui import *
    from PyQt6.QtWidgets import *
elif PyQtModule == "PySide6":
    from PySide6.QtCore import *
    from PySide6.QtGui import *
    from PySide6.QtWidgets import *
elif PyQtModule == "PyQt5":
    from PyQt5.QtCore import *
    from PyQt5.QtGui import *
    from PyQt5.QtWidgets import *
elif PyQtModule == "PySide2":
    from PySide2.QtCore import *
    from PySide2.QtGui import *
    from PySide2.QtWidgets import *
elif PyQtModule == "PyQt4":
    from PyQt4.QtCore import *
    from PyQt4.QtGui import *
elif PyQtModule == "PySide":
    from PySide.QtCore import *
    from PySide.QtGui import *
else:
    print("import error: could not import PyQt or PySide!")
    exit()

import QtVar

if getPyQtModule.UiCode == "Qt4":
    import createSetFieldsDictQt4 as createSetFieldsDictQt
elif getPyQtModule.UiCode == "Qt6":
    import createSetFieldsDictQt6 as createSetFieldsDictQt

import pyTreeFoam
import saveGoQtDialog
import inputTextQtDialog
import selectNamesQtDDialog
import QtParts
import copyPasteTextQt
import topoSetEditorCommand

import logFileCreater
logFileCreater.log()


#------------------------
#  setFieldsDialog class
#------------------------
class setFieldsDialog(createSetFieldsDictQt.Ui_MainWindow):
    
    def __init__(self, caseDir, currTime, selRegion, cellSets, fields):
        self.caseDir = caseDir
        self.currTime = currTime
        self.selRegion = selRegion
        self.cellSets = cellSets
        self.fields = fields
        self.colNames = []
        self.rowNames = []
        self.fieldTypes = []
        self.setDataFlag = 0        #dataがセットされると「１」
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        self.clipboardFile = os.getenv("TreeFoamPath") + "/data/clipBoard"
        #QTableWidget関連
        #self.MainWindow = QtGui.QMainWindow()
        self.MainWindow = QMainWindow()
        self.Table = QtParts.Table()
        self.setupUi(self.MainWindow, self.Table)   #Tableを引数として追加
        #window関連
        self.windowTitle = _("setFieldsDict作成")
        self.createEvent()
        #window位置をmousePointerに合わせる
        self.setMousePosition()

    #
    #  main
    #-------
    def main(self):
        """ メインルーチン"""
        self.setWindowTitle()
        #tableData取得
        self.getColRowLabel()
        self.getColRowData()
        self.getFieldTypes()
        #table作成
        self.createTable()
        self.setDataToTable()
        #flagをセット
        self.setDataFlag = 1
        #window表示
        self.MainWindow.show()

    #  setMousePosition
    def setMousePosition(self):
        """ windowの中心位置をmouseに合わせる"""
        #windowの中心位置
        geo = self.MainWindow.geometry()
        xy = (geo.left() + geo.width()//2, geo.top() + geo.height()//2)
        pos = QPoint(xy[0], xy[1])
        #mousePosition
        mPos = QCursor.pos()
        #設定値を取得
        newPos = mPos - pos
        self.MainWindow.move(newPos)
        

    #
    #  createEvent
    #--------------
    def createEvent(self):
        """ 各eventを作成する"""
        self.btn_close.clicked.connect(self.onClose)    #閉じる
        self.btn_loadDict.clicked.connect(self.onLoadDict)  #Dict読み込み
        self.btn_saveDict.clicked.connect(self.onSaveDict)  #Dict保存
        self.btn_saveRun.clicked.connect(self.onSaveRun)   #Dict保存実行
        self.btn_loadCsv.clicked.connect(self.onLoadCsv)    #csv読み込み
        self.btn_saveCsv.clicked.connect(self.onSaveCsv)    #csv保存
        self.btn_addCellSet.clicked.connect(self.onAddCellSet)  #cellSet追加
        self.btn_addField.clicked.connect(self.onAddField)  #field追加
        self.btn_runTopoSet.clicked.connect(self.onRunTopoSet)  #topoSet起動
        self.btn_runGridEditor.clicked.connect(self.onRunGridEditor)    #gridEditor起動
        self.btn_addBox.clicked.connect(self.onAddBox)      #box追加
        self.btn_addCylinder.clicked.connect(self.onAddCylinder)    #cylinder追加
        self.btn_addSphere.clicked.connect(self.onAddSphere)        #sphere追加
        self.btn_addGeomFile.clicked.connect(self.onAddGeomFile)    #geomFile追加
        self.btn_deleteCellSet.clicked.connect(self.onDeleteCellSet)    #cellSet削除
        self.btn_deleteField.clicked.connect(self.onDeleteField)    #field削除
        self.btn_clearCell.clicked.connect(self.onClearCell)    #cellクリア
        self.btn_copy.clicked.connect(self.onCopy)          #コピー
        self.btn_paste.clicked.connect(self.onPaste)        #貼り付け

        #cell内容が変更された時のevent
        self.tableWidget.itemChanged.connect(self.onItemChanged)

        #Ctrl+Cのevent
        self.Table.copySignal.connect(self.onCopy)
        #Crtl+Vのevent
        self.Table.pasteSignal.connect(self.onPaste)
        #<enter> or <return> のevent
        self.Table.enterSignal.connect(self.onEditorClose)

        #マウスenent
        #cellのクリックevent
        #self.tableWidget.itemClicked.connect(self.onItemChanged)
        #colLabelのDクリックevent
        #self.Dialog.connect(self.tableWidget.horizontalHeader(), QtCore.SIGNAL("sectionDoubleClicked(int)"), self.onTest)
        #rowLabelのDクリックevent
        #self.Dialog.connect(self.tableWidget.verticalHeader(), QtCore.SIGNAL("sectionDoubleClicked(int)"), self.onTest)
        #colLabelの右クリックevent
        self.hHeaders = self.tableWidget.horizontalHeader()
        self.hHeaders.setContextMenuPolicy(QtVar.CustomContextMenu)
        self.hHeaders.customContextMenuRequested.connect(self.showPopupColMenu)
        #rowLabelの右クリックevent
        self.vHeaders = self.tableWidget.verticalHeader()
        self.vHeaders.setContextMenuPolicy(QtVar.CustomContextMenu)
        self.vHeaders.customContextMenuRequested.connect(self.showPopupRowMenu)
        #cellの右クリックevent
        self.tableWidget.setContextMenuPolicy(QtVar.CustomContextMenu)
        self.tableWidget.customContextMenuRequested.connect(self.showPopupCellMenu)

        # #Ctrl+Cのevent
        # self.Table.copySignal.connect(self.onCopy)
        # #Crtl+Vのevent
        # self.Table.pasteSignal.connect(self.onPaste)
        # #<enter> or <return> のevent
        # self.Table.enterSignal.connect(self.onEditorClose)
        # #cell内容が変更された時のevent
        # self.tableWidget.itemChanged.connect(self.onItemChanged)


    #
    #  showPopupRowMenu
    #-------------------
    def showPopupRowMenu(self, pos):
        """ rowのpopupMenuを表示する。"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            cellName = _("cellZone追加...")
        else:
            cellName = _("cellSet追加...")
        menu = QMenu()
        #menu.addAction(_("cellSet追加..."), self.onAddCellSet)
        menu.addAction(cellName, self.onAddCellSet)
        menu.addAction(_("box追加"), self.onAddBox)
        menu.addAction(_("cylinder追加"), self.onAddCylinder)
        menu.addAction(_("sphere追加"), self.onAddSphere)
        menu.addAction(_("行(cellSet)削除"), self.onDeleteCellSet)
        try:
            menu.exec_(self.tableWidget.mapToGlobal(pos))
        except:
            menu.exec(self.tableWidget.mapToGlobal(pos))

    #
    #  showPopupColMenu
    #-------------------
    def showPopupColMenu(self, pos):
        """ columnのpopupMenuを表示する"""
        menu = QMenu()
        menu.addAction(_("field追加..."), self.onAddField)
        menu.addAction(_("列(field)削除"), self.onDeleteField)
        try:
            menu.exec(self.tableWidget.mapToGlobal(pos))
        except:
            menu.exec_(self.tableWidget.mapToGlobal(pos))

    #
    #  showPopupCellMenu
    #--------------------
    def showPopupCellMenu(self, pos):
        """ cellのpopupMenuを表示する。"""
        menu = QMenu()
        menu.addAction(_("cellコピー\tCtrl+C"), self.onCopy)
        menu.addAction(_("cell貼り付け\tCtrl+V"), self.onPaste)
        menu.addAction(_("cellクリア"), self.onClearCell)
        try:
            menu.exec(self.tableWidget.mapToGlobal(pos))
        except:
            menu.exec_(self.tableWidget.mapToGlobal(pos))

    #
    #  createTable
    #--------------
    def createTable(self):
        """ row x col のtableを作成する。"""
        self.setDataFlag = 0
        table = QtParts.tableWidget(self.tableWidget)
        table.createTable(self.rowNames, self.colNames)
        self.setDataFlag = 1

    #
    #  setDataToTable
    #-----------------
    def setDataToTable(self):
        """ tableにcolrowDataをセットし、cell幅を調整する。"""
        self.setDataFlag = 0
        #値をセット
        table = QtParts.tableWidget(self.tableWidget)
        for col in range(len(self.colRowData)):
            fieldData = self.colRowData[col]
            for row in range(len(fieldData)):
                cellData = fieldData[row]
                #item = self.tableWidget.item(row, col)
                table.setCellValue(row, col, cellData)
        table.adjustCells()
        self.setDataFlag = 1

    #
    #  close
    #---------
    def close(self):
        """ dialogを閉じる"""
        global app
        app.quit()

    #----------- event handler -----------
    #閉じる
    def onClose(self):       
        #self.close()
        self.closeCheck()
    #Dict読み込み
    def onLoadDict(self):        
        self.loadDict()
    #Dict保存
    def onSaveDict(self):         
        self.saveDict()
    #Dict保存・実行
    def onSaveRun(self):      
        self.saveRun()
    #csv読み込み
    def onLoadCsv(self):
        self.loadCsv()
    #csv保存
    def onSaveCsv(self):
        self.saveCsv()
    #cellSet追加
    def onAddCellSet(self):
        self.addCellSet()
    #field追加
    def onAddField(self):
        self.addField()
    #topoSet追加
    def onRunTopoSet(self):
        self.runTopoSet()
    #gridEditor起動
    def onRunGridEditor(self):
        self.runGridEditor()
    #box追加
    def onAddBox(self):
        self.addBox()
    #cylinder追加
    def onAddCylinder(self):
        self.addCylinder()
    #sphere追加
    def onAddSphere(self):
        self.addSphere()
    #geomFile追加
    def onAddGeomFile(self):
        self.addGeomFile()
    #行削除
    def onDeleteCellSet(self):
        self.deleteCellSet()
    #列削除
    def onDeleteField(self):
        self.deleteField()
    #cellクリア
    def onClearCell(self):
        self.clearCell()
    #コピー
    def onCopy(self):
        self.cellCopy()
    #貼り付け
    def onPaste(self):
        self.cellPaste()
    #cell修正
    def onItemChanged(self, event):
        self.itemChanged()
    #editor終了
    def onEditorClose(self):
        self.editorClose()
    #----------------------------------

    #
    #  closeCheck
    #-------------
    def closeCheck(self):
        if self.windowTitle[0] != "*":
            self.close()
            return
        title     = _("setFieldsDict作成")
        message   = _("cell内容が変更されています。\n  このまま終了すると、変更内容が失われます。")
        buttonsMsg = [_("キャンセル"), _("このまま終了"), _("保存して終了")]
        msgBox = saveGoQtDialog.Ui_Dialog(title, message, buttonsMsg, self.MainWindow)
        msgBox.main()
        button = msgBox.pushButton
        if button == "CANCEL":
            return
        elif button == "GO":
            self.close()
            return
        else:
            self.saveDict()
            self.close()

    #
    #  loadDict
    #-----------
    def loadDict(self):
        """ setFieldsDictを読み込む。
        ”CANCEL"、"GO"、"SAVE" を返す。"""
        stat = self.checkSaveFlagGo()
        if stat == "CANCEL":
            return
        elif stat == "SAVE":
            self.saveDict()
        #Dict読み込み
        self.getColRowLabel()
        self.getColRowData()
        #table再作成
        self.createTable()
        self.setDataToTable()
        self.clearSaveFlag()
        #messageを表示
        title = _(u"Dict読み込み")
        mess = _(u"setFieldsDictを読み込みました")
        print(mess)
        self.okDialog(title, mess)

    #
    #  checkSaveFlagGo
    def checkSaveFlagGo(self):
        """ 処理前にsaveFlagをチェックする
        "CANCEL", "GO", "SAVE"を返す。 """
        #flagを確認
        if self.windowTitle[0] != "*":
            return "GO"
        #修正されている場合、確認dialog表示
        title     = _("setFieldsDict作成")
        message   = _("cell内容が変更されています。\n  このまま処理を継続すると、変更内容が失われます。")
        buttonsMsg = [_("キャンセル"), _("このまま続行"), _("保存して続行")]
        msgBox = saveGoQtDialog.Ui_Dialog(title, message, buttonsMsg, self.MainWindow)
        msgBox.main()
        button = msgBox.pushButton
        return button

    #
    #  saveDict
    #------------
    def saveDict(self):
        """ cellDataをDictFileに保存する"""
        #cellDataを取得
        table = QtParts.tableWidget(self.tableWidget)
        saveData = []
        col = 0
        while col < self.tableWidget.columnCount():
            row = 0
            fieldData = []
            while row < self.tableWidget.rowCount():
                cellValue = table.getCellValue(row, col)
                fieldData.append(cellValue)
                row += 1
            saveData.append(fieldData)
            col += 1
        self.colRowData = saveData
        #データをDictに保存
        error = self.saveDictData()
        if error[0] == "OK":
            #saveFlagをクリア
            self.clearSaveFlag()
            #message表示
            title = _(u"Dictへ保存")
            mess = _(u"cellDataをsetFieldsDictに保存しました")
            print(mess)
            self.okDialog(title, mess)
        else:
            title = _("エラー")
            mess = error[1]
            print(mess)
            self.errDialog(title, mess)

    #
    #  saveDictData
    def saveDictData(self):
        """ cellDataをdictに保存"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        fileName = self.caseDir + "/system/" + self.selRegion + "/setFieldsDict"
        if len(glob.glob(fileName)) == 0:
            #Dict作成
            foamContents = pyTreeFoam.getFoamContents(self.caseDir)
            #dictCont = foamContents.defaultSetFieldsDict()
            dictCont = foamContents.defaultSetFieldsDict(OFversion)
            header = foamContents.makeFoamHeaderVersion(OFversion)
            foamFile = foamContents.makeFoamFile("2.0", "ascii", "dictionary",
                            "", "setFieldsDict")
            footer = foamContents.makeFoamFooter()
            cont = header + foamFile + dictCont + footer
            f = open(fileName, "w"); f.write(cont); f.close()
        f = open(fileName); cont = f.read(); f.close()
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        if numVer >= 13.0:
            #OF-13以上の場合
            (cont, p) = self.setDefaultFieldValuesToDict13(cont, p)
            (cont, p) = self.setRegionsToDict13(cont, p)
        else:
            #OF-12以下、v、extの場合
            (cont, p) = self.setDefaultFieldValuesToDict(cont, p)
            (cont, p) = self.setRegionsToDict(cont, p)
        f = open(fileName, "w"); f.write(cont); f.close()
        return ("OK", "")

    #
    #  setDefaultFieldValuesToDict13
    def setDefaultFieldValuesToDict13(self, cont, p):
        """ OF-13用のdefaultFieldValueをDictにセット"""
        contOp = pyTreeFoam.strOp(cont)
        (keyword, p) = contOp.getKeyword(p)
        while keyword != "" and keyword != "defaultValues":
            (keyword, p) = contOp.getKeyword(p)
        if keyword == "defaultValues":
            ps = p
            (valCont, p) = contOp.getMiddlePair(p)
            pe = p
            fieldValCont = ""
            row = 0
            col = 1
            while col < len(self.colNames):
                val = self.colRowData[col][row]
                if val != "":
                    field = self.colNames[col]
                    fieldVal = " "*4 + " ".join([field, val]) + ";\n"
                    fieldValCont += fieldVal
                col += 1
            valCont = "\n{\n" + fieldValCont + "}"
            newCont = cont[:ps] + valCont + cont[pe:]
            pp = len(cont[:ps] + valCont)
            return (newCont, pp)
        else:
            return (cont, p)

    #
    #  setDefaultFieldValuesToDict
    def setDefaultFieldValuesToDict(self, cont, p):
        """ defaultFieldValuesをDictにセット"""
        contOp = pyTreeFoam.strOp(cont)
        (keyword, p) = contOp.getKeyword(p)
        while keyword != "" and keyword != "defaultFieldValues":
            (keyword, p) = contOp.getKeyword(p)
        if keyword == "defaultFieldValues":
            ps = p
            (valCont, p) = contOp.getSmallPair(p)
            pe = p
            fieldValCont = ""
            row = 0
            col = 1
            while col < len(self.colNames):
                val = self.colRowData[col][row]
                if val != "":
                    field = self.colNames[col]
                    fieldType = self.fieldTypes[col] + "Value"
                    fieldVal = " "*4 + " ".join([fieldType, field, val]) + "\n"
                    fieldValCont += fieldVal
                col += 1
            valCont = "\n(\n" + fieldValCont + ")"
            newCont = cont[:ps] + valCont + cont[pe:]
            pp = len(cont[:ps] + valCont)
            return (newCont, pp)
        else:
            return (cont, p)

    #
    #  setRegionsToDict13
    def setRegionsToDict13(self, cont, p):
        """ OF-13用のregionsをDictにセット"""
        contOp = pyTreeFoam.strOp(cont)
        (keyword, p) = contOp.getKeyword(p)
        while keyword != "" and keyword != "zones":
            (keyword, p) = contOp.getKeyword(p)
        if keyword == "zones":
            ps = p
            (valCont, p) = contOp.getMiddlePair(p)
            pe = p
            setValCont = ""
            row = 1
            nBoxes = 0
            while row < len(self.rowNames):
                col = 1
                setName = self.rowNames[row]
                if (setName == "(box)" or setName == "(cylinder)" or setName == "(sphere)"):
                    setName = setName[1:-1]
                    #setName = setName[1:-1] + str(nBoxes) + "_"
                    nBoxes += 1
                    cellCont  = " "*4 + setName + str(nBoxes) + "_ToCell\n"
                    cellCont += " "*4 + "{\n"
                    cellCont += " "*8 + "type       " + setName + ";\n"
                    cellCont += " "*8 + "zoneType   cell;\n"
                    #cellCont += " "*8 + "box        " + self.colRow[0][row] + ";\n"
                    lines = self.colRowData[0][row].split("\n")
                    for line in lines:
                        cellCont += " "*8 + line + "\n"
                elif setName == "(geomFile)":
                    setName = setName[1:-1]
                    nBoxes += 1
                    cellCont  = " "*4 + setName + str(nBoxes) + "_ToCell\n"
                    cellCont += " "*4 + "{\n"
                    cellCont += " "*8 + "type       insideSurface;\n"
                    cellCont += " "*8 + "zoneType   cell;\n"
                    cellCont += " "*8 + "surface    closedTriSurface;\n"
                    cellCont += " "*8 + "file       " + self.colRowData[0][row] + ";\n"
                else:
                    cellCont = " "*4 + setName + "\n"
                    cellCont += " "*4 + "{\n"
                    cellCont += " "*8 + "type       lookup;\n"
                    cellCont += " "*8 + "zoneType   cell;\n"
                cellCont += " "*8 + "values\n"
                cellCont += " "*8 + "{\n"
                fieldVal = ""
                while col < len(self.colNames):
                    val = self.colRowData[col][row]
                    if val != "":
                        field = self.colNames[col]
                        fieldVal = " "*12 + " ".join([field, val]) + ";\n"
                        cellCont += fieldVal
                    col += 1
                cellCont += " "*8 + "}\n"
                cellCont += " "*4 + "}\n"
                row += 1
                setValCont += cellCont
            valCont = "\n{\n" + setValCont + "}"
            newCont = cont[:ps] + valCont + cont[pe:]
            pp = len(cont[:ps] + valCont)
            return (newCont, pp)
        else:
            return (cont, p)

    #
    #  setRegionsToDict
    def setRegionsToDict(self, cont, p):
        """ regionsをDictにセット"""
        contOp = pyTreeFoam.strOp(cont)
        (keyword, p) = contOp.getKeyword(p)
        while keyword != "" and keyword != "regions":
            (keyword, p) = contOp.getKeyword(p)
        if keyword == "regions":
            ps = p
            (valCont, p) = contOp.getSmallPair(p)
            pe = p
            setValCont = ""
            row = 1
            while row < len(self.rowNames):
                col = 1
                setName = self.rowNames[row]
                if (setName == "(box)" or setName == "(cylinder)" or setName == "(sphere)"):
                    setName = setName[1:-1]
                    cellCont = " "*4 + setName + "ToCell\n"
                    cellCont += " "*4 + "{\n"
                    lines = self.colRowData[0][row].split("\n")
                    for line in lines:
                        cellCont += " "*8 + line + "\n"
                else:
                    cellCont = " "*4 + "cellToCell\n"
                    cellCont += " "*4 + "{\n"
                    cellCont += " "*8 + "set " + setName + ";\n"
                cellCont += " "*8 + "fieldValues\n"
                cellCont += " "*8 + "(\n"
                fieldVal = ""
                while col < len(self.colNames):
                    val = self.colRowData[col][row]
                    if val != "":
                        field = self.colNames[col]
                        fieldType = self.fieldTypes[col] + "Value"
                        fieldVal = " "*12 + " ".join([fieldType, field, val]) + "\n"
                        cellCont += fieldVal
                    col += 1
                cellCont += " "*8 + ");\n"
                cellCont += " "*4 + "}\n"
                row += 1
                setValCont += cellCont
            valCont = "\n(\n" + setValCont + ")"
            newCont = cont[:ps] + valCont + cont[pe:]
            pp = len(cont[:ps] + valCont)
            return (newCont, pp)
        else:
            return (cont, p)

    #
    #  saveRun
    #-----------
    def saveRun(self):
        """ Dict保存して、setFieldsを実行する"""
        print(_("setFieldsDictを作成します..."))
        table = QtParts.tableWidget(self.tableWidget)
        #cellDataを取得
        saveData = []
        col = 0
        while col < self.tableWidget.columnCount():
            row = 0
            fieldData = []
            while row < self.tableWidget.rowCount():
                #cellValue = self.tableWidget.item(row, col).text()
                cellValue = table.getCellValue(row, col)
                fieldData.append(cellValue)
                row += 1
            saveData.append(fieldData)
            col += 1
        self.colRowData = saveData
        #データをDictに保存
        error = self.saveDictData()
        if error[0] != "OK":
            title = _("エラー")
            mess = error[1]
            print(mess)
            self.errDialog(title, mess)
            return
        #saveFlagをクリア
        self.clearSaveFlag()
        #setFields実行
        print(_("setFieldsを実行します..."))
        bashrcFOAM = self.configDict["bashrcFOAM"]
        comm = ". " + bashrcFOAM + "; "
        comm += "setFields -region " + self.selRegion
        (stat, _res, _err) = pyTreeFoam.run(self.caseDir).commandReturnCont(comm)
        if stat == "OK":
            title = _(u"setFields実行")
            mess = _(u"setFieldsを実行して、dataをfieldにセットしました。")
            self.okDialog(title, mess)
        else:
            title = _(u"エラー")
            mess  = _(u"setFields実行時に、")
            mess += _(u"エラーが発生しました。\n  logを確認してください。")
            self.errDialog(title, mess)
        return

    #
    #  loadCsv
    #----------
    def loadCsv(self):
        """ csvデータを読み込む"""
        stat = self.checkSaveFlagGo()
        if stat == "CANCEL":
            return
        elif stat == "SAVE":
            self.saveDict()
        title = _(u"csvデータ読み込み")
        mess = _(u"setFieldsDict用のcsvファイルを読み込みます。\n")
        mess += _(u"読み込むcsvファイルを選択してください。")
        fileFilter = "*.csv"
        fileDialog = QFileDialog()
        path = self.caseDir
        try:
            (fileName, _filter) = QFileDialog.getOpenFileNameAndFilter(fileDialog, title, path, fileFilter)
        except:
            (fileName, _filter) = QFileDialog.getOpenFileName(fileDialog, title, path, fileFilter)
        if fileName == "":
            return
        #csv読み込み
        csvFileName = fileName
        rowCols = pyTreeFoam.readCsvData(csvFileName)
        colNames = []
        rowNames = []
        colRowData = []
        #title行の読み込み
        colNames = rowCols[1][1:]
        colRowData = [[] for i in range(len(colNames))]
        #rowNameとcolRowDataを取得
        for i in range(2, len(rowCols)):
            rowName = rowCols[i][0]
            rowNames.append(rowName)
            for col in range(1, len(rowCols[i])):
                colData = rowCols[i][col]
                colRowData[col-1].append(colData)
        self.colNames = colNames
        self.rowNames = rowNames
        self.colRowData = colRowData
        #table再作成
        self.createTable()
        self.setDataToTable()
        self.clearSaveFlag()
        #messageを表示
        title = _("csvファイル読み込み")
        mess = _("csvFileを読み込みました。")
        print(mess)
        self.okDialog(title, mess)

    #
    #  saveCsv
    #----------
    def saveCsv(self):
        """ csvとして保存する"""
        #file名の取得
        title = _("file名の入力")
        mess = _("csvFileのファイル名を入力してください")
        dialog = inputTextQtDialog.Ui_Dialog(title, mess, "", self.MainWindow)
        name = dialog.main()
        if name == "":
            return
        if name.split(".")[-1] != "csv":
            name += ".csv"
        csvFileName = self.caseDir + "/" + name
        if len(glob.glob(csvFileName)) != 0:
            title = _(u"csvファイルの作成")
            mess = _(u"既に同じファイルが存在します。")
            mess += _(u"上書きされます。")
            stat = self.okCancelDialog(title, mess)
            if stat != "OK":
                return
        #表中のdataをcolRowDataに反映させる
        table = QtParts.tableWidget(self.tableWidget)
        colRowData = []
        col = 0
        while col < self.tableWidget.columnCount():
            row = 0
            fieldData = []
            while row < self.tableWidget.rowCount():
                cellValue = table.getCellValue(row, col)
                fieldData.append(cellValue)
                row += 1
            colRowData.append(fieldData)
            col += 1
        self.colRowData = colRowData
        #csvファイル作成
        error = self.createCsvSaveData(csvFileName)
        if error[0] == "OK":
            #saveFlagをクリア
            self.clearSaveFlag()
            #message表示
            title = _(u"csv保存")
            mess = _(u"csv形式で保存しました。")
            self.okDialog(title, mess)
        else:
            title = _(u"エラー")
            mess = error[1]
            self.errDialog(title, mess)
        return

    #
    #  createCsvSaveData
    def createCsvSaveData(self, fileName):
        """ csvファイル作成、保存"""
        lines = [["<setFieldsDict>"]]
        #タイトル行
        line = ["items"]
        for item in self.colNames:
            line.append('"' + item + '"')
        lines.append(line)
        #データ行
        row = 0
        for rowName in self.rowNames:
            line = ['"'+rowName+'"', '"'+self.colRowData[0][row]+'"']
            col = 1
            while col < len(self.colNames):
                colRow = self.colRowData[col][row]
                line.append(colRow)
                col += 1
            lines.append(line)
            row += 1
        #saveCsv
        f = open(fileName, "w")
        for line in lines:
            line = ",".join(line) + "\n"
            f.write(line)
        f.close()
        return ("OK", "")

    #
    #  addCellSet
    #-------------
    def addCellSet(self):
        """ 表にcellSetを追加する"""
        #選択できるcellSetsを取得
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            title = _("cellZone追加")
            mess = _("追加するcellZoneを選択してください")
            subMess = _("選択できるcellZone")
            allCellSets = self.getCellZoneNames()
        else:
            title = _("cellSet追加")
            mess = _("追加するcellSetを選択してください")
            subMess = _("選択できるcellSet")
            allCellSets = self.getCellSetNames()
        selCellSets = list(set(allCellSets).intersection(self.rowNames))
        remCellSets = list(set(allCellSets).difference(selCellSets))
        remCellSets.sort()
        #追加するcellSetsを取得
        # title = _("cellSet追加")
        # mess = _("追加するcellSetを選択してください")
        # subMess = _("選択できるcellSet")
        dialog = selectNamesQtDDialog.Ui_Dialog(title, mess, subMess, remCellSets, self.MainWindow)
        selNames = dialog.main()
        if len(selNames) == 0:
            return
        self.rowNames += selNames
        col = 0
        while col < len(self.colNames):
            cols = []
            for _name in selNames:
                cols.append("")
            self.colRowData[col] += cols
            col += 1
        #table再作成
        self.createTable()
        self.setDataToTable()
        self.clearSaveFlag()
        #saveFlagをセット
        self.setSaveFlag()        

    #
    #  getCellZoneNames
    def getCellZoneNames(self):
        """ 定義されているcellZoneを取得する"""
        cellZones = []
        case = pyTreeFoam.case(self.caseDir)
        relativeDir = case.getCurrMeshDir(self.currTime, self.selRegion, "sets")
        polyMeshDir = self.caseDir + "/" + relativeDir
        cellZones = pyTreeFoam.getFoamContents(self.caseDir).cellZoneNames(polyMeshDir)
        return cellZones

    #
    #  getCellSetNames
    def getCellSetNames(self):
        """ 定義されているcellSetを取得する"""
        cellSets = []
        case = pyTreeFoam.case(self.caseDir)
        relativeDir = case.getCurrMeshDir(self.currTime, self.selRegion, "sets")
        polyMeshDir = self.caseDir + "/" + relativeDir
        wkDir = polyMeshDir + "/sets"
        files = pyTreeFoam.getFileNames(wkDir)
        if len(files) == 0:
            return cellSets
        for fileName in files:
            foamFile = pyTreeFoam.foamFile()
            cont = foamFile.read(wkDir + "/" + fileName)
            className = foamFile.getClassInFoamFile(cont)
            className = className.decode()
            if className == "cellSet":
                cellSets.append(fileName)
        cellSets.sort()
        return cellSets

    #
    #  addField
    #------------
    def addField(self):
        """ fieldを追加する"""
        #選択できるfieldを追加
        allFields = pyTreeFoam.case(self.caseDir).getFieldNames(self.currTime, self.selRegion)
        selFields = self.colNames[1:]
        remFields = list(set(allFields).difference(selFields))
        remFields.sort()
        #追加するfieldを取得
        title = _(u"field追加")
        mess = _(u"追加するFieldを選択してください。\n複数選択が可能。")
        subMess = _("選択できるfield")
        dialog = selectNamesQtDDialog.Ui_Dialog(title, mess, subMess, remFields, self.MainWindow)
        selNames = dialog.main()
        if len(selNames) == 0:
            return
        #表の変数を設定
        self.colNames += selNames
        for _field in selNames:
            #空のcolsを作成
            cols = [ "" for row in range(len(self.rowNames))]
            #空のcolsを追加
            self.colRowData.append(cols)
        #fieldTypeを取得
        self.getFieldTypes()
        #table再作成
        self.createTable()
        self.setDataToTable()
        self.clearSaveFlag()
        #saveFlagをセット
        self.setSaveFlag()        

    #
    #  runTopoSet
    #-------------
    def runTopoSet(self):
        """ toposetEditorの起動"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            title = _("エラー")
            mess = _("「OpenFOAM-13」では、topoSetEditorが使えません。")
            self.errDialog(title, mess)
            return
        commPath = os.getenv("TreeFoamPath") + "/python"
        comm = commPath + "/topoSetEditorDialog.py " + self.caseDir
        comm += " --stl ./model" + " --time " + self.currTime
        comm += " --rigion " + self.selRegion + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  runGridEditor
    #----------------
    def runGridEditor(self):
        """ gridEditorの起動"""
        if len(self.colNames) < 2:
            return
        #gridEditorの起動
        bashrc = pyTreeFoam.readConfigTreeFoam()["bashrcFOAM"]
        nTreeFoam = "-1"
        caseDir = self.caseDir
        currTime = self.currTime
        commPath = os.getenv("TreeFoamPath") + "/python"
        comm = ". " + bashrc + ";"
        comm += commPath + "/gridEditorQtDialog.py " + nTreeFoam + " " + caseDir
        comm += " " + currTime
        pyTreeFoam.run(caseDir).command(comm)

    #
    #  addBox
    #--------
    def addBox(self):
        """ box追加"""
        OFversion = self.configDict["OFversion"]
        verNum = pyTreeFoam.getNumVersion(OFversion)
        if verNum >= 13.0:
            boxCont = "box (0 0 0) (1 1 1);"
        else:
            makeRunTopoSet = self.getTopoSetDB()
            boxCont = self.getBoxCont(makeRunTopoSet, "box")
        name = "(box)"
        self.addGeometry(name, boxCont)

    #
    #  getTopoSetDB
    def getTopoSetDB(self):
        """ OFversionに応じたtopoSetDBを取得する"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion >= "v1812":
            #v1812以上のOF
            topoSetDict = pyTreeFoam.getFoamContents().defaultDictPathInDataFolder(OFversion, "system/topoSetSourcesDict")
            makeRunTopoSet = topoSetEditorCommand.makeTopoSet1812(topoSetDict)
        elif numVer >= 10.0:
            #OF-10以上
            topoSetDict = pyTreeFoam.getFoamContents().defaultDictPathInDataFolder(OFversion, "system/topoSetDict")
            makeRunTopoSet = topoSetEditorCommand.makeTopoSet_10(topoSetDict)
        else:
            #OF-9以下、v1806以下
            topoSetDict = pyTreeFoam.getFoamContents().defaultDictPathInDataFolder(OFversion, "system/topoSetDict")
            makeRunTopoSet = topoSetEditorCommand.makeTopoSet(topoSetDict)
        return makeRunTopoSet

    #
    #  getBoxCont
    def getBoxCont(self, makeRunTopoSet, sourceType):
        """ topoSetDict内のsourceInfoからboxContを取得する"""
        #                                  sourceType, sourceName, action, resultType
        sourceInfo = makeRunTopoSet.getSourceInfo(sourceType, "", "new", "cellSet")
        if sourceInfo.find("{") > 0:
            (boxCont, p) = pyTreeFoam.strOp(sourceInfo).getMiddlePair(0)
            boxCont = pyTreeFoam.strOp(boxCont).deleteNullLines()
        else:
            (_lineCont, p, _kind) = pyTreeFoam.strOp(sourceInfo).get1line(0)
            boxCont = pyTreeFoam.strOp(sourceInfo[p:]).deleteNullLines()
        boxCont = pyTreeFoam.strOp(boxCont).compressSPCR()
        return boxCont

    #
    #  addGeometry
    def addGeometry(self, name, cont):
        """ geometryを追加"""
        self.rowNames.append(name)
        row = len(self.rowNames) - 1
        col = 0
        while col < len(self.colNames):
            self.colRowData[col].append("")
            col += 1
        self.colRowData[0][row] = cont
        #table再作成
        self.createTable()
        self.setDataToTable()
        self.clearSaveFlag()
        #saveFlagをセット
        self.setSaveFlag()        

    #
    #  addCylinder
    #--------------
    def addCylinder(self):
        """ cylinder追加"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            cylinderCont  = "point1 (0 0 0);\n"
            cylinderCont += "point2 (1 0 0);\n"
            cylinderCont += "radius 0.5;"
        else:
            makeRunTopoSet = self.getTopoSetDB()
            cylinderCont = self.getBoxCont(makeRunTopoSet, "cylinder")
        name = "(cylinder)"
        self.addGeometry(name, cylinderCont)

    #
    #  addSphere
    #------------
    def addSphere(self):
        """ sphere追加"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            sphereCont  = "centre (0 0 0);\n"
            sphereCont += "radius 0.5;"
        else:
            makeRunTopoSet = self.getTopoSetDB()
            sphereCont = self.getBoxCont(makeRunTopoSet, "sphere")
        name = "(sphere)"
        self.addGeometry(name, sphereCont)
 
    #
    #  addGeomFile
    #--------------
    def addGeomFile(self):
        """ geomFile追加"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            name = "(geomFile)"
            geomFile = '"${FOAM_CASE}/model/geometry.stl"'
            self.addGeometry(name, geomFile)
        else:
            title = _("エラー")
            mess = "OpenFOAM-" + OFversion + " " + _("では、geometryFileは使えません。")
            self.errDialog(title, mess)

    #
    #  deleteCellSet
    #-----------------
    def deleteCellSet(self):
        """ 行（cellSet）を削除する。"""
        indexes = self.tableWidget.selectionModel().selectedRows()
        delRows = list(map(lambda x: x.row(), indexes))
        if 0 in delRows:
            title = _(u"エラー")
            mess = _(u"最初の行は、削除できません。")
            self.errDialog(title, mess)
            return
        elif len(delRows) == 0:
            title = _(u"行削除")
            mess = _(u"削除する行が選択されていません。\n")
            mess += _(u"行を選択してください。")
            self.errDialog(title, mess)
            return
        #rowLabel削除
        newRowNames = []
        for i in range(len(self.rowNames)):
            if not(i in delRows):
                newRowNames.append(self.rowNames[i])
        #colRowData削除
        newColRowData = [ [] for i in range(len(self.colNames)) ]
        for col in range(len(self.colRowData)):
            for i in range(len(self.colRowData[col])):
                if not(i in delRows):
                    newColRowData[col].append(self.colRowData[col][i])
        self.rowNames = newRowNames
        self.colRowData = newColRowData
        #table再作成
        self.createTable()
        self.setDataToTable()
        self.clearSaveFlag()
        #saveFlagをセット
        self.setSaveFlag()        
        
    #
    #  deleteField
    #--------------
    def deleteField(self):
        """ 列（field）削除"""
        indexes = self.tableWidget.selectionModel().selectedColumns()
        delCols = list(map(lambda x: x.column(), indexes))
        if 0 in delCols:
            title = _(u"エラー")
            mess = _(u"最初の列は、削除できません。")
            self.errDialog(title, mess)
            return
        elif len(delCols) == 0:
            title = _(u"列削除")
            mess = _(u"削除する列が選択されていません。\n")
            mess += _(u"列を選択してください。")
            self.errDialog(title, mess)
            return
        #colLabel削除
        newColNames = []
        newColRowData = []
        for i in range(len(self.colNames)):
            if not(i in delCols):
                newColNames.append(self.colNames[i])
                newColRowData.append(self.colRowData[i])
        self.colNames = newColNames
        self.colRowData = newColRowData
        #table再作成
        self.createTable()
        self.setDataToTable()
        self.clearSaveFlag()
        #saveFlagをセット
        self.setSaveFlag()        

    #
    #  clearCell
    #------------
    def clearCell(self):
        """ 選択cellをクリアする。"""
        table = QtParts.tableWidget(self.tableWidget)
        cells = table.selectedCells()
        if len(cells) == 0:
            cell = table.currentCell()
            cells.append(cell)
        else:
            for row, col in cells:
                table.setCellValue(row, col, "")

    #
    #  cellCopy
    #-----------
    def cellCopy(self):
        """ 選択cellをclipboardにcopyする"""
        #clipboardFile = self.clipboardFile
        #copyTextを取得
        table = QtParts.tableWidget(self.tableWidget)
        copyText = table.getCopyText()
        copyPasteTextQt.copyText(copyText)
        #copyText = tableOperation(self.tableWidget).getCopyText()
        #clipboardに貼り付け
        # f = open(clipboardFile, "w"); f.write(copyText); f.close()
        # path = os.getenv("TreeFoamPath") + "/python"
        # comm = "./copyTextToSystemClipBoard.py " + clipboardFile
        # pyTreeFoam.run(path).command(comm)

    #
    #  cellPaste
    #------------
    def cellPaste(self):
        """ clipboard内のdataをcellに貼り付ける"""
        #clipboardFile = self.clipboardFile
        #clipboardの内容を取得する。
        cont = copyPasteTextQt.pasteText()
        # commPath = os.getenv("TreeFoamPath") + "/python"
        # comm = commPath + "/pasteTextFromSystemClipBoard.py " + clipboardFile
        # pyTreeFoam.run().command(comm)
        # f = open(clipboardFile); cont = f.read(); f.close()
        #tableに貼り付ける
        table = QtParts.tableWidget(self.tableWidget)
        table.setPasteText(cont)
        #tableOperation(self.tableWidget).setPasteText(cont)

    #
    #  itemChanged
    #--------------
    def itemChanged(self):
        """ 内容が修正された場合のflagをセット。
        Dialogのtitleを修正"""
        if self.setDataFlag == 0:
            return
        self.setSaveFlag()
        table = QtParts.tableWidget(self.tableWidget)
        table.adjustCells()

    #
    #  editorClose
    #--------------
    def editorClose(self):
        """ cellに組み込まれたtextEditorが終了した時の処理。
        編集後のtextをcellに挿入し、editorを閉じる"""
        #編集itemを取得
        items = self.Table.selectedItems()
        self.Table.closePersistentEditor(items[0])
        index = self.Table.indexFromItem(items[0])
        #編集したtextを取得
        newText = self.Table.newText
        #textをitemに設定
        if newText != None:
            row = index.row()
            col = index.column()
            QtParts.tableWidget(self.tableWidget).setCellValue(row, col, newText)
            #self.Table.item(index.row(), index.column()).setText(newText)
        self.Table.newText = None

    #
    #  setSaveFlag
    #--------------
    def setSaveFlag(self):
        #titleを修正
        self.windowTitle = _("*setFieldsDict作成")
        self.setWindowTitle()

    #
    #  clearSaveFlag
    #----------------
    def clearSaveFlag(self):
        """ saveFlagをクリアする"""
        #titleを修正
        self.windowTitle = _("setFieldsDict作成")
        self.setWindowTitle()



    #------------- 初期化関連 -------------

    #
    #  setWindowTitle
    #-----------------
    def setWindowTitle(self):
        self.MainWindow.setWindowTitle(self.windowTitle)

    #
    #  getColRowLabel
    #-----------------
    def getColRowLabel(self):
        """ tableのcolRowのlabelを取得する"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            self.rowNames = ["defaultValues"] + self.cellSets
            self.colNames = ["geometry\ndata"] + self.fields
        else:
            self.rowNames = ["defaultFieldValues"] + self.cellSets
            self.colNames = ["geometry\ndata"] + self.fields

    #
    #  getColRowData
    #---------------
    def getColRowData(self):
        """ tableの各cellにセットするdataをsetFieldsDictから取得"""
        
        def setDefaultValues(fields, values):
            """ defaultValueをcolRowDataにセット"""
            col = 0
            while col < len(self.colNames):
                i = 0
                val = ""
                for field in fields:
                    if self.colNames[col] == field:
                        val = values[i]
                        break
                    i += 1
                self.colRowData[col][0] = val
                col += 1
        
        def setCellToCellValues(cellSet, fields, values):
            """ cellToCellValueをcolRowDataにセット"""
            n = -1
            i = 1
            while i < len(self.rowNames):
                if self.rowNames[i] == cellSet:
                    n = i
                    break
                i += 1
            if n >= 0:
                i = 1
                while i < len(self.colNames):
                    f = 0
                    for field in fields:
                        if field == self.colNames[i]:
                            self.colRowData[i][n] = values[f]
                            break
                        f += 1
                    i += 1

        def setGeoDataValues(keyword, geoCont, fields, values):
            """ geometryのdataをセット"""
            name = "(" + keyword[:-len("ToCell")] + ")"
            nameCont = geoCont
            self.rowNames.append(name)
            self.colRowData[0].append(nameCont)
            i = 1
            while i < len(self.colNames):
                self.colRowData[i].append("")
                i += 1
            col = 1
            while col < len(self.colNames):
                i = 0
                while i < len(fields):
                    if self.colNames[col] == fields[i]:
                        self.colRowData[col][-1] = values[i]
                        break
                    i += 1
                col += 1

        def getFieldsValues(linesCont):
            """ fieldとvalueを取得"""
            contOp = pyTreeFoam.strOp(linesCont)
            contents, p = contOp.getKeywordContents("values", 0)
            contentsOp = pyTreeFoam.strOp(contents)
            fields = []; vals = []
            lineCont, p, kind = contentsOp.get1line(0)
            while lineCont != "":
                if kind == "line":
                    words = lineCont.split()
                    words = self.mergeWords(words)
                    field = words[0]
                    fields.append(field)
                    val = words[1]
                    if val[-1] == ";":
                        val = val[:-1]
                    vals.append(val)
                lineCont, p, kind = contentsOp.get1line(p)
            return fields, vals

        def getKeywordContents(linesCont, wordType):
            contOp = pyTreeFoam.strOp(linesCont)
            lineCont, p, kind = contOp.get1line(0)
            typeConts = ""
            if wordType == "box":
                while lineCont != "":
                    if kind == "line":
                        words = lineCont.split()
                        if words[0] == "box":
                            typeConts = " ".join(lineCont.split())
                            break
                    lineCont, p, kind = contOp.get1line(p)
            elif wordType == "cylinder":
                while lineCont != "":
                    if kind == "line":
                        words = lineCont.split()
                        if words[0] == "point1" or words[0] == "point2" or words[0] == "radius":
                            typeConts += " ".join(words) + "\n"
                    lineCont, p, kind = contOp.get1line(p)
            elif wordType == "sphere":
                while lineCont != "":
                    if kind == "line":
                        words = lineCont.split()
                        if words[0] == "centre" or words[0] == "radius":
                            typeConts += " ".join(words) + "\n"
                    lineCont, p, kind = contOp.get1line(p)
            elif wordType == "insideSurface":
                while lineCont != "":
                    if kind == "line":
                        words = lineCont.split()
                        if words[0] == "file":
                            typeConts = words[1]
                            if typeConts[-1] == ";":
                                typeConts = typeConts[:-1]
                            break
                    lineCont, p, kind = contOp.get1line(p)
            if typeConts[-1] == "\n":
                typeConts = typeConts[:-1]
            return typeConts

        nCol = len(self.colNames)
        nRow = len(self.rowNames)
        self.colRowData = [ [ "" for i in range(nRow)] for ii in range(nCol)]
        fileName = self.caseDir + "/system/" + self.selRegion + "/setFieldsDict"
        if len(glob.glob(fileName)) == 0:
            return
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        #OF-13？
        if numVer >= 13.0:
            #OF-13以上の場合
            f = open(fileName); cont = f.read(); f.close()
            contOp = pyTreeFoam.strOp(cont)
            p = contOp.skipFoamFile()
            #defaultFieldValueを取得
            valCont, p = contOp.getKeywordContents("defaultValues", p)
            (fields, values) = self.getDefaultFieldValues13(valCont)
            setDefaultValues(fields, values)
            #cellZone or boxを取得
            zones, p = contOp.getKeywordContents("zones", p)
            zonesOp = pyTreeFoam.strOp(zones)
            (zonesName, p) = zonesOp.getKeyword(0)
            while zonesName != "":
                lines, p = zonesOp.getMiddlePair(p)
                linesOp = pyTreeFoam.strOp(lines)
                lineCont, pp, kind = linesOp.get1line(0)
                while lineCont != "":
                    keyword, _ppp = pyTreeFoam.strOp(lineCont).getKeyword(0)
                    if keyword == "type":
                        words = lineCont.split()
                        wordType = words[1]
                        if wordType[-1] == ";":
                            wordType = wordType[:-1]
                        if wordType == "lookup":
                            #cellZone
                            cellZoneName = zonesName
                            fields, values = getFieldsValues(lines)
                            setCellToCellValues(cellZoneName, fields, values)
                        elif (wordType == "box" or wordType == "cylinder" or wordType == "sphere"):
                            #box, cylinder or sphere?
                            fields, values = getFieldsValues(lines)
                            typeCont = getKeywordContents(lines, wordType)
                            setGeoDataValues(wordType+"ToCell", typeCont, fields, values)
                        elif wordType == "insideSurface":
                            #geomFile
                            fields, values = getFieldsValues(lines)
                            typeCont = getKeywordContents(lines, wordType)
                            setGeoDataValues("geomFile"+"ToCell", typeCont, fields, values)
                    lineCont, pp, kind = linesOp.get1line(pp) 
                (zonesName, p) = zonesOp.getKeyword(p) 
        else:
            #OF-12以下、v, extの場合
            f = open(fileName); cont = f.read(); f.close()
            contOp = pyTreeFoam.strOp(cont)
            p = contOp.skipFoamFile()
            (keyword, p) = contOp.getKeyword(p)
            while keyword != "":
                if keyword == "defaultFieldValues":
                    (valCont, p) = contOp.getSmallPair(p)
                    (fields, values) = self.getDefaultFieldValues(valCont)
                    setDefaultValues(fields, values)
                elif keyword == "cellToCell":
                    (valCont, p) = contOp.getMiddlePair(p)
                    (cellSet, fields, values) = self.getCellToCellValues(valCont)
                    setCellToCellValues(cellSet, fields, values)
                elif (keyword == "boxToCell" or keyword == "cylinderToCell"
                        or keyword == "sphereToCell"):
                    (valCont, p) = contOp.getMiddlePair(p)
                    (geoCont, fields, values) = self.getBoxToCellValues(valCont)
                    setGeoDataValues(keyword, geoCont, fields, values)
                (keyword, p) = contOp.getKeyword(p)

    def mergeWords(self, words):
        """ ()付きの数値を結合させて1ヶの値にする"""
        merge = []
        flag = 0
        for word in words:
            if (word[0] == "+" or word[0] == "-" or word[0] == "(" or word[0] == ")" or
                (word[0] >= "0" and word[0] <= "9")):
                if flag == 0:
                    merge.append(word)
                    flag = 1
                else:
                    merge[-1] += " " + word
            else:
                merge.append(word)
                flag = 0
        return merge

    #
    #  getDefaultFieldValues
    def getDefaultFieldValues(self, cont):
        """ fieldと値を取得"""
        fields = []
        vals = []
        cont = pyTreeFoam.strOp(cont).deleteComments()
        words = cont.split()
        words = self.mergeWords(words)
        i = 0
        while i < len(words):
            field = words[i+1]
            val = words[i+2]
            fields.append(field)
            vals.append(val)
            i += 3
        return (fields, vals)

    #
    # getDefaultFieldValues13(valCont)
    def getDefaultFieldValues13(self, cont):
        fields = []
        vals = []
        cont = pyTreeFoam.strOp(cont).deleteComments()
        words = cont.split()
        words = self.mergeWords(words)
        i = 0
        while i < len(words):
            field = words[i]
            val = words[i+1]
            if val[-1] == ";":
                val = val[:-1]
            fields.append(field)
            vals.append(val)
            i += 2
        return (fields, vals)

    #
    #  getCellToCellValues
    def getCellToCellValues(self, cont):
        """ cellToCellの内容を取得"""
        fields = []
        vals = []
        cellSet = ""
        contOp = pyTreeFoam.strOp(cont)
        (keyword, p) = contOp.getKeyword(0)
        while keyword != "":
            #cellSet
            if keyword == "set":
                (cellSet, p) = contOp.getKeyword(p)
            #fieldValue
            elif keyword == "fieldValues":
                (valCont, p) = contOp.getSmallPair(p)
                (fields, vals) = self.getDefaultFieldValues(valCont)
            (keyword, p) = contOp.getKeyword(p)
        return (cellSet, fields, vals)

    #
    #  getBoxToCellValues
    def getBoxToCellValues(self, cont):
        """ boxToCellの内容を取得"""
        fields = []
        vals = []
        boxCont = ""
        contOp = pyTreeFoam.strOp(cont)
        (keyword, p) = contOp.getKeyword(0)
        while keyword != "":
            #box
            if keyword != "fieldValues":
                (valCont, p, _kind) = contOp.get1line(p)
                boxCont += keyword + " " + valCont + "\n"
            else:
                #fieldValues
                (valCont, p) = contOp.getSmallPair(p)
                (fields, vals) = self.getDefaultFieldValues(valCont)
            (keyword, p) = contOp.getKeyword(p)
        boxCont = boxCont[:-1]
        return (boxCont, fields, vals)

    #
    #  getFieldTypes
    def getFieldTypes(self):
        """ fieldのtype（class）を取得する"""
        fieldDir = self.caseDir + "/" + self.currTime + "/" + self.selRegion
        self.fieldTypes = [""]
        i = 1
        while i < len(self.colNames):
            field = self.colNames[i]
            cont = pyTreeFoam.foamFile().read(fieldDir + "/" + field)
            typeName = pyTreeFoam.foamFile().getClassInFoamFile(cont)
            typeName = typeName.decode()
            self.fieldTypes.append(typeName)
            i += 1
        return


    def okDialog(self, title, mess):
        msgBox = QMessageBox()
        msgBox.information(self.MainWindow, title, mess, QtVar.OkButton)

    def errDialog(self, title, mess):
        msgBox = QMessageBox()
        msgBox.critical(self.MainWindow, title, mess, QtVar.OkButton)

    def okCancelDialog(self, title, mess):
        msgBox = QMessageBox()
        res = msgBox.information(self.MainWindow, title, mess, QtVar.CancelButton, QtVar.OkButton)
        if res == QtVar.OkButton:
            ans = "OK"
        else:
            ans = "CANCEL"
        return ans

#
#  getArguments
#---------------
def getArguments():
    """ 引数の内容を取得する"""
    caseDir = ""
    fields = []
    cellSets = []
    timeFolder = ""
    region = ""
    flag = ""
    for i in range(len(sys.argv)):
        if sys.argv[i] == "-case":
            flag = "case"
        elif sys.argv[i] == "-f":
            flag = "field"
        elif sys.argv[i] == "-c":
            flag = "cellSet"
        elif sys.argv[i] == "-t":
            flag = "time"
        elif sys.argv[i] == "-r":
            flag = "region"
        else:
            if flag == "case":
                caseDir = sys.argv[i]
            elif flag == "field":
                fields.append(sys.argv[i])
            elif flag == "cellSet":
                cellSets.append(sys.argv[i])
            elif flag == "time":
                timeFolder = sys.argv[i]
            elif flag == "region":
                region = sys.argv[i]
    return (caseDir, timeFolder, region, cellSets, fields)

#
#  showGui
#----------
def showGui(caseDir, currTime, selRegion, cellSets, fields):
    """ QtGUIを表示する"""
    global app
    #app = QtGui.QApplication(sys.argv)
    app = QApplication(sys.argv)
    ui = setFieldsDialog(caseDir, currTime, selRegion, cellSets, fields)
    ui.main()
    try:
        #sys.exit(app.exec_())
        app.exec_()
    except:
        #sys.exit(app.exec())
        app.exec()


if __name__ == "__main__":
    import gettext
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    (caseDir, currTime, selRegion, cellSets, fields) = getArguments()
    showGui(caseDir, currTime, selRegion, cellSets, fields)




