#!/usr/bin/python3
# coding: utf-8
#
#   createCfMeshDict.py
#
#   15/10/01    新規作成
#      10/11    layerの設定に「maxFirstLayerThickness」を追加
#   19/10/23    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#   21/07/29    import editStlFilesDialogをpyTreeFoamに変更
#

import os
import glob
import math

import createBlockAndSnappyDict
import pyTreeFoam
import stringOp


#
#  setCsvData
def setCsvData():
    """ defaultのcsvファイル作成"""
    row = []
    row.append(["", "", "", "", "", "", ""])
    row.append(["<cfMesh>", "", "", "", "", "", ""])
    row.append(["", "maxCellSize", "", "", "", "", _(":最大のcellサイズ")])
    row.append(["", "minCellSize", "", "", "", "", _(":最小のcellサイズ")])
    row.append(["", "featureAngle", "30", "", "", "", _(":特徴線抽出用の角度")])
    row.append(["stlFile", '"sect\n(patch/wall/\nempty/symmetry(Plane)/\nregBox/regSph/face)"', "cellSize", "nLayer", "ratio", "maxThickness", ""])
    return row

#
#  addStlFileName
def addStlFileName(stlDir, data):
    """ stlFileを追加"""
    files = []
    stlFiles = glob.glob(stlDir + "/*.stl")
    for stlFile in stlFiles:
        name = stlFile.split("/")[-1].split(".")[0]
        #assy.stl以外を取得
        if name != "assy":
            files.append(name)
    files.sort()
    for name in files:
        line = [name, "", "", "", "", "", ""]
        data.append(line)
    return data

#
#  getMinMaxValueStl
def getMinMaxValueStl(fileName):
    (mins, maxs) = createBlockAndSnappyDict.getMinMaxValueStl(fileName)
    return (mins, maxs)

#
#  getStlBlockSize
def getStlBlockSize(fileName):
    """ stlのblockサイズを取得"""
    minMax = getMinMaxValueStl(fileName)
    dx = minMax[1][0] - minMax[0][0]
    dy = minMax[1][1] - minMax[0][1]
    dz = minMax[1][2] - minMax[0][2]
    dx = pyTreeFoam.getValue3digit(dx)
    dy = pyTreeFoam.getValue3digit(dy)
    dz = pyTreeFoam.getValue3digit(dz)
    return (dx, dy, dz)

#
#  addMaxCellSize
def addMaxCellSize(stlDir, data):
    """ maxCellSizeの設定（30分割）"""
    flag = 0
    xVals=[]; yVals=[]; zVals=[]
    i = 0
    maxCellSize = -1
    for row in data:
        if flag == 1:
            fileName = stlDir + "/" + row[0] + ".stl"
            [mins, maxs] = getMinMaxValueStl(fileName)
            xVals.append(mins[0])
            xVals.append(maxs[0])
            yVals.append(mins[1])
            yVals.append(maxs[1])
            zVals.append(mins[2])
            zVals.append(maxs[2])
        if row[0] == "stlFile":
            flag = 1
        if row[1] == "maxCellSize":
            maxCellSize = i
        i += 1
    if len(xVals) != 0:
        minx = min(xVals); maxx = max(xVals)
        miny = min(yVals); maxy = max(yVals)
        minz = min(zVals); maxz = max(zVals)
    else:
        minx = 0.0; maxx = 0.0
        miny = 0.0; maxy = 0.0
        minz = 0.0; maxz = 0.0
    #cellSizeの設定
    dxf = (maxx - minx) * 1.0 / 30.0
    dyf = (maxy - miny) * 1.0 / 30.0
    dzf = (maxz - minz) * 1.0 / 30.0
    dx = pyTreeFoam.getValue3digit(dxf)
    dy = pyTreeFoam.getValue3digit(dyf)
    dz = pyTreeFoam.getValue3digit(dzf)
    cellSize = max([dx, dy, dz])
    if maxCellSize >= 0:
        data[maxCellSize][2] = str(cellSize)
    return data

#
#  addStlBlockSize
def addStlBlockSize(stlDir, data):
    """ stl のblockサイズを取得"""
    flag = 0
    i = 0
    for row in data:
        if flag == 1:
            fileName = stlDir + "/" + row[0] + ".stl"
            [x, y, z] = getStlBlockSize(fileName)
            if not (x==0.0 and y==0.0 and z==0.0):
                data[i][6] = '"(' + ' '.join([str(x), str(y), str(z)]) + ')"'
        if row[0] == "stlFile":
            flag = 1
        i += 1
    return data

#
#  writeCsvData
def writeCsvData(caseDir, fileName, data):
    """ csvデータの書き込み"""
    f=open(caseDir+"/"+fileName, "w")
    for line in data:
        cont = ",".join(line) + "\n"
        f.write(cont)
    f.close()
    return

#
#  separateContents
def separateContents(cont):
    """ contの内容をheader, cont, footerに分ける"""
    contOp = stringOp.strings(cont)
    p = contOp.skipFoamFile()
    (lineCont, p, kind) = contOp.get1line(p)
    ps = p - len(lineCont)
    header = cont[:ps]
    #footerを取得
    p = ps
    pe = p
    (lineCont, p, kind) = contOp.get1line(p)
    while lineCont != b"":
        pe = p
        (lineCont, p, kind) = contOp.get1line(p)
    newCont = cont[ps:pe]
    footer = cont[pe:]
    return (header, newCont, footer)

#
#  setOrDeleteLine
def setOrDeleteLine(cont, key, value, newLine):
    """ set or delete line"""
    if value != "":
        contOp = stringOp.strings(cont)
        (cont, p, keyword) = contOp.replace1lineKeyword(0, newLine)
        #置き換えが成功したか？
        if keyword == b"":
            #失敗の場合
            if key == "surfaceFile" or key == "maxCellSize" or key == "minCellSize":
                #先頭行にnewLineを追加
                if newLine[-1] == "\n":
                    cont = newLine + cont
                else:
                    cont = newLine + "\n" + cont
            else:
                #最終行にnewLineを追加
                if cont[-1] == "\n":
                    cont += newLine
                else:
                    cont += "\n" + newLine
    else:
        #key行を削除
        contOp = stringOp.strings(cont)
        (lineCont, p, kind) = contOp.get1line(0)
        (keyword, pp) = stringOp.strings(lineCont).getKeyword(0)
        ps = p
        while keyword != key.encode() and keyword != b"":
            ps = p
            (lineCont, p, kind) = contOp.get1line(p)
            (keyword, pp) = stringOp.strings(lineCont).getKeyword(0)
        if keyword == key.encode():
            cont = cont[:ps] + cont[p:]
    return cont


#
#  setSurfaceFile
def setSurfaceFile(caseDir, stlDir, surfaceFile, cont):
    """ meshDictにsurfaceFile名を設定"""
    absFileName = stlDir + "/" + surfaceFile
    relFileName = os.path.relpath(absFileName, caseDir)
    surfaceFile = relFileName
    if len(relFileName) > len(absFileName):
        surfaceFile = absFileName
    newLine = 'surfaceFile "' + surfaceFile + '";'
    cont = setOrDeleteLine(cont, "surfaceFile", surfaceFile, newLine)
    return cont

#
#  setMaxCellSize
def setMaxCellSize(maxCellSize, cont):
    """ meshDictにmaxCellSizeの設定"""
    newLine = "maxCellSize " + maxCellSize + ";"
    cont = setOrDeleteLine(cont, "maxCellSize", maxCellSize, newLine)
    return cont

#
#  setMinCellSize
def setMinCellSize(minCellSize, cont):
    """ meshDictにminCellSizeを設定"""
    newLine = "minCellSize " + minCellSize + ";"
    cont = setOrDeleteLine(cont, "minCellSize", minCellSize, newLine)
    return cont

#
#  setLocalRefinement
def setLocalRefinement(data, cont):
    """ meshDictにlocalRefinementを設定"""
    newLine  = "localRefinement\n"
    newLine += "{\n"
    for row in data:
        patchName = row[0]
        sect = row[1]
        cellSize = row[2]
        if (sect == "wall" or sect == "patch" or sect == "empty" or
            sect == "symmetry" or sect == "symmetryPlane"):
            try:
                #cellSizeが実数値の場合
                a = float(cellSize)
                newLine += "    " + patchName + "\n"
                newLine += "    {\n"
                newLine += "        cellSize " + cellSize + ";\n"
                newLine += "    }\n"
            except:
                pass
    newLine += "}"
    cont = setOrDeleteLine(cont, "localRefinement", newLine, newLine)
    return cont

#
#  setRenameBoundary
def setRenameBoundary(data, cont):
    """ patchNameとpatchTypeを設定"""
    value = ""
    for row in data:
        patchName = row[0]
        sect = row[1]
        if (sect == "wall" or sect == "patch" or sect == "empty" or
            sect == "symmetry" or sect == "symmetryPlane"):
            value += "        " + patchName + "\n"
            value += "        {\n"
            value += "            newName " + patchName + ";\n"
            value += "            type    " + sect + ";\n"
            value += "        }\n"
    newLine  = "renameBoundary\n"
    newLine += "{\n"
    newLine += "    newPatchNames\n"
    newLine += "    {\n"
    newLine += value
    newLine += "    }\n"
    newLine += "}"
    cont = setOrDeleteLine(cont, "localRefinement", newLine, newLine)
    return cont

#
#  setSurfaceMeshRefinement
def setSurfaceMeshRefinement(caseDir, stlDir, data, cont):
    """ 表面のcellSizeを設定"""
    absDir = stlDir
    relDir = os.path.relpath(absDir, caseDir)
    setDir = relDir
    if len(relDir) > len(absDir):
        setDir = absDir 
    value = ""
    for row in data:
        patchName = row[0]
        sect = row[1]
        cellSize = row[2]
        if sect == "face":
            surfaceFile = patchName + ".stl"
            value += "    " + patchName + "\n"
            value += "    {\n"
            value += '        surfaceFile "' + setDir + "/" + surfaceFile + '";\n'
            value += "        cellSize    " + cellSize + ";\n"
            value += "    }\n"
    newLine  = "surfaceMeshRefinement\n"
    newLine += "{\n"
    newLine += value
    newLine += "}"
    cont = setOrDeleteLine(cont, "surfaceMeshRefinement", newLine, newLine)
    return cont

#
#  setObjectRefinements
def setObjectRefinements(data, stlDir, cont):
    """ objectのcellSizeを設定"""

    #  getRegionSize
    #    regionSizeを取得
    def getRegionSize(stlFile):
        
        def f2str(value):
            val = pyTreeFoam.float2strAuto(value, prec=3)
            return val

        [[minx, miny, minz], [maxx, maxy, maxz]] = getMinMaxValueStl(stlFile)
        dx = maxx - minx
        dy = maxy - miny
        dz = maxz - minz
        locx = (maxx + minx) / 2.0
        locy = (maxy + miny) / 2.0
        locz = (maxz + minz) / 2.0
        centre = [f2str(locx), f2str(locy), f2str(locz)]
        return [centre, f2str(dx), f2str(dy), f2str(dz)]

    value = ""
    for row in data:
        patchName = row[0]
        sect = row[1]
        cellSize = row[2]
        if sect == "regBox":
            stlFile = stlDir + "/" + patchName + ".stl"
            [centre, dx, dy, dz] = getRegionSize(stlFile)
            value += "    " + patchName + "\n"
            value += "    {\n"
            value += "        cellSize " + cellSize + ";\n"
            value += "        centre   (" + " ".join(centre) + ");\n"
            value += "        lengthX  " + dx + ";\n"
            value += "        lengthY  " + dy + ";\n"
            value += "        lengthZ  " + dz + ";\n"
            value += "        type     box;\n"
            value += "    }\n"
        if sect == "regSph":
            stlFile = stlDir + "/" + patchName + ".stl"
            [centre, dx, dy, dz] = getRegionSize(stlFile)
            radius = (float(dx) + float(dy) + float(dz)) / 3.0 / 2.0
            value += "    " + patchName + "\n"
            value += "    {\n"
            value += "        cellSize " + cellSize + ";\n"
            value += "        centre   (" + " ".join(centre) + ");\n"
            value += "        radius   " + str(radius) + ";"
            value += "        type     sphere;"
            value += "    }\n"
    newLine  = "objectRefinements\n"
    newLine += "{\n"
    newLine += value
    newLine += "}"
    cont = setOrDeleteLine(cont, "objectRefinements", newLine, newLine)
    return cont

#
#  setBoundaryLayers
def setBoundaryLayers(data, cont):
    """ layerの設定"""
    value = ""
    for row in data:
        patchName = row[0]
        sect = row[1]
        nLayer = row[3]
        ratio = row[4]
        maxThickness = row[5]
        #ratioのチェック
        lineRa = "            thicknessRatio 1.0;\n"
        if ratio != "":
            try:
                ratioV = float(ratio)
                lineRa = "            thicknessRatio " + str(ratioV) + ";\n"
            except:
                pass
        #maxThicknessのチェック
        lineTh = ""
        if maxThickness != "":
            try:
                maxTh = float(maxThickness)
                lineTh = "            maxFirstLayerThickness " + str(maxTh) + ";\n"
            except:
                pass
        #layer設定文作成
        if nLayer != "":
            try:
                if int(nLayer) >= 1:
                    value += "        " + patchName + "\n"
                    value += "        {\n"
                    value += "            nLayers " + nLayer + ";\n"
                    value += lineRa
                    value += lineTh
                    value += "        }\n"
            except:
                pass
    newLine  = "boundaryLayers\n"
    newLine += "{\n"
    newLine += "    patchBoundaryLayers\n"
    newLine += "    {\n"
    newLine += value
    newLine += "    }\n"
    newLine += "}"
    cont = setOrDeleteLine(cont, "boundaryLayers", newLine, newLine)
    return cont



#
#  createCsvFile
#  -------------
def createCsvFile(caseDir, stlDir, fileName):
    """ defaultのcsvファイル作成"""
    data = setCsvData()
    data = addStlFileName(stlDir, data)
    data = addMaxCellSize(stlDir, data)
    data = addStlBlockSize(stlDir, data)
    writeCsvData(stlDir, fileName, data)

#
#  createMeshDict
#  ----------------
def createMeshDict(caseDir, stlDir, surfaceFile, otherData, data):
    """ meshDictの作成"""
    maxCellSize = otherData[0]
    minCellSize = otherData[1]
    featureAngle = otherData[2]
    meshDict = caseDir + "/system/meshDict"
    f=open(meshDict); cont=f.read(); f.close()
    (header, cont, footer) = separateContents(cont)
    cont = setSurfaceFile(caseDir, stlDir, surfaceFile, cont)
    cont = setMaxCellSize(maxCellSize, cont)
    cont = setMinCellSize(minCellSize, cont)
    cont = setLocalRefinement(data, cont)
    cont = setRenameBoundary(data, cont)
    cont = setSurfaceMeshRefinement(caseDir, stlDir, data, cont)
    cont = setObjectRefinements(data, stlDir, cont)
    cont = setBoundaryLayers(data, cont)
    newCont = header + cont.decode() + footer
    f = open(meshDict, "w"); f.write(newCont); f.close()
    
    


if __name__ == "__main__":
    import gettext
    localeDir = os.getenv("TreeFoamPath") + "/date/locale"
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

