#!/usr/bin/python3
#  coding: utf-8
#
#       QtParts.py
#
#           Qt関連の定義、処理をまとめたもの
#
#   20/02/08    新規作成
#      05/02    PyQt4, PyQt5, PySide, PySide2に対応
#      05/19    selectCol, selectRow, unselectAll:selectRangeのsaize設定方法
#               バグ（機能していない）あり。修正。
#      05/24    PyQtの取得方法を修正（一元化）
#      08/10    setRowLabelValue, getRowLabelValue:新規に追加。
#               treeWidgetのrowLabelの表示が枠幅であり、余裕無いので、
#               label両端に空白を追加して表示させる様に修正。
#      08/11    adjustVerticalHeader:追加
#               verticalHderのwidthをverticalHeader().setFixedWidth(int)で設定
#               する様に修正。headerの値は、生の値をセット。
#   21/01/04    mouseMoveEvent:TableClassにmouseMoveEventを追加。
#               event発生すると、Treeのfocusさせる。
#   22/09/06    keyPressEvent:shift+矢印keyと矢印keyの操作を追加。
#               keyReleaseEvent:shiftKeyの操作を追加。
#               mousePressEvent:左クリック時にcerrCellの保存を追加
#   24/06/21    mouseMoveEvent:setFocus(True)→setFocus()に修正。
#               pySide2上のgridEditorでエラー発生し出した為。
#      08/27    Qt4系とQt6系に対応するように修正。
#

""" QtのGuiPartの操作関連をまとめたもの。

Parts:
    QTreeWidget:
    QListView:
    QTableWidget:
"""

import getPyQtModule
PyQtModule = getPyQtModule.PyQtModule

#  import されているPyQtに応じてimportする
if PyQtModule == "PyQt6":
    from PyQt6.QtCore import *
    from PyQt6.QtGui import *
    from PyQt6.QtWidgets import *
elif PyQtModule == "PySide6":
    from PySide6.QtCore import *
    from PySide6.QtGui import *
    from PySide6.QtWidgets import *
elif PyQtModule == "PyQt5":
    from PyQt5.QtCore import *
    from PyQt5.QtGui import *
    from PyQt5.QtWidgets import *
elif PyQtModule == "PySide2":
    from PySide2.QtCore import *
    from PySide2.QtGui import *
    from PySide2.QtWidgets import *
elif PyQtModule == "PyQt4":
    from PyQt4.QtCore import *
    from PyQt4.QtGui import *
elif PyQtModule == "PySide":
    from PySide.QtCore import *
    from PySide.QtGui import *
else:
    print("import error: could not import PyQt or PySide!")
    exit()

#Qtのversionに応じた変数を定義
import QtVar

#------------------------- QTreeWidget関連 ----------------------------
#  treeWidget class  (Qt4,Qt5,PySide,PySide2用)
#-------------------
class treeWidget:
    """ 
    QTreeWidgetの設定、操作関連
    使い方
        ・辞書を準備
        ・createTree(headers)でtreeのformatを作成
        ・createTreeData(dataDict)でtreeDataを作成
        ・setItems()でtreeDataと辞書からtreeWidgetが完成
    """
    def __init__(self, treeWidget):
        self.treeWidget = treeWidget
        self.dataDict = {}              #辞書
        self.treeData = []              #dirを変換

    #
    #  createTree
    def createTree(self, headers):
        """ treeのcol数を設定し、各colのheaderを設定する。
        
        Args:
            headers (list(str)) :各列のheader名をlist形式で与える"""
        nCols = len(headers)
        self.treeWidget.setColumnCount(nCols)
        for col in range(nCols):
            header = headers[col]
            self.treeWidget.headerItem().setText(col, header)
    
    #
    #  createTreeData
    def createTreeData(self, dataDict):
        """ dataDictからtreeDataを作成する。
        ここで、クラス属性のdataDict, treeDataを作成する。
        辞書データには、treeで表示するdirectoryのiconを含む。
        
        Args:
            dataDict (dict) :tree作成用の辞書
                            :{<dir>: [icon, [0, 1]]} 
                            : dir名   icon   col内容のlist"""
        folderDirs = list(dataDict.keys())
        folderDirs.sort()
        rootDir = folderDirs[0]
        treeData = [rootDir, []]
        folders = []
        for folder in folderDirs[1:]:
            words = [rootDir] + folder[len(rootDir)+1:].split("/")
            folders.append(words)
        for folderWords in folders:
            treeData = self.setTreeDataFolder(0, treeData, folderWords)
        self.dataDict = dataDict
        self.treeData = treeData

    def setTreeDataFolder(self, nest, tree, folder):
        """ folderをtree状に作成する"""
        nest += 1
        if nest > 50:
            print("error: folder nesting is too deep!")
            return
        #親folderまでskipする
        parentDir = folder[nest]
        ii = 0
        while ii < len(tree[1]):
            if type(tree[1][ii]) == list:
                if tree[1][ii][0] == parentDir:
                    self.setTreeDataFolder(nest, tree[1][ii], folder)
                    break
            else:
                if tree[1][ii] == parentDir:
                    tree[1][ii] = [parentDir, []]
                    self.setTreeDataFolder(nest, tree[1][ii], folder)
                    break
            ii += 1
        #親folderの位置ならchildを追加する
        if nest == len(folder) - 1:
            child = folder[-1]
            tree[1].append(child)
        return tree

    #
    #  setItems
    def setItems(self):
        """ treeDataを使ってtreeWidgetのitemをセットする
        
        Args:
            None"""
        treeData = self.treeData
        dataDict = self.dataDict
        rootName = treeData[0]
        items = treeData[1]
        [icon, colConts] = dataDict[rootName]
        parentItem = QTreeWidgetItem(self.treeWidget)
        parentItem.setText(0, rootName)
        parentItem.setIcon(0, icon)
        self.setColConts(parentItem, colConts)
        self.addTreeNodes(parentItem, [rootName], items)

    #
    #  setItemCont
    def setItemCont(self, item, colConts):
        """ 指定したitemにcolContsを設定する。
        
        Args:
            item (QTreeWidgetItem)  :colContsをセットするitem
            colConts (list(str))    :セットする内容"""
        self.setColConts(item, colConts)

    #
    #  addTreeNodes
    def addTreeNodes(self, parentItem, parentDir, items):
        """ parentItem以下のTreeItemをitems（treeData）から作成する。
        
        Args:
            parentItem (QTreeWidgetItem)    :親item
            parentDir (str)                 :親のdir
            items (list(treeData))          :dirをtreeDataに変換したitem"""
        for item in items:
            if type(item) == str:
                itemDir = parentDir + [item]
                [icon, colConts] = self.dataDict["/".join(itemDir)]
                child = QTreeWidgetItem(parentItem)
                child.setIcon(0, icon)
                child.setText(0, item)
                self.setColConts(child, colConts)
            else:
                itemDir = parentDir + [item[0]]
                [icon, colConts] = self.dataDict["/".join(itemDir)]
                child = QTreeWidgetItem(parentItem)
                child.setText(0, item[0])
                child.setIcon(0, icon)
                self.setColConts(child, colConts)
                self.addTreeNodes(child, itemDir, item[1])

    def setColConts(self, item, colConts):
        """ item中にcol内容をセットする"""
        col = 1
        for i in range(len(colConts)):
            value = colConts[i]
            if type(value) == str:
                item.setText(col, value)
                col += 1
            elif type(value) == int or type(value) == float:
                item.setText(col, str(value))
                col += 1
            else:
                item.setIcon(col, value)

    #
    #  selectDir
    def selectDir(self, selDir):
        """ selDirまで展開し、scrollして、selDirを選択表示する。
        
        Args:
            selDir (str)    :選択するdir"""
        #selDirのitemを取得
        selItem = self.getItemFromDir(selDir)
        #selItemまで展開
        self.expandToItem(selItem)
        #selItemを選択
        #----
        #self.treeWidget.setItemSelected(selItem, True)
        self.treeWidget.setSelected(selItem, True)
        #selItemまでscroll
        self.treeWidget.scrollToItem(selItem, QAbstractItemView.PositionAtCenter)

    #
    #  selectItem
    def selectItem(self, selItem):
        """ selItemまで展開し、scrollして、selItemを選択表示する。
        
        Args:
            selItem (QTreeWidgetItem)   :選択するitem"""
        #selItemまで展開
        self.expandToItem(selItem)
        #selItemを選択
        self.treeWidget.setItemSelected(selItem, True)
        #selItemまでscroll
        self.treeWidget.scrollToItem(selItem, QAbstractItemView.PositionAtCenter)

    #
    #  getItemCont
    def getItemCont(self, item):
        """ 指定したitemのcol内容を取得する
        
        Args:
            item (QTreeWidgetItem)  :col内容を取得するitem
        Returns:
            colConts (list(str))    :col内容"""

        def convQstringToText(string):
            """ python2, 3で動作が異なるので、ここで修正。"""
            try:
                #python2の場合、変換する。
                string = string.toUtf8().data()
            except:
                #python3の場合、エラー発生。変換せず。
                pass
            return string

        nCols = item.columnCount()
        colConts = []
        for i in range(nCols):
            text = item.text(i)
            text = convQstringToText(text)
            colConts.append(text)
        return colConts

    #
    #  getItemFromDir
    def getItemFromDir(self, selDir):
        """ dirからitemを取得して返す
        
        Args:
            selDir (str)            :取得するdir
        Returns:
            item (QTreeWidgetItem)  :dirに対応するitem"""
        rootDir = self.treeData[0]
        folders = selDir[len(rootDir)+1:].split("/")
        parentItem = self.treeWidget.topLevelItem(0)
        for folder in folders:
            for idx in range(parentItem.childCount()):
                childItem = parentItem.child(idx)
                text = childItem.text(0)
                if text == folder:
                    break
            parentItem = childItem
        return parentItem

    #
    #  getDirFromItem
    def getDirFromItem(self, selItem):
        """ itemからdirを取得する。
        
        Args:
            selItem (QTreeWidgetItem)   :dirを取得するitem
        Returns:
            selDir (str)                :取得したdir"""

        def convQstringToText(string):
            """ python2, 3で動作が異なるので、ここで修正。"""
            try:
                #python2の場合、変換する。
                string = string.toUtf8().data()
            except:
                #python3の場合、エラー発生。変換せず。
                pass
            #PySide, PyQt4で異なるので、修正
            try:
                if type(string) == unicode:
                    string = string.encode("utf-8")
            except:
                pass
            return string

        words = []
        rootItem = self.treeWidget.topLevelItem(0)
        while selItem is not rootItem:
            text = selItem.text(0)
            text = convQstringToText(text)
            words = [text] + words
            selItem = selItem.parent()
        rootText = rootItem.text(0)
        rootText = convQstringToText(rootText)
        words = [rootText] + words
        selDir = "/".join(words)
        return selDir

    #
    #  expandToItem
    def expandToItem(self, selItem):
        """ rootからselItemまで展開する
        
        Args:
            selItem (QTreeWidgetItem)   :item"""
        #展開するitemを取得
        items = []
        rootItem = self.treeWidget.topLevelItem(0)
        #while selItem != rootItem:
        while selItem is not rootItem:
            selItem = selItem.parent()
            items = [selItem] + items
        items = [rootItem] + items
        #itemを展開
        for item in items:
            self.treeWidget.expandItem(item)

    #
    #  adjustColWidth
    def adjustColWidth(self):
        """ column幅をadjustする
        
        Args:
            None"""
        nCols = self.treeWidget.columnCount()
        for col in range(nCols):
            self.treeWidget.resizeColumnToContents(col)
        
    #
    #  deleteChildren
    def deleteChldren(self, item):
        """ 指定したitemの子itemを全て削除する。辞書も該当部を削除する。
        
        Args:
            item (QTreeWidgetItem)  :指定するitem"""
        #辞書から削除
        itemDir = self.getDirFromItem(item)
        folders = list(self.dataDict.keys())
        for folderDir in folders:
            if folderDir[:len(itemDir)+1] == itemDir + "/":
                dummy = self.dataDict.pop(folderDir)
        #treeWidgetから削除
        for _i in range(item.childCount()):
            delItem = item.child(0)
            item.removeChild(delItem)

    #
    #  getItemsFromTreeData
    def getItemsFromTreeData(self, selDir):
        """ dirをたどって、TreeDataの該当するitemsを取得する
        
        Args:
            selDir (str)            :指定するdir
        Returns:
            items (list(treeData))  :取得したtreeData内のitems"""
        
        def getTreeNodes(items, name):
            """ treeData内のitems内からnameが持っているitemsを取得する"""
            for item in items:
                newItems = ""
                if type(item) == str:
                    newItems = name
                else:
                    if item[0] == name:
                        newItems = item[1]
                        break
            return newItems

        rootDir = self.treeData[0]
        items = self.treeData[1]
        words = selDir[len(rootDir)+1:].split("/")
        for name in words:
            items = getTreeNodes(items, name)
        return items


#------------------------- QListView関連 ------------------------------
#  listView class
#------------------
class listView:
    """ ListViewの設定、操作関連"""

    def __init__(self, listView):
        self.listView = listView

    def setItems(self, items, multi=True):
        """ listViewを作成し、itemをセットする。
        
        Args:
            items (list(item))  :セットする項目
                                :itemがstrのみは、nameのみ表示。
                                :itemが[icon, str]の場合は、iconも表示。
            multi (bool)        :Trueは、multipleSelection"""
        if multi == True:
            self.listView.setSelectionMode(QtVar.ExtendedSelection)
        listModel = QStandardItemModel()
        self.listView.setModel(listModel)
        for itemCont in items:
            if (itemCont) == str:
                item = QStandardItem(itemCont)
            else:
                icon = itemCont[0]
                name = itemCont[1]
                item = QStandardItem(icon, name)
            listModel.appendRow(item)

    def getSelectedNames(self):
        """ listViewから選択行名（name）を取得する
        
        Returns:
            names (list(str))   :選択行名（name）"""
        idxes = self.listView.selectionModel().selectedRows()
        names = []
        for idx in idxes:
            row = idx.row()
            name = self.listView.model().item(row).text()
            names.append(name)
        return names

    def getSelectedIndexes(self):
        """ listViewから選択行のrowNoのlistを取得する。
        
        Returns:
            rows (list(int))    :選択行Noのlist"""
        idxes = self.listView.selectionModel().selectedRows()
        rows = []
        for idx in idxes:
            row = idx.row()
            rows.append(row)
        return rows
        
    def selectNames(self, selNames):
        """ listView内の指定されたitem名を選択する。
        
        Args:
            selNames (list(str))    :選択する項目名のlist"""
        listModel = self.listView.model()
        selection = self.listView.selectionModel()
        for row in range(listModel.rowCount()):
            text = listModel.item(row).text()
            if text in selNames:
                idx = listModel.item(row).index()
                selection.select(idx, QtVar.Select)


#------------------------- QTableWidget関連 ---------------------------
#  tableWidget class
#---------------------
class tableWidget:
    """ tableWidgetの作成、操作関連"""

    def __init__(self, tableWidget):
        self.tableWidget = tableWidget      #widget

    #
    #  createTable
    #--------------
    def createTable(self, rowNames, colNames):
        """ row x col のtableを作成する。rowNames、colNamesをセット。"""
        self.tableWidget.setRowCount(len(rowNames))
        self.tableWidget.setColumnCount(len(colNames))
        #colラベル（title）を作成
        for col in range(len(colNames)):
            #label(name)を設定
            self.tableWidget.setHorizontalHeaderItem(col, QTableWidgetItem(colNames[col]))
            #水平,垂直方向のセンタリング
            self.tableWidget.horizontalHeaderItem(col).setTextAlignment(QtVar.AlignCenter)
        #rowラベル（title）を作成
        for row in range(len(rowNames)):
            #label（name）を設定
            self.setRowLabelValue(row, rowNames[row])
            #水平垂直のセンタリング
            self.tableWidget.verticalHeaderItem(row).setTextAlignment(QtVar.AlignCenter)
        #headerFontをboldに変更
        font = self.tableWidget.font()
        font.setBold(True)
        self.tableWidget.horizontalHeader().setFont(font)
        self.tableWidget.verticalHeader().setFont(font)
        #全cellに空白のQTableWidgetItemをセット
        for col in range(len(colNames)):
            for row in range(len(rowNames)):
                #空白をセット
                self.tableWidget.setItem(row, col, QTableWidgetItem(""))
                #cellの上詰めで配置（defaultは縦方向がセンタリング）
                self.tableWidget.item(row, col).setTextAlignment(QtVar.AlignTop)
        #tableWidgetにtextEditorを設定
        #  引数としてtableWidgetを設定。
        #  editorがtableWidgetにアクセスできる様にする為
        
        delegate = Delegate(self.tableWidget)
        self.tableWidget.setItemDelegate(delegate)
        
        #cell幅をadjustする
        #self.tableWidget.resizeColumnsToContents()
        #self.tableWidget.resizeRowsToContents()
        # col = self.tableWidget.columnCount() - 1
        # for row in range(self.tableWidget.rowCount()):
        #     self.tableWidget.verticalHeader().setResizeMode(row, QtGui.QHeaderView.ResizeToContents)


    #---------- label関係 --------------------
    #
    #  setLabelsFontColor_darkBlue
    def setLabelsFontColor_darkBlue(self):
        """ labelのfontColorをdarkBlue設定する"""
        for col in range(self.tableWidget.columnCount()):
            self.tableWidget.horizontalHeaderItem(col).setForeground(QColor(2, 0, 139))     #darkBlue
        for row in range(self.tableWidget.rowCount()):
            self.tableWidget.verticalHeaderItem(row).setForeground(QColor(2, 0, 139))       #darkBlue
    #
    #  setRowLabelValue
    def setRowLabelValue(self, row, value):
        """ rowのlabelをセット。行の両端に空白を加えて、スペースを確保"""
        self.tableWidget.setVerticalHeaderItem(row, QTableWidgetItem(value))
    #
    #  getColLabelValue
    def getColLabelValue(self, col):
        """ columnのlabelの値を取得する"""
        return self.tableWidget.horizontalHeaderItem(col).text()
    #
    #  getRowLabelValue
    def getRowLabelValue(self, row):
        """ rowのlabelの値を取得する。
        rowLabelには空白を追加しているので、空白を削除して返す。"""
        value = self.tableWidget.verticalHeaderItem(row).text()
        return value

    #-------- current select scrollCell copyPaste-------------
    #
    #  currentCell
    def currentCell(self):
        """ currentCellの（row, column）を返す。"""
        idx = self.tableWidget.currentIndex()
        return (idx.row(), idx.column())
    
    #
    #  scrollToCell
    def scrollToCell(self, row, col):
        """ cell(row, col)までscrollして表示する。"""
        item = self.tableWidget.item(row, col)
        self.tableWidget.scrollToItem(item, QtVar.PositionAtCenter)

    #
    #  selectedCells
    def selectedCells(self):
        """ 選択しているcellの(row, col)をtupleで戻す。"""
        items = self.tableWidget.selectedItems()
        rowCols = []
        for item in items:
            index = self.tableWidget.indexFromItem(item)
            rowCols.append((index.row(), index.column()))
        return rowCols

    #
    #  selectRow
    def selectRow(self, row):
        """ row行を選択する"""
        selRange = QTableWidgetSelectionRange(row, 0, row, self.tableWidget.columnCount()-1)
        self.tableWidget.setRangeSelected(selRange, True)
        # for col in range(self.tableWidget.columnCount()):
        #     item = self.tableWidget.item(row, col)
        #     self.tableWidget.setItemSelected(item, True)
    #
    #  selectCol
    def selectCol(self, col):
        """ col列を選択する"""
        selRange = QTableWidgetSelectionRange(0, col, self.tableWidget.rowCount()-1, col)
        self.tableWidget.setRangeSelected(selRange, True)
        # for row in range(self.tableWidget.rowCount()):
        #     item = self.tableWidget.item(row, col)
        #     self.tableWidget.setItemSelected(item, True)
    #
    #  selectCell
    def selectCell(self, row, col):
        """ cell(row, col)を選択する"""
        selRange = QTableWidgetSelectionRange(row, col, row, col)
        self.tableWidget.setRangeSelected(selRange, True)
        #item = self.tableWidget.item(row, col)
        #self.tableWidget.setItemSelected(item, True)
    #
    #  unselectAll
    def unselectAll(self):
        """ 全cellを非選択状態に設定"""
        selRange = QTableWidgetSelectionRange(0, 0, self.tableWidget.rowCount()-1, self.tableWidget.columnCount()-1)
        self.tableWidget.setRangeSelected(selRange, False)

    #
    #  getCopyText
    def getCopyText(self):
        """ 選択項目をclipboardに渡す形式で取得し、戻す"""
        items = self.tableWidget.selectedIndexes()
        rowCols = []
        for item in items:
            row = item.row()
            col = item.column()
            rowCols.append([row, col])
        rowCols.sort()
        conts = []
        rowConts = []
        rowb = rowCols[0][0]
        for row, col in rowCols:
            cell = self.tableWidget.item(row, col).text()
            if row == rowb:
                rowConts.append('"' + cell + '"')
            else:
                conts.append(rowConts)
                rowConts = ['"' + cell + '"']
                rowb = row
        conts.append(rowConts)
        lines = []
        for rowConts in conts:
            lines.append("\t".join(rowConts))
        copyCont = "\n".join(lines)
        return copyCont
    #
    #  setPasteText
    def setPasteText(self, pasteText):
        """ pasteTextをtableに貼り付ける"""

        def getMultiLines(cont):
            """ pasteContを行listに変換する"""
            lines = []
            p = 0
            line = ""
            ps = 0
            while p < len(cont):
                chara = cont[p]
                if chara == '"':
                    p += 1
                    while p < len(cont) and cont[p] != '"':
                        p += 1
                    p += 1
                elif chara == "\n":
                    line = cont[ps:p]
                    lines.append(line)
                    while p < len(cont) and cont[p] == "\n":
                        p += 1
                    ps = p
                else:
                    p += 1
            if ps < len(cont):
                line = cont[ps:p]
                lines.append(line)
            return lines

        def pasteValsToCells(vals):
            """ 値をcellにセット"""
            #選択場所を取得
            firstSelection = []
            endSelection = []
            selRanges = self.tableWidget.selectedRanges()
            if len(selRanges) >= 0:
                selRange = selRanges[0]
                top = selRange.topRow()
                bottom = selRange.bottomRow()
                left = selRange.leftColumn()
                right = selRange.rightColumn()
                firstSelection = [top, left]
                endSelection = [bottom, right]
            else:
                currRow = self.tableWidget.currentRow()
                currCol = self.tableWidget.currentColumn()
                endSelection = [currRow, currCol]
                firstSelection = [currRow, currCol]
            #paste開始
            #  1ヶ分をpaste
            selItems = []
            selRows = []
            selCols = []
            row = firstSelection[0]
            vRow = 0
            while vRow < len(vals):
                col = firstSelection[1]
                vCol = 0
                while vCol < len(vals[vRow]):
                    val = vals[vRow][vCol]
                    if row < self.tableWidget.rowCount() and col < self.tableWidget.columnCount():
                        item = self.tableWidget.item(row, col)
                        item.setText(val)
                        selRows.append(row)
                        selCols.append(col)
                        selItems.append(item)
                    col += 1
                    vCol += 1
                row += 1
                vRow += 1
            #  残りをpaste（row側）
            if row < endSelection[0] + 1:
                vRow = 0
                while row < endSelection[0] + 1:
                    col = firstSelection[1]
                    vCol = 0
                    while vCol < len(vals[vRow]):
                        if row < self.tableWidget.rowCount() and col < self.tableWidget.columnCount():
                            val = vals[vRow][vCol]
                            item = self.tableWidget.item(row, col)
                            item.setText(val)
                            selRows.append(row)
                            selCols.append(col)
                            selItems.append(item)
                        col += 1
                        vCol += 1
                    row += 1
                    vRow += 1
                    if vRow >= len(vals):
                        vRow = 0
                endPaste = [row, col]
            else:
                endPaste = [row, col]
            #  残りpaste（col側）
            if col < endSelection[1] + 1:
                vCol = 0
                col = endPaste[1]
                while col < endSelection[1] + 1:
                    row = firstSelection[0]
                    vRow = 0
                    while row < endPaste[0]:
                        if row < self.tableWidget.rowCount() and col < self.tableWidget.columnCount():
                            val = vals[vRow][vCol]
                            item = self.tableWidget.item(row, col)
                            item.setText(val)
                            selRows.append(row)
                            selCols.append(col)
                            selItems.append(item)
                        row += 1
                        vRow += 1
                        if vRow >= len(vals):
                            vRow = 0
                    col += 1
                    vCol += 1
                    if vCol >= len(vals[vRow]):
                        vCol = 0
            #pasteしたcellを選択表示
            top = min(selRows); bottom = max(selRows)
            left = min(selCols); right = max(selCols)
            selRange = QTableWidgetSelectionRange(top, left, bottom, right)
            self.tableWidget.setRangeSelected(selRange, True)

        #行を取得
        vals = getMultiLines(pasteText)
        if len(vals) == 0:
            return
        #1行毎に値を取得しlistを作成
        values = []
        for line in vals:
            row = line.split("\t")
            rows = []
            for val in row:
                #「"」マークの処理
                if len(val) == 0:
                    b = ""
                elif len(val) == 1:
                    b = val
                elif val[0] == '"':
                    b = val[1:-1]
                else:
                    b = val
                rows.append(b)
            values.append(rows)
        #値をgridにセット
        pasteValsToCells(values)

    #--------- cellの値を取得、設定 -----------------
    #
    #  getCellValue
    def getCellValue(self, row, col):
        """ 指定したcellの値を取得"""
        return self.tableWidget.item(row, col).text()
    #
    #  setCellValue
    def setCellValue(self, row, col, value):
        """ 指定したcellにvalue（text）をセットする"""
        #文字列をcellにセット
        self.tableWidget.item(row, col).setText(value)
    #
    #  adjustCells
    def adjustCells(self):
        """ 全cellの幅、高さを自動調整する。"""
        #textのwidth、heightを取得
        rowColWidth = []
        rowColHeight = []
        for row in range(self.tableWidget.rowCount()):
            colWidth = []; colHeight = []
            for col in range(self.tableWidget.columnCount()):
                text = self.tableWidget.item(row, col).text()
                width, height = self.getTextRect(text)
                colWidth.append(width)
                colHeight.append(height)
            rowColWidth.append(colWidth)
            rowColHeight.append(colHeight)
        #列幅を設定
        for col in range(self.tableWidget.columnCount()):
            rowWidth = list(map(lambda x: x[col], rowColWidth))
            maxWidth = max(rowWidth)
            #headerを確認
            header = self.tableWidget.horizontalHeaderItem(col).text()
            #  header幅は、余裕を持たせるため、空白2ヶをプラス
            width, height = self.getTextRect(header+"  ")
            #最大値を取得しセット
            maxWidth = max([maxWidth, width])
            self.tableWidget.setColumnWidth(col, maxWidth)
        #verticalHerderのwidthを設定
        self.adjustVerticalHeader()
        #行高さを設定
        for row in range(self.tableWidget.rowCount()):
            colHeight = rowColHeight[row]
            maxHeight = max(colHeight)
            #headerを確認
            #header = self.tableWidget.verticalHeaderItem(row).text()
            header = self.getRowLabelValue(row)
            width, height = self.getTextRect(header)
            #最大値を取得しセット
            maxHeight = max([maxHeight, height])
            self.tableWidget.setRowHeight(row, maxHeight)

    #
    #  adjustVerticalHeader
    def adjustVerticalHeader(self):
        """ verticalHeaderのwidthを調整する。空白2ヶ分を増やす。"""
        widthes = []
        for row in range(self.tableWidget.rowCount()):
            value = self.tableWidget.verticalHeaderItem(row).text()
            #value = rowNames[row]
            lines = value.split("\n")
            newLines = list(map(lambda x: " " + x + " ", lines))
            label = "\n".join(newLines)
            width, height = self.getTextRect(label)
            widthes.append(width)
        self.tableWidget.verticalHeader().setFixedWidth(max(widthes))
    #
    #  adjustCell
    def adjustCell(self, curRow, curCol):
        """ 特定cellの幅、高さを自動調整する。"""
        #curCol列の幅を確認、セット
        vals = []
        text = self.tableWidget.horizontalHeaderItem(curCol).text()
        width, height = self.getTextRect(text)
        vals.append(width)
        for row in range(self.tableWidget.rowCount()):
            text = self.tableWidget.item(row, curCol).text()
            #  header幅は、余裕を持たせるため、空白2ヶをプラス
            width, height = self.getTextRect(text+"  ")
            vals.append(width)
        self.tableWidget.setColumnWidth(curCol, max(vals))
        #curRow行の高さを確認、セット
        vals = []
        #text = self.tableWidget.verticalHeaderItem(curRow).text()
        text = self.getRowLabelValue(curRow)
        width, height = self.getTextRect(text)
        vals.append(height)
        for col in range(self.tableWidget.columnCount()):
            text = self.tableWidget.item(curRow, col).text()
            width, height = self.getTextRect(text)
            vals.append(height)
        self.tableWidget.setRowHeight(curRow, max(vals))
    #
    #  getTextRect
    def getTextRect(self, text):
        """ text（文字列）の幅、高さを計算し、戻す。"""
        #文字列のサイズ（whidth、height）を取得
        maxWidth, maxHeight = 1000, 1000                #最大サイズ
        #サイズを取得
        text += " "
        valRect = self.tableWidget.fontMetrics().boundingRect(
            QRect(0,0,maxWidth, maxHeight),      #maxRectSize
            Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignTop,   #alignmentMethod
            text)                                       #text
        w = int(valRect.width() * 1.095 + 2)              #大きめの値をセット
        h = int(valRect.height() * 1.095 + 2)
        return w, h

    #-------- cellの背景色設定 ------
    #
    #  setCellColor_lightGreen
    def setCellColor_lightGreen(self, row, col):
        """ cell背景色をlightGreenに設定"""
        #self.tableWidget.item(row, col).setBackgroundColor(QtGui.QColor(204, 255, 255))
        #self.tableWidget.item(row, col).setBackgroundColor(QtGui.QColor(230, 255, 255))
        self.tableWidget.item(row, col).setBackground(QColor(230, 255, 255))
    #
    #  setCellColor_pink
    def setCellColor_pink(self, row, col):
        """ cell背景色をpinkに設定"""
        self.tableWidget.item(row, col).setBackground(QColor(247, 171, 166))
    #
    #  setCellColor_yellow
    def setCellColor_yellow(self, row, col):
        """ 指定したcellの背景色をyellowに設定する。"""
        self.tableWidget.item(row, col).setBackground(QColor(255, 255, 0))      #yellow

    #------------ cellのfontColorを設定-------------------
    #
    #  setCellFontColor_blue
    def setCellFontColor_blue(self, row, col):
        """ 指定したcellのfonfColorをblueに設定する"""
        self.tableWidget.item(row, col).setForeground(QColor(0,0,255))      #blue
    #
    #  setCellFontColor_brown
    def setCellFontColor_brown(self, row, col):
        """ 指定したcellのfonfColorをbrownに設定する"""
        self.tableWidget.item(row, col).setForeground(QColor(153, 76, 0))   #brown
    #
    #  setCellFontColor_darkGreen
    def setCellFontColor_darkGreen(self, row, col):
        """ 指定したcellのfonfColorをdarkGreenに設定する"""
        self.tableWidget.item(row, col).setForeground(QColor(0,100,0))        #darkGreen
    #
    #  setCellFontColor_sysFontColor
    def setCellFontColor_sysFontColor(self, row, col):
        """ 指定したcellのfonfColorをsystemFontColor（黒）に設定する"""
        self.tableWidget.item(row, col).setForeground(QColor(0, 0, 0))      #black


#------------------
#  Delegate class
#------------------
#class Delegate(QtGui.QStyledItemDelegate):
class Delegate(QStyledItemDelegate):
    """ QTableWidgetのeditorをにQTextEditに変更するために作成。
    親のQTableWidgetを引数として受け取り、editorに渡す。"""

    def __init__(self, parent=None):
        super(Delegate, self).__init__(parent)
        self.table = parent

    def createEditor(self, parent, option, index):
        #return QtGui.QLineEdit(parent)
        #return QtGui.QTextEdit(parent)
        value = self.table.item(index.row(), index.column()).text()
        if value[-3:] == "...":
            #signalを発信してeditorを起動しない
            self.table.largeTextSignal.emit()
            return
        return TextEditor(parent, self.table)
    
    def setEditorData(self, editor, index):
        #value = index.model().data(index, QtCore.Qt.DisplayRole)
        #editor.setText(value)
        editor.setText(index.data())
        editor.selectAll() 
    
    def setModelData(self, editor, model, index):
        #model.setData(index, editor.text())
        model.setData(index, editor.toPlainText())


#---------------
#  Table class
#---------------
#class Table(QtGui.QTableWidget):
class Table(QTableWidget):
    """ eventを追加したQTableWidgetを作成。
    追加したeventは、keyPressEvent, keyReleaseEvent。"""

    CtrlFlag = 0                         #controlKey pressing:1
    ShiftFlag = 0                        #shiftKey pressing:1
    CurrRowCol = [0, 0]                  #currentCellを定義
    try:
        copySignal = pyqtSignal()        #copyのsignal
        pasteSignal = pyqtSignal()       #pasteのsignal
        enterSignal = pyqtSignal()       #enterKeyのsignal
        largeTextSignal = pyqtSignal()   #largetext(...)のsignal
    except:
        copySignal = Signal()        #copyのsignal
        pasteSignal = Signal()       #pasteのsignal
        enterSignal = Signal()       #enterKeyのsignal
        largeTextSignal = Signal()   #largetext(...)のsignal
    newText = None

    #
    #  mousePressEvent
    #------------------
    def mousePressEvent(self, event):
        """ mouseボタンの操作event"""
        super().mousePressEvent(event)
        #左クリック？
        if event.buttons() == QtVar.LeftButton:
            #currentCellを保存
            row = self.currentRow()
            col = self.currentColumn()
            self.CurrRowCol = [row, col]
        return

    #
    #  mouseMoveEvent
    #-----------------
    def mouseMoveEvent(self, event):
        """ mouseを移動させた時のevent。
        有効にする為には、呼び出し側で「self.Table.setMouseTracking(True)」が必要。
        """
        global editFlag
        #TextEditorが停止中（cell編集中ではない時）？
        if editFlag == 0:
            #TextEditor停止中の場合
            #self.setFocus(True)         #table内でmouseが動くとtableにfocusする
            self.setFocus()
        super().mouseMoveEvent(event)
        return

    #
    #  keyPressEvent
    #----------------
    def keyPressEvent(self, event):
        """ keyboardのevent取得と処理"""
        key = event.key()
        #enterKey?
        if self.CtrlFlag == 0 and (key == QtVar.Key_Enter or key == QtVar.Key_Return):            #enterKeyの場合、eventを発生させない。
            self.enterSignal.emit()
            newEvent = QKeyEvent(QtVar.KeyPress, QtVar.Key_Down, QtVar.NoModifier)
            return QTableWidget.keyPressEvent(self, newEvent)
        #deleteKey?
        elif self.CtrlFlag == 0 and key == QtVar.Key_Delete:
            #deleteKeyの場合、cellをクリア
            indexes = self.selectedIndexes()
            for index in indexes:
                row = index.row()
                col = index.column()
                self.item(row, col).setText("")
        #Ctrl+C?
        elif self.CtrlFlag == 1 and key == QtVar.Key_C:
            #signalを発信
            self.copySignal.emit()
        #Ctrl+V?
        elif self.CtrlFlag == 1 and key == QtVar.Key_V:
            #signalを発信
            self.pasteSignal.emit()
        #shift+矢印key?
        elif self.ShiftFlag == 1 and (key == QtVar.Key_Right or
                                     key == QtVar.Key_Left or
                                     key == QtVar.Key_Up or
                                     key == QtVar.Key_Down):
            #通常の処理で戻る（currentCellを保存しない）
            QTableWidget.keyPressEvent(self, event)
            return
        #矢印key?
        elif self.ShiftFlag == 0 and (key == QtVar.Key_Right or
                                     key == QtVar.Key_Left or
                                     key == QtVar.Key_Up or
                                     key == QtVar.Key_Down):
            #currentCellを保存場所から戻す
            self.setCurrentCell(self.CurrRowCol[0], self.CurrRowCol[1])
            #cursorを動かす
            QTableWidget.keyPressEvent(self, event)
            #currentCellを保存
            row = self.currentRow()
            col = self.currentColumn()
            self.CurrRowCol = [row, col]
            return
        #controlKey?
        elif key == QtVar.Key_Control:
            #CtrlFlagをセット
            self.CtrlFlag = 1
        #shiftKey?
        elif key == QtVar.Key_Shift:
            #ShitFlagをセット
            self.ShiftFlag = 1
            #currentCellをセット
            row = self.currentRow()
            col = self.currentColumn()
            self.CurrRowCol = [row, col]
        #eventを返す
        return QTableWidget.keyPressEvent(self, event)

    #
    #  keyReleaseEvent
    #-----------------
    """ keyを離した時のevent。controlKeyをチェックする。"""
    def keyReleaseEvent(self, event):
        key = event.key()
        #controlKey?
        if key == QtVar.Key_Control:
            #CtrlFlagをクリアする
            self.CtrlFlag = 0
        #shiftKey?
        elif key == QtVar.Key_Shift:
            #ShiftFlagをクリアする
            self.ShiftFlag = 0
        return QTableWidget.keyReleaseEvent(self, event)

#
#  editFlag
""" TextEditorが起動中かどうか確認用のglobal変数
 停止中: editFlag=0
 起動中: editFlag=1"""
editFlag = 0

#-------------------
#  TextEditor class
#-------------------
#class TextEditor(QtGui.QTextEdit):
class TextEditor(QTextEdit):
    """ eventを追加したQTextEditを作成。
    追加したeventは、keyPressEvent, keyReleaseEvent。
    editorから親のtableWidgetにアクセスできる様に、引数を追加。"""

    def __init__(self, parent, table):
        global editFlag
        super(TextEditor, self).__init__(parent)
        self.table = table              #tableWidget（親）
        self.table.newText = None       #編集後のtext
        self.CtrlFlag = 0               #shiftKey pressing:1
        self.AltFlag = 0                #AltKey pressing:1
        editFlag = 1                    #textEditor起動中

    def keyPressEvent(self, event):
        """ keyboardのevent取得と処理"""
        global editFlag
        key = event.key()
        #ctrl(alt) + <enter> <Return> ?
        if self.CtrlFlag == 1 or self.AltFlag == 1:
            if key == QtVar.Key_Enter or QtVar.Key_Return:
                #eventの内容を変更して、再発行
                enterEvent = QKeyEvent(QtVar.KeyPress, QtVar.Key_Enter, QtVar.NoModifier)
                editFlag = 0
                return QTextEdit.keyPressEvent(self, enterEvent)
        #<enter> or <return> ?
        elif self.CtrlFlag == 0 and self.AltFlag == 0:
            #editorの終了?
            if key == QtVar.Key_Enter or key == QtVar.Key_Return:
                #編集後のtextを取得
                self.table.newText = self.toPlainText()
                #tableのenterSignalを発行
                #  signalを発行して、main側に知らせる
                self.table.enterSignal.emit()
                editFlag = 0
                return
        #ctrlKey?
        if key == QtVar.Key_Control:
            #flagをセット
            self.CtrlFlag = 1
        #AltKey?
        elif key == QtVar.Key_Alt:
            #flagをセット
            self.AltFlag = 1
        editFlag = 0
        return QTextEdit.keyPressEvent(self, event)

    def keyReleaseEvent(self, event):
        key = event.key()
        #CtrlKeyを離した？
        if key == QtVar.Key_Control:
            #CtrlFlagをクリアする
            self.CtrlFlag = 0
        elif key == QtVar.Key_Alt:
            #AltFlagをクリア
            self.AltFlag = 0
        return QTextEdit.keyReleaseEvent(self, event)
