#!/usr/bin/python3
# -*- coding: UTF-8 -*-
#
#
#       pyFistr.py
#
#   15/09/02    windows対応の為、"\n"→os.linesepに修正
#      10/30    deleteLsOfLineを追加
#      12/17    addPrincipalStressDataInVtk:主応力のベクトルを追加
#      12/18    addPrincipalStrainDataInVtk:主ひずみを追加
#      12/25    行末コードlsを修正
#   16/01/06    addPrincipalStressDataInVtk:tresca応力
#               addPrincipalStrainDataInVtk:trescaひずみを追加
#      01/16    getEndPoint:追加
#      03/27    addPrincipalStressDataInVtk:取得する物性値名を出力
#               addPrincipalStrainDataInVtk:取得する物性値名を出力
#      04/07    getSolutionTypeFromCnt:追加
#      04/17    changeMeshScale:修正
#   17/08/23    readUsingApplications:追加
#      09/23    高速化の為、meshをmeshHeaderDataとして保存して処理する様に修正
#   18/07/16    getValName:大文字同士で比較する様に修正
#      11/30    getHeaderNumData:node数が20以下の場合、読み込み時出力しない様に修正
#   19/01/04    python3用に書き換え
#      01/06    readCsvData:修正
#      01/25    fileOpen時encodingを追加。読み書き共（windows共用の為）
#               f=open(fileName, encoding="utf-8")
#      03/21    getHeaderData:コメント行の取得方法を修正
#      04/22    vsCodeにより修正
#      05/03    vsCodeにより再修正
#      06/22    getHeaderNumData:並列処理に変更
#   20/07/08    readUsingApplications:戻り値をappDictに変更
#               runAppを追加
#      07/10    readUsingApplications:usingAppの読み込みでエラー発生した時
#               shift-jisで読み直す。（defaultは、utf-8）
#      07/12    writeUsingApplications:追加。
#      10/31    createLinesFistrModelMshHeader:!SGROUPの出力を修正。
#   21/02/18    deleteComment:関数内関数から他で使えるようにmoduleの関数に変更
#      02/26    getNoGroups:nameの比較に"dummy_"を追加して比較
#      05/17    run classを追加
#   22/03/13    readUsingApplications:languageを追加
#      04/25    run.commandWithLog, commandWithLogBG:新規追加
#      10/08    class server, class serversを追加。
#               commandReturnCont:追加。
#      10/21    getElmData:エラー発生時に、hungupするので、修正。
#               line="\n"の場合、splitできずエラー、hungupする。
#   23/01/10    readUsingApplications:REVOCAPを削除
#      04/13    cntFileName:hecmw_ctrl.datからcntファイル名を取得を追加
#               writeFistrModelCnt,writeHeaderData:削除。(cntファイルの読込方法変更の為)
#      04/18    getCntMshFileNameDict:追加。
#      04/20    shrinkKeywords:"!INITIAL CONDITION"を追加
#               setCntMshFileNameDict:辞書dataをhcmw_ctrl.datファイルに保存
#

#import abaqus2fistr
#import calcPrincipalStress
import csv
import os
import glob
import math
import multiprocessing as mp
import subprocess

ls = "\n"

#------ 全体の変数定義 -------------
#Appのpathを定義
#binAppPath = os.getenv("easyIstrPath") + os.sep + "bin" + os.sep
#pythonAppPath = os.getenv("easyIstrPath") + os.sep + "python" + os.sep

#tempFolderにコピーする時、shrinkするkeywordを定義（これ以外はそのままコピーする）
shrinkKeywords = ["!NODE", "!ELEMENT", "!EGROUP", "!SGROUP", "!NGROUP",
                  "!EQUATION", "!INITIAL CONDITION", "!INITIALCONDITION"]

#------ data 変換--------------------
#
#  getElementTypeFistr2inp
#    fistrの要素タイプをinp用に変換
def getElementTypeFistr2inp(nameType):
    data = [["341", "tet"], ["342", "tet2"], ["361", "hex"], ["362", "hex2"],
            ["351", "prism"], ["352", "prism2"]]
    for names in data:
        if names[0] == nameType:
            inpType = names[1]
    return inpType

#
#  getNodesOrderOfElement
#    faceNoからelementのnodeNoを取得
#    nodeNoの並びは、fistrの並び順
def getNodesOrderOfElement(elType, faceNo):
    data = [["tet", [[1,2,3], [1,2,4], [2,3,4], [1,3,4]]],
            ["tet2", [[1,2,3], [1,2,4], [2,3,4], [1,3,4]]],
            ["hex", [[1,2,3,4], [5,6,7,8], [1,2,5,6], [2,3,6,7], [3,4,7,8],
                     [1,4,5,8]]],
            ["hex2", [[1,2,3,4], [5,6,7,8], [1,2,5,6], [2,3,6,7], [3,4,7,8],
                     [1,4,5,8]]],
            ["prism", [[1,2,3], [4,5,6], [1,2,4,5], [2,3,5,6], [1,3,4,6]]],
            ["prism2", [[1,2,3], [4,5,6], [1,2,4,5], [2,3,5,6], [1,3,4,6]]]]
    nodes = []
    for typeFaceNo in data:
        if typeFaceNo[0] == elType:
            nodes = typeFaceNo[1][faceNo-1]
            break
    return nodes

#
#  changeMeshScale
#    meshのscaleを変更
def changeMeshScale(meshHeaderData, scale):
    for n in range(len(meshHeaderData)):
        line = meshHeaderData[n][0]
        if line[:len("!NODE")] == "!NODE":
            data = meshHeaderData[n][1]
            for i in range(len(data)):
                data[i][1] = data[i][1] * scale
                data[i][2] = data[i][2] * scale
                data[i][3] = data[i][3] * scale
            meshHeaderData[n][1] = data
    return meshHeaderData

#
#  changeEquationScale
#----------------------
def changeEquationScale(meshHeaderData,scale):
    """ solidとshell or beamを接続する為のequationの値をscale変更する。"""
    for n in range(len(meshHeaderData)):
        line = meshHeaderData[n][0]
        if line[:len("!EQUATION")] == "!EQUATION":
            dataLines = meshHeaderData[n][1]
            for i in range(1, len(dataLines), 2):
                words = dataLines[i][:-1].split(",")
                words[8] = " " + str(float(words[8]) * scale)
                words[11] = " " + str(float(words[11]) * scale)
                dataLines[i] = ",".join(words) + ls
            meshHeaderData[n][1] = dataLines
    return meshHeaderData

#
#  getNodesFromElementSurface
#    elementとfaceNoからnodesを取得
def getNodesFromElementSurface(elements, elementNo, faceNo):
    elType = elements[elementNo][0]
    nodesAll = elements[elementNo][1:]
    nodesStd = getNodesOrderOfElement(elType, faceNo)
    nodes = []
    for node in nodesStd:
        nodes.append(nodesAll[node-1])
    return nodes

#
#  getSurfaceGroupsInLines
#    surfaceGroupを取得
def getSurfaceGroupsInLines(lines):
    surGrp = []
    i = 0
    while i < len(lines):
        i = getNextHeaderName(lines, i, "!SGROUP", "")
        if i < len(lines):
            i += 1
            [surs, i] = getOnlyDataInLines(lines, i)
            surGrp.append(surs)
    return surGrp

#
#  getNodeGroupsInLines
#    nodeGroupを取得
def getNodeGroupsInLines(lines):
    nodeGrp = []
    i = 0
    while i < len(lines):
        i = getNextHeaderName(lines, i, "!NGROUP", "")
        if i < len(lines):
            i += 1
            [nodes, i] = getOnlyDataInLines(lines, i)
            nodeGrp.append(nodes)
    return nodeGrp

#  getOnlyDataInLines
#    指定行からdataのみ取得
def getOnlyDataInLines(lines, i):
    data = []
    while i < len(lines):
        if lines[i][0] != "!":
            if lines[i][0] != "#":
                words = deleteSp(lines[i]).split(",")
                for word in words:
                    if word != "":
                        data += [word]
        else:
            break
        i += 1
    return [data, i]

#  deleteOnlyDataInLines
#    指定行からdataのみ削除
def deleteOnlyDataInLines(lines, i):
    ip = i
    while i < len(lines):
        if lines[i][0] == "!" or lines[i][0] == "#":
            break
        i += 1
    newLines = lines[:ip] + lines[i:]
    return [newLines, ip]

#
#  getMeshDataForParaview
#    paraview用のmeshData(nodeとelement）を取得
def getMeshDataForParaview(fileName):
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    nNode = int(lines[3].split()[0])
    nElement = int(lines[3].split()[1])
    meshLines = lines[4:nNode+nElement+4]
    return [meshLines, nNode, nElement]

#
#  addMeshDataConvVisData
#    fistr用結果dataをvis用に変換
def addMeshDataConvVisData(lines, meshData):
    #結果データを成形
    #newLine = []
    i = 1
    nNode = int(lines[i].split()[0])
    nElement = int(lines[i].split()[1])
    i += 1
    nNodeVal = int(lines[i].split()[0])
    nElementVal = int(lines[i].split()[1])
    i += 1
    #nodeのデータ取得
    nodeValName = lines[i:i+nNodeVal+1]
    words = lines[i].split()
    nValue = 0
    for word in words:
        nValue += int(word)
    i += nNodeVal + 1
    newNodeVal = []
    n = 0
    while n < nNode:
        No = lines[i].split()[0]
        line = No + " "
        i += 1
        nVal = 0
        while nVal < nValue:
            #subLine = lines[i].split(ls)[0]
            subLine = deleteLsOfLine(lines[i])
            line += subLine + " "
            nVal += len(subLine.split())
            i += 1
        newNodeVal.append(line + ls)
        n += 1
    #elementのデータを取得
    if i < len(lines):
        elementValName = lines[i:i+nElementVal+1]
        nValue = 0
        words = lines[i].split()
        for word in words:
            nValue += int(word)
        i += nElementVal + 1
        newElementVal = []
        n = 0
        while n < nElement:
            No = lines[i].split()[0]
            line = No + " "
            i += 1
            nVal = 0
            while nVal < nValue:
                subLine = deleteLsOfLine(lines[i])
                line += subLine + " "
                nVal += len(subLine.split())
                i += 1
            newElementVal.append(line + ls)
            n += 1
    else:
        elementValName = []
        newElementVal = []
    #変数名の修正
    nodeValName[0] = str(nNodeVal) + " " + " ".join(nodeValName[0].split()) + ls
    i = 0
    while i < nNodeVal:
        nodeValName[i+1] = deleteLsOfLine(nodeValName[i+1]) + ", unit_unknown" + ls
        i += 1
    if len(elementValName) != 0:
        elementValName[0] = (str(nElementVal) + " " +
                " ".join(elementValName[0].split()) + ls)
        i = 0
        while i < nElementVal:
            elementValName[i+1] = (deleteLsOfLine(elementValName[i+1]) +
                ", unit_unknown" + ls)
            i += 1
    #newData作成
    newLines = [" ".join([str(nNode), str(nElement), str(nNodeVal), str(nElementVal), str(nNodeVal+nElementVal)]) + ls]
    newLines += meshData
    newLines += nodeValName
    newLines += newNodeVal
    newLines += elementValName
    newLines += newElementVal
    return newLines

#
#  readCsvData
#    csvデータの読み込み a☓bのtableDataを読み込む
def readCsvData(fileName):
    #file読込
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    #tab区切りorカンマ区切りのチェック
    line = lines[0]
    if len(line.split("\t")) != 1:
        #tab区切り
        #  tabの代わりに「,」を設定
        contents = []
        for line in lines:
            contents.append(line.replace("\t", ","))
        cont = csv.reader(contents)
    else:
        #カンマ区切り
        cont = csv.reader(lines)
    data = []
    for row in cont:
        if len(row) > 0:
            data.append(row)
    return data

#
#  isPrincipalStressInVtk
#    結果fileに主応力が追加されているかどうかチェック
def isPrincipalStressInVtk(lines):
    flag = False
    for line in lines:
        n = line.find("principalSigma1")
        if n >= 0:
            flag = True
            break
    return flag

#
#  isPrincipalStressVecInVtk
#    結果fileに主応力Vectorが追加されているかどうかチェック
def isPrincipalStressVecInVtk(lines):
    flag = False
    for line in lines:
        n = line.find("principalVectorSigma1")
        if n >= 0:
            flag = True
            break
    return flag

#
#  isStressMisesInVtk
#    結果fileにmises応力が追加されているかどうかチェック
def isStressMisesInVtk(lines):
    flag = False
    for line in lines:
        n = line.find("ndMisesStress")
        if n >= 0:
            flag = True
            break
    return flag

#
#  isPrincipalStrainInVtk
#    結果fileに主ひずみが追加されているかどうかチェック
def isPrincipalStrainInVtk(lines):
    flag = False
    for line in lines:
        n = line.find("principalEpsilon1")
        if n >= 0:
            flag = True
            break
    return flag

#
#  isPrincipalStrainVecInVtk
#    結果fileに主ひずみVectorが追加されているかどうかチェック
def isPrincipalStrainVecInVtk(lines):
    flag = False
    for line in lines:
        n = line.find("principalVectorEpsilon1")
        if n >= 0:
            flag = True
            break
    return flag

#
#  isStrainMisesInVtk
#    結果fileにmisesひずみが追加されているかどうかチェック
def isStrainMisesInVtk(lines):
    flag = False
    for line in lines:
        n = line.find("ndMisesStrain")
        if n >= 0:
            flag = True
            break
    return flag

#
#  isNodalStressInVtk
#    結果fileに応力tensorがあるかチェック
def isNodalStressInVtk(lines, stressTensor):
    flag = False
    for line in lines:
        n = line.find(stressTensor+" ")
        if n >= 0:
            flag = True
            break
    return flag

#
#  isNodalStrainInVtk
#    結果fileにひずみtensorがあるかチェック
def isNodalStrainInVtk(lines, strainTensor):
    flag = False
    for line in lines:
        n = line.find(strainTensor+" ")
        if n >= 0:
            flag = True
            break
    return flag

#
# isNameInVtk
#    結果file内の単語を検索する
def isNameInVtk(name, lines):
    flag = False
    for line in lines:
        n = line.find(name)
        if n >= 0:
            flag = True
            break
    return flag

#  findNumberFromList
#    挟み撃ち法でlist中からnewNoを返す
#      入力:[[oldNo, newNo]...], No
#      戻り:newNo
def findNumberFromList(nodes, No):
    newNode = "0"
    if len(nodes) > 10:
        i = int(len(nodes)/2)
        node = nodes[i][0]
        if int(No) < int(node):
            newNodes = nodes[:i]
            newNode = findNumberFromList(newNodes, No)
            return newNode
        else:
            newNodes = nodes[i:]
            newNode = findNumberFromList(newNodes, No)
            return newNode
    else:
        for node in nodes:
            if node[0] == No:
                newNode = node[1]
                break
        return newNode


#
#  renumberMeshData
#    meshDataのみをrenumberする
def renumberMeshData(meshData, nNode, nElement):
    nodeData = meshData[:nNode]
    elementData = meshData[nNode:]
    #tmpデータ（list）を作成
    tmpNodes = []
    i = 1
    for line in nodeData:
        words = line.split()
        nodeNo = int(words[0])
        tmpNodes.append([nodeNo, i])
        nodeData[i-1] = str(i) + " " + " ".join(words[1:]) + ls
        i += 1
    maxNodeNo = max(tmpNodes)[0]
    nodeNoList = []
    #配列を準備
    nodeNoList = [ "" for i in range(maxNodeNo+1) ]
    #list作成
    for [nodeNo, newNo] in tmpNodes:
        nodeNoList[nodeNo] = newNo
    #elementを修正
    i = 1
    for line in elementData:
        words = line.split()
        words[0] = str(i)
        newList = []
        for node in words[3:]:
            n = int(node)
            newNode = str(nodeNoList[n])
            newList.append(newNode)
        lineList = words[:3] + newList
        elementData[i-1] = " ".join(lineList) + ls
        i += 1
    newMeshData = nodeData + elementData
    return newMeshData

#
#  renumberRssultDataForParaview
#    node, elementのNoを修正する。
def renumberResultDataForParaview(lines):
    words = lines[0].split()
    nNode = int(words[0])
    nElem = int(words[1])
    #meshDataがrenumberされているかチェック
    if (nNode == int(lines[nNode].split()[0]) and
                nElem == int(lines[nNode+nElem].split()[0])):
        pass
    else:
        #meshDataをrenumberする
        meshLines = lines[1:1+nNode+nElem]
        meshLines = renumberMeshData(meshLines, nNode, nElem)
        lines = lines[:1] + meshLines + lines[1+nNode+nElem:]
    #nodeデータを修正
    i = nNode + nElem + 1
    words = lines[i].split()
    ir = i + int(words[0]) + 1
    i = 0
    while i < nNode:
        line = lines[i+ir]
        words = line.split()
        lines[i+ir] = str(i+1) + " " + " ".join(words[1:]) + ls
        i += 1
    #elementデータを修正
    ip = i + ir
    if ip < len(lines):
        nVal = int(lines[ip].split()[0])
        ir = ip + nVal + 1
        n = 0
        while n < nElem:
            words = lines[ir+n].split()
            lines[ir+n] = str(n+1) + " " + " ".join(words[1:]) + ls
            n += 1
    return lines

#
#  convertResultDataForParaview
#    UCD formatを旧のformatに変換する
def convertResultDataForParaview(lines):
    #3行skip
    lines = lines[3:]
    #node数、element数を取得
    words = lines[0].split()
    nNodes = int(words[0])
    nElements = int(words[1])
    #変換する全行を取得（要素座標直後の行を削除）
    i = nNodes + nElements
    newLines = lines[:i+1] + lines[i+2:]
    #出力変数の数を取得
    while i < len(lines):
        if lines[i][0] >= "A":
            break
        i += 1
    nVal = 0
    while i < len(lines):
        if "A" <= lines[i][0]:
            nVal += 1
        else:
            break
        i += 1
    #1行目を修正（節点の変数の数、要素の変数の数、モデルの変数の数）
    words = newLines[0].split()
    words += [str(nVal), "0", str(nVal)]
    newLines[0] = " ".join(words) + ls
    return newLines    

#
#  getEndPoint
#    !ENDを最後から検索してpointerを返す
def getEndPoint(lines):
    ip = 0
    i = len(lines) - 1
    while i > 0:
        line = lines[i]
        if line[:4] == "!END" or line[:4] == "!end" or line[:4] == "!End":
            ip = i
            break
        i -= 1
    return ip

#
#  getInsertPointInCnt
#    lineを追加する時の挿入場所を取得
#    keywordsの次の行No、存在しない場合はheader(!DLOAD等）の行Noが戻る
def getInsertPointInCnt(lines, header, keywords):
    ip = 0
    #keywordを検索
    i = 0
    while i < len(lines):
        line = lines[i]
        if line[0] == "#":
            flag = 0
            for key in keywords:
                if line.find(key) >= 0:
                    flag = 1
                    break
            if flag == 1:
                i += 1
                if lines[i][0] == "#":
                    i += 1
                ip =i
                break
        i += 1
    #keywordに該当したか？
    if ip == 0:
        #headerを検索
        i = 0
        while i < len(lines):
            line = lines[i]
            if line[:len(header)] == header:
                ip = i
                break
            i += 1
    return ip

#
#  skipStepId
#    nStep分skipする
def skipStepId(lines, nStep):
    if nStep <= 0:
        return 0

    i = 0
    while i < len(lines):
        if lines[i][:len("!STEP")] == "!STEP":
            nStep -= 1
            if nStep <= 0:
                i += 1
                break
        i += 1
    return i

#
#  getBndInsertPointInCnt
#    境界条件の挿入行取得専用（nStepも検索する）
#    lineを追加する時の挿入場所を取得
#    keywordsの次の行No、存在しない場合はheader(!DLOAD等）の行Noが戻る
def getBndInsertPointInCnt(lines, header, keywords, nStep):
    if nStep <= 0:
        ip = getInsertPointInCnt(lines, header, keywords)
    else:
        ip = 0
        i = skipStepId(lines, nStep)
        while i < len(lines):
            if lines[i][0] == "!" or lines[i][0] == "#":
                ip = i
                break
            i += 1
    return ip 

#
#  remakeGRPID
#    header（!DLOAD等）のGRPIDを連番に修正する
def remakeGRPID(lines, header):
    if header == "!CLOAD" or header == "!DLOAD" or header == "!TEMPERATURE":
        #同じGRPIDとして連番で設定する
        grpId = 1
        i = 0
        while i < len(lines):
            line = lines[i]
            if line[0] == "!":
                lineName = deleteSp(line).split(",")[0]
                if (lineName == "!CLOAD" or lineName == "!DLOAD"
                    or lineName == "!TEMPERATURE"):
                    lines[i] = setValName(line, "GRPID="+str(grpId))
                    grpId += 1
            i += 1
    else:
        #単独で連番を設定する
        grpId = 1
        i = 0
        while i < len(lines):
            line = lines[i]
            if line[:len(header)] == header:
                if deleteSp(lines[i]).split(",")[0] == header:
                    lines[i] = setValName(line, "GRPID="+str(grpId))
                    grpId += 1
            i += 1
    return lines

#
#  getNextHeaderName
#    header名（!SOLVER等）を取得
def getNextHeaderName(lines, i, headerName, itemName):
    while i < len(lines):
        if lines[i][:len(headerName)] == headerName:
            if deleteSp(lines[i]).split(",")[0] == headerName:
                if itemName == "":
                    break
                else:
                    words = itemName.split("=")
                    if getValName(lines[i], words[0]) == words[1]:
                        break
        i += 1
    return i

#
#  skipNextHeader
#
def skipNextHeader(lines, i):
    ans = -1
    while i < len(lines):
        if lines[i][0] == "!" or lines[i][0] == "#":
            ans = i
            break
        i += 1
    return ans

#
#  deleteHeaderName
#    指定したheaderを削除
def deleteHeaderName(lines, i, header, keyName):
    while i < len(lines):
        i = getNextHeaderName(lines, i, header, keyName)
        if i < len(lines):
            lines = delete1header(lines, i)
    return lines

#
#  delete1header
#    指定のheader位置からheader1ヶ分を削除（「#」または「!」まで
def delete1header(lines, i):
    ip = i
    i += 1
    while i < len(lines):
        if lines[i][0] == "#" or lines[i][0] == "!":
            break
        i += 1
    newLines = lines[:ip] + lines[i:]
    return newLines

#
#  getNextDloadGrpType
#    DLOADの値を取得
#    戻り値　grepTypes:[[grep, bndType, line, i], ...]
def getNextDloadGrpType(lines, i):
    ans = []
    flag = 0
    while i < len(lines):
        line = lines[i]
        if line[:len("!DLOAD")] == "!DLOAD":
            defLine = line
            i += 1
            flag = 1
            grpType = []
            while lines[i][0] != "!" and i < len(lines):
                if lines[i][0] != "#":
                    line = deleteSp(lines[i])
                    words = line.split(",")
                    #grpType += [[words[0], words[1], line, i]]
                    vals = [words[0], words[1], line, i]
                    if words[1] == "S":
                        forceType = getValName(defLine, "forceType")
                        mapFile = getValName(defLine, "mapFile")
                        mapCoeff = getValName(defLine, "mapCoeff")
                        mapShift = getValName(defLine, "mapShift")
                        vals += [forceType, mapFile, mapCoeff, mapShift]
                    grpType +=[vals]
                i += 1
        else:
            i += 1
        if flag == 1:
            ans = grpType
            break
    return ans

#
#  getNoGroups
#    allGroupとselGroupからunselectのgroupを返す
def getNoGroups(allGroups, selGroups):
    noSels = []
    for allGroup in allGroups:
        flag = 0
        for selGroup in selGroups:
            if allGroup == selGroup or "dummy_"+allGroup == selGroup:
                flag = 1
                break
        if flag == 0:
            noSels.append(allGroup)
    return noSels

#
#  getSolutionTypeFromCnt
#    solutionTypeを取得
def getSolutionTypeFromCnt():
    #values = []
    lines = readFistrModelCnt()
    i = getNextHeaderName(lines, 0, "!SOLUTION", "")
    solType = getValName(lines[i], "TYPE")
    if solType == "DYNAMIC":
        i = getNextHeaderName(lines, 0, "!DYNAMIC", "")
        i += 1
        if i < len(lines):
            words = deleteSp(lines[i]).split(",")
            if len(words) >= 2:
                if words[1] == "2":
                    solType = "DYNAMIC_F"
    elif solType == "HEAT":
        i = getNextHeaderName(lines, 0, "!HEAT", "")
        i += 1
        if i < len(lines):
            words = deleteSp(lines[i]).split(",")
            try:
                dt = float(words[0])
            except:
                dt = 0.0
            if dt > 0.0:
                solType = "HEAT_D"
    return solType

#
#  cntMshFileName
#-----------------
def cntMshFileName(workDir):
    """ hecmw_ctrl.datから、cnt, mshファイル名を取得して返す"""
    cntName = "FistrModel.cnt"
    mshName = "FistrModel.msh"
    fileName = workDir + os.sep + "hecmw_ctrl.dat"
    if os.path.exists(fileName) == False:
        return cntName, mshName
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    #cntFile
    i = getNextHeaderName(lines, 0, "!CONTROL", "NAME=fstrCNT")
    i += 1
    if i < len(lines):
        cntName = lines[i].split()[0]
    #mshFile
    i = getNextHeaderName(lines, 0, "!MESH", "TYPE=HECMW-ENTIRE")
    i += 1
    if i < len(lines):
        mshName = lines[i].split()[0]
    return cntName, mshName

#
#  getCntMshFileNameDict
#--------------------
def getCntMshFileNameDict(workDir, datFile="hecmw_ctrl.dat"):
    """ hecmw_ctrl.datからcnt, mshファイル名を辞書形式で取得する。
    
    Args:
        workDir:    directoryName
    Returns:
        nameDict (dict): 辞書(keys="cnt", "msh", "fistr", "res", "vis","restart")
        """
    nameDict = {
        "cnt":  "FistrModel.cnt",
        "msh":  "FistrModel.msh",
        "fstrMSH": "FistrModel.msh",
        "res":  "",
        "vis":  "",
        "restart": ""
        }
    #fileName = workDir + os.sep + "hecmw_ctrl.dat"
    fileName = workDir + os.sep + datFile
    if os.path.exists(fileName) == False:
        return nameDict
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    i = 0
    while i < len(lines):
        line = lines[i]
        if line[:len("!MESH")] == "!MESH":
            if line.find("TYPE=HECMW-ENTIRE") >= 0:
                nameDict["msh"] = lines[i+1].split()[0]
            if line.find("NAME=fstrMSH") >= 0:
                nameDict["fstrMSH"] = lines[i+1].split()[0]
            i += 1
        elif line[:len("!CONTROL")] == "!CONTROL":
            if line.find("NAME=fstrCNT") >= 0:
                nameDict["cnt"] = lines[i+1].split()[0]
            i += 1
        elif line[:len("!RESULT")] == "!RESULT":
            if line.find("NAME=fstrRES") >= 0:
                nameDict["res"] = lines[i+1].split()[0]
            elif line.find("NAME=vis_out") >= 0:
                nameDict["vis"] = lines[i+1].split()[0]
            i += 1
        elif line[:len("!RESTART")] == "!RESTART":
            if line.find("IO=OUT") >= 0:
                nameDict["restart"] = lines[i+1].split()[0]
            i += 1
        i += 1
    return nameDict

#
#  setCntMshFileNameDict
#-------------------------
def setCntMshFileNameDict(workDir, nameDict):
    """ hecmw_ctrl.datの内容をdictの内容に書き変える。"""
    
    def changeMshName(newName, lines):
        i = getNextHeaderName(lines, 0, "!MESH", "TYPE=HECMW-ENTIRE")
        i += 1
        if i < len(lines):
            lines[i] = newName + "\n"
        i = getNextHeaderName(lines, i, "!MESH", "TYPE=HECMW-ENTIRE")
        i += 1
        if i < len(lines):
            lines[i] = newName + "\n"
        return lines

    def changeName(newName, lines, keyName, typeName):
        i = getNextHeaderName(lines, 0, keyName, typeName)
        i += 1
        if i < len(lines):
            lines[i] = newName + "\n"
        return lines

    fileName = workDir + os.sep + "hecmw_ctrl.dat"
    if os.path.exists(fileName) == False:
        return nameDict
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    for key in nameDict.keys():
        if key == "msh":
            newName = nameDict["msh"]
            lines = changeMshName(newName, lines)
        elif key == "cnt":
            newName = nameDict["cnt"]
            lines = changeName(newName, lines, "!CONTROL", "NAME=fstrCNT")
        elif key == "res":
            newName = nameDict["res"]
            lines = changeName(newName, lines, "!RESULT", "NAME=fstrRES")
            lines = changeName(newName, lines, "!RESULT", "IO=IN")
        elif key == "vis":
            newName = nameDict["vis"]
            lines = changeName(newName, lines, "!RESULT", "NAME=vis_out")
        elif key == "restart":
            newName = nameDict["restart"]
            lines = changeName(newName, lines, "!RESTART", "IO=OUT")
            lines = changeName(newName, lines, "!RESTART", "IO=IN")
    f = open(fileName, "w"); f.writelines(lines); f.close()

#
#  setCntFileName
#-----------------
def setCntFileName(workDir, cntName):
    """ hecmw_ctrl.datから、cntファイル名をセットする"""
    fileName = workDir + os.sep + "hecmw_ctrl.dat"
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    i = getNextHeaderName(lines, 0, "!CONTROL", "NAME=fstrCNT")
    i += 1
    lines[i] = cntName + ls
    f=open(fileName, "w"); f.writelines(lines); f.close()

#
#  readFistrModelCnt
#    cntファイルを読み込み
def readFistrModelCnt():
    #fileName = "FistrModel.cnt"
    workDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = cntMshFileName(workDir)
    fileName = FistrModel_cnt
    if len(glob.glob(fileName)) == 0:
        return []
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    return lines

#
#  readFistrModelMsh
#    mshファイルを読み込み
def readFistrModelMsh():
    #fileName = "FistrModel.msh"
    workDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = cntMshFileName(workDir)
    fileName = FistrModel_msh
    if len(glob.glob(fileName)) == 0:
        return []
    f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    return lines

# #
# #  writeFistrModelCnt
# #
# def writeFistrModelCnt(lines):
#     """ FistrModel.cntに保存する"""
#     f = open("FistrModel.cnt", "w")
#     f.writelines(lines)
#     f.close()

#
#  writeFistrModelMsh
#
def writeFistrModelMsh(lines):
    workDir = os.getcwd()
    FistrModel_cnt, FistrModel_msh = cntMshFileName(workDir)
    fileName = FistrModel_msh
    #f = open("FistrModel.msh", "w")
    f = open(fileName, "w")
    f.writelines(lines)
    f.close()

#
#  getHeaderData
#    headerとそのdataをテキストで取得
def getHeaderData(lines):
    flag = 0
    headerData = []
    data = []
    header = ""
    for i in range(len(lines)):
        line = lines[i]
        if line == "":
            #空文の処理
            flag = 1
        elif line[:1] == "!" and line[:2] != "!!":
            #headerの処理
            headerData.append([header, data])
            header = lines[i]
            data = []
            flag = 1
        elif line[:2] == "!!" or line[0] == "#":
            #commentの処理
            headerData.append([header, data])
            header = line
            data = []
            flag = 1
        else:
            #header以外の処理
            flag = 0
        #data部を作成
        if flag == 0:
            data.append(lines[i])
    headerData.append([header, data])
    #最初に["", []]が入るので、最初を除いて戻す
    return headerData[1:]

# #
# #  writeHeaderData
# #
# def writeHeaderData(headerData):
#     """ headerDataを保存する"""
#     lines = []
#     for header in headerData:
#         lines += [header[0]]
#         lines += header[1]
#     writeFistrModelCnt(lines)

#  getNodeData
#
def getNodeData(headerData, pFlag, q):
    """ node行を文字から数値に変換する。

    Args:
        headerData list :nodeデータ
                         [[<header行>, list(str)], ...]
        pFlag  int      :印刷flag　0:印刷する, 1:印刷しない
        q  0 or Queue   :subprocessの場合は、queueObject,
                         mainの場合は、「0」
    Returns:
        list            :subprocessの場合は、q.put(),
                        :mainの場合は、returnで戻す"""
    i = 0
    for hData in headerData:
        if pFlag == 0:
            print("  getting NODE...")
        dLines = hData[1]
        data = []
        for line in dLines:
            words = line.split(",")
            dataLine = [int(words[0])] + list(map(float, words[1:]))
            data.append(dataLine)
        headerData[i][1] = data
        i += 1
    if q == 0:
        return headerData
    else:
        q.put(headerData)

#  getElmData
#
def getElmData(headerData, pFlag):
    """ element行を文字から数値に変換する。
    
    Args:
        headerData list :elementデータ
                         [[<header行>, list(str)], ...]
        pFlag  int      :印刷flag　0:印刷する, 1:印刷しない
    Returns:
        list            :変換結果を戻す"""
    i = 0
    for hData in headerData:
        if pFlag == 0:
            typeName = getValName(hData[0], "TYPE")
            print("  getting ELEMENT " + typeName + "...")
        dLines = hData[1]
        data = []
        for line in dLines:
            line = line.rstrip()
            if line != "":
                words = line.split(",")
                elData = list(map(int, words))      #wordsをintListに変換
            else:
                elData = []
            data.append(elData)
        headerData[i][1] = data
        i += 1
    return headerData

#  getGrpData
#
def getGrpData(headerData, pFlag, q):
    """ group行を文字から数値に変換する。
        （NGROUP, SGROUP, EGROUPの行）

    Args:
        headerData list :groupデータ
                         [[<header行>, list(str)], ...]
        pFlag  int      :印刷flag　0:印刷する, 1:印刷しない
        q  0 or Queue   :subprocessの場合は、queueObject,
                         mainの場合は、「0」
    Returns:
        list            :subprocessの場合は、q.put(),
                        :mainの場合は、returnで戻す"""
    i = 0
    for hData in headerData:
        header = hData[0]
        if pFlag == 0:
            item = deleteSp(header).split(",")[0][1:]
            if item == "EGROUP":
                typeName = getValName(header, "EGRP")
            elif item == "SGROUP":
                typeName = getValName(header, "SGRP")
            elif item == "NGROUP":
                typeName = getValName(header, "NGRP")
            print("  getting " + item + " " + typeName + "...")
        dLines = hData[1]
        data = []
        for line in dLines:
            words = line.split(",")
            elData = list(map(int, words[:-1]))
            #words最後が空白以上の文字かどうかチェック
            if skipSp(words[-1], 0) != len(words[-1]):
                elData.append(int(words[-1]))
            data += elData
        headerData[i][1] = data
        i += 1
    if q == 0:
        return headerData
    else:
        q.put(headerData)

#
#  getHeaderNumData
#    headerとそのnumDataを取得
#    NODE:      [nodeNo, locx, locy, locz]形式で数値dataを取得
#    ELEMENT:   [elNo, node1, node2, node3, ...]形式で数値データを取得
#    EGROUP:    [ele1, ...]形式で数値データを取得
#    SGROUP:    [sur1, ...]形式で数値データを取得
#    NGROUP:    [node1, ...]形式で数値データを取得
def getHeaderNumData(lines):
    print (_(u"メッシュデータを読み込み中..."))
    headerData = getHeaderData(lines)
    #printFlagを設定
    pFlag = 0
    for hData in headerData:
        if hData[0][:len("!NODE")] == "!NODE":
            if len(hData[1]) < 20:
                #node数が20以下の場合、flagを設定
                pFlag = 1
                break
    nodeHeaders = []; nodeHeaderNo = []
    elmHeaders = []; elmHeaderNo = []
    grpHeaders = []; grpHeaderNo = []
    i = 0
    for hData in headerData:
        header = hData[0]
        #nodeDataを取得
        if header[:len("!NODE")] == "!NODE":
            nodeHeaders.append(hData)
            nodeHeaderNo.append(i)
        #elementDataを取得
        elif header[:len("!ELEMENT")] == "!ELEMENT":
            elmHeaders.append(hData)
            elmHeaderNo.append(i)
        #groupDataを取得
        elif (header[:len("!EGROUP")] == "!EGROUP" or header[:len("!SGROUP")] == "!SGROUP"
            or header[:len("!NGROUP")] == "!NGROUP"):
            grpHeaders.append(hData)
            grpHeaderNo.append(i)
        i += 1
    #nCpusを確認
    nCpus = mp.cpu_count()
    #nCpu数に応じて実行
    if nCpus == 1:
        #singleCoreで順番に実行
        nodeData = getNodeData(nodeHeaders, pFlag, 0)
        elmData = getElmData(elmHeaders, pFlag)
        grpData = getGrpData(grpHeaders, pFlag, 0)
    elif nCpus == 2:
        #2coreで実行（nodeをsubprocessで実行）
        nodeQueue = mp.Queue()
        nodeProcess = mp.Process(
            target=getNodeData,
            args=(nodeHeaders, pFlag, nodeQueue) )
        nodeProcess.start()
        elmData = getElmData(elmHeaders, pFlag)
        nodeData = nodeQueue.get()
        grpData = getGrpData(grpHeaders, pFlag, 0)
    elif nCpus > 2:
        #3coreで実行（node、groupをsubprocessで実行）
        nodeQueue = mp.Queue()
        nodeProcess = mp.Process(
            target=getNodeData,
            args=(nodeHeaders, pFlag, nodeQueue) )
        nodeProcess.start()
        grpQueue = mp.Queue()
        grpProcess = mp.Process(
            target=getGrpData,
            args=(grpHeaders, pFlag, grpQueue) )
        grpProcess.start()
        elmData = getElmData(elmHeaders, pFlag)
        nodeData = nodeQueue.get()
        grpData = grpQueue.get()
    #全結果をheaderDataに格納
    for i in range(len(nodeHeaderNo)):
        hNo = nodeHeaderNo[i]
        headerData[hNo][1] = nodeData[i][1]
    for i in range(len(elmHeaderNo)):
        hNo = elmHeaderNo[i]
        headerData[hNo][1] = elmData[i][1]
    for i in range(len(grpHeaderNo)):
        hNo = grpHeaderNo[i]
        headerData[hNo][1] = grpData[i][1]
    if pFlag == 0:
        print("... done")
    return headerData

#
#  shrinkHeaderNumData
#    headerNumDataをtempCase用に縮める
def shrinkHeaderNumData(headerNumData):
    newHeaderData = []
    for headerData in headerNumData:
        header = headerData[0]
        data = headerData[1][:20]
        newHeaderData.append([header, data])
    return newHeaderData

#
#  createLinesFistrModelMshHeader
#    meshHeaderDataからlinesを作成
def createLinesFistrModelMshHeader(meshHeaderData):
    lines = []
    for headerData in meshHeaderData:
        line = headerData[0]
        lines.append(line)
        data = headerData[1]
        if line[:len("!NODE")] == "!NODE":
            for vals in data:
                #座標値（float）変換後、指数表記の場合、小数点が無いとFist側でエラー
                #  が発生するので、このチェックを含む変換。
                x = convNum2str(vals[1])
                y = convNum2str(vals[2])
                z = convNum2str(vals[3])
                strVals = [str(vals[0]), x, y, z]
                line = ", ".join(strVals) + "\n"
                lines.append(line)
        elif line[:len("!ELEMENT")] == "!ELEMENT":
            for vals in data:
                strVals = list(map(str, vals))      #valsをstrListに変換
                line = ", ".join(strVals) + "\n"
                lines.append(line)
        elif line[:len("!SGROUP")] == "!SGROUP":
            #5セット（10ヶ）毎に行を作成する
            for n in range(0, len(data), 10):
                sData = data[n:n+10]
                line = ", ".join(list(map(str, sData))) + "\n"
                lines.append(line)
        elif line[:len("!EGROUP")] == "!EGROUP" or line[:len("!NGROUP")] == "!NGROUP":
            #8ヶ毎に行を作成する。
            for n in range(0, len(data), 8):
                vals = data[n:n+8]
                line = ", ".join(list(map(str, vals))) + "\n"
                lines.append(line)
        else:
            lines += data
    return lines

#
#  convNum2str
#    数値を文字列に変換する
#    指数表記で小数点が無い場合は、少数点を追加する
def convNum2str(num):
    word = str(num)
    if word.find(".") >= 0:
        return word
    n = word.find("e")
    if n >= 0:
        word = word[:n] + ".0" + word[n:]
    return word

#
#  readUsingApplications
#    Applicationの設定を読み込む
def readUsingApplications():

    #空白をskip
    def skipSp(line, n):
        while n < len(line):
            if line[n] != " ":
                break
            n += 1
        return n

    #空白以外をskip
    def skipNsp(line, n):
        while n < len(line):
            if line[n] == " ":
                break
            n += 1
        return n

    #行末の空白を削除
    def deleteLastSp(command):
        i = len(command) - 1
        while i >= 0:
            if command[i] > " ":
                break
            i -= 1
        command = command[:i+1]
        return command

    #command（設定値）を取得
    def getCommand(line):
        n = skipSp(line, 0)
        n = skipNsp(line, n)
        n = skipSp(line, n)
        command = line[n:]
        if command != "":
            command = deleteLastSp(command)
        return command

    #usingAppの読み込み
    fileName = os.getenv("easyIstrUserPath") + os.sep + "data" + os.sep + "usingApp"
    try:
        f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    except:
        f=open(fileName, encoding="shift-jis"); lines=f.readlines(); f.close()
    if os.name == "nt":
        osName = "windows"
    else:
        osName = "linux"
    #コメント部を削除
    lines = deleteComment(lines)
    #変数の初期化
    if osName == "linux":
        #linux
        language = "Japanese"
        office = ""; terminal = "gnome-terminal"; terminalRun = "gnome-terminal -- "
        fileManager = ""; editor = ""; frontIstrFolder = ""; paraView = ""
        gnuplot = ""; cad = ""; mesher = ""
    else:
        #windows
        language = "Japanese"
        office = ""; terminal = "start cmd"; terminalRun = "start cmd /c "
        fileManager = ""; editor = ""; frontIstrFolder = ""; paraView = ""
        gnuplot = ""; cad = ""; mesher = ""
    #辞書作成（default）
    appDict = {
        "language": language,
        "office": office,
        "terminal": terminal,
        "terminalRun": terminalRun,
        "fileManager": fileManager,
        "editor": editor,
        "frontIstrFolder": frontIstrFolder,
        "paraView": paraView,
        #"REVOCAP": revocap,
        "gnuplot": gnuplot,
        "CAD": cad,
        "mesher": mesher
        }
    setFlag = [0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0]
    flag = 0
    #keyWordと設定値を取得
    for line in lines:
        words = line.split()
        #osに従う内容を読み込むflagを設定
        if line[0] > " ":
            if words[0] == osName:
                flag = 1
            else:
                flag = 0
        #keywordと設定値読み込む
        if flag == 1 and line[0] == " " and len(words) > 0:
            if words[0] == "office":
                if setFlag[0] == 0:
                    office = getCommand(line)
                    appDict["office"] = office
                    setFlag[0] = 1
            elif words[0] == "terminal":
                if setFlag[1] == 0:
                    terminal = getCommand(line)
                    appDict["terminal"] = terminal
                    setFlag[1] = 1
            elif words[0] == "fileManager":
                if setFlag[2] == 0:
                    fileManager = getCommand(line)
                    appDict["fileManager"] = fileManager
                    setFlag[2] = 1
            elif words[0] == "editor":
                if setFlag[3] == 0:
                    editor = getCommand(line)
                    appDict["editor"] = editor
                    setFlag[3] = 1
            elif words[0] == "frontIstrFolder":
                if setFlag[4] == 0:
                    frontIstrFolder = getCommand(line)
                    appDict["frontIstrFolder"] = frontIstrFolder
                    setFlag[4] = 1
            elif words[0] == "paraView":
                if setFlag[5] == 0:
                    paraView = getCommand(line)
                    appDict["paraView"] = paraView
                    setFlag[5] = 1
            # elif words[0] == "REVOCAP":
            #     if setFlag[6] == 0:
            #         revocap = getCommand(line)
            #         appDict["REVOCAP"] = revocap
            #         setFlag[6] = 1
            elif words[0] == "gnuplot":
                if setFlag[7] == 0:
                    gnuplot = getCommand(line)
                    appDict["gnuplot"] = gnuplot
                    setFlag[7] = 1
            elif words[0] == "CAD":
                if setFlag[8] == 0:
                    cad = getCommand(line)
                    appDict["CAD"] = cad
                    setFlag[8] = 1
            elif words[0] == "mesher":
                if setFlag[9] == 0:
                    mesher = getCommand(line)
                    appDict["mesher"] = mesher
                    setFlag[9] = 1
            elif words[0] == "language":
                if setFlag[10] == 0:
                    language = getCommand(line)
                    appDict["language"] = language
                    setFlag[10] = 1
            elif words[0] == "terminalRun":
                if setFlag[11] == 0:
                    terminalRun = getCommand(line)
                    appDict["terminalRun"] = terminalRun
                    setFlag[11] = 1
        if sum(setFlag) >= len(setFlag):
            break
    if sum(setFlag) != len(setFlag):
        writeUsingApplications(appDict)
    return appDict

#
#  writeUsingApplications
#-------------------------
def writeUsingApplications(appDict):
    """ appの設定（appDict）をusingAppに書き込み"""
    #file読み込み
    fileName = os.getenv("easyIstrUserPath") + os.sep + "data" + os.sep + "usingApp"
    try:
        f=open(fileName, encoding="utf-8"); lines=f.readlines(); f.close()
    except:
        f=open(fileName, encoding="shift-jis"); lines=f.readlines(); f.close()
    #該当部範囲を取得
    if os.name == "nt":
        osName = "windows"
    else:
        osName = "linux"
    #  先頭を検索
    ist = -1
    for i in range(len(lines)):
        line = lines[i]
        if line[:len(osName)] == osName:
            ist = i + 1
            break
    if ist < 0:
        #見つからない場合
        if lines[-1][-1] != ls:
            lines[-1] += ls
        ist = len(lines)
    #  最後を検索
    ied = -1
    for i in range(ist, len(lines)):
        line = lines[i]
        if line[0] != " " and line[0] != ls and line[0] != "\t":
            ied = i - 1
            break
    if ied < 0:
        #見つからない場合
        if lines[-1][-1] != ls:
            lines[-1] += ls
        ied = len(lines) - 1
    #該当部を書き換え
    addLines = []
    for key in appDict.keys():
        flag = 0
        newLine = "    " + "%-11s" % key + " " + appDict[key] + ls
        for i in range(ist, ied):
            line = lines[i]
            words = line.split()
            if len(words) > 0:
                if words[0] == key:
                    lines[i] = newLine
                    flag = 1
                    break
        if flag == 0:
            addLines.append(newLine)
    #追加する内容があるか？
    if len(addLines) > 0:
        #  addLinesを追加する場所を検索
        irem = ist
        for i in range(ied, ist, -1):
            line = lines[i]
            words = line.split()
            if len(words) > 1:
                irem = i + 1
                break
        #addLinesを追加
        lines = lines[:irem] + addLines + lines[irem:]
    #usingAppに書き込み
    f = open(fileName, "w", encoding="utf-8"); f.writelines(lines); f.close()

#
#  deleteComment
#    commentを削除
def deleteComment(lines):
    """ commentを削除する"""
    newLines = []
    for line in lines:
        newLine = ""
        for chara in line:
            if chara == "#":
                newLine += ls
                break
            else:
                newLine += chara
        newLines.append(newLine)
    return newLines

#
#  deleteSp
#    空白とCRを削除して返す
def deleteSp(line):
    line = line.replace(" ", "")
    line = line.replace("\n", "")
    return line

#
#  getValName
#    lineからvalName(TYPEなど）を取得
#    大文字同士で比較
def getValName(line, valName):
    valName = valName.upper()
    words = deleteSp(line).split(",")
    name = ""
    for word in words:
        # 「TYPE=」などを検索
        if word.upper().find(valName+"=") >= 0:
            name = word.split("=")[1]
            break
    return name

#  setValName
#    変数の値をセット
def setValName(line, val):
    valNames = val.split("=")
    words = deleteSp(line).split(",")
    i = 0
    flag = 0
    for word in words:
        name = word.split("=")[0].upper()
        if name == valNames[0].upper():
            words[i] = " " + val
            flag = 1
            break
        i += 1
    if flag == 0:
        words.append(" "+val)
    newLine = ", ".join(words) + ls
    return newLine

#
#  deleteValName
#    line内から変数名を削除する
def deleteValName(line, delName):
    words = deleteSp(line).split(",")
    newWords = []
    for word in words:
        valSet = word.split("=")
        if len(valSet) == 2 and valSet[0] == delName:
            pass
        else:
            newWords.append(word)
    newLine = ", ".join(newWords) + ls
    return newLine

#
#  deleteSameData
#    list内から同じdataを削除する
def deleteSameData(dataList):
    if len(dataList) == 0:
        return dataList
    dataList.sort()
    bData = dataList[0]
    newDataList = [bData]
    for data in dataList:
        if data != bData:
            newDataList.append(data)
            bData = data
    return newDataList    

#
#  delFiles
#    fileを削除
def removes(filePattern):
    delNames = glob.glob(filePattern)
    if len(delNames) > 0:
        for delName in delNames:
            os.remove(delName)
    return

#
#  deleteLsOfLine
#    行末ゴードを削除
def deleteLsOfLine(line):
    if line[-2:] == "\r\n":
        newLine = line[:-2]
    else:
        newLine = line[:-1]
    return newLine

#
#  skipChara
#    空白以下をskipする
def skipChara(cont, p):
    while cont[p] > " " and p < len(cont):
        p += 1
    return p

#
#  skipSp
#    空白以下をskipする
def skipSp(cont, p):
    while p < len(cont) and cont[p] <= " ":
        p += 1
    return p

#----------------
#  run class
#----------------
class run:
    """ コマンドの実行関連"""

    def __init__(self, runDir=os.getcwd()):
        self.runDir = runDir

    def commandBG(self, comm):
        """ commandを裏で実行する"""
        # if os.name == "nt":
        #     #windowsの場合
        #     comm = "start /b cmd /c " + comm
        # else:
        #     #linuxの場合
        #     comm = comm + " &"
        # print (comm)
        # self.command(comm)
        saveDir = os.getcwd()
        os.chdir(self.runDir)
        proc = subprocess.Popen(comm, shell=True)       #直ぐに戻る
        #proc = subprocess.run(comm, shell=True)        #終了まで待つ
        os.chdir(saveDir)

    def command(self, comm):
        """ commandを実行する。終了するまで待ち、終了後に戻る。
        終了を待たずに戻る場合は、commandBGを使う。"""
        saveDir = os.getcwd()
        os.chdir(self.runDir)
        #proc = subprocess.Popen(comm, shell=True)      #直ぐに戻る
        proc = subprocess.run(comm, shell=True)         #終了まで待つ
        os.chdir(saveDir)

    def commandWithLog(self, comm):
        """ commandをlogFileをupdateしながら実行する。
        command終了するまで待ち、終了後に戻る。
        終了を待たずに戻る場合は、commandWithLogBGを使う。"""
        logFile = os.getenv("LogFile")
        commLine = "python3 " + pythonAppPath + "commTee.py -a " + comm + " " + logFile
        self.command(commLine)

    def commandWithLogBG(self, comm):
        """ commandをlogFileをupdateしながら裏で実行する。"""
        logFile = os.getenv("LogFile")
        commLine = "python3 " + pythonAppPath + "commTee.py -a " + comm + " " + logFile
        self.commandBG(commLine)

    def commandReturnCont(self, comm, isOut=True):
        """ commandを実行しその結果を受け取る
        
        Args:
            comm (str)  :コマンド
            isOut (bool):標準出力にも出力する（default=True)
        Returns:
            stat (str)      :"OK" or "ERROR"
            resCont (str)   :実行結果
            errCont (str) ) :エラーの内容"""

        def bytes2unicode(line):
            try:
                #utf-8に変換
                line = line.decode("utf-8")
            except:
                #shift-jisで読み込む。errorがあれば置き換える
                line = line.decode("shift-jis", errors="replace")
                #utf-8のbytes型に変換
                b = line.encode(encoding="utf-8", errors="replace")
                #utf-8に変換
                line = b.decode("utf-8")
            return line

        saveDir = os.getcwd()
        os.chdir(self.runDir)
        logFile = os.getenv("LogFile") 
        f = open(logFile, "a", encoding="utf-8")
        results = []
        proc = subprocess.Popen(
            comm,
            shell=True,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)
        #実行結果を行単位で取得
        for line in iter(proc.stdout.readline, b""):
            line = bytes2unicode(line)
            if line[-2:] == "\r\n":
                line = line[:-2] + "\n"
            #標準出力に出力
            if isOut == True:
                print (line, end="")
            #出力結果を保存
            results.append(line)
            #出力結果をlogFileに保存
            f.write(line)
            f.flush()
        #実行結果を確認
        resCont = "".join(results)
        errCont = bytes2unicode(proc.stderr.read())
        if errCont != "":
            stat = "ERROR"
            if isOut == True:
                print(errCont)
            f.write(errCont)
            f.flush()
        else:
            stat = "OK"
        f.close()
        os.chdir(saveDir)
        return (stat, resCont, errCont)

#--------------
#  server class
#--------------
class server:
    """ 特定のserverの操作を行う。
    
    Attribute:
        serverName (str)    :server名
        sshfsDict (dict)    :serverの詳細データ
        sshDict (dict)      :sshConfigの内容
        sshfsFile (str)     :sshfs_dataのfilePath
        sshConfigFile (str) :sshConfigのfilePath
    
    sshfsdictの内容:
        host        :host名
        hostName    :sshConfigで定義しているhost名
        hostDir     :mountする時のhost側Dir
        mountPoint  :mountする時のlocal側Dir
        login       :loginサーバ名(hostNameと同じ)
        setEnviron  :login時の環境設定内容
        sshHeader   :sshのheader内容
        sshPassFile :秘密鍵を使わずに接続する時のpasswordファイル名
        jobConScript:Job管理用のscript名

    sshDictの内容
        url     :hostのurl
        user    :user名
    """
    def __init__(self, serverName):
        self.serverName = serverName
        self.sshfsFile = os.getenv("easyIstrUserPath") + "/data/sshfs_data"
        self.sshConfigFile = os.getenv("HOME") + "/.ssh/config"
        #hostの設定を読み込む
        self.sshfsDict = self.readSshfsData()
        hostName = self.sshfsDict["hostName"]
        self.sshDict = self.readSshConfig(hostName)

    #
    #  readSshConfig
    def readSshConfig(self, host):
        """ .ssh/configの読み込み。
        urlとuserを取得"""
        sshDict = {
            "url": "",
            "user": ""
            }
        fileName = self.sshConfigFile
        f = open(fileName); lines = f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        flag = 0; edFlag = 0
        for line in lines:
            if line[:len("Host ")].lower() == "Host ".lower():
                words = line.split()
                if words[1] == host:
                    flag = 1
                else:
                    if flag == 1:
                        edFlag = 1
                    flag = 0
            else:
                if flag == 1:
                    words = line.split()
                    if len(words) >= 2:
                        if words[0].lower() == "HostName".lower():
                            sshDict["url"] = words[1]
                        elif words[0].lower() == "User".lower():
                            sshDict["user"] = words[1]
            if edFlag == 1:
                break
        return sshDict

    #
    #  readSshfsData
    def readSshfsData(self):
        """ 該当serverのsshfsData（接続内容）を取得する"""
        sshfsDict = {
            "host": "",
            "hostName": "",
            "hostDir": "",
            "mountPoint": "",
            "login": "",
            "setEnviron": "",
            "sshHeader": "",
            "sshPassFile": "",
            "jobConScript": ""
            }
        fileName = self.sshfsFile
        f=open(fileName); lines=f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        #必要なserverName部を取得
        newLines = []
        flag = 0
        for line in lines:
            if line[:len("Host ")] == "Host ":
                words = line.split()
                if words[1] == self.serverName:
                    flag = 1
                else:
                    flag = 0
            elif flag == 1:
                newLines.append(line)
        #serverNameの設定値を取得
        setEnv = 0
        sshfsDict["host"] = self.serverName
        for i in range(len(newLines)):
            line = newLines[i]
            words = line.split()
            if line[0] != " ":
                if len(words) > 1:
                    if line[:len("HostName")].lower() == "HostName".lower():
                        sshfsDict["hostName"] = words[1]
                    elif line[:len("HostDir")].lower() == "HostDir".lower():
                        sshfsDict["hostDir"] = words[1]
                    elif line[:len("MountPoint")].lower() == "MountPoint".lower():
                        sshfsDict["mountPoint"] = words[1]
                    elif line[:len("sshHeader")].lower() == "sshHeader".lower():
                        sshfsDict["sshHeader"] = " ".join(words[1:]) + " "    #空白1文字を追加
                    elif line[:len("sshPassFile")].lower() == "sshPassFile".lower():
                        sshfsDict["sshPassFile"] = words[1]
                    elif line[:len("jobConScript")].lower() == "jobConScript".lower():
                        sshfsDict["jobConScript"] = words[1]
                if line[:len("login")].lower() == "login".lower():
                    words = lines[i+1].split()
                    sshfsDict["login"] = " ".join(words) 
                elif line[:len("setEnviron")].lower() == "setEnviron".lower():
                    setEnv = 1
                else:
                    setEnv = 0
            else:
                if setEnv == 1:
                    if len(words) > 0:
                        sshfsDict["setEnviron"] += " ".join(words) + "\n"
        #loginを設定
        if sshfsDict["login"] == "":
            sshfsDict["login"] = "ssh " + sshfsDict["hostName"]
        #環境変数をセット、fullpathで設定（sshPassFile）
        sshPassFile = sshfsDict["sshPassFile"]
        sshPassFile = os.path.expanduser(sshPassFile)
        sshPassFile = os.path.abspath(sshPassFile)
        os.environ["sshPassFile"] = sshPassFile
        return sshfsDict

    #
    #  writeSshfsData
    def writeSshfsData(self):
        """ 該当serverのsshfsData（接続内容）を書き込む"""
        fileName = self.sshfsFile
        f=open(fileName); lines=f.readlines(); f.close()
        #必要なserverName部を検索
        #  serverNameの前側linesを取得
        uLines = []
        ip = 0
        for i in range(len(lines)):
            line = lines[i]
            if line[:len("Host ")] == "Host ":
                words = line.split()
                if words[1] == self.serverName:
                    uLines = lines[:i]
                    ip = i
                    break
        #  serverNameの後側linesを取得
        dLines = []
        ie = len(lines)
        i = ip + 1
        while i < len(lines):
            line = lines[i]
            if line[:len("Host ")] == "Host ":
                dLines = lines[i:]
                ie = i
                break
            i += 1
        #  serverName部を取得
        mLines = lines[ip:ie]
        #severName部に上書き
        flags = [0,0,0,0]
        for i in range(len(mLines)):
            line = mLines[i]
            words = line.split()
            if len(words) > 0:
                if words[0] == "HostName":
                    flags[0] = 1
                    mLines[i] = "HostName " + self.sshfsDict["hostName"] + "\n"
                elif words[0] == "HostDir":
                    flags[1] = 1
                    mLines[i] = "HostDir " + self.sshfsDict["hostDir"] + "\n"
                elif words[0] == "MountPoint":
                    flags[2] = 1
                    mLines[i] = "MountPoint " + self.sshfsDict["mountPoint"] + "\n"
                elif words[0] == "sshHeader":
                    flags[3] = 1
                    mLines[i] = "sshHeader " + self.sshfsDict["sshHeader"] + "\n"
        if flags[0] == 0 and self.sshfsDict["hostName"] != "":
            mLines.append("HostName " + self.sshfsDict["hostName"] + "\n")
        if flags[1] == 0 and self.sshfsDict["hostDir"] != "":
            mLines.append("HostDir " + self.sshfsDict["hostname"] + "\n")
        if flags[2] == 0 and self.sshfsDict["mountPoint"] != "":
            mLines.append("MountPoint " + self.sshfsDict["mountpoint"] + "\n")
        if flags[3] == 0 and self.sshfsDict["sshHeader"] != "":
            mLines.append("sshHeader " + self.sshfsDict["sshHeader"] + "\n")
        #setEnvironを設定
        #  setEnvironの内容を削除
        newLines = []
        si = 0; flag = 0
        for i in range(len(mLines)):
            line = mLines[i]
            words = line.split()
            if len(words) > 0:
                if words[0] == "setEnviron":
                    newLines.append(line)
                    flag = 1
                    si = i + 1
                else:
                    if line[0] == " " and flag == 1:
                        pass
                    else:
                        newLines.append(line)
            else:
                newLines.append(line)
        #  setEnvironを設定
        if flag == 1:
            envLines = self.sshfsDict["setEnviron"].split("\n")
            if envLines[-1] == "":
                envLines = envLines[:-1]
            for i in range(len(envLines)):
                envLines[i] = "    " + envLines[i] + "\n"
            mLines = newLines[:si] + envLines + newLines[si:]
        else:
            if self.sshfsDict["setEnviron"] != "":
                envLines = self.sshfsDict["setenviron"].split("\n")
                if envLines[-1] == "":
                    envLines = envLines[:-1]
                for i in range(len(envLines)):
                    envLines[i] = "    " + envLines[i] + "\n"
                mLines += envLines
        #  linesを再作成して保存
        lines = uLines + mLines + dLines
        f = open(fileName, "w"); f.writelines(lines); f.close()

    #
    #  mount
    def mount(self, mountDir):
        self.sshfsDict["mountPoint"] = mountDir
        self.writeSshfsData()
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        hostDir = self.sshfsDict["hostDir"]
        sshHeader = self.sshfsDict["sshHeader"]
        comm = "sshfs " + userName + "@" + hostName + ":" + hostDir
        comm += " " + mountDir
        commLine = sshHeader + comm
        print(commLine)
        (stat, _res, err) = run().commandReturnCont(commLine)
        return (stat, err)

    #
    #  sendSshCommand
    def sendSshCommand(self, comm):
        """ serverに対し、sshでコマンドを送る"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        #hostDir = self.sshfsDict["hostDir"]
        #mountDir = self.sshfsDict["mountPoint"]
        sshHeader = self.sshfsDict["sshHeader"]
        commLine = "ssh " + userName + "@" + hostName + " '" + comm + "'"
        commLine = sshHeader + commLine
        print(commLine)
        run().command(commLine)

    #
    #  scpCommandNoTerm
    def scpCommandNoTerm(self, source, paste):
        """ 端末を開かずにscpを実行する。
        
        Args:
            source (str)    :転送するfile名
            paste (str)     :転送先"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        #hostDir = self.sshfsDict["hostDir"]
        #mountDir = self.sshfsDict["mountPoint"]
        sshHeader = self.sshfsDict["sshHeader"]
        header = userName + "@" + hostName + ":"
        source = self.checkServerDir(source)
        paste = self.checkServerDir(paste)
        #errorCheck
        if source[:len(header)] == header and paste[:len(header)] == header:
            return ["ERROR", "unmatch server directory"]
        elif source[:len(header)] != header and paste[:len(header)] != header:
            return ["ERROR", "unmatch server directory"]
        #実行
        commLine = "scp -Cr " + source + " " + paste
        commLine = sshHeader + commLine
        print(commLine)
        run().command(commLine)

    #
    #  moveRunPython
    def moveRunPython(self, scriptPath, args):
        """ python3のスクリプトをserverに転送し、そのスクリプトを実行する。
        実行は、python2でスクリプトを実行する。
        引数（args）は、文字列として入力する。"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        host = userName + "@" + hostName
        comm = "scp " + scriptPath + " " + host + ":~/runPython.py"
        comm += "; ssh " + host + " python runPython.py " + args
        run().command(comm)

    #
    #  moveRunPythonReturnCont
    def moveRunPythonReturnCont(self, scriptPath, args):
        """ python3のスクリプトをserverに転送し、そのスクリプトを実行し、結果を返す。
        実行は、python2でスクリプトを実行する。
        引数（args）は、文字列として入力する。"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        host = userName + "@" + hostName
        comm = "scp " + scriptPath + " " + host + ":~/runPython.py"
        comm += "; ssh " + host + " python runPython.py " + args
        result = run().commandReturnCont(comm, isOut=False)
        return result

    #
    #  runLoginTerm
    def runLoginTerm(self):
        """ login端末を起動"""
        appDict = readUsingApplications()
        runTerminalRunComm = appDict["terminalRun"]
        sshHeader = self.sshfsDict["sshHeader"]
        login = self.sshfsDict["login"]
        #端末を起動
        login = sshHeader + login
        fileName = os.getenv("easyIstrUserPath") + os.sep + "loginTerm"
        line = "#!/bin/bash\n"
        line += login + "\n"                                            #login
        f=open(fileName, "w"); f.write(line); f.close()
        run().command("chmod a+x " + fileName)
        comm = runTerminalRunComm + " " + fileName
        run().command(comm)

    #
    #  checkServerDir
    def checkServerDir(self, dirName):
        """ serverDirかどうか調べ、serverの場合は、userName等を付加する。"""
        hostName = self.sshfsDict["hostName"]
        userName = self.sshDict["user"]
        hostDir = self.sshfsDict["hostDir"]
        mountDir = self.sshfsDict["mountPoint"]
        header = userName + "@" + hostName + ":"
        if dirName[:len(mountDir)] == mountDir:
            dirName = header + hostDir + dirName[len(mountDir):]
        elif dirName[:len(hostDir)] == hostDir:
            dirName = header + dirName
        elif dirName[:len("~/TreeFoamServer/")] == "~/TreeFoamServer/":
            dirName = header + dirName
        elif dirName[0] == "~":
            dirName = header + dirName
        return dirName

    #
    #  deleteBashComment
    def deleteBashComment(self, lines):
        """ lines内からコメント部「#」を削除する"""
        newLines = []
        for line in lines:
            if line[0] != "#":
                n = line.find("#")
                if n < 0:
                    newLines.append(line)
                else:
                    newLine = line[:n] + "\n"
                    newLines.append(newLine)
        return newLines

#-----------------
#  servers class
#-----------------
class servers:
    """ 登録してある全serverの内容を取得する。

    Attribute:
        sshfsFile (str)     :sshfs_dataのfilePath
        sshConfigFile (str) :sshConfigのfilePath
    """
    def __init__(self):
        self.sshfsFile = os.getenv("easyIstrUserPath") + "/data/sshfs_data"
        self.sshConfigFile = os.getenv("HOME") + "/.ssh/config"

    #
    #  getAllServerNames
    def getAllServerNames(self):
        """ TreeFoamに登録されているserverを取得する。
        $TreeFoamUserPath/data/sshfs_data から取得
        
        Returns:
            names (list(str))       :全serverName
            mountNames (list(str))  :mountされているserver名
        """
        fileName = self.sshfsFile
        f=open(fileName); lines=f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        names = []; hostName = ""; mountNames = []
        for line in lines:
            words = line.split()
            if len(words) >= 1:
                if words[0] == "Host":
                    hostName = words[1]
                    names.append(hostName)
                elif words[0] == "MountPoint":
                    if len(words) > 1:
                        mountNames.append(hostName)
        names.sort()
        mountNames.sort()
        return (names, mountNames)

    #
    #  getMountedServers
    def getMountedServers(self):
        """ マウントされているserver(server, host, mountPoint)を検索し返す。
        mountPointが設定されているserver, hostNameを取得。
        
        Returns:
            mountServers (list(str, str, str)) :[[serverName, hostName, mountPoint]]
        """
        fileName = self.sshfsFile
        f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
        lines = self.deleteBashComment(lines)
        mountServers = []
        serverName = ""; hostName = ""; mountPoint = ""
        for line in lines:
            words = line.split()
            if len(words) >= 1:
                if words[0] == "Host":
                    if hostName != "" and mountPoint != "":
                        mountServers.append([serverName, hostName, mountPoint]) 
                    serverName = words[1]
                    hostName = ""; mountPoint = ""
                elif words[0] == "HostName":
                    hostName = words[1]
                elif words[0] == "MountPoint":
                    if len(words) > 1:
                        mountPoint = words[1]
        if hostName != "" and mountPoint != "":
            mountServers.append([serverName, hostName, mountPoint])
        return mountServers

    #
    #  getServerName
    def getServerName(self, getDir):
        """ directoryからserverNameを取得する。
        
        Attribute:
            getServer (str)  :Dir系列にmountされているserver名を取得する"""
        mountServers = self.getMountedServers()
        getServer = ""
        getDirList = getDir.split("/")
        for (serverName, _hostName, mountP) in mountServers:
            mountList = mountP.split("/")
            if mountList == getDirList[:len(mountList)]:
                getServer = serverName
                break
        return getServer

    #
    #  deleteBashComment
    def deleteBashComment(self, lines):
        """ lines内からコメント部「#」を削除する"""
        newLines = []
        for line in lines:
            if line[0] != "#":
                n = line.find("#")
                if n < 0:
                    newLines.append(line)
                else:
                    newLine = line[:n] + "\n"
                    newLines.append(newLine)
        return newLines



if __name__ == "__main__":
    import gettext
    gettext.install("app") # replace with the appropriate catalog name
    _ = gettext.gettext
