#!/usr/bin/python3  
# -*- coding: utf-8 -*-    
#
#   geometryFunc.py
#
#       幾何学関数、寸法を計算
#
#   18/02/01    新規作成
#      02/19    isPointInFaceLarge:追加（辺の長さを1.5倍拡大してチェック）
#      07/15    dispFaceToPoint:追加（面からの距離を正負で取得）
#      09/26    triA（三角形面積）, quadG（四角形の重心）を追加
#   19/01/08    python3用に書き換え
#   21/02/07    midPoint:2点の中間点の座標をlistで返す。
#      03/03    vecEdgeToPoint, addVec, subVec, vecByC:追加
#      03/05    convDispCtoD:座標変換を追加
#      04/08    vecsAngle:2ヶのvectorの角度を算出を追加
#   22/04/03    lenLineToLine:2直線間距離算出を追加
#               lenLineToPoint:lineと点間の距離を算出
#      05/08    tetV:四面体の体積取得を追加
#               hexV:六面体の体積取得を追加
#               priV:五面体の体積取得を追加
#      11/11    polygonAG:多角形の面積、重心位置算出追加
#

import math

#
#  addVec
#    ベクトルの加算(a+b)
def addVec(a, b):
    x = a[0] + b[0]
    y = a[1] + b[1]
    z = a[2] + b[2]
    return [x, y, z]

#
#  subVec
#    ベクトルの減算(a-b)
def subVec(a, b):
    vec = vector(b, a)
    return vec

#
#  vecByC
#    ベクトルに系数（スカラー値）を掛け算
def vecByC(vec, C):
    vec = list(map(lambda x: x * C, vec))
    return vec

#
#  crossProduct
#    ベクトル積（外積）
def crossProduct(a, b):
    x = a[1]*b[2] - a[2]*b[1]
    y = a[2]*b[0] - a[0]*b[2]
    z = a[0]*b[1] - a[1]*b[0]
    return [x, y, z]

#
#  dotProduct
#    スカラー積（内積）
def dotProduct(a, b):
    ans = a[0]*b[0] + a[1]*b[1] + a[2]*b[2]
    return ans

#
#  mag
#    絶対値の取得
def mag(a):
    ans = pow((a[0]*a[0] + a[1]*a[1] + a[2]*a[2]), 0.5)
    return ans

#
#  vector
#    2点からベクトルを取得
def vector(loc1, loc2):
    x = loc2[0] - loc1[0]
    y = loc2[1] - loc1[1]
    z = loc2[2] - loc1[2]
    return [x,y,z]
    
#
#  normal
#    単位ベクトルを取得
def normal(vec):
    l = mag(vec)
    if l != 0.0:
        x = vec[0] / l
        y = vec[1] / l
        z = vec[2] / l
    else:
        x = 0.0
        y = 0.0
        z = 0.0
    return [x,y,z]

#
#  midPoint
#    2点の中間点の座標を取得
def midPoint(loc1, loc2):
    x = (loc1[0] + loc2[0]) / 2.0
    y = (loc1[1] + loc2[1]) / 2.0
    z = (loc1[2] + loc2[2]) / 2.0
    return [x,y,z]

#
#  length
#    2点間の距離を取得
def length(a, b):
    x = a[0] - b[0]
    y = a[1] - b[1]
    z = a[2] - b[2]
    l = pow((x*x + y*y + z*z), 0.5)
    return l

#
#  length2
#    2点間の距離を取得（2次元）
def length2(a, b):
    x = a[0] - b[0]
    y = a[1] - b[1]
    l = pow((x*x + y*y), 0.5)
    return l

#
#  tri3length
#    三角形の3辺の長さを取得
def tri3length(tri):
    l1 = length(tri[0], tri[1])
    l2 = length(tri[1], tri[2])
    l3 = length(tri[2], tri[0])
    return [l1, l2, l3]

#
#  triG
#    三角形の重心の座標を取得する
def triG(tri):
    lx = (tri[1][0] + tri[0][0]) / 2.0
    ly = (tri[1][1] + tri[0][1]) / 2.0
    lz = (tri[1][2] + tri[0][2]) / 2.0
    Vx = (tri[2][0] - lx) / 3.0
    Vy = (tri[2][1] - ly) / 3.0
    Vz = (tri[2][2] - lz) / 3.0
    Gx = lx + Vx
    Gy = ly + Vy
    Gz = lz + Vz
    return [Gx, Gy, Gz]

#
#  triA
#    三角形の面積を取得
def triA(tri):
    [a, b, c] = tri3length(tri)
    s = (a + b + c) / 2.0
    A = pow( (s * (s-a) * (s-b) * (s-c)), 0.5 )
    return A

#
#  quadG
#    四角形の重心座標を取得
def quadG(quad):
    tri1 = [quad[0], quad[1], quad[2]]
    tri2 = [quad[0], quad[2], quad[3]]
    a1 = triA(tri1)
    a2 = triA(tri2)
    g1 = triG(tri1)
    g2 = triG(tri2)
    r = a2 / (a1 + a2)
    vec12 = vector(g2, g1)
    x = g1[0] - vec12[0] * r
    y = g1[1] - vec12[1] * r
    z = g1[2] - vec12[2] * r
    return [x,y,z]

#
#  tetG
#    四面体の重心の座標を取得
def tetG(tet):
    triC = triG(tet[:3])
    vecG = vector(triC, tet[3])
    Gx = triC[0] + vecG[0]/3
    Gy = triC[1] + vecG[1]/3
    Gz = triC[2] + vecG[2]/3
    return [Gx, Gy, Gz]    

#
#  tetV
#    四面体の体積を取得
def tetV(tet):
    tri = tet[:3]
    A = triA(tri)
    h = lenFaceToPoint(tri, tet[3])
    V = A * h / 3.0
    return V

#
#  hexV
#    六面体の体積を取得
def hexV(locs):
    loc0 = locs[0]
    loc1 = locs[6]
    locC = midPoint(loc0, loc1)     #対角線の中点
    face1 = [locs[0], locs[1], locs[2], locs[3]]
    face2 = [locs[4], locs[5], locs[6], locs[7]]
    face3 = [locs[0], locs[1], locs[5], locs[4]]
    face4 = [locs[1], locs[2], locs[6], locs[5]]
    face5 = [locs[2], locs[3], locs[7], locs[6]]
    face6 = [locs[3], locs[0], locs[4], locs[7]]
    faces = [face1, face2, face3, face4, face5, face6]
    V = 0.0
    for face in faces:
        tri0 = face[:3]
        tri1 = [face[0], face[2], face[3]]
        A = triA(tri0) + triA(tri1)
        h = lenFaceToPoint(face, locC)
        V += A * h / 3.0
    return V

#
#  priV
#    五面体の体積を取得
def priV(locs):
    face1 = [locs[0], locs[1], locs[2]]
    face2 = [locs[3], locs[4], locs[5]]
    face3 = [locs[0], locs[1], locs[4], locs[3]]
    face4 = [locs[1], locs[2], locs[5], locs[4]]
    face5 = [locs[2], locs[0], locs[3], locs[5]]
    faces = [face1, face2, face3, face4, face5]
    G0 = triG(face1)
    G1 = triG(face2)
    locC = midPoint(G0, G1)
    V = 0.0
    for i in range(2):
        tri0 = faces[i]
        A = triA(tri0)
        h = lenFaceToPoint(tri0, locC)
        V += A * h / 3.0
    for i in range(2, 5):
        face = faces[i]
        tri0 = face[:3]
        tri1 = [face[0], face[2], face[3]]
        A = triA(tri0) + triA(tri1)
        h = lenFaceToPoint(face, locC)
        V += A * h / 3.0
    return V

#
#  polygonAG
#    面（多角形）の面積と重心位置を返す
def polygonAG(locs):
    #基準値算出
    loc0 = locs[0]
    triLocs0 = [locs[0], locs[1], locs[2]]
    A0 = triA(triLocs0)
    G0 = triG(triLocs0)
    n1 = 2
    #2個目以降の三角形をチェック
    for n in range(len(locs)-3):
        triLocs = [loc0, locs[n1], locs[n1+1]]
        A = triA(triLocs)
        G = triG(triLocs)
        r = A / (A0 + A)
        vec12 = vector(G, G0)
        x = G0[0] - vec12[0] * r
        y = G0[1] - vec12[1] * r
        z = G0[2] - vec12[2] * r
        #基準値を更新
        G0 = [x, y, z]
        A0 += A
        n1 += 1
    return A0, G0

#
#  faceNormal
#    面に垂直なnormalVector
def faceNormal(face):
    [a,b,c] = face
    vec1 = vector(a, b)
    vec2 = vector(b, c)
    vec = crossProduct(vec1, vec2)
    faceNormal = normal(vec)
    return faceNormal

#
#  facesAngle
#    2面の角度(cos)を算出
def facesAngle(face1, face2):
    nVec1 = faceNormal(face1)
    nVec2 = faceNormal(face2)
    cosAng = dotProduct(nVec1, nVec2)
    return cosAng

#
#  vecsAngle
#    vec1からvec2の角度（cos）を算出する
def vecsAngle(vec1, vec2):
    nVec1 = normal(vec1)
    nVec2 = normal(vec2)
    cosAng = dotProduct(nVec1, nVec2)
    return cosAng

#
#  isPointInBlock
#    点がblock内に存在するかどうかチェック
def isPointInBlock(minMax, point):
    #min側をcheck
    minx = minMax[0][0]
    miny = minMax[0][1]
    minz = minMax[0][2]
    if minx > point[0]:
        return False
    if miny > point[1]:
        return False
    if minz > point[2]:
        return False
    #max側をcheck
    maxx = minMax[1][0]
    maxy = minMax[1][1]
    maxz = minMax[1][2]
    if maxx < point[0]:
        return False
    if maxy < point[1]:
        return False
    if maxz < point[2]:
        return False
    return True

#
#  isPointInEdge
#    edgeに投影した点がedge（線分）内かどうかcheck
#     edge内の場合は、edgeとpoint間の距離を返す
#     edge外の場合は、負数の距離を返す　　
def isPointInEdge(edge, point):
    vec = vector(edge[0], edge[1])
    vecPE = vector(point, edge[0])
    vecz = crossProduct(vec, vecPE)
    vecy = crossProduct(vecz, vec)    #edgeからpの垂線方向
    nVecy = normal(vecy)
    l = dotProduct(nVecy, vecPE)
    if l < 0.0:
        return l
    #反対側から距離算出
    vec = vector(edge[1], edge[0])
    vecPE = vector(point, edge[1])
    vecz = crossProduct(vec, vecPE)
    vecy = crossProduct(vecz, vec)
    nVecy = normal(vecy)
    l = dotProduct(nVecy, vecPE)
    return l

#
#  isPointInFace
#    pointをfaceに投影した点がface内に存在するかチェック
#    存在しない場合:「-1.0」を返す
#    存在する場合:faceとの距離（絶対値）を返す
def isPointInFace(tri, point):
    #org0（基点）
    vec1 = vector(tri[0], tri[1])
    vec2 = vector(tri[0], tri[2])
    vecz = crossProduct(vec1, vec2)     # 面に垂直方向（Z）
    vecy = crossProduct(vec2, vecz)     # Y方向
    vec0p = vector(tri[0], point)       # pointのvector
    if dotProduct(vecy, vec0p) < 0.0:     #line(0-1)の外？
        return -1.0
    #org1（基点）
    vec1 = vector(tri[1], tri[2])
    vec2 = vector(tri[1], tri[0])
    vecy = crossProduct(vec2, vecz)
    vec1p = vector(tri[1], point)
    if dotProduct(vecy, vec1p) < 0.0:
        return -1.0
    #org2（基点）
    vec1 = vector(tri[2], tri[0])
    vec2 = vector(tri[2], tri[1])
    vecy = crossProduct(vec2, vecz)
    vec2p = vector(tri[2], point)
    if dotProduct(vecy, vec2p) < 0.0:
        return -1.0
    #faceとの距離を算出
    nVecz = normal(vecz)
    l = abs(dotProduct(nVecz, vec0p))
    return l

#
#  isPointInFaceLarge
#    pointをfaceに投影した点がface内に存在するかチェック
#    face形状は、辺の長さを1.5倍まで拡大してチェックする
#    戻り値：[n, l]
#       n=-1: 存在しない
#          0: face内に存在する
#          1: 1.5倍に拡大したface内に存在する
#       l: 存在する場合のfaceまでの距離
def isPointInFaceLarge(tri, point):
    isPointInFaceOverValue(tri, point, 0.5)

#
#  isPointInFaceOverValue
#    pointをfaceに投影した点がface内に存在するかチェック
#    face形状は、各辺の長さを指定した倍率まで拡大してチェック
#    引数：[tri, p, ov]
#       tri: 三角形の座標のlist
#         p: 点の座標
#        ov: 倍率（0.1）
#    戻り値：[n, l]
#       n=-1: 存在しない
#          0: face内に存在する
#          1: 拡大したface内に存在する
#       l: 存在する場合のfaceまでの距離
def isPointInFaceOverValue(tri, point, ov):
    flag = 0
    #org0（基点）
    vec1 = vector(tri[0], tri[1])
    vec2 = vector(tri[0], tri[2])
    lx = mag(vec2)
    vecz = crossProduct(vec1, vec2)             # 面に垂直方向（Z）
    vecy = normal(crossProduct(vec2, vecz))     # Y方向
    vec0p = vector(tri[0], point)               # pointのvector
    diff = dotProduct(vecy, vec0p)
    #if diff < -lx*0.5:
    if diff < -lx*ov:
        return [-1, -1.0]
    elif diff < 0.0:
        flag = 1
    #org1（基点）
    vec1 = vector(tri[1], tri[2])
    vec2 = vector(tri[1], tri[0])
    lx = mag(vec2)
    vecy = normal(crossProduct(vec2, vecz))
    vec1p = vector(tri[1], point)
    diff = dotProduct(vecy, vec1p)
    #if diff < -lx*0.5:
    if diff < -lx*ov:
        return [-1, -1.0]
    elif diff < 0.0:
        flag = 1
    #org2（基点）
    vec1 = vector(tri[2], tri[0])
    vec2 = vector(tri[2], tri[1])
    lx = mag(vec2)
    vecy = normal(crossProduct(vec2, vecz))
    vec2p = vector(tri[2], point)
    diff = dotProduct(vecy, vec2p)
    #if diff < -lx*0.5:
    if diff < -lx*ov:
        return [-1, -1.0]
    elif diff < 0.0:
        flag = 1
    #faceとの距離を算出
    nVecz = normal(vecz)
    l = abs(dotProduct(nVecz, vec0p))
    if flag == 1:
        return [1, l]
    return [0, l]

#
#  isPointInTetra
#    pointがtetra内に存在するかどうか
def isPointInTetra(tetra, point):

    #  checkFace
    #    faceNormalの方向に点があるかチェック
    def checkFace(face, t4, p):
        nVec = faceNormal(face)         #faceNormalベクトル
        G = triG(face)                  #faceの重心
        vecG4 = vector(G, t4)           #face重心からtetraの頂点へのベクトル
        vecP = vector(G, p)             #face重心から点へのベクトル
        dotP = dotProduct(nVec, vecP)   #faceNormalVecと点のベクトルの方向チェック
        #faceの向きをチェック
        if dotProduct(nVec, vecG4) > 0:
            #faceの向きとtetra頂点が同じ向きの場合
            if dotP > 0:
                return True
            else:
                return False
        else:
            #faceの向きとtetra頂点が逆向きの場合
            if dotP < 0:
                return True
            else:
                return False

    #face1
    face1 = [tetra[0], tetra[1], tetra[2]]
    t4 = tetra[3]
    if checkFace(face1, t4, point) == False:
        return False
    #face2
    face2 = [tetra[0], tetra[1], tetra[3]]
    t4 = tetra[2]
    if checkFace(face2, t4, point) == False:
        return False
    #face3
    face3 = [tetra[1], tetra[2], tetra[3]]
    t4 = tetra[0]
    if checkFace(face3, t4, point) == False:
        return False
    #face4
    face4 = [tetra[2], tetra[0], tetra[3]]
    t4 = tetra[1]
    if checkFace(face4, t4, point) == False:
        return
    return True

#
#  lenLineToPoint
#    lineとpoint間の距離算出
def lenLineToPoint(line, point):
    vec = vecEdgeToPoint(line, point)
    l = mag(vec)
    return l

#
#  lenLineToLine
#    lineとlineの最短距離（line間の距離）
def lenLineToLine(line1, line2):
    p00 = line1[0]; p01 = line1[1]
    p10 = line2[0]; p11 = line2[1]
    vec1 = vector(p00, p01)
    vec2 = vector(p10, p11)
    #2直線の外積
    crossV = crossProduct(vec1, vec2)
    #単位ベクトルと各直線の任意点vecの内積
    n = normal(crossV)          #単位vector
    vec3 = vector(p00, p10)     #line間のvector
    d = abs(dotProduct(n, vec3))
    return d

#
#  lenFaceToPoint
#    faceとpoint間の距離算出
def lenFaceToPoint(tri, point):
    vec1 = vector(tri[0], tri[1])
    vec2 = vector(tri[0], tri[2])
    vecz = crossProduct(vec1, vec2)     # Z方向
    nVecz = normal(vecz)
    vecp = vector(tri[0], point)
    l = abs(dotProduct(nVecz, vecp))
    return l

#
#  dispFaceToPoint
#    faceとpoint間の距離を算出
#    pointがfaceの表なら「正」の距離
#    pointがfaceの裏なら「負」の距離を返す
def dispFaceToPoint(tri, point):
    vec1 = vector(tri[0], tri[1])
    vec2 = vector(tri[1], tri[2])
    vecz = crossProduct(vec1, vec2)     # Z方向
    nVecz = normal(vecz)
    vecp = vector(tri[1], point)
    l = dotProduct(nVecz, vecp)
    return l

#
#  vecEdgeToPoint
#    edgeからpointへの垂直ベクトル算出
def vecEdgeToPoint(edge, point):
    vec = vector(edge[0], edge[1])
    #vecPE = vector(point, edge[0])
    vecPE = vector(edge[0], point)
    vecz = crossProduct(vec, vecPE)
    vecy = crossProduct(vecz, vec)    #edgeからpの垂線方向
    nVecy = normal(vecy)
    l = dotProduct(nVecy, vecPE)
    vecPE[0] = nVecy[0] * l
    vecPE[1] = nVecy[1] * l
    vecPE[2] = nVecy[2] * l
    return vecPE

#
#  vecPointToEdge
#    pointからedgeへの垂直ベクトル算出
def vecPointToEdge(edge, point):
    vec = vector(edge[0], edge[1])
    vecPE = vector(point, edge[0])
    vecz = crossProduct(vec, vecPE)
    vecy = crossProduct(vecz, vec)    #edgeからpの垂線方向
    nVecy = normal(vecy)
    l = dotProduct(nVecy, vecPE)
    vecPE[0] = nVecy[0] * l
    vecPE[1] = nVecy[1] * l
    vecPE[2] = nVecy[2] * l
    return vecPE

#
#  vecFaceToPoint
#    faceからpointへの垂直ベクトル算出
def vecFaceToPoint(tri, point):
    vec1 = vector(tri[0], tri[1])
    vec2 = vector(tri[0], tri[2])
    vecz = crossProduct(vec1, vec2)     # Z方向
    nVecz = normal(vecz)
    vecp = vector(tri[0], point)
    l = dotProduct(nVecz, vecp)
    x = nVecz[0] * l
    y = nVecz[1] * l
    z = nVecz[2] * l
    return [x,y,z]

#
#  vecPointToFace
#    pointからfaceへの垂直ベクトル算出
def vecPointToFace(tri, point):
    vec1 = vector(tri[0], tri[1])
    vec2 = vector(tri[0], tri[2])
    vecz = crossProduct(vec1, vec2)     # Z方向
    nVecz = normal(vecz)
    vecPF = vector(point, tri[0])
    l = dotProduct(nVecz, vecPF)
    x = nVecz[0] * l
    y = nVecz[1] * l
    z = nVecz[2] * l
    return [x,y,z]

#----------- 座標変換 -------------------------------
#
#  convDispCtoD
#    直行座標pointの円筒座標変位dispから直行座標変位を取得
#       point: 直行座標(x,y,z)
#       edgeZ: 円筒座標の回転軸(a,b) 方向:a->b
#       disp:  円筒座標の変位(r,theta,z)
def convDispCtoD(point, edgeZ, disp):
    a, b = edgeZ
    radius, theta, cz = disp
    vecZ = vector(a, b)
    nVecZ = normal(vecZ)
    #半径方向変位
    vecP = vecEdgeToPoint(edgeZ, point)
    nVecP = normal(vecP)
    r = mag(vecP) + radius              #半径を加算
    #円周方向変位
    #  vecPをX軸とした高さhと幅Bを取得
    h = r * math.sin(theta)         #Y軸
    B = r * math.cos(theta)         #X軸
    #  各方向のベクトルを取得
    vec = crossProduct(vecP, nVecZ)
    nVecH = normal(vec)
    vecH = vecByC(nVecH, h)
    vecB = vecByC(nVecP, B)
    vecBH = addVec(vecB, vecH)          #回転方向加算
    #高さ方向変位
    vecCz = vecByC(nVecZ, cz)
    vecBHZ = addVec(vecBH, vecCz)       #高さ方向加算
    #変位XYZ
    newXYZ = vector(vecP, vecBHZ)       #xyz変位
    return newXYZ
