#!/usr/bin/python3
#
#           fsiCommonConcurrent.py
#
#       同時計算用のfsi連成の共通関数群
#
#   23/11/11    新規作成
#   24/04/23    deleteResSetRestartFileFsi:time2str関数を削除
#               correctTimeNameRestartFileName:restartFile名のtimeNameが
#               演算誤差で正しく設定されない事があるので、これを修正。
#      05/01    createMappingDispFile_proc:mapping処理に並列処理追加
#

import os
import shutil
import glob
import multiprocessing
import pyFistr_ptc as pyFistr
import mappingFromPointCloud as mapping
import pyCoupling
import fsiCommonAlternate as fsiCommAltn

couplingDict = {}
waitCount = 5000
waitTime = 0.01

FistrModel_cnt = "FistrModel_fsi.cnt"
resHeaderName = "FistrModel.res"
hecmwCtrlFile = "hecmw_ctrl_fsi.dat"

#
#  waitUntilExistingFile
#------------------------
def waitUntilExistingFile(fileName):
    """ fileが出来上がるまで待つ"""
    fsiCommAltn.waitUntilExistingFile(fileName)

#---------------------
#  createPressPtcFile
#---------------------
def createPressPtcFile(dataDict, pressFile):
    """ 圧力の点群fileを作成する"""
    fsiCommAltn.createPressPtcFile(dataDict, pressFile)

#-------------------------
#  createMappingPressFile
#-------------------------
def createMappingPressFile(dataDict):
    """ 各procNo毎の圧力点群fileをFistr側にmappingする"""
    fsiCommAltn.createMappingPressFile(dataDict)

#---------------------
#  setDeltaTInCnt_fsi
#---------------------
def setDeltaTInCnt_fsi(dataDict):
    """ fsiのcntファイルを更新する。"""
    global couplingDict
    couplingDict = dataDict
    deltaT = couplingDict["deltaT"]
    solidCaseDir = couplingDict["solidCaseDir"]
    timeIndex = couplingDict["timeIndex"]
    currTime = couplingDict["timeName"]
    startTime = couplingDict["startTime"]
    stepTime = float(deltaT)
    runStep = int(timeIndex)
    #cnt読み込み
    cntFile = solidCaseDir + "/FistrModel_fsi.cnt"
    f = open(cntFile); lines = f.readlines(); f.close()
    #時間増分の設定
    i = pyFistr.getNextHeaderName(lines, 0, "!DYNAMIC", "")
    i += 2
    words = pyFistr.deleteSp(lines[i]).split(",")
    startStep = runStep
    cTime = float(currTime)
    orgStep = 0
    writeInterval = float(couplingDict["writeInterval"])
    nSteps = int(couplingDict["nStepsOF"])
    endStep, endTime = pyCoupling.getEndStepFromCouplingWriteInterval(
        startStep, cTime, stepTime, nSteps, writeInterval, orgStep)
    words[2] = str(endStep)         #endStep
    #時間増分セット
    words[3] = str(stepTime)        #時間増分
    lines[i] = ", ".join(words) + "\n"
    #restartの設定
    if float(currTime) == 0.0:
        #初回は、restartさせずに起動
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=1")
    else:
        #restartで起動する様に修正
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=-1")
    #cntFile保存
    f = open(cntFile, "w"); f.writelines(lines); f.close()
    return endStep, endTime

#-----------------------------
#  getDispDictFromResFile_fsi
#-----------------------------
def getDispDictFromResFile_fsi(dataDict, endStep, np):
    """ resFileから変位を辞書形式（key=nodeNo）で取得する"""
    dispDict = fsiCommAltn.getDispDictFromResFile(dataDict, endStep, np)
    return dispDict

#-----------------
#  createDispPtc
#-----------------
def createDispPtc(dataDict, dispDict):
    """ 変位の辞書から変位のptcを作成して返す"""
    dispPtc = fsiCommAltn.createDispPtc(dataDict, dispDict)
    return dispPtc

#-------------------------
#  createMappingDispFile
#-------------------------
def createMappingDispFile(dataDict, dispPtc, endStep):
    """ 変位をpatchにmappingしたfileを作成する"""
    global couplingDict
    couplingDict = dataDict
    nProcs = couplingDict["nProcs"]
    np = int(nProcs)
    queue = []
    procs = []
    for n in range(np):
        q = multiprocessing.Queue()
        queue.append(q)
        procNo = str(n)
        p = multiprocessing.Process(
            target=createMappingDispFile_proc,
            args=(dispPtc, endStep, procNo, q)
            )
        p.start()
        procs.append(p)
    #全セットprocessの終了を待つ
    for n in range(np):
        a = queue[n].get()

#
#  createMappingDispFile_proc
def createMappingDispFile_proc(dispPtc, endStep, procNo, q):
    """ 指定したprocNoの変位をpatchにmappingしたfileを作成する"""
    solidCaseDir = couplingDict["solidCaseDir"]
    nProcs = couplingDict["nProcs"]
    nThreads = couplingDict["nThreads"]
    method = couplingDict["method"]
    mesh = mapping.fistrMesh(solidCaseDir) 
    #mappingするpointの座標を取得
    fsiCommAltn.couplingDict = couplingDict
    mapPointFaceDict, mapPointDict = fsiCommAltn.getMappedPointLocsFsi(procNo)
    #mappingTriFileがあるか？
    if fsiCommAltn.isMappingTriFileFsi(procNo) == False:
        #存在しない場合、mappingTriDictを取得
        if method == "altn":
            #alternate（交互計算）
            np = max(int(nProcs), int(nThreads))
        else:
            #concurrent（同時計算）
            np = int(nProcs) + int(nThreads)
        mapStat, mappingTriDict = mesh.getMappingTriLocs(dispPtc, mapPointFaceDict, mapPointDict, np)
        if mapStat != "OK":
            return "error in displacement mapping."
        #mappingTriFileを作成
        fsiCommAltn.createMappingTriFileFsi(mappingTriDict, procNo)
    else:
        #fileを読み込む
        mappingTriDict = fsiCommAltn.readMappingTriFileFsi(procNo)
    #変位をmapping
    mappingDict = mesh.setMappingDataFromTriLocs(dispPtc, mapPointFaceDict, mappingTriDict)
    #mapping結果を保存(このfileをOpenFOAM側が読み込む)
    saveMappingDisp(mappingDict, endStep, procNo)
    q.put("")

#
#  saveMappingDisp
def saveMappingDisp(mappingDict, endStep, procNo):
    """ mappingDispの結果を保存。このファイルをOF側が読み込む"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    #procNo = couplingDict["procNo"]
    patchName = couplingDict["mappingPatch"]
    saveDir = fluidCaseDir + "/coupling_FrontISTR/data"
    name = patchName + "_mapPointFaceNodes_" + procNo
    fileName = saveDir + "/" + name
    f = open(fileName); lines = f.readlines(); f.close()
    #保存のfilename
    if procNo == "0":
        print("(py) --- displacement file is creating for OpenFOAM at " + str(endStep) + "...")
    name = patchName + "_disp_" + str(endStep) + "_" + procNo + "tm"
    dispFile = saveDir + "/" + name
    f = open(dispFile, "w")
    for line in lines:
        words = line.split()
        nodeNo = int(words[0])
        disp = mappingDict[nodeNo]
        line = " ".join(list(map(str, disp))) + "\n"
        f.write(line)
    f.close()
    return


#-----------------------------
#  deleteResSetRestartFileFsi
#-----------------------------
def deleteResSetRestartFileFsi(dataDict, np):
    """ 結果fileを削除する。
    ただし、writeInterval時は、結果fileとrestartFileを残す"""
    global couplingDict
    couplingDict = dataDict
    timeName = couplingDict["timeName"]
    writeInterval = couplingDict["writeInterval"]
    deltaT = couplingDict["deltaT"]
    timeIndex = couplingDict["timeIndex"]
    solidCaseDir = couplingDict["solidCaseDir"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    patchName = couplingDict["mappingPatch"]
    nStepsOF = couplingDict["nStepsOF"]
    nProcs = couplingDict["nProcs"]
    #writeIntervalかどうか？
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data/"
    delTime = timeName
    delStep = timeIndex
    delFiles = []
    delFolders = []
    #OpenFOAM側を確認
    if pyCoupling.isWriteInterval(timeName, writeInterval, deltaT) == False:
        #削除可能な時間
        #  圧力の点群ファイルを削除
        pattern = solidCaseDir + "/" + patchName + "_faceCenterPress_" + delTime + "_*.particles"
        fileNames = fsiCommAltn.getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  dispを削除
        pattern = dataDir + patchName + "_disp_" + delStep + "_*tm"
        fileNames = fsiCommAltn.getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
        #  facePressを削除
        pattern = dataDir + patchName + "_facePress_" + delTime + "_*tm"
        fileNames = fsiCommAltn.getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
    else:
        #writeInterval時の場合
        #  圧力の点群ファイル
        pattern = solidCaseDir + "/" + patchName + "_faceCenterPress_" + delTime + "_*.particles"
        fileNames = fsiCommAltn.getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  dispをrename
        pattern = dataDir + patchName + "_disp_" + delStep + "_*tm"
        fileNames = fsiCommAltn.getAllProcsFiles(pattern, int(nProcs))
        fsiCommAltn.delete2charaFromEndName(fileNames)
        #  facePressをrename
        pattern = dataDir + patchName + "_facePress_" + delTime + "_*tm"
        fileNames = fsiCommAltn.getAllProcsFiles(pattern, int(nProcs))
        fsiCommAltn.delete2charaFromEndName(fileNames)
    #FrontISTR側を確認
    if pyCoupling.isWriteInterval(timeName, writeInterval, deltaT) == False:
        #削除可能な時間
        #  resFileを削除
        #resFile = solidCaseDir + "/FistrModel.res.0." + delStep
        #delFiles += [resFile]
        resPattern = solidCaseDir + "/FistrModel.res.*." + delStep
        fileNames = fsiCommAltn.getAllProcsFiles(resPattern, np)
        delFiles += fileNames
        #  vtuFileを削除
        vtuFile = solidCaseDir + "/FistrModel.vis_psf." + delStep.zfill(4) + ".pvtu"
        delFiles += [vtuFile]
        #  vtuFolder
        vtuFolder = solidCaseDir + "/FistrModel.vis_psf." + delStep.zfill(4)
        delFolders += [vtuFolder]
    else:
        #timeNameがwriteIntervalの時
        #  timeNameのrestartFile名を修正
        correctTimeNameRestartFileName(timeName, timeIndex)
    #FrontISTRのrestartFileを保存
    calcEnd = float(timeName) + int(nStepsOF) * float(deltaT)
    #calcEndTime = pyCoupling.time2str(calcEnd)
    calcEndTime = str(calcEnd)
    if pyCoupling.isWriteInterval(calcEndTime, writeInterval, deltaT) == True:
        #writeInterval時の時
        #  resFileをrenameして保存
        calcEndIdx = str(int(timeIndex) + int(nStepsOF)) 
        fsiCommAltn.saveRestartFileFsi(calcEndTime, calcEndIdx, np)
    #fileFolderを削除
    for delFile in delFiles:
        if os.path.exists(delFile) == True:
            os.remove(delFile)
    for delFolder in delFolders:
        if os.path.isdir(delFolder) == True:
            shutil.rmtree(delFolder)

#
#  correctTimeNameRestartFileName
def correctTimeNameRestartFileName(timeName, timeIndex):
    """ timeIndexのrestartFile名の時間をtimeNameに設定する"""
    solidCaseDir = couplingDict["solidCaseDir"]
    pattern = "FistrModel.restart_*_" + timeIndex + ".*"
    files = glob.glob(solidCaseDir + "/" + pattern)
    for fileName in files:
        words = fileName.split("_")
        if words[-2] != timeName:
            words[-2] = timeName
            newName = "_".join(words)
            os.rename(fileName, newName)

