#!/usr/bin/python3
#
#           chtCommonAlternate.py
#
#       交互計算用のcht連成の共通関数群
#
#   23/11/06    新規作成
#      11/19    温度の計算方法を変更により修正
#               逐次計算→restartによる連続計算に変更
#   24/04/23    setDeltaTInCnt_cht:time2str関数部を削除
#      04/24    saveRestartFileCht:保存前に同じtimeNameの
#               restartFileがあれば削除することを追加
#      05/01    createMappingHeatFlux,createMappingTempFile_proc:
#               mappoing処理に並列処理を追加
#

import os
import time
import pickle
import shutil
import glob
import multiprocessing
import pyFistr_ptc as pyFistr
import mappingFromPointCloud as mapping
import pyCoupling

couplingDict = {}
waitCount = 5000
waitTime = 0.01
FistrModel_cnt = "FistrModel_cht.cnt"
resHeaderName = "FistrModel_temp.res"
hecmwCtrlFile = "hecmw_ctrl_cht.dat"
restartHeader = "FistrModel_cht.restart"


#
#  waitUntilExistingFile
#------------------------
def waitUntilExistingFile(fileName):
    """ fileが出来上がるまで待つ"""
    count = int(couplingDict["maxWaitTime"])
    sleepTime = waitTime
    while True:
        if os.path.exists(fileName) == True:
            #time.sleep(sleepTime)
            break
        count -= 1
        if count < 0:
            break
        time.sleep(sleepTime)
    return

#------------------------
#  createHeatFluxPtcFile
#------------------------
def createHeatFluxPtcFile(dataDict, htFile):
    """ heatFluxの点群file作成"""
    global couplingDict
    couplingDict = dataDict
    fluidCaseDir = couplingDict["fluidCaseDir"]
    solidCaseDir = couplingDict["solidCaseDir"]
    timeName = couplingDict["timeName"]
    patchName = couplingDict["mappingPatch"]
    procNo = couplingDict["procNo"]
    #faceCenterLoc読み込み
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    faceCenterFile = dataDir + "/" + patchName + "_faceCenterLocs_" + procNo
    f = open(faceCenterFile); lines = f.readlines(); f.close()
    #faceValueを読み込み
    fp = open(htFile); pLines = fp.readlines(); fp.close()
    newLines = []
    for i in range(len(lines)):
        newLine = lines[i][:-1] + " " + pLines[i]
        newLines.append(newLine)
    #heatFluxのptcファイルを保存
    name = patchName + "_faceCenterHeatFlux_" + timeName + "_" + procNo + ".particles"
    ptcFileName = solidCaseDir + "/" + name
    f = open(ptcFileName, "w"); f.writelines(newLines); f.close()
    return

#------------------------
#  createMappingHeatFlux
#------------------------
def createMappingHeatFlux(dataDict):
    """ 各procNo毎のheatFlux点群fileをFistr側にmappingする"""
    global couplingDict
    couplingDict = dataDict
    procNo = couplingDict["procNo"]
    nProcs = couplingDict["nProcs"]
    patchName = couplingDict["mappingPatch"]
    timeName = couplingDict["timeName"]
    sgrpName = couplingDict["mappingSgrp"]
    solidCaseDir = couplingDict["solidCaseDir"]
    nThreads = couplingDict["nThreads"]
    method = couplingDict["method"]
    sgrpDictName = solidCaseDir + "/sgrpDict.bin"
    f = open(sgrpDictName, "rb")
    sgrpDict = pickle.load(f)
    f.close()
    nodeDictName = solidCaseDir + "/nodeDict.bin"
    f = open(nodeDictName, "rb")
    nodeDict = pickle.load(f)
    f.close()
    #各procno毎の点群fileをまとめる
    lines = []
    if int(procNo) == 0:
        for n in range(int(nProcs)):
            pNo = str(n)
            name = patchName + "_faceCenterHeatFlux_" + timeName + "_" + pNo + ".particles"
            eachPtcFile = solidCaseDir + "/" + name
            waitUntilExistingFile(eachPtcFile)
            time.sleep(waitTime/10)
            f = open(eachPtcFile)
            lines += f.readlines()
            f.close()
        newName = patchName + "_faceCenterHeatFlux.particles"
        allPtcFile = solidCaseDir + "/" + newName
        f = open(allPtcFile, "w"); f.writelines(lines); f.close()
        #ptcData(数値data）を取得
        mesh = mapping.fistrMesh(solidCaseDir)
        ptcData = mesh.getPtcDataFromLines(lines)
        if isMappingTriFileHeatFlux() == False:
            #存在しない場合、mapping用の3点を取得
            if method == "altn":
                #alternate（交互計算）
                np = max(int(nProcs), int(nThreads))
            else:
                #concurrent（同時計算）
                np = int(nProcs) + int(nThreads)
            mapStat, mappingTriDict = mesh.getMappingTriLocs(ptcData, sgrpDict, nodeDict, np)
            if mapStat != "OK":
                return "error in heat flux mapping."
            #mappingTriFileHeatFluxを作成
            createMappingTriFileHeatFlux(mappingTriDict)
        else:
            #fileを読み込む
            mappingTriDict = readMappingTriFileHeatFlux()
        #heatFluxをmapping
        print("(py) --- heat flux value is mapping to SGRP '" + sgrpName + "' of FrontISTR...", flush=True)
        mappingDict = mesh.setMappingDataFromTriLocs(ptcData, sgrpDict, mappingTriDict)
        #mapping結果を保存
        lines = mesh.createMappingFile(mappingDict, sgrpName, allPtcFile)
        name = "mappingSHF_" + sgrpName + ".map"
        fileName = solidCaseDir + os.sep + name
        f = open(fileName, "w"); f.writelines(lines); f.close()
        #mapping結果をcntファイルに反映
        remakeFistrCntFileForHeatFlux(lines)

#
#  isMappingTriFileHeatFlux
def isMappingTriFileHeatFlux():
    """ 圧力mapping用の3点を取得したfileが存在するかチェック"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFileHeatFlux.bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    return os.path.exists(fileName)

#
#  createMappingTriFileHeatFlux
def createMappingTriFileHeatFlux(mappingTriDict):
    """ mapping用の3点の座標をpickleでfileに保存する"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFileHeatFlux.bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    #保存
    f = open(fileName, "wb")
    pickle.dump(mappingTriDict, f)
    f.close()

#
#  readMappingTriFileHeatFlux
def readMappingTriFileHeatFlux():
    """ 保存したmapping用の3点の座標を読み込む"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFileHeatFlux.bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    #読み込み
    f = open(fileName, "rb")
    mappingTriDict = pickle.load(f)
    f.close()
    return mappingTriDict

#
#  remakeFistrCntFileForHeatFlux
def remakeFistrCntFileForHeatFlux(mapLines):
    """ heatFluxのmapping結果をcntファイルに反映"""
    solidCaseDir = couplingDict["solidCaseDir"]
    sgrpName = couplingDict["mappingSgrp"]
    addLines = []
    for line in mapLines:
        if line[0] != "#":
            words = line.split()
            if len(words) >= 3:
                words[1] = "S" + words[1]
                newLine = ", ".join(words) + "\n"
                addLines.append(newLine)
    #cntFileを修正
    cntFile = solidCaseDir + os.sep + "FistrModel_cht.cnt"
    f = open(cntFile); lines = f.readlines(); f.close()
    #挿入場所は、最初の!DFLUXの該当SGRPに挿入する。
    i = 0
    while i < len(lines):
        i = pyFistr.getNextHeaderName(lines, i, "!DFLUX", "")
        i += 1
        if i >= len(lines):
            mess = "error: '" + sgrpName + "' does not exist in !DFLUX.\n"
            print(mess)
            si = i
            break
        words = pyFistr.deleteSp(lines[i]).split(",")
        if words[0] == sgrpName:
            si = i - 1
            ei = pyFistr.skipNextHeader(lines, i)
            break
    if si < len(lines):
        #cntFile修正、書き込み
        name = "mappingSHF_" + sgrpName + ".map"
        grpid = pyFistr.getValName(lines[si], "GRPID")
        words = ["!DFLUX", "GRPID="+grpid, "forceType=mapping", "mapFile="+name,
                 "mapCoeff=1.0"]
        lines[si] = ", ".join(words) + "\n"
        lines[si+1] = sgrpName + ", S1, 0.0\n"
        newLines = lines[:si+2] + addLines + lines[ei:]
        f = open(cntFile, "w"); f.writelines(newLines); f.close()

#---------------------
#  setDeltaTInCnt_cht
#---------------------
def setDeltaTInCnt_cht(dataDict, bfrStep, bfrTime):
    """ chtのcntファイルを更新する。"""
    global couplingDict
    couplingDict = dataDict
    startTime = couplingDict["startTime"]
    currTime = couplingDict["timeName"]         #endTime
    deltaT = couplingDict["deltaT"]
    timeIndex = couplingDict["timeIndex"]       #endStep
    solidCaseDir = couplingDict["solidCaseDir"]
    endStep = int(timeIndex)
    eTime = float(currTime)
    sStep, sTime = bfrStep, bfrTime
    #計算開始時？
    if float(sTime) < 0.0:
        sTime = startTime
        #sStep数を取得
        if float(sTime) == 0.0:
            #初回計算
            sStep = "0"
        else:
            #restart開始時のsStep数をrestartFileから取得
            #ssTime = pyCoupling.time2str(float(sTime))
            ssTime = sTime
            name = restartHeader + "_" + ssTime + "_*"
            pattern = solidCaseDir + "/" + name
            files = glob.glob(pattern)
            restartFile = files[0]
            restFile = ".".join(restartFile.split(".")[:-1])
            runStep = int(restFile.split("_")[-1])
            sStep = runStep
    stepTime = (eTime - float(sTime)) / (int(timeIndex) - int(sStep))
    if stepTime == 0.0:
        stepTime = float(deltaT)
    endTime = eTime
    #cnt読み込み
    cntFile = solidCaseDir + "/FistrModel_cht.cnt"
    f = open(cntFile); lines = f.readlines(); f.close()
    #時間増分、終了stepを修正
    i = pyFistr.getNextHeaderName(lines, 0, "!HEAT", "")
    i += 1
    words = pyFistr.deleteSp(lines[i]).split(",")
    words[0] = str(stepTime)            #時間増分
    words[1] = currTime                 #終了時間
    lines[i] = ", ".join(words) + "\n"
    #restartの設定
    if float(sTime) == 0.0:
        #初回は、restartさせずに起動
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=1")
    else:
        #restartで起動する様に修正
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=-1")
    #cntFile保存
    f = open(cntFile, "w"); f.writelines(lines); f.close()
    return endStep, endTime

#--------------------------
#  getTempDictFromResFile
#--------------------------
def getTempDictFromResFile(dataDict, endStep, np):
    """ FrontISTRの温度計算結果を辞書形式で返す。
    dispDict={<nodeNo>: <temp>}"""
    global couplingDict
    couplingDict = dataDict
    #resFile名を取得
    solidCaseDir = couplingDict["solidCaseDir"]
    resHeader = resHeaderName
    #並列数を取得
    tempDict = {}
    queues = []
    procs = []
    for n in range(np):
        resFile = solidCaseDir + "/" + resHeader + "." + str(n) + "." + str(endStep)
        q = multiprocessing.Queue()
        queues.append(q)
        p = multiprocessing.Process(
            target=createTempDictFromResFile_proc,
            args=(resFile, endStep, n, q)
            )
        p.start()
        procs.append(p)
    #全セットprocessの結果を取得
    for n in range(np):
        addDict = queues[n].get()
        tempDict.update(addDict)
    return tempDict
        
#
#  createTempDictFromResFile_proc
def createTempDictFromResFile_proc(resFile, endStep, procNo, q):
    """ proc毎にdispDictを作成する。"""
    global couplingDict, resHeaderName
    solidCaseDir = couplingDict["solidCaseDir"]
    resFile = solidCaseDir + "/" + resHeaderName + "." + str(procNo) + "." + str(endStep)
    #waitUntilExistingFile(resFile)
    addDict = readTempFromResFile(resFile)
    q.put(addDict)

#
#  readTempFromResFile
#----------------------
def readTempFromResFile(resFile):
    """ 温度データをresFileから取得し、辞書形式で返す。"""

    def readTextLine(f):
        """ fileObject fから1行読み込む。
        eof（空行）読み込みでも指定時間まで読み直す。"""
        line = f.readline()
        if line != "":
            return line
        #空文の場合、読み直す
        count = waitCount
        sleepTime = waitTime / 10.0
        while count > 0:
            line = f.readline()
            if line != "":
                break
            count -= 1
            time.sleep(sleepTime)
        return line

    #
    #resFileを読み込み
    f = open(resFile)
    #*dataまでskip
    while True:
        #line = f.readline()
        line = readTextLine(f)
        if line[:len("*data")] == "*data":
            break
    #node数読み込み
    #line = f.readline()
    line = readTextLine(f)
    nNodes = int(line.split()[0])
    #TEMPERATUREまで進む
    while True:
        valName = readTextLine(f).split()[0]
        if valName == "TEMPERATURE":
            break
    #温度を読み込む
    valDict = {}
    n = 0
    while n < nNodes:
        line = readTextLine(f)
        nodeNo = int(line.split()[0])
        line = readTextLine(f)
        temp = float(line.split()[0])
        valDict[nodeNo] = temp
        n += 1
    return valDict

#--------------------
#  createTempPtc
#--------------------
def createTempPtc(dataDict, tempDict):
    """ 温度の点群fileを作成"""
    global couplingDict
    couplingDict = dataDict
    #FrontISTRのnodeDictを取得
    solidCaseDir = couplingDict["solidCaseDir"]
    nodeDictName = solidCaseDir + "/nodeDict.bin"
    f = open(nodeDictName, "rb")
    nodeDict = pickle.load(f)
    f.close()
    #温度の点群データを取得作成
    tempPtc = []
    for nodeNo in nodeDict.keys():
        temp = tempDict[nodeNo]
        loc = nodeDict[nodeNo]
        tempPtc.append([loc, temp])
    return tempPtc

#------------------------
#  createMappingTempFile
#------------------------
def createMappingTempFile(dataDict, tempPtc, endStep):
    """ 温度をpatchにmappingしたfileを作成する"""
    global couplingDict
    couplingDict = dataDict
    nProcs = couplingDict["nProcs"]
    np = int(nProcs)
    queue = []
    procs = []
    for n in range(np):
        q = multiprocessing.Queue()
        queue.append(q)
        procNo = str(n)
        p = multiprocessing.Process(
            target=createMappingTempFile_proc,
            args=(tempPtc, endStep, procNo, q)
            )
        p.start()
        procs.append(p)
    #全セットprocessの終了を待つ
    for n in range(np):
        a = queue[n].get()

#
#  createMappingTempFile_proc
def createMappingTempFile_proc(tempPtc, endStep, procNo, q):
    """ 指定したprocNoの変位をpatchにmappingしたfileを作成する"""
    solidCaseDir = couplingDict["solidCaseDir"]
    nProcs = couplingDict["nProcs"]
    nThreads = couplingDict["nThreads"]
    method = couplingDict["method"]
    mesh = mapping.fistrMesh(solidCaseDir) 
    #mappingするpointの座標を取得(face中心座標、point座標の辞書)
    mapFaceCenterDict, mapPointDict = getMappedPointLocsCht(procNo)
    #mappingTriFileがあるか？
    if isMappingTriFileCht(procNo) == False:
        #存在しない場合、mappingTriDictを取得
        if method == "altn":
            #alternate（交互計算）
            np = max(int(nProcs), int(nThreads))
        else:
            #concurrent（同時計算）
            np = int(nProcs) + int(nThreads)
        mapStat, mappingTriDict = mesh.getMappingTriLocs(tempPtc, mapFaceCenterDict, mapPointDict, np)
        if mapStat != "OK":
            return "error in displacement mapping."
        #mappingTriFileを作成保存
        createMappingTriFileCht(mappingTriDict, procNo)
    else:
        #fileを読み込む
        mappingTriDict = readMappingTriFileCht(procNo)
    #変位をmapping
    mappingDict = mesh.setMappingDataFromTriLocs(tempPtc, mapFaceCenterDict, mappingTriDict)
    #mapping結果を保存(このfileをOpenFOAM側が読み込む)
    saveMappingTemp(mappingDict, endStep, procNo)
    q.put("")

#
#  getMappedPointLocsCht
def getMappedPointLocsCht(procNo):
    """ 温度をmappingするpointの座標（face中心座標）を取得する"""
    patchName = couplingDict["mappingPatch"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    #mapPointDictを取得
    name = patchName + "_mapPointFaceNodes_" + procNo
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    f = open(fileName); lines = f.readlines(); f.close()
    mapPointDict = {}
    for line in lines:
        words = line.split()
        if len(words) > 4:
            nodeNo = int(words[0])
            if not (nodeNo in mapPointDict.keys()):
                pointLoc = list(map(float, words[1:4]))
                mapPointDict[nodeNo] = pointLoc
    #mapFaceCenterDictを作成
    name = patchName + "_mapFaceCenterNodes_" + procNo
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    f = open(fileName); lines = f.readlines(); f.close()
    mapFaceCenterDict = {}
    for line in lines:
        words = line.split()
        if len(words) >= 7:
            faceNo = int(words[0])
            faceLoc = list(map(float, words[1:4]))
            nodeNos = list(map(int, words[4:]))
            mapFaceCenterDict[faceNo] = [faceLoc, nodeNos]
    return mapFaceCenterDict, mapPointDict

#
#  isMappingTriFileCht
def isMappingTriFileCht(procNo):
    """ 温度mapping用の3点を取得したfileが存在するかチェック"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFileCht_" + procNo
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    return os.path.exists(fileName)

#
#  createMappingTriFileCht
def createMappingTriFileCht(mappingTriDict, procNo):
    """ mapping用の3点の座標をfileに保存する"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFileCht_" + procNo
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    lines = []
    for key in mappingTriDict.keys():
        faceNo = key
        word = str(faceNo)
        triLocs = mappingTriDict[key]
        loc0, loc1, loc2 = triLocs
        word0 = " ".join(list(map(str, loc0)))
        word1 = " ".join(list(map(str, loc1)))
        word2 = " ".join(list(map(str, loc2)))
        line = " ".join([word, word0, word1, word2]) + "\n"
        lines.append(line)
    f = open(fileName, "w"); f.writelines(lines); f.close()

#
#  readMappingTriFileCht
def readMappingTriFileCht(procNo):
    """ 保存したmapping用の3点の座標を読み込む"""
    #procNo = couplingDict["procNo"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFileCht_" + procNo
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    f = open(fileName); lines = f.readlines(); f.close()
    mappingTriDict = {}
    for line in lines:
        words = line.split()
        faceNo = int(words[0])
        key = faceNo
        loc0 = list(map(float, words[1:4]))
        loc1 = list(map(float, words[4:7]))
        loc2 = list(map(float, words[7:10]))
        mappingTriDict[key] = [loc0, loc1, loc2]
    return mappingTriDict

#
#  saveMappingTemp
def saveMappingTemp(mappingDict, endStep, procNo):
    """ mappingした温度をfileに保存。このファイルをOF側が読み込む"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    patchName = couplingDict["mappingPatch"]
    #timeIndex = couplingDict["timeIndex"]
    saveDir = fluidCaseDir + "/coupling_FrontISTR/data"
    name = patchName + "_mapFaceCenterNodes_" + procNo
    fileName = saveDir + "/" + name
    f = open(fileName); lines = f.readlines(); f.close()
    #保存のfilename
    #name = patchName + "_temp_" + timeIndex + "_" + procNo + "tm"
    name = patchName + "_temp_" + str(endStep) + "_" + procNo + "tm"
    if procNo == "0":
        print("(py) --- temperature file is creating for OpenFOAM at " + name + "...")
    #name = patchName + "_temp_" + str(endStep) + "_" + procNo + "tm"
    tempFile = saveDir + "/" + name
    f = open(tempFile, "w")
    for line in lines:
        words = line.split()
        nodeNo = int(words[0])
        temp = mappingDict[nodeNo]
        line = str(temp) + "\n"
        f.write(line)
    f.close()
    return

#-----------------------------
#  deleteResSetRestartFileCht
#-----------------------------
def deleteResSetRestartFileCht(dataDict, bfrStep, bfrTime, np):
    """ 結果fileを削除する。
    ただし、writeInterval時は、結果fileとrestartFileを残す"""
    global couplingDict
    couplingDict = dataDict
    timeName = couplingDict["timeName"]
    beforeTime = couplingDict["beforeTime"]
    writeInterval = couplingDict["writeInterval"]
    #deltaT = couplingDict["deltaT"]
    timeIndex = couplingDict["timeIndex"]
    solidCaseDir = couplingDict["solidCaseDir"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    patchName = couplingDict["mappingPatch"]
    #nSteps = couplingDict["nSteps"]
    nProcs = couplingDict["nProcs"]
    deltaT = str(float(timeName) - float(beforeTime))
    #writeIntervalかどうか？
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data/"
    #delStep, delTime = getBeforeCouplingTime()
    delStep, delTime = bfrStep, bfrTime
    delFiles = []
    delFolders = []
    #OpenFOAM側を確認
    if pyCoupling.isWriteInterval(delTime, writeInterval, deltaT) == False:
        #削除可能な時間
        #  heatFluxの点群ファイルを削除
        pattern = solidCaseDir + "/" + patchName + "_faceCenterHeatFlux_" + delTime + "_*.particles"
        fileNames = getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  温度tempを削除
        pattern = dataDir + patchName + "_temp_" + delStep + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
        #  faceHeatFluxを削除
        pattern = dataDir + patchName + "_faceHeatFlux_" + delTime + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
    else:
        #writeInterval時の場合
        #  heatFluxの点群ファイル
        pattern = solidCaseDir + "/" + patchName + "_faceCenterHeatFlux_" + delTime + "_*.particles"
        fileNames = getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  温度tempをrename
        pattern = dataDir + patchName + "_temp_" + delStep + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delete2charaFromEndName(fileNames)
        #  faceHeatFluxをrename
        pattern = dataDir + patchName + "_faceHeatFlux_" + delTime + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delete2charaFromEndName(fileNames)
    #FrontISTR側を確認
    if pyCoupling.isWriteInterval(delTime, writeInterval, deltaT) == False:
        #削除可能な時間
        #  resFileを削除
        resPattern = solidCaseDir + "/FistrModel_temp.res.*." + delStep
        fileNames = getAllProcsFiles(resPattern, np)
        delFiles += fileNames
        #  vtuFileを削除
        vtuFile = solidCaseDir + "/FistrModel_cht.vis_psf." + delStep.zfill(4) + ".pvtu"
        delFiles += [vtuFile]
        #  vtuFolder
        vtuFolder = solidCaseDir + "/FistrModel_cht.vis_psf." + delStep.zfill(4)
        delFolders += [vtuFolder]
    #FrontISTRのrestartFileを保存
    calcEndTime = timeName
    calcEndIdx = timeIndex
    if pyCoupling.isWriteInterval(calcEndTime, writeInterval, deltaT) == True:
        #writeInterval時の時
        #  resFileをrenameして保存
        saveRestartFileCht(calcEndTime, calcEndIdx, np)
    #fileFolderを削除
    for delFile in delFiles:
        if os.path.exists(delFile) == True:
            os.remove(delFile)
    for delFolder in delFolders:
        if os.path.isdir(delFolder) == True:
            shutil.rmtree(delFolder)

#
#  getAllProcsFiles
def getAllProcsFiles(pattern, np):
    """ 全procsのfile名を取得して返す"""
    names = pattern.split("*")
    files = []
    for i in range(np):
        fileName = names[0] + str(i) + names[1]
        files.append(fileName)
    return files

#
#  delete2charaFromEndName
def delete2charaFromEndName(fileNames):
    """ fileNameの最後の2文字(tm)を削除"""
    for fileName in fileNames:
        if os.path.exists(fileName):
            os.rename(fileName, fileName[:-2])

#
#  saveRestartFileCht
def saveRestartFileCht(timeName, timeIndex, np):
    """ FrontISTRのrestartFileを設定する"""
    if int(timeIndex) == 0:
        return
    solidCaseDir = couplingDict["solidCaseDir"]
    #同じtimeNameのrestartFileがあれば、保存前に削除
    pattern = "FistrModel_cht.restart_*"
    files = glob.glob(solidCaseDir + "/" + pattern)
    for name in files:
        words = name.split("_")
        if float(words[-2]) == float(timeName):
            #削除
            os.remove(name)
    #restartFile保存
    for n in range(np):
        #orgName = solidCaseDir + "/FistrModel_cht.restart.0"
        orgName = solidCaseDir + "/FistrModel_cht.restart." + str(n)
        #newName = solidCaseDir + "/FistrModel_cht.restart_" + timeName + "_" + timeIndex + ".0"
        newName = solidCaseDir + "/FistrModel_cht.restart_" + timeName + "_" + timeIndex + "." + str(n)
        shutil.copy(orgName, newName)

