#!/usr/bin/python3
# coding: utf-8
#
#   splitMeshCellZones.py
#       メッシュをcellZone毎に分割する。
#       cellZone名とcellNoは、cellZonesファイル内容から取得する。
#       領域分割後、分割したファイルは、初期時間フォルダに移動する。オリジナルの初期時間
#       フォルダは、「before-region.<初期時間フォルダ>」でバックアップされる。
#
#       使い方
#       splitMeshCellZones
#
#       10/03/26    新規作成
#       11/02/25    -cellZonesOnlyに変更（domainが作成される為）
#       11/04/15    OpenFOAMの環境設定ファイルをpyTreeFoam側から読み込む
#       12/06/15    importをpyOFLauncher→pyTreeFoamに変更
#          12/16    親case/systemフォルダ内に各regionフォルダが残るので、削除した
#       13/02/22    dict内容に空白4ヶを追加
#          03/17    multiRegion：polyMeshをtimeFolderからconstantへ移動
#          05/08      ↑　constantへ移動を廃止（constantへコピー）
#          05/20    changeDict作成をOF-2.2.0に対応
#          05/21    changeDict作成を廃止。meshOperationDalog（呼出側）で作成。
#          10/12    多言語化のため、修正
#          12/21    stdout、stderrの設定（import logFileCreater）を追加
#       14/06/07    createRegCase:regCaseDir作成時case直下のfileもコピーを追加
#          07/26    movePolyMeshToConstant:timeFolder内のpolyMeshを削除
#          10/02    createRegCase:fluidRegions, solidRegions内fileを各regionにｺﾋﾟｰ追加
#          10/05    createRegCase:元に戻す。meshOperationDialogに追加
#          10/24    timeFolder内のfluidRegion, solidRegionsフォルダをコピーする様に修正
#          10/28    timeFolder内のfluidRegion, solidRegionsﾌｫﾙﾀﾞが無い場合、ｺﾋﾟｰしない
#       15/02/07    makeRegNameIntoConstant、remakeBoundaryFile:regionNameの取得方法
#                   を修正。
#       17/09/09    splitMeshRegions:command実行をpyTreeFoam内のrunOFUtilityに変更
#       19/05/22    makeRegNameIntoConstant:mkdirをos.mkdirに修正
#          10/24    GTK+3, python3用に大幅修正
#       20/04/22    多言語化対応
#       24/02/08    splitMeshRegions:OF-11用(splitMeshRegionsコマンドのoptionが変わった為)
#                   copyFvschemesFvsolution:削除
#          07/20    delFvschemesFvsolution:delFolderが存在する時のみ削除するように修正
#                   createRegCase:OF-12対応systemFolder内もregFolderのcheckする様に修正
#       25/07/15    splitMeshRegions:OF-13対応。「-noOverwrite」を追加。
#

import os
import glob
import shutil

import stringOp
import pyTreeFoam

import logFileCreater; logFileCreater.log()

#
#  splitMeshRegions
def splitMeshRegions(caseDir):
    """ cellZone毎に領域分割する"""
    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer >= 13.0:
        #OF-13以降の場合
        envOpenFoam = configDict["bashrcFOAM"]
        comm = ". " + envOpenFoam + "; "
        comm += "splitMeshRegions -cellZones -noOverwrite"
    elif numVer >= 11.0:
        #OF-11以降の場合
        envOpenFoam = configDict["bashrcFOAM"]
        comm = ". " + envOpenFoam + "; "
        comm += "splitMeshRegions -cellZones"
    else:
        #OF-11未満の場合
        envOpenFoam = configDict["bashrcFOAM"]
        comm = ". " + envOpenFoam + "; "
        comm += "splitMeshRegions -cellZonesOnly"
    pyTreeFoam.run(caseDir).commandWithLog(comm)

#
#  remakeBoundaryFile
def remakeBoundaryFile(regCaseDir, timeFolder, regNames):
    """ boundaryのpatchを修正"""
    print(_(u"\nbounadaryファイルのpatchTypeを修正します。"))
    regNameDir = []
    for name in regNames:
        regNameDir.append(regCaseDir + "/" + timeFolder + "/" + name)
    boundaryFiles = []
    for nameDir in regNameDir:
        boundaryFiles.append(nameDir + "/polyMesh/boundary")
    for boundary in boundaryFiles:
        f=open(boundary)
        contents = f.readlines()
        f.close()
        for beforeName in regNames:
            for afterName in regNames:
                patch = beforeName + "_to_" + afterName
                i=0
                while i<len(contents):
                    if (contents[i].find(patch) >= 0):
                        flag = 0
                        while i<len(contents) and flag < 2:
                            n = contents[i].find("region0")
                            m = contents[i].find("none")
                            if n>=0:
                                contents[i] = contents[i][:n] + afterName + contents[i][n+7:]
                                flag = flag + 1
                            if m>=0:
                               rePatch = afterName + "_to_" + beforeName
                               contents[i] = contents[i][:m] + rePatch + contents[i][m+4:]
                               flag = flag + 1
                            i=i+1
                    i=i+1
        regName = boundary.split("/")[-3]
        print("    " + regName + _("の boundary を修正しました。"))
        f=open(boundary,"w")
        for line in contents:
            f.write(line)
        f.close()

#
#  remakeBoundaryZeroGradient
def remakeBoundaryZeroGradient(regCaseDir, timeFolder):
    """ 各regionのbaundary（calculatedのみ）をzeroGradientに書き換える"""
    regNameDir = glob.glob(regCaseDir + "/" + timeFolder + "/*")
    regNames = []
    for name in regNameDir:
        regNames.append(name.split("/")[-1])
    for nameDir in regNameDir:
        fields = glob.glob(nameDir + "/*")
        for field in fields:
            if os.path.isfile(field) == True:
                fileName = field
                cont = pyTreeFoam.foamFile().read(fileName)
                p = stringOp.strings(cont).getKeywordPointer("boundaryField", 0)
                #cont = getFoamContents.readField(fileName, 100)
                #p = getFoamContents.getKeywordPointerMem("boundaryField", cont, 0)
                n = 0
                while n>=0:
                    n = cont[p:].find("calculated")
                    if n>=0:
                        p += n
                        cont = cont[:p] + "zeroGradient" + cont[p+len("calculated"):]
                #getFoamContents.writeField(fileName, cont)
                pyTreeFoam.foamFile().write(fileName, cont)
    print(_("\n各RegionのboundaryFieldをzeroGradientに修正しました。"))

#
#  copyFvschemesFvsolution
def copyFvschemesFvsolution(regCaseDir, timeFolder, regNames):
    """ 各regionにfvSchemes, fvSolutionをコピーする"""
    regNames.sort()
    for regName in regNames:
        regSys = regCaseDir + "/system/" + regName
        os.system("rm " + regSys + "/*")
        source = regCaseDir + "/system/fvSchemes*"
        copy = regCaseDir + "/system/" + regName + "/"
        os.system("cp " + source + " " + copy)
        source = regCaseDir + "/system/fvSolution*"
        copy = regCaseDir + "/system/" + regName + "/"
        os.system("cp " + source + " " + copy)
    print(_("\nfvSchemes, fvSolutionを各regionにコピーしました。"))

#
#  delFvschemesFvsolution
def delFvschemesFvsolution(caseDir, timeFolder, regNames):
    """ 親case/system内のregionフォルダを削除"""
    regNames.sort()
    for regName in regNames:
        delFolder = caseDir + "/system/" + regName
        if os.path.exists(delFolder) == True:
            shutil.rmtree(delFolder)

#
#  createRegCase
def createRegCase(caseDir, flag):
    """ multiRegion用にcaseを作成"""
    print(_(u"出来上がったregionを整形します。"))
    timeFolders = pyTreeFoam.case(caseDir).getTimeFolders()
    if len(timeFolders) != 2:
        msg = _("error: 時間フォルダがないか、領域分割できていません。")
        error = ["NG", msg]
        print(msg)
        return error

    regCaseDir = caseDir + "/regCase"
    if len(glob.glob(regCaseDir)) != 0:
        shutil.rmtree(regCaseDir)
    os.mkdir(regCaseDir)
    #systemFolderをコピー
    src = caseDir + "/system"
    dst = regCaseDir + "/system"
    shutil.copytree(src,dst)
    #constantFolderをコピー
    src = caseDir + "/constant"
    dst = regCaseDir + "/constant"
    shutil.copytree(src,dst)
    #timeFolderをコピー
    src = caseDir + "/" + timeFolders[1]
    dst = regCaseDir + "/" + timeFolders[0]
    shutil.copytree(src,dst)
    
    #regionNamesを取得
    regNames = glob.glob(src+"/*")
    regionNames = []
    for regName in regNames:
        regionNames.append(regName.split("/")[-1])
    #timeFolders[1]を削除
    shutil.rmtree(src)
    #timeFolder直下のfieldをコピー
    fields = pyTreeFoam.case(caseDir).getFieldNames(timeFolders[0])
    for name in fields:
        src = caseDir + "/" + timeFolders[0] + "/" + name
        dst = regCaseDir + "/" + timeFolders[0]
        shutil.copy(src, dst)
    files = pyTreeFoam.case(caseDir).getIniCaseFiles()
    for nameDir in files:
        shutil.copy(nameDir, regCaseDir)
    
    #regNameのフォルダをconstant内に作成
    for folder in regionNames:
        path = regCaseDir + "/constant/" + folder
        if os.path.exists(path) == False:
            os.mkdir(path)
        path = regCaseDir + "/system/" + folder
        if os.path.exists(path) == False:
            os.mkdir(path)
    print(_("\nconstant, systemフォルダ内にregionNameのフォルダを作成しました。"))
    #boundaryファイルのpatchTypeを修正
    OFversion = pyTreeFoam.readConfigTreeFoam()["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer < 11.0:
        #OF-11以降はboundaryの整形不要
        remakeBoundaryFile(regCaseDir, timeFolders[0], regionNames)
    #各RegionのboundaryFieldをzeroGradientに変更
    remakeBoundaryZeroGradient(regCaseDir, timeFolders[0])
    
    #multiRegion用に修正する
    #polyMeshをconstant内にコピー
    regNames = pyTreeFoam.getFolderNames(regCaseDir + "/" + timeFolders[0])
    for region in regNames:
        source = regCaseDir + "/" + timeFolders[0] + "/" + region + "/polyMesh"
        copy = regCaseDir + "/constant/" + region + "/polyMesh"
        shutil.copytree(source, copy)
        shutil.rmtree(source)
    #timeFolder内のfluidRegions, solidRegionsフォルダを親folderからコピー
    src = caseDir + "/" + timeFolders[0] + "/fluidRegions"
    if len(glob.glob(src)) != 0:
        dst = regCaseDir + "/" + timeFolders[0] + "/fluidRegions"
        shutil.copytree(src,dst)
    src = caseDir + "/" + timeFolders[0] + "/solidRegions"
    if len(glob.glob(src)) != 0:
        dst = regCaseDir + "/" + timeFolders[0] + "/solidRegions"
        shutil.copytree(src,dst)

    #各regionにfvSchemes,fvSolutionをコピーする。
    #copyFvschemesFvsolution(regCaseDir, timeFolders[0], regionNames)
    #親case/system内のregionフォルダを削除する。
    delFvschemesFvsolution(caseDir, timeFolders[0], regionNames)
    error = ["OK", ""]
    return error


#
#  Run
#  ---
def Run(caseDir):
    """ multiRegion用にcaseを作成する"""
    splitMeshRegions(caseDir)
    error = createRegCase(caseDir, "reg")
    return error


if __name__ == "__main__":
    import gettext
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

