#!/usr/bin/python3
# coding: utf-8
#
#   setMultiRegionDialog.py
#
#       multiRegionCaseの編集
#
#   13/04/13    新規作成
#      05/22    changeDictを読み込んで自動作成するを追加
#      06/22    regionの追加・削除ボタン追加
#      09/18    import先を修正
#      09/28    changeDictのdialogを変更
#      10/15    国際化のため修正
#   14/02/08    fileOperationInRegionを起動した時、closeする様に修正
#      05/11    loadFromGaugeDialogを修正（defineDataを追加）
#      06/07    loadFromGaugeDialogのバグ修正
#      10/05    setBoundaryConditionsを追加（境界面の境界条件）
#      11/15    regionWallBoundaryの編集、設定のﾎﾞﾀﾝ2ヶ追加
#      12/22    getBoundaryFieldData:getDimensionInternalFieldsDataからの
#               戻り値を修正
#   15/02/27    getFieldBoundaryData:変数の書き込み追加
#      07/17    dialogの起動方法を修正
#   16/12/30    getFieldBoundaryData:OF-4.0用にchangeDictionaryの書式修正（バグ）
#      12/31    getFieldBoundaryData:OF-ext用に修正
#   17/09/08    currCaseDirを引数で取得する様に修正
#   19/10/25    GTK+3, python3用に大幅修正
#   20/04/22    多言語化対応
#      05/27    バグ修正。import editBoudaryConditionsDialogでエラー発生
#      10/29    「import gridEditorQtDialog」を「import readBoundaryAndFields」
#               に変更。gridEditorでは、vtkをimportするので、時間が掛かる為、
#               該当部分をreadBoundaryAndFieldsを作成し、移動した。
#   22/07/19    setBoundaryConditions,getFieldBoundaryData:OFversionのcheck方法修正（OF-10対応）
#   24/08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/05    inputTextDDialog:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os, sys
import glob

import pyTreeFoam
import inputTextDDialog
import stringOp
#import progressDialog
#import editBoundaryConditionsDialog
#import gridEditorQtDialog
import readBoundaryAndFields
from createBCsForMultiRegionDialog import checkAllFields, replaceRegionFieldBoundaryConditions, readDictFile
import getOpenRunFileDDialog
import universalDialogs as unvDlg

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()

configDict = {}

#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"setMultiRegionDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.caseDir = caseDir
        #GUIのobject名を取得
        self.setGtkObject()
        #configDictを取得
        self.configDict = pyTreeFoam.readConfigTreeFoam()

    def setGtkObject(self):
        """ glade内のobject名を取得する"""
        pass

    #
    #  main
    #  ----
    def main(self):
        """ GUIの表示"""
        #GUIを表示
        self.mainWindow.show()
        #multiRegionのチェック
        self.checkMultiRegion()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    #-------- event handlar ----------
    #境界条件編集・設定
    def onSetBoundaryConditions(self, event):
        self.setBoundaryConditions()
    #textEditorによるDict編集
    def onEditDict(self, event):
        self.editDict()
    #Dict実行（設定）
    def onRunDict(self, event):
        self.runDict()
    #現設定を取得・作成（changeDict作成）
    def onMakeChangeDict(self, event):
        self.makeChangeDict()
    #changeDict編集・実行
    def onEditRunChangeDict(self, event):
        self.editRunChangeDict()
    #region追加・削除
    def onAddDelRegion(self, event):
        self.addDeleteRegion(event)
    #file操作
    def onFileOperation(self, event):
        self.fileOperation(event)
    #閉じる
    def onClose(self, event):
        self.close()


    #
    #  setBoundaryConditions
    #  ---------------------
    def setBoundaryConditions(self):
        """ 境界面の境界条件を設定"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion[:3] == "ext":
            title = _("エラー")
            mess = _("この機能は、OpenFOAM-ext では使えません。\n")
            mess += _("OpenFOAM-2.2 以降で使用できます。")
            self.errDialog(title, mess)
            return
        #elif OFversion >= "2.2.0":
        elif numVer >= 2.2 or OFversion[0] == "v":
            #caseDir = os.getcwd()
            path = os.getenv("TreeFoamPath") + os.sep + "python"
            comm = path + "/createBCsForMultiRegionDialog.py " + self.caseDir + " &"
            pyTreeFoam.run(self.caseDir).command(comm)
        else:
            title = _("エラー")
            mess = _("この機能は、OpenFOAM-2.1 以前では使えません。\n")
            mess += _("OpenFOAM-2.2 以降で使用できます。")
            self.errDialog(title, mess)

    #
    #  editDict
    #  --------
    def editDict(self):
        """ regionWallBoundaryConditionsDictをeditorで編集する"""
        editFile = self.caseDir + "/system/regionWallBoundaryConditionsDict"
        pyTreeFoam.run(self.caseDir).editor([editFile])

    #
    #  runDict
    #  -------
    def runDict(self):
        """ regionWallBoundaryConditionsを実行する"""
        #dict有無の確認
        dictName = "regionWallBoundaryConditionsDict"
        fileName = self.caseDir + "/system/" + dictName
        if len(glob.glob(fileName)) == 0:
            title = _("エラー")
            mess = dictName + _("が存在しない為、実行（設定）できません。")
            self.errDialog(title, mess)
            return
        #各fieldの界面の境界条件が既に設定されているかどうかをチェック
        caseDir = self.caseDir
        [error, mess] = checkAllFields(caseDir)
        if error != "OK":
            title = _("警告")
            okArgs = [self.runDict_run]
            self.okCancelDialog(title, mess, funcOk=okArgs)
            return
        self.runDict_run()

    #  runDict_run
    def runDict_run(self):
        #fieldの置き換え
        OFversion = self.configDict["OFversion"]
        (fluidRegions, solidRegions) = pyTreeFoam.case(self.caseDir).getZonesInRegionProperties(OFversion)
        (error, errFields) = replaceRegionFieldBoundaryConditions(caseDir, 
                                fluidRegions, solidRegions)
        if error == "OK":
            title = _("界面の境界条件設定")
            mess = _("領域間の境界条件とinternalFieldを再設定しました。\n")
            if len(errFields) > 0:
                mess += _("以下のfieldは、存在しない為、修正していません。\n")
                if len(errFields) > 5:
                    addMess = ", ".join(errFields[:5]) + "..."
                else:
                    addMess = ", ".join(errFields)
                mess += addMess
            print(mess)
            self.okDialog(title, mess)
        else:
            title = _("エラー")
            self.errDialog(title, errFields)
        return

    #
    #  makeChangeDict
    #  --------------
    def makeChangeDict(self):
        """ fieldデータを読み込みchangeDictを作成する"""
        if self.checkMultiRegion() == False:
            return

        timeFolder = pyTreeFoam.case(self.caseDir).getCurrTimeFolder()
        title = _("ファイル名の入力")
        message = _("保存するファイル名を入力してください。\nboundaryFieldは、「") + timeFolder + _(u"」フォルダ(firstTime)から取得します。")
        inputText = "changeDictionaryAllRegionsDict-auto"
        okArgs = [self.makeChangeDict_run, timeFolder]
        dialog = inputTextDDialog.getInputText(title, message, inputText, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return
        # if result != "":

    #  makeChangeDict_run
    def makeChangeDict_run(self, result, timeFolder):
        fileName = self.caseDir + "/system/" + result
        contents = getFieldBoundaryData(self.caseDir, timeFolder)
        f=open(fileName, "w"); f.write(contents); f.close()
        title = _("ファイルの保存")
        message = _("ファイルを保存しました\nboundaryFieldのデータは、「") + timeFolder + _(u"」フォルダから取得してchangeDictを作成しました。")
        self.okDialog(title, message)

    #
    #  checkMultiRegion
    def checkMultiRegion(self):
        """ multiRegion用のcaseかどうかチェック。"""
        (ans, _loc, _regNames) = pyTreeFoam.case(self.caseDir).isMultiRegion()
        if ans == False:
            title = _("警告")
            mess = _("現在のcaseは、multiRegionではありません。")
            self.warningDialog(title, mess)
        return ans

    #
    #  editRunChangeDict
    #  -----------------
    def editRunChangeDict(self):
        """ changeDictionaryの編集・実行。
        編集はdialog内で実行するので、ここでは実行のみ記述。"""
        if self.checkMultiRegion() == False:
            return
        #runFileの取得
        title = _(u"実行fileの選択")
        message = _(u"changeDictionaryAllRegionsを編集・実行します。\n  Dict fileを選択してください。")
        currDir = self.caseDir + "/system"
        fileFilter = "changeDictionaryAllRegionsDict*"
        okArgs = [self.editRunChangeDict_check, currDir]
        dialog = getOpenRunFileDDialog.getOpenRunFile(
            title, message, currDir, fileFilter, 
            funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #  editRunChangeDict_check
    def editRunChangeDict_check(self, selFileName, currDir):
        #fileName = dialog.selFileName
        fileName = selFileName
        runFile = currDir + "/" + fileName
        if len(glob.glob(runFile)) == 0:
            title = _(u"エラー")
            msg = _(u"「") + runFile + _(u"」ファイルがありません。")
            self.errDialog(title, msg)
            return
        #実行
        title = _(u"changeDictionaryAllRegionsの実行")
        mess = _(u"changeDictionaryAllRegionsを実行します。\n")
        mess += _(u"全regionのboundaryFieldの設定が変更されます。\n")
        mess += _(u"  実行しますか？")
        okArgs = [self.editRunChangeDict_run, fileName]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return

    #  editRunChangeDict_run
    def editRunChangeDict_run(self, fileName):
        path = os.getenv("TreeFoamPath") + "/python/"
        comm = path + "changeDictionaryAllRegions.py " + fileName
        (stat, _res, _err) = pyTreeFoam.run(self.caseDir).commandReturnCont(comm)
        if stat == "OK":
            title = _(u"changeDictionaryAllRegionsの実行")
            msg = _(u"全てのregionに対して「changeDictionary」を適用しました。")
            self.okDialog(title, msg)
        else:
            msg = _(u"エラーが発生しました。\n　logを確認してください。")
            self.errDialog(title, msg)

    #
    #  addDeleteRegion
    #  ---------------
    def addDeleteRegion(self, event):
        """ regionの追加・削除"""
        path = os.getenv("TreeFoamPath") + "/python/"
        comm = path + "addDeleteRegionDialog.py " + self.caseDir
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  fileOperation
    #  -------------
    def fileOperation(self, event):
        """ region内のfile操作"""
        if self.checkMultiRegion() == False:
            return
        #実行
        path = os.getenv("TreeFoamPath") + "/python"
        comm = path + "/fileOperationInRegionDialog.py " + self.caseDir + " 1 &"
        pyTreeFoam.run(self.caseDir).command(comm)
        self.close()


    #---- universalDialogs ---------------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def warningDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.warningDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()


#
#  getFieldBoundaryData
def getFieldBoundaryData(caseDir, timeFolder):
    """ 全fieldのinternal、boundaryFieldとboundaryの設定を取得。
    internalField:uniform形式のデータ。
    boundaryField:U, T fieldは、全データ。それ以外は、zeroGradient以外のデータを取得。
    boundary:全データを取得"""
    (flag, _loc, regNames) = pyTreeFoam.case(caseDir).isMultiRegion()
    if flag == False:
        contents = ""
        return contents

    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    foamContents = pyTreeFoam.getFoamContents(caseDir)
    contents = foamContents.makeFoamHeaderVersion(OFversion)
    contents += foamContents.makeFoamFile("2.0", "ascii", "dictionary",
                "system", "changeDictionaryAllRegionsDict")
    for i in range(len(regNames)):
        regName = regNames[i]
        #boundaryデータの取得
        boundaryDir = pyTreeFoam.case(caseDir).getCurrMeshDir(timeFolder, regName, "boundary")
        rowNames = getRowNamesFromBoundary(boundaryDir)
        patchData = getPatchData(boundaryDir, rowNames)
        patchesCont = getAllPatchData(rowNames, patchData)
        if OFversion[:3] == "ext":
            patchesCont = addSpace(8, patchesCont)
            #boundaryFieldデータの取得
            fields = pyTreeFoam.case(caseDir).getFieldNames(timeFolder, regName)
            bcCont = getBoundaryFieldData(caseDir, timeFolder, regName, fields)
            bcCont = addSpace(8, bcCont)
            contents += regName + "\n"
            contents += "{\n"
            contents += "    dictionaryReplacement\n"
            contents += "    {\n"
            contents += patchesCont
            contents += bcCont
            contents += "    }\n"
            contents += "}\n"
        #elif OFversion >= "4.0":
        elif numVer >= 4.0 or OFversion[0] == "v":
            patchesCont = addSpace(4, patchesCont)
            #boundaryFieldデータの取得
            fields = pyTreeFoam.case(caseDir).getFieldNames(timeFolder, regName)
            bcCont = getBoundaryFieldData(caseDir, timeFolder, regName, fields)
            bcCont = addSpace(4, bcCont)
            contents += regName + "\n"
            contents += "{\n"
            contents += patchesCont
            contents += bcCont
            contents += "}\n"
        else:
            patchesCont = addSpace(8, patchesCont)
            #boundaryFieldデータの取得
            fields = pyTreeFoam.case(caseDir).getFieldNames(timeFolder, regName)
            bcCont = getBoundaryFieldData(caseDir, timeFolder, regName, fields)
            bcCont = addSpace(8, bcCont)
            contents += regName + "\n"
            contents += "{\n"
            contents += "    dictionaryReplacement\n"
            contents += "    {\n"
            contents += patchesCont
            contents += bcCont
            contents += "    }\n"
            contents += "}\n"
        i += 1
    contents += pyTreeFoam.getFoamContents(caseDir).makeFoamFooter()
    return contents

#
#  getBoundaryFieldData
def getBoundaryFieldData(caseDir, timeFolder, region, fields):
    """ 各fieldのbaoundaryFieldのデータ作成"""
    #読み込み    
    #load = editBoundaryConditionsDialog.load(caseDir, timeFolder, region, fields)
    #load = gridEditorQtDialog.load(caseDir, timeFolder, region, fields)
    load = readBoundaryAndFields.load(caseDir, timeFolder, region, fields)
    error = load.data()
    if error != "":
        print(error)
    #読み込みデータを取得
    intData = load.intData
    variables = load.varData
    variablesBoundary = load.varBndData
    data = load.patchData
    patchConts = load.patchConts
    rowNames = list(map(lambda x: x[0], patchConts))
    #variablesを整形
    i = 0
    for variable in variables:
        variables[i] = addSpace(4, variable)
        i += 1
    #variablesBoundaryを整形
    i = 0
    for variable in variablesBoundary:
        variablesBoundary[i] = addSpace(8, variable)
        i += 1
    #internalDataを選別（uniform形式のみ取得）
    i=0
    while i<len(intData):
        lines = intData[i].split("\n")
        if lines[0].split()[0] != "uniform":
            intData[i] = ""
        i += 1
    #boundaryFieldDataを選別（calculatedと20行以上は取得せず）
    i=0
    while i<len(data):
        j = 0
        while j<len(data[i]):
            lines = data[i][j].split("\n")
            if len(lines) > 20:
                data[i][j] = ""
            j += 1
        i += 1
    #field毎にdata作成
    cont = ""
    i=0
    while i<len(fields):
        #field名書き出し
        field = fields[i]
        cont += field + "\n"
        cont += "{\n"
        #variables書き出し
        cont += variables[i]
        #internalField書き出し
        if intData[i] != "":
            cont += "    internalField " + intData[i] + "\n"
        #TとU fieldの境界条件は全て書き出し
        if field == "T" or field == "U":
            cont += "    boundaryField\n"
            cont += "    {\n"
            if data[i] != "":
                #variableBoundaryの書き出し
                cont += variablesBoundary[i]
                #境界条件を書き出し
                j = 0
                labels = data[i]
                while j<len(labels):
                    cont += "        " + rowNames[j] + "\n"
                    cont += "        {\n"
                    cont += addSpace(12, labels[j])
                    cont += "        }\n"
                    j += 1
                cont += "    }\n"
        #T、U以外の境界条件を書き出し
        else:
            cont += "    boundaryField\n"
            cont += "    {\n"
            #variablesBoundaryの書き出し
            cont += variablesBoundary[i]
            #境界条件を書き出し
            j = 0
            labels = data[i]
            while j<len(labels):
                if labels[j] != "":
                    #zeroGradientは、書き出しせず
                    if labels[j].split("\n")[0].find("zeroGradient") < 0:
                        cont += "        " + rowNames[j] + "\n"
                        cont += "        {\n"
                        cont += addSpace(12, labels[j])
                        cont += "        }\n"
                j += 1
            cont += "    }\n"
        cont += "}\n"
        i += 1
    #cont = addSpace(8, cont)
    return cont

#
#  getRowNamesFromBoundary
def getRowNamesFromBoundary(boundaryDir):
    """ rowNamesをboundaryから取得する"""
    ans = []
    #values = getFoamContents.getPatchNameFromBoundary(boundaryDir)
    meshDir = "/".join(boundaryDir.split("/")[:-1])
    values = pyTreeFoam.case(caseDir).getPatchNameFromBoundary(meshDir)
    for patchG in values:
        ans.append(patchG[0])
    #if sortPatchName == "yes":
    ans.sort()
    return ans

#
#  getPatchData
def getPatchData(boundaryDir, rowNames):
    """ boundaryのpatch定義を取得"""
    patchData = getDefinePatchData(boundaryDir, rowNames)
    ans = []
    for data in patchData:
        lines = data.split("\n")
        cont = ""
        for line in lines:
            words = line.split()
            if words[0] != "nFaces" and words[0] != "startFace":
                cont += line + "\n"
        ans.append(cont)
    return ans

#
#  getDefinePatchData
def getDefinePatchData(wdir, names):
    """ boundaryの定義内容を取得"""
    fileName = wdir + "/boundary"
    if len(glob.glob(fileName)) == 0:
        return []

    f=open(fileName); contents=f.read(); f.close()
    defineD = []
    contentsOp = stringOp.strings(contents)
    p = contentsOp.skipFoamFile()
    (boundCont, p) = contentsOp.getSmallPair(p)
    for keyword in names:
        p=0
        boundContOp = stringOp.strings(boundCont)
        ((patchName, cont), p) = boundContOp.getNextKeywordAndContents(p)
        while(p <= len(boundCont) and patchName != b""): 
            if keyword == patchName.decode():
                defineD.append(cont.decode())
                break
            ((patchName, cont), p) = boundContOp.getNextKeywordAndContents(p)

    #空白を詰める
    newDefine = []
    for define in defineD:
        newLines = []
        lines = define.split("\n")
        for line in lines:
            words = line.split()
            if len(words)>0:
                wd = " ".join(words)
                newLines.append(wd)
        if len(newLines)>0:
            newDefine.append("\n".join(newLines))
        else:
            newDefine.append("")
    return newDefine

# #
# #  deleteNonuniformListData
# def deleteNonuniformListData(defineDataSub):
#     """ patch定義データからnonuniform形式のデータを削除する"""
#     defines = []
#     for definePatch in defineDataSub:
#         define = ""
#         lines = definePatch.split(";")
#         for line in lines:
#             words = line.split()
#             if len(words)>=3:
#                 if not (words[1] == "nonuniform" and words[2][:len("List")] == "List"):
#                     define += line + ";\n"
#             elif len(line) != 0:
#                 define += line + ";\n"
#         defines.append(define)
#     return defines


#
#  getAllPatchData
def getAllPatchData(rowNames, patchData):
    """ boundaryのpatch dataを作成"""
    cont = "boundary\n{\n"
    i=0
    while i<len(rowNames):
        patchLines = patchData[i].split("\n")
        cont += "    " + rowNames[i] + "\n"
        cont += "    {\n"
        for line in patchLines:
            if line != "":
                cont += "        " + line + "\n"
        cont += "    }\n"
        i += 1
    cont += "}\n"
    return cont

#
#  addSpace
def addSpace(n, cont):
    """ 行頭にスペース追加"""
    space = " "*n
    ans = ""
    lines = cont.split("\n")
    for line in lines:
        ans += space + line + "\n"
    return ans


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    configDict = pyTreeFoam.readConfigTreeFoam()
    winApp = windowApp(caseDir)
    winApp.main()
