#!/usr/bin/python3
#  coding: utf-8
#
#   setDataInFieldsDialog.py
#
#   各fieldへのデータセット（クリア）
#
#       programed by s.fujii
#   11/04/10    新規作成
#   12/07/16    編集するtimeFolderの変更ができる様に変更
#   13/05/09    field選択dialogを変更
#      06/30    timeFolder選択方法、region選択追加
#      07/02    fields, cellSetsのリスト表示追加
#      07/13    選択fieldsをclearInternal、zeroGradientに引き継ぐ様に修正
#               setFieldsDict時にfield、setsを選択するとこれを使ってDict作成
#      07/17    gridEditor, topoSetEditorの起動を追加
#      07/27    setFieldsでOF2.1以下は「-region」オプション使えない事の表示追加
#      09/25    getFieldTypeのバグ修正
#      10/15    国際化のため、修正
#      12/02    setFieldsDict作成をgridで編集できる様に修正
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/11/04    mapFieldsを追加
#      12/03    mapFields:バグ修正（terget→targetに修正）
#   15/02/16    setSourceTimeFolders:バグ修正（startFrom→sourceStartFrom）
#      02/18    mapFieldsの全ﾊﾟﾗﾒｰﾀ保存を追加
#      07/13    setDataInFields_dataの場所変更
#      11/28    getSourceCaseDir:バグ修正。（sourceCaseを変更した時、sourcefields
#               が取り込めていない。）
#               initialize:OF-3.0.0用のmapMethodを追加
#      12/02    runMapFields:終了時にメッセージを追加。
#               「mapFieldsDict編集」ボタンを追加
#               mapFields実行時に、確認用dialogを追加
#      12/30    initialize:OF-ext用に修正
#   17/03/08    getSetFieldsCellToCell:バグ修正。fieldValues内からfield名の取得
#               方法に誤りがあり、vectorFieldの次のfield名が取得できない。
#   19/05/21    onRunTopoSetEditor:topoSetEditorの起動方法を修正
#                   OF-v1812対応として、v1812用のtopoSetEditorを作成した為。
#   20/01/16    GTK+3, python3用に大幅変更
#      04/22    多言語化対応
#   22/07/18    createCellSet:OFversionのチェック方法を修正（OF-10対応）
#      07/19    iniGui,runSetFields,runMapFields:OFversionのチェック方法を修正（OF-10対応）
#      07/22    mapFieldsの処理を削除
#   23/03/01    getFirstStartLatestTime:startTime:0.0で設定した場合、エラー発生
#               timeFoldersを数値に変換して、startTimeを取得する様に修正
#   24/08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#   25/07/20    getSeledFieldsCellZones:setFields内からfieldsとcellZoneを取得。新規追加。
#

import os

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GdkPixbuf
import sys
import glob
import shutil

import pyTreeFoam
import GtkParts
import selectFolderFilesDialog
import universalDialogs as unvDlg

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()

#-------------------
#  windowApp class
#-------------------
class windowApp:

    def __init__(self, caseDir, currTime):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"setDataInFieldsDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を取得
        self.setGtkObject()
        #属性の設定
        self.caseDir = caseDir
        self.currTime = currTime
        self.selRegion = "."
        self.regions = []
        self.meshDir = ""
        self.contDict = pyTreeFoam.case(caseDir).getControlDict()
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        #self.setDataDict = self.readSetDataInFields_data()
        iconDir = os.getenv("TreeFoamPath") + "/icons"
        iconFile = GdkPixbuf.Pixbuf.new_from_file
        self.iconFolder = iconFile(iconDir + "/folder16.png")
        self.iconFile = iconFile(iconDir + "/text8.png")
        self.iconFolderLink = iconFile(iconDir + "/folderLink16.png")
        self.iconFileLink = iconFile(iconDir + "/textLink8.png")
        self.maskEvent = True
        #time, regionを取得
        self.setTimeFolderRegion()
        #Guiの初期化
        self.iniGui()
        #dictファイルを確認（無ければdefaultをコピー）
        self.checkDictFiles()


    #
    #  main
    #--------
    def main(self):
        self.maskEvent = False
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()


    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ GUIのobject名を取得"""
        self.combo_time = self.builder.get_object("combo_time")
        self.combo_time.set_sensitive(False)
        self.combo_region = self.builder.get_object("combo_region")
        self.tree_cellSets = self.builder.get_object("tree_cellSets")
        self.tree_setFields = self.builder.get_object("tree_setFields")
        self.button_cellSet = self.builder.get_object("button_cellSet")
        self.label_cellSet = self.builder.get_object("label_cellSet")

    #--------- event handler --------------
    #閉じる
    def onClose(self, event):
        self.close()
    #--------- 編集するfolder ---------
    #time
    def onChangeTime(self, event):
        if self.maskEvent == False:
            self.changeTime()
    #region
    def onChangeRegion(self, event):
        if self.maskEvent == False:
            self.changeRegion()
    #---------- setFields -----------
    #cellSet作成（topoSet起動）
    def onCreateCellSet(self, event):
        self.createCellSet()
    #field内容確認（gridEditor起動）
    def onShowFieldCont(self, event):
        self.showFieldCont()
    #setFieldsDict作成
    def onCreateSetFieldsDict(self, event):
        self.createSetFieldsDict()
    #setFieldsDict編集
    def onEditSetFieldsDict(self, event):
        self.editSetFieldsDict()
    #setFields実行
    def onRunSetFields(self, event):
        self.runSetFields()
    #systemフォルダ開く
    def onOpenSystemFolder(self, event):
        self.openSystemFolder()
    #------ fieldクリア ------------
    #internalFieldのクリア
    def onClearInternalField(self, event):
        self.clearInternalField()
    #boundaryFieldのクリア
    def onClearBoundaryField(self, event):
        self.clearBoundaryField()
    #--------------------------------------

    #
    #  setTimeFolderRegion
    #----------------------
    def setTimeFolderRegion(self):
        """ timeFolder, regionを取得する"""
        case = pyTreeFoam.case(self.caseDir)
        if self.currTime == "":
            self.currTime = case.getCurrTimeFolder()
        else:
            if len(glob.glob(self.currTime)) == 0:
                self.currTime = case.getCurrTimeFolder()
        (_flag, _loc, regions) = case.isMultiRegion()
        if self.selRegion == "":
            self.selRegion = "."
        elif len(glob.glob(self.caseDir + "/constant/" + self.selRegion)) == 0:
            self.selRegion = "."
        self.regions = ["."] + regions

    #
    #  getFirstStartLatestTime
    #--------------------------
    def getFirstStartLatestTime(self):
        """ first, start, latestTimeを取得する"""
        timeFolders = pyTreeFoam.case(self.caseDir).getTimeFolders()
        startTime = self.contDict["startTime"]
        times = list(map(float, timeFolders))
        if float(startTime) in times:
            idx = times.index(float(startTime))
            startTime = timeFolders[idx]
        items = ["firstTime:" + timeFolders[0]]
        items += ["startTime:" + startTime]
        items += ["latestTime:" + timeFolders[-1]]
        return items

    #
    #  iniGui
    #---------
    def iniGui(self):
        """ GUIの初期化"""
        #titleの設定
        self.mainWindow.set_title("Fieldへのデータセット")
        #comboBoxへのデータセット
        self.setDataToAllComboBox()
        #treeListの初期化(setFields)
        GtkParts.treeListIcon(self.tree_cellSets).create()
        GtkParts.treeListIcon(self.tree_setFields).create()
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            self.button_cellSet.set_sensitive(False)
            self.label_cellSet.set_text("cellZone")
        else:
            self.button_cellSet.set_sensitive(True)
            self.label_cellSet.set_text("cellSet")
        #TreeListに値をセット(setFields)
        #  cellsets or cellZonesを取得
        self.setCellSetsToTreelist()
        #  fieldsをセット
        self.setFieldsToTreeList()
        #  setFieldsDictの読み込み選択表示させる
        if numVer >= 13.0:
            (selFields, selCellSets) = self.getSeledFieldsCellZones()
        else:
            (selFields, selCellSets) = self.getSeledFieldsCellSets()
        GtkParts.treeListIcon(self.tree_setFields).selectItemNames(selFields)
        GtkParts.treeListIcon(self.tree_cellSets).selectItemNames(selCellSets)

    #
    #  getSeledFieldsCellZones
    def getSeledFieldsCellZones(self):
        """ setFieldsDict内から選択しているfieldsとcellZonesを取得する"""

        def getSetFieldsCellToCell(name, cont):
            """ cont内からcellZone名とfieldsを取得"""
            cellZone = ""
            fields = []
            p = 0
            contOp = pyTreeFoam.strOp(cont)
            (keyword, p) = contOp.getKeyword(p)
            while keyword != "":
                if keyword == "type":
                    (word, p) = contOp.getKeyword(p)
                    if word == "lookup":
                        #cellZone名を取得
                        cellZone = name
                elif keyword == "values":
                    #fieldsを取得
                    valCont, p = contOp.getMiddlePair(p)
                    words = valCont.split()
                    for i in range(len(words)):
                        field = words[i]
                        fields.append(field)
                        i += 2
                (keyword, p) = contOp.getKeyword(p)
            return (cellZone, fields)

        fileName = self.caseDir + "/system/" + self.selRegion + "/setFieldsDict"
        if len(glob.glob(fileName)) == 0:
            #setFieldsが存在しない場合
            return ([], [])
        #cellZones、cellFieldsを求める
        cellZones = []
        cellFields = []
        f = open(fileName); cont = f.read(); f.close()
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        #defaultValue内からcellFfieldsを取得
        (keyword, p) = contOp.getKeyword(p)
        while keyword != "zones" and keyword != "":
            if keyword == "defaultValues":
                (valCont, p) = contOp.getMiddlePair(p)
                words = valCont.split()
                i = 0
                while i < len(words):
                    cellFields.append(words[i])
                    i += 2
            (keyword, p) = contOp.getKeyword(p)
        if keyword != "zones":
            return (cellFields, cellZones)
        
        #cellZonesとcellFieldsを取得
        (regCont, p) = contOp.getMiddlePair(p)
        regContOp = pyTreeFoam.strOp(regCont)
        ((keyword, cont), p) = regContOp.getNextKeywordAndContents(0)
        while keyword != "":
            (cellZone, fields) = getSetFieldsCellToCell(keyword, cont)
            if cellZone != "":
                cellZones.append(cellZone)
            cellFields += fields
            ((keyword, cont), p) = regContOp.getNextKeywordAndContents(p)
        #同名のcellZoneを削除
        cellZones = list(set(cellZones))
        cellZones.sort()
        #同名のcellFieldsを削除
        cellFields = list(set(cellFields))
        cellFields.sort()
        return (cellFields, cellZones)

    #
    #  getSeledFieldsCellSets
    def getSeledFieldsCellSets(self):
        """ setFieldsDict内から選択しているfieldsとcellSetsを取得する"""
        
        def getSetFieldsCellToCell(cont):
            cellSet = ""
            fields = []
            p = 0
            contOp = pyTreeFoam.strOp(cont)
            (keyword, p) = contOp.getKeyword(p)
            while keyword != "":
                if keyword == "set":
                    (cellSet, p) = contOp.getKeyword(p)
                elif keyword == "fieldValues":
                    (fieldCont, p) = contOp.getSmallPair(p)
                    lines = fieldCont.split("\n")
                    for line in lines:
                        words = line.split()
                        if len(words) > 1:
                            field = words[1]
                            fields.append(field)
                (keyword, p) = contOp.getKeyword(p)
            return (cellSet, fields)

        fileName = self.caseDir + "/system/" + self.selRegion + "/setFieldsDict"
        if len(glob.glob(fileName)) == 0:
            return ([""], [""])

        cellSets = []
        cellFields = []
        f = open(fileName); cont = f.read(); f.close()
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        (keyword, p) = contOp.getKeyword(p)
        while keyword != "regions" and keyword != "":
            if keyword == "defaultFieldValues":
                (valCont, p) = contOp.getSmallPair(p)
                words = valCont.split()
                i = 0
                while i < len(words):
                    cellFields.append(words[i+1])
                    i += 3
            (keyword, p) = contOp.getKeyword(p)
        if keyword != "regions":
            return (cellFields, cellSets)

        (regCont, p) = contOp.getSmallPair(p)
        regContOp = pyTreeFoam.strOp(regCont)
        ((keyword, cont), p) = regContOp.getNextKeywordAndContents(0)
        while keyword != "":
            if keyword == "cellToCell":
                (cellSet, fields) = getSetFieldsCellToCell(cont)
                cellSets.append(cellSet)
                cellFields += fields
            elif keyword == "boxToCell":
                (cellSet, fields) = getSetFieldsCellToCell(cont)
                cellFields += fields
            elif keyword == "cylinderToCell":
                (cellSet, fields) = getSetFieldsCellToCell(cont)
                cellFields += fields
            elif keyword == "sphereToCell":
                (cellSet, fields) = getSetFieldsCellToCell(cont)
                cellFields += fields
            ((keyword, cont), p) = regContOp.getNextKeywordAndContents(p)
        cellSets = list(set(cellSets))
        cellSets.sort()
        cellFields = list(set(cellFields))
        cellFields.sort()
        return (cellFields, cellSets)

    #
    #  setCellSetsToTreelist
    def setCellSetsToTreelist(self):
        """ 指定したtime、region内のcellSetsをTreeListに表示する"""
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 13.0:
            cellZones = self.getCellZones()
            items = []
            icon = self.iconFile
            for cellZone in cellZones:
                item = [icon, cellZone]
                items.append(item)
        else:
            cellSets = self.getCellSets()
            items = []
            for cellSet in cellSets:
                cellDir = self.meshDir + "/sets/" + cellSet
                icon = self.iconFile
                if os.path.islink(cellDir):
                    icon = self.iconFileLink
                item = [icon, cellSet]
                items.append(item)
        items.sort()
        GtkParts.treeListIcon(self.tree_cellSets).setItems(items)

    #
    #  setFieldsToTreeList
    def setFieldsToTreeList(self):
        """ 指定したtime、region内のfieldをTreeListに表示する"""
        fields = pyTreeFoam.case(self.caseDir).getFieldNames(self.currTime, self.selRegion)
        fields.sort()
        items = []
        for field in fields:
            fieldDir = self.caseDir + "/" + self.currTime + "/" + self.selRegion + "/" + field
            icon = self.iconFile
            if os.path.islink(fieldDir):
                icon = self.iconFileLink
            item = [icon, field]
            items.append(item)
        GtkParts.treeListIcon(self.tree_setFields).setItems(items)

    #
    #  setDataToAllComboBox
    def setDataToAllComboBox(self):
        """ comboBoxへのデータセット"""
        #comboTimeへのデータセット
        items = self.getFirstStartLatestTime()
        comboBox = GtkParts.comboBoxText(self.combo_time)
        comboBox.setItems(items)
        startFrom = self.contDict["startFrom"]
        if startFrom == "firstTime":
            selNo = 0
        elif startFrom == "startTime":
            selNo = 1
        else:
            selNo = 2
        comboBox.selectItem(items[selNo])
        #comboRegionへのデータセット
        items = self.regions[:]
        items[0] = "(region0)"
        comboBox = GtkParts.comboBoxText(self.combo_region)
        comboBox.setItems(items)
        comboBox.selectItem(items[0])

    #
    #  getCellZones
    def getCellZones(self):
        """ cellZone名を取得する"""
        maxTime = self.currTime
        region = self.selRegion
        relMeshDir = pyTreeFoam.case(self.caseDir).getCurrMeshDir(maxTime, region, "boundary")
        self.meshDir = self.caseDir + "/" + relMeshDir
        foamConts = pyTreeFoam.getFoamContents(self.caseDir)
        cellZones = foamConts.cellZoneNames(self.meshDir)
        return cellZones

    #
    #  getCellSets
    def getCellSets(self):
        maxTime = self.currTime
        region = self.selRegion
        relMeshDir = pyTreeFoam.case(self.caseDir).getCurrMeshDir(maxTime, region, "boundary")
        self.meshDir = self.caseDir + "/" + relMeshDir
        setsDir = self.meshDir + "/sets"
        files = pyTreeFoam.getFileNames(setsDir)
        cellSets = []
        if len(files) > 0:
            for fileName in files:
                foamFile = pyTreeFoam.foamFile()
                cont = foamFile.read(setsDir + "/" + fileName)
                className = foamFile.getClassInFoamFile(cont)
                className = className.decode()
                if className == "cellSet":
                    cellSets.append(fileName)
            cellSets.sort()
        return cellSets

    #
    #  checkDictFiles
    #-----------------
    def checkDictFiles(self):
        """ setFieldsDict, mapFieldsDictファイルを確認し、無ければ作成する"""
        setDict = self.caseDir + "/system/setFieldsDict"
        mapDict = self.caseDir + "/system/mapFieldsDict"
        getFoam = pyTreeFoam.getFoamContents(self.caseDir)
        OFversion = self.configDict["OFversion"]
        if len(glob.glob(setDict)) == 0:
            #defaultをコピーする
            header = getFoam.makeFoamHeaderVersion(OFversion)
            #foamFile = getFoam.makeFoamFile(OFversion, "ascii", "dictionary", "system", "setFieldsDict")
            foamFile = getFoam.makeFoamFile("2.0", "ascii", "dictionary", "system", "setFieldsDict")
            dictCont = getFoam.defaultSetFieldsDict(OFversion)
            footer = getFoam.makeFoamFooter()
            cont = header + foamFile + dictCont + footer
            f = open(setDict, "w"); f.write(cont); f.close()
        if len(glob.glob(mapDict)) == 0:
            #defaultをコピー
            newDict = getFoam.defaultDictPathInDataFolder(OFversion, "system/mapFieldsDict")
            shutil.copy(newDict, mapDict)

    #-------- 編集するfolderのevent ---------------

    #
    #  changeTime
    #-------------
    def changeTime(self):
        """ caseDirのtimeを変更した時"""
        currTime = GtkParts.comboBoxText(self.combo_time).getSelectedItem()
        self.currTime = currTime.split(":")[-1].split()[0]
        self.setFieldsToTreeList()
        #setFieldsを読込itemを選択
        (selFields, selCellSets) = self.getSeledFieldsCellSets()
        GtkParts.treeListIcon(self.tree_setFields).selectItemNames(selFields)
        GtkParts.treeListIcon(self.tree_cellSets).selectItemNames(selCellSets)

    #
    #  changeRegion
    #---------------
    def changeRegion(self):
        """ caseDirのregionを変更した時"""
        self.selRegion = GtkParts.comboBoxText(self.combo_region).getSelectedItem()
        if self.selRegion == "(region0)":
            self.selRegion = "."
        #cellSetsをTreeListにセット
        self.setCellSetsToTreelist()
        #fieldをTreeListにセット
        self.setFieldsToTreeList()
        #setFieldsを読込itemを選択
        (selFields, selCellSets) = self.getSeledFieldsCellSets()
        GtkParts.treeListIcon(self.tree_setFields).selectItemNames(selFields)
        GtkParts.treeListIcon(self.tree_cellSets).selectItemNames(selCellSets)


    #------- setFields関連の処理 -------------------

    #
    #  createCellSet
    #---------------
    def createCellSet(self):
        """ topoSetEditorを起動"""
        print(_("topoSetEditorを起動"))
        configDict = self.configDict
        caseDir = self.caseDir
        treeFoamPath = os.getenv("TreeFoamPath")
        currTime = GtkParts.comboBoxText(self.combo_time).getSelectedItem()
        currTime = currTime.split(":")[-1]
        region = GtkParts.comboBoxText(self.combo_region).getSelectedItem()
        if region == "(region0)":
            region = "."
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion[:3] == "ext":
            title = _(u"エラー")
            message = _(u"OpenFOAM-") + OFversion + _(u" ではTopoSetEditorは使えません\n")
            message += _(u"OpenFOAM-2.0.0 以降で使用できます。")
            self.errDialog(title, message)
            return
        #elif OFversion >= "2.0.0" and OFversion <= "v1806":
        elif ( numVer >= 2.0 or 
               (OFversion[0] == "v" and OFversion <= "v1806") ):
            modelDict = pyTreeFoam.case(caseDir).getCreateMeshData()
            stlDir = modelDict["stlDir"]
            path = treeFoamPath + os.sep + "python"
            comm = "./topoSetEditorDialog.py " + caseDir + " --stl " + stlDir
            comm += " --time " + currTime + " --region " + region
            pyTreeFoam.run(path).command(comm)
        elif OFversion >= "v1812":
            modelDict = pyTreeFoam.case(caseDir).getCreateMeshData()
            stlDir = modelDict["stlDir"]
            path = treeFoamPath + os.sep + "python"
            comm = "./topoSetEditorDialog.py " + caseDir + " --stl " + stlDir
            comm += " --time " + currTime + " --region " + region
            pyTreeFoam.run(path).command(comm)
        else:
            title = _(u"エラー")
            message = _(u"OpenFOAM-") + OFversion + _(u" ではTopoSetEditorは使えません\n")
            message += _(u"OpenFOAM-2.0.0 以降で使用できます。")
            self.errDialog(title, message)
            return
        #cellSetを設定する
        self.setCellSetsToTreelist()

    #
    #  showFieldCont
    #----------------
    def showFieldCont(self):
        """ gridEditor起動"""
        print(_("gridEditorを起動"))
        caseDir = self.caseDir
        commPath = os.getenv("TreeFoamPath") + "/python"
        currTime = GtkParts.comboBoxText(self.combo_time).getSelectedItem()
        currTime = currTime.split(":")[-1]
        region = GtkParts.comboBoxText(self.combo_region).getSelectedItem()
        if region == "(region0)":
            region = "."
        selItems = GtkParts.treeListIcon(self.tree_setFields).getSelectedItems()
        selFields = list(map(lambda x: x[1], selItems))
        if len(selFields) == 0:
            title = _("エラー")
            mess = _("fieldを選択してください。")
            self.errDialog(title, mess)
            return
        #displayFieldsファイルを修正
        timeFolders = pyTreeFoam.case(caseDir).getTimeFolders()
        dispFile = caseDir + "/" + timeFolders[0] + "/" + region + "/.displayField"
        allItems = GtkParts.treeListIcon(self.tree_setFields).getAllItems()
        allFields = list(map(lambda x: x[1], allItems))
        if len(selFields) != len(allFields):
            cont = "\n".join(selFields) + "\n"
            f = open(dispFile, "w"); f.write(cont); f.close()
        else:
            if len(glob.glob(dispFile)) != 0:
                os.remove(dispFile)
        #gridEditorを起動
        # comm = commPath + "/editBoundaryConditionsDialog.py "
        # comm += "-1 " + caseDir + " " + currTime + " " + region + " &"
        comm = commPath + "/gridEditorQtDialog.py "
        comm += "-1 " + caseDir + " " + currTime + " " + region + " &"
        pyTreeFoam.run(caseDir).command(comm)

    #
    #  createSetFieldsDict
    #----------------------
    def createSetFieldsDict(self):
        """ setFieldsDictを作成する"""
        selItem = GtkParts.comboBoxText(self.combo_time).getSelectedItem()
        timeFolder = selItem.split(":")[-1].split()[0]
        selRegion = GtkParts.comboBoxText(self.combo_region).getSelectedItem()
        if selRegion == "(region0)":
            selRegion = "."
        (selFields, selCellSets) = self.getSelFieldsCellSets()
        selectFields = " ".join(selFields)
        selectCellSets = " ".join(selCellSets)
        commPath = os.getenv("TreeFoamPath") + "/python"
        comm = commPath + "/createSetFieldsDictQtDialog.py "
        comm += " -case " + self.caseDir
        comm += " -f " + selectFields
        comm += " -c " + selectCellSets + " -r " + selRegion
        comm += " -t " + timeFolder + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  getSelFieldsCellSets
    def getSelFieldsCellSets(self):
        """ 選択しているfieldsとcellSetsを取得する"""
        selItems = GtkParts.treeListIcon(self.tree_setFields).getSelectedItems()
        selFields = list(map(lambda x: x[1], selItems))
        selItems = GtkParts.treeListIcon(self.tree_cellSets).getSelectedItems()
        selCellSets = list(map(lambda x: x[1], selItems))
        return (selFields, selCellSets)

    #
    #  editSetFieldsDict
    #--------------------
    def editSetFieldsDict(self):
        """ setFieldsDictを編集する"""
        caseDir = self.caseDir
        region = GtkParts.comboBoxText(self.combo_region).getSelectedItem()
        if region == "(region0)":
            region = "."
        fileName = caseDir + "/system/" + region + "/setFieldsDict"
        #fileName = caseDir + "/system/setFieldsDict"
        pyTreeFoam.run(caseDir).editor([fileName])

    #
    #  runSetFields
    #---------------
    def runSetFields(self):
        """ setFields実行"""
        print(_("setFieldsを実行します。"))
        title = _("setFields")
        mess = _("Fieldにデータをセットします。\n　（setFieldsの実行）")
        okArgs = [self.runSetFields_run]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return
        # if stat != "OK":
        #     return
        
    #  runSetFields_run
    def runSetFields_run(self):
        bashrcFOAM = self.configDict["bashrcFOAM"]
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        selRegion = GtkParts.comboBoxText(self.combo_region).getSelectedItem()
        if selRegion == "(region0)":
            comm = ". " + bashrcFOAM + "; "
            comm += "setFields"
        else:
            #if OFversion >= "2.2.0":
            if numVer >= 2.2 or OFversion[:3] == "ext" or OFversion[0] == "v":
                comm = ". " + bashrcFOAM + "; "
                comm += "setFields -region " + selRegion
            else:
                mess = _("OF-2.1 以下は、「-region」オプションが使えません。")
                self.errDialog(_("エラー"), mess)
                return
            
        print(comm)
        (stat, _res, _err) = pyTreeFoam.run(self.caseDir).commandReturnCont(comm)
        title = "setFields"
        if stat == "OK":
            mess = _("「setFields」でFieldにデータをセットしました。")
            self.okDialog(title, mess)
        else:
            mess = _("エラーが発生しました。\n　logを確認してください。")
            self.errDialog(title, mess)
    
    #
    #  openSystemFolder
    #------------------
    def openSystemFolder(self):
        """ systemフォルダを開く"""
        openDir = self.caseDir + "/system"
        pyTreeFoam.run(openDir).fileManager()

    #
    #  clearInternalField
    #---------------------
    def clearInternalField(self):
        """ internalFieldのクリア"""
        currTime = GtkParts.comboBoxText(self.combo_time).getSelectedItem()
        currTime = currTime.split(":")[-1].split()[0]
        getFields = selectFolderFilesDialog.clearInternalField(self.caseDir, currTime)
        selFields = getFields.main()
        if len(selFields) == 0:
            return
        region = selFields[0].split("/")[0]
        fields = list(map(lambda x: x.split("/")[-1], selFields))
        fieldDir = self.caseDir + "/" + currTime + "/" + region
        pyTreeFoam.case(self.caseDir).clearInternalFields(fieldDir, fields)
        title = _("internalFieldのクリア")
        mess = _("選択したfieldのinternalFieldをクリアしました。")
        self.okDialog(title, mess)
        
    #
    #  clearBoundaryField
    #---------------------
    def clearBoundaryField(self):
        """ boundaryFieldのクリア"""
        currTime = GtkParts.comboBoxText(self.combo_time).getSelectedItem()
        currTime = currTime.split(":")[-1].split()[0]
        getFields = selectFolderFilesDialog.clearBoundaryField(self.caseDir, currTime)
        selFields = getFields.main()
        if len(selFields) == 0:
            return
        region = selFields[0].split("/")[0]
        fields = list(map(lambda x: x.split("/")[-1], selFields))
        relMeshDir = pyTreeFoam.case(self.caseDir).getCurrMeshDir(currTime, region, "boundary")
        meshDir = self.caseDir + "/" + relMeshDir
        pyTreeFoam.case(self.caseDir).clearNullBoundaryFields(currTime, meshDir, fields, region)
        title = _("boundaryFieldのクリア")
        mess = _("選択したfieldのboundaryFieldをクリアしました。")
        self.okDialog(title, mess)

    #----- universalDialogs ------------------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    currTime = sys.argv[2]
    
    winApp = windowApp(caseDir, currTime)
    winApp.main()
