#!/usr/bin/python3
# coding: utf-8
#
#   runParallelDialog.py
#
#       並列計算の設定、実行
#
#   12/04/30    新規作成
#      11/02    バグ修正（接続状態確認時、currentDirを設定する様に変更）
#      12/09    multiRegionの分割方法、結果再構築方法を変更
#   13/01/11    メッセージ内容を修正
#      01/19    multiRegionの場合のreconstructParを修正
#      03/16    regionPropertiesの有無でmultiRegionかどうかを判定する様に修正
#      05/09    分割後、timeFolder内にmeshをコピーするように修正
#      05/23    decompose,reconstructでzone取得にバグあり、修正
#      10/15    国際化のため、修正
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/11/13    decompose, reconstruct:OF-2.3以上で処理変更
#      11/30    couplingPatchの設定を追加
#      12/02    initialize:decomposeParDict有無、regionCouplingPatchを取得追加
#      12/03    couplingPatch部分を削除
#      12/05    initialize:ﾊﾞｸﾞ修正
#      12/13    runDecomposePar:「-constant」optionを削除
#   15/06/29    decomposeParDict中のmethodとmethodCoeffsの表示、編集追加
#      07/07    decomposeParDict内のsetPreserveを追加
#      07/17    dialogの起動方法を修正
#   16/12/31    runDecomposePar,reconstruct:OF-ext用に修正
#   17/02/13    runMpi:gnome-terminalの起動オプションを「-t」→「--name=」に修正。
#      09/10    reconstruct,runDecomposePar:OFversion毎の処理方法を修正
#   18/06/05    gnome-terminalのオプション「-x」を「--」に変更
#   20/01/06    GTK+3, python3用に大幅修正
#      04/22    多言語化対応
#      12/21    fileOperation:裏で起動するように「&」を追加。
#   21/06/05    initialize:preserveの書式変更に対応する為、keywordsに
#               「"constraints"」を追加。
#      11/5     runMpi:並列計算の並列数とコア数のチェックを追加
#               initialize:regionPropertiesで定義されているregionが
#               systemフォルダ内に存在するかチェック実施（エラー発生する為）
#   22/01/23    setNcpu,setNcpuMethodInDict:v2106対応として、"coeffs"の定義を追加
#      07/17    runMpi:teeをteeCorrect.pyに変更
#               runReconstruct:OFversionチェック方法を修正（OF-10対応）
#      07/19    runDecomposePar,runReconstruct:OFversionチェック方法を修正（OF-10対応）
#   23/01/30    runMpi:シェルスクリプトに「2>&1」を追加
#      05/01    runReconstruct:並列処理を追加
#   24/01/18    runMpi:OF-11対応で修正。
#      02/08    runReconstruct:OF-11用で修正
#      02/14    runMpi:teeCorrect.py→teeに戻す。
#      04/15    runDecomposePar:commandのエラーチェックを追加
#      08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#   25/07/16    runMpi:controlDict内のapplicationが空白の場合の処理追加。
#               application=""でsolver!=""の場合、application="foamRun"を追加
#      07/21    runMpi:applicationチェック修正をpyTreeFoam.getControlDictに移動
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import os, sys
import glob
import shutil
import multiprocessing

import pyTreeFoam
import fileOperationInRegionDialog
import getReconstructParMethodDDialog
import universalDialogs as unvDlg
import GtkParts

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()


#------------------
#  windowApp class
#------------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"runParallelDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を取得
        self.setGtkObject()
        #属性の設定
        self.caseDir = caseDir
        self.nCpu = ""
        self.method = ""
        self.methods = ["simple", "hierarchical", "scotch", "metis", "manual"]
        self.machines = "machines"
        self.regions = []
        self.configDict = pyTreeFoam.readConfigTreeFoam()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ GUIのobject名を取得する"""
        self.entry_nCpu = self.builder.get_object("entry_nCpu")
        self.combo_method = self.builder.get_object("combo_method")
        self.text_method = self.builder.get_object("text_method")
        self.check_preserve = self.builder.get_object("check_preserve")
        self.check_useMachines = self.builder.get_object("check_useMachines")
        self.button_nCpu = self.builder.get_object("button_nCpu")
        self.button_preserve = self.builder.get_object("button_preserve")

    #
    #  main
    #---------
    def main(self):
        self.initialize()
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        Gtk.main_quit()

    #------- event handler ----------------
    #machines編集
    def onEditMachines(self, event):
        self.editMachines()
    #接続状態確認
    def onCheckNodes(self, event):
        self.checkNodes()
    #nCpu, method設定
    def onSetNcpu(self, event):
        self.setNcpu()
    #preserve設定
    def onSetPreserve(self, event):
        self.setPreserve()
    #Dict編集
    def onEditDecomposeParDict(self, event):
        self.editDecomposeParDict()
    #defaultDictに戻す
    def onSetDefaultDict(self, event):
        self.setDefaultDict()
    #mesh分割
    def onRunDecomposePar(self, event):
        self.runDecomposePar()
    #proc内のfile操作
    def onFileOperation(self, event):
        self.fileOperation()
    #並列計算開始
    def onRunMpi(self, event):
        self.runMpi()
    #並列計算停止
    def onHaltMpi(self, event):
        self.haltMpi()
    #結果の再構築
    def onRunReconstruct(self, event):
        self.runReconstruct()
    #method変更
    def onChangeMethod(self, event):
        self.changeMethod()
    #checkPreserveを変更
    def onChangeCheckPreserve(self, event):
        self.changeCheckPreserve()
    #nCpu変更
    def onChangeNcpu(self, event):
        self.changeNcpu()
    #mesh分割方法編集
    def onPressMouseButton(self, widget, event):
        self.pressMouseButton()
    #閉じる
    def onClose(self, event):
        self.close()

    #
    #  editMachines
    #---------------
    def editMachines(self):
        """ machinesを編集"""
        fileName = self.caseDir + "/" + self.machines
        if len(glob.glob(fileName)) == 0:
            foamContents = pyTreeFoam.getFoamContents(self.caseDir)
            cont = foamContents.makeDefaultmachines()
            f = open(fileName, "w"); f.write(cont); f.close()
            msg = _(u"machinesがなかったので、標準の「machines」を作成しました。\n 「localhost cpu=4」で作成しています。")
            okArgs = [self.editMachine_run, fileName]
            self.okDialog(_(u"並列計算"), msg, funcOk=okArgs)
            return

    #  editMachine_run       
    def editMachine_run(self, fileName):
        pyTreeFoam.run(self.caseDir).editor([fileName])

    #
    #  checkNodes
    #-------------
    def checkNodes(self):
        """ 接続状態を確認する"""
        fileName = self.caseDir + "/" + self.machines
        if len(glob.glob(fileName)) == 0:
            title = _(u"nodeの表示")
            message = _(u"現在の解析Case内に「mashines」ファイルがありません。\n「machines」で並列計算させるノードを設定してください。")
            self.errDialog(title, message)
            return
        run = pyTreeFoam.run(self.caseDir)
        comm = "which lamboot"
        (_stat, res, _err) = run.commandReturnCont(comm)
        if res.split() == []:
            title = _("エラー")
            mess = _("LAM runtime がインストールされていません。")
            self.errDialog(title, mess)
            return
        tempFile = self.caseDir + "/nodes.temp"
        comm = "lamboot -v machines; lamnodes | tee " + tempFile + "; lamhalt -d"
        pyTreeFoam.run(self.caseDir).command(comm)
        f = open(tempFile); cont = f.read(); f.close()
        title = _(u"nodeの表示")
        message = _(u"接続されているノードを表示します。\n\nnode\t内容\n-------------------------------------\n") + cont
        self.okDialog(title, message)

    #
    #  setNcpu
    #----------
    def setNcpu(self):
        """ nCpu, methodを設定する。decomposeParDictを作成する。"""
        self.nCpu = self.entry_nCpu.get_text()
        self.method = self.combo_method.get_child().get_text()
        methodCont = self.getDataFromTextView(self.text_method)
        #errorチェック
        #  simple, hierarchicalの分割数とcpu数を確認
        if self.method == "simple" or self.method == "hierarchical":
            (lineCont, _p, _kind) = self.getKeywordLine("n", methodCont)
            (nums, _pp) = pyTreeFoam.strOp(lineCont).getSmallPair(0)
            nSplit = 1
            for num in nums.split():
                nSplit = nSplit * int(num)
            if int(self.nCpu) != nSplit:
                title = _("エラー")
                mess = _("cpu数と分割数が合致していません")
                self.errDialog(title, mess)
                return
        #decomposeParDictを書き換え（region0を含む）
        for region in self.regions:
            fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
            f = open(fileName); cont = f.read(); f.close()
            cont = self.setNcpuMethodInDict(cont, self.nCpu, self.method, methodCont)
            f = open(fileName, "w"); f.write(cont); f.close()
        #ボタンを非アクティブに設定
        self.button_nCpu.set_sensitive(False)
        title = _("decomposeParDictの作成")
        mess = ""
        if len(self.regions) > 1:
            mess += _("region毎に")
        mess += _("decomposeParDictを書き換えました。")
        print(mess)
        self.okDialog(title, mess)

    #
    #  setNcpuMethodInDict
    def setNcpuMethodInDict(self, cont, nCpu, method, methodCoeffs):
        """ nCpu, method, methodContをDictに設定する。"""

        def getInsertPointFromEnd(cont):
            """ 挿入場所を検索する。挿入場所は、footerの上"""
            #footerを検索
            n = cont.find("*********")
            if n < 0:
                #footerが存在しない場合
                ip = len(cont)
                return ip
            #位置「n」から遡って、「\n」を検索
            ip = len(cont)
            i = n
            while i >= 0:
                i -= 1
                if cont[i] == "\n":
                    ip = i
                    break
            return ip

        #nCpuを書き換え
        contOp = pyTreeFoam.strOp(cont)
        newLine = "numberOfSubdomains " + nCpu + ";"
        (cont, _p, _keyword) = contOp.replace1lineKeyword(0, newLine)
        #methodを書き換え
        newLine = "method          " + method + ";"
        (cont, _p, _keyword) = contOp.replace1lineKeyword(0, newLine)
        #methodCoeffsを書き換え
        newLine = ""
        if methodCoeffs != "" or (method != "simple" and method != "hierarchical"):
            lines = methodCoeffs.split("\n")
            for i in range(len(lines)):
                words = lines[i].split()
                nSp = " "
                if len(words) > 1:
                    if len(words[0]) < 16:
                        nSp = " " * (16 - len(words[0])) 
                    lines[i] = "    " + words[0] + nSp + " ".join(words[1:]) + "\n"
                    newLine += lines[i]
                elif len(words) == 1:
                    newLine += "    " + words[0] + "\n"
            allNewLine = method + "Coeffs\n{\n" + newLine + "}"
            (retCont, _p, keyword) = contOp.replace1lineKeyword(0, allNewLine)
            if keyword == "":
                #置き換えできなかった場合、追加する
                p = getInsertPointFromEnd(cont)
                cont = cont[:p] + allNewLine + "\n" + cont[p:]
            else:
                cont = retCont
        return cont

    #
    #  setPreserve
    #--------------
    def setPreserve(self):
        """ preserveの設定"""
        path = os.getenv("TreeFoamPath") + "/python"
        comm = path + "/runParallelSetPreserveDialog.py " + self.caseDir + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  editDecomposeParDict
    #-----------------------
    def editDecomposeParDict(self):
        """ decomposeParDictの編集"""
        if len(self.regions) == 1:
            #singleRegionの場合
            fileName = self.caseDir + "/system/decomposeParDict"
            if len(glob.glob(fileName)) == 0:
                OFversion = self.configDict["OFversion"]
                foamContents = pyTreeFoam.getFoamContents()
                sourceDict = foamContents.defaultDictPathInDataFolder(OFversion, "system/decomposeParDict")
                shutil.copy(sourceDict, self.caseDir + "/system")
                msg = _(u"deconposeParDictが無かったので、標準のdecomposeParDictを作成しました。")
                okArgs = [self.editDecomposeParDict_edit, fileName]
                self.okDialog(_(u"decomposeParDictの編集"), msg, funcOk=okArgs)
                return
            pyTreeFoam.run(self.caseDir).editor([fileName])
        else:
            #multiRegionの場合
            path = os.getenv("TreeFoamPath") + "/python"
            comm = path + "/fileOperationInRegionDialog.py " + self.caseDir + " 2"
            pyTreeFoam.run(self.caseDir).command(comm)

    #  editDecomposeParDict_edit
    def editDecomposeParDict_edit(self, fileName):
        pyTreeFoam.run(self.caseDir).editor([fileName])

    #
    #  setDefaultDict
    #-----------------
    def setDefaultDict(self):
        """ defaultのdecomposeParDictに戻す"""
        OFversion = self.configDict["OFversion"]
        foamContents = pyTreeFoam.getFoamContents()
        sourceDict = foamContents.defaultDictPathInDataFolder(OFversion, "system/decomposeParDict")
        for region in self.regions:
            distDir = self.caseDir + "/system/" + region
            shutil.copy(sourceDict, distDir)
        #textBoxに表示
        self.initialize()
        title = _(u"並列計算")
        mess = _(u"defaultのdecomposeParDictの内容に戻しました")
        print(mess)
        self.okDialog(title, mess)

    #
    #  runDecomposePar
    #------------------
    def runDecomposePar(self):
        """ decomposeParを実行してmeshを分割する"""
        #既に分割ファイルがある場合には削除する
        comm = "rm -rf processor*"
        pyTreeFoam.run(self.caseDir).command(comm)
        #領域分割
        bashrcFOAM = self.configDict["bashrcFOAM"]
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if len(self.regions) == 1:
            #singleRegionの場合
            comm = ". " + bashrcFOAM + "; "
            comm += "decomposePar"
            #pyTreeFoam.run(self.caseDir).commandWithLog(comm)
            mess = _(u"cpu 毎にメッシュを分割しました。")
        else:
            #multiRegionの場合
            #if OFversion <= "2.2.0" or OFversion[:3] == "ext":
            if (0.0 < numVer and numVer <= 2.2) or OFversion[:3] == "ext":
                #OF-2.2以下、ext版
                for region in self.regions[1:]:
                    comm = ". " + bashrcFOAM + "; "
                    comm += "decomposePar -region " + region
                    print(comm)
                    #pyTreeFoam.run(self.caseDir).commandWithLog(comm)
            else:
                #OF-2.3以上、v1610
                comm = ". " + bashrcFOAM + "; "
                comm += "decomposePar -allRegions"
                #pyTreeFoam.run(self.caseDir).commandWithLog(comm)
            mess = _(u"cpu > region 毎にメッシュを分割しました。")
        #command実行
        stat, resCont, errCont = pyTreeFoam.run(self.caseDir).commandReturnCont(comm)
        if stat != "OK":
            title = _("エラー")
            errMess = _(u"エラーが発生しました。\n  logを確認してください。")
            self.errDialog(title, errMess)
            return
        title = _("並列計算")
        self.okDialog(title, mess)
        return

    #
    #  fileOperation
    #----------------
    def fileOperation(self):
        """ processor内のfile操作"""
        path = os.getenv("TreeFoamPath") + "/python"
        comm = path + "/fileOperationInProcessorDialog.py " + self.caseDir + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  runMpi
    #---------
    def runMpi(self):
        """ 並列計算開始"""
        case = pyTreeFoam.case(self.caseDir)
        contDict = case.getControlDict()
        solver = contDict["application"]
        # if solver == "":
        #     if contDict["solver"] != "":
        #         solver = "foamRun"
        #         dataDict = {"application": "foamRun"}
        #         case.setControlDict(dataDict)
        #     elif contDict["regionSolvers"] != "":
        #         solver = "foamMultiRun"
        #         dataDict = {"application": "foamMultiRun"}
        #         case.setControlDict(dataDict)
        nCpu = len(glob.glob(self.caseDir + "/processor*"))
        nCpu_machine = multiprocessing.cpu_count()
        if nCpu > nCpu_machine:
            title = _("並列計算とコア数")
            mess = _("並列計算の並列数がPCのコア数を超えています。") + "\n"
            mess += _("    PCのコア数: ") + str(nCpu_machine) + _(" コア") + "\n"
            mess += _("        並列数: ") + str(nCpu) + _(" 並列") + "\n"
            mess += _("このまま続けますか？")
            okArgs = [self.runMpi_run, solver, nCpu]
            self.okCancelDialog(title, mess, funcOk=okArgs)
            return
        self.runMpi_run(solver, nCpu)

    #  runMpi_run
    def runMpi_run(self, solver, nCpu):
        if solver != "" and nCpu > 1:
            #solve.logのクリア
            f = open(self.caseDir + "/solve.log", "w"); f.close()
            bashrcFOAM = self.configDict["bashrcFOAM"]            
            if self.check_useMachines.get_active() == True:
                #machinesファイルを使う場合
                machines = self.getMachinesData()
                cont = "#!/bin/bash\n"
                cont += ". " + bashrcFOAM + "\n"
                cont += "lamboot -v " + self.machines + "\n"
                comm = "mpirun" + machines + " -np " + str(nCpu) + " " + solver + " -parallel"
                #cont += comm + " | tee solve.log\n"
                #cont += comm + " 2>&1 | teeCorrect.py solve.log\n"
                cont += comm + " 2>&1 | tee solve.log\n"
            else:
                #machinesファイルを使わない場合
                cont = "#!/bin/bash\n"
                cont += ". " + bashrcFOAM + "\n"
                comm = "mpirun -np " + str(nCpu) + " " + solver + " -parallel"
                #cont += comm + " | tee solve.log\n"
                #cont += comm + " 2>&1 | teeCorrect.py solve.log\n"
                cont += comm + " 2>&1 | tee solve.log\n"
            runFile = self.caseDir + "/run"
            f=open(runFile, "w"); f.write(cont); f.close()
            #実行権付与
            pyTreeFoam.run(self.caseDir).command("chmod a+x " + runFile)
            #実行
            print(comm)
            pyTreeFoam.run(self.caseDir).foamTerminal(runFile)
        else:
            title = _(u"エラー")
            if solver == "":
                message = _(u"「controlDict」は、存在しません。\n")
                message += _(u"  case directoryが間違っています。")
            else:
                message = _(u"メッシュが並列計算用に領域分割されていません。") 
            print(message)
            self.errDialog(title, message)

    def getMachinesData(self):
        text = ""
        fileName = self.caseDir + "/" + self.machines
        if len(glob.glob(fileName)) != 0:
            text = " -hostfile " + self.machines
        else:
            mess = _(u"machinesファイルが存在しません。\n")
            mess += _(u"hostのみで並列処理します。")
            print(mess)
        return text

    #
    #  haltMpi
    #----------
    def haltMpi(self):
        """ lam, mpiの中断"""
        title = _("並列計算の停止")
        mess = _("killall mpirun を実行して、停止させます。") + "\n"
        mess += _("  停止させますか？")
        okArgs = [self.haltMpi_halt]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return
        # if stat == "OK":
        #     comm = "killall mpirun"
        #     pyTreeFoam.run(self.caseDir).command(comm)

    #  haltMpi_halt
    def haltMpi_halt(self):
        comm = "killall mpirun"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  runReconstruct
    #-----------------
    def runReconstruct(self):
        OFversion = self.configDict["OFversion"]
        bashrcFOAM = self.configDict["bashrcFOAM"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        #OF-2.2以下、ext版？
        if (0.0 < numVer and numVer <= 2.2) or OFversion[:3] == "ext":
            #multiRegion?
            if len(glob.glob(self.caseDir + "/constant/regionProperties")) != 0:
                (fluids, solids) = pyTreeFoam.case(self.caseDir).getZonesInRegionProperties(OFversion)
                zones = fluids + solids
                for zone in zones:
                    comm = ". " + bashrcFOAM + "; "
                    comm += "reconstructPar -region " + zone + " -zeroTime"
                    print(comm)
                    pyTreeFoam.run(self.caseDir).commandWithLog(comm)
            #singleRegion
            else:
                comm = ". " + bashrcFOAM + "; "
                comm += "reconstructPar"
                print(comm)
                pyTreeFoam.run(self.caseDir).commandWithLog(comm)
            title = _(u"結果の再構築")
            message = _(u"並列計算結果を再構築しました。")
            self.okDialog(title, message)
        #OF-2.3以上、v1606？
        else:
            okArgs = [self.runReconstruct_over23, bashrcFOAM]
            dialog = getReconstructParMethodDDialog.getReconstructParMethod(
                self.caseDir, funcOk=okArgs, parent=self.mainWindow)
            dialog.show()
            return
            
    #  runReconstruct_over23
    def runReconstruct_over23(self, options, bashrcFOAM):
        #comm作成（option除く）
        #multiRegion?
        case = pyTreeFoam.case(self.caseDir)
        (flag, loc, regs) = case.isMultiRegion()
        if flag == True: 
            #multiregion
            comm = ". " + bashrcFOAM + "; "
            comm += "reconstructPar -allRegions -constant"
        else:
            #singleRegion
            comm = ". " + bashrcFOAM + "; "
            comm += "reconstructPar -constant"
        #parallel処理？
        if options[:len("parallel")] != "parallel":
            #通常処理　optionを追加して実行
            comm += options
            print(comm)
            pyTreeFoam.run(self.caseDir).commandWithLog(comm)
        else:
            #並列処理で再構築する
            nProcs = int(options.split("_")[1])
            self.reconstructParallel(comm, nProcs)
        title = _(u"結果の再構築")
        message = _(u"並列計算結果を再構築しました。")
        self.okDialog(title, message)

    #
    #  reconstructParallel
    #----------------------
    def reconstructParallel(self, comm, nProcs):
        """ 計算結果の再構築を並列処理で実行する"""
        #timeFolder名取得
        case = pyTreeFoam.case(self.caseDir)
        timeFolders = case.getTimeFolders("processor0")
        if len(timeFolders) < nProcs:
            nProcs = len(timeFolders)
        #各procが担当するfolder数を取得
        a = len(timeFolders) // nProcs
        num = [a for i in range(nProcs)]
        nRe = len(timeFolders) % nProcs
        for i in range(nRe):
            num[i] += 1
        #-timeオプション作成
        options = []
        ii = 0
        for i in range(nProcs):
            option  = " -time '" + timeFolders[ii] + ":"
            option += timeFolders[ii+num[i]-1] + "'"
            options.append(option)
            ii += num[i]
        #並列処理開始
        queue = []
        procs = []
        for i in range(nProcs):
            q = multiprocessing.Queue()
            queue.append(q)
            procComm = comm + options[i]
            print("[" + str(i) + "] " + procComm)
            p = multiprocessing.Process(
                target=self.reconstructParallel_proc,
                args=(procComm, i, q)
               )
            p.start()
            procs.append(p)
        #処理終了を待つ
        for i in range(nProcs):
            a = queue[i].get()

    #
    #  reconstructParallel_proc
    def reconstructParallel_proc(self, comm, i, q):
        """ 並列処理用のジョブ"""
        if i > 0:
            comm += " >/dev/null"
        pyTreeFoam.run(self.caseDir).commandWithLog(comm)
        q.put(0)

    #
    #  changeMethod
    #---------------
    def changeMethod(self):
        """ methodを変更した時の処理"""
        method = GtkParts.comboBoxText(self.combo_method).getSelectedData()
        if method != self.method:
            self.method = method
            if self.button_nCpu.get_sensitive() == False:
                self.button_nCpu.set_sensitive(True)
        #method内容を読み込み
        fileName = self.caseDir + "/system/decomposeParDict"
        f = open(fileName); cont = f.read(); f.close()
        methodCont = ""
        keyName = self.method + "Coeffs"
        (lineCont, _p, _kind) = self.getKeywordLine(keyName, cont)
        if lineCont != "":
            lineContOp = pyTreeFoam.strOp(lineCont)
            (methodCont, _pp) = lineContOp.getMiddlePair(0)
            methodContOp = pyTreeFoam.strOp(methodCont)
            methodCont = methodContOp.compressSPCR()
            methodCont = methodContOp.deleteCommentsNullLines()
        else:
            keyName = "coeffs"
            (lineCont, _p, _kind) = self.getKeywordLine(keyName, cont)
            if lineCont != "":
                lineContOp = pyTreeFoam.strOp(lineCont)
                (methodCont, _pp) = lineContOp.getMiddlePair(0)
                methodContOp = pyTreeFoam.strOp(methodCont)
                methodCont = methodContOp.compressSPCR()
                methodCont = methodContOp.deleteCommentsNullLines()
        self.setDataToTextView(self.text_method, methodCont)

    #
    #  changeCheckPreserve
    #----------------------
    def changeCheckPreserve(self):
        """ checkPreserveを変更した時の処理"""
        if self.check_preserve.get_active() == True:
            self.button_preserve.set_sensitive(True)
        else:
            self.button_preserve.set_sensitive(False)

    #
    #  changeNcpu
    #-------------
    def changeNcpu(self):
        """ nCpuを変更した時の処理。nCpuボタンが非表示の場合、表示させる。"""
        if self.button_nCpu.get_sensitive() == False:
            self.button_nCpu.set_sensitive(True)

    #
    #  pressMouseButton
    #-------------------
    def pressMouseButton(self):
        """ textViewをクリックした時、設定ボタンをアクティブに設定する"""
        self.changeNcpu()

    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        #decomposeParDictを確認する
        case = pyTreeFoam.case(self.caseDir)
        OFversion = self.configDict["OFversion"]
        (fluids, solids) = case.getZonesInRegionProperties(OFversion)
        (fluids, solids) = self.checkRegionInSystem(fluids, solids)
        regions = ["."] + fluids + solids
        regions.sort()
        self.regions = regions
        flag = 0
        dictName = "system/decomposeParDict"
        sourceFileName = self.caseDir + "/" + dictName
        foamContents = pyTreeFoam.getFoamContents(self.caseDir)
        if len(glob.glob(sourceFileName)) == 0:
            defaultDictDir = foamContents.defaultDictPathInDataFolder(OFversion, dictName)
            shutil.copy(defaultDictDir, self.caseDir + "/system")
            flag = 1
        for region in regions:
            fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
            if len(glob.glob(fileName)) == 0:
                shutil.copy(sourceFileName, fileName)
                flag = 1
        if flag == 1:
            msg = _(u"deconposeParDictが無かったので、標準のdecomposeParDictを作成しました。")
            self.okDialog(_(u"decomposeParDictの編集"), msg)

        #nCpuを読み込み
        f = open(sourceFileName); cont = f.read(); f.close()
        self.nCpu = ""
        (lineCont, _p, _kind) = self.getKeywordLine("numberOfSubdomains", cont)
        if lineCont != "":
            self.nCpu = lineCont.split(";")[0].split()[1]
        self.entry_nCpu.set_text(self.nCpu)
        #methodを読み込み
        self.method = ""
        (lineCont, _p, _kind) = self.getKeywordLine("method", cont)
        if lineCont != "":
            self.method = lineCont.split(";")[0].split()[1]
        self.setDataToComboBox(self.combo_method, self.methods)
        if self.method in self.methods:
            selNo = self.methods.index(self.method)
        else:
            selNo = 0
        self.combo_method.set_active(selNo)
        #methodの内容を読み込み
        methodCont = ""
        keyName = self.method + "Coeffs"
        (lineCont, _p, _kind) = self.getKeywordLine(keyName, cont)
        if lineCont != "":
            lineContOp = pyTreeFoam.strOp(lineCont)
            (methodCont, _pp) = lineContOp.getMiddlePair(0)
            methodContOp = pyTreeFoam.strOp(methodCont)
            methodCont = methodContOp.compressSPCR()
            methodCont = methodContOp.deleteCommentsNullLines()
        else:
            #method+"Coeffs"が存在しない場合は、
            #  "coeffs"を読み込み
            keyName = "coeffs"
            (lineCont, p, kind) = self.getKeywordLine(keyName, cont)
            if lineCont != "":
                lineContOp = pyTreeFoam.strOp(lineCont)
                (methodCont, _pp) = lineContOp.getMiddlePair(0)
                methodContOp = pyTreeFoam.strOp(methodCont)
                methodCont = methodContOp.compressSPCR()
                methodCont = methodContOp.deleteCommentsNullLines()
        self.setDataToTextView(self.text_method, methodCont)
        #Dict作成ボタンを非アクティブ設定
        self.button_nCpu.set_sensitive(False)
        #preserveの設定読み込み
        self.check_preserve.set_active(False)
        self.button_preserve.set_sensitive(False)
        keywords = ["preserveFaceZones", "preservePatches", "preserveBaffles", "constraints"]
        ## region内をチェック
        flag = 0
        for region in regions:
            fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
            f = open(fileName); cont = f.read(); f.close()
            for key in keywords:
                (lineCont, _p, _kind) = self.getKeywordLine(key, cont)
                if lineCont != "":
                    flag = 1
                    break
            if flag == 1:
                self.check_preserve.set_active(True)
                self.button_preserve.set_sensitive(True)

    def checkRegionInSystem(self, fluids, solids):
        """ fluids, solidsのregion名がsystem内に存在するかチェックする。
        存在するregionのみを返す。"""
        sysDir = self.caseDir + "/system"
        newFluids = []
        for fluid in fluids:
            if os.path.exists(sysDir + "/" + fluid) == True:
                newFluids.append(fluid)
        newSolids = []
        for solid in solids:
            if os.path.exists(sysDir + "/" + solid) == True:
                newSolids.append(solid)
        return (newFluids, newSolids) 

    def setDataToTextView(self, textView, cont):
        """ textViewにデータをセットする"""
        GtkParts.textView(textView).setData(cont)

    def getDataFromTextView(self, textView):
        """ textViewからデータを取得する"""
        text = GtkParts.textView(textView).getData()
        return text

    def setDataToComboBox(self, comboBox, items):
        """ comboBoxにデータをセットする"""
        GtkParts.comboBoxText(comboBox).setData(items)

    def getKeywordLine(self, key, cont):
        contOp = pyTreeFoam.strOp(cont)
        lineCont = " "
        p = 0
        while lineCont != "":
            (lineCont, p, kind) = contOp.get1line(p)
            (keyword, _pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
            if keyword == key:
                break
        return (lineCont, p, kind)

    #---- universalDialogs -----------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()
