#!/usr/bin/python3
#  coding: utf-8
#
#           readBoundaryAndFields.py
#
#       boundaryと各fieldを読み込む。
#       gridEditorQtDialog.py内のclassをここに移動。
#
#   20/10/29    新規作成
#      11/23    バグ修正（「import glob」を追加）
#      11/25    remakePatchTypeVariable, remakePatchTypeLevelVariable,
#               remakeValueLevelVariable:OF-v2006対応。
#               changeDictionary実行後、$:xxx.xx → "$:xxx.xx"の様に
#               出力されるので、これに対応。
#   20/05/31    OF-8に対応。「$!output/T」の構文解釈を追加。
#   24/02/07    getAbsPath:getAbsPathをpyTreeFoam.caseに移動
#      07/22    getInternalFieldVariable,remakeValueLevelVariable,
#               remakePatchTypeLevelVariable,remakePatchTypeVariable,
#               getVariableLevelAndCheck,getVariableLevelCont
#               :変数定義「$/」を追加。OF-v2406対応
#      08/19    getAbsPath:バグ修正include解釈時、相対path指定の場合、
#               currDirを「caseDir/timeFolder」に設定して解釈する様に修正。
#
#


import os
import glob
import progressBar
import pyTreeFoam


#--------------
#  load class
#--------------
class load:
    """ boundaryと各fieldを読み込む。
    
    Attrubute:
        caseDir     :caseDir
        timeFolder  :時間folder
        region      :reion名
        fields      :読み込むfield
        nMaxInter   :internalFieldの設定行数(default=5)
        nMaxPatch   :patchContの設定行数(default=10)
        nMaxLines   :数値データを読み込む数(default=20)
        error       :読み込み結果。「""」の場合は、エラー無し
        """

    def __init__(self, caseDir, timeFolder, region, fields,
                    nMaxInter = 5,
                    nMaxPatch = 10,
                    nMaxLines = 20):
        self.caseDir = caseDir
        self.timeFolder = timeFolder
        self.region = region
        self.maxRowsInCellInternal = nMaxInter
        self.maxRowsInCell = nMaxPatch
        self.nMaxLinesBinToAscii = nMaxLines
        self.fields = fields
        self.error = ""

    #
    #  data
    #
    def data(self):
        """ boundary, 各fieldを読み込む。
        
        Results:
            self.boundaryDir    :boundaryの親dir
            self.fieldDir       :各fieldの親dir
            self.fields         :読み取るfield名のlist
            self.patchConts     :boundaryのpatchとpatchType
            self.patchData      :各fieldのpatchとpatch内容
            self.dimData        :各fieldのdimensionの内容
            self.intData        :各fieldのinternalFieldの内容
            self.varData        :各fieldの全体変数の内容
            self.varBndData     :各fieldのboundary変数の内容
        Returns:
            error (str)         :読み込み結果
            """
        #settingDialog表示
        progress = progressBar.dialog()
        progress.showSettingDialog(_("fieldの読み込み"), "reading.....")
        progress.numSettingDialog(0, "setting...")
        #boundaryのdirを取得
        case = pyTreeFoam.case(self.caseDir)
        currTime = self.timeFolder
        relMeshDir = case.getCurrMeshDir(currTime, self.region, "boundary")
        self.boundaryDir = os.path.abspath(self.caseDir + "/" + relMeshDir)
        self.fieldDir = os.path.abspath(self.caseDir + "/" + currTime + "/" + self.region)        
        #boundaryの内容を取得
        patchConts = case.getPatchNameFromBoundary(self.boundaryDir)
        self.patchConts = self.deleteNonuniformListData(patchConts)
        rowNames = list(map(lambda x: x[0], self.patchConts))
        #fieldの内容を取得
        fieldsTop = []
        fieldsPatch = []
        fieldsDimension = []
        fieldsInternal = []
        fieldsVariables = []
        fieldsVariablesBoundary = []
        #settingDialogの設定
        Value = 10.0
        if len(self.fields) > 0:
            dValue = 80.0 / len(self.fields)
            #fieldの読み込み
            for field in self.fields:
                #settingDialogの設定
                stat = progress.numSettingDialog(Value, "reading " + field + " file...")
                if stat == "CANCEL":
                    mess = _("fieldの読み込みをキャンセルしました。")
                    self.error = mess
                    break
                Value += dValue
                #fieldの読み込み
                (topCont, patchesCont, dimData, intData,
                    variables, variablesBoundary) = self.getFieldConts(field)
                fieldsTop.append(topCont)
                fieldsPatch.append(patchesCont)
                fieldsDimension.append(dimData)
                fieldsInternal += intData
                fieldsVariables += variables
                fieldsVariablesBoundary += variablesBoundary
        #settingDialogの設定
        progress.numSettingDialog(90, "ending...")
        #patchデータ整形
        self.patchData = []
        for fi in range(len(fieldsPatch)):
            fieldPatch = fieldsPatch[fi]
            topCont = fieldsTop[fi]
            #patchNamesをfieldから取得
            patches = []
            for patchC in fieldPatch:
                patches.append(patchC[0])
            #fieldのpatch内容を取得
            col = [ "" for i in range(len(patches))]
            for i in range(len(fieldPatch)):
                fpatch = fieldPatch[i]
                if fpatch[0] in patches:
                    col[i] = pyTreeFoam.strOp(fpatch[1]).compressSPCR()
                else:
                    patches.append(fpatch[0])
                    col.append(pyTreeFoam.strOp(fpatch[1]).compressSPCR())
            #boundaryのpatchに対するpatchContに変換
            colPatch = self.getFitBoundaryPatchCont(col, patches, rowNames)
            #inGroupの処理
            colPatch = self.addInGroupsPatchCont(colPatch, fieldPatch,
                                rowNames, topCont)
            #wildCard「".*"」の処理
            colPatch = self.addWildCardCont(col, patches, rowNames, colPatch,
                                fieldPatch, topCont)
            self.patchData.append(colPatch)
        #patchDataの文字数をチェック修正
        maxLines = self.maxRowsInCell
        data = []
        for field in self.patchData:
            i = 0
            while i < len(field):
                patchLines = field[i].split("\n")
                if len(patchLines) > maxLines:
                    field[i] = "\n".join(patchLines[:maxLines]) + "..."
                i += 1
            data.append(field)
        self.patchData = data                       #fieldのpatch
        #patchデータ以外を保存
        self.dimData = fieldsDimension              #dimension
        self.intData = fieldsInternal               #internalField
        self.varData = fieldsVariables              #全体変数
        self.varBndData = fieldsVariablesBoundary   #boundary変数
        #settingDialogを閉じる
        progress.closeSettingDialog()
        #読み込みデータのチェック
        self.checkMisFields()
        return self.error

    def checkMisFields(self):
        """ fieldが読み込めているかどうか確認する。
        空白cellがあるかどうかチェックし、空白cellが存在するfieldを返す。"""
        badFields = []
        for i in range(len(self.patchData)):
            field = self.fields[i]
            patches = self.patchData[i]
            for patchType in patches:
                if patchType == "":
                    badFields.append(field)
                    break
        if len(badFields) > 0:
            #title = _(u"警告")
            mess = _(u"「") + ", ".join(badFields) + _(u"」のfieldが正しく読み込めていません。")
            #self.warningDialog(title, mess)
            self.error = mess

    def addWildCardCont(self, col, patches, rowNames, colPatch, fieldPatch, topCont):
        """ wildCardの内容をチェック、取得"""
        #colPatchの内容をansに転送
        ans = colPatch[:]
        #wildCardのチェック
        p = 0
        while p < len(patches):
            patch = patches[p]
            if patch[0] == '"':
                i = 0
                while i < len(rowNames):
                    #patternマッチングをチェック
                    pattern = patch[1:-1]
                    if pyTreeFoam.isPatternMatch([pattern], rowNames[i]) == True:
                        if colPatch[i] == "":
                            ans[i] = col[p]
                    i += 1
            p += 1
        #ansの内容をcolPatchに転送
        colPatch = ans
        #patchType「$」のチェック
        ans = self.remakePatchTypeVariable(ans, topCont+fieldPatch)
        #patchType「$:」のチェック修正(field頭の変数を検索)
        ans = self.remakePatchTypeLevelVariable(ans, topCont)
        #value「$:」のチェック修正(field頭の変数を検索）
        ans = self.remakeValueLevelVariable(ans, topCont)
        return ans

    def addInGroupsPatchCont(self, col, fieldPatch, rowNames, topCont):
        """ boundary内のinGroupsを追加する"""
        #boundary内の各patchに定義されているpatch内容
        defineData = list(map(lambda x: x[1], self.patchConts))

        def checkRemakePatchCont(cont):
            """ patchContをチェックする。"""
            if cont == "":
                return cont
            if cont[:len("type")] == "type" or cont[:len("$")] == "$":
                return cont
            else:
                #1行目が「type」でない場合、type行を探す
                lines = cont.split("\n")
                lineType = ""
                linesNoType = []
                for line in lines:
                    if line[:len("type")] == "type":
                        lineType = line
                    else:
                        linesNoType.append(line)
                if lineType != "":
                    #type行を1行目に移動して返す
                    newLines = [lineType] + linesNoType
                    newCont = "\n".join(newLines)
                    return newCont
                else:
                    #type行が無い場合、空文を返す
                    return ""

        def isInGroups(defineCont, fieldPatchName):
            """ fieldContがpatchのinGroupsに属するかどうかチェック。"""
            ans = False
            lineCont = " "; p = 0
            defineContOp = pyTreeFoam.strOp(defineCont)
            while lineCont != "":
                (lineCont, p, kind) = defineContOp.get1line(p)
                if kind == "line":
                    if lineCont.split()[0] == "inGroups":
                        (words, _pp) = pyTreeFoam.strOp(lineCont).getSmallPair(0)
                        inGroups = words.split()
                        for inGroup in inGroups:
                            if inGroup == fieldPatchName:
                                ans = True
                                break
                if ans == True:
                    break
            return ans

        #patchContが正しい内容かチェック。正しくない場合、「""」をセット
        #チェック結果をansに転送
        ans = []
        for fpCont in col:
            fpCont = checkRemakePatchCont(fpCont)
            ans.append(fpCont)
        #field側のpatchを順番に確認
        for p in range(len(col)):
            fpCont = col[p]
            #field側のpatchを順番に解釈し、該当はansに保存
            for fieldCont in fieldPatch:
                fieldPatchName = fieldCont[0]
                #field側のpatch名がinGroupsに属するかチェック
                if isInGroups(defineData[p], fieldPatchName) == True:
                    #属する場合、groupの内容を取得
                    groupCont = pyTreeFoam.strOp(fieldCont[1]).compressSPCR()
                    groupCont = checkRemakePatchCont(groupCont)
                    #patch内容が定義されていない(「""」)場合、groupの内容をセット
                    if groupCont != "" and fpCont == "":
                        ans[p] = groupCont
        #ansを元に戻す
        col = ans
        #patchType「$」のチェック修正
        col = self.remakePatchTypeVariable(col, fieldPatch)
        #patchType「$:」のチェック修正（field頭の変数のみ検索）
        col = self.remakePatchTypeLevelVariable(col, topCont)
        #value「$:」のチェック修正（field頭の変数のみ検索）
        col = self.remakeValueLevelVariable(col, topCont)
        #value「$」は、修正しない。（そのまま）
        return col

    def remakeValueLevelVariable(self, col, topCont):
        """ value「$:」のチェック修正(field頭の変数のみ検索）"""
        end = False; nLoop = 0
        while end == False and nLoop < 10:
            end = True
            i = 0
            for fpCont in col:
                lines = fpCont.split("\n")
                l = 0
                for line in lines:
                    n = line[1:].find("$:")
                    if n >= 0:
                        wsep = "."
                    else:
                        n = line[1:].find("$!")
                        if n >= 0:
                            wsep = "/"
                        else:
                            n = line[1:].find("$/")
                            if n >= 0:
                                wsep = "/"
                    if n >= 0:
                        ps = n + 1
                        p = ps
                        chara = line[p]
                        while (chara != " " and chara != "\n" and chara != "\t"
                                and chara != ";" and p < len(line)):
                            p += 1
                            chara = line[p]
                        lcont = line[ps+2:p].replace(" ", "")
                        lcont = lcont.replace('"', '')
                        #keyList = line[ps+2:p].split(".")
                        #keyList = lcont.split(".")
                        keyList = lcont.split(wsep)
                        keyCont = self.getKeyContFromVariableValue(keyList, col, topCont)
                        if keyCont != "":
                            newLine = line[:ps] + keyCont + line[p:]
                            newLines = lines[:l] + [newLine] + lines[l+1:]
                            col[i] = "\n".join(newLines)
                            end = False
                            break
                    l += 1
                i += 1
            nLoop += 1
        return col    

    def getKeyContFromVariableValue(self, keyList, col, fieldPatch):
        """ 「$:」Value変数の内容を取得"""
        ans = ""
        if len(keyList) == 1:
            #「$:xx」の場合
            for fpCont in fieldPatch:
                if fpCont[0] == keyList[0]:
                    ans = fpCont[1].split(";")[0]
                    break
        elif len(keyList) == 2:
            #「$:xx.xx」の場合
            for fpCont in fieldPatch:
                if fpCont[0] == keyList[0]:
                    fpContOp = pyTreeFoam.strOp(fpCont[1])
                    (lineCont, p, kind) = fpContOp.get1line(0)
                    while lineCont != "" and p < len(fpCont[1]):
                        if kind == "line":
                            lineContOp = pyTreeFoam.strOp(lineCont)
                            (keyword, pp) = lineContOp.getKeyword(0)
                            if keyword == keyList[1]:
                                (lcont, pp, dummy) = lineContOp.get1line(pp)
                                ans = lcont[:-1]    #「;」を削除
                                break
                        (lineCont, p, kind) = fpContOp.get1line(p)
        elif len(keyList) >= 3:
            #「$:xx.xx.xx」以上の場合
            for fpCont in fieldPatch:
                if fpCont[0] == keyList[0]:
                    contents = fpCont[1]
                    contentsOp = pyTreeFoam.strOp(contents)
                    keyCont = ""
                    for key in keyList[1:-1]:
                        (keyCont, p) = contentsOp.getKeywordContents(key, 0)
                        if keyCont == "":
                            break
                        contents = keyCont
                    if keyCont != "":
                        keyContOp = pyTreeFoam.strOp(keyCont)
                        (lineCont, p, kind) = keyContOp.get1line(0)
                        while lineCont != "":
                            if kind == "line":
                                (keyword, pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
                                if keyword == keyList[-1]:
                                    ans = " ".join(lineCont.split()[1:])[:-1]
                                    break
                            (lineCont, p, kind) = keyContOp.get1line(p)
                        if ans != "":
                            break
        return ans

    def remakePatchTypeLevelVariable(self, col, topCont):
        """ patchType「$:」「$!」「$/」のチェック修正（field頭の変数のみ検索）"""
        end = False; nLoop = 0
        while end == False and nLoop < 10:
            i = 0
            end = True
            for fpCont in col:
                if len(fpCont) > 4:     #「type」の4文字以上
                    if fpCont[:2] == "$:" or fpCont[:2] == "$!" or fpCont[:2] == "$/":
                        #separaterの文字を設定
                        if fpCont[:2] == "$:":
                            wsep = "."
                        else:
                            wsep = "/"
                        cont = fpCont[2:].split(";")[0]
                        cont = cont.replace(" ", "")    #空白を削除
                        keyList = cont.split(wsep)
                        keyCont = self.getKeyContFromVariable(keyList, col, topCont)
                        if keyCont != "":
                            col[i] = pyTreeFoam.strOp(keyCont).compressSPCR()
                            end = False
                    elif fpCont[:3] == '"$:' or fpCont[:3] == '"$!' or fpCont[:3] == '"$/':
                        #「"$:xxx"」の様に「"」で囲まれている場合
                        #  separaterの文字を設定
                        if fpCont[:3] == '"$:':
                            wsep = "."
                        else:
                            wsep = "/"
                        fpCont = fpCont.split('"')[1]   #
                        cont = fpCont[2:].split(";")[0]
                        cont = cont.replace(" ", "")    #空白を削除
                        keyList = cont.split(wsep)
                        keyCont = self.getKeyContFromVariable(keyList, col, topCont)
                        if keyCont != "":
                            col[i] = pyTreeFoam.strOp(keyCont).compressSPCR()
                            end = False
                i += 1
            nLoop += 1
        return col

    def getKeyContFromVariable(self, keyList, col, fieldPatch):
        """ 「$:」Type変数の内容を取得"""
        ans = ""
        if len(keyList) == 1:
            #「$:xx」の場合
            for fpCont in fieldPatch:
                if fpCont[0] == keyList[0]:
                    ans = fpCont[1]
        if len(keyList) == 2:
            #「$:xx.xx」の場合
            for fpCont in fieldPatch:
                if fpCont[0] == keyList[0]:
                    fpContOp = pyTreeFoam.strOp(fpCont[1])
                    (lineCont, p, kind) = fpContOp.get1line(0)
                    while lineCont != "" and p<len(fpCont[1]):
                        if kind == "keyword":
                            lineContOp = pyTreeFoam.strOp(lineCont)
                            (keyword, _pp) = lineContOp.getKeyword(0)
                            if keyword == keyList[1]:
                                (lastCont, _pp) = lineContOp.getMiddlePair(0)
                                #line文のみ取得
                                ans = lastCont
                                break
                        (lineCont, p, kind) = fpContOp.get1line(p)
        if len(keyList) >= 3:
            #「$:xx.xx.xx」以上の場合
            for fpCont in fieldPatch:
                if fpCont[0] == keyList[0]:
                    contents = fpCont[1]
                    contentsOp = pyTreeFoam.strOp(contents)
                    #2番目以降はloopして確認
                    for key in keyList[1:]:
                        (keyCont, p) = contentsOp.getKeywordContents(key, 0)
                        if keyCont == "":
                            break
                        contents = keyCont
                    if keyCont != "":
                        #line文のみ取得
                        ans = keyCont
                        break
        return ans

    def remakePatchTypeVariable(self, col, fieldPatch):
        """ patchType「$」のチェック修正（field頭の変数のみ検索）
        変数を解釈して、変数の内容(patchType)を取得"""

        def getKeyContFromSingleVariable(key, col, fieldPatch):
            """ 「$」type変数の内容を取得"""
            ans = ""
            for fpCont in fieldPatch:
                if fpCont[0] == key:
                    ans = fpCont[1]
                    break
            return ans

        end = False; nLoop = 0
        while end == False and nLoop < 10:
            i = 0
            end = True
            for fpCont in col:
                if len(fpCont) > 4:     #「type」の4文字以上
                    #「$」変数か？
                    if (fpCont[:2] != "$:" and fpCont[:2] != "$!" and
                        fpCont[:2] != "$/" and fpCont[0] == "$"):
                        key = fpCont[1:].split(";")[0]
                        keyCont = getKeyContFromSingleVariable(key, col, fieldPatch)
                        if keyCont != "":
                            col[i] = pyTreeFoam.strOp(keyCont).compressSPCR()
                            end = False
                    #「"$xxx"」変数か？「"」で囲まれている場合
                    elif (fpCont[:3] != '"$:' and fpCont[:3] != '"$!' and 
                          fpCont[:3] != '"$/' and fpCont[:2] == '"$'):
                        fpCont = fpCont.split('"')[1]
                        key = fpCont[1:].split(";")[0]
                        keyCont = getKeyContFromSingleVariable(key, col, fieldPatch)
                        if keyCont != "":
                            col[i] = pyTreeFoam.strOp(keyCont).compressSPCR()
                            end = False
                i += 1
        return col

    def getFitBoundaryPatchCont(self, col, patches, rowNames):
        """ boundaryのpatchに対するpatchContに変換する。
        boundaryFieldに「"patch_to_.*"」が存在する場合がある為。
        """
        ans = [ "" for i in range(len(rowNames))]
        p = 0
        while p < len(patches):
            patch = patches[p]
            if patch in rowNames:
                id = rowNames.index(patch)
                ans[id] = col[p]
            p += 1
        return ans

    def getFieldConts(self, field):
        """ fieldの内容を取得する。"""
        dimData = []
        intData = []
        varNames = []
        varBndNames = []
        fileName = self.fieldDir + "/" + field
        nMaxLines = self.nMaxLinesBinToAscii
        contents = pyTreeFoam.foamFile(nMaxLines).read(fileName)
        contentsOp = pyTreeFoam.strOp(contents)
        #dimensionデータを取得
        dimData = self.getDimensionData(contentsOp)
        #field頭のinclude行、keyword行を取得
        topCont = []
        p = contentsOp.skipFoamFile()
        (lineCont, p, kind) = contentsOp.get1line(p)
        (keyword, _pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
        while lineCont != "" and keyword != "boundaryField":
            if kind == "include":
                #includeFileの読み込み
                fileName = (lineCont.split()[1]).split('"')[1]
                fileName = self.getAbsPath(fileName)     #環境変数を変換
                (includeHeadCont, varCont) = self.readIncludeGetPatchCont(fileName)
                topCont += includeHeadCont
                varNames += varCont
            elif kind == "includeEtc":
                #includeFileの読み込み
                fileName ="${WM_PROJECT_DIR}/etc/" + (lineCont.split()[1]).split('"')[1]
                fileName = self.getAbsPath(fileName)
                (includeHeadCont, varCont) = self.readIncludeGetPatchCont(fileName)
                topCont += includeHeadCont
                varNames += varCont
            elif kind == "keyword":
                #keyword行を取得
                (patchCont, _pp) = pyTreeFoam.strOp(lineCont).getNextKeywordAndContents(0)
                topCont.append(patchCont)
            elif kind == "line" and keyword == "internalField":
                a = self.refoamInternalFieldCellData(lineCont)
                intData.append(a)
            elif (kind == "line" and keyword != "dimensions" and keyword != "internalField"):
                #line行を取得
                keyCont = " ".join(lineCont.split()[1:])
                topCont.append([keyword, keyCont])
                varNames.append(lineCont+"\n")
            elif kind == "other":
                varNames.append(lineCont+"\n")
            (lineCont, p, kind) = contentsOp.get1line(p)
            (keyword, _pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
        #boundaryFieldの内容を取得
        patchesCont = []
        (boundary, _pp) = pyTreeFoam.strOp(lineCont).getMiddlePair(0)
        #patch内容を取得
        boundaryOp = pyTreeFoam.strOp(boundary)
        (lineCont, p, kind) = boundaryOp.get1line(0)
        while lineCont != "":
            if kind == "keyword":
                (patchCont, _pp) = pyTreeFoam.strOp(lineCont).getNextKeywordAndContents(0)
                patchesCont.append(patchCont)
            elif kind == "include":
                #includeFileの読み込み
                fileName = (lineCont.split()[1]).split('"')[1]
                fileName = self.getAbsPath(fileName)
                (includePatchesCont, varCont) = self.readIncludeGetPatchCont(fileName)
                patchesCont += includePatchesCont
                varBndNames += varCont
            elif kind == "includeEtc":
                #includefileの読み込み
                fileName = "${WM_PROJECT_DIR}/etc/" + (lineCont.split()[1]).split('"')[1]
                fileName = self.getAbsPath(fileName)
                (includePatchesCont, varCont) = self.readIncludeGetPatchCont(fileName)
                patchesCont += includePatchesCont
                varBndNames += varCont
            elif kind == "line" or "other":
                varBndNames.append(lineCont+"\n")
            #次のpatchを取得
            (lineCont, p, kind) = boundaryOp.get1line(p)
        #internalFieldのvalue「$:」「$!」「$/」のチェック修正
        intData = self.getInternalFieldVariable(intData, field, topCont)

        #変数を取得
        variables = []
        variablesBoundary = []
        #  変数名を取得
        nameCont = ""
        if len(varNames) > 0:
            #変数名を整形する
            nameCont = self.reformNameCont(varNames)
        #  「$:」をチェック修正
        newNameCont = self.getVariableLevelAndCheck(nameCont, topCont)
        #  変数名を保存
        variables.append(newNameCont)

        #boundaryの変数をセット
        nameCont = ""
        if len(varBndNames) > 0:
            #変数名を整形
            nameCont = self.reformNameCont(varBndNames)
        #  「$:」をチェック修正
        newNameCont = self.getVariableLevelAndCheck(nameCont, topCont)
        #  変数名を保存
        variablesBoundary.append(newNameCont)
        #取得したデータを戻す
        return (topCont, patchesCont, dimData, intData, variables, variablesBoundary)

    def getVariableLevelAndCheck(self, nameCont, topCont):
        """ 「$:」が無くなるまで取得する。"""
        end = False; nloop = 0
        while end == False and nloop < 10:
            nameCont = self.getVariableLevelCont(nameCont, topCont)
            n = nameCont.find("$:")
            if n < 0:
                n = nameCont.find("$!")
                if n < 0:
                    n = nameCont.find("$/")
            if n < 0:
                end = True
            nloop += 1
        return nameCont

    def getVariableLevelCont(self, variCont, topCont):
        """ 「$:」を取得する。"""
        if variCont == " ":
            return variCont

        #変数のlistを取得
        variContOp = pyTreeFoam.strOp(variCont)
        variables = []
        lineCont = " "; p = 0
        while lineCont != "":
            (lineCont, p, kind) = variContOp.get1line(p)
            if kind == "line":
                variables.append(lineCont)

        ans = []
        for vari in variables:
            words = vari.split()
            if len(words) > 1:
                variName = words[1]
                if variName[:2] == "$:" or variName[:2] == "$!" or variName[:2] == "$/":
                    if variName[-1] == ";":
                        variName = variName[:-1]
                    variCont = self.getVariableLevelContFromContents(variName, topCont, variCont)
                    ans.append(words[0] + " " + variCont + ";")
                else:
                    ans.append(vari)
            else:
                ans.append(vari)
        return "\n".join(ans)

    def getVariableLevelContFromContents(self, variName, topCont, variCont):
        """ 「$:」を取得"""

        def getKeyCont(key, contents):
            """ keyの内容をcontentsから取得"""
            ans = ""
            lineCont = " "; keyword = " "; p = 0
            contentsOp = pyTreeFoam.strOp(contents)
            while lineCont != "" and keyword != "boundaryField":
                (lineCont, p, kind) = contentsOp.get1line(p)
                lineContOp = pyTreeFoam.strOp(lineCont)
                (keyword, _pp) = lineContOp.getKeyword(0)
                if keyword == key and kind == "keyword":
                    (ans, _pp) = lineContOp.getMiddlePair(0)
                    break
            return ans

        def getKeyLastCont(key, contents):
            """ 最終のkeyの内容をcontentsから取得"""
            ans = ""
            lineCont = " "; keyword = " "; p = 0
            contentsOp = pyTreeFoam.strOp(contents)
            while lineCont != "" and keyword != "boundaryField":
                (lineCont, p, kind) = contentsOp.get1line(p)
                (keyword, _pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
                if keyword == key and kind == "line":
                    keyLine = lineCont.split(";")[0]
                    ans = " ".join(keyLine.split()[1:])
                    break
            return ans

        #処理----------------------
        ans = ""
        keyList = variName[2:].replace(" ", "").split(".")
        if len(keyList) == 1:
            contents = variCont
            ans = getKeyLastCont(keyList[0], contents)
            if ans == "":
                ans = variName
            return ans

        #keyListを順番に解釈
        contents = topCont
        for key in keyList[:-1]:
            keyCont = getKeyCont(key, contents)
            if keyCont == "":
                break
            contents = keyCont
        #最後のkeyを解釈
        if keyCont == "":
            ans = variName
        else:
            contents = keyCont
            ans = getKeyLastCont(keyList[-1], contents)
            if ans == "":
                ans = variName
        return ans

    def reformNameCont(self, varNames):
        """ 変数名を整形する。"""
        varList = []
        for var in varNames:
            nameList = var.split()
            if nameList[0] != "#":
                flag = 0
                i = 0
                for newVar in varList:
                    if newVar.split()[0] == nameList[0]:
                        flag = 1
                        break
                    i += 1
                if flag == 1:
                    varList[i] = " ".join(nameList)
                else:
                    varList.append(" ".join(nameList))
        ans = "\n".join(varList)
        return ans

    def getInternalFieldVariable(self, intData, field, topCont):
        """ internalFieldのvalue「$:」「$!」「$/」のチェック修正"""
        i = 0
        for val in intData:
            p = val.find("$:")
            if p >= 0:
                wsep = "."
            else:
                p = val.find("$!")
                if p >= 0:
                    wsep = "/"
                else:
                    p = val.find("$/")
                    if p >= 0:
                        wsep = "/"
            if p >= 0:
                ps = p
                chara = val[p]
                while (chara != " " and chara != "\n" and chara != "\t"
                    and chara != ";" and p < len(val)):
                    p += 1
                    chara = val[p]
                #keyList = val[ps+2:p].split(".")
                #keyList = val[ps+2:p].replace(" ", "").split(".")
                keyList = val[ps+2:p].replace(" ", "").split(wsep)
                keyCont = self.getKeyContFromVariableValue(keyList, field, topCont)
                if keyCont != "":
                    newVal = val[:ps] + keyCont + val[p:]
                    intData[i] = newVal
                    break
            i += 1
        return intData

    def refoamInternalFieldCellData(self, contents):
        """ internalFieldデータを整形する。
        「internalField」文字を削除して、
        maxRowsInCellInternalの値（5行）の行数に縮める"""
        contentsOp = pyTreeFoam.strOp(contents)
        (keyword, pp) = contentsOp.getKeyword(0)
        if keyword == "internalField":
            (keyword , pp) = contentsOp.getKeyword(pp)
        #pointerをkeywordの頭に戻す        
        pp = pp - len(keyword)
        #p = pyTreeFoam.strOp(contents[pp:]).skipSP(0)   #空白をskip
        #表示内容を取得
        cont = contents[pp:]
        if keyword == "uniform":
            a = cont
        else:
            lines = contents[pp:].split("\n")
            if len(lines) > self.maxRowsInCellInternal:
                a = "\n".join(lines[:self.maxRowsInCellInternal]) + "..."
            else:
                a = "\n".join(lines)
        return a

    def reformBoundaryFieldCellData(self, newCont):
        """ 空白を圧縮する。"""
        patchCont = newCont.split("\n")
        if len(patchCont) > self.maxRowsInCell:
            conts = patchCont[:self.maxRowsInCell]
            conts[-1] += "..."
        else:
            conts = patchCont
        for i in range(len(conts)):
            words = conts[i].split()
            contOp = pyTreeFoam.strOp(conts[i])
            (_keyword, p) = contOp.getKeyword(0)
            conts[i] = conts[i][:p] + " ".join(words[1:])
        newCont = "\n".join(conts)
        return newCont

    def getDimensionData(self, contentsOp):
        """ dimensionデータを取得する"""
        (foamCont, pp) = contentsOp.getKeywordContents("FoamFile", 0)
        foamContOp = pyTreeFoam.strOp(foamCont)
        value = ""
        (lineCont, p, kind) = foamContOp.get1line(0)
        while lineCont != "":
            words = lineCont.split()
            if words[0] == "class":
                value = words[1]
                break
            (lineCont, p, kind) = foamContOp.get1line(p)
        dimValue = ""
        (lineCont, p, kind) = contentsOp.get1line(pp)
        while lineCont != "":
            if kind == "line":
                lineContOp = pyTreeFoam.strOp(lineCont)
                (keyword, pp) = lineContOp.getKeyword(0)
                if keyword == "dimensions":
                    dimValue = " ".join(lineCont.split()[1:])
                    break
            (lineCont, p, kind) = contentsOp.get1line(p)
        dimData = value + "\n" + dimValue
        return dimData

    def readIncludeGetPatchCont(self, fileName):
        """ includeFileを読み込み、patchContを取得"""
        cont = []; varCont = []
        (cont, varCont) = self.readIncludeSub(fileName, cont, varCont)
        return (cont, varCont)

    def readIncludeSub(self, fileName, cont, varCont):
        if len(glob.glob(fileName)) == 0:
            print(_(u"\nerror: includeファイル「") + fileName + _(u"」が存在しません。"))
            return (cont, varCont)
        f = open(fileName); fileCont = f.read(); f.close()
        fileContOp = pyTreeFoam.strOp(fileCont)
        fileCont = fileContOp.deleteCommentsNullLines()
        patchesCont = cont
        p = fileContOp.skipFoamFile()
        (lineCont, p, kind) = fileContOp.get1line(p)
        while lineCont != "":
            if kind == "keyword":
                (patchCont, _pp) = pyTreeFoam.strOp(lineCont).getNextKeywordAndContents(0)
                patchesCont.append(patchCont)
            elif kind == "include":
                fileName = (lineCont.split()[1]).split('"')[1]
                fileName = self.getAbsPath(fileName)
                (patchCont, var) = self.readIncludeSub(fileName, patchesCont, varCont)
                patchesCont += patchCont
                varCont += var
            elif kind == "includeEtc":
                fileName = "${WM_PROJECT_DIR}/etc/" + (lineCont.split()[1]).split('"')[1]
                fileName = self.getAbsPath(fileName)
                (patchCont, var) = self.readIncludeSub(fileName, patchesCont, varCont)
                patchesCont += patchCont
                varCont += var
            else:
                varCont += [lineCont + "\n"]
            (lineCont, p, kind) = fileContOp.get1line(p)
        return [patchesCont, varCont]

    def getAbsPath(self, cont):
        """ dirのpathを取得。環境変数を変換する。
        相対pathの場合、currDirを「caseDir/timeFolder」として変換する。"""
        case = pyTreeFoam.case(self.caseDir)
        if cont[0] == "$" or cont[0] == "/" or cont[0] == "<":
            #bashの環境変数を解釈しpathを設定
            cont = case.getAbsPath(self.caseDir, cont)
        else:
            #相対pathの場合
            currDir  = self.caseDir + "/" + self.timeFolder
            cont = currDir + "/" + cont
        return cont


    def deleteNonuniformListData(self, patchConts):
        """ boundary中のpatch内容からnonuniformデータを削除する。"""
        for i in range(len(patchConts)):
            (_patch, patchCont, _nFaces, _startFace) = patchConts[i]
            lines = patchCont.split(";")
            define = ""
            for line in lines:
                words = line.split()
                if len(words) >= 3:
                    if not (words[1] == "nonuniform" and words[2][:len("List") == "List"]):
                        define += line + ";\n"
                elif len(line) != 0 and line != "\n":
                    define += line + ";\n"
            #データを整形
            lines = define.split("\n")
            newCont = ""
            for line in lines:
                if line != "":
                    newCont += line + "\n"
            patchConts[i][1] = newCont
        return patchConts

