#!/usr/bin/python3
# coding: utf-8
#
#  openGridEditorDialog.py
#
#     Fieldsの選択
#                   folder1                     folder2
#       入力データ:[[folder1,field1,field2,..], [folder2,field1,field2,..], ...]
#
#       12/03/25    新規作成（selectFieldsForEditDialogを変更して作成）
#          05/28    ﾊﾞｸﾞ修正（openコマンド時、設定されたfield順番が反映されなかった）
#          08/25    ﾊﾞｸﾞ修正（fieldﾘｽﾄ読み込み時、隠しﾌｧｲﾙも読み込んでいた）
#          09/20    gridEditor起動方法変更
#          10/29    mainでのdialog表示に変更
#       13/01/11    multiRegionで計算結果が入っているtimeで開く時ｴﾗｰ発生を修正
#          04/17    polyMeshの場所をfieldDirから探すように修正（第1優先）
#          04/19    fieldが存在しない場合でも起動するように修正
#          04/27    複数のregionのgridEditorを表示させる事ができる様に修正。
#          05/01    複数起動時にnRnningGridが更新されるまで待つ様に修正
#          05/10    getBoundaryDir:boundaryの場所検索方法変更
#          06/08    連続してopenする時、稀にopenできない時があるので更新後待ち追加
#          07/11    boundaryDirの検索方法を変更（latest、start、firstTime、constantの順）
#          08/13    comboBox(編集する時間folder)のセット方法を変更
#          09/07    fieldの圧縮ファイル（gzip）の読込に対応
#          09/08    time選択方法を修正
#          10/12    多言語化の為修正
#          12/21    stdout、stderrの設定（import logFileCreater）を追加
#       14/09/16    showGrid:openFOAMの環境でgridEditorを起動する様に変更
#          10/25    changeTimeFolder:listBoxの選択項目を保存する様に修正
#       15/03/08    readColOrderを追加（表示設定fileの順番で表示されない為）
#          07/13    TreeFoamPathの取得方法を修正
#          09/27    getSelectionTime:first,start,latestTimeの取得方法を修正（バグ修正）
#       19/09/12    dialog中に「other case」ボタンを追加
#          12/03    GTK+3, python3用に大幅修正
#       20/04/22    多言語化対応
#          08/02    setComboTimeData:timeFolder取得をfloatに変換して取得する様に修正
#                   「0.0」と「0」folderを区別しないため。
#          08/22    起動時、window位置をmouseに合わせる様修正。
#       25/07/04    setTreeFolderData:multiRegionの場合、fluidRegions,solidRegionsの
#                   folder名を非表示設定を追加。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GdkPixbuf

import os
import sys
import glob

import pyTreeFoam
import getOpenFolderDDialog
import GtkParts

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

#-------------
#  windowApp
#-------------
class windowApp:

    def __init__(self, nTreeFoam, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        gladeFile = os.getenv("TreeFoamPath") + "/glade/openGridEditorDialog.glade"
        self.builder.add_from_file(gladeFile)
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.nTreeFoam = nTreeFoam
        self.caseDir = caseDir
        self.controlDict = {}
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        self.maskEvent = True
        #iconの定義
        iconDir = os.getenv("TreeFoamPath") + "/icons"
        iconFile = GdkPixbuf.Pixbuf.new_from_file
        self.iconFolder = iconFile(iconDir+"/folder16.png")
        self.iconFolderLink = iconFile(iconDir+"/folderLink16.png")
        self.iconFile = iconFile(iconDir+"/text8.png")
        self.iconFileLink = iconFile(iconDir+"/textLink8.png")
        #GUIのobject名を取得
        self.setGtkObject()
        #初期化
        self.initialize()
        #GUIの初期化
        self.iniGui()

    #
    #  main
    #-------
    def main(self):
        """ GUIを表示"""
        self.maskEvent = False
        self.mainWindow.show()
        Gtk.main()

    #
    #  close
    #---------
    def close(self, *args):
        """ GUIを閉じる"""
        Gtk.main_quit()

    def setGtkObject(self):
        """ glade内のobject名を取得する"""
        self.entry_caseDir = self.builder.get_object("entry_caseDir")
        self.entry_time = self.builder.get_object("entry_time")
        self.combo_time = self.builder.get_object("combo_time")
        self.check_allFields = self.builder.get_object("check_allFields")
        self.tree_folder = self.builder.get_object("tree_folder")
        self.tree_field = self.builder.get_object("tree_field")
        self.imgIcon = self.builder.get_object("imgIcon")
        self.label_titleCont = self.builder.get_object("label_titleCont")

    #------ event handler -------------
    #参照
    def onReference(self, event):
        self.reference()
    #開く
    def onOpen(self, event):
        self.openGridEditor()
    #キャンセル
    def onCancel(self, event):
        self.close()
    #時間を変更（comboBox）
    def onChangeTime(self, event):
        if self.maskEvent == False:
            self.changeTime()
    #folder（region）選択
    def onSelectFolder(self, event):
        self.selectFolder()
    #check変更時
    def onChangeCheckBox(self, event):
        self.changeCheckBox()
    #-----------------------------------


    #
    #  iniGui
    #----------
    def iniGui(self):
        """ GUIの初期化"""
        self.mainWindow.set_title(_("gridEditor開く"))
        self.label_titleCont.set_text(_("選択した boundaryField を gridEditor で開く"))
        fileName = os.getenv("TreeFoamPath") + "/icons/gridEditor.png"
        self.imgIcon.set_from_file(fileName)

    #
    #  initialize
    #--------------
    def initialize(self):
        """ 初期化する"""
        #caseDirをentryに表示
        self.entry_caseDir.set_text(self.caseDir)
        #初期値を取得
        case = pyTreeFoam.case(self.caseDir)
        self.timeFolders = case.getTimeFolders()
        self.controlDict = case.getControlDict()
        #comboにデータをセット
        self.setComboTimeData()
        #treeListを初期化
        self.iniTreeList()          #2回実行すると、list内容が壊れる
        #tree_folderにデータセット
        self.setTreeFolderData()
        #tree_fieldにデータセット
        self.setTreeFieldData()

    #
    #  iniTreeList
    def iniTreeList(self):
        """ TreeListを初期化"""
        #tree_folder側
        GtkParts.treeListIcon(self.tree_folder).create(multi=False)
        #tree_field側
        GtkParts.treeListIcon(self.tree_field).create()

    #
    #  setTreeFolderData
    def setTreeFolderData(self):
        """ tree_folder(region)にデータをセット"""
        #itemsを作成
        timeFolder = self.getTimeFolderFromCombo()
        fieldDir = self.caseDir + "/" + timeFolder
        if len(glob.glob(fieldDir)) == 0:
            folderNames = []
        else:
            folderNames = pyTreeFoam.getFolderNames(fieldDir)
            flag, loc, regNames =  pyTreeFoam.case(self.caseDir).isMultiRegion()
            if flag == True:
                newNames = []
                for name in folderNames:
                    if name == "fluidRegions" or name == "solidRegions":
                        pass
                    else:
                        newNames.append(name)
                folderNames = newNames
        folderNames.sort()
        #itemsを作成
        items = []
        icon = self.iconFolder
        if os.path.islink(fieldDir):
            icon = self.iconFolderLink
        items.append([icon, _("(現在のfolder)")])
        for name in folderNames:
            nameDir = fieldDir + "/" + name
            icon = self.iconFolder
            if os.path.islink(nameDir):
                icon = self.iconFolderLink
            item = [icon, name]
            items.append(item)
        #値をセット
        treeList = GtkParts.treeListIcon(self.tree_folder)
        treeList.setItems(items)
        treeList.selectItems([items[0]])
    
    #
    #  setTreeFieldData
    def setTreeFieldData(self):
        """ tree_field(field)にデータセット"""
        #folderを取得
        selFolder = self.getFolderFromTreeFolder()
        #itemsを作成
        timeFolder = self.getTimeFolderFromCombo()
        fields = pyTreeFoam.case(self.caseDir).getFieldNames(timeFolder, selFolder)
        fields.sort()
        items = []
        fieldDir = self.caseDir + "/" + timeFolder + "/" + selFolder
        for field in fields:
            icon = self.iconFile
            if os.path.islink(fieldDir + "/" + field):
                icon = self.iconFileLink
            item = [icon, field]
            items.append(item)
        #itemを表示
        treeList = GtkParts.treeListIcon(self.tree_field)
        treeList.setItems(items)
        #displayFieldファイルを確認
        fileName = self.caseDir + "/" + self.timeFolders[0] + "/" + selFolder + "/.displayField"
        if len(glob.glob(fileName)) == 1:
            f = open(fileName); cont = f.read(); f.close()
            dispFields = cont.split()
            if dispFields != fields:
                #fieldを選択する
                self.check_allFields.set_active(False)
                treeList.unselectAllItems()
                treeList.selectItemNames(dispFields)
                self.tree_field.set_sensitive(True)
            else:
                self.check_allFields.set_active(True)
                self.tree_field.set_sensitive(False)
        else:
            self.check_allFields.set_active(True)
            self.tree_field.set_sensitive(False)

    #
    #  getFolderFromTreeFolder
    def getFolderFromTreeFolder(self):
        """ 選択しているfolderを取得する"""
        treeList = GtkParts.treeListIcon(self.tree_folder)
        selFolders = treeList.getSelectedItems()
        if len(selFolders) == 0:
            folder = "."
        else:
            folder = selFolders[0][1]
            if folder == _("(現在のfolder)"):
                folder = "."
        return folder

    #
    #  setComboTimeData
    def setComboTimeData(self):
        """ comboBoxにデータをセットする。"""
        timeFolders = self.timeFolders
        firstTime = timeFolders[0]
        startTime = self.controlDict["startTime"]
        for timeFolder in timeFolders:
            if float(startTime) == float(timeFolder):
                startTime = timeFolder
                break
        latestTime = timeFolders[-1]
        times = []
        times.append("firstTime:" + firstTime)
        times.append("startTime:" + startTime)
        times.append("latestTime:" + latestTime)
        times += timeFolders
        self.setComboData(times)
        startFrom = self.controlDict["startFrom"]
        selNo = 0
        if startFrom == "firstTime":
            selNo = 0
        elif startFrom == "startTime":
            selNo = 1
        elif startFrom == "latestTime":
            selNo = 2
        else:
            selNo = 1
        self.combo_time.set_active(selNo)

    #
    #  setComboData
    def setComboData(self, items):
        """ comboBoxにitemsをセットする。"""
        self.combo_time.clear()
        listModel = Gtk.ListStore(str)
        for name in items:
            listModel.append([name])
        cell = Gtk.CellRendererText()
        self.combo_time.pack_start(cell, True)
        self.combo_time.add_attribute(cell, "text", 0)
        self.combo_time.set_model(listModel)
        self.combo_time.set_entry_text_column(0)

    #
    #  getTimeFolderFromCombo
    def getTimeFolderFromCombo(self):
        """ comboBoxから選択されているtimeFolderを取得する"""
        timeCont = self.entry_time.get_text()
        conts = timeCont.split(":")
        if len(conts) == 2:
            timeFolder = conts[1]
        else:
            timeFolder = conts[0]
        return timeFolder

    #
    #  changeTime
    #-------------
    def changeTime(self):
        """ 時間を変更した時の処理"""
        self.setTreeFolderData()

    #
    #  selectFolder
    #----------------
    def selectFolder(self):
        """ folder(region)を選択した時の処理"""
        self.setTreeFieldData()

    #
    #  changeCheckBox
    #-----------------
    def changeCheckBox(self):
        """ checkBoxのon, offの処理"""
        flag = self.check_allFields.get_active()
        treeList = GtkParts.treeListIcon(self.tree_field)
        if flag == True:
            #checkBoxをチェック
            treeList.unselectAllItems()
            self.tree_field.set_sensitive(False)
        else:
            #checkBoxのチェックを外す
            self.tree_field.set_sensitive(True)
            #folderを取得
            selFolder = self.getFolderFromTreeFolder()
            #itemsを作成
            timeFolder = self.getTimeFolderFromCombo()
            fields = pyTreeFoam.case(self.caseDir).getFieldNames(timeFolder, selFolder)
            fields.sort()
            #displayFieldファイルを確認
            fileName = self.caseDir + "/" + self.timeFolders[0] + "/" + selFolder + "/.displayField"
            if len(glob.glob(fileName)) == 1:
                f = open(fileName); cont = f.read(); f.close()
                dispFields = cont.split()
                if dispFields != fields:
                    #fieldを選択する
                    treeList.selectItemNames(fields)
                    self.tree_field.set_sensitive(True)
                else:
                    #全選択
                    treeList.selectItemNames(fields)
            else:
                #全選択
                treeList.selectItemNames(fields)

    #
    #  openGridEditor
    #-----------------
    def openGridEditor(self):
        """ gridEditorを起動する"""
        #dispFieldを作成する
        self.caseDir = self.entry_caseDir.get_text()
        selFolder = self.getFolderFromTreeFolder()
        dispFileName = self.caseDir + "/" + self.timeFolders[0] + "/" + selFolder + "/.displayField"
        existsDispFile = os.path.exists(dispFileName)
        #  選択fieldを取得する
        if self.check_allFields.get_active() == True:
            #全fieldを取得するする為、.dispFieldを削除
            #if len(glob.glob(dispFileName)) != 0:
            if existsDispFile == True:
                os.remove(dispFileName)
        else:
            #selitemを取得
            treeList = GtkParts.treeListIcon(self.tree_field)
            selItems = treeList.getSelectedItems()
            allItems = treeList.getAllItems()
            if len(selItems) == len(allItems):
                #全選択の為、displyFiedを削除
                if existsDispFile == True:
                    os.remove(dispFileName)
            else:
                #.displayFieldを再作成する
                selNames = list(map(lambda x: x[1], selItems))
                if existsDispFile == True:
                    f = open(dispFileName); cont = f.read(); f.close()
                    dispFields = cont.split()
                    newFields = []
                    for field in dispFields:
                        if field in selNames:
                            newFields.append(field)
                    for field in selNames:
                        if not (field in newFields):
                            newFields.append(field)
                else:
                    newFields = selNames
                cont = "\n".join(newFields) + "\n"
                f = open(dispFileName, "w"); f.write(cont); f.close()
        #gridEditor起動
        self.runApp()

    def runApp(self):
        #gridEditorを起動
        caseDir = self.caseDir
        nTreeFoam = self.nTreeFoam
        currTime = self.getTimeFolderFromCombo()
        selFolder = self.getFolderFromTreeFolder()
        bashrcFOAM = self.configDict["bashrcFOAM"]
        comm = ". " + bashrcFOAM + "; " 
        commPath = os.getenv("TreeFoamPath") + "/python"
        #comm = commPath + "/editBoundaryConditionsDialog.py "
        comm += commPath + "/gridEditorQtDialog.py "
        comm += nTreeFoam + " " + caseDir + " " + currTime + " " + selFolder + " &"
        print(comm)
        pyTreeFoam.run(caseDir).command(comm)
        #自身を閉じる
        self.close()

    #
    #  reference
    #------------
    def reference(self):
        """ caseDirを変更する"""
        rootDir = pyTreeFoam.readConfigTreeFoam()["rootDir"]
        currDir = self.caseDir
        selDir = self.caseDir
        title = _("case選択")
        mess = _("gridEditorで開くcaseを選択してください。")
        okArgs = [self.reference_getDir]
        cancelArgs = [self.reference_setCaseDir]
        dialog = getOpenFolderDDialog.getOpenFolder(
            title, mess, rootDir, currDir, selDir, 
            funcOk=okArgs, funcCancel=cancelArgs, parent=self.mainWindow)
        dialog.show()
        return
    
    #  reference_getDir
    def reference_getDir(self, selDir):
        self.caseDir = selDir
        self.reference_setCaseDir()

    #  reference_setCaseDir
    def reference_setCaseDir(self):
        self.entry_caseDir.set_text(self.caseDir)
        #初期値を取得
        case = pyTreeFoam.case(self.caseDir)
        self.timeFolders = case.getTimeFolders()
        self.controlDict = case.getControlDict()
        #comboにデータをセット
        self.setComboTimeData()
        #tree_folderにデータセット
        self.setTreeFolderData()
        #tree_fieldにデータセット
        self.setTreeFieldData()


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    if len(sys.argv) <= 2:
        print(_("error: 引数が不足しています。"))
        print(_("  引数1  nTreefoam（TreeFoam起動No）"))
        print(_("  引数2  caseDir"))
        exit()

    if os.getenv("TreeFoamPath") == None:
        print(_("error: 環境変数 $TreeFoamPath が設定されていません"))
        exit()

    if os.getenv("TreeFoamUserPath") == None:
        print(_("error: 環境変数 $TreeFoamUserPath が設定されていません"))
        exit()

    #引数を取得
    nTreeFoam = sys.argv[1]
    caseDir = sys.argv[2]
    #GUIを表示
    winApp = windowApp(nTreeFoam, caseDir)
    winApp.main()


