#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#       meshViewerDialog.py
#
#   コマンドの内容
#
#   start           command処理開始
#
#   show            指定したparts（patch, zone, set, stl）を表示
#   showActor        例: show patch:inW cellZone:block faceSet:faceF stl:outW
#
#   addShow         指定したparts（patch, zone, set, stl）を追加表示
#   addShowActor    例: addShow patch:inW cellZone:block faceSet:faceF
#
#   clear           vtk画面をクリアする
#                   例: clear
#
#   print           statusBar上に出力する
#                   例: print 'loading ...'\n
#
#   load            指定したvtkDataを読み込む
#   reload          例: load [all] [patchesZones] [sets] [stls]
#                       load sets stls     #setsとstlを再読込する。
#
#   case            caseDirを指定する。実行後は「load all」で再読込する
#   caseDir         例: case ~/test/cavity
#
#   region          regionを指定する。実行後は「load all」で再読込する。
#                   例: region .
#
#   time            timeFolderを指定する。実行後は、「load all」で再読込する。
#                   例: time 0.6, time latestTime etc.
#
#   mess            messageを表示する
#   message         例: mess "topoSetEditor\n 選択partsが表示されます"
#
#   addMess         messageを追加表示する
#   addMessage      例: addMess 追加表示
#
#   stl             stlFileの場所を指定する。実行後は「load stls」で再読込する
#   stlDir          例: stl ./model1
#
#   edge            edge表示を制御する
#                   例: edge <option>  option: show or hide
#
#   outline         outline表示を制御する
#                   例: outline <option>  option: show or hide
#
#   face            face表裏の表示を制御する
#                   例: face <option>  option: both, front or back
#
#   wait            GUIを表示させる為の待ち時間を設定
#                   例: wait 0.1s  or wait 100ms  or wait 0.1
#
#   end             入力threadを終了させる
#
#   close           meshViewerを閉じる
#   exit
#   quit
#
#
#   21/08/05    新規作成
#      09/04    showItemSize:minLocationの座標の表示を追加。
#      09/15    setsのvtk作成をmeshViewerCreateSetsVtk.pyで作成。
#               foamToVtkコマンドを使っていたが、OF-v2006では出力結果
#               が異なる為、共用できない。
#      09/23    commandLoad:delayを設定して実行する様に修正。
#               大規模メッシュの場合、loadに時間がかかる為。
#      11/18    setViewerTitle:GUIのtitle表示を追加。
#      11/23    close:終了時にos.killpgで強制終了する様に変更
#               threadでinputを使っており、これが終了できていない為。
#      11/25    waitコマンド、clearコマンド追加
#      11/30    setCaseDir:「self.label_caseDir.set_text(title)」を
#               「GLib.idle_add(self.label_caseDir.set_text, title)」
#               に修正。threadでlabelをセットする場合、以下error発生の為
#               gtk_label_update_layout_width: assertion failed
#      12/01    threadでGUIを修正sしていたものをmain側でGUI操作する様に
#               書き換え
#      12/08    multiRegionCaseの場合、outlineを全regionのedgeに変更。
#               regionの表示を追加。
#               getPatchTypesAllRegions:boundaryからpatchTypeの読込追加
#      12/14    close:close時、winSizeを保存する事を追加
#               起動時は、保存したwinSizeを反映させる。
#      12/15    createZoneActors,createPointVtkActors:pointSizeを設定
#               createCellVtkActors,etc.:setsの色をpinkに変更
#      12/25    reloadMesh:threadとtimerEventの起動チェックを追加
#               エラー等で停止している場合は、再起動させる。
#      12/30    getRegionData:vtk-9.1.0からvtkOpenFOAMReaderが変更されて
#               おり、これに対応。
#   21/01/05    commandTime:startFrom, currTime, firstTime, startTime,
#               latestTime を追加。
#      01/07    reloadMesh:選択itemを保存、reloadMesh後、選択itemを再設定
#               する様に修正。
#               vtkのimport方法を修正。
#      01/13    close:close時に自身と子process全てのpidをkillする様に修正。
#      01/23    iniGetPatchesZonesActors:timeFolderのチェックを追加。
#      03/26    writeCancelFlag:flag書き込み後、timerEventのinterval時間
#               (50ms)分waitするを追加。flagが読めない時があるため。
#      05/02    timerEvent:self.threadCancelFlagのcheckを ==1 → ==True
#               に変更。
#      05/07    ubuntu起動直後のcancel時に、自身のprocessが実行中かどうか
#               のチェックが働かない時があったので、cancel処理方法を修正
#      09/20    addOriginAxis:原点表示を追加
#   23/08/13    getPatchTypesAllRegions:currTimeをcontrolDictのstartTime
#               に設定していたので、patchTypeが取得できない事があった。
#               viewer起動時にtimeFolderを指定しているので、このtimeFolderに設定
#   24/01/19    checkFaceZones:faceZonesファイル内容のチェックを追加。（起動時）
#               OF-11以降のチェック
#      01/22    checkFaceZones:バグ修正。
#               pyTreeFoam.findDataListPositionの修正に伴いバグ表面化。
#      02/12    getPatchesZonesActorsAllRegions:filmを追加。（OF-11対応）
#      07/28    checkFaceZones:バグ修正。（（caseFolderでは無いfolerでは、エラー発生する）
#      08/22    起動時、window位置をmouseに合わせる様修正。
#      10/14    universalDialogs:okDialog,errDialogを追加
#   25/02/05    大幅変更。internalField, clip, planeの表示を追加。
#      03/30    setIniVtkModel:vtk-9.4対応でGradientBackgroundOnを追加
#      07/08    getPatchesZonesActorsAllRegions:「constant/polyMesh」無しのmultiRegion
#               に対応させる為修正。
# 

import logFileCreater
logFileCreater.log()

import os, sys
print("importing vtk...")
sys.stdout.flush()
import checkVtkModule
isVtk = checkVtkModule.isVtk("meshViewer")
if isVtk == True:
    import vtk
    from vtk.util import colors
    vtkVersion = vtk.vtkVersion.GetVTKSourceVersion().split()[-1]
    print("  imported vtk version " + vtkVersion)
else:
    print("error: could not import vtk !")
    exit()

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GLib
#import os, sys
import glob
import threading
import time
import signal
import select

import GtkVtkRenderWindowInteractor as gtkVtk
import pyTreeFoam
import geometryFunc as geom
import meshViewerCreateSetsVtk as createSetsVtk
import universalDialogs as unvDlg

#-------------------
#  windowApp class
#-------------------
class windowApp:

    gladeFileName = "meshViewerDialog.glade"

    def __init__(self, caseDir, timeFolder, region, inputFile, title):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + self.gladeFileName)
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobjectを取得
        self.maskEvent = True
        self.setGtkObject()
        #変数の設定
        self.caseDir = self.getAbsPath(caseDir)
        self.timeFolder = timeFolder
        self.region = region
        self.relativeStlDir = "model"
        self.stlDir = caseDir + "/" + self.relativeStlDir
        self.inputFile = inputFile
        self.title = title
        self.commandTitle(title)
        self.closeFlag = 0
        self.idleFlag = True
        self.allRegions = []
        self.winSize = (600, 500)
        self.thGetCommand = ""          #thread名の定義
        #vtk関連
        self.vtkObj = gtkVtk.GtkVtkRenderWindowInteractor(self)
        self.vtkSize = (250, 200)
        #clip(planeWidget)の定義
        self.planeWidget = vtk.vtkImplicitPlaneWidget()
        #internalMeshの辞書
        self.internalMeshDict = {}
        #actorの辞書
        self.clipPlaneDict = {}
        self.clippedMeshActorDict = {}
        self.internalActorDict = {}
        self.patchActorDict = {}
        self.outlineActorDict = {}
        self.cellZoneActorDict = {}
        self.faceZoneActorDict = {}
        self.pointZoneActorDict = {}
        self.cellSetActorDict = {}
        self.faceSetActorDict = {}
        self.pointSetActorDict = {}
        self.stlActorDict = {}
        self.patchTypesDict = {}
        self.currItemActors = []
        self.currItemNames = []
        self.vtkFilesDict = {}
        self.axes = ""
        self.originAxis = None
        self.renderer = ""
        #signal関係
        signal.signal(signal.SIGUSR1, self.signal_usr1)
        self.cancelFile = os.getenv("TreeFoamUserPath") + "/temp/" + self.title + ".stat"
        self.threadCancelFlag = False
        #cancelFileを作成
        self.writeCancelFlag(False)
        #初期command設定(起動時に実行するコマンド)
        #例：
        #  self.commandList  = [_("mess TreeFoamにlinkしています。\n")]
        #  self.commandList += [_("addMess '  patch, zone, setが表示できます。'\n")]
        self.commandList = []
        #OF-11以降用のチェック(faceZonesのtrue, falseをチェック修正)
        self.checkFaceZones()

    #
    #  main
    #--------
    def main(self):
        """ GUIの表示"""
        #winSizeを設定
        self.setWindowSize()
        #vtkAreaをセット
        self.setVtkArea()
        #caseDirを表示
        self.setCaseDir(self.caseDir)
        #GUIを表示
        self.mainWindow.show_all()
        #vtkを表示
        self.setIniVtkModel()               #vtkを表示
        self.currItemActors = []
        self.showActors(self.currItemActors)
        self.resetCamera()
        self.maskEvent = False              #eventのmaskを解除
        #threadを起動
        self.runThreadGetCommand()
        #timerEventを設定
        self.setTimerEvent()
        Gtk.main()

    #
    #  close
    #---------
    def close(self, *args):
        """ 閉じる"""
        # #winSizeを保存
        self.saveWindowSize()
        #閉じる
        self.closeFlag = 1
        Gtk.main_quit()
        #viewerStatファイルを削除する
        self.removeCancelFile()
        #threadが起動しているので、pidを取得してkillする
        pid = os.getpid()
        #os.killpg(pid, signal.SIGTERM)  #threadを強制終了させる
        os.killpg(os.getpgid(pid), signal.SIGKILL)

    #
    #  saveWindowSize
    def saveWindowSize(self):
        """ windowSizeを保存する"""
        #winSizeを取得
        (x, y) = self.mainWindow.get_size()
        #splitterを取得してxを修正
        winSizeDict = pyTreeFoam.readWindowSize()
        if "meshViewer_splitter" in winSizeDict.keys():
            splitWidth = winSizeDict["meshViewer_splitter"]
            if splitWidth != "":
                splitWidth = int(splitWidth)
                x = x + splitWidth
        winSize = str(x) + " " + str(y)
        winDict = {"meshViewer": winSize}
        pyTreeFoam.writeWindowSize(winDict)

    #
    #  signal_usr1
    #---------------
    def signal_usr1(self, signum, frame):
        """ signal.SIGUSR1のhandler"""
        sys.stdout.flush()
        print("---->> received signal. load is canceled")
        sys.stdout.flush()
        #threadCancelFlagをセットしてrunCommandListを停止する
        self.threadCancelFlag = True
        
    #
    #  writeCancelFlag
    #------------------
    def writeCancelFlag(self, flag):
        """ cancelの内容をセットする"""
        fileName = self.cancelFile
        if flag == True:
            cont = "threadCancel True\n"
        else:
            cont = "threadCancel False\n"
        f = open(fileName, "w"); f.write(cont); f.close()

    #
    #  getCancelFlag
    #----------------
    def getCancelFlag(self):
        """ cancelFileの内容を取得する。
        viewerが起動中は、cancelFileが作成されている。

        Returns:
            True:   cancel受付
            False:  cancel処理済"""
        fileName = self.cancelFile
        f = open(fileName); lines = f.readlines(); f.close()
        if len(lines) > 0:
            words = lines[0].split()
            if words[1] == "True":
                return True
            else:
                return False

    #
    #  removeCancelFile
    #-------------------
    def removeCancelFile(self):
        """ cancelFileを削除する"""
        if os.path.exists(self.cancelFile) == True:
            os.remove(self.cancelFile)

    #
    #  checkFaceZones
    #-----------------
    def checkFaceZones(self):
        """ faceZonesのtrue,falseをチェック。
        OF-11以降のチェック"""
        #OFversionをチェック
        OFversion = pyTreeFoam.readConfigTreeFoam()["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer < 11.0:
            return
        #caseDirをチェック
        if pyTreeFoam.isCaseDir(self.caseDir) == False:
            return
        #faceZonesの内容をチェックし修正。
        case = pyTreeFoam.case(self.caseDir)
        timeFolder = self.timeFolder
        flag, loc, regions = case.isMultiRegion()
        if flag == False:
            regions = ["."]
        for region in regions:
            relMeshDir = case.getCurrMeshDir(timeFolder, region, "faceZones")
            faceZonesFile = self.caseDir + os.sep + relMeshDir + os.sep + "faceZones"
            if os.path.exists(faceZonesFile) == True:
                #faceZonesの内容をチェック(asciiのみ)
                #  true, false → 1, 0 に修正
                foamFile = pyTreeFoam.foamFile()
                cont = foamFile.readFull(faceZonesFile)
                try:
                    cont = cont.decode()
                    flag = 0
                except:
                    flag = 1
                if flag == 1:
                    #binaryは、チェックせず
                    return
                #asciiファイは、書き換える
                l = len(cont)
                cont = cont.replace("true", "1")
                cont = cont.replace("false", "0")
                if l != len(cont):
                    foamFile.writeFull(faceZonesFile, cont)

    #--------- event handler ----------
    def onChangeShowEdge(self, event):      #edge表示
        if self.maskEvent == False:
            self.changeShowEdge()
    def onChangeShowOutline(self, event):   #outline表示
        if self.maskEvent == False:
            self.changeShowOutline()
    def onChangeShowAxis(self, event):      #原点表示
        if self.maskEvent == False:
            self.changeShowAxis()
    def onResetCamera(self, event):         #resetCamera
        self.resetCamera()
    def onSetViewX(self, event):            #X軸視
        self.setViewX()
    def onSetViewY(self, event):            #Y軸視
        self.setViewY()
    def onSetViewZ(self, event):            #Z軸視
        self.setViewZ()
    def onRightToLeft(self, event):         #左右前後入れ替え
        self.rightToLeft()
    def onRolling90(self, event):           #回転90°
        self.rolling90()
    def onReloadMesh(self, event):          #再読込
        self.reloadMesh()
    def onChangeShowBoth(self, event):      #face両面表示
        self.changeShowBoth()
    def onChangeShowFront(self, event):     #face表面表示
        self.changeShowFront()
    def onChangeShowBack(self, event):      #face裏面表示
        self.changeShowBack()

    #
    #  setGtkObject
    #----------------
    def setGtkObject(self):
        """ glade内のobjectを取得"""
        #vtk関連
        self.box_vtk = self.builder.get_object("box_vtk")
        self.check_showEdge = self.builder.get_object("check_showEdge")
        self.check_showOutline = self.builder.get_object("check_showOutline")
        self.check_showAxis = self.builder.get_object("check_showAxis")
        self.button_resetCamera = self.builder.get_object("button_resetCamera")
        self.button_viewX = self.builder.get_object("button_viewX")
        self.button_viewY = self.builder.get_object("button_viewY")
        self.button_viewZ = self.builder.get_object("button_viewZ")
        self.button_rightToLeft = self.builder.get_object("button_rightToLeft")
        self.button_rolling90 = self.builder.get_object("button_rolling90")
        self.button_reload = self.builder.get_object("button_reload")
        self.radio_showBoth = self.builder.get_object("radio_showBoth")
        self.radio_showFront = self.builder.get_object("radio_showFront")
        self.radio_showBack = self.builder.get_object("radio_showBack")
        self.label_caseDir = self.builder.get_object("label_caseDir")
        self.label_message = self.builder.get_object("label_message")
        self.label_status = self.builder.get_object("label_status")
        self.label_itemSize = self.builder.get_object("label_itemSize")
        #初期設定
        self.check_showEdge.set_active(True)
        self.check_showOutline.set_active(True)
        #tooltip
        self.check_showEdge.props.tooltip_text = _("表示モデルのedgeを表示する")
        self.check_showOutline.props.tooltip_text = _("モデル全体の概形状を表示する")
        self.button_resetCamera.props.tooltip_text = _("画面にフィットさせる（reset camera）")
        self.button_viewX.props.tooltip_text = _("x軸視（y-z 面）を表示")
        self.button_viewY.props.tooltip_text = _("y軸視（z-x 面）を表示")
        self.button_viewZ.props.tooltip_text = _("z軸視（x-y 面）を表示")
        self.button_rightToLeft.props.tooltip_text = _("視点を反転(左右前後反転)")
        self.button_rolling90.props.tooltip_text = _("時計回りに90度回転")
        self.button_reload.props.tooltip_text = _("vtkDataを再読込する")
        self.radio_showBoth.props.tooltip_text = _("faceの表裏両面を表示する")
        self.radio_showFront.props.tooltip_text = _("faceの表面のみ表示する\nfaceの方向が確認できる")
        self.radio_showBack.props.tooltip_text = _("faceの裏面のみ表示する\n隠れた内部のstlが確認できる")

    #
    #  setWindowSize
    #----------------
    def setWindowSize(self):
        #sizeを取得
        winDict = pyTreeFoam.readWindowSize()
        if "meshViewer" in winDict.keys():
            if winDict["meshViewer"] != "":
                words = winDict["meshViewer"].split()
                self.winSize = (int(words[0]), int(words[1]))
                # #sizeを取得
            if "meshViewer_splitter" in winDict.keys():
                words = winDict["meshViewer_splitter"].split()
                splitWidth = int(words[0])
                if self.winSize[0] > splitWidth:
                    x, y = self.winSize
                    x = x - splitWidth
                    self.winSize = (x, y)
        self.mainWindow.set_default_size(*self.winSize)

    #
    #  setVtkArea
    #-------------
    def setVtkArea(self):
        """ vtkAreaを設定する"""
        #sizeを設定
        vtkObj = self.vtkObj
        vtkObj.set_size_request(*self.vtkSize)
        self.box_vtk.pack_start(vtkObj, True, True, 0)

    #
    #  addCornerAxis
    #----------------
    def addCornerAxis(self):
        """ XYZ軸（compas）をrendererに追加する"""
        interactor = self.renWin.GetInteractor()
        axesActor = vtk.vtkAxesActor()
        self.axes = vtk.vtkOrientationMarkerWidget()
        self.axes.SetOrientationMarker(axesActor)
        self.axes.SetInteractor(interactor)
        self.axes.EnabledOn()
        self.axes.InteractiveOn()

    #
    #  addOriginAxis
    #----------------
    def addOriginAxis(self, actors):
        """ 原点を表示する"""
        if len(actors) == 0:
            #表示するactorsが存在しない場合
            if self.originAxis != None:
                self.renderer.RemoveActor(self.originAxis)
                self.originAxis = None
            return
        if self.check_showAxis.get_active() == True:
            #原点を表示
            if self.originAxis == None:
                #原点を作成
                bounds = self.getBoundsActors()
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes = vtk.vtkAxesActor()
                axes.SetTotalLength(length, length, length)
                axes.GetXAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetXAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetYAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetYAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetZAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetZAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                self.renderer.AddActor(axes)
                self.originAxis = axes
            else:
                #原点を修正
                axes = self.originAxis
                bounds = self.getBoundsActors()
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes.SetTotalLength(length, length, length)
                #削除して追加する（axesを最後に描画する為）
                self.renderer.RemoveActor(axes)
                self.renderer.AddActor(axes)
                self.originAxis = axes
        else:
            #原点を削除
            self.renderer.RemoveActor(self.originAxis)
            self.originAxis = None
        return

    #
    #  setIniVtkModel
    #---------------
    def setIniVtkModel(self):
        """ vtk画面の初期化"""
        self.renderer = vtk.vtkRenderer()
        #backgroundColorを設定
        self.renderer.SetBackground(0.85, 0.85, 0.85)
        self.renderer.SetBackground2(0.4, 0.55, 0.75)
        self.renderer.GradientBackgroundOn()
        #renderWindowをセット
        self.renWin = self.vtkObj.GetRenderWindow()
        self.renWin.NewInstance()
        #rendererを追加
        self.renWin.AddRenderer(self.renderer)
        #axisを追加
        self.addCornerAxis()
        #vtk表示
        self.vtkObj.GlobalWarningDisplayOff()   #windowsのエラー対策
        self.vtkObj.Initialize()
        self.vtkObj.Start()

    #
    #  showActors
    #-------------
    def showActors(self, itemActors, clipRegion=""):
        """ actorを表示する"""
        #現在のactorを削除する
        delActors = self.renderer.GetActors()
        for actor in delActors:
            self.renderer.RemoveActor(actor)
        #outlineをチェック
        actors = []
        if self.check_showOutline.get_active() == True:
            for region in self.allRegions:
                if region in self.outlineActorDict.keys():
                    outlineDict = self.outlineActorDict[region]
                    outlineActors = list(outlineDict.values())
                    actors += outlineActors
        #指定のactorを取得し、propertyを設定
        for item, actor in itemActors:
            if item == "outline":
                pass
            else:
                actor = self.setPropertyOfActor(actor, item)
                actors.append(actor)
        #指定のactorを表示させる
        for actor in actors:
            self.renderer.AddActor(actor)
        #原点を表示
        self.addOriginAxis(actors)
        #itemSizeを表示
        bounds = self.getBoundsActors()
        self.showItemSize(bounds)
        #clipPlaneを表示？
        self.planeWidget.Off()
        if clipRegion != "":
            #clipPlaneを表示
            region = clipRegion
            normal = self.clipPlaneDict[region]["normal"]
            bounds = self.clipPlaneDict[region]["bounds"]
            origin = self.clipPlaneDict[region]["origin"]
            invert = self.clipPlaneDict[region]["invert"]
            self.planeWidget.OutlineTranslationOff()
            self.planeWidget.SetPlaceFactor( 1.0 )
            self.planeWidget.PlaceWidget(bounds)
            self.planeWidget.SetOrigin(origin)
            self.planeWidget.SetNormal(normal)
            prop = self.planeWidget.GetSelectedPlaneProperty()
            prop.SetColor(colors.grey)
            prop.SetOpacity(0.5)
            prop = self.planeWidget.GetPlaneProperty()
            prop.SetColor(colors.grey)
            prop.SetOpacity(0.8)
            self.planeWidget.On()
        self.vtkObj.Initialize()

    #
    #  showItemSize
    def showItemSize(self, bounds):
        """ 表示itemのサイズを表示する"""
        xyz = []; minxyz = []
        if len(bounds) > 0:
            #lengthを取得
            x = bounds[1]-bounds[0]
            y = bounds[3]-bounds[2]
            z = bounds[5]-bounds[4]
            xstr = pyTreeFoam.getValue3digit_shrink(x)
            ystr = pyTreeFoam.getValue3digit_shrink(y)
            zstr = pyTreeFoam.getValue3digit_shrink(z)
            xyz = [xstr, ystr, zstr]
            #minLocationを取得
            minx = bounds[0]
            miny = bounds[2]
            minz = bounds[4]
            minxyz = [pyTreeFoam.getValue3digit_shrink(minx),
                      pyTreeFoam.getValue3digit_shrink(miny),
                      pyTreeFoam.getValue3digit_shrink(minz)]
        cont = _("表示itemのサイズ XYZ: ") + " ".join(xyz)
        cont += _("  (minLoc: ") + " ".join(minxyz) + " )"
        self.label_itemSize.set_text(cont)

    #--------------------
    #  timerEvent関連
    #--------------------
    #
    #  setTimerEvent
    #----------------
    def setTimerEvent(self):
        """ timerEventを設定。"""
        self.nCommandPass = 0       #timeEventをpassする回数
        self.timerValue = 100       #timer時間(ms)を設定
        GLib.timeout_add(self.timerValue, self.timerEvent)

    #
    #  timerEvent
    #-------------
    def timerEvent(self):
        """ self.commandListの内容を取り出し実行する。"""
        #flagをセット
        self.isRunningTimer = 1
        #commandが保存されている？
        if len(self.commandList) > 0:
            #保存されているcommandを実行する
            #  threadCancelFlag=Trueの時、途中でcancelする
            self.runCommandList()
        #cancelFlagのクリア
        if self.threadCancelFlag == True:
            self.writeCancelFlag(False)
            self.threadCancelFlag = False
        #closeFlagをチェック
        if self.closeFlag == 0:
            #closeでない場合、timerを設定
            GLib.timeout_add(self.timerValue, self.timerEvent)

    #--------------------
    #  eventの処理
    #--------------------
    #
    #  changeShowEdge
    #-----------------
    def changeShowEdge(self):
        """ showEdgeを変更した時"""
        self.showActors(self.currItemActors)

    #
    #  changeShowOutline
    #--------------------
    def changeShowOutline(self):
        """ showOutlineを変更した時"""
        self.showActors(self.currItemActors)

    #
    #  changeShowAxis
    #-----------------
    def changeShowAxis(self):
        """ showOriginAxisを変更所た時"""
        self.showActors(self.currItemActors)

    #
    #  resetCamera
    #---------------
    def resetCamera(self):
        """ 画面に合わせてVTKを表示させる"""
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  setViewX
    #-----------
    def setViewX(self):
        """ X軸方向から見る"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter+maxLeng*2.0, Ycenter, Zcenter)
        #X軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  setViewY
    #-----------
    def setViewY(self):
        """ Y軸視（Z-X面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter+maxLeng*2.0, Zcenter)
        #Y軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(1, 0, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  setViewZ
    #-----------
    def setViewZ(self):
        """ Z軸視（X-Y面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter, Zcenter+maxLeng*2.0)
        #Z軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 1, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  rightToLeft
    #--------------
    def rightToLeft(self):
        """ camera位置を反転（左右反転）させる"""
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        center = (Xcenter, Ycenter, Zcenter)
        camera = self.renderer.GetActiveCamera()
        pos = camera.GetPosition()
        relPos = (pos[0]-center[0], pos[1]-center[1], pos[2]-center[2])
        newRelPos = list(map(lambda x: -x, relPos))
        newPos = (newRelPos[0]+center[0], newRelPos[1]+center[1], newRelPos[2]+center[2])
        camera.SetPosition(*newPos)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  rolling90
    #-------------
    def rolling90(self):
        """ 視点を右回転（90°）させる。"""
        camera = self.renderer.GetActiveCamera()
        camera.Roll(-90.0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  reloadMesh
    #-------------
    def reloadMesh(self):
        """ meshDataをloadする"""
        #currItemNamesを取得
        currItemNames = self.currItemNames[:]
        #全itemをload
        print("loading patches, zones, sets, stl...")
        commLines  = ["start\n"]
        commLines += ["clear\n"]
        commLines += ["print 'loading ...'\n"]
        commLines += ["wait 0.2s\n"]
        commLines += ["load all\n"]
        argLine = " ".join(currItemNames)
        commLines += ["show " + argLine + "\n"]
        self.commandList += commLines
        #threadとtimerEventが作動しているかcheck
        self.checkThreadTimer()

    #
    #  remakeSetsVtk
    def remakeSetsVtk(self):
        """ setsのvtkを作り直す"""
        caseDir = self.caseDir
        region = self.region
        timeFolder = self.timeFolder
        #vtkを無条件で作り直す。
        createSetsVtk.createAllSetsVtkFiles(caseDir, timeFolder, region, checkFlag=False)

    #
    #  checkThreadTimer
    def checkThreadTimer(self):
        """ threadとtimerEventが作動しているかcheck。
        作動していない場合は、起動させる。"""
        #threadを確認
        if self.thGetCommand._started.is_set() == False:
            #停止している場合は、再起動
            self.thGetCommand.start()
        #timerEventを確認
        #flagをクリアして、checkTimerEventへ
        self.isRunningTimer = 0
        GLib.timeout_add(self.timerValue*1.5, self.checkTimerEvent)

    #  checkTimerEvent
    def checkTimerEvent(self):
        """ timerEventが停止しているかチェック。
        停止している場合は、再起動。"""
        #flagを確認
        if self.isRunningTimer == 0 and self.closeFlag == 0:
            #停止している場合は、再起動
            self.timerEvent()

    #
    #  changeShowBoth
    #-----------------
    def changeShowBoth(self):
        """ face両面を表示"""
        self.showActors(self.currItemActors)

    #
    #  changeShowFront
    #-------------------
    def changeShowFront(self):
        """ face表面を表示"""
        self.showActors(self.currItemActors)

    #
    #  changeShowBack
    #-----------------
    def changeShowBack(self):
        """ face裏面を表示"""
        self.showActors(self.currItemActors)

    #  getCameraBounds
    def getCameraBounds(self):
        """ 表示させるactorのboundsを取得する。
        マージンを含めて取得する。"""
        bounds = self.getBoundsActors()
        if len(bounds) == 0:
            return ()
        #マージンを確保
        xmin = bounds[0]
        xmax = bounds[1]
        ymin = bounds[2]
        ymax = bounds[3]
        zmin = bounds[4]
        zmax = bounds[5]
        a = 0.1
        xmin = xmin - (xmax-xmin) * a
        xmax = xmax + (xmax-xmin) * a
        ymin = ymin - (ymax-ymin) * a
        ymax = ymax + (ymax-ymin) * a
        zmin = zmin - (zmax-xmin) * a
        zmax = zmax + (zmax-xmin) * a
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #  getBoundsActors
    def getBoundsActors(self):
        """ 全actorのboundsを取得して返す"""
        actors = self.renderer.GetActors()
        bounds = []
        for actor in actors:
            #if actor in self.actorDict.values():
            bound = actor.GetBounds()
            bounds.append(bound)
        if len(bounds) == 0:
            return ()
        Xmin = min(map(lambda x: x[0], bounds))
        Xmax = max(map(lambda x: x[1], bounds))
        Ymin = min(map(lambda x: x[2], bounds))
        Ymax = max(map(lambda x: x[3], bounds))
        Zmin = min(map(lambda x: x[4], bounds))
        Zmax = max(map(lambda x: x[5], bounds))
        return (Xmin, Xmax, Ymin, Ymax, Zmin, Zmax)


    #--------------------------
    #  showActorのcommand関連
    #--------------------------
    #
    #  runThreadGetCommand
    #----------------------
    def runThreadGetCommand(self):
        """ コマンド取得するthreadを起動する"""
        self.thGetCommand = threading.Thread(target=self.getCommandLine)
        self.thGetCommand.start()

    #
    #  getCommandLine
    def getCommandLine(self):
        """ コマンドを取得し、commandListに保存する"""
        if self.inputFile == "":
            # key入力
            while self.closeFlag == 0:
                try:
                    line = input()
                    #line = sys.stdin.readline()
                except:
                    line = ""
                words = line.split()
                if len(words) >= 1:
                    self.commandList.append(line)
        else:
            #fileから取得する
            pid = os.getpid()
            path = os.getenv("TreeFoamUserPath")
            fileName = path + "/temp/meshViewer_" + str(pid)
            f = open(fileName, encoding="utf-8")
            while self.closeFlag == 0:
                line = f.readline()
                words = line.split()
                if len(words) >= 1:
                    self.commandList.append()
            f.close()

    #
    #  runCommandList
    #--------------------
    def runCommandList(self):
        """ self.commandListに保存されているコマンドを実行する"""
        #wait時間分待つ
        if self.nCommandPass > 0:
            self.nCommandPass -= 1
            return
        #コマンドを実行
        while len(self.commandList) > 0:
            #threadCancel?
            if self.threadCancelFlag == True:
                self.commandList = []
                break
            line = self.commandList.pop(0)
            if line[-1] == "\n":
                line = line[:-1]
            words = line.split()
            if words[0] == "close" or words[0] == "exit" or words[0] == "quit":
                #meshViewerを閉じる
                if self.closeFlag == 0:
                    self.close()
                break
            elif words[0] == "wait":
                #command実行をwaitする
                wTime = self.getWaitTime(words[1])
                self.nCommandPass = wTime // self.timerValue
                break
            else:
                #command実行
                self.runCommand(line)
        return

    #
    #  getWaitTime
    def getWaitTime(self, word):
        try:
            wTime = int(word)
        except:
            if word[-2:] == "ms":
                wTime = float(word[:-2])
            elif word[-1] == "s":
                wTime = float(word[:-1]) * 1000.0
        return wTime

    #
    #  createCommandDict
    #--------------------
    def createCommandDict(self):
        """ commandの辞書を定義する"""
        commandDict = {
            #command        関数名
            "start":        self.commandNone,
            "end":          self.commandEnd,
            "title":        self.commandTitle,
            "show":         self.commandShowActor,
            "showActor":    self.commandShowActor,
            "clear":        self.commandClear,
            "clearActor":   self.commandClear,
            "addShow":      self.commandAddShowActor,
            "addShowActor": self.commandAddShowActor,
            "print":        self.commandPrint,
            "load":         self.commandLoad,
            "reload":       self.commandLoad,
            "case":         self.commandCaseDir,
            "caseDir":      self.commandCaseDir,
            "region":       self.commandRegion,
            "message":      self.commandMessage,
            "mess":         self.commandMessage,
            "addMessage":   self.commandAddMessage,
            "addMess":      self.commandAddMessage,
            "stl":          self.commandStlDir,
            "stlDir":       self.commandStlDir,
            "time":         self.commandTime,
            "outline":      self.commandOutline,
            "edge":         self.commandEdge,
            "face":         self.commandFace,
            "resetCamera":  self.commandResetCamera
        }
        return commandDict

    #
    #  runCommand
    def runCommand(self, line):
        """ コマンドを解釈し、実行する。"""

        #  skipUntilNextWord
        def skipUntilNextWord(line):
            lineOp = pyTreeFoam.strOp(line)
            keyword, p = lineOp.getKeyword(0)
            pm = lineOp.skipSP(p)
            if pm >= len(line):
                keyword += line[-1]
            #「"」「'」を削除
            newLine = line[pm:].replace("'", "")
            newLine = newLine.replace('"', '')
            newLine = newLine.replace("\\n", "\n")
            return newLine

        #cancelの場合、直ぐに戻る
        if self.threadCancelFlag == True:
            return
        #command無い場合、直ぐに戻る
        words = line.split()
        if len(words) == 0:
           return
        #command解釈し実行
        command = words[0]
        argLine = skipUntilNextWord(line)
        commandDict = self.createCommandDict()
        if command[0] == "#":
            pass
        elif command in commandDict.keys():
            commandFunc = commandDict[command]
            self.writeStatus(line)
            commandFunc(argLine)
        else:
            print(_("error: コマンド「") + command + _("」が間違っています。"))

    #
    #  commandNone
    def commandNone(self, line):
        """ 何もせず戻る"""
        return

    #
    #  commandEnd
    def commandEnd(self, line):
        """ commandListの終了処理"""
        #statusBarに「Ready」を表示
        self.writeStatus("Ready\n")
        return

    #
    #  commandPrint
    def commandPrint(self, line):
        self.writeStatus(line)
        return

    #
    #  commandTitle
    def commandTitle(self, line):
        """ titleを設定する"""
        title = line
        self.mainWindow.set_title(title)

    #
    #  commandShowActor
    def commandShowActor(self, line):
        """ vtkのactorを表示する"""
        args = line.split()
        actors = self.getCommandArgsActors(args)
        self.currItemNames = args[:]
        self.currItemActors = actors[:]
        self.showActors(self.currItemActors)

    #
    #  commandAddShowActor
    def commandAddShowActor(self, line):
        """ vtkのactorを追加して表示する"""
        args = line.split()
        currActors = self.currItemActors[:]
        actors = self.getCommandArgsActors(args)
        allActors = currActors + actors
        self.currItemNames += args[:]
        self.currItemActors = allActors[:]
        self.showActors(self.currItemActors)

    #
    #  commandClear
    def commandClear(self, line):
        """ vtk画面をクリアする"""
        #現在のactorを削除する
        delActors = self.renderer.GetActors()
        for actor in delActors:
            self.renderer.RemoveActor(actor)
        #辞書初期化
        #self.clipPlaneDict = {}
        self.clippedMeshActorDict = {}
        self.internalActorDict = {}
        self.patchActorDict = {}
        self.outlineActorDict = {}
        self.cellZoneActorDict = {}
        self.faceZoneActorDict = {}
        self.pointZoneActorDict = {}
        self.cellSetActorDict = {}
        self.faceSetActorDict = {}
        self.pointSetActorDict = {}
        self.stlActorDict = {}
        self.patchTypesDict = {}
        self.currItemActors = []
        self.currItemNames = []
        #self.showActors(self.currItemActors)
        self.vtkObj.Initialize()

    #
    #  commandLoad
    def commandLoad(self, line):
        """ meshDataをloadする"""
        self.idleFlag = False
        args = line.split()
        if len(args) == 0:
            self.iniGetPatchesZonesActors()
            self.getPatchesZonesActorsAllRegions()
            self.iniGetSetsActorsAllRegions()
            self.getSetsActorsAllRegions()
            self.getStlActors()
            self.getPatchTypesAllRegions()
        else:
            for arg in args:
                if arg == "all":
                    self.iniGetPatchesZonesActors()
                    self.getPatchesZonesActorsAllRegions()
                    self.iniGetSetsActorsAllRegions()
                    self.getSetsActorsAllRegions()
                    self.getStlActors()
                    self.getPatchTypesAllRegions()
                elif arg == "patches" or arg == "zones" or arg == "patchesZones":
                    self.iniGetPatchesZonesActors()
                    self.getPatchesZonesActorsAllRegions()
                elif arg == "sets":
                    self.iniGetSetsActorsAllRegions()
                    self.getSetsActorsAllRegions()
                elif arg == "stls" or arg == "stl":
                    self.getStlActors()
                elif arg == "patchTypes":
                    self.getPatchTypesAllRegions()
                else:
                    print(_("error: 引数「") + arg + _("」が間違っています。"))
        if self.threadCancelFlag == True:
            print("load ---->>> cancel")
        else:
            print("load ... done")
        self.idleFlag = True
        self.currItemActors = []
        self.showActors(self.currItemActors)
        self.writeStatus(line)

    #
    #  commandCaseDir
    def commandCaseDir(self, argLine):
        """ caseDirを変更する"""
        self.caseDir = self.getAbsPath(argLine)
        self.setCaseDir(self.caseDir)

    #  getAbsPath
    def getAbsPath(self, path):
        """ 絶対pathに変換する"""
        path = os.path.normpath(path)
        path = os.path.expanduser(path)
        path = os.path.abspath(path)
        return path

    #  setCaseDir
    def setCaseDir(self, caseDir):
        words = caseDir.split("/")
        if len(words) > 2:
            title = "/".join(words[-2:])
            if len(title) > 100:
                title = "..." + title[-100:]
            else:
                title = ".../" + title
        else:
            title = caseDir
            if len(title) > 100:
                title = "..." + title[-100:]
        title = "caseDir: " + title
        self.label_caseDir.set_text(title)

    #
    #  commandRegion
    def commandRegion(self, argLine):
        """ regionを変更する"""
        region = argLine
        self.region = region

    #
    #  commandMessage
    def commandMessage(self, argLine):
        """ messageを取得し、表示する"""
        if argLine[-1] == "\n":
            argLine = argLine[:-1]
        self.label_message.set_text(argLine)

    #
    #  commandAddMessage
    def commandAddMessage(self, argLine):
        """ 追加するmessageを取得し、追加表示する"""
        text = self.label_message.get_text()
        if argLine[-1] == "\n":
            argLine = argLine[:-1]
        newText = text + "\n" + argLine
        self.label_message.set_text(newText)

    #
    #  commandStlDir
    def commandStlDir(self, argLine):
        """ stlDirを設定する"""
        try:
            if argLine[0] != "/":
                #相対path
                self.relativeStlDir = argLine
            else:
                #絶対path
                self.relativeStlDir = os.path.relpath(argLine, self.caseDir)
        except:
            self.relativeStlDir = "model"
        stlDir = self.caseDir + "/" + self.relativeStlDir
        self.stlDir = os.path.normpath(stlDir)

    #
    #  commandTime
    def commandTime(self, argLine):
        """ timeFolderを設定する。"""
        case = pyTreeFoam.case(self.caseDir)
        if argLine == "startFrom" or argLine == "currTime":
            self.timeFolder = case.getCurrTimeFolder()
            return
        elif argLine == "firstTime":
            timeFolders = case.getTimeFolders()
            self.timeFolder = timeFolders[0]
            return
        elif argLine == "startTime":
            contDict = case.getControlDict()
            self.timeFolder = contDict["startTime"]
            return
        elif argLine == "latestTime":
            timeFolders = case.getTimeFolders()
            self.timeFolder = timeFolders[-1]
            return
        try:
            a = float(argLine)
            self.timeFolder = argLine
        except:
            pass

    #
    #  commandOutline
    def commandOutline(self, argLine):
        """ outlineの表示を制御"""
        if argLine == "show":
            self.check_showOutline.set_active(True)
        elif argLine == "hide":
            self.check_showOutline.set_active(False)

    #
    #  commandEdge
    def commandEdge(self, argLine):
        """ edgeの表示を制御"""
        if argLine == "show":
            self.check_showEdge.set_active(True)
        elif argLine == "hide":
            self.check_showEdge.set_active(False)

    #
    #  commandFace
    def commandFace(self, argLine):
        """ faceの表示を制御"""
        if argLine == "both":
            self.radio_showBoth.set_active(True)
        elif argLine == "front":
            self.radio_showFront.set_active(True)
        elif argLine == "back":
            self.radio_showBack.set_active(True)

    #
    #  commandResetCamera
    def commandResetCamera(self, argLine):
        """ resetCameraを実行する。"""
        self.resetCamera()

    #  writeStatus
    def writeStatus(self, line):
        """ status行に表示させる"""
        print(line)
        line = line.split("\n")[0]
        if len(line) > 97:
            line = line[:97] + "..."
        self.label_status.set_text(line)

    #  getCommandArgsActors
    def getCommandArgsActors(self, args):
        """ commandのargs内のactorを翻訳して取得する。"""
        region = self.region
        actorDict = {
            "internalMesh":self.internalActorDict[region],
            "patch":    self.patchActorDict[region],
            "cellZone": self.cellZoneActorDict[region],
            "faceZone": self.faceZoneActorDict[region],
            "pointZone":self.pointZoneActorDict[region],
            "cellSet":  self.cellSetActorDict[region],
            "faceSet":  self.faceSetActorDict[region],
            "pointSet": self.pointSetActorDict[region],
            "stl":      self.stlActorDict
            }
        actors = []
        for arg in args:
            words = arg.split(":")
            item = words[0]
            if item in actorDict.keys():
                getDict = actorDict[item]
                name = words[1]
                if item == "stl":
                    if name[:-4] != ".stl":
                        name += ".stl"
                if name in getDict.keys():
                    actor = getDict[name]
                    itemActor = [item, actor]
                    actors.append(itemActor)
            elif item == "region":
                name = words[1]
                for actor in self.patchActorDict[name].values():
                    itemActor = ["patch", actor]
                    actors.append(itemActor)
        return actors

    #
    #  setPropertyOfActor
    def setPropertyOfActor(self, actor, item):
        """ actorのpropertyを設定"""
        if item == "outline":
            return
        #edge表示
        prop = actor.GetProperty()
        if self.check_showEdge.get_active() == True:
            prop.EdgeVisibilityOn()
        else:
            prop.EdgeVisibilityOff()
        #表裏表示の設定
        if self.radio_showBoth.get_active() == True:
            #両面
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOff()
        elif self.radio_showFront.get_active() == True:
            #表面
            prop.BackfaceCullingOn()
            prop.FrontfaceCullingOff()
        elif self.radio_showBack.get_active() == True:
            #裏面
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOn()
        return actor

    #----------------------
    #  actor取得関連
    #----------------------
    #
    #  iniGetAllActors
    #---------------
    def iniGetAllActors(self):
        """ 全actor取得関連の定義"""
        self.iniGetPatchesZonesActors()
        self.iniGetSetsActorsAllRegions()

    #
    #  iniGetPatchesZonesActors
    def iniGetPatchesZonesActors(self):
        """ internalActor, patchActor, zoneActor取得関連の定義"""
        if self.threadCancelFlag == True:
            return
        #patch, zonesの初期化定義
        print("  setting vtkOpenFOAMReader...")
        contFile = self.caseDir + "/system/controlDict"
        self.readerOF = vtk.vtkOpenFOAMReader()
        self.readerOF.SetFileName(contFile)
        #clip関連
        self.readerOF.DecomposePolyhedraOn()            #clipの為追加
        #self.planeWidget = vtk.vtkImplicitPlaneWidget()              #outline
        self.planeWidget.SetInteractor( self.vtkObj )
        #  patch読込
        if self.threadCancelFlag == True:
            return
        print("  setting internalMesh, patches...")
        #  timeFolderをチェック
        try:
            #timeFolderが存在する場合
            floatTime = float(self.timeFolder)
            self.readerOF.UpdateTimeStep(floatTime)
        except:
            #timeFolderが存在しない場合
            self.readerOF.Update()
        #  設定
        self.readerOF.EnableAllPatchArrays()
        self.readerOF.SetReadZones(True)
        if self.threadCancelFlag == True:
            return
        #  zones読込
        print("  setting zones...")
        self.readerOF.Update()

    #
    #  iniGetSetsActorsAllRegions
    def iniGetSetsActorsAllRegions(self):
        """ 全regionのsetActor取得関連の定義"""
        if self.threadCancelFlag == True:
            return
        caseDir = self.caseDir
        timeFolder = self.timeFolder
        self.vtkFilesDict = {}
        print("  setting sets...")
        for region in self.allRegions:
            createSetsVtk.createAllSetsVtkFiles(caseDir, timeFolder, region, checkFlag=True)
            if self.threadCancelFlag == True:
                break
            folderDir = createSetsVtk.getMeshDir(caseDir, timeFolder, region)
            vtkFiles = [[], [], []]
            vtkDir = folderDir + "/VTK"
            files = glob.glob(vtkDir + "/*")
            for fileName in files:
                name = os.path.basename(fileName)
                setType = name.split("_")[0]
                if setType == "cellSet":
                    vtkFiles[0].append(fileName)
                elif setType == "faceSet":
                    vtkFiles[1].append(fileName)
                elif setType == "pointSet":
                    vtkFiles[2].append(fileName)
            self.vtkFilesDict[region] = vtkFiles
        return

    #  checkItemVtkFile
    def checkItemVtkFile(self, vtkFilePtn):
        """ vtkFileをチェックし、存在すれば削除する。"""
        delFiles = glob.glob(vtkFilePtn)
        if len(delFiles) > 0:
            for delFile in delFiles:
                os.remove(delFile)

    #
    #  getPatchesZonesActorsAllRegions
    #----------------------------------
    def getPatchesZonesActorsAllRegions(self):
        """ 全regionのpatchとzonesのactorを辞書で取得
        region名を取得するので、self.allRegionsを作成"""
        if self.threadCancelFlag == True:
            return
        #全blockDataを取得
        blockDataDict = {}
        self.internalMeshDict = {}
        output = self.readerOF.GetOutput()
        blockData = None
        #region名とdataを取得
        metaData = output.GetMetaData(0)
        blockName = self.getBlockName(metaData)
        if blockName == "internalMesh":
            blockData = output
            blockDataDict["."] = blockData
        #film？（OF-11対応）
        elif blockName == "film":
            blockData = output.GetBlock(0)
            blockDataDict["film"] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData
        #普通のmultiRegion？
        #else:
        elif blockName == "defaultRegion":
            blockData = output.GetBlock(0)
            blockDataDict["."] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData
        #constant/polyMesh無しのmultiRegion？
        else:
            blockData = output.GetBlock(0)
            blockDataDict[blockName] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData

        #region名を取得
        self.allRegions = list(blockDataDict.keys())
        #region毎にactorを取得
        for region in self.allRegions:
            blockData = blockDataDict[region]
            if blockData != None:
                #actorを取得
                (patchActorDicts, zoneActorDicts, 
                internalActor, internalBlock) = self.getRegionData(blockData)
                #region毎にactor辞書を保存
                self.internalMeshDict[region] = internalBlock
                self.internalActorDict[region] = internalActor
                self.patchActorDict[region]   = patchActorDicts[0]
                self.outlineActorDict[region] = patchActorDicts[1]
                self.cellZoneActorDict[region]  = zoneActorDicts[0]
                self.faceZoneActorDict[region]  = zoneActorDicts[1]
                self.pointZoneActorDict[region] = zoneActorDicts[2]
            else:
                self.internalMeshDict[region] = {}
                self.internalActorDict[region] = {}
                self.patchActorDict[region] = {}
                self.outlineActorDict[region] = {}
                self.cellZoneActorDict[region] = {}
                self.faceZoneActorDict[region] = {}
                self.pointZoneActorDict[region] = {}
        #self.region"."が、self.allRegionsに含まれているか？（OF-11対応）
        if not self.region in self.allRegions:
            #含まれていない場合、修正
            self.region = self.allRegions[0]
        return

    #
    #  getBlockDataForReading
    def getBlockDataForReading(self):
        """ 最初のblockを取得（region有無で変わるため）"""
        output = self.readerOF.GetOutput()
        blockData = None
        if self.region == ".":
            #region0を取得
            metaData = output.GetMetaData(0)
            blockName = self.getBlockName(metaData)
            if blockName == "internalMesh":
                blockData = output
            else:
                blockData = output.GetBlock(0)
        else:
            #regionのblockDataを取得
            nBlocks = output.GetNumberOfBlocks()
            for i in range(nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                if blockName == self.region:
                    blockData = output.GetBlock(i)
                    break
        return blockData

    #
    #  getBlockName
    def getBlockName(self, metaData):
        """ metaDataからそのblockのblockNameを取得する"""
        lines = str(metaData).split("\n")
        blockName = ""
        for line in lines:
            words = line.split()
            if words[0] == "NAME:":
                blockName = words[1]
                break
        return blockName

    #
    #  getRegionData
    def getRegionData(self, regionBlock):
        """ regionのblockDataからactorDict,internalBlockを取得する"""
        nBlocks = regionBlock.GetNumberOfBlocks()
        #blockDataからpatchData,internalMeshを取得
        patchActorDict = {}; edgeActorDict = {}; internalActor = ""
        flag = [0, 0]
        for i in range(nBlocks):
            metaData = regionBlock.GetMetaData(i)
            blockName = self.getBlockName(metaData)
            #vtk 9.1.0から名称変更
            #           9.0.3以前               9.1.0以降
            if blockName == "Patches" or blockName == "boundary":
                patchBlock = regionBlock.GetBlock(i)
                patchNamesIds = self.getPatchNamesIds(patchBlock)
                patchActorDict, edgeActorDict = self.createPatchActors(regionBlock, patchNamesIds)
                #break
                flag[0] = 1
            elif blockName == "internalMesh":
                internalBlock = regionBlock.GetBlock(i)
                internalActor = self.createInternalActor(internalBlock)
                flag[1] = 1
            if sum(flag) == 2:
                break
        #blockDataからzoneDataを取得
        zoneActorDicts = [{}, {}, {}]
        for i in range(nBlocks):
            metaData = regionBlock.GetMetaData(i)
            blockName = self.getBlockName(metaData)
            #vtk 9.1.0から名称変更
            #           9.0.3以前               9.1.0以降
            if blockName == "Zones" or blockName == "zones":
                zoneBlock = regionBlock.GetBlock(i)
                zoneActorDicts = self.createAllZoneActors(zoneBlock)
                break
        patchActorDicts = [patchActorDict, edgeActorDict]
        return (patchActorDicts, zoneActorDicts, 
                internalActor, internalBlock)

    #  getPatchNamesIds
    def getPatchNamesIds(self, patchBlock):
        """ blockDataからpatchNameとpatchIdを取得する"""
        patchIds = []
        patchNames = []
        index = 3
        nBlocks = patchBlock.GetNumberOfBlocks()
        for i in range(nBlocks):
            metaData = patchBlock.GetMetaData(i)
            patchName = self.getBlockName(metaData)
            patchIds.append(index)
            patchNames.append(patchName)
            index += 1
        return patchNames, patchIds

    #  createPatchActors
    def createPatchActors(self, regionBlock, patchNamesIds):
        """ patchActor, outlineActor(featureEdge)を作成する"""
        (patchNames, patchIds) = patchNamesIds
        patchActorDict = {}
        outlineActorDict = {}
        for i in range(len(patchIds)):
            index = patchIds[i]
            extract_block = vtk.vtkExtractBlock()
            extract_block.SetInputData(regionBlock)
            extract_block.AddIndex(index)
            extract_block.Update()
            #filter
            geom_filter = vtk.vtkGeometryFilter()
            geom_filter.SetInputConnection(extract_block.GetOutputPort())
            #featureEdge
            feature_edge = vtk.vtkFeatureEdges()
            feature_edge.SetInputConnection(geom_filter.GetOutputPort())
            #mapper
            mapper = vtk.vtkCompositePolyDataMapper()
            mapper.SetInputConnection(geom_filter.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(feature_edge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            prop = actor.GetProperty()
            prop.SetColor(colors.green)
            name = patchNames[i]
            patchActorDict[name] = actor
            edgeActor = vtk.vtkActor()
            edgeActor.SetMapper(edgeMapper)
            outlineActorDict[name] = edgeActor
        return patchActorDict, outlineActorDict

    #  createInternalActor
    def createInternalActor(self, internalBlock):
        """ internalActor（internalMesh）を作成する。"""
        #internalBlockは、unstructuredGridのため、直接mapperにセット
        #mapper
        mapper = vtk.vtkDataSetMapper()
        mapper.SetInputData(internalBlock)
        mapper.ScalarVisibilityOff()
        #actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper)
        prop = actor.GetProperty()
        prop.SetColor(colors.green)
        return actor

    #  createInitialClip
    def createInitialClip(self, region):
        """ 初期のclipMeshを作成する"""

        def recalcBounds(bnds):
            """ boundsに長辺の10%のmerginを追加する"""
            lx = (bnds[1] - bnds[0])
            ly = (bnds[3] - bnds[2])
            lz = (bnds[5] - bnds[4])
            l = max([lx, ly, lz])
            dl = l * 0.1
            bounds = (bnds[0]-dl, bnds[1]+dl,
                      bnds[2]-dl, bnds[3]+dl,
                      bnds[4]-dl, bnds[5]+dl)
            return bounds

        #normal, bounds, origin, invertを設定
        #normal = [0, 1, 0]      #y方向
        #invert = False
        normal = self.default_normal      #y方向
        invert = self.default_invert
        bnds = self.internalActorDict[region].GetBounds()
        bounds = recalcBounds(bnds)
        origin = self.getOriginFromRegion(region)
        #clipPlaneDictを設定
        planeDict = {}
        planeDict["normal"] = normal
        planeDict["bounds"] = bounds
        planeDict["origin"] = origin
        planeDict["invert"] = invert
        self.clipPlaneDict[region] = planeDict
        #clipActorを作成
        internalMesh = self.internalMeshDict[region]
        plane = vtk.vtkPlane()
        plane.SetNormal(normal)
        plane.SetOrigin(origin)
        #alg = vtk.vtkClipDataSet()             #triangleができる
        alg = vtk.vtkTableBasedClipDataSet()    #triangleを削除するclip
        #alg.SetInputConnection(blockData)
        alg.SetInputData(internalMesh)          #internalMeshを直接セット
        alg.SetClipFunction(plane)
        alg.SetInsideOut(invert)
        alg.Update()
        filter = vtk.vtkGeometryFilter()
        filter.SetInputConnection(alg.GetOutputPort())
        filter.Update()
        # mapper
        mapper = vtk.vtkCompositePolyDataMapper2()
        mapper.SetInputConnection(filter.GetOutputPort()) #mapperにfilterを設定
        mapper.SetScalarModeToUseCellFieldData() #scalarデータ用に設定
        # actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper) 
        prop = actor.GetProperty()            #actorにmapperを設定
        prop.SetColor(colors.green)
        self.clippedMeshActorDict[region] = actor
        return actor

    #  createNewClipFromPlaneWidget
    def createNewClipFromPlaneWidget(self, region):
        """ planeWidgetで設定されたplaneでclipする"""
        #clipActorを作成
        internalMesh = self.internalMeshDict[region]
        normal = self.clipPlaneDict[region]["normal"]
        origin = self.clipPlaneDict[region]["origin"]
        invert = self.clipPlaneDict[region]["invert"]
        plane = vtk.vtkPlane()
        plane.SetNormal(normal)
        plane.SetOrigin(origin)
        alg = vtk.vtkTableBasedClipDataSet()    #triangleを削除するclip
        #alg.SetInputConnection(blockData)
        alg.SetInputData(internalMesh)          #internalMeshを直接セット
        alg.SetClipFunction(plane)
        alg.SetInsideOut(invert)
        alg.Update()
        filter = vtk.vtkGeometryFilter()
        filter.SetInputConnection(alg.GetOutputPort())
        filter.Update()
        # mapper
        mapper = vtk.vtkCompositePolyDataMapper2()
        mapper.SetInputConnection(filter.GetOutputPort()) #mapperにfilterを設定
        mapper.SetScalarModeToUseCellFieldData() #scalarデータ用に設定
        # actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper) 
        prop = actor.GetProperty()            #actorにmapperを設定
        prop.SetColor(colors.green)
        self.clippedMeshActorDict[region] = actor
        return actor

    #  createAllZoneActors
    def createAllZoneActors(self, zoneBlock):
        """ 全zoneのactorを取得"""
        cellZoneActorDict = {}
        faceZoneActorDict = {}
        pointZoneActorDict = {}
        nBlocks = zoneBlock.GetNumberOfBlocks()
        for i in range(nBlocks):
            metaData = zoneBlock.GetMetaData(i)
            zoneName = self.getBlockName(metaData)
            if zoneName == "cellZones":
                block = zoneBlock.GetBlock(i)
                cellZoneNamesIds = self.getZoneNamesIds(block)
                cellZoneActorDict = self.createCellActors(block, cellZoneNamesIds)
            elif zoneName == "faceZones":
                block = zoneBlock.GetBlock(i)
                faceZoneNamesIds = self.getZoneNamesIds(block)
                faceZoneActorDict = self.createZoneActors(block, faceZoneNamesIds)
            elif zoneName == "pointZones":
                block = zoneBlock.GetBlock(i)
                pointZoneNamesIds = self.getZoneNamesIds(block)
                pointZoneActorDict = self.createZoneActors(block, pointZoneNamesIds)
        actorDicts = (cellZoneActorDict, faceZoneActorDict, pointZoneActorDict)
        return actorDicts

    #  createCellActors
    def createCellActors(self, block, cellZoneNamesIds):
        """ 表面のみのcellZoneActorを作成する。"""
        zoneNames, zoneIds = cellZoneNamesIds
        cellActorDict = {}
        for i in range(len(zoneIds)):
            index = zoneIds[i]
            extract_block = vtk.vtkExtractBlock()
            extract_block.SetInputData(block)
            extract_block.AddIndex(index)
            extract_block.Update()
            #filter
            #geom_filter = vtk.vtkGeometryFilter()
            geom_filter = vtk.vtkDataSetSurfaceFilter()
            geom_filter.SetInputConnection(extract_block.GetOutputPort())
            #mapper
            mapper = vtk.vtkCompositePolyDataMapper()
            mapper.SetInputConnection(geom_filter.GetOutputPort())
            mapper.SetScalarModeToUseCellFieldData()
            mapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            prop = actor.GetProperty()
            prop.SetColor(colors.blue)
            name = zoneNames[i]
            cellActorDict[name] = actor
        return cellActorDict

    #
    #  createZoneActors
    def createZoneActors(self, block, zoneNamesIds):
        """ faceとpointのzoneActorを作成する"""
        zoneNames, zoneIds = zoneNamesIds
        zoneActorDict = {}
        for i in range(len(zoneIds)):
            index = zoneIds[i]
            extract_block = vtk.vtkExtractBlock()
            extract_block.SetInputData(block)
            extract_block.AddIndex(index)
            extract_block.Update()
            #filter
            geom_filter = vtk.vtkGeometryFilter()
            geom_filter.SetInputConnection(extract_block.GetOutputPort())
            #mapper
            mapper = vtk.vtkCompositePolyDataMapper()
            mapper.SetInputConnection(geom_filter.GetOutputPort())
            mapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            name = zoneNames[i]
            prop = actor.GetProperty()
            prop.SetColor(colors.blue)
            prop.SetPointSize(8)
            #prop.BackfaceCullingOn()
            #prop.SetOpacity(0.5)
            zoneActorDict[name] = actor
        return zoneActorDict

    #  getZoneNamesIds
    def getZoneNamesIds(self, block):
        """ blockDataからzoneNameとzoneIdを取得する。
        要素数が「0」のzoneは、省く。"""
        zoneNames = []
        zoneIds = []
        index = 1
        nBlocks = block.GetNumberOfBlocks()
        for i in range(nBlocks):
            subBlock = block.GetBlock(i)
            if subBlock.GetNumberOfCells() != 0:
                #zoneのcell数が「0」以外を取得
                metaData = block.GetMetaData(i)
                zoneName = self.getBlockName(metaData)
                zoneNames.append(zoneName)
                zoneIds.append(index)
            index += 1
        return zoneNames, zoneIds

    #
    #  getSetsActorsAllRegions
    #--------------------------
    def getSetsActorsAllRegions(self):
        """ 全regionのsetsのactorを辞書で取得する"""
        if self.threadCancelFlag == True:
            return
        for region in self.allRegions:
            if region in self.vtkFilesDict.keys():
                vtkFiles = self.vtkFilesDict[region]
                self.cellSetActorDict[region] = self.createCellVtkActors(vtkFiles[0])
                self.faceSetActorDict[region] = self.createFaceVtkActors(vtkFiles[1])
                self.pointSetActorDict[region] = self.createPointVtkActors(vtkFiles[2])

    #
    #  getPatchTypesAllRegions
    #---------------------------
    def getPatchTypesAllRegions(self):
        """ 全regionのpatchTypeを取得する"""

        def getPatchType(cont):
            contOp = pyTreeFoam.strOp(cont)
            line, p, kind = contOp.get1line(0)
            words = line.split()
            while line != "" and words[0] != "type":
                line, p, kind = contOp(p)
                words = line.split()
            if words[0] == "type":
                if words[1][-1] == ";":
                    patchType = words[1][:-1]
                else:
                    patchType = words[1]
            else:
                patchType = ""
            return patchType

        if self.threadCancelFlag == True:
            return
        self.patchTypesDict = {}
        folderType = pyTreeFoam.getFolderType(self.caseDir)
        if not (folderType == "case" or folderType == "caseLink"):
            return
        case = pyTreeFoam.case(self.caseDir)
        flag, loc, regNames = case.isMultiRegion()
        if len(self.allRegions) == 0:
            regions = ["."]
        else:
            regions = self.allRegions
        for region in regions:
            if region == ".":
                #currTime = case.getCurrTimeFolder()
                currTime = self.timeFolder
                relMeshDir = case.getCurrMeshDir(currTime, region, "boundary")
                meshDir = self.caseDir + "/" + relMeshDir
            else:    
                meshDir = self.caseDir + "/" + loc + "/" + region
            results = case.getPatchNameFromBoundary(meshDir)
            if len(results) > 0:
                patchDict = {}
                for result in results:
                    patchName = result[0]
                    patchType = getPatchType(result[1])
                    if patchType != "":                    
                        patchDict[patchName] = patchType
                self.patchTypesDict[region] = patchDict

    #
    #  getStlActors
    #---------------
    def getStlActors(self):
        """ stlのactorの辞書を作成する。"""
        if self.threadCancelFlag == True:
            return
        print("  setting stl files...")
        actorDict = {}
        stlFiles = glob.glob(self.stlDir + "/*.stl")
        for stlFile in stlFiles:
            reader = vtk.vtkSTLReader()
            reader.SetFileName(stlFile)
            reader.Update()
            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputConnection(reader.GetOutputPort())
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            name = os.path.basename(stlFile)
            actorDict[name] = actor
        self.stlActorDict = actorDict

    #  createCellVtkActors
    def createCellVtkActors(self, fileNames):
        """ cellSetのactorを作成"""
        actorDict = {}
        for fileName in fileNames:
            reader = vtk.vtkUnstructuredGridReader()
            reader.SetFileName(fileName)
            reader.Update()
            output = reader.GetOutput()
            mapper = vtk.vtkDataSetMapper()
            mapper.SetInputData(output)
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            prop = actor.GetProperty()
            #prop.SetColor(colors.blue_light)
            prop.SetColor(colors.pink)
            name = self.getSetNameFromFile(fileName)
            actorDict[name] = actor
        return actorDict

    #  getSetNameFromFile
    def getSetNameFromFile(self, fileName):
        """ fileNameからsetNameを取得"""
        name = os.path.basename(fileName)
        #name = "_".join(name.split("_")[:-1])
        name = "_".join(name.split("_")[1:])
        name = name[:-4]
        return name

    #  createFaceVtkActors
    def createFaceVtkActors(self, fileNames):
        """ faceSetのactorを作成"""
        actorDict = {}
        for fileName in fileNames:
            #reader = vtk.vtkPolyDataReader()
            reader = vtk.vtkUnstructuredGridReader()
            reader.SetFileName(fileName)
            reader.Update()
            output = reader.GetOutput()
            mapper = vtk.vtkDataSetMapper()
            mapper.SetInputData(output)
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            prop = actor.GetProperty()
            #prop.SetColor(colors.blue_light)
            prop.SetColor(colors.pink)
            name = self.getSetNameFromFile(fileName)
            actorDict[name] = actor
        return actorDict

    #  createPointVtkActors
    def createPointVtkActors(self, fileNames):
        """ pointSetのactorを作成
        vtk-9.1.0対応で、GlyphFilterを追加。
        pointのみの場合は、vtkVertexGlyphFilterを通さないと
        表示されない。"""
        actorDict = {}
        for fileName in fileNames:
            reader = vtk.vtkPolyDataReader()
            reader.SetFileName(fileName)
            reader.Update()
            output = reader.GetOutput()
            glyFilter = vtk.vtkVertexGlyphFilter()
            glyFilter.SetInputData(output)
            #mapper = vtk.vtkPointGaussianMapper()
            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputConnection(glyFilter.GetOutputPort())
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            prop = actor.GetProperty()
            #prop.SetColor(colors.blue_light)
            prop.SetColor(colors.pink)
            prop.SetPointSize(8)
            name = self.getSetNameFromFile(fileName)
            actorDict[name] = actor
        return actorDict

    #------- universalDialogs ---------------------
    #  okDialog
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()
    
    #errDialog
    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()


#
#  getArguments
#---------------
def getArguments():
    """ 引数を取得する。"""
    caseDir = os.getcwd()
    timeFolder = "0"
    region = "."
    inputFile = ""
    title = "meshViewer"
    i = 1
    while i < len(sys.argv):
        arg = sys.argv[i]
        if arg == "-case":
            i += 1
            caseDir = sys.argv[i]
        elif arg == "-time":
            i += 1
            timeFolder = sys.argv[i]
        elif arg == "-region":
            i += 1
            region = sys.argv[i]
        elif arg == "-input":
            i += 1
            inputFile = sys.argv[i]
        elif arg == "-title":
            i += 1
            title = sys.argv[i]
        i += 1
    return caseDir, timeFolder, region, inputFile, title


if __name__ == "__main__":
    import gettext
    import locale
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    locale.bindtextdomain("treefoam", localeDir)
    gettext.install("treefoam", localeDir)
    #gettext.install("app")

    #引数を取得
    (caseDir, timeFolder, region,
     inputFile, title) = getArguments()
    #GUIを表示
    winApp = windowApp(caseDir, timeFolder, region,
                        inputFile, title)
    winApp.main()
