#!/usr/bin/python3
#  coding: utf-8
#
#       couplingToFistrDialog.py
#
#       FrontISTRと連成解析する。
#
#   23/02/20    新規作成
#      03/13    clearResults:並列処理の場合を検出していなかったので追加
#      03/21    readCouplingDataFile:solidAnaComm, easyistrの読込を
#               configTreeFoamからに変更。
#               writeCouplingDataFile:solidAnaCommは、configTreeFoamと
#               couplingDataに書込。easyistrは、configTreeFoamのみに変更。
#      03/31    setInitialSettingGUI:easystr, fistr1のpathの設定を
#               configTreeFoamからに修正。
#      05/20    cht, chtssを追加
#      06/13    replaceCouplingPatchConts:patchContを名を修正。
#      07/02    patchContの置き換えは、single, parallelとも「0」フォルダに限定
#               置き換え時、value行は、元の内容を使う。
#      07/15    refEasyistr,referFistr1:selNameを設定し、選択表示する様に修正
#      07/23    referSolidCaseDir:selNameを設定し、選択表示する様に修正
#      07/27    getSetting:nThreadsの取得方法を修正。（バグ）
#               chekを外してもtextBox内の値を取得していた為
#      08/01    getPatchFileDict:OF10用のpatchContFileを追加
#               OF-10用のtutorialsを追加
#      08/17    setInitialSettingGUI,setCoupligFileSetting:fsi,cht,chtssの
#               checkに応じてpFieldの定数、comboBoxの非表示設定を追加
#      08/27    changeCoupling:連成内容によるlabelの表示、非表示を追加
#      10/11    codedをpatch内からcontrolDictに移動。
#               getSetting:pointDisplacement,T fieldのpatchTypeの確認を追加
#      10/31    getSetting:scalarFieldの設定に「p,T」を追加。
#      11/24    getSetting:Fistr側の並列処理チェックを追加
#      11/29    runEasyistr:選択した項目をcouplingDictに保存する様に修正
#      11/30    runCoupling:diolog表示内容を修正。
#   24/01/26    replaceCouplingToFistr:置き換えがうまくできない事がある為、修正
#      01/27    getSetting:連成項目を選択せずに「設定取得」すると、エラー発生。
#               エラー発生しないよう修正
#      07/29    openFileDialog:universalDialogs内の関数名変更による修正
#               openFolderDialog:universalDialogs内の関数名変更による修正
#               universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import os, sys
import shutil
import glob
import pyTreeFoam
import GtkParts
import selectEditStructureCaseDDialog as selectEditStructureCase
import universalDialogs as unvDlg
import pyFistr_ptc as pyFistr
import pyCoupling

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()


#------------------
#  windowApp class
#------------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"couplingToFistrDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        self.mainWindow.set_title(_("流体-構造の連成解析"))
        #GUIのobect名を設定
        self.setGtkObject()
        self.setToolTips()
        #属性の設定
        self.caseDir = caseDir
        self.fields = []
        self.patches = []
        self.sgrps = []
        #GUIの設定
        self.setInitialSettingGUI()
        #coupling_FrontISTRの有無を確認、無ければ作成
        self.checkFolderDir()
        #couplingFileの内容を設定
        self.couplingDataDict = readCouplingDataFile(self.caseDir)
        self.setCoupligFileSetting()


    #
    #  setGtkObject
    #----------------
    def setGtkObject(self):
        """ GUIのobject名を設定する"""
        self.label_caseDir = self.builder.get_object("label_caseDir")
        self.label_fsiConts = self.builder.get_object("label_fsiConts")
        self.label_pCoeff = self.builder.get_object("label_pCoeff")
        self.label_pShift = self.builder.get_object("label_pShift")
        self.label_scalarField = self.builder.get_object("label_scalarField")
        self.label_pointField = self.builder.get_object("label_pointField")
        self.label_nStepsFsi = self.builder.get_object("label_nStepsFsi")
        self.label_heatConts = self.builder.get_object("label_heatConts")
        self.label_nStepsHeat = self.builder.get_object("label_nStepsHeat")
        self.entry_coeff = self.builder.get_object("entry_coeff")
        self.entry_shift = self.builder.get_object("entry_shift")
        self.entry_easyistr = self.builder.get_object("entry_easyistr")
        self.entry_fistr1 = self.builder.get_object("entry_fistr1")
        self.entry_solidCaseDir = self.builder.get_object("entry_solidCaseDir")
        self.entry_nOF = self.builder.get_object("entry_nOF")
        self.entry_nFistr = self.builder.get_object("entry_nFistr")
        self.entry_nThreads = self.builder.get_object("entry_nThreads")
        #self.entry_buffer = self.builder.get_object("entry_buffer")
        self.entry_maxWaitTime = self.builder.get_object("entry_maxWaitTime")
        self.entry_nOF = self.builder.get_object("entry_nOF")
        self.entry_nFistr = self.builder.get_object("entry_nFistr")
        combo_fields = self.builder.get_object("combo_fields")
        self.comboFields = GtkParts.comboBoxText(combo_fields)
        combo_pointField = self.builder.get_object("combo_pointField")
        self.comboPointField = GtkParts.comboBoxText(combo_pointField)
        tree_patches = self.builder.get_object("tree_patches")
        self.treePatches = GtkParts.treeList(tree_patches)
        self.treePatches.create()
        tree_sgrps = self.builder.get_object("tree_sgrps")
        self.treeSgrps = GtkParts.treeList(tree_sgrps)
        self.treeSgrps.create()
        self.check_fsi = self.builder.get_object("check_fsi")
        self.check_cht = self.builder.get_object("check_cht")
        self.check_chtss = self.builder.get_object("check_chtss")
        self.check_parallel = self.builder.get_object("check_parallel")
        self.check_thread = self.builder.get_object("check_thread")
        self.button_refFistr1 = self.builder.get_object("button_refFistr1")
        self.button_refSolidCaseDir = self.builder.get_object("button_refSolidCaseDir")
        self.button_clearResults = self.builder.get_object("button_clearResults")
        self.button_showFistrLog = self.builder.get_object("button_showFistrLog")
        self.button_getSetting = self.builder.get_object("button_getSetting")
        self.button_runCoupling = self.builder.get_object("button_runCoupling")
        self.button_close = self.builder.get_object("button_close")
        self.radio_concurr = self.builder.get_object("radio_concurr")
        self.radio_altn = self.builder.get_object("radio_altn")

    #
    #  setToolTips
    #---------------
    def setToolTips(self):
        """ toolTipを設定する"""
        clearResults = _("OpenFOAM, FrontISTRの計算結果folder, fileを削除する。")
        runShowLog = _("FrontISTRの計算logを表示する。")
        getSetting = _("入力した設定を保存\n OpenFOAMのpatch内容を連成用に修正する。")
        runCoupling = _("流体-構造の連成計算を開始する。")
        self.button_clearResults.props.tooltip_text = clearResults
        self.button_showFistrLog.props.tooltip_text = runShowLog
        self.button_getSetting.props.tooltip_text = getSetting
        self.button_runCoupling.props.tooltip_text = runCoupling

    #
    #  setInitialsettingGUI
    #--------------------
    def setInitialSettingGUI(self):
        """ GUIの初期設定"""
        OF = openFoam(caseDir)
        self.fields = OF.fields()
        if not ("p" in self.fields):
            self.fields = ["p"] + self.fields
        if "T" in self.fields:
            self.fields = ["p,T"] + self.fields
        self.patches = OF.patches()
        #基本dirを設定
        showDir = caseDir
        if len(showDir) > 80:
            showDir = "..." + showDir[-77:]
        self.label_caseDir.set_text(showDir)
        #checkBoxを設定
        if "T" in self.fields:
            self.check_cht.set_sensitive(True)
        else:
            self.check_cht.set_active(False)
            self.check_cht.set_sensitive(False)
        if "pointDisplacement" in self.fields:
            self.check_fsi.set_sensitive(True)
        else:
            self.check_fsi.set_active(False)
            self.check_fsi.set_sensitive(False)
        if "T" in self.fields and "pointDisplacement" in self.fields:
            self.check_chtss.set_sensitive(True)
        else:
            self.check_chtss.set_active(False)
            self.check_chtss.set_sensitive(False)
        #OpenFOAMの内容を取得
        self.entry_coeff.set_text("1.0")
        self.entry_shift.set_text("0.0")
        self.comboFields.setItems([""] + self.fields)
        if "p" in self.fields:
            self.comboFields.selectItem("p")
        self.comboPointField.setItems([""] + self.fields)
        if "pointDisplacement" in self.fields:
            self.comboPointField.selectItem("pointDisplacement")
        self.treePatches.setItems(self.patches)
        self.entry_nOF.set_text("1")
        #FrontISTR側の設定
        configDict = pyTreeFoam.readConfigTreeFoam()
        easyistr = "easyistr"
        if configDict["easyistr"] != "":
            easyistr = configDict["easyistr"]
        fistr1 = "fistr1"
        if configDict["fistr1"] != "":
            fistr1 = configDict["fistr1"]
        self.entry_easyistr.set_text(easyistr)
        self.entry_fistr1.set_text(fistr1)
        self.entry_nFistr.set_text("1")
        self.entry_nThreads.set_text("1")
        #bufferの設定
        #self.entry_buffer.set_text("10")
        self.entry_maxWaitTime.set_text("60")

    #
    #  checkFolderDir
    #------------------
    def checkFolderDir(self):
        """ folderDirの有無を確認し、無ければ作成する"""
        folderDir = self.caseDir + "/coupling_FrontISTR"
        if os.path.exists(folderDir) == False:
            os.mkdir(folderDir)
        dataDir = folderDir + "/data"
        if os.path.exists(dataDir) == False:
            os.mkdir(dataDir)
        pythonDir = folderDir + "/python"
        if os.path.exists(pythonDir) == False:
            os.mkdir(pythonDir)

    #
    #  setCoupligFileSetting
    #------------------------
    def setCoupligFileSetting(self):
        """ couplingDataファイルの内容を設定する"""
        dataDict = self.couplingDataDict
        if len(dataDict.keys()) == 0:
            return
        #連成の内容
        self.check_fsi.set_active(False)
        self.check_cht.set_active(False)
        self.check_chtss.set_active(False)
        self.radio_altn.set_active(False)
        if dataDict["fsi"] == "yes" and self.check_fsi.get_sensitive() == True:
            self.check_fsi.set_active(True)
        if dataDict["cht"] == "yes" and self.check_cht.get_sensitive() == True:
            self.check_cht.set_active(True)
        if dataDict["chtss"] == "yes" and self.check_chtss.get_sensitive() == True:
            self.check_chtss.set_active(True)
        if dataDict["method"] == "altn":
            self.radio_altn.set_active(True)
        #OpenFOAM側
        if "mappingCoeff" in dataDict.keys():
            self.entry_coeff.set_text(dataDict["mappingCoeff"])
        if "mappingShift" in dataDict.keys():
            self.entry_shift.set_text(dataDict["mappingShift"])
        if "field" in dataDict.keys():
            self.comboFields.selectItem(dataDict["field"])
        if "pointField" in dataDict.keys():
            self.comboPointField.selectItem(dataDict["pointField"])
        if "mappingPatch" in dataDict.keys():
            self.treePatches.selectItems([dataDict["mappingPatch"]])
        if "nStepsOF" in dataDict.keys():
            self.entry_nOF.set_text(dataDict["nStepsOF"])
        if "nProcs" in dataDict.keys():
            if int(dataDict["nProcs"]) > 1:
                self.check_parallel.set_active(True)
            else:
                self.check_parallel.set_active(False)
        #FrontISTR側
        if "easyistr" in dataDict.keys():
            self.entry_easyistr.set_text(dataDict["easyistr"])
        if "solidAnaComm" in dataDict.keys():
            self.entry_fistr1.set_text(dataDict["solidAnaComm"])
        if "solidCaseDir" in dataDict.keys():
            self.entry_solidCaseDir.set_text(dataDict["solidCaseDir"])
            solidCaseDir = dataDict["solidCaseDir"]
            if os.path.exists(solidCaseDir) == True:
                self.sgrps = fistr(solidCaseDir).sgrps()
                self.treeSgrps.setItems(self.sgrps)
        if "mappingSgrp" in dataDict.keys():
            if dataDict["mappingSgrp"] in self.sgrps:
                self.treeSgrps.selectItems([dataDict["mappingSgrp"]])
        if "nSteps" in dataDict.keys():
            self.entry_nFistr.set_text(dataDict["nSteps"])
        if "nThreads" in dataDict.keys():
            self.entry_nThreads.set_text(dataDict["nThreads"])
            if int(dataDict["nThreads"]) > 1:
                self.check_thread.set_active(True)
            else:
                self.check_thread.set_active(False)
        #streamBuffer
        # if "streamBuffer" in dataDict.keys():
        #     bufferSize = int(float(dataDict["streamBuffer"]) / 1000.0)
        #     self.entry_buffer.set_text(str(bufferSize))
        if "maxWaitTime" in dataDict.keys():
            maxTime = int(float(dataDict["maxWaitTime"]) / 100.0)
            self.entry_maxWaitTime.set_text(str(maxTime))

    #
    #  main
    #----------
    def main(self):
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        Gtk.main_quit()

    #------------ event handler -------------------------
    #
    def onRefEasyistr(self, event):     #参照...(EasyIstr)
        self.refEasyistr()
    def onRefFistr1(self, event):       #参照...(fistr1)
        self.referFistr1()
    def onRefSolidCaseDir(self, event): #参照...(solidCaseDir)
        self.referSolidCaseDir()
    def onClose(self, event):           #閉じる
        self.close()
    def onClearResults(self, event):    #結果クリア
        self.clearResults()
    def onGetSetting(self, event):      #設定取得・保存
        self.getSetting()
    def onRunCoupling(self, event):     #連成計算開始
        self.runCoupling()
    def onShowFistrLog(self, event):    #Fistrのlog
        self.showFistrLog()
    def onRunEasyistr(self, event):     #easyIstrの起動
        self.runEasyistr()
    def onOpenFluidCase(self, event):   #流体folder開く
        self.openFluidCase()
    def onOpenSolidCase(self, event):   #個体floder開く
        self.openSolidCase()
    def onChangeCoupling(self, event):  #checkBoxを変更した時
        self.changeCoupling()

    #
    #  refEasyistr
    #--------------
    def refEasyistr(self):
        """ easyIstrの参照"""
        title = _("command「easyistr」を指定")
        selFile = self.entry_easyistr.get_text()
        path = os.path.dirname(selFile)
        if os.path.exists(path) == False:
            path = os.path.dirname(self.caseDir)
        name = "easyistr"
        funcOk = [self.refEasyistr_ok]
        self.openFileDialog(title, path, selName=name, funcOk=funcOk)
        
    #  file設定時
    def refEasyistr_ok(self, newPath):
        self.entry_easyistr.set_text(newPath)

    #
    #  referFistr1
    #--------------
    def referFistr1(self):
        """ fistr1の参照..."""
        title = _("command「fistr1」を指定")
        path = self.entry_fistr1.get_text()
        path = os.path.dirname(path)
        if os.path.exists(path) == False:
            path = os.path.dirname(self.caseDir)
        #path = os.path.dirname(self.caseDir)
        selFile = "fistr1"
        funcOk = [self.referFistr1_ok]
        self.openFileDialog(title, path, selName=selFile, funcOk=funcOk)

    #  file設定時
    def referFistr1_ok(self, newPath):
        self.entry_fistr1.set_text(newPath)

    #
    #  referSolidCaseDir
    #--------------------
    def referSolidCaseDir(self):
        """ solidCaseDirを取得"""
        title = _("FrontISTRの計算folder（workFolder）を指定")
        path = os.path.dirname(self.caseDir)
        selFolder = os.path.basename(self.entry_solidCaseDir.get_text())
        funcOk = [self.referSolidCaseDir_ok]
        self.openFolderDialog(title, path, funcOk=funcOk, selName=selFolder)

    #  folder選択時
    def referSolidCaseDir_ok(self, newPath):
        relPath = os.path.relpath(newPath, self.caseDir)
        self.entry_solidCaseDir.set_text(relPath)
        self.sgrps = fistr(newPath).sgrps()
        self.treeSgrps.setItems(self.sgrps)

    #
    #  clearResults
    #---------------
    def clearResults(self):
        """ OpenFOAM, FrontISTR両方の結果をクリアする"""
        #途中再開の場合、クリアできない
        OF = openFoam(self.caseDir)
        if self.check_parallel.get_active() == True:
            if os.path.exists(self.caseDir + "/processor0") == True:
                startTime = OF.runStartTime("processor0")
            else:
                startTime = OF.runStartTime()    
        else:
            startTime = OF.runStartTime()
        if float(startTime) > 0.0:
            title = _("エラー")
            mess = _("startTimeが「0.0」以外の設定になっている為、結果をクリアできません。")
            print(mess)
            self.errDialog(title, mess)
            return
        #OpenFOAM側----------------
        case = pyTreeFoam.case(self.caseDir)
        #case直下
        timeFolders = case.getTimeFolders()
        timeFolders.sort()
        delFolders = timeFolders[1:]
        for delFolder in delFolders:
            delDir = self.caseDir + os.sep + delFolder
            shutil.rmtree(delDir)
        #並列の場合
        folders = pyTreeFoam.getFolderNames(self.caseDir)
        for folder in folders:
            if folder[:len("processor")] == "processor":
                procFolders = case.getTimeFolders(folder)
                procFolders.sort()
                delFolders = procFolders[1:]
                for delFolder in delFolders:
                    delDir = self.caseDir + os.sep + folder + os.sep + delFolder
                    shutil.rmtree(delDir)
        #controlDict内をクリア(couplingDataをクリア)
        self.clearCouplingDataInControlDict()
        #dataFolder内をクリア
        dataFolder = self.caseDir + "/coupling_FrontISTR/data"
        files = glob.glob(dataFolder + "/*")
        for name in files:
            if os.path.isfile(name):
                os.remove(name)
            if os.path.isdir(name):
                shutil.rmtree(name)
        #FrontISTR側------------
        resFiles = ["*.res.*", "*.inp", "*.restart.*", "*.restart_*", "*.vtk",
                   "conv*.", "*.log", "*_log", "FSTR.dbg.*", "FSTR.msg",
                    "FSTR.sta", "*.pvtu", "*.vtu", "*.plt"]
        resFolders = ["*_psf.*"]
        solidCaseDir = self.entry_solidCaseDir.get_text()
        #fileを削除
        for pattern in resFiles:
            delNames = glob.glob(solidCaseDir + os.sep + pattern)
            if len(delNames) > 0:
                for delName in delNames:
                    os.remove(delName)
        #folderを削除
        for folder in resFolders:
            delNames = glob.glob(solidCaseDir + os.sep + folder)
            for delName in delNames:
                if os.path.isdir(delName) == True:
                    shutil.rmtree(delName)
        #message出力
        title = _("結果のクリア")
        mess = _("OpenFOAMとFrontISTRの計算結果folder、fileを削除しました。")
        self.okDialog(title, mess)
        print(mess)

    #
    #  clearCouplingDataInControlDict
    def clearCouplingDataInControlDict(self):
        """ controlDict内のcouplingDataを削除"""
        fileName = self.caseDir + "/system/controlDict"
        f = open(fileName); cont = f.read(); f.close()
        cont = self.clearCouplingSource(cont)
        cont = self.clearCouplingToFistr(cont)
        f = open(fileName, "w"); f.write(cont); f.close()

    #
    #  clearCouplingSource
    def clearCouplingSource(self, cont):
        """ controlDict内のfunctions.couplingSource部をクリアする"""
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.getKeywordPointer("functions", 0)
        if p < 0:
            #functionsが存在しない場合
            return cont
        #functionsが存在する場合
        ps = p
        p = contOp.getKeywordPointer("couplingSource", ps)
        if p < 0:
            #couplingSourceが存在しない場合
            return cont
        #couplingSourceが存在する場合、クリアする
        lineCont, p, kind = contOp.get1line(p)
        pp = p - len(lineCont)
        #ppをCRの前まで戻す
        pp = self.backSkipUntilCR(cont, pp)
        if cont[p+1] == "\n":
            p += 1
        newCont = cont[:pp] + cont[p:]
        return newCont

    #
    #  backSkipUntilCR
    def backSkipUntilCR(self, cont, p):
        """ 空白をskipして、pointerをCRの前まで戻す"""
        p -= 1
        while p >= 0:
            if cont[p] == " ":
                p -= 1
            elif cont[p] == "\n":
                break
            else:
                break
        return p


    #
    #  clearCouplingToFistr
    def clearCouplingToFistr(self, cont):
        """ controlDict内のcouplingToFistr部をクリアする"""
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        p = contOp.getKeywordPointer("couplingToFistr", p)
        if p >= 0:
            lineCont, p, kind = contOp.get1line(p)
            pp = p - len(lineCont)
            if cont[p+1] == "\n":
                p += 1
            newCont = cont[:pp] + cont[p:]
            #f = open(fileName, "w"); f.write(newCont); f.close()
            return newCont
        return cont

    #
    #  getSetting
    #-------------
    def getSetting(self):
        """ 連成解析の為の設定を取得する"""
        dataDict = self.couplingDataDict
        messTitle = _("エラー")
        #OpenFOAMのtimeStepをチェック
        if self.radio_concurr.get_active() == True:
            dataDict["method"] = "conc"
        else:
            dataDict["method"] = "altn"
        OF = openFoam(self.caseDir)
        contDict = OF.controlDict()
        if (contDict["adjustTimeStep"].lower() == "yes" or
            contDict["adjustTimeStep"].lower() == "true" or
            contDict["adjustTimeStep"].lower() == "on"):
            if dataDict["method"] == "conc":
                mess = _("system/controlDict内の「adjustTimeStep」の設定を「no」に設定してください。")
                mess += "\n" + _("deltaTが変化する設定では、連成計算ができません。")
                self.errDialog(messTitle, mess)
                return
        else:
            if dataDict["method"] == "altn":
                title = _("警告")
                mess = _("controlDict内のadjustTimeStepが「no」に設定されています。")
                mess += "\n" + _("流体個体の交互計算の場合、")
                mess += "\n  " + _("「adjustTimeStep yes;」で計算した方が安定します。")
                mess += "\n  " + _("(maxCo制御で計算できます。)")
                mess += "\n" + _("このままの設定で計算しますか？")
                funcOk = [self.getSetting_checkDict, dataDict]
                self.okCancelDialog(title, mess, funcOk=funcOk)
                return
        self.getSetting_checkDict(dataDict)

    #
    #  getSetting_checkDict                
    def getSetting_checkDict(self, dataDict):
        #OpenFOAM側取得
        dataDict["mappingCoeff"] = self.entry_coeff.get_text()
        dataDict["mappingShift"] = self.entry_shift.get_text()
        dataDict["field"] = self.comboFields.getSelectedItem()
        dataDict["pointField"] = self.comboPointField.getSelectedItem()
        selItems = self.treePatches.getSelectedItems()
        messTitle = _("エラー")
        if len(selItems) == 0:
            mess = _("OpenFOAMのpatch名が選択されていません。") + "\n"
            mess += _("patchを選択してください。")
            self.errDialog(messTitle, mess)
            return
        dataDict["mappingPatch"] = selItems[0]
        dataDict["nStepsOF"] = self.entry_nOF.get_text()
        folNames = pyTreeFoam.getFolderNames(self.caseDir)
        nProcs = len(list(filter(lambda x: x[:len("processor")] == "processor", folNames)))
        if self.check_parallel.get_active() == True:
            if nProcs == 0:
                mess  = _("メッシュが並列計算用に分割されていません。") + "\n"
                mess += _("decomposeParを実行して、メッシュ分割してください。")
                self.errDialog(messTitle, mess)
                return
        else:
            if nProcs > 0:
                title = _("警告")
                mess = _("メッシュが並列計算用に分割されています。") + "\n"
                mess += _("このまま、シングルコアで計算しますか？")
                funcOk = [self.getSetting_run, dataDict, 1]
                self.okCancelDialog(title, mess, funcOk=funcOk)
                return
            nProcs = 1
        self.getSetting_run(dataDict, nProcs)

    #
    #  getSetting_run
    def getSetting_run(self, dataDict, nProcs=1):
        """ 連成解析の為の設定を取得する"""
        dataDict["nProcs"] = str(nProcs)
        #FrontISTR側を取得
        dataDict["easyistr"] = self.entry_easyistr.get_text()
        dataDict["solidAnaComm"] = self.entry_fistr1.get_text()
        dataDict["solidCaseDir"] = self.entry_solidCaseDir.get_text()
        selItems = self.treeSgrps.getSelectedItems()
        if len(selItems) != 1:
            mess = _("FrontISTRのSGRP名が選択されていません。") + "\n"
            mess += _("SGRPを選択してください。")
            self.errDialog(_("エラー"), mess)
            return
        #連成項目を取得
        dataDict["fsi"] = "no"
        if self.check_fsi.get_active() == True:
            dataDict["fsi"] = "yes"
        dataDict["cht"] = "no"
        if self.check_cht.get_active() == True:
            dataDict["cht"] = "yes"
        dataDict["chtss"] = "no"
        if self.check_chtss.get_active() == True:
            dataDict["chtss"] = "yes"
        #全て未選択？
        if (dataDict["fsi"] == "no") and (dataDict["cht"] == "no") and (dataDict["chtss"] == "no"):
            title = _("エラー")
            mess = _("連成項目(fsi, cht, chtss)が選択されていません。")
            self.errDialog(title, mess)
            return
        #couplingDataファイルを更新
        dataDict["mappingSgrp"] = selItems[0]
        dataDict["nSteps"] = self.entry_nFistr.get_text()
        if self.check_thread.get_active() == True:
            dataDict["nThreads"] = self.entry_nThreads.get_text()
        else:
            dataDict["nThreads"] = "1"
        #bufferSize = self.entry_buffer.get_text()
        #dataDict["streamBuffer"] = str(int(bufferSize) * 1000)
        maxTime = self.entry_maxWaitTime.get_text()
        dataDict["maxWaitTime"] = str(int(maxTime) * 100)
        #controlDictから取得
        OF = openFoam(self.caseDir)
        contDict = OF.controlDict()
        dataDict["deltaT"] = contDict["deltaT"]
        dataDict["writeInterval"] = contDict["writeInterval"]
        dataDict["endTime"] = contDict["endTime"]
        if int(dataDict["nProcs"]) > 1:
            #並列処理
            startTime = OF.runStartTime("processor0")
        else:
            #シングルコア
            startTime = OF.runStartTime()
        dataDict["startTime"] = startTime
        writeCouplingDataFile(self.caseDir, dataDict)
        self.couplingDataDict = dataDict

        #Fistrの並列数をチェック(全core数とmesh分割数)
        allCores = int(dataDict["nThreads"])
        solidCaseDir = dataDict["solidCaseDir"]
        if self.check_thread.get_active() == True:
            #Fistr並列処理する場合
            title = _("エラー")
            mess  = _("FrontISTRのmesh分割数が、全core数より多くなっています。") + "\n"
            mess += _("mesh分割数を減らすか、全core数を増やしてください。")
            if dataDict["fsi"] == "yes":
                datFile = "hecmw_ctrl_fsi.dat"
                np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile=datFile)
                if allCores < np:
                    self.errDialog(title, mess)
                    return
            if dataDict["cht"] == "yes":
                datFile = "hecmw_ctrl_cht.dat"
                np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile=datFile)
                if allCores < np:
                    self.errDialog(title, mess)
                    return
            if dataDict["chtss"] == "yes":
                datFile = "hecmw_ctrl_chtss.dat"
                np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile=datFile)
                if allCores < np:
                    self.errDialog(title, mess)
                    return
        else:
            #Fistr並列しない（1コア）
            title = _("エラー")
            mess  = _("FrontISTRのmeshが並列処理用で分割されています。") + "\n"
            mess += _("「並列計算する」をチェックしてください。")
            nps = []
            if dataDict["fsi"] == "yes":
                datFile = "hecmw_ctrl_fsi.dat"
                np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile=datFile)
                nps.append(np)
            if dataDict["cht"] == "yes":
                datFile = "hecmw_ctrl_cht.dat"
                np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile=datFile)
                nps.append(np)
            if dataDict["chtss"] == "yes":
                datFile = "hecmw_ctrl_chtss.dat"
                np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile=datFile)
                nps.append(np)
            if len(nps) < sum(nps):
                self.errDialog(title, mess)
                return

        #scalarFieldのチェック
        if dataDict["field"] == "":
            title = _("エラー")
            mess = _("scalarFieldが設定されていません。")
            self.errDialog(title, mess)
            return
        if dataDict["fsi"] == "yes" and dataDict["cht"] == "yes":
            if dataDict["field"] != "p,T":
                title = _("エラー")
                mess = _("scalarFieldを「p,T」に設定してください。")
                self.errDialog(title, mess)
                return
        elif dataDict["fsi"] == "yes":
            if dataDict["field"] != "p":
                title = _("エラー")
                mess = _("scalarFieldを「p」に設定してください")
                self.errDialog(title, mess)
                return
        elif dataDict["cht"] == "yes":
            if dataDict["field"] != "T":
                title = _("エラー")
                mess = _("scalarFieldを「T」に設定してください")
                self.errDialog(title, mess)
                return

        #設定内容のエラーチェック
        if dataDict["fsi"] == "yes" or dataDict["chtss"] == "yes":
            if dataDict["pointField"] != "pointDisplacement":
                title = _("エラー")
                mess = _("fsi or chtss に関する計算は、pointFieldを「pointDisplacement」に設定してください。")
                self.errDialog(title, mess)
                return
            if self.isFieldPatchFixedValue(dataDict["pointField"]) == False:
                title = _("エラー")
                mess = _("pointFieldの「" + dataDict["mappingPatch"] + "」のpatchTypeは、「fixedValue」に設定してください。")
                self.errDialog(title, mess)
                return
        if dataDict["cht"] == "yes":
            if dataDict["field"] == "p,T":
                field = "T"
            else:
                field = dataDict["field"]
            if self.isFieldPatchFixedValue(field) == False:
                title = _("エラー")
                mess = _("scalarFieldの「" + dataDict["mappingPatch"] + "」のpatchTypeは、「fixedValue」に設定してください。")
                self.errDialog(title, mess)
                return

        #OpenFOAMのcontrolDictを修正
        stat = self.replaceCouplingDataToControlDict()
        #Pythonスクリプトをコピー
        self.copyPythonScript()
        #runFsiを作成
        self.createRunFsi()
        #message dialog表示
        title = _("設定取得・保存")
        mess = _("OpenFOAM, FrontISTRの設定を取得し、保存しました。") + "\n"
        mess += _("連成解析の為の実行ファイルをコピーしました。")
        self.okDialog(title, mess)

    #
    #  isFieldPatchFixedValue
    def isFieldPatchFixedValue(self, field):
        """ scalarFieldのcouplingPatchがfixedValueかどうか確認する"""
        ans = False
        OF = openFoam(self.caseDir)
        #scalarField = self.couplingDataDict["field"]
        patches, patchData = OF.fieldPatches(field)
        for i in range(len(patches)):
            patch = patches[i]
            if patch == self.couplingDataDict["mappingPatch"]:
                patchCont = patchData[i]
                patchContOp = pyTreeFoam.strOp(patchCont)
                p = patchContOp.getKeywordPointer("fixedValue", 0)
                if p > 0:
                    ans = True
                    break
        return ans

    #
    #  createRunFsi
    #----------------
    def createRunFsi(self):
        configDict = pyTreeFoam.readConfigTreeFoam()
        nth = self.couplingDataDict["nThreads"]
        bashrc = configDict["bashrcFOAM"]
        runComm  = "#!/bin/bash\n"
        runComm += ". " + bashrc + "\n"
        runComm += "export OMP_NUM_THREADS=" + nth + "\n"
        runComm += "python3 coupling_FrontISTR/python/runCoupling.py"
        fileName = self.caseDir + "/runFsi"
        f = open(fileName, "w"); f.write(runComm); f.close()
        runCase = pyTreeFoam.run(self.caseDir)
        runCase.command("chmod a+x runFsi")

    #
    #  replaceCouplingDataToControlDict
    #-------------------------------------
    def replaceCouplingDataToControlDict(self):
        """ controlDict内couplingDataを修正"""
        fileName = self.caseDir + "/system/controlDict"
        f = open(fileName); cont = f.read(); f.close()
        #functions.couplingSource部を作成
        cont = self.replaceCouplingSource(cont)
        #couplingToFistr部を作成
        cont = self.replaceCouplingToFistr(cont)
        f = open(fileName, "w"); f.write(cont); f.close()

    #
    #  replaceCouplingSource
    def replaceCouplingSource(self, cont):
        """ controlDict内のfunctions.couplingSource部を作成"""
        dataDict = self.couplingDataDict
        OFversion = pyTreeFoam.readConfigTreeFoam()["OFversion"]
        contFileDict = pyCoupling.getCouplingScriptDict(OFversion)
        key = (dataDict["fsi"], dataDict["cht"], dataDict["chtss"], dataDict["method"])
        contFile, contRunFile = contFileDict[key]
        #追加行を作成
        addCont  = "    couplingSource\n"
        addCont += "    {\n"
        addCont += '        #include "${FOAM_CASE}/coupling_FrontISTR/python/' + contFile + '"\n'
        addCont += "    }\n"
        contOp = pyTreeFoam.strOp(cont)
        #functionsを検索
        p = contOp.getKeywordPointer("functions", 0)
        if p < 0:
            #functionsが存在しない場合
            newCont  = "functions\n"
            newCont += "{\n"
            newCont += addCont
            newCont += "}\n"
            #  最終行を取得
            lineCont = " "
            p = 0
            while lineCont != "":
                pp = p
                lineCont, p, kind = contOp.get1line(p)
            #  最後の行頭から次の次の行頭
            p = contOp.skipNextCR(pp)
            p = contOp.skipNextCR(p)
            if p >= len(contOp.line):
                p = len(contOp.line) - 1
            newCont = cont[:p] + newCont + cont[p:]
            return newCont
        #couplingSourceを検索
        ps = p
        p = contOp.getKeywordPointer("couplingSource", ps)
        if p >= 0:
            #couplingSourceが存在する場合
            cont, p, key = contOp.replace1lineKeyword(p, addCont[4:-1])
            return cont
        #couplingSourceが存在しない場合
        newCont, _p = contOp.getKeywordContents("functions", 0)
        newCont = "functions\n{" + newCont + addCont + "}"
        cont, _p, key = contOp.replace1lineKeyword(ps, newCont)
        return cont

    #
    #  replaceCouplingToFistr
    def replaceCouplingToFistr(self, cont):
        """ controlDict内のcouplingToFistr部を作成"""
        dataDict = self.couplingDataDict
        fieldCont = ""
        if dataDict["field"] == "p,T":
            fieldCont += "        pressField  p;\n"
            fieldCont += "        tempField   T;\n"
        elif dataDict["field"] == "T":
            fieldCont += "        tempField   T;\n"
        elif dataDict["field"] == "p":
            fieldCont += "        pressField   p;\n"
        if dataDict["pointField"] != "":
            fieldCont += "        pointField  " + dataDict["pointField"] + ";\n"
        contOp = pyTreeFoam.strOp(cont)
        addCont  = "couplingToFistr\n"
        addCont += "{\n"
        addCont += "    couplingInterval\n"
        addCont += "    {\n"
        addCont += "        pressure    " + dataDict["nStepsOF"] + ";\n"
        addCont += "        temperature " + dataDict["nSteps"] + ";\n"
        addCont += "    }\n"
        addCont += "    jobWaitingTime\n"
        addCont += "    {\n"
        addCont += "        maxWaitingTime  " + dataDict["maxWaitTime"] + ";\n"
        addCont += "    }\n"
        addCont += "    couplingField\n"
        addCont += "    {\n"
        addCont += fieldCont
        addCont += "    }\n"
        addCont += "    couplingPatch   " + dataDict["mappingPatch"] + ";\n"
        addCont += "}\n"
        sp = contOp.skipFoamFile()
        #削除
        p = contOp.getKeywordPointer("couplingToFistr", sp)
        if p >= 0:
            stp = p
            contents, pp = contOp.getKeywordContents("couplingToFistr", 0)
            edp = pp
            if contOp.line[edp] == "\n":
                edp += 1
            #削除
            cont = cont[:stp] + cont[edp:]
        contOp = pyTreeFoam.strOp(cont)
        #coupling行を追加
        lineCont = " "
        p = sp
        while lineCont != "":
            pp = p
            lineCont, p, kind = contOp.get1line(p)
        #最後の行頭から次の次の行頭
        p = contOp.skipNextCR(pp)
        p = contOp.skipNextCR(p)
        if p >= len(contOp.line):
            #p = len(contOp.line) - 1
            p = contOp.skipBackCR(p)
            addCont = "\n" + addCont + "\n"
        newCont = cont[:p] + addCont + cont[p:]
        return newCont

    #
    #  copyPythonScript
    def copyPythonScript(self):
        """ pythonスクリプトをコピーする"""
        #folder有無確認
        couplingFolder = self.caseDir + "/coupling_FrontISTR"
        if os.path.exists(couplingFolder) == False:
            os.mkdir(couplingFolder)
        pythonFolder = couplingFolder + "/python"
        if os.path.exists(pythonFolder) == False:
            os.mkdir(pythonFolder)
        #pythonスクリプトをコピー
        #  sourceを取得
        scrDir = os.getenv("TreeFoamPath") + os.sep + "frontIstr"
        scripts = glob.glob(scrDir + "/*.py")
        #scripts += glob.glob(scrDir + "/cont_*")
        for fileName in scripts:
            name = os.path.basename(fileName)
            if name[:len("contRun_")] != "contRun_":
                shutil.copy(fileName, pythonFolder)
        #実行codeを取得
        OFversion = pyTreeFoam.readConfigTreeFoam()["OFversion"]
        contDict = pyCoupling.getCouplingScriptDict(OFversion)
        dataDict = self.couplingDataDict
        key = (dataDict["fsi"], dataDict["cht"], dataDict["chtss"], dataDict["method"])
        addNames = contDict[key]
        addScripts = list(map(lambda x: scrDir + "/" + x, addNames))
        for fileName in addScripts:
            shutil.copy(fileName, pythonFolder)

    #
    #  runCoupling
    #--------------
    def runCoupling(self):
        """ 連成計算開始"""
        #並列処理の確認(OpenFOAM)
        folNames = pyTreeFoam.getFolderNames(self.caseDir)
        nProcs = len(list(filter(lambda x: x[:len("processor")] == "processor", folNames)))
        if self.check_parallel.get_active() == True:
            if nProcs == 0:
                title = _("エラー")
                mess  = _("メッシュが並列計算用に分割されていません。") + "\n"
                mess += _("decomposeParを実行して、メッシュ分割してください。")
                self.errDialog(title, mess)
                return
        else:
            if nProcs > 0:
                title = _("警告")
                mess = _("メッシュが並列計算用に分割されています。") + "\n"
                mess += _("このまま、シングルコアで計算しますか？")
                funcOk = [self.runCoupling_run]
                self.okCancelDialog(title, mess, funcOk=funcOk)
                return
        self.runCoupling_run()

    #
    #  runCoupling_run
    def runCoupling_run(self):
        #deltaT, writeIntervalの取得
        OF = openFoam(self.caseDir)
        contDict = OF.controlDict()
        deltaT = contDict["deltaT"]
        writeInterval = contDict["writeInterval"]
        endTime = contDict["endTime"]
        self.couplingDataDict = readCouplingDataFile(self.caseDir)
        #startTimeのfolderが存在するか確認
        if int(self.couplingDataDict["nProcs"]) > 1:
            #並列処理
            timeFolders = pyTreeFoam.case(self.caseDir).getTimeFolders("processor0")
            startTime = OF.runStartTime("processor0")
        else:
            #シングルコア
            timeFolders = pyTreeFoam.case(self.caseDir).getTimeFolders()
            startTime = OF.runStartTime()
        timeNums = list(map(float, timeFolders))
        if float(startTime) in timeNums == False:
            title = _("エラー")
            mess = _("timeFolder「") + startTime + _("」が存在しません。") + "\n"
            mess += _("計算開始できません。")
            self.errDialog(title, mess)
            return
        #fistr1の有無を確認
        fistr1 = self.couplingDataDict["solidAnaComm"]
        comm = "which " + fistr1
        stat, res, err = pyTreeFoam.run(self.caseDir).commandReturnCont(comm)
        if stat != "OK" or len(res) <= 1:
            title = _("エラー")
            mess = _("command「fistr1」が見つかりません。") + "\n"
            mess += _("計算開始できません。")
            self.errDialog(title, mess)
            return
        #定数を取得
        self.couplingDataDict["startTime"] = startTime
        self.couplingDataDict["deltaT"] = deltaT
        self.couplingDataDict["writeInterval"] = writeInterval
        self.couplingDataDict["endTime"] = endTime
        solidCaseDir = self.entry_solidCaseDir.get_text()
        if self.radio_concurr.get_active() == True:
            method = "concurrent"
        else:
            method = "alternate"
        writeCouplingDataFile(self.caseDir, self.couplingDataDict)
        #起動確認
        fistrMess = []
        if self.couplingDataDict["fsi"] == "yes":
            np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile="hecmw_ctrl_fsi.dat")
            nt = int(self.couplingDataDict["nThreads"]) // np
            mess = "cores=" + self.couplingDataDict["nThreads"] + "   fsi(procs:" + str(np) + ", threads:" + str(nt) + ")\n"
            fistrMess.append(mess)
        if self.couplingDataDict["cht"] == "yes":
            np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile="hecmw_ctrl_cht.dat")
            nt = int(self.couplingDataDict["nThreads"]) // np
            mess = "cores=" + self.couplingDataDict["nThreads"] + "   cht(procs:" + str(np) + ", threads:" + str(nt) + ")\n"
            fistrMess.append(mess)
        if self.couplingDataDict["chtss"] == "yes":
            np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile="hecmw_ctrl_chtss.dat")
            nt = int(self.couplingDataDict["nThreads"]) // np
            mess = "cores=" + self.couplingDataDict["nThreads"] + " chtss(procs:" + str(np) + ", threads:" + str(nt) + ")\n"
            fistrMess.append(mess)
        title = _("流体-構造連成解析")
        mess  = _("OpenFOAMとFrontISTRの連成解析を開始します。") + "\n\n"
        mess += _("      startTime: ") + startTime + "\n"
        mess += _("  writeInterval: ") + writeInterval + "\n"
        mess += _("        endTime: ") + endTime + "\n"
        mess += _("         method: ") + method + "\n"
        mess += _("       OpenFOAM: ") + "procs=" + self.couplingDataDict["nProcs"] + "\n"
        mess += _("     FrontISTR : ") + fistrMess[0]
        for i in range(1, len(fistrMess)):
            mess += _("               : ") + fistrMess[i]
        funcOk = [self.runCoupling_runOk]
        self.okCancelDialog(title, mess, funcOk=funcOk)

    #  runOk時
    def runCoupling_runOk(self):
        #実行
        runCase = pyTreeFoam.run(self.caseDir)
        runCase.foamTerminal("runFsi")

    #
    #  showFistrLog
    #----------------
    def showFistrLog(self):
        """ FrontISTRのlog表示"""
        comm = pyTreeFoam.TerminalRun + "tail -f solidSolve.log"
        runCase = pyTreeFoam.run(self.caseDir)
        runCase.command(comm)

    #
    #  runEasyistr
    #--------------
    def runEasyistr(self):
        """ EasyISTRの起動"""
        solidCaseDir = self.entry_solidCaseDir.get_text()
        if solidCaseDir[0] != os.path:
            solidCaseDir = os.path.abspath(solidCaseDir)
        easyistrPath = self.entry_easyistr.get_text()
        comm = "which " + easyistrPath
        stat, res, err = pyTreeFoam.run().commandReturnCont(comm)
        if stat != "OK" or len(res) <= 1:
            title = _("エラー")
            mess = _("command「easyistr」が見つかりません。") + "\n"
            mess += _("計算開始できません。")
            self.errDialog(title, mess)
            return
        #編集する構造解析のcaseを取得
        cases = []
        if self.check_fsi.get_active() == True:
            cases.append("fsi")
        if self.check_cht.get_active() == True:
            cases.append("cht")
        if self.check_chtss.get_active() == True:
            cases.append("chtss")
        if len(cases) == 0:
            title = _("エラー")
            mess = _("連成する項目の組み合わせが誤っています。") + "\n"
            mess += _("選択し直してください。")
            self.errDialog(title, mess)
            return
        couplingData = pyCoupling.couplingData(os.getcwd())
        couplingDict = couplingData.read()
        anaItem = couplingDict["solidAnaItem"]
        okArgs = [self.runEasyistr_run, easyistrPath, solidCaseDir, couplingData, couplingDict]
        dialog = selectEditStructureCase.selectEditStructureCase(
            cases, anaItem, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return
        
    #  runEasyistr_run
    def runEasyistr_run(self, selCase, easyistrPath, solidCaseDir, couplingData, couplingDict):
        couplingDict["solidAnaItem"] = selCase
        couplingData.write(couplingDict)
        #cnt,msh名チェック修正
        cntMshFiles = pyCoupling.cntMshFiles(solidCaseDir)
        #標準のcnt,msh名か確認する。
        nameDict = pyFistr.getCntMshFileNameDict(solidCaseDir)
        if os.path.exists(solidCaseDir + os.sep + "FistrModel.cnt") == False:
            copyName = solidCaseDir + os.sep + nameDict["cnt"]
            pasteName = solidCaseDir + os.sep + "FistrModel.cnt"
            shutil.copy(copyName, pasteName)
        if os.path.exists(solidCaseDir + os.sep + "FistrModel.msh") == False:
            copyName = solidCaseDir + os.sep + nameDict["msh"]
            pasteName = solidCaseDir + os.sep + "FistrModel.msh"
            shutil.copy(copyName, pasteName)
        #編集するhecmw_ctrl.datを作成する
        pasteCase = solidCaseDir + os.sep + "hecmw_ctrl.dat"
        if selCase == "fsi":
            cntMshFiles.checkFsiFiles()
            copyCase = solidCaseDir + os.sep + "hecmw_ctrl_fsi.dat"
            shutil.copy(copyCase, pasteCase)
        elif selCase == "cht":
            cntMshFiles.checkChtFiles()
            copyCase = solidCaseDir + os.sep + "hecmw_ctrl_cht.dat"
            shutil.copy(copyCase, pasteCase)
        elif selCase == "chtss":
            cntMshFiles.checkChtssFiles()
            copyCase = solidCaseDir + os.sep + "hecmw_ctrl_chtss.dat"
            shutil.copy(copyCase, pasteCase)
        else:
            return
        #EasyISTRを起動
        print("edit (" + selCase + ") case at EasyISTR.")
        comm = easyistrPath + " -d " + solidCaseDir + " &"
        print(comm)
        pyTreeFoam.run().command(comm)

    #
    #  openFluidCase
    #----------------
    def openFluidCase(self):
        """ 流体folderを開く"""
        pyTreeFoam.run(self.caseDir).fileManager()

    #
    #  openSolidCase
    #----------------
    def openSolidCase(self):
        """ 固体folderを開く"""
        solidCaseDir = self.entry_solidCaseDir.get_text()
        if solidCaseDir[0] != os.path:
            solidCaseDir = os.path.abspath(solidCaseDir)
        pyTreeFoam.run(solidCaseDir).fileManager()

    #
    #  changeCoupling
    #-----------------
    def changeCoupling(self):
        """ checkBoxを変更した時"""
        if ( self.check_fsi.get_active() == True or
             self.check_chtss.get_active() == True ):
            #fsiをcheck
            self.label_fsiConts.set_sensitive(True)
            self.label_nStepsFsi.set_sensitive(True)
            self.entry_nOF.set_sensitive(True)
        else:
            #fsiのcheckを外す
            self.label_fsiConts.set_sensitive(False)
            self.label_nStepsFsi.set_sensitive(False)
            self.entry_nOF.set_sensitive(False)
        if self.check_cht.get_active() == True:
            #chtをcheck
            self.label_heatConts.set_sensitive(True)
            self.label_nStepsHeat.set_sensitive(True)
            self.entry_nFistr.set_sensitive(True)
        else:
            #chtのcheckを外す
            self.label_heatConts.set_sensitive(False)
            self.label_nStepsHeat.set_sensitive(False)
            self.entry_nFistr.set_sensitive(False)

        #OpenFOAM側のlabel, entry, comboの設定
        combo_fields = self.builder.get_object("combo_fields")
        combo_pointField = self.builder.get_object("combo_pointField")
        #fsi ?
        if self.check_fsi.get_active() == True:
            #全て表示
            self.label_pCoeff.set_sensitive(True)
            self.label_pShift.set_sensitive(True)
            self.label_scalarField.set_sensitive(True)
            self.label_pointField.set_sensitive(True)
            self.entry_coeff.set_sensitive(True)
            self.entry_shift.set_sensitive(True)
            combo_fields.set_sensitive(True)
            combo_pointField.set_sensitive(True)
        else:
            #fsi off, cht on, chtss off ?
            if (self.check_fsi.get_active() == False
                and self.check_cht.get_active() == True
                and self.check_chtss.get_active() == False):
                #scalarFieldのみ表示
                self.label_pCoeff.set_sensitive(False)
                self.label_pShift.set_sensitive(False)
                self.label_scalarField.set_sensitive(True)
                self.label_pointField.set_sensitive(False)
                self.entry_coeff.set_sensitive(False)
                self.entry_shift.set_sensitive(False)
                combo_fields.set_sensitive(True)
                combo_pointField.set_sensitive(False)
            #fsi off, cht on, chtss on ?
            elif (self.check_fsi.get_active() == False
                and self.check_cht.get_active() == True
                and self.check_chtss.get_active() == True):
                #scalar, pointFieldを表示
                self.label_pCoeff.set_sensitive(False)
                self.label_pShift.set_sensitive(False)
                self.label_scalarField.set_sensitive(True)
                self.label_pointField.set_sensitive(True)
                self.entry_coeff.set_sensitive(False)
                self.entry_shift.set_sensitive(False)
                combo_fields.set_sensitive(True)
                combo_pointField.set_sensitive(True)
            else:
                #全て非表示
                self.label_pCoeff.set_sensitive(False)
                self.label_pShift.set_sensitive(False)
                self.label_scalarField.set_sensitive(False)
                self.label_pointField.set_sensitive(False)
                self.entry_coeff.set_sensitive(False)
                self.entry_shift.set_sensitive(False)
                combo_fields.set_sensitive(False)
                combo_pointField.set_sensitive(False)

    # --------- universal dialog ----------------

    #  okDialog
    def okDialog(self, title, mess, funcOk=[]):
        """ okDialog表示"""
        dialog = unvDlg.okDialog(title, mess, parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    #  errDialog
    def errDialog(self, title, mess, funcOk=[]):
        """ errDialog表示"""
        dialog = unvDlg.errDialog(title, mess, parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    #  okCancelDialog
    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        """ okCancel dialog表示"""
        dialog = unvDlg.okCancelDialog(title, mess, parentWin=self.mainWindow, funcOk=funcOk, funcCancel=funcCancel)
        dialog.show()

    # openFolderDialog
    def openFolderDialog(self, title, path, funcOk=[], funcCancel=[], selName="", patterns=[]):
        dialog = unvDlg.openFolderDialog(title, path, funcOk=funcOk, funcCancel=funcCancel, parentWin=self.mainWindow, selName=selName, filterSets=patterns)
        dialog.show()

    #openFileDialog
    def openFileDialog(self, title, path, funcOk=[], funcCancel=[], selName="", patterns=[]):
        dialog = unvDlg.openFileDialog(title, path, funcOk=funcOk, funcCancel=funcCancel, parentWin=self.mainWindow, selName=selName, filterSets=patterns)
        dialog.show()

#----------------------
#  openFoam class
#----------------------
class openFoam:
    """ openFOAM関連の内容を取得"""
    def __init__(self, caseDir):
        self.caseDir = caseDir

    def timeFolders(self):
        case = pyTreeFoam.case(self.caseDir)
        timeFolders = case.getTimeFolders()
        return timeFolders

    def fields(self):
        """ fieldを取得し、list形式で返す"""
        case = pyTreeFoam.case(self.caseDir)
        timeFolders = case.getTimeFolders()
        timeFolder = timeFolders[0]
        fields = case.getFieldNames(timeFolder)
        fields.sort()
        return fields

    def patches(self):
        """ patchを取得し、list形式で返す"""
        case = pyTreeFoam.case(self.caseDir)
        timeFolders = case.getTimeFolders()
        maxTime = timeFolders[0]
        meshDir = case.getCurrMeshDir(maxTime, ".", "boundary")
        patchConts = case.getPatchNameFromBoundary(meshDir)
        patches = list(map(lambda x: x[0], patchConts))
        return patches

    def runStartTime(self, processor="."):
        """ 計算開始時間を取得"""
        getTime = ""
        case = pyTreeFoam.case(self.caseDir)
        contDict = case.getControlDict()
        if contDict["startFrom"] == "firstTime":
            timeFolders = case.getTimeFolders(processor)
            getTime = timeFolders[0]
        elif contDict["startFrom"] == "startTime":
            getTime = contDict["startTime"]
        elif contDict["startFrom"] == "latestTime":
            timeFolders = case.getTimeFolders(processor)
            getTime = timeFolders[-1]
        return getTime

    def controlDict(self):
        """ controlDictを取得"""
        case = pyTreeFoam.case(self.caseDir)
        controlDict = case.getControlDict()
        return controlDict

    def deltaT(self):
        """ deltaTを取得"""
        case = pyTreeFoam.case(self.caseDir)
        contDict = case.getControlDict()
        deltaT = contDict["deltaT"]
        return deltaT

    def writeInterval(self):
        """ writeIntervalを取得"""
        case = pyTreeFoam.case(self.caseDir)
        contDict = case.getControlDict()
        writeInterval = contDict["writeInterval"]
        return writeInterval

    def adjustTimeStep(self):
        case = pyTreeFoam.case(self.caseDir)
        contDict = case.getControlDict()
        adjustTimeStep = contDict["adjustTimeStep"]
        return adjustTimeStep

    def fieldPatches(self, field):
        """指定したfieldの全patch名とpatch内容を取得"""
        case = pyTreeFoam.case(self.caseDir)
        currTimeFolder = case.getCurrTimeFolder()
        _fields, patches, patchData = case.getFieldsPatchNames(currTimeFolder, [field])
        return patches, patchData[0]


#----------------
#  fistr class
#----------------
class fistr:

    def __init__(self, solidCaseDir):
        self.solidCaseDir = solidCaseDir
        saveDir = os.getcwd()
        os.chdir(solidCaseDir)
        lines = pyFistr.readFistrModelMsh()
        self.headerData = pyFistr.getHeaderData(lines)
        os.chdir(saveDir)

    def sgrps(self):
        """ sgrpを取得し、list形式で返す。"""
        sgrps = []
        for header in self.headerData:
            line = header[0]
            item = pyFistr.deleteSp(line).split(",")[0]
            if item == "!SGROUP":
                name = pyFistr.getValName(line, "SGRP")
                sgrps.append(name)
        return sgrps

#
#  readCouplingDataFile
#-----------------------
def readCouplingDataFile(caseDir):
    """ couplingDataファイル読み込み"""
    fileName = caseDir + "/coupling_FrontISTR/couplingData"
    dataDict = {}
    if os.path.exists(fileName) == False:
        return dataDict
    #couplingData読み込み
    couplingData = pyCoupling.couplingData(caseDir)
    dataDict = couplingData.read()
    #configTreeFoamの内容を追加
    configDict = pyTreeFoam.readConfigTreeFoam()
    dataDict["solidAnaComm"] = configDict["fistr1"]
    dataDict["easyistr"] = configDict["easyistr"]
    return dataDict

#
#  writeCouplingDataFile
#-------------------------
def writeCouplingDataFile(caseDir, dataDict):
    """ couplingDataFileに保存する"""
    fileName = caseDir + "/coupling_FrontISTR/couplingData"
    if os.path.exists(fileName) == False:
        folder = os.path.dirname(fileName)
        if os.path.exists(folder) == False:
            os.mkdir(folder)
        #空ファイルを作成
        f = open(fileName, "w"); f.close()
    #configTreeFoamに保存
    configDict = {}
    if "solidAnaComm" in dataDict.keys():
        if dataDict["solidAnaComm"] != "":
            configDict["fistr1"] = dataDict["solidAnaComm"]
    if "easyistr" in dataDict.keys():
        if dataDict["easyistr"] != "":
            easyistr = dataDict.pop("easyistr")
            configDict["easyistr"] = easyistr
    if len(configDict) > 0:
        pyTreeFoam.writeConfigTreeFoam(configDict)
    #couplingDataに保存
    couplingData = pyCoupling.couplingData(caseDir)
    couplingData.write(dataDict)



if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    caseDir = os.path.abspath(caseDir)
    winApp = windowApp(caseDir)
    winApp.main()
