#!/usr/bin/python3
#  coding:  utf-8
#
#       copyPasteFolders.py
#
#       fileやfolderをclipboardを使ってcopy, pasteする。
#       Qt以外のappに使う。
#
#   20/05/01    新規作成
#      05/02    PyQt4, PyQt5, PySide, PySide2に対応
#      05/23    GTK+3ベースに戻す（PyQt4のみで動かなかった為）
#   22/09/26    xclip, wl-clipboardを使うように修正。
#               clipboard保存のformatがgnomeバージョンによって変わるので
#               copyPasteができなくなる事がある為。
#               以下のwebを参考にした。
#               「https://stackoverflow.com/questions/73234123/how-copy-file-to-clipboard-using-python-or-cl-to-paste-it-using-strgv-later-on」
#      09/29    copyFolders, pasteFolders:xclipなどがinstallされていない
#               場合には、clipboardにtextとして保存する様に修正。
#      10/06    paste_from_clipboard:debug用のprint文残っていたので削除
#   23/05/24    base_paste_cmd:if os.environ["GDK_BACKEND"] != "x11"を追加
#               ubuntu2204をwaylandで起動する様に変更した為
#   24/07/15    base_paste_cmd:os.environ[] を os.getenv() に修正
#      09/03    checkClipApp:messageの出力優先度を修正（flush=Trueを追加）
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk
clipboard = Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD)

import os
import subprocess
import sys
from pathlib import Path

import pyTreeFoam

gnome_desktops = ['X-Cinnamon', 'XFCE']


def is_gnome(desktop):
    """ desktopを確認。
    GNOME, mint, xface, KDEかどうか確認"""
    if desktop.endswith("GNOME") or desktop in gnome_desktops:
        return True
    return False


def target():
    """ targetを確認
    GNOME, mint, xface:'x-special/gnome-copied-files'
    KDE:'text/uri-list'
    """
    current_desktop = os.environ['XDG_CURRENT_DESKTOP']
    if is_gnome(current_desktop):
        return 'x-special/gnome-copied-files'
    elif current_desktop == 'KDE':
        return 'text/uri-list'
    else:
        sys.exit(f'unsupported desktop {current_desktop}')

#
#  copyFolder関連-----------------------------
#
def base_copy_cmd():
    """ commandを取得
    Wayland: wl-copy
    X: xclip -i -selection clipboard 
    """
    if 'WAYLAND_DISPLAY' in os.environ:
        return 'wl-copy'
    return 'xclip -i -selection clipboard'


def copy_clipboard_cmd():
    """ commandにtargetを追加して返す。"""
    return f"{base_copy_cmd()} -t '{target()}'"


def content(files_to_copy):
    """ clipboardに保存する内容を作成"""
    uris = '\n'.join([Path(f).as_uri() for f in files_to_copy])
    current_desktop = os.environ['XDG_CURRENT_DESKTOP']
    if is_gnome(current_desktop):
        return f"copy\n{uris}".encode("utf-8")
    return uris.encode("utf-8")


def copy_to_clipboard(files_to_copy):
    """ clipboardへ書き込み"""
    copy_process = subprocess.Popen(
        copy_clipboard_cmd(),
        shell=True, 
        stdin=subprocess.PIPE)
    copy_process.stdin.write(content(files_to_copy))
    copy_process.stdin.close()
    copy_process.wait()

#
#  pasteFolder関連----------------------------
#
def base_paste_cmd():
    """ commandを取得
    Wayland: wl-paste
    X: xclip -o -selection clipboard
    """
    if 'WAYLAND_DISPLAY' in os.environ:
        #if os.environ["GDK_BACKEND"] != "x11":
        if os.getenv("GDK_BACKEND") != "x11":
            return 'wl-paste'
    return 'xclip -o -selection clipboard'


def paste_clipboard_cmd():
    """ commandにtargetを追加して返す。"""
    return f"{base_paste_cmd()}  -t '{target()}'"


def paste_from_clipboard():
    """ clipboardの内容を読み込む"""
    #clipboardの内容を取得
    paste_process = subprocess.Popen(
        paste_clipboard_cmd(), 
        shell=True, 
        stdout=subprocess.PIPE)
    #stdoutの内容を取得
    pasteLines = []
    for line in iter(paste_process.stdout.readline, b""):
        pasteLine = line.rstrip().decode("utf-8")
        pasteLines.append(pasteLine)
        print(pasteLine)
    #clipboard内のpath名を取得
    pasteLines = pasteLines[1:]
    pasteFiles = list(map(lambda x: x[len("file://"):], pasteLines))
    return pasteFiles

#
#  checkClipApp
#----------------
def checkClipApp():
    """ clipManagerがインストールされているか確認する"""
    app = base_paste_cmd().split()[0]
    comm = "which " + app
    stat, res, err = pyTreeFoam.run().commandReturnCont(comm, isOut=False)
    if len(res.split()) == 0:
        #インストールなし
        if app == "wl-paste" or app == "wl-copy":
            app = "wl-clipboard"
        print("\nplease install '" + app + "'.  uris are copied for text.", flush=True)
        return False
    else:
        #インストールされている場合
        return True

#変数を設定
#clipActive = checkClipApp()

#
#  copyToText
#--------------
def copyToText(folders):
    """ textとしてclipboardに保存"""
    text = content(folders).decode("utf-8")
    clipboard.set_text(text, -1)

#
#  pasteFromText
#---------------
def pasteFromText():
    """ clipboard textからpasteFoldersを取得する"""
    text = clipboard.wait_for_text()
    pasteLines = []
    for line in text.split():
        pasteLine = line.rstrip()
        pasteLines.append(pasteLine)
    #clipboard内のpath名を取得
    pasteLines = pasteLines[1:]
    pasteFiles = list(map(lambda x: x[len("file://"):], pasteLines))
    return pasteFiles

#--------------
#  copyFolders
#--------------
def copyFolders(folders):
    #global clipActive
    #clipboardをクリア
    clipboard.clear()
    #clipboardのappのinstall確認
    clipActive = checkClipApp()
    if clipActive == True:
        #system clipboardへcopy
        copy_to_clipboard(folders)
    else:
        #clipboardのtextへcopy
        copyToText(folders)
    return

#---------------
#  pasteFolders
#---------------
def pasteFolders():
    #global clipActive
    #clipboardのappのinstall確認
    clipActive = checkClipApp()
    if clipActive == True:
        #system clipboardから取得
        folders = paste_from_clipboard()
    else:
        #clipboardのtextから取得
        folders = pasteFromText()
    return folders


if __name__ == '__main__':
    files = ['/etc/hosts', '/etc/group', '/home/caeuser/work/treefoam']
    #copy_to_clipboard(files)
    #paste_from_clipboard()
    copyFolders(files)
    folders = pasteFolders()
