#!/usr/bin/python3
# coding: utf-8
#
#   addDeleteRegionDialog.py
#
#       multiRegion内のregionを追加・削除する
#
#   13/06/22    新規作成
#      06/23    region名変更時、boundaryのsampleRegion、samplePatchの内容も修正。
#               （名称変更時、実行できなくなるため）
#      06/24    region削除時、boundaryのpatch名、patchTypeも修正
#      07/03    region削除、変更時にconstant内のregionが修正されない場合があり修正
#      09/28    gzip, binaryに対応。region削除時、boundaryFieldのクリアを追加。
#               region名変更後、boundaryFieldが読み込めないバグを修正。
#      10/15    国際化のため、修正
#               全region削除にバグあり、修正
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/10/21    remakeFieldPatchName:unicodeｴﾗｰ発生したため、修正。
#   15/02/26    region操作時、結果folder有無を確認する様に修正
#      02/27    addRegionFolder:追加したregionにファイルの配布、boundaryの作成追加
#      03/02    copyFilesFluidSolid:修正（regName付きfileの配布を追加）
#      07/17    dialogの起動方法を修正
#   19/05/22    addFolder:mkdirをos.mkdirに修正
#               copyFilesFluidSolid:cpをshutil.copyに修正
#      12/13    GTK+3, python3用に大幅修正
#   20/04/20    国際化対応で修正
#   24/02/05    deleteAllRegions:OF-11用を追加
#      08//01   universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/04    deleteSolidRegions,deleteFluidRegions,addSolidRegion,addFluidRegion:
#               OF-11以降についてregionPropertiesの分岐を追加。（バグ）
#      08/05    changeSolidRegion,changeFluidRegion:OF-11以降についてregionProperties
#               の分岐を追加。（バグ）
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os, sys
import glob
import shutil

import pyTreeFoam
import inputTextDDialog
import universalDialogs as unvDlg

import logFileCreater
logFileCreater.log()


#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"addDeleteRegionDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #属性
        self.caseDir = caseDir
        self.fluidRegions = []
        self.solidRegions = []
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        #GUIのobjectを取得
        self.setGtkObject()
        #treeListの初期化
        self.iniTreeList()
        #region名を取得
        self.getRegions()
        #treeListにデータをセット
        self.setTreeDataToTreeList()

    #
    #  main
    #  ----
    def main(self):
        """ GUIの表示"""
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    def setGtkObject(self):
        self.tree_fluid = self.builder.get_object("tree_fluid")
        self.tree_solid = self.builder.get_object("tree_solid")

    def iniTreeList(self):
        """ teeListの初期化"""
        #fluid側
        #selectModeをMULTIPLEに設定
        treeSelection = self.tree_fluid.get_selection()
        treeSelection.set_mode(Gtk.SelectionMode.MULTIPLE)
        #titleを非表示
        self.tree_fluid.set_headers_visible(False)
        #treeColumnの設定
        cell = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn()
        col.pack_start(cell, True)
        col.add_attribute(cell, "text", 0)
        self.tree_fluid.append_column(col)
        #solid側
        #selectModeをMULTIPLEに設定
        treeSelection = self.tree_solid.get_selection()
        treeSelection.set_mode(Gtk.SelectionMode.MULTIPLE)
        #titleを非表示
        self.tree_solid.set_headers_visible(False)
        #treeColumnの設定
        cell = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn()
        col.pack_start(cell, True)
        col.add_attribute(cell, "text", 0)
        self.tree_solid.append_column(col)

    def getRegions(self):
        """ fluid, solidのregionを取得"""
        #configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = self.configDict["OFversion"]
        (fluid, solid) = pyTreeFoam.case(self.caseDir).getZonesInRegionProperties(OFversion)
        fluid.sort()
        solid.sort()
        self.fluidRegions = fluid
        self.solidRegions = solid

    def setTreeDataToTreeList(self):
        """ region名を各treeListに表示する"""
        #fluid側
        listStore = Gtk.ListStore(str)
        for name in self.fluidRegions:
            listStore.append([name])
        self.tree_fluid.set_model(listStore)
        #solid側
        listStore = Gtk.ListStore(str)
        for name in self.solidRegions:
            listStore.append([name])
        self.tree_solid.set_model(listStore)

    def getFluidSelNames(self):
        """ fluid側の選択region名を取得する"""
        treeSelection = self.tree_fluid.get_selection()
        treeStore, pathes = treeSelection.get_selected_rows()
        selItems = []
        for path in pathes:
            iter = treeStore.get_iter(path)
            selItem = treeStore[iter][0]
            selItems.append(selItem)
        return selItems

    def getSolidSelNames(self):
        """ solid側の選択region名を取得する"""
        treeSelection = self.tree_solid.get_selection()
        treeStore, pathes = treeSelection.get_selected_rows()
        selItems = []
        for path in pathes:
            iter = treeStore.get_iter(path)
            selItem = treeStore[iter][0]
            selItems.append(selItem)
        return selItems

    def selectItemsInTreeList(self, treeList, selItems):
        """ 指定しているlistBox内のitemsを選択する"""
        #selNoを取得
        treeStore = treeList.get_model()
        selNos = []
        for i in range(len(treeStore)):
            item = treeStore[i][0]
            if item in selItems:
                selNos.append(i)
        #selNoを選択する
        for selNo in selNos:
            path = Gtk.TreePath(selNo)
            treeSelection = treeList.get_selection()
            treeSelection.select_path(path)

    #---- event handler ------------
    #fluid追加
    def onAddFluid(self, event):
        self.addFluidRegion()
    #fluid変更
    def onChangeFluid(self, event):
        self.changeFluidRegion()
    #fluid削除
    def onDeleteFluid(self, event):
        self.deleteFluidRegions()
    #solid追加
    def onAddSolid(self, event):
        self.addSolidRegion()
    #solid変更
    def onChangeSolid(self, event):
        self.changeSolidRegion()
    #solid削除
    def onDeleteSolid(self, event):
        self.deleteSolidRegions()
    #全region削除
    def onDeleteAllRegions(self, event):
        self.deleteAllRegions()
    #閉じる
    def onClose(self, event):
        self.close()


    #
    #  addFluidRegion
    #  --------------
    def addFluidRegion(self):
        #newRegion名を取得
        newName = self.getIniRegionName()
        title = _(u"region名の入力")
        message = _(u"新しいregion名を入力してください。")
        inputText = newName
        okArgs = [self.addFluidRegion_run]
        dialog = inputTextDDialog.getInputText(title, message, inputText, funcOk=okArgs)
        dialog.show()
        return
    
    #  addFluidRegion_run
    def addFluidRegion_run(self, newName):
        allRegions = self.fluidRegions + self.solidRegions
        if newName in allRegions:
            mess = _(u"同じ名前のregionがあります。")
            self.errDialog(_(u"エラー"), mess)
            return
        self.fluidRegions.append(newName)
        #treeListを再表示
        self.setTreeDataToTreeList()
        #regionPropertiesを作成
        cont = pyTreeFoam.getFoamContents().makeRegionProperties(self.fluidRegions, self.solidRegions)
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 11:
            rsOp = pyTreeFoam.strOp(cont)
            regionSolvers, p = rsOp.getMiddlePair(0)
            contDict = {"regionSolvers":regionSolvers}
            pyTreeFoam.case(self.caseDir).setControlDict(contDict)
        else:        
            fileName = self.caseDir + "/constant/regionProperties"
            f = open(fileName, "w"); f.write(cont); f.close()
        #folderを作成
        self.addFluidRegionFolder(newName)
        mess = _(u"新しいregion「") + newName + _(u"」を追加しました。")
        self.okDialog(_(u"region追加"), mess)

    #
    #  addSolidRegion
    #  --------------
    def addSolidRegion(self):
        """ solidRegionを追加"""
        #newRegion名を取得
        newName = self.getIniRegionName()
        title = _(u"region名の入力")
        message = _(u"新しいregion名を入力してください。")
        inputText = newName
        okArgs = [self.addSolidRegion_run]
        dialog = inputTextDDialog.getInputText(title, message, inputText, funcOk=okArgs)
        dialog.show()
        return

    #  addSolidRegion_run
    def addSolidRegion_run(self, newName):
        allRegions = self.fluidRegions + self.solidRegions
        if newName in allRegions:
            mess = _(u"同じ名前のregionがあります。")
            self.errDialog(_(u"エラー"), mess)
            return
        self.solidRegions.append(newName)
        #treeListを再表示
        self.setTreeDataToTreeList()
        #regionPropertiesを作成
        cont = pyTreeFoam.getFoamContents().makeRegionProperties(self.fluidRegions, self.solidRegions)
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 11:
            rsOp = pyTreeFoam.strOp(cont)
            regionSolvers, p = rsOp.getMiddlePair(0)
            contDict = {"regionSolvers":regionSolvers}
            pyTreeFoam.case(self.caseDir).setControlDict(contDict)
        else:
            fileName = self.caseDir + "/constant/regionProperties"
            f = open(fileName, "w"); f.write(cont); f.close()
        #folderを作成
        self.addSolidRegionFolder(newName)
        mess = _(u"新しいregion「") + newName + _(u"」を追加しました。")
        self.okDialog(_(u"region追加"), mess)

    def getIniRegionName(self):
        """ 初期のregionNameを取得する"""
        newName = "newRegion_"
        allRegions = self.fluidRegions + self.solidRegions
        i = 0
        while True:
            name = newName + str(i)
            if name in allRegions:
                i += 1
            else:
                break
        return name

    def addFluidRegionFolder(self, newRegion):
        """ fluidRegionのfolderにnewRegionを追加する"""
        case = pyTreeFoam.case(self.caseDir)
        iniTime = case.getTimeFolders()[0]
        loc = self.getBoundaryLocation(iniTime)
        iniTimeDir = self.caseDir + "/" + iniTime
        constantDir = self.caseDir + "/" + loc
        systemDir = self.caseDir + "/system"
        os.mkdir(iniTimeDir + "/" + newRegion)
        self.copyFilesFluidSolid(iniTimeDir, newRegion, "fluid")
        os.mkdir(constantDir + "/" + newRegion)
        self.copyFilesFluidSolid(constantDir, newRegion, "fluid")
        os.mkdir(systemDir + "/" + newRegion)
        self.copyFilesFluidSolid(systemDir, newRegion, "fluid")
        #boundaryを作成
        polyMeshDir = self.caseDir + "/" + loc + "/" + newRegion
        os.mkdir(polyMeshDir + "/polyMesh")
        fileName = polyMeshDir + "/polyMesh/boundary"
        comm = "touch " + fileName
        pyTreeFoam.run(self.caseDir).command(comm)

    def addSolidRegionFolder(self, newRegion):
        """ solidRegionのfolderにnewRegionを追加する"""
        case = pyTreeFoam.case(self.caseDir)
        iniTime = case.getTimeFolders()[0]
        loc = self.getBoundaryLocation(iniTime)
        iniTimeDir = self.caseDir + "/" + iniTime
        constantDir = self.caseDir + "/" + loc
        systemDir = self.caseDir + "/system"
        os.mkdir(iniTimeDir + "/" + newRegion)
        self.copyFilesFluidSolid(iniTimeDir, newRegion, "solid")
        os.mkdir(constantDir + "/" + newRegion)
        self.copyFilesFluidSolid(constantDir, newRegion, "solid")
        os.mkdir(systemDir + "/" + newRegion)
        self.copyFilesFluidSolid(systemDir, newRegion, "solid")
        #boundaryを作成
        polyMeshDir = self.caseDir + "/" + loc + "/" + newRegion
        os.mkdir(polyMeshDir + "/polyMesh")
        fileName = polyMeshDir + "/polyMesh/boundary"
        comm = "touch " + fileName
        pyTreeFoam.run(self.caseDir).command(comm)

    def getBoundaryLocation(self, timeFolder):
        """ boundaryのlocationを返す"""
        regions = self.fluidRegions + self.solidRegions
        relPolyMeshDir = pyTreeFoam.case(self.caseDir).getCurrMeshDir(
                                timeFolder, regions[0], "boundary")
        loc = relPolyMeshDir.split("/")[-3]
        return loc

    def copyFilesFluidSolid(self, folderDir, newFolder, fluidSolid):
        """ 必要なfileをコピーする"""
        if fluidSolid == "fluid":
            copyFolder = folderDir + "/fluidRegions"
        else:
            copyFolder = folderDir + "/solidRegions"
        files = glob.glob(copyFolder+"/*")
        specNames = []
        #ファイルを配布
        for fileName in files:
            name = fileName.split("/")[-1]
            if len(name.split(".")) == 1:
                shutil.copy(fileName, folderDir + "/" + newFolder)
            else:
                specNames.append(fileName)
        #regName付きfileを配布
        for fileName in specNames:
            name = fileName.split("/")[-1]
            words = name.split(".")
            regName = words[1]
            if regName == newFolder:
                newName = folderDir + "/" + newFolder + "/" + words[0]
                shutil.copy(fileName, newName)

    #
    #  changeFluidRegion
    #  -----------------
    def changeFluidRegion(self):
        """ fluidRegion名を修正する"""
        #選択regionを取得
        selNames = self.getFluidSelNames()
        #erroeチェック
        if len(selNames) == 0:
            mess = _(u"regionを選択してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        if len(selNames) > 1:
            mess = _(u"複数のregionは、選択できません。")
            self.errDialog(_(u"エラー"), mess)
            return
        if len(self.fluidRegions) == 0:
            mess = _(u"regionが存在しません。")
            self.errDialog(_(u"エラー"), mess)
            return
        timeFolders = pyTreeFoam.case(self.caseDir).getTimeFolders()
        if len(timeFolders) > 1:
            mess = _(u"timeFolderが複数あります。\n")
            mess += _(u"結果folderを削除してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        #newNameの取得
        selName = selNames[0]
        title = _(u"region名の入力")
        message = _(u"新しいregion名を入力してください。")
        inputText = selName
        okArgs = [self.changeFluidRegion_run, selName]
        dialog = inputTextDDialog.getInputText(title, message, inputText, funcOk=okArgs)
        dialog.show()
        return

    #  changeFluidRegion_run
    def changeFluidRegion_run(self, newName, selName):
        #newNameのチェック
        if newName in (self.fluidRegions + self.solidRegions):
            mess = _(u"入力したregion名は、既に使われています。\n別の名前で入力しなおしてください。")
            self.errDialog(_(u"エラー"), mess)
            return
        #fluidRegionsを追加
        self.fluidRegions.remove(selName)
        self.fluidRegions.append(newName)
        self.fluidRegions.sort()
        #listBoxを修正
        self.setTreeDataToTreeList()
        #listBox中のitemを選択
        self.selectItemsInTreeList(self.tree_fluid, [newName])
        #regionPropertiesの修正
        cont = pyTreeFoam.getFoamContents().makeRegionProperties(self.fluidRegions, self.solidRegions)
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 11:
            rsOp = pyTreeFoam.strOp(cont)
            regionSolvers, p = rsOp.getMiddlePair(0)
            contDict = {"regionSolvers":regionSolvers}
            pyTreeFoam.case(self.caseDir).setControlDict(contDict)
        else:
            fileName = self.caseDir + "/constant/regionProperties"
            f = open(fileName, "w"); f.write(cont); f.close()
        #folder名をrenameする
        self.renameRegionFolder(selName, newName)
        #boundaryの内容を修正
        self.renameBoundaryContents(selName, newName)
        #boundaryFieldの境界patch名修正
        self.renameBoundaryFieldContents(selName, newName)
        mess = _(u"region名を変更しました。")
        self.okDialog(_(u"region名の変更"), mess)

    #
    #  changeSolidRegion
    #  -----------------
    def changeSolidRegion(self):
        """ solidRegion名を修正する"""
        #選択regionを取得
        selNames = self.getSolidSelNames()
        #erroeチェック
        if len(selNames) == 0:
            mess = _(u"regionを選択してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        if len(selNames) > 1:
            mess = _(u"複数のregionは、選択できません。")
            self.errDialog(_(u"エラー"), mess)
            return
        if len(self.solidRegions) == 0:
            mess = _(u"regionが存在しません。")
            self.errDialog(_(u"エラー"), mess)
            return
        timeFolders = pyTreeFoam.case(self.caseDir).getTimeFolders()
        if len(timeFolders) > 1:
            mess = _(u"timeFolderが複数あります。\n")
            mess += _(u"結果folderを削除してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        #newNameの取得
        selName = selNames[0]
        title = _(u"region名の入力")
        message = _(u"新しいregion名を入力してください。")
        inputText = selName
        okArgs = [self.changeSolidRegion_run, selName]
        dialog = inputTextDDialog.getInputText(title, message, inputText, funcOk=okArgs)
        dialog.show()
        return

    #  changeSolidRegion_run       
    def changeSolidRegion_run(self, newName, selName):
        #newNameのチェック
        if newName in (self.fluidRegions + self.solidRegions):
            mess = _(u"入力したregion名は、既に使われています。\n別の名前で入力しなおしてください。")
            self.errDialog(_(u"エラー"), mess)
            return
        #solidRegionsを追加
        self.solidRegions.remove(selName)
        self.solidRegions.append(newName)
        self.solidRegions.sort()
        #listBoxを修正
        self.setTreeDataToTreeList()
        #listBox中のitemを選択
        self.selectItemsInTreeList(self.tree_solid, [newName])
        #regionPropertiesの修正
        cont = pyTreeFoam.getFoamContents().makeRegionProperties(self.fluidRegions, self.solidRegions)
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 11:
            rsOp = pyTreeFoam.strOp(cont)
            regionSolvers, p = rsOp.getMiddlePair(0)
            contDict = {"regionSolvers":regionSolvers}
            pyTreeFoam.case(self.caseDir).setControlDict(contDict)
        else:
            fileName = self.caseDir + "/constant/regionProperties"
            f = open(fileName, "w"); f.write(cont); f.close()
        #folder名をrenameする
        self.renameRegionFolder(selName, newName)
        #boundaryの内容を修正
        self.renameBoundaryContents(selName, newName)
        #boundaryFieldの境界patch名修正
        self.renameBoundaryFieldContents(selName, newName)
        mess = _(u"region名を変更しました。")
        self.okDialog(_(u"region名の変更"), mess)

    def renameRegionFolder(self, selFolder, newFolder):
        """ region名を変更する。folder名の変更のみ。"""
        
        def renameFolder(folderDir, selFolder, newFolder):
            selName = folderDir + "/" + selFolder
            newName = folderDir + "/" + newFolder
            if len(glob.glob(selName)) == 1:
                os.rename(selName, newName)

        times = pyTreeFoam.case(self.caseDir).getTimeFolders()
        for time in times:
            timeDir = self.caseDir + "/" + time
            renameFolder(timeDir, selFolder, newFolder)
        constantDir = self.caseDir + "/constant"
        systemDir = self.caseDir + "/system"
        renameFolder(constantDir, selFolder, newFolder)
        renameFolder(systemDir, selFolder, newFolder)

    def renameBoundaryContents(self, selName, newName):
        """ boundaryの内容を修正する。"""
        allRegions = self.fluidRegions + self.solidRegions
        iniTime = pyTreeFoam.case(self.caseDir).getTimeFolders()[0]
        loc = self.getBoundaryLocation(iniTime)
        for regName in allRegions:
            boundary = self.caseDir + "/" + loc + "/" + regName + "/polyMesh/boundary"
            f = open(boundary); cont = f.read(); f.close()
            cont = self.remakeBoundaryCont(cont, selName, newName)
            f=open(boundary, "w"); f.write(cont); f.close()
        if loc != "constant":
            for regName in allRegions:
                boundary = self.caseDir + "/constant/" + regName + "/polyMesh/boundary"
                f = open(boundary); cont = f.read(); f.close()
                cont = self.remakeBoundaryCont(cont, selName, newName)
                f = open(boundary, "w"); f.write(cont); f.close()

    def remakeBoundaryCont(self, cont, selName, newName):
        """ boundaryのpatch名を修正する"""

        def checkPatchName(patchName, line, selName, newName):
            """ patch名を修正する"""
            newLine = line
            n = len(selName+"_to_")
            if (patchName[:n] == selName+"_to_"):
                newLine = newName + "_to_" + line[n:]
            elif (patchName[-n:] == "_to_" + selName):
                newLine = patchName[:-n] + "_to_" + newName + line[len(patchName):]
            return newLine

        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        (patchesConts, p) = contOp.getSmallPair(p)
        #boundary以外（header, footer）を取得
        ep = p - 1
        sp = p - 1 - len(patchesConts)
        header = cont[:sp]
        footer = cont[ep:]
        #boundary修正
        patchesContsOp = pyTreeFoam.strOp(patchesConts)
        lines = []
        lineCont = " "
        p = 0
        while lineCont != "":
            #1行を取得
            (lineCont, p, _kind) = patchesContsOp.get1line(p)
            (patchName, pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
            n = patchName.find("_to_")
            if n > 0:
                #patch名をチェック修正
                tline = checkPatchName(patchName, lineCont, selName, newName)
                lineCont = tline
                #境界面のpatch内容を修正する
                keyword = " "
                pn = len(selName + "_to_")
                while keyword != "" and pp < len(lineCont):
                    lineContOp = pyTreeFoam.strOp(lineCont)
                    #patch内容に旧のregion名があるか確認
                    (keyword, pp) = lineContOp.getKeyword(pp)
                    if keyword == selName:
                        #旧region名を新に置き換え
                        newCont = lineCont[:pp-len(selName)] + newName + lineCont[pp:]
                        lineCont = newCont
                    elif keyword[:pn] == selName + "_to_":
                        #「旧region_to_xxxx」形式のregion名を新に置き換え
                        newKey = newName + "_to_" + keyword[pn:]
                        newCont = lineCont[:pp-len(keyword)] + newKey + lineCont[pp:]
                        lineCont = newCont
                    elif keyword[-pn:] == "_to_" + selName:
                        #「xxxx_to_旧region」形式のregion名を新に置き換え
                        newKey = keyword[:-pn] + "_to_" + newName
                        newCont = lineCont[:pp-len(keyword)] + newKey + lineCont[pp:]
                        lineCont = newCont
                #1行を保存
                lines.append("    " + lineCont)
            else:
                words = lineCont.split()
                if len(words) == 0:
                    lines.append(lineCont)
                else: 
                    #通常のpatchはそのまま保存
                    lines.append("    " + lineCont)
        #全内容を作成
        allCont = header + "\n" + "\n".join(lines) + "\n" + footer
        return allCont

    def renameBoundaryFieldContents(self, selName, newName):
        """ region内の各fieldの境界patch名変更"""
        regions = self.fluidRegions + self.solidRegions
        case = pyTreeFoam.case(self.caseDir)
        iniTime = case.getTimeFolders()[0]
        iniTimeDir = self.caseDir + "/" + iniTime
        for region in regions:
            fieldDir = iniTimeDir + "/" + region
            fields = case.getFieldNames(iniTime, region)
            for field in fields:
                fileName = fieldDir + "/" + field
                foamFile = pyTreeFoam.foamFile()
                cont = foamFile.read(fileName)
                newCont = self.remakeFieldPatchName(cont, selName, newName)
                foamFile.write(fileName, newCont)

    def remakeFieldPatchName(self, cont, selName, newName):
        """ fieldの境界patchを修正"""

        def checkPatchName(patchName, line, selName, newName):
            """ patch名を修正する"""
            newLine = line
            n = len(selName+"_to_")
            if (patchName[:n] == selName+"_to_"):
                newLine = newName + "_to_" + line[n:]
            elif (patchName[-n:] == "_to_" + selName):
                newLine = patchName[:-n] + "_to_" + newName + line[len(patchName):]
            return newLine

        #boundaryFieldの内容を取得
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        (bCont, p) = contOp.getKeywordContents("boundaryField", p)
        ep = p - 1
        #sp = ep - len(bCont)
        sp = p - len(bCont)
        header = cont[:sp]
        footer = cont[ep:]
        #boundaryを修正
        bContOp = pyTreeFoam.strOp(bCont)
        lines = []
        line = " "
        p = 0
        while line != "" and p < len(bCont):
            #1行を取得
            (line, p, _kind) = bContOp.get1line(p)
            (patchName, _pp) = pyTreeFoam.strOp(line).getKeyword(0)
            n = patchName.find("_to_")
            if n > 0:
                #境界面のpatch名をチェック修正
                newLine = checkPatchName(patchName, line, selName, newName)
                #lineを追加
                lines.append("    " + newLine)
            else:
                words = line.split()
                if len(words) == 0:
                    lines.append(line)
                else: 
                    #通常のpatchはそのまま追加
                    lines.append("    " + line)
        #全内容を追加
        allCont = header + "\n".join(lines) + footer
        return allCont

    #
    #  deleteFluidRegions
    #  ------------------
    def deleteFluidRegions(self):
        """ 複数のfluidRegionを削除する。"""
        delNames = self.getFluidSelNames()
        if len(delNames) == 0:
            mess = _(u"regionを選択してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        #削除の確認
        title = _(u"regionの削除")
        mess = _(u"regionを削除しますか？\n\n削除後は、元に戻りません。")
        okArgs = [self.deleteFluidRegions_run, delNames]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return

    def deleteFluidRegions_run(self, delNames):
        #削除後のnewRegionsを作成
        newRegions = list(set(self.fluidRegions).difference(delNames))
        self.fluidRegions = newRegions
        self.fluidRegions.sort()
        #regionPropertiesの修正
        cont = pyTreeFoam.getFoamContents().makeRegionProperties(self.fluidRegions, self.solidRegions)
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 11:
            rsOp = pyTreeFoam.strOp(cont)
            regionSolvers, p = rsOp.getMiddlePair(0)
            contDict = {"regionSolvers":regionSolvers}
            pyTreeFoam.case(self.caseDir).setControlDict(contDict)
        else:
            fileName = self.caseDir + "/constant/regionProperties"
            f = open(fileName, "w"); f.write(cont); f.close()
        #listBox修正
        self.setTreeDataToTreeList()
        #folder削除
        for delName in delNames:
            self.deleteRegionFolder(delName)
            print(_(u"削除: ") + delName)
        #polyMesh/boundaryのpatchName、patchTypeを修正
        for delName in delNames:
            self.renameBoundaryPatchType(delName)
        #空白boundaryFieldをzerogradientに設定
        regions = self.fluidRegions + self.solidRegions
        case = pyTreeFoam.case(self.caseDir)
        iniTime = case.getTimeFolders()[0]
        for region in regions:
            meshRelDir = case.getCurrMeshDir(iniTime, region, "boundary")
            meshRel = "/".join(meshRelDir.split("/")[:-1])
            polyMeshDir = caseDir + "/" + meshRel
            case.clearNullBoundaryAllFields(iniTime, polyMeshDir, region)
        #終了メッセージ
        mess = _(u"regionを削除しました。\n削除したregionに関係するpolyMesh/boundaryのpatch名とpatchTypeを修正しています。\nboundaryFieldのpatchTypeは、「zeroGradient」に修正しています。")
        self.okDialog(_(u"regionの削除"), mess)

    #
    #  deleteSolidRegions
    #  -----------------
    def deleteSolidRegions(self):
        """ 複数のfluidRegionを削除する。"""
        delNames = self.getSolidSelNames()
        if len(delNames) == 0:
            mess = _(u"regionを選択してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        #削除の確認
        title = _(u"regionの削除")
        mess = _(u"regionを削除しますか？\n\n削除後は、元に戻りません。")
        okArgs = [self.deleteSolidRegions_run, delNames]
        self.okCancelDialog(title, mess, okArgs)
        
    #  deleteSolidRegions_run
    def deleteSolidRegions_run(self, delNames):
        #削除後のnewRegionsを作成
        newRegions = list(set(self.solidRegions).difference(delNames))
        self.solidRegions = newRegions
        self.solidRegions.sort()
        #regionPropertiesの修正
        cont = pyTreeFoam.getFoamContents().makeRegionProperties(self.fluidRegions, self.solidRegions)
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 11:
            rsOp = pyTreeFoam.strOp(cont)
            regionSolvers, p = rsOp.getMiddlePair(0)
            contDict = {"regionSolvers":regionSolvers}
            pyTreeFoam.case(self.caseDir).setControlDict(contDict)
        else:
            fileName = self.caseDir + "/constant/regionProperties"
            f = open(fileName, "w"); f.write(cont); f.close()
        #listBox修正
        self.setTreeDataToTreeList()
        #folder削除
        for delName in delNames:
            self.deleteRegionFolder(delName)
            print(_(u"削除: ") + delName)
        #polyMesh/boundaryのpatchName、patchTypeを修正
        for delName in delNames:
            self.renameBoundaryPatchType(delName)
        #空白boundaryFieldをzerogradientに設定
        regions = self.fluidRegions + self.solidRegions
        case = pyTreeFoam.case(self.caseDir)
        iniTime = case.getTimeFolders()[0]
        for region in regions:
            meshRelDir = case.getCurrMeshDir(iniTime, region, "boundary")
            meshRel = "/".join(meshRelDir.split("/")[:-1])
            polyMeshDir = caseDir + "/" + meshRel
            case.clearNullBoundaryAllFields(iniTime, polyMeshDir, region)
        #終了メッセージ
        mess = _(u"regionを削除しました。\n削除したregionに関係するpolyMesh/boundaryのpatch名とpatchTypeを修正しています。\nboundaryFieldのpatchTypeは、「zeroGradient」に修正しています。")
        self.okDialog(_(u"regionの削除"), mess)

    def deleteRegionFolder(self, delRegion):
        """ regionを削除する"""
        
        def deleteFolder(folderDir, delFolder):
            delName = folderDir + "/" + delFolder
            if len(glob.glob(delName)) != 0:
                shutil.rmtree(delName)
        
        iniTime = pyTreeFoam.case(self.caseDir).getTimeFolders()[0]
        #loc = self.getBoundaryLocation(iniTime)
        iniTimeDir = self.caseDir + "/" + iniTime
        constantDir = self.caseDir + "/constant"
        systemDir = self.caseDir + "/system"
        deleteFolder(iniTimeDir, delRegion)
        deleteFolder(constantDir, delRegion)
        deleteFolder(systemDir, delRegion)

    def renameBoundaryPatchType(self, delRegion):
        """ 削除したregionに関係するpatch名とpatchTypeを修正する"""
        regions = self.fluidRegions + self.solidRegions
        case = pyTreeFoam.case(self.caseDir)
        iniTime = case.getTimeFolders()[0]
        for region in regions:
            boundaryRelDir = case.getCurrMeshDir(iniTime, region, "boundary")
            boundary = self.caseDir + "/" + boundaryRelDir + "/boundary"
            f = open(boundary); cont = f.read(); f.close()
            newCont = self.renameBoundaryPatchType_sub(cont, delRegion)
            f = open(boundary, "w"); f.write(newCont); f.close()

    def renameBoundaryPatchType_sub(self, cont, delRegion):
        """ boundaryの内容を修正"""

        def checkPatchType(patchName, line, delRegion):
            newLine = line
            n = len("_to_" + delRegion)
            if (patchName[-n:] == "_to_" + delRegion):
                lines = []
                lines.append(patchName + "Del")
                lines.append("    {")
                lines.append("        type            wall;")
                conts = line.split("\n")[1:]
                for cont in conts:
                    (keyword, _p) = pyTreeFoam.strOp(cont).getKeyword(0)
                    if keyword == "nFaces" or keyword == "startFace":
                        lines.append(cont)
                lines.append("    }")
                newLine = "\n".join(lines) + "\n"
            return newLine

        #boundaryの内容を取得
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        (patchesConts, p) = contOp.getSmallPair(p)
        #boundary以外（header、footer）を取得
        ep = p -1
        sp = p -1 - len(patchesConts)
        header = cont[:sp]
        footer = cont[ep:]
        #boundaryを修正
        patchesContsOp = pyTreeFoam.strOp(patchesConts)
        lines = []
        line = " "
        p = 0
        while line != "" and p < len(patchesConts):
            #1行を取得
            (line, p, _kind) = patchesContsOp.get1line(p)
            (patchName, _pp) = pyTreeFoam.strOp(line).getKeyword(0)
            n = patchName.find("_to_")
            if n > 0:
                #patchTypeをチェック修正
                tline = checkPatchType(patchName, line, delRegion)
                line = tline
                #1行を保存
                lines.append("    " + line)
            else:
                words = line.split()
                if len(words) == 0:
                    lines.append(line)
                else:
                    #通常の行はそのまま保存
                    lines.append("    " + line)
        #全内容を作成
        allCont = header + "\n" + "\n".join(lines) + footer
        return allCont

    #
    #  deleteAllRegions
    #  ----------------
    def deleteAllRegions(self):
        """ 全regionを削除する。"""
        allRegions = self.fluidRegions + self.solidRegions
        if len(allRegions) == 0:
            mess = _(u"このcaseには、regionがありません。")
            self.errDialog(_(u"エラー"), mess)
            return
        timeFolders = pyTreeFoam.case(self.caseDir).getTimeFolders()
        if len(timeFolders) > 1:
            mess = _(u"timeFolderが複数あります。\n")
            mess += _(u"結果folderを削除してください。")
            self.errDialog(_(u"エラー"), mess)
            return
        #削除の確認
        title = _(u"全regionの削除")
        mess = _(u"全regionをゴミ箱に移動しますか。")
        okArgs = [self.deleteAllRegions_recheck, allRegions]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return
        
    #  deleteAllRegions_recheck
    def deleteAllRegions_recheck(self, allRegions):
        mess = _(u"本当に、全regionを削除しますか？")
        okArgs = [self.deleteAllRegions_run, allRegions]
        self.okCancelDialog(_(u"警告"), mess, funcOk=okArgs)

    #  deleteAllRegions_run
    def deleteAllRegions_run(self, allRegions):
        #全regionを削除
        for region in allRegions:
            self.deleteRegionFolder(region)
        #削除後の終了処理
        self.fluidRegions = []
        self.solidRegions = []
        #treeListを更新
        self.setTreeDataToTreeList()
        #regionPropertiesを削除
        OFversion = pyTreeFoam.readConfigTreeFoam()["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if numVer >= 11.0:
            setDict = {"regionSolvers": ""}
            pyTreeFoam.case(self.caseDir).setControlDict(setDict)
        else:
            delName = self.caseDir + "/constant/regionProperties"
            os.remove(delName)
        mess = _(u"全regionを削除しました。")
        self.okDialog(_(u"全regionの削除"), mess)

    #------------ universalDialogs ---------------------
    #  okDialog
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    #errDialog
    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, 
            parentWin = self.mainWindow, funcOk=funcOk)
        dialog.show()

    #okCancelDialog
    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, 
            funcOk=funcOk, funcCancel=funcCancel, parentWin=self.mainWindow)
        dialog.show()


if __name__ == "__main__":
    import gettext
    import locale
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    locale.bindtextdomain("treefoam", localeDir)
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()
