#!/usr/bin/python3
#  coding: utf-8
#
#       runCoupling.py
#
#       OpenFOAMとFrontISTRの連成解析を開始する。
#
#   23/09/23    新規作成
#      10/20    reateCommTimeFile:共有メモリ用file追加
#      10/27    createInitialTemperatureFile:追加。（バグ）
#               createInitialDispChtssDict:追加。（バグ）
#      10/31    getInitialFieldData:fieldのpatchDataを読込、初期値と
#               していたが、計算前にfieldが存在しない場合、
#               「0」を設定する様に修正。
#      11/06    setInitialSettingHecmwFile:hecmw_ctrl.datファイルの初期化
#               を追加（single or parallelの内容）
#      11/20    renameInitialDataFile:chtとchtssの同時計算時、dispの初期値が
#               設定されない事を修正（バグ）
#      11/28    mmap関連をallCommonConcAttnモジュールから取得する様に修正
#   24/02/14    getOpenFOAMScript:teeCorrect.py→teeに戻す。
#      04/23    setRestartFile_fsi:startTimeをそのまま使ってrestartFile名を作成
#               する様に修正。
#               setRestartFile_fsi,setRestartFile_cht:time2str関数部を削除
#      04/24    createInitialDispFiles:pressFile名の初期timeNameを「0.000000」から「0」に変更
#               get1stRunScript:初期scriptのtimeNameを「0.000000」から「0」に変更
#               getTimeNameFromTimeFolder:timeNameを実在するfolder名に修正する関数を追加
#               renamePressFile,renameTempFile:time2str関数部を削除
#               setRestartFile_fsi,setRestartFile_cht:restartFileをtimeNameで検索して
#               発見できなかった場合、timeNameを数値に変換して、数値を比較する事を追加。
#

import os
import pyCoupling
import glob
import pickle
import shutil
import time
import mmap
import pyTreeFoam_ptc as pyTreeFoam
import pyFistr_ptc as pyFistr
import createPointCloudFromOpenFoam as createPtc
import mappingFromPointCloud as mapping
import allCommonConcAltn as allComm

waitCount = 50000
waitTime = 0.01


#-------------
#  initialize
#-------------
def initialize():
    """ 連成解析の初期化"""
    dataDir = "coupling_FrontISTR/data"
    #--- OpenFOAM --------------
    #dataフォルダのクリア
    clearDataFolder(dataDir)
    #全procsの流体側patch内容を取得
    createPatchCont()
    #--- FrontISTR --------------
    #固体側のsgrp内容を取得
    (sgrpDict, nodeDict, allNodeDict) = createSgrpCont()
    #固体のdictをbinで保存
    saveSgrpNodeDict(sgrpDict, nodeDict)
    #FrontISTRのrestartの設定、folderをクリア
    clearRestartSettingAndFolder()
    #Fistrの計算初期温度fileを作成
    err = createInitialTemperatureFile()
    if err != "":
        exit()
    #Fistrの初期設定
    setInitialSettingFistr()
    setInitialSettingHecmwFile()
    #---- 両方 --------------------
    #初期のdataFile作成
    createInitialFiles(allNodeDict)

#
#  clearDataFolder
#------------------
def clearDataFolder(dataDir):
    """ dataフォルダのクリア"""
    dataDir = fluidCaseDir + "/coupling_FrontISTR"
    files = glob.glob(dataDir + "/data/*")
    for file in files:
        name = os.path.basename(file)
        if name != "times":
            os.remove(file)
    #FrontISTRのlogファイルを初期化
    name = fluidCaseDir + "/solidSolve.log"
    f = open(name, "w"); f.close()

#
#  createPatchCont
#------------------
def createPatchCont():
    """ OpenFOAM側のfaceCenterLocs（face中心座標）などmesh情報を取得"""
    patchName = couplingDict["mappingPatch"]
    nProcs = couplingDict["nProcs"]
    if nProcs == "1":
        #singleコアの場合
        fluidCase = pyTreeFoam.case(fluidCaseDir)
        time = fluidCase.getCurrTimeFolder()
        region= "."
        pMesh = fluidCase.getCurrMeshDir(time, region, "boundary")
        meshDir = fluidCaseDir + os.sep + pMesh.split(os.sep)[0]
        polyMesh = createPtc.ofMesh(fluidCaseDir, meshDir)
        faceCenterDict = polyMesh.getFaceCenterDict([patchName])
        #faceCenter座標を取得
        faceCenterLocs = getFaceCenterLocs(faceCenterDict)
        #mapPointFaceDict, nodeOrderDictを作成
        allFaceDict = polyMesh.faceDict()
        allNodeDict = polyMesh.nodeDict()
        mapPointFaceDict, nodeOrderDict = getDictsFromFaceCenterDict(
            faceCenterDict, allFaceDict, allNodeDict)
        #取得したmesh情報を保存
        # _faceCenterLocs_, _mapPointFaceNodes_を作成
        saveFaceNodesPara(faceCenterLocs, mapPointFaceDict, nodeOrderDict, "0")
        #face中心にmappingする為のdata作成
        mapFaceCenterDict = getMapFaceCenterDict(faceCenterDict, allFaceDict)
        #ファイルに保存
        saveMapFaceCenterDictPara(mapFaceCenterDict, "0")
    else:
        #parallelの場合
        fluidParaCase = pyTreeFoam.paraCase(fluidCaseDir)
        time = fluidParaCase.getCurrTimeFolder()
        region= "."
        pMesh = fluidParaCase.getCurrMeshDir(time, region, "boundary")
        for p in range(int(nProcs)):
            pNo = str(p)
            meshDir = fluidCaseDir + "/processor" + pNo + "/" + pMesh.split(os.sep)[0]
            polyMesh = createPtc.ofMesh(fluidCaseDir + "/processor" + pNo, meshDir)
            faceCenterDict = polyMesh.getFaceCenterDict([patchName])
            #faceCenter座標を取得
            faceCenterLocs = getFaceCenterLocs(faceCenterDict)
            #mapPointFaceDict, nodeOrderDictを作成
            allFaceDict = polyMesh.faceDict()
            allNodeDict = polyMesh.nodeDict()
            mapPointFaceDict, nodeOrderDict = getDictsFromFaceCenterDict(
                faceCenterDict, allFaceDict, allNodeDict)
            saveFaceNodesPara(faceCenterLocs, mapPointFaceDict, nodeOrderDict, pNo)
            mapFaceCenterDict = getMapFaceCenterDict(faceCenterDict, allFaceDict)
            saveMapFaceCenterDictPara(mapFaceCenterDict, pNo)

#
#  getFaceCenterLocs
def getFaceCenterLocs(faceCenterDict):
    """ faceCenterDictからfaceCenterLocsを取得して返す"""
    faceNos = list(faceCenterDict.keys())
    faceNos.sort()
    faceCenterLocs = []
    for faceNo in faceNos:
        loc = faceCenterDict[faceNo]
        strLoc = list(map(str, loc))
        faceCenterLocs.append(strLoc)
    return faceCenterLocs

#
#  getDictsFromFaceCenterDict
def getDictsFromFaceCenterDict(faceCenterDict, allFaceDict, allNodeDict):
    """ mapPointFaceDictとnodeOrderDictを作成。"""
    mapPointFaceDict = {}
    nodeOrderDict = {}
    order = 0
    for faceNo in faceCenterDict.keys():
        nodeNos = allFaceDict[faceNo]["nodes"]
        for nodeNo in nodeNos:
            if not nodeNo in nodeOrderDict.values():
                loc = allNodeDict[nodeNo]
                mapPointFaceDict[nodeNo] = [loc, nodeNos]
                nodeOrderDict[order] = nodeNo
                order += 1
    return mapPointFaceDict, nodeOrderDict

#
#  saveFaceNodesPara
def saveFaceNodesPara(faceCenterLocs, mapPointFaceDict, nodeOrderDict, pNo):
    """ couplingで使用するmesh内容を保存する"""
    patchName = couplingDict["mappingPatch"]
    #faceCenterLocを保存
    saveDir = fluidCaseDir + "/coupling_FrontISTR/data/"
    name = patchName + "_faceCenterLocs_" + pNo
    fileName = saveDir + name
    lines = []
    for strLoc in faceCenterLocs:
        line = " ".join(strLoc) + "\n"
        lines.append(line)
    f = open(fileName, "w"); f.writelines(lines); f.close()

    #mapPointFaceDictを保存
    name = patchName + "_mapPointFaceNodes_" + pNo
    fileName = saveDir + name
    lines = []
    nodeOrders = list(nodeOrderDict.keys())
    nodeOrders.sort()
    num = len(mapPointFaceDict.keys())
    for orderNo in nodeOrders:
        nodeNo = nodeOrderDict[orderNo]
        loc = mapPointFaceDict[nodeNo][0]
        strLoc = list(map(str, loc))
        faceNodeNos = mapPointFaceDict[nodeNo][1]
        strNodeNos = list(map(str, faceNodeNos))
        line = " ".join([str(nodeNo)] + strLoc + strNodeNos) + "\n"
        lines.append(line)
    f = open(fileName, "w"); f.writelines(lines); f.close()
    return

#
#  getMapFaceCenterDict
def getMapFaceCenterDict(faceCenterDict, allFaceDict):
    """ face中心にmappingする為のmapFaceCenterDictを作成する。
    key=faceNo, value=[[faceCenterLoc], [nodeNos]]"""
    mapDict = {}
    for faceNo in faceCenterDict.keys():
        loc = faceCenterDict[faceNo]
        nodeNos = allFaceDict[faceNo]["nodes"]
        mapDict[faceNo] = [loc] + [nodeNos]
    return mapDict

#
#  saveMapFaceCenterDictPara
def saveMapFaceCenterDictPara(mapFaceCenterDict, pNo):
    """ face中心にmappingする為のdataを保存する"""
    patchName = couplingDict["mappingPatch"]
    saveDir = fluidCaseDir + "/coupling_FrontISTR/data/"
    name = patchName + "_mapFaceCenterNodes_" + pNo
    fileName = saveDir + name
    lines = []
    for faceNo in mapFaceCenterDict.keys():
        loc, nodeNos = mapFaceCenterDict[faceNo]
        words = map(str, [faceNo] + loc + nodeNos)
        line = " ".join(words) + "\n"
        lines.append(line)
    f = open(fileName, "w"); f.writelines(lines); f.close()

#
#  createSgrpCont
#------------------
def createSgrpCont():
    """ FrontISTR側のSGRPの情報を取得して保存する"""
    solidCaseDir = couplingDict["solidCaseDir"]
    sgrpName = couplingDict["mappingSgrp"]
    mesh = mapping.fistrMesh(solidCaseDir)
    meshHeaderData = mesh.getMeshHeaderData()
    nodeDict = mesh.nodeDict(meshHeaderData)
    elementDict = mesh.elementDict(meshHeaderData)
    sgrpData = mesh.getSgrpData(sgrpName, meshHeaderData)
    sgrpDict = mesh.getSgrpDict(sgrpData, nodeDict, elementDict)
    newNodeDict = getNeedNodesFromDict(sgrpDict, nodeDict)
    #Dictを保存
    saveFistrFaceData(sgrpDict, newNodeDict)
    return (sgrpDict, newNodeDict, nodeDict)

#
#  getNeedNodesFromDict
def getNeedNodesFromDict(sgrpDict, nodeDict):
    """ 必要なnodeを抽出したnodeDictを取得して返す。"""
    newNodeDict = {}
    for key in sgrpDict.keys():
        nodeNos = sgrpDict[key][1]
        for nodeNo in nodeNos:
            if not nodeNo in newNodeDict.keys():
                newNodeDict[nodeNo] = nodeDict[nodeNo]
    return newNodeDict

#
#  saveFistrFaceData
def saveFistrFaceData(sgrpDict, nodeDict):
    """ FrontISTR側のfaceDataを保存する"""
    sgrpName = couplingDict["mappingSgrp"]
    #sgrpDictを保存
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    name = sgrpName + "_mapFaceCenterFaceNodes"
    fileName = dataDir + "/" + name
    lines = []
    for key in sgrpDict.keys():
        elmNo, faceNo = key
        loc = sgrpDict[key][0]
        nodeNos = sgrpDict[key][1]
        nums = [elmNo, faceNo] + loc + nodeNos
        line = " ".join(map(str, nums)) + "\n"
        lines.append(line)
    f = open(fileName, "w"); f.writelines(lines); f.close()
    #nodeDictを保存
    name = sgrpName + "_nodeLocs"
    fileName = dataDir + "/" + name
    lines = []
    for nodeNo in nodeDict.keys():
        loc = nodeDict[nodeNo]
        nums = [nodeNo] + loc
        line = " ".join(map(str, nums)) + "\n"
        lines.append(line)
    f = open(fileName, "w"); f.writelines(lines); f.close()

#
#  saveSgrpNodeDict
#-------------------
def saveSgrpNodeDict(sgrpDict, nodeDict):
    """ sgrpDict, nodeDictをbinで保存"""
    solidCaseDir = couplingDict["solidCaseDir"]
    sgrpDictName = solidCaseDir + "/sgrpDict.bin"
    nodeDictName = solidCaseDir + "/nodeDict.bin"
    f = open(sgrpDictName, "wb")
    pickle.dump(sgrpDict, f)
    f.close()
    f = open(nodeDictName, "wb")
    pickle.dump(nodeDict, f)
    f.close()

#
#  clearRestartSettingAndFolder
#-------------------------------
def clearRestartSettingAndFolder():
    """ FistrMode.cntファイルのresartの設定とfolder内をクリアする"""
    fsi = couplingDict["fsi"]
    cht = couplingDict["cht"]
    solidCaseDir = couplingDict["solidCaseDir"]
    nStepsOF = couplingDict["nStepsOF"]
    patchName = couplingDict["mappingPatch"]
    deltaT = couplingDict["deltaT"]
    if fsi == "yes":
        fileName = solidCaseDir + "/FistrModel_fsi.cnt"
        f = open(fileName); lines = f.readlines(); f.close()
        #restartの設定
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=1")
        #nStepsをクリア
        i = pyFistr.getNextHeaderName(lines, 0, "!DYNAMIC", "")
        i += 2
        if i >= len(lines):
            print("error: it is not DYNAMIC analysis. must set at DYNMIC.")
            print()
            return
        words = pyFistr.deleteSp(lines[i]).split(",")
        words[2] = nStepsOF
        lines[i] = ", ".join(words) + "\n"
        #保存
        f = open(fileName, "w"); f.writelines(lines); f.close()
    if cht == "yes":
        fileName = solidCaseDir + "/FistrModel_cht.cnt"
        f = open(fileName); lines = f.readlines(); f.close()
        #restartの設定
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=1")
        #nStepsをクリア
        i = pyFistr.getNextHeaderName(lines, 0, "!HEAT", "")
        i += 1
        if i >= len(lines):
            print("error: it is not HEAT analysis. must set at HEAT.")
            print()
            return
        words = pyFistr.deleteSp(lines[i]).split(",")
        num = int(nStepsOF) * float(deltaT)
        #endTime = roundVal(num, 8)
        endTime = num
        words[1] = str(endTime)
        lines[i] = ", ".join(words) + "\n"
        #保存
        f = open(fileName, "w"); f.writelines(lines); f.close()
    #結果fileを全て削除
    delFiles = glob.glob(solidCaseDir + "/*.pvtu")
    delFiles += glob.glob(solidCaseDir + "/FistrModel.res.*")
    delFiles += glob.glob(solidCaseDir + "/FistrModel_temp.res.*")
    delFiles += glob.glob(solidCaseDir + "/*.particles")
    delFiles += glob.glob(solidCaseDir + "/*.vtk")
    delFiles += glob.glob(solidCaseDir + patchName + "_faceCenterPress*.particles")
    for delFile in delFiles:
        os.remove(delFile)
    delFolders = glob.glob(solidCaseDir + "/FistrModel.vis_psf.*")
    for delFolder in delFolders:
        if os.path.isdir(delFolder) == True:
            shutil.rmtree(delFolder)

#
#  createInitialTemperatureFile
#-------------------------------
def createInitialTemperatureFile():
    """ 温度解析時の計算初期温度fileを作成"""
    if couplingDict["cht"] == "yes":
        #initialTemperatureファイルを作成
        solidCaseDir = couplingDict["solidCaseDir"]
        _cntName, mshName = pyFistr.cntMshFileName(solidCaseDir)
        mshFile = solidCaseDir + os.sep + mshName
        f = open(mshFile); lines = f.readlines(); f.close()
        headerData = pyFistr.getHeaderData(lines)
        tempLines = []
        for i in range(len(headerData)):
            header = headerData[i][0]
            if header[:len("!INITIAL CONDITION")] == "!INITIAL CONDITION":
                tempLines = headerData[i][1]
                break
        if len(tempLines) == 0:
            tempLines = ["ALL, 300.0\n"]
        iniTempFile = solidCaseDir + os.sep + "initialTemperature"
        if len(tempLines) == 1:
            #initialTemperatureに書き込み
            f = open(iniTempFile, "w"); f.writelines(tempLines); f.close()
        else:
            #複数行のinitialTempの場合、initialTemperatureファイル有無check
            if os.path.exists(iniTempFile) == False:
                print("ERROR: 'initialtemperature' file does not exist!!")
                print("       set initial temperature using EasyISTR.")
                print("       after that, its file will be created.")
                return "error"
    return ""

#
#  setInitialSettingFistr
#-------------------------
def setInitialSettingFistr():
    """ Fistr側の初期設定"""
    deltaT = couplingDict["deltaT"]
    nStepsOF = couplingDict["nStepsOF"]
    nSteps = couplingDict["nSteps"]
    solidCaseDir = couplingDict["solidCaseDir"]
    fsi = couplingDict["fsi"]
    cht = couplingDict["cht"]
    nSubSteps = getSubStepsNumInFistr()
    #step時間を取得
    dT = float(deltaT)
    #step数とstep時間を設定（時間増分）
    if fsi == "yes":
        couplingInterval = dT * int(nStepsOF)
        stepTime = str(couplingInterval / int(nStepsOF))
        cntFile = solidCaseDir + os.sep + "FistrModel_fsi.cnt"
        #cntFileを修正
        fileName = cntFile
        f = open(fileName); lines = f.readlines(); f.close()
        i = pyFistr.getNextHeaderName(lines, 0, "!DYNAMIC", "")
        i += 2
        if i >= len(lines):
            i = pyFistr.getNextHeaderName(lines, 0, "!SOLUTION", "")
            lines[i] = pyFistr.setValName(lines[i], "TYPE=DYNAMIC")
            i = pyFistr.getNextHeaderName(lines, 0, "!HEAT", "")
            lines[i] = "!DYNAMIC, TYPE=NONLINEAR\n"
            si = i
            header = lines[:i+1]
            i = pyFistr.skipNextHeader(lines, i+1)
            footer = lines[i:]
            #時間増分:0.0001, 全step数:2 を設定
            contLines  = ["1, 1\n"]
            contLines += ["0.0, 1.0, 2, 0.0001\n"]
            contLines += ["0.5, 0.25\n"]
            contLines += ["1, 1, 0.01, 0.01\n"]
            contLines += ["100, 1, 1\n"]
            contLines += ["0, 0, 0, 0, 0, 0\n"]
            lines = header + contLines + footer
            i = si + 2
        words = pyFistr.deleteSp(lines[i]).split(",")
        words[2] = nStepsOF       #全step数
        words[3] = stepTime     #時間増分
        lines[i] = ", ".join(words) + "\n"
        #結果の書き込み頻度を設定
        fr = str(int(nStepsOF) * nSubSteps)
        i = pyFistr.getNextHeaderName(lines, 0, "!WRITE", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY="+fr)
        i = pyFistr.getNextHeaderName(lines, i+1, "!WRITE", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY="+fr)
        #cntファイル書き込み
        fw = open(fileName, "w"); lines = fw.writelines(lines); fw.close()
    if cht == "yes":
        couplingInterval = dT * int(nSteps)
        stepTime = str(couplingInterval / int(nSteps))
        cntFile = solidCaseDir + os.sep + "FistrModel_cht.cnt"
        mshFile = solidCaseDir + os.sep + "FistrModel_cht.msh"
        #cntFileを修正
        fileName = cntFile
        f = open(fileName); lines = f.readlines(); f.close()
        i = pyFistr.getNextHeaderName(lines, 0, "!HEAT", "")
        i += 1
        if i >= len(lines):
            i = pyFistr.getNextHeaderName(lines, 0, "!SOLUTION", "")
            lines[i] = pyFistr.setValName(lines[i], "TYPE=HEAT")
            i = pyFistr.getNextHeaderName(lines, 0, "!DYNAMIC", "")
            si = i
            header = lines[:i]
            i = pyFistr.skipNextHeader(lines, i+1)
            footer = lines[i:]
            #時間増分:0.0001, 終了時間:0.0002(全step数:2) を設定
            solLines  = "!HEAT\n"
            solLines += "0.0001, 0.0002, 0, 20.0, 20, 1.0e-6\n"
            lines = header + solLines + footer
            i = si + 1
        words = pyFistr.deleteSp(lines[i]).split(",")
        endTime = str(couplingInterval)
        words[0] = stepTime     #時間増分
        words[1] = endTime      #終了時間
        lines[i] = ", ".join(words) + "\n"
        #結果の書き込み頻度を設定
        fr = nSteps
        #fr = nStepsOF
        i = pyFistr.getNextHeaderName(lines, 0, "!WRITE", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY="+fr)
        i = pyFistr.getNextHeaderName(lines, i+1, "!WRITE", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY="+fr)
        #cntファイル書き込み
        fw = open(fileName, "w"); lines = fw.writelines(lines); fw.close()
        #mshファイルのINITIAL_CONDITIONの初期設定
        iniTempFile = solidCaseDir + os.sep + "initialTemperature"
        f = open(iniTempFile); tempLines = f.readlines(); f.close()
        fileName = mshFile
        f = open(fileName); lines = f.readlines(); f.close()
        headerData = pyFistr.getHeaderData(lines)
        for i in range(len(headerData)):
            header = headerData[i][0]
            if header[:len("!INITIAL CONDITION")] == "!INITIAL CONDITION":
                headerData[i][1] = tempLines
                break
        #mshファイル書込
        lines = []
        for header in headerData:
            lines += [header[0]]
            lines += header[1]
        f = open(fileName, "w"); f.writelines(lines); f.close()

#
#  getSubStepsNumInFistr
def getSubStepsNumInFistr():
    """ FistrModel.cntファイルからsubStep数を取得する。
    !STEPがない場合は、「1」を返す。"""
    solidCaseDir = couplingDict["solidCaseDir"]
    nSubSteps = 1
    fileName = solidCaseDir + "/FistrModel.cnt"
    f = open(fileName); lines = f.readlines(); f.close()
    i = pyFistr.getNextHeaderName(lines, 0, "!STEP", "")
    if i < len(lines):
        name = pyFistr.getValName(lines[i], "SUBSTEPS")
        nSubSteps = int(name)
    return nSubSteps

#
#  setInitialSettingHecmwFile
#-----------------------------
def setInitialSettingHecmwFile():
    """ hecmw_ctrl.datファイルの初期設定
    （singleCore or parallelを設定する）"""
    if couplingDict["fsi"] == "yes":
        hecmwFile = "hecmw_ctrl_fsi.dat"
        setHecmwContSingleOrParallel(hecmwFile)
    if couplingDict["cht"] == "yes":
        hecmwFile = "hecmw_ctrl_cht.dat"
        setHecmwContSingleOrParallel(hecmwFile)
    if couplingDict["chtss"] == "yes":
        hecmwFile = "hecmw_ctrl_chtss.dat"
        setHecmwContSingleOrParallel(hecmwFile)
        #singleCoreに設定
        #mshHeader = "FistrModel"
        #setHecmwContSingle(hecmwFile, mshHeader)

#
#  setHecmwContSingleOrParallel
def setHecmwContSingleOrParallel(hecmwFile):
    """ hecmw_cnt.datファイルをmeshFileの状況によって設定する。
    meshファイルが複数個に分割されていれば、parallelに設定する。"""
    #hecmw_cnt.datファイル読み込み
    solidCaseDir = couplingDict["solidCaseDir"]
    FistrModelDict = pyFistr.getCntMshFileNameDict(solidCaseDir, datFile=hecmwFile)
    mshName = FistrModelDict["fstrMSH"]
    if mshName[-4:] == ".msh":
        mshHeader = mshName.split(".")[0]
    else:
        mshHeader = "_".join(mshName.split("_")[:-1])
    if mshHeader == "":
        print("error: could not read " + hecmwFile + " !!!")
        return
    #メッシュfileの分割数から並列数を取得
    np = pyCoupling.getNumProcsFistrFromMshNum(solidCaseDir, datFile=hecmwFile)
    #hcemwFileを設定
    if np == 1:
        if mshName[-4:] == ".msh":
            return
        #singleCoreの設定
        setHecmwContSingle(hecmwFile, mshHeader)
    else:
        words = mshName.split("_")
        if len(words) >= 2:
            if words[-1] == "p" + str(np):
                return
        #並列処理の設定
        setHecmwContParallel(hecmwFile, mshHeader, np)

#
# setHecmwContSingle
def setHecmwContSingle(hecmwFile, mshHeader):
    """ 指定したhecmwFileの内容をparallel→singleに変更する"""
    #lines = readHecmwCtrlDat()
    fileName = couplingDict["solidCaseDir"] + "/" + hecmwFile
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    FistrModel_msh = mshHeader + ".msh"
    #for partitionを削除
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=part_in")
    if i < len(lines):
        lines = pyFistr.delete1header(lines, i)
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=part_out")
    if i < len(lines):
        lines = pyFistr.delete1header(lines, i)
    #for solver
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=fstrMSH")
    lines[i] = pyFistr.setValName(lines[i], "TYPE=HECMW-ENTIRE")
    #lines[i+1] = "FistrModel.msh" + ls
    lines[i+1] = FistrModel_msh + "\n"
    #for visualizer
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=mesh")
    lines[i] = pyFistr.setValName(lines[i], "TYPE=HECMW-ENTIRE")
    #lines[i+1] = "FistrModel.msh" + ls
    lines[i+1] = FistrModel_msh + "\n"
    #writeHecmwCtrlDat(lines)
    f = open(fileName, "w"); f.writelines(lines); f.close()

#
# setHecmwContParallel
def setHecmwContParallel(hecmwFile, mshHeader, np):
    """ 指定したhecmwFileの内容をsingle→parallelに変更する"""
    #lines = readHecmwCtrlDat()
    fileName = couplingDict["solidCaseDir"] + "/" + hecmwFile
    f = open(fileName, encoding="utf-8"); lines = f.readlines(); f.close()
    nCpu = str(np)
    ls = "\n"
    FistrModel_msh = mshHeader + ".msh"
    #for partitionerを削除（無ければ何もしない）
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=part_in")
    if i < len(lines):
        lines = pyFistr.delete1header(lines, i)
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=part_out")
    if i < len(lines):
        lines = pyFistr.delete1header(lines, i)
    #for partitionerを追加
    #headerName = pyFistr.getHeaderName(FistrModel_msh)
    addLines =  ["!MESH, NAME=part_in, TYPE=HECMW-ENTIRE" + ls]
    #addLines += ["FistrModel.msh" + ls]
    addLines += [FistrModel_msh + ls]
    addLines += ["!MESH, NAME=part_out, TYPE=HECMW-DIST" + ls]
    #addLines += ["FistrModel_p" + nCpu + ls]
    #addLines += [headerName + "_p" + nCpu + ls]
    addLines += [mshHeader + "_p" + nCpu + ls]
    i = pyFistr.getInsertPointInCnt(lines, "!MESH", ["partition"])
    lines = lines[:i] + addLines + lines[i:]
    #for solver
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=fstrMSH")
    lines[i] = pyFistr.setValName(lines[i], "TYPE=HECMW-DIST")
    #lines[i+1] = "FistrModel_p" + nCpu + ls
    lines[i+1] = mshHeader + "_p" + nCpu + ls
    #for visualizer
    i = pyFistr.getNextHeaderName(lines, 0, "!MESH", "NAME=mesh")
    lines[i] = pyFistr.setValName(lines[i], "TYPE=HECMW-DIST")
    #lines[i+1] = "FistrModel_p" + nCpu + ls
    lines[i+1] = mshHeader + "_p" + nCpu + ls
    #writeHecmwCtrlDat(lines)
    f = open(fileName, "w"); f.writelines(lines); f.close()

#
#  createInitialFiles
#---------------------
def createInitialFiles(allNodeDict):
    """ 計算初期で読み込むdataFileを作成する"""
    #初期変位dataFileを作成
    if couplingDict["fsi"] == "yes" or couplingDict["chtss"] == "yes":
        if float(couplingDict["startTime"]) == 0.0:
            #初期から開始の場合
            createInitialDispFiles()
    if couplingDict["cht"] == "yes":
        if float(couplingDict["startTime"]) == 0.0:
            createInitialHeatFluxFiles()
    if couplingDict["fsi"] == "yes" and couplingDict["chtss"] == "yes":
        if float(couplingDict["startTime"]) == 0.0:
            createInitialDispChtssDict(allNodeDict)

#
#  createInitialDispFiles
def createInitialDispFiles():
    """ 初期の変位データを作成する"""
    nProcs = couplingDict["nProcs"]
    patchName = couplingDict["mappingPatch"]
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    for i in range(int(nProcs)):
        procNo = str(i)
        name = patchName + "_mapPointFaceNodes_" + procNo
        fileName = dataDir + "/" + name
        f = open(fileName); lines = f.readlines(); f.close()
        #保存のfileName
        endStep = "0"
        name = patchName + "_disp_" + endStep + "_" + procNo + "tm"
        dispFile = dataDir + "/" + name
        dataLine = "0.0 0.0 0.0\n"
        f = open(dispFile, "w")
        for ii in range(len(lines)):
            f.write(dataLine)
        f.close()
        #圧力の初期値作成
        name = patchName + "_mapFaceCenterNodes_" + procNo
        fileName = dataDir + "/" + name
        f = open(fileName); lines = f.readlines(); f.close()
        procNo = str(i)
        #name = patchName + "_facePress_0.000000_" + procNo + "tm"
        name = patchName + "_facePress_0_" + procNo + "tm"
        pressFile = dataDir + "/" + name
        #lines = getInitialFieldData(procNo)
        lines = getInitialPressFieldData(procNo)
        f = open(pressFile, "w"); f.writelines(lines); f.close()

#
#  getInitialTempFieldData
def getInitialTempFieldData(procNo):
    """ 温度T fieldのpatchdataを取得して返す"""
    tempField = couplingDict["field"]
    #p,Tの場合？
    fields = tempField.split(",")
    if len(fields) == 2:
        tempField = fields[1]
    lines = getInitialFieldData(procNo, tempField)
    return lines 

#
#  getInitialPressFieldData
def getInitialPressFieldData(procNo):
    """ 圧力p fieldのpatchdataを取得して返す"""
    pressField = couplingDict["field"]
    #p,Tの場合？
    fields = pressField.split(",")
    if len(fields) == 2:
        pressField = fields[0]
    lines = getInitialFieldData(procNo, pressField)
    return lines 

#
#  getInitialFieldData
def getInitialFieldData(pNo, field):
    """ patchの dataを取得して返す"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    nProcs = couplingDict["nProcs"]
    patchName = couplingDict["mappingPatch"]
    if int(nProcs) == 1:
        #シングルコア
        fluidCase = pyTreeFoam.case(fluidCaseDir) 
        time = fluidCase.getCurrTimeFolder()
        region= "."
        pMesh = fluidCase.getCurrMeshDir(time, region, "boundary")
        meshDir = fluidCaseDir + os.sep + pMesh.split(os.sep)[0]
        polyMesh = createPtc.ofMesh(fluidCaseDir, meshDir)
        fieldDir = fluidCaseDir + "/" + time + "/" + region
    else:
        #MPI並列の場合
        fluidParaCase = pyTreeFoam.paraCase(fluidCaseDir) 
        time = fluidParaCase.getCurrTimeFolder()
        region= "."
        pMesh = fluidParaCase.getCurrMeshDir(time, region, "boundary")
        meshDir = fluidCaseDir + "/processor" + pNo + "/" + pMesh.split(os.sep)[0]
        polyMesh = createPtc.ofMesh(fluidCaseDir + "/processor" + pNo, meshDir)
        fieldDir = fluidCaseDir + "/processor" + pNo + "/" + time + "/" + region
    fieldNames = glob.glob(fieldDir + "/*")
    fields = list(map(lambda x: os.path.basename(x), fieldNames))
    if field in fields:
        #patchDataを取得
        patchDataDict = polyMesh.patchDataDictFromField(time, field, [patchName], region)
        lines = []
        keys = list(patchDataDict.keys())
        keys.sort()
        for key in keys:
            line = str(patchDataDict[key]) + "\n"
            lines.append(line)
    else:
        #fieldが存在しない場合：初期値を「0」で設定
        lines = []
        value = couplingDict["mappingShift"]
        dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
        name = patchName + "_faceCenterLocs_" + pNo
        fileName = dataDir + "/" + name
        f = open(fileName); locLines = f.readlines(); f.close()
        for i in range(len(locLines)):
            lines.append(value + "\n")
    return lines

#
#  createInitialHeatFluxFiles
#-----------------------------
def createInitialHeatFluxFiles():
    """ 初期のheatFlux、温度データを作成する"""
    nProcs = couplingDict["nProcs"]
    patchName = couplingDict["mappingPatch"]
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    for i in range(int(nProcs)):
        procNo = str(i)
        #保存のfileName
        endStep = "0"
        name = patchName + "_temp_" + endStep + "_" + procNo + "tm"
        dispFile = dataDir + "/" + name
        #温度（fieldData）を取得
        #lines = getInitialFieldData(procNo)
        lines = getInitialTempFieldData(procNo)
        f = open(dispFile, "w"); f.writelines(lines); f.close()
        #圧力の初期値作成
        #name = patchName + "_faceHeatFlux_0.000000_" + procNo + "tm"
        name = patchName + "_faceHeatFlux_0_" + procNo + "tm"
        pressFile = dataDir + "/" + name
        dataLine = "0.0\n"
        f = open(pressFile, "w")
        for ii in range(len(lines)):
            f.write(dataLine)
        f.close()

#
#  createInitialDispChtssDict
def createInitialDispChtssDict(allNodeDict):
    """ 初期のchtss変位を「0」で作成"""
    solidCaseDir = couplingDict["solidCaseDir"]
    dispChtssDict = {}
    for nodeNo in allNodeDict.keys():
        #変位を設定
        dispChtssDict[nodeNo] = [0.0, 0.0, 0.0]
    fileName = solidCaseDir + "/dispChtssDict.bin"
    f = open(fileName, "wb")
    pickle.dump(dispChtssDict, f)
    f.close()

#------------------
#  createRunScript
#------------------
def createRunScript():
    """ 起動スクリプトを作成する"""
    comm = "#!/bin/bash\n"
    #FrontISTRの待ち受けを起動
    comm += "python3 coupling_FrontISTR/python/"
    comm += "waitToRunFrontISTR.py &\n"
    #FrontISTRの初回起動
    #  v2206系は、pythonスクリプトを起動する
    #    v2206系:時間loopの最後でcontrolDictのcodedを実行
    #    OF-9系:時間loopの最初でcodedを実行する為
    OFversion = os.getenv("WM_PROJECT_VERSION")
    if OFversion[0] == "v":
        comm += get1stRunScript()
    #OpenFOAMの起動
    comm += getOpenFOAMScript()
    #fileに保存
    fluidCaseDir = couplingDict["fluidCaseDir"]
    fileName = fluidCaseDir + "/startCoupling"
    f = open(fileName, "w"); f.write(comm); f.close()
    pyFistr.run().command("chmod a+x " + fileName)
    return fileName

#
#  get1stRunScript
def get1stRunScript():
    """ 初回起動のスクリプトを作成"""
    
    comm = ""
    nProcs = couplingDict["nProcs"]
    method = couplingDict["method"]
    if method == "conc":
        lineEnd = " &\n"        #裏で起動
    else:
        lineEnd = "\n"
    for np in range(int(nProcs)):
        procNo = str(np)
        comm += "python3 coupling_FrontISTR/python/"
        comm += getPythonScriptName()
        comm += " 0"                #timeName
        comm += " 0"                #beforeTime
        comm += " " + procNo        #procNo
        comm += " " + nProcs        #全procs数
        comm += " 0"                #timeIndex
        comm += lineEnd
        if np == 0:
            print(comm, flush=True)
    return comm

#
#  getPythonScriptName
def getPythonScriptName():
    """ 解析内容に応じたpythonScript名を取得"""
    OFversion = os.getenv("WM_PROJECT_VERSION")
    contFileDict = pyCoupling.getCouplingScriptDict(OFversion)
    fsi = couplingDict["fsi"]
    cht = couplingDict["cht"]
    chtss = couplingDict["chtss"]
    method = couplingDict["method"]
    key = (fsi, cht, chtss, method)
    _contFile, scriptName = contFileDict[key]
    return scriptName

#
#  getOpenFOAMScript
def getOpenFOAMScript():
    """ OpenFOAM起動スクリプトを作成"""
    nProcs = couplingDict["nProcs"]
    contDict = pyTreeFoam.case().getControlDict()
    solver = contDict["application"]
    if int(nProcs) > 1:
        #parallel設定
        #comm = "mpirun -np " + nProcs + " " + solver + " -parallel 2>&1 | teeCorrect.py solve.log"
        comm = "mpirun -np " + nProcs + " " + solver + " -parallel 2>&1 | tee solve.log"
    else:
        #singleコア
        #comm = solver + " 2>&1 | teeCorrect.py solve.log"
        comm = solver + " 2>&1 | tee solve.log"
    return comm

#-------------------
#  setRestartFile
#-------------------
def setRestartFile():
    """ frontISTRのrestartFileを設定"""
    fsi = couplingDict["fsi"]
    cht = couplingDict["cht"]
    if fsi == "yes":
        #renamePressFile()
        runStep = setRestartFile_fsi()
    if cht == "yes":
        #renameTempFile()
        runStep = setRestartFile_cht()
    return runStep

#
#  setRestartFile_fsi
#---------------------
def setRestartFile_fsi():
    """ fsiのrestartFileを設定する"""
    startTime = couplingDict["startTime"]
    solidCaseDir = couplingDict["solidCaseDir"]
    nStepsOF = couplingDict["nStepsOF"]
    deltaT = couplingDict["deltaT"]
    nProcs = couplingDict["nProcs"]
    #sTime = pyCoupling.time2str(float(startTime))
    sTime = startTime
    name = "FistrModel.restart_" + sTime + "_*"
    pattern = solidCaseDir + "/" + name
    files = glob.glob(pattern)
    if len(files) == 0:
        #restartFileが見つからない場合は、再検索する。
        files = []
        pattern = solidCaseDir + "/" + "FistrModel.restart_*"
        fileNames = glob.glob(pattern)
        for fileName in fileNames:
            words = fileName.split("_")
            if float(words[-2]) == float(sTime):
                files.append(fileName)
        if len(files) == 0:
            print("error: restart file does not exist. could not restart !!!!\n", flush=True)
            return
    #runStepを取得
    restartFile = files[0]
    restFile = ".".join(restartFile.split(".")[:-1])
    runStep = int(restFile.split("_")[-1])
    #restartFileを設定
    baseName = os.path.basename(restartFile)
    restartName = ".".join(baseName.split(".")[:-1])
    for n in range(len(files)):
        #orgName = "/FistrModel.restart_" + sTime + "_" + str(runStep) + "." + str(n)
        orgName = "/" + restartName + "." + str(n)
        newName = "/FistrModel.restart." + str(n)
        orgFile = solidCaseDir + orgName
        newFile = solidCaseDir + newName
        shutil.copy(orgFile, newFile)
    #コピーが完了するまで待つ(シェルが認識できるまで待つ)
    time.sleep(1.0)
    comm = ["#!/bin/bash\n"]
    for n in range(len(files)):
        newName = solidCaseDir + "/FistrModel.restart." + str(n)
        comm += ["FILE=" + newName + "\n" ]
        comm += ["while [ ! -e $FILE ]\n"]
        comm += ["do\n"]
        comm += ["  sleep 0.1\n"]
        comm += ["done\n"]
    exeFile = solidCaseDir + "/run"
    f = open(exeFile, "w"); f.writelines(comm); f.close()
    pyFistr.run(solidCaseDir).command("chmod a+x run")
    pyFistr.run(solidCaseDir).command(exeFile)
    #endStepをcntファイルに書き込み
    endStep = runStep + int(nStepsOF)
    #初期の時間増分を設定
    stepTime = float(deltaT)
    cntFile = solidCaseDir + "/FistrModel_fsi.cnt"
    f = open(cntFile); lines = f.readlines(); f.close()
    i = pyFistr.getNextHeaderName(lines, 0, "!DYNAMIC", "")
    i += 2
    words = pyFistr.deleteSp(lines[i]).split(",")
    words[2] = str(endStep)     #endStepを設定
    words[3] = str(stepTime)    #時間増分を設定
    lines[i] = ", ".join(words) + "\n"
    f = open(cntFile, "w"); f.writelines(lines); f.close()
    return runStep

#
#  setRestartFile_cht
#---------------------
def setRestartFile_cht():
    """ chtのrestartFileを設定する"""
    startTime = couplingDict["startTime"]
    solidCaseDir = couplingDict["solidCaseDir"]
    #nStepsOF = couplingDict["nStepsOF"]
    nSteps = couplingDict["nSteps"]
    deltaT = couplingDict["deltaT"]
    nProcs = couplingDict["nProcs"]
    #sTime = pyCoupling.time2str(float(startTime))
    sTime = startTime
    name = "FistrModel_cht.restart_" + sTime + "_*"
    pattern = solidCaseDir + "/" + name
    files = glob.glob(pattern)
    if len(files) == 0:
        #restartFileが見つからない場合は、再検索する。
        files = []
        pattern = solidCaseDir + "/" + "FistrModel_cht.restart_*"
        fileNames = glob.glob(pattern)
        for fileName in fileNames:
            words = fileName.split("_")
            if float(words[-2]) == float(sTime):
                files.append(fileName)
        if len(files) == 0:
            print("error: restart file does not exist. could not restart !!!!\n", flush=True)
            return
    #runStepを取得
    restartFile = files[0]
    restFile = ".".join(restartFile.split(".")[:-1])
    runStep = int(restFile.split("_")[-1])
    #restartFileを設定
    baseName = os.path.basename(restartFile)
    restartName = ".".join(baseName.split(".")[:-1])
    for n in range(len(files)):
        #orgName = "/FistrModel_cht.restart_" + sTime + "_" + str(runStep) + "." + str(n)
        orgName = "/" + restartName + "." + str(n)
        newName = "/FistrModel_cht.restart." + str(n)
        orgFile = solidCaseDir + orgName
        newFile = solidCaseDir + newName
        shutil.copy(orgFile, newFile)
    #コピーが完了するまで待つ（シェルが認識できるまで待つ）
    time.sleep(1.0)
    comm = ["#!/bin/bash\n"]
    for np in range(int(nProcs)):
        newName = solidCaseDir + "/FistrModel_cht.restart." + str(n)
        comm += ["FILE=" + newName + "\n" ]
        comm += ["while [ ! -e $FILE ]\n"]
        comm += ["do\n"]
        comm += ["  sleep 0.1\n"]
        comm += ["done\n"]
    exeFile = solidCaseDir + "/run"
    f = open(exeFile, "w"); f.writelines(comm); f.close()
    pyFistr.run(solidCaseDir).command("chmod a+x run")
    pyFistr.run(solidCaseDir).command(exeFile)
    #初期の時間増分を設定
    stepTime = float(deltaT)
    #endTimeを設定
    endTime = float(sTime) + (stepTime * int(nSteps))
    #cntFileに書き込み
    cntFile = solidCaseDir + "/FistrModel_cht.cnt"
    f = open(cntFile); lines = f.readlines(); f.close()
    i = pyFistr.getNextHeaderName(lines, 0, "!HEAT", "")
    i += 1    
    words = pyFistr.deleteSp(lines[i]).split(",")
    words[0] = str(stepTime)            #時間増分
    words[1] = str(endTime)             #終了時間
    lines[i] = ", ".join(words) + "\n"
    #restartの設定
    if float(sTime) == 0.0:
        #初回は、restartさせずに起動
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=1")
    else:
        #restartで起動する様に修正
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=-1")
    #cntFile保存
    f = open(cntFile, "w"); f.writelines(lines); f.close()
    return runStep

#
#  getTempLinesInResFile
def getTempLinesInResFile(resFile):
    """ resFileから温度データを取得する"""
    tempLines = []
    f = open(resFile); lines = f.readlines(); f.close()
    flag = 0
    i = 0
    while i < len(lines):
        if flag == 1:
            nodeNo = lines[i].split()[0]
            temp = lines[i+1].split()[0]
            tempLines.append(", ".join([nodeNo, temp]) + "\n")
            i += 1
        elif lines[i][:len("TEMPERATURE")] == "TEMPERATURE":
            flag = 1
        i += 1
    return tempLines

#
#  addTempDataToMeshFile
def addTempDataToMeshFile(tempLines):
    """ mshFileに温度データを追加"""
    solidCaseDir = couplingDict["solidCaseDir"]
    mshFile = solidCaseDir + os.sep + "FistrModel_cht.msh"
    f = open(mshFile); lines = f.readlines(); f.close()
    headerData = pyFistr.getHeaderData(lines)
    for i in range(len(headerData)):
        header = headerData[i][0]
        if header[:len("!INITIAL CONDITION")] == "!INITIAL CONDITION":
            headerData[i][1] = tempLines
            break
    lines = []
    for header in headerData:
        lines += [header[0]]
        lines += header[1]
    f = open(mshFile, "w"); f.writelines(lines); f.close()

#
#  renameInitialDataFile()
#------------------------
def renameInitialDataFile():
    """ restart時の入力ファイルをrenameする"""
    if couplingDict["fsi"] == "yes" or couplingDict["chtss"] == "yes": 
        renamePressFile()
    if couplingDict["cht"] == "yes":
        renameTempFile()

#
#  renamePressFile
def renamePressFile():
    """ pressFileをrenameしてtm付きに変更する"""
    patchName = couplingDict["mappingPatch"]
    startTime = couplingDict["startTime"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    timeIndex = couplingDict["timeIndex"]
    nProcs = couplingDict["nProcs"]
    #sTime = pyCoupling.time2str(float(startTime))
    sTime = startTime
    for np in range(int(nProcs)):
        name = patchName + "_facePress_" + sTime + "_" + str(np)
        pressFile = fluidCaseDir + "/coupling_FrontISTR/data/" + name
        #rename
        if os.path.exists(pressFile) == True:
            newName = pressFile + "tm"
            os.rename(pressFile, newName)
        name = patchName + "_disp_" + timeIndex + "_" + str(np)
        dispFile = fluidCaseDir + "/coupling_FrontISTR/data/" + name
        if os.path.exists(dispFile) == True:
            newName = dispFile + "tm"
            os.rename(dispFile, newName)

#
#  renameTempFile
def renameTempFile():
    """ tempFileをrenameしてtm付きに変更する"""
    patchName = couplingDict["mappingPatch"]
    startTime = couplingDict["startTime"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    timeIndex = couplingDict["timeIndex"]
    nProcs = couplingDict["nProcs"]
    #sTime = pyCoupling.time2str(float(startTime))
    sTime = startTime
    for np in range(int(nProcs)):
        name = patchName + "_faceHeatFlux_" + sTime + "_" + str(np)
        htFile = fluidCaseDir + "/coupling_FrontISTR/data/" + name
        #rename
        if os.path.exists(htFile) == True:
            newName = htFile + "tm"
            os.rename(htFile, newName)
        name = patchName + "_temp_" + timeIndex + "_" + str(np)
        tempFile = fluidCaseDir + "/coupling_FrontISTR/data/" + name
        if os.path.exists(tempFile) == True:
            newName = tempFile + "tm"
            os.rename(tempFile, newName)

#----------------------
#  createRestartScript
#----------------------
def createRestartScript(runStep):
    """ restartする場合のscriptを作成する"""
    comm = "#!/bin/bash\n"
    #FrontISTRの待ち受けを起動
    comm += "python3 coupling_FrontISTR/python/"
    comm += "waitToRunFrontISTR.py &\n"
    #FrontISTRの初回起動
    OFversion = os.getenv("WM_PROJECT_VERSION")
    if OFversion[0] == "v":
        #ESI版は、ここで初回の起動スクリプトを作成する。
        #（Foundation版は、OFが起動スクリプトを出力。）
        comm += get1stRestartScript(runStep)
    #OpenFOAMの起動
    comm += getOpenFOAMScript()
    #fileに保存
    fluidCaseDir = couplingDict["fluidCaseDir"]
    fileName = fluidCaseDir + "/startCoupling"
    f = open(fileName, "w"); f.write(comm); f.close()
    pyFistr.run().command("chmod a+x " + fileName)
    return fileName

#
#  get1stRestartScript
def get1stRestartScript(runStep):
    """ 初回起動のスクリプトを作成"""
    comm = ""
    nProcs = couplingDict["nProcs"]
    startTime = couplingDict["startTime"]
    deltaT = couplingDict["deltaT"]
    method = couplingDict["method"]
    #timeName, beforeTimeを設定
    #sTime = pyCoupling.time2str(float(startTime))
    #bTime = pyCoupling.time2str((float(startTime) - float(deltaT)))
    sTime = getTimeNameFromTimeFolder(startTime)
    bTime = str(float(startTime) - float(deltaT))
    #timeIndexを取得
    idx = str(runStep)
    #スクリプト作成
    if method == "conc":
        lineEnd = " &\n"    #裏で起動
    else:
        lineEnd = "\n"
    for np in range(int(nProcs)):
        procNo = str(np)
        comm += "python3 coupling_FrontISTR/python/"
        comm += getPythonScriptName()
        comm += " " + sTime         #timeName
        comm += " " + bTime         #beforeTime
        comm += " " + procNo        #procNo
        comm += " " + nProcs        #全procs数
        comm += " " + idx           #timeIndex
        #comm += " &\n"
        comm += lineEnd
        if np == 0:
            print(comm, flush=True)
    return comm

#
#  getTimeNameFromTimeFolder
def getTimeNameFromTimeFolder(timeName):
    """ timeNameと同じ値のfolder名を取得する"""
    fluidDir = couplingDict["fluidCaseDir"]
    case = pyTreeFoam.case(fluidDir)
    if int(couplingDict["nProcs"]) > 1:
        region = "processor0"
    else:
        region = "."
    timeFolders = case.getTimeFolders(region)
    getTime = timeName
    for timeFolder in timeFolders:
        if float(timeFolder) == float(timeName):
            getTime = timeFolder
            break
    return getTime


if __name__ == "__main__":
    import gettext
    gettext.install("app") # replace with the appropriate catalog name
    _ = gettext.gettext

    fluidCaseDir = os.getcwd()
    couplingData = pyCoupling.couplingData(fluidCaseDir)
    couplingDict = couplingData.read()
    couplingDict["fluidCaseDir"] = fluidCaseDir
    waitCount = int(couplingDict["maxWaitTime"])
    startTime = couplingDict["startTime"]
    #連成解析開始
    if float(startTime) == 0.0:
        #初期から計算-----------
        #初期化
        initialize()
        #起動スクリプト作成
        runFile = createRunScript()
    else:
        #途中から計算開始-------
        #restartFileの設定
        runStep = setRestartFile()
        couplingDict["timeIndex"] = str(runStep)
        #起動スクリプト作成
        runFile = createRestartScript(runStep)
        #入力fileをrename
        renameInitialDataFile()
    #共有メモリの設定
    mmCommFile = allComm.createCommonFile(couplingDict)
    mmCommPressFile = allComm.createCommTimePressFile(couplingDict)
    mmCommTempFile = allComm.createCommTimeTempFile(couplingDict)
    #スクリプト実行(連成開始)
    pyFistr.run().command(runFile)
