#!/usr/bin/python3
#
#           allCommonConcAltn.py
#
#       同時・交互計算用の共通関数群
#
#   23/11/16    新規作成
#      11/28    mmapのfile作成関数を追加
#

import time
import mmap
import pyCoupling


couplingDict = {}
waitCount = 5000
waitTime = 0.01


#-------------------
#  createCommonArea
#-------------------
#--- fileを定義 ------
def createCommonFile(dataDict):
    """ 共有メモリ用のfileを設定"""
    global couplingDict
    couplingDict = dataDict
    fileName = "common"
    mmCommon = createCommFile(fileName)
    return mmCommon

def createCommTimePressFile(dataDict):
    """ 共有メモリ用のfileを作成
    couplingTime保存用"""
    global couplingDict
    couplingDict = dataDict
    fileName = "commTimePress"
    mmCommTime = createCommFile(fileName)
    return mmCommTime

def createCommTimeTempFile(dataDict):
    """ 共有メモリ用のfileを作成
    couplingTime保存用"""
    global couplingDict
    couplingDict = dataDict
    fileName = "commTimeTemp"
    mmCommTime = createCommFile(fileName)
    return mmCommTime

def createCommFile(fileName):
    """ 共有メモリ用のfileを作成する"""
    length = 200
    cont = "\00" * length
    fluidCaseDir = couplingDict["fluidCaseDir"]
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    commonFile = dataDir + "/" + fileName
    #file作成
    f = open(commonFile, "w"); f.write(cont); f.close()
    #共有file作成
    f = open(commonFile, "r+b"); mmCommon = mmap.mmap(f.fileno(), 0); f.close()
    return mmCommon

#--- areaを定義 ------
def createCommonArea(dataDict):
    """ common fileを共有メモリとして設定する。
    200文字で設定。"""
    global couplingDict
    couplingDict = dataDict
    fileName = "common"
    mmCommTime = createCommArea(fileName)
    return mmCommTime

def createCommTimePressArea(dataDict):
    """ 共有メモリ作成
    couplingTime, stepのセットを最新から3組保存する領域"""
    global couplingDict
    couplingDict = dataDict
    fileName = "commTimePress"
    mmCommTime = createCommArea(fileName)
    return mmCommTime

def createCommTimeTempArea(dataDict):
    """ 共有メモリ作成
    couplingTime, stepのセットを最新から3組保存する領域"""
    global couplingDict
    couplingDict = dataDict
    fileName = "commTimeTemp"
    mmCommon = createCommArea(fileName)
    return mmCommon

def createCommArea(fileName):
    """ 共有メモリ用のareaを定義する"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    commonFile = dataDir + "/" + fileName
    f = open(commonFile, "r+b"); mmCommon = mmap.mmap(f.fileno(), 0); f.close()
    return mmCommon


#--------------------
#  mmCommonArea class
#--------------------
class mmCommonArea:
    """ 共有メモリへの操作"""

    def __init__(self, couplingDict, mmComm):
        self.couplingDict = couplingDict
        self.waitCount = self.couplingDict["maxWaitTime"]
        self.waitTime = waitTime
        self.mmCommon = mmComm

    #
    #  writeCommonArea
    def writeCommonArea(self, cont):
        """ 共有メモリ（200文字）への書き込み。"""
        length = 200
        space = "\00" * (length - len(cont))
        cont += space
        self.mmCommon.seek(0)
        self.mmCommon.write(cont.encode())
        self.mmCommon.flush()

    #
    #  readCommonArea
    def readCommonArea(self):
        """ 共有メモリの内容を読み込む"""
        self.mmCommon.seek(0)
        cont = self.mmCommon.read().decode()
        n = cont.find("\00")
        cont = cont[:n]
        return cont

    #
    #  waitUntilFreeCommon
    def waitUntilFreeCommon(self):
        """ busy状態かチェック。
        busy（前回起動されたmainが起動中）なら、終了するまで待つ.
        起動中の場合は、contに文字が入っている。"""
        count = self.waitCount
        sleepTime = self.waitTime
        while True:
            cont = self.readCommonArea()
            if len(cont) == 0:          #bussy？
                break
            count -= 1
            if count < 0:
                break
            time.sleep(sleepTime)
        return

    #
    #  getBeforeCouplingTime
    def getBeforeCouplingTime(self):
        """ 直前のcouplingTime、stepを共有メモリから取得"""
        #cont = readCommTimePressArea()
        cont = self.readCommonArea()
        lines = cont.split("\n")[:-1]
        if len(lines) <= 1:
            cpTime = pyCoupling.time2str(-1)
            cpStep = "-1"
            return cpStep, cpTime
        line = lines[-2]
        cpTime, cpStep = line.split()
        return cpStep, cpTime

    #
    #  saveCouplingTime
    def saveCouplingTime(self):
        """ couplintTime, stepのペアを保存"""
        timeName = couplingDict["timeName"]
        timeIndex = couplingDict["timeIndex"]
        procNo = couplingDict["procNo"]
        if int(procNo) == 0:
            addLine = timeName + " " + timeIndex
            cont = self.readCommonArea()
            cont += addLine + "\n"
            lines = cont.split("\n")[:-1]
            lines = lines[-3:]
            cont = "\n".join(lines) + "\n"
            #writeCommTimePressArea(cont)
            self.writeCommonArea(cont)

    #
    #  waitUntilFreeCommonArea
    def waitUntilFreeCommonArea(self):
        """ busy状態かチェック。
        busy（前回起動されたmainが起動中）なら、終了するまで待つ.
        起動中の場合は、contに文字が入っている。"""
        count = waitCount
        sleepTime = waitTime
        while True:
            cont = self.readCommonArea()
            if len(cont) == 0:          #bussy？
                break
            count -= 1
            if count < 0:
                break
            time.sleep(sleepTime)
        return


#------------
#  runFistr
#------------
def runFistr(dataDict, mmCommArea, sect, endStep, endTime):
    """ FrontISTRを起動"""
    global couplingDict
    couplingDict = dataDict

    timeName = couplingDict["timeName"]
    title = "title runFistr\n"
    time = "time " + timeName + "\n"
    #fsi？
    if sect == "fsi":
        nSteps = couplingDict["nStepsOF"]
        startStep = endStep - int(nSteps)
        print("(py) --- FrontISTR is running for structure from step "
           + str(startStep) + " to " + str(endStep) + " time "
           + str(endTime) + " ...", flush=True)
    #cht？
    elif sect == "cht":
        nSteps = couplingDict["nSteps"]
        startStep = endStep - int(nSteps)
        print("(py) --- FrontISTR is running for heat transfer from step "
            + str(startStep) + " to " + str(endStep) + " time "
            + str(endTime) + " ...", flush=True)
    #chtss？
    elif sect == "chtss":
        print("(py) --- FrontISTR is running for heat stress...", flush=True)
        #起動設定
    stepCont = sect + " -> " + str(endStep) + " steps"
    footer = "footer " + "FrontISTR totla time (" + stepCont + ") ----> \n"
    cont = title + time + footer + "#end\n"
    #Fistrがfreeまで待つ
    waitUntilFreeFistr(mmCommArea)
    #計算開始
    mmCommArea.writeCommonArea(cont)
    #Fistr終了まで待つ
    waitUntilFreeFistr(mmCommArea)

#
#  waitUntilFreeFistr
def waitUntilFreeFistr(mmCommArea):
    """ Fistrの終了を待つ。
    共有メモリの内容から確認する"""
    count = waitCount
    sleepTime = waitTime
    while True:
        cont = mmCommArea.readCommonArea()
        if len(cont) > 0:
            lines = cont.split("\n")
            if len(lines) > 0:
                words = lines[0].split()
                if len(words) >= 2:
                    if (words[1] == "FistrFinished" or
                        words[1] == "python"    #同時計算用で追加
                        ):
                        break
        else:   #交互計算用で追加
            break
        count -= 1
        if count < 0:
            break
        time.sleep(sleepTime)
    return

