#!/usr/bin/python3
#  coding: utf-8
#
#   runParallelSetPreserveDialog.py
#
#       decomposeParDict内のpreserveの設定
#
#   15/07/07    新規作成
#      07/12    writeDecomposeParDict:修正（insertした時、一行追加されるのを修正）
#   16/12/31    iniClass:OF-ext用に修正
#   20/01/07    GTK+3, python3用に大幅修正
#      03/02    buttonの位置修正。バグ取り
#      04/22    多言語化対応
#   21/06/05    writeDecomposeParDict:preserveの書式をOFversionによって修正。
#   22/07/19    showFaceZonePatchBaffle:OFversionのcheck方法修正（OF-10対応）
#               getPreserveCont,getPreserveBaffles:  ↑
#   24/04/15    initialize:region無いcaseの場合、エラー発生する為、修正。
#      08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import os, sys
import glob
import shutil

import pyTreeFoam
import GtkParts
import universalDialogs as unvDlg

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()

#------------------
#  windowApp class
#------------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"runParallelSetPreserveDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を取得
        self.setGtkObject()
        #属性の設定
        self.maskEvent = True
        self.caseDir = caseDir
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        self.allRegions = []
        self.preserveRegions = []
        self.keywords = ("preserveFaceZones", "preservePatches", "preserveBaffles", "constraints")
        #treeListの初期化
        self.iniTreeList()
        #title
        self.mainWindow.set_title(_("preserveの設定"))

    #
    #  main
    #-------
    def main(self):
        """ GUIの表示"""
        self.initialize()
        self.mainWindow.show()
        self.maskEvent = False
        Gtk.main()

    #
    #  close
    #--------
    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    #------- event handler -----------
    #設定>>faceZone
    def onSetFaceZones(self, event):
        self.setPreserveFaceZones()
    def onReturnFaceZones(self, event):
        self.returnPreserveFaceZone()
    def onSetPatches(self, event):
        self.setPreservePatches()
    def onReturnPatches(self, event):
        self.returnPreservePatches()
    def onOpenDict(self, event):
        self.openDict()
    def onSetPreserve(self, event):
        self.setPreserveToDict()
    #閉じる
    def onClose(self, event):
        self.close()
    def onChangeRegion(self, event):
        if self.maskEvent == False:
            self.changeRegion()
    #------------------------------------

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        self.label_preserve = self.builder.get_object("label_preserve")
        self.label_baffle = self.builder.get_object("label_baffle")
        self.tree_region = self.builder.get_object("tree_region")
        self.tree_faceZone = self.builder.get_object("tree_faceZone")
        self.tree_setFaceZone = self.builder.get_object("tree_setFaceZone")
        self.tree_patch = self.builder.get_object("tree_patch")
        self.tree_setPatch = self.builder.get_object("tree_setPatch")
        self.check_baffle = self.builder.get_object("check_baffle")

    #
    #  iniTreeList
    #----------------
    def iniTreeList(self):
        GtkParts.treeList(self.tree_region).create(multi=False, header=False)
        GtkParts.treeList(self.tree_faceZone).create(multi=True, header=False)
        GtkParts.treeList(self.tree_setFaceZone).create(multi=True, header=False)
        GtkParts.treeList(self.tree_patch).create(multi=True, header=False)
        GtkParts.treeList(self.tree_setPatch).create(multi=True, header=False)

    #
    #  setPreserveFaceZones
    #----------------------
    def setPreserveFaceZones(self):
        """ faceZoneを設定側に移動"""
        self.moveItemsInTreeList(self.tree_faceZone, self.tree_setFaceZone)

    #
    #  returnPreserveFaceZone
    #-------------------------
    def returnPreserveFaceZone(self):
        """ faceZoneを非設定側に戻す"""
        self.moveItemsInTreeList(self.tree_setFaceZone, self.tree_faceZone)

    #
    #  setPreservePatches
    #---------------------
    def setPreservePatches(self):
        """ patchを設定側に移動する"""
        self.moveItemsInTreeList(self.tree_patch, self.tree_setPatch)

    #
    #  returnPreservePatches
    #------------------------
    def returnPreservePatches(self):
        """ patchを非設定側に戻す"""
        self.moveItemsInTreeList(self.tree_setPatch, self.tree_patch)

    #
    #  moveItemsInTreeList
    def moveItemsInTreeList(self, fromList, toList):
        """ fromListの選択項目をtoList側へ移動する。"""
        fromTreeList = GtkParts.treeList(fromList)
        selNames = fromTreeList.getSelectedItems()
        if len(selNames) == 0:
            title = _(u"エラー")
            mess = _(u"項目が選択されていません\n")
            mess += _(u"項目を選択後。クリックしてください")
            self.errDialog(title, mess)
            return
        #toList側の設定
        toTreeList = GtkParts.treeList(toList)
        allItems = toTreeList.getAllItems()
        allItems += selNames
        allItems.sort()
        toTreeList.setData(allItems)
        toTreeList.selectItems(selNames)
        #fromList側の設定
        allItems = fromTreeList.getAllItems()
        allItems = list(set(allItems).difference(selNames))
        fromTreeList.setData(allItems)

    #
    #  openDict
    #-----------
    def openDict(self):
        """ decomposeParDictを開く"""
        regions = GtkParts.treeList(self.tree_region).getSelectedItems()
        region = regions[0]
        if region == _("(現在のfolder)"):
            region = "."
        fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
        pyTreeFoam.run(self.caseDir).editor([fileName])

    #
    #  setPreserveToDict
    #--------------------
    def setPreserveToDict(self):
        """ 設定内容をdecomposeParDictに反映する。"""
        #regionを取得
        regions = GtkParts.treeList(self.tree_region).getSelectedItems()
        region = regions[0]
        if region == _("(現在のfolder)"):
            region = "."
        #設定内容を取得
        faceZones = GtkParts.treeList(self.tree_setFaceZone).getAllItems()
        patches = GtkParts.treeList(self.tree_setPatch).getAllItems()
        bafflesBool = self.check_baffle.get_active()
        clearNames = self.writeDecomposeParDict(region, faceZones, patches, bafflesBool)
        #終了メセージ
        title = _(u"preserveの設定")
        mess = _(u"decomposeParDici内で、\n")
        preserve = []
        if len(faceZones) != 0:
            preserve.append("preserveFaceZones")
        if len(patches) != 0:
            preserve.append("preservePatches")
        if bafflesBool == True:
            preserve.append("preserveBaffles")
        if len(preserve) != 0:
            mess += "  " + ", ".join(preserve) + _(u"\nを設定しました\n")
        else:
            mess += _("preserveの設定をクリアしました")
        print(mess)
        self.okDialog(title, mess)
        #preserveの設定を取得し、表示
        self.preserveRegions = []
        for region in self.allRegions:
            fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
            f = open(fileName); cont = f.read(); f.close()
            for key in self.keywords:
                (lineCont, _p, _kind) = self.getKeywordLine(key, cont, 0)
                if lineCont != "":
                    self.preserveRegions.append(region)
                    break
        regions = []
        cont = _("無し")
        for region in self.preserveRegions:
            if region == ".":
                regions.append(_("(現在のfolder)"))
            else:
                regions.append(region)
            if len(regions) != 0:
                cont = "\n".join(regions)
        self.label_preserve.set_text(cont)

    #
    #  writeDecomposeParDict
    def writeDecomposeParDict(self, region, faceZones, patches, bafflesBool):
        """ OFversionに応じた書式でdecomposeParDictを書き換える"""
        OFversion = self.configDict["OFversion"]
        ver = pyTreeFoam.getNumVersion(OFversion)
        if OFversion >= "v1906":
            #OF-v1906以上
            self.writeDecomposeParDict_v1906(region, faceZones, patches, bafflesBool)
        elif OFversion[0] == "v":
            #OF-v1906未満
            self.writeDecomposeParDict_4(region, faceZones, patches, bafflesBool)
        elif ver >= 4.0:
            #OF-4.0以上
            self.writeDecomposeParDict_4(region, faceZones, patches, bafflesBool)
        elif (0.0 < ver and ver < 4.0):
            #OF-4未満
            self.writeDecomposeParDict_0(region, faceZones, patches, bafflesBool)

    #
    #  writeDecomposeParDict_0
    def writeDecomposeParDict_0(self, region, faceZones, patches, bafflesBool):
        """ OF-4.0未満、decomposeParDictを書き換える"""
        clearNames = []
        fileName = caseDir + "/system/" + region + "/decomposeParDict"
        f=open(fileName); cont=f.read(); f.close()
        #faceZoneの設定
        if len(faceZones) == 0:
            [cont, act] = self.deleteKeywordLine("preserveFaceZones", cont)
            if act == "del":
                clearNames.append("faceZones")
        else:
            [cont, act] = self.replaceKeywordLine("preserveFaceZones", faceZones, cont)
        #patchesの設定
        if len(patches) == 0:
            [cont, act] = self.deleteKeywordLine("preservePatches", cont)
            if act == "del":
                clearNames.append("patches")
        else:
            [cont, act] = self.replaceKeywordLine("preservePatches", patches, cont)
        #bafflesの設定
        if bafflesBool == False:
            [cont, act] = self.deleteKeywordLine("preserveBaffles", cont)
            if act == "del":
                clearNames.append("baffles")
        else:
            [cont, act] = self.replaceBafflesLine(cont)
        #書き込み
        f=open(fileName, "w"); f.write(cont); f.close()
        return []

    #
    #  writeDecomposeParDict_v1906
    def writeDecomposeParDict_v1906(self, region, faceZones, patches, bafflesBool):
        """ OF-4.0以上、decomposeParDictを書き換える"""
        fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
        f = open(fileName); cont = f.read(); f.close()
        #全設定を削除する
        cont = self.deleteAllPreserve(cont)
        if len(faceZones) > 0 or len(patches) > 0 or bafflesBool == True:
            addCont = ""
            #faceZonesの設定を作成
            if len(faceZones) > 0:
                newCont  = "    faces\n"
                newCont += "    {\n"
                newCont += "        type    preserveFaceZones;\n"
                newCont += "        zones   ( " + " ".join(faceZones) + " );\n"
                newCont += "    }\n"
                addCont += newCont 
            #patchesの設定を作成
            if len(patches) > 0:
                newCont  = "    patches\n"
                newCont += "    {\n"
                newCont += "        type    preservePatches;\n"
                newCont += "        patches ( " + " ".join(patches) + " );\n"
                newCont += "    }\n"
                addCont += newCont 
            #bafflesの設定を作成
            if bafflesBool == True:
                newCont  = "    baffles\n"
                newCont += "    {\n"
                newCont += "        type    preserveBaffles;\n"
                newCont += "    }\n"
                addCont += newCont
            #constraintsを作成
            if addCont != "":
                newCont  = "constraints\n"
                newCont += "{\n"
                newCont += addCont
                newCont += "}\n"
            #newContを追加する
            cont = self.addNewCont(cont, newCont)

        #内容を書き込み
        f = open(fileName, "w"); f.write(cont); f.close()
        return []

    #
    #  writeDecomposeParDict_4
    def writeDecomposeParDict_4(self, region, faceZones, patches, bafflesBool):
        """ OF-4.0以上、decomposeParDictを書き換える"""
        fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
        f = open(fileName); cont = f.read(); f.close()
        #全設定を削除する
        cont = self.deleteAllPreserve(cont)
        if len(faceZones) > 0 or len(patches) > 0 or bafflesBool == True:
            addCont = ""
            #faceZonesの設定を作成
            if len(faceZones) > 0:
                newCont  = "    preserveFaceZones\n"
                newCont += "    {\n"
                newCont += "        type    preserveFaceZones;\n"
                newCont += "        zones   ( " + " ".join(faceZones) + " );\n"
                newCont += "    }\n"
                addCont += newCont 
            #patchesの設定を作成
            if len(patches) > 0:
                newCont  = "    preservePatches\n"
                newCont += "    {\n"
                newCont += "        type    preservePatches;\n"
                newCont += "        patches ( " + " ".join(patches) + " );\n"
                newCont += "    }\n"
                addCont += newCont 
            #bafflesの設定を作成
            if bafflesBool == True:
                newCont  = "    preserveBaffles\n"
                newCont += "    {\n"
                newCont += "        type    preserveBaffles;\n"
                newCont += "    }\n"
                addCont += newCont
            #constraintsを作成
            if addCont != "":
                newCont  = "constraints\n"
                newCont += "{\n"
                newCont += addCont
                newCont += "}\n"
            #newContを追加する
            cont = self.addNewCont(cont, newCont)

        #内容を書き込み
        f = open(fileName, "w"); f.write(cont); f.close()
        return []

    def deleteAllPreserve(self, cont):
        """ 全preserveの設定をクリアする"""
        (cont, act) = self.deleteKeywordLine("preserveFaceZones", cont)
        (cont, act) = self.deleteKeywordLine("preservePatches", cont)
        (cont, act) = self.deleteKeywordLine("preserveBaffles", cont)
        (cont, act) = self.deleteKeywordLine("constraints", cont)
        return cont

    def addNewCont(self, cont, addCont):
        """ subDomainの次の行にaddContを追加する"""
        ps = 0
        contOp = pyTreeFoam.strOp(cont)
        lineCont, p, kind = contOp.get1line(0)
        while lineCont != "":
            keyword, _pp = pyTreeFoam.strOp(lineCont).getKeyword(0)
            if keyword == "numberOfSubdomains":
                ps = p
                break
            lineCont, p, kind = contOp.get1line(p)
        
        if ps > 0:
            ps = contOp.skipUntilCR(ps+1)
            cont = cont[:ps] + addCont + cont[ps:]
        return cont

    def deleteKeywordLine(self, key, cont):
        """ keywordLineを削除"""
        (lineCont, p, _kind) = self.getKeywordLine(key, cont, 0)
        if lineCont != "":
            ps = p - len(lineCont)
            newCont = cont[:ps-1] + cont[p:]
            return (newCont, "del")
        else:
            return (cont, "")

    #
    #  replaceKeywordLine
    def replaceKeywordLine(self, key, faceZones, cont):
        """ keywordラインを置き換え"""
        vals = "    " + " ".join(faceZones) + "\n"
        newLine =  key + "\n"
        newLine += "(\n"
        newLine += vals
        newLine += ");"
        contOp = pyTreeFoam.strOp(cont)
        newCont, p, keyword = contOp.replace1lineKeyword(0, newLine)
        if keyword == "":
            lineCont, p, kind = contOp.get1line(0)
            ps = 0
            while lineCont != "":
                keyword, pp = pyTreeFoam.strOp(lineCont).getKeyword(0)
                if keyword == "numberOfSubdomains":
                    ps = p
                    break
                lineCont, p, kind = contOp.get1line(p)
            newCont = cont[:ps] + "\n" + newLine + cont[ps:]
            retVal = [newCont, "add"]
        else:
            retVal = [newCont, "rep"]
        return retVal

    #
    #  replaceBafflesLine
    def replaceBafflesLine(self, cont):
        """ bafflesの行を置き換える"""
        newLine = "preserveBaffles  true;"
        contOp = pyTreeFoam.strOp(cont)
        newCont, p, keyword = contOp.replace1lineKeyword(0, newLine)
        if keyword == "":
            lineCont, p, kind = contOp.get1line(0)
            ps = 0
            while lineCont != "":
                keyword, pp = pyTreeFoam.strOp(lineCont).getKeyword(0)
                if keyword == "numberOfSubdomains":
                    ps = p
                    break
                lineCont, p, kind = contOp.get1line(p)
            newCont = cont[:ps] + "\n" + newLine + cont[ps:]
            return [newCont, "add"]
        else:
            return [newCont, "rep"]

    #
    #  changeRegion
    #---------------
    def changeRegion(self):
        """ regionを変更した時の処理"""
        regions = GtkParts.treeList(self.tree_region).getSelectedItems()
        region = regions[0]
        if region == _("(現在のfolder)"):
            region = "."
        self.showFaceZonePatchBaffle(region)        

    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        OFversion = self.configDict["OFversion"]
        (fluids, solids) = pyTreeFoam.case(self.caseDir).getZonesInRegionProperties(OFversion)
        if fluids == "":
            fluids = []
        if solids == "":
            solids = []
        self.allRegions = ["."] + fluids + solids
        self.allRegions.sort()
        self.preserveRegions = []
        for region in self.allRegions:
            fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
            f = open(fileName); cont = f.read(); f.close()
            for key in self.keywords:
                (lineCont, _p, _kind) = self.getKeywordLine(key, cont, 0)
                if lineCont != "":
                    self.preserveRegions.append(region)
                    break
        #preserveの設定を取得し、表示
        regions = []
        cont = _("無し")
        for region in self.preserveRegions:
            if region == ".":
                regions.append(_("(現在のfolder)"))
            else:
                regions.append(region)
            if len(regions) != 0:
                cont = "\n".join(regions)
        self.label_preserve.set_text(cont)
        #regionの設定
        regions = [_("(現在のfolder)")] + self.allRegions[1:]
        treeList = GtkParts.treeList(self.tree_region)
        treeList.setData(regions)
        selItems = [_("(現在のfolder)")]
        treeList.selectItems(selItems)
        #TreeListを表示
        region = "."
        self.showFaceZonePatchBaffle(region)

    #
    #  showFaceZonePatchBaffle
    def showFaceZonePatchBaffle(self, region):
        """ region内のpreserveの設定を表示する。TreeListに表示する"""
        #faceZone, patchの設定
        self.setPreserveToListBox(region)
        #bafflesの設定
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        #if OFversion < "2.3" or OFversion[:3] == "ext":
        if (0.0 < numVer and numVer < 2.3) or OFversion[:3] == "ext":
            self.label_baffle.set_sensitive(False)
            self.check_baffle.set_sensitive(False)

    #
    #  setPreserveToListBox
    def setPreserveToListBox(self, region):
        #decomposeParDict読み込み
        cont = self.readDecomposeParDict(region)
        #設定を取得
        setFaceZones = self.getPreserveCont("preserveFaceZones", cont)
        setPatches = self.getPreserveCont("preservePatches", cont)
        bafflesBool = self.getPreserveBaffles(cont)
        #faceZonesの設定
        faceZones = self.getFaceZones(region)
        noFaceZones = list(set(faceZones).difference(setFaceZones))
        noFaceZones.sort()
        GtkParts.treeList(self.tree_faceZone).setData(noFaceZones)
        GtkParts.treeList(self.tree_setFaceZone).setData(setFaceZones)
        #patchesの設定
        patches = self.getPatches(region)
        noPatches = list(set(patches).difference(setPatches))
        noPatches.sort()
        GtkParts.treeList(self.tree_patch).setData(noPatches)
        GtkParts.treeList(self.tree_setPatch).setData(setPatches)
        #baffleの設定
        self.check_baffle.set_active(bafflesBool)

    #
    #  getPatches
    def getPatches(self, region):
        polyMeshDir = self.caseDir + "/constant/" + region + "/polyMesh"
        patchConts = pyTreeFoam.case(self.caseDir).getPatchNameFromBoundary(polyMeshDir)
        patches = list(map(lambda x: x[0], patchConts))
        patches.sort()
        return patches

    #
    #  getFaceZones
    def getFaceZones(self, region):
        polyMeshDir = self.caseDir + "/constant/" + region + "/polyMesh"
        faceZones = pyTreeFoam.getFoamContents(self.caseDir).faceZoneNames(polyMeshDir)
        faceZones.sort()
        return faceZones

    #
    #  getPreserveCont
    def getPreserveCont(self, keyword, cont):

        def getPreserveContents(keyword, cont):
            vals = []
            (lineCont, p, _kind) = self.getKeywordLine(keyword, cont, 0)
            while lineCont != "":
                line, _pp = pyTreeFoam.strOp(lineCont).getSmallPair(0)
                vals += line.split()
                (lineCont, p, _kind) = self.getKeywordLine(keyword, cont, p)
            return vals

        OFversion = self.configDict["OFversion"]
        (lineCont, p, _kind) = self.getKeywordLine("constraints", cont, 0)
        if lineCont != "":
            if OFversion >= "v1906":
                #OF-v1906以上
                faceNames = "faces"
                patchNames = "patches"
            else:
                #OF-4以上、v1906未満
                faceNames = "preserveFaceZones"
                patchNames = "preservePatches"
            lineCont, pp = pyTreeFoam.strOp(lineCont).getMiddlePair(0)
            if keyword == "preserveFaceZones":
                #(line, p, _kind) = self.getKeywordLine("faces", lineCont, 0)
                (line, p, _kind) = self.getKeywordLine(faceNames, lineCont, 0)
            elif keyword == "preservePatches":
                #(line, p, _kind) = self.getKeywordLine("patches", lineCont, 0)
                (line, p, _kind) = self.getKeywordLine(patchNames, lineCont, 0)
            line, pp = pyTreeFoam.strOp(line).getMiddlePair(0)
            line, pp = pyTreeFoam.strOp(line).getSmallPair(0)
            vals = line.split()
        else:
            #OF-4未満、OF-ext
            vals = getPreserveContents(keyword, cont)
        vals.sort()
        return vals

    #
    #  getPreserveBaffles
    def getPreserveBaffles(self, cont):

        def getPreserveBafflsContents(cont):
            bafflesBool = False
            (lineCont, _p, _kind) = self.getKeywordLine("preserveBaffles", cont, 0)
            if lineCont != "":
                word = lineCont.split(";")[0].split()[1]
                if word == "true":
                    bafflesBool = True
            return bafflesBool

        OFversion = self.configDict["OFversion"]
        (lineCont, p, _kind) = self.getKeywordLine("constraints", cont, 0)
        if lineCont != "":
            if OFversion >= "v1906":
                #OF-v1906以上
                names = "baffles"
            else:
                #OF-4以上、OF-v1906未満
                names = "preserveBaffles"
            bafflesBool = False
            lineCont, pp = pyTreeFoam.strOp(lineCont).getMiddlePair(0)
            #line, pp, kind = self.getKeywordLine("baffles", lineCont, 0)
            line, pp, kind = self.getKeywordLine(names, lineCont, 0)
            line, pp = pyTreeFoam.strOp(line).getMiddlePair(0)
            if line != "":
                if line.split()[1][:len("preserveBaffles")] == "preserveBaffles":
                    bafflesBool = True
        else:
            #OF-4未満、OF-ext
            bafflesBool = getPreserveBafflsContents(cont)
        return bafflesBool

    #
    #  readDecomposeParDict
    def readDecomposeParDict(self, region):
        fileName = self.caseDir + "/system/" + region + "/decomposeParDict"
        f = open(fileName); cont = f.read(); f.close()
        return cont

    #
    #  getKeywordLine
    def getKeywordLine(self, key, cont, p):
        """ keywordを検索し、該当する行を戻す。"""
        contOp = pyTreeFoam.strOp(cont)
        lineCont = " "
        #p = 0
        while lineCont != "":
            (lineCont, p, kind) = contOp.get1line(p)
            (keyword, _pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
            if keyword == key:
                break
        return (lineCont, p, kind)

    #---- universalDialogs ----------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()
