#!/usr/bin/python3
# coding: utf-8
#
#   pasteMeshDialog.py
#
#       copy元のmeshを貼り付ける
#
#   22/01/28    新規作成
#      06/09    onStartPaste:paste後、close()を追加
#   24/08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk

import sys, os
import glob
import shutil

import GtkParts
import pyTreeFoam
import progressBar
import universalDialogs as unvDlg

import logFileCreater
logFileCreater.log()


#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, copyDir, pasteDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"pasteMeshDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を定義
        self.setGtkObject()
        #変数の定義
        self.copyDir = copyDir
        self.pasteDir = pasteDir

    #
    #  main
    #-------
    def main(self):
        """ GUIの表示"""
        self.initialize()
        self.mainWindow.show()
        Gtk.main()

    #
    #  close
    #--------
    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ glade内のobject名を定義"""
        self.label_copyDir = self.builder.get_object("label_copyDir")
        self.label_pasteDir = self.builder.get_object("label_pasteDir")
        self.label_copyCase = self.builder.get_object("label_copyCase")
        self.label_pasteCase = self.builder.get_object("label_pasteCase")
        tree_copy = self.builder.get_object("tree_copy")
        self.listBoxCopy = GtkParts.treeList(tree_copy)
        self.listBoxCopy.create()
        tree_paste = self.builder.get_object("tree_paste")
        self.listBoxPaste = GtkParts.treeList(tree_paste)
        self.listBoxPaste.create(multi=False)


    #---------- event handler -----------
    #貼り付け開始
    def onStartPaste(self, event):
        self.startPaste()
        #self.close()
    #キャンセル
    def onCancel(self, event):
        self.close()


    #
    #  startPaste
    #-------------
    def startPaste(self):
        """ 指定されたmeshを貼り付ける"""
        #選択しているmeshを取得
        selCopyMeshes = self.listBoxCopy.getSelectedItems()
        selPasteMeshes = self.listBoxPaste.getSelectedItems()
        if len(selPasteMeshes) != 1:
            title = _("エラー")
            mess = _("paste先は、複数選択できません。")
            self.errDialog(title, mess)
            return

        #copy元の複数のpolyMeshを選択している場合
        if len(selCopyMeshes) > 1:
            pasteName = selPasteMeshes[0]
            if len(pasteName.split("/")) != 1:
                mess = _("copy元を複数選択する場合は、paste先を「constant」または「timefilder」に設定してください。")
                self.errDialog(_("エラー"), mess)
                return
            #errorチェック
            errFlag = 0
            for selCopyMesh in selCopyMeshes:
                meshLoc = selCopyMesh.split("/")[-2]
                if meshLoc == "constant":
                    errFlag = 1
                else:
                    try:
                        float(meshLoc)
                        errFlag = 1
                    except:
                        pass
                if errFlag == 1:
                    break
            if errFlag == 1:
                mess = _("copy元を複数選択する場合は、regionのメッシュを選択してください。")
                self.errDialog(_("エラー"), mess)
                return
            #copyMesh, pasteMeshのfullPathを取得
            copyMeshDirs = []
            pasteMeshDirs = []
            pasteDir = self.pasteDir + "/" + selPasteMeshes[0]
            for selCopyMesh in selCopyMeshes:
                meshDir = self.copyDir + "/" + selCopyMesh
                pasteMeshDir = pasteDir + "/" + selCopyMesh.split("/")[1]
                if meshDir != pasteMeshDir+"/polyMesh":
                    copyMeshDirs.append(meshDir)
                    pasteMeshDirs.append(pasteMeshDir)
            #paste先のmeshを削除する
            for pasteMeshDir in pasteMeshDir:
                if len(glob.glob(pasteMeshDir+"/polyMesh")) != 0:
                    shutil.rmtree(pasteMeshDir + "/polyMesh")
            #paste開始
            stat = progressBar.pasteLocalToLocal(copyMeshDirs, pasteMeshDirs).paste()
            if stat != "OK":
                return
            #regionPropertiesのコピー
            error = copyRegionProperties(self.copyDir, self.pasteDir)
            if error != "":
                mess = _("regionPropertiesがコピーできませんでした。\n新たにregionPopertiesを作成してください。")
                self.errDialog(_("エラー"), mess)
                return
            clearInternalBoundaryFieldMultiPolyMesh(self.pasteDir, pasteDir,
                    copyMeshDirs)
            mess = _("regionのメッシュをコピーしました。")
            self.okDialog(_("meshの入れ替え"), mess)
            return
        else:
            #copyDir側が1個のpolyMeshの場合
            #fullPathを取得
            copyMeshDir = self.copyDir + os.sep + selCopyMeshes[0]
            pasteMeshDir = self.pasteDir + os.sep + selPasteMeshes[0]
            if copyMeshDir == pasteMeshDir + "/polyMesh":
                mess = _("copy元とpaste先が同じ設定になっています。")
                self.errDialog(_("エラー"), mess)
                return
            #cell数を取得
            print("getting number of cells in source mesh...")
            nCellCopy = self.getNCellPolyMesh(self.copyDir, copyMeshDir)
            print("getting number of cells in paste mesh...")
            nCellPaste = self.getNCellPolyMesh(self.pasteDir, pasteMeshDir)
            #paste先のpolyMeshを削除する。
            if len(glob.glob(pasteMeshDir+"/polyMesh")) != 0:
                shutil.rmtree(pasteMeshDir+"/polyMesh")
            #コピー開始
            stat = progressBar.pasteLocalToLocal([copyMeshDir], [pasteMeshDir]).paste()
            if stat != "OK":
                return
            #boundaryFieldとinternalFieldを整形
            if nCellCopy != nCellPaste and nCellCopy != 0:
                #cell数が違う場合（異なるmeshの場合）
                title = _(u"メッシュの入れ替え")
                mess = _(u"copy元とpaste先のcell数が異なっています。\n")
                mess += _(u"paste後のメッシュに合わせる為、")
                mess += _(u"internalFieldをクリアしますか？\n")
                mess += _(u"（nonuniform形式をuniform形式にクリアする。）")
                yesArgs = [self.startPaste_clearBoundary1M, pasteMeshDir]
                noArgs  = [self.startPaste_nullBoundary1M, pasteMeshDir]
                stat = self.yesNoDialog(title, mess, funcYes=yesArgs, funcNo=noArgs)
                return
            else:
                #cell数が同じ場合（同じメッシュの場合）
                clearOnlyNullBoundaryFields(self.pasteDir, pasteMeshDir)
            #終了messageを出力
            self.startPaste_ending1M()
            return
    
    #  startPaste_clearBoundary1M
    def startPaste_clearBoundary1M(self, pasteMeshDir):
        """ boundaryField, internalFieldをクリアする。"""
        clearNullBoundaryFieldsInternalFields(self.pasteDir, pasteMeshDir)
        self.startPaste_ending1M()

    #  startPaste_nullBoundary1M
    def startPaste_nullBoundary1M(self, pasteMeshDir):
        """ 未定義のboundaryFieldのみクリアする"""
        clearOnlyNullBoundaryFields(self.pasteDir, pasteMeshDir)
        self.startPaste_ending1M()

    #  startPaste_ending1M
    def startPaste_ending1M(self):
        """ messageを出力し、dialogを閉じる"""
        title = _("meshの copy & paste")
        mess = _("meshを貼り付けました。")
        self.okDialog(title, mess)
        self.close()    #dialogをcloseする

    #
    #  getNCellPolyMesh
    def getNCellPolyMesh(self, caseDir, meshDir):
        """ cell数を取得。ouner, neighbourが存在しない場合は「0」を戻す"""
        #owner側のcellNoを確認
        fileName = meshDir + "/owner"
        fileName = pyTreeFoam.case(caseDir).checkFileName(fileName)
        if len(glob.glob(fileName)) == 0:
            return 0
        foam = pyTreeFoam.foamFile()
        (ascii, data) = foam.readSeparate(fileName)
        if foam.checkBinary(ascii) == "binary":
            numData = foam.structBinData(data[0])
            ownerMax = max(numData)
        else:
            sList = data[0][1].split()
            ownerMax = 0
            for n in sList:
                if len(n) > 0:
                    num = int(n)
                    if ownerMax < num:
                        ownerMax = num
            return ownerMax

    #
    #  shrinkLenStr
    def shrinkLenStr(self, name, nMax):
        """ 文字列を指定の長さに縮める。"""
        if len(name) > nMax:
            name = "..." + name[-(nMax-3):]
        return name

    #
    #  setCaseNameToLabel
    def setCaseNameToLabel(self):
        """ copy, pasteのcase名をlabelに設定する"""
        copyDir = self.shrinkLenStr(self.copyDir, 80)
        self.label_copyDir.set_text(copyDir)
        pasteDir = self.shrinkLenStr(self.pasteDir, 80)
        self.label_pasteDir.set_text(pasteDir)
        copyName = self.copyDir.split("/")[-1]
        copyName = self.shrinkLenStr(copyName, 40)
        pasteName = self.pasteDir.split("/")[-1]
        pasteName = self.shrinkLenStr(pasteName, 40)
        self.label_copyCase.set_text(_("case名: ") + copyName)
        self.label_pasteCase.set_text(_("case名: ") + pasteName)

    #
    #  iniSetMeshesToListBox
    def iniSetMeshesToListBox(self):
        """ meshをlistBoxに書き込み"""
        #copyDirのmesh読み込み
        meshDirs = getAllPolyMeshDirsInCase(self.copyDir)
        meshDirs.sort()
        #  listBoxに設定
        names = list(map(lambda x: x[len(self.copyDir)+1:]+"/polyMesh", meshDirs))
        self.listBoxCopy.setItems(names)
        self.listBoxCopy.selectItemNos([0])
        #pasteDirのmesh読み込み
        meshDirs = getAllPolyMeshDirsInCase(self.pasteDir)
        meshDirs.sort()
        #  region, 数字folderをチェックし追加
        meshDirs = addNumRegionFolders(self.pasteDir, meshDirs)
        #  listBoxに設定
        names = list(map(lambda x: x[len(self.pasteDir)+1:], meshDirs))
        self.listBoxPaste.setItems(names)
        self.listBoxPaste.selectItemNos([0])

    #
    #  initialize
    #-------------
    def initialize(self):
        #case名をセット
        self.setCaseNameToLabel()
        #meshをlistBoxにセット
        self.iniSetMeshesToListBox()


    #  ------- universal dialog ---------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def yesNoDialog(self, title, mess, funcYes=[], funcNo=[]):
        dialog = unvDlg.yesNoDialog(
            title, mess, funcYes=funcYes, funcNo=funcNo, 
            parentWin=self.mainWindow)
        dialog.show()


#
#  getAllPolyMeshDirsInCase
#----------------------------
def getAllPolyMeshDirsInCase(caseDir):
    """ caseDir内の全polyMesh（region含む）を取得する"""
    meshDirs = getConstantPolyMesh(caseDir)
    meshDirs += getTimePolyMeshDirs(caseDir)
    return meshDirs

#
#  getConstantPolyMesh
def getConstantPolyMesh(caseDir):
    """ constant内のpolyMesh（region含む）を取得する"""
    dirs = glob.glob(caseDir + "/constant/polyMesh/boundary")
    dirs += glob.glob(caseDir + "/constant/*/polyMesh/boundary")
    meshDirs = []
    for name in dirs:
        #polyMesh/boundaryを削除
        nameList = name.split("/")[:-2]
        meshDirs.append("/".join(nameList))
    meshDirs.sort()
    return meshDirs

#
#  getTimePolyMeshDirs
def getTimePolyMeshDirs(caseDir):
    """ timeFolder内のpolyMesh（region含む）を取得する"""
    dirs = glob.glob(caseDir + "/*/polyMesh/boundary")
    dirs += glob.glob(caseDir + "/*/*/polyMesh/boundary")
    aMeshDirs = []
    for name in dirs:
        nameList = name[len(caseDir)+1:].split("/")[:-2]
        try:
            a = float(nameList[0])
            aMeshDirs.append([a, nameList])
        except:
            pass
    aMeshDirs.sort()
    meshDirs = []
    for aMeshDir in aMeshDirs:
        meshDir = caseDir + "/" + "/".join(aMeshDir[1])
        meshDirs.append(meshDir)
    return meshDirs

#
#  addNumRegionFolders
def addNumRegionFolders(caseDir, meshDirs):
    """ meshDirs中に数字folderとregionFolderを追加する"""
    #caseDirからの相対dirを取得
    meshes = []
    for meshDir in meshDirs:
        polyMesh = meshDir[len(caseDir)+1:]
        meshes.append(polyMesh)
    #constantフォルダをチェックし、無ければ追加。
    regions = []
    numFolders = []
    cFlag = 0
    for mesh in meshes:
        words = mesh.split("/")
        if words[0] == "constant":
            cFlag = 1
        if len(words) >= 2:
            #regionを取得
            regions.append(words[1])
        try:
            #timeFolder有無確認
            a = float(words[0])
            numFolders.append([a, words[0]])
        except:
            pass
    if cFlag == 0:
        #constantフォルダを追加
        meshes.append("constant")
    #regionsを整形
    if len(regions) > 0:
        regions = list(set(regions))
        regions.sort()
    #計算startFolderを取得
    currTimeFolder = pyTreeFoam.case(caseDir).getCurrTimeFolder()
    if len(numFolders) == 0:
        #timeFolder内にmeshが存在しない場合
        #  meshの無いtimeFolderを追加
        meshes.append(currTimeFolder)
        if len(regions) > 0:
            #regionも追加
            for region in regions:
                meshes.append(currTimeFolder+"/"+region)
    else:
        #timeFolderが存在する場合
        #  currTimeFolder内にmeshがあるか確認
        flag = 0
        for mesh in meshes:
            words = mesh.split("/")
            if words[0] == currTimeFolder:
                flag = 1
                break
        if flag == 0:
            #currTimeFolder内にmeshがない場合
            meshes.append(currTimeFolder)
            if len(regions) > 0:
                #regionがあれば追加
                for region in regions:
                    meshes.append(currTimeFolder+"/"+region)
    #絶対pathに戻す
    meshDirs = list(map(lambda x: caseDir+"/"+x, meshes))
    return meshDirs

#
#  copyRegionProperties
def copyRegionProperties(selCaseDir, toCopyCaseDir):
    """ regionPropertiesをコピーする"""
    copyFile = selCaseDir + "/constant/regionProperties"
    pasteDir = toCopyCaseDir + "/constant"
    error = ""
    if len(glob.glob(copyFile)) == 0:
        error = "NG"
    else:
        shutil.copy(copyFile, pasteDir)
        error = ""
    return error

#
#  clearInternalBoundaryFieldMultiPolyMesh
def clearInternalBoundaryFieldMultiPolyMesh(caseDir, meshDir, sources):
    """ 複数polyMeshをコピーした時、internal, boundaryFieldクリアする"""
    regionDirs = glob.glob(meshDir + "/*/polyMesh")
    regions = []
    for regionDir in regionDirs:
        region = regionDir.split("/")[-2]
        regions.append(region)
    currTime = pyTreeFoam.case(caseDir).getCurrTimeFolder()
    i = 0
    for region in regions:
        fieldDir = caseDir + "/" + currTime + "/" + region
        if len(glob.glob(fieldDir)) == 0:
            #timeFolderにregionがない場合、fieldをコピーする
            copyFieldParentToRegion(caseDir+"/"+currTime, region)
        fields = pyTreeFoam.case(caseDir).getFieldNames(currTime, region)
        polyMeshDir = meshDir + "/" + region
        setCase = pyTreeFoam.case(caseDir)
        setCase.clearNullBoundaryFields(currTime, polyMeshDir, fields, region)
        setCase.clearInternalFields(fieldDir, fields)
        i += 1
    return

#
#  copyFieldParentToRegion
def copyFieldParentToRegion(parentDir, region):
    """ timeFolderのfieldをregionにコピーする"""
    files = glob.glob(parentDir + "/*")
    os.mkdir(parentDir + "/" + region)
    for file in files:
        if os.path.isfile(file) == True:
            shutil.copy(file, parentDir + "/" + region)

#
#  clearNullBoundaryFieldsInternalFields
def clearNullBoundaryFieldsInternalFields(caseDir, polyMeshDir):
    """ メッシュ入れ替え時のnullpatchのみboundaryFieldとinternalFieldをクリアする"""
    #fieldDir, fieldsを取得
    foamCase = pyTreeFoam.case(caseDir)
    time = foamCase.getCurrTimeFolder()
    region = polyMeshDir.split("/")[-1]
    if region == "constant":
        region = "."
        fieldDir = caseDir + "/" + time
    else:
        try:
            float(region)
            region = "."
        except:
            pass
        fieldDir = caseDir + "/" + time + "/" + region
    fields = foamCase.getFieldNames(time, region)
    #currMeshDirの取得
    relativeCurrPolyMeshDir = foamCase.getCurrMeshDir(time, region, "boundary")
    currPolyMeshDir = caseDir + "/" + relativeCurrPolyMeshDir
    meshDir = "/".join(currPolyMeshDir.split("/")[:-1])
    #boundaruFieldのクリア
    foamCase.clearNullBoundaryAllFields(time, meshDir, region)
    #internalFieldのクリア
    foamCase.clearInternalFields(fieldDir, fields)
    return

#
#  clearOnlyNullBoundaryFields
def clearOnlyNullBoundaryFields(caseDir, polyMeshDir):
    """ メッシュ入れ替え時のnullPatchのみboundaryfieldをクリアする"""
    #fieldDir, fieldsを取得
    foamCase = pyTreeFoam.case(caseDir)
    time = foamCase.getCurrTimeFolder()
    region = polyMeshDir.split("/")[-1]
    if region == "constant":
        region = "."
        fieldDir = caseDir + "/" + time
    else:
        try:
            float(region)
            region = "."
        except:
            pass
        fieldDir = caseDir + "/" + time + "/" + region
    fields = foamCase.getFieldNames(time, region)
    #currMeshDirの取得
    relativeCurrPolyMeshDir = foamCase.getCurrMeshDir(time, region, "boundary")
    currPolyMeshDir = caseDir + "/" + relativeCurrPolyMeshDir
    meshDir = "/".join(currPolyMeshDir.split("/")[:-1])
    foamCase.clearNullBoundaryFields(time, meshDir, fields, region)



if __name__ == "__main__":
    import gettext
    import locale
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    locale.bindtextdomain("treefoam", localeDir)
    gettext.install("treefoam", localeDir)

    copyDir = sys.argv[1]
    pasteDir = sys.argv[2]

    winApp = windowApp(copyDir, pasteDir)
    winApp.main()
