#!/usr/bin/python3
# -*- coding: utf-8 -*-
#
#       meshViewerFullDialog.py
#
#   コマンドの内容
#
#   start           command処理開始
#
#   show            指定したparts（patch, zone, set, stl）を表示
#   showActor        例: show patch:inW cellZone:block faceSet:faceF stl:outW
#
#   addShow         指定したparts（patch, zone, set, stl）を追加表示
#   addShowActor    例: addShow patch:inW cellZone:block faceSet:faceF
#
#   clear           vtk画面をクリアする
#                   例: clear
#
#   print           statusBar上に出力する
#                   例: print 'loading ...'\n
#
#   load            指定したvtkDataを読み込む
#   reload          例: load [all] [patchesZones] [sets] [stls]
#                       load sets stls     #setsとstlを再読込する。
#
#   case            caseDirを指定する。実行後は「load all」で再読込する
#   caseDir         例: case ~/test/cavity
#
#   region          regionを指定する。実行後は「load all」で再読込する。
#                   例: region .
#
#   time            timeFolderを指定する。実行後は、「load all」で再読込する。
#                   例: time 0.6
#
#   mess            messageを表示する
#   message         例: mess "topoSetEditor\n 選択partsが表示されます"
#
#   addMess         messageを追加表示する
#   addMessage      例: addMess 追加表示
#
#   stl             stlFileの場所を指定する。実行後は「load stls」で再読込する
#   stlDir          例: stl ./model1
#
#   edge            edge表示を制御する
#                   例: edge <option>  option: show or hide
#
#   outline         outline表示を制御する
#                   例: outline <option>  option: show or hide
#
#   face            face表裏の表示を制御する
#                   例: face <option>  option: both, front or back
#
#   setTreeList     TreeListを表示する
#                   例:  setTreeList
#
#   wait            GUIを表示させる為の待ち時間を設定
#                   例: wait 0.1s  or wait 100ms  or wait 0.1
#
#   end             入力threadを終了させる
#
#   close           meshViewerを閉じる
#   exit
#   quit
#
#
#   21/08/05    新規作成
#      09/04    showItemSize:minLocationの座標の表示を追加。
#      09/15    setsのvtk作成をmeshViewerCreateSetsVtk.pyで作成。
#               foamToVtkコマンドを使っていたが、OF-v2006では出力結果
#               が異なる為、共用できない。
#      09/23    commandLoad:delayを設定して実行する様に修正。
#               大規模メッシュの場合、loadに時間がかかる為。
#      11/18    setViewerTitle:GUIのtitle表示を追加。
#      11/23    close:終了時にos.killpgで強制終了する様に変更
#               threadでinputを使っており、これが終了できていない為。
#      11/25    waitコマンド、clearコマンド追加
#      11/30    setCaseDir:「self.label_caseDir.set_text(title)」を
#               「GLib.idle_add(self.label_caseDir.set_text, title)」
#               に修正。threadでlabelをセットする場合、以下error発生の為
#               gtk_label_update_layout_width: assertion failed
#      12/01    threadでGUIを修正sしていたものをmain側でGUI操作する様に
#               書き換え
#      12/08    windowApp:meshViewerDialog内のwindowAppを継承して定義
#               する様に修正。
#      12/14    選択item(region, patch, zones, sets)の表示を追加
#      12/24    setWindowSize:splitterのdefault値を設定
#      12/25    reloadMesh:threadとtimerEventの起動チェックを追加
#               エラー等で停止している場合は、再起動させる。
#      12/29    gladeからのevent処理を修正。
#               commandEndを追加（この処理でtreeにfocusする）
#   22/01/02    onFocusTree:mauseがtree上に来たらtreeにFOCUSするを追加。
#      01/05    timeの設定用のcomboBoxを追加。
#      01/06    commandEnd:meshが読み込めない場合、「noMesh」を表示。
#               reloadMesh:選択表示項目の保存を追加。
#               commandSetTreeList:設定後、選択表示させる様に修正。
#      03/26    commandTime:solvCaseDirがfolderの場合、timeのcomboBox
#               をクリアする様に修正。
#      03/27    commandSetTreeList:stlをtreeNodesに追加
#               getSelectItems_showItems:stlの表示を追加
#               entry_stlDirを追加。
#      04/04    editStlAtStlViewer:stl編集用のstlViewer起動ボタンを追加
#               閉じるボタンを追加
#   24/01/19    changeTime:faceZonesファイルの内容のチェックを追加。
#               OF-11以上でチェックを追加。
#      10/14    editStlAtStlViewer:editStlFiles:実在しないstlDirの場合、エラー発生する為、修正。
#      10/16    changeStlDirFromRef:selDirを直接entryからstlDirを取得する様に修正。
#   25/02/05    大幅変更。internalMesh, clip, planeの表示を追加。
#               commandSetTreeList:multiRegionCaseのTree表示を修正（region名まで）
#      03/30    「import vtk」を「import gi」の前に移動。vtk-9.4対応
#      08/08    changeTime:高速表示させる為、「clear」「wait」コマンドを削除。
#               button_decTime,button_incTimeのボタンを追加。
#


import checkVtkModule
isVtk = checkVtkModule.isVtk("meshViewer")
if isVtk == True:
    import vtk

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GLib

import os, sys

import meshViewerDialog
import GtkParts
import pyTreeFoam
import getOpenFolderDDialog

#-------------------
#  windowApp class
#-------------------
class windowApp(meshViewerDialog.windowApp):

    gladeFileName = "meshViewerFullDialog.glade"

    def __init__(self, caseDir, timeFolder, region, inputFile, title):
        super().__init__(caseDir, timeFolder, region, inputFile, title)
        #変数の設定
        self.treeData = []
        self.timeFolders = []
        self.selectedDirInTree = []
        self.maskEvent = True           #eventをmaskする
        #treeWidgetの設定
        self.treeWidget = GtkParts.treeWidget(self.tree_item)
        #treeのitemを定義(item、headerの内容、itemの位置変数を定義)
        (headers, itemDefines, colWidthes) = self.defineTreeItem()
        #treeViewの定義
        self.treeWidget.defineTreeView(headers, itemDefines,
                    colWidthes, multi=True)
        #comboの定義
        self.comboTime = GtkParts.comboBoxText(self.combo_time)
        #entryの値を設定
        self.entry_stlDir.set_text(self.relativeStlDir)
        #default値を設定
        self.default_invert = False         #invertなし
        self.default_normal = (0, 1, 0)     #y方向
        #clip関連objectを初期値に設定
        #radio, checkをdefaultに設定する
        self.setDefaultSettingOfClipObject()
        #clip関連objectを隠す
        self.hideClipObject()

    #
    #  hideClipObject
    def hideClipObject(self):
        """ clip関連のobjectを隠す"""
        self.showHideClipObject(False)

    #
    #  showClipObject
    def showClipObject(self):
        """ clip関連のobjectを表示する"""
        self.showHideClipObject(True)

    #  showHideClipObject
    def showHideClipObject(self, flag):
        """ clipObjectをflagに従ってshow or hide する"""
        self.label_clip.set_sensitive(flag)
        self.radio_x.set_sensitive(flag)
        self.radio_y.set_sensitive(flag)
        self.radio_z.set_sensitive(flag)
        self.check_invert.set_sensitive(flag)
        self.check_default.set_sensitive(flag)
        self.button_clip.set_sensitive(flag)

    #
    #  setDefaultSettingOfClipObject
    def setDefaultSettingOfClipObject(self):
        """ clipObjectを初期の設定にする"""
        normal = self.default_normal
        invert = self.default_invert
        if normal[0] == 1:
            self.radio_x.set_active(True)
        elif normal[1] == 1:
            self.radio_y.set_active(True)
        elif normal[2] == 1:
            self.radio_z.set_active(True)
        else:
            self.radio_y.set_active(True)
        self.check_invert.set_active(invert)
        self.check_default.set_active(True)

    #
    #  saveWindowSize
    def saveWindowSize(self):
        """ windowSizeを保存する"""
        #winSizeを保存
        (x, y) = self.mainWindow.get_size()
        winSize = str(x) + " " + str(y)
        splitWidth = str(self.panedWin.get_position())
        winDict = {"meshViewer": winSize, "meshViewer_splitter": splitWidth}
        pyTreeFoam.writeWindowSize(winDict)

    #
    #  setWindowSize
    def setWindowSize(self):
        #sizeを取得
        winDict = pyTreeFoam.readWindowSize()
        if "meshViewer" in winDict.keys():
            if winDict["meshViewer"] != "":
                words = winDict["meshViewer"].split()
                self.winSize = (int(words[0]), int(words[1]))
        splitWidth = self.winSize[0] // 2
        if "meshViewer_splitter" in winDict.keys():
            if winDict["meshViewer_splitter"] != "":
                words = winDict["meshViewer_splitter"].split()
                splitWidth = int(words[0])
        #windowSizeを設定
        self.mainWindow.set_default_size(*self.winSize)
        #splitterの設定
        if splitWidth != "":
            if self.winSize[0] > splitWidth:
                self.panedWin.set_position(splitWidth)

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        """ glade内のobjectを取得"""
        #super(windowApp, self).setGtkObject()
        super().setGtkObject()
        #entryを追加
        self.entry_stlDir = self.builder.get_object("entry_stlDir")
        #comboを追加
        self.combo_time = self.builder.get_object("combo_time")
        #treeを追加
        self.treestore = Gtk.TreeStore(str)
        self.tree_item = self.builder.get_object("tree_item")
        self.panedWin = self.builder.get_object("panedWin")
        #checkを追加
        self.check_invert = self.builder.get_object("check_invert")
        self.check_default = self.builder.get_object("check_default")
        #buttonを追加
        self.button_refStlDir = self.builder.get_object("button_refStlDir")
        self.button_editStl = self.builder.get_object("button_editStl")
        self.button_clip = self.builder.get_object("button_clip")
        self.button_decTime = self.builder.get_object("button_decTime")
        self.button_incTime = self.builder.get_object("button_incTime")
        #radioを追加
        self.radio_x = self.builder.get_object("radio_x")
        self.radio_y = self.builder.get_object("radio_y")
        self.radio_z = self.builder.get_object("radio_z")
        #labelを追加
        self.label_clip = self.builder.get_object("label_clip")
        #tooltipを追加
        self.entry_stlDir.props.tooltip_text = _("stlファイルの保存directory（相対or絶対path）を設定")
        self.button_refStlDir.props.tooltip_text = _("dialog上でstlファイルの保存directoryを設定する")
        self.button_editStl.props.tooltip_text = _("stlViewerの起動\nstlの拡縮、回転、移動等ができる")
        mess = _("internalMeshをclipする面を指定する。")
        self.radio_x.props.tooltip_text = mess
        self.radio_y.props.tooltip_text = mess
        self.radio_z.props.tooltip_text = mess
        self.check_invert.props.tooltip_text = _("clip面を反転する。")
        self.check_default.props.tooltip_text = _("clip面をdefault面に戻す。")
        self.button_clip.props.tooltip_text = _("現在のclip面で、internalMeshをclipする。")
        self.button_decTime.props.tooltip_text = _("時間を戻す")
        self.button_incTime.props.tooltip_text = _("時間を進める")


    #----------- event handler ---------------
    def onPressMouseButton(self, widget, event):
        #self.changeCursor(widget)
        pass
    def onSelectAll(self, widget):
        self.changeCursor(widget)
    def onUnselectAll(self, widget):
        self.changeCursor(widget)
    def onChangeCursor(self, treeview):         #treeItemの選択変更時
        self.changeCursor(treeview)
    def onFocusTree(self, widget, event):       #mouse位置のwidgetにfocus
        #self.treeWidget.treeView.grab_focus()
        widget.grab_focus()
    def onChangeTime(self, event):              #timeFolder変更時
        if self.maskEvent == False:
            self.changeTime()
    def onDecTime(self, event):                 #decTime時間戻す
        self.decrementTime()
    def onIncTime(self, event):                 #incTime時間進める
        self.incrementTime()
    def onSetStlDirToEntry(self, event):        #stlDir変更時
        if self.maskEvent == False:
            self.setStlDirToEntry()
    def onChangeStlDir(self, event):            #参照...
        self.changeStlDirFromRef()
    def onEditStl(self, event):                 #stl編集
        self.editStlAtStlViewer()
    def onChangeClipXYZ(self, event):           #clip面変更時
        if self.maskEvent == False:             #  起動までmask
            self.changeClipXYZ()
    def onChangeDefault(self, event):           #default設定
        if self.maskEvent == False:
            self.changeDefault()
    def onChangeInvert(self, event):            #invert
        if self.maskEvent == False:
            self.changeInvert()
    def onCreateClip(self, event):              #clip作成
        self.createClip()
    def onClose(self, event):                   #閉じる
        self.close()
    #---- GtkVtkRenderWindowInteractorから呼び出される -----
    def onMouseDraged(self):                    #ドラッグ時
        self.mouseDraged()

    #
    #  reloadMesh
    #----------
    def reloadMesh(self):
        """ mesh再読込。
        読込前に、選択dirを取得し、読込後に選択dirを表示させる。"""
        #選択項目を保存
        self.selectedDirInTree = self.treeWidget.getSelectedDirs()
        #timeを取得
        selTime = self.comboTime.getSelectedItem()
        folderName = selTime.split(":")[0]
        #全itemをload
        print("loading patches, zones, sets, stl...")
        commLines  = ["start\n"]
        commLines += ["time " + folderName + "\n"]
        commLines += ["stlDir " + self.relativeStlDir + "\n"]
        commLines += ["clear\n"]
        commLines += ["print 'loading ...'\n"]
        commLines += ["wait 0.2s\n"]
        commLines += ["load all\n"]
        commLines += ["setTreeList\n"]
        commLines += ["end\n"]
        self.commandList += commLines
        #threadとtimerEventをcheck、停止している場合は、再起動
        self.checkThreadTimer()

    #
    #  changeTime
    #-------------
    def changeTime(self):
        #選択項目を保存
        self.selectedDirInTree = self.treeWidget.getSelectedDirs()
        #timeを取得
        selTime = self.comboTime.getSelectedItem()
        words = selTime.split(":")
        if len(words) == 1:
            self.timeFolder = selTime
        else:
            self.timeFolder = words[1]
        #faceZonesの内容をチェック(OF-11以上)---------------
        self.checkFaceZones()
        #全itemをload
        print("loading patches, zones, sets, stl...")
        commLines  = ["start\n"]
        #commLines += ["clear\n"]
        commLines += ["print 'loading ...'\n"]
        #commLines += ["wait 0.2s\n"]
        commLines += ["load all\n"]
        commLines += ["setTreeList\n"]
        commLines += ["end\n"]
        self.commandList += commLines
        #threadとtimerEventをcheck、停止している場合は、再起動
        self.checkThreadTimer()

    #
    #  decrementTime
    def decrementTime(self):
        """ 時間を戻す"""
        #全itemを取得
        items = self.comboTime.getAllItems()
        #timeを取得
        selTime = self.comboTime.getSelectedItem()
        idx = items.index(selTime) - 1
        while True:
            if idx < 0:
                idx = len(items) - 1
            selItem = items[idx]
            if len(selItem.split(":")) > 1:
                idx -= 1
            else:
                break
        self.comboTime.selectItem(idx)                

    #
    #  incrementTime
    def incrementTime(self):
        """ 時間を進める"""
        #全itemを取得
        items = self.comboTime.getAllItems()
        #timeを取得
        selTime = self.comboTime.getSelectedItem()
        idx = items.index(selTime) + 1
        while True:
            if idx >= len(items):
                idx = 0
            selItem = items[idx]
            if len(selItem.split(":")) > 1:
                idx += 1
            else:
                break
        self.comboTime.selectItem(idx)

        # words = selTime.split(":")
        # if len(words) == 1:
        #     self.timeFolder = selTime
        # else:
        #     self.timeFolder = words[1]


    #
    #  changeCursor
    #----------------
    def changeCursor(self, treeview):
        """ delayを設定して、treeViewのselectItemsを取得し、表示する。"""
        GLib.timeout_add(50, self.getSelectItems_showItems)

    #
    #  setStlDirToEntry
    #--------------------
    def setStlDirToEntry(self):
        """ entryにstlDirをセットしたとき"""
        stlDir = self.entry_stlDir.get_text()
        self.commandStlDir(stlDir)
        self.getStlActors()
        #treeNodesを修正
        stlFiles = list(self.stlActorDict.keys())
        #dataDictからstlを削除
        dataDict = self.treeWidget.dataDict
        items = list(dataDict.keys())
        for item in items:
            words = item.split("/")
            if words[0] == "stlFiles":
                _value = dataDict.pop(item)
        #dataDictにstlを追加
        if len(stlFiles) > 0:
            folderList = []
            headerList = ["stlFiles"]
            folderList.append(headerList)
            headerList = ["stlFiles", "stlDir"]
            folderList.append(headerList)
            for i in range(len(stlFiles)):
                stlFile = stlFiles[i]
                nameList = headerList + [stlFile]
                folderList.append(nameList)
            for folder in folderList:
                folderDir = "/".join(folder)
                dataDict[folderDir] = [""]
        #新dataDictを保存
        self.treeWidget.dataDict = dataDict
        #itemの辞書を作成
        itemDict = self.createItemDict(dataDict)
        self.treeWidget.itemDict = itemDict
        #treeDataを作成
        items = list(itemDict.keys())
        items.sort()
        rootName = ""
        tempDict = {}
        treeDataAll = []
        for i in range(len(items)):
            words = items[i].split(os.sep)
            if words[0] != rootName:
                rootName = words[0]
                if len(list(tempDict.keys())) > 0:
                    treeData = self.treeWidget.createTreeData(tempDict)
                    treeDataAll.append(treeData)
                tempDict = {}
            tempDict[items[i]] = [words[-1]]
        if len(list(tempDict.keys())) > 0:
            treeData = self.treeWidget.createTreeData(tempDict)
            treeDataAll.append(treeData)
        self.treeWidget.treeData = treeDataAll
        #treeNodesを作成し、表示する
        self.treeWidget.createTreeNodesMultiRoot()
        #treeを展開する
        if len(self.allRegions) == 1:
            parDir = "regions/. region/patches"
            self.treeWidget.expandChildDir(parDir)
        else:
            for region in self.allRegions:
                if region != ".":
                    reg = region + " region"
                    parDir = "regions/" + reg + "/patches"
                    self.treeWidget.expandChildDir(parDir)
        #選択項目を設定する。
        selDirs = []
        parDirs = []
        for itemDir in self.treeWidget.dataDict.keys():
            words = itemDir.split("/")
            words[-1] = words[-1].split()
            for selDir in self.selectedDirInTree:
                selWords = selDir.split("/")
                selWords[-1] = selWords[-1].split()
                if words == selWords:
                    selDirs.append(itemDir)
                    parDir = "/".join(selWords[:-1])
                    parDirs.append(parDir)
        #treeを展開する
        for parDir in parDirs:
            self.treeWidget.expandChildDir(parDir)
        #選択表示
        self.treeWidget.selectDirs(selDirs)
        self.getSelectItems_showItems()

    #
    #  changeStlDirFromRef
    #----------------------
    def changeStlDirFromRef(self):
        """ stlDirをdialogから取得して設定する"""
        title = _("stlDirの取得")
        mess = _("stlファイルが保存されているDirを選択してください")
        rootDir = os.getenv("HOME")
        currDir = self.caseDir
        #selDir = self.stlDir
        selDir = self.entry_stlDir.get_text()
        okArgs = [self.changeStlDirFromRef_setStlDir]
        dialog = getOpenFolderDDialog.getOpenFolder(title, mess, rootDir, currDir, selDir, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #  changeStlDirFromRef_setStlDir
    def changeStlDirFromRef_setStlDir(self, selDir):
        if selDir[0] == "/":
            selDir = os.path.relpath(selDir, self.caseDir)
        self.entry_stlDir.set_text(selDir)
        self.setStlDirToEntry()

    #
    #  editStlAtStlViewer
    #---------------------
    def editStlAtStlViewer(self):
        """ stlViewerによるstl編集。"""
        caseDir = self.caseDir
        stlDir = self.stlDir
        if os.path.exists(stlDir) == False or os.path.isdir(stlDir) == False:
            title = _("エラー")
            mess = _("stlDir「" + stlDir + "」は、存在しません。") + "\n"
            mess += _("stlDirを再設定してください。")
            self.errDialog(title, mess)
            return
        comm = " ".join(["editStlFilesDialog.py", caseDir, stlDir, "&"])
        pyTreeFoam.run().command(comm)

    #
    #  changeClipXYZ
    #-----------------
    def changeClipXYZ(self):
        """ clip面変更時"""
        self.maskEvent = True
        #check_defaultをclear
        self.check_default.set_active(False)
        #normalを取得
        if self.radio_x.get_active() == True:
            normal = (1, 0, 0)
        elif self.radio_y.get_active() == True:
            normal = (0, 1, 0)
        elif self.radio_z.get_active() == True:
            normal = (0, 0, 1)
        #regionを取得
        region = self.getRegionOfPlane()
        #planeWidgetにnormalを設定
        self.clipPlaneDict[region]["normal"] = normal
        self.planeWidget.SetNormal(normal)
        #再描画
        self.vtkObj.Initialize()
        self.maskEvent = False

    #
    #  changeDefault
    #----------------
    def changeDefault(self):
        """ defaultのcheckを変更した時のevent"""
        if self.check_default.get_active() == True:
            self.maskEvent = True
            #regionを取得
            region = self.getRegionOfPlane()
            #origin, normalを設定
            origin = self.getOriginFromRegion(region)
            self.planeWidget.SetOrigin(origin)
            normal = self.default_normal
            self.planeWidget.SetNormal(normal)
            #clipObjectの設定
            self.setDefaultSettingOfClipObject()
            #再描画
            self.vtkObj.Initialize()
            self.maskEvent = False

    #
    #  getOriginFromRegion
    def getOriginFromRegion(self, region):
        """ regionのactorからorigin（中心座標）を取得する"""
        actor = self.internalActorDict[region]
        bnds = actor.GetBounds()
        org = ((bnds[0]+bnds[1])/2.0,
               (bnds[2]+bnds[3])/2.0,
               (bnds[4]+bnds[5])/2.0)
        origin = []
        for val in org:
            val = pyTreeFoam.float2strAuto(val)
            origin.append(float(val))
        return origin

    #
    #  changeInvert
    #---------------
    def changeInvert(self):
        """ invertを変更したときのevent"""
        region = self.getRegionOfPlane()
        if self.check_invert.get_active() == True:
            self.check_default.set_active(False)
            self.clipPlaneDict[region]["invert"] = True
        else:
            self.clipPlaneDict[region]["invert"] = False

    #
    #  createClip
    #-------------
    def createClip(self):
        """ clipを作成する"""
        region = self.getRegionOfPlane()
        if region != "":
            #設定値を取得
            normal = self.planeWidget.GetNormal()
            origin = self.planeWidget.GetOrigin()
            invert = self.check_invert.get_active()
            #actorを作成
            self.clipPlaneDict[region]["normal"] = normal
            self.clipPlaneDict[region]["origin"] = origin
            self.clipPlaneDict[region]["invert"] = invert
            act = self.createNewClipFromPlaneWidget(region)
            #作成したactorを保存
            self.clippedMeshActorDict[region] = act
        #全actorを表示
        self.getSelectItems_showItems()

    #
    #  getRegionOfPlane
    def getRegionOfPlane(self):
        """ 選択item内からplaneItemのregionを取得する。"""
        region = ""
        selDirs = self.treeWidget.getSelectedDirs()
        for selDir in selDirs:
            words = selDir.split("/")
            if words[2] == "internalMesh":
                if words[3].split()[0] == "plane":
                    region = words[1].split()[0]
                    break
        return region

    #
    #  mouseDraged
    #--------------
    def mouseDraged(self):
        """ viewer内でマウスをドラッグした時に処理するevent"""
        #defaultのcheckを外す
        if self.check_default.get_sensitive() == True:
            self.check_default.set_active(False)

    #
    #  getSelectItems_showItems
    #---------------------------
    def getSelectItems_showItems(self):
        """ 選択itemを取得して表示する"""

        def getNewClippedMeshActor(region):
            """ clippngPlaneを取得し、今のclipedMeshのplaneと比較。
            planeが異なっていれば、clipedMeshを作り直す。"""
            #初めての場合は、初期actorを作成
            if not region in self.clipPlaneDict.keys():
                #初めての場合(planeが定義されていない場合)
                act = self.createInitialClip(region)
            else:
                #actorが定義されているか？
                if region in self.clippedMeshActorDict.keys():
                    #actorが存在する場合
                    act = self.clippedMeshActorDict[region]
                else:
                    #actorがない場合、planeから作成
                    act = self.createNewClipFromPlaneWidget(region)
            return act

        def getShowActor(region, sect, name, type):
            #clip関連のobjectを隠す
            self.hideClipObject()
            #選択itemを検索
            if sect == "internalMesh":
                if type == "plane":
                    #planeを選択
                    #  clip関連objectを表示
                    self.showClipObject()
                    #  clipPlaneが定義？
                    if not region in self.clipPlaneDict.keys():
                        #定義されていない場合、defaultでclip
                        act = self.createInitialClip(region)
                    else:
                        #clipActorが定義？
                        if not region in self.clippedMeshActorDict.keys():
                            #定義されていない場合、clipPlaneでclip
                            act = self.createNewClipFromPlaneWidget(region)
                        else:
                            #actorを取得
                            act = self.clippedMeshActorDict[region]
                elif type == "clip":
                    #planeが更新されている場合、clipを更新する
                    act = getNewClippedMeshActor(region)
                item = type
            elif sect == "patches":
                act = self.patchActorDict[region][name]
                item = "patch"
            elif sect == "zones":
                if type == "cellZone":
                    act = self.cellZoneActorDict[region][name]
                    item = "cellZone"
                elif type == "faceZone":
                    act = self.faceZoneActorDict[region][name]
                    item = type
                elif type == "pointZone":
                    act = self.pointZoneActorDict[region][name]
                    item = type
            elif sect == "sets":
                if type == "cellSet":
                    act = self.cellSetActorDict[region][name]
                    item = type
                elif type == "faceSet":
                    act = self.faceSetActorDict[region][name]
                    item = type
                elif type == "pointSet":
                    act = self.pointSetActorDict[region][name]
                    item = type
            actor = [item, act]
            return actor

        def addActorFromDir(wdir, actors):
            words = wdir.split("/")
            region = words[1].split()[0]
            self.region = region
            sect = words[2]
            type = words[3].split()[0]
            name = words[3].split()[1]
            actor = getShowActor(region, sect, name, type)
            if not actor in actors:
                actors.append(actor)
            return actors

        def addStlActorFromDir(wdir, actors):
            words = wdir.split("/")
            stlFile = words[-1]
            actorName = self.stlActorDict[stlFile]
            actor = ["stl", actorName]
            if not actor in actors:
                actors.append(actor)
            return actors

        def getAllChildren(wdir):
            folders = []
            for folderDir in self.treeWidget.dataDict.keys():
                if folderDir[:len(wdir)+1] == wdir + "/":
                    folders.append(folderDir)
            return folders

        actors = []
        selDirs = self.treeWidget.getSelectedDirs()
        clipRegion = ""
        for selDir in selDirs:
            words = selDir.split("/")
            if words[0] == "regions":
                if len(words) == 2:
                    #region選択
                    selDir = selDir + "/patches"
                    folders = getAllChildren(selDir)
                    for folder in folders:
                        actors = addActorFromDir(folder, actors)
                elif len(words) == 3:
                    #internalField?
                    if words[2] == "internalMesh":
                        region = words[1].split()[0]
                        act = self.internalActorDict[region]
                        actor = ["internalMesh", act]
                        actors.append(actor)
                    #patches等選択
                    else:
                        folders = getAllChildren(selDir)
                        for folder in folders:
                            actors = addActorFromDir(folder, actors)
                elif len(words) == 4:
                    #選択部位に応じたactorを取得
                    actors = addActorFromDir(selDir, actors)
                    #planeを選択?
                    if words[2] == "internalMesh" and words[3].split()[0] == "plane":
                        #planeが選択された時、regionを取得
                        region = words[1].split()[0]
                        clipRegion = region                        
            elif words[0] == "stlFiles":
                if len(words) == 2:
                    folders = getAllChildren(selDir)
                    for folder in folders:
                        actors = addStlActorFromDir(folder, actors)
                elif len(words) == 3:
                    actors = addStlActorFromDir(selDir, actors)
        self.currItemActors = actors[:]
        self.showActors(self.currItemActors, clipRegion=clipRegion)

    #
    #  createCommandDict
    #--------------------
    def createCommandDict(self):
        """ commandの辞書を作成"""
        commandDict = super().createCommandDict()
        commandDict["setTreeList"] = self.commandSetTreeList
        return commandDict

    #
    # commandCaseDir
    def commandCaseDir(self, line):
        """ 選択項目を取得後、caseDirを変更する。"""
        #Treeの選択項目を保存
        self.selectedDirInTree = self.treeWidget.getSelectedDirs()
        #caseDirを変更
        self.caseDir = self.getAbsPath(line)
        self.setCaseDir(self.caseDir)

    #
    #  commandSetTreeList
    def commandSetTreeList(self, line):
        """ TreeListにTreeを表示する"""
        #dataDictを作成
        dataDict = self.createDataDict()
        self.treeWidget.dataDict = dataDict
        #itemの辞書（itemDict）を作成
        itemDict = self.createItemDict(dataDict)
        self.treeWidget.itemDict = itemDict
        #treeDataを作成
        items = list(itemDict.keys())
        items.sort()
        rootName = ""
        tempDict = {}
        treeDataAll = []
        for i in range(len(items)):
            words = items[i].split(os.sep)
            if words[0] != rootName:
                rootName = words[0]
                if len(list(tempDict.keys())) > 0:
                    treeData = self.treeWidget.createTreeData(tempDict)
                    treeDataAll.append(treeData)
                tempDict = {}
            tempDict[items[i]] = [words[-1]]
        if len(list(tempDict.keys())) > 0:
            treeData = self.treeWidget.createTreeData(tempDict)
            treeDataAll.append(treeData)
        #treeData = self.treeWidget.createTreeData(itemDict)
        self.treeWidget.treeData = treeDataAll
        #treeNodesを作成し、表示する
        self.treeWidget.createTreeNodesMultiRoot()
        #treeを展開する
        if len(self.allRegions) == 1:
            parDir = "regions/. region/patches"
            self.treeWidget.expandChildDir(parDir)
        else:
            parDir = "regions"
            self.treeWidget.expandChildDir(parDir) 
            # for region in self.allRegions:
            #     if region != ".":
            #         reg = region + " region"
            #         parDir = "regions/" + reg + "/patches"
            #         self.treeWidget.expandChildDir(parDir)
        #選択項目を設定する。
        selDirs = []
        parDirs = []
        for itemDir in self.treeWidget.dataDict.keys():
            words = itemDir.split("/")
            words[-1] = words[-1].split()
            for selDir in self.selectedDirInTree:
                selWords = selDir.split("/")
                selWords[-1] = selWords[-1].split()
                if words == selWords:
                    selDirs.append(itemDir)
                    parDir = "/".join(selWords[:-1])
                    parDirs.append(parDir)
        #treeを展開する
        for parDir in parDirs:
            self.treeWidget.expandChildDir(parDir)
        #選択表示
        self.treeWidget.selectDirs(selDirs)
        self.getSelectItems_showItems()

    #
    #  commandEnd
    def commandEnd(self, line):
        """ commandListの終了処理"""
        if len(list(self.treeWidget.dataDict.keys())) <= 2:
            self.writeStatus("noMesh\n")    
        else:
            #statusBarに「Ready」を表示
            self.writeStatus("Ready\n")
        #treeViewにfocus（keyEventを効かす為）
        #self.treeWidget.treeView.grab_focus()

    #
    #  commandClear
    def commandClear(self, line):
        #vtkをクリア
        super().commandClear(line)
        #treeをクリア
        self.treeWidget.treeData = ["regions", ['. region']]
        self.treeWidget.itemDict = {'regions': ['regions'], 'regions/. region': ['. region']}
        self.treeWidget.createTreeNodes()

    #
    # commandTime
    def commandTime(self, line):
        """ timeFolderを指定し、combo_timeを設定する"""
        #changeTimeEventをmask
        self.maskEvent = True
        #timeを取得、設定
        case = pyTreeFoam.case(self.caseDir)
        self.timeFolders = case.getTimeFolders()
        try:
            contDict = case.getControlDict()
        except:
            self.maskEvent = False
            self.comboTime.setItems([""])
            self.comboTime.selectItem(0)
            return
        items = []
        if len(self.timeFolders) == 0:
            self.timeFolders = [""]
        items.append("firstTime:" + self.timeFolders[0])
        items.append("startTime:" + contDict["startTime"])
        items.append("latestTime:" + self.timeFolders[-1])
        items += self.timeFolders
        self.comboTime.setItems(items)
        if line == "startFrom" or line == "currTime":
            if contDict["startFrom"] == "firstTime":
                self.timeFolder = self.timeFolders[0]
                self.comboTime.selectItem(0)
            elif contDict["startFrom"] == "startTime":
                self.timeFolder = contDict["startTime"]
                self.comboTime.selectItem(1)
            elif contDict["startFrom"] == "latestTime":
                self.timeFolder = self.timeFolders[-1]
                self.comboTime.selectItem(2)
            else:
                self.timeFolder = self.timeFolders[0]
                self.comboTime.selectItem(0)
        elif line == "firstTime":
            self.timeFolder = self.timeFolders[0]
            self.comboTime.selectItem(0)
        elif line == "startTime":
            self.timeFolder = contDict["startTime"]
            self.comboTime.selectItem(1)
        elif line == "latestTime":
            self.timeFolder = self.timeFolders[-1]
            self.comboTime.selectItem(2)
        else:
            try:
                a = float(line)
                self.timeFolder = line
                self.comboTime.selectItem(line)
            except:
                self.timeFolder = contDict["startTime"]
                self.comboTime.selectItem(1)
        #eventのmaskを解除
        self.maskEvent = False

    #
    #  commandStlDir
    #----------------
    def commandStlDir(self, argLine):
        """ stlDirをセット"""
        super().commandStlDir(argLine)
        self.maskEvent = True
        if len(self.relativeStlDir) < len(self.stlDir):
            self.entry_stlDir.set_text(self.relativeStlDir)    
        else:
            self.entry_stlDir.set_text(self.stlDir)
        self.maskEvent = False

    #  -------------- tree 作成 ---------------------------
    #
    #  defineTreeItem
    def defineTreeItem(self):
        """ itemの内容とheaderを定義する。"""
        itemDefines = [str]
        headers     = ["regions > items > type name"]
        colWidthes  = [None]
        self.COL_DIR = 0
        self.treeWidget.COL_DIR = self.COL_DIR
        return (headers, itemDefines, colWidthes)

    #
    #  createItemDict
    def createItemDict(self, dataDict):
        """ dirをkeyとして、treeNodesに保存する内容の辞書を作成する。
        辞書の内容は、itemDefinesに従った内容。"""
        itemDict = {}
        folders = dataDict.keys()
        for folder in folders:
            words = folder.split("/")
            name = words[-1]
            itemDict[folder] = [name]
        return itemDict

    #
    #  createDataDict
    def createDataDict(self):
        """ tree作成の為のdataDictを作成する。"""

        def getNamesInRegion(region, nameDict):
            names = []
            if region in nameDict.keys():
                names = list(nameDict[region].keys())
                names.sort()
            return names

        #folderListを取得
        folderList = []
        regions = self.allRegions
        rootName = "regions"
        folderList.append([rootName])
        for region in regions:
            dataList = [rootName, region + " region"]
            folderList.append(dataList)
            #項目を取得
            patches = getNamesInRegion(region, self.patchActorDict)
            cellZones = getNamesInRegion(region, self.cellZoneActorDict)
            faceZones = getNamesInRegion(region, self.faceZoneActorDict)
            pointZones = getNamesInRegion(region, self.pointZoneActorDict)
            cellSets = getNamesInRegion(region, self.cellSetActorDict)
            faceSets = getNamesInRegion(region, self.faceSetActorDict)
            pointSets = getNamesInRegion(region, self.pointSetActorDict)
            #treeDataを追加
            #internalMesh
            itemList = dataList + ["internalMesh"]
            folderList.append(itemList)
            nameList = itemList + ["plane editClippingPlane"]
            folderList.append(nameList)
            nameList = itemList + ["clip showClippedMesh"]
            folderList.append(nameList)
            #patch
            for i in range(len(patches)):
                #patches
                itemList = dataList + ["patches"]
                folderList.append(itemList)
                #patch
                patch = patches[i]
                patchType = "-"
                if region in self.patchTypesDict.keys():
                    if patch in self.patchTypesDict[region].keys():
                        patchType = self.patchTypesDict[region][patch]
                #nameList = itemList + [patch + " " + patchType]
                nameList = itemList + [patchType + " " + patch]
                folderList.append(nameList)
            #cellZone
            for i in range(len(cellZones)):
                #zones
                itemList = dataList + ["zones"]
                folderList.append(itemList)
                #cellZone
                cellZone = cellZones[i]
                #nameList = itemList + [cellZone + " cellZone"]
                nameList = itemList + ["cellZone " + cellZone]
                folderList.append(nameList)
            #faceZone
            for i in range(len(faceZones)):
                #zones
                itemList = dataList + ["zones"]
                folderList.append(itemList)
                #faceZone
                faceZone = faceZones[i]
                #nameList = itemList + [faceZone + " faceZone"]
                nameList = itemList + ["faceZone " + faceZone]
                folderList.append(nameList)
            #pointZone
            for i in range(len(pointZones)):
                #zones
                itemList = dataList + ["zones"]
                folderList.append(itemList)
                #pointZone
                pointZone = pointZones[i]
                #nameList = itemList + [pointZone + " pointZone"]
                nameList = itemList + ["pointZone " + pointZone]
                folderList.append(nameList)
            #cellSet
            for i in range(len(cellSets)):
                #sets
                itemList = dataList + ["sets"]
                folderList.append(itemList)
                #cellSet
                cellSet = cellSets[i]
                #nameList = itemList + [cellSet + " cellSet"]
                nameList = itemList + ["cellSet " + cellSet]
                folderList.append(nameList)
            #faceSet
            for i in range(len(faceSets)):
                #sets
                itemList = dataList + ["sets"]
                folderList.append(itemList)
                #faceSet                
                faceSet = faceSets[i]
                #nameList = itemList + [faceSet + " faceSet"]
                nameList = itemList + ["faceSet " + faceSet]
                folderList.append(nameList)
            #pointSet
            for i in range(len(pointSets)):
                #sets
                itemList = dataList + ["sets"]
                folderList.append(itemList)
                #pointSet
                pointSet = pointSets[i]
                #nameList = itemList + [pointSet + " pointSet"]
                nameList = itemList + ["pointSet " + pointSet]
                folderList.append(nameList)
        #itemNameTypeを修正（幅を合わせる
        #  最大文字数を取得
        maxLen = 0
        for folder in folderList:
            if len(folder) == 4:
                nameType = folder[-1]
                n = len(nameType.split()[0])
                if n > maxLen:
                    maxLen = n
        #  nameTypeを再設定
        width = maxLen + 2
        for i in range(len(folderList)):
            folder = folderList[i]
            if len(folder) == 4:
                words = folder[-1].split()
                newNameType = words[0] + " "*(width-len(words[0])) + words[1]
                folderList[i][-1] = newNameType
        #stlをfolderListに追加
        stls = list(self.stlActorDict.keys())
        if len(stls) > 0:
            #stlをlistに追加
            headerList = ["stlFiles"]
            folderList.append(headerList)
            headerList = ["stlFiles", "stlDir"]
            folderList.append(headerList)
            for i in range(len(stls)):
                stlFile = stls[i]
                nameList = headerList + [stlFile]
                folderList.append(nameList)
        #dataDict作成
        dataDict = {}
        for folder in folderList:
            folderDir = "/".join(folder)
            dataDict[folderDir] = [""]
        return dataDict

#
#  getArguments
#---------------
def getArguments():
    """ 引数を取得する。"""
    caseDir = os.getcwd()
    timeFolder = "0"
    region = "."
    inputFile = ""
    title = "meshViewer"
    i = 1
    while i < len(sys.argv):
        arg = sys.argv[i]
        if arg == "-case":
            i += 1
            caseDir = sys.argv[i]
        elif arg == "-time":
            i += 1
            timeFolder = sys.argv[i]
        elif arg == "-region":
            i += 1
            region = sys.argv[i]
        elif arg == "-input":
            i += 1
            inputFile = sys.argv[i]
        elif arg == "-title":
            i += 1
            title = sys.argv[i]
        i += 1
    return caseDir, timeFolder, region, inputFile, title


if __name__ == "__main__":
    import gettext
    import locale
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    locale.bindtextdomain("treefoam", localeDir)
    gettext.install("treefoam", localeDir)
    #gettext.install("app")

    #引数を取得
    (caseDir, timeFolder, region,
     inputFile, title) = getArguments()
    #GUIを表示
    winApp = windowApp(caseDir, timeFolder, region,
                        inputFile, title)
    winApp.main()
