#!/usr/bin/python3
#  coding : utf-8
#
#       meshViewerControl
#
#           meshViewerDialogを制御する。
#           起動、コマンド実行、閉じる
#
#   21/08/20    新規作成
#      11/22    killViewer:新規追加（強制終了）
#      11/23    isAlive:新規追加（実行中かどうか確認）
#      11/25    isIdle:新規追加（viewerが待機中かどうか確認）
#      12/01    timerEventを設定してcancel時のコマンドを実行
#   22/01/15    openViewer:viewer起動時のpidとmeshViewerのpidを取得。
#               getViewerPid:viewerのpidを取得を追加
#               isAlive:pidで実行、停止を確認する様に修正。
#               self.viewer.poll()では、確認できない場合がある為。
#      03/26    timerEvent:viewerが停止している場合は、timerEventの
#               停止を追加。
#      05/02    timerEvent:viewerの生死をself.isAlive()に変更
#               timerEvent停止する事がある為。
#      05/07    ubuntu起動直後に、isAlive()が機能しない時があったので
#               isAliveの検出方法を変更。
#      05/11    cancelViewer:vtkをimport中にcurrDirを変更した時、
#               エラー発生。エラー発生しない様に修正。
#

import subprocess
import os
import signal
import time
import threading
import gi
gi.require_version("Gtk", "3.0")
from gi.repository import GLib

import pyTreeFoam


#-----------------
#  control class
#-----------------
class control:
    """ meshViewerDialogを制御する。
    起動、コマンド実行、閉じる"""

    def __init__(self, caseDir, timeFolder, region, title):
        self.caseDir = caseDir
        self.timeFolder = timeFolder
        self.region = region
        self.title = title
        self.viewer = None
        self.procId = None
        self.parId = None
        self.viewerName = ""
        self.idleFlag = "yes"
        self.cancelFile = os.getenv("TreeFoamUserPath") + "/temp/" + title + ".stat"
        self.closeFlag = 0
        self.delayCommandList = []

    #
    #  openViewer
    #--------------
    def openViewer(self, viewerType):
        """ meshViewerDialogを起動する。 
        """
        if viewerType == "full":
            self.viewerName = "meshViewerFullDialog.py"
        else:
            self.viewerName = "meshViewerDialog.py"
        #command作成
        comm  = self.viewerName
        comm += " -case " + self.caseDir
        comm += " -time " + self.timeFolder
        comm += " -region " + self.region
        comm += " -title " + self.title
        #起動する
        self.viewer = subprocess.Popen(
            comm,
            shell=True,
            stdin=subprocess.PIPE,
            preexec_fn = os.setsid
            )
        #子プロセスのPIDを取得
        self.parId = self.viewer.pid
        self.procId = self.getViewerPid(self.parId, self.viewerName)
        #viewerStatファイル作成
        #self.writeCancelFlag(False)
        #timerEventを起動
        GLib.timeout_add(10, self.timerEvent)

    #
    #  getViewerPid
    def getViewerPid(self, pid, viewerName):
        """ viewerのPIDを取得する"""
        #自身のpidがviewerかどうか確認
        comm = "pgrep -l -f '" + viewerName + "'"
        stat, res, err = pyTreeFoam.run().commandReturnCont(comm, isOut=False)
        lines = res.split("\n")
        #  viewerのpidを取得
        viewerPids = []
        for line in lines:
            words = line.split()
            if len(words) > 1:
                if words[1].find("meshViewer") >= 0:
                    viewerPids.append(int(words[0]))
        #  自身のpidがviewerの場合、直ぐに戻る
        if pid in viewerPids:
            return pid
        #子pidを確認する
        viewerPid = pid
        #  全ての子pidを取得
        comm = "pgrep -d' ' -P " + str(pid)
        stat, res, err = pyTreeFoam.run().commandReturnCont(comm, isOut=False)
        words = res.split()
        pids = list(map(int, words))
        #  子pidがviewerかどうか確認
        for id in pids:
            if id in viewerPids:
                viewerPid = id
                break
        return viewerPid
            
    #
    #  timerEvent
    #-------------
    def timerEvent(self):
        """ delayCommandListの内容をタイミングを図り実行する。"""
        if os.path.exists(self.cancelFile) == True:
            #cancel？
            cancelFlag = self.getCancelFlag()
            if cancelFlag == False:
                if len(self.delayCommandList) > 0:
                    #delayCommandListを実行
                    commLines = self.delayCommandList[:]
                    _stat = self.runMultiCommands(commLines)
                    self.delayCommandList = []
        GLib.timeout_add(100, self.timerEvent)

    #
    #  runCommand
    #-------------
    def runCommand(self, commLine):
        """ meshViewerにline（コマンドライン）入力する
        途中でCANCELまたは、dialogを閉じた場合は、
        BrokenPipeエラーが発生する。この場合、"CANCEL"が戻る。
        """
        stat = "OK"
        try:
            #subprocessのstdinにlineを出力する
            if commLine[-1] != "\n":
                commLine += "\n"
            commLine = commLine.encode()
            self.viewer.stdin.write(commLine)
            self.viewer.stdin.flush()
        except:
            stat = "CANCEL"
        return stat

    #
    #  runMultiCommands
    #-------------------
    def runMultiCommands(self, commLines):
        """ list形式の複数のコマンドを実行する"""
        stat = "OK"
        for line in commLines:
            stat = self.runCommand(line)
            if stat != "OK":
                break
        return stat

    #
    #  runDelayCommands
    #--------------------
    def runDelayCommands(self, commLines):
        """ commandをdelayCommandListに保存する
        実行はできる状態になってから、timerEvent内で実行する"""
        self.delayCommandList = commLines

    #
    #  runEndCommands
    #-----------------
    def runEndCommands(self, commLines):
        """ list形式の複数コマンドを実行する。
        endコマンドがくると抜け出る"""
        if commLines[-1] != "end\n" and commLines[-1] != "end":
            commLines += ["end\n"]
        self.runMultiCommands(commLines)

    #
    #  isAlive
    #-----------
    def isAlive(self):
        """ viewerが起動中かどうか確認する"""
        poll = self.viewer.poll()
        if poll == None:
            return True         #実行中
        else:
            #cancelFile有無で確認する
            if os.path.exists(self.cancelFile) == True:
                return True     #実行中
            else:
                return False    #停止中

    #
    #  closeViewer
    #--------------
    def closeViewer(self):
        """ meshViewerDialogを閉じる"""
        self.viewer.stdin.write(b"close\n")
        #if self.viewer.poll() == None:
        if self.isAlive() == True:
            #BrokenPipeでは無い時にflushする
            self.viewer.stdin.flush()
        self.procId = None
        self.parId = None

    #
    #  cancelViewer
    #-----------------
    def cancelViewer(self):
        """ meshViewerを中断させる
        signalを送信する"""
        if self.procId != None:
            #signalを子プロセスに送信(signal.SIGUSR1を送信)
            print("---->> send_signal")
            try:
                os.killpg(self.parId, signal.SIGUSR1)
                #cancelFlagをセット
                self.writeCancelFlag(True)
            except:
                print("  meshViewer does not run!")
                self.removeCancelFile()

    #
    #  getCancelFlag
    def getCancelFlag(self):
        """ cancelFileの内容を取得する。
        viewerが起動中は、cancelFileが作成されている。

        Returns:
            True:   cancel受付
            False:  cancel処理済"""
        f = open(self.cancelFile); lines = f.readlines(); f.close()
        if len(lines) > 0:
            words = lines[0].split()
            if words[1] == "True":
                return True
            else:
                return False

    #
    #  writeCancelFlag
    def writeCancelFlag(self, flag):
        """ cancelをセットする"""
        if flag == True:
            cont = "threadCancel True\n"
        else:
            cont = "threadCancel False\n"
        f = open(self.cancelFile, "w"); f.write(cont); f.close()

    #
    #  removeCancelFile
    def removeCancelFile(self):
        """ cancelFileを削除する"""
        if os.path.exists(self.cancelFile) == True:
            os.remove(self.cancelFile)
