#!/usr/bin/python3
# coding: utf-8
#
#   getOpenFolderDDialog.py
#
#       treeListからfolderを選択して返す
#
#   12/11/25    新規作成
#      12/02    相対参照も取得できるよう変更
#   13/01/07    ﾀﾞﾌﾞﾙｸﾘｯｸでfolderをopenする事を追加
#      01/12    currentDirにﾁｪｯｸﾏｰｸを追加
#      01/31    open, newFolderﾎﾞﾀﾝを追加
#      02/01    ﾎﾟｯﾌﾟｱｯﾌﾟﾒﾆｭｰを追加
#      02/09    treeにcurrentDirも表示させる事を追加
#      02/10    ｱｲｺﾝが正しく表示されないﾊﾞｸﾞ修正。
#      02/11    ｱｲｺﾝをimageFileから読むように修正
#      02/22    選択行をreload時ｽｸﾛｰﾙする様に修正
#      04/27    messageの表示をhorizontal centerに修正。
#      10/15    国際化のため、修正
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   15/06/25    wxPython-2.8→3.0変更に伴う修正
#   19/09/15    python3, GTK+3用に大幅修正
#   20/03/12    serverDir、linkDirを表示（バグ修正）
#      04/21    多言語化対応
#   21/07/30    __init__:selDirが相対pathの場合、絶対pathに変更する
#               様に修正。
#   22/02/01    __init__:rootDir, currDir, selDirをself変数に変換する時
#               os.normpath()関数を通して変換するように修正。
#   24/08/08    funcOk,funcCancel,parentの設定を追加。
#      11/08    getFolderContDict:rootDir,currDir,selDirのnormalPathの取得追加
#               selDirが「aa/./mm」形式のdirの時、エラー発生した為、修正。
#      12/04    __init__:一部英訳できていなかった為修正（バグ修正）
#      12/13    __init__:親のfreezeを削除
#   25/01/29    __init__:selDirが相対pathの場合、選択dirとして選択されなかった
#               為、修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GLib, GdkPixbuf, Pango
import cairo
import os
import glob

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

try:
    import pyTreeFoam
except:
    import python.pyTreeFoam as pyTreeFoam


#  ------------------
#  getOpenFolder class
#  ------------------
class getOpenFolder:
    """ folderを選択し、そのfolderDirを戻す。
    treeFoamライクにtree表示された中からfolderを取得する。
    
    Args:
        title (str)         :dialogのtitle
        dialogCont (str)    :dialogに表示するmessage
        rootDir (str)       :treeの最上位dir
        currDir (str)       :基準のdir
        selDir (str)        :選択したfolderDir
        funcOk (list)       :「OK」時に実行する関数名と引数のlist
        funcCancel (list)   :「cancel」時に実行する関数名と引数のlist
        parent (obj)        :親window
    """
                
    def __init__(self, title, dialogCont, rootDir, currDir, selDir, 
                 funcOk=[], funcCancel=[], parent=None):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"getOpenFolderDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        #親windowを設定
        self.dialog.set_transient_for(parent)
        #modal（freeze）設定
        #self.dialog.set_modal(True)
        self.dialog.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を取得
        self.setGtkObject()
        #titleと内容をセット
        self.dialog.set_title(title)
        self.label_dialogCont.set_text(dialogCont)
        self.label_currDir.set_text(_("現在のフォルダ: ")+currDir)
        if len(currDir) > 80:
            self.label_currDir.set_text(_("現在のフォルダ: ")+"..."+currDir[-70:])
        #変数をセット
        self.rootDir = os.path.normpath(rootDir)
        self.currDir = os.path.normpath(currDir)
        self.selDir = os.path.normpath(selDir)
        if self.selDir[0] != "/":
            #selDirが相対pathの場合
            self.selDir = self.currDir + "/" + self.selDir
            self.selDir = os.path.normpath(self.selDir)
            self.check_relative.set_active(True)
        else:
            #selDirが絶対pathの場合
            self.check_relative.set_active(False)
        if len(glob.glob(self.selDir)) == 0:
            self.selDir = self.currDir
        self.funcOk = funcOk
        self.funcCancel = funcCancel
        #self.status = "CANCEL"
        self.maskEvent = False
        self.serverDirs = self.getServerDirs()
        #folderDirを取得
        self.folderContDict = getFolderContDict(self.rootDir, self.currDir, self.selDir)
        #treeDataを取得
        self.treeData = createTreeData(self.folderContDict, self.rootDir)
        #treeViewを作成
        self.setTreeDataCaseConts()
        #dialog window sizeを設定
        (x,y) = self.windowSize
        self.dialog.set_default_size(x,y)

    #
    #  show
    #  ----------
    def show(self):
        """ dialogの表示"""
        self.dialog.show()
        # self.dialog.run()
        # #結果を取得する
        # if self.status != "OK":
        #     self.selDir = ""
        # return self.selDir

    def close(self, *args):
        """ dialogを閉じる"""
        # self.status = "CANCEL"
        # self.selDir = ""
        self.dialog.destroy()

    def setGtkObject(self):
        """ dialog内のwidgetObjectを設定"""
        self.label_dialogCont = self.builder.get_object("label_dialogCont")
        self.label_currDir = self.builder.get_object("label_currDir")
        self.treeView = self.builder.get_object("tree_treeDir")
        self.check_relative = self.builder.get_object("check_relative")
        #iconの定義
        iconDir = os.getenv("TreeFoamPath") + "/icons"
        iconFile = GdkPixbuf.Pixbuf.new_from_file
        self.iconFolder = iconFile(iconDir+"/folder16.png")
        self.iconFolderCk = iconFile(iconDir+"/folderCheck16.png")
        self.iconFolderLn = iconFile(iconDir+"/folderLink16.png")
        self.iconFolderLnCk = iconFile(iconDir+"/folderLinkCheck16.png")
        self.iconCase = iconFile(iconDir+"/x-office-document16.png")
        self.iconCaseCk = iconFile(iconDir+"/x-office-document_check16.png")
        self.iconCaseLn = iconFile(iconDir+"/x-office-document_link16.png")
        self.iconCaseLnCk = iconFile(iconDir+"/x-office-document_link_check16.png")
        self.iconSalome = iconFile(iconDir+"/edit-copy16.png")
        self.iconSalomeCk = iconFile(iconDir+"/edit-copyCheck16.png")
        self.iconSalomeLn = iconFile(iconDir+"/edit-copyLink16.png")
        self.iconSalomeLnCk = iconFile(iconDir+"/edit-copyLinkCheck16.png")
        self.iconServerFolder = iconFile(iconDir+"/serverFolder16.png")
        self.iconServerFolderCk = iconFile(iconDir+"/serverFolderCheck16.png")
        self.iconServerFolderLn = iconFile(iconDir+"/serverFolderLink16.png")
        self.iconServerFolderLnCk = iconFile(iconDir+"/serverFolderLinkCheck16.png")
        self.iconSolver = iconFile(iconDir+"/folder-documents16.png")
        self.iconBlank = None
        #fontの定義
        self.NORMAL = Pango.Weight.NORMAL   #標準
        self.BOLD = Pango.Weight.BOLD       #bold
        #treeStoreの位置定義
        self.COL_ICON = 0               #folderIcon
        self.COL_DIR = 1                #dir名
        self.COL_ICON_SOLVER = 2        #solverIcon
        self.COL_SOLVER = 3             #solver名
        self.COL_BCPN = 4               #BCPn
        self.COL_NR = 5                 #nR
        self.COL_ST = 6                 #st
        self.COL_ED = 7                 #ed
        self.COL_WEIGHT = 8             #Pango.weight(bold)
        self.COL_COLOR = 9              #fontColor
        self.windowSize = (700, 300)    #doalogのサイズ
        self.colWidth = [300,           #Tree width
                         200,           #solver width
                         40,            #BCPn width
                         40,            #nR width
                         60,            #st width
                         60]            #ed width
        #local用のicon
        self.iconLocalDict = {
            #foltype        iconTree            iconSolber       checkIcon
            "folder":       (self.iconFolder,   self.iconBlank,  self.iconFolderCk),
            "folderLink":   (self.iconFolderLn, self.iconBlank,  self.iconFolderLnCk),
            "case":         (self.iconCase,     self.iconSolver, self.iconCaseCk),
            "caseLink":     (self.iconCaseLn,   self.iconSolver, self.iconCaseLnCk),
            "salome":       (self.iconSalome,   self.iconBlank,  self.iconSalomeCk),
            "salomeLink":   (self.iconSalomeLn, self.iconBlank,  self.iconSalomeLnCk)
            }
        #server用のicon
        self.iconServerDict = {
            "folder":       (self.iconServerFolder,   self.iconBlank,  self.iconServerFolderCk),
            "folderLink":   (self.iconServerFolderLn, self.iconBlank,  self.iconServerFolderLnCk),
            "case":         (self.iconCase,           self.iconSolver, self.iconCaseCk),
            "caseLink":     (self.iconCaseLn,         self.iconSolver, self.iconCaseLnCk),
            "salome":       (self.iconSalome,         self.iconBlank,  self.iconSalomeCk),
            "salomeLink":   (self.iconSalomeLn,       self.iconBlank,  self.iconSalomeLnCk)
            }

    #  ------- event handler ---------------
    def onExpandDir(self, treeview, treeiter, path):
        if self.maskEvent == False:
            self.expandDir(path)
    def onApply(self, event):
        selDir = self.getSelFolderDir()
        self.close()
        if len(self.funcOk) > 0:
            funcName = self.funcOk[0]
            args = self.funcOk[1:]
            #funcName(selDir, *args)
            GLib.timeout_add(100, funcName, selDir, *args)
    def onCancel(self, event):
        self.close()
        if len(self.funcCancel) > 0:
            funcName = self.funcCancel[0]
            args = self.funcCancel[1:]
            #funcName(*args)
            GLib.timeout_add(100, funcName, *args)
    #  ------------------------------------

    def getServerDirs(self):
        """ serverDisを取得する"""
        mountServers = pyTreeFoam.servers().getMountedServers()
        if len(mountServers) == 0:
            return []
        serverDirs = list(map(lambda x: x[2], mountServers))
        return serverDirs

    def getSelFolderDir(self):
        """ 選択したfolderDirを取得"""
        treeSelection = self.treeView.get_selection()
        _treestore, pathes = treeSelection.get_selected_rows()
        path = pathes[0]
        selDirList = self.getDirFromTreePath(path)
        selDir = os.sep.join(selDirList)
        if self.check_relative.get_active() == True:
            selDir = os.path.relpath(selDir, self.currDir)
        return selDir

    def expandDir(self, path):
        """ treeView上のtreeを展開(expand)する"""
        #expandDirを取得
        expandDir = os.sep.join(self.getDirFromTreePath(path))
        #subFolder名を取得
        treeiter = self.treestore.get_iter(path)
        subFolderIter = self.getSubFoldersInTree(treeiter)
        #subFolderの下にfolderが存在するかどうか
        if self.isSubfolderInfolders(subFolderIter) == True:
            #folderが存在する場合、戻る
            pass
        else:
            #実際のfolderを確認し、treeに追加する
            self.addSubFoldersInTree(expandDir, subFolderIter)
        
    def getSubFoldersInTree(self, treeiter):
        """ treeiterをparentとしてchildrenのnameとiterを取得"""
        iter = self.treestore.iter_children(treeiter)   #firstChild
        subFolders = []
        while iter:
            subFolders.append([self.treestore.get_value(iter, self.COL_DIR), iter])
            iter = self.treestore.iter_next(iter)
        return subFolders

    def isSubfolderInfolders(self, folderIter):
        """ flderIterをparentとしてchildren有無を確認"""
        flag = False
        for [dummy, treeiter] in folderIter:
            subFolders = self.getSubFoldersInTree(treeiter)
            if len(subFolders) != 0:
                flag = True
                break
        return flag

    def addSubFoldersInTree(self, expandDir, subFolderIter):
        """ 実際のfolderを取得し、Treeに追加する"""
        #追加するfolderContDataを取得
        names = list(map(lambda x: x[0], subFolderIter))
        folderConts = self.getFolderContData(expandDir, names)
        #folderContDictを更新
        for folderCont in folderConts:
            self.folderContDict[folderCont[0]] = folderCont[1:]
        #treeDataを再作成
        rootDir = self.treeData[0]
        self.treeData = createTreeData(self.folderContDict, rootDir)
        #treeViewにnodeを追加
        for folderCont in folderConts:
            itemDir = folderCont[0]
            parentDir = os.sep.join(itemDir.split(os.sep)[:-1])
            parentIter = self.getIterInTreeStore(rootDir, parentDir)
            itemName = itemDir.split(os.sep)[-1]
            weight = self.NORMAL
            color = "black"
            (folType, solver, BCPn, nR, st, ed) = self.getFolderConts(itemDir)
            (iconFol, iconSolv) = self.getImageIcon(folType, itemDir)
            #treeViewに追加
            self.treestore.insert(parentIter, -1, 
                            [iconFol, itemName, iconSolv, solver,
                             BCPn, nR, st, ed, weight, color])

    def getFolderContData(self, expandDir, names):
        """ expandDir + name 以下のfolderContDataを取得する"""
        #folderDirを取得
        folderDirs = []
        for name in names:
            currDir = os.sep.join([expandDir, name])
            folders = pyTreeFoam.getFoldersOnlyCurrDir(currDir)
            folders = list(filter(pyTreeFoam.isNeedFolder, folders))
            folderDirs += folders
        #folderDirsにfolderの内容を追加
        folderConts = addFolderContsData(folderDirs)
        folderConts.sort()
        return folderConts

    # ----- treeView 作成 ----------------------------
    def setTreeDataCaseConts(self):
        """ treeViewを作成する"""
        #treeEventをmaskする
        self.maskEvent = True
        #treeStoreを定義
        self.defineTreeStore()
        #treeを作成
        self.createTreeNodes()
        #treeの内容を定義
        self.defineTreeConts()
        #currDirのTreeを設定
        self.setCurrCaseInTree(self.rootDir, self.currDir)
        #currDir, selDirまでexpandする
        self.expandCurrSelDir(self.rootDir, self.currDir, self.selDir)
        #eventのmaskを解除
        self.maskEvent = False

    def defineTreeStore(self):
        pixbuf = GdkPixbuf.Pixbuf
        weight = Pango.Weight
        self.treestore = Gtk.TreeStore(
            #icon   dir  icon    slv  BCPn nR   st   ed   (bold)
            pixbuf, str, pixbuf, str, str, str, str, str, weight, str)

    def createTreeNodes(self):
        """ treeData(list)からtreeViewのnodeを作成する"""
        rootName = self.treeData[0]
        items = self.treeData[1]
        rootIter = self.treestore.append(
            None, [self.iconFolder, rootName, self.iconBlank, "",
                   "", "", "", "", self.NORMAL, "black"])
        self.addTreeNodes([rootName], self.treestore, rootIter, items)

    def addTreeNodes(self, names, treeStore, parentIter, items):
        iconFol = self.iconFolder; iconSolv = self.iconSolver; weight = self.NORMAL
        solver = ""; BCPn = ""; nR = ""; st = ""; ed = ""; color = "black"
        path = treeStore.get_path(parentIter)
        parentDir = self.getDirFromTreePath(path)
        for item in items:
            if type(item) == str:
                itemDir = parentDir + [item]
                (folType, solver, BCPn, nR, st, ed) = self.getTreeConts(itemDir)
                (iconFol, iconSolv) = self.getImageIcon(folType, itemDir)
                newIter = treeStore.append(
                    parentIter, [iconFol, item, iconSolv, solver,
                                 BCPn, nR, st, ed, weight, color])
            else:
                itemDir = parentDir + [item[0]]
                (folType, solver, BCPn, nR, st, ed) = self.getTreeConts(itemDir)
                (iconFol, iconSolv) = self.getImageIcon(folType, itemDir)
                newIter = treeStore.append(
                    parentIter, [iconFol, item[0], iconSolv, solver,
                                 BCPn, nR, st, ed, weight, color])
                self.addTreeNodes(names, treeStore, newIter, item[1])

    def getDirFromTreePath(self, path):
        """ treeのpathからdirectoryのlistを取得する。"""
        words = []
        for i in range(len(path)):
            treeiter = self.treestore.get_iter(path[:i+1])
            value = self.treestore.get_value(treeiter, self.COL_DIR)
            words.append(value)
        return words

    def getTreeConts(self, treeDir):
        """ treeDir(list)から、その内容を取得する"""
        treeDir = "/".join(treeDir)
        return self.getFolderConts(treeDir)

    def getFolderConts(self, folderDir):
        """ folderDirの内容を取得する"""
        conts = self.folderContDict[folderDir]
        solver = ""; BCPn = ""; nR = ""; st = ""; ed = ""
        folType = conts[0]
        if folType == "case":
            [solver, BCPn, nR, st, ed] = conts[1:]
        return (folType, solver, BCPn, nR, st, ed)

    def isServerDir(self, itemDir):
        """ Dirがserver内のDirかどうかチェックする"""
        if type(itemDir) == list:
            itemDir = "/".join(itemDir)
        itemList = itemDir.split("/")
        ans = False
        for serverDir in self.serverDirs:
            serverList = serverDir.split("/")
            n = len(serverList)
            if serverList == itemList[:n]:
                ans = True
                break
        return ans

    def getImageIcon(self, folType, itemDir):
        """ folTypeからiconを取得する"""
        if self.isServerDir(itemDir) == True:
            iconDict = self.iconServerDict
        else:
            iconDict = self.iconLocalDict
        (iconFol, iconSolve, _iconCk) = iconDict[folType]
        return (iconFol, iconSolve)

    def defineTreeConts(self):
        """ treeViewを定義する"""
        self.treeView.set_model(model=self.treestore)
        cellRenderer = Gtk.CellRendererText()
        iconRenderer = Gtk.CellRendererPixbuf()
        #Tree
        tvcolumn = Gtk.TreeViewColumn("Tree")               #title
        tvcolumn.pack_start(iconRenderer, False)
        tvcolumn.pack_start(cellRenderer, False)
        tvcolumn.add_attribute(iconRenderer, "pixbuf", self.COL_ICON)
        tvcolumn.add_attribute(cellRenderer, "text", self.COL_DIR)
        tvcolumn.add_attribute(cellRenderer, "foreground", self.COL_COLOR)
        tvcolumn.set_fixed_width(self.colWidth[0])  #cell幅を設定
        tvcolumn.set_resizable(True)                #cell幅を可変
        self.treeView.append_column(tvcolumn)
        #solver
        tvcolumn = Gtk.TreeViewColumn("solver")              #title
        tvcolumn.pack_start(iconRenderer, False)
        tvcolumn.pack_start(cellRenderer, False)
        tvcolumn.add_attribute(iconRenderer, "pixbuf", self.COL_ICON_SOLVER)
        tvcolumn.add_attribute(cellRenderer, "text", self.COL_SOLVER)
        tvcolumn.add_attribute(cellRenderer, "weight", self.COL_WEIGHT)
        tvcolumn.set_fixed_width(self.colWidth[1])  #cell幅を設定
        tvcolumn.set_resizable(True)                #cell幅を可変
        self.treeView.append_column(tvcolumn)
        #BCPn
        tvcolumn = Gtk.TreeViewColumn("BCPn")              #title
        tvcolumn.pack_start(cellRenderer, False)
        tvcolumn.add_attribute(cellRenderer, "text", self.COL_BCPN)
        self.treeView.append_column(tvcolumn)
        #nR
        tvcolumn = Gtk.TreeViewColumn("nR")              #title
        tvcolumn.pack_start(cellRenderer, False)
        tvcolumn.add_attribute(cellRenderer, "text", self.COL_NR)
        self.treeView.append_column(tvcolumn)
        #st
        tvcolumn = Gtk.TreeViewColumn("st")              #title
        tvcolumn.pack_start(cellRenderer, False)
        tvcolumn.add_attribute(cellRenderer, "text", self.COL_ST)
        self.treeView.append_column(tvcolumn)
        #ed
        tvcolumn = Gtk.TreeViewColumn("ed")              #title
        tvcolumn.pack_start(cellRenderer, False)
        tvcolumn.add_attribute(cellRenderer, "text", self.COL_ED)
        self.treeView.append_column(tvcolumn)

    def setCurrCaseInTree(self, rootDir, currDir):
        """ treeView上のcurrDir（基本dir）の設定。
        （check付icon、bold、red）"""
        #currDirの位置を取得
        treeiter = self.getIterInTreeStore(rootDir, currDir)
        #iconの設定
        dirCont = currDir.split("/")
        conts = self.getTreeConts(dirCont)
        folType = conts[0]
        #folTypeからiconを取得
        if self.isServerDir(currDir) == True:
            iconDict = self.iconServerDict
        else:
            iconDict = self.iconLocalDict
        (_iconFol, _iconSolve, icon) = iconDict[folType]
        self.treestore.set_value(treeiter, self.COL_ICON, icon)
        self.treestore.set_value(treeiter, self.COL_WEIGHT, self.BOLD)
        self.treestore.set_value(treeiter, self.COL_COLOR, "red")

    def expandCurrSelDir(self, rootDir, currDir, selDir):
        """ currDir, selDirまでTreeを展開し、選択する"""
        #currDirの展開
        expDir = os.sep.join(currDir.split(os.sep)[:-1])
        path = self.getPathInTreeStore(rootDir, expDir)
        self.treeView.expand_to_path(path)
        #selDirの展開
        expDir = os.sep.join(selDir.split(os.sep)[:-1])
        path = self.getPathInTreeStore(rootDir, expDir)
        self.treeView.expand_to_path(path)
        #dirの選択
        treeSelection = self.treeView.get_selection()
        path = self.getPathInTreeStore(rootDir, selDir)
        treeSelection.select_path(path)
        self.treeView.scroll_to_cell(path)


    def getIterInTreeStore(self, rootDir, selCaseDir):
        """ selCaseDirからtree内の位置:treeiterを取得する。
            treeiterで直接treeの場所にアクセスできる"""
        path = self.getPathInTreeStore(rootDir, selCaseDir)
        treeiter = self.treestore.get_iter(path)
        return treeiter

    #  getPathInTreeStore
    #
    def getPathInTreeStore(self, rootDir, selCaseDir):
        """ selcaseDirからtreeのpathを取得する。
            path: 0:1:4:1 """
        rootWords = rootDir.split("/")
        selWords = selCaseDir.split("/")
        #pathを取得
        selCont = selWords[len(rootWords):]
        items = self.treeData[1]
        pathInt = [0]
        for word in selCont:
            for i in range(len(items)):
                if type(items[i]) == str:
                    if word == items[i]:
                        pathInt.append(i)
                        #items = []
                        break
                else:
                    if word == items[i][0]:
                        pathInt.append(i)
                        items = items[i][1]
                        break
        path = Gtk.TreePath.new_from_indices(pathInt)
        return path
    #------------ treeView作成(ここまで) --------------------

def getFolderContDict(rootDir, currDir, selDir):
    """ folderの内容を辞書形式で取得して返す"""

    def addFoldersToDict(folders, folderDict):
        for folder in folders:
            folderDict[folder] = []
        return folderDict

    rootDir = os.path.normpath(rootDir)
    currDir = os.path.normpath(currDir)
    selDir = os.path.normpath(selDir)
    #辞書をクリア
    folderContDict = {}
    folderDict = {}
    #rootDirを設定
    folderDict[rootDir] = []
    #currDir系列のfolderを取得
    if currDir != rootDir:
        folders = pyTreeFoam.getFoldersBetweenDir(rootDir, currDir)
        folderDict = addFoldersToDict(folders, folderDict)
    #selDir系列のfolderを取得
    if currDir != selDir:
        #startDirを取得
        selWords = selDir.split("/")
        currWords = currDir.split("/")
        nLoop = len(selWords)
        if len(selWords) > len(currWords):
            nLoop = len(currWords)
        #for i in range(len(selWords)):
        for i in range(nLoop):
            if selWords[i] != currWords[i]:
                break
        st = i + 1
        startDir = "/".join(selWords[:st])
        #folderを取得
        folders = pyTreeFoam.getFoldersBetweenDir(startDir, selDir)
        folderDict = addFoldersToDict(folders, folderDict)    
    #子folderを追加
    folders = pyTreeFoam.getChildFolders(folderDict.keys())
    folderDict = addFoldersToDict(folders, folderDict)
    #folderの内容を追加（foltypeなど）
    folders = list(folderDict.keys())
    folderConts = addFolderContsData(folders)
    #結果を保存
    for folderCont in folderConts:
        folderContDict[folderCont[0]] = folderCont[1:]
    return folderContDict

def addFolderContsData(folderDirs):
    """ folderDirsにfolderの内容を追加して戻す。
        folderContDataのformat形式で戻す"""
    #folderの種類を取得
    folderConts = []
    for folder in folderDirs:
        folType = pyTreeFoam.getFolderType(folder)
        folderConts.append([folder, folType])
    #caseFolderの内容を設定
    for i in range(len(folderConts)):
        folderCont = folderConts[i]
        if folderCont[1] == "case":
            folder = folderCont[0]
            caseCont = pyTreeFoam.case(folder).getCaseCont()
            folderConts[i] += caseCont
    return folderConts

def createTreeData(folderContDict, rootDir):
    """ folderContDictからtreeDataを作成し戻す"""
    rootName = "root"
    treeData = [rootDir, []]
    folders = []
    folderDirs = list(folderContDict.keys())
    folderDirs.sort()
    for folder in folderDirs:
        if folder != rootDir:
            folderDir = rootName + folder[len(rootDir):]
            words = folderDir.split("/")
            folders.append(words)
    for folderWords in folders:
        treeData = setTreeDataFolder(0, treeData, folderWords)
    return treeData

def setTreeDataFolder(nest, tree, folder):
    """ folderをtree上に作成する"""
    nest += 1
    if nest > 50:
        print(_("フォルダのネスティングが深すぎます。"))
        return

    #親folderまでskipする
    parentDir = folder[nest]
    ii = 0
    while ii < len(tree[1]):
        if type(tree[1][ii]) == list:
            if tree[1][ii][0] == parentDir:
                setTreeDataFolder(nest, tree[1][ii], folder)
                break
        else:
            if tree[1][ii] == parentDir:
                tree[1][ii] = [parentDir, []]
                setTreeDataFolder(nest, tree[1][ii], folder)
                break
        ii += 1
    #親folderの位置ならchildを追加する
    if nest == len(folder) - 1:
        child = folder[-1]
        tree[1].append(child)
    return tree


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext
