#!/usr/bin/python3
#  coding: utf-8
#
#       focusJobControlDialog.py
#
#       FOCUSのJob管理用
#       JobFileの選択、作成、投入。slurmコマンドは自動作成される。
#
#   16/10/20    新規作成
#      10/28    「#sbatch」を大文字「#SBATCH」に変更。logファイルが設定されないため。
#               「#SBATCH -J」の場所を変更。jib名をjobFile名と同じ設定に修正。
#      11/06    caseDirを引数から取得する様に修正。
#      12/05    runJob:submit時にjobIdファイル作成を追加
#               setQueueCpu:logFile名をfocus_data内データから設定
#      12/10    showLog、getLogFile追加。tail-fコマンドでlogを表示。
#   17/01/15    plotWatcherを追加
#      02/26    「nCPUs」→「nPs」に修正。
#      03/06    scancel:停止確認のdialogの表示を追加。
#      05/20    sshHeaderをssh実行前に追加
#      05/28    runFocusCommandReturnCont:sshHeader追加時、「" "」の追加をしない様に修正。
#      07/01    setCurrDir:追加（caseDirを表示させる。）
#      07/13    runTerminal:バグ修正（login端末起動時のdir設定を修正）
#      08/05    showLog, plotWatcher:jobを表示させていない場合、currentDirでlogFileを検索
#      08/09    setCurrDir:currDirの表示がおかしくなる事があり、修正。
#   18/06/05    gnome-terminalのオプション「-x」を「--」に変更
#   19/11/07    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#   22/10/07    squeues:コマンドを修正。
#               "squeues"→"env LD_LIBRARY_PATH=/opt/munge/lib64 /home1/share/bin/squeues"
#      07/29    openFileDialog:universalDialogs内の関数名変更による修正
#      07/21    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os, sys
import glob
import shutil
import subprocess
import datetime

import pyTreeFoam
import universalDialogs as unvDlg

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()

serverName = ""
caseDir = ""
jobFile = ""
queue = ""
nCpu = ""
nNode = ""
queueNames = []         #queue名のlist

#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"focusJobControlDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.caseDir = caseDir
        self.serverName = ""            #接続するserver名
        self.queueNames = []            #comboにセットする内容
        #GUIのobject名を取得
        self.setGtkObject()
        #初期化
        self.initialize()
        self.setComboBox()              #comboにデータセット
        #self.hideSbatchOption()         #sbatchOptionを隠す
        self.setSqueueLabel()           #squeue実行時のtitle行を設定
        self.setCurrDir()               #currDirをlabelに設定
        self.iniTreeView()              #treeView（実行状況のlist）初期化

    def setGtkObject(self):
        """ glade内のobject名を取得する"""
        self.entry_jobFile = self.builder.get_object("entry_jobFile")
        self.entry_nProcs = self.builder.get_object("entry_nProcs")
        self.entry_nNodes = self.builder.get_object("entry_nNodes")
        self.entry_otherComm = self.builder.get_object("entry_otherComm")
        self.combo_queue = self.builder.get_object("combo_queue")
        self.text_commList = self.builder.get_object("text_commList")
        self.tree_jobList = self.builder.get_object("tree_jobList")
        self.button_setJobCont = self.builder.get_object("button_setJobCont")
        self.button_editQueueCont = self.builder.get_object("button_editQueueCont")
        self.button_runJob = self.builder.get_object("button_runJob")
        self.label_squeueTitle = self.builder.get_object("label_squeueTitle")
        self.label_currDir = self.builder.get_object("label_currDir")

    #
    #  main
    #  ----
    def main(self):
        """ GUIを表示する"""
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    #-------- event handler ------------------
    #参照...
    def onRefJobFile(self, event):
        self.selectJob()
    #Job名入力
    def onSetJobFile(self, event):
        self.showSbatchOption()
    #Job編集
    def onEditJobFile(self, event):
        self.editJob()
    #設定
    def onSetJobCont(self, event):
        self.setJobCont()
    #queue設定file編集
    def onEditQueueCont(self, event):
        self.editQueueCont()
    #Job投入実行
    def onRunJob(self, event):
        self.runJob()
    #freenodes
    def onFreenodes(self, event):
        self.freenodes()
    #squeues
    def onSqueues(self, event):
        self.squeues()
    #sinfo
    def onSinfo(self, event):
        self.sinfo()
    #enterOtherComm
    def onEnterOtherComm(self, event):
        self.runOtherComm()
    #run
    def onRunOtherComm(self, event):
        self.runOtherComm()
    #squeue
    def onSqueue(self, event):
        self.squeue()
    #tail_f
    def onTail_f(self, event):
        self.showLog()
    #plotWatcher
    def onPlotWatcher(self, event):
        self.plotWatcher()
    #scancel
    def onScancel(self, event):
        self.scancel()
    #folder開く
    def onOpenFolder(self, event):
        self.openFolder()
    #login端末起動
    def onRunLoginTerm(self, event):
        self.runLoginTerm()
    #閉じる
    def onClose(self, event):
        self.close()
    #-------------------------------------------

    #
    #  initialize
    #  ----------
    def initialize(self):
        """ 初期化"""
        global jobFile, queue, nCpu, nNode
        (conts, _logFiles) = self.readFocusData()
        self.queueNames = conts
        self.serverName = pyTreeFoam.servers().getServerName(self.caseDir)
        #optionを隠す
        self.hideSbatchOption()
        #jobFileNameを取得
        fileName = self.caseDir + "/.jobFileName"
        if os.path.exists(fileName):
            #jobFile有無を確認
            f = open(fileName); cont = f.read(); f.close()
            jobFile = cont.split()[0]
            if os.path.exists(self.caseDir + "/" + jobFile):            
                #jobFileをentryに設定
                self.entry_jobFile.set_text(jobFile)
                (queue, nCpu, nNode) = self.readJobFile(jobFile)
                self.setSbatchOption(queue, nCpu, nNode)
                self.showSbatchOption()

    #
    #  setComboBox
    #  -----------
    def setComboBox(self):
        """ comboBoxにqueue名をセットする"""
        self.combo_queue.clear()
        listModel = Gtk.ListStore(str)
        for name in self.queueNames:
            listModel.append([name])
        cell = Gtk.CellRendererText()
        self.combo_queue.pack_start(cell, True)
        self.combo_queue.add_attribute(cell, "text", 0)
        self.combo_queue.set_model(listModel)
        self.combo_queue.set_entry_text_column(0)

    #
    #  hideSbatchOption
    #  ----------------
    def hideSbatchOption(self):
        """ sbatchオプション関連を隠す"""
        self.combo_queue.set_sensitive(False)
        self.entry_nProcs.set_sensitive(False)
        self.entry_nNodes.set_sensitive(False)
        self.button_setJobCont.set_sensitive(False)
        self.button_editQueueCont.set_sensitive(False)
        self.button_runJob.set_sensitive(False)

    #
    #  showSbatchOption
    #  ----------------
    def showSbatchOption(self):
        """ sbatchオプション関連を表示させる"""
        self.combo_queue.set_sensitive(True)
        self.entry_nProcs.set_sensitive(True)
        self.entry_nNodes.set_sensitive(True)
        self.button_setJobCont.set_sensitive(True)
        self.button_editQueueCont.set_sensitive(True)
        self.button_runJob.set_sensitive(True)

    #
    #  setSqueueLabel
    #  --------------
    def setSqueueLabel(self):
        """ squeue実行時のtitle行を設定"""
        titleLine = "   JOBID PARTITION     NAME     USER ST       TIME  NODES NODELIST(REASON)"
        self.label_squeueTitle.set_label(titleLine)

    #
    #  setCurrDir
    #  ----------
    def setCurrDir(self):
        """ currDirをlabelにセットして表示"""
        caseDir = self.caseDir
        if len(caseDir) > 73:
            caseDir = "..." + caseDir[-70:]
        currDirCont = "currDir: " + caseDir
        self.label_currDir.set_label(currDirCont)

    #
    #  iniTreeView
    #  -----------
    def iniTreeView(self):
        """ treeViewを初期化する"""
        #title行を非表示
        self.tree_jobList.set_headers_visible(False)
        #treeColumnの設定
        cell = Gtk.CellRendererText()
        col = Gtk.TreeViewColumn()
        col.pack_start(cell, True)
        col.add_attribute(cell, "text", 0)
        self.tree_jobList.append_column(col)

    #
    #  readJobfile
    def readJobFile(self, jobName):
        """ JobFileを読み込み、queue, nCpu, nNodeを返す。
        定義されていない場合は、「""」を返す。
        """
        fileName = self.caseDir + "/" + jobName
        f = open(fileName); lines = f.readlines(); f.close()
        queue = ""; nCpu = ""; nNode = ""
        for line in lines:
            if line[:len("#SBATCH ")].upper() == "#SBATCH ":
                words = line.split()
                if len(words) >= 3:
                    if words[1] == "-n":
                        nCpu = words[2]
                    elif words[1] == "-N":
                        nNode = words[2]
                    elif words[1] == "-p":
                        queue = words[2]
        return (queue, nCpu, nNode)

    #
    #  setSbatchOption
    def setSbatchOption(self, queue, nCpu, nNode):
        """ sbatchのオプションをtextBoxに設定する。"""
        entry = self.combo_queue.get_child()
        entry.set_text(queue)
        self.entry_nProcs.set_text(nCpu)
        self.entry_nNodes.set_text(nNode)

    #
    #  selectJob
    #  ---------
    def selectJob(self):
        global jobFile, queue, nCpu, nNode
        title = _("Jobファイルの選択")
        #mess = _("編集するJobファイルを選択する。")
        fileName = self.entry_jobFile.get_text()
        okArgs = [self.selectJob_run]
        self.openFileDialog(title, self.caseDir, funcOk=okArgs, selName=fileName)

    #  selectJob_run
    def selectJob_run(self, fileName):
        global jobFile, queue, nCpu, nNode
        """ jobFileを選択する"""
        jobFile = fileName.split("/")[-1]
        self.entry_jobFile.set_text(jobFile)
        (queue, nCpu, nNode) = self.readJobFile(jobFile)
        self.setSbatchOption(queue, nCpu, nNode)
        self.showSbatchOption()
        #jobFileNameを保存
        jobFileName = self.caseDir + "/.jobFileName"
        f = open(jobFileName, "w"); f.write(jobFile); f.close()

    #
    #  editJob
    #  -------
    def editJob(self):
        """ JobFileの編集"""
        jobName = self.entry_jobFile.get_text()
        if jobName == "":
            title = _("エラー")
            mess = _("job名が入力されていません。") + "\n"
            mess += _("  job名を入力してください。")
            self.errDialog(title, mess)
            return
        self.showSbatchOption()
        fileName = self.caseDir + "/" + jobName
        pyTreeFoam.run(self.caseDir).editor([fileName])

    #
    #  setJobCont
    #  ----------
    def setJobCont(self):
        """ sbatchのoptionを書き込む"""
        jobFile = self.entry_jobFile.get_text()
        job = jobFile
        queue = self.combo_queue.get_child().get_text()
        nCpu = self.entry_nProcs.get_text()
        nNode = self.entry_nNodes.get_text()
        (oFile, eFile) = self.getFocusOutputLogFileNames(job)
        error = self.writeJobFile(jobFile, job, queue, nCpu, nNode, oFile, eFile)
        if error != "":
            title = _("エラー")
            self.errDialog(title, error)
            return
        title = _("Jobファイルの修正")
        mess = _("以下の内容を書き込みました。\n")
        mess += ("  #SBATCH -p " + queue + "\n")
        mess += ("  #SBATCH -n " + nCpu + "\n")
        if nNode != "":
            mess += ("  #SBATCH -N " + nNode + "\n")
        mess += ("  #SBATCH -J " + job + "\n")
        mess += ("  #SBATCH -e " + eFile + "\n")
        mess += ("  #SBATCH -o " + oFile + "\n")
        self.okDialog(title, mess)

    #
    #  editQueueCont
    #  -------------
    def editQueueCont(self):
        """ focusの設定file編集"""
        fileName = os.getenv("TreeFoamUserPath") + "/data/focus_data"
        pyTreeFoam.run(self.caseDir).editorWait([fileName])
        (data, _logFiles) = self.readFocusData()
        self.queueNames = data
        selQueue = self.combo_queue.get_child().get_text()
        self.setComboBox()
        if selQueue in self.queueNames:
            idNo = self.queueNames.index(selQueue)
        else:
            idNo = 0
        self.combo_queue.set_active(idNo)

    #
    #  runJob
    #  ------
    def runJob(self):
        """ job投入・実行"""
        jobFile = self.entry_jobFile.get_text()
        (queue, nCpu, nNode) = self.readJobFile(jobFile)
        error = ""
        if queue == "":
            error = _("キューが設定されていません。")
        elif nCpu == "":
            error = _("nPs（プロセス数）が設定されていません。")
        if error != "":
            title = _("エラー")
            self.errDialog(title, error)
            return

        title = _("Jobの投入")
        mess = _("以下の条件でJobを投入します。\n")
        mess += _("  キュー名:    ") + queue + "\n"
        mess += _("  processor数: ") + nCpu + "\n"
        if nNode != "":
            mess += _("  node数:      ") + nNode + "\n"
        funcOk = [self.runJob_run, jobFile]
        self.okCancelDialog(title, mess, funcOk=funcOk)
        
    #  runJob_run
    def runJob_run(self, jobFile):
        """ jobを投入する"""
        #今回の結果を書き込む
        cont = self.runAddThisJobToJobListCont(jobFile)
        #jobFileNameを保存
        fileName = self.caseDir + "/.jobFileName"
        f = open(fileName, "w"); f.write(jobFile); f.close()
        #実行の状況を表示
        title = _("Jobの投入")
        mess = _("Jobを投入しました。") + "\n\n"
        mess += cont
        self.okDialog(title, mess)

    #
    #  runAddThisJobToJobListCont
    def runAddThisJobToJobListCont(self, jobFile):
        """ jobを投入し、その結果を返す。
        jobListContに今回のjobを追加し、古いjobは削除する。"""
        script = os.getenv("TreeFoamPath") + "/python/focus_addJobList.py"
        server = pyTreeFoam.server(self.serverName)
        hostDir = server.sshfsDict["hostDir"]
        mountDir = server.sshfsDict["mountPoint"]
        jobDir = hostDir + self.caseDir[len(mountDir):]
        args = jobDir + " " + jobFile
        #スクリプトを転送し、実行
        (_stat, res, err) = server.moveRunPythonReturnCont(script, args)
        cont = res + err
        return cont

    #
    #  getJobListCont
    def getJobListCont(self, server):
        """ ~/.jobListContの内容を取得"""
        mountDir = server.sshfsDict["mountPoint"]
        fileName = mountDir + "/.jobList"
        if len(glob.glob(fileName)) == 0:
            return ""
        else:
            f = open(fileName); jobListCont = f.read(); f.close()
            return jobListCont

    #
    #  freenodes
    #  ---------
    def freenodes(self):
        """ freenodesコマンド実行"""
        comm = "freenodes"
        cont = self.runFocusCommandReturnCont("~", comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  squeues
    #  -------
    def squeues(self):
        """ squeuesコマンド実行"""
        #comm = "squeues"
        comm = "env LD_LIBRARY_PATH=/opt/munge/lib64 /home1/share/bin/squeues"
        cont = self.runFocusCommandReturnCont("~", comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  sinfo
    #  -----
    def sinfo(self):
        """ sinfoコマンド実行"""
        comm = "sinfo"
        cont = self.runFocusCommandReturnCont("~", comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  runOtherComm
    #  ------------
    def runOtherComm(self):
        """ その他コマンド"""
        server = pyTreeFoam.server(self.serverName)
        hostDir = server.sshfsDict["hostDir"]
        mountPoint = server.sshfsDict["mountPoint"]
        workDir = hostDir + self.caseDir[len(mountPoint):]
        comm = self.entry_otherComm.get_text()
        cont = self.runFocusCommandReturnCont(workDir, comm)
        buffer = self.text_commList.get_buffer()
        buffer.set_text(cont)

    #
    #  squeue
    #  ------
    def squeue(self):
        """ squeueコマンドの実行（実行状況のリスト）"""
        cont = self.getClearJobList()
        lines = cont.split("\n")[:-1]
        items = []
        for line in lines[1:]:
            i = 0
            while i <= 10:
                chara = line[i]
                if chara == " ":
                    i += 1
                else:
                    break
            items.append(line[i:])
        #treeに値をセット
        listStore = Gtk.ListStore(str)
        for name in items:
            listStore.append([name])
        self.tree_jobList.set_model(listStore)

    #
    #  getClearJobList
    def getClearJobList(self):
        """ jobListを取得し現状に合わせる"""
        script = os.getenv("TreeFoamPath") + "/python/focus_clearJobList.py"
        server = pyTreeFoam.server(self.serverName)
        args = ""
        #スクリプトを転送し、実行
        (_stat, cont, _err) = server.moveRunPythonReturnCont(script, args)
        return cont

    #
    #  showLog
    #  -------
    def showLog(self):
        """ 選択されたJobのlogを表示（tail-f）"""
        #選択しているjobを取得
        listSelection = self.tree_jobList.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        if len(pathes) == 0:
            #itemを選択していない場合
            if listStore == None:
                n = 0
            else:
                n = len(listStore)
            if n == 0:
                #listBox内が空の場合
                logFile = self.caseDir + "/solve.log"
                title = _("log表示")
                mess = _("currentDirectoryのlogを表示します。")
                funcOk = [self.showLog_run, logFile]
                self.okCancelDialog(title, mess, funcOk=funcOk)
                return
            else:
                #listBox内にjobが表示されている状態でjobを選択していない場合
                title = _("エラー")
                mess = _("jobが選択されていません。")
                self.errDialog(title, mess)
                return
        else:
            iter = listStore.get_iter(pathes[0])
            item = listStore.get_value(iter, 0)
            jobId = item.split()[0]
            serverLogFile = self.getLogFileFromjobId(jobId)
            server = pyTreeFoam.server(self.serverName)
            hostDir = server.sshfsDict["hostDir"]
            mountDir = server.sshfsDict["mountPoint"]
            logFile = mountDir + serverLogFile[len(hostDir):]
            #log出力
            self.showlog_run(logFile)

    #  showLog_run
    def showLog_run(self, logFile):
        """ log出力を開始する"""
        comm = "tail -n 100 -f " + logFile
        fileName = os.getenv("TreeFoamUserPath") + "/temp/runBash"
        f = open(fileName, "w"); f.write(comm); f.close()
        #pyTreeFoam.run().foamTerminal(fileName)
        commLine = pyTreeFoam.foamTerminalRun + " " + fileName
        pyTreeFoam.run().command(commLine)

    #
    #  plotWatcher
    #  -----------
    def plotWatcher(self):
        """ plotWatcherの起動"""
        #logFileを取得する
        server = pyTreeFoam.server(self.serverName)
        #選択しているjobを取得
        listSelection = self.tree_jobList.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        if len(pathes) == 0:
            #itemを選択していない場合
            if listStore == None:
                n = 0
            else:
                n = len(listStore)
            if n == 0:
                #listBox内が空の場合
                #currentDir内でlogFileを取得
                logFileLocal = self.caseDir + "/solve.log"
                #server側のlogFileに変換
                mountDir = server.sshfsDict["mountPoint"]
                hostDir = server.sshfsDict["hostDir"]
                logFile = hostDir + logFileLocal[len(mountDir):]
                #okCancelDialog表示
                title = _("log表示")
                mess = _("currentDirectoryのlogを表示します。")
                okArgs = [self.plotWatcher_run, server, logFile]
                self.okCancelDialog(title, mess, funcOk=okArgs)
                return
            else:
                #listBox内にjobが表示されている状態でjobを選択していない場合
                title = _("エラー")
                mess = _("jobが選択されていません。")
                self.errDialog(title, mess)
                return
        else:
            iter = listStore.get_iter(pathes[0])
            item = listStore.get_value(iter, 0)
            jobId = item.split()[0]
            logFile = self.getLogFileFromjobId(jobId)
            self.plotWatcher_run(server, logFile)

    #  plotWatcher_run
    def plotWatcher_run(self, server, logFile):
        """ logFileを使って、plotWatcherを起動する"""
        #  plotWatcherは、local側で起動する
        configDict = pyTreeFoam.readConfigTreeFoam()
        envOpenFoam = configDict["bashrcFOAM"]
        #  logFileをlocal側に転送
        userName = server.sshDict["user"]
        hostName = server.sshfsDict["hostName"]
        comm = "scp " + userName + "@" + hostName + ":" + logFile + " " + os.getenv("TreeFoamUserPath") + "/temp"
        pyTreeFoam.run().command(comm)
        #起動スクリプト作成
        if os.getenv("venvPython") == "yes":
            pythonName = os.getenv("venvPythonName")
        else:
            pythonName = "python3"
        localLogFile = os.getenv("TreeFoamUserPath") + "/temp/solve.log"
        cont = "#!/bin/bash\n"
        cont += ". " + envOpenFoam + "\n"
        cont += "echo \n"
        cont += 'echo -n "running plotWatcher\n"\n'
        cont += pythonName + " $pyFoamDir/bin/pyFoamPlotWatcher.py " + localLogFile + "\n"
        #実行fileを作成
        fileName = os.getenv("TreeFoamUserPath") + "/temp/runBash"
        f = open(fileName, "w"); f.write(cont); f.close()
        run = pyTreeFoam.run(os.getenv("TreeFoamUserPath")+"/temp")
        run.command("chmod a+x " + fileName)
        comm = pyTreeFoam.TerminalRun + " " + fileName
        run.command(comm)

    #
    #  scancel
    #  -------
    def scancel(self):
        """ 選択Jobをキャンセルする"""
        #選択しているjobを取得
        listSelection = self.tree_jobList.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        if len(pathes) == 0:
            title = _("エラー")
            mess = _("停止させるjobが選択されていません。")
            self.errDialog(title, mess)
            return
        #jobIdを取得
        iter = listStore.get_iter(pathes[0])
        item = listStore.get_value(iter, 0)
        jobId = item.split()[0]
        #確認message
        title = _("jobの停止")
        mess = _("選択しているJobを停止させますか？")
        okArgs = [self.scancel_run, jobId]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return
    
    #  scancel_run
    def scancel_run(self, jobId):
        """ jobをキャンセルする。"""
        comm = "scancel " + jobId
        _cont = self.runFocusCommandReturnCont("~", comm)
        title = _("jobの停止")
        mess = _("以下のjobIdに対し、停止コマンド（scancel）を送出しました。") + "\n\n"
        mess += "jobId: " + jobId
        self.okDialog(title, mess)

    #
    #  getLogFileFromjobId
    def getLogFileFromjobId(self, jobId):
        """ jobIdからlogfile(serverのpath）を取得する"""
        server = pyTreeFoam.server(self.serverName)
        jobList = self.getJobListCont(server)
        jobLines = jobList.split("\n")
        jobDir = ""
        for line in jobLines:
            words = line.split()
            if len(words) > 0:
                if jobId == words[0]:
                    jobDir = words[2]
                    break
        logFile = jobDir + "/solve.log"
        return logFile

    #
    #  openFolder
    #  ----------
    def openFolder(self):
        """ folderを開く"""
        pyTreeFoam.run(self.caseDir).fileManager()

    #
    #  runLoginTerm
    #  ------------
    def runLoginTerm(self):
        """ login端末を開く"""
        server = pyTreeFoam.server(self.serverName)
        hostDir = server.sshfsDict["hostDir"]
        mountDir = server.sshfsDict["mountPoint"]
        setEnviron = server.sshfsDict["setEnviron"]
        #loginDir, setEnvironを作成
        if mountDir == self.caseDir[:len(mountDir)]:
            openDir = hostDir + self.caseDir[len(mountDir):]
            lines = setEnviron.split("\n")
            i = len(lines) - 1
            while i >= 0:
                if lines[i][:len("cd")] == "cd":
                    #words = lines[i].split()
                    lines[i] = "cd " + openDir
                    break
                i -= 1
            setEnviron = "\n".join(lines)
        else:
            setEnviron = ""
        #tempFile作成
        name = "setEnviron"
        fileName = os.getenv("TreeFoamUserPath") + "/temp/" + name
        f=open(fileName, "w"); f.write(setEnviron); f.close()
        #environを転送
        server.scpCommandNoTerm(fileName, "~")
        #login端末を起動 
        server.runLoginTerm()

    #
    #  writeJobFile
    def writeJobFile(self, jobFile, job, queue, nCpu, nNode, oFile, eFile):
        """ jobFileに書き込む"""
        error = ""
        #errorチェック
        if job == "":
            error = _("Jobファイル名が設定されていません。")
            return error
        if queue == "" and nCpu == "" and nNode == "":
            error = _("queue, nCpu, nNodeが入力されていません。")
            return error

        jobFile = self.caseDir + "/" + jobFile
        #ファイルが存在しない場合、作成する
        if len(glob.glob(jobFile)) == 0:
            f=open(jobFile, "w"); f.write("#!/bin/bash\n"); f.close()
        #ファイルを読み込む
        f = open(jobFile); lines = f.readlines(); f.close()
        #関係する行を削除
        newLines = []
        for line in lines:
            if line[:len("#SBATCH ")].upper() == "#SBATCH ":
                words = line.split()
                if not (words[1] == "-J" or words[1] == "-n" or words[1] == "-N" 
                    or words[1] == "-p" or words[1] == "-e" or words[1] == "-o"):
                    newLines.append(line)
            else:
                newLines.append(line)
        #追加する行を作成
        addLines = ["#SBATCH -p " + queue + "\n"]
        addLines += ["#SBATCH -n " + nCpu + "\n"]
        if nNode != "":
            addLines += ["#SBATCH -N " + nNode + "\n"]
        addLines += ["#SBATCH -J " + job + "\n"]
        addLines += ["#SBATCH -e " + eFile + "\n"]
        addLines += ["#SBATCH -o " + oFile + "\n"]
        #行を追加
        newLines = newLines[:1] + addLines + newLines[1:]
        #書き込み
        f = open(jobFile, "w")
        for line in newLines:
            f.write(line)
        f.close()
        return error

    #
    #  getFocusOutputLogFileNames
    def getFocusOutputLogFileNames(self, job):
        """ logFile名を取得"""
        (_data, logFiles) = self.readFocusData()
        (logFile, errLogFile) = logFiles
        logFile = self.getStringNameFromEnv(logFile, job)
        errLogFile = self.getStringNameFromEnv(errLogFile, job)
        return (logFile, errLogFile)

    #
    #  getStringNameFromEnv
    def getStringNameFromEnv(self, name, job):
        """ 環境変数(name)を文字列に変換する"""

        def getEnvName(name, j):
            if name[j] == "{":
                n = name[j:].find("}")
                if n >= 0:
                    jj = j + n
                    strName = name[j+1:jj]
                else:
                    strName = name[j+1:]
                    jj = len(name)
            else:
                strName = name[j:]
                jj = len(name)
            return (strName, jj)

        newName = ""
        i = 0
        while i < len(name):
            chara = name[i]
            if chara == "$":
                (envName, i) = getEnvName(name, i+1)
                if envName == "jobName":
                    strName = job
                else:
                    strName = os.getenv(envName)
                    if strName == None:
                        strName = ""
                newName += strName
            else:
                newName += chara
            i += 1
        return newName



    #
    #  readFocusData
    def readFocusData(self):
        """ FOCUSのデータを読み込む"""
        fileName = os.getenv("TreeFoamUserPath") + "/data/focus_data"
        if len(glob.glob(fileName)) == 0:
            orgName = os.getenv("TreeFoamPath") + "/data/focus_data"
            shutil.copy(orgName, os.getenv("TreeFoamUserPath") + "/data")
        f = open(fileName); lines = f.readlines(); f.close()
        lines = self.deleteComment(lines)
        flag = 0; logFlag = 0; eLogFlag = 0
        queueNames = []; logFile = "solve.log"; errLogFile = "${jobName}.e%J"
        for line in lines:
            if line[0] != " ":
                words = line.split()
                if len(words) > 0:
                    if words[0] == "queueNames":
                        flag = 1
                    elif words[0] == "logFileName":
                        flag = 0
                        if len(words) >= 2:
                            logFile = words[1]
                            logFlag = 1
                    elif words[0] == "errLogFileName":
                        flag = 0
                        if len(words) >= 2:
                            errLogFile = words[1]
                            eLogFlag = 1
                    else:
                        flag = 0
            else:
                if flag == 1:
                    words = line.split()
                    if len(words) > 0:
                        queueNames.append(words[0])
        queueNames.sort()
        logFiles = [logFile, errLogFile]
        if logFlag == 1 and eLogFlag == 1:
            return (queueNames, logFiles)

        #fileを読み込み再作成する
        f = open(fileName); lines = f.readlines(); f.close()
        if logFlag == 0 and eLogFlag == 0:
            #logFile errLogFile共、再作成
            addLines = ["#  Log file names\n"]
            addLines += ["#    These names are able to change at following lines.\n"]
            addLines += ["logFileName     " + logFile + "\n"]
            addLines += ["errLogFileName  " + errLogFile + "\n"]
            lines += addLines
        elif logFlag == 0:
            #logFileを追加
            addLines = ["logFileName     " + logFile + "\n"]
            for i in range(len(lines)):
                line = lines[i]
                if line[:len("errLogFileName ")] == "errLogFileName ":
                    break
            lines = lines[:i] + addLines + lines[i:]
        elif eLogFlag == 0:
            #errLogFileを追加
            addLines = ["errLogFileName  " + errLogFile + "\n"]
            for i in range(len(lines)):
                line = lines[i]
                if line[:len("logFileName ")] == "logFileName ":
                    break
            lines = lines[:i] + addLines + lines[i:]
        #書き込み
        f = open(fileName, "w")
        for line in lines:
            f.write(line)
        f.close()
        #戻る
        return (queueNames, logFiles)

    #
    #  runFocusCommandReturnCont
    def runFocusCommandReturnCont(self, workDir, comm):
        """ FOCUSコマンドを実行し、その結果を返す。"""
        server = pyTreeFoam.server(self.serverName)
        hostName = server.sshfsDict["hostName"]
        sshHeader = server.sshfsDict["sshHeader"]
        userName = server.sshDict["user"]
        #コマンド作成
        cont = "cd " + workDir + "; "
        cont += comm
        #コマンド実行
        commLine = "ssh " + userName + "@" + hostName + " '" + cont + "'" 
        print(commLine)
        commLine = sshHeader + commLine
        result = self.runShellCommand(commLine)
        return result

    #
    #  runShellCommand
    def runShellCommand(self, comm):
        result = ""
        (_stat, res, err) = pyTreeFoam.run().commandReturnCont(comm)
        result = res + err
        return result

    #
    #  deleteComment
    def deleteComment(self, lines):
        """ コメント文を削除"""
        newLines = []
        for line in lines:
            n = line.find("#")
            if n >= 0:
                newLine = line[:n] + "\n"
            else:
                newLine= line
            newLines.append(newLine)
        return newLines

    #--- universalDialogs -----------------------------
    #  okDialog
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    #  errDialog
    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    #  okCancelDialog
    def okCancelDialog(self, title, mess, 
                       funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, 
            funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()

    #openFilaDialog
    def openFileDialog(self, title, path, 
                       funcOk=[], funcCancel=[], 
                       selName="", pattern=[]):
        dialog = unvDlg.openFileDialog(
            title, path, 
            funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow, selName=selName, 
            filterSets=pattern)
        dialog.show()


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()
