#!/usr/bin/python3
# coding:utf-8
#
#   fileOperationInRegionDialog.py
#
#       各region内のfile操作
#
#   13/01/06    新規作成
#      01/07    DBの場所（defaultの値）を変更
#      01/09    固体の材料選択を修正
#      01/12    DBの場所（defaultの値）を変更
#      01/30    全region削除ボタンを追加
#      02/21    材料設定をlistBox→listCtrlに変更
#      04/07    バグ修正（材料DB内のfile編集ボタンでエラー発生）
#      04/11    file名変更ボタン追加
#      05/01    csvFileで一括設定を追加
#      05/14    csvFile設定した後、表示が変わらないので、表示も変えるよう変更
#      05/19    setMaterialToRegionを修正（copy→delete後、copy）
#      05/20    solidの物性値設定をOF2.2.0に対応
#      05/21    OFversionチェックをinitialize内に統一
#      05/25    checkMaterialDBDirのチェック方法修正
#      05/29    regionを削除した場合の処理を追加、戻すボタンを追加
#      06/22    全region削除ボタン削除
#      07/14    DBフォルダ内にfileが無かった場合、fileをコピーする。
#      09/01    liveDVDで起動したとき、
#                   setMultiRegionでDBdirが設定できない、ゴミ箱が無いエラーを修正
#      09/20    変更ボタンの設定が削除になっていたのを修正
#      10/04    iniTimeの取得をcurrTimeに変更
#      10/15    国際化のため、修正
#      11/12    iniTime, constant, systemフォルダ直下をtree表示に変更
#      12/19    コピー、戻すボタンをfluid側、solid側各々に設置（昔に戻す）
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/01/05    各regionへのコピー時、無印を全regionへコピーを追加
#      04/07    editorでfileOpen時、binaryのみtempフォルダでOpenする様修正
#      05/01    popupMenuを追加、toolTipを追加
#      05/03    materialFileの読み込みにcodecsで読み込み（unucodeでエラー）
#      06/21    getCopyFileNameFromBox:「戻す」時、「_all_」を廃止
#      10/25    clearBoundaryFieldsを追加（fieldコピー時）
#               textBox内の選択項目を保持
#      11/07    radCsvData:pyTreeFoamから読み込む様に修正
#      11/24    setMaterialToListBox:候補材を全てｾｯﾄする様に修正（同名はｾｯﾄせず）
#      11/29    「配布」「戻す」で項目を選択表示させる
#   15/02/08    region選択時、fileListを再表示させる
#      02/09    ﾎﾟｯﾌﾟｱｯﾌﾟﾒﾆｭｰに「名前からregion名を削除」を追加
#      02/23    showRegionFileList:バグ修正（checkFileNameを追加）
#      03/02    fileコピー時、同名があるとｴﾗｰ発生を修正
#      07/13    clipBoardの場所をTreeFoam/dataに変更
#      07/17    dialogの起動方法修正
#      08/09    setMaterialToSelRegions:新規追加（材料設定ボタンを追加）
#   16/12/31    checkMaterialDBDir:OF-ext用に修正
#   17/09/10    checkMaterialDBDir:OFversion毎の処理を修正
#   19/02/27    copyFiles:file名に「.」が有る場合にコピーできなかったので修正
#      05/22    checkMaterialDBDir,checkFluidSolidFolder:mkdirをos.mkdirに変更
#      12/16    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#      05/27    checkMaterialDBDir:バグ修正。DBdirが無かった時にエラー発生
#      06/02    copyToClipboard, pasteFileFromClipboard:
#               copyPasteFolders.pyに変更。
#      07/30    showPopupMenu:材料設定のtreeWidgetで右クリックすると
#               エラー発生するので、修正。
#   21/06/02    checkMaterialDBDir, getSolidPhysicalDB:OF-8対応で修正。
#               thermophysicalPropertesの内容が変わった為、これに対応。
#               「thermophysicalPropertes_energy」を追加。
#               deleteFilesInTree:ゴミ箱へ移動できなかった場合の処理を追加
#   22/01/25    clearMterialFromRegion,editRegionMaterial,renameRegionMaterial,
#               saveMaterialDB,setMarerialReqDC:thermoPhysicalproperties名を修正
#      07/19    checkMaterialDBDir:OFversionのcheck方法を修正（OF-10対応）
#      07/22    checkMaterialDBDir:OF-10以降から固体の物性値file名が
#               「thermophysicalProperties → physicalProperties」に変更され、対応。
#   23/02/05    initialize:OF-11ではregionPropertiesが存在しない為、
#               regionProperties有無チェックを廃止
#   24/02/13    initialize:流体or固体のregionが定義されていない場合、
#               終了するよう修正。
#      07/29    openFileDialog:universalDialogs内の関数名変更による修正
#      07/30    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/05    inputTextDDialog:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#   25/07/04    glade:全scrollWindowが水平方向の伸びに対して応答する様に修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GdkPixbuf
import os, sys
import glob
import shutil
import threading
import time

import pyTreeFoam
import inputTextDDialog
import getOpenFolderDDialog
import GtkParts
import universalDialogs as unvDlg
import copyPasteFolders

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()


#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir, openPage):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"fileOperationInRegionDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #属性をセット
        self.caseDir = caseDir
        self.iniTime = ""                   #対象の時間folder
        self.openPage = openPage            #起動時にnotebookを表示させるpage
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        self.multiRegion_data = os.getenv("TreeFoamUserPath") + "/data/multiRegion_data"
        self.iniDBdata = ""                 #defaultのmatDBファイル名（fullPath）
        self.solidPhysicalDBdir = ""        #matDBの場所
        self.solidProperties = ""           #matDB名
        self.solidPhysicalDB = []           #matDBの内容
        self.constantSolidPhysical = []     #constantの定義内容
        self.timeFiles = []                 #timePageのデータ
        self.timeFluidFiles = []
        self.timeSolidFiles = []
        self.constantFiles = []             #constantPageのデータ
        self.constantFluidFiles = []
        self.constantSolidFiles = []
        self.systemFiles = []               #systemPageのデータ
        self.systemFluidFiles = []
        self.systemSolidFiles = []
        self.ctrlKeyFlag = 0                #ctrlKeyのflag
        iconDir = os.getenv("TreeFoamPath") + "/icons"
        iconFile = GdkPixbuf.Pixbuf.new_from_file
        self.iconFolder = iconFile(iconDir+"/folder16.png")
        self.iconFile = iconFile(iconDir+"/text8.png")
        self.iconFolderLink = iconFile(iconDir+"/folderLink16.png")
        self.iconFileLink = iconFile(iconDir+"/textLink8.png")
        self.clipboardFile = os.getenv("TreeFoamUserPath") + "/data/clipBoard"
        self.pasteContFile = os.getenv("TreeFoamUserPath") + "/data/pasteNames"

        #GUIのobject名を取得
        self.setGtkObject()
        #matDBの場所を確認
        self.checkMaterialDBDir()
        #TreeListを初期化
        self.iniTreeList()
        #対象のtimeFolderを取得、設定
        self.getTimeFolderForOperation()
        #popupMenuを設定
        self.setPopupMenu()
        #tooltipを設定
        self.setTooltipText()
        #固体の材料設定のtitle設定
        self.setTitleForSolid()

    #
    #  main
    #--------
    def main(self):
        """ GUIの表示"""
        #初期化
        self.initialize()
        #dataをlistBoxにセット
        self.setFileNameMaterialToListBox()
        #最初のregionを選択する
        self.selectFirstRegion()
        #setPage
        page = int(self.openPage)
        if 0 <= page and page <= 3:
            self.noteBook.set_current_page(page)
        else:
            self.noteBook.set_current_page(1)
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    def setGtkObject(self):
        """ GUIのobject名を取得する"""
        self.noteBook = self.builder.get_object("noteBook")
        self.tree_time = self.builder.get_object("tree_time")
        self.tree_constant = self.builder.get_object("tree_const")
        self.tree_system= self.builder.get_object("tree_sys")
        self.tree_timeFluidFile = self.builder.get_object("tree_timeFluidFile")
        self.tree_constantFluidFile = self.builder.get_object("tree_constFluidFile")
        self.tree_systemFluidFile = self.builder.get_object("tree_sysFluidFile")
        self.tree_timeFluidRegion = self.builder.get_object("tree_timeFluidRegion")
        self.tree_constantFluidRegion = self.builder.get_object("tree_constFluidRegion")
        self.tree_systemFluidRegion = self.builder.get_object("tree_sysFluidRegion")
        self.tree_timeSolidFile = self.builder.get_object("tree_timeSolidFile")
        self.tree_constantSolidFile = self.builder.get_object("tree_constSolidFile")
        self.tree_systemSolidFile = self.builder.get_object("tree_sysSolidFile")
        self.tree_timeSolidRegion = self.builder.get_object("tree_timeSolidRegion")
        self.tree_constantSolidRegion = self.builder.get_object("tree_constSolidRegion")
        self.tree_systemSolidRegion = self.builder.get_object("tree_sysSolidRegion")
        self.tree_setMaterial = self.builder.get_object("tree_setMaterial")
        self.tree_matDB = self.builder.get_object("tree_matDB")
        self.entry_matDB = self.builder.get_object("entry_matDB")
        self.label_solidTitle = self.builder.get_object("label_solidTitle")


    #------ event handler -----------
    #--------- 開く
    def onEditTime(self, event):
        self.editTimeTreeFiles()
    def onEditTimeFluid(self, event):
        self.editTimeFluidFiles()
    def onEditTimeSolid(self, event):
        self.editTimeSolidFiles()
    def onEditConstant(self, event):
        self.editConstantTreeFiles()
    def onEditConstantFluid(self, event):
        self.editConstantFluidFiles()
    def onEditConstantSolid(self, event):
        self.editConstantSolidFiles()
    def onEditSystem(self, event):
        self.editSystemTreeFiles()
    def onEditSystemFluid(self, event):
        self.editSystemFluidFiles()
    def onEditSystemSolid(self, event):
        self.editSystemSolidFiles()
    #--------- 変更（rename）
    def onRenameTime(self, event):
        self.renameTimeFile()
    def onRenameTimeFluid(self, event):
        self.renameTimeFluidFile()
    def onRenameTimeSolid(self, event):
        self.renameTimeSolidFile()
    def onRenameConstant(self, event):
        self.renameConstantFile()
    def onRenameConstantFluid(self, event):
        self.renameConstantFluidFile()
    def onRenameConstantSolid(self, event):
        self.renameConstantSolidFile()
    def onRenameSystem(self, event):
        self.renameSystemFile()
    def onRenameSystemFluid(self, event):
        self.renameSystemFluidFile()
    def onRenameSystemSolid(self, event):
        self.renameSystemSolidFile()
    #--------- 削除
    def onDeleteTime(self, event):
        self.deleteTimeFiles()
    def onDeleteTimeFluid(self, event):
        self.deleteTimeFluidFiles()
    def onDeleteTimeSolid(self, event):
        self.deleteTimeSolidFiles()
    def onDeleteConstant(self, event):
        self.deleteConstantFiles()
    def onDeleteConstantFluid(self, event):
        self.deleteConstantFluidFiles()
    def onDeleteConstantSolid(self, event):
        self.deleteConstantSolidFiles()
    def onDeleteSystem(self, event):
        self.deleteSystemFiles()
    def onDeleteSystemFluid(self, event):
        self.deleteSystemFluidFiles()
    def onDeleteSystemSolid(self, event):
        self.deleteSystemSolidFiles()
    #--------- 配布>>
    def onCopyFluidTime(self, event):
        self.distFilesTimeFluid()
    def onCopyFluidConstant(self, event):
        self.distFilesConstantFluid()
    def onCopyFluidSystem(self, event):
        self.distFilesSystemFluid()
    def onCopySolidTime(self, event):
        self.distFilesTimeSolid()
    def onCopySolidConstant(self, event):
        self.distFilesConstantSolid()
    def onCopySolidSystem(self, event):
        self.distFilesSystemSolid()
    #--------- <<戻す
    def onReturnFluidTime(self, event):
        self.returnFilesTimeFluid()
    def onReturnFluidConstant(self, event):
        self.returnFilesConstantFluid()
    def onReturnFluidSystem(self, event):
        self.returnFilesSystemFluid()
    def onReturnSolidTime(self, event):
        self.returnFilesTimeSolid()
    def onReturnSolidConstant(self, event):
        self.returnFilesConstantSolid()
    def onReturnSolidSystem(self, event):
        self.returnFilesSystemSolid()
    #--------regin選択
    def onChangeRegionTimeFluid(self, event):
        self.changeRegionTimeFluid()
    def onChangeRegionTimeSolid(self, event):
        self.changeRegionTimeSolid()
    def onChangeRegionConstantFluid(self, event):
        self.changeRegionConstantFluid()
    def onChangeRegionConstantSolid(self, event):
        self.changeRegionConstantSolid()
    def onChangeRegionSystemFluid(self, event):
        self.changeRegionSystemFluid()
    def onChangeRegionSystemSolid(self, event):
        self.changeRegionSystemSolid()
    #--------固体の材料設定
    def onRefMatDB(self, event):
        self.referenceMatDB()
    def onSetMaterialToListBox(self, event):
        self.setMaterialNameToListBox()
    def onSetCsvFile(self, event):
        self.setCsvFile()
    def onOutputCsvFile(self, event):
        self.outputCsvFile()
    def onEditRegionMaterial(self, event):
        self.editRegionMaterial()
    def onRenameRegionMaterial(self, event):
        self.renameRegionMaterial()
    def onSaveMaterialDB(self, event):
        self.saveMaterialDB()
    def onClearMaterialFromRegion(self, event):
        self.clearMterialFromRegion()
    def onEditMaterialDB(self, event):
        self.editMaterialDB()
    def onOpenMaterialDBfolder(self, event):
        self.openMaterialDBfolder()
    #--------欄外
    def onClose(self, event):
        self.close()
    #------- mouse関連 
    def onPressMouseButton(self, widget, event):
        self.pressMouseButton(widget, event)
    def onTooltipMatDB(self, treeView, event):
        self.showTooltipMatDB(treeView, event)
    def onTooltipSetMaterial(self, treeView, event):
        self.showTooltipSetMaterial(treeView, event)
    #------- key event ----------------
    def onKeyPress(self, widget, event):
        self.keyPress(widget, event)
    def onKeyRelease(self, widget, event):
        self.keyRelease(widget, event)
    #----- popup event -----------------
    #-- timeTreeView
    def popTimeEdit(self, menuItem):
        self.editTimeTreeFiles()
    def popTimeRename(self, menuItem):
        self.renameTimeFile()
    def popTimeDelRegion(self, menuItem):
        self.deleteRegionNameTime()
    def popTimeDelete(self, menuItem):
        self.deleteTimeFiles()
    def popTimeCopy(self, menuItem):
        self.copyTimeFiles()
    def popTimePaste(self, menuItem):
        self.pasteTimeFiles()
    #-- constantTreeView
    def popConstEdit(self, menuItem):
        self.editConstantTreeFiles()
    def popConstRename(self, menuItem):
        self.renameConstantFile()
    def popConstDelRegion(self, menuItem):
        self.deleteRegionNameConstant()
    def popConstDelete(self, menuItem):
        self.deleteConstantFiles()
    def popConstCopy(self, menuItem):
        self.copyConstantFiles()
    def popConstPaste(self, menuItem):
        self.pasteConstantFiles()
    #-- systemTreeView
    def popSystemEdit(self, menuItem):
        self.editSystemTreeFiles()
    def popSystemRename(self, menuItem):
        self.renameSystemFile()
    def popSystemDelRegion(self, menuItem):
        self.deleteRegionNameSystem()
    def popSystemDelete(self, menuItem):
        self.deleteSystemFiles()
    def popSystemCopy(self, menuItem):
        self.copySystemFiles()
    def popSystemPaste(self, menuItem):
        self.pasteSystemFiles()
    #--timeFluidSolid
    def popTimeFluidEdit(self, menuItem):
        self.editTimeFluidFiles()
    def popTimeFluidRename(self, menuItem):
        self.renameTimeFluidFile()
    def popTimeFluidDelete(self, menuItem):
        self.deleteTimeFluidFiles()
    def popTimeFluidCopy(self, menuItem):
        self.copyTimeFluidFiles()
    def popTimeFluidPaste(self, menuItem):
        self.pasteTimeFluidFiles()
    def popTimeSolidEdit(self, menuItem):
        self.editTimeSolidFiles()
    def popTimeSolidRename(self, menuItem):
        self.renameTimeSolidFile()
    def popTimeSolidDelete(self, menuItem):
        self.deleteTimeSolidFiles()
    def popTimeSolidCopy(self, menuItem):
        self.copyTimeSolidFiles()
    def popTimeSolidPaste(self, menuItem):
        self.pasteTimeSolidFiles()
    #-- constantFluidSolid
    def popConstFluidEdit(self, menuItem):
        self.editConstantFluidFiles()
    def popConstFluidRename(self, menuItem):
        self.renameConstantFluidFile()
    def popConstFluidDelete(self, menuItem):
        self.deleteConstantFluidFiles()
    def popConstFluidCopy(self, menuItem):
        self.copyConstantFluidFiles()
    def popConstFluidPaste(self, menuItem):
        self.pasteConstantFluidFiles()
    def popConstSolidEdit(self, menuItem):
        self.editConstantSolidFiles()
    def popConstSolidRename(self, menuItem):
        self.renameConstantSolidFile()
    def popConstSolidDelete(self, menuItem):
        self.deleteConstantSolidFiles()
    def popConstSolidCopy(self, menuItem):
        self.copyConstantSolidFiles()
    def popConstSolidPaste(self, menuItem):
        self.pasteConstantSolidFiles()
    #-- systemFluidSolid
    def popSystemFluidEdit(self, menuItem):
        self.editSystemFluidFiles()
    def popSystemFluidRename(self, menuItem):
        self.renameSystemFluidFile()
    def popSystemFluidDelete(self, menuItem):
        self.deleteSystemFluidFiles()
    def popSystemFluidCopy(self, menuItem):
        self.copySystemFluidFiles()
    def popSystemFluidPaste(self, menuItem):
        self.pasteSystemFluidFiles()
    def popSystemSolidEdit(self, menuItem):
        self.editSystemSolidFiles()
    def popSystemSolidRename(self, menuItem):
        self.renameSystemSolidFile()
    def popSystemSolidDelete(self, menuItem):
        self.deleteSystemSolidFiles()
    def popSystemSolidCopy(self, menuItem):
        self.copySystemSolidFiles()
    def popSystemSolidPaste(self, menuItem):
        self.pasteSystemSolidFiles()
    #----- double click
    def onDCtime(self, pos):
        self.editTimeTreeFiles()
    def onDCtimeFluid(self, pos):
        self.editTimeFluidFiles()
    def onDCtimeSolid(self, pos):
        self.editTimeSolidFiles()
    def onDCconst(self, pos):
        self.editConstantTreeFiles()
    def onDCconstFluid(self, pos):
        self.editConstantFluidFiles()
    def onDCconstSolid(self, pos):
        self.editConstantSolidFiles()
    def onDCsystem(self, pos):
        self.editSystemTreeFiles()
    def onDCsystemFluid(self, pos):
        self.editSystemFluidFiles()
    def onDCsystemSolid(self, pos):
        self.editSystemSolidFiles()
    def onDCmaterial(self, pos):
        self.setMarerialReqDC(pos)
    def onDCmatDB(self, pos):
        self.editMaterialDB()


    #----- mouse event 関連 -----------------
    #
    #  pressMouseButton
    #-------------------
    def pressMouseButton(self, widget, event):
        """ マウスボタンのevent"""
        pos = (event.x, event.y)
        #rightClick?
        if event.button == 3:
            #rightClickの場合
            #  rightClickの場所を取得
            pathinfo = widget.get_path_at_pos(event.x, event.y)
            if pathinfo == None:
                return
            path, _col, _x, _y = pathinfo
            #ctrlKeyのチェック
            treeSelection = widget.get_selection()
            if self.ctrlKeyFlag == 1:
                #選択が反転する（非選択）ので、0.1s後に再度選択する
                thread = threading.Thread(target=self.afterSelectRow, args=(widget, [path]))
                thread.start()
            else:
                #ctrKeyを押していない場合
                _treestore, selPathes = treeSelection.get_selected_rows()
                if len(selPathes) == 1:
                    treeSelection.unselect_all()
                #右クリック行のみ選択
                treeSelection.select_path(path)
                #現在のselectedPathを取得
                _treestore, selPathes = treeSelection.get_selected_rows()
                #0.1s後に再度選択する
                th = threading.Thread(target=self.afterSelectRow, args=(widget, selPathes))
                th.start()
            #widget毎に処理
            self.showPopupMenu(widget, pos)
        #doubleClick?
        elif event.button == 1:
            if event.type.value_name == "GDK_2BUTTON_PRESS":
                #doubleClickの場合
                self.doubleClickEvent(widget, pos)
    
    def afterSelectRow(self, widget, pathes):
        """ 0.1s後にpathを選択する"""
        time.sleep(0.1)
        for path in pathes:
            treeSelection = widget.get_selection()
            treeSelection.select_path(path)

    def doubleClickEvent(self, widget, pos):
        """ doubleClick時の処理"""
        widgetDict = {
            self.tree_time:             self.onDCtime,
            self.tree_timeFluidFile:    self.onDCtimeFluid,
            self.tree_timeSolidFile:    self.onDCtimeSolid,
            self.tree_constant:         self.onDCconst,
            self.tree_constantFluidFile: self.onDCconstFluid,
            self.tree_constantSolidFile: self.onDCconstSolid,
            self.tree_system:           self.onDCsystem,
            self.tree_systemFluidFile:  self.onDCsystemFluid,
            self.tree_systemSolidFile:  self.onDCsystemSolid,
            self.tree_setMaterial:      self.onDCmaterial,
            self.tree_matDB:            self.onDCmatDB
            }
        func = widgetDict[widget]
        func(pos)

    #
    #  showTooltipMatDB
    #-------------------
    def showTooltipMatDB(self, treeView, event):
        """ treeView(materialDB)のtoolTipを表示する。"""
        pos = (event.x, event.y)
        pathInfo = treeView.get_path_at_pos(*pos)
        if pathInfo == None:
            return
        (path, _col, _x, _y) = pathInfo
        if path == None:
            return
        text = _("ここをダブルクリックすると") + "\n"
        text += _("ファイルを開きます。")
        treeView.props.tooltip_text = text

    #
    #  showTooltipSetMaterial
    #-------------------------
    def showTooltipSetMaterial(self, treeView, event):
        """ treeView(材料設定)のtooltipを表示する"""
        pos = (event.x, event.y)
        pathInfo = treeView.get_path_at_pos(*pos)
        if pathInfo == None:
            return
        (path, col, _x, _y) = pathInfo
        if path == None:
            return
        #tooltipを表示
        for i in range(len(treeView.get_columns())):
            if treeView.get_column(i) == col:
                break
        if i == 0:
            text = _("ここをダブルクリックすると") + "\n"
            text += _("  regionのfolderを開きます。")
        elif i == 1:
            text = _("ここをダブルクリックすると") + "\n"
            text += _("  選択されたmaterialDBの材料を設定します。")
        else:
            text = ""
        treeView.props.tooltip_text = text

    
    #--------- key event ------------------
    #
    #  keyPress
    #-----------
    def keyPress(self, widget, event):
        """ key pressの処理"""
        keyName = Gdk.keyval_name(event.keyval)
        if keyName == "Control_L":
            self.ctrlKeyFlag = 1
    
    #
    #  keyRelease
    #-------------
    def keyRelease(self, widget, event):
        """ key releaseの処理"""
        keyName = Gdk.keyval_name(event.keyval)
        if keyName == "Control_L":
            self.ctrlKeyFlag = 0


    #---------- popupMenu関連---------
    #
    #  setPopupMenu
    #---------------
    def setPopupMenu(self):
        """ popupMenuを定義"""
        self.timePopupMenuItems = [
            [_("開く"),                     self.popTimeEdit],
            [_("名前変更"),                 self.popTimeRename],
            [_("名前からregion名を削除"),   self.popTimeDelRegion],
            [_("削除"),                     self.popTimeDelete],
            ["-",                           ""],
            [_("コピー"),                   self.popTimeCopy],
            [_("貼り付け"),                 self.popTimePaste]
            ]

        self.constantPopupMenuItems = [
            [_("開く"),                     self.popConstEdit],
            [_("名前変更"),                 self.popConstRename],
            [_("名前からregion名を削除"),   self.popConstDelRegion],
            [_("削除"),                     self.popConstDelete],
            ["-",                           ""],
            [_("コピー"),                   self.popConstCopy],
            [_("貼り付け"),                 self.popConstPaste]
            ]

        self.systemPopupMenuItems = [
            [_("開く"),                     self.popSystemEdit],
            [_("名前変更"),                 self.popSystemRename],
            [_("名前からregion名を削除"),   self.popSystemDelRegion],
            [_("削除"),                     self.popSystemDelete],
            ["-",                           ""],
            [_("コピー"),                   self.popSystemCopy],
            [_("貼り付け"),                 self.popSystemPaste]
            ]

        self.timeFluidPopupMenuItems = [
            [_("開く"),     self.popTimeFluidEdit],
            [_("名前変更"), self.popTimeFluidRename],
            [_("削除"),     self.popTimeFluidDelete],
            ["-",           ""],
            [_("コピー"),   self.popTimeFluidCopy],
            [_("貼り付け"), self.popTimeFluidPaste]
            ]

        self.timeSolidPopupMenuItems = [
            [_("開く"),     self.popTimeSolidEdit],
            [_("名前変更"), self.popTimeSolidRename],
            [_("削除"),     self.popTimeSolidDelete],
            ["-",           ""],
            [_("コピー"),   self.popTimeSolidCopy],
            [_("貼り付け"), self.popTimeSolidPaste]
            ]

        self.constantFluidPopupMenuItems = [
            [_("開く"),     self.popConstFluidEdit],
            [_("名前変更"), self.popConstFluidRename],
            [_("削除"),     self.popConstFluidDelete],
            ["-",           ""],
            [_("コピー"),   self.popConstFluidCopy],
            [_("貼り付け"), self.popConstFluidPaste]
            ]

        self.constantSolidPopupMenuItems = [
            [_("開く"),     self.popConstSolidEdit],
            [_("名前変更"), self.popConstSolidRename],
            [_("削除"),     self.popConstSolidDelete],
            ["-",           ""],
            [_("コピー"),   self.popConstSolidCopy],
            [_("貼り付け"), self.popConstSolidPaste]
            ]

        self.systemFluidPopupMenuItems = [
            [_("開く"),     self.popSystemFluidEdit],
            [_("名前変更"), self.popSystemFluidRename],
            [_("削除"),     self.popSystemFluidDelete],
            ["-",           ""],
            [_("コピー"),   self.popSystemFluidCopy],
            [_("貼り付け"), self.popSystemFluidPaste]
            ]

        self.systemSolidPopupMenuItems = [
            [_("開く"),     self.popSystemSolidEdit],
            [_("名前変更"), self.popSystemSolidRename],
            [_("削除"),     self.popSystemSolidDelete],
            ["-",           ""],
            [_("コピー"),   self.popSystemSolidCopy],
            [_("貼り付け"), self.popSystemSolidPaste]
            ]

    #
    #  showPopupMenu
    #----------------
    def showPopupMenu(self, widget, pos):
        """ popupMenuを表示する"""
        widgetMenuDict = {
            self.tree_time:            self.timePopupMenuItems,
            self.tree_timeFluidFile:   self.timeFluidPopupMenuItems,
            self.tree_timeSolidFile:   self.timeSolidPopupMenuItems,
            self.tree_constant:        self.constantPopupMenuItems,
            self.tree_constantFluidFile: self.constantFluidPopupMenuItems,
            self.tree_constantSolidFile: self.constantSolidPopupMenuItems,
            self.tree_system:          self.systemPopupMenuItems,
            self.tree_systemFluidFile: self.systemFluidPopupMenuItems,
            self.tree_systemSolidFile: self.systemSolidPopupMenuItems
            }
        if widget in widgetMenuDict.keys():
            menuItems = widgetMenuDict[widget]
            self.showPopupMenuItems(menuItems)

    def showPopupMenuItems(self, menuItems):
        """ popupメニュー表示"""
        #menuItems = self.timePopupMenuItems
        menu = Gtk.Menu()
        for i in range(len(menuItems)):
            menuName = menuItems[i][0]
            if menuName == "-":
                #セパレータをセット
                item = Gtk.SeparatorMenuItem()
            else:
                #menuItemをセット
                item = Gtk.MenuItem()
                item.set_label(menuName)
                item.connect("activate", menuItems[i][1])
            menu.append(item)
        #menuを表示
        menu.show_all()
        menu.popup(None, None, None, None, 0, Gtk.get_current_event_time())

    
    #------ tooltip 関連 -------------
    #
    #  setTooltipText
    #-----------------
    def setTooltipText(self):
        """ tooltipTextを設定する。"""
        tooltipTreeView = _("ここをダブルクリックすると、\n　選択したfileを開きます。")
        tooltipTreeList = _("ここをダブルクリックすると、\n　選択したregion内の選択したfileを開きます。")
        self.tree_time.props.tooltip_text = tooltipTreeView
        self.tree_constant.props.tooltip_text = tooltipTreeView
        self.tree_system.props.tooltip_text = tooltipTreeView

        self.tree_timeFluidFile.props.tooltip_text = tooltipTreeList
        self.tree_timeSolidFile.props.tooltip_text = tooltipTreeList
        self.tree_constantFluidFile.props.tooltip_text = tooltipTreeList
        self.tree_constantSolidFile.props.tooltip_text = tooltipTreeList
        self.tree_systemFluidFile.props.tooltip_text = tooltipTreeList
        self.tree_systemSolidFile.props.tooltip_text = tooltipTreeList


    #
    #  setTitleForSolid
    #-------------------
    def setTitleForSolid(self):
        """ 固体の材料設定pageのtitleを設定"""
        title = _("「") + self.solidProperties + _("」の設定")
        self.label_solidTitle.set_text(title)


    #
    #  editTimeTreeFiles
    #--------------------
    def editTimeTreeFiles(self):
        """ timeTree内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeView(self.tree_time)
        if len(relFiles) == 0:
            return
        parentDir = self.caseDir + "/" + self.iniTime + "/"
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)

    def editOpenFiles(self, fileNames):
        """ relFiles内のfile:editorで開く、folder:fileManagerで開く。"""
        #pathNames = map(lambda x: parentDir+x, relFiles)
        files = []
        #folders = []
        for pathName in fileNames:
            if os.path.isfile(pathName) == True:
                files.append(pathName)
            else:
                pyTreeFoam.run(pathName).fileManager()
        if len(files) > 0:
            #pyTreeFoam.run().editor(files)
            pyTreeFoam.run().foamEditor(files)

    def getSelectedFilesInTreeView(self, treeView):
        """ treeView内の選択fileを取得する"""
        relDirs = []
        treeSelection = treeView.get_selection()
        _treeStore, pathes = treeSelection.get_selected_rows()
        for path in pathes:
            pathDir = self.getDirFromTreePath(treeView, path)
            relDirs.append(pathDir)
        return relDirs

    def getDirFromTreePath(self, treeView, path):
        """ treePathからdir(相対path)を取得する"""
        treeStore = treeView.get_model()
        words = []
        for i in range(len(path)):
            treeIter = treeStore.get_iter(path[:i+1])
            value = treeStore.get_value(treeIter, 1)
            words.append(value)
        wdir = "/".join(words)
        return wdir

    #
    #  editConstantTreeFiles
    #------------------------
    def editConstantTreeFiles(self):
        """ constantTree内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeView(self.tree_constant)
        if len(relFiles) == 0:
            return
        parentDir = self.caseDir + "/constant/"
        #pyTreeFoam.run(parentDir).editor(relFiles)
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)

    #
    #  editSystemTreeFiles
    #----------------------
    def editSystemTreeFiles(self):
        """ systemTree内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeView(self.tree_system)
        if len(relFiles) == 0:
            return
        parentDir = self.caseDir + "/system/"
        #pyTreeFoam.run(parentDir).editor(relFiles)
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)


    #
    #  editTimeFluidFiles
    #---------------------
    def editTimeFluidFiles(self):
        """ timeFluid内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_timeFluidFile)
        regions = self.getSelectedFilesInTreeList(self.tree_timeFluidRegion)
        if len(relFiles) == 0 or len(regions) == 0:
            return
        parentDir = self.caseDir + "/" + self.iniTime + "/" + regions[0] + "/"
        #pyTreeFoam.run(parentDir).editor(relFiles)
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)
 
    def getSelectedFilesInTreeList(self, treeList):
        """ treeView内の選択fileを取得する"""
        relDirs = GtkParts.treeListIcon(treeList).getSelectedItems()
        names = list(map(lambda x: x[1], relDirs))
        return names

    #
    #  editTimeSolidFiles
    #---------------------
    def editTimeSolidFiles(self):
        """ timeSolid内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_timeSolidFile)
        regions = self.getSelectedFilesInTreeList(self.tree_timeSolidRegion)
        if len(relFiles) == 0 or len(regions) == 0:
            return
        parentDir = self.caseDir + "/" + self.iniTime + "/" + regions[0] + "/"
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)

    #
    #  editConstantFluidFiles
    #-------------------------
    def editConstantFluidFiles(self):
        """ constantFluid内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_constantFluidFile)
        regions = self.getSelectedFilesInTreeList(self.tree_constantFluidRegion)
        if len(relFiles) == 0 or len(regions) == 0:
            return
        parentDir = self.caseDir + "/constant/" + regions[0] + "/"
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)

    #
    #  editConstantSolidFiles
    #-------------------------
    def editConstantSolidFiles(self):
        """ constantSolid内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_constantSolidFile)
        regions = self.getSelectedFilesInTreeList(self.tree_constantSolidRegion)
        if len(relFiles) == 0 or len(regions) == 0:
            return
        parentDir = self.caseDir + "/constant/" + regions[0] + "/"
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)

    #
    #  editSystemFluidFiles
    #----------------------
    def editSystemFluidFiles(self):
        """ systemFluid内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_systemFluidFile)
        regions = self.getSelectedFilesInTreeList(self.tree_systemFluidRegion)
        if len(relFiles) == 0 or len(regions) == 0:
            return
        parentDir = self.caseDir + "/system/" + regions[0] + "/"
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)

    #
    #  editSystemSolidFiles
    #-----------------------
    def editSystemSolidFiles(self):
        """ systemSolid内のfileを編集する。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_systemSolidFile)
        regions = self.getSelectedFilesInTreeList(self.tree_systemSolidRegion)
        if len(relFiles) == 0 or len(regions) == 0:
            return
        parentDir = self.caseDir + "/system/" + regions[0] + "/"
        fileNames = list(map(lambda x: parentDir+x, relFiles))
        self.editOpenFiles(fileNames)

    #
    #  renameTimeFile
    #-----------------
    def renameTimeFile(self):
        """ timeTreeView内のfile名を変更"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.renameFileInTreeView(self.tree_time, parentDir)

    #
    #  renameFileInTreeView
    def renameFileInTreeView(self, treeView, parentDir):
        """ treeView内の選択fileをrenameする"""
        relFiles = self.getSelectedFilesInTreeView(treeView)
        if self.isSingleSelection(relFiles) == False:
            return
        fileName = parentDir + "/" + relFiles[0]
        #選択pathを取得
        treeSelection = treeView.get_selection()
        _treeStore, pathes = treeSelection.get_selected_rows()
        selPath = pathes[0]
        #rename
        name = fileName.split("/")[-1]
        if name == "regionProperties":
            mess = _("「regionProperties」は、renameできません。")
            self.errDialog(_("エラー"), mess)
            return
        title = _("file名の変更")
        mess = _("新しい名前を入力してください。")
        okArgs = [self.renameFileInTreeView_run, treeView, selPath, parentDir, fileName]        
        dialog = inputTextDDialog.getInputText(
            title, mess, name, 
            funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return
        
    #  renameFileInTreeView_run
    def renameFileInTreeView_run(self, newName, treeView, selPath, parentDir, fileName):
        newFileName = "/".join(fileName.split("/")[:-1]) + "/" + newName
        os.rename(fileName, newFileName)
        #再表示
        treeData = self.getTreeDataFromDir(parentDir)
        self.setItemToTreeView(treeView, treeData)
        #選択表示する
        treeView.expand_to_path(selPath)
        treeSelection = treeView.get_selection()
        treeSelection.select_path(selPath)

    def isSingleSelection(self, relFiles):
        """ 選択個数を確認し、複数選択の場合は、エラーメッセージを出力する。"""
        if len(relFiles) == 0:
            return False
        if len(relFiles) != 1:
            mess = _("複数選択しています。") + "\n"
            mess += _("  1ヶのfileを選択してください。")
            self.errDialog(_("エラー"), mess)
            return False
        return True

    #
    #  renameConstantFile
    #---------------------
    def renameConstantFile(self):
        """ constantTreeView内のfile名を変更"""
        parentDir = self.caseDir + "/constant"
        self.renameFileInTreeView(self.tree_constant, parentDir)

    #
    #  renameSystemFile
    #-------------------
    def renameSystemFile(self):
        """ systemTreeView内のfile名を変更"""
        parentDir = self.caseDir + "/system"
        self.renameFileInTreeView(self.tree_system, parentDir)
        
    #
    #  renameTimeFluidFile
    #----------------------
    def renameTimeFluidFile(self):
        """ timeFluidTreeList内のfile名を変更"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.renameFileInTreeList(self.tree_timeFluidFile, self.tree_timeFluidRegion, parentDir)
    
    #
    #  renameFileInTreeList
    def renameFileInTreeList(self, treeFile, treeRegion, parentDir):
        """ treeList内のfile名を変更する。"""
        relFiles = self.getSelectedFilesInTreeList(treeFile)
        regions = self.getSelectedFilesInTreeList(treeRegion)
        if self.isSingleSelection(relFiles) == False or len(regions) == 0:
            return
        fileName = parentDir + "/" + regions[0] + "/" + relFiles[0]
        #選択pathを取得
        treeSelection = treeFile.get_selection()
        _treeStore, pathes = treeSelection.get_selected_rows()
        selPath = pathes[0]
        #rename
        name = fileName.split("/")[-1]
        title = _("file名の変更")
        mess = _("新しい名前を入力してください。")
        okArgs = [self.renameFileInTreeList_run, treeFile, selPath, parentDir, regions, fileName]
        dialog = inputTextDDialog.getInputText(title, mess, name, funcOk=okArgs)
        dialog.show()
        return
        
    #  renameFileInTreeList_run
    def renameFileInTreeList_run(self, newName, treeFile, selPath, parentDir, regions, fileName):
        newFileName = "/".join(fileName.split("/")[:-1]) + "/" + newName
        os.rename(fileName, newFileName)
        #dataを取得
        files = pyTreeFoam.getFileNames(parentDir+"/"+regions[0])
        files = self.checkFileName(files)
        files.sort()
        #再表示
        wkDir = parentDir+"/"+regions[0]
        self.setItemToTreeListFil(treeFile, files, wkDir)
        #選択表示する
        treeSelection = treeFile.get_selection()
        treeSelection.select_path(selPath)

    #
    #  renameTimeSolidFile
    #----------------------
    def renameTimeSolidFile(self):
        """ timeSolidTreeList内のfile名を変更"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.renameFileInTreeList(self.tree_timeSolidFile, self.tree_timeSolidRegion, parentDir)

    #
    #  renameConstantFluidFile
    #--------------------------
    def renameConstantFluidFile(self):
        """ constantFluidTreeList内のfile名を変更"""
        parentDir = self.caseDir + "/constant"
        self.renameFileInTreeList(self.tree_constantFluidFile, self.tree_constantFluidRegion, parentDir)

    #
    #  renameConstantSolidFile
    #--------------------------
    def renameConstantSolidFile(self):
        """ constantSolidTreeList内のfile名を変更"""
        parentDir = self.caseDir + "/constant"
        self.renameFileInTreeList(self.tree_constantSolidFile, self.tree_constantSolidRegion, parentDir)

    #
    #  renameSystemFluidFile
    #-----------------------
    def renameSystemFluidFile(self):
        """ systemFluidTreeList内のfile名を変更"""
        parentDir = self.caseDir + "/system"
        self.renameFileInTreeList(self.tree_systemFluidFile, self.tree_systemFluidRegion, parentDir)

    #
    #  renameSystemSolidFile
    #------------------------
    def renameSystemSolidFile(self):
        """ systemSolidTreeList内のfile名を変更"""
        parentDir = self.caseDir + "/system"
        self.renameFileInTreeList(self.tree_systemSolidFile, self.tree_systemSolidRegion, parentDir)

    #
    #  deleteTimeFiles
    #------------------
    def deleteTimeFiles(self):
        """ timeTreeView内のfileを削除する"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.deleteFilesInTreeView(self.tree_time, parentDir)

    def deleteFilesInTreeView(self, treeView, parentDir):
        """ treeViewから削除するfileを取得する。"""
        relFiles = self.getSelectedFilesInTreeView(treeView)
        delFiles = map(lambda x: parentDir + "/" + x, relFiles)
        delFiles = list(filter(lambda x: os.path.isfile(x), delFiles))
        #選択pathを取得
        treeSelection = treeView.get_selection()
        _treeStore, delPathes = treeSelection.get_selected_rows()
        showArgs = ["treeView", treeView, parentDir, delPathes]
        self.deleteFilesInTree(delFiles, showArgs)
        
    def deleteFiles_showTreeView(self, treeView, parentDir, delPathes):
        #treeViewを再表示
        treeData = self.getTreeDataFromDir(parentDir)
        self.setItemToTreeView(treeView, treeData)
        #treeExpand確認
        expPathes = filter(lambda x: len(x)==2, delPathes)
        for expPath in expPathes:
            #treeをexpandする。
            treeView.expand_to_path(expPath)

    def deleteFilesInTree(self, delFiles, showArgs):
        """ fileをゴミ箱へ移動する。"""
        if len(delFiles) == 0:
            return
        title = _("fileの削除")
        mess = _("fileをゴミ箱に移動します。")
        funcOk = [self.deleteFilesInTree_runDel, delFiles, showArgs]
        self.okCancelDialog(title, mess, funcOk=funcOk)
        return

    #  deleteFilesInTree_runDel    
    def deleteFilesInTree_runDel(self, delFiles, showArgs):
        #fileを削除する
        undelFiles = pyTreeFoam.moveToTrash(delFiles)
        #削除error？
        if len(undelFiles) != 0:
            #error発生時の処理
            title = _("ゴミ箱へ移動")
            names = list(map(lambda x: os.path.basename(x), undelFiles))
            if len(names) > 5:
                nameCont = ", ".join(names[:5]) + "..."
            else:
                nameCont = ", ".join(names)
            mess = _("「") + nameCont + _("」がゴミ箱へ移動できませんでした。" ) + "\n"
            mess +=_("これらのファイルを完全に削除しますか？")
            funcOk = [self.deleteFilesInTree_conf, undelFiles, showArgs]
            self.okCancelDialog(title, mess, funcOk=funcOk)
            return
        #treeView再表示
        self.deleteFiles_show(showArgs)
        #self.deleteFiles_showTreeView(treeView, parentDir, delPathes)        

    #  deleteFilesInTree_conf
    def deleteFilesInTree_conf(self, undelFiles, showArgs):
        for name in undelFiles:
            os.remove(name)
        #treeView再表示
        self.deleteFiles_show(showArgs)
        #self.deleteFiles_showTreeView(treeView, parentDir, delPathes)        

    #
    #  deleteFiles_show
    def deleteFiles_show(self, showArgs):
        """ treeView or treeListを再表示する"""
        if showArgs[0] == "treeView":
            #treeViewを再表示
            _name, treeView, parentDir, delPathes = showArgs
            self.deleteFiles_showTreeView(treeView, parentDir, delPathes)
        else:
            #treeListを再表示
            _name, treeFile, parentDir, region = showArgs
            self.deleteFilesInTreeList_showTreeList(treeFile, parentDir, region)

    #
    #  deleteConstantFiles
    #----------------------
    def deleteConstantFiles(self):
        """ constantTreeView内のfileを削除する"""
        parentDir = self.caseDir + "/constant"
        self.deleteFilesInTreeView(self.tree_constant, parentDir)

    #
    #  deleteSystemFiles
    #--------------------
    def deleteSystemFiles(self):
        """ systemTreeView内のfileを削除する"""
        parentDir = self.caseDir + "/system"
        self.deleteFilesInTreeView(self.tree_system, parentDir)

    #
    #  deleteTimeFluidFiles
    #----------------------
    def deleteTimeFluidFiles(self):
        """ timeFluidTreeList内のfileを削除する"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.deleteFilesInTreeList(self.tree_timeFluidFile, self.tree_timeFluidRegion, parentDir)

    def deleteFilesInTreeList(self, treeFile, treeRegion, parentDir):
        """ treeList内のfile名を削除する。"""
        relFiles = self.getSelectedFilesInTreeList(treeFile)
        region = self.getSelectedFilesInTreeList(treeRegion)[0]
        delFiles = list(map(lambda x: parentDir + "/" + region + "/" + x, relFiles))
        showArgs = ["treeList", treeFile, parentDir, region]
        self.deleteFilesInTree(delFiles, showArgs)
        
    def deleteFilesInTreeList_showTreeList(self, treeFile, parentDir, region):        
        #dataを取得
        files = pyTreeFoam.getFileNames(parentDir+"/"+region)
        files = self.checkFileName(files)
        files.sort()
        #再表示
        wkDir = parentDir+"/"+region
        self.setItemToTreeListFil(treeFile, files, wkDir)

    #
    #  deleteTimeSolidFiles
    #-----------------------
    def deleteTimeSolidFiles(self):
        """ timeSolidTreeList内のfileを削除する"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.deleteFilesInTreeList(self.tree_timeSolidFile, self.tree_timeSolidRegion, parentDir)

    #
    #  deleteConstantFluidFiles
    #---------------------------
    def deleteConstantFluidFiles(self):
        """ constantFluidTreeList内のfileを削除する"""
        parentDir = self.caseDir + "/constant"
        self.deleteFilesInTreeList(self.tree_constantFluidFile, self.tree_constantFluidRegion, parentDir)

    #
    #  deleteConstantSolidFiles
    #---------------------------
    def deleteConstantSolidFiles(self):
        """ constantSolidTreeList内のfileを削除する"""
        parentDir = self.caseDir + "/constant"
        self.deleteFilesInTreeList(self.tree_constantSolidFile, self.tree_constantSolidRegion, parentDir)

    #
    #  deleteSystemFluidFiles
    #-------------------------
    def deleteSystemFluidFiles(self):
        """ systemFluidTreeList内のfileを削除する"""
        parentDir = self.caseDir + "/system"
        self.deleteFilesInTreeList(self.tree_systemFluidFile, self.tree_systemFluidRegion, parentDir)

    #
    #  deleteSystemSolidFiles
    #-------------------------
    def deleteSystemSolidFiles(self):
        """ systemSolidTreeList内のfileを削除する"""
        parentDir = self.caseDir + "/system"
        self.deleteFilesInTreeList(self.tree_systemSolidFile, self.tree_systemSolidRegion, parentDir)

    #
    #  copyTimeFiles
    #---------------
    def copyTimeFiles(self):
        """ time内のfileをclipboardへコピー"""
        parentDir = self.caseDir + "/" + self.iniTime
        relFiles = self.getSelectedFilesInTreeView(self.tree_time)
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    def copyToClipboard(self, files):
        """ systemのclipboardへコピーする。"""
        files = list(filter(lambda x: os.path.isfile(x), files))
        f = open(self.clipboardFile, "w")
        f.write("foldersCopy\n")
        for copyDir in files:
            print("copy--->", copyDir)
            f.write(copyDir+"\n")
        f.close()
        #comm = "./copyToSystemClipBoard.py " + self.clipboardFile + " &"
        #path = os.getenv("TreeFoamPath") + os.sep + "python"
        #pyTreeFoam.run(path).commandWithLog(comm)
        copyPasteFolders.copyFolders(files)

    #
    #  copyConstantFiles
    #--------------------
    def copyConstantFiles(self):
        """ constant内のfileをclipboardへコピー"""
        parentDir = self.caseDir + "/constant"
        relFiles = self.getSelectedFilesInTreeView(self.tree_constant)
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  copySystemFiles
    #------------------
    def copySystemFiles(self):
        """ system内のfileをclipboardへコピー"""
        parentDir = self.caseDir + "/system"
        relFiles = self.getSelectedFilesInTreeView(self.tree_system)
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  copyTimeFluidFiles
    #--------------------
    def copyTimeFluidFiles(self):
        """ timeFluid内のfileをclipboardへコピー"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_timeFluidFile)
        region = self.getSelectedFilesInTreeList(self.tree_timeFluidRegion)[0]
        parentDir = self.caseDir + "/" + self.iniTime + "/" + region
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  copyTimeSolidFiles
    #---------------------
    def copyTimeSolidFiles(self):
        """ timeSolid内のfileをclipboardへコピー"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_timeSolidFile)
        region = self.getSelectedFilesInTreeList(self.tree_timeSolidRegion)[0]
        parentDir = self.caseDir + "/" + self.iniTime + "/" + region
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  copyConstantFluidFiles
    #-------------------------
    def copyConstantFluidFiles(self):
        """ constantFluid内のfileをclipboardへコピー"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_constantFluidFile)
        region = self.getSelectedFilesInTreeList(self.tree_constantFluidRegion)[0]
        parentDir = self.caseDir + "/constant/" + region
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  copyConstantSolidFiles
    #-------------------------
    def copyConstantSolidFiles(self):
        """ constantSolid内のfileをclipboardへコピー"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_constantSolidFile)
        region = self.getSelectedFilesInTreeList(self.tree_constantSolidRegion)[0]
        parentDir = self.caseDir + "/constant/" + region
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  copySystemFluidFiles
    #-----------------------
    def copySystemFluidFiles(self):
        """ systemFluid内のfileをclipboardへコピー"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_systemFluidFile)
        region = self.getSelectedFilesInTreeList(self.tree_systemFluidRegion)[0]
        parentDir = self.caseDir + "/system/" + region
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  copySystemSolidFiles
    #-----------------------
    def copySystemSolidFiles(self):
        """ systemSolid内のfileをclipboardへコピー"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_systemSolidFile)
        region = self.getSelectedFilesInTreeList(self.tree_systemSolidRegion)[0]
        parentDir = self.caseDir + "/system/" + region
        files = list(map(lambda x: parentDir + "/" + x, relFiles))
        self.copyToClipboard(files)

    #
    #  pasteTimeFiles
    #-----------------
    def pasteTimeFiles(self):
        """ clipboardの内容をtimeに貼り付ける。"""
        parentDir = self.caseDir + "/" + self.iniTime
        relFiles = self.getSelectedFilesInTreeView(self.tree_time)
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir + "/" + relFiles[0]
        self.pasteFileRemakeTreeView(self.tree_time, parentDir, pasteDir)

    def pasteFileRemakeTreeView(self, treeView, parentDir, pasteDir):
        #pastedirのチェック
        if os.path.isdir(pasteDir) == False:
            pasteDir = "/".join(pasteDir.split("/")[:-1])
        #貼り付け
        newFileNames = self.pasteFileFromClipboard(pasteDir)
        #再表示させる
        treeData = self.getTreeDataFromDir(parentDir)
        self.setItemToTreeView(treeView, treeData)
        #選択表示させる
        self.selectPasteNames(treeView, treeData, parentDir, newFileNames)

    def selectPasteNames(self, treeView, treeData, rootDir, selNames):
        """ treeView内のselNamesを選択表示させる。"""
        rootWords = rootDir.split("/")
        pathes = []
        for selName in selNames:
            selWords = selName.split("/")
            #pathを取得
            selCont = selWords[len(rootWords):]
            items = treeData[1]
            #pathInt = [0]
            pathInt = []        #rootがtree上に存在しないため
            for word in selCont:
                for i in range(len(items)):
                    if type(items[i]) == str:
                        if word == items[i]:
                            pathInt.append(i)
                            break
                    else:
                        if word == items[i][0]:
                            pathInt.append(i)
                            items = items[i][1]
                            break
            path = Gtk.TreePath.new_from_indices(pathInt)
            pathes.append(path)
        #treeをexpandさせる
        for path in pathes:
            if len(path) > 1:
                treeView.expand_to_path(path)
        #pathを選択表示させる
        treeSelection = treeView.get_selection()
        for path in pathes:
            treeSelection.select_path(path)

    def pasteFileFromClipboard(self, pasteDir):
        """ clipboardの内容をpasteDirに貼り付ける。"""
        # pasteContFile = self.pasteContFile
        # comm = "./pasteFromSystemClipBoard.py " + pasteContFile
        # path = os.getenv("TreeFoamPath") + os.sep + "python"
        # pyTreeFoam.run(path).commandWithLog(comm)
        # f = open(pasteContFile); cont = f.read(); f.close()
        # copyDirs = cont.split()
        copyDirs = copyPasteFolders.pasteFolders()
        #file名のチェック
        newFileNames = []
        for copyDir in copyDirs:
            baseName = os.path.basename(copyDir)
            newFileName = pasteDir + "/" + baseName
            if len(glob.glob(newFileName)) != 0:
                i = 0
                while True:
                    newBaseName = baseName + "_copy" + str(i)
                    newFileName = pasteDir + "/" + newBaseName
                    if len(glob.glob(newFileName)) == 0:
                        break
                    i += 1
            shutil.copy(copyDir, newFileName)
            newFileNames.append(newFileName)
        return newFileNames

    #
    #  pasteConstantFiles
    #---------------------
    def pasteConstantFiles(self):
        """ clipboardの内容をconstantに貼り付ける。"""
        parentDir = self.caseDir + "/constant"
        relFiles = self.getSelectedFilesInTreeView(self.tree_constant)
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir + "/" + relFiles[0]
        self.pasteFileRemakeTreeView(self.tree_constant, parentDir, pasteDir)

    #
    #  pasteSystemFiles
    #-------------------
    def pasteSystemFiles(self):
        """ clipboardの内容をsystemに貼り付ける。"""
        parentDir = self.caseDir + "/system"
        relFiles = self.getSelectedFilesInTreeView(self.tree_system)
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir + "/" + relFiles[0]
        self.pasteFileRemakeTreeView(self.tree_system, parentDir, pasteDir)

    #
    #  pasteTimeFluidFiles
    #----------------------
    def pasteTimeFluidFiles(self):
        """ clipboardの内容をtimeFluidに貼り付ける。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_timeFluidFile)
        region = self.getSelectedFilesInTreeList(self.tree_timeFluidRegion)[0]
        parentDir = self.caseDir + "/" + self.iniTime + "/" + region
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir
        self.pasteFileRemakeTreeList(self.tree_timeFluidFile, pasteDir)

    def pasteFileRemakeTreeList(self, treeFile, pasteDir):
        #貼り付け
        newFileNames = self.pasteFileFromClipboard(pasteDir)
        #再表示させる
        files = pyTreeFoam.getFileNames(pasteDir)
        files = self.checkFileName(files)
        files.sort()
        #再表示
        wkDir = pasteDir
        self.setItemToTreeListFil(treeFile, files, wkDir)
        #pasteFilesを選択表示させる
        treeSelection = treeFile.get_selection()
        treeStore = treeFile.get_model()
        pathes = []
        pasteNames = list(map(lambda x: x.split("/")[-1], newFileNames))
        for i in range(len(treeStore)):
            value = treeStore[i][0]
            if value in pasteNames:
                path = Gtk.TreePath(i)
                pathes.append(path)
        for path in pathes:
            treeSelection.select_path(path)


    #
    #  pasteTimeSolidFiles
    #----------------------
    def pasteTimeSolidFiles(self):
        """ clipboardの内容をtimeSolidに貼り付ける。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_timeSolidFile)
        region = self.getSelectedFilesInTreeList(self.tree_timeSolidRegion)[0]
        parentDir = self.caseDir + "/" + self.iniTime + "/" + region
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir
        self.pasteFileRemakeTreeList(self.tree_timeSolidFile, pasteDir)

    #
    #  pasteConstantFluidFiles
    #--------------------------
    def pasteConstantFluidFiles(self):
        """ clipboardの内容をconstantFluidに貼り付ける。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_constantFluidFile)
        region = self.getSelectedFilesInTreeList(self.tree_constantFluidRegion)[0]
        parentDir = self.caseDir + "/constant/" + region
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir
        self.pasteFileRemakeTreeList(self.tree_constantFluidFile, pasteDir)

    #
    #  pasteConstantSolidFiles
    #--------------------------
    def pasteConstantSolidFiles(self):
        """ clipboardの内容をconstantSolidに貼り付ける。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_constantSolidFile)
        region = self.getSelectedFilesInTreeList(self.tree_constantSolidRegion)[0]
        parentDir = self.caseDir + "/constant/" + region
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir
        self.pasteFileRemakeTreeList(self.tree_constantSolidFile, pasteDir)

    #
    #  pasteSystemFluidFiles
    #------------------------
    def pasteSystemFluidFiles(self):
        """ clipboardの内容をsystemFluidに貼り付ける。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_systemFluidFile)
        region = self.getSelectedFilesInTreeList(self.tree_systemFluidRegion)[0]
        parentDir = self.caseDir + "/system/" + region
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir
        self.pasteFileRemakeTreeList(self.tree_systemFluidFile, pasteDir)

    #
    #  pasteSystemSolidFiles
    #------------------------
    def pasteSystemSolidFiles(self):
        """ clipboardの内容をsystemSolidに貼り付ける。"""
        relFiles = self.getSelectedFilesInTreeList(self.tree_systemSolidFile)
        region = self.getSelectedFilesInTreeList(self.tree_systemSolidRegion)[0]
        parentDir = self.caseDir + "/system/" + region
        if self.isSingleSelection(relFiles) == False:
            return
        pasteDir = parentDir
        self.pasteFileRemakeTreeList(self.tree_systemSolidFile, pasteDir)

    #
    #  changeRegionTimeFluid
    #------------------------
    def changeRegionTimeFluid(self):
        """ timeFluidのregionを変更した時、fileを読み直し再表示する。"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.changeRegionRemakeFiles(self.tree_timeFluidFile, self.tree_timeFluidRegion, parentDir)

    #
    #  changeRegionRemakeFiles
    def changeRegionRemakeFiles(self, treeFile, treeRegion, parentDir):
        """ 選択regionを変更した時、fileを読み直して再表示する。"""
        regions = self.getSelectedFilesInTreeList(treeRegion)
        if len(regions) == 0:
            return
        region = regions[0]
        #fileを読み直す
        files = pyTreeFoam.getFileNames(parentDir + "/" + region)
        files = self.checkFileName(files)
        files.sort()
        #treeListを再表示
        wkDir = parentDir + "/" + region
        self.setItemToTreeListFil(treeFile, files, wkDir)

    #
    #  changeRegionTimeSolid
    #------------------------
    def changeRegionTimeSolid(self):
        """ timeSoldの選択regionを変更した時、fileを読み直し再表示する。"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.changeRegionRemakeFiles(self.tree_timeSolidFile, self.tree_timeSolidRegion, parentDir)

    #
    #  changeRegionConstantFluid
    #-----------------------------
    def changeRegionConstantFluid(self):
        """ constantFluidの選択regionを変更した時、fileを読み直し再表示する。"""
        parentDir = self.caseDir + "/constant"
        self.changeRegionRemakeFiles(self.tree_constantFluidFile, self.tree_constantFluidRegion, parentDir)

    #
    #  changeRegionConstantSolid
    #----------------------------
    def changeRegionConstantSolid(self):
        """ constantSoldの選択regionを変更した時、fileを読み直し再表示する。"""
        parentDir = self.caseDir + "/constant"
        self.changeRegionRemakeFiles(self.tree_constantSolidFile, self.tree_constantSolidRegion, parentDir)

    #
    #  changeRegionSystemFluid
    #--------------------------
    def changeRegionSystemFluid(self):
        """ systemFluidの選択regionを変更した時、fileを読み直し再表示する。"""
        parentDir = self.caseDir + "/system"
        self.changeRegionRemakeFiles(self.tree_systemFluidFile, self.tree_systemFluidRegion, parentDir)

    #
    #  changeRegionSystemSolid
    #--------------------------
    def changeRegionSystemSolid(self):
        """ systemSolidの選択regionを変更した時、fileを読み直し再表示する。"""
        parentDir = self.caseDir + "/system"
        self.changeRegionRemakeFiles(self.tree_systemSolidFile, self.tree_systemSolidRegion, parentDir)

    #
    #  distFilesTimeFluid
    #---------------------
    def distFilesTimeFluid(self):
        """ timeの選択fileをfluidRegionに配布する。"""
        parentDir = self.caseDir + "/" + self.iniTime
        regions = self.fluidRegions
        stat = self.distributeFilesToRegions(self.tree_time, self.tree_timeFluidFile,
                    self.tree_timeFluidRegion, parentDir, regions)
        if stat == "OK":
            #distFilesのpatch、internalFieldを確認
            self.remakePatchCont(parentDir, regions)

    def remakePatchCont(self, parentDir, regions):
        """ distFieldsのpatch内容、internalfieldを修正する。"""
        relFiles = self.getSelectedFilesInTreeView(self.tree_time)
        fieldDirs = map(lambda x: parentDir + "/" + x, relFiles)
        fieldDirs = filter(lambda x: os.path.isfile(x), fieldDirs)
        fields = list(map(lambda x: os.path.basename(x), fieldDirs))
        case = pyTreeFoam.case(self.caseDir)
        for region in regions:
            relMeshDir = case.getCurrMeshDir(self.iniTime, region, "boundary")
            meshDir = self.caseDir + "/" + relMeshDir
            case.clearNullBoundaryFields(self.iniTime, meshDir, fields, region)
            fieldDir = parentDir + "/" + region
            case.clearInternalFields(fieldDir, fields)

    def distributeFilesToRegions(self, treeView, treeFile, treeRegion, parentDir, regions):
        """ distDir内の全fileを各regionに配布する。"""
        relFiles = self.getSelectedFilesInTreeView(treeView)
        files = map(lambda x: parentDir + "/" + x, relFiles)
        distFiles = list(filter(lambda x: os.path.isfile(x), files))
        if len(distFiles) == 0:
            title = _("エラー")
            mess = _("配布するfileを選択してください。")
            self.errDialog(title, mess)
            return "ERROR"
        #配布regionsに配布
        for file in distFiles:
            for region in regions:
                distDir = parentDir + "/" + region
                shutil.copy(file, distDir)
        #treeListを再表示
        region = self.getSelectedFilesInTreeList(treeRegion)[0]
        #fileを読み直す
        files = pyTreeFoam.getFileNames(parentDir + "/" + region)
        files = self.checkFileName(files)
        files.sort()
        #treeListを再表示
        wkDir = parentDir + "/" + region
        self.setItemToTreeListFil(treeFile, files, wkDir)
        #distFilesを選択表示させる
        treeSelection = treeFile.get_selection()
        treeStore = treeFile.get_model()
        pathes = []
        distNames = list(map(lambda x: x.split("/")[-1], distFiles))
        for i in range(len(treeStore)):
            value = treeStore[i][0]
            if value in distNames:
                path = Gtk.TreePath(i)
                pathes.append(path)
        for path in pathes:
            treeSelection.select_path(path)
        return "OK"

    #
    #  distFilesTimeSolid
    #---------------------
    def distFilesTimeSolid(self):
        """ timeの選択fileをsolidRegionに配布する。"""
        parentDir = self.caseDir + "/" + self.iniTime
        regions = self.solidRegions
        stat = self.distributeFilesToRegions(self.tree_time, self.tree_timeSolidFile,
                    self.tree_timeSolidRegion, parentDir, regions)
        if stat == "OK":
            #distFilesのpatch、internalFiedを確認
            self.remakePatchCont(parentDir, regions)

    #
    #  distFilesConstantFluid
    #-------------------------
    def distFilesConstantFluid(self):
        """ constantの選択fileをfluidRegionに配布する。"""
        parentDir = self.caseDir + "/constant"
        regions = self.fluidRegions
        self.distributeFilesToRegions(self.tree_constant, self.tree_constantFluidFile,
                    self.tree_constantFluidRegion, parentDir, regions)

    #
    #  distFilesConstantSolid
    #-------------------------
    def distFilesConstantSolid(self):
        """ constantの選択fileをsolidRegionに配布する。"""
        parentDir = self.caseDir + "/constant"
        regions = self.solidRegions
        self.distributeFilesToRegions(self.tree_constant, self.tree_constantSolidFile,
                    self.tree_constantSolidRegion, parentDir, regions)

    #
    #  distFilesSystemFluid
    #-----------------------
    def distFilesSystemFluid(self):
        """ systemの選択fileをfluidRegionに配布する。"""
        parentDir = self.caseDir + "/system"
        regions = self.fluidRegions
        self.distributeFilesToRegions(self.tree_system, self.tree_systemFluidFile,
                    self.tree_systemFluidRegion, parentDir, regions)

    #
    #  distFilesSystemSolid
    #-----------------------
    def distFilesSystemSolid(self):
        """ systemの選択fileをsolidRegionに配布する。"""
        parentDir = self.caseDir + "/system"
        regions = self.solidRegions
        self.distributeFilesToRegions(self.tree_system, self.tree_systemSolidFile,
                    self.tree_systemSolidRegion, parentDir, regions)

    #
    #  returnFilesTimeFluid
    #-----------------------
    def returnFilesTimeFluid(self):
        """ timeのfluid選択fileをfluidRegionsに戻す。"""
        parentDir = self.caseDir + "/" + self.iniTime
        retFolder = "fluidRegions"
        self.returnFilesFromRegions(self.tree_time, self.tree_timeFluidFile,
                    self.tree_timeFluidRegion, parentDir, retFolder)

    def returnFilesFromRegions(self, treeView, treeFile, treeRegion, parentDir, retFolder):
        retDir = parentDir + "/" + retFolder
        relFiles = self.getSelectedFilesInTreeList(treeFile)
        selRegion = self.getSelectedFilesInTreeList(treeRegion)[0]
        retFiles = map(lambda x: parentDir + "/"+ selRegion + "/" + x, relFiles)
        newFileNames = []
        for retFile in retFiles:
            baseName = os.path.basename(retFile)
            newFileName = retDir + "/" + baseName + "." + selRegion
            shutil.copy(retFile, newFileName)
            newFileNames.append(newFileName)
        #再表示させる
        treeData = self.getTreeDataFromDir(parentDir)
        self.setItemToTreeView(treeView, treeData)
        #選択表示させる
        self.selectPasteNames(treeView, treeData, parentDir, newFileNames)

    #
    #  returnFilesTimeSolid
    #-----------------------
    def returnFilesTimeSolid(self):
        """ timeのsolid選択fileをsolidRegionsに戻す。"""
        parentDir = self.caseDir + "/" + self.iniTime
        retFolder = "solidRegions"
        self.returnFilesFromRegions(self.tree_time, self.tree_timeSolidFile,
                    self.tree_timeSolidRegion, parentDir, retFolder)

    #
    #  returnFilesConstantFluid
    #---------------------------
    def returnFilesConstantFluid(self):
        """ constantのfluid選択fileをfluidRegionsに戻す。"""
        parentDir = self.caseDir + "/constant"
        retFolder = "fluidRegions"
        self.returnFilesFromRegions(self.tree_constant, self.tree_constantFluidFile,
                    self.tree_constantFluidRegion, parentDir, retFolder)

    #
    #  returnFilesConstantSolid
    #---------------------------
    def returnFilesConstantSolid(self):
        """ constantのsolid選択fileをsolidRegionsに戻す。"""
        parentDir = self.caseDir + "/constant"
        retFolder = "solidRegions"
        self.returnFilesFromRegions(self.tree_constant, self.tree_constantSolidFile,
                    self.tree_constantSolidRegion, parentDir, retFolder)

    #
    #  returnFilesSystemFluid
    #-------------------------
    def returnFilesSystemFluid(self):
        """ systemのfluid選択fileをfluidRegionsに戻す。"""
        parentDir = self.caseDir + "/system"
        retFolder = "fluidRegions"
        self.returnFilesFromRegions(self.tree_system, self.tree_systemFluidFile,
                    self.tree_systemFluidRegion, parentDir, retFolder)

    #
    #  returnFilesSystemSolid
    #-------------------------
    def returnFilesSystemSolid(self):
        """ systemのsolid選択fileをsolidRegionsに戻す。"""
        parentDir = self.caseDir + "/system"
        retFolder = "solidRegions"
        self.returnFilesFromRegions(self.tree_system, self.tree_systemSolidFile,
                    self.tree_systemSolidRegion, parentDir, retFolder)

    #
    #  deleteRegionNameTime
    #----------------------
    def deleteRegionNameTime(self):
        """ timeTree内の選択したfile名に付加されているregion名を削除する。"""
        parentDir = self.caseDir + "/" + self.iniTime
        self.deleteRegionName(self.tree_time, parentDir)

    def deleteRegionName(self, treeView, parentDir):
        """ treeView内のfile名に付加されているregion名を削除。"""
        relFiles = self.getSelectedFilesInTreeView(treeView)
        files = map(lambda x: parentDir + "/" + x, relFiles)
        files = list(filter(lambda x: os.path.isfile(x), files))
        if len(files) == 0:
            return
        regions = self.fluidRegions + self.solidRegions
        errFiles = []
        selectFileNames = []
        for file in files:
            baseName = os.path.basename(file)
            words = baseName.split(".")
            if words[-1] in regions:
                newName = ".".join(words[:-1])
                newFileName = "/".join(file.split("/")[:-1]) + "/" + newName
                if len(glob.glob(newFileName)) == 0:
                    os.rename(file, newFileName)
                    selectFileNames.append(newFileName)
                else:
                    errFiles.append(baseName)
                    selectFileNames.append(file)
        if len(errFiles) > 0:
            title = _("エラー")
            mess = _("「") + ", ".join(errFiles) + _("」は、regionが削除できません。")
            self.errDialog(title, mess)
        #再表示させる
        treeData = self.getTreeDataFromDir(parentDir)
        self.setItemToTreeView(treeView, treeData)
        #選択表示させる
        self.selectPasteNames(treeView, treeData, parentDir, selectFileNames)
        
    #
    #  deleteRegionNameConstant
    #---------------------------
    def deleteRegionNameConstant(self):
        """ constantTree内の選択したfile名に付加されているregion名を削除する。"""
        parentDir = self.caseDir + "/constant"
        self.deleteRegionName(self.tree_constant, parentDir)

    #
    #  deleteRegionNameSystem
    #-------------------------
    def deleteRegionNameSystem(self):
        """ systemTree内の選択したfile名に付加されているregion名を削除する。"""
        parentDir = self.caseDir + "/system"
        self.deleteRegionName(self.tree_system, parentDir)

    #
    #  referenceMatDB
    #-----------------
    def referenceMatDB(self):
        """ materialDBを参照し、設定する。"""
        title = _(u"固体用の物性値DBフォルダのセット")
        mess = _(u"固体の物性値データベースの保存先フォルダを設定します。")
        root = os.getenv("HOME")
        currDir = self.caseDir
        selDir = self.solidPhysicalDBdir
        #selDir = getOpenFolderDDialog.Show(title, mess, root, currDir, selDir)
        okArgs = [self.referenceMatDB_run]
        dialog = getOpenFolderDDialog.getOpenFolder(title, mess, root, currDir, selDir, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #  referenceMatDB_run
    def referenceMatDB_run(self, selDir):
        self.solidPhysicalDBdir = selDir
        self.entry_matDB.set_text(selDir)
        self.setSolidDirMultiRegion_data(selDir)
        self.solidPhysicalDB = self.getSolidPhysicalDB(selDir)
        wkDir = self.solidPhysicalDBdir
        self.setItemToTreeListFil(self.tree_matDB, self.solidPhysicalDB, wkDir)

    #
    #  setMaterialNameToListBox
    #---------------------------
    def setMaterialNameToListBox(self):
        """ 選択した材料名をlistBoxにセットする。"""

        def checkPhysicalNameCont(fileName):
            """ file名とfile内容のmaterialが合致しているか確認する。
            これらが異なっている場合は、materialをfile名に合わせる。
            materialが定義されていない場合は、定義する。"""
            f = open(fileName); cont = f.read(); f.close()
            p = cont.find("material ")
            if p < 0:
                #materialが定義されていない場合
                contOp = pyTreeFoam.strOp(cont)
                p = contOp.skipFoamFile()
                (keyword, p) = contOp.getKeyword(p)
                sp = p - len(keyword)
                header = cont[:sp]
                footer = cont[sp:]
                addLine = addLine = "material    " + fileName.split("/")[-1] + ";\n\n"
                newCont = header + addLine + footer
                f = open(fileName, "w"); f.write(newCont); f.close()
            else:
                #materialが定義されている場合
                material = fileName.split("/")[-1]
                contOp = pyTreeFoam.strOp(cont)
                p = contOp.skipFoamFile()
                lineCont = " "
                while lineCont != "":
                    ps = p
                    (lineCont, p, _kind) = contOp.get1line(p)
                    lineContOp = pyTreeFoam.strOp(lineCont)
                    (keyword, pp) = lineContOp.getKeyword(0)
                    if keyword == "material":
                        (matName, pp) = lineContOp.getKeyword(pp)
                        if matName != material:
                            #異なっている場合、書き換える
                            addLine = "material    " + material + ";"
                            (newCont, p, _keyword) = contOp.replace1lineKeyword(ps, addLine)
                            f = open(fileName, "w"); f.write(newCont); f.close()
                        break

        #材料DBの選択行を取得
        mats = self.getSelectedFilesInTreeList(self.tree_matDB)
        if len(mats) == 0:
            title = _(u"エラー")
            mess = _(u"材料DB内の材料名を選択していません。\n材料名を選択後、クリックして材料を設定してください。")
            self.errDialog(title, mess)
            return
        selMat = mats[0]
        fileName = self.solidPhysicalDBdir + "/" + selMat
        #file名の材料とfile内で定義されている材料をチェックする
        checkPhysicalNameCont(fileName)
        selRegions = self.getSelectedFilesInTreeList(self.tree_setMaterial)
        if len(selRegions) == 0:
            title = _(u"エラー")
            mess = _(u"設定先のregionが選択されていません。\nregion名を選択して材料を設定してください。")
            self.errDialog(title, mess)
            return
        #選択したregionに材料を設定する。
        listStore = self.tree_setMaterial.get_model()
        for i in range(len(listStore)):
            region = listStore[i][1]
            if region in selRegions:
                listStore[i][3] = selMat
                self.constantSolidPhysical[i] = selMat
                prop = self.solidProperties.split("_")[0]
                #pasteName = "/".join([self.caseDir, "constant", region, self.solidProperties])
                pasteName = "/".join([self.caseDir, "constant", region, prop])
                fileName = self.solidPhysicalDBdir + "/" + selMat
                shutil.copy(fileName, pasteName)

    #
    #  setCsvFile
    #-------------
    def setCsvFile(self):
        """ csvFileを読み込み、各regionに一括して材料名をセット"""
        title = _(u"csvFileの選択")
        currDir = self.caseDir
        fileFilters = [["csv", "*.csv"]]
        funcOk = [self.setCsvFile_csvRead]
        self.openFileDialog(title, currDir, funcOk=funcOk, fileFilters=fileFilters)
        return
    
    #
    #  setCsvFile_csvRead
    def setCsvFile_csvRead(self, fileName):

        def csvErrorCheck(data):
            """ 読み込んだcsvデータのエラーチェック"""
            #未定義材料のチェック
            regions = list(map(lambda x: x[0], data))
            unDefineRegs = set(self.solidRegions).difference(regions)
            if len(unDefineRegs) > 0:
                title = _(u"エラー")
                mess = _("csvFile内に以下のregionが足りません。")
                mess += _(" 「")
                if len(unDefineRegs) > 10:
                    names = ",".join(unDefineRegs[:10]) + "..."
                else:
                    names = ",".join(unDefineRegs)
                mess += _("」")
                self.errDialog(title, mess)
                return "ERROR"
            materials = list(map(lambda x: x[1], data))
            unDefineMats = set(materials).difference(self.solidPhysicalDB)
            if len(unDefineMats) > 0:
                title = _(u"エラー")
                message  = _(u"csvFile内に、DBで定義されていない材料があります。\n")
                message += _(u"「")
                if len(unDefineMats) > 10:
                    names = ",".join(unDefineMats[:10]) + "..."
                else:
                    names = ",".join(unDefineMats)
                message += names + _(u"」が未定義です。")
                self.errDialog(title, message)
                return "ERROR"
            return "OK"

        #scvFile読み込み
        dataFile = fileName
        readData = pyTreeFoam.readCsvData(dataFile)
        data = readData[1:]
        #エラーチェック
        stat = csvErrorCheck(data)
        if stat != "OK":
            return
        #読み込んだデータをmaterialsにセット
        materials = self.constantSolidPhysical[:]
        for i in range(len(self.solidRegions)):
            region = self.solidRegions[i]
            for reg, mat in data:
                if region == reg:
                    materials[i] = mat
                    break
        #データをセット
        listStore = self.tree_setMaterial.get_model()
        for i in range(len(self.solidRegions)):
            region = self.solidRegions[i]
            matReq = materials[i]
            listStore[i][3] = matReq
            if self.constantSolidPhysical[i] != matReq:
                fileName = self.solidPhysicalDBdir + "/" + matReq
                pasteName = self.caseDir + "/constant/" + region + "/" + self.solidProperties
                shutil.copy(fileName, pasteName)
        title = _("csvによる材料データセット")
        mess = _("csvfileを使って材料を各regionに設定しました。")
        self.okDialog(title, mess)

    #
    #  outputCsvFile
    #----------------
    def outputCsvFile(self):
        """ csvFileを出力する。"""
        #file名を取得
        title = _("file名の入力")
        mess = _("出力するcsvファイルのファイル名を入力してください。")
        text = "regionMaterial.csv"
        okArgs = [self.outputCsvFile_run]
        dialog = inputTextDDialog.getInputText(title, mess, text, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return

    #  outputCsvFile_run
    def outputCsvFile_run(self, text):
        fileName = self.caseDir + "/" + text
        #出力データの作成
        cont = "region,material\n"
        for i in range(len(self.solidRegions)):
            material = self.constantSolidPhysical[i]
            region = self.solidRegions[i]
            cont += ",".join([region, material]) + "\n"
        #fileを保存
        f = open(fileName, "w"); f.write(cont); f.close()
        title = _(u"csvFileの作成")
        message = _(u"現在の材料設定（region,material）を出力しました。")
        self.okDialog(title, message)

    #
    #  editRegionMaterial
    #---------------------
    def editRegionMaterial(self):
        """ regionに設定されている材料fileを編集する"""
        selRegions = self.getSelectedFilesInTreeList(self.tree_setMaterial)
        if len(selRegions) == 0:
            title = _(u"エラー")
            mess = _(u"regionの材料名を選択してください。")
            self.errDialog(title, mess)
            return
        #fileを開く
        fileDir = map(lambda x: self.caseDir + "/constant/" + x, selRegions)
        properties = self.solidProperties.split("_")[0]
        fileNames = map(lambda x: x + "/" + properties, fileDir)
        pyTreeFoam.run(self.caseDir).editor(fileNames)

    #
    #  renameRegionMaterial
    #-----------------------
    def renameRegionMaterial(self):
        """ regionに設定されている材料名を変更する"""
        selRegions = self.getSelectedFilesInTreeList(self.tree_setMaterial)
        if len(selRegions) == 0:
            title = _(u"エラー")
            mess = _(u"regionの材料名を選択してください。")
            self.errDialog(title, mess)
            return
        elif len(selRegions) != 1:
            title = _(u"エラー")
            mess = _(u"複数のregion材料名が選択されています。") + "\n"
            mess += _("  1ヶのregion材料名を選択してください。")
            self.errDialog(title, mess)
            return
        selRegion = selRegions[0]
        propName = self.solidProperties.split("_")[0]
        #fileName = self.caseDir + "/constant/" + selRegion + "/" + self.solidProperties
        fileName = self.caseDir + "/constant/" + selRegion + "/" + propName
        material = self.getMaterial(fileName)
        title = _("材料名の変更")
        mess = _("新しい材料名を入力してください。")
        okArgs = [self.renameRegionMaterial_run, selRegion, fileName]
        dialog = inputTextDDialog.getInputText(title, mess, material, funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return
        
    #  renameRegionMaterial_run
    def renameRegionMaterial_run(self, newMaterial, selRegion, fileName):
        self.setMaterial(fileName, newMaterial)
        listStore = self.tree_setMaterial.get_model()
        for i in range(len(listStore)):
            region = listStore[i][1]
            if region == selRegion:
                self.constantSolidPhysical[i] = newMaterial
                listStore[i][3] = newMaterial

    #
    #  saveMaterialDB
    #-----------------
    def saveMaterialDB(self):
        """ regionに設定されている材料名をDBに登録する。"""
        selRegions = self.getSelectedFilesInTreeList(self.tree_setMaterial)
        if len(selRegions) == 0:
            title = _(u"エラー")
            mess = _(u"regionの材料名を選択してください。")
            self.errDialog(title, mess)
            return
        #材料を登録
        propName = self.solidProperties.split("_")[0]
        #fileNames = list(map(lambda x: self.caseDir + "/constant/" + x + "/" + self.solidProperties, selRegions))
        fileNames = list(map(lambda x: self.caseDir + "/constant/" + x + "/" + propName, selRegions))
        materials = list(map(self.getMaterial, fileNames))
        if "-" in materials:
            title = _(u"エラー")
            mess = _(u"材料名が設定されていないregionが選択されています。")
            self.errDialog(title, mess)
            return
        for i in range(len(fileNames)):
            material = materials[i]
            shutil.copy(fileNames[i], self.solidPhysicalDBdir + "/" + material)
        allMaterials = pyTreeFoam.getFileNames(self.solidPhysicalDBdir)
        allMaterials.sort()
        listStore = self.tree_matDB.get_model()
        listStore.clear()
        for material in allMaterials:
            if os.path.islink(self.solidPhysicalDBdir + "/" + material):
                icon = self.iconFileLink
            else:
                icon = self.iconFile
            listStore.append([icon, material])

    #
    #  clearMterialFromRegion
    #-------------------------
    def clearMterialFromRegion(self):
        """ 現設定の材料名をクリアする。
        thermoPhysicalProperties内のmaterial行を削除する。"""
        selRegions = self.getSelectedFilesInTreeList(self.tree_setMaterial)
        if len(selRegions) == 0:
            title = _("エラー")
            mess = _("regionを選択してください。")
            self.errDialog(title, mess)
            return
        #material行を削除
        propName = self.solidProperties.split("_")[0]
        for i in range(len(selRegions)):
            #fileName = self.caseDir + "/constant/" + selRegions[i] + "/" + self.solidProperties
            fileName = self.caseDir + "/constant/" + selRegions[i] + "/" + propName
            self.clearMaterial(fileName)
        #再表示
        listStore = self.tree_setMaterial.get_model()
        for i in range(len(self.solidRegions)):
            region = self.solidRegions[i]
            #fileName = self.caseDir + "/constant/" + region + "/" + self.solidProperties
            fileName = self.caseDir + "/constant/" + region + "/" + propName
            material = self.getMaterial(fileName)
            self.constantSolidPhysical[i] = material
            listStore[i][3] = material

    #
    #  editMaterialDB
    #-----------------
    def editMaterialDB(self):
        """ 選択されたDB内の材料fileを編集する。"""
        selNames = self.getSelectedFilesInTreeList(self.tree_matDB)
        if len(selNames) == 0:
            title = _(u"エラー")
            mess = _(u"材料名を選択してください。")
            self.errDialog(title, mess)
            return
        fileDir = self.solidPhysicalDBdir
        fileNames = list(map(lambda x: fileDir + "/" + x, selNames))
        pyTreeFoam.run(self.caseDir).editor(fileNames)

    #
    #  openMaterialDBfolder
    #----------------------
    def openMaterialDBfolder(self):
        """ materialDBfolderを開く"""
        openDir = self.solidPhysicalDBdir
        pyTreeFoam.run(openDir).fileManager()

    #
    #  setMarerialReqDC
    #-------------------
    def setMarerialReqDC(self, pos):
        """ ダブルクリックでregionに材料を設定する。"""
        treeList = GtkParts.treeListIcon(self.tree_matDB)
        items = treeList.getSelectedItems()
        if len(items) == 0:
            title = _("エラー")
            mess = _("材料名が選択されていません。") + "\n"
            mess += _("材料DB内の材料名を選択してください")
            self.errDialog(title, mess)
            return
        listSelection = self.tree_setMaterial.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        path = pathes[0]
        #colNoを取得
        for i in range(len(self.tree_setMaterial.get_columns())):
            #cellサイズを取得
            # (rect.x 〜 rect.x + rect.width)
            rect = self.tree_setMaterial.get_cell_area(path, self.tree_setMaterial.get_column(i))
            if pos[0] < rect.x + rect.width:
                break
        colNo = i
        #colNoに応じた処理
        treeIter = listStore.get_iter(path)
        region = listStore.get_value(treeIter, 1)
        if colNo == 0:
            #regionをダブルクリック
            regionDir = self.caseDir + "/constant/" + region
            pyTreeFoam.run(regionDir).fileManager()
        elif colNo == 1:
            #設定材料名をダブルクリック
            idx = self.solidRegions.index(region)
            selMat = self.getSelectedFilesInTreeList(self.tree_matDB)[0]
            self.constantSolidPhysical[idx] = selMat
            listStore[idx][3] = selMat
            propName = self.solidProperties.split("_")[0]
            #pasteName = "/".join([self.caseDir, "constant", region, self.solidProperties])
            pasteName = "/".join([self.caseDir, "constant", region, propName])
            fileName = self.solidPhysicalDBdir + "/" + selMat
            shutil.copy(fileName, pasteName)


    #
    #  iniTreeList
    #--------------
    def iniTreeList(self):
        """ Tree関係を初期化"""
        #TreeViewを初期化
        self.iniTreeView_multiple(self.tree_time)
        self.iniTreeView_multiple(self.tree_constant)
        self.iniTreeView_multiple(self.tree_system)
        #treeListをMULTIPLEで初期化
        self.iniTreeList_Multiple(self.tree_timeFluidFile)
        self.iniTreeList_Multiple(self.tree_timeSolidFile)
        self.iniTreeList_Multiple(self.tree_constantFluidFile)
        self.iniTreeList_Multiple(self.tree_constantSolidFile)
        self.iniTreeList_Multiple(self.tree_systemFluidFile)
        self.iniTreeList_Multiple(self.tree_systemSolidFile)
        #treeListをSINGLEで初期化
        self.iniTreeList_Single(self.tree_timeFluidRegion)
        self.iniTreeList_Single(self.tree_timeSolidRegion)
        self.iniTreeList_Single(self.tree_constantFluidRegion)
        self.iniTreeList_Single(self.tree_constantSolidRegion)
        self.iniTreeList_Single(self.tree_systemFluidRegion)
        self.iniTreeList_Single(self.tree_systemSolidRegion)
        #材料設定の初期化
        self.iniTreeList_setMaterial()
        self.iniTreeList_matDB()

    def iniTreeView_multiple(self, treeView):
        """ treeViewをmultiSelectionで初期化"""
        treeSelection = treeView.get_selection()
        treeSelection.set_mode(Gtk.SelectionMode.MULTIPLE)
        pixbuf = GdkPixbuf.Pixbuf
        treeStore = Gtk.TreeStore(pixbuf, str)      #icon, dir
        treeView.set_model(model=treeStore)
        treeView.set_headers_visible(False)
        cell = Gtk.CellRendererText()
        icon = Gtk.CellRendererPixbuf()
        col = Gtk.TreeViewColumn()
        col.pack_start(icon, False)
        col.pack_start(cell, False)
        col.add_attribute(icon, "pixbuf", 0)
        col.add_attribute(cell, "text", 1)
        treeView.append_column(col)

    def iniTreeList_Multiple(self, treeList):
        """ treeListをMULTIPLEで初期化する"""
        GtkParts.treeListIcon(treeList).create()

    def iniTreeList_Single(self, treeList):
        """ treeListをSINGLEselectionで初期化"""
        GtkParts.treeListIcon(treeList).create(multi=False)

    def iniTreeList_setMaterial(self):
        """ self.tree_setMaterialの初期化"""
        pixbuf = GdkPixbuf.Pixbuf
        prop = self.solidProperties.split("_")[0]
        GtkParts.treeListCols(self.tree_setMaterial).create(
            model = (pixbuf, str, pixbuf, str),
            multi = True,
            header = True,
            #headerConts = [_("region名"), _("設定材料")+"("+self.solidProperties+")"]
            headerConts = [_("region名"), _("設定材料")+"("+prop+")"]
            )

    def iniTreeList_matDB(self):
        """ self.tree_matDBの初期化"""
        GtkParts.treeListIcon(self.tree_matDB).create(multi=False, header=True, headerCont="material")

    #
    #  getTimeFolderForOperation
    #----------------------------
    def getTimeFolderForOperation(self):
        """ 操作対象のtimeFolderを取得し、設定する。"""
        #iniTimeを取得
        case = pyTreeFoam.case(self.caseDir)
        self.iniTime = case.getTimeFolders()[0]
        #pageLabeを設定
        page = self.noteBook.get_nth_page(0)
        self.noteBook.set_tab_label_text(page, self.iniTime)

    #
    #  setFileNameMaterialToListBox
    #-------------------------------
    def setFileNameMaterialToListBox(self):
        """ file名、materialをlistBoxに表示する"""
        #setFileName
        self.setFileNameToListBox()     #time, constant, systemPage
        self.setMaterialToListBox()     #materialPage

    #
    #  setFileNameToListBox
    def setFileNameToListBox(self):
        """ time, constant, system Pageにfolder, fileをtreeViewに表示"""
        #timePage
        self.setItemToTreeView(self.tree_time, self.timeFiles)
        wkDir = self.caseDir + "/" + self.iniTime + "/" + self.fluidRegions[0]
        self.setItemToTreeListFil(self.tree_timeFluidFile, self.timeFluidFiles, wkDir)
        wkDir = self.caseDir + "/" + self.iniTime
        self.setItemToTreeListFol(self.tree_timeFluidRegion, self.fluidRegions, wkDir)
        wkDir = self.caseDir + "/" + self.iniTime + "/" + self.solidRegions[0]
        self.setItemToTreeListFil(self.tree_timeSolidFile, self.timeSolidFiles, wkDir)
        wkDir = self.caseDir + "/" + self.iniTime
        self.setItemToTreeListFol(self.tree_timeSolidRegion, self.solidRegions, wkDir)
        #constantPage
        self.setItemToTreeView(self.tree_constant, self.constantFiles)
        wkDir = self.caseDir + "/constant/" + self.fluidRegions[0]
        self.setItemToTreeListFil(self.tree_constantFluidFile, self.constantFluidFiles, wkDir)
        wkDir = self.caseDir + "/constant"
        self.setItemToTreeListFol(self.tree_constantFluidRegion, self.fluidRegions, wkDir)
        wkDir = self.caseDir + "/constant/" + self.solidRegions[0]
        self.setItemToTreeListFil(self.tree_constantSolidFile, self.constantSolidFiles, wkDir)
        wkDir = self.caseDir + "/constant"
        self.setItemToTreeListFol(self.tree_constantSolidRegion, self.solidRegions, wkDir)
        #syatemPage
        self.setItemToTreeView(self.tree_system, self.systemFiles)
        wkDir = self.caseDir + "/system/" + self.fluidRegions[0]
        self.setItemToTreeListFil(self.tree_systemFluidFile, self.systemFluidFiles, wkDir)
        wkDir = self.caseDir + "/system"
        self.setItemToTreeListFol(self.tree_systemFluidRegion, self.fluidRegions, wkDir)
        wkDir = self.caseDir + "/system/" + self.solidRegions[0]
        self.setItemToTreeListFil(self.tree_systemSolidFile, self.systemSolidFiles, wkDir)
        wkDir = self.caseDir + "/system"
        self.setItemToTreeListFol(self.tree_systemSolidRegion, self.solidRegions, wkDir)

    #
    #  setMaterialToListBox
    def setMaterialToListBox(self):
        """ 材料設定Pageにdataをセット"""
        #材料設定Page
        items = []
        for i in range(len(self.solidRegions)):
            region = self.solidRegions[i]
            setName = self.constantSolidPhysical[i]
            if os.path.islink(self.caseDir + "/constant/" + region):
                iconFol = self.iconFolderLink
            else:
                iconFol = self.iconFolder
            if os.path.islink(self.caseDir + "/constant/" + region + "/" + self.solidProperties):
                iconFil = self.iconFileLink
            else:
                iconFil = self.iconFile
            items.append([iconFol, region, iconFil, setName])
        GtkParts.treeListCols(self.tree_setMaterial).setItems(items)

        #  tree_matDBにセット
        wkDir = self.solidPhysicalDBdir
        self.setItemToTreeListFil(self.tree_matDB, self.solidPhysicalDB, wkDir)
        #  solidPhysicalDBdirをセット
        self.entry_matDB.set_text(self.solidPhysicalDBdir)

    def setItemToTreeListFil(self, treeList, items, wkDir):
        setItems = []
        for file in items:
            if os.path.islink(wkDir + "/" + file):
                icon = self.iconFileLink
            else:
                icon = self.iconFile
            setItems.append([icon, file])
        GtkParts.treeListIcon(treeList).setItems(setItems)

    def setItemToTreeListFol(self, treeList, items, wkDir):
        setItems = []
        for name in items:
            if os.path.islink(wkDir + "/" + name):
                icon = self.iconFolderLink
            else:
                icon = self.iconFolder
            setItems.append([icon, name])
        GtkParts.treeListIcon(treeList).setItems(setItems)

    def setItemToTreeView(self, treeView, treeData):
        """ 指定されたtreeViewにtreeDataをセットする"""
        #dataSet
        _rootName = treeData[0]
        items = treeData[1]
        treeStore = treeView.get_model()
        treeStore.clear()
        #treeを作成する
        self.addTreeNodes(treeStore, None, items)

    def addTreeNodes(self, treeStore, parentIter, items):
        """ treeNodesを作成する"""
        for item in items:
            if type(item) == str:
                #icon設定
                if item == "." or item == "fluidRegions" or item == "solidRegions":
                    icon = self.iconFolder
                else:
                    #icon = self.iconBlank
                    icon = self.iconFile
                #treeStore設定
                newIter = treeStore.append(parentIter, [icon, item])
            else:
                #icon設定
                if item[0] == "." or item[0] == "fluidRegions" or item[0] == "solidRegions":
                    icon = self.iconFolder
                else:
                    #icon = self.iconBlank
                    icon = self.iconFile
                #treeStore設定
                newIter = treeStore.append(parentIter, [icon, item[0]])
                #subFolderを設定
                self.addTreeNodes(treeStore, newIter, item[1])

    #
    #  selectFirstRegion
    #--------------------
    def selectFirstRegion(self):
        """ 最初のregionを選択する"""

        def selectRegion(treeList):
            GtkParts.treeList(treeList).selectItemNos([0])
            # selection = treeList.get_selection()
            # path = Gtk.TreePath(0)
            # selection.select_path(path)

        selectRegion(self.tree_timeFluidRegion)
        selectRegion(self.tree_timeSolidRegion)
        selectRegion(self.tree_constantFluidRegion)
        selectRegion(self.tree_constantSolidRegion)
        selectRegion(self.tree_systemFluidRegion)
        selectRegion(self.tree_systemSolidRegion)

    #
    #  checkMaterialDBDir
    #----------------------
    def checkMaterialDBDir(self):
        """ materialDBDirをチェックし、無ければ作成する。
        defaultの場所は、「~/OpenFOAM/multiRegionDB/solid」
        self.solidProperties, self.iniDBdataを確定 """
        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        #solidProperties取得
        #if OFversion <= "2.1.x" or OFversion[:3] == "ext":
        if (0.0 < numVer and numVer <= 2.1) or OFversion[:3] == "ext": 
            #OF-2.1以前、ext版
            self.solidProperties = "solidThermophysicalProperties"
        #elif OFversion <= "7" or OFversion[0] == "v":
        elif (0.0 < numVer and numVer <= 7.0) or OFversion[0] == "v":
            #OF-2.2以降〜OF-7まで、v系のOF
            self.solidProperties = "thermophysicalProperties"
        #elif OFversion >= "8":
        #elif numVer >= 8.0:
        elif (8.0 <= numVer and numVer < 10.0):
            #OF-8, 9
            self.solidProperties = "thermophysicalProperties_energy"
        elif (10.0 <= numVer):
            #OF-10以降
            self.solidProperties = "physicalProperties_10"
        #現在の設定場所を確認する
        fileName = self.multiRegion_data
        setDir = ""
        if len(glob.glob(fileName)) != 0:
            f = open(fileName); cont = f.readlines(); f.close()
            for line in cont:
                if line.split()[0] == "solidPhysicalDBdir":
                    setDir = line.split()[1]
                    break

        setFolder = setDir.split("/")[-1]
        if (len(glob.glob(setDir)) == 0 or setFolder != self.solidProperties):
            #materialDBDirが無い場合、defaultの場所に設定する
            homeDir = os.environ["HOME"]
            DBdir = homeDir + "/OpenFOAM"
            if len(glob.glob(DBdir)) == 0:
                os.mkdir(DBdir)
            DBdir += "/multiRegionDB"
            if len(glob.glob(DBdir)) == 0:
                os.mkdir(DBdir)
            DBdir += "/solid"
            if len(glob.glob(DBdir)) == 0:
                os.mkdir(DBdir)
            properties = self.solidProperties
            DBdir += "/" + properties
            if len(glob.glob(DBdir)) == 0:
                os.mkdir(DBdir)
            setDir = DBdir

        self.iniDBdata = os.getenv("TreeFoamPath") + "/data/multiRegionDB/solid/" + self.solidProperties
        if len(glob.glob(setDir + "/*")) == 0:
            #defaultの設定場所からコピーして作成する
            sourceDir = self.iniDBdata
            #shutil.copy(sourceDir, setDir)
            files = glob.glob(sourceDir + "/*")
            for file in files:
                shutil.copy(file, setDir)
            #書き込めなかった場合、treeFoam内のdirを設定
            if len(glob.glob(setDir + "/*")) == 0:
                setDir = self.iniDBdata
        self.setSolidDirMultiRegion_data(setDir)

    #
    #  getSolidDirMultiRegion_data
    def getSolidDirMultiRegion_data(self):
        """ multiRegion_dataファイルからデータを取得する"""
        fileName = self.multiRegion_data
        try:
            f = open(fileName); cont = f.readlines(); f.close()
            for line in cont:
                p = line.find("solidPhysicalDBdir")
                if p >= 0:
                    DBdir = line.split()[1]
                    break
            if len(glob.glob(DBdir)) == 0:
                DBdir = self.iniDBdata
        except:
            DBdir = self.iniDBdata
        #OFversionに合わせDBdirを設定
        wdir = DBdir.split("/")[:-1]
        DBdir = "/".join(wdir) + "/" + self.solidProperties
        return DBdir

    #
    #  setSolidDirMultiRegion_data
    def setSolidDirMultiRegion_data(self, selDir):
        """ multiRegion_dataファイルに保存する"""
        fileName = self.multiRegion_data
        if len(glob.glob(fileName)) != 0:
            f = open(fileName); cont = f.readlines(); f.close()
            i = 0
            while i < len(cont):
                p = cont[i].find("solidPhysicalDBdir")
                if p >= 0:
                    cont[i] = "solidPhysicalDBdir " + selDir + "\n"
                    break
                i += 1
        else:
            cont = ["solidPhysicalDBdir " + selDir + "\n"]
        f = open(fileName, "w")
        for line in cont:
            f.write(line)
        f.close()

    #
    #  getSolidPhysicalDB
    def getSolidPhysicalDB(self, DBdir):
        """ matDB内のfilesを取得する。"""

        def setDBFilesToDBDir(DBdir):
            folder = DBdir.split("/")[-1]
            iniDBdataDir = "/".join(self.iniDBdata.split("/")[:-1]) + "/" + folder
            comm = "cp " + iniDBdataDir + "/* " + DBdir
            print(comm)
            pyTreeFoam.run(self.caseDir).command(comm)
            #os.system(comm)

        def isSolidThermophysicalProperties(fileName):
            """ folder名（thermophysicalPropertes）が
            本文に存在するかどうか確認する。"""
            f = open(fileName); cont = f.read(); f.close()
            #folder名の20文字を取得してチェック
            name = fileName.split("/")[-2]
            word = name.split("_")[0]
            p = cont.find(word)
            if p >= 0:
                return True
            else:
                return False

        if len(glob.glob(DBdir)) != 0:
            #[folders, fileNames] = pyTreeFoam.getFoldersFiles(DBdir)
            fileNames = pyTreeFoam.getFileNames(DBdir)
            if len(fileNames) == 0:
                #fileが無かった場合、fileをコピーする。
                setDBFilesToDBDir(DBdir)
            files = []
            for fileName in fileNames:
                files.append(fileName.split("/")[-1])
            solidPhysicalDB = self.checkFileName(files)
            files = []
            for fileName in solidPhysicalDB:
                fileDir = DBdir + "/" + fileName
                if isSolidThermophysicalProperties(fileDir) == True:
                    files.append(fileName)
            solidPhysicalDB = self.checkFileName(files)
            solidPhysicalDB.sort()
            return solidPhysicalDB

    #
    #  getMaterial
    def getMaterial(self, fileName):
        """ material名を取得する"""
        f = open(fileName); cont = f.read(); f.close()
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.getKeywordPointer("material", 0)
        if p < 0:
            material = "-"
        else:
            while p < len(cont) and cont[p] != " ":
                p += 1
            (material, p) = contOp.getKeyword(p)
        return material

    #
    #  setMaterial
    def setMaterial(self, fileName, material):
        """ marerial名を設定する。"""
        f = open(fileName); cont = f.read(); f.close()
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        pp = contOp.getKeywordPointer("material", p)
        if pp < 0:
            #定義されていない場合
            (keyword, p) = contOp.getKeyword(p)
            ps = p - len(keyword)
            header = cont[:ps]
            footer = cont[ps:]
            newLine = "material    " + material + ";\n\n"
            newCont = header + newLine + footer
        else:
            #材料名が定義されている場合
            newLine = "material    " + material + ";"
            (newCont, p, _keyword) = contOp.replace1lineKeyword(p, newLine)
        f = open(fileName, "w"); f.write(newCont); f.close()

    #
    #  clearMaterial
    def clearMaterial(self, fileName):
        """ material行を削除する"""
        f = open(fileName); cont = f.read(); f.close()
        contOp = pyTreeFoam.strOp(cont)
        p = contOp.skipFoamFile()
        pp = contOp.getKeywordPointer("material", p)
        if pp < 0:
            #定義されていない場合
            return
        else:
            #材料名が定義されている場合
            header = cont[:pp]
            (lineCont, p, _kind) = contOp.get1line(pp)   #material行を取得
            (lineCont, p, _kind) = contOp.get1line(p)    #materialの次の行を取得
            ps = p - len(lineCont)
            footer = cont[ps:]
            newCont = header + footer
            f = open(fileName, "w"); f.write(newCont); f.close()

    #
    #  makeTreeData
    def makeTreeData(self, fluidFiles, solidFiles, files):
        """ TreeDataを作成する"""
        treeData = []
        treeData.append("root")
        items = []
        items.append(".")
        fluid = ["fluidRegions", fluidFiles]
        solid = ["solidRegions", solidFiles]
        items.append(fluid)
        items.append(solid)
        for name in files:
            items.append(name)
        treeData.append(items)
        return treeData

    #
    #  checkFileName
    def checkFileName(self, files):
        """ 有効なfileのみを戻す"""
        newFiles = []
        for name in files:
            if (name[0] != "." and name[-1] != "~"
                    and name != "README"):
                newFiles.append(name)
        return newFiles

    #
    #  getTreeDataFromDir
    def getTreeDataFromDir(self, parentDir):
        """ parentDir内のTreeDataを作成する"""
        files = pyTreeFoam.getFileNames(parentDir)
        parFiles = self.checkFileName(files)
        parFiles.sort()
        files = pyTreeFoam.getFileNames(parentDir + "/fluidRegions")
        parFluidFiles = self.checkFileName(files)
        parFluidFiles.sort()
        files = pyTreeFoam.getFileNames(parentDir + "/solidRegions")
        parSolidFiles = self.checkFileName(files)
        parSolidFiles.sort()
        treeData = self.makeTreeData(parFluidFiles, parSolidFiles, parFiles)
        return treeData

    #
    #  initialize
    #-------------
    def initialize(self):
        """ 初期化"""
        
        def checkFluidSolidFolder(checkDir):
            """ fluidRegions, solidRegions folderをチェックし、無ければ作成する"""
            if len(glob.glob(checkDir + "/fluidRegions")) == 0:
                os.mkdir(checkDir + "/fluidRegions")
            if len(glob.glob(checkDir + "/solidRegions")) == 0:
                os.mkdir(checkDir + "/solidRegions")

        #regionの取得
        case = pyTreeFoam.case(self.caseDir)
        OFversion = self.configDict["OFversion"]
        (self.fluidRegions, self.solidRegions) = case.getZonesInRegionProperties(OFversion)
        if self.fluidRegions == "":
            mess = _("regionPropertiesが存在しません")
            print(mess)
            self.warningDialog(_("警告"), mess)
            return
        if len(self.fluidRegions) == 0 or len(self.solidRegions) == 0:
            mess = _("流体 or 固体のregionが定義されていません。") 
            self.warningDialog(_("警告"), mess)
            exit()
            #return
        #timeFolderを取得
        iniTime = self.iniTime
        timeDir = self.caseDir + "/" + iniTime
        checkFluidSolidFolder(timeDir)
        self.timeFiles = self.getTreeDataFromDir(timeDir)
        self.timeFluidFiles = case.getFieldNames(iniTime, self.fluidRegions[0])
        self.timeFluidFiles.sort()
        self.timeSolidFiles = case.getFieldNames(iniTime, self.solidRegions[0])
        self.timeSolidFiles.sort()
        #constantFileを取得
        constDir = self.caseDir + "/constant"
        checkFluidSolidFolder(constDir)
        self.constantFiles = self.getTreeDataFromDir(constDir)
        files = pyTreeFoam.getFileNames(constDir + "/" + self.fluidRegions[0])
        self.constantFluidFiles = self.checkFileName(files)
        self.constantFluidFiles.sort()
        files = pyTreeFoam.getFileNames(constDir + "/" + self.solidRegions[0])
        self.constantSolidFiles = self.checkFileName(files)
        self.constantSolidFiles.sort()
        #systemFileを取得
        sysDir = self.caseDir + "/system"
        checkFluidSolidFolder(sysDir)
        self.systemFiles = self.getTreeDataFromDir(sysDir)
        files = pyTreeFoam.getFileNames(sysDir + "/" + self.fluidRegions[0])
        self.systemFluidFiles = self.checkFileName(files)
        self.systemFluidFiles.sort()
        files = pyTreeFoam.getFileNames(sysDir + "/" + self.solidRegions[0])
        self.systemSolidFiles = self.checkFileName(files)
        self.systemSolidFiles.sort()
        #DBdata取得
        self.solidPhysicalDBdir = self.getSolidDirMultiRegion_data()
        self.solidPhysicalDB = self.getSolidPhysicalDB(self.solidPhysicalDBdir)
        self.solidPhysicalDB.sort()
        #各regionのsolidPhysicalFile取得
        self.constantSolidPhysical = []
        properties = self.solidPhysicalDBdir.split("/")[-1]
        propNames = properties.split("_")[0]
        for solidRegion in self.solidRegions:
            #fileName = self.caseDir + "/constant/" + solidRegion + "/" + properties
            fileName = self.caseDir + "/constant/" + solidRegion + "/" + propNames
            if len(glob.glob(fileName)) != 0:
                material = self.getMaterial(fileName)
            else:
                material = ""
            self.constantSolidPhysical.append(material)


    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    def warningDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.warningDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, 
            parentWin=self.mainWindow, funcOk=funcOk)
        dialog.show()

    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, 
            funcCancel=funcCancel, parentWin=self.mainWindow)
        dialog.show()

    def openFileDialog(self, title, currDir, funcOk=[], funcCancel=[], selName="", fileFilters=[]):
        dialog = unvDlg.openFileDialog(
            title, currDir, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow, filterSets=fileFilters)
        dialog.show()


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = os.sys.argv[1]         #caseDir
    openPage = os.sys.argv[2]        #起動時にnotebookを表示させるpage
    winApp = windowApp(caseDir, openPage)
    winApp.main()
