#!/usr/bin/python3
#  coding: utf-8
#
#   fileOperationInProcessorDialog.py
#
#       並列計算時のprocessor*フォルダ内のファイル操作（コピー、削除）
#
#   12/11/24    新規作成
#      12/02    processor数を表示
#   13/01/08    ボタン形状変更
#      01/29    数字ﾌｫﾙﾀﾞをsortするように修正
#      02/04    バグ修正（processor内のfaile削除時、ｴﾗｰmsg出ていた）
#      06/07    file表示で日本語可能に修正(日本語でｴﾗｰ発生していた）
#      10/15    国際化のため、修正
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   15/07/17    dialogの起動方法を修正
#   20/01/08    GTK+3, python3用に大幅修正
#      04/21    多言語化対応
#      07/28    getFoldersFiles:数値folderを数値でsortして返す様に修正
#      12/21    getAllFoldersFiles:processor0フォルダ有無確認を追加
#               起動時にエラー発生する為。
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GdkPixbuf

import os, sys
import glob
import shutil

import pyTreeFoam
import GtkParts

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()


#------------------
#  windowApp class
#------------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"fileOperationInProcessorDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobjectを取得
        self.setGtkObject()
        #属性の設定
        self.caseDir = caseDir
        self.caseItems = [[], []]   #[[folders], [files]]
        self.procItems = [[], []]
        self.caseFolderDict = {}
        self.caseFileDict = {}
        self.procFolderDict = {}
        self.procFileDict = {}
        iconDir = os.getenv("TreeFoamPath") + "/icons"
        iconFile = GdkPixbuf.Pixbuf.new_from_file
        self.iconFolder = iconFile(iconDir + "/folder16.png")
        self.iconFile = iconFile(iconDir + "/text8.png")
        self.iconFolderLink = iconFile(iconDir + "/folderLink16.png")
        self.iconFileLink = iconFile(iconDir + "/textLink8.png")
        #GUIの初期化
        self.iniGui()
        #treeListの初期化
        self.iniTreeList()


    #
    #  main
    #--------
    def main(self):
        #case内のfolders, filesを取得
        self.getAllFoldersFiles()
        #treeListに表示
        self.setCaseItemsToTreeList()
        self.setProcItemsToTreeList()
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        Gtk.main_quit()

    #----- event handler ----------------
    def onCopyItems(self, event):
        self.copyItems()
    def onDeleteInCase(self, event):
        self.deleteInCase()
    def onDeleteInProc(self, event):
        self.deleteInProc()
    def onClose(self, event):
        self.close()

    #
    #  setGtkObject
    #---------------
    def setGtkObject(self):
        self.tree_case = self.builder.get_object("tree_case")
        self.tree_proc = self.builder.get_object("tree_proc")
        self.label_nProcs = self.builder.get_object("label_nProcs")

    #
    #  iniGui
    #----------
    def iniGui(self):
        """ GUIの初期化"""
        self.mainWindow.set_title(_("processor内の操作"))
        nProcs = len(glob.glob(self.caseDir + "/processor*"))
        self.label_nProcs.set_text(_("processor数: ") + str(nProcs) + _("ヶ"))

    #
    #  iniTreeList
    #--------------
    def iniTreeList(self):
        """ treeListの初期化"""
        GtkParts.treeListIcon(self.tree_case).create()
        GtkParts.treeListIcon(self.tree_proc).create()

    #
    #  getAllFoldersFiles
    #---------------------
    def getAllFoldersFiles(self):
        (self.caseItems, dicts) = self.getFoldersFiles(self.caseDir)
        (self.caseFolderDict, self.caseFileDict) = dicts[:] 
        if os.path.exists(self.caseDir + "/processor0") == True:
            (self.procItems, dicts) = self.getFoldersFiles(self.caseDir + "/processor0")
            (self.procFolderDict, self.procFileDict) = dicts[:]

    #
    #  getFoldersFiles
    #-------------------
    def getFoldersFiles(self, folderDir):
        """ folderDir内のfolder名とfile名を取得して返す。"""
        #folderを取得
        names = pyTreeFoam.getFolderNames(folderDir)
        names.sort()
        folders = []
        folderDict = {}
        for name in names:
            if not(name == "constant" or name == "system"
                    or name[:len("processor")] == "processor" or name[0] == "."):
                if os.path.islink(folderDir + "/" + name) == True:
                    folderDict[name] = "link"
                else: 
                    folderDict[name] = ""
                folders.append(name)
        #数値foldersをsortする。
        numFolders = []
        strFolders = []
        for folder in folders:
            try:
                num = float(folder)
                numFolders.append([num, folder])
            except:
                strFolders.append(folder)
        numFolders.sort()
        newFolders = list(map(lambda x: x[1], numFolders))
        strFolders.sort()
        folders = newFolders + strFolders
        #filesを取得
        names = pyTreeFoam.getFileNames(folderDir)
        names.sort()
        files = []
        fileDict = {}
        for name in names:
            if not(name[0] == "." or name[0] == "~"):
                if os.path.islink(folderDir + "/" + name) == True:
                    fileDict[name] = "link"
                else:
                    fileDict[name] = ""
                files.append(name)
        foldersFiles = [folders, files]
        dicts = [folderDict, fileDict]
        return (foldersFiles, dicts)

    #
    #  setCaseItemsToTreeList
    #-------------------------
    def setCaseItemsToTreeList(self):
        """ caseDir内のfolder、fileをTreeListに表示する"""
        items = []
        for folder in self.caseItems[0]:
            if self.caseFolderDict[folder] == "link":
                icon = self.iconFolderLink
            else:
                icon = self.iconFolder
            item = [icon, folder]
            items.append(item)
        for file in self.caseItems[1]:
            if self.caseFileDict[file] == "link":
                icon = self.iconFileLink
            else:
                icon = self.iconFile
            item = [icon, file]
            items.append(item)
        GtkParts.treeListIcon(self.tree_case).setData(items)

    #
    #  setProcItemsToTreeList
    #-------------------------
    def setProcItemsToTreeList(self):
        """ processor内のfolder、fileをTreeListに表示する"""
        items = []
        for folder in self.procItems[0]:
            if self.procFolderDict[folder] == "link":
                icon = self.iconFolderLink
            else:
                icon = self.iconFolder
            item = [icon, folder]
            items.append(item)
        for file in self.procItems[1]:
            if self.procFileDict[file] == "link":
                icon = self.iconFileLink
            else:
                icon = self.iconFile
            item = [icon, file]
            items.append(item)
        GtkParts.treeListIcon(self.tree_proc).setData(items)

    #
    #  copyItems
    #------------
    def copyItems(self):
        """ 選択しているitemをprocessorにコピーする"""
        #選択しているfolder、fileを取得
        selItems = GtkParts.treeListIcon(self.tree_case).getSelectedItems()
        selFolders = []
        selFiles = []
        for icon, name in selItems:
            if icon == self.iconFolder or icon == self.iconFolderLink:
                selFolders.append(name)
            else:
                selFiles.append(name)
        #copy開始
        pasteDirs = glob.glob(self.caseDir + "/processor*")
        for folder in selFolders:
            folderDir = self.caseDir + "/" + folder
            for pasteDir in pasteDirs:
                pasteFolder = pasteDir + "/" + folder
                shutil.copytree(folderDir, pasteFolder, symlinks=True)
        for file in selFiles:
            fileDir = self.caseDir + "/" + file
            for pasteDir in pasteDirs:
                pasteFile = pasteDir + "/" + file
                shutil.copy(fileDir, pasteFile)
        #treeListを再表示
        (self.procItems, dicts) = self.getFoldersFiles(self.caseDir + "/processor0")
        self.procFolderDict, self.procFileDict = dicts
        self.setProcItemsToTreeList()
        #copyした項目を選択表示
        GtkParts.treeListIcon(self.tree_proc).selectItems(selItems)

    #
    #  deleteInCase
    #---------------
    def deleteInCase(self):
        """ case内の選択したfolder、fileを削除する"""
        selItems = GtkParts.treeListIcon(self.tree_case).getSelectedItems()
        selFolders = []
        selFiles = []
        selFolderLinks = []
        for icon, name in selItems:
            if icon == self.iconFolder:
                selFolders.append(name)
            elif icon == self.iconFolderLink:
                selFolderLinks.append(name)
            else:
                selFiles.append(name)
        #削除開始
        for folder in selFolders:
            delFolderDir = self.caseDir + "/" + folder
            shutil.rmtree(delFolderDir)
        for file in selFiles:
            delFileDir = self.caseDir + "/" + file
            os.remove(delFileDir)
        for folder in selFolderLinks:
            delDir = self.caseDir + "/" + folder
            comm = "rm -rf " + delDir
            pyTreeFoam.run(self.caseDir).command(comm)
        #treeListを再表示
        (self.caseItems, dicts) = self.getFoldersFiles(self.caseDir)
        self.caseFolderDict, self.caseFileDict = dicts
        self.setCaseItemsToTreeList()

    #
    #  deleteInProc
    #---------------
    def deleteInProc(self):
        """ processor内から選択した項目（folder、file）を削除する"""
        selItems = GtkParts.treeListIcon(self.tree_proc).getSelectedItems()
        selFolders = []
        selFiles = []
        selFolderLinks = []
        for icon, name in selItems:
            if icon == self.iconFolder:
                selFolders.append(name)
            elif icon == self.iconFolderLink:
                selFolderLinks.append(name)
            else:
                selFiles.append(name)
        #削除開始
        procDirs = glob.glob(self.caseDir + "/processor*")
        for procDir in procDirs:
            for folder in selFolders:
                delFolderDir = procDir + "/" + folder
                shutil.rmtree(delFolderDir)
            for file in selFiles:
                delFileDir = procDir + "/" + file
                os.remove(delFileDir)
            for folder in selFolderLinks:
                delDir = procDir + "/" + folder
                comm = "rm -rf " + delDir
                pyTreeFoam.run(self.caseDir).command(comm)
        #TreeList再表示
        (self.procItems, dicts) = self.getFoldersFiles(self.caseDir + "/processor0")
        self.procFolderDict, self.procFileDict = dicts
        self.setProcItemsToTreeList()




if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    #os.environ["TreeFoamPath"] = "/home/caeuser/TreeFoam-glade3"
    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()

