#!/usr/bin/python3
# coding: utf-8
#
#   editStlFilesDialog.py
#
#       stlファイルのsolid名や倍率、face向き等を変更する
#
#   13/10/31    新規作成
#      11/02    binaryの場合は、solid名、サイズとも空白で表示
#      11/03    binaryの場合も表示、ascii変換追加
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/12/27    solid名の取得、変更方法を修正、stl結合ボタンを追加
#   15/01/10    mergeStlFile:バグ修正（stl結合時、cancelできない）
#      01/13    「face向き反転」「solid名設定」「stl内部結合」「folder開く」ﾎﾞﾀﾝ追加
#               toolTip追加、ﾀﾞﾌﾞﾙｸﾘｯｸの動作追加、「paraview起動」ボタン追加
#      01/15    setSolidName:バグ修正
#      06/27    windowのサイズをwxのバージョンによって方法を変える
#      07/14    winSizeの保存場所変更
#      07/17    dialog起動方法修正
#   16/05/09    binaryの読み込み方法を修正
#      08/28    windowサイズ設定を修正
#   19/02/01    windowサイズ設定を修正(wx.vrsion()のgtkVerを追加）
#               setSolidName:stl内のsolid名に「,」が含まれる事があるので、これに対応
#      05/21    copyFileToTriSurface:「mkdir」を「os.mkdir」に修正
#      09/13    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#      06/02    copyStlFiles, pasteStlFiles:copyPasteFolders.pyに変更。
#      07/13    axisRotation:軸周りの回転を追加
#               transform:移動を追加
#      07/23    stlViewerを追加。
#      07/25    onKeyPress,onKeyRelease:新規追加
#               pressMouseButton:ctrlKeyの処理を追加
#      07/27    afterSelectRow:threadで実行していたものをGlib.timeout_add
#               のtimerEventに変更。
#               keyPress:cursorKeyで操作できる様に修正
#      07/28    stl読込関連、有効数字取得をpyTreeFoamに移動。
#               各処理後にvtk表示を追加
#               vtk-7対応でrenderer.ResetCamera()に引数を追加した。
#      09/23    getStlFilesCont:getValue3digit→getValue3digit_shrinkに変更
#   22/01/07    vtkのimport方法を修正。
#      01/09    changeScale,binToAscii,transform:OF-9対応で修正。
#               surfaceTransformPointsコマンドがOF-9から変更された。
#      01/18    pressMouseButton:大幅修正。treeListのwidget操作処理を
#               defaultの処理を使うように修正。
#      02/03    vtkの表示方法を変更。
#               stl選択毎にresetCamera → 起動時のみresetCamea
#      03/26    setDataToListBox:指定folder内にstlファイルが存在しない場合、
#               エラーが発生していたので、修正。
#      07/17    createTransformCommLine:OF-10対応
#      08/17    stlCheck,stlSolidCheck:新しくstlのチェックを追加。
#      09/06    stlCheck,stlSolidChek:終了時にdialog表示を追加。
#      11/07    copyFilesToTriSurface:caseDirではない場合、copyせずを追加
#      11/25    changeScale:commandを出力する様に修正
#      11/29    transform:stlの中心座標を原点に移動するを追加。
#   24/08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/05    inputTextDDialog変更（funcOk, 親windowの設定）
#      08/22    起動時、window位置をmouseに合わせる様修正。
#      09/10    stlファイルのedgeをoutlineとして表示させる事を追加。
#      09/30    reloadVtk:reload時のedgeActor漏れあった為、追加。
#               changeScale,mergeStlFile:inputTextDDialog前後で関数を分割
#      10/01    treeList変数をself.treeListに変更。全itemが取得できない時があった為。
#      10/14    pasteStlFiles:selftreeList→self.treeListに修正。（バグ）
#   25/03/30    setVtkModel:vtk-9.4対応でGradientBackgroundOnを追加

#

import logFileCreater
logFileCreater.log()

import os, sys
print("importing vtk...")
sys.stdout.flush()
import checkVtkModule
isVtk = checkVtkModule.isVtk("editStlFiles")
if isVtk == True:
    import vtk
    from vtk.util import colors
    vtkVersion = vtk.vtkVersion.GetVTKSourceVersion().split()[-1]
    print("  imported vtk version " + vtkVersion)
else:
    print("error: could not import vtk !")
    exit()

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, Gdk, GdkPixbuf, GLib
import glob
import shutil

import pyTreeFoam
import universalDialogs as unvDlg
import inputTextDDialog
import GtkParts
import copyPasteFolders
import surfaceRotationDDialog
import surfaceTransformDDialog
import GtkVtkRenderWindowInteractor as gtkVtk

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

print("initializing...")

#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir, stlDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"editStlFilesDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #GUIのobject名を取得
        self.setGtkObject()
        #変数の設定
        self.caseDir = caseDir
        self.stlDir = stlDir
        self.clipboardFile = os.getenv("TreeFoamUserPath") + "/data/clipBoard"
        self.ctrlKeyFlag = 0
        self.shiftKeyFlag = 0
        #iconの設定
        iconDir = os.getenv("TreeFoamPath") + "/icons"
        iconFile = GdkPixbuf.Pixbuf.new_from_file
        self.iconFile = iconFile(iconDir + "/text8.png")
        self.iconFileLink = iconFile(iconDir + "/textLink8.png")
        #popupMenuItemsを設定
        self.setPopupMenuItems()
        #vtk関連
        self.maskEvent = True
        self.vtkSize = (300, 250)
        self.vtkInitialRun = True           #最初のvtk表示flag
        self.vtkObj = gtkVtk.GtkVtkRenderWindowInteractor(self)
        self.readerDict = {}
        self.actorDict = {}
        self.outlineActorDict = {}
        self.renderer = ""
        #self.renderer = vtk.vtkRenderer()
        self.originAxes = None
        self.allElms = []
        self.hideElms = []
        self.showElms = []
        self.currElms = []
        self.selectNos = 0
        self.modelBounds = (0,0, 0,0, 0,0)
        self.vtkInitialize()
        #初期化
        self.initialize()

    def main(self):
        """ GUIの表示"""
        self.setVtkArea()           #vtkAreaをセット
        self.mainWindow.show_all()
        self.setVtkModel()          #vtkを表示
        self.treeView.grab_focus()  #treeViewにfocus（keyEventを効かす為）
        self.maskEvent = False
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    def setGtkObject(self):
        """ glade内のobject名を取得"""
        self.treeView = self.builder.get_object("treeView")
        self.treeList = GtkParts.treeListCols(self.treeView)
        self.paned_win = self.builder.get_object("paned_win")
        #vtk関連
        self.box_vtk = self.builder.get_object("box_vtk")
        self.check_showEdge = self.builder.get_object("check_showEdge")
        self.check_showOutline = self.builder.get_object("check_showOutline")
        self.check_showOrigin = self.builder.get_object("check_showOrigin")
        self.radio_both = self.builder.get_object("radio_both")
        self.radio_frontFace = self.builder.get_object("radio_frontFace")
        self.radio_backFace = self.builder.get_object("radio_backFace")
        self.label_vtkMess = self.builder.get_object("label_vtkMess")
        self.button_reset = self.builder.get_object("button_reset")
        self.button_Xaxis = self.builder.get_object("button_Xaxis")
        self.button_Yaxis = self.builder.get_object("button_Yaxis")
        self.button_Zaxis = self.builder.get_object("button_Zaxis")
        self.button_rightToLeft = self.builder.get_object("button_rightToLeft")
        self.button_rolling90 = self.builder.get_object("button_rolling90")
        self.button_reloadVtk = self.builder.get_object("button_reloadVtk")

    #--------- vtk関連 --------------------------
    #
    #  vtkInitialize
    #----------------
    def vtkInitialize(self):
        """ vtk関連の初期化"""
        self.label_vtkMess.set_text(_("選択しているstlファイルを表示する。"))
        self.check_showEdge.set_active(True)
        self.check_showOutline.set_active(True)
        #toolTip
        self.check_showEdge.props.tooltip_text = _("stlのedgeを表示する")
        self.check_showOrigin.props.tooltip_text = _("原点を表示する")
        self.check_showOutline.props.tooltip_text = _("全stlファイルのoutline（feature edge）を表示する。")
        self.button_reset.props.tooltip_text = _("画面にフィットさせる（reset camera）")
        self.button_Xaxis.props.tooltip_text = _("x軸視（y-z 面）を表示")
        self.button_Yaxis.props.tooltip_text = _("y軸視（z-x 面）を表示")
        self.button_Zaxis.props.tooltip_text = _("z軸視（x-y 面）を表示")
        self.button_rightToLeft.props.tooltip_text = _("視点を反転(左右前後反転)")
        self.button_rolling90.props.tooltip_text = _("時計回りに90度回転")
        self.button_reloadVtk.props.tooltip_text = _("stlを再読込する")
        self.radio_both.props.tooltip_text = _("faceの表裏両面を表示する")
        self.radio_frontFace.props.tooltip_text = _("faceの表面のみ表示する\nfaceの方向が確認できる")
        self.radio_backFace.props.tooltip_text = _("faceの裏面のみ表示する\n隠れた内部のstlが確認できる")

    #
    #  setVtkArea
    #-------------
    def setVtkArea(self):
        """ vtkAreaを設定する"""
        vtkObj = self.vtkObj
        vtkObj.set_size_request(*self.vtkSize)
        self.box_vtk.pack_start(vtkObj, True, True, 0)

    #
    #  addCornerAxis
    #----------------
    def addCornerAxis(self):
        """ 最初の1回のみ、XYZの軸（compas）をrendererに追加する。"""
        if self.vtkInitialRun == True:
            #add axis(最初の起動時のみ)
            interactor = self.renWin.GetInteractor()
            axesActor = vtk.vtkAxesActor()
            self.axes = vtk.vtkOrientationMarkerWidget()
            self.axes.SetOrientationMarker(axesActor)
            self.axes.SetInteractor(interactor)
            self.axes.EnabledOn()
            self.axes.InteractiveOn()
            #flagをセット
            self.vtkInitialRun = False
        return

    #
    #  addOriginAxis
    #----------------
    def addOriginAxis(self):
        """ 原点にaxisを表示する"""
        #if len(self.showElms) > 0:
        if self.check_showOrigin.get_active() == True:
            #原点を表示
            if self.originAxes == None:
                #原点を作成
                #bounds = self.getBoundsActors()
                bounds = self.getOriginBounds()
                if len(bounds) == 0:
                    return
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes = vtk.vtkAxesActor()
                axes.SetTotalLength(length, length, length)
                axes.GetXAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetXAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetYAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetYAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                axes.GetZAxisCaptionActor2D().GetTextActor().SetTextScaleModeToNone()
                axes.GetZAxisCaptionActor2D().GetCaptionTextProperty().SetFontSize(12)
                self.renderer.AddActor(axes)
                self.originAxes = axes
            else:
                #原点を修正
                axes = self.originAxes
                #bounds = self.getBoundsActors()
                bounds = self.getOriginBounds()
                lenX = (bounds[1] - bounds[0]) / 2.0
                lenY = (bounds[3] - bounds[2]) / 2.0
                lenZ = (bounds[5] - bounds[4]) / 2.0
                length = max([lenX, lenY, lenZ])
                axes.SetTotalLength(length, length, length)
                #削除して追加する（axesを最後に描画する為）
                self.renderer.RemoveActor(axes)
                self.renderer.AddActor(axes)
                self.originAxes = axes
        else:
            #原点を削除
            self.renderer.RemoveActor(self.originAxes)
            self.originAxes = None
        return

    #
    #  setVtkModel
    #---------------
    def setVtkModel(self):
        """ vtkを表示する"""
        #stlFileを取得
        #treeList = GtkParts.treeListCols(self.treeView)
        allItems = self.treeList.getAllItemNames()
        stlNames = list(map(lambda x: x[0], allItems))
        self.allElms = stlNames[:]
        #全stlのactor辞書を作成
        self.actorDict = {}
        self.outlineActorDict = {}
        print("reading stl files...")
        for stlName in stlNames:
            #reader
            fileName = stlDir + "/" + stlName
            reader = vtk.vtkSTLReader()
            reader.SetFileName(fileName)
            reader.Update()
            self.readerDict[stlName] = reader
            #filter
            geomFilter = vtk.vtkGeometryFilter()
            geomFilter.SetInputConnection(reader.GetOutputPort())
            #featureEdge
            featureEdge = vtk.vtkFeatureEdges()
            featureEdge.SetInputConnection(geomFilter.GetOutputPort())
            #mapper
            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputConnection(reader.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(featureEdge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            self.actorDict[stlName] = actor
            edgeActor = vtk.vtkActor()
            edgeActor.SetMapper(edgeMapper)
            self.outlineActorDict[stlName] = edgeActor
        #rendererをチェック
        if self.renderer == "":
            self.renderer = vtk.vtkRenderer()
        else:
            self.renderer.NewInstance()
        #assyを定義
        self.assy = vtk.vtkAssembly()
        #stlを表示（選択しているstlを取得して）
        self.showSelectedItems()

        #backgroundColorを設定
        self.renderer.SetBackground2(0.5, 0.5, 1)
        self.renderer.SetBackground(0.8, 0.8, 1)
        self.renderer.GradientBackgroundOn()
        self.renderer.SetTexturedBackground(True)
        #self.renderer.ResetCamera()
        self.renWin = self.vtkObj.GetRenderWindow()
        self.renWin.NewInstance()
        #新しいrendererを追加
        self.renWin.AddRenderer(self.renderer)
        self.addCornerAxis()
        #vtk表示
        bounds = self.getCameraBounds()
        #self.renderer.ResetCamera(*self.modelBounds)
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.GlobalWarningDisplayOff()   #windowsのエラー対策
        self.vtkObj.Initialize()

        self.vtkObj.Start()
        return

    #
    #  selectItem
    #-------------
    def selectItem(self):
        """ 100ms後に選択されているitemを表示する
        直ぐには、選択itemが取得できない為。"""
        GLib.timeout_add(100, self.showSelectedItems)

    #
    #  showSelectedItems
    #--------------------
    def showSelectedItems(self):
        """ TreeItemの選択したitemをvtk表示させる"""
        if self.renderer == "":
            return
        #選択item名を取得
        #treeList = GtkParts.treeListCols(self.treeView)
        selItems = self.treeList.getSelectedItemNames()
        self.showElms = list(map(lambda x: x[0], selItems))
        self.hideElms = list(set(self.allElms) - set(self.showElms))
        parts = list(self.assy.GetParts())
        if len(parts) > 0:
            self.renderer.RemoveActor(self.assy)
            self.renderer.SetErase(True)
            self.assy = vtk.vtkAssembly()
        #actorを再設定
        for showElm in self.showElms:
            if showElm in self.actorDict.keys():
                actor = self.actorDict[showElm]
                actor = self.setActorProperty(actor)
                self.assy.AddPart(actor)
        #  outlineを追加
        if self.check_showOutline.get_active() == True:
            actors = self.outlineActorDict.values()
            for actor in actors:
                self.assy.AddPart(actor)
        self.renderer.AddActor(self.assy)
        #原点を表示
        self.addOriginAxis()
        #初めて（削除するactorが無かった）の場合、表示を合わせる
        #self.fitWindow()
        #vtk表示
        self.vtkObj.Initialize()
        return

    #
    #  setActorProperty
    def setActorProperty(self, actor):
        """ actorのpropertyを設定"""
        prop = actor.GetProperty()
        #edgeの表示
        if self.check_showEdge.get_active() == True:
            #edge表示
            prop.EdgeVisibilityOn()
        else:
            #edge非表示
            prop.EdgeVisibilityOff()
        #表裏面の表示
        if self.radio_both.get_active() == True:
            #両面表示
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOff()
        elif self.radio_frontFace.get_active() == True:
            #表面を表示
            prop.BackfaceCullingOn()
            prop.FrontfaceCullingOff()
        elif self.radio_backFace.get_active() == True:
            #裏面を表示
            prop.BackfaceCullingOff()
            prop.FrontfaceCullingOn()
        #色の設定
        prop.SetAmbient(0.3)
        prop.SetColor(colors.grey)
        return actor

    #
    #  fitWindow
    #------------
    def fitWindow(self):
        """ 画面に合わせてvtk表示させる"""
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromX
    #------------
    def viewFromX(self):
        """ X軸方向から見る"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter+maxLeng*2.0, Ycenter, Zcenter)
        #X軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 0, 1)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromY
    #------------
    def viewFromY(self):
        """ Y軸視（Z-X面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter+maxLeng*2.0, Zcenter)
        #Y軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(1, 0, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  viewFromZ
    #------------
    def viewFromZ(self):
        """ Z軸視（X-Y面）"""
        #camera位置を算出
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        maxLeng = max(bounds[1]-bounds[0], bounds[3]-bounds[2], bounds[5]-bounds[4])
        cameraPos = (Xcenter, Ycenter, Zcenter+maxLeng*2.0)
        #Z軸視に設定
        camera = self.renderer.GetActiveCamera()
        #  camera位置を設定
        camera.SetPosition(*cameraPos)
        #  cameraの縦軸の単位vectorを設定
        camera.SetViewUp(0, 1, 0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  setReverseCamera
    #-------------------
    def setReverseCamera(self):
        """ camera位置を反転（左右反転）させる"""
        bounds = self.getBoundsActors()
        Xcenter = (bounds[0] + bounds[1]) / 2.0
        Ycenter = (bounds[2] + bounds[3]) / 2.0
        Zcenter = (bounds[4] + bounds[5]) / 2.0
        center = (Xcenter, Ycenter, Zcenter)
        camera = self.renderer.GetActiveCamera()
        pos = camera.GetPosition()
        relPos = (pos[0]-center[0], pos[1]-center[1], pos[2]-center[2])
        newRelPos = list(map(lambda x: -x, relPos))
        newPos = (newRelPos[0]+center[0], newRelPos[1]+center[1], newRelPos[2]+center[2])
        camera.SetPosition(*newPos)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  rolling90
    #------------
    def rolling90(self):
        """ 視点を右回転（90°）させる。"""
        camera = self.renderer.GetActiveCamera()
        camera.Roll(-90.0)
        #  表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkObj.Initialize()

    #
    #  reloadVtk
    #------------
    def reloadVtk(self):
        """ stlを再読込"""
        #stlFileを取得
        #treeList = GtkParts.treeListCols(self.treeView)
        allItems = self.treeList.getAllItemNames()
        stlNames = list(map(lambda x: x[0], allItems))
        self.allElms = stlNames[:]
        #actor辞書を作成
        self.actorDict = {}
        self.outlineActorDict = {}
        for stlName in stlNames:
            #reader
            fileName = stlDir + "/" + stlName
            reader = vtk.vtkSTLReader()
            reader.SetFileName(fileName)
            reader.Update()
            self.readerDict[stlName] = reader
            #filter
            geomFilter = vtk.vtkGeometryFilter()
            geomFilter.SetInputConnection(reader.GetOutputPort())
            #featureEdge
            featureEdge = vtk.vtkFeatureEdges()
            featureEdge.SetInputConnection(geomFilter.GetOutputPort())
            #mapper
            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputConnection(reader.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(featureEdge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            self.actorDict[stlName] = actor
            edgeActor = vtk.vtkActor()
            edgeActor.SetMapper(edgeMapper)
            self.outlineActorDict[stlName] = edgeActor
        #vtk表示
        self.showSelectedItems()        

    #
    #  changeShowEdge
    #-----------------
    def changeShowEdge(self):
        """ check_showEdgeを変更した時のevent"""
        self.showSelectedItems()

    #
    #  changeShowOutline
    #--------------------
    def changeShowOutline(self):
        """ check_showOutlineを変更したときのevent"""
        self.showSelectedItems()

    #
    #  changeShowOrigin
    #-------------------
    def chengeShowOrigin(self):
        """ 原点表示チェックを変更した時のevent"""
        self.showSelectedItems()

    #
    #  changeShowBoth
    #-----------------
    def changeShowBoth(self):
        """ radio_showBothを変更した時のevent"""
        if self.radio_both.get_active() == True:
            self.showSelectedItems()
        
    #
    #  changeShowFront
    #------------------
    def changeShowFront(self):
        """ radio_showFrontFaceを変更した時のevent"""
        if self.radio_frontFace.get_active() == True:
            self.showSelectedItems()

    #
    #  changeShowBack
    #-----------------
    def changeShowBack(self):
        """ radio_showbackFaceを変更した時のevent"""
        if self.radio_backFace.get_active() == True:
            self.showSelectedItems()

    #
    #  getCameraBounds
    def getCameraBounds(self):
        """ 表示させるactorのboundsを取得する。
        原点、マージンも含めて取得する。"""
        bounds = self.getOriginBounds()
        # if self.originAxes == None:
        #     #原点無しの場合
        #     return bounds
        #マージンを確保
        xmin = bounds[0]
        xmax = bounds[1]
        ymin = bounds[2]
        ymax = bounds[3]
        zmin = bounds[4]
        zmax = bounds[5]
        a = 0.1
        xmin = xmin - (xmax-xmin) * a
        xmax = xmax + (xmax-xmin) * a
        ymin = ymin - (ymax-ymin) * a
        ymax = ymax + (ymax-ymin) * a
        zmin = zmin - (zmax-xmin) * a
        zmax = zmax + (zmax-xmin) * a
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getOriginBounds
    def getOriginBounds(self):
        """ 表示させるactorと原点のboundsを取得する。"""
        bounds = self.getBoundsActors()
        if self.originAxes == None:
            #原点無しの場合
            return bounds
        #原点を追加する
        xmin = min([bounds[0], 0.0])
        xmax = max([bounds[1], 0.0])
        ymin = min([bounds[2], 0.0])
        ymax = max([bounds[3], 0.0])
        zmin = min([bounds[4], 0.0])
        zmax = max([bounds[5], 0.0])
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getBoundsActors
    def getBoundsActors(self):
        """ 全actorのboundsを取得して返す。"""
        actors = self.renderer.GetActors()
        bounds = []
        for actor in actors:
            # if actor in self.actorDict.values():
            #     bound = actor.GetBounds()
            #     bounds.append(bound)
            bound = actor.GetBounds()
            bounds.append(bound)
        if len(bounds) == 0:
            return ()
        Xmin = min(map(lambda x: x[0], bounds))
        Xmax = max(map(lambda x: x[1], bounds))
        Ymin = min(map(lambda x: x[2], bounds))
        Ymax = max(map(lambda x: x[3], bounds))
        Zmin = min(map(lambda x: x[4], bounds))
        Zmax = max(map(lambda x: x[5], bounds))
        return (Xmin, Xmax, Ymin, Ymax, Zmin, Zmax)
    #--------------------------------------------


    #----------- event handler -------
    #-------- vtk画面 -----------
    #サイズを合わせる
    def onFitWindow(self, event):
        self.fitWindow()
    #X軸視
    def onViewFromX(self, event):
        self.viewFromX()
    #Y軸視
    def onViewFromY(self, event):
        self.viewFromY()
    #Z軸視
    def onViewFromZ(self, event):
        self.viewFromZ()
    #左右前後入れ替え
    def onSetReverseCamera(self, event):
        self.setReverseCamera()
    #回転90°
    def onRolling90(self, event):
        self.rolling90()
    #vtk再読込
    def onReloadVtk(self, event):
        self.setDataToListBox()
        self.reloadVtk()
    #edge表示のチェック
    def onChangeShowEdge(self, event):
        self.changeShowEdge()
    #outline表示
    def onChangeShowOutline(self, event):
        self.changeShowOutline()
    #原点表示のチェック
    def onChangeShowOrigin(self, event):
        self.chengeShowOrigin()
    #radio_bothを変更
    def onChangeShowBoth(self, event):
        self.changeShowBoth()
    #radio_frontFaceを変更
    def onChangeShowFront(self, event):
        self.changeShowFront()
    #radio_backFaceを変更
    def onChangeShowBack(self, event):
        self.changeShowBack()
    #-------- ボタン ------------
    #ascii変換
    def onBinToAscii(self, event):
        self.binToAscii()
    #solid名セット
    def onSetSolidName(self, event):
        self.setSolidName()
    #solid名変更
    def onChangeSolidName(self, event):
        self.changeSolidName()
    #scale変更
    def onChangeScale(self, event):
        self.changeScale()
    #各軸回転
    def onAxisRotation(self, event):
        self.axisRotation()
    #移動
    def onTransform(self, event):
        self.transform()
    #stlCheck
    def onStlCheck(self, event):
        self.stlCheck()
    #stlSolidCheck(solid抽出)
    def onStlSolidCheck(self, event):
        self.stlSolidCheck()
    #faceの向きを反転
    def onReverseSurface(self, event):
        self.reverseSurface()
    #stlFile結合
    def onMergeStlFiles(self, event):
        self.mergeStlFile()
    #内部solid名削除（内部結合）
    def onDeleteInnerNames(self, event):
        self.deleteInnerNames()
    #paraview起動
    def onRunParaview(self, event):
        self.runParaview()
    #folder開く
    def onOpenFolder(self, event):
        self.openFolder()
    #閉じる
    def onClose(self, event):
        #ending(stlDir、winSizeを保存)
        self.ending()
        self.close()
    #item選択
    def onSelectItem(self, event):
        if self.maskEvent == False:
            self.selectItem()
    
    #-------- mouse操作------
    #selectCursor
    def onChangeCursor(self, widget):
        self.changeCursor(widget)
    #selectAll
    def onSelectAll(self, widget):
        self.changeCursor(widget)
    #unselectAll
    def onUnselectAll(self, widget):
        self.changeCursor()
    #pressButton
    def onPressMouseButton(self, widget, event):
        self.pressMouseButton(widget, event)
    #moseMove
    def onFocusTreeList(self, widget, event):
        widget.grab_focus()
    #-------- key操作 -------------
    def onKeyPress(self, widget, event):
        self.keyPress(widget, event)
    def onKeyRelease(self, widget, event):
        self.keyRelease(widget, event)
    #-------- popupMenu ------------
    #開く
    def onOpenStlFiles(self, event):
        self.openStlFiles()
    #コピー
    def onCopyStlFiles(self, event):
        self.copyStlFiles()
    #貼り付け
    def onPasteStlFiles(self, event):
        self.pasteStlFiles()
    #ファイル名変更
    def onRenameStlFile(self, event):
        self.renameStlFile()
    #削除
    def onDeleteStlFiles(self, event):
        self.deleteStlFiles()


    #
    #  setPopupMenuItems
    #--------------------
    def setPopupMenuItems(self):
        """ popupMenu項目を設定する"""
        self.popupMenuItems = [
            #ラベル                     handler
            [_("開く"),                 self.onOpenStlFiles],
            ["-", ""],
            [_("コピー"),               self.onCopyStlFiles],
            [_("フォルダに貼付け"),     self.onPasteStlFiles],
            ["-", ""],
            [_("ファイル名変更"),       self.onRenameStlFile],
            [_("削除"),                 self.onDeleteStlFiles]
            ]

    #
    #  initialize
    #--------------
    def initialize(self):
        """ 初期化"""
        #configDictを取得
        self.configDict = pyTreeFoam.readConfigTreeFoam()
        #windowSizeを取得
        winSizeDict = pyTreeFoam.readWindowSize()
        #windowSizeを設定
        if "editStlFiles" in winSizeDict.keys():
            xy = winSizeDict["editStlFiles"]
            if xy != "":
                (x, y) = map(int, xy.split()[:2])
                self.mainWindow.set_default_size(x, y)
        #panedSplitterを設定
        if "editStlFiles_splitter" in winSizeDict.keys():
            splitWidth = winSizeDict["editStlFiles_splitter"]
            if splitWidth != "":
                width = int(splitWidth.split()[0])
                self.paned_win.set_position(width)
        #treeViewのcolumnWidthを取得
        colWidthes = winSizeDict["editStlFiles_colWidth"]
        if colWidthes != "":
            widthes = list(map(int, colWidthes.split()))
        else:
            widthes = []
        #treeListを作成
        pixbuf = GdkPixbuf.Pixbuf
        model = (pixbuf, str, str, str, str)
        header = ["stl", "format", "solid", "size(xyz)"]
        #treeList = GtkParts.treeListCols(self.treeView)
        self.treeList.create(model, 
                multi=True, 
                header=True, 
                headerConts=header,
                colWidthes=widthes)
        #tree表示
        self.setDataToListBox()

    #
    #  ending
    #
    def ending(self):
        """ 終了処理。stlDirとwinSizeを保存する"""
        #stlDirを保存
        dataDict = {}
        dataDict["stlDir"] = self.stlDir
        pyTreeFoam.case(self.caseDir).setCreateMeshData(dataDict)
        #winSizeを保存
        (width, height) = self.mainWindow.get_size()
        splitWidth = self.paned_win.get_position()
        colWidthes = []
        for i in range(3):
            colWidth = self.treeView.get_column(i).get_width()
            colWidthes.append(str(colWidth))
        winDict = {}
        winDict["editStlFiles"] = str(width) + " " + str(height)
        winDict["editStlFiles_colWidth"] = " ".join(colWidthes)
        winDict["editStlFiles_splitter"] = str(splitWidth)
        pyTreeFoam.writeWindowSize(winDict)

    #
    #  binToAscii
    #  ----------
    def binToAscii(self):
        """ 選択ファイルをascii形式に変換する"""
        stlNames = self.getSelectStlNames()
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        #倍率を変更(asciiに変換)
        envOpenFoam = self.configDict["bashrcFOAM"]
        for name in stlNames:
            fileName = self.stlDir + os.sep + name
            comm = ". " + envOpenFoam + "; "
            #comm += "surfaceTransformPoints -scale '(1.0 1.0 1.0)' " + fileName + " " + fileName
            comm += self.createTransformCommLine("scale", ["1.0", "1.0", "1.0"], fileName)
            print(comm)
            pyTreeFoam.run(self.caseDir).commandWithLog(comm)
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface(stlNames)
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        print(_("ascii形式に変換しました。"))

    #
    #  setSolidName
    #  ------------
    def setSolidName(self):
        """ file名のsolid名をセットする"""
        stlNames = self.getSelectStlNames()
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        #asciiチェック
        flag = 0
        for stlName in stlNames:
            fileName = stlDir + "/" + stlName
            #if isBinaryStl == True:
            if pyTreeFoam.isBinaryStl(fileName) == True:
                flag = 1
                break
        if flag == 1:
            title = _("エラー")
            mess = _("選択したstlファイルは、binaryが含まれています。")
            mess += _("ascii形式に変換後、実行してください。")
            self.errDialog(title, mess)
            return
        #solid名をセット
        for stlName in stlNames:
            flag = 0
            fileName = stlDir + "/" + stlName
            name = stlName[:-4]
            f=open(fileName); lines=f.readlines(); f.close()
            #旧のsolid名を取得
            words = lines[0].split()
            oldName = ""
            if len(words) > 1:
                oldName = words[1]
                if oldName.find(",") >= 0:
                    #1行目のnameに「,」が含まれているかどうか
                    oldName = oldName.split(",")[0]
            #solid名を設定
            lines[0] = " ".join([words[0], name]) + "\n"
            i = len(lines) - 1
            while i > 0:
                words = lines[i].split()
                if words[0] == "endsolid":
                    endName = ""
                    if len(words) > 1:
                        endName = words[1]
                    break
                i -= 1
            if oldName != endName:
                flag += 1
            else:
                lines[i] = " ".join([words[0], name]) + "\n"
                f=open(fileName, "w")
                for line in lines:
                    f.write(line)
                f.close()
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface(stlNames)
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        print(_("solid名をセットしました。"))

    #
    #  changeSolidName
    #  ---------------
    def changeSolidName(self):
        """ 選択しているstlファイルのsolid名を変更する"""
        #treeList = GtkParts.treeListCols(self.treeView)
        stlNames = self.treeList.getSelectedItemNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        if len(stlNames) != 1:
            title = _("エラー")
            mess = _("複数のstlファイルを選択しています。")
            mess += _("\n1ヶのstlファイルを選択してください。")
            self.errDialog(title, mess)
            return
        #変更するsolidNameを取得
        solidName = stlNames[0][2]
        title = _("solid名の入力")
        mess = _("新しいsolid名を入力してください")
        funcOk = [self.changeSolidName_run, solidName, stlNames]
        self.inputTextDialog(title, mess, solidName, funcOk)
        #newName = self.inputTextDialog(title, mess, solidName)
        #if newName == "":
        #    return
        
    #
    #  changeSolidName_run
    def changeSolidName_run(self, newName, solidName, stlNames):
        #solid名を変更
        oldName = " ".join(solidName.split())
        changeFlag = 0
        selName = stlNames[0][0]
        fileName = stlDir + "/" + selName
        #fileFormatをチェック
        if pyTreeFoam.isBinaryStl(fileName) == False:
            #asciiの場合
            f=open(fileName); lines=f.readlines(); f.close()
            i = 0
            for line in lines:
                if line[:5] == "solid":
                    if " ".join(line.split()[1:]) == oldName:
                        lines[i] = "solid " + newName + "\n"
                        changeFlag += 1
                if line[:8] == "endsolid":
                    if changeFlag == 1:
                        lines[i] = "endsolid " + newName + "\n"
                        changeFlag += 1
                i += 1
            if changeFlag == 2:
                f=open(fileName, "w")
                for line in lines:
                    f.write(line)
                f.close()
                print(_("solid名を変更しました。"), solidName, "->", newName)
            else:
                title = _("エラー")
                mess = _("solid名が変更できませんでした。")
                mess += _("\nstlファイルの書式を確認してください。")
                okArgs = [self.changeSolidName_ending, selName]
                self.errDialog(title, mess, funcOk=okArgs)
                return
        else:
            #binaryの場合
            f=open(fileName, "rb"); cont=f.read(); f.close()
            header = newName.encode("utf-8") + b"\00"*(80-len(newName))
            cont = header + cont[80:]
            f=open(fileName, "wb"); f.write(cont); f.close()
            print(_("solid名を変更しました。"), solidName, "->", newName)
        self.changeSolidName_ending(selName)

    #  changeSolidName_ending
    def changeSolidName_ending(self, selName):
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface([selName])
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames([selName])
        #vtk再表示
        self.reloadVtk()

    #
    #  changeScale
    #  -----------
    def changeScale(self):
        """ scaleの変更"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        #変更する倍率を取得
        title = _("倍率の入力")
        mess = _("変更する倍率を入力してください")
        entryText = "1.0"
        funcOk = [self.changeScale_run, stlNames]
        self.inputTextDialog(title, mess, entryText, funcOk)

    #
    #  changeScale_run
    def changeScale_run(self, scale, stlNames):
        if scale == "" or scale == "1.0":
            return        
        #倍率を変更
        envOpenFoam = self.configDict["bashrcFOAM"]
        for name in stlNames:
            fileName = self.stlDir + os.sep + name
            comm = ". " + envOpenFoam + "; "
            #comm += "surfaceTransformPoints -scale '(" + scale + " " + scale + " " + scale + ")' " + fileName + " " + fileName
            comm += self.createTransformCommLine("scale", [scale, scale, scale], fileName)
            print(comm)
            pyTreeFoam.run(self.caseDir).commandWithLog(comm)
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface(stlNames)
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        print(_("scaleを変更しました。"), " x", scale)

    #
    #  createTransformCommLine
    def createTransformCommLine(self, commType, args, fileName):
        """ surfaceTransformPoints コマンド行を作成する。
        OF-9からコマンド変更"""
        
        def createCommLine_Org(commType, args, fileName):
            """
            従来の表記
            surfaceTransformPoints -scale '(2.0 2.0 2.0)' file file
            surfaceTransformPoints -translate '(0.1 0 0)' file file
            surfaceTransformPoints -rollPitchYaw '(90 0 0)' fileName filename
            """
            if commType == "scale":
                argLine = " '(" + " ".join(args) + ")' "
                line = "surfaceTransformPoints -scale" + argLine + fileName + " " + fileName
            elif commType == "translate":
                argLine = " '(" + " ".join(args) + ")' "
                line = "surfaceTransformPoints -translate" + argLine + fileName + " " + fileName
            elif commType == "rotation":
                argLine = " '(" + " ".join(args) + ")' "
                line = "surfaceTransformPoints -rollPitchYaw" + argLine + fileName + " " + fileName
            return line

        def createCommLine_9(commType, args, fileName):
            """
            OF-9以降の表記
            surfaceTransformPoints 'scale=(2.0 2.0 2.0)' file file
            surfaceTransformPoints 'translate=(0.1 0.0 0.0)' file file
            surfaceTransformPoints 'Rx=90, Ry=0, Rz=0' file file
            """
            if commType == "scale":
                argLine = " 'scale=(" + " ".join(args) + ")' "
                line = "surfaceTransformPoints " + argLine + fileName + " " + fileName
            elif commType == "translate":
                argLine = " 'translate=(" + " ".join(args) + ")' "
                line = "surfaceTransformPoints " + argLine + fileName + " " + fileName
            elif commType == "rotation":
                argLine = " 'Rx=" + args[0] + ", Ry=" + args[1] + ", Rz=" + args[2] + "' "
                line = "surfaceTransformPoints " + argLine + fileName + " " + fileName
            return line

        OFversion = self.configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        if OFversion >= "v1804":
            line = createCommLine_Org(commType, args, fileName)
        elif OFversion[:3] == "ext":
            line = createCommLine_Org(commType, args, fileName)
        #elif OFversion >= "9" or OFversion >= "10":
        elif numVer >= 9.0:
            line = createCommLine_9(commType, args, fileName)
        else:
            line = createCommLine_Org(commType, args, fileName)
        return line

    #
    #  axisRotation
    #---------------
    def axisRotation(self):
        """ 各軸回りの回転"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        #角度の入力
        okArgs = [self.axisRotation_run, stlNames]
        dialog = surfaceRotationDDialog.surfaceRotation(funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return
        
    #  axisRotation_run
    def axisRotation_run(self, rotations, stlNames):
        #角度のエラーチェック
        try:
            _a = float(rotations[0])
            _b = float(rotations[1])
            _c = float(rotations[2])
        except:
            title = _("エラー")
            mess = _("角度の入力値が間違っています")
            self.errDialog(title, mess)
            return
        #角度を修正
        envOpenFoam = self.configDict["bashrcFOAM"]
        for name in stlNames:
            fileName = self.stlDir + os.sep + name
            comm = ". " + envOpenFoam + "; "
            #comm += "surfaceTransformPoints -rollPitchYaw "
            #comm += "'(" + " ".join(rotations) + ")' " 
            #comm += fileName + " " + fileName
            comm += self.createTransformCommLine("rotation", rotations, fileName)
            print(comm)
            pyTreeFoam.run(self.caseDir).commandWithLog(comm)
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface(stlNames)
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        print(_("軸回りに回転しました。"))

    #
    #  transform
    #------------
    def transform(self):
        """ stlを移動する"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        #移動量の入力
        okArgs = [self.transform_run, stlNames]
        dialog = surfaceTransformDDialog.surfaceTransform(funcOk=okArgs, parent=self.mainWindow)
        dialog.show()
        return
    
    #  transform_run
    def transform_run(self, transforms, stlNames):
        #中心を原点移動？
        if "org" in transforms:
            transforms = self.checkGetOrigin(stlNames, transforms)
            #error?
            if len(transforms) == 0:
                #error発生
                return
        #角度のエラーチェック
        try:
            _a = float(transforms[0])
            _b = float(transforms[1])
            _c = float(transforms[2])
        except:
            title = _("エラー")
            mess = _("移動量の入力値が間違っています")
            self.errDialog(title, mess)
            return
        #移動
        envOpenFoam = self.configDict["bashrcFOAM"]
        for name in stlNames:
            fileName = self.stlDir + os.sep + name
            comm = ". " + envOpenFoam + "; "
            #comm += "surfaceTransformPoints -translate "
            #comm += "'(" + " ".join(transforms) + ")' " 
            #comm += fileName + " " + fileName
            print(comm)
            comm += self.createTransformCommLine("translate", transforms, fileName)
            pyTreeFoam.run(self.caseDir).commandWithLog(comm)
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface(stlNames)
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        print(_("移動しました。"))

    #
    #  checkGetOrigin
    def checkGetOrigin(self, stlNames, transforms):
        #asciiチェック
        flag = 0
        for stlName in stlNames:
            fileName = self.stlDir + "/" + stlName
            if pyTreeFoam.isBinaryStl(fileName) == True:
                flag = 1
                break
        if flag == 1:
            title = _("エラー")
            mess = _("選択したstlファイルは、binaryが含まれています。")
            mess += _("ascii形式に変換後、実行してください。")
            self.errDialog(title, mess)
            return []
        #中心座標を取得
        cx, cy, cz = self.getStlsCenterLocation(stlNames)
        #移動量を設定
        if transforms[0] == "org":
            transforms[0] = str(-cx)
        if transforms[1] == "org":
            transforms[1] = str(-cy)
        if transforms[2] == "org":
            transforms[2] = str(-cz)
        return transforms

    #
    #  getStlsCenterLocation
    def getStlsCenterLocation(self, stlNames):
        """ stlFilesの中心座標を取得し返す"""
        #minMax座標を取得
        locs = []
        for stlName in stlNames:
            fileName = self.stlDir + "/" + stlName
            f = open(fileName); lines = f.readlines(); f.close()
            for line in lines:
                words = line.split()
                if words[0] == "vertex":
                    #ベクトルを反転
                    [x, y, z] = map(float, words[1:4])
                    locs.append([x, y, z])
        xLocs = list(map(lambda x: x[0], locs))
        xMin = min(xLocs); xMax = max(xLocs)
        yLocs = list(map(lambda x: x[1], locs))
        yMin = min(yLocs); yMax = max(yLocs)
        zLocs = list(map(lambda x: x[2], locs))
        zMin = min(zLocs); zMax = max(zLocs)
        #transformsを修正
        cx = (xMax + xMin) / 2.0
        cy = (yMax + yMin) / 2.0
        cz = (zMax + zMin) / 2.0
        return (cx, cy, cz)

    #
    #  stlCheck
    #------------
    def stlCheck(self):
        """ stlをチェック、修正する"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        resLines = []
        envOpenFoam = self.configDict["bashrcFOAM"]
        for name in stlNames:
            fileName = self.stlDir + os.sep + name
            comm = ". " + envOpenFoam + "; "
            comm += "stlChecker.py -i " + fileName + " -oc onlyCorrect"
            #pyTreeFoam.run(self.caseDir).commandWithLog(comm)
            stat, res, err = pyTreeFoam.run(self.caseDir).commandReturnCont(comm, isOut=True)
            lines = res.split("\n")
            for line in lines:
                if line.find("-->") >= 0:
                    if line.find("good ") >=0:
                        addLine = "  " + name + " --> good"
                    else:
                        addLine = "  " + name + " --> Bad " + _("（修正fileを保存）")
                    resLines.append(addLine)
                    break
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        #dialog表示
        title = "stlCheck"
        cont = _("stlFileのチェック結果") + "\n\n"
        cont += "\n".join(resLines) + "\n\n"
        cont += _("詳細なチェック結果は、logを参照。")
        self.okDialog(title, cont)

    #
    #  stlSolidCheck
    #----------------
    def stlSolidCheck(self):
        """ stlの中から閉じたstl（solid）を抽出"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        resLines = []
        envOpenFoam = self.configDict["bashrcFOAM"]
        for name in stlNames:
            fileName = self.stlDir + os.sep + name
            comm = ". " + envOpenFoam + "; "
            comm += "stlSolidChecker.py -i " + fileName + " -oc onlyCorrect"
            #pyTreeFoam.run(self.caseDir).commandWithLog(comm)
            stat, res, err = pyTreeFoam.run(self.caseDir).commandReturnCont(comm, isOut=True)
            lines = res.split("\n")
            for line in lines:
                if line.find("-->") >= 0:
                    if line.find("good ") >=0:
                        addLine = "  " + name + " --> good"
                    elif line.find("not exist") >= 0:
                        addLine = "  " + name + " --> no solid"
                    else:
                        addLine = "  " + name + " --> Bad " + _("（修正fileを保存）")
                    resLines.append(addLine)
                    break
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        #dialog表示
        title = _("solid抽出")
        cont = _("stlFileのsolidチェック結果") + "\n\n"
        cont += "\n".join(resLines) + "\n\n"
        cont += _("詳細なチェック結果は、logを参照。")
        self.okDialog(title, cont)

    #
    #  reverseSurface
    #  --------------
    def reverseSurface(self):
        """ faceの向きを反転する"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        #asciiチェック
        flag = 0
        for stlName in stlNames:
            fileName = stlDir + "/" + stlName
            #if isBinaryStl == True:
            if pyTreeFoam.isBinaryStl(fileName) == True:
                flag = 1
                break
        if flag == 1:
            title = _("エラー")
            mess = _("選択したstlファイルは、binaryが含まれています。")
            mess += _("ascii形式に変換後、実行してください。")
            self.errDialog(title, mess)
            return
        #faceの向き反転
        newNames = []
        for stlName in stlNames:
            fileName = self.stlDir + "/" + stlName
            f = open(fileName); lines = f.readlines(); f.close()
            i = 0
            while i < len(lines):
                words = lines[i].split()
                if words[0] == "facet" and words[1]== "normal":
                    #ベクトルを反転
                    words[2] = str(-float(words[2]))
                    words[3] = str(-float(words[3]))
                    words[4] = str(-float(words[4]))
                    lines[i] = " " + " ".join(words) + "\n"
                    #三角形の座標の順番を入れ替え
                    point1 = lines[i+3]
                    point2 = lines[i+4]
                    lines[i+3] = point2
                    lines[i+4] = point1
                    i += 5
                else:
                    i += 1
            newFileName = fileName[:-4] + "_rev.stl"
            f = open(newFileName, "w"); f.writelines(lines); f.close()
            newName = newFileName.split(os.sep)[-1]
            newNames.append(newName)
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface(newNames)
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(newNames)
        #vtk再表示
        self.reloadVtk()
        print(_("faceの向きを反転させました。"))

    #
    #  mergeStlFile
    #  ------------
    def mergeStlFile(self):
        """ 選択した複数のstlFileを結合する"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) < 2:
            title = _("エラー")
            mess = _("stlファイルを複数選択してください")
            self.errDialog(title, mess)
            return
        #asciiチェック
        flag = 0
        for stlName in stlNames:
            fileName = stlDir + "/" + stlName
            #if isBinaryStl == True:
            if pyTreeFoam.isBinaryStl(fileName) == True:
                flag = 1
                break
        if flag == 1:
            title = _("エラー")
            mess = _("選択したstlファイルは、binaryが含まれています。")
            mess += _("ascii形式に変換後、実行してください。")
            self.errDialog(title, mess)
            return
        #solid名のチェック
        solids = []
        for stlName in stlNames:
            names = []
            fileName = self.stlDir + os.sep + stlName
            f = open(fileName); lines = f.readlines(); f.close()
            for line in lines:
                if line[:5] == "solid":
                    name = " ".join(line.split()[1:])
                    names.append(name)
            solids += names
        flag = 0
        solids.sort()
        oldName = ""
        for solid in solids:
            if len(solid.split()) == 0:
                flag = 1
                break
            elif oldName == solid:
                flag = 2
                break
            oldName = solid
        if flag == 1:
            title = _("エラー")
            mess = _("solid名が設定されていないファイルがあります")
            mess += _("\nstlファイルが結合できません")
            self.errDialog(title, mess)
            return
        if flag == 2:
            title = _("エラー")
            mess = _("同じsolid名が使われているファイルがあります。")
            mess += _("\nstlファイルが結合できません")
            self.errDialog(title, mess)
            return
        #defaultのnewStl名を設定
        allStlNames = glob.glob(self.stlDir + os.sep + "*.stl")
        allStlNames = list(map(lambda x: x.split(os.sep)[-1], allStlNames))
        stlNo = 0
        loop = True
        while loop == True:
            newName = "newStl_" + str(stlNo) + ".stl"
            if newName in allStlNames:
                stlNo += 1
            else:
                loop = False
                break
        #newStl名を取得
        title = _(u"stlファイルの結合")
        mess = _(u"新しいstlファイル名を入力してください")
        inputText = newName
        funcOk = [self.mergeStlFile_run, stlNames, allStlNames]
        self.inputTextDialog(title, mess, inputText, funcOk)
        
    #
    #  mergeStlFile_run
    def mergeStlFile_run(self, text, stlNames, allStlNames):
        """ stlNamesをmergeして1個のfileを作成する。"""
        if text.split(".")[-1] != "stl":
            text += ".stl"
        if text in allStlNames:
            title = _("エラー")
            mess = _("同じファイル名が存在します。")
            mess += _("\nファイル名を変更してください。")
            self.errDialog(title, mess)
            return
        writeName = self.stlDir + os.sep + text
        #stlFileを結合
        for stlName in stlNames:
            readName = self.stlDir + os.sep + stlName
            f = open(readName); cont = f.read(); f.close()
            if cont[-1] != "\n":
                cont += "\n"
            f = open(writeName, "a"); f.write(cont); f.close()
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface([text])
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames([text])
        #vtk再表示
        self.reloadVtk()
        title = _("stlファイルの結合")
        mess = _("stlファイルを結合しました。")
        self.okDialog(title, mess)
        print(mess)

    #
    #  deleteInnerNames
    #  ----------------
    def deleteInnerNames(self):
        """ stlファイル内の内部solid名を削除する"""
        stlNames = self.getSelectStlNames()
        #エラーチェック
        if len(stlNames) == 0:
            title = _("エラー")
            mess = _("stlファイルを選択してください")
            self.errDialog(title, mess)
            return
        #asciiチェック
        flag = 0
        for stlName in stlNames:
            fileName = stlDir + "/" + stlName
            #if isBinaryStl == True:
            if pyTreeFoam.isBinaryStl(fileName) == True:
                flag = 1
                break
        if flag == 1:
            title = _("エラー")
            mess = _("選択したstlファイルは、binaryが含まれています。")
            mess += _("ascii形式に変換後、実行してください。")
            self.errDialog(title, mess)
            return
        #内部solid名を削除する
        for stlName in stlNames:
            fileName = self.stlDir + os.sep + stlName
            f = open(fileName); lines = f.readlines(); f.close()
            solidName = stlName[:-4]
            #header部以外を取得
            newLines = []
            for line in lines:
                words = line.split()
                if len(words) > 0:
                    name = words[0].lower()
                    if name != "solid" and name != "endsolid":
                        newLines.append(line)
            #header, footerを追加
            header = "solid " + solidName + "\n"
            footer = "endsolid " + solidName + "\n"
            if newLines[-1][-1] != "\n":
                newLines[-1] += "\n"
            newLines = [header] + newLines + [footer]
            f = open(fileName, "w"); f.writelines(newLines); f.close()
        #fileをtriSurfaceにコピー
        self.copyFilesToTriSurface(stlNames)
        #listBoxに反映
        self.setDataToListBox()
        #変換したfileを選択する
        self.selectStlNames(stlNames)
        #vtk再表示
        self.reloadVtk()
        print(_("内部solid名を削除しました。"))

    #
    #  runParaview
    #  -----------
    def runParaview(self):
        """ paraViewを起動する"""
        print(_("paraviewを起動します"))
        envOpenFoam = self.configDict["bashrcFOAM"]
        comm = ". " + envOpenFoam + "; "
        comm += "paraview &"
        pyTreeFoam.run(self.stlDir).commandWithLog(comm)

    #
    #  openFolder
    #  ----------
    def openFolder(self):
        """ folderを開く"""
        print(_("folderを開く"))
        openDir = self.stlDir
        fileManager = pyTreeFoam.fileManager
        comm = fileManager + " " + openDir + " &"
        pyTreeFoam.run(self.stlDir).command(comm)

    #
    #  changeCursor
    #---------------
    def changeCursor(self, widget):
        """ delayを設定して、treeListのselectItemsを取得し、表示する。"""
        GLib.timeout_add(50, self.showSelectedItems)

    #
    #  pressMouseButton
    #-------------------
    def pressMouseButton(self, widget, event):
        """ 右クリックの動作"""
        pos = (event.x, event.y)
        pathInfo = widget.get_path_at_pos(event.x, event.y)
        if pathInfo == None:
            return
        path, _col, _x, _y = pathInfo
        #right click?
        if event.button == 3:
            selection = widget.get_selection()
            #selection.select_path(path)
            (_listStore, pathes) = selection.get_selected_rows()
            #右クリック場所が戦隊されているか？
            if path in pathes:
                #選択されている場合
                self.afterSelectPathes(pathes, selection)
            self.changeCursor(widget)
            #popupMenuを表示
            self.showPopupMenu(pos)

    #
    #  showPopupMenu
    def showPopupMenu(self, pos):
        """ popupMenuを表示する"""
        #popupMenu作成
        menu = Gtk.Menu()
        menuItems = self.popupMenuItems
        for i in range(len(menuItems)):
            menuName = menuItems[i]
            if menuName[0] == "-":
                #セパレータをセット
                item = Gtk.SeparatorMenuItem()
            else:
                item = Gtk.MenuItem()
                item.set_label(menuName[0])
                item.connect("activate", menuName[1])
            menu.append(item)
        #menuを表示
        menu.show_all()
        menu.popup(None, None, None, None, 0, Gtk.get_current_event_time())

    #
    #  afterSelectPathes
    def afterSelectPathes(self, pathes, selection):
        """ 0.05s後に選択する"""
        GLib.timeout_add(50, self.afterSelectPathes_sub, pathes, selection)

    def afterSelectPathes_sub(self, pathes, selection):
        for path in pathes:
            selection.select_path(path)

    #
    #  keyPress
    #------------
    def keyPress(self, widget, event):
        """ key pressの処理"""
        keyName = Gdk.keyval_name(event.keyval)
        if keyName == "Control_L" or keyName == "Control_R":
            self.ctrlKeyFlag = 1
        elif keyName == "Shift_L" or keyName == "Shift_R":
            self.shiftKeyFlag = 1

    #
    #  keyRelease
    #-------------
    def keyRelease(self, widget, event):
        """ key releaseの処理"""
        keyName = Gdk.keyval_name(event.keyval)
        if keyName == "Control_L" or keyName == "Control_R":
            self.ctrlKeyFlag = 0
        elif keyName == "Shift_L" or keyName == "Shift_R":
            self.shiftKeyFlag = 0

    #
    #  openStlFiles
    #--------------
    def openStlFiles(self):
        """ stlFileをeditorで開く"""
        #treeList = GtkParts.treeListCols(self.treeView)
        selNames = self.treeList.getSelectedItemNames()
        openNames = list(map(lambda x: x[0], selNames))
        openFiles = list(map(lambda x: self.stlDir + "/" + x, openNames))
        pyTreeFoam.run(self.stlDir).editor(openFiles)

    #
    #  copyStlFiles
    #---------------
    def copyStlFiles(self):
        """ stlFileをclipboardにコピーする"""
        #treeList = GtkParts.treeListCols(self.treeView)
        selNames = self.treeList.getSelectedItemNames()
        copyNames = list(map(lambda x: x[0], selNames))
        copyFiles = list(map(lambda x: self.stlDir + "/" + x, copyNames))
        f = open(self.clipboardFile, "w")
        f.write("foldersCopy\n")
        for fileName in copyFiles:
            print("copy--->", fileName)
            f.write(fileName + "\n")
        f.close()
        copyPasteFolders.copyFolders(copyFiles)

    #
    #  pasteStlFiles
    #----------------
    def pasteStlFiles(self):
        """ stlFileを貼り付ける"""
        #clipboardの内容を取得
        copyDirs = copyPasteFolders.pasteFolders()
        pasteDir = self.stlDir
        pasteNames = []        
        for copyDir in copyDirs:
            name = os.path.basename(copyDir)
            if len(glob.glob(self.stlDir + "/" + name)) != 0:
                #同じfile名が存在する場合
                #file名を修正する
                i = 0
                pasteFileName = pasteDir + "/" + name[:-4] + "_copy" + str(i) + ".stl"
                while len(glob.glob(pasteFileName)) == 1:
                    i += 1
                    pasteFileName = pasteDir + "/" + name[:-4] + "_copy" + str(i) + ".stl"
                #コピー
                shutil.copy(copyDir, pasteFileName)
                pasteNames.append(os.path.basename(pasteFileName))
        #tree再表示
        self.setDataToListBox()
        #選択表示する。
        #treeList = GtkParts.treeListCols(self.treeView)
        allNames = self.treeList.getAllItemNames()
        selNos = []
        for i in range(len(allNames)):
            words = allNames[i]
            if words[0] in pasteNames:
                selNos.append(i)
        self.treeList.selectItemNos(selNos)
        #vtk表示
        self.reloadVtk()

    #
    #  renameStlFile
    #----------------
    def renameStlFile(self):
        """ stlFile名の変更"""
        #treeList = GtkParts.treeListCols(self.treeView)
        selNames = self.treeList.getSelectedItemNames()
        if len(selNames) >= 2:
            title = _("エラー")
            mess = _("複数選択しています。") + "\n"
            mess += _("1ヶのファイルを選択してください。")
            self.errDialog(title, mess)
            return
        #新しいファイル名を取得
        selName = selNames[0][0]
        title = _("ファイル名の変更")
        mess = _("新しいファイル名を入力してください。")
        defText = selName
        okArgs = [self.renameStlFile_run, selName]
        self.inputTextDialog(title, mess, defText, funcOk=okArgs)
        return

    #  renameStlFile_run
    def renameStlFile_run(self, newName, selName):
        #renameする
        if newName.split(".")[-1] != "stl":
            newName += ".stl"
        newNamePath = self.stlDir + "/" + newName
        selNamePath = self.stlDir + "/" + selName
        os.rename(selNamePath, newNamePath)
        #treeListを再表示
        self.setDataToListBox()
        #選択表示させる
        #treeList = GtkParts.treeListCols(self.treeView)
        allNames = self.treeList.getAllItemNames()
        selNo = 0
        for i in range(len(allNames)):
            words = allNames[i]
            if words[0] == newName:
                selNo = i
                break
        self.treeList.selectItemNos([selNo])
        #vtk表示
        self.reloadVtk()

    #
    #  deleteStlFiles
    #-----------------
    def deleteStlFiles(self):
        """ 選択したstlFileを削除する"""
        #treeList = GtkParts.treeListCols(self.treeView)
        selNames = self.treeList.getSelectedItemNames()
        title = _("stlFileの削除")
        mess = _("以下のファイルを削除します。")
        files = ""
        for selName in selNames:
            files += selName[0] + ", "
            if len(files) > 50:
                files = files + "..."
                break
        if len(files) <= 50:
            files = files[:-2]
        mess += "\n  " + files
        okArgs = [self.deleteStlFiles_run, selNames]
        self.okCancelDialog(title, mess, funcOk=okArgs)
        return

    #  deleteStlFiles_run
    def deleteStlFiles_run(self, selNames):
        #削除する
        for selName in selNames:
            namePath = self.stlDir + "/" + selName[0]
            os.remove(namePath)
        #再表示させる
        self.setDataToListBox()
        #vtk表示
        self.reloadVtk()
        title = _("stlFileの削除")
        mess = _("「") + str(len(selNames)) + _("」ヶのファイルを削除しました。")
        self.okDialog(title, mess)


    #  copyFilesToTriSurface
    def copyFilesToTriSurface(self, stlNames):
        """ stlFileをtiSurfaceにコピーする"""
        #caseDir?
        if os.path.exists(self.caseDir + "/constant") == False:
            return
        folder = self.caseDir + "/constant/triSurface"
        if len(glob.glob(folder)) == 0:
            os.mkdir(folder)
        for name in stlNames:
            copyName = self.stlDir + os.sep + name
            pasteName = folder + os.sep + name
            shutil.copy(copyName, pasteName)

    #  setDataToListBox
    def setDataToListBox(self):
        """ TreeViewにdataをセットし直す"""
        stlFiles = glob.glob(self.stlDir + os.sep + "*.stl")
        stlConts = getStlFilesCont(stlFiles)
        #itemを作成
        items = []
        minxyz = [[], [], []]
        maxxyz = [[], [], []]
        for stlCont in stlConts:
            stl = stlCont[0]
            stlPath = self.stlDir + "/" + stl
            icon = self.iconFile
            if os.path.islink(stlPath) == True:
                icon = self.iconFileLink
            fmt = stlCont[1]
            solid = stlCont[2]
            size = "  ".join(map(str, stlCont[3]))
            items.append([icon, stl, fmt, solid, size])
            #minMaxLocを取得
            minLoc = list(map(float, stlCont[4]))
            maxLocx = minLoc[0] + float(stlCont[3][0])
            maxLocy = minLoc[1] + float(stlCont[3][1])
            maxLocz = minLoc[2] + float(stlCont[3][2])
            maxLoc = [maxLocx, maxLocy, maxLocz]
            for i in range(3):
                minxyz[i].append(minLoc[i])
                maxxyz[i].append(maxLoc[i])
        #treeListにitemをセット
        #treeList = GtkParts.treeListCols(self.treeView)
        self.treeList.setItems(items)
        if len(items) > 0:
            #modelのboundsを取得
            minx = min(minxyz[0]); miny = min(minxyz[1]); minz = min(minxyz[2])
            maxx = max(maxxyz[0]); maxy = max(maxxyz[1]); maxz = max(maxxyz[2])
            self.modelBounds = (minx, maxx, miny, maxy, minz, maxz)


    #  selectStlNames
    def selectStlNames(self, stlNames):
        """ 指定stlを選択表示する"""
        #treeList = GtkParts.treeListCols(self.treeView)
        allNames = self.treeList.getAllItemNames()
        selNos = []
        for i in range(len(allNames)):
            if allNames[i][0] in stlNames:
                selNos.append(i)
        self.treeList.selectItemNos(selNos)

    #  getSelectStlNames
    def getSelectStlNames(self):
        """ listBoxから選択しているstlNameを取得する"""
        #treeList = GtkParts.treeListCols(self.treeView)
        selNames = self.treeList.getSelectedItemNames()
        selNames = list(map(lambda x: x[0], selNames))
        return selNames


    #----- universalDialog -----------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def inputTextDialog(self, title, message, entryText, funcOk=[]):
        dialog = inputTextDDialog.getInputText(
            title, message, entryText, funcOk=funcOk, 
            parent=self.mainWindow)
        inputText = dialog.show()
        return inputText
    
    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()


#
#  getStlFilesCont
#
def getStlFilesCont(stlFiles):
    """ stlFileの内容（ascii or binary, size etc.)を取得する
    
    Args:
        stlFiles (list) :確認するstlファイル(fullPath)
    Returns:
        stlファイル毎に以下のlistを返す
        stlName (str)           :stlFileのbaseName
        fileFmts (str)          :"ascii" or "bin"
        solidNames (str)        :solid名
        stlSizes (list(float))  :stlファイルのサイズ[x,y,z]
        minLoc (list(float)     :stlファイルのmin座標"""
    stlConts = []
    for stlFile in stlFiles:
        stlCont = []
        #stlNameを取得
        stlName = stlFile.split(os.sep)[-1]
        stlCont.append(stlName)
        #solidName取得
        if pyTreeFoam.isBinaryStl(stlFile) == False:
            stlCont.append("ascii")
            (solidName, xyzVals) = pyTreeFoam.getStlContFromAscii(stlFile)
        else:
            stlCont.append("bin")
            (solidName, xyzVals) = pyTreeFoam.getStlContFromBin(stlFile)
        stlCont.append(solidName)
        #stlSizeを取得
        (xVals, yVals, zVals) = xyzVals
        if len(xVals) != 0:
            maxx = max(xVals); maxy = max(yVals); maxz = max(zVals)
            minx = min(xVals); miny = min(yVals); minz = min(zVals)
            dx = maxx - minx
            dy = maxy - miny
            dz = maxz - minz
        else:
            dx, dy, dz = 0.0, 0.0, 0.0
        #値を丸める
        dx = pyTreeFoam.getValue3digit_shrink(dx)
        dy = pyTreeFoam.getValue3digit_shrink(dy)
        dz = pyTreeFoam.getValue3digit_shrink(dz)
        stlCont.append([dx, dy, dz])
        #minLocを取得
        minxs = pyTreeFoam.getValue3digit_shrink(minx)
        minys = pyTreeFoam.getValue3digit_shrink(miny)
        minzs = pyTreeFoam.getValue3digit_shrink(minz)
        minLoc = [minxs, minys, minzs]
        stlCont.append(minLoc)
        stlConts.append(stlCont)
    stlConts.sort()
    return stlConts


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    stlDir = sys.argv[2]
    #caseDir = os.getenv("HOME") + "/vtkTest/normalMesh_copy0"
    #caseDir = "/mnt/hgfs/share/CAE/OpenFOAM/OF-7.0/taiyo/4_tanbayama/05_tabayamaMesh4_windTunnel_ESE/1_tanbayamaMesh_2_sqHouse"
    #stlDir = caseDir + "/model"
    winApp = windowApp(caseDir, stlDir)
    winApp.main()
