#!/usr/bin/python3
# coding: utf-8
#
#   createNewCaseDDialog.py
#
#       新しいcaseをtutorialsから選択する。
#
#   13/03/24    新規作成
#      04/18    casesを取得時に、subCaseがダブって表示されていたものを修正
#      04/20    tutorialsのsolver選択時、稀にエラーが発生していたのを修正
#      05/12    cases取得時、取得洩れがあったものを修正
#      10/15    国際化のため、修正
#      10/20    英語版のsolverListデータを準備
#   15/07/13    clipBoardの場所をTreeFoam/dataに変更
#      07/17    dialogの起動方法修正
#   16/12/31    iniSetTutorialsDir:OF-ext用に修正
#   19/09/14    python3, GTK+3に置き換えにより大幅修正
#   20/04/21    多言語化対応
#   22/02/12    iniSetTutorialsDir:tutorialsDirの取得方法を変更
#   24/01/16    OF-11用のtutorials対応で大幅修正。
#      01/18    getItemsSolversCases:case取得時、sortを追加（バグ修正）
#               getCaseDir:itemの説明語の削除を追加（バグ修正）
#      07/29    openFolderDialog:universalDialogs内の関数名変更による修正
#      07/31    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/07    GtkPartsを使用、親window,funcOk,funcCancelを設定
#      08/19    selectItemShowSolver,selectSolversShowCase:item選択時エラー発生。修正
#      08/31    getItemsSolversCases:表示内容に誤りがあり、修正。
#      11/28    getFolderDirsUnderTutorials:コピーできない表示項目があった為、修正
#      12/13    __init__:親のfreezeを削除
#   25/07/10    OF-13への対応。
#      07/11    getItemsSolversCases:区分けを修正（film、legacyを修正）
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib
import os
import sys
import glob

import pyTreeFoam
import GtkParts
import universalDialogs as unvDlg

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

tutorials_data = os.getenv("TreeFoamUserPath") + "/data/tutorials_data"
tutorialsDir = ""
tutoDict = {}
solverDict = {}
OFversion = ""
items = []
solvers = []
solverConts = []
cases = []
solverList = ""
selItemNo = 0
selSolverNo = 0
selCaseNo = 0


#  ---------------
#  createNewCase class
#  ---------------
class createNewCase:
    """ tutorialsからcaseを選択して、dirを返す"""
    
    def __init__(self, dialogTitle, dialogMessage, caseDir, 
                 funcOk=[], funcCancel=[], parent=None):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "createNewCaseDDialog.glade")
        self.dialog = self.builder.get_object("dialog1")
        #親を設定
        self.dialog.set_transient_for(parent)
        #modal(freeze)を設定
        #self.dialog.set_modal(True)
        self.dialog.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        self.maskEvent = True
        self.setGtkObject()
        #変数をセット
        self.dialogTitle = dialogTitle
        self.dialogMessage = dialogMessage
        self.caseDir = caseDir
        self.funcOk = funcOk
        self.funcCancel = funcCancel
        #tutorialsDirを取得
        iniSetTutorialsDir()
        #csvデータ（solverList）読み込み
        getSolverList()
        #tutorials内の項目、solver、caseを取得
        getItemsSolversCases()
        #GUIの初期化
        self.maskEvent = False
        self.subInitialize()

    #
    #  show
    #  ----------
    def show(self):
        """ dialogを表示する"""
        self.dialog.show()
        #self.dialog.run()

    def close(self, *args):
        """ dialogのwindowを直接閉じた時"""
        # self.selectDir = ""
        # self.status = "CANCEL"
        self.dialog.destroy()

    #  ---------- event handler ----------------
    def onChangeDir(self, event):
        self.changeTutorialsDir()
    def onReference(self, event):
        self.getReferenceDir()
    def onChangeItem(self, event):
        if self.maskEvent == False:
            self.selectItemShowSolver()
    def onChangeSolver(self, event):
        if self.maskEvent == False:
            self.selectSolversShowCase()
    def onOk(self, event):
        selectDir, application = self.getTutoCaseDir()
        self.dialog.destroy()
        if len(self.funcOk) > 0:
            funcName = self.funcOk[0]
            args = self.funcOk[1:]
            #funcName(selectDir, *args)
            GLib.timeout_add(100, funcName, selectDir, application, *args)
    def onCancel(self, event):
        self.dialog.destroy()
        if len(self.funcCancel) > 0:
            funcName = self.funcCancel[0]
            args = self.funcCancel[1:]
            #funcName(*args)
            GLib.timeout_add(100, funcName, *args)

    #---------------------------------------------

    def setGtkObject(self):
        """ GUIのid名を取得する"""
        self.label_conts = self.builder.get_object("label_conts")
        self.label_solverCont = self.builder.get_object("label_solverCont")
        self.check_changeDir = self.builder.get_object("check_changeDir")
        self.entry_tutorialsDir = self.builder.get_object("entry_tutorialsDir")
        self.button_reference = self.builder.get_object("button_reference")
        tree_items = self.builder.get_object("tree_items")
        tree_solvers = self.builder.get_object("tree_solvers")
        tree_cases = self.builder.get_object("tree_cases")
        self.tree_items = GtkParts.treeList(tree_items)
        self.tree_solvers = GtkParts.treeList(tree_solvers)
        self.tree_cases = GtkParts.treeList(tree_cases)
        self.tree_solvers.create(multi=False)
        self.tree_items.create(multi=False)
        self.tree_cases.create(multi=False)

    def subInitialize(self):
        """ GUIの初期化"""
        #dialogのtitle、labelをセット
        self.dialog.set_title(self.dialogTitle)
        self.label_conts.set_text(self.dialogMessage)
        #textBox, 参照buttonを隠す
        self.entry_tutorialsDir.set_sensitive(False)
        self.button_reference.set_sensitive(False)
        self.entry_tutorialsDir.set_text(tutorialsDir)
        #listBoxにデータをセット
        self.setListBoxData()

    def setListBoxData(self):
        """ listBoxにデータをセット"""
        global selItemNo, selSolverNo, selCaseNo, tutoDict
        global items, solvers, cases, solverConts
        selItemNo = 0
        selSolverNo = 0
        selCaseNo = 0
        #itemsをセット
        self.tree_items.setItems(items)
        self.tree_items.selectItemNos([0])
        #solversをセット
        selItem = items[0].split(":")[0]
        solvers = list(tutoDict[selItem].keys())
        solvers.sort()
        self.tree_solvers.setItems(solvers)
        self.tree_solvers.selectItemNos([0])
        #casesをセット
        cases = tutoDict[selItem][solvers[0]]
        self.tree_cases.setItems(cases)
        self.tree_cases.selectItemNos([0])
        #solverContをセット
        solver = self.tree_solvers.getSelectedItems()[0]
        solverCont = ""
        if solver in solverDict.keys():
            solverCont = solverDict[solver]
        self.label_solverCont.set_text(solverCont)

    def changeTutorialsDir(self):
        if self.check_changeDir.get_active() == True:
            self.entry_tutorialsDir.set_sensitive(True)
            self.button_reference.set_sensitive(True)
        else:
            self.entry_tutorialsDir.set_sensitive(False)
            self.button_reference.set_sensitive(False)

    def getReferenceDir(self):
        """ 他のtutorialsDirを取得する"""
        global tutorialsDir
        currDir = self.caseDir
        selDir = tutorialsDir
        title = _("tutorialsDirの設定")
        mess = _("使用するTutorialsのdirを選択してください") + "\n"
        mess += _("  変更したTutorialsDirは、保存されます")
        okArgs = [self.getReferenceDir_run]
        self.openFolderDialog(title, selDir, funcOk=okArgs)
        return

    #  getReferenceDir_run
    def getReferenceDir_run(self, newTutoDir):
        """ tutorialsDirを取得する"""
        tutorialsDir = newTutoDir
        self.entry_tutorialsDir.set_text(tutorialsDir)
        writeTutorials_data(OFversion, tutorialsDir)
        print(_("tutorialsDirを変更し、保存しました。"))
        print(_("次回起動時に反映されます。"))
        getItemsSolversCases()
        self.setListBoxData()

    def selectItemShowSolver(self):
        """ itemを選択した時、solversとcasesを再表示する"""
        global items, solvers, selItemNo, selSolverNo, selCaseNo, tutoDict
        global solverDict
        if len(items) == 0:
            return
        selItems = self.tree_items.getSelectedItems()
        if len(selItems) == 0:
            return
        item = selItems[0].split(":")[0]
        selItemNo = items.index(selItems[0])
        #solversをセット
        solvers = list(tutoDict[item].keys())
        solvers.sort()
        selSolverNo = 0
        self.tree_solvers.setItems(solvers)
        self.tree_solvers.selectItemNos([0])
        selSolvers = self.tree_solvers.getSelectedItems()
        if len(selSolvers) == 0:
            self.tree_cases.setItems([])
            return
        solver = selSolvers[0]
        solverCont = ""
        if solver in solverDict.keys():
            solverCont = solverDict[solver]
        self.label_solverCont.set_text(solverCont)
        #casesをセット
        selCaseNo = 0
        self.selectSolversShowCase()

    def selectSolversShowCase(self):
        """ solverを選択した時、casesを再表示する"""
        global items, solvers, cases, selItemNo, selSolverNo, selCaseNo, tutoDict
        if len(items) == 0:
            return
        item = items[selItemNo].split(":")[0]
        if len(solvers) == 0:
            self.tree_solvers.setItems([])
            self.label_solverCont.set_text("")
            self.tree_cases.setItems([])
            return
        selSolvers = self.tree_solvers.getSelectedItems()
        if len(selSolvers) == 0:
            return
        solver = selSolvers[0]
        allSolvers = self.tree_solvers.getAllItems()
        selSolverNo = allSolvers.index(solver)
        #solver内容をセット
        solverCont = ""
        if solver in solverDict.keys():
            solverCont = solverDict[solver]
        self.label_solverCont.set_text(solverCont)
        #caseを取得
        cases = tutoDict[item][solver]
        self.tree_cases.setItems(cases)
        self.tree_cases.selectItemNos([0])
        selCaseNo = 0

    def getTutoCaseDir(self):
        """ 選択しているTutorialsのcaseDirを取得"""
        global selCaseNo
        item = items[selItemNo]
        solver = solvers[selSolverNo]
        caseName = self.tree_cases.getSelectedItems()[0]
        allItems = self.tree_cases.getAllItems()
        selCaseNo = allItems.index(caseName)
        ansDir = getCaseDir(item, solver, caseName)
        application = getApplicationNameFromTutoData(item, solver,caseName)
        return ansDir, application

    #----- universalDialogs ---------------
    #  openFolderDialog
    def openFolderDialog(self, title, showDir, 
                         funcOk=[], funcCancel=[], 
                         selName="", patterns=[]):
        dialog = unvDlg.openFolderDialog(
            title, showDir,
            funcOk=funcOk, funcCancel=funcCancel,
            parentWin=self.dialog, selName=selName,
            filterSets=patterns)
        dialog.show()

#
#  iniSetTutorialsDir
#
def iniSetTutorialsDir():
    """ tutorialsDirを取得して、tutorials_dataファイルに保存する"""
    global tutorialsDir, OFversion
    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    if len(glob.glob(tutorials_data)) != 0:
        [ver, tutoDir] = readTutorials_data()
        if (len(glob.glob(tutoDir)) != 0) and (ver == OFversion):
            tutorialsDir = tutoDir
            return
    #tutorialsDirを取得
    tutorialsDir = ""
    if OFversion[:3] == "ext":
        #foam-extend
        ver = OFversion[4:]
        tutoDir = os.getenv("HOME") + "/foam/" + os.getenv("USER") + "-" + ver + "/run/tutorials"
        if len(glob.glob(tutoDir)) == 0:
            tutoDir = os.getenv("HOME") + "/foam/foam-extend-" + ver + "/tutorials"
        tutorialsDir = tutoDir
    else:
        #OpenFOAM
        tutoDir = os.getenv("HOME") + "/OpenFOAM/" + os.getenv("USER") + "-" + OFversion + "/run/tutorials"
        if len(glob.glob(tutoDir)) == 0:
            tutoDir = os.getenv("HOME") + "/OpenFOAM/OpenFOAM-" + OFversion + "/tutorials"
        tutorialsDir = tutoDir
    if len(glob.glob(tutorialsDir)) == 0:
        #環境変数からdirを取得
        bashrcFOAM = configDict["bashrcFOAM"]
        comm = ". " + bashrcFOAM + "; "
        comm += "echo $WM_PROJECT_DIR"
        _stat, res, _err = pyTreeFoam.run().commandReturnCont(comm)
        resLines = res.split("\n")
        if resLines[-1] != "":
            tutorialsDir = resLines[-1] + "/tutorials"
        else:
            tutorialsDir = resLines[-2] + "/tutorials"
    #ファイルに書き込み
    writeTutorials_data(OFversion, tutorialsDir)
    return

#
#  readTutorials_data
#
def readTutorials_data():
    """ OFversionとtutorialsDirをファイルから読み込み"""
    f=open(tutorials_data); cont=f.readlines(); f.close()
    ver = ""
    tutoDir = ""
    for line in cont:
        comms = line.split()
        if len(comms) > 1:
            item = comms[0]
            data = comms[1]
            if item == "OFversion":
                ver = data
            elif item == "tutorialsDir":
                tutoDir = data
    return [ver, tutoDir] 

#
#  writeTutorials_data
#
def writeTutorials_data(ver, tutoDir):
    """ tutorials_dataの書き込み"""
    cont  = "\n"
    cont += "OFversion    " + ver + "\n"
    cont += "tutorialsDir " + tutoDir + "\n"
    f=open(tutorials_data, "w")
    f.write(cont)
    f.close()

#
#  getSolverList
#
def getSolverList():
    global listData, OFversion
    #環境に応じてsolverListファイルを設定する
    if os.getenv("LANG")[:2] == "ja":
        solverList = os.getenv("TreeFoamUserPath") + "/data/solverList.csv"
    else:
        solverList = os.environ["TreeFoamUserPath"] + "/data/solverList_en.csv"
    fileName = solverList
    solverConts = pyTreeFoam.readCsvData(fileName)
    listData = solverConts[1:]

#
#  getItemName
#
def getItemName(itemCont):
    """ 区分からitemを取得する"""
    item = itemCont.split(":")[0]
    if len(item) == len(itemCont):
        item = itemCont.split("：")[0]
    return item

#
#  getSolverNames
#
def getSolverNames(item):
    """ itemに該当するsolver名を取得（nestingは2段まで）"""
    itemName = getItemName(item)
    solverList = glob.glob(tutorialsDir + "/" + itemName + "/*")
    solvers = []
    if len(solverList) > 0:
        # solverを取得
        for solverDir in solverList:
            name = solverDir.split("/")[-1]
            if name[-4:] == "Foam" or name[-4:] == "Mesh":
                solvers.append(name)
            else:
                subList = glob.glob(solverDir + "/*")
                if len(subList) > 0:
                    for subDir in subList:
                        name = subDir.split("/")[-1]
                        if name[-4:] == "Foam" or name[-4:] == "Mesh":
                            solvers.append(name)
        solvers.sort()
    return solvers

#
#  getSolverDir
#
def getSolverDir(item, solver):
    """ itemとsolverNameからsolverDirを取得"""
    itemName = getItemName(item)
    checkDir = tutorialsDir + "/" + itemName
    solverDir = checkDir + "/" + solver
    return solverDir

#
#  getApplicationNameFromTutoData
#
def getApplicationNameFromTutoData(item, solver,caseName):
    """ tutorialsのsolver名（application）を取得"""
    item = getItemName(item)
    app = ""
    if item == "foamRun" or item == "foamMultiRun":
        app = item
    elif solver == ".":
        if item[-4:] == "Foam":
            app = item
    elif solver[-4:] == "Foam":
        app = solver
    elif item[:4] == "mesh":
        if solver[-4:] == "Mesh":
            app = solver
    return app

#
#  getCaseDir
#
def getCaseDir(item, solver, case):
    """ tutorialsのcaseDirを取得"""
    global tutoDict, tutorialsDir
    # solverDir = getSolverDir(item, solver)
    # caseDir = solverDir + "/" + case
    
    item = item.split(":")[0]
    if item == "foamRun" or item == "foamMultiRun":
        itemDir = tutorialsDir
    else:
        itemDir = tutorialsDir + "/" + item
    solverDir = itemDir + "/" + solver
    caseDir = solverDir + "/" + case
    return caseDir

#
#  getCaseNames
#
def getCaseNames(item, solver):
    """ solverに該当するcase名を取得"""
    #solverDirの取得
    solverDir = getSolverDir(item, solver)
    #caseを取得
    #  3レベルまで探す
    caseNames = []
    folders = pyTreeFoam.getFolders(solverDir)
    for checkDir in folders:
        if pyTreeFoam.getFolderType(checkDir) == "case":
            folder = checkDir.split("/")[-1]
            caseNames.append(folder)
        else:
            folders1 = pyTreeFoam.getFolders(checkDir)
            for checkDir1 in folders1:
                if pyTreeFoam.getFolderType(checkDir1) == "case":
                    folder = "/".join(checkDir1.split("/")[-2:])
                    caseNames.append(folder)
                else:
                    folders2 = pyTreeFoam.getFolders(checkDir1)
                    for checkDir2 in folders2:
                        if pyTreeFoam.getFolderType(checkDir2) == "case":
                            folder = "/".join(checkDir2.split("/")[-3:])
                            caseNames.append(folder)
    caseNames.sort()
    return caseNames

#
#  getApplicationName
def getApplicationName(folder):
    case = pyTreeFoam.case(folder)
    if os.path.exists(folder + "/system/controlDict") == False:
        files = glob.glob(folder + "/system/controlDict.*")
        if len(files) == 0:
            return ""
        else:
            name = os.path.basename(files[0])
            contDict = case.getControlDict(name)
            app = contDict["application"]
    else:
        contDict = case.getControlDict()
        app = contDict["application"]
    return app

#
#  getApplicationName13
def getApplicationName13(folder):
    case = pyTreeFoam.case(folder)
    if os.path.exists(folder + "/system/controlDict") == False:
        files = glob.glob(folder + "/system/controlDict.*")
        if len(files) == 0:
            return ""
        else:
            contName = os.path.basename(files[0])
            contDict = case.getControlDict(name=contName, remake=False)
    else:
        contDict = case.getControlDict(remake=False)
    #contDict内を確認
    solvers = contDict["regionSolvers"]
    if len(solvers) != 0:
        app = "foamMultiRun"
    else:
        solver = contDict["solver"]
        if solver != "":
            app = "foamRun"
        else:
            app = ""
    return app

#
#  getMainSolverNameFromTutorials
def getMainSolverNameFromTutorials(folder):
    """ folderNameのsolver名を調べる"""
    global OFversion

    def getAppInFolder(folder):
        folders = pyTreeFoam.getFolders(folder)
        app = ""
        for i in range(len(folders)):
            folder = folders[i]
            if pyTreeFoam.isCaseDir(folder) == True:
                app = getApplicationName(folder)
                if app == "foamRun" or app == "foamMultiRun":
                    break
        return app
    
    def getAppInFolder13(folder):
        folders = pyTreeFoam.getFolders(folder)
        app = ""
        for i in range(len(folders)):
            folder = folders[i]
            if pyTreeFoam.isCaseDir(folder) == True:
                app = getApplicationName13(folder)
                if app == "foamRun" or app == "foamMultiRun":
                    break
        return app

    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer >= 13.0:
        #OF-13以上
        app = getApplicationName13(folder)
        if app == "foamRun" or app == "foamMultiRun":
            return app
        app = getAppInFolder13(folder)
        return app
    else:
        #通常OF、OF-12以下
        app = getApplicationName(folder)
        if app == "foamRun" or app == "foamMultiRun":
            return app
        app = getAppInFolder(folder)
        return app

#
#  getFolderDirsUnderTutorials
def getFolderDirsUnderTutorials():
    """ tutorialsDir以下のcase、folderを取得"""
    global tutorialsDir
    caseDirs = []; folderDirs = []
    caseDir = ""
    for dirPath, dirNames, files in os.walk(tutorialsDir):
        if ("system" in dirNames or "constant" in dirNames):
            caseDir = dirPath
            caseDirs.append(caseDir)
        elif len(dirNames) == 0:
            if dirPath[:len(caseDir)+1] != caseDir + "/":
                folderDirs.append(dirPath)
    folderDirs += caseDirs
    #folderDirsを整形
    tutoDataSet = set([])
    for folder in folderDirs:
        words = folder[len(tutorialsDir)+1:].split("/")
        words = words[:5]
        if len(words) >= 2:
            if len(words) == 2:
                words = [words[0], ".", words[1]]
            elif (words[0] != "mesh" and words[0] != "legacy"):
                #for n in range(3, len(words)):
                for n in range(2, len(words)):
                    dir = tutorialsDir + "/" + "/".join(words[:n])
                    #if os.path.exists(dir + "/Allclean") == True:
                    if os.path.exists(dir + "/Allclean") == True or os.path.exists(dir + "/Allrun"):
                        words = words[:n]
                        if len(words) == 2:
                            words = [words[0], ".", words[1]]
                        break
            wordsTpl = tuple(words)
            tutoDataSet.add(wordsTpl)
    tutoDataList = list(tutoDataSet)
    tutoDataList.sort()
    return tutoDataList

#
#  remakeTutoDict
def remakeTutoDict(tutoDict, foamRun):
    """ tutoDictから漏れてitemに登録されているsolverを
    foamRun, foamMultiRunに組み込む"""
    if not foamRun in tutoDict.keys():
        return tutoDict
    solvers = tutoDict[foamRun].keys()
    delItems = []
    for itemName in tutoDict.keys():
        if itemName != foamRun:
            if itemName in solvers:
                delItems.append(itemName)
                solverCase = tutoDict[itemName]["."]
                #漏れたitemをfoamRunに追加
                if not itemName in tutoDict[foamRun].keys():
                    tutoDict[foamRun][itemName] = []
                tutoDict[foamRun][itemName] += solverCase
    #  漏れた分をitemから削除
    for itemName in delItems:
        delCase = tutoDict.pop(itemName)
    return tutoDict

#
#  getItemsSolversCases
#
def getItemsSolversCases():
    """ tutorials内から項目、solver、caseを取得する"""
    global items, solvers, solverConts, cases, tutoDict, solverDict
    global listData, tutorialsDir, OFversion

    #tutorials内の全folderDirsを取得
    tutoDataList = getFolderDirsUnderTutorials()

    #辞書を作成
    tutoDict = {}
    for tutoData in tutoDataList:
        itemName = tutoData[0]
        solverName = tutoData[1]
        case = list(tutoData[2:])
        case = "/".join(case)
        caseDir = "/".join([tutorialsDir, itemName, solverName, case])
        app = getMainSolverNameFromTutorials(caseDir)
        #OF-13の場合appが取得できていない為、訂正
        if app == "":
            if itemName == "multiRegion":
                app = "foamMultiRun"
            elif itemName == "film":
                app = "foamRun"
        #legacyを訂正
        if itemName == "legacy":
            itemName = "/".join(tutoData[:2])
            solverName = tutoData[2]
            case = list(tutoData[3:])
            case = "/".join(case)
        #itemとsolverを設定
        if app == "foamRun" or app == "foamMultiRun":
            #itemとsolverを入れ替え
            solverName = itemName
            itemName = app
            if app == "foamMultiRun":
                #foamMultiRunは、case再取得(25/07/10追加)
                case = list(tutoData[1:])
                case = "/".join(case)
        #itemとsolverを取得
        if not itemName in tutoDict.keys():
            tutoDict[itemName] = {}
        if not solverName in tutoDict[itemName].keys():
            tutoDict[itemName][solverName] = []
        #caseを取得
        tutoDict[itemName][solverName].append(case)
    #formRunのsolverをチェック
    tutoDict = remakeTutoDict(tutoDict, "foamRun")
    #tutoDict = remakeTutoDict(tutoDict, "foamMultiRun")

    solvers = list(tutoDict.keys())
    solvers.sort()
    #tutorialsの区分（items）をlistDataから取得
    itemContDict = {}
    for i in range(len(listData)):
        item = listData[i][0].split(":")[0]     #solver名
        itemName = listData[i][0]
        if item in tutoDict.keys():
            itemContDict[item] = [i, itemName]
    items = []
    for item in tutoDict.keys():
        if item in itemContDict.keys():
            itemNo = itemContDict[item][0]
            itemName = itemContDict[item][1]
        else:
            itemNo = 100
            itemName = item
        items.append([itemNo, itemName])
    items.sort()
    items = list(map(lambda x: x[1], items))

    #solverDict（solverの内容）を取得
    solverDict = {}
    for i in range(len(listData)):
        solver = listData[i][1]
        solverCont = listData[i][2]
        solverDict[solver] = solverCont
    


if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    # os.environ["TreeFoamPath"] = "/home/caeuser/TreeFoam-glade3"
    # app = dialogApp("title", "mmesage", ".")
    # app.showDialog()
