#!/usr/bin/python3
# coding: utf-8
#
#   createBlockAndSnappyDict.py
#
#       blockMesh、snappyHexMeshDict作成用のcsvFileの作成
#       csvFileからmeshDictを作成する。
#
#   13/10/20    新規作成
#      10/26    バグ修正
#      12/02    sectにregを追加（refinementRegionのみを追加）
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/01/05    blockMeshDict中の「convertToMeters」のチェック追加
#      02/04    modelSize取得（getmodelSize）時、patchのみ→全stlに変更
#      03/18    checkStlFileName:圧縮時のfileNameが取得できなかった事を修正
#      07/12    sectに「face」を追加
#      07/17    sectに「faceZone」を追加
#      07/18    allowFaceZoneを操作する様に修正
#      09/07    setCastellatedSnapFlagを追加（snappyHexMeshのflagをセット）
#   15/01/22    setGeometryData:バグ修正（"\n"が余分にｾｯﾄされていた）
#      09/18    sectに「wall」を追加
#      10/03    sectに「empty、symmetry」を追加。
#               addLayersのflagを「false」に固定。
#      11/11    readCsvData:snappy用のcsvファイルかどうかエラーチェック追加
#      12/12    getFoamContents.getKeywordPointerMemをgetKeywordPointerに変更
#               getKeywordPointerは、コメントが有っても検索できる。
#   19/05/20    checkBlockMeshMag:倍率（convertToMeters, scale）を「1」に設定を追加
#      05/22    checkStlFileName:mkdirをos.mkdir、cpをshutil.copyに修正
#      10/18    GTK+3, python3用に大幅修正
#   20/02/10    stringOpをpyTreeFoam.strOpに変更
#      04/21    多言語化対応
#   21/07/28    import editStlFilesDialogを廃止。
#   22/01/24    createMeshDict:patchTypeNameのチェックを追加
#      07/17    setLocationInMesh:OFversionによる処理追加（OF-10対応）
#               createMeshDict:checkBlockMeshEmptyを追加（OF-10対応）
#   24/04/07    getValueCsvData:sectItemsに"overset”を追加。
#               setRefinementSurfaces:patchTypeにoversetを追加。
#      07/16    setGeometryData:OF-12からDict内のgeometryの書式が変わった為、
#               OFversionの分岐を追加。
#      11/17    TYPE_NAMES,checkStlFileName,setRefinementRegions:"region"を追加。
#               "reg" or "region"でも使える様に修正。
#               setRefinementSurfaces:surfaceは、surfaceのみのlevelで分割する様に修正
#      11/20    readCsvData:cellZoneOtherをcellZoneに変えてdataの最後に移動を追加
#      11/26    setRefinementSurfaces:refinementSurfaceのlevelをmaxLevelに統一。
#      12/10    readCsvData:stlのdataをcellZone,cellZoneOther,faceZone,その他に
#               分類後、記述順をその他、cellZone,cellZoneOther,faceZoneに修正
#


import glob
import math
import os
import shutil

import pyTreeFoam

#patchTypeを定義
TYPE_NAMES = [
    "wall", "patch", "overset", "cellZone", "reg", "region", "faceZone", "face", 
    "empty", "symmetry", "symmetryPlane"
    ]

#
#  setCsvData
def setCsvData():
    """ defaultのcsvの内容を作成"""
    row = []
    row.append(["", "", "", "", "", ""])
    row.append(["<blockMesh>", "", "x", "y", "z", _("備考")])
    row.append(["", "cellSize", "0.002", "0.002", "0.002", _("blockMeshのcellSize")])
    row.append(["", "overBlockSize", "5", "5", "5" ,_('"cells: stlのMinMax値を越えるcell数"')])
    row.append(["", "", "", "", "", ""])
    row.append(["<snappyHexMesh>", "", "", "", "", ""])
    row.append(["", "mesh", "0.0", "0.0", "0.0", _("meshの位置(materialPoint)")])
    row.append(["stlFile", '"sect\n(patch/wall/\nempty/symmetry(Plane)/\nfaceZone/face/\ncellZone/reg)"', '"featureEdge\ncellSize"',
            '"base\ncellSize"', '"fine\ncellSize"', _('"featureEdge: cellSizeを入力したstlのみ抽出。\nbase: surface, regionとも設定する。"')])
    return row

#
#  addStlFileName
def addStlFileName(stlDir, data):
    """ csvにstlファイル名を追加"""
    files = []
    stlFiles = glob.glob(stlDir + "/*.stl")
    for stlFile in stlFiles:
        name = stlFile.split("/")[-1].split(".")[0]
        files.append(name)
    files.sort()
    for name in files:
        line = [name, "", "", "", "", ""]
        data.append(line)
    return data

#
#  getMinMaxValueStl
def getMinMaxValueStl(fileName):
    """ stlファイルのminMax値を取得"""
    if pyTreeFoam.isBinaryStl(fileName) == True:
        #binaryの読み込み
        (_solidName, xyzVals) = pyTreeFoam.getStlContFromBin(fileName)
    else:
        #asciiの読み込み
        (_solidName, xyzVals) = pyTreeFoam.getStlContFromAscii(fileName)
    (xVals, yVals, zVals) = xyzVals
    if len(xVals) != 0:
        minx = min(xVals); maxx = max(xVals)
        miny = min(yVals); maxy = max(yVals)
        minz = min(zVals); maxz = max(zVals)
    else:
        minx=0.0; maxx=0.0
        miny=0.0; maxy=0.0
        minz=0.0; maxz=0.0
    return [[minx, miny, minz], [maxx, maxy, maxz]]

#
#  addBlockCellSize
def addBlockCellSize(stlDir, data):
    """ blockMeshのcellSize（30分割）"""
    flag = 0
    xVals=[]; yVals=[]; zVals=[]
    i = 0
    cellSize = -1
    for row in data:
        if flag == 1:
            fileName = stlDir + "/" + row[0] + ".stl"
            [mins, maxs] = getMinMaxValueStl(fileName)
            xVals.append(mins[0])
            xVals.append(maxs[0])
            yVals.append(mins[1])
            yVals.append(maxs[1])
            zVals.append(mins[2])
            zVals.append(maxs[2])
        if row[0] == "stlFile":
            flag = 1
        if row[1] == "cellSize":
            cellSize = i
        if row[1] == "mesh":
            meshLoc = i
        i += 1
    if len(xVals) != 0:
        minx = min(xVals); maxx = max(xVals)
        miny = min(yVals); maxy = max(yVals)
        minz = min(zVals); maxz = max(zVals)
    else:
        minx = 0.0; maxx = 0.0
        miny = 0.0; maxy = 0.0
        minz = 0.0; maxz = 0.0
    #cellSizeの設定
    dxf = (maxx - minx) * 1.0 / 30.0
    dyf = (maxy - miny) * 1.0 / 30.0
    dzf = (maxz - minz) * 1.0 / 30.0
    dx = pyTreeFoam.getValue3digit(dxf)
    dy = pyTreeFoam.getValue3digit(dyf)
    dz = pyTreeFoam.getValue3digit(dzf)
    if cellSize >= 0:
        data[cellSize][2:5] = [str(dx), str(dy), str(dz)]
    #meshLocationの設定
    meshx = (maxx + minx)/2.0
    meshy = (maxy + miny)/2.0
    meshz = (maxz + minz)/2.0
    data[meshLoc][2:5] = [str(meshx), str(meshy), str(meshz)]
    return data

#
#  addStlBlockSize
def addStlBlockSize(stlDir, data):
    """ stlのblockサイズを取得"""
    flag = 0
    i = 0
    for row in data:
        if flag == 1:
            fileName = stlDir + "/" + row[0] + ".stl"
            [x, y, z] = getStlBlockSize(fileName)
            if not (x==0.0 and y==0.0 and z==0.0):
                data[i][5] = '"(' + ' '.join([str(x), str(y), str(z)]) + ')"'
        if row[0] == "stlFile":
            flag = 1
        i += 1
    return data

#
#  getStlBlockSize
def getStlBlockSize(fileName):
    """ stlのblockSizeを取得"""
    minMax = getMinMaxValueStl(fileName)
    dx = minMax[1][0] - minMax[0][0]
    dy = minMax[1][1] - minMax[0][1]
    dz = minMax[1][2] - minMax[0][2]
    dx = pyTreeFoam.getValue3digit(dx)
    dy = pyTreeFoam.getValue3digit(dy)
    dz = pyTreeFoam.getValue3digit(dz)
    return [dx, dy, dz]

#
#  writeCsvData
def writeCsvData(stlDir, fileName, data):
    """ csvデータの書き込み"""
    f=open(stlDir+"/"+fileName, "w")
    for line in data:
        cont = ",".join(line) + "\n"
        f.write(cont)
    f.close()
    return

#
#  readCsvData
def readCsvData(fileName):
    """ csvデータの読み込み"""
    conts = pyTreeFoam.readCsvData(fileName)
    #dataの取り込み
    data = []; otherData = []; cellSize = []; blockSize = []
    meshLoc = []; readStl = 0
    cellZoneOther = []
    for row in conts:
        flag = 0
        for val in row:
            if val != "":
                flag = 1
                break
        if flag == 1:
            if row[1] == "cellSize":
                cellSize = [row[2], row[3], row[4]]
            elif row[1] == "blockSize" or row[1] == "overBlockSize":
                blockSize = [row[2], row[3], row[4]]
            elif row[1] == "mesh":
                meshLoc = [row[2], row[3], row[4]]
            elif row[0] == "stlFile":
                readStl = 1
            else:
                if readStl == 1:
                    data.append(row)
    #cellZone, cellZoneOther, faceZone, その他に分類する
    cellZones = []; cellZoneOther = []; faceZones = []; others = []
    for row in data:
        if row[1] == "cellZone":
            cellZones.append(row)
        elif row[1] == "cellZoneOther":
            row[1] = "cellZone"
            cellZoneOther = row
        elif row[1] == "faceZone":
            faceZones.append(row)
        else:
            others.append(row)
    #dataを再作成
    newData = others
    if len(cellZones) > 0:
        #通常patchの後にcellZoneを追加
        newData += cellZones
    if len(cellZoneOther) > 0:
        #cellZoneOtherは、cellZoneの最後
        newData += [cellZoneOther]
    if len(faceZones) > 0:
        #faceZoneは最後
        newData += faceZones
    data = newData
    #空白cellに値をセット
    cellSize = setNullValue(cellSize)
    blockSize = setNullValue(blockSize)
    #その他データをセット
    otherData = [cellSize, blockSize, meshLoc]
    return (otherData, data)

#
#  setNullValue
def setNullValue(vals):
    """ 空白cellに値をセット"""
    if len(vals) == 0:
        return vals
    #空白のチェック
    if vals[0] == "":
        vals = [0, 0, 0]
    if vals[1] == "":
        vals[1] = vals[0]
    if vals[2] == "":
        vals[2] = vals[0]
    #数字のチェック
    i = 0
    while i < len(vals):
        try:
            float(vals[i])
        except:
            vals[i] = "0.0"
        i += 1
    return vals

#
#  getBaseCellSize
def getBaseCellSize(loc):
    """ csvのblockCellSizeをチェックし、取得する"""
    size = []
    error = 0
    for val in loc:
        if val != "":
            try:
                size.append(float(val))
            except:
                error = 1
                size.append(0.0)
    if error == 0:
        return (0, size)
    else:
        print(_("error: blockMeshのcellSizeの値が間違っています。"))
        return (1, size)

#
#  getValueCsvData
def getValueCsvData(cellSize, data):
    """ csv データをstlMinMaxのlist変数に代入する"""
    stl = []; sect = []; featureEdge = []; minLevel = []; maxLevel = []
    error = 0
    for line in data:
        try:
            stl.append(line[0])
            sect.append(line[1])
            featureEdge.append(line[2])
            minLevel.append(line[3])
            maxLevel.append(line[4])
        except:
            error = 1
            print(_("error: csvファイルから値が取得できません。"))
            break
    dAve = sum(cellSize) / len(cellSize)
    if error == 0:
        i = 0
        sectItems = TYPE_NAMES
        while i<len(featureEdge):
            #sect, minLevelの関係をチェック
            if sect[i] in sectItems:
                if minLevel[i] == "":
                    error = 1
                    print(_("error: patch/cellZoneに対し、cellSizeが設定されていません。\nbaseのcellSizeは、省略できません。"))
                else:
                    try:
                        a = float(minLevel[i])
                        if a == 0:
                            error = 1
                            print(_("error: cellSizeが「0」になっています。\ncellSize=0では、メッシュが作れません。"))
                    except:
                        error = 1
                        print(_("error: cellSizeは、数字を入力してください。\ncellSizeに文字が入力されています。"))
            #cellSizeをlevelに変換
            if featureEdge[i] != "":
                level = calcCellLevel(dAve, featureEdge[i])
                featureEdge[i] = str(level)
            if minLevel[i] != "":
                level = calcCellLevel(dAve, minLevel[i])
                minLevel[i] = str(level)
            if maxLevel[i] != "":
                level = calcCellLevel(dAve, maxLevel[i])
                maxLevel[i] = str(level)
            else:
                #maxLevelが空白の場合、minLvelと同じ値にセット
                maxLevel[i] = minLevel[i]
            if error != 0:
                break
            i += 1
    stlMinMax = [stl, sect, featureEdge, minLevel, maxLevel]
    return (error, stlMinMax)

#
#  calcCellLevel
def calcCellLevel(dAve, strDx):
    """ cell寸法をcellLevelに変換する"""
    level = 0
    try:
        a = dAve / float(strDx)
        if a > 1.0:
            level = int(math.log(a) / math.log(2) + 0.5)
        else:
            level = 0
    except:
        level = 0
    return level


#
#  checkStlFileName
def checkStlFileName(caseDir, stlDir, stlMinMax):
    """ stlFile有無チェック"""
    [stl, sect, featureEdge, _minLevel, _maxLevel] = stlMinMax
    error = 0
    fileNames = []
    i = 0
    for name in stl:
        if sect[i] in TYPE_NAMES:
            fileNames.append(name+".stl")
        if featureEdge[i] != "":
            fileNames.append(name+".eMesh")
        i += 1
    noFiles = []
    if len(fileNames) != 0:
        for name in fileNames:
            fileName = stlDir + "/" + name
            if len(glob.glob(fileName)) == 0:
                fileName = pyTreeFoam.case().checkFileName(fileName)
                if len(glob.glob(fileName)) == 0:
                    noFiles.append(name)
        if len(noFiles) != 0:
            error = 1 
    else:
        print(_("error: patch/cellZone/regが未記入です。"))
        error = 1
    if error == 0:
        #必要なfileをセット
        distDir = caseDir + "/constant/triSurface"
        if len(glob.glob(distDir)) == 0:
            os.mkdir(distDir)
        for name in fileNames:
            sourceName = stlDir + "/" + name
            distName = distDir + "/" + name
            sourceName = pyTreeFoam.case().checkFileName(sourceName)
            distName = pyTreeFoam.case().checkFileName(distName)
            shutil.copy(sourceName, distName)
    return (error, noFiles)

#
#  checkNeededEMeshFileName
def checkNeededEMeshFileName(stlMinMax):
    [stl, _sect, featureEdge, _minLevel, _maxLevel] = stlMinMax
    eMeshFiles = []
    i = 0
    for name in stl:
        if featureEdge[i] != "":
            eMeshFiles.append(name+".stl")
        i += 1
    return eMeshFiles

#
#  getModelSize
def getModelSize(stlDir, stl, sect):
    """ modelのmin,max値を取得"""
    fileNames=[]
    for name in stl:
        fileNames.append(name+".stl")
    xValues=[]; yValues=[]; zValues=[]
    for fileName in fileNames:
        [minxyz, maxxyz] = getMinMaxValueStl(stlDir+"/"+fileName)
        xValues.append(minxyz[0])
        xValues.append(maxxyz[0])
        yValues.append(minxyz[1])
        yValues.append(maxxyz[1])
        zValues.append(minxyz[2])
        zValues.append(maxxyz[2])
    minx = min(xValues); maxx = max(xValues)
    miny = min(yValues); maxy = max(yValues)
    minz = min(zValues); maxz = max(zValues)
    return [[minx, miny, minz], [maxx, maxy, maxz]]

#
#  getBlockSize
def getBlockSize(modelSize, bSize, nAddCells):
    """ blockMeshのサイズを求める"""
    nCellx = float(nAddCells[0])
    nCelly = float(nAddCells[1])
    nCellz = float(nAddCells[2])
    [dx, dy, dz] = bSize
    minx = float(int((modelSize[0][0] - nCellx*dx)/dx - 0.5))*dx
    miny = float(int((modelSize[0][1] - nCelly*dy)/dy - 0.5))*dy
    minz = float(int((modelSize[0][2] - nCellz*dz)/dz - 0.5))*dz
    maxx = float(int((modelSize[1][0] + nCellx*dx)/dx + 0.5))*dx
    maxy = float(int((modelSize[1][1] + nCelly*dy)/dy + 0.5))*dy
    maxz = float(int((modelSize[1][2] + nCellz*dz)/dz + 0.5))*dz
    return [[minx, miny, minz], [maxx, maxy, maxz]]

#
#  readBlockMeshDict
def readBlockMeshDict(caseDir):
    """ blockMeshDictの内容を読み込み返す"""
    fileName = caseDir + "/system/blockMeshDict"
    if len(glob.glob(fileName)) == 0:
        fileName = caseDir + "/constant/polyMesh/blockMeshDict"
        if len(glob.glob(fileName)) == 0:
            cont = ""
            return
    f=open(fileName); cont=f.read(); f.close()
    return cont

#
#  readSnappyHexMeshDict
def readSnappyHexMeshDict(caseDir):
    """ Dictファイルを読み込む"""
    fileName = caseDir + "/system/snappyHexMeshDict"
    if len(glob.glob(fileName)) == 0:
        cont = ""
    else:
        f=open(fileName); cont=f.read(); f.close()
    return cont

#
#  setBlockPointData
def setBlockPointData(blockCont, blockSize, baseCell):
    """ blockMeshDictの座標作成"""
    blockContOp = pyTreeFoam.strOp(blockCont)
    pVer = blockContOp.getKeywordPointer("vertices", 0)
    if pVer < 0:
        return blockCont
    (dummy, pEnd) = blockContOp.getSmallPair(pVer)
    minx=blockSize[0][0]; miny=blockSize[0][1]; minz=blockSize[0][2]
    maxx=blockSize[1][0]; maxy=blockSize[1][1]; maxz=blockSize[1][2]
    p0 = " ".join([str(minx), str(miny), str(minz)])
    p1 = " ".join([str(maxx), str(miny), str(minz)])
    p2 = " ".join([str(maxx), str(maxy), str(minz)])
    p3 = " ".join([str(minx), str(maxy), str(minz)])
    p4 = " ".join([str(minx), str(miny), str(maxz)])
    p5 = " ".join([str(maxx), str(miny), str(maxz)])
    p6 = " ".join([str(maxx), str(maxy), str(maxz)])
    p7 = " ".join([str(minx), str(maxy), str(maxz)])
    cont = "vertices\n" + 4*" " + "(\n"
    cont += 8*" " + "( " + p0 + " )\n"
    cont += 8*" " + "( " + p1 + " )\n"
    cont += 8*" " + "( " + p2 + " )\n"
    cont += 8*" " + "( " + p3 + " )\n"
    cont += 8*" " + "( " + p4 + " )\n"
    cont += 8*" " + "( " + p5 + " )\n"
    cont += 8*" " + "( " + p6 + " )\n"
    cont += 8*" " + "( " + p7 + " )\n"
    cont += 4*" " + ")"
    blockCont = blockCont[:pVer] + cont + blockCont[pEnd:]
    blockContOp = pyTreeFoam.strOp(blockCont)
    pBlo = blockContOp.getKeywordPointer("blocks", pVer)
    (dummy, pEnd) = blockContOp.getSmallPair(pBlo)
    [dx, dy, dz] = baseCell
    nx = int((maxx-minx)/dx + 0.5)
    ny = int((maxy-miny)/dy + 0.5)
    nz = int((maxz-minz)/dz + 0.5)
    cont = ("blocks\n" + 4*" " + "(\n"
            + 4*" " + "hex (0 1 2 3 4 5 6 7) ("
            + " ".join([str(nx), str(ny), str(nz)])
            + ") simpleGrading (1 1 1)\n"
            + 4*" " + ")")
    newCont = blockCont[:pBlo] + cont + blockCont[pEnd:]
    return newCont

#
#  checkBlockMeshMag
def checkBlockMeshMag(cont):
    """ blockMeshDictの倍率（convertToMeters, scale）のチェック修正"""
    contOp = pyTreeFoam.strOp(cont)
    #FoamFileをskip
    p = contOp.skipFoamFile(0)
    ps = p
    #"convertToMeters"をチェック
    p = contOp.getKeywordPointer("convertToMeters", ps)
    if p >= 0:
        (lineCont, pEnd, _kind) = contOp.get1line(p)
        if lineCont != "":
            changeCont = "convertToMeters 1.0;"
            newCont = cont[:p] + changeCont + cont[pEnd:]
        else:
            newCont = cont
        return newCont

    #"scale"をチェック
    p = contOp.getKeywordPointer("scale", ps)
    if p >= 0:
        (lineCont, pEnd, _kind) = contOp.get1line(p)
        if lineCont != "":
            changeCont = "scale 1.0;"
            newCont = cont[:p] + changeCont + cont[pEnd:]
        else:
            newCont = cont
        return newCont
    return cont

#
#  checkBlockMeshEmpty
def checkBlockMeshEmpty(cont):
    """ blockMeshDictの「type empty;」を「type wall;」に修正する。
    OF-10からemptyでは、エラーが発生する様になった為。"""
    contOp = pyTreeFoam.strOp(cont)
    #FoamFileをskip
    p = contOp.skipFoamFile(0)
    ps = p
    #"boundary"を検索
    (lineCont, p, kind) = contOp.get1line(p)
    (keyword, pp) = contOp.getKeyword(p)
    while keyword != "" and keyword != "boundary":
        ps = p
        (lineCont, p, kind) = contOp.get1line(p)
        (keyword, pp) = pyTreeFoam.strOp(lineCont).getKeyword(0)
    if keyword == "boundary":
        newCont = cont[ps:]
        newContOp = pyTreeFoam.strOp(newCont)
        (keyword, p) = newContOp.getKeyword(0)
        while keyword != "":
            if keyword == "type":
                (nextWord, pp) = newContOp.getKeyword(p)
                if nextWord == "empty":
                    #置き換え
                    newCont = newCont[:p] + newCont[p:].replace("empty", "patch")
                    p = pp
            (keyword, p) = newContOp.getKeyword(p)
        cont = cont[:ps] + newCont
    return cont

#
#  writeBlockMeshDict
def writeBlockMeshDict(caseDir, blockCont):
    """ blockMeshを書き込む"""
    fileName = caseDir + "/system/blockMeshDict"
    if len(glob.glob(fileName)) == 0:
        fileName = caseDir + "/constant/polyMesh/blockMeshDict"
    f=open(fileName, "w"); f.write(blockCont); f.close()
    return

#
#  writeSnappyHexMeshDict
def writeSnappyHexMeshDict(caseDir, snappyCont):
    """ snappyHexMeshを書き込む"""
    fileName = caseDir + "/system/snappyHexMeshDict"
    f=open(fileName, "w"); f.write(snappyCont); f.close()
    return

#
#  getKeywordPointer
def getKeywordPointer(word, contents, ps):
    """ contents内をpsからwordを検索する。
    コメント文があっても検索できる。検索wordが見つからない場合は、「-1」を戻す"""
    contentsOp = pyTreeFoam.strOp(contents)
    p = ps
    keyword = " "
    while keyword != "" and p < len(contents) and keyword != word:
        [keyword, p] = contentsOp.getKeyword(p)
    if keyword == "":
        return -1
    return p - len(word)

#
#  setCastellatedSnapFlag
def setCastellatedSnapFlag(snappyCont):
    """ snappyHexMeshのflagをセット"""
    #"castellatedMesh true;"を設定
    snappyOp = pyTreeFoam.strOp(snappyCont)
    ps = snappyOp.skipFoamFile(0)
    newLine = "castellatedMesh true;"
    (snappyCont, _pp, _stat) = snappyOp.replace1lineKeyword(ps, newLine)
    #"snap true;"を設定
    newLine = "snap true;"
    (snappyCont, _pp, _stat) = snappyOp.replace1lineKeyword(ps, newLine)    #"addLayers false;"
    #"addLayers false;"を設定
    newLine = "addLayers false;"
    (snappyCont, _pp, _stat) = snappyOp.replace1lineKeyword(ps, newLine)
    return snappyCont

#
#  setGeometryData
def setGeometryData(snappyCont, stl, sect, OFversion):
    """ snappyHexMeshDictのgeometry部を書き換え"""
    ver = pyTreeFoam.getNumVersion(OFversion)
    if OFversion[0] == "v" or OFversion[:3] == "ext":
        newCont = setGeometryData_11(snappyCont, stl, sect)
    elif ver >= 12.0:
        #OF-12以降
        newCont = setGeometryData_12(snappyCont, stl, sect)
    else:
        #OF-11以前
        newCont = setGeometryData_11(snappyCont, stl, sect)
    return newCont

#
def setGeometryData_11(snappyCont, stl, sect):
    """ OF-11以下、OF-v、OF-extの場合
    snappyHexMeshDictのgeometry部を書き換え"""
    snappyOp = pyTreeFoam.strOp(snappyCont)
    pGeom = getKeywordPointer("geometry", snappyCont, 0)
    if pGeom < 0:
        return snappyCont
    (dummy, pEnd) = snappyOp.getKeywordContents("geometry", pGeom)
    cont = "geometry\n" + "{\n"
    i = 0
    for name in stl:
        if sect[i] in TYPE_NAMES:
            cont += 4*" " + name + ".stl\n"
            cont += 4*" " + "{\n"
            cont += 8*" " + "type triSurfaceMesh;\n"
            cont += 8*" " + "name " + name + ";\n"
            cont += 4*" " + "}\n\n"
        i += 1
    cont += "}"
    newCont = snappyCont[:pGeom] + cont + snappyCont[pEnd:]
    return newCont

#
def setGeometryData_12(snappyCont, stl, sect):
    """ OF-12以降の場合
    snappyHexMeshDictのgeometry部を書き換え"""
    snappyOp = pyTreeFoam.strOp(snappyCont)
    pGeom = getKeywordPointer("geometry", snappyCont, 0)
    if pGeom < 0:
        return snappyCont
    (dummy, pEnd) = snappyOp.getKeywordContents("geometry", pGeom)
    cont = "geometry\n" + "{\n"
    i = 0
    for name in stl:
        if sect[i] in TYPE_NAMES:
            cont += 4*" " + name + "\n"
            cont += 4*" " + "{\n"
            cont += 8*" " + "type triSurfaceMesh;\n"
            cont += 8*" " + 'file "' + name + '.stl";\n'
            cont += 4*" " + "}\n\n"
        i += 1
    cont += "}"
    newCont = snappyCont[:pGeom] + cont + snappyCont[pEnd:]
    return newCont

#
#  setFeatureEdgeData
def setFeatureEdgeData(snappyCont, stl, sect, featureEdge):
    """ featureEdgeを書き換え"""
    snappyOp = pyTreeFoam.strOp(snappyCont)
    pFt = getKeywordPointer("features", snappyCont, 0)
    if pFt < 0:
        return snappyCont

    (dummy, pEnd) = snappyOp.getSmallPair(pFt)
    cont = "features\n" + 4*" " + "(\n"
    i = 0
    for name in stl:
        if featureEdge[i] != "":
            cont += 8*" " + "{\n"
            cont += 12*" " + 'file "' + name + '.eMesh";\n'
            cont += 12*" " + "level " + featureEdge[i] + ";\n"
            cont += 8*" " + "}\n\n"
        i += 1
    cont += 4*" " + ")"
    newCont = snappyCont[:pFt] + cont + snappyCont[pEnd:]
    return newCont

#
#  setRefinementSurfaces
def setRefinementSurfaces(snappyCont, stl, sect, minLevel, maxLevel):
    """ MinMaxレベル、cellZoneの設定"""
    snappyOp = pyTreeFoam.strOp(snappyCont)
    pRs = getKeywordPointer("refinementSurfaces", snappyCont, 0)
    if pRs < 0:
        return snappyCont
    (dummy, pEnd) = snappyOp.getMiddlePair(pRs)
    cont = "refinementSurfaces\n" + 4*" " + "{\n"
    i = 0
    for name in stl:
        #faceLevel = maxLevel[i] + " " + maxLevel[i]
        faceLevel = minLevel[i] + " " + maxLevel[i]
        if sect[i] == "wall":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "regions\n"
            cont += 12*" " + "{\n"
            cont += 12*" " + "}\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "patch":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "regions\n"
            cont += 12*" " + "{\n"
            cont += 12*" " + "}\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 12*" " + "patchInfo\n"
            cont += 12*" " + "{\n"
            cont += 16*" " + "type patch;\n"
            cont += 16*" " + "inGroups (patches);\n"
            cont += 12*" " + "}\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "overset":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "regions\n"
            cont += 12*" " + "{\n"
            cont += 12*" " + "}\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 12*" " + "patchInfo\n"
            cont += 12*" " + "{\n"
            cont += 16*" " + "type overset;\n"
            cont += 16*" " + "inGroups (overset);\n"
            cont += 12*" " + "}\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "cellZone":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "cellZoneInside inside;\n"
            cont += 12*" " + "cellZone " + name + ";\n"
            cont += 12*" " + "faceZone " + name + ";\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "faceZone":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "faceZone " + name + ";\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "empty":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "regions\n"
            cont += 12*" " + "{\n"
            cont += 12*" " + "}\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 12*" " + "patchInfo\n"
            cont += 12*" " + "{\n"
            cont += 16*" " + "type empty;\n"
            cont += 16*" " + "inGroups (empty);\n"
            cont += 12*" " + "}\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "symmetry":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "regions\n"
            cont += 12*" " + "{\n"
            cont += 12*" " + "}\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 12*" " + "patchInfo\n"
            cont += 12*" " + "{\n"
            cont += 16*" " + "type symmetry;\n"
            cont += 16*" " + "inGroups (symmetry);\n"
            cont += 12*" " + "}\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "symmetryPlane":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "regions\n"
            cont += 12*" " + "{\n"
            cont += 12*" " + "}\n"
            #cont += 12*" " + "level (" + minLevel[i] + " " + maxLevel[i] + ");\n"
            cont += 12*" " + "level (" + faceLevel + ");\n"
            cont += 12*" " + "patchInfo\n"
            cont += 12*" " + "{\n"
            cont += 16*" " + "type symmetryPlane;\n"
            cont += 16*" " + "inGroups (symmetryPlane);\n"
            cont += 12*" " + "}\n"
            cont += 8*" " + "}\n\n"
        i += 1
    cont += 4*" " + "}"
    newCont = snappyCont[:pRs] + cont + snappyCont[pEnd:]
    return newCont

#
#  setRefinementRegions
def setRefinementRegions(snappyCont, stl, sect, minLevel, baseCellSize):
    """ regionのレベルの設定"""
    snappyOp = pyTreeFoam.strOp(snappyCont)
    pRs = getKeywordPointer("refinementRegions", snappyCont, 0)
    if pRs < 0:
        return snappyCont
    
    (dummy, pEnd) = snappyOp.getMiddlePair(pRs)
    #"refinementRegions"の内容を入れ替え
    cont = "refinementRegions\n" + 4*" " + "{\n"
    i = 0
    for name in stl:
        if sect[i] == "cellZone" or sect[i] == "reg" or sect[i] == "region":
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "levels (( 1E5 " + minLevel[i] + " ));\n"
            cont += 12*" " + "mode inside;\n"
            cont += 8*" " + "}\n\n"
        elif sect[i] == "face":
            distance = baseCellSize[0]
            disStr = str(distance)
            cont += 8*" " + name + "\n"
            cont += 8*" " + "{\n"
            cont += 12*" " + "levels (( "+ disStr + " "  + minLevel[i] + " ));\n"
            cont += 12*" " + "mode distance;\n"
            cont += 8*" " + "}\n\n"
        i += 1
    cont += 4*" " + "}"
    newCont = snappyCont[:pRs] + cont + snappyCont[pEnd:]
    return newCont

#
#  checkFaceZone
def checkFaceZone(snappyCont, sect):
    """ faceZoneを作成する場合はallowFaceZZoneをtrueに設定"""
    snappyOp = pyTreeFoam.strOp(snappyCont)
    faceZone = False
    for item in sect:
        if item == "faceZone":
            faceZone = True
            break
    if faceZone == True:
        pRs = getKeywordPointer("allowFreeStandingZoneFaces", snappyCont, 0)
        if pRs < 0:
            newCont = snappyCont
        else:
            [_lineCont, pEnd, _kind] = snappyOp.get1line(pRs)
            cont = "allowFreeStandingZoneFaces true;"
            newCont = snappyCont[:pRs] + cont + snappyCont[pEnd:]
    else:
        newCont = snappyCont
    return newCont

#
#  setLocationInMesh
def setLocationInMesh(snappyCont, meshLoc):
    """ mesh位置（materialPoint）の座標をセット"""
    configDict = pyTreeFoam.readConfigTreeFoam()
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer >= 10.0:
        #OF-10 以降
        locationMesh = "insidePoint"
    else:
        #OF-9以下、ext,vを含む
        locationMesh = "locationInMesh"
    snappyOp = pyTreeFoam.strOp(snappyCont)
    pRs = getKeywordPointer(locationMesh, snappyCont, 0)
    if pRs < 0:
        return snappyCont

    (_lineCont, pEnd, _kind) = snappyOp.get1line(pRs)
    cont = locationMesh + " (" + " ".join(meshLoc) + ");"
    newCont = snappyCont[:pRs] + cont + snappyCont[pEnd:]
    return newCont

#
#  createCsvFile
#  -------------
def createCsvFile(stlDir, fileName):
    """ defaultのcsvファイルを作成"""
    data = setCsvData()
    data = addStlFileName(stlDir, data)
    data = addBlockCellSize(stlDir, data)
    data = addStlBlockSize(stlDir, data)
    writeCsvData(stlDir, fileName, data)
    return

#
#  checkStlEMeshFiles
#  ------------------
def checkStlEMeshFiles(caseDir, stlDir, fileName):
    """ file有無チェック"""
    stl = []; sect = []; featureEdge = []; minLevel = []; maxLevel = []
    stlMinMax = [stl, sect, featureEdge, minLevel, maxLevel]
    nonFiles = []
    (otherData, data) = readCsvData(fileName)
    if len(otherData[0]) == 0 or len(otherData[1]) == 0 or len(otherData[2]) == 0:
        error = 1
        return (error, nonFiles)
    (error, baseCellSize) = getBaseCellSize(otherData[0])
    if error == 0:
        (error, stlMinMax) = getValueCsvData(baseCellSize, data)
    if error == 0:
        (error, nonFiles) = checkStlFileName(caseDir, stlDir, stlMinMax)
    return (error, nonFiles)

#
#  getNeededEMeshFiles
#  -------------------
def getNeededEMeshFiles(caseDir, stlDir, fileName):
    """ featureEdgeを作成するstlファイル名を取得して返す"""
    #stl=[]; featureEdge=[]
    (otherData, data) = readCsvData(fileName)
    (error, baseCellSize) = getBaseCellSize(otherData[0])
    if error == 0:
        (error, stlMinMax) = getValueCsvData(baseCellSize, data)
    if error == 0:
        eMeshFiles = checkNeededEMeshFileName(stlMinMax)
    return eMeshFiles

#
#  createMeshDict
#  --------------
def createMeshDict(caseDir, stlDir, fileName):
    """ csvファイルからblockMesh, snappyHexMeshDictを作成する"""        
    (otherData, data) = readCsvData(fileName)
    (blockCellSize, nAddCells, meshLoc) = otherData
    if len(blockCellSize) == 0 or len(nAddCells) == 0 or len(meshLoc) == 0:
        error = 1
        return error
    (error, baseCellSize) = getBaseCellSize(blockCellSize)
    if error == 0:
        (error, stlMinMax) = getValueCsvData(baseCellSize, data)
    if error == 0:
        (error, nonFiles) = checkStlFileName(caseDir, stlDir, stlMinMax)
        if error != 0:
            print(_("以下のファイルが存在しません"))
            print(nonFiles)
    if error == 0:
        #patchTypeをcheck
        [stl, sect, featureEdge, minLevel, maxLevel] = stlMinMax
        errNames = set(sect) - set(TYPE_NAMES)
        if len(errNames) > 0:
            print()
            print(_("以下のtypeは、使用できません"))
            print(list(errNames))
            error = 1
    if error == 0:
        #OFversionを取得
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        #blockMeshDictを作成
        #[stl, sect, featureEdge, minLevel, maxLevel] = stlMinMax
        modelSize = getModelSize(stlDir, stl, sect)
        blockSize = getBlockSize(modelSize, baseCellSize, nAddCells)
        blockCont = readBlockMeshDict(caseDir)
        if blockCont == "":
            error = 1
            print(_("blockMeshDictが存在しません。"))
            return error
        blockCont = setBlockPointData(blockCont, blockSize, baseCellSize)
        #blockMeshのscale（倍率）をチェック
        blockCont = checkBlockMeshMag(blockCont)
        #blockMeshの「type empty」をチェック
        blockCont = checkBlockMeshEmpty(blockCont)
        #blockMeshDictを書き込む
        writeBlockMeshDict(caseDir, blockCont)
        #snappyHexMeshDictを作成
        snappyCont = readSnappyHexMeshDict(caseDir)
        if snappyCont == "":
            error = 1
            print(_("snappyHexMeshDictが存在しません。"))
            return error
        snappyCont = setCastellatedSnapFlag(snappyCont)
        snappyCont = setGeometryData(snappyCont, stl, sect, OFversion)
        snappyCont = setFeatureEdgeData(snappyCont, stl, sect, featureEdge)
        snappyCont = setRefinementSurfaces(snappyCont, stl, sect, minLevel,
                        maxLevel)
        snappyCont = setRefinementRegions(snappyCont, stl, sect, minLevel, baseCellSize)
        snappyCont = checkFaceZone(snappyCont, sect)
        snappyCont = setLocationInMesh(snappyCont, meshLoc)
        writeSnappyHexMeshDict(caseDir, snappyCont)
    return error


if __name__ == "__main__":
    import gettext
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    gettext.install("treefoam", locale)
    #_ = gettext.gettext

