#!/usr/bin/python3
# coding: utf-8
#
#   createBafflesDialog.py
#
#       内部patch作成（OF-2.0以降）
#
#   13/01/01    新規作成
#      04/19    nullPatch作成ボタンを追加
#      12/21    stdout、stderrの設定（import logFileCreater）を追加
#   14/10/23    initialize:nullPatch取得時にmeshが無いとｴﾗｰが発生するので修正
#      12/17    runCommand:caseDir直下のfileをsubCaseにコピーする様に修正
#   15/07/17    dialog起動方法を修正
#   19/12/31    GTK+3, python3用に大幅修正
#               OF-2.2.0以降専用（createBafflesDialog220の後継）
#   20/04/21    多言語化対応
#   22/01/09    createPatch,getPatchContFromPatchType:patchをwallに変更。
#   24/02/04    createBafflesDict:OF-11用に修正。OF-11は、mappedの内容が変わっている。
#      08/01    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#   25/07/05    runBafflesDict_run:baffle作成後、解析caseをsubCaseに設定するを追加。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os, sys
import glob
import shutil
import signal

import pyTreeFoam
import GtkParts
import universalDialogs as unvDlg

import logFileCreater
logFileCreater.log()

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

treefoamPid = ""

#
#  Show
#-------
def Show(caseDir, trPid):
    global treefoamPid
    treefoamPid = trPid
    winApp = windowApp(caseDir)
    winApp.main()


#-------------------
#  windowApp class
#-------------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path + "createBafflesDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.caseDir = caseDir
        self.selRegion = ""
        self.currTime = ""
        self.currMeshDir = ""
        self.subCaseDir = ""
        self.maskEvent = True
        #GUIのobject名を取得
        self.setGtkObject()
        #treeListの初期化
        self.iniTreeList()

    def setGtkObject(self):
        """ glade内のobject名を取得"""
        self.combo_time = self.builder.get_object("combo_time")
        self.combo_region = self.builder.get_object("combo_region")
        self.tree_faceZone = self.builder.get_object("tree_faceZone")
        self.tree_baffleList = self.builder.get_object("tree_baffleList")
        self.entry_master = self.builder.get_object("entry_master")
        self.entry_slave = self.builder.get_object("entry_slave")

    def iniTreeList(self):
        """ treeListの初期化"""
        #SINGLESELECTIONの設定
        GtkParts.treeList(self.tree_faceZone).create(multi=False)
        #MULTISELECTIONの設定
        GtkParts.treeList(self.tree_baffleList).create()


    #
    #  main
    #--------
    def main(self):
        """ GUIの表示"""
        #subCaseを作成
        stat = self.initialize()
        if stat != "OK":
            return
        self.maskEvent = False
        self.mainWindow.show()
        #Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        #Gtk.main_quit()
        self.mainWindow.destroy()


    #-------- event handler ----------------
    #--- combo, treeList ---
    def onClose(self, event):
        self.close()
    def onChangeTime(self, event):
        if self.maskEvent == False:
            self.changeTime()
    def onChangeRegion(self, event):
        if self.maskEvent == False:
            self.changeRegion()
    def onChangeFaceZone(self, event):
        self.changeFaceZone()
    #--- button ----
    def onRunTopoSetEditor(self, event):
        self.runTopoSetEditor()
    def onRunGridEditor(self, event):
        self.runGridEditor()
    def onCreatePatch(self, event):
        self.createPatch()
    def onCreateCyclicPatch(self, event):
        self.createCyclicPatch()
    def onCreateMappedPatch(self, event):
        self.createMappedPatch()
    def onCreateBafflesDict(self, event):
        self.createBafflesDict()
    def onEditBafflesDict(self, event):
        self.editBafflesDict()
    def onRunBafflesDict(self, event):
        self.runBafflesDict()
    def onDeleteNameList(self, event):
        self.deleteNameList()
    #----------------------------------------

    #
    #  changeTime
    #-------------
    def changeTime(self):
        """ timeを変更した時の処理"""
        #entry = self.combo_time.get_child()
        #selTime = entry.get_text()
        selTime = GtkParts.comboBoxText(self.combo_time).getSelectedData()
        time = selTime.split(":")[-1]
        self.currTime = time
        case = pyTreeFoam.case(self.caseDir)
        self.currMeshDir = case.getCurrMeshDir(self.currTime, self.selRegion, "boundary")
        #faceZoneの取得
        faceZoneNames = pyTreeFoam.getFoamContents().faceZoneNames(self.currMeshDir)
        self.setDataToTreeList(self.tree_faceZone, faceZoneNames)
        
    #
    #  changeRegion
    #---------------
    def changeRegion(self):
        """ regionを変更した時の処理"""
        entry = self.combo_region.get_child()
        selRegion = entry.get_text()
        if selRegion == "(region0)":
            selRegion = "."
        self.selRegion = selRegion
        case = pyTreeFoam.case(self.caseDir)
        self.currMeshDir = case.getCurrMeshDir(self.currTime, self.selRegion, "boundary")
        #faceZoneの取得
        faceZoneNames = pyTreeFoam.getFoamContents().faceZoneNames(self.currMeshDir)
        self.setDataToTreeList(self.tree_faceZone, faceZoneNames)

    #
    #  changeFaceZone
    #-----------------
    def changeFaceZone(self):
        """ faceZoneを選択した時の処理"""
        selNames = self.getSelectDataInTreeList(self.tree_faceZone)
        if len(selNames) != 0:
            faceZone = selNames[0]
            self.entry_master.set_text(faceZone + "_master")
            self.entry_slave.set_text(faceZone + "_slave")

    #
    #  runTopoSetEditor
    #-------------------
    def runTopoSetEditor(self):
        """ topoSetEditorを起動する"""
        modelDict = pyTreeFoam.case(self.caseDir).getCreateMeshData()
        stlDir = modelDict["stlDir"]
        path = os.getenv("TreeFoamPath") + "/python"
        comm = path + "/topoSetEditorDialog.py " + self.caseDir + " --stl " + stlDir
        comm += " --time " + self.currTime + " --region " + self.selRegion + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  runGridEditor
    #----------------
    def runGridEditor(self):
        """ gridEditorを起動する"""
        nTreeFoam = "-1"
        path = os.getenv("TreeFoamPath") + "/python"
        # comm = path + "/editBoundaryConditionsDialog.py "
        # comm += nTreeFoam + " " + self.caseDir + " " + self.currTime + " " + self.selRegion + " &"
        comm = path + "/gridEditorQtDialog.py "
        comm += nTreeFoam + " " + self.caseDir + " " + self.currTime + " " + self.selRegion + " &"
        pyTreeFoam.run(self.caseDir).command(comm)

    #
    #  createPatch
    #--------------
    def createPatch(self):
        """ 標準のpatch(wall)を作成する（nameListを作成）"""
        self.setPatchNamesList("wall")

    #
    #  createCyclicPath
    #-------------------
    def createCyclicPatch(self):
        """ cyclicPatchを作成する（nameListを作成）"""
        self.setPatchNamesList("cyclic")

    #
    #  createMappedPatch
    #--------------------
    def createMappedPatch(self):
        """ mappedPatchを作成する(nameListを作成）"""
        self.setPatchNamesList("mapped")

    #
    #  setPatchNamesList
    def setPatchNamesList(self, patchType):
        """ 名前リスト作成"""
        faceZones = self.getSelectDataInTreeList(self.tree_faceZone)
        master = self.entry_master.get_text()
        slave = self.entry_slave.get_text()
        if len(faceZones) == 0 or master == "" or slave == "":
            title = _(u"エラー")
            mess = _(u"名前が選択されていません。")
            self.errDialog(title, mess)
            return
        faceZone = faceZones[0]
        patchConts = pyTreeFoam.case(self.caseDir).getPatchNameFromBoundary(self.currMeshDir)
        patchNames = list(map(lambda x: x[0], patchConts))
        if master in patchNames or slave in patchNames:
            title = _(u"警告")
            mess = _(u"master or slaveのpatch名は、既に使用されています。\n")
            mess += _(u"このまま設定しますか？")
            yesArgs = [self.setPatchNamesList_run, faceZone, patchType, master, slave]
            self.yesNoDialog(title, mess, funcYes=yesArgs)
            return
        self.setPatchNamesList_run(faceZone, patchType, master, slave)

    #  setPatchNamesList_run           
    def setPatchNamesList_run(self, faceZone, patchType, master, slave):
        """ 名前Listを作成する"""
        nameList = patchType + " " + " ".join([faceZone, master, slave])
        self.appendDataToListBox(self.tree_baffleList, nameList)

    #
    #  appendDataToListBox
    def appendDataToListBox(self, treeList, item):
        """ treeListに項目を追加する"""
        names = GtkParts.treeList(treeList).getAllItems()
        names.append(item)
        names = list(set(names))
        names.sort()
        GtkParts.treeList(treeList).setData(names)
        GtkParts.treeList(treeList).selectItems([item])

    #
    #  createBafflesDict
    #--------------------
    def createBafflesDict(self):
        """ createBafflesDictをnameListから作成する"""
        #OFversionを取得
        OFversion = pyTreeFoam.readConfigTreeFoam()["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        nameLists = GtkParts.treeList(self.tree_baffleList).getAllItems()
        cont  = "// Whether to convert internal faces only (so leave boundary faces intact).\n"
        cont += "// This is only relevant if your face selection type can pick up boundary\n"
        cont += "// faces.\n"
        cont += "internalFacesOnly true;\n"
        cont += "\n"
        cont += "// Baffles to create.\n"
        cont += "baffles\n"
        cont += "{\n"
        sp = " "
        for i in range(len(nameLists)):
            nameList = nameLists[i]
            (patchType, faceZone, master, slave) = nameList.split()
            if numVer >= 11.0:
                #OF-11以上の場合
                (patchMaster, patchSlave) = self.getPatchContFromPatchType_OF11(patchType, master, slave)
            else:
                #OF-11未満の場合（通常）
                (patchMaster, patchSlave) = self.getPatchContFromPatchType(patchType, master, slave)
            cont += "    " + faceZone + sp*5 + "//baffles is created\n"
            cont += "    {\n"
            cont += "        //- Use predefined faceZone to select faces and orientation.\n"
            cont += "        type        faceZone;\n"
            cont += "        zoneName    " + faceZone + ";\n"
            cont += "        patches\n"
            cont += "        {\n"
            cont += "            master\n"
            cont += "            {\n"
            cont += "                //- Master side patch\n"
            cont += "                name        " + master + ";\n"
            cont += patchMaster
            cont += "            }\n"
            cont += "            slave\n"
            cont += "            {\n"
            cont += "                //- Slave side patch\n"
            cont += "                name        " + slave + ";\n"
            cont += patchSlave
            cont += "            }\n"
            cont += "        }\n"
            cont += "    }\n"
        cont += "}\n"
        configDict = pyTreeFoam.readConfigTreeFoam()
        OFversion = configDict["OFversion"]
        loc = "system/" + self.selRegion
        getFoamContents = pyTreeFoam.getFoamContents(self.subCaseDir)
        header = getFoamContents.makeFoamHeaderVersion(OFversion)
        foamFile = getFoamContents.makeFoamFile("2.0", "ascii", "dictionary", loc, "createBafflesDict")
        footer = getFoamContents.makeFoamFooter()
        contents = header + foamFile + cont + footer
        fileName = self.subCaseDir + "/" + loc + "/createBafflesDict"
        f = open(fileName, "w"); f.write(contents); f.close()
        title = _(u"createBafflesDict作成")
        mess = _(u"createBafflesDictを作成しました。")
        print(mess)
        self.okDialog(title, mess)

    #
    #
    def getPatchContFromPatchType_OF11(self, patchType, master, slave):
        """ OF-11以上用のpatch内容を取得"""
        contMaster = ""
        contSlave = ""
        if patchType == "wall":
            contMaster += "                type        wall;\n"
            contSlave = contMaster
        elif patchType == "cyclic":
            contMaster += "                type        cyclic;\n"
            contMaster += "                neighbourPatch " + slave + ";\n"
            contSlave  += "                type        cyclic;\n"
            contSlave  += "                neighbourPatch " + master + ";\n"
        elif patchType == "mapped":
            region = self.selRegion
            if region == ".":
                region = "region0"
            contMaster += "                type        patch;\n"
            contSlave  += "                type        mapped;\n"
            contSlave  += "                neighbourRegion " + region + ";\n" 
            contSlave  += "                neighbourPatch " + master + ";\n"
        else:
            contMaster += "                type        patch;\n"
            contSlave = contMaster
        return (contMaster, contSlave)


    #
    #  getPatchContFromPatchType
    def getPatchContFromPatchType(self, patchType, master, slave):
        """ patchTypeからそのpatch内容を取得する。"""
        contMaster = ""
        contSlave = ""
        if patchType == "wall":
            contMaster += "                type        wall;\n"
            contSlave = contMaster
        elif patchType == "cyclic":
            contMaster += "                type        cyclic;\n"
            contMaster += "                neighbourPatch " + slave + ";\n"
            contSlave  += "                type        cyclic;\n"
            contSlave  += "                neighbourPatch " + master + ";\n"
        elif patchType == "mapped":
            region = self.selRegion
            if region == ".":
                region = "region0"
            contMaster += "                type        patch;\n"
            contSlave  += "                type        mappedPatch;\n"
            contSlave  += "                sampleRegion " + region + ";\n" 
            contSlave  += "                sampleMode  nearestPatchFace;\n"
            contSlave  += "                samplePatch " + master + ";\n"
        else:
            contMaster += "                type        patch;\n"
            contSlave = contMaster
        return (contMaster, contSlave)

    #
    #  editBafflesDict
    #------------------
    def editBafflesDict(self):
        """ createBafflesDictを編集する。"""
        fileName = self.subCaseDir + "/system/" + self.selRegion + "/createBafflesDict"
        pyTreeFoam.run(self.subCaseDir).editor([fileName])

    #
    #  runBafflesDict
    #-----------------
    def runBafflesDict(self):
        """ createBafflesを実行。"""
        #createBaffles実行
        configDict = pyTreeFoam.readConfigTreeFoam()
        bashrcFOAM = configDict["bashrcFOAM"]
        comm  = ". " + bashrcFOAM + "; "
        comm += "createBaffles -region " + self.selRegion + " -overwrite"
        print(comm)
        (stat, _res, _err) = pyTreeFoam.run(self.subCaseDir).commandReturnCont(comm)
        if stat != "OK":
            title = _(u"エラー")
            mess = _(u"実行エラーが発生しました。\nlogを確認してください。")
            self.errDialog(title, mess)
            return
        #終了処理
        case = pyTreeFoam.case(self.subCaseDir)
        timeFolders = case.getTimeFolders()
        if len(timeFolders) > 0:
            #boundaryの整合性をとる。
            timeFolder = timeFolders[0]
            maxTime = timeFolders[-1]
            relMeshDir = case.getCurrMeshDir(maxTime, self.selRegion, "boundary")
            meshDir = self.subCaseDir + "/" + relMeshDir
            meshDir = "/".join(meshDir.split("/")[:-1])
            case.clearNullBoundaryAllFields(timeFolder, meshDir, self.selRegion)
        self.entry_master.set_text("")
        self.entry_slave.set_text("")
        self.setDataToTreeList(self.tree_baffleList, [])
        if os.path.basename(self.caseDir) == "subCase":
            #終了メッセージ
            title = _(u"内部patch作成")
            mess = _(u"内部patchを作成しました。")
            funcOk = []
        else:
            #終了メッセージ
            title = _(u"内部patch作成")
            mess = _(u"内部patchを作成しました。\n解析caseを「subCase」に設定します。")
            funcOk = [self.runBafflesDict_run]
        self.okDialog(title, mess, funcOk=funcOk)

    #  runBafflesDict_run
    def runBafflesDict_run(self):
        fileName = os.getenv("TreeFoamUserPath") + os.sep + "temp" + os.sep + "signalData"
        newCaseDir = self.subCaseDir
        lines = ["changeCursor\n"]
        lines += ["newDir " + newCaseDir + "\n"]
        f = open(fileName, "w"); f.writelines(lines); f.close()
        #treefoam.pyにsignal送信
        pid = int(treefoamPid)
        os.kill(pid, signal.SIGUSR1)
        print("send signal to TreeFoam.")


    #
    #  deleteNameList
    #------------------
    def deleteNameList(self):
        """ nameListを1行削除する。"""
        nameLists = GtkParts.treeList(self.tree_baffleList).getAllItems()
        selNames = self.getSelectDataInTreeList(self.tree_baffleList)
        if len(selNames) == 0:
            title = _("エラー")
            mess = _("nameListが選択されていません。") + "\n"
            mess += _("  削除するnameListを選択してください。")
            self.errDialog(title, mess)
            return
        nameLists = list(filter(lambda x: not(x in selNames), nameLists))
        self.setDataToTreeList(self.tree_baffleList, nameLists)


    #
    #  initialize
    #-------------
    def initialize(self):
        """ subCaseを作成する"""
        stat = self.makeSubCase()
        if stat == "OK":
            #timeを取得
            case = pyTreeFoam.case(self.caseDir)
            timeFolders = case.getTimeFolders()
            firstTime = timeFolders[0]
            latestTime = timeFolders[-1]
            contDict = case.getControlDict()
            startTime = contDict["startTime"]
            times = []
            times.append("firstTime:" + firstTime)
            times.append("startTime:" + startTime)
            times.append("latestTime:" + latestTime)
            if contDict["startFrom"] == "firstTime":
                selNo = 0
                self.currTime = firstTime
            elif contDict["startFrom"] == "startTime":
                selNo = 1
                self.currTime = startTime
            else:
                selNo = 2
                self.currTime = latestTime
            self.setDataToComboBox(self.combo_time, times)
            self.combo_time.set_active(selNo)
            #region, polyMeshDirの取得
            self.selRegion = "."
            (isRegion, _loc, regions) = case.isMultiRegion()
            if isRegion == True:
                self.setRegionNames(regions)
            else:
                self.setRegionNames([])
            self.combo_region.set_active(0)
            self.currMeshDir = case.getCurrMeshDir(self.currTime, self.selRegion, "boundary")
            #faceZoneの取得
            faceZoneNames = pyTreeFoam.getFoamContents().faceZoneNames(self.currMeshDir)
            self.setDataToTreeList(self.tree_faceZone, faceZoneNames)
        return stat

    #
    #  makeSubCase
    def makeSubCase(self):
        """ subCaseの作成"""
        caseName = os.path.basename(self.caseDir)
        if caseName == "subCase":
            self.subCaseDir = self.caseDir
            return "OK"
        if len(glob.glob(self.caseDir + "/subCase")) != 0:
            title = _(u"エラー")
            msg = _(u"既に「subCase」が存在しています。\n　「subCase」を削除してください。")
            self.errDialog(title, msg)
            return "ERROR"
        else:
            #subCase作成
            os.mkdir(self.caseDir + "/subCase")
            times = pyTreeFoam.case(self.caseDir).getTimeFolders()
            shutil.copytree(self.caseDir + "/" + times[0], self.caseDir + "/subCase/" + times[0])
            shutil.copytree(self.caseDir + "/constant", self.caseDir + "/subCase/constant")
            shutil.copytree(self.caseDir + "/system", self.caseDir + "/subCase/system")
            self.subCaseDir = self.caseDir + "/subCase"
            return "OK"

    #
    #  setRegionNames
    def setRegionNames(self, regions):
        """ regioNamesをcomboにセット"""
        names = []
        names.append(_("(region0)"))
        names += regions
        self.setDataToComboBox(self.combo_region, names)

    #
    #  setDataToComboBox
    def setDataToComboBox(self, comboBox, setNames):
        """ comboBoxにsetNamesを設定する。"""
        GtkParts.comboBoxText(comboBox).setData(setNames)

    #
    #  setDataToTreeList
    def setDataToTreeList(self, treeList, setNames):
        """ treeListにsetNamesを設定する"""
        GtkParts.treeList(treeList).setData(setNames)

    #
    #  getSelectDataInTreeList
    def getSelectDataInTreeList(self, treeList):
        """ treeList内から選択している項目を返す"""
        selItems = GtkParts.treeList(treeList).getSelectedItems()
        return selItems

    #--- universalDialogs -----------------------------
    def okDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def errDialog(self, title, mess, funcOk=[]):
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    def yesNoDialog(self, title, mess, funcYes=[], funcNo=[]):
        dialog = unvDlg.yesNoDialog(
            title, mess, funcYes=funcYes, funcNo=funcNo, parentWin=self.mainWindow)
        dialog.show()


if __name__ == "__main__":
    import gettext
    #import locale
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    #locale.bindtextdomain("treefoam", localeDir)
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    #os.environ["TreeFoamPath"] = "/home/caeuser/TreeFoam-glade3"
    #caseDir = "/home/caeuser/CAE/CAE-FOAM/OF-v1906/multiRegionHeater_copy0"
    caseDir = sys.argv[1]
    winApp = windowApp(caseDir)
    winApp.main()
