#!/usr/bin/python3
# coding: utf-8
#
#   createBCsMultiRegionDialog.py
#
#       multiReginモデルの境界面の境界条件設定
#       OF-2.2以降
#
#   14/10/05    新規作成
#      10/06    replaceRegionFieldBoundaryConditions:includeファイルの挿入方法修正
#      10/15    dict読み込みを修正
#      10/16    変数名のエラーチェックを追加
#      11/08    replaceRegionFieldBoundaryConditions:自由なwildCard設定を追加
#      11/10    saveVariableToFile:Dict内にunicodeが有った場合にｴﾗｰ発生を修正
#      11/11    checkSpecWildCard:wildCardのfield名チェックを追加
#      11/12    checkBoundaryField:check方法を修正（誤った表示をしてしまう。）
#      11/14    setWildCard:バグ修正（行末の「;」が抜ける
#   15/07/17    dialog起動方法修正
#   18/09/23    getAbsPath:#includeのpath指定「<constant>」に対応。
#      09/27    getAbsPath:#includeのpath指定「<system>」に対応。
#   19/03/03    errorCheckBoundaryFieldName:変数名に「internalField」を追加
#      05/22    makeSpecFieldPatchCont:mkdirをos.makedirsに修正
#      10/26    GTK+3, python3用に大幅修正
#   20/04/21    多言語化対応
#      06/03    readDictFile:OF-7.0用のregionWallBoundaryConditionsDict_7を準備
#   21/06/07    readDictFile:OF-8.0用のregionWallBoundaryConditionsDict_8を追加した為、
#               OF-v系の条件がOF-8に変わってしまうので、OF-v系専用の
#               regionWallBoundaryConditionsDict_vを追加
#   22/07/20    readDictFile:OFversionのcheck方法修正（OF-10対応）
#      08/14    readDictFile:OFversionのcheck方法修正（バグ修正）
#   24/02/07    getAbsPath:getAbsPathをpyTreeFoam.caseに移動。
#      08/02    universalDialogs:全dialogに対しfuncOk、親windowを設定
#      08/22    起動時、window位置をmouseに合わせる様修正。
#

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk
import os, sys
import glob
import shutil
import re

import pyTreeFoam
import stringOp
import universalDialogs as unvDlg

import locale
localeDir = os.getenv("TreeFoamPath") + "/data/locale"
locale.bindtextdomain("treefoam", localeDir)

import logFileCreater
logFileCreater.log()

caseDir = ""
variables = ""
beforeFluidField = ""
beforeSolidField = ""
fluidRegions = []
fluidFields = []                #[[<field>, [<variable>, <yesNo>]], ...]
fluidBoundaryCont = []          #[<field>, <boundary>], .. , [..]]
solidRegions = []
solidFields = []
solidBoundaryCont = []
errFields = []                  #存在しないfield
specFluidBoundary = []          #[[<wildCard>,[<field>, <boundary]...]
specSolidBoundary = []

includeDir = ""
includeDirOrg = ""
uniform = ""
noChange = ""

configDict = {}


#------------
#  windowApp
#------------
class windowApp:

    def __init__(self, caseDir):
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("treefoam")
        path = os.getenv("TreeFoamPath") + os.sep + "glade" + os.sep
        self.builder.add_from_file(path+"createBCsForMultiRegionDialog.glade")
        self.mainWindow = self.builder.get_object("window1")
        #window位置をmouseに合わせる
        self.mainWindow.set_position(2)
        self.mainWindow.connect("delete-event", self.close)
        self.builder.connect_signals(self)
        #変数の設定
        self.caseDir = caseDir
        #GUIのobject名を取得
        self.setGtkObject()
        #変数をtextBoxにセット
        self.setVariablesTextBox()
        #fieldをTreeにセット
        self.iniTreeView()
        self.setDataToTreeView()
        #internal, boundaryFieldの内容をセット
        self.showFluidInternalBoundaryField(fluidFields[0][0])
        self.showSolidInternalBoundaryField(solidFields[0][0])


    def setGtkObject(self):
        """ glade内のobject名を取得する"""
        self.textView_variables = self.builder.get_object("textView_variables")
        self.tree_fluid = self.builder.get_object("tree_fluid")
        self.tree_solid = self.builder.get_object("tree_solid")
        self.textView_fluid = self.builder.get_object("textView_fluid")
        self.textView_solid = self.builder.get_object("textView_solid")
        self.combo_fluid = self.builder.get_object("combo_fluid")
        self.combo_solid = self.builder.get_object("combo_solid")

    #
    #  main
    #  ----
    def main(self):
        """ GUIを表示"""
        self.mainWindow.show()
        Gtk.main()

    def close(self, *args):
        """ 閉じる"""
        Gtk.main_quit()

    #------- event handlar ----------
    #fluidField選択
    def onSelectFluidField(self, event):
        self.selectFluidField()
    #solidField選択
    def onSelectSolidField(self, event):
        self.selectSolidField()
    #設定
    def onApply(self, event):
        self.getAllVariablesFromTextBox()
        stat = self.saveVariableToFile()
        if stat != "OK":
            return
        self.replaceBoundaryFields()
        # stat = self.replaceBoundaryFields()
        # if stat != "OK":
        #     return
        # self.close()

    #閉じる
    def onClose(self, event):
        self.close()
    #--------------------------------

    #
    #  setComboDataInComboBox
    def setComboDataInComboBox(self, comboBox, setNames):
        """ comboBoxにsetNamesを設定する。"""
        comboBox.clear()
        listModel = Gtk.ListStore(str)
        for name in setNames:
            listModel.append([name])
        cell = Gtk.CellRendererText()
        comboBox.pack_start(cell, True)
        comboBox.add_attribute(cell, "text", 0)
        comboBox.set_model(listModel)
        comboBox.set_entry_text_column(0)

    #
    #  setFluidComboBox
    def setFluidComboBox(self, field):
        """ fluidComboBoxにデータセット"""
        variables = []
        variables.append(noChange)
        for fieldCont in fluidFields:
            if fieldCont[0] == field:
                variables.append(uniform+fieldCont[1][0])
                break
        self.setComboDataInComboBox(self.combo_fluid, variables)

    #
    #  setSolidComboBox
    def setSolidComboBox(self, field):
        """ solidComboBoxにデータセット"""
        variables = []
        variables.append(noChange)
        for fieldCont in solidFields:
            if fieldCont[0] == field:
                #variables.append(uniform+fieldCont[1][0])
                variables.append(uniform)
                break
        self.setComboDataInComboBox(self.combo_solid, variables)

    #
    #  getSelectValueInComboBox
    def getSelectValueInComboBox(self, comboBox):
        """ 指定されたcomboBoxから選択されている内容を取得する"""
        entry = comboBox.get_child()
        selCont = entry.get_text()
        return selCont

    #
    #  setVariablesTextBox
    def setVariablesTextBox(self):
        """ 変数をtextBoxにセット"""
        buffer = self.textView_variables.get_buffer()
        buffer.set_text(variables)

    #
    #  getSelectFieldInTree
    def getSelectFieldInTree(self, treeList):
        """ treeList内から選択したfieldを返す"""
        listSelection = treeList.get_selection()
        (listStore, pathes) = listSelection.get_selected_rows()
        if len(pathes) != 0:
            iter = listStore.get_iter(pathes[0])
            name = listStore.get_value(iter, 0)
        else:
            name = ""
        return name

    #
    #  iniTreeView
    def iniTreeView(self):
        """ treeViewを設定する"""
        #title行を非表示
        self.tree_fluid.set_headers_visible(False)
        self.tree_solid.set_headers_visible(False)
        #treeColumnの設定
        cellRenderer = Gtk.CellRendererText()
        colFluid = Gtk.TreeViewColumn()
        colFluid.pack_start(cellRenderer, True)
        colFluid.add_attribute(cellRenderer, "text", 0)
        self.tree_fluid.append_column(colFluid)
        colSolid = Gtk.TreeViewColumn()
        colSolid.pack_start(cellRenderer, True)
        colSolid.add_attribute(cellRenderer, "text", 0)
        self.tree_solid.append_column(colSolid)

    #
    #  setDataToTreeView
    def setDataToTreeView(self):
        """ listBoxにデータを設定"""
        global beforeFluidField, beforeSolidField
        #fluid側をセット
        fields = []
        for fieldCont in fluidFields:
            fields.append(fieldCont[0])
        fields.sort()
        listStore = Gtk.ListStore(str)
        for name in fields:
            listStore.append([name])
        self.tree_fluid.set_model(listStore)
        listSelection = self.tree_fluid.get_selection()
        path = Gtk.TreePath(0)
        listSelection.select_path(path)
        beforeFluidField = fields[0]
        #solid側をセット
        fields = []
        for fieldCont in solidFields:
            fields.append(fieldCont[0])
        fields.sort()
        listStoreSolid = Gtk.ListStore(str)
        for name in fields:
            listStoreSolid.append([name])
        self.tree_solid.set_model(listStoreSolid)
        listSelection = self.tree_solid.get_selection()
        path = Gtk.TreePath(0)
        listSelection.select_path(path)
        beforeSolidField = fields[0]

    #
    #  getFluidSettingContFromTextBox
    def getFluidSettingContFromTextBox(self):
        """ textBoxのデータを取得"""
        global fluidFields, solidFields, fluidBoundaryCont
        #internalFieldを取得
        intValue = self.getSelectValueInComboBox(self.combo_fluid)
        if intValue == noChange:
            i = 0
            for fieldCont in fluidFields:
                if fieldCont[0] == beforeFluidField:
                    fluidFields[i][1][1] = "off"
                    break
                i += 1
        elif intValue != "":
            variable = intValue.split()[-1][1:]
            #fieldと変数の関連を書き換え
            i = 0
            for fieldCont in fluidFields:
                if fieldCont[0] == beforeFluidField:
                    fluidFields[i][1][0] = variable
                    fluidFields[i][1][1] = "on"
                    break
                i += 1

        #boundaryFieldを取得
        buffer = self.textView_fluid.get_buffer()
        start = buffer.get_start_iter()
        end = buffer.get_end_iter()
        boundary = buffer.get_text(start, end, True)
        i = 0
        for field in fluidBoundaryCont:
            if field[0] == beforeFluidField:
                fluidBoundaryCont[i][1] = boundary
                break
            i += 1    

    #
    #  getSolidSettingContFromTextBox(self):
    def getSolidSettingContFromTextBox(self):
        """ textBoxのデータを取得"""
        global fluidFields, solidFields, solidBoundaryCont
        #internalFieldを取得
        intValue = self.getSelectValueInComboBox(self.combo_solid)
        if intValue == noChange:
            #switch off設定
            i = 0
            for fieldCont in solidFields:
                if fieldCont[0] == beforeSolidField:
                    solidFields[i][1][1] = "off"
                    break
                i += 1
        elif intValue != "":
            #switch on 設定
            variable = intValue.split()[-1][1:]
            i = 0
            for fieldCont in solidFields:
                if fieldCont[0] == beforeSolidField:
                    solidFields[i][1][0] = variable
                    solidFields[i][1][1] = "on"
                    break
                i += 1

        #boundaryFieldを取得
        buffer = self.textView_solid.get_buffer()
        start = buffer.get_start_iter()
        end = buffer.get_end_iter()
        boundary = buffer.get_text(start, end, True)
        i = 0
        for field in solidBoundaryCont:
            if field[0] == beforeSolidField:
                solidBoundaryCont[i][1] = boundary
                break
            i += 1

    #
    #  showFluidInternalBoundaryField
    def showFluidInternalBoundaryField(self, field):
        """ internalField, boundaryFieldの内容を表示する"""
        #internalFieldの表示
        self.setFluidComboBox(field)
        show = 0
        i = 0
        for setField in fluidFields:
            if setField[0] == field:
                if isString(setField[1][1]) == True:
                    show = 1
                    break
            i += 1
        if show == 0:
            self.combo_fluid.set_active(0)
        else:
            self.combo_fluid.set_active(1)
        #boundaryの表示
        boundary = ""
        for bCont in fluidBoundaryCont:
            if field == bCont[0]:
                boundary = bCont[1]
                break
        buffer = self.textView_fluid.get_buffer()
        buffer.set_text(boundary)

    #
    #  showSolidInternalBoundaryField
    def showSolidInternalBoundaryField(self, field):
        """ internalField, boundaryFieldの内容を表示する"""
        #internalFieldの表示
        self.setSolidComboBox(field)
        show = 0
        i = 0
        for setField in solidFields:
            if setField[0] == field:
                if isString(setField[1][1]) == True:
                    show = 1
                    break
            i += 1
        if show == 0:
            self.combo_solid.set_active(0)
        else:
            self.combo_solid.set_active(1)
        #boundaryの表示
        boundary = ""
        for bCont in solidBoundaryCont:
            if field == bCont[0]:
                boundary = bCont[1]
                break
        buffer = self.textView_solid.get_buffer()
        buffer.set_text(boundary)

    #
    #  getAllVariablesFromTextBox
    def getAllVariablesFromTextBox(self):
        """ dialog内の全てのデータを取得"""
        global beforeFluidField, variables
        #変数の設定を取得
        buffer = self.textView_variables.get_buffer()
        start = buffer.get_start_iter()
        end = buffer.get_end_iter()
        variables = buffer.get_text(start, end, True)
        #fluid側の設定を取得
        field = self.getSelectFieldInTree(self.tree_fluid)
        beforeFluidField = field
        self.getFluidSettingContFromTextBox()
        #solid側の設定を取得
        field = self.getSelectFieldInTree(self.tree_solid)
        beforeSolidField = field
        #全ての設定をtextBoxから取得
        self.getSolidSettingContFromTextBox()

    #
    #  saveVariableToFile
    def saveVariableToFile(self):
        """ 設定した内容（変数、internalField、boundaryField）をDictに保存"""
        (stat, mess) = errorCheckInternalFieldName()
        if stat != "OK":
            self.errDialog("エラー", mess)
            return stat
        (stat, mess) = errorCheckBoundaryFieldName()
        if stat != "OK":
            self.errDialog("エラー", mess)
            return stat
        #設定内容を保存
        fileName = caseDir + "/system/regionWallBoundaryConditionsDict"
        f=open(fileName); dictCont=f.read(); f.close()
        dictCont = saveVariableToFile_variable(dictCont)
        dictCont = saveVariableToFile_internalField(dictCont)
        dictCont = saveVariableToFile_boundary(dictCont)
        f=open(fileName, "w"); f.write(dictCont); f.close()
        return "OK"

    #
    #  replaceBoundaryFields
    def replaceBoundaryFields(self):
        """ 各fieldのinternalField、boundaruFieldを書き換える"""
        #各fieldの界面の境界条件が既に設定されているかどうかをチェック
        (error, mess) = checkAllFields(caseDir)
        if error != "OK":
            title = _("警告")
            okArgs = [self.replaceBoundaryFields_run]
            self.okCancelDialog(title, mess, funcOk=okArgs)
            return
        #fieldの置き換え
        self.replaceBoundaryFields_run()

    #  replaceBoundaryFields_run
    def replaceBoundaryFields_run(self):
        #fieldの置き換え
        (error, errFields) = replaceRegionFieldBoundaryConditions(caseDir, 
                        fluidRegions, solidRegions)
        if error == "OK":
            title = _("界面の境界条件設定")
            mess = _("領域間の境界条件とinternalFieldを再設定しました。\n")
            if len(errFields) > 0:
                mess += _("以下のfieldは、存在しない為、修正していません。\n")
                if len(errFields) > 5:
                    addMess = ", ".join(errFields[:5]) + "..."
                else:
                    addMess = ", ".join(errFields)
                mess += addMess
            print(mess)
            self.okDialog(title, mess)
            #dialogをclose
            self.close()
        else:
            title = _("エラー")
            print(errFields)
            self.errDialog(title, errFields)
        return

    #
    #  selectFluidField
    #  ----------------
    def selectFluidField(self):
        """ fluid側のfieldを選択した時"""
        global beforeFluidField
        field = self.getSelectFieldInTree(self.tree_fluid)
        print(field, _("fluid fieldの状態を表示します。"))
        self.getFluidSettingContFromTextBox()
        self.showFluidInternalBoundaryField(field)
        beforeFluidField = field
        
    #
    #  selectSolidField
    #  ----------------
    def selectSolidField(self):
        """ solid側のfieldを選択した時"""
        global beforeSolidField
        field = self.getSelectFieldInTree(self.tree_solid)
        print(field, _("solid fieldの状態を表示します。"))
        self.getSolidSettingContFromTextBox()
        self.showSolidInternalBoundaryField(field)
        beforeSolidField = field

    #----------- universal dialog -------------
    #  okDialog
    def okDialog(self, title, mess, funcOk=[]):
        """ okDialog表示"""
        dialog = unvDlg.okDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    #  errDialog
    def errDialog(self, title, mess, funcOk=[]):
        """ errDialog表示"""
        dialog = unvDlg.errDialog(
            title, mess, funcOk=funcOk, parentWin=self.mainWindow)
        dialog.show()

    #  okCancelDialog
    def okCancelDialog(self, title, mess, funcOk=[], funcCancel=[]):
        """ okCancel dialog表示"""
        dialog = unvDlg.okCancelDialog(
            title, mess, funcOk=funcOk, funcCancel=funcCancel, 
            parentWin=self.mainWindow)
        dialog.show()


#
#  initialize
#  ----------
def initialize(caseDir):
    """ 初期化"""
    global fluidRegions, solidRegions
    global uniform, noChange, configDict
    configDict = pyTreeFoam.readConfigTreeFoam()
    #各region名を取得
    OFversion = configDict["OFversion"]
    (fluidRegions, solidRegions) = pyTreeFoam.case(caseDir).getZonesInRegionProperties(OFversion)
    #boundaryDict読み込み
    dictCont = readDictFile(caseDir)
    #includeDirを取得
    _error = getIncludeDirFromDict(dictCont)
    #変数(variables)を取得
    _error = getVariableContentsFromDict(dictCont)
    #internalFieldの設定(fluidFields, solidFields)読み込み
    _error = getInternalFieldsFromDict(dictCont)
    #境界条件の読み込み(fluidBoundaryCont, solidBoundaryCont, 
    #                   specFluidBoundary, specSolidBoundary)
    _error = getBoundaryConditionsFromDict(dictCont)
    #field名のチェック修正
    checkOperationField()
    #変数の設定
    uniform = _("uniform $")
    noChange = _("(変更せず)")

#
#  readDictFile
def readDictFile(caseDir):
    """ regionWallBoundaryConditionsDictの読み込み"""
    sName = "regionWallBoundaryConditionsDict"
    fileName = caseDir + "/system/" + sName
    if len(glob.glob(fileName)) == 0:
        #dictが存在しない場合、defaultのdictFileをコピーする
        OFversion = configDict["OFversion"]
        numVer = pyTreeFoam.getNumVersion(OFversion)
        treeFoamPath = os.getenv("TreeFoamPath")
        orgName = treeFoamPath + "/data/multiRegionDB/" + sName
        names = glob.glob(orgName+"*")
        #namesをsortする
        namesList = []
        for name in names:
            strVer = (name.split("/")[-1]).split("_")[-1]
            ver = pyTreeFoam.getNumVersion(strVer)
            if ver < 0:
                ver = 9999
            namesList.append([ver, name])
        namesList.sort()
        names = list(map(lambda x: x[1], namesList))
        #namesの全て検索して、最後のdictを選択する
        dictFileName = ""
        for name in names:
            if name[-1] != "~" and name[0] != ".":
                dictName = name.split("/")[-1]
                #OFversionを確認
                #readName = sName + "_" + OFversion
                strVer = dictName.split("_")[-1]
                dictVer = pyTreeFoam.getNumVersion(strVer)
                if dictVer < 0:
                    dictVer = 9999
                #if dictName <= readName:
                if dictVer <= numVer:
                    #数値比較でOFversionが低ければ取得
                    dictFileName = dictName
                elif strVer <= OFversion:
                    #文字列比較でOFversionが低ければ取得
                    dictFileName = dictName
        orgName = treeFoamPath + "/data/multiRegionDB/" + dictFileName
        distName = caseDir + "/system/" + sName
        shutil.copy(orgName, distName)
        print(_("警告:regionWallBoundaryConditionsDictファイルが存在しなかったので、systemフォルダにコピーしました。"))
    #dict読み込み
    f=open(fileName); cont=f.read(); f.close()
    return cont

#
#  getIncludeDirFromDict
def getIncludeDirFromDict(dictCont):
    """ includeDirを取得"""
    global includeDir, includeDirOrg
    dictContOp = stringOp.strings(dictCont)
    p = dictContOp.skipFoamFile()
    lineCont = b" "; keyword = b" "
    while lineCont != b"" and keyword != b"includeDir":
        (lineCont, p, kind) = dictContOp.get1line(p)
        lineContOp = stringOp.strings(lineCont)
        (keyword, _pp) = lineContOp.getKeyword(0)
    if lineCont != b"" and kind == b"line":
        includeDir = (lineCont.decode().split()[1]).split(";")[0]
        includeDirOrg = includeDir
        return ("OK", "")
    else:
        return ("NG", _("Dict内にincludeDirがありません。"))

#
#  getVariableContentsFromDict
def getVariableContentsFromDict(dictCont):
    """ 変数を取得"""
    global variables
    dictContOp = stringOp.strings(dictCont)
    p = dictContOp.skipFoamFile()
    lineCont = b" "; keyword = b" "
    while lineCont != b"" and keyword != b"variableSetting":
        (lineCont, p, kind) = dictContOp.get1line(p)
        lineContOp = stringOp.strings(lineCont)
        (keyword, _pp) = lineContOp.getKeyword(0)
    if keyword != b"variableSetting":
        return ("NG", _("Dict内にvariableSettingがありません。"))

    (contents, p) = lineContOp.getMiddlePair(0)
    contentsOp = stringOp.strings(contents)
    variables = ""
    (lineCont, p, kind) = contentsOp.get1line(0)
    while lineCont != b"" and p < len(contents):
        if kind == b"line":
            variables += lineCont.decode() + "\n"
        (lineCont, p, kind) = contentsOp.get1line(p)
    return ("OK", "")

#
#  getInternalFieldsFromDict
def getInternalFieldsFromDict(dictCont):
    """ field名とfieldに対応する変数名、internalFieldを変数に設定するfield名を取得する"""
    global fluidFields, solidFields

    #  getNameCont
    def getNameCont(name, dictCont):
        #nameを取得
        lineCont = b" "; keyword = b" "; p = 0; name = name.encode()
        dictContOp = stringOp.strings(dictCont)
        while lineCont != b"" and keyword != name:
            (lineCont, p, _kind) = dictContOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
        (cont, p) = stringOp.strings(lineCont).getMiddlePair(0)
        return cont.decode()

    #  operationFields
    def operationFields(cont):
        ans = []
        contOp = stringOp.strings(cont)
        (fieldCont, p, _kind) = contOp.get1line(0)
        while fieldCont != b"" and p < len(cont):
            field = fieldCont.split()[0].decode()
            variable = fieldCont.split()[1].decode()
            if variable[-1] == ";":
                variable = variable[:-1]
            ans.append([field, [variable, "on"]])
            (fieldCont, p, _kind) = contOp.get1line(p)
        return ans

    #処理
    dictContOp = stringOp.strings(dictCont)
    p = dictContOp.skipFoamFile()
    lineCont = b" "; keyword = b" "
    while lineCont != b"" and keyword != b"internalFields":
        (lineCont, p, _kind) = dictContOp.get1line(p)
        (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
    if lineCont == b"":
        return ("NG", _("Dict内にinternalFieldsの設定がありません。"))
    (contents, p) = stringOp.strings(lineCont).getMiddlePair(0)
    contents = contents.decode()
    fluidCont = getNameCont("fluidRegions", contents)
    solidCont = getNameCont("solidRegions", contents)
    fluidFields = operationFields(fluidCont)
    solidFields = operationFields(solidCont)
    return ("OK", "")

#
#  getBoundaryConditionsFromDict
def getBoundaryConditionsFromDict(dictCont):
    """ 境界条件を取得"""
    global fluidBoundaryCont, solidBoundaryCont
    global specFluidBoundary, specSolidBoundary

    #  getNameCont
    def getNameCont(name, dictCont):
        #nameを取得
        lineCont = b" "; keyword = b" "; p = 0; name = name.encode()
        dictContOp = stringOp.strings(dictCont)
        while lineCont != "" and keyword != name:
            (lineCont, p, _kind) = dictContOp.get1line(p)
            lineContOp = stringOp.strings(lineCont)
            (keyword, _pp) = lineContOp.getKeyword(0)
        (cont, p) = lineContOp.getMiddlePair(0)
        return cont.decode()

    #  boundaryConditions
    def boundaryConditions(nameCont):
        boundaryCont = []
        nameContOp = stringOp.strings(nameCont)
        (fieldCont, p, _kind) = nameContOp.get1line(0)
        while fieldCont != b"" and p < len(nameCont):
            fieldContOp = stringOp.strings(fieldCont)
            (field, _pp) = fieldContOp.getKeyword(0)
            (boundary, _pp) = fieldContOp.getMiddlePair(0)
            boundary = stringOp.strings(boundary).compressSPCR()
            boundaryCont.append([field.decode(), boundary.decode()])
            (fieldCont, p, _kind) = nameContOp.get1line(p)
        return boundaryCont

    #  wildCardのRegionWallBC
    def getSpecRegionWallBC(name, dictCont):

        #get1wildCard
        def get1wildCard(dictCont, p):
            dictContOp = stringOp.strings(dictCont)
            (lineCont, p, _kind) = dictContOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
            while lineCont != b"" and len(keyword.split(b'"')) != 3:
                (lineCont, p, _kind) = dictContOp.get1line(p)
                (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
            return (lineCont, p)

        #specRegionWallを取得
        Bcont = []
        (lineCont, p) = get1wildCard(dictCont, 0)
        while lineCont != b"" and dictCont[p:] != b"":
            #wildCard取得
            lineContOp = stringOp.strings(lineCont)
            (wildCard, _pp) = lineContOp.getKeyword(0)
            #name（fluid or solid）の内容を取得
            (cont, _pp) = lineContOp.getMiddlePair(0)
            contents = getNameCont(name, cont)
            bound = boundaryConditions(contents)
            if len(bound) != 0:
                Bcont.append([wildCard.decode()] + boundaryConditions(contents))
            [lineCont, p] = get1wildCard(dictCont, p)
        return Bcont

    #処理
    dictContOp = stringOp.strings(dictCont)
    p = dictContOp.skipFoamFile()
    lineCont = b" "; keyword = b" "
    while lineCont != b"" and keyword != b"regionWallBoundaryConditions":
        (lineCont, p, _kind) = dictContOp.get1line(p)
        (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
    if lineCont == b"":
        return ("NG", _("Dict内にregionWallBoundaryConditionsの設定がありません。"))

    (boundaryCont, _pp) = stringOp.strings(lineCont).getMiddlePair(0)
    #fluid側を取得
    name = "fluidRegions"
    contents = getNameCont(name, boundaryCont)
    fluidBoundaryCont = boundaryConditions(contents)
    #solid側を取得
    name = "solidRegions"
    contents = getNameCont(name, boundaryCont)
    solidBoundaryCont = boundaryConditions(contents)

    #wildCardの設定を取得
    specFluidBoundary = getSpecRegionWallBC("fluidRegions", dictCont[p:]) 
    specSolidBoundary = getSpecRegionWallBC("solidRegions", dictCont[p:])

#
#  checkOperationFields
def checkOperationField():
    """ 操作するfield名をinternalField, boundaryFieldから確認し、足りないfieldを追加する。"""
    global fluidFields, fluidBoundaryCont, solidFields, solidBoundaryCont

    def addInFields(outFields, inFields):
        addFields = []
        for outField in outFields:
            same = 0
            for inField in inFields:
                if outField[0] == inField[0]:
                    same = 1
                    break
            if same == 0:
                addFields.append([outField[0], []])
        inFields += addFields
        return inFields

    #fluid側
    #  fluidFieldsをチェック
    fluidFields = addInFields(fluidBoundaryCont, fluidFields)
    for fluidField in fluidFields:
        if len(fluidField[1]) == 0:
            fluidField[1] = [" ", "off"]
    #  fluidBoundaryをチェック
    fluidBoundaryCont = addInFields(fluidFields, fluidBoundaryCont)
    for fluidBoundField in fluidBoundaryCont:
        if len(fluidBoundField[1]) == 0:
            fluidBoundField[1] = ""
    fluidFields.sort()
    #solid側
    #  solidFieldsをチェック
    solidFields = addInFields(solidBoundaryCont, solidFields)
    for solidField in solidFields:
        if len(solidField[1]) == 0:
            solidField[1] = [" ", "off"]
    #  fluidBoundaryをチェック
    solidBoundaryCont = addInFields(solidFields, solidBoundaryCont)
    for solidBoundField in solidBoundaryCont:
        if len(solidBoundField[1]) == 0:
            solidBoundField[1] = ""
    solidFields.sort()

#
#  isString
def isString(string):
    """ 文字列をチェック"""
    string = string.upper()
    if (string == "YES" or string == "ON" or string == "TRUE"):
        ans = True
    else:
        ans = False
    return ans

#
#  errorCheckInternalFieldName
def errorCheckInternalFieldName():
    """ internalFieldで使用する変数名をチェック"""

    def errorCheck(fluidSolid, intConts, variNames):
        errCont = ""
        for intCont in intConts:
            if isString(intCont[1][1]) == True:
                same = 0
                intName = intCont[1][0]
                for variName in variNames:
                    if intName == variName:
                        same = 1
                        break
                if same == 0:
                    errCont = [fluidSolid, intCont[0], intCont[1][0]]
                    break
        return errCont

    def makeMessage(errorCont):
        mess  = errorCont[0] + _(": ") + errorCont[1]
        mess += _(" internalField で使用している変数「")
        mess += errorCont[2] + _("」は、変数設定エリア内に定義されていません。")
        return mess        

    #変数名を取得
    variableList = variables.split("\n")
    variNames = []
    for line in variableList:
        varis = line.split()
        if len(varis) > 1:
            variNames.append(varis[0])
    #fluid側
    errorCont = errorCheck("fluid", fluidFields, variNames)
    if len(errorCont) != 0:
        return ("NG", makeMessage(errorCont))
    #solid側
    errorCont = errorCheck("solid", solidFields, variNames)
    if len(errorCont) != 0:
        return ("NG", makeMessage(errorCont))
    return ("OK", "")

#
#  errorCheckBoundaryFieldName
def errorCheckBoundaryFieldName():
    """ boundaryFieldで使用する変数名のエラーチェック"""

    def errorCheck(fluidSolid, boundaryConts, variNames):
        errorCont = ""
        for boundCont in boundaryConts:
            text = boundCont[1]
            n = text.find(" $")
            if n >= 0 and text[n:n+3] != " $:":
                textOp = stringOp.strings(text)
                (keyword, _p) = textOp.getKeyword(n) 
                boundName = keyword[1:].decode()
                same = 0
                for variName in variNames:
                    if boundName == variName:
                        same = 1
                        break
                if same == 0:
                    errorCont = [fluidSolid, boundCont[0], boundName]
                    break
        return errorCont

    def makeMessage(errorCont):
        mess  = errorCont[0] + _(u": ") + errorCont[1]
        mess += _(" boundaryField で使用している変数「")
        mess += errorCont[2] + _("」は、変数設定エリア内に定義されていません。")
        return mess

    #変数名を取得
    variableList = variables.split("\n")
    variNames = ["internalField"]
    for line in variableList:
        varis = line.split()
        if len(varis) > 1:
            variNames.append(varis[0])
    #fluid側
    errorCont = errorCheck("fluid", fluidBoundaryCont, variNames)
    if len(errorCont) != 0:
        return ("NG", makeMessage(errorCont))
    #solid側
    errorCont = errorCheck("solid", solidBoundaryCont, variNames)
    if len(errorCont) != 0:
        return ("NG", makeMessage(errorCont))
    return ("OK", "")

#
#  checkAllFields
def checkAllFields(caseDir):
    """ 各fieldのboundaryFieldをチェックする。
    各fieldの界面の境界条件が既に設定されているかどうかチェック。"""

    #  checkBoundaryField
    def checkBoundaryField(regName, fieldName):
        """ fieldの境界条件をチェック"""
        error = ""
        boundaryCont = b""
        if len(glob.glob(fieldName)) == 0:
            return ""
        (ascii, _data) = pyTreeFoam.foamFile().readSeparate(fieldName)
        asciiOp = stringOp.strings(ascii)
        (lineCont, p, _kind) = asciiOp.get1line(0)
        while lineCont != b"":
            lineContOp = stringOp.strings(lineCont)
            (keyword, _pp) = lineContOp.getKeyword(0)
            if keyword == b"boundaryField":
                (boundaryCont, _pp) = lineContOp.getMiddlePair(0)
                break
            (lineCont, p, _kind) = asciiOp.get1line(p)
        if boundaryCont != b"":
            boundaryContOp = stringOp.strings(boundaryCont)
            (lineCont, p, _kind) = boundaryContOp.get1line(0)
            while lineCont != b"":
                lineContOp = stringOp.strings(lineCont)
                (keyword, _pp) = lineContOp.getKeyword(0)
                #patch名にwaildCardを使わずにregion名があれば警告
                if keyword[0] != b'"' and keyword.find(regName.encode() + b"_to_") >= 0:
                    error = regName
                    break
                (lineCont, p, _kind) = boundaryContOp.get1line(p)
        return error

    #処理------------------------
    timeFolder = pyTreeFoam.case(caseDir).getTimeFolders()[0]
    #fluid側のチェック
    fluidErrRegions = []
    for regName in fluidRegions:
        #flag = 0
        fieldDir = caseDir + "/" + timeFolder + "/" + regName
        for fieldCont in fluidFields:
            field = fieldCont[0]
            fieldName = fieldDir + "/" + field
            error = checkBoundaryField(regName, fieldName)
            if error != "":
                fluidErrRegions.append(regName)
                break
    #solid側のチェック
    solidErrRegions = []
    for regName in solidRegions:
        fieldDir = caseDir + "/" + timeFolder + "/" + regName
        for fieldCont in solidFields:
            field = fieldCont[0]
            fieldName = fieldDir + "/" + field
            #patch名に「_to_」の文字があるかどうかチェック
            error = checkBoundaryField(regName, fieldName)
            if error != "":
                solidErrRegions.append(regName)
                break
    #終了処理
    if len(fluidErrRegions) == 0 and len(solidErrRegions) == 0:
        mess = ""
        stat = "OK"
    else:
        stat = "NG"
        mess  = _("以下のregion内には、既に界面の境界条件が設定されています。\n")
        mess += _("界面の境界条件を再設定しますか？\n")
        if len(fluidErrRegions) != 0:
            mess += _("fluid:")
            if len(fluidErrRegions) > 5:
                mess += ", ".join(fluidErrRegions[:5]) + "..."
            else:
                mess += ", ".join(fluidErrRegions)
        if len(solidErrRegions) != 0:
            mess += _("\nsolid:")
            if len(solidErrRegions) > 5:
                mess += ", ".join(solidErrRegions[:5]) + "..."
            else:
                mess += ", ".join(solidErrRegions)
    return (stat, mess)

#
#  replaceRegionFieldBoundaryConditions
def replaceRegionFieldBoundaryConditions(caseDir, fluidRegs, solidRegs):
    """ region内のboundaryFieldのpatchを再設定する"""

    #  deleteErrVariable
    def deleteErrVariable(cont, variablesLine):
        """ 余分な変数名を削除。（includeFile内の変数名と同じ変数名は削除）"""
        variables = variablesLine.split("\n")
        #pointerをdimensionsまで進める
        contOp = stringOp.strings(cont)
        p = contOp.skipFoamFile()
        lineCont = b" "; keyword = b" "
        while lineCont != b"" and keyword != b"dimensions":
            (lineCont, p, kind) = contOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
        #dimensionsまでを保存
        newAscii = cont[:p] + "\n"
        #include内の変数（variables）と同じ変数は削除する
        lineCont = b" "; keyword = b" "
        while lineCont != b"" and keyword != b"internalField":
            (lineCont, p, kind) = contOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
            if kind == b"line" and keyword != b"internalField":
                lineCont = lineCont.decode()
                name = lineCont.split()[0]
                same = 0
                for variable in variables:
                    if len(variable) > 0:
                        if variable.split()[0] == name:
                            same = 1
                            break
                if same == 0:
                    newAscii += lineCont + "\n"
            else:
                if keyword == b"internalField":
                    newAscii += "\n" + lineCont.decode()
                else:
                    newAscii += lineCont.decode() + "\n"
        newAscii += cont[p:]
        return newAscii

    #  deleteErrIncludeFile
    def deleteErrIncludeFile(cont, includeFiles):
        """ 余分なincludeFileを削除。
        （同じfile名でincludeDirが異なっているincludeFile）"""
        contOp = stringOp.strings(cont)
        p = contOp.skipFoamFile()
        lineCont = b" "; keyword = b" "
        while lineCont != b"" and keyword != b"dimensions":
            (lineCont, p, kind) = contOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
        newAscii = cont[:p] + "\n"
        lineCont = b" "; keyword = b" "
        while lineCont != b"" and keyword != b"internalField":
            (lineCont, p, kind) = contOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
            if kind == b"include":
                lineCont = lineCont.decode()
                incFile = lineCont.split()[1]
                name = incFile.split("/")[-1]
                if (name == 'variableSetting"' or
                    name == 'boundaryConditionsFluid"' or
                    name == 'boundaryConditionsSolid"'):
                    if (incFile == includeFiles[0] or
                        incFile == includeFiles[1]):
                        newAscii += lineCont + "\n"
            else:
                if keyword == b"internalField":
                    newAscii += "\n" + lineCont.decode()
                else:
                    newAscii += lineCont.decode() + "\n"
        newAscii += cont[p:]
        return newAscii

    #  replaceField
    def replaceField(fluidSolid, fieldName, intCont):
        """ fieldの置き換え"""

        #  setWildCard
        def setWildCard(fluidSolid, field):
            """ specWildCar内容を作成"""
            #使用する変数を決定
            if fluidSolid == "fluid":
                specBoundary = specFluidBoundary
            else:
                specBoundary = specSolidBoundary
            #patch内容を作成
            lines = ""
            n = 0
            for wildCont in specBoundary:
                wildCard = wildCont[0]
                for (specField, _pCont) in wildCont[1:]:
                    if specField == field:
                        lines += "    " + wildCard + "\n"
                        lines += "    {\n"
                        lines += "        $:specRegionWall_" + str(n) + "." + field + ";\n"
                        lines += "    }\n\n"
                n += 1
            return lines

        #fieldの読み込み
        ascii = pyTreeFoam.foamFile().read(fieldName)
        asciiOp = stringOp.strings(ascii)
        p = asciiOp.skipFoamFile()
        #includeの置き換え
        #  includeFileを設定
        incDir = includeDirOrg
        if incDir[0] == '"':
            incDir = includeDirOrg.split('"')[1]
        includeFiles = ['"' + incDir + '/variableSetting"']
        if fluidSolid == "fluid":
            includeFiles.append('"' + incDir + '/boundaryConditionsFluid"')
        else:
            includeFiles.append('"' + incDir + '/boundaryConditionsSolid"')
        #  pointerをdimensionsの次まで進める
        (lineCont, p, kind) = asciiOp.get1line(p)
        while lineCont != b"":
            (keyword, pp) = stringOp.strings(lineCont).getKeyword(0)
            if keyword == b"dimensions":
                ps = p+1
                break
            (lineCont, p, kind) = asciiOp.get1line(p)
        #  同じincludeFileが存在しているかどうかチェック
        isInclude = [False, False]
        (lineCont, p, kind) = asciiOp.get1line(p)
        (keyword, pp) = stringOp.strings(lineCont).getKeyword(0)
        while lineCont != b"" and keyword != b"internalField":
            if kind == b"include":
                lineCont = lineCont.decode()
                incFile = lineCont.split()[1]
                if incFile == includeFiles[0]:
                    isInclude[0] = True
                if incFile == includeFiles[1]:
                    isInclude[1] = True
                ps = p
            (lineCont, p, kind) = asciiOp.get1line(p)
            (keyword, pp) = stringOp.strings(lineCont).getKeyword(0)
        #  includeFileを設定
        addLines = ""
        if isInclude[0] == False:
            addLines += '#include ' + includeFiles[0] + "\n"
        if isInclude[1] == False:
            addLines += '#include ' + includeFiles[1] + "\n"
        header = ascii[:ps]
        footer = ascii[ps:]
        ascii = header + addLines + footer
        #  余分なincludeFileを削除
        #  （同じfile名でincludeDirが異なっているincludeFile）
        ascii = deleteErrIncludeFile(ascii, includeFiles)
        #  余分な変数名を削除
        #  （includeFile内の変数名と同じ変数名は削除）
        ascii = deleteErrVariable(ascii, variables)
        #pointerを保存
        asciiOp = stringOp.strings(ascii)
        pss = asciiOp.skipFoamFile()
        pss = asciiOp.getKeywordPointer("dimensions", pss)
        #internalField置き換え
        internalVar = intCont[0]
        if isString(intCont[1]) == True:
            ps = asciiOp.getKeywordPointer("internalField", pss)
            (lineCont, p, kind) = asciiOp.get1line(ps)
            lineContOp = stringOp.strings(lineCont)
            (keyword, pp) = lineContOp.getKeyword(0)
            (keyword, pp) = lineContOp.getKeyword(pp)
            if keyword == b"uniform":
                line = "internalField   uniform $" + internalVar + ";"
                (ascii, p, keyword) = asciiOp.replace1lineKeyword(ps, line)
                ascii = ascii.decode()
        #boundaryの書き換え
        #  boundary部を取得
        field = fieldName.split("/")[-1]
        ps = asciiOp.getKeywordPointer("boundaryField", pss)
        header = ascii[:ps] + "boundaryField\n{\n"
        (boundCont, pp) = asciiOp.getMiddlePair(ps)
        footer = "}" + ascii[pp:]
        #  region間のpatchを削除
        newCont = ""
        pattern = '".*._to_.*"'
        boundContOp = stringOp.strings(boundCont)
        (lineCont, p, kind) = boundContOp.get1line(0)
        (keyword, pp) = stringOp.strings(lineCont).getKeyword(0)
        while lineCont != b"":
            if isMatchName(pattern, keyword.decode()) == False:
                if keyword[0] != b'"':
                    newCont += "    " + lineCont.decode() + "\n"
            (lineCont, p, kind) = boundContOp.get1line(p)
            (keyword, pp) = stringOp.strings(lineCont).getKeyword(0)
        #  ".*._to_.*"の設定
        line  = "    " + pattern + "\n"
        line += "    {\n"
        line += "        $:wallToRegion." + field + ";\n"
        line += "    }\n"
        newCont += line
        #  汎用wildCardの設定
        lines = setWildCard(fluidSolid, field)
        newCont += lines
        ascii = header + newCont + footer
        #再設定したfieldを保存
        #pyTreeFoam.foamFile().writeSeparate(fieldName, [ascii, data])
        pyTreeFoam.foamFile().write(fieldName, ascii)

    #処理----------------------
    errFields = []
    #Dictの読み込み
    initialize(caseDir)
    #includeFileの作成
    error = createIncludeFiles(caseDir)
    if error[0] != "OK":
        return error

    #wildCard部のfield名チェック
    dictCont = readDictFile(caseDir)
    (error, mess) = checkSpecWildCard(dictCont)
    if error != "OK":
        return (error, mess)

    #fluid側の設定
    timeFolder = pyTreeFoam.case(caseDir).getTimeFolders()[0]
    error = ""
    for regName in fluidRegs:
        for fieldCont in fluidFields:
            field = fieldCont[0]
            fieldName = caseDir + "/" + timeFolder + "/" + regName + "/" + field
            if len(glob.glob(fieldName)) == 0:
                #存在しないfield名を保存
                errFields.append(fieldName.split("/")[-1])
            else:
                #fieldの置き換え（修正）
                replaceField("fluid", fieldName, fieldCont[1])
    #solid側の設定
    for regName in solidRegs:
        for fieldCont in solidFields:
            field = fieldCont[0]
            fieldName = caseDir + "/" + timeFolder + "/" + regName + "/" + field
            if len(glob.glob(fieldName)) == 0:
                #存在しないfield名を保存
                errFields.append(fieldName.split("/")[-1])
            else:
                #fieldの置き換え（修正）
                replaceField("solid", fieldName, fieldCont[1])
    #errFieldsの修正
    errFields.sort()
    fields = []
    name = ""
    for errName in errFields:
        if name != errName:
            fields.append(errName)
            name = errName            
    return ("OK", fields)

#
#  isMatchName
def isMatchName(pattern, name):
    """ nameがpatternと一致するかどうかをチェック。
    一致：Trueが戻る。"""
    if pattern[0] == '"':
        pattern = pattern[1:-1]
    ans = False
    obj = re.match(pattern, name)   #正規表現でチェック
    if obj != None:
        ans = True
    return ans

#
#  setTab
def setTab(word, tab):
    """ wordの長さを空白を入れて揃える"""
    if len(word) < tab:
        newWord = word + " "*(tab - len(word))
    else:
        newWord = word + " "
    return newWord

#
#  saveVariableToFile_variable
def saveVariableToFile_variable(contents):
    """ 変数定義をfileに保存"""
    varList = variables.split("\n")
    corrVar = []
    for var in varList:
        if len(var) > 0:
            words = var.split()
            name = setTab(words[0], 16)
            value = " ".join(words[1:])
            corrVar.append("    " + name + value)
    corrVariables = "\n".join(corrVar)
    newCont  = "variableSetting\n"
    newCont += "{\n"
    newCont += corrVariables + "\n"
    newCont += "}"
    contentsOp = stringOp.strings(contents)
    p = contentsOp.skipFoamFile()
    (newCont, p, _keyword) = contentsOp.replace1lineKeyword(p, newCont)
    return newCont.decode()

#
#  saveVariableToFile_internalField
def saveVariableToFile_internalField(contents):
    """ internalFieldの設定をfileに保存"""

    def deleteNullInternalData(conts):
        newCont = []
        for cont in conts:
            if len(cont[1][0].split()) != 0:
                if isString(cont[1][1]) == True:
                    newCont.append(cont)
        return newCont

    newCont  = "internalFields\n"
    newCont += "{\n"
    #fluid側を設定
    newCont += "    fluidRegions\n"
    newCont += "    {\n"
    internalData = deleteNullInternalData(fluidFields)
    for fieldCont in internalData:
        name = fieldCont[0]
        line  = "    " + setTab(name, 12) + fieldCont[1][0] + ";\n"
        newCont += "    " + line
    newCont += "    }\n"
    #solid側を設定
    newCont += "    solidRegions\n"
    newCont += "    {\n"
    internalData = deleteNullInternalData(solidFields)
    for fieldCont in internalData:
        name = fieldCont[0]
        line  = "    " + setTab(name, 12) + fieldCont[1][0] + ";\n"
        newCont += "    " + line
    newCont += "    }\n"
    newCont += "}"
    contentsOp = stringOp.strings(contents)
    p = contentsOp.skipFoamFile()
    (newCont, p, _kind) = contentsOp.replace1lineKeyword(p, newCont)
    return newCont.decode()

#
#  saveVariableToFile_boundary
def saveVariableToFile_boundary(contents):
    """ boundaryのデータをfileに保存"""

    def deleteNullBoundaryCont(conts):
        newCont = []
        for cont in conts:
            line = stringOp.strings(cont[1]).compressSPCR()
            line = line.decode()
            if not (line == "" or line == "\n" or line == " \n" or
                    line == noChange):
                cont[1] = line
                newCont.append(cont)
        return newCont

    newCont  = "regionWallBoundaryConditions\n"
    newCont += "{\n"
    newCont += "    fluidRegions\n"
    newCont += "    {\n"
    #fluid側
    fieldData = deleteNullBoundaryCont(fluidBoundaryCont)
    for fieldCont in fieldData:
        newCont += "        " + fieldCont[0] + "\n"
        newCont += "        {\n"
        lines = fieldCont[1].split("\n")
        for line in lines:
            newCont += "            " + line + "\n"
        newCont += "        }\n\n"
    newCont += "    }\n"
    #solid側
    newCont += "    solidRegions\n"
    newCont += "    {\n"
    fieldData = deleteNullBoundaryCont(solidBoundaryCont)
    for fieldCont in fieldData:
        newCont += "        " + fieldCont[0] + "\n"
        newCont += "        {\n"
        lines = fieldCont[1].split("\n")
        for line in lines:
            newCont += "            " + line + "\n"
        newCont += "        }\n\n"
    newCont += "    }\n"
    newCont += "}"
    contentsOp = stringOp.strings(contents)
    p = contentsOp.skipFoamFile()
    (newCont, p, _kind) = contentsOp.replace1lineKeyword(p, newCont)
    return newCont.decode()

#
#  getAbsPath
def getAbsPath(caseDir, cont):
    """ 絶対pathに変換"""
    case = pyTreeFoam.case(caseDir)
    cont = case.getAbsPath(caseDir, cont)
    return cont

#
#  createIncludeFiles
def createIncludeFiles(caseDir):
    """ includeFileを指定されたdirectory内に作成する。"""
    global includeDir

    #  getIncludeDir
    def getIncludeDir(dictCont):
        """ includeDirを取得"""
        dictContOp = stringOp.strings(dictCont)
        pCont = dictContOp.skipFoamFile()
        (lineCont, p, _kind) = dictContOp.get1line(pCont)
        (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
        while lineCont != b"" and keyword != b"includeDir":
            (lineCont, p, _kind) = dictContOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
        if keyword == b"includeDir":
            lineCont = lineCont.decode()
            includeDir = lineCont.split()[1]
            if includeDir[-1] == ";":
                includeDir = includeDir[:-1]
            includeDir = includeDir.split('"')[1]
            includeDir = getAbsPath(caseDir, includeDir)
        else:
            includeDir = ""
        return includeDir

    #  makeFieldPatchCont
    def makeFieldPatchCont(boundaryCont):
        Bcont = "wallToRegion\n{\n"
        for [field, Pcont] in boundaryCont:
            Bcont += "    " + field + "\n"
            Bcont += "    {\n"
            lines = Pcont.split("\n")
            for line in lines:
                Bcont += "        " + line + "\n"
            Bcont += "    }\n\n"
        Bcont += "}\n\n"
        return Bcont

    #  makeSpecFieldPatchCont
    def makeSpecFieldPatchCont(specBoundary):
        n = 0
        Bcont = ""
        for wildCardCont in specBoundary:
            wildCard = wildCardCont[0]
            Bcont += "// " + wildCard + "\n"
            Bcont += "specRegionWall_" + str(n) + "\n{\n"
            for [field, Pcont] in wildCardCont[1:]:
                Bcont += "    " + field + "\n"
                Bcont += "    {\n"
                lines = Pcont.split("\n")
                for line in lines:
                    Bcont += "        " + line + "\n"
                Bcont += "    }\n\n"
            Bcont += "}\n\n"
            n += 1
        return Bcont

    #dict読み込み
    dictCont = readDictFile(caseDir)
    #header, footer作成
    OFversion = configDict["OFversion"]
    foamContents = pyTreeFoam.getFoamContents(caseDir)
    header = foamContents.makeFoamHeaderVersion(OFversion)
    header += foamContents.makeFoamFile("2.0", "ascii", "dictionary", "",
            "regionWallBoundaryConditionsDict")
    footer = foamContents.makeFoamFooter()
    #includeDirを取得
    includeDir = getIncludeDir(dictCont)
    if includeDir != "":
        if len(glob.glob(includeDir)) == 0:
            os.makedirs(includeDir)
    else:
        error = ("NG", _(u"Dict内にincludeDirが設定されていません。"))
        return error

    #variableSettingを作成
    variCont = header + variables + footer
    fileName = includeDir + "/variableSetting"
    f=open(fileName, "w"); f.write(variCont); f.close()
    #fluidBoundaryConditionsを作成
    #  wallToRegionの作成
    fluidCont = makeFieldPatchCont(fluidBoundaryCont)
    #  specRegionWallの作成
    fluidCont += makeSpecFieldPatchCont(specFluidBoundary)
    fluidCont = header + fluidCont + footer
    fileName = includeDir + "/boundaryConditionsFluid"
    f=open(fileName, "w"); f.write(fluidCont); f.close()
    #solidBoundaryConditionsを作成
    #  wallToRegionの作成
    solidCont = makeFieldPatchCont(solidBoundaryCont)
    #  specRegionWallの作成
    solidCont += makeSpecFieldPatchCont(specSolidBoundary)
    solidCont = header + solidCont + footer
    fileName = includeDir + "/boundaryConditionsSolid"
    f=open(fileName, "w"); f.write(solidCont); f.close()
    return ("OK", "")

#
#  checkSpecWildCard
def checkSpecWildCard(dictCont):
    """ specWildCardのfield名のチェック"""

    def getErrFields(name, wildCont):
        lineCont = b" "; keyword = b" "; p = 0
        wildContOp = stringOp.strings(wildCont)
        while lineCont != b"" and keyword != name.encode():
            (lineCont, p, _kind) = wildContOp.get1line(p)
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
        fields = []
        if lineCont != b"":
            (nameCont, _pp) = stringOp.strings(lineCont).getMiddlePair(0)
            lineCont = b" "; p = 0
            while lineCont != b"":
                (lineCont, p, _kind) = stringOp.strings(nameCont).get1line(p)
                if lineCont != b"":
                    (field, _pp) = stringOp.strings(lineCont).getKeyword(0)
                    fields.append(field.decode())
        if name == "fluidRegions":
            defineFields = fluidBoundaryCont
        else:
            defineFields = solidBoundaryCont
        errFields = []
        for field in fields:
            same = 0
            for defField in defineFields:
                if defField[0] == field:
                    same = 1
                    break
            if same == 0:
                errFields.append(field)
        errFields.sort()
        err = []
        errField = ""
        for field in errFields:
            if errField != field:
                err.append(field)
                errField = field
        return err

    errFluidFields = []
    errSolidFields = []
    lineCont = b" "; p = 0
    dictContOp = stringOp.strings(dictCont)
    while lineCont != b"":
        (lineCont, p, _kind) = dictContOp.get1line(p)
        if len(lineCont) > 0:
            (keyword, _pp) = stringOp.strings(lineCont).getKeyword(0)
            if keyword[0] == b'"':
                (wildCont, _pp) = stringOp.strings(lineCont).getMiddlePair(0)
                fields = getErrFields("fluidRegions", wildCont.decode())
                errFluidFields += fields
                fields = getErrFields("solidRegions", wildCont.decode())
                errSolidFields += fields
                #errorCheck
    if len(errFluidFields) > 0 or len(errSolidFields) > 0:
        mess = _("\nerror:wildCardのpatch名内で使用している以下のfieldは、regionWallBoundaryConditions内で定義されていない為、反映できません。\n")
        mess += _("regionWallBoundaryConditions内でfieldとpatchTypeを定義してください。\n")
        if len(errFluidFields) > 0:
            mess += _("  fluidRegions: ") + ", ".join(errFluidFields) + "\n"
        if len(errSolidFields) > 0:
            mess += _("  solidRegions: ") + ", ".join(errSolidFields) + "\n"
        mess = mess[:-1]
        print(mess)
        error = ("NG", mess)
    else:
        error = ("OK", "")
    return error




if __name__ == "__main__":
    import gettext
    gettext.install("treefoam", localeDir)
    #_ = gettext.gettext

    caseDir = sys.argv[1]
    initialize(caseDir)
    winApp = windowApp(caseDir)
    winApp.main()

