#!/usr/bin/python3
#  coding: utf-8
#
#       QtVtk.py
#
#       vtkをQt上で実現させるためのモジュール
#
#   20/05/24    新規作成
#      05/25    getPatchNamesBlockIndex:バグ修正
#               multiRegionで親のpolyMeshが存在しない場合、indexがずれる
#      08/02    getPatchNamesBlockIndex:multiRegionCaseのチェックを追加
#               修正前は、lagrangianがmultiRegionとして設定されてしまう。
#      09/15    vtkのimport状況、vtkのバージョンをを出力。
#      10/29    logFileCreaterのimportを追加。（logを出力させるため）
#   21/06/12    reloadRemakeRenderWindow:再読込用を追加
#      12/21    selectPatches:faceの両面を表示する様に修正。
#   22/01/01    vtk-9.1.0対応で大幅修正。
#      01/02    makeRenderWindow, reloadRemakeRenderWindow:引数として
#               selPatchNamesを追加。（vtk起動時に選択patchを表示させる為）
#   24/02/12    getPatchNamesBlockIndex:OF-11対応で、filmを追加
#      08/30    QtVtkRenderWindowInteractorのimport方法を修正
#      10/25    mouseSelectActor:OpenFoamPatchMeshクラス内から外に出す。
#   25/03/30    initialVtkArea:vtk-9.4対応でGradientBackgroundOn()を追加
#


try:
    import logFileCreater
    logFileCreater.log()
except:
    pass

print("importing vtk...")
import vtk
print("  imported " + vtk.vtkVersion.GetVTKSourceVersion())

#import QtVtkRenderWindowInteractor as Qvtk

VtkFlag = None
#vtkのチェック
try:
    #Qt6に対応したQtVtk
    import QtVtkRenderWindowInteractorVtk9 as Qvtk
    VtkFlag = "ok"
except:
    try:
        import QtVtkRenderWindowInteractorVtk7 as Qvtk
        VtkFlag = "ok"
    except:
        VtkFlag = None
        raise ImportError("Cannot load QtVtkRenderWindowInteractor")

from vtk.util import colors
import time
import glob
import os
import pyTreeFoam

#from PyQt4 import QtCore, QtGui
import getPyQtModule
PyQtModule = getPyQtModule.PyQtModule

#  import されているPyQtに応じてimportする
if PyQtModule == "PyQt6":
    from PyQt6.QtCore import *
    from PyQt6.QtGui import *
    from PyQt6.QtWidgets import *
elif PyQtModule == "PySide6":
    from PySide6.QtCore import *
    from PySide6.QtGui import *
    from PySide6.QtWidgets import *
elif PyQtModule == "PyQt5":
    from PyQt5.QtCore import *
    from PyQt5.QtGui import *
    from PyQt5.QtWidgets import *
elif PyQtModule == "PySide2":
    from PySide2.QtCore import *
    from PySide2.QtGui import *
    from PySide2.QtWidgets import *
elif PyQtModule == "PyQt4":
    from PyQt4.QtCore import *
    from PyQt4.QtGui import *
elif PyQtModule == "PySide":
    from PySide.QtCore import *
    from PySide.QtGui import *
else:
    print("import error: could not import PyQt or PySide!")
    exit()


#--------------------
#  vtkQWidget class
#--------------------
class vtkQWidget(Qvtk.QVTKRenderWindowInteractor):
    """ mouseの操作を「vtkInteractorStyleTrackballCamera()」に設定
    patchのselectSignalを定義
    したものにoverrideする。"""

    #signalを設定
    try:
        #PyQt系
        selectedSignal   = pyqtSignal()     #patch選択時
        rightClickSignal = pyqtSignal()     #右クリック時
    except:
        #PySide系
        selectedSignal   = Signal()         #patch選択時
        rightClickSignal = Signal()         #右クリック時

    def __init__(self, parent=None):
        super(vtkQWidget, self).__init__(parent)
        #super(vtkQWidget, self).__init__()
        inter = self.GetRenderWindow().GetInteractor()
        #inter.SetInteractorStyle(vtk.vtkInteractorStyleTrackballCamera())

#--------------------------
#  mouseSelectActor class
#--------------------------
class mouseSelectActor(vtk.vtkInteractorStyleTrackballCamera):
    """ vtk画面のmouseのeventStyleをoverrideする。"""

    def __init__(self, parent):
        self.AddObserver("LeftButtonPressEvent",self.leftButtonPressEvent)
        self.AddObserver("RightButtonPressEvent",self.rightButtonPressEvent)
        

        self.LastPickedActor = None
        self.LastPickedProperty = vtk.vtkProperty()

        self.actorNo = 0                #選択したactorNo
        self.oldTime = time.time()
        self.parent = parent

    #
    #  leftButtonPressEvent
    #-----------------------
    def leftButtonPressEvent(self, obj, event):
        """ 左クリックのeventをチェック。"""

        clickPos = self.GetInteractor().GetEventPosition()

        picker = vtk.vtkPropPicker()
        picker.Pick(clickPos[0], clickPos[1], 0, self.GetDefaultRenderer())

        # get the new
        self.NewPickedActor = picker.GetActor()

        #doubleClick?
        currTime = time.time()
        dTime = currTime - self.oldTime
        self.oldTime = currTime
        if dTime <= 0.5:
            # If something was selected
            if self.NewPickedActor:
                # If we picked something before, reset its property
                #if self.LastPickedActor:
                #    self.LastPickedActor.GetProperty().DeepCopy(self.LastPickedProperty)
                
                # Save the property of the picked actor so that we can
                # restore it next time
                #self.LastPickedProperty.DeepCopy(self.NewPickedActor.GetProperty())

                # Highlight the picked actor by changing its properties
                # self.NewPickedActor.GetProperty().SetColor(1.0, 0.0, 1.0)
                # self.NewPickedActor.GetProperty().SetDiffuse(1.0)
                # self.NewPickedActor.GetProperty().SetSpecular(0.0)
                
                # save the last picked actor
                #self.LastPickedActor = self.NewPickedActor

                # actorのNoを取得
                actors = self.GetDefaultRenderer().GetActors()
                i = 0
                for actor in actors:
                    if actor == self.NewPickedActor:
                        break
                    i += 1
                self.actorNo = i

                #ctrlKeyを確認
                ctrl, shift = self.parent.vtkWidget._GetCtrlShift(event)
                if ctrl == True:
                    #actor選択を追加する
                    self.parent.selectAddActor(self.actorNo)
                else:
                    #actorを選択する
                    self.parent.selectActor(self.actorNo)

        self.OnLeftButtonDown()
        return

    #
    #  rightButtonPressEvent
    #------------------------
    def rightButtonPressEvent(self, obj, event):
        #self.parent.vtkWidget.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        #self.parent.vtkWidget.setContextMenuPolicy(Qt.CustomContextMenu)
        #self.parent.vtkWidget.customContextMenuRequested.connect(self.parent.showPopupMenu)
        self.parent.vtkWidget.rightClickSignal.emit()
        #self.OnRightButtonDown()
        return


#--------------------------
#  OpenFoamPatchMesh class
#--------------------------
class OpenFoamPatchMesh():
    """ OpenFOAMのpatchをvtkを使って3D表示させ、各patchの境界条件を設定する。"""

    def __init__(self, vtkWidget, caseDir, region=".", timeFolder="0"):
        self.caseDir = caseDir
        self.fileName = caseDir + "/system/controlDict"
        self.region = region
        self.timeFolder = timeFolder
        self.vtkWidget = vtkWidget  #vtkRenderWindowInteractor
        self.enabled_block = []     #表示する全patch名
        self.selected_block = []    #選択したpatch名
        self.current_block = []     #表示しているpatch名
        self.regionBlock = None

        self.showEdge = True        #edge表示/非表示flag
        self.setOpacity = False     #透明flag

        self.vtkWidget.setToolTip(_("ダブルクリックしてpatchを選択する"))

        #reader
        self.reader = vtk.vtkOpenFOAMReader()
        self.reader.SetFileName(self.fileName)
        #self.reader.Update()
        self.reader.UpdateTimeStep(float(self.timeFolder))
        self.reader.EnableAllPatchArrays()
        self.reader.Update()
        #patch名とblock_indexを取得
        self.getPatchNamesBlockIndex()

    #
    #  getPatchNamesBlockIndex
    def getPatchNamesBlockIndex(self):
        """
        regionのblockを取得
        """
        #blockDataを取得
        blockDataDict = {}
        output = self.reader.GetOutput()
        blockData = None
        #region名とdataを取得
        metaData = output.GetMetaData(0)
        blockName = self.getBlockName(metaData)
        #通常のcase
        if blockName == "internalMesh":
            blockData = output
            blockDataDict["."] = blockData
        #film?
        elif blockName == "film":
            blockData = output.GetBlock(0)
            blockDataDict["film"] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData
        #multiRegion(fluidとsolid)？
        else:
            blockData = output.GetBlock(0)
            blockDataDict["."] = blockData
            nBlocks = output.GetNumberOfBlocks()
            for i in range(1, nBlocks):
                metaData = output.GetMetaData(i)
                blockName = self.getBlockName(metaData)
                blockData = output.GetBlock(i)
                blockDataDict[blockName] = blockData
        #regionBlockを取得
        self.regionBlock = blockDataDict[self.region]

    #
    #  getBlockName
    def getBlockName(self, metaData):
        """ metaDataからそのblockのblockNameを取得する"""
        lines = str(metaData).split("\n")
        blockName = ""
        for line in lines:
            words = line.split()
            if words[0] == "NAME:":
                blockName = words[1]
                break
        return blockName

    #
    #  makeRenderWindow
    #-------------------
    def makeRenderWindow(self, selPatchNames):
        """ renderWindowを作成する。
        enabled_block[0]のpatchを選択表示させる。"""
        #patchのactorを作成
        self.actorDict = self.getRegionActors()
        self.current_block = list(self.actorDict.keys())
        self.enabled_block = self.current_block[:]
        #選択面をクリア
        #selPatchNames = []
        self.selectPatches(selPatchNames)
        # renderer
        self.ren = vtk.vtkRenderer()
        self.ren.GradientBackgroundOn()
        self.ren.SetBackground(1, 1, 1)
        self.ren.SetBackground2(0.4, 0.55, 0.75)
        #actorを追加
        for name in self.current_block:
            self.ren.AddActor(self.actorDict[name])

        renWin = self.vtkWidget.GetRenderWindow()
        renWin.NewInstance()
        renWin.AddRenderer(self.ren)

        #styleを設定(event)
        #style = self.mouseSelectActor(self)
        style = mouseSelectActor(self)
        style.SetDefaultRenderer(self.ren)
        interactor = renWin.GetInteractor()
        interactor.SetInteractorStyle(style)

        #add axis
        self.axesActor = vtk.vtkAxesActor()
        self.axes = vtk.vtkOrientationMarkerWidget()
        self.axes.SetOrientationMarker(self.axesActor)
        self.axes.SetInteractor(interactor)
        self.axes.EnabledOn()
        self.axes.InteractiveOn()
        self.ren.ResetCamera()

    #
    #  getRegionActors
    def getRegionActors(self):
        """ region内のpatchActorを取得"""
        regionBlock = self.regionBlock
        nBlocks = regionBlock.GetNumberOfBlocks()
        patchActorDict = {}
        for i in range(nBlocks):
            metaData = regionBlock.GetMetaData(i)
            blockName = self.getBlockName(metaData)
            #vtk 9.1.0から名称変更
            #           9.0.3以前               9.1.0以降
            if blockName == "Patches" or blockName == "boundary":
                patchBlock = regionBlock.GetBlock(i)
                patchNamesIds = self.getPatchNamesIds(patchBlock)
                patchActorDict = self.createPatchActors(regionBlock, patchNamesIds)
                break
        return patchActorDict

    #
    #  getPatchNamesIds
    def getPatchNamesIds(self, patchBlock):
        """ blockDataからpatchNameとpatchIdを取得する"""
        patchIds = []
        patchNames = []
        index = 3
        nBlocks = patchBlock.GetNumberOfBlocks()
        for i in range(nBlocks):
            metaData = patchBlock.GetMetaData(i)
            patchName = self.getBlockName(metaData)
            if self.region != ".":
                patchName = self.region + "/" + patchName
            patchIds.append(index)
            patchNames.append(patchName)
            print(patchName, index)
            index += 1
        return patchNames, patchIds

    #
    #  createPatchActors
    def createPatchActors(self, regionBlock, patchNamesIds):
        """ patchActor, outlineActor(featureEdge)を作成する"""
        (patchNames, patchIds) = patchNamesIds
        patchActorDict = {}
        for i in range(len(patchIds)):
            index = patchIds[i]
            extract_block = vtk.vtkExtractBlock()
            extract_block.SetInputData(regionBlock)
            extract_block.AddIndex(index)
            extract_block.Update()
            #filter
            geom_filter = vtk.vtkGeometryFilter()
            geom_filter.SetInputConnection(extract_block.GetOutputPort())
            #featureEdge
            feature_edge = vtk.vtkFeatureEdges()
            feature_edge.SetInputConnection(geom_filter.GetOutputPort())
            #mapper
            mapper = vtk.vtkCompositePolyDataMapper()
            mapper.SetInputConnection(geom_filter.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(feature_edge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            prop = actor.GetProperty()
            prop.SetColor(colors.green)
            name = patchNames[i]
            patchActorDict[name] = actor
        return patchActorDict

    #
    #  deleteAllActors
    #------------------
    def deleteAllActors(self):
        """ 表示されている全てのactorを削除する"""
        #renderWindowから全actorを削除する
        for delActorName in self.actorDict.keys():
            if delActorName in self.current_block:
                self.ren.RemoveActor(self.actorDict[delActorName])
                self.current_block.remove(delActorName)

    #
    #  reloadRemakeRenderWindow
    #---------------------------
    def reloadRemakeRenderWindow(self, selPatchNames):
        """ reloadしてrenderWindowを再作成する。"""
        #actorを再作成
        self.actorDict = self.getRegionActors()
        self.current_block = list(self.actorDict.keys())
        self.enabled_block = self.current_block[:]
        #select表示させる（actorのpropertyを設定）
        #selPatchNames = []
        self.selectPatches(selPatchNames)
        #renderWindowにactorを追加
        for name in self.enabled_block:
            self.ren.AddActor(self.actorDict[name])
        #表示させる
        self.vtkWidget.Initialize()

    #
    #  selectActor
    #---------------
    def selectActor(self, actorNo):
        """ mouseのdoubleClickで選択したactorを選択表示に設定する。
        
        Args:
            actorNo (int)   :actorのindexNo"""
        selPatch = self.current_block[actorNo]
        selPatchNames = [selPatch]
        self.selectPatches(selPatchNames)

    #
    #  selectAddActor
    #-----------------
    def selectAddActor(self, actorNo):
        """ mouseのdoubleClickで選択したactorを選択表示に追加設定する。
        
        Args:
            actorNo (int)   :actorのindexNo"""
        selPatch = self.current_block[actorNo]
        selPatchNames = self.selected_block[:]
        selPatchNames.append(selPatch)
        self.selectPatches(selPatchNames)

    #
    #  selectPatches
    #----------------
    def selectPatches(self, selPatchNames):
        """
        指定したpatchを選択表示、それ以外を非選択表示に設定する。
        
        Args:
            selPatchNames (list(str))   :選択表示させるpatch名
        """
        for name in self.enabled_block:
            if name in selPatchNames:
                #選択部
                prop = self.actorDict[name].GetProperty()
                #prop.BackfaceCullingOn()
                #prop.FrontfaceCullingOn()
                #両面表示
                prop.BackfaceCullingOff()
                prop.FrontfaceCullingOff()
                prop.SetAmbient(0.5)
                prop.SetColor(colors.red)
                if self.showEdge == True:
                    prop.EdgeVisibilityOn()
                else:
                    prop.EdgeVisibilityOff()
                if self.setOpacity == True:
                    prop.SetOpacity(0.5)
                else:
                    prop.SetOpacity(1.0)
                prop.SetEdgeColor(colors.white)
                prop.SetRepresentationToSurface()
            else:
                #非選択部
                prop = self.actorDict[name].GetProperty()
                #prop.BackfaceCullingOn()
                #prop.FrontfaceCullingOn()
                #両面表示
                prop.BackfaceCullingOff()
                prop.FrontfaceCullingOff()
                prop.SetAmbient(0.3)
                prop.SetColor(colors.green)
                if self.showEdge == True:
                    prop.EdgeVisibilityOn()
                else:
                    prop.EdgeVisibilityOff()
                #prop.EdgeVisibilityOff()
                if self.setOpacity == True:
                    prop.SetOpacity(0.5)
                else:
                    prop.SetOpacity(1.0)
                prop.SetEdgeColor(colors.black)
                prop.SetRepresentationToSurface()
        self.selected_block = selPatchNames
        self.vtkWidget.selectedSignal.emit()

    #
    #  remakeRenderWindow
    #---------------------
    def remakeRenderWindow(self):
        """
        隠したpatchを表示し、元の形状を表示する。
        """
        #全actorを削除
        for actorName in self.current_block:
            self.ren.RemoveActor(self.actorDict[actorName])
        #全actorを表示
        for actorName in self.enabled_block:
            self.ren.AddActor(self.actorDict[actorName])
        self.current_block = self.enabled_block[:]
        # addPatches = list(set(self.enabled_block).difference(self.current_block))
        # for patch in addPatches:
        #     self.ren.AddActor(self.actorDict[patch])
        #     self.current_block.append(patch)

    #
    #  hidePatches
    #---------------
    def hidePatches(self, hidePatchNames):
        """ 指定したpatchを隠し、表示させない。
        """
        if len(self.current_block) <= 1:
            return
        for hideName in hidePatchNames:
            if hideName in self.current_block:
                self.ren.RemoveActor(self.actorDict[hideName])
                self.current_block.remove(hideName)

    #
    #  showPopupMenu
    #-----------------
    def showPopupMenu(self, pos):
        """ popupMenuを表示する。"""
        #menu = QtGui.QMenu()
        menu = QMenu()
        menu.addAction(_("edge表示/非表示の切替え"), self.onSwitchShowEdge)
        menu.addAction(_("透明/不透明の切替え"), self.onSwitchOpacity)
        menu.addAction(_("選択patchを隠す"), self.onHideSelectedPatches)
        menu.addAction(_("全patchを表示"), self.onShowAllPatches)
        menu.exec_(self.vtkWidget.mapToGlobal(pos))

    def onSwitchShowEdge(self):
        if self.showEdge == True:
            self.showEdge = False
        else:
            self.showEdge = True
        selectPatchNames = self.selected_block
        self.selectPatches(selectPatchNames)

    def onSwitchOpacity(self):
        if self.setOpacity == True:
            self.setOpacity = False
        else:
            self.setOpacity = True
        selectPatchNames = self.selected_block
        self.selectPatches(selectPatchNames)

    def onHideSelectedPatches(self):
        selectedPatches = self.selected_block
        self.hidePatches(selectedPatches)

    def onShowAllPatches(self):
        self.remakeRenderWindow()


#----------------------
#  stlPatchView class
#----------------------
class stlPatchView:
    """ stlのpatchをvtkを使って3D表示する。"""

    def __init__(self, vtkWidget, caseDir, stlDir, stlNames, blockMinMax, locMesh, flags):
        self.vtkWidget = vtkWidget
        self.caseDir = caseDir
        self.stlDir = stlDir
        self.stlNames = stlNames
        self.blockMinMax = blockMinMax
        self.locationInMesh = locMesh
        self.flags = flags
        self.actorDict = {}
        self.outlineActorDict = {}
        self.renderer = ""
        self.allStls = []
        self.hideStls = []
        self.showStls = []
        self.currStls = []        
        self.vtkInitialRun = True           #最初のvtk表示flag
        self.initialVtkArea()               #vtkAreaを作成

    #
    #  initialVtkArea
    #------------------
    def initialVtkArea(self):
        """ vtkの辞書を取得する"""
        #actorDictを作成する
        self.createActorDict()

        #rendererをチェック
        if self.renderer == "":
            self.renderer = vtk.vtkRenderer()
        else:
            self.renderer.NewInstance()
        self.assy = vtk.vtkAssembly()
        #stlを表示（選択しているstlを取得して）
        selNames = []
        flags = self.flags
        self.showSelectedItems(selNames, flags)

        #backgroundColorを設定
        self.renderer.SetBackground2(0.5, 0.5, 1)
        self.renderer.SetBackground(0.8, 0.8, 1)
        self.renderer.GradientBackgroundOn()
        self.renderer.SetTexturedBackground(True)
        self.renWin = self.vtkWidget.GetRenderWindow()
        self.renWin.NewInstance()
        #新しいrendererを追加
        self.renWin.AddRenderer(self.renderer)
        #styleを設定
        style = mouseSelectActor(self)
        style.SetDefaultRenderer(self.renderer)
        interactor = self.renWin.GetInteractor()
        interactor.SetInteractorStyle(style)
        #xyz軸を表示        
        self.addCorneAxis()
        #vtkを表示
        bounds = self.getCameraBounds()
        self.renderer.ResetCamera(*bounds)
        self.vtkWidget.Initialize()
        self.vtkWidget.Start()

    #
    #  createActorDict
    def createActorDict(self):
        """ actorを作成する"""
        self.allStls = {}
        self.outlineActorDict = {}
        #blockMeshのstlをcaseDir内に作成
        blockMeshStl = self.createBlockMeshStlFile()
        self.blockMeshName = os.path.basename(blockMeshStl)
        stlFiles = list(map(lambda x: self.stlDir + "/" + x, self.stlNames))
        stlFiles.append(blockMeshStl)
        #全stlを読み込み
        for fileName in stlFiles:
            stlName = os.path.basename(fileName)
            #reader
            reader = vtk.vtkSTLReader()
            reader.SetFileName(fileName)
            reader.Update()
            #filter
            geomFilter = vtk.vtkGeometryFilter()
            geomFilter.SetInputConnection(reader.GetOutputPort())
            #featureEdge
            featureEdge = vtk.vtkFeatureEdges()
            featureEdge.SetInputConnection(geomFilter.GetOutputPort())
            #mapper
            mapper = vtk.vtkPolyDataMapper()
            mapper.SetInputConnection(reader.GetOutputPort())
            mapper.ScalarVisibilityOff()
            edgeMapper = vtk.vtkCompositePolyDataMapper()
            edgeMapper.SetInputConnection(featureEdge.GetOutputPort())
            edgeMapper.ScalarVisibilityOff()
            #actor
            actor = vtk.vtkActor()
            actor.SetMapper(mapper)
            self.actorDict[stlName] = actor
            edgeActor = vtk.vtkActor()
            edgeActor.SetMapper(edgeMapper)
            self.outlineActorDict[stlName] = edgeActor
        #locMeshのactorを取得
        locMeshActor = self.getLocationInMeshActor()
        self.actorDict["locationInMesh"] = locMeshActor

    #
    #  createBlockMeshStlFile
    def createBlockMeshStlFile(self):
        """ blockMeshのstlファイルを作成する
        caseDir内にtemp.stlで作成する"""

        def convNum2str(num):
            ans = pyTreeFoam.float2strAuto(num, 6)
            return ans

        def createTet(tet, normal):
            """ 三角形1個分を作成"""
            lines  = [" facet normal " + normal + "\n"]
            lines += ["  outer loop\n"]
            p0 = list(map(convNum2str, tet[0]))
            p1 = list(map(convNum2str, tet[1]))
            p2 = list(map(convNum2str, tet[2]))
            lines += ["   vertex " + " ".join(p0) + "\n"]
            lines += ["   vertex " + " ".join(p1) + "\n"]
            lines += ["   vertex " + " ".join(p2) + "\n"]
            lines += ["  endloop\n"]
            lines += [" endfacet\n"]
            return lines

        def createTets(tet1, tet2, normal):
            lines = createTet(tet1, normal)
            lines += createTet(tet2, normal)
            return lines

        [x0, y0, z0], [x1, y1, z1] = self.blockMinMax
        p0 = [x0, y0, z0]
        p1 = [x1, y0, z0]
        p2 = [x1, y0, z1]
        p3 = [x0, y0, z1]
        p4 = [x0, y1, z0]
        p5 = [x1, y1, z0]
        p6 = [x1, y1, z1]
        p7 = [x0, y1, z1]
        #x-z面
        tet1 = [p0, p1, p2]
        tet2 = [p0, p2, p3]
        tet3 = [p4, p6, p5]
        tet4 = [p4, p7, p6]
        #y-z面
        tet5 = [p1, p5, p6]
        tet6 = [p1, p6, p2]
        tet7 = [p0, p7, p4]
        tet8 = [p0, p3, p7]
        #x-y面
        tet9 = [p0, p5, p1]
        tet10 = [p0, p4, p5]
        tet11 = [p3, p2, p6]
        tet12 = [p3, p6, p7]
        #stlデータ作成
        lines = []
        lines += ["solid blockMesh\n"]
        normal = "0 -1 0"
        lines += createTets(tet1, tet2, normal)
        normal = "0 1 0"
        lines += createTets(tet3, tet4, normal)
        normal = "1 0 0"
        lines += createTets(tet5, tet6, normal)
        normal = "-1 0 0"
        lines += createTets(tet7, tet8, normal)
        normal = "0 0 -1"
        lines += createTets(tet9, tet10, normal)
        normal = "0 0 1"
        lines += createTets(tet11, tet12, normal)
        lines += ["endsolid blockMesh\n"]
        #stlファイル保存
        userDir = os.getenv("TreeFoamUserPath")
        fileName = userDir + "/temp/blockMesh.stl"
        f = open(fileName, "w"); f.writelines(lines); f.close
        return fileName

    #
    #  getCameraBounds
    def getCameraBounds(self):
        """ 表示させるactorのboundsを取得する。
        原点、マージンも含めて取得する。"""
        bounds = self.getOriginBounds()
        #マージンを確保
        bounds = self.addMarginToBounds(bounds)
        return bounds
        # xmin = bounds[0]
        # xmax = bounds[1]
        # ymin = bounds[2]
        # ymax = bounds[3]
        # zmin = bounds[4]
        # zmax = bounds[5]
        # a = 0.1
        # xmin = xmin - (xmax-xmin) * a
        # xmax = xmax + (xmax-xmin) * a
        # ymin = ymin - (ymax-ymin) * a
        # ymax = ymax + (ymax-ymin) * a
        # zmin = zmin - (zmax-xmin) * a
        # zmax = zmax + (zmax-xmin) * a
        # return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  addMarginToBounds
    def addMarginToBounds(self, bounds):
        """ boundsにmargin0.1を加える"""
        xmin = bounds[0]
        xmax = bounds[1]
        ymin = bounds[2]
        ymax = bounds[3]
        zmin = bounds[4]
        zmax = bounds[5]
        a = 0.1
        xmin = xmin - (xmax-xmin) * a
        xmax = xmax + (xmax-xmin) * a
        ymin = ymin - (ymax-ymin) * a
        ymax = ymax + (ymax-ymin) * a
        zmin = zmin - (zmax-xmin) * a
        zmax = zmax + (zmax-xmin) * a
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getOriginBounds
    def getOriginBounds(self):
        """ 表示させるactorと原点のboundsを取得する。"""
        bounds = self.getBoundsActors()
        xmin = min([bounds[0], 0.0])
        xmax = max([bounds[1], 0.0])
        ymin = min([bounds[2], 0.0])
        ymax = max([bounds[3], 0.0])
        zmin = min([bounds[4], 0.0])
        zmax = max([bounds[5], 0.0])
        return (xmin, xmax, ymin, ymax, zmin, zmax)

    #
    #  getBoundsActors
    def getBoundsActors(self):
        """ 全actorのboundsを取得して返す。"""
        actors = self.renderer.GetActors()
        bounds = []
        for actor in actors:
            bound = actor.GetBounds()
            bounds.append(bound)
        if len(bounds) == 0:
            return ()
        Xmin = min(map(lambda x: x[0], bounds))
        Xmax = max(map(lambda x: x[1], bounds))
        Ymin = min(map(lambda x: x[2], bounds))
        Ymax = max(map(lambda x: x[3], bounds))
        Zmin = min(map(lambda x: x[4], bounds))
        Zmax = max(map(lambda x: x[5], bounds))
        return (Xmin, Xmax, Ymin, Ymax, Zmin, Zmax)

    #
    #  getLocationInMeshActor
    def getLocationInMeshActor(self):
        """ locationInMeshのactorを取得"""
        #source
        minVals, maxVals = self.blockMinMax
        length = [maxVals[0] - minVals[0], 
                  maxVals[1] - minVals[1],
                  maxVals[2] - minVals[2]]
        radius = sum(length) / 3.0 / 50.0
        location = list(map(float, self.locationInMesh))
        sphere = vtk.vtkSphereSource()
        sphere.SetRadius(radius)
        sphere.SetCenter(location)
        #mapper
        mapper = vtk.vtkPolyDataMapper()
        mapper.SetInputConnection(sphere.GetOutputPort())
        #actor
        actor = vtk.vtkActor()
        actor.SetMapper(mapper)
        prop = actor.GetProperty()
        prop.SetColor(colors.red)
        return actor

    #
    #  addCorneAxis
    #----------------
    def addCorneAxis(self):
        """ 最初の1回のみ、XYZの軸（compas）をrendererに追加する。"""
        if self.vtkInitialRun == True:
            #add axis(最初の起動時のみ)
            interactor = self.renWin.GetInteractor()
            axesActor = vtk.vtkAxesActor()
            self.axes = vtk.vtkOrientationMarkerWidget()
            self.axes.SetOrientationMarker(axesActor)
            self.axes.SetInteractor(interactor)
            self.axes.EnabledOn()
            self.axes.InteractiveOn()
            #flagをセット
            self.vtkInitialRun = False
        return

    #
    #  showSelectedItems
    #------------------
    def showSelectedItems(self, selNames, flags):
        """ 選択したitemを表示させる"""
        if self.renderer == "":
            return
        #flagsを展開
        self.flags = flags
        (edgeFlag, outlineFlag, locMeshFlag, outlineBlockFlag) = flags
        #選択item名を取得
        self.allStls = self.stlNames
        self.showStls = selNames
        self.hideStls = list(set(self.allStls) - set(self.showStls))
        parts = list(self.assy.GetParts())
        if len(parts) > 0:
            self.renderer.RemoveActor(self.assy)
            self.renderer.SetErase(True)
            self.assy = vtk.vtkAssembly()
        #actor再設定
        for showStl in self.showStls:
            if showStl in self.actorDict.keys():
                actor = self.actorDict[showStl]
                actor = self.setActorProperty(actor, edgeFlag)
                self.assy.AddPart(actor)
        #outlineを追加
        if outlineFlag == True:
            for stlName in self.allStls:
                actor = self.outlineActorDict[stlName]
                self.assy.AddPart(actor)
        #outlineBlockを追加
        if outlineBlockFlag == True:
            actor = self.outlineActorDict[self.blockMeshName]
            self.assy.AddPart(actor)
        #locMeshを追加
        if locMeshFlag == True:
            actor = self.actorDict["locationInMesh"]
            self.assy.AddPart(actor)
        self.renderer.AddActor(self.assy)
        #vtk表示
        self.vtkWidget.Initialize()
        return
    
    #
    #  setActorProperty
    def setActorProperty(self, actor, edgeFlag):
        """ actorのpropertyを設定（color、edge有無）"""
        prop = actor.GetProperty()
        #edgeの表示
        if edgeFlag == True:
            #edge表示
            prop.EdgeVisibilityOn()
        else:
            #edge非表示
            prop.EdgeVisibilityOff()
        #色の設定
        prop.SetAmbient(0.3)
        prop.SetColor(colors.grey)
        return actor

    #
    #  createShowSelectedItems
    #--------------------------
    def createShowSelectedItems(self, selNames, locMesh, blockMinMax, flags):
        """ actorを作り直し、再表示させる。"""
        self.locationInMesh = locMesh
        self.blockMinMax = blockMinMax
        self.flags = flags
        self.createActorDict()
        self.showSelectedItems(selNames, flags)


if __name__ == "__main__":
    import gettext
    gettext.install("app")
    #_ = gettext.gettext
