#!/usr/bin/python3
# coding: utf-8
#
#       GtkVtkRenderWindowInteractor.py
#
#   20/03/12    新規作成（originalを修正）
#   20/06/10    OnWheelEventを追加（wheelで拡縮）
#      07/15    OnConfigure:vtk.vtkRenderWindow.SetSize()を追加
#               vtk9の場合、windowSize変更時、vtk画面が変わらない為。
#      10/01    OnButtonDown:クリック時にcellPickerを追加
#               GtkVtkRenderWindowInteractor.__init__:引数にparent追加
#                 pickしたcellの処理を実行する為。
#      11/05    OnButtonDown:処理内容を修正
#   21/07/21    ConnectSignals:Gdk.EventMask.SCROLL_MASKを追加
#               OnWheelEvent:elseをelifに変更
#      07/25    OnKeyPress,OnKeyRelease:ctrlKeyの処理をparent側の
#               self.parent.ctrlKeyFlagに設定する事を追加
#      07/28    ConnectSignals:"key_release_event"を追加。
#               OnKeyReleaseが働いていなかった為。
#   25/02/05    parent_onMouseDraged:マウスをドラッグした時に
#               呼び出す親classの関数（onMouseDraged）を追加。
#      03/30    「import vtk」を「import gi」の前に移動。vtk-9.4対応
#      04/02    OnMouseMove:「self.canMove」flagを追加して、actorを
#               動かさずに固定するモードを追加。
#


import vtk

import gi
gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GLib, Gdk

#import vtk
import sys
import ctypes
import time

#-------------------------------------
#  GtkVtkRenderWindowInteractor class
#-------------------------------------
class GtkVtkRenderWindowInteractor(Gtk.DrawingArea):
    """ Gtk.DrawingAreaにvtkを表示させる。
    vtk.gtk.GtkGLExtVTKRenderWindowInteractorを参考に、Gtk、python3用に書き換え。"""

    def __init__(self, parent, *args):
        Gtk.DrawingArea.__init__(self)
        #self.set_double_buffered(False)
        self.parent = parent

        self._RenderWindow = vtk.vtkRenderWindow()

        # private attributes
        self.__Created = False  #0
        self._ActiveButton = 0

        self._Iren = vtk.vtkGenericRenderWindowInteractor()
        self._Iren.SetRenderWindow(self._RenderWindow)

        #self._Iren.GetInteractorStyle().SetCurrentStyleToTrackballCamera()
        inter = self._RenderWindow.GetInteractor()
        inter.SetInteractorStyle(vtk.vtkInteractorStyleTrackballCamera())

        self._Iren.AddObserver('CreateTimerEvent', self.CreateTimer)
        self._Iren.AddObserver('DestroyTimerEvent', self.DestroyTimer)
        self.ConnectSignals()

        # need this to be able to handle key_press events.
        #self.set_flags(gtk.CAN_FOCUS)
        self.set_can_focus(True)

        #ダブルクリック確認用の変数
        self.oldTime = time.time()
        
        #ドラッグ確認用の変数（button1をクリックした時間を保存）
        self.buttonDownTime = time.time()

        #rendererの回転を止めてfixする場合は、Falseを設定する
        self.canMove = True

    def parent_onMouseClicked(self):
        """ 親classを呼び出す。マウスをクリックした時"""
        try:
            self.parent.onMouseClicked()
        except:
            pass

    def parent_onMouseDraged(self):
        """ 親classを呼び出す。マウスをドラッグした時"""
        try:
            self.parent.onMouseDraged()
        except:
            pass

    def set_size_request(self, w, h):
        #gtk.gtkgl.DrawingArea.set_size_request(self, w, h)
        Gtk.DrawingArea.set_size_request(self, w, h)
        self._RenderWindow.SetSize(w, h)
        self._Iren.SetSize(w, h)
        self._Iren.ConfigureEvent()

    def ConnectSignals(self):
        self.connect("realize", self.OnRealize)
        #self.connect("expose_event", self.OnExpose)
        self.connect("draw", self.OnExpose)
        self.connect("configure_event", self.OnConfigure)       #windowSize変更時
        self.connect("button_press_event", self.OnButtonDown)
        self.connect("button_release_event", self.OnButtonUp)
        self.connect("motion_notify_event", self.OnMouseMove)
        self.connect("scroll_event", self.OnWheelEvent)         #追加(wheel回転)
        self.connect("enter_notify_event", self.OnEnter)
        self.connect("leave_notify_event", self.OnLeave)
        self.connect("key_press_event", self.OnKeyPress)
        self.connect("key_release_event", self.OnKeyRelease)    #追加
        self.connect("delete_event", self.OnDestroy)
        # self.add_events(gdk.EXPOSURE_MASK| gdk.BUTTON_PRESS_MASK |
        #                 gdk.BUTTON_RELEASE_MASK |
        #                 gdk.KEY_PRESS_MASK |
        #                 gdk.POINTER_MOTION_MASK |
        #                 gdk.POINTER_MOTION_HINT_MASK |
        #                 gdk.ENTER_NOTIFY_MASK | gdk.LEAVE_NOTIFY_MASK)
        self.add_events(Gdk.EventMask.EXPOSURE_MASK |
                        Gdk.EventMask.BUTTON_PRESS_MASK |
                        Gdk.EventMask.BUTTON_RELEASE_MASK |
                        Gdk.EventMask.KEY_PRESS_MASK |
                        Gdk.EventMask.POINTER_MOTION_MASK |
                        Gdk.EventMask.POINTER_MOTION_HINT_MASK |
                        Gdk.EventMask.ENTER_NOTIFY_MASK |
                        Gdk.EventMask.LEAVE_NOTIFY_MASK |
                        Gdk.EventMask.SCROLL_MASK)              #scrollを追加

    def __getattr__(self, attr):
        """Makes the object behave like a
        vtkGenericRenderWindowInteractor"""
        if attr == '__vtk__':
            return lambda t=self._Iren: t
        elif hasattr(self._Iren, attr):
            return getattr(self._Iren, attr)
        else:
            raise AttributeError(self.__class__.__name__ +
                  " has no attribute named " + attr)

    def CreateTimer(self, obj, event):
        GLib.timeout_add(10, self._Iren.TimerEvent)

    def DestroyTimer(self, obj, event):
        return True

    def GetRenderWindow(self):
        return self._RenderWindow

    def Render(self):
        if self.__Created:
            self._RenderWindow.Render()

    def OnRealize(self, *args):
        #if self.__Created == 0:
        if self.__Created == False:
            # you can't get the xid without the window being realized.
            self.realize()
            if sys.platform=='win32':
                #win_id = str(self.widgetTemp.window.handle)
                draw_window = self.widgetTemp.get_window()
                ctypes.pythonapi.PyCapsule_GetPointer.restype = ctypes.c_void_p
                ctypes.pythonapi.PyCapsule_GetPointer.argtypes = [ctypes.py_object]
                drawingarea_gpointer = ctypes.pythonapi.PyCapsule_GetPointer(draw_window.__gpointer__, None)
                gdkdll = ctypes.CDLL ("libgdk-3-0.dll")
                win_id = gdkdll.gdk_win32_window_get_handle(drawingarea_gpointer)
                win_id = str(win_id)
            else:
                #win_id = str(self.widget.window.xid)
                win_id = str(self.widgetTemp.get_window().get_xid())

            self.win_id = win_id
            self._RenderWindow.SetWindowInfo(win_id)
            #self._Iren.Initialize()
            #self.__Created = 1
            self.__Created = True
        #return gtk.TRUE
        return True

    def OnConfigure(self, widget, event):
        """ windowのサイズを変更した時"""
        self.widgetTemp = widget
        #以下の1行を追加
        vtk.vtkRenderWindow.SetSize(self._RenderWindow, event.width, event.height)
        self._Iren.SetSize(event.width, event.height)
        self._Iren.ConfigureEvent()
        self.Render()
        #return gtk.TRUE
        return True     

    #def OnExpose(self, *args):
    def OnExpose(self, widget, context):
        # #drawingSizeを取得
        # aw = widget.get_allocated_width()
        # ah = widget.get_allocated_height()
        # #黒で塗りつぶす
        # context.set_source_rgb(0, 0, 0)
        # context.rectangle(0, 0, aw, ah)
        # context.fill()
        self.Render()
        #return gtk.TRUE
        return True

    def OnDestroy(self, event=None):
        self.hide()
        del self._RenderWindow
        self.destroy()
        #return gtk.TRUE
        return True

    def _GetCtrlShift(self, event):
        ctrl = (event.state & Gdk.ModifierType.CONTROL_MASK)
        shift = (event.state & Gdk.ModifierType.SHIFT_MASK)
        # ctrl, shift = 0, 0
        # if ((event.state & Gdk.CONTROL_MASK) == Gdk.CONTROL_MASK):
        #     ctrl = 1
        # if ((event.state & Gdk.SHIFT_MASK) == Gdk.SHIFT_MASK):
        #     shift = 1
        return ctrl, shift

    def OnButtonDown(self, wid, event):
        """Mouse button pressed."""
        m = self.get_pointer()
        ctrl, shift = self._GetCtrlShift(event)
        self._Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift,
                                            chr(0), 0, None)
        button = event.button
        if button == 3:
            self._Iren.RightButtonPressEvent()
            #return gtk.TRUE
            return True
        elif button == 1:
            #doubleClickの確認
            currTime = time.time()
            #drag確認の為、clickした時のtimeを保存
            self.buttonDownTime = currTime
            if (currTime - self.oldTime) < 0.5:
                #doubleClick
                pass
            else:
                #singleClick
                self.oldTime = currTime
                self.parent_onMouseClicked()
            #defaultのevent
            self._Iren.LeftButtonPressEvent()
            #return gtk.TRUE
            return True
        elif button == 2:
            self._Iren.MiddleButtonPressEvent()
            #return gtk.TRUE
            return True
        else:
            #return gtk.FALSE
            return False

    def OnButtonUp(self, wid, event):
        """Mouse button released."""
        m = self.get_pointer()
        ctrl, shift = self._GetCtrlShift(event)
        self._Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift,
                                            chr(0), 0, None)
        button = event.button
        if button == 3:
            self._Iren.RightButtonReleaseEvent()
            #return gtk.TRUE
            return True
        elif button == 1:
            #pushしている時間を取得
            pushingTime = time.time() - self.buttonDownTime
            if pushingTime < 0.5:
                #通常のclick時
                pass
            else:
                #drag時（長押し時）
                self.parent_onMouseDraged()
            self._Iren.LeftButtonReleaseEvent()
            #return gtk.TRUE
            return True
        elif button == 2:
            self._Iren.MiddleButtonReleaseEvent()
            #return gtk.TRUE
            return True

        #return gtk.FALSE
        return False

    def OnMouseMove(self, wid, event):
        """Mouse has moved."""
        m = self.get_pointer()
        ctrl, shift = self._GetCtrlShift(event)
        self._Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift,
                                            chr(0), 0, None)
        if self.canMove == True:
            self._Iren.MouseMoveEvent()
        #return gtk.TRUE
        return True

    def OnWheelEvent(self, widget, event):
        """ mouse wheel event (拡縮）
        event.directionには、UP, DOWN 以外にSMOOTHも戻ってくる。"""
        if event.direction == Gdk.ScrollDirection.UP:
            self._Iren.MouseWheelForwardEvent()
        elif event.direction == Gdk.ScrollDirection.DOWN:
            self._Iren.MouseWheelBackwardEvent()

    def OnEnter(self, wid, event):
        """Entering the vtkRenderWindow."""
        self.grab_focus()
        m = self.get_pointer()
        ctrl, shift = self._GetCtrlShift(event)
        self._Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift,
                                            chr(0), 0, None)
        self._Iren.EnterEvent()
        #return gtk.TRUE
        return True

    def OnLeave(self, wid, event):
        """Leaving the vtkRenderWindow."""
        m = self.get_pointer()
        ctrl, shift = self._GetCtrlShift(event)
        self._Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift,
                                            chr(0), 0, None)
        self._Iren.LeaveEvent()
        #return gtk.TRUE
        return True

    def OnKeyPress(self, wid, event):
        """Key pressed."""
        m = self.get_pointer()
        ctrl, shift = self._GetCtrlShift(event)
        keycode, keysym = event.keyval, event.string
        key = chr(0)
        if keycode < 256:
            key = chr(keycode)
        self._Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift,
                                            key, 0, keysym)
        self._Iren.KeyPressEvent()
        self._Iren.CharEvent()
        #return gtk.TRUE
        #  ctrlKeyの場合は、parent側へ
        keyName = Gdk.keyval_name(event.keyval)
        if keyName == "Control_L" or keyName == "Control_R":
            self.parent.ctrlKeyFlag = 1
        return True

    def OnKeyRelease(self, wid, event):
        "Key released."
        m = self.get_pointer()
        ctrl, shift = self._GetCtrlShift(event)
        keycode, keysym = event.keyval, event.string
        key = chr(0)
        if keycode < 256:
            key = chr(keycode)
        self._Iren.SetEventInformationFlipY(m[0], m[1], ctrl, shift,
                                            key, 0, keysym)
        self._Iren.KeyReleaseEvent()
        #return gtk.TRUE
        #  ctrlKeyの場合はparentへ
        keyName = Gdk.keyval_name(event.keyval)
        if keyName == "Control_L" or keyName == "Control_R":
            self.parent.ctrlKeyFlag = 0
        return True

    def Initialize(self):
        if self.__Created:
            self._Iren.Initialize()

    def SetPicker(self, picker):
        self._Iren.SetPicker(picker)

    def GetPicker(self, picker):
        return self._Iren.GetPicker()


def main():
    # The main window
    #window = Gtk.Window(Gtk.WINDOW_TOPLEVEL)
    window = Gtk.Window()
    window.set_title("A GtkVTKRenderWindow Demo!")
    window.connect("destroy", Gtk.main_quit)
    window.connect("delete_event", Gtk.main_quit)
    window.set_border_width(10)

    # A VBox into which widgets are packed.
    vbox = Gtk.Box(spacing=3)
    window.add(vbox)
    vbox.show()

    # The GtkVTKRenderWindow
    gvtk = GtkVtkRenderWindowInteractor()
    vtkWinSize = (300, 300)
    gvtk.set_size_request(vtkWinSize[0]+1, vtkWinSize[1]+1)
    #vbox.pack_start(gvtk)
    vbox.pack_start(gvtk, True, True, 0)
    #gvtk.show()
    #gvtk.Initialize()
    #gvtk.Start()

    # prevents 'q' from exiting the app.
    gvtk.AddObserver("ExitEvent", lambda o,e,x=None: x)

    # The VTK stuff.
    cone = vtk.vtkConeSource()
    cone.SetResolution(80)
    coneMapper = vtk.vtkPolyDataMapper()
    coneMapper.SetInputConnection(cone.GetOutputPort())
    coneActor = vtk.vtkActor()
    coneActor.SetMapper(coneMapper)
    coneActor.GetProperty().SetColor(0.5, 0.5, 1.0)
    ren = vtk.vtkRenderer()
    gvtk.GetRenderWindow().AddRenderer(ren)
    ren.AddActor(coneActor)

    # A simple quit button
    quit = Gtk.Button.new_with_mnemonic("Quit!")
    quit.connect("clicked", Gtk.main_quit)
    vbox.pack_start(quit, True, True, 5)
    #quit.show()

    # show the main window and start event processing.
    window.show_all()
    #Gtk.mainloop()
    
    gvtk.Initialize()
    gvtk.Start()
    gvtk.set_size_request(*vtkWinSize)
    
    Gtk.main()


if __name__ == "__main__":
    main()

