#!/usr/bin/python3
# coding: utf-8
#
#   stringOp.py
#
#       文字列操作、検索のmodule
#
#   19/09/20    新規作成
#

import os, sys, glob

class strings:
    """ 文字列操作。文字列は、bytes型で検索操作する。結果もbytes型文字列が戻る。
    メソッドの引数（例:keyword等）は、str型で与えても、内部でbytes型に変換して処理する。
    対象の文字列が変わる操作（replace、delete関連）は、対象の文字列も変える。

    Attribute:
        line (bytes)    :対象の文字列（strで与えると内部でbytesに変換する）"""

    def __init__(self, line):
        #lineをbytesに変換
        if type(line) == str:
            line = line.encode()
        self.line = line

    #
    #  skipFoamFile
    #
    def skipFoamFile(self, p=0):
        """ FoamFileブロックをskipしたpointerを返す。"""
        (_line, p) = self.getKeywordContents(b"FoamFile", p)
        return p

    #
    #  replace1line
    #
    def replace1line(self, ps, oldLine, newLine):
        """ line[ps:]以降を検索し、のoldLineをnewLineに入れ替える。
        置き換え後は、元のself.lineの内容も置き換える。
        戻り値:[newCont, p, keyword]。
        失敗した場合、keywordが「""」になる。
        pointerは、更新される。"""
        if type(newLine) == str:
            newLine = newLine.encode()
        if type(oldLine) == str:
            oldLine = oldLine.encode()
        #置き換える
        newCont = self.line[:ps] + self.line[ps:].replace(oldLine, newLine, 1)
        #置き換え結果を確認
        p = newCont[ps:].find(newLine)
        if p >= 0:
            #置き換え成功
            p = ps + p + len(newLine)
            keyword = newLine
            self.line = newCont
        else:
            #置き換え失敗
            p = len(newCont)
            keyword = b""
        return (newCont, p, keyword)

    #
    #  replace1lineKeyword
    #
    def replace1lineKeyword(self, ps, newLine):
        """ line[ps:]を検索してnewLineに置き換える。検索は、newLineで使われているkeywordを検索する。
        置き換え後は、元のself.lineも置き換える。
        戻り値：(newCont, p, keyword)。
        失敗した場合、keywordが「""」になる。"""
        (keyword, _pp) = strings(newLine).getKeyword(0)
        (lineCont, p, _kind) = self.get1line(ps)
        while lineCont != b"":
            (tempKey, _pp) = strings(lineCont).getKeyword(0)
            if tempKey == keyword:
                oldLine = lineCont
                break
            ps = p
            (lineCont, p, _kind) = self.get1line(ps)
        if lineCont == b"":
            return (b"", -1, b"")
        else:
            (newCont, p, keyword) = self.replace1line(ps, oldLine, newLine)
            self.line = newCont
            return (newCont, p, keyword)

    #
    #  get1line
    #
    def get1line(self, p):
        """ 指定したpointerから1行を取得する
        line:「...;」、keyword:「keyword{}」、include:「# include "xxx.H"」、
        other:「# xxx yyy」を取得する。
        pointerがeofの場合は、lineCont=""が戻る。
        コメント文もそのまま取得する。
        戻り値:[lineCont, p, kind]"""
        lineCont = b""
        kind = b""
        if p >= len(self.line):
            return (lineCont, p, kind)
        
        chara, p = self.skipNoChara(p)      #pointerを空白以外に設定
        if p >= len(self.line) or chara == b"":
            return (lineCont, p, kind)
        
        if chara == b";":
            lineCont = b";"
            p += 1
        elif chara == b"#":
            #「#include ...」の取得
            sp = p
            p += 1                          #pointerを「#」の次に進める
            (chara, p) = self.skipNoChara(p)
            (name, p) = self.getKeyword(p)      #「include」取得
            (_keyword, p) = self.getKeyword(p)   #「"xxx.H"」取得
            lineCont = self.line[sp:p-1]
            if name == b"include" or name == b"includeEtc":
                kind = name
            else:
                kind = b"other"
        else:
            sp = p
            (_keyword, p) = self.getKeyword(p)
            (chara, p) = self.skipNoChara(p)
            if chara == b"{":
                #「keyword{}」の取得
                p -= 1                      #pointerを戻す
                (_cont, p) = self.getMiddlePair(p)
                lineCont = self.line[sp:p]
                kind = b"keyword"
            else:
                #「...;」を取得
                while chara != b";" and chara != b"":
                    if chara == b"(":
                        p -= 1              #pointerを戻す
                        (dummy, p) = self.getSmallPair(p)
                    elif chara == b"{":
                        p -= 1              #pointerを戻す
                        (dummy, p) = self.getMiddlePair(p)
                    (chara, p) = self.get1chara(p)
                if chara != b"" and p < len(self.line):
                    if self.line[p] == b";":
                        p += 1
                lineCont = self.line[sp:p]
                kind = b"line"
        return (lineCont, p, kind)

    #
    #  getKeywordPointer
    #
    def getKeywordPointer(self, keyword, p):
        """ 指定したkeywordのpointerを取得する。
        （高速検索している為、keywordがコメント中でも、検索するの注意！）
        pointerは、keywordの頭の文字位置。
        合致しない場合は、戻り値「-1」になる。"""
        if type(keyword) == str:
            keyword = keyword.encode()
        if p >= 0 and p < len(self.line):
            #高速検索のため、string.find()を使用してポインタを設定
            flag = 0
            while (p >= 0) and (flag == 0):
                n = self.line[p:].find(keyword)
                if n >= 0:
                    #keyword前後の文字をチェック
                    ps = p + n - 1
                    if ps < 0:
                        stChara = self.line[ps+1]
                    else:
                        stChara = self.line[ps]
                    pe = p + n + len(keyword)
                    if pe >= len(self.line):
                        edChara = b"\n"
                    else:
                        edChara = self.line[pe]
                    if ((self.checkChara(stChara) == False) and
                        (self.checkChara(edChara) == False)):
                        flag = 1
                        p = pe
                        break
                    else:
                        p = p + n + 1
                else:
                    break
            if flag == 0:
                return -1
            else:
                p = ps + 1
                return p
        return -1

    #
    #  getKeywordContents
    #
    def getKeywordContents(self, keyword, p):
        """ 指定したkeywordの内容を取得する。
        str.find()で検索するため、コメント内の文字列も検索する。
        {}内のデータを取得する。()内は取得しない。"""
        if type(keyword) == str:
            keyword = keyword.encode()
        if p >= 0 and p < len(self.line):
            #高速検索のため、str.find()で検索する
            flag = 0
            while p >= 0 and flag == 0:
                n = self.line[p:].find(keyword)
                if n >= 0:
                    #keyword前後の文字をチェック
                    ps = p + n - 1
                    if ps < 0:
                        #stChara = self.line[ps+1]
                        stChara = b"\n"
                    else:
                        stChara = self.line[ps]
                    pe = p + n + len(keyword)
                    if pe >= len(self.line):
                        edChara = b"\n"
                    else:
                        edChara = self.line[pe]
                    if (self.checkChara(stChara) == False and
                        self.checkChara(edChara) == False):
                        #ループ脱出
                        flag = 1
                        p = pe
                        break
                    else:
                        p = p + n + 1
                else:
                    break
            if flag == 0:
                contents = b""
                p = len(self.line) + 1
            else:
                #内容を取得
                (contents, p) = self.getMiddlePair(p)
        else:
            contents = b""
            p = len(self.line) + 1
        return (contents, p)

    #
    #  getBackKeyword
    #
    def getBackKeyword(self, p):
        """ 遡ってkeywordを取得"""
        #文字をbackする
        chara = self.line[p]
        while self.checkChara(chara) == True:
            p -= 1
            chara = self.line[p]
        #空白をbackSkipする
        while self.checkChara(chara) == False:
            p -= 1
            chara = self.line[p]
        #文字をbackする
        while self.checkChara(chara) == True:
            p -= 1
            chara = self.line[p]
        #keywordを取得
        (keyword, p) = self.getKeyword(p)
        return (keyword, p)

    #
    #  getNextKeywordAndContents
    #
    def getNextKeywordAndContents(self, p):
        """ 次のkeywordと{}の内容を取得する
        戻り値：((keyword, contents), p)"""
        flag = 0
        chara = b" "
        while flag == 0 and chara != b"":
            (keyword, p) = self.getKeyword(p)
            (chara, p) = self.skipNoChara(p)
            if chara == b"{":
                flag = 1
        p -= 1
        (contents, p) = self.getMiddlePair(p)
        return ((keyword, contents), p)

    #
    #  getMiddlePair
    #
    def getMiddlePair(self, p):
        """ {}内の文字列を取得し、その内容とpointerを返す。
        検索はstr.find()で検索する。"""
        pair = b""
        while pair != b"{" and p < len(self.line):
            #"{" or "}" を探す
            (pair, p) = self.getNextMiddlePair(p)
        #p += 1
        ap = p
        count  = 1
        while count > 0 and p < len(self.line):
            (pair, p) = self.getNextMiddlePair(p)
            if pair == b"{":
                count += 1
            elif pair == b"}":
                count -= 1
        contents = self.line[ap:p-1]
        return (contents, p)
    
    #  getNextMiddlePair
    #
    def getNextMiddlePair(self, p):
        """ pointerに近い"{" or "}" 文字を検索し、文字種とpointerを返す。
        検索はstr.finf()で検索する。"""
        lp = self.line[p:].find(b"{")
        if lp < 0:
            lp = len(self.line)
        rp = self.line[p:].find(b"}")
        if rp < 0:
            rp = len(self.line)
        if lp < rp:
            ans = b"{"
            ap = p + lp + 1
        elif rp < lp:
            ans = b"}"
            ap = p + rp + 1
        else:
            ans = b""
            ap = len(self.line)
        return (ans, ap)

    def getSmallPair(self, p):
        """ ()内データを取得し返す。"""
        pair = ""
        while pair != b"(" and p < len(self.line):
            #"(" or ")" を探す
            (pair, p) = self.getNextSmallPair(p)
        ap = p
        count  = 1
        while count > 0 and p < len(self.line):
            (pair, p) = self.getNextSmallPair(p)
            if pair == b"(":
                count += 1
            elif pair == b")":
                count -= 1
        contents = self.line[ap:p-1]
        return (contents, p)

    #  getNextSmallPair
    #
    def getNextSmallPair(self, p):
        """ pointerに近い"(" or ")" 文字を検索し、文字種とpointerを返す。
        検索はstr.finf()で検索する。"""
        lp = self.line[p:].find(b"(")
        if lp < 0:
            lp = len(self.line)
        rp = self.line[p:].find(b")")
        if rp < 0:
            rp = len(self.line)
        if lp < rp:
            ans = b"("
            ap = p + lp + 1
        elif rp < lp:
            ans = b")"
            ap = p + rp + 1
        else:
            ans = b""
            ap = len(self.line)
        return (ans, ap)

    #
    #  getKeyword
    #
    def getKeyword(self, p):
        """ keywordを取得して返す"""
        (chara, p) = self.get1chara(p)
        #文字まで進む
        while (self.checkChara(chara) != True and chara != b""):
            (chara, p) = self.get1chara(p)
        #keyword取得
        keyword = b""
        if chara == b'"':
            #""で囲まれる場合
            keyword += chara
            (chara, p) = self.get1chara(p)
            while (chara != b'"') and (chara != b""):
                keyword += chara
                (chara, p) = self.get1chara(p)
            #最後の"をkeywordに追加しポインタを1ヶ進める
            keyword += chara
            p += 1
        else:
            #通常の文字列
            while (self.checkChara(chara) == True) and (chara != b""):
                keyword += chara
                (chara, p) = self.get1chara(p)
            if chara != b"":
                p -= 1      #1文字読みすぎているので、ポインタを1ヶ戻す
        return (keyword, p)

    #
    #  get1chara
    #
    def get1chara(self, p):
        """ 1文字を取得"""
        if p >= len(self.line):
            return (b"", p)

        (chara, p) = self.get1c(p)
        if chara == b"/":
            ps = p
            (nextChara, p) = self.get1c(p)
            if nextChara == b"/":
                p = self.skipUntilCR(p)      #行末までスキップ
                (chara, p) = self.get1c(p)
            elif nextChara == b"*":
                p= self.skipUntilSS(p)       #"*/"までスキップ
                (chara, p) = self.get1c(p)
            else:
                p = ps                       #pointerを戻す
                #p-=1                            #1文字戻す
        return (chara, p)

    #行末までスキップ
    def skipUntilCR(self, p):
        """ 行末までskipする"""
        if p>=0 and p<len(self.line):
            (chara, p) = self.get1c(p)
            while (chara != b"\n") and (chara != b""):
                [chara, p] = self.get1c(p)
            if chara == b"\n":
                p-=1                            #1文字戻す
        else:
            p=len(self.line)
        return p

    #*/までスキップ
    def skipUntilSS(self, p):
        """ 「*/」までskipする"""
        if p >= 0 and p < len(self.line)-1:
            charas = self.line[p:p+2]; p+=1
            while (charas != b"*/") and (p < len(self.line)-1):
                charas = self.line[p:p+2]; p+=1
            if charas == b"*/":
                p+=1
            else:
                p=len(self.line)
        else:
            p=len(self.line)
        return p

    #spスキップ
    def skipSP(self, p):
        """ 空白をskipする"""
        if p >= 0 and p < len(self.line)-1:
            while self.line[p] == ord(" "):
                p += 1
        else:
            p=len(self.line)
        return p

    #1文字取得
    def get1c(self, p):
        """ 1文字取得"""
        #if p >= 0 and p < len(self.line)-1:
        if p >= 0 and p < len(self.line):
            chara = self.line[p]; p += 1
            if type(chara) == int:
                chara = chr(chara).encode()
        else:
            chara = b""; p = len(self.line)
        return (chara, p)

    #
    #  skipNoChara
    #
    def skipNoChara(self, p):
        """ 空白以外の文字までskipする
        ﾎﾟｲﾝﾀは空白以外の文字位置を返す。charaはﾎﾟｲﾝﾀが示す文字を返す"""
        if p >= len(self.line):
            return (b"", p)
        (chara, p) = self.get1chara(p)
        while (chara == b" " or chara == b"\n" or chara == b"\t"):
            (chara, p) = self.get1chara(p)
        p -= 1
        return (chara, p)

    #
    #  checkChara
    #
    def checkChara(self, chara):
        """ keywordとして使える文字のチェック"""
        if type(chara) == int:
            chara = chr(chara).encode()
        if chara > b" " and chara <= b"z":
            if chara == b"{" or chara == b"}":
                flag = False
            elif chara == b"(" or chara == b")":
                flag = False
            elif chara == b";":
                flag = False
            else:
                flag = True
        else:
            flag = False
        return flag

    #
    #  skipIntoPair
    #
    def skipIntoPair(self, p, pair=b"{"):
        """ 現在のpointerからpair文字（default:"{"）までskipしてpointerを返す """
        if p >= 0 and p < len(self.line)-1:
            #asciiコードで比較
            while (self.line[p] != ord(pair)) and (p < len(self.line)-1):
                p += 1
            p += 1
        else:
            p=len(self.line)
        return p

    #
    #  deleteCommentsNullLines
    #
    def deleteCommentsNullLines(self):
        """ コメント、空行の削除。削除後は、元のself.lineも置き換わる"""
        contents = self.deleteComments()
        contents = self.deleteNullLines()
        return contents

    #
    #  deleteComments
    #
    def deleteComments(self):
        """ commentを削除して返す。削除後は、元のself.lineも置き換わる"""
        #/*を削除
        nocomm = b""
        st = 0
        flag = 0
        while flag == 0:
            pst = self.line[st:].find(b"/*")
            if pst == -1:
                #comm無し、終了
                flag = 1
            else:
                #/*をskip
                commS = st + pst
                commE = self.skipComm(b"*/", commS)
            if flag == 0:
                nocomm += self.line[st:commS]
                st = commE + 1
            else:
                nocomm += self.line[st:]
        #//を削除
        contOp = strings(nocomm)
        nocomm = b""
        st = 0
        flag = 0
        while flag == 0:
            pst = contOp.line[st:].find(b"//")
            if pst == -1:
                flag = 1
            else:
                #//をskip
                commS = st + pst
                commE = contOp.skipComm(b"\n", commS)
            if flag == 0:
                nocomm += contOp.line[st:commS]
                st = commE + 1
            else:
                nocomm += contOp.line[st:]
        self.line = nocomm
        return nocomm

    #
    #  skipComm
    #
    def skipComm(self, echara, st):
        """ commentをskip"""
        p = self.line[st:].find(echara)
        if p == -1:
            p = len(self.line[st:]) + st
        else:
            p = p + len(echara) + st
            if echara == b"\n":
                p -= 2
        return p

    #
    #  deleteNullLines
    #
    def deleteNullLines(self):
        """ 空行を削除する。削除後は、元のself.lineも置き換わる"""
        lines = self.line.split(b"\n")
        newLines = []
        for line in lines:
            if len(line.split()) != 0:
                newLines.append(line)
        newCont = b"\n".join(newLines) + b"\n"
        self.line = newCont
        return newCont

    #
    #  compressSPCR
    #
    def compressSPCR(self):
        """ SPとCRを削除する。削除後は、元のself.lineも置き換える"""
        lines = self.line.split(b"\n")
        ans = []
        for line in lines:
            if line != b"":
                words = line.split()
                ans.append(b" ".join(words))
        ansR = b"\n".join(ans)
        if len(ansR) > 0:
            if ansR[-1] == b"\n":
                ansR = ansR[:-1]
        else:
            ansR = b"\n"
        self.line = ansR        
        return ansR



if __name__ == "__main__":
    f = open("boundary"); conts = f.read(); f.close()
    lineOp = strings(conts)
    p = lineOp.skipFoamFile(0)
    p = lineOp.getKeywordPointer("inlet", p)
