#!/usr/bin/python3
#  coding: utf-8
#
#           pyCoupling.py
#
#       fsi, cht, chtssに応じたcnt, mshファイルを作成する
#
#   23/04/21    新規作成
#      05/28    couplingData.write:修正。
#      10/11    getCouplingScriptDict:OFversionに応じたsourceの辞書を返す
#               codedをpatch内からcontrolDictに変更。
#   24/01/26    getCouplingScriptDict:OF-11用の辞書を追加
#      04/13    couplingData.write:バグ修正。
#               項目内容が「""」の場合、書き換わらなかった。
#      07/18    getCouplingScriptDict:OF-12用で辞書定義を追加。
#   25/07/14    getCouplingScriptDict:OFversionチェック方法を変更。
#

import os
import shutil
import time
import mmap
import glob
import pyFistr_ptc as pyFistr



waitCount = 5000
waitTime = 0.01
commonSize = 10000

#-----------------
#  stream class
#-----------------
class stream:
    """ 双方向通信"""
    def __init__(self, mmComm, commonSize, waitCount, waitTime):
        self.mmComm = mmComm            #mmapFile
        self.commonSize = commonSize    #bufferサイズ
        self.waitCount = waitCount      #count数
        self.waitTime = waitTime        #count当たりの待ち時間

    #
    #  put
    #--------
    def put(self, title, text):
        """ データ送信"""
        size = self.commonSize
        mmCommFile = self.mmComm
        myPid = os.getpid()
        header = "PID:" + str(myPid) + " title:" + title
        cont = header + "\01" + text + "\04"
        cont = cont.encode()
        dataList = []
        for i in range(0, len(cont), size):
            data = cont[i:i+size]
            dataList.append(data)
        #textをstreamにセットする
        for i in range(len(dataList)):
            if i == 0:
                self.waitUntilBufferInitialSet()
            else:
                self.waitUntilBufferEmpty()
            data = dataList[i]
            mmCommFile.seek(0)
            mmCommFile.write(data)
            mmCommFile.flush()
        mmCommFile.seek(commonSize-1)
        mmCommFile.write(b"\04")        #最後に書き込み
        mmCommFile.flush()
        self.waitUntilBufferEmpty()     #相手が読み込むまで待つ

    #
    #  waitUntilBufferInitialSet
    def waitUntilBufferInitialSet(self):
        """ bufferがemptyの場合、直ぐにPIDをセットし、bufferを使用中にする"""
        count = self.waitCount
        sleepTime = self.waitTime
        mmCommFile = self.mmComm
        bPid = str(os.getpid()).encode()
        while True:
            mmCommFile.flush()
            if mmCommFile[:2] == b"\00\00" and mmCommFile[-1:] == b"\00":
                mmCommFile.seek(0)
                mmCommFile.write(bPid)
                mmCommFile.flush()
                mmCommFile.seek(0)
                break
            count -= -1
            if count < 0:
                break
            time.sleep(sleepTime)
        return

    #
    #  waitUntilBufferEmpty
    def waitUntilBufferEmpty(self):
        """ 送信可能になるまで待つ。bufferが空になるまで"""
        count = self.waitCount
        sleepTime = self.waitTime
        mmCommFile = self.mmComm
        while True:
            if mmCommFile[:2] == b"\00\00" and mmCommFile[-1:] == b"\00":
                break
            count -= -1
            if count < 0:
                break
            time.sleep(sleepTime)
        return

    #
    #  waitUntilBufferSet
    def waitUntilBufferSet(self):
        """ streamに文字がセットされるまで待つ"""
        count = self.waitCount
        sleepTime = self.waitTime
        mmCommFile = self.mmComm
        while True:
            mmCommFile.flush()
            if mmCommFile[:2] != b"\00\00" and mmCommFile[-1:] != b"\00":
                break
            count -= 1
            if count < 0:
                break
            time.sleep(sleepTime)
        return  

    #
    #  get
    #-------
    def get(self, title):
        """ データ受信"""
        mmCommFile = self.mmComm
        commonSize = self.commonSize
        #dataがセットされるまで待つ
        self.waitUntilBufferSet()
        #buffer読み込み
        mmCommFile.seek(0)
        conts = mmCommFile.read().decode()
        #bufferクリア
        mmCommFile.seek(0)
        mmCommFile.write(b"\00\00")
        mmCommFile.seek(commonSize-1)
        mmCommFile.write(b"\00")
        mmCommFile.flush()
        #streamのheaderを取得
        ns = conts.find("\01")
        header = conts[:ns]
        #header内のtitleを取得
        nt = header.find("title:")
        title = header[nt + len("title:"):]
        #streamのtextを取得
        text = ""
        cont = conts[ns+1:]
        ne = cont.find("\04")
        if ne >= 0:
            text = cont[:ne]
            return title, text
        text += cont
        while True:
            self.waitUntilBufferSet()
            #読み込み
            mmCommFile.seek(0)
            cont = mmCommFile.read().decode()
            #bufferクリア
            mmCommFile.seek(0)
            mmCommFile.write(b"\00\00")
            mmCommFile.seek(commonSize-1)
            mmCommFile.write(b"\00")
            mmCommFile.flush()
            #読み込み内容を処理
            ne = cont.find("\04")
            if ne >= 0:
                text += cont[:ne]
                break
            text += cont
        return title, text

    #
    #  getLines
    #------------
    def getLines(self, title):
        """ 受信データをlinesで取得"""
        _title, text = self.get(title)
        if text[-1] == "\n":
            text = text[:-1]
        lines = text.split("\n")
        lines = list(map(lambda x: x+"\n", lines))
        return lines

    #
    #  getLinesAtSteps
    #-------------------
    def getLinesAtSteps(self, title):
        """ 受信データをlinesで取得し、title内のstep数も取得して返す"""
        title, text = self.get(title)
        if text[-1] == "\n":
            text = text[:-1]
        lines = text.split("\n")
        lines = list(map(lambda x: x+"\n", lines))
        #stepsを取得(titleの最後の数字がstep)
        steps = title.split()[-1]
        try:
            steps = int(steps)
        except:
            steps = -1
        return lines, steps



#--------------------
#  cntMshFiles class
#--------------------
class cntMshFiles:
    """ cnt, msh, datファイルをチェックし、無ければ作成する。"""

    def __init__(self, solidCaseDir):
        self.solidCaseDir = solidCaseDir

    #
    #  checkCntMshFiles
    def checkCntMshFiles(self):
        """ 標準のcnt、mshファイルかどうかチェックし、
        標準でない場合、標準に戻す。"""
        changeDict = {}
        nameDict = pyFistr.getCntMshFileNameDict(self.solidCaseDir)
        if nameDict["cnt"] != "FistrModel.cnt":
            orgName = self.solidCaseDir + os.sep + nameDict["cnt"]
            newName = self.solidCaseDir + os.sep + "FistrModel.cnt"
            os.rename(orgName, newName)
            changeDict["cnt"] = "FistrModel.cnt"
        if nameDict["msh"] != "FistrModel.msh":
            orgName = self.solidCaseDir + os.sep + nameDict["msh"]
            newName = self.solidCaseDir + os.sep + "FistrModel.msh"
            os.rename(orgName, newName)
            changeDict["msh"] = "FistrModel.msh"
        if len(changeDict) > 0:
            pyFistr.setCntMshFileNameDict(self.solidCaseDir, changeDict)

    #
    #  checkFsiFiles
    def checkFsiFiles(self):
        """ hecmw_ctrl_fsi.datが無い場合、作成する。
        既に存在すれば、何もしない"""
        solidCaseDir = self.solidCaseDir
        datFsiFile = solidCaseDir + os.sep + "hecmw_ctrl_fsi.dat"
        cntFsiFile =  solidCaseDir + os.sep + "FistrModel_fsi.cnt"
        if os.path.exists(datFsiFile) and os.path.exists(cntFsiFile):
            return
        #fsi専用のcntを作成
        FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(solidCaseDir)
        copyFile = solidCaseDir + os.sep + FistrModel_cnt
        pasteFile = solidCaseDir + os.sep + "FistrModel_fsi.cnt"
        if os.path.exists(pasteFile) == False:
            shutil.copy(copyFile, pasteFile)
        #fsi専用のdatを作成
        nameDict = {
            "msh":      "FistrModel.msh",
            "cnt":      "FistrModel_fsi.cnt",
            "res":      "FistrModel.res",
            "vis":      "FistrModel.vis",
            "restart":  "FistrModel.restart"
            }
        pyFistr.setCntMshFileNameDict(solidCaseDir, nameDict)
        copyFile = solidCaseDir + os.sep + "hecmw_ctrl.dat"
        if os.path.exists(datFsiFile) == False:
            shutil.copy(copyFile, datFsiFile)

    #
    #  checkChtFiles
    def checkChtFiles(self):
        """ hecmw_ctrl_cht.datが無い場合、作成する。
        既に存在すれば、何もしない"""
        solidCaseDir = self.solidCaseDir
        datChtFile = solidCaseDir + os.sep + "hecmw_ctrl_cht.dat"
        cntChtFile = solidCaseDir + os.sep + "FistrModel_cht.cnt" 
        if os.path.exists(datChtFile) and os.path.exists(cntChtFile):
            return
        #cht専用のcntを作成
        FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(solidCaseDir)
        copyFile = solidCaseDir + os.sep + FistrModel_cnt
        pasteFile = solidCaseDir + os.sep + "FistrModel_cht.cnt"
        if os.path.exists(pasteFile) == False:
            shutil.copy(copyFile, pasteFile)
        #cht専用のmshを作成
        copyFile = solidCaseDir + os.sep + FistrModel_msh
        pasteFile = solidCaseDir + os.sep + "FistrModel_cht.msh"
        if os.path.exists(pasteFile) == False:
            shutil.copy(copyFile, pasteFile)
        #cht専用のdatを作成
        nameDict = {
            "msh":      "FistrModel_cht.msh",
            "cnt":      "FistrModel_cht.cnt",
            "res":      "FistrModel_temp.res",
            "vis":      "FistrModel_cht.vis",
            "restart":  "FistrModel_cht.restart"
            }
        pyFistr.setCntMshFileNameDict(solidCaseDir, nameDict)
        copyFile = solidCaseDir + os.sep + "hecmw_ctrl.dat"
        if os.path.exists(datChtFile) == False:
            shutil.copy(copyFile, datChtFile)

    #
    #  checkChtssFiles
    def checkChtssFiles(self):
        """ hecmw_ctrl_chtss.datが無い場合、作成する。
        既に存在すれば、何もしない"""
        solidCaseDir = self.solidCaseDir
        datChtssFile = solidCaseDir + os.sep + "hecmw_ctrl_chtss.dat"
        cntChtssFile = solidCaseDir + os.sep + "FistrModel_chtss.cnt"
        if os.path.exists(datChtssFile) and os.path.exists(cntChtssFile):
            return
        #chtss専用のcntを作成
        FistrModel_cnt, FistrModel_msh = pyFistr.cntMshFileName(solidCaseDir)
        copyFile = solidCaseDir + os.sep + FistrModel_cnt
        pasteFile = solidCaseDir + os.sep + "FistrModel_chtss.cnt"
        if os.path.exists(pasteFile) == False:
            shutil.copy(copyFile, pasteFile)
        #chtss専用のdatを作成
        nameDict = {
            "msh":      "FistrModel.msh",
            "cnt":      "FistrModel_chtss.cnt",
            "res":      "FistrModel.res",
            "vis":      "FistrModel_chtss.vis",
            "restart":  "FistrModel_chtss.restart"
            }
        pyFistr.setCntMshFileNameDict(solidCaseDir, nameDict)
        copyFile = solidCaseDir + os.sep + "hecmw_ctrl.dat"
        if os.path.exists(datChtssFile) == False:
            shutil.copy(copyFile, datChtssFile)

#----------------------
#  couplingData class
#----------------------
class couplingData:
    """ couplingDataのread、write"""

    def __init__(self, fluidCaseDir):
        self.fluidCaseDir = fluidCaseDir
        self.dataFile = fluidCaseDir + "/coupling_FrontISTR/couplingData"

    #
    #  readMmapLines
    def readMmapLines(self):
        """ fileObjectの共有メモリからlineを読込、返す。"""
        f = open(self.dataFile, "r+b")
        mm = mmap.mmap(f.fileno(), 0)
        f.close()
        mm.seek(0)
        conts = mm.read().decode()
        lines = conts.split("\n")
        mm.close()
        return lines

    #
    #  read
    #---------
    def read(self):
        """ couplingDataを辞書形式で読み込み、返す。"""
        #値の初期化
        dataDict = {
            "field": "",                #OpenFOAM関連
            "pointField": "",
            "mappingPatch": "",
            "mappingCoeff": "1.0",
            "mappingShift": "0.0",
            "nStepsOF": "1",
            "nProcs": "",
            "startTime": "",
            "deltaT": "",
            "writeInterval": "",
            "endTime": "",
            "timeName": "",
            "beforeTime": "",
            "timeIndex": "",

            "solidAnaComm": "fistr1",   #FrontISTR関連
            "solidCaseDir": "",
            "mappingSgrp": "",
            "nSteps": "1",
            "nThreads": "1",

            #"streamBuffer": "10000",         #その他
            "maxWaitTime": "6000",
            "fsi": "",
            "cht": "",
            "chtss": "",
            "method": "conc",
            "solidAnaItem": "fsi"
            }
        #共有メモリから読み込む
        lines = self.readMmapLines()
        for line in lines:
            words = line.split()
            if len(words) >= 2:
                if words[0][0] != "#":
                    dataDict[words[0]] = " ".join(words[1:])
        return dataDict
    
    #
    #  write
    #---------
    def write(self, dataDict):
        """ couplingDataを書き込む"""
        fileName = self.dataFile
        f = open(fileName); lines = f.readlines(); f.close()
        addLines = []
        for key in dataDict.keys():
            cont = dataDict[key]
            flag = 0
            for i in range(len(lines)):
                line = lines[i]
                words = line.split()
                if len(words) >= 1:
                    if words[0] == key:
                        lines[i] = ("%-15s" % words[0] + " ") + cont + "\n"
                        flag = 1
                        break
            if flag == 0:
                word = ("%-15s" % key + " ")
                line = word + cont + "\n"
                addLines.append(line)
        lines += addLines
        f = open(fileName, "w"); f.writelines(lines); f.close()

#
#  time2str
#-------------
def time2str(time):
    """ time(float)を決められた長さの文字列に変換する"""
    strTime = "%.6f" % time
    return strTime

#
#  isWriteInterval
#------------------
def isWriteInterval(currTime, writeInterval, deltaT):
    """ currTimeがwriteIntervalのタイミングかどうかチェックする"""
    dt = float(deltaT)
    curr = float(currTime)
    wInt = float(writeInterval)
    diffR = curr / wInt - int(curr/ wInt + 0.5)
    diff = diffR * wInt
    if abs(diff) < dt/1000.0:
        return True
    else:
        return False
    
#
#  createTimeCont
#--------------------
def createTimeCont(beforeTime, currTime, timeIndex, c1, c2, c3):
    """ 共有メモリmmTimesの内容を作成して返す"""
    lines =  ["beforeTime   " + beforeTime + "\n"]
    lines += ["currTime     " + currTime + "\n"]
    lines += ["couplePress  " + c1 + "\n"]
    lines += ["coupleTemp   " + c2 + "\n"]
    lines += ["coupleOther  " + c3 + "\n"]
    lines += ["timeIndex    " + timeIndex + "\n"]
    lines += ["#end\n"]
    cont = "".join(lines)
    return cont

#
#  writeTimeCont
#----------------
def writeTimeCont(mmTimes, timesDict):
    """ 共有メモリに辞書形式の内容を書き込む"""
    lines = []
    for key in timesDict.keys():
        line = ("%-13s" % key) + timesDict[key] + "\n"
        lines.append(line)
    lines += ["#end\n"]
    mmTimes.seek(0)
    newCont = "".join(lines)
    mmTimes.write(newCont.encode())
    mmTimes.flush()

#
#  getTimeCont
#---------------
def getTimeCont(mmTimes):
    """ 共有メモリmmTimesの内容を取得し、辞書形式で返す"""
    mmTimes.seek(0)
    timesDict = {}
    conts = mmTimes.read().decode()
    lines = conts.split("\n")
    for line in lines:
        words = line.split()
        if len(words) > 1:
            timesDict[words[0]] = words[1]
    return timesDict

#
#  getEndStepFromCouplingWriteInterval
#--------------------------------------
def getEndStepFromCouplingWriteInterval(startStep, currTime, deltaT, nSteps, writeInterval, orgStep):
    """ 次の計算のendStepを取得する
    endStepは、writeInterval時と(endStep % nSteps == 0)の時。
    startStepとcurrTimeは、同じ時点の値を設定する。"""
    nextEndStep = startStep + nSteps
    corr = nextEndStep % nSteps
    if corr != 0:
        nextEndStep = nextEndStep - corr
        if startStep < nextEndStep - nSteps:
            nextEndStep -= nSteps
    nextEndTime = currTime + nSteps * deltaT
    nextWriteTime = int((currTime + deltaT) / writeInterval + 0.5) * writeInterval
    if int(nextWriteTime / deltaT + 0.5) <= int(currTime / deltaT + 0.5):
        nextWriteTime += writeInterval
    if nextEndTime <= nextWriteTime:
        #通常のcouplingIntervalに設定
        endStep = nextEndStep
        endTime = nextEndTime
    else:
        #writeIntervalに設定
        endStep = startStep + int((nextWriteTime - currTime) / deltaT + 0.5)
        endTime = nextWriteTime
    return endStep, endTime

#
#  getBeforeEndStepFromCoupling
#-------------------------------
def getBeforeEndStepFromCoupling(startStep, currTime, deltaT, nSteps, writeInterval):
    """ 前の計算のendStepを取得する。
    startStepとcurrTimeは、同じ時点の値を設定する。"""
    beforeTime = currTime - deltaT * nSteps
    beforeStep = startStep - nSteps
    nEndStep, nEndTime = getEndStepFromCouplingWriteInterval(
        beforeStep, beforeTime, deltaT, nSteps, writeInterval, 0)
    if nEndStep == startStep:
        bEndStep = beforeStep
    else:
        bEndStep = nEndStep
    if time2str(nEndTime) == time2str(currTime):
        bEndTime = beforeTime
    else:
        bEndTime = nEndTime
    return bEndStep, bEndTime

#
#  getCouplingScriptFile
#------------------------
def getCouplingScriptDict(OFversion):
    """ 解析内容に応じたcontFile名の辞書を取得する。"""
    contFileDict = {
            #OF全般
            #fsi    cht    chtss  method        scalarFieldの処理                   pointFieldの処理
            ("yes", "yes", "yes", "conc"):  ["cont_couplingSource_fsiChtChtss", "contRun_couplingSource_fsiChtChtss.py"],
            ("yes", "yes", "no",  "conc"):  ["cont_couplingSource_fsiCht", "contRun_couplingSource_fsiCht.py"],
            ("yes", "no",  "yes", "conc"):  ["", ""],
            ("yes", "no",  "no",  "conc"):  ["cont_couplingSource_fsi", "contRun_couplingSource_fsi.py"],
            ("no",  "yes", "yes", "conc"):  ["cont_couplingSource_chtChtss", "contRun_couplingSource_chtChtss.py"],
            ("no",  "yes", "no",  "conc"):  ["cont_couplingSource_cht", "contRun_couplingSource_cht.py"],
            ("no",  "no",  "yes", "conc"):  ["", ""],
            ("no",  "no",  "no",  "conc"):  ["", ""],
            ("yes", "yes", "yes", "altn"):  ["cont_altnSource_fsiChtChtss", "contRun_altnSource_fsiChtChtss.py"],
            ("yes", "yes", "no",  "altn"):  ["cont_altnSource_fsiCht", "contRun_altnSource_fsiCht.py"],
            ("yes", "no",  "yes", "altn"):  ["", ""],
            ("yes", "no",  "no",  "altn"):  ["cont_altnSource_fsi", "contRun_altnSource_fsi.py"],
            ("no",  "yes", "yes", "altn"):  ["cont_altnSource_chtChtss", "contRun_altnSource_chtChtss.py"],
            ("no",  "yes", "no",  "altn"):  ["cont_altnSource_cht", "contRun_altnSource_cht.py"],
            ("no",  "no",  "yes", "altn"):  ["", ""],
            ("no",  "no",  "no",  "altn"):  ["", ""]
    }

    #OFversionを確認し、Dictを置き換え
    numVer = getNumVersion(OFversion)
    if numVer >= 12.0:
        contFileDict = {
            #OF-12以上
            #fsi    cht    chtss   method       scalarFieldの処理                   pointFieldの処理
            ("yes", "yes", "yes", "conc"):  ["cont_couplingSource_fsiChtChtss_OF12", "contRun_couplingSource_fsiChtChtss.py"],
            ("yes", "yes", "no",  "conc"):  ["cont_couplingSource_fsiCht_OF12", "contRun_couplingSource_fsiCht.py"],
            ("yes", "no",  "yes", "conc"):  ["", ""],
            ("yes", "no",  "no",  "conc"):  ["cont_couplingSource_fsi_OF12", "contRun_couplingSource_fsi.py"],
            ("no",  "yes", "yes", "conc"):  ["cont_couplingSource_chtChtss_OF12", "contRun_couplingSource_chtChtss.py"],
            ("no",  "yes", "no",  "conc"):  ["cont_couplingSource_cht_OF12", "contRun_couplingSource_cht.py"],
            ("no",  "no",  "yes", "conc"):  ["", ""],
            ("no",  "no",  "no",  "conc"):  ["", ""],
            ("yes", "yes", "yes", "altn"):  ["cont_altnSource_fsiChtChtss_OF12", "contRun_altnSource_fsiChtChtss.py"],
            ("yes", "yes", "no",  "altn"):  ["cont_altnSource_fsiCht_OF12", "contRun_altnSource_fsiCht.py"],
            ("yes", "no",  "yes", "altn"):  ["", ""],
            ("yes", "no",  "no",  "altn"):  ["cont_altnSource_fsi_OF12", "contRun_altnSource_fsi.py"],
            ("no",  "yes", "yes", "altn"):  ["cont_altnSource_chtChtss_OF12", "contRun_altnSource_chtChtss.py"],
            ("no",  "yes", "no",  "altn"):  ["cont_altnSource_cht_OF12", "contRun_altnSource_cht.py"],
            ("no",  "no",  "yes", "altn"):  ["", ""],
            ("no",  "no",  "no",  "altn"):  ["", ""]
        }
    elif numVer >= 10.0:
        contFileDict = {
            #OF-10, OF-11
            #fsi    cht    chtss   method       scalarFieldの処理                   pointFieldの処理
            ("yes", "yes", "yes", "conc"):  ["cont_couplingSource_fsiChtChtss_OF10", "contRun_couplingSource_fsiChtChtss.py"],
            ("yes", "yes", "no",  "conc"):  ["cont_couplingSource_fsiCht_OF10", "contRun_couplingSource_fsiCht.py"],
            ("yes", "no",  "yes", "conc"):  ["", ""],
            ("yes", "no",  "no",  "conc"):  ["cont_couplingSource_fsi", "contRun_couplingSource_fsi.py"],
            ("no",  "yes", "yes", "conc"):  ["cont_couplingSource_chtChtss_OF10", "contRun_couplingSource_chtChtss.py"],
            ("no",  "yes", "no",  "conc"):  ["cont_couplingSource_cht_OF10", "contRun_couplingSource_cht.py"],
            ("no",  "no",  "yes", "conc"):  ["", ""],
            ("no",  "no",  "no",  "conc"):  ["", ""],
            ("yes", "yes", "yes", "altn"):  ["cont_altnSource_fsiChtChtss_OF10", "contRun_altnSource_fsiChtChtss.py"],
            ("yes", "yes", "no",  "altn"):  ["cont_altnSource_fsiCht_OF10", "contRun_altnSource_fsiCht.py"],
            ("yes", "no",  "yes", "altn"):  ["", ""],
            ("yes", "no",  "no",  "altn"):  ["cont_altnSource_fsi", "contRun_altnSource_fsi.py"],
            ("no",  "yes", "yes", "altn"):  ["cont_altnSource_chtChtss_OF10", "contRun_altnSource_chtChtss.py"],
            ("no",  "yes", "no",  "altn"):  ["cont_altnSource_cht_OF10", "contRun_altnSource_cht.py"],
            ("no",  "no",  "yes", "altn"):  ["", ""],
            ("no",  "no",  "no",  "altn"):  ["", ""]
        }

    return contFileDict

    # if OFversion == "10" or OFversion == "11":
    #     contFileDict = {
    #         #fsi    cht    chtss   method       scalarFieldの処理                   pointFieldの処理
    #         ("yes", "yes", "yes", "conc"):  ["cont_couplingSource_fsiChtChtss_OF10", "contRun_couplingSource_fsiChtChtss.py"],
    #         ("yes", "yes", "no",  "conc"):  ["cont_couplingSource_fsiCht_OF10", "contRun_couplingSource_fsiCht.py"],
    #         ("yes", "no",  "yes", "conc"):  ["", ""],
    #         ("yes", "no",  "no",  "conc"):  ["cont_couplingSource_fsi", "contRun_couplingSource_fsi.py"],
    #         ("no",  "yes", "yes", "conc"):  ["cont_couplingSource_chtChtss_OF10", "contRun_couplingSource_chtChtss.py"],
    #         ("no",  "yes", "no",  "conc"):  ["cont_couplingSource_cht_OF10", "contRun_couplingSource_cht.py"],
    #         ("no",  "no",  "yes", "conc"):  ["", ""],
    #         ("no",  "no",  "no",  "conc"):  ["", ""],
    #         ("yes", "yes", "yes", "altn"):  ["cont_altnSource_fsiChtChtss_OF10", "contRun_altnSource_fsiChtChtss.py"],
    #         ("yes", "yes", "no",  "altn"):  ["cont_altnSource_fsiCht_OF10", "contRun_altnSource_fsiCht.py"],
    #         ("yes", "no",  "yes", "altn"):  ["", ""],
    #         ("yes", "no",  "no",  "altn"):  ["cont_altnSource_fsi", "contRun_altnSource_fsi.py"],
    #         ("no",  "yes", "yes", "altn"):  ["cont_altnSource_chtChtss_OF10", "contRun_altnSource_chtChtss.py"],
    #         ("no",  "yes", "no",  "altn"):  ["cont_altnSource_cht_OF10", "contRun_altnSource_cht.py"],
    #         ("no",  "no",  "yes", "altn"):  ["", ""],
    #         ("no",  "no",  "no",  "altn"):  ["", ""]
    #     }
    # elif OFversion == "12" or OFversion == "13":
    #     contFileDict = {
    #         #fsi    cht    chtss   method       scalarFieldの処理                   pointFieldの処理
    #         ("yes", "yes", "yes", "conc"):  ["cont_couplingSource_fsiChtChtss_OF12", "contRun_couplingSource_fsiChtChtss.py"],
    #         ("yes", "yes", "no",  "conc"):  ["cont_couplingSource_fsiCht_OF12", "contRun_couplingSource_fsiCht.py"],
    #         ("yes", "no",  "yes", "conc"):  ["", ""],
    #         ("yes", "no",  "no",  "conc"):  ["cont_couplingSource_fsi_OF12", "contRun_couplingSource_fsi.py"],
    #         ("no",  "yes", "yes", "conc"):  ["cont_couplingSource_chtChtss_OF12", "contRun_couplingSource_chtChtss.py"],
    #         ("no",  "yes", "no",  "conc"):  ["cont_couplingSource_cht_OF12", "contRun_couplingSource_cht.py"],
    #         ("no",  "no",  "yes", "conc"):  ["", ""],
    #         ("no",  "no",  "no",  "conc"):  ["", ""],
    #         ("yes", "yes", "yes", "altn"):  ["cont_altnSource_fsiChtChtss_OF12", "contRun_altnSource_fsiChtChtss.py"],
    #         ("yes", "yes", "no",  "altn"):  ["cont_altnSource_fsiCht_OF12", "contRun_altnSource_fsiCht.py"],
    #         ("yes", "no",  "yes", "altn"):  ["", ""],
    #         ("yes", "no",  "no",  "altn"):  ["cont_altnSource_fsi_OF12", "contRun_altnSource_fsi.py"],
    #         ("no",  "yes", "yes", "altn"):  ["cont_altnSource_chtChtss_OF12", "contRun_altnSource_chtChtss.py"],
    #         ("no",  "yes", "no",  "altn"):  ["cont_altnSource_cht_OF12", "contRun_altnSource_cht.py"],
    #         ("no",  "no",  "yes", "altn"):  ["", ""],
    #         ("no",  "no",  "no",  "altn"):  ["", ""]
    #     }

    # return contFileDict

#
#  getNumProcsFistrFromDat
#--------------------------
def getNumProcsFistrFromDat(solidCaseDir, datFile="hecmw_ctrl.dat"):
    """ Fistrの並列数をhecmw_ctrl.datファイルから取得する"""
    FistrModelDict = pyFistr.getCntMshFileNameDict(solidCaseDir, datFile=datFile)
    mshName = FistrModelDict["fstrMSH"]
    if mshName[-4:] == ".msh":
        np = 1
    else:
        np = int(mshName.split("_")[-1][1:])
    return np

#
#  getNumProcsFistrFromMshNum
#--------------------------
def getNumProcsFistrFromMshNum(solidCaseDir, datFile="hecmw_ctrl.dat"):
    """ Fistrの並列数をmshの分割数から取得する"""
    FistrModelDict = pyFistr.getCntMshFileNameDict(solidCaseDir, datFile=datFile)
    mshName = FistrModelDict["msh"]
    header = ".".join(mshName.split(".")[:-1])
    mshHeader = solidCaseDir + "/" + header + "_p*"
    files = glob.glob(mshHeader)
    if len(files) == 0:
        np = 1
    else:
        np = len(files)
    return np

#
#  isDispCoupling
#------------------
def isDispCoupling(couplingDict):
    """ 変位のcouplingかどうか確認してその結果を返す"""
    timeIndex = couplingDict["timeIndex"]
    nStepsOF = couplingDict["nStepsOF"]     #変位のstep数
    timeName = couplingDict["timeName"]
    writeInterval = couplingDict["writeInterval"]
    beforeTime = couplingDict["beforeTime"]
    deltaT = str(float(timeName) - float(beforeTime))
    if int(timeIndex) % int(nStepsOF) == 0:
        return True
    if isWriteInterval(timeName, writeInterval, deltaT) == True:
        return True
    return False

#
#  isTempCoupling
#------------------
def isTempCoupling(couplingDict):
    """ 温度のcouplingかどうか確認してその結果を返す"""
    timeIndex = couplingDict["timeIndex"]
    nSteps = couplingDict["nSteps"]     #変位のstep数
    timeName = couplingDict["timeName"]
    writeInterval = couplingDict["writeInterval"]
    beforeTime = couplingDict["beforeTime"]
    deltaT = str(float(timeName) - float(beforeTime))
    if int(timeIndex) % int(nSteps) == 0:
        return True
    if isWriteInterval(timeName, writeInterval, deltaT) == True:
        return True
    return False

#
#  getNumVersion
#----------------
def getNumVersion(OFversion):
    """ versionを数値(float)に変換する。
    数値変換できない場合は、「-1.0」を返す"""
    numStr = ".".join(OFversion.split(".")[:2])
    if numStr[-1].lower() == "x":
        numStr = numStr[:-1]
    try:
        numVer = float(numStr)
    except:
        numVer = -1.0
    return numVer
    
