#!/usr/bin/python3
#  coding: utf-8
#
#       mappingFromPointCloud.py
#
#           点群fileからFrontIstr側のSGRPにmappingする。
#           easyIstrでも「import pyFistr」を修正して使っている。
#   
#   22/11/21    新規作成
#      11/23    getPointOnFace:data内から余分な点を削除するを追加
#      11/27    setMappingData:四角形の場合、faceNodes数が4個になるので、
#               faceNodes数を3個に固定して取得するように修正。
#      12/03    isLenInTriSize:点群から取得した3点とSGRP側のface面との
#               垂直距離がface三角形の辺の長さ以上かチェックを追加
#      12/08    getPtcData:data取得できなかった場合、警告を発しているが、
#               最大の警告回数を設定し、超えた場合は、中断する。
#   23/01/26    getValsInLine:mapping対象がvectorの場合、データが欠ける為、修正
#      02/11    getPtcDataFromLines:linesでも取得できるように追加
#      02/18    getMappingTriLocs,setMappingDataFromTriLocs:追加。
#               setMappingDataを分割した。
#      05/24    fistrMesh.__init__:mshファイル名をpyFistr（datファイル）
#               から取得する様に修正。
#   24/04/17    getMappingTriLocs:取得したPTCの3点の角度チェックを追加
#               角度が小さく3点で三角形ができない場合があるため。
#      04/18    mappingFromTri:0割でエラー発生する時有り、修正。
#      04/26    getMappingTriLocs:アルゴリズムを見直す。
#               （face角度、faceとの距離、三角形の角度の全てがOKの3点を取得）
#               getPointOnFace:上記見直しにより削除
#      04/28    getMappingTriLocs:再修正。lenFaceToPointの距離側に重みつけ追加
#      05/01    getMappingTriLocs:並列処理ができる様に修正。
#               mappingPressureToVtu,mappingPressure:並列処理ができる様に修正。
#      07/18    getMappingTriLocs:各procのkey数が10以下の場合の処理を削除（バグ）
#

import os
import multiprocessing
#import pyFistr
import pyFistr_ptc as pyFistr
import convertMesh as cm
import geometryFunc as geo

ls = "\n"

#--------------------
#  fistrMesh class
#--------------------
class fistrMesh:

    def __init__(self, caseDir):
        self.caseDir = caseDir
        #self.meshFile = "FistrModel.msh"
        _cntName, self.meshFile = pyFistr.cntMshFileName(caseDir)

    def getMeshHeaderData(self):
        fileName = self.caseDir + os.sep + self.meshFile
        f = open(fileName); lines = f.readlines(); f.close()
        meshHeaderData = pyFistr.getHeaderNumData(lines)
        return meshHeaderData

    #
    #  nodeDict
    #------------
    def nodeDict(self, meshHeaderData):
        """ nodeDict[<nodeNo>] = [<x>, <y>, <z>]を取得する"""
        nodeDict = {}
        for headerData in meshHeaderData:
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if len(words) > 0:
                if words[0] == "!NODE":
                    for nodeLoc in headerData[1]:
                        nodeNo = nodeLoc[0]
                        loc = nodeLoc[1:]
                        nodeDict[nodeNo] = loc
        return nodeDict

    #
    #  elementDict
    #--------------
    def elementDict(self, meshHeaderData):
        """ elementDict[<elmNo>] = {"type":<elmType>, "nodes":[<node0>, <node1>, ...]}を取得する"""
        elementDict = {}
        for headerData in meshHeaderData:
            header = headerData[0]
            words = pyFistr.deleteSp(header).split(",")
            if len(words) > 0:
                if words[0] == "!ELEMENT":
                    elmType = pyFistr.getValName(header, "TYPE")
                    for elmData in headerData[1]:
                        elmNo = elmData[0]
                        nodes = elmData[1:]
                        elementDict[elmNo] = {"type":elmType, "nodes":nodes}
        return elementDict


    def getSgrpData(self, sgrpName, meshHeaderData):
        """ SGRPの値（<要素No>, 、<面No>）の組をlistで取得"""
        sgrpData = []
        for headerData in meshHeaderData:
            header = headerData[0]
            data = headerData[1]
            words = pyFistr.deleteSp(header).split(",")
            if len(words) > 1:
                if words[0] == "!SGROUP":
                    if pyFistr.getValName(header, "SGRP") == sgrpName:
                        for i in range(0, len(data), 2):
                            elmNo = data[i]
                            faceNo = data[i+1]
                            sgrp = (elmNo, faceNo)
                            sgrpData.append(sgrp)
        return sgrpData

    #
    #  getSgrpDict
    #--------------
    def getSgrpDict(self, sgrpData, nodeDict, elementDict):
        """ facecenterLocationを取得する。"""
        sgrpDict = {}
        for (elmNo, faceNo) in sgrpData:
            elType = elementDict[elmNo]["type"]
            elNodes = elementDict[elmNo]["nodes"]
            faceNodes = cm.fistr_elFaceNode(elType, elNodes, faceNo)
            #2次要素の場合、mainの節点のみ取得する
            if len(faceNodes) == 6 or len(faceNodes) == 8:
                newNodes = []
                for i in range(0, len(faceNodes), 2):
                    nodeNo = faceNodes[i]
                    newNodes.append(nodeNo)
                faceNodes = newNodes
            locs = list(map(lambda x: nodeDict[x], faceNodes))
            _A, centerLoc = geo.polygonAG(locs)
            sgrp = (elmNo, faceNo)
            sgrpDict[(elmNo, faceNo)] = [centerLoc, faceNodes]
        return sgrpDict

    #
    #  getPtcData
    #--------------
    def getPtcData(self, ptcFile):
        """ ファイルから点群データを取得"""
        f = open(ptcFile); lines = f.readlines(); f.close()
        ptcData = self.getPtcDataFromLines(lines)
        return ptcData

    #
    #  getPtcDataFromLines
    #----------------------
    def getPtcDataFromLines(self, lines):
        """ linesから点群データを取得"""
        count = 0
        maxCount = 30
        ptcData = []
        for line in lines:
            ne = line.find("#")
            if ne >= 0:
                line = line[:ne] + "\n"
            words = line.split()
            try:
                nums = list(map(float, words))
            except:
                print("could not convert line -->", line, end="")
                nums = []
                count += 1
            if count > maxCount:
                ptcData = []
                break
            if len(nums) >= 4:
                loc = nums[:3]
                if len(nums[3:]) == 1:
                    vals = nums[-1]
                elif len(nums[3:]) > 1:
                    vals = nums[3:]
                ptcData.append([loc, vals])
        return ptcData

    #
    #  mappingFromTri
    #-----------------
    def mappingFromTri(self, locs, vals, loc):
        """ 三角形の座標(locs)と頂点の値(vals)から任意座標(loc:p)の値を推定。
                
        conts:    Y
            c:    |   2
            c:        /\
            c:       /  \
            c:      / .p \
            c:     /      \
            c:     --------    --> X
            c:    0         1
            c:   原点
        
        """
        #三角形の3辺の長さからtolを算出
        tol = min(geo.tri3length(locs)) / 1.0e5
        #三角形のXYZ軸を設定。原点はlocs[0]
        vec01 = geo.vector(locs[0], locs[1])     #X軸方向
        vec02 = geo.vector(locs[0], locs[2])
        vecx = geo.normal(vec01)                             #X軸
        vecz = geo.normal(geo.crossProduct(vec01, vec02))    #Y軸
        vecy = geo.normal(geo.crossProduct(vecz, vecx))      #Z軸
        #直線0-pの方程式算出 y=ax
        #  （点pを三角形のfaceに投影）
        vecp = geo.vector(locs[0], loc)
        xp = geo.dotProduct(vecx, vecp)     #点pの座標
        yp = geo.dotProduct(vecy, vecp)
        x1 = geo.length(locs[1], locs[0])   #点1の座標
        y1 = 0.0
        x2 = geo.dotProduct(vecx, vec02)    #点2の座標
        y2 = geo.dotProduct(vecy, vec02)    
        #X1,X2座標のチェック
        if abs(xp) >= tol and abs(x1 - x2) < tol:
            #x1=x2の場合
            a1 = yp / xp
            xc = x1
            yc = a1 * xc
            #xp<tolの場合は、pがY軸上で計算
        #p点の座標をチェック
        elif abs(xp) >= tol and abs(yp) >= tol:
            #点pがXY軸上に無い場合（通常）
            a1 = yp / xp                    #勾配
            a2 = (y2 - y1) / (x2 - x1)      #勾配2
            #直線0-pと直線1-2の交点cの座標
            xc = (a2 * x1) / (a2 - a1)      #点cの座標
            yc = a1 * xc
        elif (abs(xp) < tol and abs(yp) >= tol) and abs(x2 - x1) > tol:
            #点pがY軸上の場合（勾配a1が計算できない）
            a2 = (y2 - y1) / (x2 - x1)
            xc = 0.0                        #点cの座標
            yc = -a2 * x1
            #点pがY軸上で、x2-x1=0の場合は、計算できない
            #　座標変換が必要。
        elif abs(xp) >= tol and abs(yp) < tol:
            #点pがx軸上の場合
            xc = x1
            yc = 0.0
        else:
            #点pが原点上の場合
            xc = 0.0
            yc = 0.0
        #点cの値
        p0 = [x1, y1]; p1=[x2, y2]; line = [p0, p1]
        pc = [xc, yc]
        if xc == 0.0 and yc == 0.0:
            #原点上の場合、原点の値を返す
            valsP = vals[0]
            return valsP
        #原点以外の場合
        vals12 = [vals[1], vals[2]] 
        valsC = self.getValsInLine(line, pc, vals12, tol)
        #点pの値
        p0 = [0.0, 0.0]; p1 = pc; line = [p0, p1]
        pp = [xp, yp]
        vals0c = [vals[0], valsC]
        valsP = self.getValsInLine(line, pp, vals0c, tol)
        return valsP

    #
    #  getValsInLine
    #-----------------
    def getValsInLine(self, line, pc, vals, tol):
        """ 2次元平面に置き換えて、line上のc点の値valsを取得する。"""
        #pc点とline[0]、line[1]の距離をチェック
        #  各点に近い場合、各点の値に設定して返す。
        if geo.length2(line[0], pc) < tol:
            return vals[0]
        if geo.length2(line[1], pc) < tol:
            return vals[1]
        #lineの全長
        lineLength = geo.length2(line[0], line[1])
        #直線p0-pcの長さ（方向を考慮）
        p0 = [0,0,0]; p1 = line[1][:] + [0.0]   #3次元点に変換
        vec0c = geo.normal(geo.vector(p0, p1))  #直線の標準vec
        pcc = pc[:] + [0.0]                     #3次元点に変換
        vec0p = geo.vector(p0, pcc)             #p0→pcのvec
        cLength = geo.dotProduct(vec0c, vec0p)  #p0→pcの長さ（方向を考慮）
        #係数を算出
        coeff = cLength / lineLength
        #pcの値を算出
        if type(vals[0]) != list:
            #floatの場合
            cVals = (vals[1] - vals[0]) * coeff + vals[0]
        else:
            #listの場合
            cVals = []
            vals0 = vals[0]; vals1 = vals[1]
            for i in range(len(vals0)):
                cVal = (vals1[i] - vals0[i]) * coeff + vals0[i]
                cVals.append(cVal)
            # for val in vals:
            #     cVal = (val[1] - val[0]) * coeff + val[0]
            #     cVals.append(cVal)
        return cVals

    #
    #  setMappingData
    #------------------
    def setMappingData(self, ptcData, sgrpDict, nodeDict, nProcs=1):
        mapStat, mappingTriDict = self.getMappingTriLocs(ptcData, sgrpDict, nodeDict, nProcs)
        if mapStat != "OK":
            sgrpValDict = {}
            return mapStat, sgrpValDict      
        sgrpValDict = self.setMappingDataFromTriLocs(ptcData, sgrpDict, mappingTriDict)
        return mapStat, sgrpValDict

    #
    #  getMappingTriLocs
    #------------------
    def getMappingTriLocs(self, ptcData, sgrpDict, nodeDict, nProcs=1):
        """ 各mapping点に対応する点群データ3点（三角形）を取得した結果を返す。
        sgrpDictのkeyは、indexとして使用しているだけなので、何でもOK。
        mapping結果は、sgrpDictのkeyを使ってsgrpValDictに保存され戻る。

        Arg:
            ptcData   :点群データ
            sgrpDict  :key=(elmNo, faceNo), value=[loc, faceNodeNos]
            nodeDict  :key=nodeNo, value=loc
            nProcs=1  :int, 並列処理数
        Returns:
            mapStat     :mapping結果（"OK" or "NG"）
            sgrpValDict :key=(elmNo, faceNo), value=[mapping結果]
            """
        sgrpKeys = list(sgrpDict.keys())
        n = len(sgrpKeys) // nProcs
        #各procのkey数が10以下の場合は、nProcsを再設定
        # if n < 10:
        #     nProcs = len(sgrpKeys) / 10
        #     n = len(sgrpKeys) // nProcs
        #各procに均等にkeyを振り分ける
        keyProcs = [[] for i in range(nProcs)]
        for i in range(nProcs):
            sn = i * n
            keys = sgrpKeys[sn:sn+n]
            keyProcs[i] = keys
        #余ったkeyを振り分け
        rem = len(sgrpKeys) % nProcs
        sn = nProcs * n
        for i in range(rem):
            keyProcs[i] += [sgrpKeys[sn]]
            sn += 1
        #並列計算
        queue = []
        procs = []
        for n in range(nProcs):
            q = multiprocessing.Queue()
            queue.append(q)
            keys = keyProcs[n]
            p = multiprocessing.Process(
                target=self.getMappingTriLocs_proc,
                args=(ptcData, sgrpDict, nodeDict, keys, q)
                )
            p.start()
            procs.append(p)
        #全procsの終了を待つ
        mapStat = "OK"
        sgrpTriDict = {}
        for n in range(nProcs):
            loc9s = queue[n].get()
            if loc9s[0] == "NG":
                mapStat = "NG"
            loc9s = loc9s[1:]
            #取得したdataから3個の座標（三角形）を取得
            locs = []
            for i in range(0, len(loc9s), 9):
                loc0 = [loc9s[i], loc9s[i+1], loc9s[i+2]]
                loc1 = [loc9s[i+3], loc9s[i+4], loc9s[i+5]]
                loc2 = [loc9s[i+6], loc9s[i+7], loc9s[i+8]]
                locs.append([loc0, loc1, loc2])
            #取得結果を辞書に保存
            for i in range(len(keyProcs[n])):
                key = keyProcs[n][i]
                sgrpTriDict[key] = locs[i]
        return mapStat, sgrpTriDict

    #
    #  getMappingTriLocs_proc
    def getMappingTriLocs_proc(self, ptcData, sgrpDict, nodeDict, sgrpKeys, q):
        """ cpu毎のprocess
        mappingに用いるtriLocsを取得する"""

        def checkTriAngle(locs):
            """ 3点(三角形）が扁平の三角形かどうかを確認する。
            角度が2.6〜177.4°の範囲:正常三角形。
            以外：扁平三角形"""
            vec1 = geo.vector(locs[0], locs[1])
            vec2 = geo.vector(locs[1], locs[2])
            vec3 = geo.vector(locs[2], locs[0])
            #角度算出
            cosAng0 = geo.vecsAngle(vec1, vec2)
            cosAng1 = geo.vecsAngle(vec2, vec3)
            #角度をチェック
            if cosAng0 > 0.999 or cosAng0 < -0.999:
                return False
            if cosAng1 > 0.999 or cosAng1 < -0.999:
                return False
            return True

        loc9s = []
        search = searchBlock(ptcData)
        mapStat = "OK"
        #for key in sgrpDict.keys():
        for key in sgrpKeys:
            loc = sgrpDict[key][0]
            faceNodes = sgrpDict[key][1]
            triNodes = faceNodes[:3]        #3点の平面を取得
            triLocs = list(map(lambda x: nodeDict[x], triNodes))
            nPcs = 10    #sampleの取得数
            block = search.nearNodes(loc, nSamples=nPcs)
            if len(block) < nPcs:
                mapStat = "NG"
                break
            #nearBlock内の点との距離計算し、data取得
            data = []
            aveLen = sum(geo.tri3length(triLocs)) / 3.0
            for sample in block:
                lfp = geo.lenFaceToPoint(triLocs, sample[0])
                if lfp <=aveLen:
                    #triLocsとの垂直距離がtriLocsの平均辺の長さ以下を取得
                    l = geo.length(loc, sample[0])
                    #data.append([l, sample])
                    #lfpに重みつけ「10」を設定
                    data.append([l+10*lfp, sample])
            data.sort()

            #data内から3点を取得
            mapStat = "NG"
            surs = [[0,1,2], [0,1,3], [0,2,3], [1,2,3]]     #4点から3点を取得する組み合わせ
            tris = []
            for n in range(len(data)-4):
                #4点を取得
                sample4 = [data[n][1], data[n+1][1], data[n+2][1], data[n+3][1]]
                for i in range(len(surs)):
                    #locs（3点）を取得
                    n0 = surs[i][0]
                    n1 = surs[i][1]
                    n2 = surs[i][2]
                    sample3 = sample4[n0], sample4[n1], sample4[n2]
                    locs = list(map(lambda x: x[0], sample3))
                    #三角形の角度チェック（扁平三角形は除く）
                    if checkTriAngle(locs) == True:
                        #face同士の角度が30°以下？
                        absFaceAngle = abs(geo.facesAngle(triLocs, locs))
                        tris.append([absFaceAngle, locs])                             
                        if absFaceAngle > 0.866:      #30°以下
                            mapStat = "OK"
                            break
                if mapStat == "OK":
                    break
            #取得できなかった場合は、absFaceAngleの最大値を取得する
            if mapStat == "NG":
                tris.sort()
                locs = tris[-1][1]      #最大値（角度最小値）を取得
                mapStat = "OK"

            # #p点（loc）の推定値を計算
            # pVals = self.mappingFromTri(locs, vals, loc)
            loc9 = [locs[0][0], locs[0][1], locs[0][2],
                    locs[1][0], locs[1][1], locs[1][2],
                    locs[2][0], locs[2][1], locs[2][2],]
            loc9s += loc9
        loc9s = [mapStat] + loc9s
        q.put(loc9s)
        #return mapStat, sgrpTriDict

    #
    #  setMappingDataFromTriLocs
    #----------------------------
    def setMappingDataFromTriLocs(self, ptcData, sgrpDict, mappingTriDict):
        #ptc辞書作成
        ptcDataDict = {}
        for l, vals in ptcData:
            loc = tuple(l)
            ptcDataDict[loc] = vals
        #mapping結果の辞書作成
        sgrpValDict = {}
        for key in sgrpDict.keys():
            locl = sgrpDict[key][0]
            loc = tuple(locl)               #mappingするpoint
            locs = mappingTriDict[key]      #三角形の頂点の座標
            vals = []                       #　　↑　　　の値
            for i in range(3):
                val = ptcDataDict[tuple(locs[i])]
                vals.append(val)
            #mappingPointの推定値
            pVals = self.mappingFromTri(locs, vals, loc)
            sgrpValDict[key] = pVals
        return sgrpValDict

    #
    #  createMappingFile
    #--------------------
    def createMappingFile(self, mappingDict, sgrpName, ptcFile):
        """ mapping結果を保存する"""
        ptcName = os.path.basename(ptcFile)
        lines  = ["#  mapping result from PoinT Cloud (PTC) File\n"]
        lines += ["#\n"]
        lines += ["#    ptcFile  " + ptcName + "\n"]
        lines += ["#\n"]
        lines += ["#   elmNo  faceNo    value\n"]
        elmFaceNos = list(mappingDict.keys())
        elmFaceNos.sort()
        for key in elmFaceNos:
            elm, face = key
            line = ("%10d" % elm) + ("%5d" % face) + ("%13.5e" % mappingDict[key]) + "\n"
            lines += [line]
        return lines

    #
    #  createVtuForSgrp
    #-------------------
    def createVtuForSgrp(self, mappingDict, nodeDict, elementDict):
        """ mapping結果のvtuファイルを作成する"""
        (vnList, veList, cellVals) = self.createNodesElementsForSgrp(mappingDict, nodeDict, elementDict)
        vtuLines = self.createVtuFile(vnList, veList, cellVals)
        return vtuLines

    #
    #  createNodesElementsForSgrp
    def createNodesElementsForSgrp(self, mappingDict, nodeDict, elementDict):
        """ SGRPのみのnodesとelementを取得し直す"""
        #使用しているnodeNoを取得
        allNodes = []
        allElms = []
        vals = []
        for key in mappingDict.keys():
            elmNo, faceNo = key
            elType = elementDict[elmNo]["type"]
            elNodes = elementDict[elmNo]["nodes"]
            faceNodes = cm.fistr_elFaceNode(elType, elNodes, faceNo)
            #2次要素の場合、mainの節点のみ取得する
            if len(faceNodes) == 6 or len(faceNodes) == 8:
                newNodes = []
                for i in range(0, len(faceNodes), 2):
                    nodeNo = faceNodes[i]
                    newNodes.append(nodeNo)
                faceNodes = newNodes
            allElms += [faceNodes]            
            allNodes += faceNodes
            vals.append(mappingDict[key])
        #newNodeNoを作成
        vnList = []
        allNodes.sort()
        old2newDict = {}
        newNo = 0
        for nodeNo in allNodes:
            loc = nodeDict[nodeNo]
            vnList.append(loc)
            old2newDict[nodeNo] = newNo
            newNo += 1
        #newElementNoを作成
        veList = []
        for faceNodes in allElms:
            newNodes = list(map(lambda x: old2newDict[x], faceNodes))
            veList.append(newNodes)
        return vnList, veList, vals

    #
    #  createVtuFile
    def createVtuFile(self, vnList, veList, cellVals):
        """ vtuFileを作成する。"""
        lines = self.createVtuHeader()
        lines += self.createVtuPieceHeader(vnList, veList)
        lines += self.createVtuNodes(vnList)
        lines += self.createVtuCells(veList)
        lines += self.createVtuCellsData(veList, cellVals)
        lines += self.createVtuPieceFooter()
        lines += self.createVtuFooter()
        return lines

    #
    def getLinesWordsN(self, words, n):
        """ n words毎に改行して返す"""
        lines = []
        for i in range(0, len(words), n):
            line = " ".join(words[i:i + n]) + ls
            lines.append(line)
        return lines

    #
    def createVtuHeader(self):
        """ vtkファイルのheader部を作成して返す"""
        lines  = ['<?xml version="1.0"?>' + ls]
        lines += ['<VTKFile type="UnstructuredGrid" version="1.0">' + ls]
        lines += ['<UnstructuredGrid>' + ls]
        return lines

    #
    def createVtuPieceHeader(self, vnList, veList):
        """ pieceSectionのheaderを設定"""
        nNodes = len(vnList)
        nCells = len(veList)
        lines = ['<Piece NumberOfPoints="' + str(nNodes) + '" NumberOfCells="' + str(nCells) + '">' + ls]
        return lines

    #
    def createVtuNodes(self, vnList):
        """ node部を作成"""
        lines = ['<Points>' + ls]
        lines += ['<DataArray type="Float32" NumberOfComponents="3" format="ascii">' + ls]
        for nData in vnList:
            loc = nData
            loc = list(map(lambda x: str(x), loc))
            line = " ".join(loc) + ls
            lines.append(line)
        lines += ['</DataArray>' + ls]
        lines += ['</Points>' + ls] 
        return lines

    #
    def createVtuCells(self, veList):
        """ cell部を作成する"""
        lines = ['<Cells>' + ls]
        #elementを作成
        lines += ['<DataArray type="Int32" Name="connectivity" format="ascii">' + ls]
        for eData in veList:
            eNodes = eData
            eNodes = list(map(lambda x: str(x), eNodes))
            line = " ".join(eNodes) + ls
            lines.append(line)
        lines += ['</DataArray>' + ls]
        lines += ['<DataArray type="Int32" Name="offsets" format="ascii">' + ls]
        #offsetを作成
        offset = 0
        data = []
        for eData in veList:
            eNodes = eData
            offset += len(eNodes)
            data.append(str(offset))
        lines += self.getLinesWordsN(data, 10)      #10 words毎に改行する
        lines += ['</DataArray>' + ls]
        lines += ['<DataArray type="UInt8" Name="types" format="ascii">' + ls]
        #typesを作成
        data = []
        for eData in veList:
            eNodes = eData
            if len(eNodes) == 3:
                #三角形
                elType = "5"
            else:
                #四角形
                elType = "9"
            data.append(elType)
        lines += self.getLinesWordsN(data, 20)
        lines += ['</DataArray>' + ls]
        lines += ['</Cells>' + ls]
        return lines

    #
    def createVtuCellsData(self, veList, cellVals):
        """ cellDataを作成する"""
        lines = ['<CellData>' + ls]
        if type(cellVals[0]) == list:
            n = len(cellVals[0])
            strVals = []
            for cellVal in cellVals:
                strVal = list(map(str, cellVal))
                strVals += strVal
        else:
            n = 1
            strVals = list(map(str, cellVals))
        lines += ['<DataArray type="Float64" Name="mappingData" NumberOfComponents="' + str(n) + '" format="ascii">' + ls]
        lines += self.getLinesWordsN(strVals, 6)
        lines += ['</DataArray>' + ls]
        lines += ['</CellData>' + ls]
        return lines

    #
    def createVtuPieceFooter(self):
        """ pieceSectionのfooterを設定"""
        lines = ['</Piece>' + ls]
        return lines

    #
    def createVtuFooter(self):
        """ vtkファイルのfooterを作成して返す"""
        lines  = ['</UnstructuredGrid>' + ls]
        lines += ['</VTKFile>' + ls]
        return lines



#----------------------
#  searchBlock class
#----------------------
class searchBlock:
    """ 点群データの中から指定した座標に近いnodesを取得する"""

    def __init__(self, ptcData):
        self.ptcData = ptcData                      #点群dataのlist
        self.dataCenterLoc = []                     #全dataの中心座標
        self.minMaxLoc = self.getMinMaxLocation()   #座標のminMax
        self.dataWidth = [self.minMaxLoc[3]-self.minMaxLoc[0],
                          self.minMaxLoc[4]-self.minMaxLoc[1],
                          self.minMaxLoc[5]-self.minMaxLoc[2]]
        self.nSep = self.getNumberOfSeparate()      #sample数に応じた分割数
        self.dSep = max(self.dataWidth) / self.nSep #分割の幅
        self.dataBlockDict = self.setNodesEachBlocks()

    def setNodesEachBlocks(self):
        """ dataを各blockに保存する"""
        blockDict = {}
        for loc, vals in self.ptcData:
            key = self.getBlockKey(loc)
            if key in blockDict.keys():
                blockDict[key].append([loc, vals])
            else:
                blockDict[key] = [[loc, vals]]
        return blockDict

    def getBlockKey(self, loc):
        """ 座標のkeyを取得"""
        nx = int((loc[0] - self.dataCenterLoc[0]) / self.dSep)
        ny = int((loc[1] - self.dataCenterLoc[1]) / self.dSep)
        nz = int((loc[2] - self.dataCenterLoc[2]) / self.dSep)
        key = (nx, ny, nz)
        return key


    def getKeysUnderW(self, key, width):
        """ 幅width内の全てのkeyを取得する。"""
        keys = []
        for wx in range(-width, width+1):
            newX = key[0] + wx
            for wy in range(-width, width+1):
                newY = key[1] + wy
                for wz in range(-width, width+1):
                    newZ = key[2] + wz
                    newKey = (newX, newY, newZ)
                    if newKey in self.dataBlockDict.keys():
                        keys.append(newKey)
        return keys


    def nearNodes(self, loc, nSamples=1):
        """ locに近いnodesを取得"""
        key = self.getBlockKey(loc)
        width = 0
        nodes = []
        while len(nodes) < nSamples:
            width += 1
            #最大分割数を超えた？
            if width > 20:
                #超えた場合は、戻る
                print("warning: number of loop has been over number of separation.")
                print("  check model size between mapping model and point cloud model.")
                break
            keys = self.getKeysUnderW(key, width)
            nodes = []
            for key in keys:
                nodes += self.dataBlockDict[key]
        return nodes


    def getMinMaxLocation(self):
        """ 座標のmin、maxをlistで取得して返す。
        [minX, minY, minZ, maxX, maxY, maxZ]を返す。"""
        locs = list(map(lambda x: x[0], self.ptcData))
        minX = min(map(lambda x: x[0], locs))
        minY = min(map(lambda x: x[1], locs))
        minZ = min(map(lambda x: x[2], locs))
        maxX = max(map(lambda x: x[0], locs))
        maxY = max(map(lambda x: x[1], locs))
        maxZ = max(map(lambda x: x[2], locs))
        #平面の場合、平面の厚さ「0」となる為、厚さ方向に厚さを持たせる
        #xyzの最小幅を確認
        width = [[(maxX - minX), 0], [(maxY - minY), 1], [(maxZ - minZ), 2]]
        width.sort()
        #        2番目 / 最小幅 > 1e4 の場合
        if width[0][0] / width[1][0] < 1.0e-4:
            #最小幅は、(2番目幅/1000/分割数)に設定
            n = self.getNumberOfSeparate()
            minWidth = width[2][0] / 1.0e3 / n
            #最小幅を設定する
            coDir = width[0][1]
            if coDir == 0:
                minX -= minWidth
                maxX += minWidth
            elif coDir == 1:
                minY -= minWidth
                maxY += minWidth
            else:
                minZ -= minWidth
                maxZ += minWidth
        #blockの中心座標を取得
        midX = (maxX - minX) / 2.0
        midY = (maxY - minY) / 2.0
        midZ = (maxZ - minZ) / 2.0
        self.dataCenterLoc = [midX, midY, midZ]
        return (minX, minY, minZ, maxX, maxY, maxZ)

    def getNumberOfSeparate(self):
        """ sample数に応じた分割数の設定"""
        n = len(self.ptcData)   #sample数
        if n < 1000:
            nSep = 5
        elif n < 10000:
            nSep = 10
        else:
            nSep = 20
        return nSep

#=========================
#  mappingPressureToVtu
#=========================
def mappingPressureToVtu(caseDir, sgrpName, ptcFile, nProcs=1):
    """ 点群データをsgrpにmappingしてvtuファイルを作成する。
    
    Args:
        caseDir (str)   :FrontISTRのcaseDir（fullPath）
        sgrpName (str)  :SGRP名
        ptcFile (str)   :点群データのfile名（fullPath）
    Returns:
        vtuFile (str)   :vtuFile名（fullPath）
                        :エラー発生時は、「""」が戻る
        """
    mesh = fistrMesh(caseDir)
    meshHeaderData = mesh.getMeshHeaderData()
    nodeDict = mesh.nodeDict(meshHeaderData)
    elementDict = mesh.elementDict(meshHeaderData)
    sgrpData = mesh.getSgrpData(sgrpName, meshHeaderData)
    sgrpDict = mesh.getSgrpDict(sgrpData, nodeDict, elementDict)
    print("reading pointCloud file...")
    ptcData = mesh.getPtcData(ptcFile)
    if len(ptcData) < 3:
        return ""
    print("mapping point cloud data to SGRP...")
    mapStat, mappingDict = mesh.setMappingData(ptcData, sgrpDict, nodeDict, nProcs)
    if mapStat != "OK":
        return ""
    #print("saving mapping result...")
    name = "mappingSP_" + sgrpName
    #lines = mesh.createMappingFile(mappingDict, sgrpName, ptcFile)
    #fileName = caseDir + os.sep + name + ".map"
    #f = open(fileName, "w"); f.writelines(lines); f.close()
    lines = mesh.createVtuForSgrp(mappingDict, nodeDict, elementDict)
    fileName = caseDir + os.sep + name + ".vtu"
    f = open(fileName, "w"); f.writelines(lines); f.close()
    print("...done")
    return fileName


#=========================
#  mappingPressure
#=========================
def mappingPressure(caseDir, sgrpName, ptcFile, nProcs=1):
    """ 点群データをsgrpにmappingする。
    
    Args:
        caseDir (str)   :FrontISTRのcaseDir（fullPath）
        sgrpName (str)  :SGRP名
        ptcFile (str)   :点群データのfile名（fullPath）
    Returns:
        mapFile (str)   :mapping結果のfile名（fullPath）
                        :エラー発生時は「""」が戻る。
        """
    mesh = fistrMesh(caseDir)
    meshHeaderData = mesh.getMeshHeaderData()
    nodeDict = mesh.nodeDict(meshHeaderData)
    elementDict = mesh.elementDict(meshHeaderData)
    sgrpData = mesh.getSgrpData(sgrpName, meshHeaderData)
    sgrpDict = mesh.getSgrpDict(sgrpData, nodeDict, elementDict)
    print("reading pointCloud file...")
    ptcData = mesh.getPtcData(ptcFile)
    if len(ptcData) < 3:
        return ""
    print("mapping point cloud data to SGRP...")
    mapStat, mappingDict = mesh.setMappingData(ptcData, sgrpDict, nodeDict, nProcs)
    if mapStat != "OK":
        return ""
    print("saving mapped result...")
    lines = mesh.createMappingFile(mappingDict, sgrpName, ptcFile)
    name = "mappingSP_" + sgrpName
    fileName = caseDir + os.sep + name + ".map"
    f = open(fileName, "w"); f.writelines(lines); f.close()
    #lines = mesh.createVtuForSgrp(mappingDict, nodeDict, elementDict)
    #fileName = caseDir + os.sep + name + ".vtu"
    #f = open(fileName, "w"); f.writelines(lines); f.close()
    print("...done")
    return fileName




if __name__ == "__main__":
    import gettext
    gettext.install("app") # replace with the appropriate catalog name
    _ = gettext.gettext

    solidCaseDir = "/home/caeuser/CAE/CAE-FOAM/OF-9/coupling_fistr/OpenFOAM-CalculiX_copy0/Solid"
    fluidCaseDir = "/home/caeuser/CAE/CAE-FOAM/OF-9/coupling_fistr/OpenFOAM-CalculiX_copy0/Fluid"
    sgrpName = "interfaceFlap"
    
    PTCfile = fluidCaseDir + "/coupling_FrontISTR/data/flapFaceCenterLocs.0.particles"
        
    mapFile = mappingPressure(solidCaseDir, sgrpName, PTCfile)
    
