#!/usr/bin/python3
#
#           fsiCommonAlternate.py
#
#       交互計算用のfsi連成の共通関数群
#
#   23/11/05    新規作成
#   24/04/20    createMappingPressFile:画面出力を追加
#               createMappingDispFile_proc:画面出力を追加
#      04/23    setDeltaTInCnt:time2str関数部を削除
#      04/24    saveRestartFileFsi:保存前に同じtimeNameの
#               restartFileがあれば削除することを追加
#      05/01    createMappingPressFile,createMappingDispFile_proc:
#               mapping処理に並列処理を追加。
#

import os
import time
import pickle
import shutil
import glob
import multiprocessing
import pyFistr_ptc as pyFistr
import mappingFromPointCloud as mapping
import pyCoupling


couplingDict = {}
waitCount = 5000
waitTime = 0.01
 
FistrModel_cnt = "FistrModel_fsi.cnt"
resHeaderName = "FistrModel.res"
hecmwCtrlFile = "hecmw_ctrl_fsi.dat"
restartHeader = "FistrModel.restart"

#
#  waitUntilExistingFile
#------------------------
def waitUntilExistingFile(fileName):
    """ fileが出来上がるまで待つ"""
    count = int(couplingDict["maxWaitTime"])
    sleepTime = waitTime
    while True:
        if os.path.exists(fileName) == True:
            #time.sleep(sleepTime)
            break
        count -= 1
        if count < 0:
            break
        time.sleep(sleepTime)
    return

#-----------------------
#  createPressPtcFile
#-----------------------
def createPressPtcFile(dataDict, pressFile):
    """ 圧力の点群fileを作成する"""
    global couplingDict
    couplingDict = dataDict
    #faceCenterLoc読み込み
    patchName = couplingDict["mappingPatch"]
    timeName = couplingDict["timeName"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    solidCaseDir = couplingDict["solidCaseDir"]
    procNo = couplingDict["procNo"]
    if int(procNo) == 0:
        print("(py) --- creating pressure PTC ...", flush=True)
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    faceCenterFile = dataDir + "/" + patchName + "_faceCenterLocs_" + procNo
    f = open(faceCenterFile); lines = f.readlines(); f.close()
    #facePressを読み込み
    fp = open(pressFile); pLines = fp.readlines(); fp.close()
    newLines = []
    for i in range(len(lines)):
        newLine = lines[i][:-1] + " " + pLines[i]
        newLines.append(newLine)
    #圧力のptcファイルを保存
    name = patchName + "_faceCenterPress_" + timeName + "_" + procNo + ".particles"
    ptcFileName = solidCaseDir + "/" + name
    f = open(ptcFileName, "w"); f.writelines(newLines); f.close()
    return

#-----------------
#  setDeltaTInCnt
#-----------------
def setDeltaTInCnt(dataDict, sStep, sTime):
    """ cntファイルを更新する"""
    global couplingDict
    couplingDict = dataDict
    startTime = couplingDict["startTime"]
    deltaT = couplingDict["deltaT"]
    solidCaseDir = couplingDict["solidCaseDir"]
    timeIndex = couplingDict["timeIndex"]       #endStep
    currTime = couplingDict["timeName"]         #endTime
    endStep = int(timeIndex)
    eTime = float(currTime)
    #計算開始時？
    if float(sTime) < 0.0:
        sTime = startTime
        #sStep数を取得
        if float(sTime) == 0.0:
            #初回計算
            sStep = "0"
        else:
            #restart開始時のsStep数をrestartFileから取得
            #ssTime = pyCoupling.time2str(float(sTime))
            ssTime = sTime
            name = restartHeader + "_" + ssTime + "_*"
            pattern = solidCaseDir + "/" + name
            files = glob.glob(pattern)
            restartFile = files[0]
            restFile = ".".join(restartFile.split(".")[:-1])
            runStep = int(restFile.split("_")[-1])
            sStep = runStep
    stepTime = (eTime - float(sTime)) / (int(timeIndex) - int(sStep))
    if stepTime == 0.0:
        stepTime = float(deltaT)
    endTime = eTime
    #cnt読み込み
    cntFile = solidCaseDir + "/FistrModel_fsi.cnt"
    f = open(cntFile); lines = f.readlines(); f.close()
    #時間増分の設定
    i = pyFistr.getNextHeaderName(lines, 0, "!DYNAMIC", "")
    i += 2
    words = pyFistr.deleteSp(lines[i]).split(",")
    words[2] = str(endStep)         #endStep
    #時間増分セット
    words[3] = str(stepTime)        #時間増分
    lines[i] = ", ".join(words) + "\n"
    #restartの設定
    if float(sTime) == 0.0:
        #初回は、restartさせずに起動
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=1")
    else:
        #restartで起動する様に修正
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=-1")
    #cntFile保存
    f = open(cntFile, "w"); f.writelines(lines); f.close()
    return endStep, endTime


#-------------------------
#  getDispDictFromResFile
#-------------------------
def getDispDictFromResFile(dataDict, endStep, np):
    """ FrontISTRの変位計算結果を辞書形式で返す。
    dispDict={<nodeNo>: [x,y,z]}"""
    global couplingDict, resHeaderName
    couplingDict = dataDict
    #resFile名を取得
    solidCaseDir = couplingDict["solidCaseDir"]
    resHeader = resHeaderName
    #並列数を取得
    dispDict = {}
    queues = []
    procs = []
    for n in range(np):
        resFile = solidCaseDir + "/" + resHeader + "." + str(n) + "." + str(endStep)
        q = multiprocessing.Queue()
        queues.append(q)
        p = multiprocessing.Process(
            target=createDispDictFromResFile_proc,
            args=(resFile, endStep, n, q)
            )
        p.start()
        procs.append(p)
    #全セットprocessの結果を取得
    for n in range(np):
        addDict = queues[n].get()
        dispDict.update(addDict)
    return dispDict

#
#  createDispDictFromResFile_proc
def createDispDictFromResFile_proc(resFile, endStep, procNo, q):
    """ proc毎にdispDictを作成する。"""
    global couplingDict, resHeaderName
    solidCaseDir = couplingDict["solidCaseDir"]
    resFile = solidCaseDir + "/" + resHeaderName + "." + str(procNo) + "." + str(endStep)
    waitUntilExistingFile(resFile)
    addDict = readDispFromResFile(resFile)
    q.put(addDict)

#
#  readDispFromResFile
#----------------------
def readDispFromResFile(resFile):
    """ 変位データをresFileから取得し、辞書形式で返す"""
    
    def readTextLine(f):
        """ fileObject fから1行読み込む。
        eof（空行）読み込みでも指定時間まで読み直す。"""
        line = f.readline()
        if line != "":
            return line
        #空文の場合、読み直す
        count = waitCount
        sleepTime = waitTime / 10.0
        while count > 0:
            line = f.readline()
            if line != "":
                break
            count -= 1
            time.sleep(sleepTime)
        return line

    #
    #resFileを読み込み
    f = open(resFile);
    #*dataまでskip
    while True:
        #line = f.readline()
        line = readTextLine(f)
        if line[:len("*data")] == "*data":
            break
    #node数読み込み
    #line = f.readline()
    line = readTextLine(f)
    nNodes = int(line.split()[0])
    #節点の値の位置を取得
    #_dummy = f.readline()
    _dummy = readTextLine(f)
    #line = f.readline()
    line = readTextLine(f)
    words = line.split()
    nums = list(map(int, words))
    nVals = sum(nums)
    dispLoc = 0
    flag = 0
    for i in range(len(words)):
        #valName = f.readline().split()[0]
        valName = readTextLine(f).split()[0]
        if valName == "DISPLACEMENT":
            flag = 1
        else:
            if flag == 0:
                dispLoc += nums[i]
    #DISPLACEMENTの値を読み込む
    dispDict = {}
    n = 0
    while True:
        #nodeNo取得
        #line = f.readline()
        line = readTextLine(f)
        nodeNo = int(line.split()[0])
        #disp = [nodeNo]
        #disp読み込み
        vals = []
        while True:
            #line = f.readline()
            line = readTextLine(f)
            words = line.split()
            vals += words
            if len(vals) >= nVals:
                break
        dispWords = vals[dispLoc:dispLoc+3]
        dispVal = list(map(float, dispWords))
        dispDict[nodeNo] = dispVal
        n += 1
        if n >= nNodes:
            break
    f.close()
    return dispDict

#----------------
#  createDispPtc
#----------------
def createDispPtc(dataDict, dispDict):
    """ 変位の辞書から変位のptcを作成して返す"""
    global couplingDict
    couplingDict = dataDict
    #nodeDictを取得
    solidCaseDir = couplingDict["solidCaseDir"]
    nodeDictName = solidCaseDir + "/nodeDict.bin"
    f = open(nodeDictName, "rb")
    nodeDict = pickle.load(f)
    f.close()
    #ptc作成
    dispPtc = []
    for nodeNo in nodeDict.keys():
        disp = dispDict[nodeNo]
        loc = nodeDict[nodeNo]
        dispPtc.append([loc, disp])
    return dispPtc

#-------------------------
#  createMappingPressFile
#-------------------------
def createMappingPressFile(dataDict):
    """ 圧力のmapping結果fileを作成する。
    圧力ptcFile名:  <patchName>_faceCenterPress_<time>_<pNo>.particle
    圧力ptcFile名:  <patchName>_faceCenterPress.particle
    sgrpDict名:     sgrpDict.bin
    sgrpNodeDict名: nodeDict.bin
    mappingFile名:  mappingSP_<sgrp>.map
    """
    global couplingDict
    couplingDict = dataDict
    #dictから取得
    procNo = couplingDict["procNo"]
    nProcs = couplingDict["nProcs"]             #OF側のcpu数
    patchName = couplingDict["mappingPatch"]
    timeName = couplingDict["timeName"]
    sgrpName = couplingDict["mappingSgrp"]
    solidCaseDir = couplingDict["solidCaseDir"]
    nThreads = couplingDict["nThreads"]         #Fistr側のcpu数
    method = couplingDict["method"]
    #sgrpDict読み込み dict={(<elmNo>, <faceNo>):[<loc>,<faceNodeNos>]}
    sgrpDictName = solidCaseDir + "/sgrpDict.bin"
    f = open(sgrpDictName, "rb")
    sgrpDict = pickle.load(f)
    f.close()
    #nodeDict読み込み dict={<nodeNo>:<loc>}
    nodeDictName = solidCaseDir + "/nodeDict.bin"
    f = open(nodeDictName, "rb")
    nodeDict = pickle.load(f)
    f.close()
    #各procno毎の点群fileをまとめる
    lines = []
    if int(procNo) == 0:
        for n in range(int(nProcs)):
            pNo = str(n)
            name = patchName + "_faceCenterPress_" + timeName + "_" + pNo + ".particles"
            eachPtcFile = solidCaseDir + "/" + name
            waitUntilExistingFile(eachPtcFile)
            time.sleep(waitTime/2)
            f = open(eachPtcFile)
            lines += f.readlines()
            f.close()
        newName = patchName + "_faceCenterPress.particles"
        allPtcFile = solidCaseDir + "/" + newName
        f = open(allPtcFile, "w"); f.writelines(lines); f.close()
        #ptcData(数値data）を取得
        mesh = mapping.fistrMesh(solidCaseDir)
        ptcData = mesh.getPtcDataFromLines(lines)
        #mappingTriPressFileがあるか？
        if isMappingTriFilePress() == False:
            #存在しない場合、mapping用の3点を取得
            if method == "altn":
                #alternate（交互計算）
                np = max(int(nProcs), int(nThreads))
            else:
                #concurrent（同時計算）
                np = int(nProcs) + int(nThreads)
            print("(py) ---  ( getting the tri nodes for mapping at " + str(np) + " procs... )", flush=True)
            mapStat, mappingTriDict = mesh.getMappingTriLocs(ptcData, sgrpDict, nodeDict, np)
            if mapStat != "OK":
                return "error in pressure mapping."
            #mappingTriFilePressを作成
            createMappingTriFilePress(mappingTriDict)
        else:
            #fileを読み込む
            mappingTriDict = readMappingTriFilePress() 
        #圧力をmapping
        print("(py) --- pressure value is mapping to SGRP '" + sgrpName + "' of FrontISTR...", flush=True)
        mappingDict = mesh.setMappingDataFromTriLocs(ptcData, sgrpDict, mappingTriDict)
        #mapping結果を保存
        lines = mesh.createMappingFile(mappingDict, sgrpName, allPtcFile)
        name = "mappingSP_" + sgrpName + ".map"
        fileName = solidCaseDir + os.sep + name
        f = open(fileName, "w"); f.writelines(lines); f.close()
        #mapping結果をcntファイルに反映
        remakeFistrCntFileForPress(lines)

#
#  isMappingTriFilePress
def isMappingTriFilePress():
    """ 圧力mapping用の3点を取得したfileが存在するかチェック"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFilePress.bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    return os.path.exists(fileName)

#
#  createMappingTriFilePress
def createMappingTriFilePress(mappingTriDict):
    """ mapping用の3点の座標をpickleでfileに保存する"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFilePress.bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    #保存
    f = open(fileName, "wb")
    pickle.dump(mappingTriDict, f)
    f.close()

#
#  readMappingTriFilePress
def readMappingTriFilePress():
    """ 保存したmapping用の3点の座標を読み込む"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFilePress.bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    #読み込み
    f = open(fileName, "rb")
    mappingTriDict = pickle.load(f)
    f.close()
    return mappingTriDict

#
#  remakeFistrCntFileForPress
def remakeFistrCntFileForPress(mapLines):
    """ 圧力のmapping結果をcntファイルに反映"""
    mapCoeff = couplingDict["mappingCoeff"]
    mapShift = couplingDict["mappingShift"]
    timeName = couplingDict["timeName"]
    solidCaseDir = couplingDict["solidCaseDir"]
    sgrpName = couplingDict["mappingSgrp"]
    #追加行を作成
    coeff = float(mapCoeff)
    shift = float(mapShift)
    addLines = []
    for line in mapLines:
        if line[0] != "#":
            words = line.split()
            if len(words) >= 3:
                #words[2] = str(float(words[2]) * coeff + shift)
                press = float(words[2]) * coeff + shift
                #計算初回のみ値を修正
                #if iniFlag == 0
                if float(timeName) == 0.0:
                    #値が「0」の場合、zeroに近い値に修正
                    if press == 0.0:
                        press = 1.0e-36     #zeroに近い値
                words[2] = str(press)
                words[1] = "P" + words[1]
                newLine = ", ".join(words) + "\n"
                addLines.append(newLine)
    #cntFileを修正
    cntFile = solidCaseDir + os.sep + FistrModel_cnt
    f = open(cntFile); lines = f.readlines(); f.close()
    #挿入場所は、最初の!DLOADの該当SGRPに挿入する。
    i = 0
    while i < len(lines):
        i = pyFistr.getNextHeaderName(lines, i, "!DLOAD", "")
        i += 1
        if i >= len(lines):
            mess = "error: '" + sgrpName + "' does not exist in !DLOAD.\n"
            print(mess)
        words = pyFistr.deleteSp(lines[i]).split(",")
        if words[0] == sgrpName:
            si = i - 1
            ei = pyFistr.skipNextHeader(lines, i)
            break
    if si < len(lines):
        #cntFile修正、書き込み
        name = "mappingSP_" + sgrpName + ".map"
        grpid = pyFistr.getValName(lines[si], "GRPID")
        words = ["!DLOAD", "GRPID="+grpid, "forceType=mapping", "mapFile="+name,
                 "mapCoeff="+mapCoeff, "mapShift="+mapShift]
        lines[si] = ", ".join(words) + "\n"
        lines[si+1] = sgrpName + ", S, 0.0\n"
        newLines = lines[:si+2] + addLines + lines[ei:]
        f = open(cntFile, "w"); f.writelines(newLines); f.close()

#------------------------
#  createMappingDispFile
#------------------------
def createMappingDispFile(dataDict, dispPtc, endStep):
    """ 変位のptcデータをpatchにmappingしたfileを作成する。"""
    global couplingDict
    couplingDict = dataDict
    nProcs = couplingDict["nProcs"]
    np = int(nProcs)
    queue = []
    procs = []
    for n in range(np):
        q = multiprocessing.Queue()
        queue.append(q)
        procNo = str(n)
        p = multiprocessing.Process(
            target=createMappingDispFile_proc,
            args=(dispPtc, endStep, procNo, q)
            )
        p.start()
        procs.append(p)
    #全セットprocessの終了を待つ
    for n in range(np):
        a = queue[n].get()

#
#  createMappingDispFile_proc
def createMappingDispFile_proc(dispPtc, endStep, procNo, q):
    """ 指定したprocNoの変位をpatchにmappingしたfileを作成する"""
    nProcs = couplingDict["nProcs"]
    nThreads = couplingDict["nThreads"]
    solidCaseDir = couplingDict["solidCaseDir"]
    method = couplingDict["method"]
    mesh = mapping.fistrMesh(solidCaseDir) 
    #mappingするpointの座標を取得
    mapPointFaceDict, mapPointDict = getMappedPointLocsFsi(procNo)
    #mappingTriFileがあるか？
    if isMappingTriFileFsi(procNo) == False:
        #存在しない場合、mappingTriDictを取得
        if method == "altn":
            #alternate（交互計算）
            np = max(int(nProcs), int(nThreads))
        else:
            #concurrent（同時計算）
            np = int(nProcs) + int(nThreads)
        print("(py) ---  ( getting the tri nodes for mapping at " + str(np) + " procs... )", flush=True)
        mapStat, mappingTriDict = mesh.getMappingTriLocs(dispPtc, mapPointFaceDict, mapPointDict, np)
        if mapStat != "OK":
            return "error in displacement mapping."
        #mappingTriFileを作成
        createMappingTriFileFsi(mappingTriDict, procNo)
    else:
        #fileを読み込む
        mappingTriDict = readMappingTriFileFsi(procNo)
    #変位をmapping
    mappingDict = mesh.setMappingDataFromTriLocs(dispPtc, mapPointFaceDict, mappingTriDict)
    #mapping結果を保存(このfileをOpenFOAM側が読み込む)
    saveMappingDisp(mappingDict, endStep, procNo)
    q.put("")

#
#  getMappedPointLocsFsi
def getMappedPointLocsFsi(procNo):
    """ 変位をmappingするpointの座標を取得する"""
    patchName = couplingDict["mappingPatch"]
    #procNo = couplingDict["procNo"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = patchName + "_mapPointFaceNodes_" + procNo
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    f = open(fileName); lines = f.readlines(); f.close()
    mapPointFaceDict = {}
    mapPointDict = {}
    for line in lines:
        words = line.split()
        if len(words) > 4:
            nodeNo = int(words[0])
            pointLoc = list(map(float, words[1:4]))
            faceNodes = list(map(int, words[4:]))
            mapPointFaceDict[nodeNo] = [pointLoc, faceNodes]
            mapPointDict[nodeNo] = pointLoc
    return mapPointFaceDict, mapPointDict

#
#  isMappingTriFileFsi
def isMappingTriFileFsi(procNo):
    """ 変位mapping用の3点を取得したfileが存在するかチェック"""
    #procNo = couplingDict["procNo"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFile_" + procNo + ".bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    return os.path.exists(fileName)

#
#  createMappingTriFileFsi
def createMappingTriFileFsi(mappingTriDict, procNo):
    """ mapping用の3点の座標をfileに保存する"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFile_" + procNo + ".bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    #保存
    f = open(fileName, "wb")
    pickle.dump(mappingTriDict, f)
    f.close()

#
#  readMappingTriFileFsi
def readMappingTriFileFsi(procNo):
    """ 保存したmapping用の3点の座標を読み込む"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    name = "mappingTriFile_" + procNo + ".bin"
    fileName = fluidCaseDir + "/coupling_FrontISTR/data/" + name
    #読み込み
    f = open(fileName, "rb")
    mappingTriDict = pickle.load(f)
    f.close()
    return mappingTriDict

#
#  saveMappingDisp
def saveMappingDisp(mappingDict, endStep, procNo):
    """ mappingDispの結果を保存。このファイルをOF側が読み込む"""
    fluidCaseDir = couplingDict["fluidCaseDir"]
    #procNo = couplingDict["procNo"]
    patchName = couplingDict["mappingPatch"]
    timeIndex = couplingDict["timeIndex"]
    saveDir = fluidCaseDir + "/coupling_FrontISTR/data"
    name = patchName + "_mapPointFaceNodes_" + procNo
    fileName = saveDir + "/" + name
    f = open(fileName); lines = f.readlines(); f.close()
    #保存のfilename
    name = patchName + "_disp_" + timeIndex + "_" + procNo + "tm"
    if procNo == "0":
        #print("(py) --- displacement file is creating for OpenFOAM at " + str(endStep) + "...")
        print("(py) --- displacement file is creating for OpenFOAM at " + name + " ...")
    #name = patchName + "_disp_" + str(endStep) + "_" + procNo + "tm"
    dispFile = saveDir + "/" + name
    f = open(dispFile, "w")
    for line in lines:
        words = line.split()
        nodeNo = int(words[0])
        disp = mappingDict[nodeNo]
        line = " ".join(list(map(str, disp))) + "\n"
        f.write(line)
    f.close()
    return


#--------------------------
#  deleteResSetRestartFile
#--------------------------
def deleteResSetRestartFile(dataDict, delStep, delTime, nP):
    """ 結果fileを削除する。
    ただし、writeInterval時は、結果fileとrestartFileを残す"""
    global couplingDict
    couplingDict = dataDict
    timeName = couplingDict["timeName"]             #calcEndTime
    beforeTime = couplingDict["beforeTime"]
    writeInterval = couplingDict["writeInterval"]
    #deltaT = couplingDict["deltaT"]
    timeIndex = couplingDict["timeIndex"]           #calcEndStep
    solidCaseDir = couplingDict["solidCaseDir"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    patchName = couplingDict["mappingPatch"]
    nStepsOF = couplingDict["nStepsOF"]
    nProcs = couplingDict["nProcs"]
    deltaT = str(float(timeName) - float(beforeTime))
    #writeIntervalかどうか？
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data/"
    delFiles = []
    delFolders = []
    #OpenFOAM側を確認
    if pyCoupling.isWriteInterval(delTime, writeInterval, deltaT) == False:
        #削除可能な時間
        #  圧力の点群ファイルを削除
        pattern = solidCaseDir + "/" + patchName + "_faceCenterPress_" + delTime + "_*.particles"
        fileNames = getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  dispを削除
        pattern = dataDir + patchName + "_disp_" + delStep + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
        #  facePressを削除
        pattern = dataDir + patchName + "_facePress_" + delTime + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
    else:
        #writeInterval時の場合
        #  圧力の点群ファイル
        pattern = solidCaseDir + "/" + patchName + "_faceCenterPress_" + delTime + "_*.particles"
        fileNames = getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  dispをrename
        pattern = dataDir + patchName + "_disp_" + delStep + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delete2charaFromEndName(fileNames)
        #  facePressをrename
        pattern = dataDir + patchName + "_facePress_" + delTime + "_*tm"
        fileNames = getAllProcsFiles(pattern, int(nProcs))
        delete2charaFromEndName(fileNames)
    #FrontISTR側を確認
    #nP = pyCoupling.getNumProcsFistrFromDat(solidCaseDir, datFile=hecmwCtrlFile)
    if pyCoupling.isWriteInterval(delTime, writeInterval, deltaT) == False:
        #削除可能な時間
        #  resFileを削除
        resPattern = solidCaseDir + "/FistrModel.res.*." + delStep
        fileNames = getAllProcsFiles(resPattern, nP)
        delFiles += fileNames
        #  vtuFileを削除
        vtuFile = solidCaseDir + "/FistrModel.vis_psf." + delStep.zfill(4) + ".pvtu"
        delFiles += [vtuFile]
        #  vtuFolder
        vtuFolder = solidCaseDir + "/FistrModel.vis_psf." + delStep.zfill(4)
        delFolders += [vtuFolder]
    #FrontISTRのrestartFileを保存
    calcEndTime = timeName
    calcEndIdx = timeIndex
    if pyCoupling.isWriteInterval(calcEndTime, writeInterval, deltaT) == True:
        #writeInterval時の時
        #  resFileをrenameして保存
        saveRestartFileFsi(calcEndTime, calcEndIdx, nP)
    #fileFolderを削除
    for delFile in delFiles:
        if os.path.exists(delFile) == True:
            os.remove(delFile)
    for delFolder in delFolders:
        if os.path.isdir(delFolder) == True:
            shutil.rmtree(delFolder)

#
#  getAllProcsFiles
def getAllProcsFiles(pattern, np):
    """ 全procsのfile名を取得して返す"""
    names = pattern.split("*")
    files = []
    for i in range(np):
        fileName = names[0] + str(i) + names[1]
        files.append(fileName)
    return files

#
#  delete2charaFromEndName
def delete2charaFromEndName(fileNames):
    """ fileNameの最後の2文字(tm)を削除"""
    for fileName in fileNames:
        if os.path.exists(fileName):
            os.rename(fileName, fileName[:-2])

#
#  saveRestartFileFsi
def saveRestartFileFsi(timeName, timeIndex, np):
    """ FrontISTRのrestartFileを設定する"""
    if int(timeIndex) == 0:
        return
    solidCaseDir = couplingDict["solidCaseDir"]
    #同じtimeNameのrestartFileがあれば、保存前に削除
    pattern = "FistrModel.restart_*"
    files = glob.glob(solidCaseDir + "/" + pattern)
    for name in files:
        words = name.split("_")
        if float(words[-2]) == float(timeName):
            #削除
            os.remove(name)
    #restartFile保存
    for n in range(np):
        #orgName = solidCaseDir + "/FistrModel.restart.0"
        orgName = solidCaseDir + "/FistrModel.restart." + str(n)
        #newName = solidCaseDir + "/FistrModel.restart_" + timeName + "_" + timeIndex + ".0"
        newName = solidCaseDir + "/FistrModel.restart_" + timeName + "_" + timeIndex + "." + str(n)
        shutil.copy(orgName, newName)
