#!/usr/bin/python3
#  coding: utf-8
#
#       createPointCloudFromOpenFoam.py
#
#           OpenFOAMの計算結果から点群データを取得する
#
#   22/11/21    新規作成
#      12/10    createPointCloudOfPressure:複数のpatch名の取得を可能にした。
#   23/02/13    getAsciiLineValues:特殊なpatchContの場合、値が取得できず、修正
#               patchTypeがcodedFixedValuの場合、取得できない時あり。
#

import os
import pyTreeFoam_ptc as pyTreeFoam
import geometryFunc as geo
import mappingFromPointCloud as ptc

#---------------
#  ofMesh class
#---------------
class ofMesh:
    """ OpenFOAMのmesh操作用のclass
    
    Attribute:
        caseDir (str)   :caseDir
                         default:<currDir>
        meshDir (str)   :polyMeshの場所
                         default:<currDir>/constant
        region (str)    :region名
                         default:「.」"""

    def __init__(self, 
                    caseDir=os.getcwd(),
                    meshDir=os.getcwd()+os.sep+"constant",
                    region = "."):
        self.caseDir = caseDir
        self.meshDir = meshDir + os.sep + region + os.sep + "polyMesh"
        self.region = region

    #
    #  faceDict
    #--------------
    def faceDict(self):
        """ faceDictを取得する。faceを構成するnodeNo、faceが属するcellNoを取得。

        faceDict[<faceNo>] = {"nodes": [<node0>, ..], "cell": <cellNo>} """
        fileName = self.meshDir + os.sep + "faces"
        foamFile = pyTreeFoam.foamFile()
        faceDict = {}
        #faceを構成するnodeNoを取得
        ascii, data = foamFile.readSeparate(fileName)
        if foamFile.checkBinary(ascii) == "binary":
            #binaryの場合
            indx = foamFile.structBinData(data[0])
            faceNos = foamFile.structBinData(data[1])
            for n in range(len(indx)-1):
                ns = indx[n]
                ne = indx[n+1]
                nums = list(faceNos[ns:ne])
                faceDict[n] = {"nodes":nums}
        else:
            #asciiの場合
            lines = data[0][1].decode().split("\n")
            i = 0
            for line in lines:
                ns = line.find("(")
                if ns >= 0:
                    ne = line.find(")")
                    words = line[ns+1:ne].split()
                    nums = list(map(int, words))            
                    faceDict[i] = {"nodes":nums}
                    i += 1
        #faceが属するcellNoを取得
        fileName = self.meshDir + os.sep + "owner"
        ascii, data = foamFile.readSeparate(fileName)
        if foamFile.checkBinary(ascii) == "binary":
            #binary
            cellNos = foamFile.structBinData(data[0])
            i = 0
            for cellNo in cellNos:
                faceDict[i]["cell"] = cellNo
                i += 1
        else:
            #ascii
            lines = data[0][1].decode().split("\n")
            i = 0
            for line in lines:
                if line != "":
                    cellNo = int(line)
                    faceDict[i]["cell"] = cellNo
                    i += 1
        return faceDict

    #
    #  nodeDict
    #--------------
    def nodeDict(self):
        """ point(node)の辞書を取得する
        
        nodeDict[<nodeNo>] = [<x>, <y>, <z>]"""
        fileName = self.meshDir + os.sep + "points"
        foamFile = pyTreeFoam.foamFile()
        ascii, data = foamFile.readSeparate(fileName)
        nodeDict = {}
        if foamFile.checkBinary(ascii) == "binary":
            nums = foamFile.structBinData(data[0])
            n = 0
            for i in range(0, len(nums), 3):
                num = list(nums[i:i+3])
                nodeDict[n] = num
                n += 1
        else:
            lines = data[0][1].decode().split("\n")
            i = 0
            for line in lines:
                ns = line.find("(")
                if ns >= 0:
                    ne = line.find(")")
                    words = line[ns+1:ne].split()
                    nums = list(map(float, words))
                    nodeDict[i] = nums
                    i += 1
        return nodeDict

    #
    #  patchDictFromBoundary
    #-------------------------
    def patchDictFromBoundary(self):
        """ boundaryFileがらpatchの内容を辞書形式で取得
        
        patchDict[<patchName>] = {"cont":<patchConts>, "nFaces":<nFaces>, "startFace":<startFace>}"""
        case = pyTreeFoam.case(self.caseDir)
        polyMeshDir = os.path.dirname(self.meshDir)
        conts = case.getPatchNameFromBoundary(polyMeshDir)
        patchDict = {}
        for cont in conts:
            patchDict[cont[0]] = {}
            patchDict[cont[0]]["cont"] = cont[1]
            patchDict[cont[0]]["nFaces"] = int(cont[2])
            patchDict[cont[0]]["startFace"] = int(cont[3])
        return patchDict

    #
    #  patchDataDictFromField
    #-------------------------
    def patchDataDictFromField(self, time, field, patchNames, region="."):
        """ 指定したtime, field, patchNameの値を辞書形式で取得する。
        
        patchDataDict[<faceNo>] = <fieldValue>"""
        patchDataDict = {}
        fileName = os.sep.join([self.caseDir, time, region, field])
        foamFile = pyTreeFoam.foamFile()
        ascii, data = foamFile.readSeparate(fileName)
        if foamFile.checkBinary(ascii) == "binary":
            #binary形式の場合
            fieldDict = {}
            asciiOp = pyTreeFoam.strOp(ascii)
            bndCont, p = asciiOp.getKeywordContents("boundaryField", 0)
            bndContOp = pyTreeFoam.strOp(bndCont)
            for patchName in patchNames:
                patchCont, pp = bndContOp.getKeywordContents(patchName, 0)
                dataDict = self.getBinaryLineValues(patchCont, data)
                if type(dataDict) == dict:
                    #nonuniform形式
                    patchDict = self.patchDictFromBoundary()
                    nFaces = patchDict[patchName]["nFaces"]
                    startFace = patchDict[patchName]["startFace"]
                    patchNameDict = {}
                    for n in range(len(dataDict)):
                        faceNo = startFace + n
                        patchNameDict[faceNo] = dataDict[n]
                elif dataDict == "":
                    #値が存在しない場合 fieldから取得
                    if len(fieldDict) == 0:
                        fieldDict = self.getBinaryInternalFieldDict(ascii, data)
                    patchNameDict = self.getPatchValsFromField(fieldDict, patchName)
                else:
                    #uniform形式の値
                    if type(dataDict) == list:
                        #listの場合
                        vals = dataDict[:]
                    else:
                        #floatの場合
                        vals = dataDict
                    patchDict = self.patchDictFromBoundary()
                    nFaces = patchDict[patchName]["nFaces"]
                    startFace = patchDict[patchName]["startFace"]
                    patchNameDict = {}
                    for n in range(startFace, startFace + nFaces):
                        patchNameDict[n] = vals
                patchDataDict.update(patchNameDict)
        else:
            #ascii形式の場合
            fieldDict = {}
            asciiOp = pyTreeFoam.strOp(ascii)
            bndCont, p = asciiOp.getKeywordContents("boundaryField", 0)
            bndContOp = pyTreeFoam.strOp(bndCont)
            for patchName in patchNames:
                patchCont, pp = bndContOp.getKeywordContents(patchName, 0)
                dataDict = self.getAsciiLineValues(patchCont, data)
                if type(dataDict) == dict:
                    #nonuniform形式の値
                    patchDict = self.patchDictFromBoundary()
                    nFaces = patchDict[patchName]["nFaces"]
                    startFace = patchDict[patchName]["startFace"]
                    patchNameDict = {}
                    for n in range(len(dataDict)):
                        faceNo = startFace + n
                        patchNameDict[faceNo] = dataDict[n]
                elif dataDict == "":
                    #値が存在しない場合 fieldから取得
                    if len(fieldDict) == 0:
                        fieldDict = self.getAsciiInternalFieldDict(ascii, data)
                    patchNameDict = self.getPatchValsFromField(fieldDict, patchName)
                else:
                    #uniform形式の値
                    if type(dataDict) == list:
                        #listの場合
                        vals = dataDict[:]
                    else:
                        #floatの場合
                        vals = dataDict
                    patchDict = self.patchDictFromBoundary()
                    nFaces = patchDict[patchName]["nFaces"]
                    startFace = patchDict[patchName]["startFace"]
                    patchNameDict = {}
                    for n in range(startFace, startFace + nFaces):
                        patchNameDict[n] = vals
                patchDataDict.update(patchNameDict)
        return patchDataDict

    #
    #  getBinaryLineValues
    def getBinaryLineValues(self, lineCont, data):
        """ bainaryLineから値を取得する"""
        dataDict = {}
        foamFile = pyTreeFoam.foamFile()
        if lineCont.find("nonuniform") >= 0:
            #"\0"で囲まれたwordからdataのidxを取得
            ns = lineCont.find("\0")
            if ns >= 0:
                ne = lineCont[ns+1:].find("\0")
                ns += 1
                words = lineCont[ns:ns+ne].split(".")
                idx = int(words[-1])
                useData = data[idx]
                nData = int(useData[0][1].decode())     #data数
                nums = foamFile.structBinData(useData)  #全データ
                nVal = len(nums) // nData               #1組のdata数
                if nVal == 1:
                    for n in range(len(nums)):
                        dataDict[n] = nums[n]    
                else:
                    for n in range(nData):
                        vals = list(nums[n:n+nVal])
                        dataDict[n] = vals
        elif lineCont.find("uniform") >= 0:
            #listで値を返す
            lineContOp = pyTreeFoam.strOp(lineCont)
            p = lineContOp.getKeywordPointer("uniform", 0)
            keyword, p = lineContOp.getKeyword(p)
            valLine = lineCont[p:]
            ns = valLine.find("(")
            if ns >= 0:
                ne = valLine.find(")")
                ns += 1
                words = valLine[ns:ne].split()
                nums = list(map(float, words))
                dataDict = nums
            else:
                keyword, p = lineContOp.getKeyword(p)
                num = float(keyword)
                dataDict = num
        else:
            #空文字で値を返す
            dataDict = ""
        return dataDict

    #
    #  getAsciiLineValues
    def getAsciiLineValues(self, lineCont, data):
        """ asciiLineから値を取得する"""

        def isInString(value, lineCont):
            ns = lineCont.find(value)
            if ns < 0:
                return -1
            elif ns == 0:
                return ns
            elif lineCont[ns - 1] > " ":
                return -1
            return ns

        # #value行を取得
        # ns = lineCont.find("value ")
        # if ns < 0:
        #     #「value」が存在しない場合
        #     return ""
        # if lineCont[ns - len("value ")] > " ":
        #     #「value」の前charが記号文字以上の場合
        #     return ""
        # #lineCont[ns:]以降で1行を取得
        # lineCont, _p, _kind = pyTreeFoam.strOp(lineCont).get1line(ns)

        ns = isInString("value ", lineCont)
        if ns < 0:
            ns = isInString("internalField ", lineCont)
            if ns < 0:
                return ""
        lineCont, _p, _kind = pyTreeFoam.strOp(lineCont).get1line(ns)
        dataDict = {}
        if lineCont.find("nonuniform") >= 0:
            #"\0"で囲まれたwordからdataのidxを取得
            ns = lineCont.find("\0")
            if ns >= 0:
                ne = lineCont[ns+1:].find("\0")
                ns += 1
                words = lineCont[ns:ns+ne].split(".")
                idx = int(words[-1])
                useData = data[idx]
                #dataから値を取得する
                lines = useData[1].decode().split("\n")
                n = 0
                for line in lines:
                    ns = line.find("(")
                    if ns >= 0:
                        #listの値を取得
                        ne = line.find(")")
                        ns += 1
                        words = line[ns:ne].split()
                        nums = list(map(float, words))
                        dataDict[n] = nums
                        n += 1
                    elif line != "":
                        #値を取得
                        num = float(line)
                        dataDict[n] = num
                        n += 1
        elif lineCont.find("uniform") >= 0:
            #listで値を返す
            lineContOp = pyTreeFoam.strOp(lineCont)
            p = lineContOp.getKeywordPointer("uniform", 0)
            keyword, p = lineContOp.getKeyword(p)
            valLine = lineCont[p:]
            ns = valLine.find("(")
            if ns >= 0:
                ne = valLine.find(")")
                ns += 1
                words = valLine[ns:ne].split()
                nums = list(map(float, words))
                dataDict = nums
            else:
                keyword, p = lineContOp.getKeyword(p)
                num = float(keyword)
                dataDict = num
        else:
            #空文字で値を返す
            dataDict = ""
        return dataDict
    
    #
    #  getPatchValsFromField
    def getPatchValsFromField(self, fieldDict, patchName):
        """ fieldの値からdataを取得する"""
        #patchのfaceNoを取得
        patchDict = self.patchDictFromBoundary()
        nFaces = patchDict[patchName]["nFaces"]
        startFace = patchDict[patchName]["startFace"]
        #faceの辞書作成
        faceDict = self.faceDict()
        #patchに属するcellを取得
        valDict = {}
        for i in range(nFaces):
            faceNo = startFace + i
            cellNo = faceDict[faceNo]["cell"]
            if type(fieldDict) == dict:
                val = fieldDict[cellNo]
            else:
                val = fieldDict
            valDict[faceNo] = val
        return valDict

    #
    #  getBinaryInternalFieldDict
    def getBinaryInternalFieldDict(self, ascii, data):
        """ binaryのinternalFieldの値を辞書形式で取得"""
        asciiOp = pyTreeFoam.strOp(ascii)
        p = asciiOp.skipFoamFile()
        lineCont, p, kind = asciiOp.get1line(p)
        keyword, pp = pyTreeFoam.strOp(lineCont).getKeyword(0)
        while (keyword != "internalField" and p < len(ascii)-1):
            lineCont, p, kind = asciiOp.get1line(p)
            keyword, pp = pyTreeFoam.strOp(lineCont).getKeyword(0)
        valDict = self.getBinaryLineValues(lineCont, data)
        return valDict

    #
    #  getAsciiInternalFieldDict
    def getAsciiInternalFieldDict(self, ascii, data):
        """ aciiのinternalFieldの値を辞書形式で取得"""
        asciiOp = pyTreeFoam.strOp(ascii)
        p = asciiOp.skipFoamFile()
        lineCont, p, kind = asciiOp.get1line(p)
        keyword, pp = pyTreeFoam.strOp(lineCont).getKeyword(0)
        while (keyword != "internalField" and p < len(ascii)-1):
            lineCont, p, kind = asciiOp.get1line(p)
            keyword, pp = pyTreeFoam.strOp(lineCont).getKeyword(0)
        valDict = self.getAsciiLineValues(lineCont, data)
        return valDict

    def checkLocs(self, locs):
        a=[]
        nvec0 = geo.normal(geo.vector(locs[0], locs[1]))
        for i in range(2, len(locs)):
            nvec = geo.normal(geo.vector(locs[0], locs[i]))
            dot = geo.dotProduct(nvec0, nvec)
            a.append(dot)
        if a[0] < a[1]:
            print(a)
        if len(locs) != 4:
            print(locs)

    #
    #  getFaceCenterDict
    #-----------------
    def getFaceCenterDict(self, patchNames):
        """ patchの中心座標の辞書を取得
        
        centerLocDict[<faceNo>] = [<x>, <y>, <z>]"""
        patchDict = self.patchDictFromBoundary()
        #nFaces = patchDict[patchName]["nFaces"]
        #startFace = patchDict[patchName]["startFace"]
        faceDict = self.faceDict()
        nodeDict = self.nodeDict()
        centerDict = {}
        for patchName in patchNames:
            nFaces = patchDict[patchName]["nFaces"]
            startFace = patchDict[patchName]["startFace"]
            for faceNo in range(startFace, startFace + nFaces):
                nodeNos = faceDict[faceNo]["nodes"]
                nodeLocs = []
                for nodeNo in nodeNos:
                    loc = nodeDict[nodeNo]
                    nodeLocs.append(loc)
                _A, centerLoc = geo.polygonAG(nodeLocs)
                centerDict[faceNo] = centerLoc
        return centerDict

    #
    #  correctFaces
    #-----------------
    def correctFaces(self, centerDict, faceDict, nodeDict):
        """ centerDictから密接しているcenterDictを削除する。"""
        centerLocs = []
        for key in centerDict.keys():
            centerLocs.append([centerDict[key], key])
        search = ptc.searchBlock(centerLocs)
        #密接点検出の寸法算出
        #  平均要素サイズの1/10以下で近接している点を削除
        faces = list(centerDict.keys())[:10]
        ls = []
        for faceNo in faces:
            nodes = faceDict[faceNo]["nodes"]
            loc0 = nodeDict[nodes[0]]
            loc1 = nodeDict[nodes[1]]
            loc2 = nodeDict[nodes[2]]
            l = geo.tri3length([loc0, loc1, loc2])
            ls += l
        delLength = sum(ls) / len(ls) / 10.0
        #削除faceを取得
        delFaces = set([])
        for faceNo in centerDict.keys():
            if not faceNo in delFaces:
                #近傍のfaceを取得
                loc = centerDict[faceNo]
                blockFaces = search.nearNodes(loc)
                #点間距離を算出
                for _loc, face in blockFaces:
                    if faceNo != face:
                        if not face in delFaces:
                            l = geo.length(centerDict[faceNo], centerDict[face])
                            if l < delLength:
                                #削除するface
                                delFaces.add(faceNo)
        dels = list(delFaces)
        return dels

    #
    #  writeDataFile
    #----------------
    def writeDataFile(self, name, patchDataDict, faceCenterDict,
                        caseDir, time, field, patches, region):
        """ file作成"""
        faceNos = list(faceCenterDict.keys())
        faceNos.sort()
        lines =  ["#  point cloud data from OpenFOAM\n"]
        lines += ["#\n"]
        lines += ["#   source  OpenFOAM\n"]
        lines += ["#       caseDir  " + caseDir + "\n"]
        lines += ["#       time  " + time + "\n"]
        lines += ["#       region  " + region + "\n"]
        lines += ["#       field  " + field + "\n"]
        lines += ["#       boundary  " + " ".join(patches) + "\n"]
        lines += ["#\n"]
        lines += ["#       x            y            z            value\n"]
        fmt = "%13.5e"
        for faceNo in faceNos:
            loc = faceCenterDict[faceNo]
            line = fmt % loc[0] + fmt % loc[1] + fmt % loc[2] + " "
            vals = patchDataDict[faceNo]
            if type(vals) == list:
                words = list(map(lambda x: fmt % x, vals))
                valLine = " ".join(words) + "\n"
            else:
                valLine = fmt % vals + "\n"
            lines += [line + valLine]
        fileName = name
        f = open(fileName, "w"); f.writelines(lines); f.close()

#=============================
#  createPointCloudOfPressure
#=============================
def createPointCloudOfPressure(caseDir, time, field, patches,
                        #mesh="constant", 
                        region=".",
                        fistrDir=os.getcwd()):
    """ OpenFOAMの結果ファイルから圧力（scalar）の点群データファイルを作成する。
    
    Args:
        caseDir (str)   :OpenFOAMのcaseDir
        time (str)      :timeFolder
        field (str)     :field名
        patches (list(str)) :patch名
        mesh (str)      :"constant"(default) or timeFolder
        region (str)    :region名(default:".")
        fistrDir (str)  :FrontISTRのcaseDir(default:currDir)"""
    case = pyTreeFoam.case(caseDir)
    pMesh = case.getCurrMeshDir(time, region, "boundary")
    meshDir = caseDir + os.sep + pMesh.split(os.sep)[0]
    polyMesh = ofMesh(caseDir, meshDir)
    print("getting boundary patch data...")
    patchDataDict = polyMesh.patchDataDictFromField(time, field, patches, region)
    print("getting patch face center location...")
    faceCenterDict = polyMesh.getFaceCenterDict(patches)
    if len(patches) == 1:
        name = "ptcSP_" + patches[0] + ".particles"
    else:
        name = "ptcSP_" + patches[0] + "Etc.particles"
    print("creating pointCloud file '" + name + "'...")
    fileName = fistrDir + os.sep + name
    polyMesh.writeDataFile(fileName, patchDataDict, faceCenterDict,
                        caseDir, time, field, patches, region)
    print("...done")
    return fileName

if __name__ == "__main__":
    import gettext
    gettext.install("app")

    #caseDir = "/home/caeuser/CAE/CAE-FOAM/OF-9/cavity_bin"
    caseDir = "/home/caeuser/CAE/OF_fistr/OF_9/flowAna_face"
    time = "200"
    field = "p"
    patch = "pressFront"
    #meshDir = caseDir + "/constant"
    mesh = "constant"
    region = "."
    fistr = "/home/caeuser/CAE/OF_fistr/fistr/plateAna_face"

    createPointCloudOfPressure(caseDir, time, field, patch, fistrDir=fistr)




