#!/usr/bin/python3
#
#       contRun_couplingSource_chtChtss.py
#
#       OpenFOAMが出力した熱流束から、FrontISTRを実行して、
#       温度、熱ひずみを出力する。
#
#   23/10/04    新規作成
#      11/14    chtCommonConcurrentモジュール使用に変更
#               chtssCommonConcurrentモジュール使用に変更
#      11/16    allCommonConcAltnモジュール使用に変更
#   24/04/24    endTime時（最後の計算）は、restartFile名のチェックを追加。
#

import sys, os
import shutil
import pyCoupling
import chtCommonConcurrent as chtComm
import fsiCommonConcurrent as fsiComm
import chtssCommonConcurrent as chtssComm
import allCommonConcAltn as allComm

waitCount = 5000
waitTime = 0.01


#---------------------
#  coupling_chtChtss
#---------------------
def coupling_chtChtss():
    """ OpenFOAMが出力した熱流束から、FrontISTRが温度を計算する。"""
    solidCaseDir = couplingDict["solidCaseDir"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    patchName = couplingDict["mappingPatch"]
    timeName = couplingDict["timeName"]
    procNo = couplingDict["procNo"]
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    #---- heatFluxをmapping ---------
    #heatFluxの点群ファイル作成
    htFile = dataDir + "/" + patchName + "_faceHeatFlux_" + timeName + "_" + procNo + "tm"
    chtComm.createHeatFluxPtcFile(couplingDict, htFile)
    if int(procNo) == 0:
        #heatFluxをmappingしてcntファイルに反映
        chtComm.createMappingHeatFlux(couplingDict)
        #---- fistr実行（温度計算） --------------
        #cht計算前の設定
        setFsiChtCntFile("cht")
        endStep, endTime = chtComm.setDeltaTInCnt_cht(couplingDict)
        #Fistr起動
        runFistr("cht", endStep, endTime)
        #--- 温度を取得 ---------
        #温度のptcデータを作成
        print("(py) --- creating temperature PTC ...", flush=True)
        np = pyCoupling.getNumProcsFistrFromDat(solidCaseDir)
        tempDict = chtComm.getTempDictFromResFile(couplingDict, endStep, np)
        tempPtc = chtComm.createTempPtc(couplingDict, tempDict)
        #温度をmappingしたfileを作成
        print("(OF) --- temperature value is mapping to patch '" + patchName + "' ...", flush=True)
        chtComm.createMappingTempFile(couplingDict, tempPtc, endStep)
        #---- fistr実行（熱ひずみ計算）----------
        #chtss計算前の設定(cnt,mshファイルを設定)
        setFsiChtCntFile("chtss")
        #温度の計算結果をcntファイルに設定
        chtssComm.remakeFistrCntFileForTemperature(couplingDict, tempDict)
        #Fistr起動(endStep, endTimeはdummy)
        runFistr("chtss", endStep, endTime)
        #---- 変位を取得 -----------------------
        #変位の点群データ作成
        print("(py) --- creating displacement PTC ...", flush=True)
        npSs = pyCoupling.getNumProcsFistrFromDat(solidCaseDir)
        chtssDispDict = chtssComm.getDispDictFromResFile(couplingDict, endStep, npSs)
        chtssDispPtc = chtssComm.createDispPtc(couplingDict, chtssDispDict)
        #変位のmappingFileを作成
        print("(py) --- displacement value is mapping to patch '" + patchName + "' ...", flush=True)
        fsiComm.createMappingDispFile(couplingDict, chtssDispPtc, endStep)
        #--- 結果file整理（file削除、restart設定）----
        chtComm.deleteResSetRestartFileCht(couplingDict, np)
    return ""

#
#  setFsiChtCntFile
def setFsiChtCntFile(sect):
    """ sectに応じたdatファイルに修正"""
    solidCaseDir = couplingDict["solidCaseDir"]
    pasteDatFile = solidCaseDir + os.sep + "hecmw_ctrl.dat"
    if sect == "fsi":
        datFile = solidCaseDir + os.sep + "hecmw_ctrl_fsi.dat"
    elif sect == "cht":
        datFile = solidCaseDir + os.sep + "hecmw_ctrl_cht.dat"
    elif sect == "chtss":
        datFile = solidCaseDir + os.sep + "hecmw_ctrl_chtss.dat"
    shutil.copy(datFile, pasteDatFile)

#
#  runFistr
#----------
def runFistr(sect, endStep, endTime):
    """ FrontISTRを起動"""
    allComm.runFistr(couplingDict, mmCommArea, sect, endStep, endTime)
    return

#------------------
#  createNextDisp
#------------------
def createNextDisp():
    """ OpenFOAMが読み込む変位のfileをコピーして作成する"""
    timeIndex = couplingDict["timeIndex"]
    nStepsOF = couplingDict["nStepsOF"]     #変位のstep数
    nSteps = couplingDict["nSteps"]         #温度のstep数
    procNo = couplingDict["procNo"]
    patchName = couplingDict["mappingPatch"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data"
    #温度のcoupling時に作成した変位file
    name = patchName + "_disp_" + timeIndex + "_" + procNo + "tm"
    oldDispFile = dataDir + "/" + name
    if os.path.exists(oldDispFile) == False:
        oldDispFile = oldDispFile[:-2]
    #OpenFOAMが読み込むfileを作成
    calcEndStep = int(timeIndex) + int(nStepsOF)
    if calcEndStep % int(nSteps) != 0:
        name = patchName + "_disp_" + str(calcEndStep) + "_" + procNo + "tm"
        newDispFile = dataDir + "/" + name
        shutil.copy(oldDispFile, newDispFile)
    #不要なfileを削除--------------
    timeName = couplingDict["timeName"]
    deltaT = couplingDict["deltaT"]
    writeInterval = couplingDict["writeInterval"]
    name = patchName + "_disp_" + timeIndex + "_" + procNo + "tm"
    orgFile = dataDir + "/" + name
    #  writeInterval？
    if pyCoupling.isWriteInterval(timeName, writeInterval, deltaT) == True:
        #orgFileをrenameする
        newFile = orgFile[:-2]
        if os.path.exists(orgFile) == True:
            shutil.move(orgFile, newFile)
    else:
        #削除する
        if os.path.exists(orgFile) == True:
            os.remove(orgFile)



if __name__ == "__main__":
    import gettext
    gettext.install("app") # replace with the appropriate catalog name
    _ = gettext.gettext

    currTime = sys.argv[1]      #currTime
    beforeTime = sys.argv[2]    #前回のtime
    procNo = sys.argv[3]        #procNo
    nProcs = sys.argv[4]        #nProcs
    timeIndex = sys.argv[5]     #timeIndex
    fluidCaseDir = os.getcwd()
    couplingData = pyCoupling.couplingData(fluidCaseDir)
    couplingDict = couplingData.read()
    couplingDict["fluidCaseDir"] = fluidCaseDir
    couplingDict["timeName"] = currTime
    couplingDict["beforeTime"] = beforeTime
    couplingDict["procNo"] = procNo
    couplingDict["nProcs"] = nProcs
    couplingDict["timeIndex"] = timeIndex
    waitCount = int(couplingDict["maxWaitTime"])
    #共有メモリの設定
    #mmCommon = createCommonArea()
    mmCommFile = allComm.createCommonArea(couplingDict)
    mmCommArea = allComm.mmCommonArea(couplingDict, mmCommFile)
    #現在時間が終了時間？
    if float(couplingDict["timeName"]) >= float(couplingDict["endTime"]):
        #最後にrestartFile名をチェック修正する
        chtComm.couplingDict = couplingDict
        chtComm.correctTimeNameRestartFileName(currTime, timeIndex)
        exit()
    #----flagをチェック --------------
    if int(procNo) == 0:
        #busy状態？（自身が起動していれば、終了まで待つ）
        mmCommArea.waitUntilFreeCommonArea()
        #setBusy（busyをセット）
        mmCommArea.writeCommonArea("running python\n")
    #温度のcoupling？
    if pyCoupling.isTempCoupling(couplingDict) == True:
        #mainへ
        stat = coupling_chtChtss()
        if stat != "":
            raise NameError(stat)
    #変位のcoupling？
    if pyCoupling.isDispCoupling(couplingDict) == True:
        #cht時に計算したdispFileを作成
        createNextDisp()
    #---- flagをクリア -------------
    if int(procNo) == 0:
        #clearBusy（freeをセット）
        mmCommArea.writeCommonArea("")
        

