#!/usr/bin/python3
#
#           chtCommonCincurrent.py
#
#       同時計算用のcht連成の共通関数群
#
#   23/11/12    新規作成
#   24/04/23    deleteResSetRestartFileCht:time2str関数部を削除
#               correctTimeNameRestartFileName:restartFile名のtimeNameが
#               演算誤差で正しく設定されない事があるので、これを修正。
#      05/01    createMappingTempFile_proc:mapping処理に並列処理を追加
#

import os
import time
import pickle
import shutil
import glob
import multiprocessing
import pyFistr_ptc as pyFistr
import mappingFromPointCloud as mapping
import pyCoupling
import chtCommonAlternate as chtCommAltn

couplingDict = {}
waitCount = 5000
waitTime = 0.01
FistrModel_cnt = "FistrModel_cht.cnt"
resHeaderName = "FistrModel_temp.res"
hecmwCtrlFile = "hecmw_ctrl_cht.dat"


#------------------------
#  createHeatFluxPtcFile
#------------------------
def createHeatFluxPtcFile(dataDict, htFile):
    """ heatFluxの点群file作成"""
    chtCommAltn.createHeatFluxPtcFile(dataDict, htFile)
    return

#------------------------
#  createMappingHeatFlux
#------------------------
def createMappingHeatFlux(dataDict):
    """ 各procNo毎のheatFlux点群fileをFistr側にmappingする"""
    chtCommAltn.createMappingHeatFlux(dataDict)
    return

#---------------------
#  setDeltaTInCnt_cht
#---------------------
def setDeltaTInCnt_cht(dataDict):
    """ chtのcntファイルを更新する。"""
    global couplingDict
    couplingDict = dataDict
    currTime = couplingDict["timeName"]
    deltaT = couplingDict["deltaT"]
    timeIndex = couplingDict["timeIndex"]
    solidCaseDir = couplingDict["solidCaseDir"]
    stepTime = float(deltaT)
    runStep = int(timeIndex)
    #cnt読み込み
    cntFile = solidCaseDir + "/FistrModel_cht.cnt"
    f = open(cntFile); lines = f.readlines(); f.close()
    #時間増分、終了stepを修正
    i = pyFistr.getNextHeaderName(lines, 0, "!HEAT", "")
    i += 1
    words = pyFistr.deleteSp(lines[i]).split(",")
    startStep = runStep
    cTime = float(currTime)
    orgStep = 0
    dT = stepTime
    writeInterval = float(couplingDict["writeInterval"])
    nSteps = int(couplingDict["nSteps"])
    endStep, endTime = pyCoupling.getEndStepFromCouplingWriteInterval(
        startStep, cTime, dT, nSteps, writeInterval, orgStep)
    words[0] = str(stepTime)            #時間増分
    words[1] = str(endTime)
    lines[i] = ", ".join(words) + "\n"
    #restartの設定
    if float(currTime) == 0.0:
        #初回は、restartさせずに起動
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=1")
    else:
        #restartで起動する様に修正
        i = pyFistr.getNextHeaderName(lines, 0, "!RESTART", "")
        lines[i] = pyFistr.setValName(lines[i], "FREQUENCY=-1")
    #cntFile保存
    f = open(cntFile, "w"); f.writelines(lines); f.close()
    return endStep, endTime

#--------------------------
#  getTempDictFromResFile
#--------------------------
def getTempDictFromResFile(dataDict, endStep, np):
    """ FrontISTRの温度計算結果を辞書形式で返す。
    dispDict={<nodeNo>: <temp>}"""
    tempDict = chtCommAltn.getTempDictFromResFile(dataDict, endStep, np)
    return tempDict

#----------------
#  createTempPtc
#----------------
def createTempPtc(dataDict, tempDict):
    """ 温度の点群fileを作成"""
    tempPtc = chtCommAltn.createTempPtc(dataDict, tempDict)
    return tempPtc

#-------------------
#  addTempRenameRes
#-------------------
def addTempRenameRes(dataDict, endStep, endTime, np):
    """ mshFileに温度の計算結果を追加してrenameする"""
    global couplingDict
    couplingDict = dataDict
    currTime = couplingDict["timeName"]
    deltaT = couplingDict["deltaT"]
    writeInterval = couplingDict["writeInterval"]
    solidCaseDir = couplingDict["solidCaseDir"]
    #温度データを取得
    edTime = endTime - float(currTime)
    nSteps = int(edTime / float(couplingDict["deltaT"]) + 0.5)
    tempDict = {}
    for n in range(np):
        resFile = solidCaseDir + os.sep + "FistrModel_temp.res." + str(n) + "." + str(nSteps)
        valDict = chtCommAltn.getTempLinesInResFile(resFile)
        tempDict.update(valDict)
    #meshFileに温度データを初期値として追加
    tempLines = []
    for nodeNo in tempDict.keys():
        line = nodeNo + ", " + tempDict[nodeNo] + "\n"
        tempLines.append(line)
    chtCommAltn.addTempDataToMeshFile(tempLines)
    #data保存？
    nextTime = endTime
    if pyCoupling.isWriteInterval(nextTime, writeInterval, deltaT) == True:
        if nSteps != endStep:
            #renameの為の定数を設定
            nn = "%04d" % nSteps
            nnEnd = "%04d" % endStep
            orgName = "FistrModel_cht.vis_psf." + nn
            newName = "FistrModel_cht.vis_psf." + nnEnd
            vtuFolder = solidCaseDir + "/" + orgName
            newFolder = solidCaseDir + "/" + newName
            if os.path.exists(newFolder) == True:
                shutil.rmtree(newFolder)
            shutil.move(vtuFolder, newFolder)
            #process毎にrename
            for n in range(np):
                #温度データをrename
                resFile = solidCaseDir + os.sep + "FistrModel_temp.res." + str(n) + "." + str(nSteps)
                newRes = solidCaseDir + os.sep + "FistrModel_temp.res." + str(n) + "." + str(endStep)
                os.rename(resFile, newRes)
                #vtuファイルをrename
                vtuFile = newFolder + "/" + orgName + "." + str(n) + ".vtu"
                newFile = newFolder + "/" + newName + "." + str(n) + ".vtu"
                shutil.move(vtuFile, newFile)
            #pvtuファイルをrename
            pvtuFile = solidCaseDir + "/" + orgName + ".pvtu"
            newFile = solidCaseDir + "/" + newName + ".pvtu"
            if os.path.exists(newFile) == True:
                os.remove(newFile)
            shutil.move(pvtuFile, newFile)
            #  内容修正（processNoに関する部分）
            chtCommAltn.replaceLineInPvtuFile(newFile, orgName, newName, np)
    return tempLines

#--------------------
#  createTempPtcFile
#--------------------
def createTempPtcFile(dataDict, tempLines):
    """ 温度の点群fileを作成"""
    locTemp = chtCommAltn.createTempPtcFile(dataDict, tempLines)
    return locTemp


#------------------------
#  createMappingTempFile
#------------------------
def createMappingTempFile(dataDict, tempPtc, endStep):
    """ 温度をpatchにmappingしたfileを作成する"""
    global couplingDict
    couplingDict = dataDict
    nProcs = couplingDict["nProcs"]
    np = int(nProcs)
    queue = []
    procs = []
    for n in range(np):
        q = multiprocessing.Queue()
        queue.append(q)
        procNo = str(n)
        p = multiprocessing.Process(
            target=createMappingTempFile_proc,
            args=(tempPtc, endStep, procNo, q)
            )
        p.start()
        procs.append(p)
    #全セットprocessの終了を待つ
    for n in range(np):
        a = queue[n].get()

#
#  createMappingTempFile_proc
def createMappingTempFile_proc(tempPtc, endStep, procNo, q):
    """ 指定したprocNoの変位をpatchにmappingしたfileを作成する"""
    solidCaseDir = couplingDict["solidCaseDir"]
    nProcs = couplingDict["nProcs"]
    nThreads = couplingDict["nThreads"]
    method = couplingDict["method"]
    mesh = mapping.fistrMesh(solidCaseDir) 
    #mappingするpointの座標を取得(face中心座標、point座標の辞書)
    mapFaceCenterDict, mapPointDict = chtCommAltn.getMappedPointLocsCht(procNo)
    #mappingTriFileがあるか？
    if chtCommAltn.isMappingTriFileCht(procNo) == False:
        #存在しない場合、mappingTriDictを取得
        if method == "altn":
            #alternate（交互計算）
            np = max(int(nProcs), int(nThreads))
        else:
            #concurrent（同時計算）
            np = int(nProcs) + int(nThreads)
        mapStat, mappingTriDict = mesh.getMappingTriLocs(tempPtc, mapFaceCenterDict, mapPointDict, np)
        if mapStat != "OK":
            return "error in displacement mapping."
        #mappingTriFileを作成保存
        chtCommAltn.createMappingTriFileCht(mappingTriDict, procNo)
    else:
        #fileを読み込む
        mappingTriDict = chtCommAltn.readMappingTriFileCht(procNo)
    #変位をmapping
    mappingDict = mesh.setMappingDataFromTriLocs(tempPtc, mapFaceCenterDict, mappingTriDict)
    #mapping結果を保存(このfileをOpenFOAM側が読み込む)
    chtCommAltn.saveMappingTemp(mappingDict, endStep, procNo)
    q.put("")

#-----------------------------
#  deleteResSetRestartFileCht
#-----------------------------
def deleteResSetRestartFileCht(dataDict, np):
    """ 結果fileを削除する。
    ただし、writeInterval時は、結果fileとrestartFileを残す"""
    global couplingDict
    couplingDict = dataDict
    timeName = couplingDict["timeName"]
    writeInterval = couplingDict["writeInterval"]
    deltaT = couplingDict["deltaT"]
    timeIndex = couplingDict["timeIndex"]
    solidCaseDir = couplingDict["solidCaseDir"]
    fluidCaseDir = couplingDict["fluidCaseDir"]
    patchName = couplingDict["mappingPatch"]
    nSteps = couplingDict["nSteps"]
    nProcs = couplingDict["nProcs"]
    #writeIntervalかどうか？
    dataDir = fluidCaseDir + "/coupling_FrontISTR/data/"
    delTime = timeName
    delStep = timeIndex
    delFiles = []
    delFolders = []
    #OpenFOAM側を確認
    #if pyCoupling.isWriteInterval(timeName, writeInterval, deltaT) == False:
    if pyCoupling.isWriteInterval(delTime, writeInterval, deltaT) == False:
        #削除可能な時間
        #  heatFluxの点群ファイルを削除
        pattern = solidCaseDir + "/" + patchName + "_faceCenterHeatFlux_" + delTime + "_*.particles"
        fileNames = chtCommAltn.getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  温度tempを削除
        pattern = dataDir + patchName + "_temp_" + delStep + "_*tm"
        fileNames = chtCommAltn.getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
        #  faceHeatFluxを削除
        pattern = dataDir + patchName + "_faceHeatFlux_" + delTime + "_*tm"
        fileNames = chtCommAltn.getAllProcsFiles(pattern, int(nProcs))
        delFiles += fileNames
    else:
        #writeInterval時の場合
        #  heatFluxの点群ファイル
        pattern = solidCaseDir + "/" + patchName + "_faceCenterHeatFlux_" + delTime + "_*.particles"
        fileNames = chtCommAltn.getAllProcsFiles(pattern, int(nProcs))                
        delFiles += fileNames
        #  温度tempをrename
        pattern = dataDir + patchName + "_temp_" + delStep + "_*tm"
        fileNames = chtCommAltn.getAllProcsFiles(pattern, int(nProcs))
        chtCommAltn.delete2charaFromEndName(fileNames)
        #  faceHeatFluxをrename
        pattern = dataDir + patchName + "_faceHeatFlux_" + delTime + "_*tm"
        fileNames = chtCommAltn.getAllProcsFiles(pattern, int(nProcs))
        chtCommAltn.delete2charaFromEndName(fileNames)
    #FrontISTR側を確認
    if pyCoupling.isWriteInterval(delTime, writeInterval, deltaT) == False:
        #削除可能な時間
        #  resFileを削除
        resPattern = solidCaseDir + "/FistrModel_temp.res.*." + delStep
        fileNames = chtCommAltn.getAllProcsFiles(resPattern, np)
        #delFiles += [resFile]
        delFiles += fileNames
        #  vtuFileを削除
        vtuFile = solidCaseDir + "/FistrModel_cht.vis_psf." + delStep.zfill(4) + ".pvtu"
        delFiles += [vtuFile]
        #  vtuFolder
        vtuFolder = solidCaseDir + "/FistrModel_cht.vis_psf." + delStep.zfill(4)
        delFolders += [vtuFolder]
    else:
        #timeNameがwriteIntervalの時
        #  timeNameのrestartFile名を修正
        correctTimeNameRestartFileName(timeName, timeIndex)
    #FrontISTRのrestartFileを保存
    calcEnd = float(timeName) + int(nSteps) * float(deltaT)
    #calcEndTime = pyCoupling.time2str(calcEnd)
    calcEndTime = str(calcEnd)
    if pyCoupling.isWriteInterval(calcEndTime, writeInterval, deltaT) == True:
        #writeInterval時の時
        #  resFileをrenameして保存
        calcEndIdx = str(int(timeIndex) + int(nSteps)) 
        chtCommAltn.saveRestartFileCht(calcEndTime, calcEndIdx, np)
    #fileFolderを削除
    for delFile in delFiles:
        if os.path.exists(delFile) == True:
            os.remove(delFile)
    for delFolder in delFolders:
        if os.path.isdir(delFolder) == True:
            shutil.rmtree(delFolder)

#
#  correctTimeNameRestartFileName
def correctTimeNameRestartFileName(timeName, timeIndex):
    """ timeIndexのrestartFile名の時間をtimeNameに設定する"""
    solidCaseDir = couplingDict["solidCaseDir"]
    pattern = "FistrModel_cht.restart_*_" + timeIndex + ".*"
    files = glob.glob(solidCaseDir + "/" + pattern)
    for fileName in files:
        words = fileName.split("_")
        if words[-2] != timeName:
            words[-2] = timeName
            newName = "_".join(words)
            os.rename(fileName, newName)


